/**
 *	@file NMEMain.c
 *	@brief Test program for Nyctergatis Markup Engine.
 *	@author Yves Piguet.
 *	@copyright 2007-2012, Yves Piguet.
 *
 *	@section Usage Usage
 *	This program filters standard input and writes the result to
 *	sandard output. It can be called as follows to convert file
 *	readme.nme to HTML file readme.html:
 *	@code
 *	./nme <readme.nme >readme.html
 *	@endcode
 *	Here is the list of options it supports:
 *	- \c --1eol           single eol as paragraph breaks
 *	- \c --2eol           double eol as paragraph breaks (default)
 *	- \c --autocclink     automatic conversion of camelCase words to links
 *	- \c --autourl        automatic conversion of URLs to links
 *	- \c --body           naked body without header and footer
 *	- \c --checkhooks     check hooks
 *	- \c --debug          XML debug output, sublists outside list items
 *	- \c --debug2         XML debug output, sublists inside list items
 *	- \c --easylink \e format
 *                        links are converted as follows: letters, digits,
 *                        hyphens, commas, dots, apostrophes, parentheses,
 *                        colons and underscores are preserved, spaces are
 *                        converted to underscores, non-ascii characters are
 *                        URL-encoded, other characters are ignored, and the
 *                        output link is obtained by replacing the dollar character
 *                        in \e format with the processed link (e.g. \e format
 *                        could be \e '/wiki/$.html'). URL are used verbatim.
 *	- \c --editfrag \e index \e length \e beg \e end
 *                        edit a fragment of the source code; in output,
 *                        replace what corresponds to input fragment starting
 *                        at \e index of length \e length with this unmodified
 *                        input fragment surrounded by \e beg and \e end
 *	- \c --epub           EPUB output
 *	- \c --headernum1     numbering of level-1 headers
 *	- \c --headernum2     numbering of level-2 headers
 *	- \c --fontsize \e s  font size (0=default)
 *	- \c --help           this help message
 *	- \c --html           HTML output (default)
 *	- \c --jspwiki        JSPWiki output
 *	- \c --latex          LaTeX output
 *	- \c --man            man page output
 *	- \c --mediawiki      Mediawiki output
 *	- \c --nme            NME output
 *	- \c --null           no output (still process input)
 *	- \c --strictcreole   dble tt, u, sub/sup, DL, ind par and esc and eble tt nowiki
 *  - \c --structdiv      display division structure
 *  - \c --structpar      display paragraph structure
 *  - \c --structspan     display span structure
 *	- \c --rtf            RTF output
 *	- \c --test           test (validation of style string nesting)
 *	- \c --testoutput     test output
 *	- \c --testoutput2    test output, sublists inside list items
 *	- \c --text           plain text output
 *	- \c --textc          compact plain text output
 *	- \c --xref           headings have hyperlink target labels
 */

/* License: new BSD license (see NME.h) */

// To compile: gcc -o nme NME.c NMEAutolink.c NMEMain.c

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "NME.h"
#include "NMETest.h"
#include "NMEEPub.h"
#include "NMEAutolink.h"
#include "NMEPluginRot13.h"
#include "NMEPluginReverse.h"
#include "NMEPluginUppercase.h"
#include "NMEPluginCalendar.h"
#include "NMEPluginRaw.h"
#include "NMEPluginTOC.h"
#include "NMEPluginWiki.h"

/// Initial size for reading input
#define INITIALSIZE (32 * 1024)

/// Format strings for slides in HTML
static NMEOutputFormat const NMEOutputFormatSlidesHTML =
{
	" ",	// space
	2,	// indentSpaces
	0,	// defFontSize
	'%',	// ctrlChar
//	"<!-- Generated by Nyctergatis Markup Engine, "
//			__DATE__ " " __TIME__ " -->\n"
//	"<html><head><title>Test of CSS</title></head>\n"
	"<style type=\"text/css\">\n"
//	"body{background:white;margin:1em 1em}\n"
	"div.SL *,div.FSL *,div.TSL * {background:inherit;color:white}\n"
	"div.TSL h1,div.SL h2,div.SL h3,div.SL h4{color:white}\n"
	"div.SL,div.TSL {background:#000099;width:300pt;height:200pt;\n"
	"padding:20pt;margin:10pt;page-break-inside:avoid;position:relative}\n"
	"div.FSL{position:absolute;bottom:10px;right:10px}\n"
	"div.SL h1,div.SL h2,div.SL h3,div.SL p,div.SL dl,div.SL dt,div.SL ul,div.SL table,div.SL td,div.SL th,div.SL address{font-family:verdana,arial,sans-serif}\n"
	"div.SL h1,div.SL h2,div.TSL h1,div.TSL h2{text-align:center}\n"
	"div.TSL h1{font-size: 20pt;margin-top:50pt}\n"
	"div.SL h2{font-size:18pt}\n"
	"div.SL h3{font-size:13pt}\n"
	"div.TSL p,div.SL p,div.SL dl,div.SL dt,div.SL ul,div.SL table,div.SL td,div.SL th,div.SL pre,div.SL address{font-size:11pt;line-height:1.3;\n"
	"margin-top:3pt;margin-bottom:6pt}\n"
	"div.SL p,div.SL dl,div.SL ul{text-align:justify}\n"
	"div.SL table{border-style=solid;border-width=1px;border-color=white}\n"
	"div.FSL p{font-size:9pt;color:#6666ff}\n"
	"div.TSL p{text-align:center}\n"
	"</style>\n"
	"<div class=\"TSL\">\n",
		"</div>"
//		"</body></html>\n"
		,	// doc
	4,	// highest heading level
	"%%{l=2&i>1}<div class=\"FSL\"><p>%{i-1}</p></div>\n%%"
	"%%{l=2}</div>\n<div class=\"SL\">\n%%"
			"<h%{l}"
			"%%{x} id=\"h%{o}\"%%"
			"%%{s>0} style=\"font-size:%{l=1&3*s|l=2&2*s|l=3&3*s/2|5*s/4}pt\"%%>",
		"</h%{l}>\n",	// heading
	"<p%%{s>0} style=\"font-size:%{s}pt\"%%>", "</p>\n",	// par
	"<br />",	// line break
	"<pre%%{s>0} style=\"font-size:%{s}pt\"%%>\n", "</pre>\n",	// pre
	"", "\n",	// pre line
	FALSE,	// sublistInListItem
	"<ul>\n", "</ul>\n",	// UL
	"<li%%{s>0} style=\"font-size:%{s}pt\"%%>", "</li>\n",	// UL line
	"<ol>\n", "</ol>\n",	// OL
	"<li%%{s>0} style=\"font-size:%{s}pt\"%%>", "</li>\n",	// OL line
	"<dl>\n", "</dl>\n",	// DL
	"<dt%%{s>0} style=\"font-size:%{s}pt\"%%>", "</dt>\n",	// DT
	NULL,	// emptyDT
	"<dd%%{s>0} style=\"font-size:%{s}pt\"%%>", "</dd>\n",	// DD
	"<div style=\"margin-left:2em%%{s>0}; font-size:%{s}pt%%\">\n",
		"</div>\n",	// indented section
	"<p%%{s>0} style=\"font-size:%{s}pt\"%%>", "</p>\n",	// indented par
	"<table align=\"center\">\n", "</table>\n",	// table
	"<tr>", "</tr>\n",	// table row
	"<th%%{s>0} style=\"font-size:%{s}pt\"%%>", "</th>\n",	// table header cell
	"<td%%{s>0} style=\"font-size:%{s}pt\"%%>", "</td>\n",	// table normal cell
	"<hr />\n",	// hr
#if defined(useHTMLEmphasisTags)
	"<strong>", "</strong>",	// strong emphasis for **...**
	"<em>", "</em>",	// emphasis for //...//
#else
	"<b>", "</b>",	// bold
	"<i>", "</i>",	// italic
#endif
	"<u>", "</u>",	// underline
	"<sup>", "</sup>",	// superscript
	"<sub>", "</sub>",	// subscript
	"<tt>", "</tt>",	// monospace
	"<a href=\"", "</a>", "\">", FALSE,	// URL
	"<img src=\"", "\" />", "\" alt=\"", FALSE, TRUE,	// image
	NULL,	// interwiki
	NULL, NULL,	// encodeURLFun
	NMEEncodeCharFunDict, (void *)NMEXMLCharDict,	// char encoder
	NMEEncodeCharFunDict, (void *)NMEXMLCharDict,	// char pre encoder
	70, NULL, NULL,	// wordwrap
	NULL, NULL,	// char hook
	NULL, NULL, NULL, NULL,	// process hooks
	NULL,	// plugins
	NULL,	// autoconverts
	NULL, NULL	// getVar
};

/// Format strings for Mediawiki output (NOT FINISHED!)
static NMEOutputFormat const NMEOutputFormatMediawiki =
{
	" ",	// space
	0,	// indentSpaces
	10,	// defFontSize
	'%',	// ctrlChar
	"", "",	// doc
	6,	// highest heading level
	"%%{l}=%%", "%%{l}=%%\n",	// heading
	"", "\n\n",	// par
	"<br>",	// line break
	"", "\n",	// pre
	" ", "\n",	// pre line
	FALSE,	// sublistInListItem
	"", "%%{l=1}\n%%",	// UL
	"%%{l}*%% ", "\n",	// UL line
	"", "%%{l=1}\n%%",	// OL
	"%%{l}#%% ", "\n",	// OL line
	"", "%%{l=1}\n%%",	// DL
	"%%{l};%% ", "\n",	// DT
	NULL,	// emptyDT
	": ", "\n",	// DD
	"", "%%{l=1}\n%%",	// indented section
	"%%{l}:%% ", "\n",	// indented par
	"{|\n", "|}\n",	// table
	"|-\n", "",	// table row
	"!", "\n",	// table header cell
	"|", "\n",	// table normal cell
	"----\n\n",	// hr									??
	"'''", "'''",	// bold
	"''", "''",	// italic
	"__", "__",	// underline							??
	"<sup>", "</sup>",	// superscript
	"<sub>", "</sub>",	// subscript
	"<code>", "</code>",	// monospace
	"[", "]", "|", FALSE,	// URL
	"[[Image:", "]]", "|", FALSE, TRUE,	// image
	NULL,	// interwiki
	NULL, NULL,	// encodeURLFun
	NULL, NULL,	// char encoder							??
	NULL, NULL,	// char pre encoder						??
	65, NULL, NULL,	// wordwrap
	NULL, NULL,	// char hook
	NULL, NULL, NULL, NULL,	// process hooks
	NULL,	// plugins
	NULL,	// autoconverts
	NULL, NULL	// getVar
};

/** Table of character substitutions for JSPWiki */
static NMEEncodeCharDict const jspwikiCharDict[] =
{
	{'*', "~*"},
	{'#', "~#"},
	{'\\', "~\\"},
	{'_', "~_"},
	{'!', "~!"},
	{'{', "~{"},
	{'}', "~}"},
	{'~', "~~"},
	{0, NULL}
};

/// Format strings for JSPWiki output
static NMEOutputFormat const NMEOutputFormatJSPWiki =
{
	" ",	// space
	0,	// indentSpaces
	10,	// defFontSize
	'?',	// ctrlChar
	"", "",	// doc
	3,	// highest heading level
	"??{4-l}!?? ", "\n",	// heading
	"", "\n\n",	// par
	"\\\\",	// line break
	"{{{\n", "}}}\n\n",	// pre
	"", "\n",	// pre line
	FALSE,	// sublistInListItem
	"", "??{l=1}\n??",	// UL
	"??{l}*?? ", "\n",	// UL line
	"", "??{l=1}\n??",	// OL
	"??{l}#?? ", "\n",	// OL line
	"", "??{l=1}\n??",	// DL
	"%%(text-indent:?{20*l}px)\n__", ":__ ",	// DT
	"%%(text-indent:?{20*l}px)\n",	// emptyDT
	"", "\n%%\n\n",	// DD
	"", "",	// indented section
	"%%(text-indent:?{20*l}px)\n", "\n%%\n\n",	// indented par
	"", "\n",	// table
	"", "\n",	// table row
	"|| ", "",	// table header cell
	"| ", "",	// table normal cell
	"----\n\n",	// hr
	"__", "__",	// bold
	"''", "''",	// italic
	"%%(text-decoration:underline)", "%%",	// underline
	"%%(vertical-align:sup)", "%%",	// superscript
	"%%(vertical-align:sub)", "%%",	// subscript
	"{{", "}}",	// monospace
	"[", "]", "|", TRUE,	// URL
	"", "", NULL, FALSE, FALSE,	// image
	NULL,	// interwiki
	NULL, NULL,	// encodeURLFun
	NMEEncodeCharFunDict, (void *)jspwikiCharDict,	// char encoder
	NULL, NULL,	// char pre encoder
	65, NULL, NULL,	// wordwrap
	NULL, NULL,	// char hook
	NULL, NULL, NULL, NULL,	// process hooks
	NULL,	// plugins
	NULL,	// autoconverts
	NULL, NULL	// getVar
};

/// User data of NMEPluginTOCEntry
static NMEPluginTocData tocData;

/// Table of plugins for conversion to HTML
static NMEPlugin const pluginsHTML[] =
{
	NMEPluginReverseEntry,
	NMEPluginRot13Entry,
	NMEPluginUppercaseEntry,
	NMEPluginRawEntry("rawinpar", kNMEPluginOptDefault),
	NMEPluginRawEntry("rawoutpar", kNMEPluginOptBetweenPar),
	NMEPluginCalendarEntry,
	NMEPluginTOCEntry(&tocData),
	
	NMEPluginTableEnd
};

/// Table of plugins for conversion to all formats but HTML/XML
static NMEPlugin const plugins[] =
{
	NMEPluginReverseEntry,
	NMEPluginRot13Entry,
	NMEPluginUppercaseEntry,
	NMEPluginCalendarEntry,
	
	NMEPluginTableEnd
};

/// Table of plugins for metadata extraction
static NMEPlugin const pluginsMetadata[] =
{
	NMEPluginKeywordsEntry,
	NMEPluginDateEntry(NMEMonthNameEn NMEMonthNameFr NMEMonthNameSp NMEMonthNameIt NMEMonthNameDe),
	NMEPluginExecuteEntry,
	
	NMEPluginTableEnd
};

/// Table of autoconvert functions
static NMEAutoconvert autoconverts[16];

/// Table of interwiki definitions
static NMEInterwiki const interwikis[] =
{
	{"Dictionary:",
		"http://www.dict.org/bin/Dict?Database=*&Form=Dict1&Strategy=*&Query="},
	{"Foldoc:", "http://www.foldoc.org/foldoc/foldoc.cgi?"},
	{"Google:", "http://www.google.com/search?q="},
	{"WhoIs:", "http://www.whois.sc/"},
	{"WikiPedia:", "http://en.wikipedia.org/wiki/"},
	{NULL, NULL}
};

/// Data passed to hookDump
typedef struct
{
	NMEInt nesting;
} HookDumpData;

/// Hook for dumping information about input
static NMEErr hookDump(NMEInt level,
		NMEInt item,
		NMEBoolean enter,
		NMEConstText markup,
		NMEInt srcIndex,
		NMEInt srcLineNumber,
		NMEContext *context,
		void *data)
{
	if (!enter)
		((HookDumpData *)data)->nesting--;
	printf("%*s", 2 * ((HookDumpData *)data)->nesting, " ");
	if (level > 0)
		printf("%-4s L%d %c %5d\n", markup, level, enter ? '<' : '>', srcIndex);
	else
		printf("%-4s    %c %5d\n", markup, enter ? '<' : '>', srcIndex);
	if (enter)
		((HookDumpData *)data)->nesting++;
	return kNMEErrOk;
}

/// Data passed to hookEdit
typedef struct
{
	NMEInt index;
	NMEInt length;
	NMEConstText begin;
	NMEConstText end;
	NMEBoolean done;	///< set to TRUE once the fragment has been processed
} HookEditData;

/// Hook for editing a fragment of input source
static NMEErr hookEdit(NMEInt level,
		NMEInt item,
		NMEBoolean enter,
		NMEConstText markup,
		NMEInt srcIndex,
		NMEInt srcLineNumber,
		NMEContext *context,
		void *data)
{
	HookEditData *hookEditData = (HookEditData *)data;
	NMEErr err;
	
	if (!hookEditData->done
			&& srcIndex >= hookEditData->index)
	{
		if (srcIndex < hookEditData->index + hookEditData->length)
		{
			if (!NMEAddString(hookEditData->begin, -1, '\0', context))
				return kNMEErrNotEnoughMemory;
			err = NMECopySource(hookEditData->length - (hookEditData->index - srcIndex),
					TRUE, TRUE, context);
			if (err != kNMEErrOk)
				return err;
			if (!NMEAddString(hookEditData->end, -1, '\0', context))
				return kNMEErrNotEnoughMemory;
		}
		
		hookEditData->done = TRUE;
	}
	return kNMEErrOk;
}

/// Maximum depth of nested constructs in HookCheckData
#define kHookCheckMaxDepth 256
/// Data passed to hookCheck
typedef struct
{
	NMEInt lastSrcIndex;	///< last srcIndex
	NMEInt depth;	///< depth of nesting (elements in stack[])
	struct
	{
		NMEChar hook;	///< 's'=span, 'p'=par, 'd'=div
		NMEInt level;
		NMEInt item;
		NMEConstText markup;
		NMEInt srcIndex;
		NMEInt srcLineNumber;
	} stack[kHookCheckMaxDepth];	///< nested constructs
} HookCheckData;

/// Hook for checking hooks
static NMEErr hookCheck(NMEChar hook,
		NMEInt level,
		NMEInt item,
		NMEBoolean enter,
		NMEConstText markup,
		NMEInt srcIndex,
		NMEInt srcLineNumber,
		HookCheckData *hookCheckData)
{
	if (srcIndex < hookCheckData->lastSrcIndex)
		fprintf(stderr, "Last srcIndex = %d\n", hookCheckData->lastSrcIndex);
	else if (!enter && hookCheckData->depth <= 0)
		fprintf(stderr, "Outside any structure\n");
	else if (!enter
			&& (hookCheckData->stack[hookCheckData->depth - 1].hook != hook
				|| hookCheckData->stack[hookCheckData->depth - 1].level != level
				|| hookCheckData->stack[hookCheckData->depth - 1].item != item
				|| strcmp(hookCheckData->stack[hookCheckData->depth - 1].markup, markup)))
	{
		fprintf(stderr, "Nonmatching exit:\n");
		fprintf(stderr,
				" hook = %c, level = %d, item = %d, markup = \"%s\", srcIndex = %d, srcLineNumber = %d\n",
				hookCheckData->stack[hookCheckData->depth - 1].hook,
				hookCheckData->stack[hookCheckData->depth - 1].level,
				hookCheckData->stack[hookCheckData->depth - 1].item,
				hookCheckData->stack[hookCheckData->depth - 1].markup,
				hookCheckData->stack[hookCheckData->depth - 1].srcIndex,
				hookCheckData->stack[hookCheckData->depth - 1].srcLineNumber);
	}
	else if (enter && hookCheckData->depth >= kHookCheckMaxDepth)
		fprintf(stderr, "Nesting too deep\n");
	else if (enter)
	{
		hookCheckData->stack[hookCheckData->depth].hook = hook;
		hookCheckData->stack[hookCheckData->depth].level = level;
		hookCheckData->stack[hookCheckData->depth].item = item;
		hookCheckData->stack[hookCheckData->depth].markup = markup;
		hookCheckData->stack[hookCheckData->depth].srcIndex = srcIndex;
		hookCheckData->stack[hookCheckData->depth].srcLineNumber = srcLineNumber;
		hookCheckData->depth++;
		return kNMEErrOk;
	}
	else
	{
		hookCheckData->depth--;
		return kNMEErrOk;
	}
	
	fprintf(stderr,
			"Hook = %c, level = %d, item = %d, %s, markup = \"%s\", srcIndex = %d, srcLineNumber = %d\n",
			hook, level, item, enter ? "enter" : "exit", markup, srcIndex, srcLineNumber);
	
	return kNMEErrInternal;
}

/// Hook for checking div hook
static NMEErr divHookCheck(NMEInt level,
		NMEInt item,
		NMEBoolean enter,
		NMEConstText markup,
		NMEInt srcIndex,
		NMEInt srcLineNumber,
		NMEContext *context,
		void *data)
{
	return hookCheck('d', level, item, enter, markup, srcIndex, srcLineNumber,
			(HookCheckData *)data);
}

/// Hook for checking par hook
static NMEErr parHookCheck(NMEInt level,
		NMEInt item,
		NMEBoolean enter,
		NMEConstText markup,
		NMEInt srcIndex,
		NMEInt srcLineNumber,
		NMEContext *context,
		void *data)
{
	return hookCheck('p', level, item, enter, markup, srcIndex, srcLineNumber,
			(HookCheckData *)data);
}

/// Hook for checking span hook
static NMEErr spanHookCheck(NMEInt level,
		NMEInt item,
		NMEBoolean enter,
		NMEConstText markup,
		NMEInt srcIndex,
		NMEInt srcLineNumber,
		NMEContext *context,
		void *data)
{
	return hookCheck('s', level, item, enter, markup, srcIndex, srcLineNumber,
			(HookCheckData *)data);
}

/// Easylink callback (replace dollar in string pointed by data with link,
/// where letters, digits, hyphens, commas, dots, apostrophes, parentheses,
/// colons and underscores are preserved, spaces are converted to underscores,
/// non-ascii characters are URL-encoded, and other characters are ignored;
//  absolute URL are copied unprocessed)
NMEErr encodeURLEasylink(NMEConstText link, NMEInt linkLen,
		NMEContext *context, void *data)
{
	NMEInt i, j;
	NMEConstText format = (NMEConstText)data;
#define isAlphaNum(c) \
	((c) >= 'a' && (c) <= 'z' || (c) >= 'A' && (c) <= 'Z' \
			|| (c) >= '0' && (c) <= '9')
#define toHexa(d) \
	((d) >= 10 ? (d) + ('a' - 10) : (d) + '0')
	
	// URL?
	for (i = 0; i < linkLen && isAlphaNum(link[i]); i++)
		;
	if (i < linkLen && link[i] == ':')
	{
		// yes: copy verbatim
		if (!NMEAddString(link, linkLen, 0, context))
			return kNMEErrNotEnoughMemory;
		return kNMEErrOk;
	}
	
	for (i = 0; format[i]; i++)
		if (format[i] == '$')
		{
			for (j = 0; j < linkLen; j++)
				if (isAlphaNum(link[j]) || link[j] == '-' || link[j] == ',' || link[j] == '.'
						|| link[j] == '\'' || link[j] == '(' || link[j] == ')'
						|| link[j] == ':')
				{
					if (!NMEAddString(&link[j], 1, 0, context))
						return kNMEErrNotEnoughMemory;
				}
				else if (link[j] == '_' || link[j] == ' ')
				{
					if (!NMEAddString("_", 1, 0, context))
						return kNMEErrNotEnoughMemory;
				}
				else if (link[j] & 0x80)	// non-ascii
				{
					char e[3];
					e[0] = '%';
					e[1] = toHexa((link[j] >> 4) & 0x0f);
					e[2] = toHexa(link[j] & 0x0f);
					if (!NMEAddString(e, 3, 0, context))
						return kNMEErrNotEnoughMemory;
				}
		}
		else if (!NMEAddString(&format[i], 1, 0, context))
			return kNMEErrNotEnoughMemory;
		
	return kNMEErrOk;
}

/// Application entry point
int main(int argc, char **argv)
{
	NMEInt size;
	NMEText src = NULL, buf, dest;
	NMEInt srcLen, destLen;
	NMEOutputFormat outputFormat = NMEOutputFormatHTML;
	NMEInt options = kNMEProcessOptDefault;
	NMEBoolean autoURLLink = FALSE, autoCCLink = FALSE;
	NMEBoolean testPhase = 0;	// no test by default
	int i;
	int fontSize = 0;
	HookDumpData hookDumpData;
	HookTOCData hookTOCData;
	HookEditData hookEditData;
	HookCheckData hookCheckData;
	NMEErr err;
	NMEInt lineNumber;
	
	outputFormat.plugins = pluginsHTML;
	
	for (i = 1; i < argc; i++)
		if (!strcmp(argv[i], "--body"))
			options |= kNMEProcessOptNoPreAndPost;
		else if (!strcmp(argv[i], "--1eol"))
			options |= kNMEProcessOptNoMultilinePar;
		else if (!strcmp(argv[i], "--2eol"))
			options &= ~kNMEProcessOptNoMultilinePar;
		else if (!strcmp(argv[i], "--xref"))
			options |= kNMEProcessOptXRef;
		else if (!strcmp(argv[i], "--autocclink"))
			autoCCLink = TRUE;
		else if (!strcmp(argv[i], "--autourllink"))
			autoURLLink = TRUE;
		else if (!strcmp(argv[i], "--nme"))
		{
			outputFormat = NMEOutputFormatNME;
			outputFormat.plugins = plugins;
		}
		else if (!strcmp(argv[i], "--html"))
		{
			outputFormat = NMEOutputFormatHTML;
			outputFormat.plugins = pluginsHTML;
		}
		else if (!strcmp(argv[i], "--slides"))
		{
			outputFormat = NMEOutputFormatSlidesHTML;
			outputFormat.plugins = pluginsHTML;
		}
		else if (!strcmp(argv[i], "--epub"))
		{
			outputFormat = NMEOutputFormatOPSXHTML;
			outputFormat.plugins = pluginsHTML;
		}
		else if (!strcmp(argv[i], "--jspwiki"))
		{
			outputFormat = NMEOutputFormatJSPWiki;
			outputFormat.plugins = plugins;
		}
		else if (!strcmp(argv[i], "--latex"))
		{
			outputFormat = NMEOutputFormatLaTeX;
			outputFormat.plugins = plugins;
		}
		else if (!strcmp(argv[i], "--mediawiki"))
		{
			outputFormat = NMEOutputFormatMediawiki;
			outputFormat.plugins = plugins;
		}
		else if (!strcmp(argv[i], "--null"))
		{
			outputFormat = NMEOutputFormatNull;
			outputFormat.plugins = NULL;
		}
		else if (!strcmp(argv[i], "--null-plugins"))
		{
			outputFormat = NMEOutputFormatNull;
			outputFormat.plugins = plugins;
		}
		else if (!strcmp(argv[i], "--rtf"))
		{
			outputFormat = NMEOutputFormatRTF;
			outputFormat.plugins = plugins;
		}
		else if (!strcmp(argv[i], "--metadata"))
		{
			outputFormat = NMEOutputFormatNull;
			outputFormat.plugins = pluginsMetadata;
		}
		else if (!strcmp(argv[i], "--editfrag") && i + 4 < argc)
		{
			outputFormat.parHookFun = hookEdit;
			hookEditData.index = strtol(argv[++i], NULL, 0);
			hookEditData.length = strtol(argv[++i], NULL, 0);
			hookEditData.begin = argv[++i];
			hookEditData.end = argv[++i];
			hookEditData.done = FALSE;
			outputFormat.hookData = (void *)&hookEditData;
		}
		else if (!strcmp(argv[i], "--structdiv"))
		{
			outputFormat = NMEOutputFormatNull;
			outputFormat.divHookFun = hookDump;
			hookDumpData.nesting = 0;
			outputFormat.hookData = (void *)&hookDumpData;
		}
		else if (!strcmp(argv[i], "--structpar"))
		{
			outputFormat = NMEOutputFormatNull;
			outputFormat.parHookFun = hookDump;
			hookDumpData.nesting = 0;
			outputFormat.hookData = (void *)&hookDumpData;
		}
		else if (!strcmp(argv[i], "--structspan"))
		{
			outputFormat = NMEOutputFormatNull;
			outputFormat.spanHookFun = hookDump;
			hookDumpData.nesting = 0;
			outputFormat.hookData = (void *)&hookDumpData;
		}
		else if (!strcmp(argv[i], "--checkhooks"))
		{
			outputFormat = NMEOutputFormatNull;
			outputFormat.divHookFun = divHookCheck;
			outputFormat.parHookFun = parHookCheck;
			outputFormat.spanHookFun = spanHookCheck;
			hookCheckData.lastSrcIndex = 0;
			hookCheckData.depth = 0;
			outputFormat.hookData = (void *)&hookCheckData;
		}
		else if (!strcmp(argv[i], "--debug"))
		{
			outputFormat = NMEOutputFormatDebug;
			outputFormat.plugins = plugins;
		}
		else if (!strcmp(argv[i], "--debug2"))
		{
			outputFormat = NMEOutputFormatDebug;
			outputFormat.sublistInListItem = TRUE;
			outputFormat.plugins = plugins;
		}
		else if (!strcmp(argv[i], "--easylink"))
		{
			outputFormat.encodeURLFun = encodeURLEasylink;
			outputFormat.encodeURLData = argv[++i];
		}
		else if (!strcmp(argv[i], "--test"))
		{
			outputFormat = NMEOutputFormatTest;
			outputFormat.plugins = plugins;
			testPhase = 1;	// sublistInListItem = FALSE
			
		}
		else if (!strcmp(argv[i], "--testoutput"))
		{
			outputFormat = NMEOutputFormatTest;
			outputFormat.plugins = plugins;
			
		}
		else if (!strcmp(argv[i], "--testoutput2"))
		{
			outputFormat = NMEOutputFormatTest;
			outputFormat.sublistInListItem = TRUE;
			outputFormat.plugins = plugins;
			
		}
		else if (!strcmp(argv[i], "--text"))
		{
			outputFormat = NMEOutputFormatText;
			outputFormat.plugins = plugins;
		}
		else if (!strcmp(argv[i], "--textc"))
		{
			outputFormat = NMEOutputFormatTextCompact;
			outputFormat.plugins = plugins;
		}
		else if (!strcmp(argv[i], "--man"))
		{
			outputFormat = NMEOutputFormatMan;
			outputFormat.plugins = plugins;
		}
		else if (!strcmp(argv[i], "--fontsize") && i + 1 < argc)
			fontSize = strtol(argv[++i], NULL, 0);
		else if (!strcmp(argv[i], "--headernum1"))
			options |= kNMEProcessOptH1Num;
		else if (!strcmp(argv[i], "--headernum2"))
			options |= kNMEProcessOptH2Num;
		else if (!strcmp(argv[i], "--strictcreole"))
			options |= kNMEProcessOptNoUnderline | kNMEProcessOptNoMonospace
					| kNMEProcessOptNoSubSuperscript | kNMEProcessOptNoIndentedPar
					| kNMEProcessOptNoDL | kNMEProcessOptVerbatimMono;
		else if (!strcmp(argv[i], "--toc"))
			NMESetTOCOutputFormat(&outputFormat, &hookTOCData);
		else
		{
			if (strcmp(argv[i], "--help"))
				fprintf(stderr, "Unknown option %s\n", argv[i]);
			fprintf(stderr, "Usage: %s [options]\n"
					"Filter NME stdin and renders it to another format.\n"
					"--1eol            single eol as paragraph breaks\n"
					"--2eol            double eol as paragraph breaks (default)\n"
					"--autocclink      automatic conversion of camelCase words to links\n"
					"--autourllink     automatic conversion of URLs to links\n"
					"--body            naked body without header and footer\n"
					"--checkhooks      check hooks\n"
					"--debug           XML debug format, sublists outside list items\n"
					"--debug2          XML debug format, sublists inside list items\n"
					"--easylink format links are converted as follows: letters, digits,\n"
					"                  hyphens commas, dots, apostrophes, parentheses,\n"
					"                  colons and underscores are preserved, spaces are\n"
					"                  converted to underscores, non-ascii characters are\n"
					"                  URL-encoded, other characters are ignored, and the\n"
					"                  output link is obtained by replacing the dollar\n"
					"                  character in format with the processed link (e.g.\n"
					"                  format could be '/wiki/$.html').\n"
					"                  URL are used verbatim.\n"
					"--editfrag i l b e   edit a fragment of the source code; in output,\n"
					"                  replace what corresponds to input fragment starting\n"
					"                  at i of length l with this unmodified input fragment\n"
					"                  surrounded by b and e\n"
					"--headernum1      numbering of level-1 headers\n"
					"--headernum2      numbering of level-2 headers\n"
					"--fontsize s      font size (0=default)\n"
					"--help            this help message\n"
					"--html            HTML output (default)\n"
					"--jspwiki         JSPWiki output\n"
					"--latex           LaTeX output\n"
					"--man             man page output\n"
					"--mediawiki       MediaWiki output\n"
					"--metadata        metadata extraction\n"
					"--nme             NME output\n"
					"--null            no output, plugins disabled (still process input)\n"
					"--null-plugins    no normal output, but process plugins\n"
					"--strictcreole    disable monospace, underline, subscript,\n"
					"                  superscript, definition lists, and indented\n"
					"                  paragraphs; and enable nowiki monospace\n"
					"--structdiv       display division structure\n"
					"--structpar       display paragraph structure\n"
					"--structspan      display span structure\n"
					"--rtf             RTF output\n"
					"--test            test (validation of style string nesting)\n"
					"--testoutput      test output\n"
					"--testoutput2     test output, sublists inside list items\n"
					"--slides          HTML slides output\n"
					"--text            plain text output\n"
					"--textc           compact plain text output\n"
					"--xref            headings have hyperlink target labels\n",
				argv[0]);
			exit(0);
		}
	
	outputFormat.interwikis = interwikis;
	if (autoCCLink || autoURLLink)
	{
		int n = 0;
		
		if (autoCCLink)
			autoconverts[n++].cb = NMEAutoconvertCamelCase;
		if (autoURLLink)
			autoconverts[n++].cb = NMEAutoconvertURL;

		outputFormat.autoconverts = autoconverts;
	}
	
	size = INITIALSIZE;
	src = malloc(size);
	for (srcLen = 0; ; )
	{
		if (!src)
			exit(1);
		i = fread(src + srcLen, 1, size - srcLen, stdin);
		if (i == 0)
			break;
		else if (i < 0)
		{
			free(src);
			exit(2);
		}
		srcLen += i;
		size *= 2;
		src = realloc(src, size);
	}
	
	size *= 4;	// likely enough
	buf = malloc(size);
	if (!buf)
		exit(1);
	
	tocData.src = src;
	tocData.srcLen = srcLen;
	
process:
	err = NMEProcess(src, srcLen,
			buf, size,
			options, "\n", &outputFormat, fontSize,
			&dest, &destLen, NULL);
	if (err == kNMEErrNotEnoughMemory)
	{
		free(buf);
		size *= 2;
		buf = malloc(size);
		if (!buf)
			exit(1);
		goto process;
	}
	
	if (err != kNMEErrOk)
		printf("Error %d\n", err);
	else switch (testPhase)
	{
		case 0:	// no test, just plain output
			printf("%.*s", destLen, dest);
			break;
		case 1:	// sublistInListItem = FALSE
			err = NMETest(dest, destLen, &lineNumber);
			if (err != kNMEErrOk)
			{
				printf("Error %d (sublistInListItem = FALSE, line = %d)\n",
						err, lineNumber);
				break;
			}
			outputFormat = NMEOutputFormatTest;
			outputFormat.sublistInListItem = TRUE;
			testPhase++;
			goto process;
		case 2:	// sublistInListItem = TRUE
			err = NMETest(dest, destLen, &lineNumber);
			if (err != kNMEErrOk)
				printf("Error %d (sublistInListItem = TRUE, line = %d)\n",
						err, lineNumber);
			break;
	}
	
	free((void *)buf);
	free((void *)src);
	
	return err != kNMEErrOk;
}
