/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2004 Masanao Izumo <iz@onicos.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

    The 8-bit uLaw to 16-bit PCM and the 13-bit-PCM to 8-bit uLaw
    tables were lifted from the rsynth-2.0 sources.  The README says:

    This is a text to speech system produced by integrating various pieces
    of code and tables of data, which are all (I believe) in the public domain.

    The bulk of the intergration was done by myself, that is Nick Ing-Simmons.
    I can be reached via my employer at nik@tiuk.ti.com.
*/

#include <stdio.h>
#include <math.h>
#include "timidity.h"
#include "tables.h"
#include "instrum.h"


namespace TimidityPlus
{

int32_t freq_table[128];
int32_t freq_table_tuning[128][128];
int32_t freq_table_pytha[24][128];
int32_t freq_table_meantone[48][128];
int32_t freq_table_pureint[48][128];


static void init_freq_table(void)
{
	int i;
	
	for (i = 0; i < 128; i++) {
		freq_table[i] = 440 * pow(2.0, (i - 69) / 12.0) * 1000 + 0.5;
	}
}

static void init_freq_table_tuning(void)
{
	int p, i;
	double f;
	
	for (i = 0; i < 128; i++)
			freq_table_tuning[0][i] = freq_table[i];
	for (i = 0; i < 128; i++) {
		f = 440 * pow(2.0, (i - 69) / 12.0);
		for (p = 1; p < 128; p++)
			freq_table_tuning[p][i] = f * 1000 + 0.5;
	}
}

static void init_freq_table_pytha(void)
{
	int i, j, k, l;
	double f;
	static const double major_ratio[] = {
		  1.0 /  1, 256.0 / 243,   9.0 /   8,  32.0 /  27,
		 81.0 / 64,   4.0 /   3, 729.0 / 512,   3.0 /   2,
		128.0 / 81,  27.0 /  16,  16.0 /   9, 243.0 / 128
	};
	static const double minor_ratio[] = {
		   1.0 /    1, 2187.0 / 2048,   9.0 /   8, 19683.0 / 16384,
		  81.0 /   64,    4.0 /    3, 729.0 / 512,     3.0 /     2,
		6561.0 / 4096,   27.0 /   16,  16.0 /   9,   243.0 /   128
	};
	
	for (i = 0; i < 12; i++)
		for (j = -1; j < 11; j++) {
			f = 440 * pow(2.0, (i - 9) / 12.0 + j - 5);
			for (k = 0; k < 12; k++) {
				l = i + j * 12 + k;
				if (l < 0 || l >= 128)
					continue;
				freq_table_pytha[i][l] = f * major_ratio[k] * 1000 + 0.5;
				freq_table_pytha[i + 12][l] = f * minor_ratio[k] * 1000 + 0.5;
			}
		}
}

static void init_freq_table_meantone(void)
{
	int i, j, k, l;
	double f;
	static double major_ratio[12], minor_ratio[12];
	static const double sc = 81.0 / 80;
	
	major_ratio[0] = 1;
	major_ratio[1] = 8 / pow(5.0, 5.0 / 4);
	major_ratio[2] = pow(5.0, 1.0 / 2) / 2;
	major_ratio[3] = 4 / pow(5.0, 3.0 / 4);
	major_ratio[4] = 5.0 / 4;
	major_ratio[5] = 2 / pow(5.0, 1.0 / 4);
	major_ratio[6] = pow(5.0, 3.0 / 2) / 8;
	major_ratio[7] = pow(5.0, 1.0 / 4);
	major_ratio[8] = 8.0 / 5;
	major_ratio[9] = pow(5.0, 3.0 / 4) / 2;
	major_ratio[10] = 4 / pow(5.0, 1.0 / 2);
	major_ratio[11] = pow(5.0, 5.0 / 4) / 4;
	minor_ratio[0] = 1;
	minor_ratio[1] = pow(10.0 / 3, 7.0 / 3) / 16;
	minor_ratio[2] = pow(10.0 / 3, 2.0 / 3) / 2;
	minor_ratio[3] = 125.0 / 108;
	minor_ratio[4] = pow(10.0 / 3, 4.0 / 3) / 4;
	minor_ratio[5] = 2 / pow(10.0 / 3, 1.0 / 3);
	minor_ratio[6] = 25.0 / 18;
	minor_ratio[7] = pow(10.0 / 3, 1.0 / 3);
	minor_ratio[8] = pow(10.0 / 3, 8.0 / 3) / 16;
	minor_ratio[9] = 5.0 / 3;
	minor_ratio[10] = 4 / pow(10.0 / 3, 2.0 / 3);
	minor_ratio[11] = pow(10.0 / 3, 5.0 / 3) / 4;
	for (i = 0; i < 12; i++)
		for (j = -1; j < 11; j++) {
			f = 440 * pow(2.0, (i - 9) / 12.0 + j - 5);
			for (k = 0; k < 12; k++) {
				l = i + j * 12 + k;
				if (l < 0 || l >= 128)
					continue;
				freq_table_meantone[i][l] =
						f * major_ratio[k] * 1000 + 0.5;
				freq_table_meantone[i + 12][l] =
						f * minor_ratio[k] * sc * 1000 + 0.5;
				freq_table_meantone[i + 24][l] =
						f * minor_ratio[k] * 1000 + 0.5;
				freq_table_meantone[i + 36][l] =
						f * major_ratio[k] * sc * 1000 + 0.5;
			}
		}
}

static void init_freq_table_pureint(void)
{
	int i, j, k, l;
	double f;
	static const double major_ratio[] = {
		 1.0 /  1, 16.0 / 15, 9.0 / 8, 6.0 / 5, 5.0 / 4,  4.0 / 3,
		45.0 / 32,  3.0 /  2, 8.0 / 5, 5.0 / 3, 9.0 / 5, 15.0 / 8
	};
	static const double minor_ratio[] = {
		 1.0 /  1, 25.0 / 24, 10.0 /  9, 75.0 / 64,  5.0 / 4,  4.0 / 3,
		25.0 / 18,  3.0 /  2, 25.0 / 16,  5.0 /  3, 16.0 / 9, 15.0 / 8
	};
	static const double sc = 81.0 / 80;
	
	for (i = 0; i < 12; i++)
		for (j = -1; j < 11; j++) {
			f = 440 * pow(2.0, (i - 9) / 12.0 + j - 5);
			for (k = 0; k < 12; k++) {
				l = i + j * 12 + k;
				if (l < 0 || l >= 128)
					continue;
				freq_table_pureint[i][l] =
						f * major_ratio[k] * 1000 + 0.5;
				freq_table_pureint[i + 12][l] =
						f * minor_ratio[k] * sc * 1000 + 0.5;
				freq_table_pureint[i + 24][l] =
						f * minor_ratio[k] * 1000 + 0.5;
				freq_table_pureint[i + 36][l] =
						f * major_ratio[k] * sc * 1000 + 0.5;
			}
		}
}


/* v=2.^((x/127-1) * 6) */
double def_vol_table[1024];

static void init_def_vol_table(void)
{
	int i;
	
	for (i = 0; i < 1024; i++)
		def_vol_table[i] = pow(2.0f,((double)i / 1023.0 - 1) * 6);
}

/* v=2.^((x/127-1) * 8) */
double gs_vol_table[1024];

static void init_gs_vol_table(void)
{
	int i;
	
	for (i = 0; i < 1024; i++)
		gs_vol_table[i] = pow(2.0f,((double)i / 1023.0 - 1) * 8);
}

double *xg_vol_table = gs_vol_table;

double bend_fine[256];
double bend_coarse[128];

static void init_bend_fine(void)
{
	int i;
	
	for (i = 0; i < 256; i++)
		bend_fine[i] = pow(2.0, i / 12.0 / 256);
}

static void init_bend_coarse(void)
{
	int i;
	
	for (i = 0; i < 128; i++)
		bend_coarse[i] = pow(2.0, i / 12.0);
}

/*
 * midi_time_table(x + 16y) = midi_time_table(x) * (2^y)
 * midi_time_table(64) = 1
 * then,
 * midi_time_table(x) := (2^(x/16))/16
 */
const double midi_time_table[128] =
{
    0.06250, 0.06527, 0.06816, 0.07117, 0.07433, 0.07762, 0.08105, 0.08464,
    0.08839, 0.09230, 0.09639, 0.10066, 0.10511, 0.10977, 0.11463, 0.11970,
    0.12500, 0.13053, 0.13631, 0.14235, 0.14865, 0.15523, 0.16210, 0.16928,
    0.17678, 0.18460, 0.19278, 0.20131, 0.21022, 0.21953, 0.22925, 0.23940,
    0.25000, 0.26107, 0.27263, 0.28470, 0.29730, 0.31046, 0.32421, 0.33856,
    0.35355, 0.36921, 0.38555, 0.40262, 0.42045, 0.43906, 0.45850, 0.47880,
    0.50000, 0.52214, 0.54525, 0.56939, 0.59460, 0.62093, 0.64842, 0.67713,
    0.70711, 0.73841, 0.77111, 0.80525, 0.84090, 0.87813, 0.91700, 0.95760,
    1.00000, 1.04427, 1.09051, 1.13879, 1.18921, 1.24186, 1.29684, 1.35426,
    1.41421, 1.47683, 1.54221, 1.61049, 1.68179, 1.75625, 1.83401, 1.91521,
    2.00000, 2.08855, 2.18102, 2.27758, 2.37841, 2.48372, 2.59368, 2.70851,
    2.82843, 2.95365, 3.08442, 3.22098, 3.36359, 3.51250, 3.66802, 3.83041,
    4.00000, 4.17710, 4.36203, 4.55515, 4.75683, 4.96743, 5.18736, 5.41702,
    5.65685, 5.90730, 6.16884, 6.44196, 6.72717, 7.02501, 7.33603, 7.66083,
    8.00000, 8.35419, 8.72406, 9.11031, 9.51366, 9.93486,10.37472,10.83404,
   11.31371,11.81461,12.33769,12.88392,13.45434,14.05002,14.67206,15.32165
};
/*
 * midi_time_table2(x) := 2^(x/16/128)  (for lsb tunning)
 */
const double midi_time_table2[128] =
{
    1.00000, 1.00034, 1.00068, 1.00102, 1.00135, 1.00169, 1.00203, 1.00237,
    1.00271, 1.00305, 1.00339, 1.00373, 1.00407, 1.00441, 1.00475, 1.00509,
    1.00543, 1.00577, 1.00611, 1.00645, 1.00679, 1.00713, 1.00747, 1.00781,
    1.00816, 1.00850, 1.00884, 1.00918, 1.00952, 1.00986, 1.01021, 1.01055,
    1.01089, 1.01123, 1.01157, 1.01192, 1.01226, 1.01260, 1.01294, 1.01329,
    1.01363, 1.01397, 1.01432, 1.01466, 1.01500, 1.01535, 1.01569, 1.01603,
    1.01638, 1.01672, 1.01707, 1.01741, 1.01776, 1.01810, 1.01844, 1.01879,
    1.01913, 1.01948, 1.01982, 1.02017, 1.02051, 1.02086, 1.02121, 1.02155,
    1.02190, 1.02224, 1.02259, 1.02294, 1.02328, 1.02363, 1.02397, 1.02432,
    1.02467, 1.02501, 1.02536, 1.02571, 1.02606, 1.02640, 1.02675, 1.02710,
    1.02745, 1.02779, 1.02814, 1.02849, 1.02884, 1.02919, 1.02953, 1.02988,
    1.03023, 1.03058, 1.03093, 1.03128, 1.03163, 1.03198, 1.03233, 1.03268,
    1.03302, 1.03337, 1.03372, 1.03407, 1.03442, 1.03477, 1.03512, 1.03548,
    1.03583, 1.03618, 1.03653, 1.03688, 1.03723, 1.03758, 1.03793, 1.03828,
    1.03863, 1.03899, 1.03934, 1.03969, 1.04004, 1.04039, 1.04075, 1.04110,
    1.04145, 1.04180, 1.04216, 1.04251, 1.04286, 1.04321, 1.04357, 1.04392
};


static double triangular_table[257];

void init_triangular_table(void)
{
	int i;
	for (i = 0; i < 257; i++) {
		triangular_table[i] = (double)(i/* - (genrand_int32() % 1)*/) / 256.0;
		if(triangular_table[i] < 0) {triangular_table[i] = 0;}
		else if(triangular_table[i] > 1.0) {triangular_table[i] = 1.0;}
	}
	triangular_table[0] = 0.0;
	triangular_table[256] = 1.0;
}

double lookup_triangular(int x)
{
  int xx = x & 0xFF;
  switch ((x>>8) & 0x03)
    {
    default:
    case 0:
      return triangular_table[xx];
    case 1:
      return triangular_table[0x100 - xx];
    case 2:
      return -triangular_table[xx];
    case 3:
      return -triangular_table[0x100 - xx];
    }
}


const uint8_t reverb_macro_presets[] =
{	/* CHARACTER,PRE-LPF,LEVEL,TIME,DELAY FEEDBACK,PREDELAY TIME */
	0,3,64,80,0,0,		/* 00: Room1 */
	1,4,64,56,0,0,		/* 01: Room2 */
	2,0,64,64,0,0,		/* 02: Room3 */
	3,4,64,72,0,0,		/* 03: Hall1 */
	4,0,64,64,0,0,		/* 04: Hall2 */
	5,0,64,88,0,0,		/* 05: Plate */
	6,0,64,32,40,0,		/* 06: Delay */
	7,0,64,64,32,0,		/* 07: Panning Delay */
};

const uint8_t chorus_macro_presets[] =
{	/* PRE-LPF,LEVEL,FEEDBACK,DELAY,RATE,DEPTH,SEND TO REVERB,SEND TO DELAY */
	0,64,0,112,3,5,0,0,		/* 00: Chorus1 */
	0,64,5,80,9,19,0,0,		/* 01: Chorus2 */
	0,64,8,80,3,19,0,0,		/* 02: Chorus3 */
	0,64,16,64,9,16,0,0,	/* 03: Chorus4 */
	0,64,64,127,2,24,0,0,	/* 04: Feedback Chorus */
	0,64,112,127,1,5,0,0,	/* 05: Flanger */
	0,64,0,127,0,127,0,0,	/* 06: Short Delay */
	0,64,80,127,0,127,0,0,	/* 07: Short Delay(Feedback) */
};

const uint8_t delay_macro_presets[] =
{	/* PRE-LPF,TIME(C),RATIO(L),RATIO(R),LEVEL(C),LEVEL(L),LEVEL(R),LEVEL,FEEDBACK,LEVEL TO REVERB */
	0,97,1,1,127,0,0,64,79,0,		/* 00: Delay1 */
	0,106,1,1,127,0,0,64,79,0,		/* 01: Delay2 */
	0,115,1,1,127,0,0,64,63,0,		/* 02: Delay3 */
	0,83,1,1,127,0,0,64,71,0,		/* 03: Delay4 */
	0,90,12,24,0,125,60,64,73,0,	/* 04: Pan Delay1 */
	0,109,12,24,0,125,60,64,70,0,	/* 05: Pan Delay2 */
	0,115,12,24,0,120,64,64,72,0,	/* 06: Pan Delay3 */
	0,93,12,24,0,120,64,64,63,0,	/* 07: Pan Delay4 */
	0,109,12,24,0,114,60,64,60,36,	/* 08: Delay to Reverb */
	0,110,21,31,97,127,67,64,39,0,	/* 09: Pan Repeat */
};

const float delay_time_center_table[] =
{	/* 0x00~0x73, 0.1ms~1000ms */
	0.1f, 0.1f, 0.2f, 0.3f, 0.4f, 0.5f, 0.6f, 0.7f, 0.8f, 0.9f, 1.0f, 1.1f, 1.2f, 1.3f, 1.4f, 1.5f, 1.6f, 1.7f, 1.8f, 1.9f,
	2.0f, 2.2f, 2.4f, 2.6f, 2.8f, 3.0f, 3.2f, 3.4f, 3.6f, 3.8f, 4.0f, 4.2f, 4.4f, 4.6f, 4.8f,
	5.0f, 5.5f, 6.0f, 6.5f, 7.0f, 7.5f, 8.0f, 8.5f, 9.0f, 9.5f,
	10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
	20, 22, 24, 26, 28, 30, 32, 34, 36, 38, 40, 42, 44, 46, 48,
	50, 55, 60, 65, 70, 75, 80, 85, 90, 95,
	100, 110, 120, 130, 140, 150, 160, 170, 180, 190,
	200, 220, 240, 260, 280, 300, 320, 340, 360, 380, 400, 420, 440, 460, 480,
	500, 550, 600, 650, 700, 750, 800, 850, 900, 950, 1000,
};

const float pre_delay_time_table[] =
{
	0.0f, 0.1f, 0.2f, 0.3f, 0.4f, 0.5f, 0.6f, 0.7f, 0.8f, 0.9f, 1.0f, 1.1f, 1.2f, 1.3f, 1.4f, 1.5f, 1.6f, 1.7f, 1.8f, 1.9f,
	2.0f, 2.1f, 2.2f, 2.3f, 2.4f, 2.5f, 2.6f, 2.7f, 2.8f, 2.9f, 3.0f, 3.1f, 3.2f, 3.3f, 3.4f, 3.5f, 3.6f, 3.7f, 3.8f, 3.9f,
	4.0f, 4.1f, 4.2f, 4.3f, 4.4f, 4.5f, 4.6f, 4.7f, 4.8f, 4.9f,
	5.0f, 5.5f, 6.0f, 6.5f, 7.0f, 7.5f, 8.0f, 8.5f, 9.0f, 9.5f,
	10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29,
	30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49,
	50, 52, 54, 56, 58, 60, 62, 64, 66, 68, 70, 72, 74, 76, 78, 80, 82, 84, 86, 88,
	90, 92, 94, 96, 98, 100, 100, 100,
};

const float chorus_delay_time_table[] =
{
    0.000000f, 0.078740f, 0.157480f, 0.236220f, 0.314961f, 0.393701f, 0.472441f, 0.551181f, 
    0.629921f, 0.708661f, 0.787402f, 0.866142f, 0.944882f, 1.023622f, 1.102362f, 1.181102f, 
    1.259843f, 1.338583f, 1.417323f, 1.496063f, 1.574803f, 1.653543f, 1.732283f, 1.811024f, 
    1.889764f, 1.968504f, 2.047244f, 2.125984f, 2.204724f, 2.283465f, 2.362205f, 2.440945f, 
    2.519685f, 2.598425f, 2.677165f, 2.755906f, 2.834646f, 2.913386f, 2.992126f, 3.070866f, 
    3.149606f, 3.228346f, 3.307087f, 3.385827f, 3.464567f, 3.543307f, 3.622047f, 3.700787f, 
    3.779528f, 3.858268f, 3.937008f, 4.015748f, 4.094488f, 4.173228f, 4.251969f, 4.330709f, 
    4.409449f, 4.488189f, 4.566929f, 4.645669f, 4.724409f, 4.803150f, 4.881890f, 4.960630f, 
    5.039370f, 5.118110f, 5.196850f, 5.275591f, 5.354331f, 5.433071f, 5.511811f, 5.590551f, 
    5.669291f, 5.748031f, 5.826772f, 5.905512f, 5.984252f, 6.062992f, 6.141732f, 6.220472f, 
    6.299213f, 6.377953f, 6.456693f, 6.535433f, 6.614173f, 6.692913f, 6.771654f, 6.850394f, 
    6.929134f, 7.007874f, 7.086614f, 7.165354f, 7.244094f, 7.322835f, 7.401575f, 7.480315f, 
    7.559055f, 7.637795f, 7.716535f, 7.795276f, 10.000000f, 10.555556f, 11.111111f, 11.666667f, 
    12.222222f, 12.777778f, 13.333333f, 13.888889f, 14.444444f, 15.000000f, 15.555556f, 16.111111f, 
    16.666667f, 17.222222f, 17.777778f, 18.333333f, 18.888889f, 19.444444f, 20.000000f, 20.555556f, 
    21.111111f, 21.666667f, 22.222222f, 22.777778f, 23.333333f, 23.888889f, 24.444444f, 25.000000f, 
};

const float rate1_table[] =
{
	0.05f, 0.10f, 0.15f, 0.20f, 0.25f, 0.30f, 0.35f, 0.40f, 0.45f, 0.50f, 0.55f, 0.60f, 0.65f, 0.70f, 0.75f, 0.80f,
	0.85f, 0.90f, 0.95f, 1.00f, 1.05f, 1.10f, 1.15f, 1.20f, 1.25f, 1.30f, 1.35f, 1.40f, 1.45f, 1.50f, 1.55f, 1.60f,
	1.65f, 1.70f, 1.75f, 1.80f, 1.85f, 1.90f, 1.95f, 2.00f, 2.05f, 2.10f, 2.15f, 2.20f, 2.25f, 2.30f, 2.35f, 2.40f,
	2.45f, 2.50f, 2.55f, 2.60f, 2.65f, 2.70f, 2.75f, 2.80f, 2.85f, 2.90f, 2.95f, 3.00f, 3.05f, 3.10f, 3.15f, 3.20f,
	3.25f, 3.30f, 3.35f, 3.40f, 3.45f, 3.50f, 3.55f, 3.60f, 3.65f, 3.70f, 3.75f, 3.80f, 3.85f, 3.90f, 3.95f, 4.00f,
	4.05f, 4.10f, 4.15f, 4.20f, 4.25f, 4.30f, 4.35f, 4.40f, 4.45f, 4.50f, 4.55f, 4.60f, 4.65f, 4.70f, 4.75f, 4.80f,
	4.85f, 4.90f, 4.95f, 5.00f, 5.10f, 5.20f, 5.30f, 5.40f, 5.50f, 5.60f, 5.70f, 5.80f, 5.90f, 6.00f, 6.10f, 6.20f,
	6.30f, 6.40f, 6.50f, 6.60f, 6.70f, 6.80f, 6.90f, 7.00f, 7.50f, 8.00f, 8.50f, 9.00f, 9.50f, 10.00f, 10.00f, 10.00f,
};

/* Derivation of Perceived Volume Curve Equation:
 *
 *    Given: delta dB = 20 * log10(amplitude_new / amplitude_old)
 *           delta dB of 10 == perceived volume change of 2x
 *
 *    10 = 20 * log10(?)
 *    0.5 = log10(?)
 *    10^0.5 = ?
 *
 *    therefore: 2x perceived volume == ~3.16x amplitude
 *               4x perceived volume ==    10x amplitude
 *
 *       Volume         Amplitude
 *       ------------   ---------------
 *       1              1
 *       0.25           0.1
 *       0.0625         0.01
 *       0.015625       0.001
 *       0.00390625     0.0001
 *       0.0009765625   0.00001
 *       0              0
 *
 *    Fit curve to table:
 *
 *       amplification = pow(volume, 1.66096404744)
 */
double perceived_vol_table[128];

void init_perceived_vol_table(void)
{
	int i;
	
	for (i = 0; i < 128; i++)
		perceived_vol_table[i] =
				127.0 * pow((double)i / 127.0, 1.66096404744);
}

double gm2_vol_table[128];

void init_gm2_vol_table(void)
{
	int i;
	
	for(i = 0; i < 128; i++)
		gm2_vol_table[i] = (i * i) / 127.0;
}

/* measured value from SC-88STPro.
   approximate expression: y = (-0.3768x6 + 0.9528x5 - 0.8253x4 + 0.2665x3 + 0.9892x2 - 0.0059x + 0.001) * 127 */
double sc_vol_table[128] = 
{
    0.000000, 0.128905, 0.146482, 0.179815, 0.228982, 0.294049, 0.375078, 0.472120, 
    0.585221, 0.714419, 0.859746, 1.021229, 1.198887, 1.392736, 1.602785, 1.829039, 
    2.071501, 2.330166, 2.605028, 2.896078, 3.203301, 3.526682, 3.866202, 4.221841, 
    4.593575, 4.981382, 5.385233, 5.805103, 6.240963, 6.692783, 7.160536, 7.644189, 
    8.143714, 8.659080, 9.190256, 9.737215, 10.299925, 10.878360, 11.472491, 12.082292, 
    12.707738, 13.348803, 14.005465, 14.677704, 15.365497, 16.068829, 16.787681, 17.522039, 
    18.271890, 19.037223, 19.818029, 20.614301, 21.426034, 22.253225, 23.095873, 23.953980, 
    24.827548, 25.716584, 26.621094, 27.541088, 28.476578, 29.427576, 30.394100, 31.376165, 
    32.373791, 33.386998, 34.415810, 35.460249, 36.520342, 37.596115, 38.687597, 39.794815, 
    40.917801, 42.056586, 43.211200, 44.381677, 45.568048, 46.770346, 47.988605, 49.222856, 
    50.473131, 51.739464, 53.021883, 54.320420, 55.635102, 56.965957, 58.313010, 59.676284, 
    61.055802, 62.451580, 63.863636, 65.291982, 66.736627, 68.197578, 69.674835, 71.168397, 
    72.678255, 74.204399, 75.746811, 77.305466, 78.880337, 80.471388, 82.078576, 83.701851, 
    85.341158, 86.996429, 88.667594, 90.354568, 92.057260, 93.775571, 95.509387, 97.258589, 
    99.023042, 100.802603, 102.597116, 104.406412, 106.230309, 108.068613, 109.921115, 111.787592, 
    113.667805, 115.561500, 117.468408, 119.388243, 121.320699, 123.265458, 125.222177, 127.000000, 
};

/* measured value from SC-88STPro.
   approximate expression: y = (-1.5374x6 + 4.4002x5 - 4.8309x4 + 2.572x3 + 0.1487x2 + 0.2412x + 0.0044) * 127 */
double sc_vel_table[128] = 
{
    0.000000, 0.801328, 1.047122, 1.297056, 1.551953, 1.812583, 2.079668, 2.353885, 
    2.635863, 2.926190, 3.225412, 3.534034, 3.852525, 4.181317, 4.520806, 4.871357, 
    5.233303, 5.606946, 5.992560, 6.390392, 6.800666, 7.223577, 7.659301, 8.107993, 
    8.569785, 9.044792, 9.533111, 10.034824, 10.549997, 11.078680, 11.620912, 12.176722, 
    12.746124, 13.329126, 13.925725, 14.535911, 15.159666, 15.796968, 16.447787, 17.112090, 
    17.789842, 18.481003, 19.185529, 19.903380, 20.634509, 21.378873, 22.136426, 22.907127, 
    23.690932, 24.487801, 25.297696, 26.120582, 26.956425, 27.805197, 28.666872, 29.541428, 
    30.428848, 31.329119, 32.242232, 33.168184, 34.106974, 35.058609, 36.023099, 37.000459, 
    37.990710, 38.993877, 40.009989, 41.039080, 42.081190, 43.136361, 44.204639, 45.286075, 
    46.380723, 47.488640, 48.609885, 49.744520, 50.892609, 52.054217, 53.229410, 54.418254, 
    55.620816, 56.837160, 58.067351, 59.311450, 60.569517, 61.841607, 63.127770, 64.428053, 
    65.742497, 67.071133, 68.413988, 69.771079, 71.142412, 72.527985, 73.927782, 75.341776, 
    76.769925, 78.212172, 79.668444, 81.138652, 82.622687, 84.120420, 85.631702, 87.156361, 
    88.694200, 90.244998, 91.808507, 93.384452, 94.972526, 96.572391, 98.183679, 99.805985, 
    101.438869, 103.081852, 104.734417, 106.396007, 108.066018, 109.743805, 111.428675, 113.119886, 
    114.816648, 116.518116, 118.223392, 119.931522, 121.641492, 123.352230, 125.062599, 127.000000, 
};

double sc_pan_table[129] = 
{
	0.000000, 0.000000, 0.999479, 2.011744, 3.036530, 4.073569, 5.122593, 6.183332,
	7.255517, 8.338874, 9.433131, 10.538014, 11.653247, 12.778552, 13.913653, 15.058271,
	16.212123, 17.374930, 18.546409, 19.726275, 20.914243, 22.110027, 23.313339, 24.523890,
	25.741391, 26.965550, 28.196074, 29.432671, 30.675045, 31.922900, 33.175939, 34.433863,
	35.696373, 36.963168, 38.233946, 39.508403, 40.786235, 42.067137, 43.350800, 44.636918,
	45.925181, 47.215278, 48.506897, 49.799726, 51.093451, 52.387755, 53.682323, 54.976837,
	56.270977, 57.564424, 58.856855, 60.147950, 61.437382, 62.724829, 64.009963, 65.292456,
	66.571981, 67.848208, 69.120804, 70.389439, 71.653778, 72.913487, 74.168230, 75.417670,
	76.661468, 77.899286, 79.130781, 80.355614, 81.573439, 82.783913, 83.986691, 85.181425,
	86.367767, 87.545369, 88.713880, 89.872949, 91.022222, 92.161346, 93.289965, 94.407723,
	95.514263, 96.609225, 97.692249, 98.762975, 99.821039, 100.866079, 101.897729, 102.915623,
	103.919394, 104.908673, 105.883091, 106.842276, 107.785858, 108.713461, 109.624713, 110.519236,
	111.396655, 112.256590, 113.098663, 113.922493, 114.727699, 115.513896, 116.280702, 117.027730,
	117.754595, 118.460908, 119.146280, 119.810321, 120.452639, 121.072843, 121.670538, 122.245328,
	122.796819, 123.324612, 123.828308, 124.307509, 124.761812, 125.190815, 125.594115, 125.971307,
	126.321986, 126.645744, 126.942172, 127.210862, 127.451402, 127.663381, 127.846385, 128.000000,
	128.000000,
};

double gm2_pan_table[129];
double *pan_table = sc_pan_table;

void init_gm2_pan_table(void)
{
	int i;

	gm2_pan_table[0] = 0;
	for(i = 0; i < 127; i++)
		gm2_pan_table[i + 1] = sin(M_PI / 2 * i / 126) * 128;
								/* lookup_sine(i * SINE_CYCLE_LENGTH / 4 / 126) */
	gm2_pan_table[128] = 128.0;
}

double sc_drum_level_table[128] = 
{
    0.007874, 0.007874, 0.031496, 0.070866, 0.125984, 0.196850, 0.283465, 0.385827, 
    0.503937, 0.637795, 0.787402, 0.952756, 1.133858, 1.330709, 1.543307, 1.771654, 
    2.015748, 2.275591, 2.551181, 2.842520, 3.149606, 3.472441, 3.811024, 4.165354, 
    4.535433, 4.921260, 5.322835, 5.740157, 6.173228, 6.622047, 7.086614, 7.566929, 
    8.062992, 8.574803, 9.102362, 9.645669, 10.204724, 10.779528, 11.370079, 11.976378, 
    12.598425, 13.236220, 13.889764, 14.559055, 15.244094, 15.944882, 16.661417, 17.393701, 
    18.141732, 18.905512, 19.685039, 20.480315, 21.291339, 22.118110, 22.960630, 23.818898, 
    24.692913, 25.582677, 26.488189, 27.409449, 28.346457, 29.299213, 30.267717, 31.251969, 
    32.251969, 33.267717, 34.299213, 35.346457, 36.409449, 37.488189, 38.582677, 39.692913, 
    40.818898, 41.960630, 43.118110, 44.291339, 45.480315, 46.685039, 47.905512, 49.141732, 
    50.393701, 51.661417, 52.944882, 54.244094, 55.559055, 56.889764, 58.236220, 59.598425, 
    60.976378, 62.370079, 63.779528, 65.204724, 66.645669, 68.102362, 69.574803, 71.062992, 
    72.566929, 74.086614, 75.622047, 77.173228, 78.740157, 80.322835, 81.921260, 83.535433, 
    85.165354, 86.811024, 88.472441, 90.149606, 91.842520, 93.551181, 95.275591, 97.015748, 
    98.771654, 100.543307, 102.330709, 104.133858, 105.952756, 107.787402, 109.637795, 111.503937, 
    113.385827, 115.283465, 117.196850, 119.125984, 121.070866, 123.031496, 125.007874, 127.000000, 
};

double attack_vol_table[1024];

void init_attack_vol_table(void)
{
	int i;
	
	for (i = 0; i < 1024; i++)
		attack_vol_table[i] = i / 1023.0;
}

float sc_eg_decay_table[128] = 
{
    81.841218f, 81.841218f, 81.841218f, 81.841218f, 81.841218f, 81.841218f, 81.841218f, 81.841218f, 
    81.841218f, 81.841218f, 81.841218f, 81.841218f, 81.841218f, 81.841218f, 81.841218f, 81.841218f, 
    81.841218f, 81.841218f, 81.841218f, 81.841218f, 81.841218f, 81.841218f, 74.928977f, 68.580327f, 
    62.750584f, 57.398521f, 52.486110f, 47.978275f, 43.842671f, 40.049475f, 36.571192f, 33.382477f, 
    30.459975f, 27.782161f, 25.329207f, 23.082845f, 21.026252f, 19.143935f, 17.421629f, 15.846202f, 
    14.405568f, 13.088604f, 11.885077f, 10.785570f, 9.781425f, 8.864676f, 8.028000f, 7.264663f, 
    6.568475f, 5.933745f, 5.355241f, 4.828153f, 4.348058f, 3.910885f, 3.512890f, 3.150618f, 
    2.820877f, 2.520709f, 2.247348f, 1.998183f, 1.770681f, 1.562261f, 1.369978f, 1.189386f, 
    1.000000f, 0.838459f, 0.726301f, 0.635581f, 0.559656f, 0.494986f, 0.439286f, 0.390934f, 
    0.348712f, 0.311669f, 0.279045f, 0.250221f, 0.224684f, 0.202006f, 0.181825f, 0.163831f, 
    0.147761f, 0.133387f, 0.120513f, 0.108967f, 0.098600f, 0.089282f, 0.080897f, 0.073346f, 
    0.066540f, 0.060399f, 0.054854f, 0.049845f, 0.045315f, 0.041217f, 0.037506f, 0.034144f, 
    0.031097f, 0.028333f, 0.025826f, 0.023549f, 0.021480f, 0.019601f, 0.017892f, 0.016337f, 
    0.014923f, 0.013635f, 0.012462f, 0.011394f, 0.011394f, 0.011394f, 0.011394f, 0.011394f, 
    0.011394f, 0.011394f, 0.011394f, 0.011394f, 0.011394f, 0.011394f, 0.011394f, 0.011394f, 
    0.011394f, 0.011394f, 0.011394f, 0.011394f, 0.011394f, 0.011394f, 0.011394f, 0.011394f, 
};

float sc_eg_release_table[128] = 
{
    27.322002f, 27.322002f, 27.322002f, 27.322002f, 27.322002f, 27.322002f, 27.322002f, 27.322002f, 
    27.322002f, 27.322002f, 27.322002f, 27.322002f, 27.322002f, 27.322002f, 27.322002f, 27.322002f, 
    27.322002f, 27.322002f, 27.322002f, 27.322002f, 27.322002f, 27.322002f, 25.299110f, 23.425992f, 
    21.691556f, 20.085537f, 18.598425f, 17.221418f, 15.946363f, 14.765711f, 13.672474f, 12.660179f, 
    11.722833f, 10.854887f, 10.051203f, 9.307023f, 8.617941f, 7.979878f, 7.389056f, 6.841978f, 
    6.335406f, 5.866339f, 5.432002f, 5.029822f, 4.657419f, 4.312589f, 3.993290f, 3.697631f, 
    3.423862f, 3.170363f, 2.935633f, 2.718282f, 2.517023f, 2.330665f, 2.158106f, 1.998322f, 
    1.850368f, 1.713369f, 1.586513f, 1.469049f, 1.360282f, 1.259569f, 1.166311f, 1.079959f, 
    1.000000f, 0.925961f, 0.857404f, 0.793923f, 0.735141f, 0.680712f, 0.630313f, 0.583645f, 
    0.540433f, 0.500420f, 0.463369f, 0.429062f, 0.397295f, 0.367879f, 0.340642f, 0.315421f, 
    0.292068f, 0.270443f, 0.250420f, 0.231879f, 0.214711f, 0.198814f, 0.184094f, 0.170464f, 
    0.157843f, 0.146157f, 0.135335f, 0.125315f, 0.116037f, 0.107446f, 0.099491f, 0.092124f, 
    0.085304f, 0.078988f, 0.073140f, 0.067724f, 0.062710f, 0.058067f, 0.053768f, 0.049787f, 
    0.046101f, 0.042688f, 0.039527f, 0.036601f, 0.036601f, 0.036601f, 0.036601f, 0.036601f, 
    0.036601f, 0.036601f, 0.036601f, 0.036601f, 0.036601f, 0.036601f, 0.036601f, 0.036601f, 
    0.036601f, 0.036601f, 0.036601f, 0.036601f, 0.036601f, 0.036601f, 0.036601f, 0.036601f, 
};

float sc_eg_attack_table[128] = 
{
    82.756924f, 82.756924f, 82.756924f, 82.756924f, 82.756924f, 82.756924f, 82.756924f, 82.756924f, 
    82.756924f, 82.756924f, 82.756924f, 82.756924f, 82.756924f, 82.756924f, 82.756924f, 82.756924f, 
    82.756924f, 82.756924f, 82.756924f, 82.756924f, 82.756924f, 82.756924f, 75.473398f, 68.815182f, 
    62.729632f, 57.168464f, 52.087395f, 47.445819f, 43.206507f, 39.335325f, 35.800987f, 32.574817f, 
    29.630534f, 26.944060f, 24.493331f, 22.258137f, 20.219967f, 18.361866f, 16.668311f, 15.125088f, 
    13.719184f, 12.438688f, 11.272700f, 10.211246f, 9.245197f, 8.366205f, 7.566631f, 6.839489f, 
    6.178391f, 5.577493f, 5.031451f, 4.535378f, 4.084805f, 3.675641f, 3.304143f, 2.966879f, 
    2.660703f, 2.382715f, 2.130237f, 1.900768f, 1.691929f, 1.501374f, 1.326560f, 1.163993f, 
    1.000000f, 0.859112f, 0.753830f, 0.666057f, 0.591041f, 0.526103f, 0.469431f, 0.419689f, 
    0.375841f, 0.337054f, 0.302650f, 0.272061f, 0.244810f, 0.220489f, 0.198750f, 0.179292f, 
    0.161854f, 0.146210f, 0.132159f, 0.119529f, 0.108164f, 0.097931f, 0.088710f, 0.080394f, 
    0.072891f, 0.066115f, 0.059994f, 0.054461f, 0.049456f, 0.044927f, 0.040827f, 0.037114f, 
    0.033749f, 0.030699f, 0.027932f, 0.025422f, 0.023145f, 0.021077f, 0.019199f, 0.017492f, 
    0.015941f, 0.014532f, 0.013250f, 0.012084f, 0.012084f, 0.012084f, 0.012084f, 0.012084f, 
    0.012084f, 0.012084f, 0.012084f, 0.012084f, 0.012084f, 0.012084f, 0.012084f, 0.012084f, 
    0.012084f, 0.012084f, 0.012084f, 0.012084f, 0.012084f, 0.012084f, 0.012084f, 0.012084f, 
};

double sb_vol_table[1024];

void init_sb_vol_table(void)
{
	int i;
	
	for (i = 0; i < 1024; i++)
		sb_vol_table[i] = pow(10.0, (double)(1023 - i) * 960.0 / (1023.0 * -200.0));
}

double modenv_vol_table[1024];

void init_modenv_vol_table(void)
{
	int i;
	double x;

	modenv_vol_table[0] = (float)0;
	for (i = 1; i < 1023; i++) {
		x = (1.0 - (-20.0 / 96.0 * log(((double)i * (double)i) / (1023.0 * 1023.0)) / log(10.0)));
		if (x < 0) {x = 0;}
		modenv_vol_table[i] = log(x + 1) / log(2);
	}
	modenv_vol_table[1023] = (float)1.0;
}

const float cb_to_amp_table[961] =
{
    1.000000f, 0.995677f, 0.991373f, 0.987088f, 0.982821f, 0.978572f, 0.974342f, 0.970130f, 
    0.965936f, 0.961761f, 0.957603f, 0.953464f, 0.949342f, 0.945238f, 0.941152f, 0.937084f, 
    0.933033f, 0.929000f, 0.924984f, 0.920985f, 0.917004f, 0.913040f, 0.909093f, 0.905163f, 
    0.901250f, 0.897355f, 0.893475f, 0.889613f, 0.885768f, 0.881939f, 0.878126f, 0.874330f, 
    0.870551f, 0.866787f, 0.863040f, 0.859310f, 0.855595f, 0.851896f, 0.848214f, 0.844547f, 
    0.840896f, 0.837261f, 0.833642f, 0.830038f, 0.826450f, 0.822878f, 0.819321f, 0.815779f, 
    0.812252f, 0.808741f, 0.805245f, 0.801764f, 0.798298f, 0.794848f, 0.791412f, 0.787990f, 
    0.784584f, 0.781192f, 0.777816f, 0.774453f, 0.771105f, 0.767772f, 0.764453f, 0.761149f, 
    0.757858f, 0.754582f, 0.751320f, 0.748072f, 0.744839f, 0.741619f, 0.738413f, 0.735221f, 
    0.732043f, 0.728878f, 0.725728f, 0.722590f, 0.719467f, 0.716357f, 0.713260f, 0.710177f, 
    0.707107f, 0.704050f, 0.701007f, 0.697976f, 0.694959f, 0.691955f, 0.688964f, 0.685986f, 
    0.683020f, 0.680068f, 0.677128f, 0.674201f, 0.671286f, 0.668384f, 0.665495f, 0.662618f, 
    0.659754f, 0.656902f, 0.654062f, 0.651235f, 0.648420f, 0.645617f, 0.642826f, 0.640047f, 
    0.637280f, 0.634525f, 0.631783f, 0.629051f, 0.626332f, 0.623625f, 0.620929f, 0.618245f, 
    0.615572f, 0.612911f, 0.610262f, 0.607624f, 0.604997f, 0.602382f, 0.599778f, 0.597185f, 
    0.594604f, 0.592033f, 0.589474f, 0.586926f, 0.584389f, 0.581862f, 0.579347f, 0.576843f, 
    0.574349f, 0.571866f, 0.569394f, 0.566933f, 0.564482f, 0.562042f, 0.559612f, 0.557193f, 
    0.554785f, 0.552387f, 0.549999f, 0.547621f, 0.545254f, 0.542897f, 0.540550f, 0.538213f, 
    0.535887f, 0.533570f, 0.531264f, 0.528967f, 0.526681f, 0.524404f, 0.522137f, 0.519880f, 
    0.517632f, 0.515395f, 0.513167f, 0.510949f, 0.508740f, 0.506541f, 0.504351f, 0.502171f, 
    0.500000f, 0.497839f, 0.495687f, 0.493544f, 0.491410f, 0.489286f, 0.487171f, 0.485065f, 
    0.482968f, 0.480880f, 0.478802f, 0.476732f, 0.474671f, 0.472619f, 0.470576f, 0.468542f, 
    0.466516f, 0.464500f, 0.462492f, 0.460493f, 0.458502f, 0.456520f, 0.454547f, 0.452582f, 
    0.450625f, 0.448677f, 0.446738f, 0.444807f, 0.442884f, 0.440969f, 0.439063f, 0.437165f, 
    0.435275f, 0.433394f, 0.431520f, 0.429655f, 0.427798f, 0.425948f, 0.424107f, 0.422274f, 
    0.420448f, 0.418631f, 0.416821f, 0.415019f, 0.413225f, 0.411439f, 0.409660f, 0.407889f, 
    0.406126f, 0.404371f, 0.402623f, 0.400882f, 0.399149f, 0.397424f, 0.395706f, 0.393995f, 
    0.392292f, 0.390596f, 0.388908f, 0.387227f, 0.385553f, 0.383886f, 0.382227f, 0.380574f, 
    0.378929f, 0.377291f, 0.375660f, 0.374036f, 0.372419f, 0.370809f, 0.369207f, 0.367611f, 
    0.366021f, 0.364439f, 0.362864f, 0.361295f, 0.359733f, 0.358178f, 0.356630f, 0.355088f, 
    0.353553f, 0.352025f, 0.350503f, 0.348988f, 0.347480f, 0.345977f, 0.344482f, 0.342993f, 
    0.341510f, 0.340034f, 0.338564f, 0.337100f, 0.335643f, 0.334192f, 0.332748f, 0.331309f, 
    0.329877f, 0.328451f, 0.327031f, 0.325617f, 0.324210f, 0.322808f, 0.321413f, 0.320024f, 
    0.318640f, 0.317263f, 0.315891f, 0.314526f, 0.313166f, 0.311812f, 0.310464f, 0.309122f, 
    0.307786f, 0.306456f, 0.305131f, 0.303812f, 0.302499f, 0.301191f, 0.299889f, 0.298593f, 
    0.297302f, 0.296017f, 0.294737f, 0.293463f, 0.292194f, 0.290931f, 0.289674f, 0.288421f, 
    0.287175f, 0.285933f, 0.284697f, 0.283466f, 0.282241f, 0.281021f, 0.279806f, 0.278597f, 
    0.277392f, 0.276193f, 0.274999f, 0.273811f, 0.272627f, 0.271448f, 0.270275f, 0.269107f, 
    0.267943f, 0.266785f, 0.265632f, 0.264484f, 0.263340f, 0.262202f, 0.261068f, 0.259940f, 
    0.258816f, 0.257697f, 0.256583f, 0.255474f, 0.254370f, 0.253270f, 0.252175f, 0.251085f, 
    0.250000f, 0.248919f, 0.247843f, 0.246772f, 0.245705f, 0.244643f, 0.243585f, 0.242533f, 
    0.241484f, 0.240440f, 0.239401f, 0.238366f, 0.237336f, 0.236310f, 0.235288f, 0.234271f, 
    0.233258f, 0.232250f, 0.231246f, 0.230246f, 0.229251f, 0.228260f, 0.227273f, 0.226291f, 
    0.225313f, 0.224339f, 0.223369f, 0.222403f, 0.221442f, 0.220485f, 0.219532f, 0.218583f, 
    0.217638f, 0.216697f, 0.215760f, 0.214827f, 0.213899f, 0.212974f, 0.212053f, 0.211137f, 
    0.210224f, 0.209315f, 0.208411f, 0.207510f, 0.206613f, 0.205719f, 0.204830f, 0.203945f, 
    0.203063f, 0.202185f, 0.201311f, 0.200441f, 0.199575f, 0.198712f, 0.197853f, 0.196998f, 
    0.196146f, 0.195298f, 0.194454f, 0.193613f, 0.192776f, 0.191943f, 0.191113f, 0.190287f, 
    0.189465f, 0.188646f, 0.187830f, 0.187018f, 0.186210f, 0.185405f, 0.184603f, 0.183805f, 
    0.183011f, 0.182220f, 0.181432f, 0.180648f, 0.179867f, 0.179089f, 0.178315f, 0.177544f, 
    0.176777f, 0.176013f, 0.175252f, 0.174494f, 0.173740f, 0.172989f, 0.172241f, 0.171496f, 
    0.170755f, 0.170017f, 0.169282f, 0.168550f, 0.167822f, 0.167096f, 0.166374f, 0.165655f, 
    0.164938f, 0.164225f, 0.163516f, 0.162809f, 0.162105f, 0.161404f, 0.160706f, 0.160012f, 
    0.159320f, 0.158631f, 0.157946f, 0.157263f, 0.156583f, 0.155906f, 0.155232f, 0.154561f, 
    0.153893f, 0.153228f, 0.152565f, 0.151906f, 0.151249f, 0.150595f, 0.149944f, 0.149296f, 
    0.148651f, 0.148008f, 0.147368f, 0.146731f, 0.146097f, 0.145466f, 0.144837f, 0.144211f, 
    0.143587f, 0.142967f, 0.142349f, 0.141733f, 0.141121f, 0.140511f, 0.139903f, 0.139298f, 
    0.138696f, 0.138097f, 0.137500f, 0.136905f, 0.136313f, 0.135724f, 0.135138f, 0.134553f, 
    0.133972f, 0.133393f, 0.132816f, 0.132242f, 0.131670f, 0.131101f, 0.130534f, 0.129970f, 
    0.129408f, 0.128849f, 0.128292f, 0.127737f, 0.127185f, 0.126635f, 0.126088f, 0.125543f, 
    0.125000f, 0.124460f, 0.123922f, 0.123386f, 0.122853f, 0.122322f, 0.121793f, 0.121266f, 
    0.120742f, 0.120220f, 0.119700f, 0.119183f, 0.118668f, 0.118155f, 0.117644f, 0.117135f, 
    0.116629f, 0.116125f, 0.115623f, 0.115123f, 0.114626f, 0.114130f, 0.113637f, 0.113145f, 
    0.112656f, 0.112169f, 0.111684f, 0.111202f, 0.110721f, 0.110242f, 0.109766f, 0.109291f, 
    0.108819f, 0.108348f, 0.107880f, 0.107414f, 0.106949f, 0.106487f, 0.106027f, 0.105568f, 
    0.105112f, 0.104658f, 0.104205f, 0.103755f, 0.103306f, 0.102860f, 0.102415f, 0.101972f, 
    0.101532f, 0.101093f, 0.100656f, 0.100221f, 0.099787f, 0.099356f, 0.098926f, 0.098499f, 
    0.098073f, 0.097649f, 0.097227f, 0.096807f, 0.096388f, 0.095972f, 0.095557f, 0.095144f, 
    0.094732f, 0.094323f, 0.093915f, 0.093509f, 0.093105f, 0.092702f, 0.092302f, 0.091903f, 
    0.091505f, 0.091110f, 0.090716f, 0.090324f, 0.089933f, 0.089545f, 0.089158f, 0.088772f, 
    0.088388f, 0.088006f, 0.087626f, 0.087247f, 0.086870f, 0.086494f, 0.086120f, 0.085748f, 
    0.085378f, 0.085008f, 0.084641f, 0.084275f, 0.083911f, 0.083548f, 0.083187f, 0.082827f, 
    0.082469f, 0.082113f, 0.081758f, 0.081404f, 0.081052f, 0.080702f, 0.080353f, 0.080006f, 
    0.079660f, 0.079316f, 0.078973f, 0.078631f, 0.078292f, 0.077953f, 0.077616f, 0.077281f, 
    0.076947f, 0.076614f, 0.076283f, 0.075953f, 0.075625f, 0.075298f, 0.074972f, 0.074648f, 
    0.074325f, 0.074004f, 0.073684f, 0.073366f, 0.073049f, 0.072733f, 0.072418f, 0.072105f, 
    0.071794f, 0.071483f, 0.071174f, 0.070867f, 0.070560f, 0.070255f, 0.069952f, 0.069649f, 
    0.069348f, 0.069048f, 0.068750f, 0.068453f, 0.068157f, 0.067862f, 0.067569f, 0.067277f, 
    0.066986f, 0.066696f, 0.066408f, 0.066121f, 0.065835f, 0.065550f, 0.065267f, 0.064985f, 
    0.064704f, 0.064424f, 0.064146f, 0.063869f, 0.063592f, 0.063318f, 0.063044f, 0.062771f, 
    0.062500f, 0.062230f, 0.061961f, 0.061693f, 0.061426f, 0.061161f, 0.060896f, 0.060633f, 
    0.060371f, 0.060110f, 0.059850f, 0.059591f, 0.059334f, 0.059077f, 0.058822f, 0.058568f, 
    0.058315f, 0.058062f, 0.057811f, 0.057562f, 0.057313f, 0.057065f, 0.056818f, 0.056573f, 
    0.056328f, 0.056085f, 0.055842f, 0.055601f, 0.055360f, 0.055121f, 0.054883f, 0.054646f, 
    0.054409f, 0.054174f, 0.053940f, 0.053707f, 0.053475f, 0.053244f, 0.053013f, 0.052784f, 
    0.052556f, 0.052329f, 0.052103f, 0.051877f, 0.051653f, 0.051430f, 0.051208f, 0.050986f, 
    0.050766f, 0.050546f, 0.050328f, 0.050110f, 0.049894f, 0.049678f, 0.049463f, 0.049249f, 
    0.049037f, 0.048825f, 0.048613f, 0.048403f, 0.048194f, 0.047986f, 0.047778f, 0.047572f, 
    0.047366f, 0.047161f, 0.046958f, 0.046755f, 0.046552f, 0.046351f, 0.046151f, 0.045951f, 
    0.045753f, 0.045555f, 0.045358f, 0.045162f, 0.044967f, 0.044772f, 0.044579f, 0.044386f, 
    0.044194f, 0.044003f, 0.043813f, 0.043624f, 0.043435f, 0.043247f, 0.043060f, 0.042874f, 
    0.042689f, 0.042504f, 0.042320f, 0.042138f, 0.041955f, 0.041774f, 0.041593f, 0.041414f, 
    0.041235f, 0.041056f, 0.040879f, 0.040702f, 0.040526f, 0.040351f, 0.040177f, 0.040003f, 
    0.039830f, 0.039658f, 0.039486f, 0.039316f, 0.039146f, 0.038977f, 0.038808f, 0.038640f, 
    0.038473f, 0.038307f, 0.038141f, 0.037976f, 0.037812f, 0.037649f, 0.037486f, 0.037324f, 
    0.037163f, 0.037002f, 0.036842f, 0.036683f, 0.036524f, 0.036366f, 0.036209f, 0.036053f, 
    0.035897f, 0.035742f, 0.035587f, 0.035433f, 0.035280f, 0.035128f, 0.034976f, 0.034825f, 
    0.034674f, 0.034524f, 0.034375f, 0.034226f, 0.034078f, 0.033931f, 0.033784f, 0.033638f, 
    0.033493f, 0.033348f, 0.033204f, 0.033060f, 0.032918f, 0.032775f, 0.032634f, 0.032492f, 
    0.032352f, 0.032212f, 0.032073f, 0.031934f, 0.031796f, 0.031659f, 0.031522f, 0.031386f, 
    0.031250f, 0.031115f, 0.030980f, 0.030846f, 0.030713f, 0.030580f, 0.030448f, 0.030317f, 
    0.030186f, 0.030055f, 0.029925f, 0.029796f, 0.029667f, 0.029539f, 0.029411f, 0.029284f, 
    0.029157f, 0.029031f, 0.028906f, 0.028781f, 0.028656f, 0.028533f, 0.028409f, 0.028286f, 
    0.028164f, 0.028042f, 0.027921f, 0.027800f, 0.027680f, 0.027561f, 0.027441f, 0.027323f, 
    0.027205f, 0.027087f, 0.026970f, 0.026853f, 0.026737f, 0.026622f, 0.026507f, 0.026392f, 
    0.026278f, 0.026164f, 0.026051f, 0.025939f, 0.025827f, 0.025715f, 0.025604f, 0.025493f, 
    0.025383f, 0.025273f, 0.025164f, 0.025055f, 0.024947f, 0.024839f, 0.024732f, 0.024625f, 
    0.024518f, 0.024412f, 0.024307f, 0.024202f, 0.024097f, 0.023993f, 0.023889f, 0.023786f, 
    0.023683f, 0.023581f, 0.023479f, 0.023377f, 0.023276f, 0.023176f, 0.023075f, 0.022976f, 
    0.022876f, 0.022777f, 0.022679f, 0.022581f, 0.022483f, 0.022386f, 0.022289f, 0.022193f, 
    0.022097f, 0.022002f, 0.021906f, 0.021812f, 0.021717f, 0.021624f, 0.021530f, 0.021437f, 
    0.021344f, 0.021252f, 0.021160f, 0.021069f, 0.020978f, 0.020887f, 0.020797f, 0.020707f, 
    0.020617f, 0.020528f, 0.020439f, 0.020351f, 0.020263f, 0.020176f, 0.020088f, 0.020001f, 
    0.019915f, 0.019829f, 0.019743f, 0.019658f, 0.019573f, 0.019488f, 0.019404f, 0.019320f, 
    0.019237f, 0.019153f, 0.019071f, 0.018988f, 0.018906f, 0.018824f, 0.018743f, 0.018662f, 
    0.018581f, 0.018501f, 0.018421f, 0.018341f, 0.018262f, 0.018183f, 0.018105f, 0.018026f, 
    0.017948f, 0.017871f, 0.017794f, 0.017717f, 0.017640f, 0.017564f, 0.017488f, 0.017412f, 
    0.017337f, 0.017262f, 0.017187f, 0.017113f, 0.017039f, 0.016966f, 0.016892f, 0.016819f, 
    0.016746f, 0.016674f, 0.016602f, 0.016530f, 0.016459f, 0.016388f, 0.016317f, 0.016246f, 
    0.016176f, 0.016106f, 0.016036f, 0.015967f, 0.015898f, 0.015829f, 0.015761f, 0.015693f, 
    0.015625f, 
};

/* Reverb Time in sec */
const float reverb_time_table[128] =
{
    0.410349f, 0.440872f, 0.468882f, 0.494640f, 0.518394f, 0.540373f, 0.560793f, 0.579854f, 
    0.597743f, 0.614635f, 0.630688f, 0.646053f, 0.660866f, 0.675251f, 0.689325f, 0.703192f, 
    0.716947f, 0.730676f, 0.744456f, 0.758358f, 0.772441f, 0.786761f, 0.801365f, 0.816293f, 
    0.831583f, 0.847262f, 0.863356f, 0.879886f, 0.896866f, 0.914308f, 0.932223f, 0.950614f, 
    0.969484f, 0.988835f, 1.008663f, 1.028967f, 1.049741f, 1.070980f, 1.092677f, 1.114826f, 
    1.137419f, 1.160450f, 1.183914f, 1.207803f, 1.232115f, 1.256845f, 1.281992f, 1.307556f, 
    1.333540f, 1.359947f, 1.386784f, 1.414061f, 1.441788f, 1.469982f, 1.498661f, 1.527845f, 
    1.557561f, 1.587836f, 1.618703f, 1.650199f, 1.682363f, 1.715240f, 1.748879f, 1.783333f, 
    1.818659f, 1.854921f, 1.892183f, 1.930517f, 1.970001f, 2.010713f, 2.052741f, 2.096173f, 
    2.141107f, 2.187641f, 2.235880f, 2.285935f, 2.337920f, 2.391955f, 2.448163f, 2.506674f, 
    2.567622f, 2.631144f, 2.697384f, 2.766490f, 2.838612f, 2.913907f, 2.992536f, 3.074662f, 
    3.160454f, 3.250085f, 3.343730f, 3.441570f, 3.543786f, 3.650566f, 3.762098f, 3.878575f, 
    4.000192f, 4.127146f, 4.259638f, 4.397868f, 4.542042f, 4.692364f, 4.849041f, 5.012281f, 
    5.182294f, 5.359289f, 5.543476f, 5.735064f, 5.934264f, 6.141286f, 6.356336f, 6.356336f, 
    6.356336f, 6.356336f, 6.356336f, 6.356336f, 6.356336f, 6.356336f, 6.356336f, 6.356336f, 
    6.356336f, 6.356336f, 6.356336f, 6.356336f, 6.356336f, 6.356336f, 6.356336f, 6.356336f, 
};

/* phase lag between left and right ear. (in ms) */
const float pan_delay_table[128] =
{
    0.000000f, 0.006136f, 0.012271f, 0.018404f, 0.024534f, 0.030660f, 0.036782f, 0.042899f, 
    0.049009f, 0.055111f, 0.061205f, 0.067290f, 0.073365f, 0.079429f, 0.085481f, 0.091520f, 
    0.097545f, 0.103556f, 0.109551f, 0.115529f, 0.121490f, 0.127433f, 0.133356f, 0.139260f, 
    0.145142f, 0.151003f, 0.156841f, 0.162655f, 0.168445f, 0.174209f, 0.179948f, 0.185659f, 
    0.191342f, 0.196996f, 0.202621f, 0.208215f, 0.213778f, 0.219308f, 0.224806f, 0.230269f, 
    0.235698f, 0.241092f, 0.246449f, 0.251769f, 0.257051f, 0.262295f, 0.267499f, 0.272662f, 
    0.277785f, 0.282866f, 0.287904f, 0.292899f, 0.297850f, 0.302756f, 0.307616f, 0.312430f, 
    0.317197f, 0.321916f, 0.326586f, 0.331208f, 0.335779f, 0.340300f, 0.344770f, 0.349188f, 
    0.353553f, 0.357865f, 0.362124f, 0.366327f, 0.370476f, 0.374568f, 0.378604f, 0.382584f, 
    0.386505f, 0.390369f, 0.394173f, 0.397918f, 0.401604f, 0.405229f, 0.408792f, 0.412295f, 
    0.415735f, 0.419112f, 0.422427f, 0.425678f, 0.428864f, 0.431986f, 0.435043f, 0.438035f, 
    0.440961f, 0.443820f, 0.446612f, 0.449337f, 0.451995f, 0.454584f, 0.457105f, 0.459557f, 
    0.461940f, 0.464253f, 0.466496f, 0.468670f, 0.470772f, 0.472804f, 0.474764f, 0.476653f, 
    0.478470f, 0.480215f, 0.481888f, 0.483488f, 0.485016f, 0.486470f, 0.487851f, 0.489159f, 
    0.490393f, 0.491553f, 0.492639f, 0.493651f, 0.494588f, 0.495451f, 0.496240f, 0.496953f, 
    0.497592f, 0.498156f, 0.498645f, 0.499059f, 0.499398f, 0.499661f, 0.499849f, 0.500000f, 
};

/* for 0dBf, 0.25dBf, 0.5dBf,...f, 24dB. */
const float chamberlin_filter_db_to_q_table[97] =
{
    1.000000f, 1.000000f, 1.000000f, 1.000000f, 1.000000f, 1.000000f, 1.000000f, 1.000000f, 
    1.000000f, 1.000000f, 1.000000f, 1.000000f, 1.000000f, 1.000000f, 1.000000f, 1.000000f, 
    1.000000f, 1.000000f, 1.000000f, 1.000000f, 1.000000f, 1.029207f, 1.113701f, 1.205132f, 
    1.304068f, 1.411127f, 1.526975f, 1.652334f, 1.787984f, 1.934771f, 2.093608f, 2.265485f, 
    2.451472f, 2.652729f, 2.870507f, 3.106165f, 3.361169f, 3.637108f, 3.935700f, 4.258806f, 
    4.608437f, 4.986772f, 5.396167f, 5.839171f, 6.318544f, 6.837272f, 7.398585f, 8.005980f, 
    8.663240f, 9.374459f, 10.144065f, 10.976853f, 11.878010f, 12.853149f, 13.908342f, 15.050163f, 
    16.285723f, 17.622717f, 19.069474f, 20.635003f, 22.329057f, 24.162185f, 26.145807f, 28.292276f, 
    30.614961f, 33.128330f, 35.848037f, 38.791022f, 41.975614f, 45.421648f, 49.150589f, 53.185661f, 
    57.551996f, 62.276791f, 67.389473f, 72.921887f, 78.908490f, 85.386569f, 92.396474f, 99.981865f, 
    108.189987f, 117.071964f, 126.683116f, 137.083307f, 148.337313f, 160.515229f, 173.692904f, 187.952416f, 
    203.382577f, 220.079495f, 238.147165f, 257.698120f, 278.854132f, 301.746971f, 326.519223f, 353.325180f, 
    382.331802f, 
};

const uint8_t multi_eq_block_table_xg[] =
{	/* Gain1, Freq1, Q1, Shape1, Gain2, Freq2, Q2, Not Used, Gain3, Freq3, Q3, Not Used,
	Gain4, Freq4, Q4, Not Used, Gain5, Freq5, Shape5 */
	64, 12, 7, 0, 64, 28, 7, 0, 64, 34, 7, 0, 64, 46, 7, 0, 64, 52, 7, 0,	/* Flat */
	58, 8, 7, 0, 66, 16, 3, 0, 68, 33, 3, 0, 60, 44, 5, 0, 58, 50, 7, 0,	/* Jazz */
	68, 16, 7, 0, 60, 24, 20, 0, 67, 34, 7, 0, 60, 40, 20, 0, 70, 48, 7, 0,	/* Pops */
	71, 16, 7, 0, 68, 20, 7, 0, 60, 36, 5, 0, 68, 41, 10, 0, 66, 50, 7, 0,	/* Rock */
	67, 12, 7, 0, 68, 24, 7, 0, 64, 34, 5, 0, 66, 50, 7, 0, 61, 52, 7, 0,	/* Concert */
};

const float eq_freq_table_xg[] =
{
	20, 22, 25, 28, 32, 36, 40, 45, 50, 56, 63, 70, 80, 90, 100, 110,
	125, 140, 160, 180, 200, 225, 250, 280, 315, 355, 400, 450, 500, 560, 630,
	700, 800, 900, 1000, 1100, 1200, 1400, 1600, 1800, 2000, 2200, 2500, 2800, 3200, 3600,
	4000, 4500, 5000, 5600, 6300, 7000, 8000, 9000, 10000, 11000, 12000, 14000, 16000, 18000, 20000,
};

const float lfo_freq_table_xg[] =
{
	0.00f, 0.04f, 0.08f, 0.13f, 0.17f, 0.21f, 0.25f, 0.29f, 0.34f, 0.38f, 0.42f, 0.46f, 0.51f, 0.55f, 0.59f, 0.63f,
	0.67f, 0.72f, 0.76f, 0.80f, 0.84f, 0.88f, 0.93f, 0.97f, 1.01f, 1.05f, 1.09f, 1.14f, 1.18f, 1.22f, 1.26f, 1.30f,
	1.35f, 1.39f, 1.43f, 1.47f, 1.51f, 1.56f, 1.60f, 1.64f, 1.68f, 1.72f, 1.77f, 1.81f, 1.85f, 1.89f, 1.94f, 1.98f,
	2.02f, 2.06f, 2.10f, 2.15f, 2.19f, 2.23f, 2.27f, 2.31f, 2.36f, 2.40f, 2.44f, 2.48f, 2.52f, 2.57f, 2.61f, 2.65f,
	2.69f, 2.78f, 2.86f, 2.94f, 3.03f, 3.11f, 3.20f, 2.28f, 3.37f, 3.45f, 3.53f, 3.62f, 3.70f, 3.87f, 4.04f, 4.21f,
	4.37f, 4.54f, 4.71f, 4.88f, 5.05f, 5.22f, 5.38f, 5.55f, 5.72f, 6.06f, 6.39f, 6.73f, 7.07f, 7.40f, 7.74f, 8.08f,
	8.41f, 8.75f, 9.08f, 9.42f, 9.76f, 10.1f, 10.8f, 11.4f, 12.1f, 12.8f, 13.5f, 14.1f, 14.8f, 15.5f, 16.2f, 16.8f,
	17.5f, 18.2f, 19.5f, 20.9f, 22.2f, 23.6f, 24.9f, 26.2f, 27.6f, 28.9f, 30.3f, 31.6f, 33.0f, 34.3f, 37.0f, 39.7f,
};

const float mod_delay_offset_table_xg[] =
{
	0.0f, 0.1f, 0.2f, 0.3f, 0.4f, 0.5f, 0.6f, 0.7f, 0.8f, 0.9f, 1.0f, 1.1f, 1.2f, 1.3f, 1.4f, 1.5f,
	1.6f, 1.7f, 1.8f, 1.9f, 2.0f, 2.1f, 2.2f, 2.3f, 2.4f, 2.5f, 2.6f, 2.7f, 2.8f, 2.9f, 3.0f, 3.1f, 
	3.2f, 3.3f, 3.4f, 3.5f, 3.6f, 3.7f, 3.8f, 3.9f, 4.0f, 4.1f, 4.2f, 4.3f, 4.4f, 4.5f, 4.6f, 4.7f,
	4.8f, 4.9f, 5.0f, 5.1f, 5.2f, 5.3f, 5.4f, 5.5f, 5.6f, 5.7f, 5.8f, 5.9f, 6.0f, 6.1f, 6.2f, 6.3f,
	6.4f, 6.5f, 6.6f, 6.7f, 6.8f, 6.9f, 7.0f, 7.1f, 7.2f, 7.3f, 7.4f, 7.5f, 7.6f, 7.7f, 7.8f, 7.9f,
	8.0f, 8.1f, 8.2f, 8.3f, 8.4f, 8.5f, 8.6f, 8.7f, 8.8f, 8.9f, 9.0f, 9.1f, 9.2f, 9.3f, 9.4f, 9.5f,
	9.6f, 9.7f, 9.8f, 9.9f, 10.0f, 11.1f, 12.2f, 13.3f, 14.4f, 15.5f, 17.1f, 18.6f, 20.2f, 21.8f,	23.3f, 24.9f,
	26.5f, 28.0f, 29.6f, 31.2f, 32.8f, 34.3f, 35.9f, 37.5f, 39.0f, 40.6f,	42.2f, 43.7f, 45.3f, 46.9f, 48.4f, 50.0f,
};

const float reverb_time_table_xg[] =
{
	0.3f, 0.4f, 0.5f, 0.6f, 0.7f, 0.8f, 0.9f, 1.0f, 1.1f, 1.2f, 1.3f, 1.4f, 1.5f, 1.6f, 1.7f, 1.8f,
	1.9f, 2.0f, 2.1f, 2.2f, 2.3f, 2.4f, 2.5f, 2.6f, 2.7f, 2.8f, 2.9f, 3.0f, 3.1f, 3.2f, 3.3f, 3.4f,
	3.5f, 3.6f, 3.7f, 3.8f, 3.9f, 4.0f, 4.1f, 4.2f, 4.3f, 4.4f, 4.5f, 4.6f, 4.7f, 4.8f, 4.9f, 5.0f,
	5.5f, 6.0f, 6.5f, 7.0f, 7.5f, 8.0f, 8.5f, 9.0f, 9.5f, 10.0f, 11.0f, 12.0f, 13.0f, 14.0f, 15.0f, 16.0f,
	17.0f, 18.0f, 19.0f, 20.0f, 25.0f, 30.0f,
};

const float delay_time_table_xg[] =
{
	0.1f, 1.7f, 3.2f, 4.8f, 6.4f, 8.0f, 9.5f, 11.1f, 12.7f, 14.3f, 15.8f, 17.4f, 19.0f, 20.6f, 22.1f, 23.7f,
	25.3f, 26.9f, 28.4f, 30.0f, 31.6f, 33.2f, 34.7f, 36.3f, 37.9f, 39.5f, 41.0f, 42.6f, 44.2f, 45.7f, 47.3f, 48.9f,
	50.5f, 52.0f, 53.6f, 55.2f, 56.8f, 58.3f, 59.9f, 61.5f, 63.1f, 64.6f, 66.2f, 67.8f, 69.4f, 70.9f, 72.5f, 74.1f,
	75.7f, 77.2f, 78.8f, 80.4f, 81.9f, 83.5f, 85.1f, 86.7f, 88.2f, 89.8f, 91.4f, 93.0f, 94.5f, 96.1f, 97.7f, 99.3f,
	100.8f, 102.4f, 104.0f, 105.6f, 107.1f, 108.7f, 110.3f, 111.9f, 113.4f, 115.0f, 116.6f, 118.2f, 119.7f, 121.3f, 122.9f, 124.4f,
	126.0f, 127.6f, 129.2f, 130.7f, 132.3f, 133.9f, 135.5f, 137.0f, 138.6f, 140.2f, 141.8f, 143.3f, 144.9f, 146.5f, 148.1f, 149.6f,
	151.2f, 152.8f, 154.4f, 155.9f, 157.5f, 159.1f, 160.6f, 162.2f, 163.8f, 165.4f, 166.9f, 168.5f, 170.1f, 171.7f, 173.2f, 174.8f,
	176.4f, 178.0f, 179.5f, 181.1f, 182.7f, 184.3f, 185.8f, 187.4f, 189.0f, 190.6f, 192.1f, 193.7f, 195.3f, 196.9f, 198.4f, 200.0f,
};

const int16_t cutoff_freq_table_gs[] =
{
	250, 250, 250, 250, 250, 250, 250, 250,
	315, 315, 315, 315, 315, 315, 315, 315,
	400, 400, 400, 400, 400, 400, 400, 400,
	500, 500, 500, 500, 500, 500, 500, 500,
	630, 630, 630, 630, 630, 630, 630, 630,
	800, 800, 800, 800, 800, 800, 800, 800,
	1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000,
	1250, 1250, 1250, 1250, 1250, 1250, 1250, 1250,
	1600, 1600, 1600, 1600, 1600, 1600, 1600, 1600,
	2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,
	2500, 2500, 2500, 2500, 2500, 2500, 2500, 2500,
	3150, 3150, 3150, 3150, 3150, 3150, 3150, 3150,
	4000, 4000, 4000, 4000, 4000, 4000, 4000, 4000,
	5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000,
	6300, 6300, 6300, 6300, 6300, 6300, 6300, 6300,
	8000, 8000, 8000, 8000, 8000, 8000, 8000, 8000,
};

const int16_t lpf_table_gs[] =
{
	250, 250, 250, 250, 250, 250, 250, 250,
	315, 315, 315, 315, 315, 315, 315, 315,
	400, 400, 400, 400, 400, 400, 400, 400,
	500, 500, 500, 500, 500, 500, 500, 500,
	630, 630, 630, 630, 630, 630, 630, 630,
	800, 800, 800, 800, 800, 800, 800, 800,
	1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000,
	1250, 1250, 1250, 1250, 1250, 1250, 1250, 1250,
	1600, 1600, 1600, 1600, 1600, 1600, 1600, 1600,
	2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,
	2500, 2500, 2500, 2500, 2500, 2500, 2500, 2500,
	3150, 3150, 3150, 3150, 3150, 3150, 3150, 3150,
	4000, 4000, 4000, 4000, 4000, 4000, 4000, 4000,
	5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000,
	6300, 6300, 6300, 6300, 6300, 6300, 6300, 6300,
	-1, -1, -1, -1, -1, -1, -1, -1,
};

const int16_t eq_freq_table_gs[] =
{
	200, 200, 200, 200, 200, 200, 200, 200,
	250, 250, 250, 250, 250, 250, 250, 250,
	315, 315, 315, 315, 315, 315, 315, 315,
	400, 400, 400, 400, 400, 400, 400, 400,
	500, 500, 500, 500, 500, 500, 500, 500,
	630, 630, 630, 630, 630, 630, 630, 630,
	800, 800, 800, 800, 800, 800, 800, 800,
	1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000,
	1250, 1250, 1250, 1250, 1250, 1250, 1250, 1250,
	1600, 1600, 1600, 1600, 1600, 1600, 1600, 1600,
	2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,
	2500, 2500, 2500, 2500, 2500, 2500, 2500, 2500,
	3150, 3150, 3150, 3150, 3150, 3150, 3150, 3150,
	4000, 4000, 4000, 4000, 4000, 4000, 4000, 4000,
	5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000,
	6300, 6300, 6300, 6300, 6300, 6300, 6300, 6300,
};

const float lofi_sampling_freq_table_xg[] =
{
	44100.0, 22100.0, 14700.0, 11000.0, 8800.0, 7400.0, 6300.0, 5500.0,
	4900.0, 4400.0, 4000.0, 3700.0, 3400.0, 3200.0, 2900.0, 2800.0,
	2600.0, 2500.0, 2300.0, 2200.0, 2100.0, 2000.0, 1920.0, 1840.0,
	1760.0, 1700.0, 1630.0, 1580.0, 1520.0, 1470.0, 1420.0, 1380.0,
	1340.0, 1300.0, 1260.0, 1230.0, 1190.0, 1160.0, 1130.0, 1110.0,
	1080.0, 1050.0, 1030.0, 1000.0, 980.0, 959.0, 938.0, 919.0,
	900.0, 882.0, 865.0, 848.0, 832.0, 817.0, 802.0, 788.0,
	774.0, 760.0, 747.0, 735.0, 723.0, 711.0, 700.0, 689.0,
	678.0, 668.0, 658.0, 649.0, 639.0, 630.0, 621.0, 613.0,
	604.0, 596.0, 588.0, 580.0, 573.0, 565.0, 558.0, 551.0,
	544.0, 538.0, 531.0, 525.0, 519.0, 513.0, 507.0, 501.0,
	496.0, 490.0, 485.0, 479.0, 474.0, 469.0, 464.0, 459.0,
	455.0, 450.0, 445.0, 441.0, 437.0, 432.0, 428.0, 424.0,
	420.0, 416.0, 412.0, 408.0, 405.0, 401.0, 397.0, 394.0,
	390.0, 387.0, 383.0, 380.0, 377.0, 374.0, 371.0, 368.0,
	364.0, 361.0, 359.0, 356.0, 353.0, 350.0, 347.0, 345.0,
};

void init_tables(void)
{
	// Only needs to be done once.
	static bool done = false;
	if (done) return;
	done = true;

	init_freq_table();
	init_freq_table_tuning();
	init_freq_table_pytha();
	init_freq_table_meantone();
	init_freq_table_pureint();
	init_bend_fine();
	init_bend_coarse();
	init_triangular_table();
	init_gm2_pan_table();
	init_attack_vol_table();
	init_sb_vol_table();
	init_modenv_vol_table();
	init_def_vol_table();
	init_gs_vol_table();
	init_perceived_vol_table();
	init_gm2_vol_table();
}

int32_t get_note_freq(Sample *sp, int note)
{
	int32_t f;
	int16_t sf, sn;
	double ratio;

	f = freq_table[note];
	/* GUS/SF2 - Scale Tuning */
	if ((sf = sp->scale_factor) != 1024) {
		sn = sp->scale_freq;
		ratio = pow(2.0, (note - sn) * (sf - 1024) / 12288.0);
		f = f * ratio + 0.5;
	}
	return f;
}
}