#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <locale.h>
#include "sgsearch.h"
#include "config.h"


int main (int argc, char *argv[]) {
	setlocale (LC_ALL, "");
	bindtextdomain (PROJECT_NAME, LOCALE_DIR);
	bind_textdomain_codeset (PROJECT_NAME, "UTF-8");
	textdomain (PROJECT_NAME);
	
	gtk_init (&argc, &argv);
	load_gui();
	
	app = g_new(T_APP, 1);
	app->foundfiles = (GtkListStore *) gtk_tree_view_get_model (gui->foundfiles);
	app->startfolder = NULL;
	app->thread = NULL;
	
	load_parameters(argc, argv);
	GAppInfo *editorapp = g_app_info_get_default_for_type ("text/plain", FALSE);
	app->texteditor = g_strdup(g_app_info_get_executable (editorapp));
	g_object_unref (editorapp);
	
	gtk_widget_show ((GtkWidget *) gui->main_application);
	gtk_main();
}


void load_gui() {
	gui = g_new(T_GUI, 1);
	gui->builder = gtk_builder_new();
	gtk_builder_add_from_file (gui->builder, UI_FILE, NULL);
	gtk_builder_connect_signals (gui->builder, NULL);
	
	gui->main_application = (GtkApplicationWindow *) gtk_builder_get_object (gui->builder, "main_application");
	gtk_window_set_icon_name((GtkWindow *) gui->main_application, PROJECT_NAME);
	gui->about_dialog = (GtkAboutDialog *) gtk_builder_get_object (gui->builder, "about_dialog");
	gtk_window_set_icon_name((GtkWindow *) gui->about_dialog, PROJECT_NAME);
	gtk_about_dialog_set_version (gui->about_dialog, PROJECT_VERSION);
	
	gui->startfolder = (GtkFileChooser*) gtk_builder_get_object (gui->builder, "startfolder");
	gui->content = (GtkEntry *) gtk_builder_get_object (gui->builder, "content");
	gui->filename = (GtkEntry *) gtk_builder_get_object (gui->builder, "filename");
	gui->search = (GtkButton *) gtk_builder_get_object (gui->builder, "search");
	gui->cancel = (GtkButton *) gtk_builder_get_object (gui->builder, "cancel");
	gui->foundfiles = (GtkTreeView *) gtk_builder_get_object (gui->builder, "files_treeview");
	gui->progressbar = (GtkProgressBar *) gtk_builder_get_object (gui->builder, "progressbar");
	gui->contextmenu = (GtkMenu *) gtk_builder_get_object (gui->builder, "context_menu");
}


void load_parameters(int argc, char *argv[]) {
	gint n = 2;
	if (argc > 1) {
		if (g_file_test(argv[1],G_FILE_TEST_IS_DIR)) {
			gtk_file_chooser_set_filename (gui->startfolder, argv[1]);
		} else {
			gtk_file_chooser_set_filename (gui->startfolder, g_getenv("HOME"));
			n = 1;
		}
		if (argc == n + 1 || argc == n + 2) {
			if (argv[n][0] == '/') {
				gtk_entry_set_text (gui->content, argv[n] + 1);
			} else {
				gtk_entry_set_text (gui->filename, argv[n]);
				n++;
				if (argc == n + 1 && argv[n][0] == '/') {
					gtk_entry_set_text (gui->content, argv[n] + 1);
				}
			}
			app->thread = g_thread_new(NULL, run_search, NULL);
		}
	} else {
		gtk_file_chooser_set_filename (gui->startfolder, g_getenv("HOME"));
	}
}


gboolean new_filefound_handler (GIOChannel *source, GIOCondition condition, gpointer data) {
	gchar *line;
	if (G_IO_STATUS_EOF != g_io_channel_read_line (app->stdout_ioc, &line, NULL, NULL, NULL)) {
		if (line != NULL ) {
			line[strlen (line)-1] = '\0';
			GtkTreeIter iter;
			gtk_list_store_append (app->foundfiles, &iter);
			const gchar *path = line + strlen(app->startfolder) + 1;
			gtk_list_store_set (app->foundfiles, &iter, 0, path, -1);
			g_free (line);
		}
	}
	return TRUE;
}


void on_cancel_clicked (GtkWidget *widget, gpointer data) {
	kill (app->pid, SIGTERM);
}


void on_default_action_activate(GtkWidget *widget, gpointer data) {
	gchar *commandline = g_strdup_printf("exo-open \"%s/%s\"", app->startfolder, app->path);
	g_spawn_command_line_async (commandline, NULL);
	g_free(commandline);
	g_free(app->path);
}


gboolean on_files_treeview_button_press_event (GtkWidget *widget, GdkEvent *event, gpointer data) {
	if (event->type == GDK_2BUTTON_PRESS) {
		GtkTreeSelection *selection = gtk_tree_view_get_selection (gui->foundfiles);
		GtkTreeIter iter;
		gchar *path;
		gtk_tree_selection_get_selected (selection, NULL, &iter);
		gtk_tree_model_get ((GtkTreeModel *) app->foundfiles, &iter, 0, &path, -1);
		gchar *commandline = g_strdup_printf("exo-open \"%s/%s\"", app->startfolder, path);
		g_spawn_command_line_async (commandline, NULL);
		g_free(path);
		g_free(commandline);
	} else if (event->button.button == 3) { //right-click: select then open context menu
		GtkTreePath *path;
		gdouble x, y;
		gdk_event_get_coords(event, &x, &y);
		gtk_tree_view_get_path_at_pos (gui->foundfiles, (gint) x, (gint) y, &path, NULL, NULL, NULL);
		GtkTreeSelection *selection = gtk_tree_view_get_selection (gui->foundfiles);
		gtk_tree_selection_select_path (selection, path);
		GtkTreeIter iter;
		gtk_tree_selection_get_selected (selection, NULL, &iter);
		gtk_tree_model_get ((GtkTreeModel *) app->foundfiles, &iter, 0, &(app->path), -1);
		gtk_menu_popup_at_pointer(gui->contextmenu, event);			
	}
	return FALSE;
}


void on_open_text_activate(GtkWidget *widget, gpointer data) {
	if (app->texteditor == NULL) {
		on_default_action_activate(widget, data);
	} else {
		gchar *commandline = g_strdup_printf("%s \"%s/%s\"", app->texteditor, app->startfolder, app->path);
		g_spawn_command_line_async (commandline, NULL);
		g_free(commandline);
		g_free(app->path);
	}
}


void on_open_folder_activate(GtkWidget *widget, gpointer data) {
	gchar *dirname = g_path_get_dirname (app->path);
	gchar *commandline = g_strdup_printf("exo-open --launch FileManager \"%s/%s\"", app->startfolder, dirname);
	g_free(dirname);
	g_spawn_command_line_async (commandline, NULL);
	g_free(commandline);
	g_free(app->path);
}


void on_process_end (GPid pid, gint status, gpointer data) {
	if (app->thread != NULL) {
		g_thread_unref(app->thread);
		app->thread = NULL;
	}
	g_spawn_close_pid(app->pid);
	g_source_remove (app->output_watcher);
	new_filefound_handler (NULL, G_IO_IN, data);
	g_io_channel_shutdown (app->stdout_ioc, FALSE, NULL);
	g_io_channel_unref (app->stdout_ioc);
	g_source_remove (app->pulse_watcher);
	gtk_progress_bar_set_fraction (gui->progressbar, 0);
	gtk_widget_set_sensitive ((GtkWidget *) gui->search, TRUE);
	gtk_widget_set_sensitive ((GtkWidget *) gui->cancel, FALSE);
}


void on_search_clicked (GtkWidget *widget, gpointer data) {
	run_search(data);
}


gboolean progress_handler (gpointer data) {
	gtk_progress_bar_pulse (gui->progressbar);
	return TRUE;
}

gpointer run_search(gpointer data) {
	if (app->startfolder != NULL) {
		g_free(app->startfolder);
	}
	app->startfolder = gtk_file_chooser_get_filename (gui->startfolder);
	const gchar *filename = gtk_entry_get_text (gui->filename);
	const gchar *content = gtk_entry_get_text (gui->content);
	gtk_list_store_clear (app->foundfiles);
	
	gchar *commandline;
	gchar **command;
	if (0 == strlen (content)) {
		commandline = g_strdup_printf ("find \"%s\" \\( -type f -or -type l -or -type d \\) -iname \"*%s*\"", app->startfolder, filename);
	} else {
		if (strlen (filename) > 0) {
			commandline = g_strdup_printf ("grep -rli \"%s\" --include \"*%s*\" \"%s\"", content, filename, app->startfolder);
		} else {
			commandline = g_strdup_printf ("grep -rli \"%s\" \"%s\"", content, app->startfolder);
		}		
	}
	g_print ("%s\n", commandline);
	g_shell_parse_argv (commandline, NULL, &command, NULL);
	g_free (commandline);
	
	gint stdoutfd;
	g_spawn_async_with_pipes (NULL, command, NULL, G_SPAWN_SEARCH_PATH|G_SPAWN_DO_NOT_REAP_CHILD, NULL, NULL, &(app->pid), NULL, &stdoutfd, NULL, NULL);
	g_strfreev (command);
	app->pulse_watcher = g_timeout_add (100, progress_handler, NULL);
	app->stdout_ioc = g_io_channel_unix_new (stdoutfd);
	app->output_watcher = g_io_add_watch (app->stdout_ioc, G_IO_IN, new_filefound_handler, NULL);
	g_child_watch_add (app->pid, on_process_end, NULL);
	
	gtk_widget_set_sensitive ((GtkWidget *) gui->search, FALSE);
	gtk_widget_set_sensitive ((GtkWidget *) gui->cancel, TRUE);
}
