// Copyright (c) 1999-2016  David Muse
// See the file COPYING for more information

#include <config.h>
#include <sqlrelay/sqlrclient.h>
#include <rudiments/memorypool.h>
#include <rudiments/file.h>
#include <rudiments/charstring.h>
#include <rudiments/permissions.h>
#include <rudiments/datetime.h>
#include <rudiments/bytestring.h>
#include <rudiments/character.h>
#include <rudiments/filesystem.h>
#include <rudiments/error.h>
#include <defines.h>
#define NEED_DATATYPESTRING
#include <datatypes.h>

#ifndef MAXPATHLEN
	#define MAXPATHLEN 256
#endif

// we're optimistic that the average query will contain 16 bind variables
#define OPTIMISTIC_BIND_COUNT 16

// we're optimistic that the average query will contain 15 columns whose names
// average 10 characters in length
#define OPTIMISTIC_COLUMN_COUNT 15
#define OPTIMISTIC_AVERAGE_COLUMN_NAME_LENGTH 10
#define OPTIMISTIC_COLUMN_DATA_SIZE OPTIMISTIC_COLUMN_COUNT*\
					OPTIMISTIC_AVERAGE_COLUMN_NAME_LENGTH

// we're optimistic that the average query will contain 15 rows whose fields
// average 15 characters in length
#define OPTIMISTIC_ROW_COUNT 15
#define OPTIMISTIC_AVERAGE_FIELD_LENGTH 15
#define OPTIMISTIC_RESULT_SET_SIZE OPTIMISTIC_COLUMN_COUNT*\
					OPTIMISTIC_ROW_COUNT*\
					OPTIMISTIC_AVERAGE_FIELD_LENGTH



class sqlrclientrow {
	friend class sqlrcursor;
	private:
			sqlrclientrow(uint32_t colcount);
			~sqlrclientrow();
		void	resize(uint32_t colcount);
		void	addField(uint32_t column, 
				const char *buffer, uint32_t length);

		char		*getField(uint32_t column) const;
		uint32_t	getFieldLength(uint32_t column) const;

		sqlrclientrow	*next;

		char		*fields[OPTIMISTIC_COLUMN_COUNT];
		uint32_t	fieldlengths[OPTIMISTIC_COLUMN_COUNT];
		char		**extrafields;
		uint32_t	*extrafieldlengths;

		uint32_t	colcount;
};

sqlrclientrow::sqlrclientrow(uint32_t colcount) {
	this->colcount=colcount;
	if (colcount>=OPTIMISTIC_COLUMN_COUNT) {
		extrafields=new char *[colcount-OPTIMISTIC_COLUMN_COUNT];
		extrafieldlengths=new uint32_t
					[colcount-OPTIMISTIC_COLUMN_COUNT];
	} else {
		extrafields=NULL;
		extrafieldlengths=NULL;
	}
}

sqlrclientrow::~sqlrclientrow() {
	delete[] extrafields;
	delete[] extrafieldlengths;
}

void sqlrclientrow::resize(uint32_t colcount) {
	this->colcount=colcount;
	if (colcount>=OPTIMISTIC_COLUMN_COUNT) {
		delete[] extrafields;
		delete[] extrafieldlengths;
		extrafields=new char *[colcount-OPTIMISTIC_COLUMN_COUNT];
		extrafieldlengths=new uint32_t
					[colcount-OPTIMISTIC_COLUMN_COUNT];
	}
}

void sqlrclientrow::addField(uint32_t column,
				const char *buffer, uint32_t length) {
	if (column<OPTIMISTIC_COLUMN_COUNT) {
		fields[column]=(char *)buffer;
		fieldlengths[column]=length;
	} else {
		extrafields[column-OPTIMISTIC_COLUMN_COUNT]=(char *)buffer;
		extrafieldlengths[column-OPTIMISTIC_COLUMN_COUNT]=length;
	}
}

char *sqlrclientrow::getField(uint32_t column) const {
	if (column<OPTIMISTIC_COLUMN_COUNT) {
		return fields[column];
	} else {
		return extrafields[column-OPTIMISTIC_COLUMN_COUNT];
	}
}

uint32_t sqlrclientrow::getFieldLength(uint32_t column) const {
	if (column<OPTIMISTIC_COLUMN_COUNT) {
		return fieldlengths[column];
	} else {
		return extrafieldlengths[column-OPTIMISTIC_COLUMN_COUNT];
	}
}


class sqlrclientcolumn {
	public:
		char		*name;
		uint16_t	type;
		char		*typestring;
		uint16_t	typestringlength;
		uint32_t	length;
		uint32_t	longest;
		unsigned char	longdatatype;
		uint32_t	precision;
		uint32_t	scale;
		uint16_t	nullable;
		uint16_t	primarykey;
		uint16_t	unique;
		uint16_t	partofkey;
		uint16_t	unsignednumber;
		uint16_t	zerofill;
		uint16_t	binary;
		uint16_t	autoincrement;
};

enum columncase {
	MIXED_CASE,
	UPPER_CASE,
	LOWER_CASE
};


class sqlrclientbindvar {
	friend class sqlrcursor;
	friend class sqlrcursorprivate;
	private:
		char	*variable;
		union {
			char	*stringval;
			int64_t	integerval;
			struct {
				double		value;
				uint32_t	precision;
				uint32_t	scale;
			} doubleval;
			struct {
				int16_t	year;
				int16_t	month;
				int16_t	day;
				int16_t	hour;
				int16_t	minute;
				int16_t	second;
				int32_t	microsecond;
				char	*tz;
			} dateval;
			char		*lobval;
			uint16_t	cursorid;
		} value;
		uint32_t	valuesize;
		uint32_t	resultvaluesize;

		sqlrclientbindvartype_t 	type;

		bool		send;

		bool		substituted;
		bool		donesubstituting;
};


class sqlrcursorprivate {
	friend class sqlrcursor;
	private:
		sqlrclientbindvar	*findVar(const char *variable,
					dynamicarray<sqlrclientbindvar> *vars);

		bool		_resumed;
		bool		_cached;

		// query
		char		*_querybuffer;
		const char	*_queryptr;
		uint32_t	_querylen;
		char		*_fullpath;
		bool		_reexecute;

		// substitution variables
		dynamicarray<sqlrclientbindvar>	*_subvars;
		bool				_dirtysubs;

		// bind variables
		dynamicarray<sqlrclientbindvar>	*_inbindvars;
		dynamicarray<sqlrclientbindvar>	*_outbindvars;
		bool				_validatebinds;
		bool				_dirtybinds;

		// result set
		uint64_t	_rsbuffersize;
		uint16_t	_sendcolumninfo;
		uint16_t	_sentcolumninfo;

		uint16_t	_suspendresultsetsent;
		bool		_endofresultset;

		uint16_t	_columntypeformat;
		uint32_t	_colcount;
		uint32_t	_previouscolcount;

		columncase	_colcase;

		sqlrclientcolumn	*_columns;
		sqlrclientcolumn	*_extracolumns;
		memorypool		*_colstorage;
		char			**_columnnamearray;

		uint64_t	_firstrowindex;
		uint64_t	_rowcount;
		uint64_t	_previousrowcount;
		uint16_t	_knowsactualrows;
		uint64_t	_actualrows;
		uint16_t	_knowsaffectedrows;
		uint64_t	_affectedrows;

		sqlrclientrow	**_rows;
		sqlrclientrow	**_extrarows;
		memorypool	*_rowstorage;
		sqlrclientrow	*_firstextrarow;
		char		***_fields;
		uint32_t	**_fieldlengths;

		bool		_returnnulls;

		// result set caching
		bool		_cacheon;
		int32_t		_cachettl;
		char		*_cachedestname;
		char		*_cachedestindname;
		file		*_cachedest;
		file		*_cachedestind;
		file		*_cachesource;
		file		*_cachesourceind;

		// error
		int64_t		_errorno;
		char		*_error;

		// copy references flag
		bool		_copyrefs;

		// parent connection
		sqlrconnection	*_sqlrc;

		// next/previous pointers
		sqlrcursor	*_next;
		sqlrcursor	*_prev;

		// cursor id
		uint16_t	_cursorid;
		bool		_havecursorid;
};

// This method is a member of sqlrcursorprivate, rather than sqlrcuror, because
// if it were a member of sqlrcursor, then it would have to be defined in the
// header file.  If it were, then since it references a
// dynamicarray<sqlrclientbindvar>, older compilers would also require that
// sqlrclientbindvar be defined in the header file as well.  To avoid all of
// that, it's part of sqlrcursorprivate.
sqlrclientbindvar *sqlrcursorprivate::findVar(const char *variable,
					dynamicarray<sqlrclientbindvar> *vars) {
	for (uint16_t i=0; i<vars->getLength(); i++) {
		if (!charstring::compare((*vars)[i].variable,variable)) {
			return &((*vars)[i]);
		}
	}
	return NULL;
}


sqlrcursor::sqlrcursor(sqlrconnection *sqlrc, bool copyreferences) {
	init(sqlrc,copyreferences);
}

sqlrcursor::sqlrcursor(sqlrconnection *sqlrc) {
	init(sqlrc,false);
}

void sqlrcursor::init(sqlrconnection *sqlrc, bool copyreferences) {

	pvt=new sqlrcursorprivate;

	// copy references
	pvt->_copyrefs=copyreferences;

	pvt->_sqlrc=sqlrc;

	// put self in connection's cursor list
	if (pvt->_sqlrc->lastcursor()) {
		pvt->_sqlrc->lastcursor()->pvt->_next=this;
		pvt->_prev=pvt->_sqlrc->lastcursor();
	} else {
		pvt->_sqlrc->firstcursor(this);
		pvt->_prev=NULL;
	}
	pvt->_sqlrc->lastcursor(this);
	pvt->_next=NULL;

	// session state
	pvt->_cached=false;

	// query
	pvt->_querybuffer=NULL;
	pvt->_fullpath=NULL;

	// result set
	pvt->_rsbuffersize=0;

	pvt->_firstrowindex=0;
	pvt->_rowcount=0;
	pvt->_previousrowcount=0;
	pvt->_actualrows=0;
	pvt->_affectedrows=0;
	pvt->_endofresultset=true;

	pvt->_errorno=0;
	pvt->_error=NULL;

	pvt->_rows=NULL;
	pvt->_extrarows=NULL;
	pvt->_firstextrarow=NULL;
	pvt->_rowstorage=new memorypool(OPTIMISTIC_RESULT_SET_SIZE,
			OPTIMISTIC_RESULT_SET_SIZE/OPTIMISTIC_ROW_COUNT,5);
	pvt->_fields=NULL;
	pvt->_fieldlengths=NULL;

	pvt->_colcount=0;
	pvt->_previouscolcount=0;
	pvt->_columns=NULL;
	pvt->_extracolumns=NULL;
	pvt->_colstorage=new memorypool(OPTIMISTIC_COLUMN_DATA_SIZE,
			OPTIMISTIC_COLUMN_DATA_SIZE/OPTIMISTIC_COLUMN_COUNT,5);
	pvt->_columnnamearray=NULL;

	pvt->_returnnulls=false;

	// cache file
	pvt->_cachesource=NULL;
	pvt->_cachesourceind=NULL;
	pvt->_cachedestname=NULL;
	pvt->_cachedestindname=NULL;
	pvt->_cachedest=NULL;
	pvt->_cachedestind=NULL;
	pvt->_cacheon=false;

	// options...
	pvt->_sendcolumninfo=SEND_COLUMN_INFO;
	pvt->_sentcolumninfo=SEND_COLUMN_INFO;
	pvt->_columntypeformat=COLUMN_TYPE_IDS;
	pvt->_colcase=MIXED_CASE;

	// cursor id
	pvt->_cursorid=0;
	pvt->_havecursorid=false;

	// initialize all bind/substitution-related variables
	pvt->_subvars=new dynamicarray<sqlrclientbindvar>(
					OPTIMISTIC_BIND_COUNT,16);
	pvt->_inbindvars=new dynamicarray<sqlrclientbindvar>(
					OPTIMISTIC_BIND_COUNT,16);
	pvt->_outbindvars=new dynamicarray<sqlrclientbindvar>(
					OPTIMISTIC_BIND_COUNT,16);
	clearVariables();
}

sqlrcursor::~sqlrcursor() {

	// abort result set if necessary
	if (pvt->_sqlrc && !pvt->_sqlrc->endsessionsent() &&
				!pvt->_sqlrc->suspendsessionsent()) {
		closeResultSet(true);
	}

	// deallocate copied references
	deleteVariables();
	delete pvt->_outbindvars;
	delete pvt->_inbindvars;
	delete pvt->_subvars;

	// deallocate the query buffer
	delete[] pvt->_querybuffer;

	// deallocate the fullpath (used for file queries)
	delete[] pvt->_fullpath;

	clearResultSet();
	delete[] pvt->_columns;
	delete[] pvt->_extracolumns;
	delete pvt->_colstorage;
	if (pvt->_rows) {
		for (uint32_t i=0; i<OPTIMISTIC_ROW_COUNT; i++) {
			delete pvt->_rows[i];
		}
		delete[] pvt->_rows;
	}
	delete pvt->_rowstorage;

	// it's possible for the connection to be deleted before the 
	// cursor is, in that case, don't do any of this stuff
	if (pvt->_sqlrc) {

		// remove self from connection's cursor list
		if (!pvt->_next && !pvt->_prev) {
			pvt->_sqlrc->firstcursor(NULL);
			pvt->_sqlrc->lastcursor(NULL);
		} else {
			sqlrcursor	*temp=pvt->_next;
			if (pvt->_next) {
				pvt->_next->pvt->_prev=pvt->_prev;
			} else {
				pvt->_sqlrc->lastcursor(pvt->_prev);
			}
			if (pvt->_prev) {
				pvt->_prev->pvt->_next=temp;
			} else {
				pvt->_sqlrc->firstcursor(pvt->_next);
			}
		}

		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Deallocated cursor\n");
			pvt->_sqlrc->debugPreEnd();
		}
	}

	if (pvt->_copyrefs && pvt->_cachedestname) {
		delete[] pvt->_cachedestname;
	}
	delete[] pvt->_cachedestindname;

	delete pvt;
}

void sqlrcursor::setResultSetBufferSize(uint64_t rows) {
	pvt->_rsbuffersize=rows;
	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Result Set Buffer Size: ");
		pvt->_sqlrc->debugPrint((int64_t)rows);
		pvt->_sqlrc->debugPrint("\n");
		pvt->_sqlrc->debugPreEnd();
	}
}

uint64_t sqlrcursor::getResultSetBufferSize() {
	return pvt->_rsbuffersize;
}

void sqlrcursor::dontGetColumnInfo() {
	pvt->_sendcolumninfo=DONT_SEND_COLUMN_INFO;
}

void sqlrcursor::getColumnInfo() {
	pvt->_sendcolumninfo=SEND_COLUMN_INFO;
}

void sqlrcursor::mixedCaseColumnNames() {
	pvt->_colcase=MIXED_CASE;
}

void sqlrcursor::upperCaseColumnNames() {
	pvt->_colcase=UPPER_CASE;
}

void sqlrcursor::lowerCaseColumnNames() {
	pvt->_colcase=LOWER_CASE;
}

void sqlrcursor::cacheToFile(const char *filename) {

	pvt->_cacheon=true;
	pvt->_cachettl=600;
	if (pvt->_copyrefs) {
		delete[] pvt->_cachedestname;
		pvt->_cachedestname=charstring::duplicate(filename);
	} else {
		pvt->_cachedestname=(char *)filename;
	}

	// create the index name
	delete[] pvt->_cachedestindname;
	size_t	cachedestindnamelen=charstring::length(filename)+5;
	pvt->_cachedestindname=new char[cachedestindnamelen];
	charstring::copy(pvt->_cachedestindname,filename);
	charstring::append(pvt->_cachedestindname,".ind");
}

void sqlrcursor::setCacheTtl(uint32_t ttl) {
	pvt->_cachettl=ttl;
}

const char *sqlrcursor::getCacheFileName() {
	return pvt->_cachedestname;
}

void sqlrcursor::cacheOff() {
	pvt->_cacheon=false;
}

void sqlrcursor::startCaching() {

	if (!pvt->_resumed) {
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Caching data to ");
			pvt->_sqlrc->debugPrint(pvt->_cachedestname);
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPreEnd();
		}
	} else {
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Resuming caching data to ");
			pvt->_sqlrc->debugPrint(pvt->_cachedestname);
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPreEnd();
		}
	}

	// create the cache file, truncate it unless we're 
	// resuming a previous session
	pvt->_cachedest=new file();
	pvt->_cachedestind=new file();
	if (!pvt->_resumed) {
		pvt->_cachedest->open(pvt->_cachedestname,
					O_RDWR|O_TRUNC|O_CREAT,
					permissions::ownerReadWrite());
		pvt->_cachedestind->open(pvt->_cachedestindname,
					O_RDWR|O_TRUNC|O_CREAT,
					permissions::ownerReadWrite());
	} else {
		pvt->_cachedest->open(pvt->_cachedestname,
					O_RDWR|O_CREAT|O_APPEND);
		pvt->_cachedestind->open(pvt->_cachedestindname,
					O_RDWR|O_CREAT|O_APPEND);
	}

	if (pvt->_cachedest && pvt->_cachedestind) {

		// calculate and set write buffer size
		// FIXME: I think rudiments bugs keep this from working...
		/*filesystem	fs;
		if (fs.initialize(pvt->_cachedestname)) {
			off64_t	optblocksize=fs.getOptimumTransferBlockSize();
			pvt->_cachedest->setWriteBufferSize(
					(optblocksize)?optblocksize:1024);
			pvt->_cachedestind->setWriteBufferSize(
					(optblocksize)?optblocksize:1024);
		}*/

		if (!pvt->_resumed) {

			// write "magic" identifier to head of files
			pvt->_cachedest->write("SQLRELAYCACHE",13);
			pvt->_cachedestind->write("SQLRELAYCACHE",13);
			
			// write ttl to files
			datetime	dt;
			dt.getSystemDateAndTime();
			int64_t	expiration=dt.getEpoch()+pvt->_cachettl;
			pvt->_cachedest->write(expiration);
			pvt->_cachedestind->write(expiration);
		}

	} else {

		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Error caching data to ");
			pvt->_sqlrc->debugPrint(pvt->_cachedestname);
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPreEnd();
		}

		// in case of an error, clean up
		clearCacheDest();
	}
}

void sqlrcursor::cacheError() {

	if (pvt->_resumed || !pvt->_cachedest) {
		return;
	}

	// write the number of returned rows, affected rows 
	// and a zero to terminate the column descriptions
	pvt->_cachedest->write((uint16_t)NO_ACTUAL_ROWS);
	pvt->_cachedest->write((uint16_t)NO_AFFECTED_ROWS);
	pvt->_cachedest->write((uint16_t)END_COLUMN_INFO);
}

void sqlrcursor::cacheNoError() {

	if (pvt->_resumed || !pvt->_cachedest) {
		return;
	}

	pvt->_cachedest->write((uint16_t)NO_ERROR_OCCURRED);
}

void sqlrcursor::cacheColumnInfo() {

	if (pvt->_resumed || !pvt->_cachedest) {
		return;
	}

	// write the number of returned rows
	pvt->_cachedest->write(pvt->_knowsactualrows);
	if (pvt->_knowsactualrows==ACTUAL_ROWS) {
		pvt->_cachedest->write(pvt->_actualrows);
	}

	// write the number of affected rows
	pvt->_cachedest->write(pvt->_knowsaffectedrows);
	if (pvt->_knowsaffectedrows==AFFECTED_ROWS) {
		pvt->_cachedest->write(pvt->_affectedrows);
	}

	// write whether or not the column info is is cached
	pvt->_cachedest->write(pvt->_sentcolumninfo);

	// write the column count
	pvt->_cachedest->write(pvt->_colcount);

	// write column descriptions to the cache file
	if (pvt->_sendcolumninfo==SEND_COLUMN_INFO && 
			pvt->_sentcolumninfo==SEND_COLUMN_INFO) {

		// write column type format
		pvt->_cachedest->write(pvt->_columntypeformat);

		// write the columns themselves
		uint16_t			namelen;
		sqlrclientcolumn		*whichcolumn;
		for (uint32_t i=0; i<pvt->_colcount; i++) {

			// get the column
			whichcolumn=getColumnInternal(i);

			// write the name
			namelen=charstring::length(whichcolumn->name);
			pvt->_cachedest->write(namelen);
			pvt->_cachedest->write(whichcolumn->name,namelen);

			// write the type
			if (pvt->_columntypeformat==COLUMN_TYPE_IDS) {
				pvt->_cachedest->write(whichcolumn->type);
			} else {
				pvt->_cachedest->write(
						whichcolumn->typestringlength);
				pvt->_cachedest->write(
						whichcolumn->typestring,
						whichcolumn->typestringlength);
			}

			// write the length, precision and scale
			pvt->_cachedest->write(whichcolumn->length);
			pvt->_cachedest->write(whichcolumn->precision);
			pvt->_cachedest->write(whichcolumn->scale);

			// write the flags
			pvt->_cachedest->write(whichcolumn->nullable);
			pvt->_cachedest->write(whichcolumn->primarykey);
			pvt->_cachedest->write(whichcolumn->unique);
			pvt->_cachedest->write(whichcolumn->partofkey);
			pvt->_cachedest->write(whichcolumn->unsignednumber);
			pvt->_cachedest->write(whichcolumn->zerofill);
			pvt->_cachedest->write(whichcolumn->binary);
			pvt->_cachedest->write(whichcolumn->autoincrement);
		}
	}
}

void sqlrcursor::cacheOutputBinds(uint32_t count) {

	if (pvt->_resumed || !pvt->_cachedest) {
		return;
	}

	// write the variable/value pairs to the cache file
	uint16_t	len;
	for (uint32_t i=0; i<count; i++) {

		pvt->_cachedest->write((uint16_t)(*pvt->_outbindvars)[i].type);

		len=charstring::length((*pvt->_outbindvars)[i].variable);
		pvt->_cachedest->write(len);
		pvt->_cachedest->write((*pvt->_outbindvars)[i].variable,len);

		len=(*pvt->_outbindvars)[i].resultvaluesize;
		pvt->_cachedest->write(len);
		if ((*pvt->_outbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_STRING ||
			(*pvt->_outbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_BLOB ||
			(*pvt->_outbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_CLOB) {
			pvt->_cachedest->write(
				(*pvt->_outbindvars)[i].value.stringval,len);
			pvt->_cachedest->write(
				(*pvt->_outbindvars)[i].value.lobval,len);
		} else if ((*pvt->_outbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_INTEGER) {
			pvt->_cachedest->write(
				(*pvt->_outbindvars)[i].value.integerval);
		} else if ((*pvt->_outbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_DOUBLE) {
			pvt->_cachedest->write(
				(*pvt->_outbindvars)[i].value.
							doubleval.value);
			pvt->_cachedest->write(
				(*pvt->_outbindvars)[i].value.
							doubleval.precision);
			pvt->_cachedest->write(
				(*pvt->_outbindvars)[i].value.
							doubleval.scale);
		}
	}

	// terminate the list of output binds
	pvt->_cachedest->write((uint16_t)END_BIND_VARS);
}

void sqlrcursor::cacheData() {

	if (!pvt->_cachedest) {
		return;
	}

	// write the data to the cache file
	uint32_t	rowbuffercount=pvt->_rowcount-pvt->_firstrowindex;
	for (uint32_t i=0; i<rowbuffercount; i++) {

		// get the current offset in the cache destination file
		int64_t	position=pvt->_cachedest->getCurrentPosition();

		// seek to the right place in the index file and write the
		// destination file offset
		pvt->_cachedestind->setPositionRelativeToBeginning(
				13+sizeof(int64_t)+
				((pvt->_firstrowindex+i)*sizeof(int64_t)));
		pvt->_cachedestind->write(position);

		// write the row to the cache file
		for (uint32_t j=0; j<pvt->_colcount; j++) {
			uint16_t	type;
			int32_t		len;
			char		*field=getFieldInternal(i,j);
			if (field) {
				type=STRING_DATA;
				len=charstring::length(field);
				pvt->_cachedest->write(type);
				pvt->_cachedest->write(len);
				if (len>0) {
					pvt->_cachedest->write(field);
				}
			} else {
				type=NULL_DATA;
				pvt->_cachedest->write(type);
			}
		}
	}

	if (pvt->_endofresultset) {
		finishCaching();
	}
}

void sqlrcursor::finishCaching() {

	if (!pvt->_cachedest) {
		return;
	}

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Finishing caching.\n");
		pvt->_sqlrc->debugPreEnd();
	}

	// terminate the result set
	pvt->_cachedest->write((uint16_t)END_RESULT_SET);
	// FIXME: I think rudiments bugs keep this from working...
	/*pvt->_cachedest->flushWriteBuffer(-1,-1);
	pvt->_cachedestind->flushWriteBuffer(-1,-1);*/

	// close the cache file and clean up
	clearCacheDest();
}

void sqlrcursor::clearCacheDest() {

	// close the cache file and clean up
	if (pvt->_cachedest) {
		pvt->_cachedest->close();
		delete pvt->_cachedest;
		pvt->_cachedest=NULL;
		pvt->_cachedestind->close();
		delete pvt->_cachedestind;
		pvt->_cachedestind=NULL;
		pvt->_cacheon=false;
	}
}

bool sqlrcursor::getDatabaseList(const char *wild) {
	return getDatabaseList(wild,SQLRCLIENTLISTFORMAT_MYSQL);
}

bool sqlrcursor::getDatabaseList(const char *wild,
					sqlrclientlistformat_t listformat) {
	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("getting database list");
		if (wild) {
			pvt->_sqlrc->debugPrint("\"");
			pvt->_sqlrc->debugPrint(wild);
			pvt->_sqlrc->debugPrint("\"");
		}
		pvt->_sqlrc->debugPrint("\n");
		pvt->_sqlrc->debugPreEnd();
	}
	return getList(GETDBLIST,listformat,NULL,wild);
}

bool sqlrcursor::getTableList(const char *wild) {
	return getTableList(wild,SQLRCLIENTLISTFORMAT_MYSQL);
}

bool sqlrcursor::getTableList(const char *wild,
					sqlrclientlistformat_t listformat) {
	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("getting table list");
		if (wild) {
			pvt->_sqlrc->debugPrint("\"");
			pvt->_sqlrc->debugPrint(wild);
			pvt->_sqlrc->debugPrint("\"");
		}
		pvt->_sqlrc->debugPrint("\n");
		pvt->_sqlrc->debugPreEnd();
	}
	return getList(GETTABLELIST,listformat,NULL,wild);
}

bool sqlrcursor::getColumnList(const char *table, const char *wild) {
	return getColumnList(table,wild,SQLRCLIENTLISTFORMAT_MYSQL);
}

bool sqlrcursor::getColumnList(const char *table,
				const char *wild,
				sqlrclientlistformat_t listformat) {
	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("getting column list for: \"");
		pvt->_sqlrc->debugPrint(table);
		pvt->_sqlrc->debugPrint("\"");
		if (wild) {
			pvt->_sqlrc->debugPrint(" - \"");
			pvt->_sqlrc->debugPrint(wild);
			pvt->_sqlrc->debugPrint("\"");
		}
		pvt->_sqlrc->debugPrint("\n");
		pvt->_sqlrc->debugPreEnd();
	}
	return getList(GETCOLUMNLIST,listformat,(table)?table:"",wild);
}

bool sqlrcursor::getList(uint16_t command, sqlrclientlistformat_t listformat,
					const char *table, const char *wild) {

	pvt->_reexecute=false;
	pvt->_validatebinds=false;
	pvt->_resumed=false;
	clearVariables();

	if (!pvt->_endofresultset) {
		closeResultSet(false);
	}
	clearResultSet();

	if (!pvt->_sqlrc->openSession()) {
		return false;
	}

	pvt->_cached=false;
	pvt->_endofresultset=false;

	// tell the server we want to get a db list
	pvt->_sqlrc->cs()->write(command);

	// tell the server whether we'll need a cursor or not
	sendCursorStatus();

	// send the list format
	pvt->_sqlrc->cs()->write((uint16_t)listformat);

	// send the wild parameter
	uint32_t	len=charstring::length(wild);
	pvt->_sqlrc->cs()->write(len);
	if (len) {
		pvt->_sqlrc->cs()->write(wild,len);
	}

	// send the table parameter
	if (table) {
		len=charstring::length(table);
		pvt->_sqlrc->cs()->write(len);
		if (len) {
			pvt->_sqlrc->cs()->write(table,len);
		}
	}

	pvt->_sqlrc->flushWriteBuffer();

	// process the result set
	bool	retval=true;
	if (pvt->_rsbuffersize) {
		if (!processResultSet(false,pvt->_rsbuffersize-1)) {
			retval=false;
		}
	} else {
		if (!processResultSet(true,0)) {
			retval=false;
		}
	}

	// set up not to re-execute the same query if executeQuery is called
	// again before calling prepareQuery on a new query
	pvt->_reexecute=false;

	return retval;
}

bool sqlrcursor::sendQuery(const char *query) {
	prepareQuery(query);
	return executeQuery();
}

bool sqlrcursor::sendQuery(const char *query, uint32_t length) {
	prepareQuery(query,length);
	return executeQuery();
}

bool sqlrcursor::sendFileQuery(const char *path, const char *filename) {
	return prepareFileQuery(path,filename) && executeQuery();
}

void sqlrcursor::prepareQuery(const char *query) {
	prepareQuery(query,charstring::length(query));
}

void sqlrcursor::prepareQuery(const char *query, uint32_t length) {
	pvt->_reexecute=false;
	pvt->_validatebinds=false;
	pvt->_resumed=false;
	clearVariables();
	pvt->_querylen=length;
	if (pvt->_copyrefs) {
		initQueryBuffer(pvt->_querylen);
		charstring::copy(pvt->_querybuffer,query,pvt->_querylen);
		pvt->_querybuffer[pvt->_querylen]='\0';
	} else {
		pvt->_queryptr=query;
	}
}

bool sqlrcursor::prepareFileQuery(const char *path, const char *filename) {

	// init some variables
	pvt->_reexecute=false;
	pvt->_validatebinds=false;
	pvt->_resumed=false;
	clearVariables();

	// init the fullpath buffer
	if (!pvt->_fullpath) {
		pvt->_fullpath=new char[MAXPATHLEN+1];
	}

	// add the path to the fullpath
	uint32_t	index=0;
	uint32_t	counter=0;
	if (path) {
		while (path[index] && counter<MAXPATHLEN) {
			pvt->_fullpath[counter]=path[index];
			index++;
			counter++;
		}

		// add the "/" to the fullpath
		if (counter<=MAXPATHLEN) {
			pvt->_fullpath[counter]='/';
			counter++;
		}
	}

	// add the file to the fullpath
	index=0;
	while (filename[index] && counter<MAXPATHLEN) {
		pvt->_fullpath[counter]=filename[index];
		index++;
		counter++;
	}

	// handle a filename that's too long
	if (counter>MAXPATHLEN) {

		// sabotage the file name so it can't be opened
		pvt->_fullpath[0]='\0';

		// debug info
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("File name ");
			if (path) {
				pvt->_sqlrc->debugPrint((char *)path);
				pvt->_sqlrc->debugPrint("/");
			}
			pvt->_sqlrc->debugPrint((char *)filename);
			pvt->_sqlrc->debugPrint(" is too long.");
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPreEnd();
		}

	} else {

		// terminate the string
		pvt->_fullpath[counter]='\0';

		// debug info
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("File: ");
			pvt->_sqlrc->debugPrint(pvt->_fullpath);
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPreEnd();
		}
	}

	// open the file
	file	queryfile;
	if (!queryfile.open(pvt->_fullpath,O_RDONLY)) {

		// set the error
		char	*err=new char[32+charstring::length(pvt->_fullpath)];
		charstring::append(err,"The file ");
		charstring::append(err,pvt->_fullpath);
		charstring::append(err," could not be opened.\n");
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint(err);
			pvt->_sqlrc->debugPreEnd();
		}
		setError(err);

		// set queryptr to NULL so executeQuery won't try to do
		// anything with it in the event that it gets called
		pvt->_queryptr=NULL;

		delete[] err;

		return false;
	}

	initQueryBuffer(queryfile.getSize());

	// read the file into the query buffer
	pvt->_querylen=queryfile.getSize();
	queryfile.read((unsigned char *)pvt->_querybuffer,pvt->_querylen);
	pvt->_querybuffer[pvt->_querylen]='\0';

	queryfile.close();

	return true;
}

void sqlrcursor::initQueryBuffer(uint32_t querylength) {
	delete[] pvt->_querybuffer;
	pvt->_querybuffer=new char[querylength+1];
	pvt->_queryptr=pvt->_querybuffer;
}

void sqlrcursor::attachToBindCursor(uint16_t bindcursorid) {
	prepareQuery("");
	pvt->_reexecute=true;
	pvt->_cursorid=bindcursorid;
}

uint16_t sqlrcursor::countBindVariables() const {

	if (!pvt->_queryptr) {
		return 0;
	}

	char	lastchar='\0';
	bool	inquotes=false;

	uint16_t	questionmarkcount=0;
	uint16_t	coloncount=0;
	uint16_t	atsigncount=0;
	uint16_t	dollarsigncount=0;

	for (const char *ptr=pvt->_queryptr; *ptr; ptr++) {

		if (*ptr=='\'' && lastchar!='\\') {
			if (inquotes) {
				inquotes=false;
			} else {
				inquotes=true;
			}
		}

		// If we're not inside of a quoted string and we run into
		// a ?, : (for oracle-style binds), @ (for sap/sybase-style
		// binds) or $ (for postgresql-style binds) and the previous
		// character was something that might come before a bind
		// variable then we must have found a bind variable.
		// count ?, :, @, $ separately
		if (!inquotes &&
			character::inSet(lastchar," \t\n\r=<>,(+-*/%|&!~^")) {
			if (*ptr=='?') {
				questionmarkcount++;
			} else if (*ptr==':') {
				coloncount++;
			} else if (*ptr=='@') {
				atsigncount++;
			} else if (*ptr=='$') {
				dollarsigncount++;
			}
		}

		lastchar=*ptr;
	}

	// if we got $'s or ?'s, ignore the :'s or @'s
	if (dollarsigncount) {
		return dollarsigncount;
	}
	if (questionmarkcount) {
		return questionmarkcount;
	}
	if (coloncount) {
		return coloncount;
	}
	if (atsigncount) {
		return atsigncount;
	}
	return 0;
}

void sqlrcursor::clearVariables() {

	deleteSubstitutionVariables();
	pvt->_subvars->clear();
	pvt->_dirtysubs=false;
	pvt->_dirtybinds=false;
	clearBinds();
}

void sqlrcursor::deleteVariables() {
	deleteSubstitutionVariables();
	deleteInputBindVariables();
	deleteOutputBindVariables();
}

void sqlrcursor::deleteSubstitutionVariables() {

	if (pvt->_copyrefs) {
		for (uint64_t i=0; i<pvt->_subvars->getLength(); i++) {
			delete[] (*pvt->_subvars)[i].variable;
			if ((*pvt->_subvars)[i].type==
					SQLRCLIENTBINDVARTYPE_STRING) {
				delete[] (*pvt->_subvars)[i].value.stringval;
			}
			if ((*pvt->_subvars)[i].type==
					SQLRCLIENTBINDVARTYPE_DATE) {
				delete[] (*pvt->_subvars)[i].value.dateval.tz;
			}
		}
	}
}

void sqlrcursor::deleteInputBindVariables() {

	if (pvt->_copyrefs) {
		for (uint64_t i=0; i<pvt->_inbindvars->getLength(); i++) {
			delete[] (*pvt->_inbindvars)[i].variable;
			if ((*pvt->_inbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_STRING) {
				delete[] (*pvt->_inbindvars)[i].value.stringval;
			}
			if ((*pvt->_inbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_BLOB ||
				(*pvt->_inbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_CLOB) {
				delete[] (*pvt->_inbindvars)[i].value.lobval;
			}
			if ((*pvt->_inbindvars)[i].type==
						SQLRCLIENTBINDVARTYPE_DATE) {
				delete[] (*pvt->_inbindvars)[i].
							value.dateval.tz;
			}
		}
	}
}

void sqlrcursor::deleteOutputBindVariables() {

	for (uint64_t i=0; i<pvt->_outbindvars->getLength(); i++) {
		if (pvt->_copyrefs) {
			delete[] (*pvt->_outbindvars)[i].variable;
		}
		if ((*pvt->_outbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_STRING) {
			delete[] (*pvt->_outbindvars)[i].value.stringval;
		}
		if ((*pvt->_outbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_BLOB ||
			(*pvt->_outbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_CLOB) {
			delete[] (*pvt->_outbindvars)[i].value.lobval;
		}
		if ((*pvt->_outbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_DATE) {
			delete[] (*pvt->_outbindvars)[i].value.dateval.tz;
		}
	}
}

void sqlrcursor::substitution(const char *variable, const char *value) {
	if (charstring::isNullOrEmpty(variable)) {
		return;
	}
	bool			preexisting=true;
	sqlrclientbindvar	*bv=pvt->findVar(variable,pvt->_subvars);
	if (!bv) {
		bv=&(*pvt->_subvars)[pvt->_subvars->getLength()];
		preexisting=false;
	}
	initVar(bv,variable,preexisting);
	stringVar(bv,variable,value);
	pvt->_dirtysubs=true;
}

void sqlrcursor::substitution(const char *variable, int64_t value) {
	if (charstring::isNullOrEmpty(variable)) {
		return;
	}
	bool			preexisting=true;
	sqlrclientbindvar	*bv=pvt->findVar(variable,pvt->_subvars);
	if (!bv) {
		bv=&(*pvt->_subvars)[pvt->_subvars->getLength()];
		preexisting=false;
	}
	initVar(bv,variable,preexisting);
	integerVar(bv,variable,value);
	pvt->_dirtysubs=true;
}

void sqlrcursor::substitution(const char *variable, double value, 
				uint32_t precision, uint32_t scale) {
	if (charstring::isNullOrEmpty(variable)) {
		return;
	}
	bool			preexisting=true;
	sqlrclientbindvar	*bv=pvt->findVar(variable,pvt->_subvars);
	if (!bv) {
		bv=&(*pvt->_subvars)[pvt->_subvars->getLength()];
		preexisting=false;
	}
	initVar(bv,variable,preexisting);
	doubleVar(bv,variable,value,precision,scale);
	pvt->_dirtysubs=true;
}

void sqlrcursor::clearBinds() {

	deleteInputBindVariables();
	pvt->_inbindvars->clear();

	deleteOutputBindVariables();
	pvt->_outbindvars->clear();
}

void sqlrcursor::inputBindBlob(const char *variable, const char *value,
							uint32_t size) {
	if (charstring::isNullOrEmpty(variable)) {
		return;
	}
	bool			preexisting=true;
	sqlrclientbindvar	*bv=pvt->findVar(variable,pvt->_inbindvars);
	if (!bv) {
		bv=&(*pvt->_inbindvars)[pvt->_inbindvars->getLength()];
		preexisting=false;
	}
	initVar(bv,variable,preexisting);
	lobVar(bv,variable,value,size,SQLRCLIENTBINDVARTYPE_BLOB);
	bv->send=true;
	pvt->_dirtybinds=true;
}

void sqlrcursor::inputBindClob(const char *variable, const char *value,
							uint32_t size) {
	if (charstring::isNullOrEmpty(variable)) {
		return;
	}
	bool			preexisting=true;
	sqlrclientbindvar	*bv=pvt->findVar(variable,pvt->_inbindvars);
	if (!bv) {
		bv=&(*pvt->_inbindvars)[pvt->_inbindvars->getLength()];
		preexisting=false;
	}
	initVar(bv,variable,preexisting);
	lobVar(bv,variable,value,size,SQLRCLIENTBINDVARTYPE_CLOB);
	bv->send=true;
	pvt->_dirtybinds=true;
}

void sqlrcursor::inputBind(const char *variable, const char *value) {
	if (charstring::isNullOrEmpty(variable)) {
		return;
	}
	bool			preexisting=true;
	sqlrclientbindvar	*bv=pvt->findVar(variable,pvt->_inbindvars);
	if (!bv) {
		bv=&(*pvt->_inbindvars)[pvt->_inbindvars->getLength()];
		preexisting=false;
	}
	initVar(bv,variable,preexisting);
	stringVar(bv,variable,value);
	bv->send=true;
	pvt->_dirtybinds=true;
}

void sqlrcursor::inputBind(const char *variable, const char *value,
						uint32_t valuesize) {
	if (charstring::isNullOrEmpty(variable)) {
		return;
	}
	bool			preexisting=true;
	sqlrclientbindvar	*bv=pvt->findVar(variable,pvt->_inbindvars);
	if (!bv) {
		bv=&(*pvt->_inbindvars)[pvt->_inbindvars->getLength()];
		preexisting=false;
	}
	initVar(bv,variable,preexisting);
	stringVar(bv,variable,value,valuesize);
	bv->send=true;
	pvt->_dirtybinds=true;
}

void sqlrcursor::inputBind(const char *variable, int64_t value) {
	if (charstring::isNullOrEmpty(variable)) {
		return;
	}
	bool			preexisting=true;
	sqlrclientbindvar	*bv=pvt->findVar(variable,pvt->_inbindvars);
	if (!bv) {
		bv=&(*pvt->_inbindvars)[pvt->_inbindvars->getLength()];
		preexisting=false;
	}
	initVar(bv,variable,preexisting);
	integerVar(bv,variable,value);
	bv->send=true;
	pvt->_dirtybinds=true;
}

void sqlrcursor::inputBind(const char *variable, double value, 
				uint32_t precision, uint32_t scale) {
	if (charstring::isNullOrEmpty(variable)) {
		return;
	}
	bool			preexisting=true;
	sqlrclientbindvar	*bv=pvt->findVar(variable,pvt->_inbindvars);
	if (!bv) {
		bv=&(*pvt->_inbindvars)[pvt->_inbindvars->getLength()];
		preexisting=false;
	}
	initVar(bv,variable,preexisting);
	doubleVar(bv,variable,value,precision,scale);
	bv->send=true;
	pvt->_dirtybinds=true;
}

void sqlrcursor::inputBind(const char *variable,
				int16_t year, int16_t month, int16_t day,
				int16_t hour, int16_t minute, int16_t second,
				int32_t microsecond, const char *tz) {
	if (charstring::isNullOrEmpty(variable)) {
		return;
	}
	bool			preexisting=true;
	sqlrclientbindvar	*bv=pvt->findVar(variable,pvt->_inbindvars);
	if (!bv) {
		bv=&(*pvt->_inbindvars)[pvt->_inbindvars->getLength()];
		preexisting=false;
	}
	initVar(bv,variable,preexisting);
	dateVar(bv,variable,year,month,day,hour,minute,second,microsecond,tz);
	bv->send=true;
	pvt->_dirtybinds=true;
}

void sqlrcursor::substitutions(const char **variables, const char **values) {
	for (uint16_t i=0; variables[i]; i++) {
		substitution(variables[i],values[i]);
	}
}

void sqlrcursor::substitutions(const char **variables, const int64_t *values) {
	for (uint16_t i=0; variables[i]; i++) {
		substitution(variables[i],values[i]);
	}
}

void sqlrcursor::substitutions(const char **variables, const double *values, 
					const uint32_t *precisions,
					const uint32_t *scales) {
	for (uint16_t i=0; variables[i]; i++) {
		substitution(variables[i],values[i],precisions[i],scales[i]);
	}
}

void sqlrcursor::inputBinds(const char **variables, const char **values) {
	for (uint16_t i=0; variables[i]; i++) {
		inputBind(variables[i],values[i]);
	}
}

void sqlrcursor::inputBinds(const char **variables, const int64_t *values) {
	for (uint16_t i=0; variables[i]; i++) {
		inputBind(variables[i],values[i]);
	}
}

void sqlrcursor::inputBinds(const char **variables, const double *values, 
					const uint32_t *precisions,
					const uint32_t *scales) {
	for (uint16_t i=0; variables[i]; i++) {
		inputBind(variables[i],values[i],
				precisions[i],scales[i]);
	}
}

void sqlrcursor::stringVar(sqlrclientbindvar *var,
					const char *variable,
					const char *value) {
	stringVar(var,variable,value,charstring::length(value));
}

void sqlrcursor::stringVar(sqlrclientbindvar *var,
					const char *variable,
					const char *value,
					uint32_t valuesize) {

	// store the value, handle NULL values too
	if (value) {
		if (pvt->_copyrefs) {
			var->value.stringval=charstring::duplicate(value);
		} else {
			var->value.stringval=(char *)value;
		}
		var->valuesize=valuesize;
		var->type=SQLRCLIENTBINDVARTYPE_STRING;
	} else {
		var->type=SQLRCLIENTBINDVARTYPE_NULL;
	}
}

void sqlrcursor::integerVar(sqlrclientbindvar *var,
					const char *variable,
					int64_t value) {
	var->type=SQLRCLIENTBINDVARTYPE_INTEGER;
	var->value.integerval=value;
}

void sqlrcursor::doubleVar(sqlrclientbindvar *var,
					const char *variable,
					double value,
					uint32_t precision,
					uint32_t scale) {
	var->type=SQLRCLIENTBINDVARTYPE_DOUBLE;
	var->value.doubleval.value=value;
	var->value.doubleval.precision=precision;
	var->value.doubleval.scale=scale;
}

void sqlrcursor::dateVar(sqlrclientbindvar *var,
					const char *variable,
					int16_t year,
					int16_t month,
					int16_t day,
					int16_t hour,
					int16_t minute,
					int16_t second,
					int32_t microsecond,
					const char *tz) {
	var->type=SQLRCLIENTBINDVARTYPE_DATE;
	var->value.dateval.year=year;
	var->value.dateval.month=month;
	var->value.dateval.day=day;
	var->value.dateval.hour=hour;
	var->value.dateval.minute=minute;
	var->value.dateval.second=second;
	var->value.dateval.microsecond=microsecond;
	if (pvt->_copyrefs) {
		var->value.dateval.tz=charstring::duplicate(tz);
	} else {
		var->value.dateval.tz=(char *)tz;
	}
}

void sqlrcursor::lobVar(sqlrclientbindvar *var,
					const char *variable,
					const char *value,
					uint32_t size,
					sqlrclientbindvartype_t type) {

	// Store the value, handle NULL values too.
	// For LOB's empty strings are handled as NULL's as well, this is
	// probably not right, but I can't get empty string lob binds to work.
	if (value && size>0) {
		if (pvt->_copyrefs) {
			var->value.lobval=new char[size];
			bytestring::copy(var->value.lobval,value,size);
		} else {
			var->value.lobval=(char *)value;
		}
		var->valuesize=size;
		var->type=type;
	} else {
		var->type=SQLRCLIENTBINDVARTYPE_NULL;
	}
}

void sqlrcursor::initVar(sqlrclientbindvar *var,
				const char *variable,
				bool preexisting) {

	// clear any old variable name that was stored and assign the new 
	// variable name also clear any old value that was stored in this 
	// variable
	if (pvt->_copyrefs) {
		if (preexisting) {
			delete[] var->variable;
			if (var->type==SQLRCLIENTBINDVARTYPE_STRING) {
				delete[] var->value.stringval;
			} else if (var->type==SQLRCLIENTBINDVARTYPE_BLOB ||
					var->type==SQLRCLIENTBINDVARTYPE_CLOB) {
				delete[] var->value.lobval;
			}
		}
		var->variable=charstring::duplicate(variable);
	} else {
		var->variable=(char *)variable;
	}

	var->substituted=false;
	var->donesubstituting=false;
}

void sqlrcursor::defineOutputBindString(const char *variable,
						uint32_t length) {
	defineOutputBindGeneric(variable,
				SQLRCLIENTBINDVARTYPE_STRING,length);
}

void sqlrcursor::defineOutputBindInteger(const char *variable) {
	defineOutputBindGeneric(variable,
				SQLRCLIENTBINDVARTYPE_INTEGER,sizeof(int64_t));
}

void sqlrcursor::defineOutputBindDouble(const char *variable) {
	defineOutputBindGeneric(variable,
				SQLRCLIENTBINDVARTYPE_DOUBLE,sizeof(double));
}

void sqlrcursor::defineOutputBindDate(const char *variable) {
	defineOutputBindGeneric(variable,
				SQLRCLIENTBINDVARTYPE_DATE,sizeof(double));
}

void sqlrcursor::defineOutputBindBlob(const char *variable) {
	defineOutputBindGeneric(variable,
				SQLRCLIENTBINDVARTYPE_BLOB,0);
}

void sqlrcursor::defineOutputBindClob(const char *variable) {
	defineOutputBindGeneric(variable,
				SQLRCLIENTBINDVARTYPE_CLOB,0);
}

void sqlrcursor::defineOutputBindCursor(const char *variable) {
	defineOutputBindGeneric(variable,
				SQLRCLIENTBINDVARTYPE_CURSOR,0);
}

void sqlrcursor::defineOutputBindGeneric(const char *variable,
						sqlrclientbindvartype_t type,
						uint32_t valuesize) {

	if (charstring::isNullOrEmpty(variable)) {
		return;
	}

	bool			preexisting=true;
	sqlrclientbindvar	*bv=pvt->findVar(variable,pvt->_outbindvars);
	if (!bv) {
		bv=&(*pvt->_outbindvars)[pvt->_outbindvars->getLength()];
		preexisting=false;
		pvt->_dirtybinds=true;
	}

	// clean up old values and set new values
	if (preexisting) {
		if (bv->type==SQLRCLIENTBINDVARTYPE_STRING) {
			delete[] bv->value.stringval;
		} else if (bv->type==SQLRCLIENTBINDVARTYPE_BLOB ||
				bv->type==SQLRCLIENTBINDVARTYPE_CLOB) {
			delete[] bv->value.lobval;
		}
	}
	if (pvt->_copyrefs) {
		if (preexisting) {
			delete[] bv->variable;
		}
		bv->variable=charstring::duplicate(variable);
	} else {
		bv->variable=(char *)variable;
	}
	bv->type=type;
	if (bv->type==SQLRCLIENTBINDVARTYPE_STRING) {
		bv->value.stringval=NULL;
	} else if (bv->type==SQLRCLIENTBINDVARTYPE_BLOB ||
				bv->type==SQLRCLIENTBINDVARTYPE_CLOB) {
		bv->value.lobval=NULL;
	}
	bv->valuesize=valuesize;
	bv->resultvaluesize=0;
	bv->send=true;
}

const char *sqlrcursor::getOutputBindString(const char *variable) {

	if (variable) {
		for (uint64_t i=0; i<pvt->_outbindvars->getLength(); i++) {
			if (!charstring::compare(
				(*pvt->_outbindvars)[i].variable,variable) &&
				(*pvt->_outbindvars)[i].type==
						SQLRCLIENTBINDVARTYPE_STRING) {
				return (*pvt->_outbindvars)[i].value.stringval;
			}
		}
	}
	return NULL;
}

uint32_t sqlrcursor::getOutputBindLength(const char *variable) {

	if (variable) {
		for (uint64_t i=0; i<pvt->_outbindvars->getLength(); i++) {
			if (!charstring::compare(
				(*pvt->_outbindvars)[i].variable,variable)) {
				return (*pvt->_outbindvars)[i].resultvaluesize;
			}
		}
	}
	return 0;
}

const char *sqlrcursor::getOutputBindBlob(const char *variable) {

	if (variable) {
		for (uint64_t i=0; i<pvt->_outbindvars->getLength(); i++) {
			if (!charstring::compare(
				(*pvt->_outbindvars)[i].variable,variable) &&
				(*pvt->_outbindvars)[i].type==
						SQLRCLIENTBINDVARTYPE_BLOB) {
				return (*pvt->_outbindvars)[i].value.lobval;
			}
		}
	}
	return NULL;
}

const char *sqlrcursor::getOutputBindClob(const char *variable) {

	if (variable) {
		for (uint64_t i=0; i<pvt->_outbindvars->getLength(); i++) {
			if (!charstring::compare(
				(*pvt->_outbindvars)[i].variable,variable) &&
				(*pvt->_outbindvars)[i].type==
						SQLRCLIENTBINDVARTYPE_CLOB) {
				return (*pvt->_outbindvars)[i].value.lobval;
			}
		}
	}
	return NULL;
}

int64_t sqlrcursor::getOutputBindInteger(const char *variable) {

	if (variable) {
		for (uint64_t i=0; i<pvt->_outbindvars->getLength(); i++) {
			if (!charstring::compare(
				(*pvt->_outbindvars)[i].variable,variable) &&
				(*pvt->_outbindvars)[i].type==
						SQLRCLIENTBINDVARTYPE_INTEGER) {
				return (*pvt->_outbindvars)[i].value.integerval;
			}
		}
	}
	return -1;
}

double sqlrcursor::getOutputBindDouble(const char *variable) {

	if (variable) {
		for (uint64_t i=0; i<pvt->_outbindvars->getLength(); i++) {
			if (!charstring::compare(
				(*pvt->_outbindvars)[i].variable,variable) &&
				(*pvt->_outbindvars)[i].type==
						SQLRCLIENTBINDVARTYPE_DOUBLE) {
				return (*pvt->_outbindvars)[i].
							value.doubleval.value;
			}
		}
	}
	return -1.0;
}

bool sqlrcursor::getOutputBindDate(const char *variable,
			int16_t *year, int16_t *month, int16_t *day,
			int16_t *hour, int16_t *minute, int16_t *second,
			int32_t *microsecond, const char **tz) {

	if (variable) {
		for (uint64_t i=0; i<pvt->_outbindvars->getLength(); i++) {
			if (!charstring::compare(
				(*pvt->_outbindvars)[i].variable,variable) &&
				(*pvt->_outbindvars)[i].type==
						SQLRCLIENTBINDVARTYPE_DATE) {
				*year=(*pvt->_outbindvars)[i].
						value.dateval.year;
				*month=(*pvt->_outbindvars)[i].
						value.dateval.month;
				*day=(*pvt->_outbindvars)[i].
						value.dateval.day;
				*hour=(*pvt->_outbindvars)[i].
						value.dateval.hour;
				*minute=(*pvt->_outbindvars)[i].
						value.dateval.minute;
				*second=(*pvt->_outbindvars)[i].
						value.dateval.second;
				*microsecond=(*pvt->_outbindvars)[i].
						value.dateval.microsecond;
				*tz=(*pvt->_outbindvars)[i].
						value.dateval.tz;
				return true;
			}
		}
	}
	return false;
}

sqlrcursor *sqlrcursor::getOutputBindCursor(const char *variable) {
	return getOutputBindCursor(variable,false);
}

sqlrcursor *sqlrcursor::getOutputBindCursor(const char *variable,
							bool copyrefs) {

	if (!outputBindCursorIdIsValid(variable)) {
		return NULL;
	}
	uint16_t	bindcursorid=getOutputBindCursorId(variable);
	sqlrcursor	*bindcursor=new sqlrcursor(pvt->_sqlrc,copyrefs);
	bindcursor->attachToBindCursor(bindcursorid);
	return bindcursor;
}

bool sqlrcursor::outputBindCursorIdIsValid(const char *variable) {
	if (variable) {
		for (uint64_t i=0; i<pvt->_outbindvars->getLength(); i++) {
			if (!charstring::compare(
				(*pvt->_outbindvars)[i].variable,variable)) {
				return true;
			}
		}
	}
	return false;
}

uint16_t sqlrcursor::getOutputBindCursorId(const char *variable) {

	if (variable) {
		for (uint64_t i=0; i<pvt->_outbindvars->getLength(); i++) {
			if (!charstring::compare(
				(*pvt->_outbindvars)[i].variable,variable)) {
				return (*pvt->_outbindvars)[i].value.cursorid;
			}
		}
	}
	return 0;
}

void sqlrcursor::validateBinds() {
	pvt->_validatebinds=true;
}

bool sqlrcursor::validBind(const char *variable) {
	performSubstitutions();
	validateBindsInternal();
	for (uint64_t in=0; in<pvt->_inbindvars->getLength(); in++) {
		if (!charstring::compare(
			(*pvt->_inbindvars)[in].variable,variable)) {
			return (*pvt->_inbindvars)[in].send;
		}
	}
	for (uint64_t out=0; out<pvt->_outbindvars->getLength(); out++) {
		if (!charstring::compare(
			(*pvt->_outbindvars)[out].variable,variable)) {
			return (*pvt->_outbindvars)[out].send;
		}
	}
	return false;
}

bool sqlrcursor::executeQuery() {

	if (!pvt->_queryptr) {
		setError("No query to execute.");
		return false;
	}

	performSubstitutions();

	// validate the bind variables
	if (pvt->_validatebinds) {
		validateBindsInternal();
	}
		
	// run the query
	bool	retval=runQuery(pvt->_queryptr);

	// set up to re-execute the same query if executeQuery is called
	// again before calling prepareQuery
	pvt->_reexecute=true;

	return retval;
}

void sqlrcursor::performSubstitutions() {

	if (!pvt->_subvars->getLength() || !pvt->_dirtysubs) {
		return;
	}

	// perform substitutions
	stringbuffer	container;
	const char	*ptr=pvt->_queryptr;
	bool		found=false;
	bool		inquotes=false;
	bool		inbraces=false;
	int		len=0;
	stringbuffer	*braces=NULL;

	// iterate through the string
	while (*ptr) {
	
		// figure out whether we're inside a quoted 
		// string or not
		if (*ptr=='\'' && *(ptr-1)!='\\') {
			if (inquotes) {
				inquotes=false;
			} else {
				inquotes=true;
			}
		}
	
		// if we find an open-brace then start 
		// sending to a new buffer
		if (*ptr=='[' && !inbraces && !inquotes) {
			braces=new stringbuffer();
			inbraces=true;
			ptr++;
		}
	
		// if we find a close-brace then process 
		// the brace buffer
		if (*ptr==']' && inbraces && !inquotes) {
	
			// look for an = sign, skipping whitespace
			const char	*bptr=braces->getString();
			while (*bptr && (*bptr==' ' || 
				*bptr=='	' || *bptr=='\n')) {
				bptr++;
			}
	
			if (*bptr=='=') {
				// if we find an equals sign first 
				// then process the rest of the buffer
				bptr++;
	
				// skip whitespace
				while (*bptr && (*bptr==' ' || 
					*bptr=='	' || 
				 	*bptr=='\n')) {
					bptr++;
				}
	
				// if the remaining contents of the 
				// buffer are '' or nothing then we 
				// must have an ='' or just an = with 
				// some whitespace, replace this
				// with "is NULL" otherwise, just write
				// out the contents of the buffer
				if (!bptr || 
					(bptr &&
					!charstring::compare(bptr,
							"''"))) {
					container.append(" is NULL ");
				} else {
					container.append(
						braces->getString());
				}
			} else {
				// if we don't find an equals sign, 
				// then write the contents out directly
				container.append(braces->getString());
			}
			delete braces;
			inbraces=false;
			ptr++;
		}
	
		// if we encounter $(....) then replace the 
		// variable within
		if ((*ptr)=='$' && (*(ptr+1))=='(') {
	
			// first iterate through the arrays passed in
			found=false;
			for (uint64_t i=0;
				i<pvt->_subvars->getLength() && !found; i++) {

	
				// if we find a match, write the 
				// value to the container and skip 
				// past the $(variable)
				len=charstring::length(
						(*pvt->_subvars)[i].variable);
				if (!(*pvt->_subvars)[i].donesubstituting &&
					!charstring::compare((ptr+2),
						(*pvt->_subvars)[i].
							variable,len) &&
						(*(ptr+2+len))==')') {
	
					if (inbraces) {
						performSubstitution(
							braces,i);
					} else {
						performSubstitution(
							&container,i);
					}
					ptr=ptr+3+len;
					found=true;
				}
			}
	
			// if the variable wasn't found, then 
			// just write the $(
			if (!found) {
				if (inbraces) {
					braces->append("$(");
				} else {
					container.append("$(");
				}
				ptr=ptr+2;
			}
	
		} else {
	
			// print out the current character and proceed
			if (inbraces) {
				braces->append(*ptr);
			} else {
				container.append(*ptr);
			}
			ptr++;
		}
	}

	// mark all vars that were substituted in as "done" so the next time
	// this method gets called, they won't be processed.
	for (uint64_t i=0; i<pvt->_subvars->getLength(); i++) {
		(*pvt->_subvars)[i].donesubstituting=
					(*pvt->_subvars)[i].substituted;
	}

	delete[] pvt->_querybuffer;
	pvt->_querylen=container.getStringLength();
	pvt->_querybuffer=container.detachString();
	pvt->_queryptr=pvt->_querybuffer;

	pvt->_dirtysubs=false;
}

void sqlrcursor::validateBindsInternal() {

	if (!pvt->_dirtybinds) {
		return;
	}

	// some useful variables
	const char	*ptr;
	const char	*start;
	const char	*after;
	bool		found;
	int		len;

	// check each input bind
	for (uint64_t in=0; in<pvt->_inbindvars->getLength(); in++) {

		// don't check bind-by-position variables
		len=charstring::length(
				(*pvt->_inbindvars)[in].variable);
		if (charstring::isInteger(
				(*pvt->_inbindvars)[in].variable,len)) {
			continue;
		}

		found=false;
		start=pvt->_queryptr+1;

		// there may be more than 1 match for the variable name as in
		// "select * from table where table_name=:table_name", both
		// table_name's would match, but only the second is a bind
		// variable
		while ((ptr=charstring::findFirst(start,
					(*pvt->_inbindvars)[in].variable))) {

			// for a match to be a bind variable, it must be 
			// preceded by a colon or at-sign and can't be followed
			// by an alphabet character, number or underscore
			after=ptr+len;
			if ((*(ptr-1)==':' || *(ptr-1)=='@') && *after!='_' &&
				!(*(after)>='a' && *(after)<='z') &&
				!(*(after)>='A' && *(after)<='Z') &&
				!(*(after)>='0' && *(after)<='9')) {
				found=true;
				break;
			} else {
				// jump past this instance to look for the
				// next one
				start=ptr+len;
			}
		}

		(*pvt->_inbindvars)[in].send=found;
	}

	// check each output bind
	for (uint64_t out=0; out<pvt->_outbindvars->getLength(); out++) {

		// don't check bind-by-position variables
		len=charstring::length(
				(*pvt->_outbindvars)[out].variable);
		if (charstring::isInteger(
				(*pvt->_outbindvars)[out].variable,len)) {
			continue;
		}

		found=false;
		start=pvt->_queryptr+1;

		// there may be more than 1 match for the variable name as in
		// "select * from table where table_name=:table_name", both
		// table_name's would match, but only 1 is correct
		while ((ptr=charstring::findFirst(start,
					(*pvt->_outbindvars)[out].variable))) {

			// for a match to be a bind variable, it must be 
			// preceded by a colon and can't be followed by an
			// alphabet character, number or underscore
			after=ptr+len;
			if (*(ptr-1)==':' && *after!='_' &&
				!(*(after)>='a' && *(after)<='z') &&
				!(*(after)>='A' && *(after)<='Z') &&
				!(*(after)>='0' && *(after)<='9')) {
				found=true;
				break;
			} else {
				// jump past this instance to look for the
				// next one
				start=ptr+len;
			}
		}

		(*pvt->_outbindvars)[out].send=found;
	}
}

void sqlrcursor::performSubstitution(stringbuffer *buffer, uint16_t which) {

	if ((*pvt->_subvars)[which].type==
				SQLRCLIENTBINDVARTYPE_STRING) {
		buffer->append((*pvt->_subvars)[which].value.stringval);
	} else if ((*pvt->_subvars)[which].type==
				SQLRCLIENTBINDVARTYPE_INTEGER) {
		buffer->append((*pvt->_subvars)[which].value.integerval);
	} else if ((*pvt->_subvars)[which].type==
				SQLRCLIENTBINDVARTYPE_DOUBLE) {
		buffer->append((*pvt->_subvars)[which].value.doubleval.value,
			(*pvt->_subvars)[which].value.doubleval.precision,
			(*pvt->_subvars)[which].value.doubleval.scale);
	}
	(*pvt->_subvars)[which].substituted=true;
}

bool sqlrcursor::runQuery(const char *query) {

	// send the query
	if (sendQueryInternal(query)) {

		sendInputBinds();
		sendOutputBinds();
		sendGetColumnInfo();

		pvt->_sqlrc->flushWriteBuffer();

		if (pvt->_rsbuffersize) {
			if (processResultSet(false,pvt->_rsbuffersize-1)) {
				return true;
			}
		} else {
			if (processResultSet(true,0)) {
				return true;
			}
		}
	}
	return false;
}

bool sqlrcursor::sendQueryInternal(const char *query) {

	// if the first 8 characters of the query are "-- debug" followed
	// by a return, then set debugging on
	if (!charstring::compare(query,"-- debug\n",9)) {
		pvt->_sqlrc->debugOn();
	}

	if (!pvt->_endofresultset) {
		closeResultSet(false);
	}
	clearResultSet();

	if (!pvt->_sqlrc->openSession()) {
		return false;
	}

	pvt->_cached=false;
	pvt->_endofresultset=false;

	// send the query to the server.
	if (!pvt->_reexecute) {

		// tell the server we're sending a query
		pvt->_sqlrc->cs()->write((uint16_t)NEW_QUERY);

		// tell the server whether we'll need a cursor or not
		sendCursorStatus();

		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Sending Client Info:");
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPrint("Length: ");
			pvt->_sqlrc->debugPrint(
					(int64_t)pvt->_sqlrc->clientinfolen());
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPrint(pvt->_sqlrc->clientinfo());
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPreEnd();
		}

		// send the client info
		// FIXME: arguably this should be its own command
		pvt->_sqlrc->cs()->write(pvt->_sqlrc->clientinfolen());
		pvt->_sqlrc->cs()->write(pvt->_sqlrc->clientinfo(),
					pvt->_sqlrc->clientinfolen());

		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Sending Query:");
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPrint("Length: ");
			pvt->_sqlrc->debugPrint((int64_t)pvt->_querylen);
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPrint(query);
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPreEnd();
		}

		// send the query
		pvt->_sqlrc->cs()->write(pvt->_querylen);
		pvt->_sqlrc->cs()->write(query,pvt->_querylen);

	} else {

		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Requesting re-execution of ");
			pvt->_sqlrc->debugPrint("previous query.");
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPrint("Requesting Cursor: ");
			pvt->_sqlrc->debugPrint((int64_t)pvt->_cursorid);
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPreEnd();
		}

		// tell the server we're sending a query
		pvt->_sqlrc->cs()->write((uint16_t)REEXECUTE_QUERY);

		// send the cursor id to the server
		pvt->_sqlrc->cs()->write(pvt->_cursorid);
	}

	return true;
}

void sqlrcursor::sendCursorStatus() {

	if (pvt->_havecursorid) {

		// tell the server we already have a cursor
		pvt->_sqlrc->cs()->write((uint16_t)DONT_NEED_NEW_CURSOR);

		// send the cursor id to the server
		pvt->_sqlrc->cs()->write(pvt->_cursorid);

		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Requesting Cursor: ");
			pvt->_sqlrc->debugPrint((int64_t)pvt->_cursorid);
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPreEnd();
		}

	} else {

		// tell the server we need a cursor
		pvt->_sqlrc->cs()->write((uint16_t)NEED_NEW_CURSOR);

		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Requesting a new cursor.\n");
			pvt->_sqlrc->debugPreEnd();
		}
	}
}

void sqlrcursor::sendInputBinds() {

	// index
	uint16_t	i=0;

	// count number of vars to send
	uint16_t	count=pvt->_inbindvars->getLength();
	for (i=0; i<count; i++) {
		if (!(*pvt->_inbindvars)[i].send) {
			count--;
		}
	}

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Sending ");
		pvt->_sqlrc->debugPrint((int64_t)count);
		pvt->_sqlrc->debugPrint(" Input Bind Variables:\n");
		pvt->_sqlrc->debugPreEnd();
	}

	// write the input bind variables/values to the server.
	pvt->_sqlrc->cs()->write(count);
	uint16_t	size;
	i=0;
	while (i<count) {

		// don't send anything if the send flag is turned off
		if (!(*pvt->_inbindvars)[i].send) {
			continue;
		}

		// send the variable
		size=charstring::length((*pvt->_inbindvars)[i].variable);
		pvt->_sqlrc->cs()->write(size);
		pvt->_sqlrc->cs()->write(
				(*pvt->_inbindvars)[i].variable,(size_t)size);
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint(
					(*pvt->_inbindvars)[i].variable);
			pvt->_sqlrc->debugPrint("(");
			pvt->_sqlrc->debugPrint((int64_t)size);
		}

		// send the type
		pvt->_sqlrc->cs()->write((uint16_t)(*pvt->_inbindvars)[i].type);

		// send the value
		if ((*pvt->_inbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_NULL) {

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPrint(":NULL)\n");
				pvt->_sqlrc->debugPreEnd();
			}

		} else if ((*pvt->_inbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_STRING) {

			pvt->_sqlrc->cs()->write(
					(*pvt->_inbindvars)[i].valuesize);
			if ((*pvt->_inbindvars)[i].valuesize>0) {
				pvt->_sqlrc->cs()->write(
					(*pvt->_inbindvars)[i].value.stringval,
					(size_t)(*pvt->_inbindvars)[i].
								valuesize);
			}

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPrint(":STRING)=");
				pvt->_sqlrc->debugPrint(
					(*pvt->_inbindvars)[i].value.stringval);
				pvt->_sqlrc->debugPrint("(");
				pvt->_sqlrc->debugPrint(
					(int64_t)(*pvt->_inbindvars)[i].
								valuesize);
				pvt->_sqlrc->debugPrint(")");
				pvt->_sqlrc->debugPrint("\n");
				pvt->_sqlrc->debugPreEnd();
			}

		} else if ((*pvt->_inbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_INTEGER) {

			pvt->_sqlrc->cs()->write(
					(uint64_t)(*pvt->_inbindvars)[i].
							value.integerval);

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPrint(":LONG)=");
				pvt->_sqlrc->debugPrint(
					(int64_t)(*pvt->_inbindvars)[i].
							value.integerval);
				pvt->_sqlrc->debugPrint("\n");
				pvt->_sqlrc->debugPreEnd();
			}

		} else if ((*pvt->_inbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_DOUBLE) {

			pvt->_sqlrc->cs()->write(
					(*pvt->_inbindvars)[i].value.
							doubleval.value);
			pvt->_sqlrc->cs()->write((*pvt->_inbindvars)[i].value.
							doubleval.precision);
			pvt->_sqlrc->cs()->write((*pvt->_inbindvars)[i].value.
							doubleval.scale);

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPrint(":DOUBLE)=");
				pvt->_sqlrc->debugPrint(
					(*pvt->_inbindvars)[i].value.
							doubleval.value);
				pvt->_sqlrc->debugPrint(":");
				pvt->_sqlrc->debugPrint(
					(int64_t)(*pvt->_inbindvars)[i].
						value.doubleval.precision);
				pvt->_sqlrc->debugPrint(",");
				pvt->_sqlrc->debugPrint(
					(int64_t)(*pvt->_inbindvars)[i].
						value.doubleval.scale);
				pvt->_sqlrc->debugPrint("\n");
				pvt->_sqlrc->debugPreEnd();
			}

		} else if ((*pvt->_inbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_DATE) {

			pvt->_sqlrc->cs()->write((uint16_t)
					(*pvt->_inbindvars)[i].
						value.dateval.year);
			pvt->_sqlrc->cs()->write((uint16_t)
					(*pvt->_inbindvars)[i].
						value.dateval.month);
			pvt->_sqlrc->cs()->write((uint16_t)
					(*pvt->_inbindvars)[i].
						value.dateval.day);
			pvt->_sqlrc->cs()->write((uint16_t)
					(*pvt->_inbindvars)[i].
						value.dateval.hour);
			pvt->_sqlrc->cs()->write((uint16_t)
					(*pvt->_inbindvars)[i].
						value.dateval.minute);
			pvt->_sqlrc->cs()->write((uint16_t)
					(*pvt->_inbindvars)[i].
						value.dateval.second);
			pvt->_sqlrc->cs()->write((uint32_t)
					(*pvt->_inbindvars)[i].
						value.dateval.microsecond);
			pvt->_sqlrc->cs()->write((uint16_t)
					charstring::length(
					(*pvt->_inbindvars)[i].
						value.dateval.tz));
			pvt->_sqlrc->cs()->write(
					(*pvt->_inbindvars)[i].
						value.dateval.tz);

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPrint(":DATE)=");
				pvt->_sqlrc->debugPrint((int64_t)
					(*pvt->_inbindvars)[i].
						value.dateval.year);
				pvt->_sqlrc->debugPrint("-");
				pvt->_sqlrc->debugPrint((int64_t)
					(*pvt->_inbindvars)[i].
						value.dateval.month);
				pvt->_sqlrc->debugPrint("-");
				pvt->_sqlrc->debugPrint((int64_t)
					(*pvt->_inbindvars)[i].
						value.dateval.day);
				pvt->_sqlrc->debugPrint(" ");
				pvt->_sqlrc->debugPrint((int64_t)
					(*pvt->_inbindvars)[i].
						value.dateval.hour);
				pvt->_sqlrc->debugPrint(":");
				pvt->_sqlrc->debugPrint((int64_t)
					(*pvt->_inbindvars)[i].
						value.dateval.minute);
				pvt->_sqlrc->debugPrint(":");
				pvt->_sqlrc->debugPrint((int64_t)
					(*pvt->_inbindvars)[i].
						value.dateval.second);
				pvt->_sqlrc->debugPrint(":");
				pvt->_sqlrc->debugPrint((int64_t)
					(*pvt->_inbindvars)[i].value.
						dateval.microsecond);
				pvt->_sqlrc->debugPrint(" ");
				pvt->_sqlrc->debugPrint(
					(*pvt->_inbindvars)[i].
						value.dateval.tz);
				pvt->_sqlrc->debugPrint("\n");
				pvt->_sqlrc->debugPreEnd();
			}

		} else if ((*pvt->_inbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_BLOB ||
				(*pvt->_inbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_CLOB) {

			pvt->_sqlrc->cs()->write(
					(*pvt->_inbindvars)[i].valuesize);
			if ((*pvt->_inbindvars)[i].valuesize>0) {
				pvt->_sqlrc->cs()->write(
					(*pvt->_inbindvars)[i].value.lobval,
					(size_t)(*pvt->_inbindvars)[i].
								valuesize);
			}

			if (pvt->_sqlrc->debug()) {
				if ((*pvt->_inbindvars)[i].type==
						SQLRCLIENTBINDVARTYPE_BLOB) {
					pvt->_sqlrc->debugPrint(":BLOB)=");
					pvt->_sqlrc->debugPrintBlob(
						(*pvt->_inbindvars)[i].
								value.lobval,
						(*pvt->_inbindvars)[i].
								valuesize);
				} else if ((*pvt->_inbindvars)[i].type==
						SQLRCLIENTBINDVARTYPE_CLOB) {
					pvt->_sqlrc->debugPrint(":CLOB)=");
					pvt->_sqlrc->debugPrintClob(
						(*pvt->_inbindvars)[i].
								value.lobval,
						(*pvt->_inbindvars)[i].
								valuesize);
				}
				pvt->_sqlrc->debugPrint("(");
				pvt->_sqlrc->debugPrint((int64_t)
						(*pvt->_inbindvars)[i].
								valuesize);
				pvt->_sqlrc->debugPrint(")");
				pvt->_sqlrc->debugPrint("\n");
				pvt->_sqlrc->debugPreEnd();
			}
		}

		i++;
	}
}

void sqlrcursor::sendOutputBinds() {

	// index
	uint16_t	i=0;

	// count number of vars to send
	uint16_t	count=pvt->_outbindvars->getLength();
	for (i=0; i<count; i++) {
		if (!(*pvt->_outbindvars)[i].send) {
			count--;
		}
	}

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Sending ");
		pvt->_sqlrc->debugPrint((int64_t)count);
		pvt->_sqlrc->debugPrint(" Output Bind Variables:\n");
		pvt->_sqlrc->debugPreEnd();
	}

	// write the output bind variables to the server.
	pvt->_sqlrc->cs()->write(count);
	uint16_t	size;
	i=0;
	while (i<count) {

		// don't send anything if the send flag is turned off
		if (!(*pvt->_outbindvars)[i].send) {
			continue;
		}

		// send the variable, type and size that the buffer needs to be
		size=charstring::length((*pvt->_outbindvars)[i].variable);
		pvt->_sqlrc->cs()->write(size);
		pvt->_sqlrc->cs()->write(
				(*pvt->_outbindvars)[i].variable,(size_t)size);
		pvt->_sqlrc->cs()->write((uint16_t)
				(*pvt->_outbindvars)[i].type);
		if ((*pvt->_outbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_STRING ||
			(*pvt->_outbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_BLOB ||
			(*pvt->_outbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_CLOB ||
			(*pvt->_outbindvars)[i].type==
					SQLRCLIENTBINDVARTYPE_NULL) {
			pvt->_sqlrc->cs()->write(
				(*pvt->_outbindvars)[i].valuesize);
		}

		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint(
				(*pvt->_outbindvars)[i].variable);
			const char	*bindtype=NULL;
			switch ((*pvt->_outbindvars)[i].type) {
				case SQLRCLIENTBINDVARTYPE_NULL:
					bindtype="(NULL)";
					break;
				case SQLRCLIENTBINDVARTYPE_STRING:
					bindtype="(STRING)";
					break;
				case SQLRCLIENTBINDVARTYPE_INTEGER:
					bindtype="(INTEGER)";
					break;
				case SQLRCLIENTBINDVARTYPE_DOUBLE:
					bindtype="(DOUBLE)";
					break;
				case SQLRCLIENTBINDVARTYPE_DATE:
					bindtype="(DATE)";
					break;
				case SQLRCLIENTBINDVARTYPE_BLOB:
					bindtype="(BLOB)";
					break;
				case SQLRCLIENTBINDVARTYPE_CLOB:
					bindtype="(CLOB)";
					break;
				case SQLRCLIENTBINDVARTYPE_CURSOR:
					bindtype="(CURSOR)";
					break;
			}
			pvt->_sqlrc->debugPrint(bindtype);
			if ((*pvt->_outbindvars)[i].type==
						SQLRCLIENTBINDVARTYPE_STRING ||
				(*pvt->_outbindvars)[i].type==
						SQLRCLIENTBINDVARTYPE_BLOB ||
				(*pvt->_outbindvars)[i].type==
						SQLRCLIENTBINDVARTYPE_CLOB ||
				(*pvt->_outbindvars)[i].type==
						SQLRCLIENTBINDVARTYPE_NULL) {
				pvt->_sqlrc->debugPrint("(");
				pvt->_sqlrc->debugPrint((int64_t)
					(*pvt->_outbindvars)[i].valuesize);
				pvt->_sqlrc->debugPrint(")");
			}
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPreEnd();
		}

		i++;
	}
}

void sqlrcursor::sendGetColumnInfo() {

	if (pvt->_sendcolumninfo==SEND_COLUMN_INFO) {
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Send Column Info: yes\n");
			pvt->_sqlrc->debugPreEnd();
		}
		pvt->_sqlrc->cs()->write((uint16_t)SEND_COLUMN_INFO);
	} else {
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Send Column Info: no\n");
			pvt->_sqlrc->debugPreEnd();
		}
		pvt->_sqlrc->cs()->write((uint16_t)DONT_SEND_COLUMN_INFO);
	}
}

bool sqlrcursor::processResultSet(bool getallrows, uint64_t rowtoget) {

	// start caching the result set
	if (pvt->_cacheon) {
		startCaching();
	}

	// parse the columninfo and data
	bool	success=true;

	// skip and fetch here if we're not reading from a cached result set
	// this way, everything gets done in 1 round trip
	if (!pvt->_cachesource) {
		success=skipAndFetch(getallrows,pvt->_firstrowindex+rowtoget);
	}

	// check for an error
	if (success) {

		uint16_t	err=getErrorStatus();
		if (err!=NO_ERROR_OCCURRED) {

			// if there was a timeout, then end
			// the session and bail immediately
			if (err==TIMEOUT_GETTING_ERROR_STATUS) {
				pvt->_sqlrc->endSession();
				return false;
			}

			// otherwise, get the error from the server
			getErrorFromServer();

			// don't get the cursor if the error was that there
			// were no cursors available
			if (pvt->_errorno!=SQLR_ERROR_NOCURSORS) {
				getCursorId();
			}

			// if we need to disconnect then end the session
			if (err==ERROR_OCCURRED_DISCONNECT) {
				pvt->_sqlrc->endSession();
			}
			return false;
		}
	}

	// get data back from the server
	if (success && ((pvt->_cachesource && pvt->_cachesourceind) ||
			((!pvt->_cachesource && !pvt->_cachesourceind)  && 
				(success=getCursorId()) && 
				(success=getSuspended()))) &&
			(success=parseColumnInfo()) && 
			(success=parseOutputBinds())) {

		// skip and fetch here if we're reading from a cached result set
		if (pvt->_cachesource) {
			success=skipAndFetch(getallrows,
					pvt->_firstrowindex+rowtoget);
		}

		// parse the data
		if (success) {
			success=parseData();
		}
	}

	// if success is false, then some kind of network error occurred,
	// end the session
	if (!success) {
		clearResultSet();
		pvt->_sqlrc->endSession();
	}
	return success;
}

bool sqlrcursor::skipAndFetch(bool getallrows, uint64_t rowtoget) {

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Skipping and Fetching\n");
		if (!getallrows) {
			pvt->_sqlrc->debugPrint("	row to get: ");
			pvt->_sqlrc->debugPrint((int64_t)rowtoget);
			pvt->_sqlrc->debugPrint("\n");
		}
		pvt->_sqlrc->debugPreEnd();
	}

	// if we're stepping through the result set, we can possibly 
	// skip a big chunk of it...
	if (!skipRows(getallrows,rowtoget)) {
		return false;
	}

	// tell the connection how many rows to send
	fetchRows();

	pvt->_sqlrc->flushWriteBuffer();
	return true;
}

void sqlrcursor::fetchRows() {

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Fetching ");
		pvt->_sqlrc->debugPrint((int64_t)pvt->_rsbuffersize);
		pvt->_sqlrc->debugPrint(" rows\n");
		pvt->_sqlrc->debugPreEnd();
	}

	// if we're reading from a cached result set, do nothing
	if (pvt->_cachesource && pvt->_cachesourceind) {
		return;
	}

	// otherwise, send to the connection the number of rows to send back
	pvt->_sqlrc->cs()->write(pvt->_rsbuffersize);
}

bool sqlrcursor::skipRows(bool getallrows, uint64_t rowtoget) {

	// if we're reading from a cached result set we have to manually skip
	if (pvt->_cachesource && pvt->_cachesourceind) {

		// skip to the next block of rows
		if (getallrows) {
			return true;
		} else {
			pvt->_rowcount=rowtoget-(rowtoget%pvt->_rsbuffersize);
		}

		// get the row offset from the index
		pvt->_cachesourceind->setPositionRelativeToBeginning(
					13+sizeof(int64_t)+
					(pvt->_rowcount*sizeof(int64_t)));
		int64_t	rowoffset;
		if (pvt->_cachesourceind->read(&rowoffset)!=sizeof(int64_t)) {
			setError("The cache file index appears to be corrupt.");
			return false;
		}

		// skip to that offset in the cache file
		pvt->_cachesource->setPositionRelativeToBeginning(rowoffset);
		return true;
	}

	// calculate how many rows to skip unless we're buffering the entire
	// result set or caching the result set
	uint64_t	skip=0;
	if (pvt->_rsbuffersize && !pvt->_cachedest && !getallrows) {
		skip=(rowtoget-(rowtoget%pvt->_rsbuffersize))-pvt->_rowcount; 
		pvt->_rowcount=pvt->_rowcount+skip;
	}
	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Skipping ");
		pvt->_sqlrc->debugPrint((int64_t)skip);
		pvt->_sqlrc->debugPrint(" rows\n");
		pvt->_sqlrc->debugPreEnd();
	}

	// if we're reading from a connection, send the connection the 
	// number of rows to skip
	pvt->_sqlrc->cs()->write(skip);
	return true;
}

uint16_t sqlrcursor::getErrorStatus() {

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Checking For An Error...\n");
		pvt->_sqlrc->debugPreEnd();
	}

	// get a flag indicating whether there's been an error or not
	uint16_t	err;
	int32_t	result=getShort(&err,pvt->_sqlrc->responsetimeoutsec(),
					pvt->_sqlrc->responsetimeoutusec());
	if (result==RESULT_TIMEOUT) {
		setError("Timeout while determining whether "
				"an error occurred or not.\n");
		return TIMEOUT_GETTING_ERROR_STATUS;
	} else if (result!=sizeof(uint16_t)) {
		setError("Failed to determine whether an "
				"error occurred or not.\n "
				"A network error may have ocurred.");
		return ERROR_OCCURRED;
	}

	if (err==NO_ERROR_OCCURRED) {
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("none.\n");
			pvt->_sqlrc->debugPreEnd();
		}
		cacheNoError();
		return NO_ERROR_OCCURRED;
	}

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("error!!!\n");
		pvt->_sqlrc->debugPreEnd();
	}
	return err;
}

bool sqlrcursor::getCursorId() {

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Getting Cursor ID...\n");
		pvt->_sqlrc->debugPreEnd();
	}
	if (pvt->_sqlrc->cs()->read(&pvt->_cursorid)!=sizeof(uint16_t)) {
		if (!pvt->_error) {
			char	*err=error::getErrorString();
			stringbuffer	errstr;
			errstr.append("Failed to get a cursor id.\n "
					"A network error may have ocurred. ");
			errstr.append(err);
			setError(errstr.getString());
			delete[] err;
		}
		return false;
	}
	pvt->_havecursorid=true;
	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Cursor ID: ");
		pvt->_sqlrc->debugPrint((int64_t)pvt->_cursorid);
		pvt->_sqlrc->debugPrint("\n");
		pvt->_sqlrc->debugPreEnd();
	}
	return true;
}

bool sqlrcursor::getSuspended() {

	// see if the result set of that cursor is actually suspended
	uint16_t	suspendedresultset;
	if (pvt->_sqlrc->cs()->read(&suspendedresultset)!=sizeof(uint16_t)) {
		setError("Failed to determine whether "
			"the session was suspended or not.\n "
			"A network error may have ocurred.");
		return false;
	}

	if (suspendedresultset==SUSPENDED_RESULT_SET) {

		// If it was suspended the server will send the index of the 
		// last row from the previous result set.
		// Initialize firstrowindex and rowcount from this index.
		if (pvt->_sqlrc->cs()->read(&pvt->_firstrowindex)!=
							sizeof(uint64_t)) {
			setError("Failed to get the index of the "
				"last row of a previously suspended result "
				"set.\n A network error may have ocurred.");
			return false;
		}
		pvt->_rowcount=pvt->_firstrowindex+1;
	
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Previous result set was ");
	       		pvt->_sqlrc->debugPrint("suspended at row index: ");
			pvt->_sqlrc->debugPrint((int64_t)pvt->_firstrowindex);
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPreEnd();
		}

	} else {

		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Previous result set was ");
	       		pvt->_sqlrc->debugPrint("not suspended.\n");
			pvt->_sqlrc->debugPreEnd();
		}
	}
	return true;
}

bool sqlrcursor::parseColumnInfo() {

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Parsing Column Info\n");
		pvt->_sqlrc->debugPrint("Actual row count: ");
		pvt->_sqlrc->debugPreEnd();
	}

	// first get whether the server knows the total number of rows or not
	if (getShort(&pvt->_knowsactualrows)!=sizeof(uint16_t)) {
		setError("Failed to get whether the server knows "
				"the number of actual rows or not.\n"
				"A network error may have occurred.");
		return false;
	}

	// get the number of rows returned by the query
	if (pvt->_knowsactualrows==ACTUAL_ROWS) {
		if (getLongLong(&pvt->_actualrows)!=sizeof(uint64_t)) {
			setError("Failed to get the number of actual rows.\n"
					"A network error may have occurred.");
			return false;
		}
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint((int64_t)pvt->_actualrows);
			pvt->_sqlrc->debugPreEnd();
		}
	} else {
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("unknown");
			pvt->_sqlrc->debugPreEnd();
		}
	}

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("\n");
		pvt->_sqlrc->debugPrint("Affected row count: ");
		pvt->_sqlrc->debugPreEnd();
	}

	// get whether the server knows the number of affected rows or not
	if (getShort(&pvt->_knowsaffectedrows)!=sizeof(uint16_t)) {
		setError("Failed to get whether the server knows "
				"the number of affected rows or not.\n"
				"A network error may have occurred.");
		return false;
	}

	// get the number of rows affected by the query
	if (pvt->_knowsaffectedrows==AFFECTED_ROWS) {
		if (getLongLong(&pvt->_affectedrows)!=sizeof(uint64_t)) {
			setError("Failed to get the number of affected rows.\n"
				"A network error may have occurred.");
			return false;
		}
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint((int64_t)pvt->_affectedrows);
			pvt->_sqlrc->debugPreEnd();
		}
	} else {
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("unknown");
			pvt->_sqlrc->debugPreEnd();
		}
	}

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("\n");
		pvt->_sqlrc->debugPreEnd();
	}

	// get whether the server is sending column info or not
	if (getShort(&pvt->_sentcolumninfo)!=sizeof(uint16_t)) {
		setError("Failed to get whether the server "
				"is sending column info or not.\n"
				"A network error may have occurred.");
		return false;
	}

	// get column count
	if (getLong(&pvt->_colcount)!=sizeof(uint32_t)) {
		setError("Failed to get the column count.\n"
				"A network error may have occurred.");
		return false;
	}
	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Column count: ");
		pvt->_sqlrc->debugPrint((int64_t)pvt->_colcount);
		pvt->_sqlrc->debugPrint("\n");
		pvt->_sqlrc->debugPreEnd();
	}

	// we have to do this here even if we're not getting the column
	// descriptions because we are going to use the longdatatype member
	// variable no matter what
	createColumnBuffers();

	if (pvt->_sendcolumninfo==SEND_COLUMN_INFO && 
			pvt->_sentcolumninfo==SEND_COLUMN_INFO) {

		// get whether column types will be predefined id's or strings
		if (getShort(&pvt->_columntypeformat)!=sizeof(uint16_t)) {
			setError("Failed to whether column types will be "
					"predefined id's or strings.\n"
					"A network error may have occurred.");
			return false;
		}

		// some useful variables
		uint16_t			length;
		sqlrclientcolumn		*currentcol;

		// get the columninfo segment
		for (uint32_t i=0; i<pvt->_colcount; i++) {
	
			// get the column name length
			if (getShort(&length)!=sizeof(uint16_t)) {
				setError("Failed to get the column name "
					"length.\n"
					"A network error may have occurred.");
				return false;
			}
	
			// which column to use
			currentcol=getColumnInternal(i);
	
			// get the column name
			currentcol->name=
				(char *)pvt->_colstorage->allocate(length+1);
			if (getString(currentcol->name,length)!=length) {
				setError("Failed to get the column name.\n "
					"A network error may have occurred.");
				return false;
			}
			currentcol->name[length]='\0';

			// upper/lowercase column name if necessary
			if (pvt->_colcase==UPPER_CASE) {
				charstring::upper(currentcol->name);
			} else if (pvt->_colcase==LOWER_CASE) {
				charstring::lower(currentcol->name);
			}

			if (pvt->_columntypeformat==COLUMN_TYPE_IDS) {

				// get the column type
				if (getShort(&currentcol->type)!=
						sizeof(uint16_t)) {
					setError("Failed to get the column "
						"type.\n"
						"A network error may have "
						"occurred.");
					return false;
				}

			} else {

				// get the column type length
				if (getShort(&currentcol->typestringlength)!=
						sizeof(uint16_t)) {
					setError("Failed to get the column "
						"type length.\n"
						"A network error may have "
						"occurred.");
					return false;
				}

				// get the column type
				currentcol->typestring=new
					char[currentcol->typestringlength+1];
				currentcol->typestring[
					currentcol->typestringlength]='\0';
				if (getString(currentcol->typestring,
						currentcol->typestringlength)!=
						currentcol->typestringlength) {
					setError("Failed to get the column "
						"type.\n"
						"A network error may have "
						"occurred.");
					return false;
				}
			}

			// get the column length
			// get the column precision
			// get the column scale
			// get whether the column is nullable
			// get whether the column is a primary key
			// get whether the column is unique
			// get whether the column is part of a key
			// get whether the column is unsigned
			// get whether the column is zero-filled
			// get whether the column is binary
			// get whether the column is auto-incremented
			if (getLong(&currentcol->length)!=
						sizeof(uint32_t) ||
				getLong(&currentcol->precision)!=
						sizeof(uint32_t) ||
				getLong(&currentcol->scale)!=
						sizeof(uint32_t) ||
				getShort(&currentcol->nullable)!=
						sizeof(uint16_t) ||
				getShort(&currentcol->primarykey)!=
						sizeof(uint16_t) ||
				getShort(&currentcol->unique)!=
						sizeof(uint16_t) ||
				getShort(&currentcol->partofkey)!=
						sizeof(uint16_t) ||
				getShort(&currentcol->unsignednumber)!=
						sizeof(uint16_t) ||
				getShort(&currentcol->zerofill)!=
						sizeof(uint16_t) ||
				getShort(&currentcol->binary)!=
						sizeof(uint16_t) ||
				getShort(&currentcol->autoincrement)!=
						sizeof(uint16_t)) {
				setError("Failed to get column info.\n"
					"A network error may have occurred.");
				return false;
			}

			// initialize the longest value
			currentcol->longest=0;
	
			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint("\"");
				pvt->_sqlrc->debugPrint(currentcol->name);
				pvt->_sqlrc->debugPrint("\",");
				pvt->_sqlrc->debugPrint("\"");
				if (pvt->_columntypeformat!=COLUMN_TYPE_IDS) {
					pvt->_sqlrc->debugPrint(
						currentcol->typestring);
				} else {
					pvt->_sqlrc->debugPrint(
						datatypestring[
							currentcol->type]);
				}
				pvt->_sqlrc->debugPrint("\", ");
				pvt->_sqlrc->debugPrint((int64_t)
							currentcol->length);
				pvt->_sqlrc->debugPrint(" (");
				pvt->_sqlrc->debugPrint((int64_t)
							currentcol->precision);
				pvt->_sqlrc->debugPrint(",");
				pvt->_sqlrc->debugPrint((int64_t)
							currentcol->scale);
				pvt->_sqlrc->debugPrint(") ");
				if (!currentcol->nullable) {
					pvt->_sqlrc->debugPrint(
							"NOT NULL ");
				}
				if (currentcol->primarykey) {
					pvt->_sqlrc->debugPrint(
							"Primary Key ");
				}
				if (currentcol->unique) {
					pvt->_sqlrc->debugPrint(
							"Unique ");
				}
				if (currentcol->partofkey) {
					pvt->_sqlrc->debugPrint(
							"Part of a Key ");
				}
				if (currentcol->unsignednumber) {
					pvt->_sqlrc->debugPrint(
							"Unsigned ");
				}
				if (currentcol->zerofill) {
					pvt->_sqlrc->debugPrint(
							"Zero Filled ");
				}
				if (currentcol->binary) {
					pvt->_sqlrc->debugPrint(
							"Binary ");
				}
				if (currentcol->autoincrement) {
					pvt->_sqlrc->debugPrint(
							"Auto-Increment ");
				}
				pvt->_sqlrc->debugPrint("\n");
				pvt->_sqlrc->debugPreEnd();
			}

		}
	}

	// cache the column definitions
	cacheColumnInfo();

	return true;
}

void sqlrcursor::createColumnBuffers() {

	// we could get really sophisticated here and keep stats on the number
	// of columns that previous queries returned and adjust the size of
	// "columns" periodically, but for now, we'll just use a static size

	// create the standard set of columns, this will hang around until
	// the cursor is deleted
	if (!pvt->_columns) {
		pvt->_columns=new sqlrclientcolumn[OPTIMISTIC_COLUMN_COUNT];
	}

	// if there are more columns than our static column buffer
	// can handle, create extra columns, these will be deleted after each
	// query
	if (pvt->_colcount>OPTIMISTIC_COLUMN_COUNT &&
			pvt->_colcount>pvt->_previouscolcount) {
		delete[] pvt->_extracolumns;
		pvt->_extracolumns=
			new sqlrclientcolumn[
				pvt->_colcount-OPTIMISTIC_COLUMN_COUNT];
	}
}

bool sqlrcursor::parseOutputBinds() {

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Receiving Output Bind Values: \n");
		pvt->_sqlrc->debugPreEnd();
	}

	// useful variables
	uint16_t	type;
	uint32_t	length;
	uint16_t	count=0;

	// get the bind values
	for (;;) {

		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("	getting type...\n");
			pvt->_sqlrc->debugPreEnd();
		}

		// get the data type
		if (getShort(&type)!=sizeof(uint16_t)) {
			setError("Failed to get data type.\n "
				"A network error may have occurred.");

			return false;
		}

		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("	done getting type: ");
			pvt->_sqlrc->debugPrint((int64_t)type);
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPreEnd();
		}

		// check for end of bind values
		if (type==END_BIND_VARS) {

			break;

		} else if (type==NULL_DATA) {

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint(
						"	NULL output bind\n");
				pvt->_sqlrc->debugPreEnd();
			}

			// handle a null value
			(*pvt->_outbindvars)[count].resultvaluesize=0;
			if ((*pvt->_outbindvars)[count].type==
						SQLRCLIENTBINDVARTYPE_STRING) {
				if (pvt->_returnnulls) {
					(*pvt->_outbindvars)[count].value.
							stringval=NULL;
				} else {
					(*pvt->_outbindvars)[count].value.
							stringval=new char[1];
					(*pvt->_outbindvars)[count].value.
							stringval[0]='\0';
				}
			} else if ((*pvt->_outbindvars)[count].type==
						SQLRCLIENTBINDVARTYPE_INTEGER) {
				(*pvt->_outbindvars)[count].value.integerval=0;
			} else if ((*pvt->_outbindvars)[count].type==
						SQLRCLIENTBINDVARTYPE_DOUBLE) {
				(*pvt->_outbindvars)[count].
						value.doubleval.value=0;
				(*pvt->_outbindvars)[count].
						value.doubleval.precision=0;
				(*pvt->_outbindvars)[count].
						value.doubleval.scale=0;
			} else if ((*pvt->_outbindvars)[count].type==
						SQLRCLIENTBINDVARTYPE_DATE) {
				(*pvt->_outbindvars)[count].
						value.dateval.year=0;
				(*pvt->_outbindvars)[count].
						value.dateval.month=0;
				(*pvt->_outbindvars)[count].
						value.dateval.day=0;
				(*pvt->_outbindvars)[count].
						value.dateval.hour=0;
				(*pvt->_outbindvars)[count].
						value.dateval.minute=0;
				(*pvt->_outbindvars)[count].
						value.dateval.second=0;
				(*pvt->_outbindvars)[count].
						value.dateval.microsecond=0;
				if (pvt->_returnnulls) {
					(*pvt->_outbindvars)[count].
						value.dateval.tz=NULL;
				} else {
					(*pvt->_outbindvars)[count].
						value.dateval.tz=new char[1];
					(*pvt->_outbindvars)[count].
						value.dateval.tz[0]='\0';
				}
			} 

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPrint("		");
				pvt->_sqlrc->debugPrint("done fetching.\n");
			}

		} else if (type==STRING_DATA) {

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint(
						"	STRING output bind\n");
				pvt->_sqlrc->debugPreEnd();
			}

			// get the value length
			if (getLong(&length)!=sizeof(uint32_t)) {
				setError("Failed to get string value length.\n "
					"A network error may have occurred.");
				return false;
			}
			(*pvt->_outbindvars)[count].resultvaluesize=length;
			(*pvt->_outbindvars)[count].value.stringval=
							new char[length+1];

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint(
						"		length=");
				pvt->_sqlrc->debugPrint((int64_t)length);
				pvt->_sqlrc->debugPrint("\n");
				pvt->_sqlrc->debugPreEnd();
			}

			// get the value
			if ((uint32_t)getString(
					(*pvt->_outbindvars)[count].value.
						stringval,length)!=length) {
				setError("Failed to get string value.\n "
					"A network error may have occurred.");
				return false;
			}
			(*pvt->_outbindvars)[count].
					value.stringval[length]='\0';

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint("		");
				pvt->_sqlrc->debugPrint("done fetching\n");
				pvt->_sqlrc->debugPreEnd();
			}

		} else if (type==INTEGER_DATA) {

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint(
						"	INTEGER output bind\n");
				pvt->_sqlrc->debugPreEnd();
			}

			// get the value
			if (getLongLong((uint64_t *)
					&(*pvt->_outbindvars)[count].value.
						integerval)!=sizeof(uint64_t)) {
				setError("Failed to get integer value.\n "
					"A network error may have occurred.");
				return false;
			}

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint("		");
				pvt->_sqlrc->debugPrint("done fetching\n");
				pvt->_sqlrc->debugPreEnd();
			}

		} else if (type==DOUBLE_DATA) {

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint(
					"	DOUBLE output bind\n");
				pvt->_sqlrc->debugPreEnd();
			}

			// get the value
			if (getDouble(&(*pvt->_outbindvars)[count].value.
						doubleval.value)!=
						sizeof(double)) {
				setError("Failed to get double value.\n "
					"A network error may have occurred.");
				return false;
			}

			// get the precision
			if (getLong(&(*pvt->_outbindvars)[count].value.
						doubleval.precision)!=
						sizeof(uint32_t)) {
				setError("Failed to get precision.\n "
					"A network error may have occurred.");
				return false;
			}

			// get the scale
			if (getLong(&(*pvt->_outbindvars)[count].value.
						doubleval.scale)!=
						sizeof(uint32_t)) {
				setError("Failed to get scale.\n "
					"A network error may have occurred.");
				return false;
			}

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint("		");
				pvt->_sqlrc->debugPrint("done fetching\n");
				pvt->_sqlrc->debugPreEnd();
			}

		} else if (type==DATE_DATA) {

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint(
					"	DATE output bind\n");
				pvt->_sqlrc->debugPreEnd();
			}

			uint16_t	temp;

			// get the year
			if (getShort(&temp)!=sizeof(uint16_t)) {
				setError("Failed to get long value.\n "
					"A network error may have occurred.");
				return false;
			}
			(*pvt->_outbindvars)[count].
					value.dateval.year=(int16_t)temp;

			// get the month
			if (getShort(&temp)!=sizeof(uint16_t)) {
				setError("Failed to get long value.\n "
					"A network error may have occurred.");
				return false;
			}
			(*pvt->_outbindvars)[count].
					value.dateval.month=(int16_t)temp;

			// get the day
			if (getShort(&temp)!=sizeof(uint16_t)) {
				setError("Failed to get long value.\n "
					"A network error may have occurred.");
				return false;
			}
			(*pvt->_outbindvars)[count].
					value.dateval.day=(int16_t)temp;

			// get the hour
			if (getShort(&temp)!=sizeof(uint16_t)) {
				setError("Failed to get long value.\n "
					"A network error may have occurred.");
				return false;
			}
			(*pvt->_outbindvars)[count].
					value.dateval.hour=(int16_t)temp;

			// get the minute
			if (getShort(&temp)!=sizeof(uint16_t)) {
				setError("Failed to get long value.\n "
					"A network error may have occurred.");
				return false;
			}
			(*pvt->_outbindvars)[count].
					value.dateval.minute=(int16_t)temp;

			// get the second
			if (getShort(&temp)!=sizeof(uint16_t)) {
				setError("Failed to get long value.\n "
					"A network error may have occurred.");
				return false;
			}
			(*pvt->_outbindvars)[count].
					value.dateval.second=(int16_t)temp;

			// get the microsecond
			uint32_t	temp32;
			if (getLong(&temp32)!=sizeof(uint32_t)) {
				setError("Failed to get long value.\n "
					"A network error may have occurred.");
				return false;
			}
			(*pvt->_outbindvars)[count].value.
					dateval.microsecond=(int32_t)temp32;

			// get the timezone length
			uint16_t	length;
			if (getShort(&length)!=sizeof(uint16_t)) {
				setError("Failed to get timezone length.\n "
					"A network error may have occurred.");
				return false;
			}
			(*pvt->_outbindvars)[count].
					value.dateval.tz=new char[length+1];

			// get the timezone
			if ((uint16_t)getString(
					(*pvt->_outbindvars)[count].value.
						dateval.tz,length)!=length) {
				setError("Failed to get timezone.\n "
					"A network error may have occurred.");
				return false;
			}
			(*pvt->_outbindvars)[count].
					value.dateval.tz[length]='\0';

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint("		");
				pvt->_sqlrc->debugPrint("done fetching\n");
				pvt->_sqlrc->debugPreEnd();
			}

		} else if (type==CURSOR_DATA) {

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint(
					"	CURSOR output bind\n");
				pvt->_sqlrc->debugPreEnd();
			}

			// get the cursor id
			if (getShort((uint16_t *)
				&((*pvt->_outbindvars)[count].value.cursorid))!=
				sizeof(uint16_t)) {
				setError("Failed to get cursor id.\n "
					"A network error may have occurred.");
				return false;
			}

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint("		");
				pvt->_sqlrc->debugPrint("done fetching\n");
				pvt->_sqlrc->debugPreEnd();
			}

		} else {

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint("	LOB/CLOB ");
				pvt->_sqlrc->debugPrint("output bind\n");
				pvt->_sqlrc->debugPreEnd();
			}

			// must be START_LONG_DATA...
			// get the total length of the long data
			uint64_t	totallength;
			if (getLongLong(&totallength)!=sizeof(uint64_t)) {
				setError("Failed to get total length.\n "
					"A network error may have occurred.");
				return false;
			}

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint(
					"		length=");
				pvt->_sqlrc->debugPrint((int64_t)totallength);
				pvt->_sqlrc->debugPrint("\n");
				pvt->_sqlrc->debugPreEnd();
			}

			// create a buffer to hold the data
			char	*buffer=new char[totallength+1];

			uint64_t	offset=0;
			uint32_t	length;
			for (;;) {

				if (pvt->_sqlrc->debug()) {
					pvt->_sqlrc->debugPreStart();
					pvt->_sqlrc->debugPrint(
							"		");
					pvt->_sqlrc->debugPrint(
							"fetching...\n");
					pvt->_sqlrc->debugPreEnd();
				}

				// get the type of the chunk
				if (getShort(&type)!=sizeof(uint16_t)) {
					delete[] buffer;
					setError("Failed to get chunk type.\n "
						"A network error may have "
						"occurred.");
					return false;
				}

				// check to see if we're done
				if (type==END_LONG_DATA) {
					break;
				}

				// get the length of the chunk
				if (getLong(&length)!=sizeof(uint32_t)) {
					delete[] buffer;
					setError("Failed to get chunk length.\n"
						" A network error may have "
						"occurred.");
					return false;
				}

				// get the chunk of data
				if ((uint32_t)getString(buffer+offset,
							length)!=length) {
					delete[] buffer;
					setError("Failed to get chunk data.\n "
						"A network error may have "
						"occurred.");
					return false;
				}

				offset=offset+length;
			}

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint("		");
				pvt->_sqlrc->debugPrint("done fetching.\n");
				pvt->_sqlrc->debugPreEnd();
			}

			// NULL terminate the buffer.  This makes 
			// certain operations safer and won't hurt
			// since the actual length (which doesn't
			// include the NULL) is available from
			// getOutputBindLength.
			buffer[totallength]='\0';
			(*pvt->_outbindvars)[count].value.lobval=buffer;
			(*pvt->_outbindvars)[count].resultvaluesize=totallength;
		}

		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint(
					(*pvt->_outbindvars)[count].variable);
			pvt->_sqlrc->debugPrint("=");
			if ((*pvt->_outbindvars)[count].type==
						SQLRCLIENTBINDVARTYPE_BLOB) {
				pvt->_sqlrc->debugPrintBlob(
					(*pvt->_outbindvars)[count].
							value.lobval,
					(*pvt->_outbindvars)[count].
							resultvaluesize);
			} else if ((*pvt->_outbindvars)[count].type==
						SQLRCLIENTBINDVARTYPE_CLOB) {
				pvt->_sqlrc->debugPrintClob(
					(*pvt->_outbindvars)[count].
							value.lobval,
					(*pvt->_outbindvars)[count].
							resultvaluesize);
			} else if ((*pvt->_outbindvars)[count].type==
						SQLRCLIENTBINDVARTYPE_CURSOR) {
				pvt->_sqlrc->debugPrint((int64_t)
						(*pvt->_outbindvars)[count].
							value.cursorid);
			} else if ((*pvt->_outbindvars)[count].type==
						SQLRCLIENTBINDVARTYPE_INTEGER) {
				pvt->_sqlrc->debugPrint(
						(*pvt->_outbindvars)[count].
							value.integerval);
			} else if ((*pvt->_outbindvars)[count].type==
						SQLRCLIENTBINDVARTYPE_DOUBLE) {
				pvt->_sqlrc->debugPrint(
						(*pvt->_outbindvars)[count].
							value.doubleval.value);
				pvt->_sqlrc->debugPrint("(");
				pvt->_sqlrc->debugPrint((int64_t)
						(*pvt->_outbindvars)[count].
							value.doubleval.
							precision);
				pvt->_sqlrc->debugPrint(",");
				pvt->_sqlrc->debugPrint((int64_t)
						(*pvt->_outbindvars)[count].
							value.doubleval.
							scale);
				pvt->_sqlrc->debugPrint(")");
			} else if ((*pvt->_outbindvars)[count].type==
						SQLRCLIENTBINDVARTYPE_DATE) {
				pvt->_sqlrc->debugPrint((int64_t)
						(*pvt->_outbindvars)[count].
							value.dateval.year);
				pvt->_sqlrc->debugPrint("-");
				pvt->_sqlrc->debugPrint((int64_t)
						(*pvt->_outbindvars)[count].
							value.dateval.month);
				pvt->_sqlrc->debugPrint("-");
				pvt->_sqlrc->debugPrint((int64_t)
						(*pvt->_outbindvars)[count].
							value.dateval.day);
				pvt->_sqlrc->debugPrint(" ");
				pvt->_sqlrc->debugPrint((int64_t)
						(*pvt->_outbindvars)[count].
							value.dateval.hour);
				pvt->_sqlrc->debugPrint(":");
				pvt->_sqlrc->debugPrint((int64_t)
						(*pvt->_outbindvars)[count].
							value.dateval.minute);
				pvt->_sqlrc->debugPrint(":");
				pvt->_sqlrc->debugPrint((int64_t)
						(*pvt->_outbindvars)[count].
							value.dateval.second);
				pvt->_sqlrc->debugPrint(" ");
				pvt->_sqlrc->debugPrint(
						(*pvt->_outbindvars)[count].
							value.dateval.tz);
			} else {
				pvt->_sqlrc->debugPrint(
						(*pvt->_outbindvars)[count].
							value.stringval);
			}
			pvt->_sqlrc->debugPrint("\n");
			pvt->_sqlrc->debugPreEnd();
		}

		count++;
	}

	// cache the output binds
	cacheOutputBinds(count);

	return true;
}

bool sqlrcursor::parseData() {

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Parsing Data\n");
		pvt->_sqlrc->debugPreEnd();
	}

	// if we're already at the end of the result set, then just return
	if (pvt->_endofresultset) {
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint(
				"Already at the end of the result set\n");
			pvt->_sqlrc->debugPreEnd();
		}
		return true;
	}

	// useful variables
	uint16_t		type;
	uint32_t		length;
	char			*buffer=NULL;
	uint32_t		colindex=0;
	sqlrclientcolumn	*currentcol;
	sqlrclientrow		*currentrow=NULL;
	bool			firstrow=true;

	// set firstrowindex to the index of the first row in the buffer
	pvt->_firstrowindex=pvt->_rowcount;

	// keep track of how large the buffer is
	uint64_t	rowbuffercount=0;

	// get rows
	for (;;) {

		// get the type of the field
		if (getShort(&type)!=sizeof(uint16_t)) {
			setError("Failed to get the field type.\n"
				"A network error may have occurred");
			return false;
		}

		// check for the end of the result set
		if (type==END_RESULT_SET) {

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint(
						"Got end of result set.\n");
				pvt->_sqlrc->debugPreEnd();
			}
			pvt->_endofresultset=true;

			// if we were stepping through a cached result set
			// then we need to close the file
			clearCacheSource();
			break;
		} 

		// if we're on the first column, start a new row,
		// reset the column pointer, and increment the
		// buffer counter and total row counter
		if (colindex==0) {

			if (rowbuffercount<OPTIMISTIC_ROW_COUNT) {
				if (!pvt->_rows) {
					createRowBuffers();
				}
				currentrow=pvt->_rows[rowbuffercount];
			} else {
				if (pvt->_sqlrc->debug()) {
					pvt->_sqlrc->debugPreStart();
					pvt->_sqlrc->debugPrint(
						"Creating extra rows.\n");
					pvt->_sqlrc->debugPreEnd();
				}
				if (!pvt->_firstextrarow) {
					currentrow=new sqlrclientrow(
								pvt->_colcount);
					pvt->_firstextrarow=currentrow;
				} else {
					currentrow->next=new sqlrclientrow(
								pvt->_colcount);
					currentrow=currentrow->next;
				}
			}
			if (pvt->_colcount>currentrow->colcount) {
				currentrow->resize(pvt->_colcount);
			}

			rowbuffercount++;
			pvt->_rowcount++;
		}

		if (type==NULL_DATA) {

			// handle null data
			if (pvt->_returnnulls) {
				buffer=NULL;
			} else {
				buffer=(char *)pvt->_rowstorage->allocate(1);
				buffer[0]='\0';
			}
			length=0;

		} else if (type==STRING_DATA) {
		
			// handle non-null data
			if (getLong(&length)!=sizeof(uint32_t)) {
				setError("Failed to get the field length.\n"
					"A network error may have occurred");
				return false;
			}

			// for non-long, non-NULL datatypes...
			// get the field into a buffer
			buffer=(char *)pvt->_rowstorage->allocate(length+1);
			if ((uint32_t)getString(buffer,length)!=length) {
				setError("Failed to get the field data.\n"
					"A network error may have occurred");
				return false;
			}
			buffer[length]='\0';

		} else if (type==START_LONG_DATA) {

			uint64_t	totallength;
			if (getLongLong(&totallength)!=sizeof(uint64_t)) {
				setError("Failed to get total length.\n"
					"A network error may have occurred");
				return false;
			}

			// create a buffer to hold the data
			buffer=new char[totallength+1];

			// handle a long datatype
			uint64_t	offset=0;
			for (;;) {

				// get the type of the chunk
				if (getShort(&type)!=sizeof(uint16_t)) {
					delete[] buffer;
					setError("Failed to get chunk type.\n"
						"A network error may have "
						"occurred");
					return false;
				}

				// check to see if we're done
				if (type==END_LONG_DATA) {
					break;
				}

				// get the length of the chunk
				if (getLong(&length)!=sizeof(uint32_t)) {
					delete[] buffer;
					setError("Failed to get chunk length.\n"
						"A network error may have "
						"occurred");
					return false;
				}

				// Oracle in particular has a function that
				// returns the number of characters in a CLOB,
				// but not the number of bytes.  Since
				// varying-width character data can be stored
				// in a CLOB, characters may be less than bytes.
				// AFAIK, there's no way to get the number of
				// bytes.  So, we use the number of characters
				// as a starting point, and extend buffer if
				// necessary.
				if (offset+length>totallength) {
					char	*newbuffer=
						new char[offset+length+1];
					bytestring::copy(
						newbuffer,buffer,offset);
					delete[] buffer;
					buffer=newbuffer;
					totallength=offset+length;
				}

				// get the chunk of data
				if ((uint32_t)getString(buffer+offset,
							length)!=length) {
					delete[] buffer;
					setError("Failed to get chunk data.\n"
						"A network error may have "
						"occurred");
					return false;
				}

				offset=offset+length;
			}
			// NULL terminate the buffer.  This makes 
			// certain operations safer and won't hurt
			// since the actual length (which doesn't
			// include the NULL) is available from
			// getFieldLength.
			buffer[totallength]='\0';
			length=totallength;
		}

		// add the buffer to the current row
		currentrow->addField(colindex,buffer,length);
	
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			if (buffer) {
				if (type==END_LONG_DATA) {
					pvt->_sqlrc->debugPrint("\nLOB data:");
					pvt->_sqlrc->debugPrintBlob(
							buffer,length);
				} else {
					pvt->_sqlrc->debugPrint("\"");
					pvt->_sqlrc->debugPrint(buffer);
					pvt->_sqlrc->debugPrint("\",");
				}
			} else {
				pvt->_sqlrc->debugPrint(buffer);
				pvt->_sqlrc->debugPrint(",");
			}
			pvt->_sqlrc->debugPreEnd();
		}

		// tag the column as a long data type or not
		currentcol=getColumnInternal(colindex);

		// set whether this column is a "long type" or not
		// (unless it's already set)
		if (firstrow || !currentcol->longdatatype) {
			currentcol->longdatatype=(type==END_LONG_DATA)?1:0;
		}

		if (pvt->_sendcolumninfo==SEND_COLUMN_INFO && 
				pvt->_sentcolumninfo==SEND_COLUMN_INFO) {

			// keep track of the longest field
			if (length>currentcol->longest) {
				currentcol->longest=length;
			}
		}

		// move to the next column, handle end of row 
		colindex++;
		if (colindex==pvt->_colcount) {

			colindex=0;

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint("\n");
				pvt->_sqlrc->debugPreEnd();
			}

			// check to see if we've gotten enough rows
			if (pvt->_rsbuffersize &&
				rowbuffercount==pvt->_rsbuffersize) {
				break;
			}

			firstrow=false;
		}
	}

	// terminate the row list
	if (rowbuffercount>=OPTIMISTIC_ROW_COUNT && currentrow) {
		currentrow->next=NULL;
		createExtraRowArray();
	}

	// cache the rows
	cacheData();

	return true;
}

void sqlrcursor::createRowBuffers() {

	// rows will hang around from now until the cursor is deleted,
	// getting reused with each query
	pvt->_rows=new sqlrclientrow *[OPTIMISTIC_ROW_COUNT];
	for (uint64_t i=0; i<OPTIMISTIC_ROW_COUNT; i++) {
		pvt->_rows[i]=new sqlrclientrow(pvt->_colcount);
	}
}

void sqlrcursor::createExtraRowArray() {

	// create the arrays
	uint64_t	howmany=pvt->_rowcount-
				pvt->_firstrowindex-
				OPTIMISTIC_ROW_COUNT;
	pvt->_extrarows=new sqlrclientrow *[howmany];
	
	// populate the arrays
	sqlrclientrow	*currentrow=pvt->_firstextrarow;
	for (uint64_t i=0; i<howmany; i++) {
		pvt->_extrarows[i]=currentrow;
		currentrow=currentrow->next;
	}
}

void sqlrcursor::getErrorFromServer() {

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Getting Error From Server\n");
		pvt->_sqlrc->debugPreEnd();
	}

	bool	networkerror=true;

	// get the error code
	if (getLongLong((uint64_t *)&pvt->_errorno)==sizeof(uint64_t)) {

		// get the length of the error string
		uint16_t	length;
		if (getShort(&length)==sizeof(uint16_t)) {

			// get the error string
			pvt->_error=new char[length+1];
			pvt->_sqlrc->cs()->read(pvt->_error,length);
			pvt->_error[length]='\0';

			networkerror=false;
		}
	}

	if (networkerror) {
		setError("There was an error, but the connection"
				" died trying to retrieve it.  Sorry.");
	}
	
	handleError();
}

void sqlrcursor::setError(const char *err) {

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Setting Error\n");
		pvt->_sqlrc->debugPreEnd();
	}
	pvt->_error=charstring::duplicate(err);
	handleError();
}

void sqlrcursor::handleError() {

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint((int64_t)pvt->_errorno);
		pvt->_sqlrc->debugPrint(":\n");
		pvt->_sqlrc->debugPrint(pvt->_error);
		pvt->_sqlrc->debugPrint("\n");
		pvt->_sqlrc->debugPreEnd();
	}

	pvt->_endofresultset=true;

	cacheError();
	finishCaching();
}

bool sqlrcursor::fetchRowIntoBuffer(bool getallrows, uint64_t row,
						uint64_t *rowbufferindex) {

	// if we getting the entire result set at once, then the result set 
	// buffer index is the requested row-pvt->_firstrowindex
	if (!pvt->_rsbuffersize) {
		if (row<pvt->_rowcount && row>=pvt->_firstrowindex) {
			*rowbufferindex=row-pvt->_firstrowindex;
			return true;
		}
		return false;
	}

	// but, if we're not getting the entire result set at once
	// and if the requested row is not in the current range, 
	// fetch more data from the connection
	while (row>=(pvt->_firstrowindex+pvt->_rsbuffersize) &&
					!pvt->_endofresultset) {

		if (pvt->_sqlrc->connected() ||
			(pvt->_cachesource && pvt->_cachesourceind)) {

			clearRows();

			// if we're not fetching from a cached result set,
			// tell the server to send one 
			if (!pvt->_cachesource && !pvt->_cachesourceind) {
				pvt->_sqlrc->cs()->write(
						(uint16_t)FETCH_RESULT_SET);
				pvt->_sqlrc->cs()->write(
						pvt->_cursorid);
			}

			if (!skipAndFetch(getallrows,row) || !parseData()) {
				return false;
			}

		} else {
			return false;
		}
	}

	// return the buffer index corresponding to the requested row
	// or -1 if the requested row is past the end of the result set
	if (row<pvt->_rowcount) {
		*rowbufferindex=row%pvt->_rsbuffersize;
		return true;
	}
	return false;
}

int32_t sqlrcursor::getShort(uint16_t *integer,
				int32_t timeoutsec, int32_t timeoutusec) {

	// if the result set is coming from a cache file, read from
	// the file, if not, read from the server
	if (pvt->_cachesource && pvt->_cachesourceind) {
		return pvt->_cachesource->read(integer);
	} else {
		return pvt->_sqlrc->cs()->read(integer,timeoutsec,timeoutusec);
	}
}

int32_t sqlrcursor::getShort(uint16_t *integer) {

	// if the result set is coming from a cache file, read from
	// the file, if not, read from the server
	if (pvt->_cachesource && pvt->_cachesourceind) {
		return pvt->_cachesource->read(integer);
	} else {
		return pvt->_sqlrc->cs()->read(integer);
	}
}

int32_t sqlrcursor::getLong(uint32_t *integer) {

	// if the result set is coming from a cache file, read from
	// the file, if not, read from the server
	if (pvt->_cachesource && pvt->_cachesourceind) {
		return pvt->_cachesource->read(integer);
	} else {
		return pvt->_sqlrc->cs()->read(integer);
	}
}

int32_t sqlrcursor::getLongLong(uint64_t *integer) {

	// if the result set is coming from a cache file, read from
	// the file, if not, read from the server
	if (pvt->_cachesource && pvt->_cachesourceind) {
		return pvt->_cachesource->read(integer);
	} else {
		return pvt->_sqlrc->cs()->read(integer);
	}
}

int32_t sqlrcursor::getString(char *string, int32_t size) {

	// if the result set is coming from a cache file, read from
	// the file, if not, read from the server
	if (pvt->_cachesource && pvt->_cachesourceind) {
		return pvt->_cachesource->read(string,size);
	} else {
		return pvt->_sqlrc->cs()->read(string,size);
	}
}

int32_t sqlrcursor::getDouble(double *value) {

	// if the result set is coming from a cache file, read from
	// the file, if not, read from the server
	if (pvt->_cachesource && pvt->_cachesourceind) {
		return pvt->_cachesource->read(value);
	} else {
		return pvt->_sqlrc->cs()->read(value);
	}
}

bool sqlrcursor::fetchFromBindCursor() {

	if (!pvt->_endofresultset || !pvt->_sqlrc->connected()) {
		return false;
	}

	// FIXME: should these be here?
	clearVariables();
	clearResultSet();

	pvt->_cached=false;
	pvt->_endofresultset=false;

	// tell the server we're fetching from a bind cursor
	pvt->_sqlrc->cs()->write((uint16_t)FETCH_FROM_BIND_CURSOR);

	// send the cursor id to the server
	pvt->_sqlrc->cs()->write((uint16_t)pvt->_cursorid);

	sendGetColumnInfo();

	pvt->_sqlrc->flushWriteBuffer();

	if (pvt->_rsbuffersize) {
		return processResultSet(false,pvt->_rsbuffersize-1);
	} else {
		return processResultSet(true,0);
	}
}

bool sqlrcursor::openCachedResultSet(const char *filename) {

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Opening cached result set: ");
		pvt->_sqlrc->debugPrint(filename);
		pvt->_sqlrc->debugPrint("\n");
		pvt->_sqlrc->debugPreEnd();
	}

	if (!pvt->_endofresultset) {
		closeResultSet(true);
	}
	clearResultSet();

	pvt->_cached=true;
	pvt->_endofresultset=false;

	// create the index file name
	size_t	indexfilenamelen=charstring::length(filename)+5;
	char	*indexfilename=new char[indexfilenamelen];
	charstring::copy(indexfilename,filename);
	charstring::append(indexfilename,".ind");

	// open the file
	pvt->_cachesource=new file();
	pvt->_cachesourceind=new file();
	if ((pvt->_cachesource->open(filename,O_RDWR)) &&
		(pvt->_cachesourceind->open(indexfilename,O_RDWR))) {

		delete[] indexfilename;

		// initialize firstrowindex and rowcount
		pvt->_firstrowindex=0;
		pvt->_rowcount=pvt->_firstrowindex;

		// make sure it's a cache file and skip the ttl
		char		magicid[13];
		uint64_t	ttl;
		if (getString(magicid,13)==13 &&
			!charstring::compare(magicid,"SQLRELAYCACHE",13) &&
			getLongLong(&ttl)==sizeof(uint64_t)) {

			// process the result set
			if (pvt->_rsbuffersize) {
				return processResultSet(false,
						pvt->_firstrowindex+
							pvt->_rsbuffersize-1);
			} else {
				return processResultSet(true,0);
			}
		} else {

			// if the test above failed, the file is either not
			// a cache file or is corrupt
			stringbuffer	errstr;
			errstr.append("File ");
			errstr.append(filename);
			errstr.append(" is either corrupt");
			errstr.append(" or not a cache file.");
			setError(errstr.getString());
		}

	} else {

		// if we couldn't open the file, set the error message
		stringbuffer	errstr;
		errstr.append("Couldn't open ");
		errstr.append(filename);
		errstr.append(" and ");
		errstr.append(indexfilename);
		setError(errstr.getString());

		delete[] indexfilename;
	}

	// if we fell through to here, then an error has ocurred
	clearCacheSource();
	return false;
}

void sqlrcursor::clearCacheSource() {
	if (pvt->_cachesource) {
		pvt->_cachesource->close();
		delete pvt->_cachesource;
		pvt->_cachesource=NULL;
	}
	if (pvt->_cachesourceind) {
		pvt->_cachesourceind->close();
		delete pvt->_cachesourceind;
		pvt->_cachesourceind=NULL;
	}
}

uint32_t sqlrcursor::colCount() {
	return pvt->_colcount;
}

sqlrclientcolumn *sqlrcursor::getColumn(uint32_t index) {
	if (pvt->_sendcolumninfo==SEND_COLUMN_INFO && 
			pvt->_sentcolumninfo==SEND_COLUMN_INFO &&
			pvt->_colcount && index<pvt->_colcount) {
		return getColumnInternal(index);
	}
	return NULL;
}

sqlrclientcolumn *sqlrcursor::getColumn(const char *name) {
	if (pvt->_sendcolumninfo==SEND_COLUMN_INFO && 
			pvt->_sentcolumninfo==SEND_COLUMN_INFO) {
		sqlrclientcolumn	*whichcolumn;
		for (uint32_t i=0; i<pvt->_colcount; i++) {
			whichcolumn=getColumnInternal(i);
			if (!charstring::compareIgnoringCase(
						whichcolumn->name,name)) {
				return whichcolumn;
			}
		}
	}
	return NULL;
}

sqlrclientcolumn *sqlrcursor::getColumnInternal(uint32_t index) {
	if (index<OPTIMISTIC_COLUMN_COUNT) {
		return &pvt->_columns[index];
	}
	return &pvt->_extracolumns[index-OPTIMISTIC_COLUMN_COUNT];
}

const char * const *sqlrcursor::getColumnNames() {

	if (pvt->_sendcolumninfo==DONT_SEND_COLUMN_INFO ||
			pvt->_sentcolumninfo==DONT_SEND_COLUMN_INFO) {
		return NULL;
	}

	if (!pvt->_columnnamearray) {
		if (pvt->_sqlrc->debug()) {
			pvt->_sqlrc->debugPreStart();
			pvt->_sqlrc->debugPrint("Creating Column Arrays...\n");
			pvt->_sqlrc->debugPreEnd();
		}
	
		// build a 2d array of pointers to the column names
		pvt->_columnnamearray=new char *[pvt->_colcount+1];
		pvt->_columnnamearray[pvt->_colcount]=NULL;
		for (uint32_t i=0; i<pvt->_colcount; i++) {
			pvt->_columnnamearray[i]=getColumnInternal(i)->name;
		}
	}
	return pvt->_columnnamearray;
}

const char *sqlrcursor::getColumnName(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?whichcol->name:NULL;
}

const char *sqlrcursor::getColumnType(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	if (whichcol) {
		if (pvt->_columntypeformat!=COLUMN_TYPE_IDS) {
			return whichcol->typestring;
		} else {
			return datatypestring[whichcol->type];
		}
	}
	return NULL;
}

uint32_t sqlrcursor::getColumnLength(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?whichcol->length:0;
}

uint32_t sqlrcursor::getColumnPrecision(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?whichcol->precision:0;
}

uint32_t sqlrcursor::getColumnScale(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?whichcol->scale:0;
}

bool sqlrcursor::getColumnIsNullable(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->nullable!=0):false;
}

bool sqlrcursor::getColumnIsPrimaryKey(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->primarykey!=0):false;
}

bool sqlrcursor::getColumnIsUnique(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->unique!=0):false;
}

bool sqlrcursor::getColumnIsPartOfKey(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->partofkey!=0):false;
}

bool sqlrcursor::getColumnIsUnsigned(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->unsignednumber!=0):false;
}

bool sqlrcursor::getColumnIsZeroFilled(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->zerofill!=0):false;
}

bool sqlrcursor::getColumnIsBinary(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->binary!=0):false;
}

bool sqlrcursor::getColumnIsAutoIncrement(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->autoincrement!=0):false;
}

uint32_t sqlrcursor::getLongest(uint32_t col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?whichcol->longest:0;
}

const char *sqlrcursor::getColumnType(const char *col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	if (whichcol) {
		if (pvt->_columntypeformat!=COLUMN_TYPE_IDS) {
			return whichcol->typestring;
		} else {
			return datatypestring[whichcol->type];
		}
	}
	return NULL;
}

uint32_t sqlrcursor::getColumnLength(const char *col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?whichcol->length:0;
}

uint32_t sqlrcursor::getColumnPrecision(const char *col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?whichcol->precision:0;
}

uint32_t sqlrcursor::getColumnScale(const char *col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?whichcol->scale:0;
}

bool sqlrcursor::getColumnIsNullable(const char *col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->nullable!=0):false;
}

bool sqlrcursor::getColumnIsPrimaryKey(const char *col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->primarykey!=0):false;
}

bool sqlrcursor::getColumnIsUnique(const char *col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->unique!=0):false;
}

bool sqlrcursor::getColumnIsPartOfKey(const char *col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->partofkey!=0):false;
}

bool sqlrcursor::getColumnIsUnsigned(const char *col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->unsignednumber!=0):false;
}

bool sqlrcursor::getColumnIsZeroFilled(const char *col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->zerofill!=0):false;
}

bool sqlrcursor::getColumnIsBinary(const char *col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->binary!=0):false;
}

bool sqlrcursor::getColumnIsAutoIncrement(const char *col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?(whichcol->autoincrement!=0):false;
}


uint32_t sqlrcursor::getLongest(const char *col) {
	sqlrclientcolumn	*whichcol=getColumn(col);
	return (whichcol)?whichcol->longest:0;
}

uint64_t sqlrcursor::firstRowIndex() {
	return pvt->_firstrowindex;
}

bool sqlrcursor::endOfResultSet() {
	return pvt->_endofresultset;
}

uint64_t sqlrcursor::rowCount() {
	return pvt->_rowcount;
}

uint64_t sqlrcursor::affectedRows() {
	if (pvt->_knowsaffectedrows==AFFECTED_ROWS) {
		return pvt->_affectedrows;
	}
	return 0;
}

uint64_t sqlrcursor::totalRows() {
	if (pvt->_knowsactualrows==ACTUAL_ROWS) {
		return pvt->_actualrows;
	}
	return 0;
}

int64_t sqlrcursor::errorNumber() {
	// if we have a code then we should have a message too,
	// the codes could be any number, including 0, so check
	// the message to see which code to return
	if (pvt->_error) {
		return pvt->_errorno;
	} else if (pvt->_sqlrc->error()) {
		return pvt->_sqlrc->errorno();
	}
	return 0;
}

const char *sqlrcursor::errorMessage() {
	if (pvt->_error) {
		return pvt->_error;
	} else if (pvt->_sqlrc->error()) {
		return pvt->_sqlrc->error();
	}
	return NULL;
}

void sqlrcursor::getNullsAsEmptyStrings() {
	pvt->_returnnulls=false;
}

void sqlrcursor::getNullsAsNulls() {
	pvt->_returnnulls=true;
}

char *sqlrcursor::getFieldInternal(uint64_t row, uint32_t col) {
	if (row<OPTIMISTIC_ROW_COUNT) {
		return pvt->_rows[row]->getField(col);
	}
	return pvt->_extrarows[row-OPTIMISTIC_ROW_COUNT]->getField(col);
}

uint32_t sqlrcursor::getFieldLengthInternal(uint64_t row, uint32_t col) {
	if (row<OPTIMISTIC_ROW_COUNT) {
		return pvt->_rows[row]->getFieldLength(col);
	}
	return pvt->_extrarows[row-OPTIMISTIC_ROW_COUNT]->getFieldLength(col);
}

const char *sqlrcursor::getField(uint64_t row, uint32_t col) {

	if (pvt->_rowcount && row>=pvt->_firstrowindex && col<pvt->_colcount) {

		// in the event that we're stepping through the result set 
		// instead of buffering the entire thing, the requested row
		// may have to be fetched into the buffer...
		uint64_t	rowbufferindex;
		if (fetchRowIntoBuffer(false,row,&rowbufferindex)) {
			return getFieldInternal(rowbufferindex,col);
		}
	}
	return NULL;
}

int64_t sqlrcursor::getFieldAsInteger(uint64_t row, uint32_t col) {
	const char	*field=getField(row,col);
	return (field)?charstring::toInteger(field):0;
}

double sqlrcursor::getFieldAsDouble(uint64_t row, uint32_t col) {
	const char	*field=getField(row,col);
	return (field)?charstring::toFloat(field):0.0;
}

const char *sqlrcursor::getField(uint64_t row, const char *col) {

	if (pvt->_sendcolumninfo==SEND_COLUMN_INFO && 
			pvt->_sentcolumninfo==SEND_COLUMN_INFO &&
			pvt->_rowcount && row>=pvt->_firstrowindex) {
		for (uint32_t i=0; i<pvt->_colcount; i++) {
			if (!charstring::compareIgnoringCase(
					getColumnInternal(i)->name,col)) {

				// in the event that we're stepping through the
				// result set instead of buffering the entire 
				// thing, the requested row may have to be 
				// fetched into the buffer...
				uint64_t	rowbufferindex;
				if (fetchRowIntoBuffer(false,row,
							&rowbufferindex)) {
					return getFieldInternal(
							rowbufferindex,i);
				}
				return NULL;
			}
		}
	}
	return NULL;
}

int64_t sqlrcursor::getFieldAsInteger(uint64_t row, const char *col) {
	const char	*field=getField(row,col);
	return (field)?charstring::toInteger(field):0;
}

double sqlrcursor::getFieldAsDouble(uint64_t row, const char *col) {
	const char	*field=getField(row,col);
	return (field)?charstring::toFloat(field):0.0;
}

uint32_t sqlrcursor::getFieldLength(uint64_t row, uint32_t col) {

	if (pvt->_rowcount && row>=pvt->_firstrowindex && col<pvt->_colcount) {

		// in the event that we're stepping through the result set 
		// instead of buffering the entire thing, the requested row
		// may have to be fetched into the buffer...
		uint64_t	rowbufferindex;
		if (fetchRowIntoBuffer(false,row,&rowbufferindex)) {
			return getFieldLengthInternal(rowbufferindex,col);
		}
	}
	return 0;
}

uint32_t sqlrcursor::getFieldLength(uint64_t row, const char *col) {

	if (pvt->_sendcolumninfo==SEND_COLUMN_INFO && 
			pvt->_sentcolumninfo==SEND_COLUMN_INFO &&
			pvt->_rowcount && row>=pvt->_firstrowindex) {

		for (uint32_t i=0; i<pvt->_colcount; i++) {
			if (!charstring::compareIgnoringCase(
					getColumnInternal(i)->name,col)) {

				// in the event that we're stepping through the
				// result set instead of buffering the entire 
				// thing, the requested row may have to be 
				// fetched into the buffer...
				uint64_t	rowbufferindex;
				if (fetchRowIntoBuffer(false,row,
							&rowbufferindex)) {
					return getFieldLengthInternal(
							rowbufferindex,i);
				}
				return 0;
			}
		}
	}
	return 0;
}

const char * const *sqlrcursor::getRow(uint64_t row) {

	if (pvt->_rowcount && row>=pvt->_firstrowindex) {

		// in the event that we're stepping through the result set 
		// instead of buffering the entire thing, the requested row
		// may have to be fetched into the buffer...
		uint64_t	rowbufferindex;
		if (fetchRowIntoBuffer(false,row,&rowbufferindex)) {
			if (!pvt->_fields) {
				createFields();
			}
			return pvt->_fields[rowbufferindex];
		}
	}
	return NULL;
}

void sqlrcursor::createFields() {
	// lets say that rowcount=5 and firstrowindex=3,
	// the fields array will contain 2 elements:
	// 	fields[0] (corresponding to row 3) and
	// 	fields[1] (corresponding to row 4)
	uint64_t	rowbuffercount=pvt->_rowcount-pvt->_firstrowindex;
	pvt->_fields=new char **[rowbuffercount+1];
	pvt->_fields[rowbuffercount]=(char **)NULL;
	for (uint64_t i=0; i<rowbuffercount; i++) {
		pvt->_fields[i]=new char *[pvt->_colcount+1];
		pvt->_fields[i][pvt->_colcount]=(char *)NULL;
		for (uint32_t j=0; j<pvt->_colcount; j++) {
			pvt->_fields[i][j]=getFieldInternal(i,j);
		}
	}
}

uint32_t *sqlrcursor::getRowLengths(uint64_t row) {

	if (pvt->_rowcount && row>=pvt->_firstrowindex) {

		// in the event that we're stepping through the result set 
		// instead of buffering the entire thing, the requested row
		// may have to be fetched into the buffer...
		uint64_t	rowbufferindex;
		if (fetchRowIntoBuffer(false,row,&rowbufferindex)) {
			if (!pvt->_fieldlengths) {
				createFieldLengths();
			}
			return pvt->_fieldlengths[rowbufferindex];
		}
	}
	return NULL;
}

void sqlrcursor::createFieldLengths() {
	// lets say that rowcount=5 and firstrowindex=3,
	// the fieldlengths array will contain 2 elements:
	// 	fieldlengths[0] (corresponding to row 3) and
	// 	fieldlengths[1] (corresponding to row 4)
	uint64_t	rowbuffercount=pvt->_rowcount-pvt->_firstrowindex;
	pvt->_fieldlengths=new uint32_t *[rowbuffercount+1];
	pvt->_fieldlengths[rowbuffercount]=0;
	for (uint64_t i=0; i<rowbuffercount; i++) {
		pvt->_fieldlengths[i]=new uint32_t[pvt->_colcount+1];
		pvt->_fieldlengths[i][pvt->_colcount]=0;
		for (uint32_t j=0; j<pvt->_colcount; j++) {
			pvt->_fieldlengths[i][j]=getFieldLengthInternal(i,j);
		}
	}
}

void sqlrcursor::suspendResultSet() {

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Suspending Result Set\n");
		pvt->_sqlrc->debugPreEnd();
	}

	if (pvt->_sqlrc->connected() && !pvt->_cached) {

		pvt->_sqlrc->cs()->write((uint16_t)SUSPEND_RESULT_SET);
		pvt->_sqlrc->cs()->write(pvt->_cursorid);

		pvt->_sqlrc->flushWriteBuffer();
	}

	clearCacheDest();
	pvt->_suspendresultsetsent=1;
}

uint16_t sqlrcursor::getResultSetId() {
	return pvt->_cursorid;
}

bool sqlrcursor::resumeResultSet(uint16_t id) {
	return resumeCachedResultSet(id,NULL);
}

bool sqlrcursor::resumeCachedResultSet(uint16_t id, const char *filename) {

	if (!pvt->_endofresultset && !pvt->_suspendresultsetsent) {
		closeResultSet(false);
	}
	clearResultSet();

	if (!pvt->_sqlrc->connected()) {
		return false;
	}

	pvt->_cached=false;
	pvt->_resumed=true;
	pvt->_endofresultset=false;

	if (pvt->_sqlrc->debug()) {
		pvt->_sqlrc->debugPreStart();
		pvt->_sqlrc->debugPrint("Resuming Result Set of Cursor: ");
		pvt->_sqlrc->debugPrint((int64_t)id);
		pvt->_sqlrc->debugPrint("\n");
		pvt->_sqlrc->debugPreEnd();
	}

	// tell the server we want to resume the result set
	pvt->_sqlrc->cs()->write((uint16_t)RESUME_RESULT_SET);

	// send the id of the cursor we want to 
	// resume the result set of to the server
	pvt->_sqlrc->cs()->write(id);

	// process the result set
	if (!charstring::isNullOrEmpty(filename)) {
		cacheToFile(filename);
	}
	if (pvt->_rsbuffersize) {
		if (processResultSet(true,
				pvt->_firstrowindex+
					pvt->_rsbuffersize-1)) {
			return true;
		}
	} else {
		if (processResultSet(false,0)) {
			return true;
		}
	}
	return false;
}

void sqlrcursor::closeResultSet() {
	closeResultSet(true);
}

void sqlrcursor::closeResultSet(bool closeremote) {

	// If the end of the previous result set was never reached, abort it.
	// If we're caching data to a local file, get the rest of the data; we
	// won't have to abort the result set in that case, the server will
	// do it.

	if (pvt->_sqlrc->connected() || pvt->_cached) {
		if (pvt->_cachedest && pvt->_cachedestind) {
			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint(
					"Getting the rest of the result set, "
					"since this is a cached result set.\n");
				pvt->_sqlrc->debugPreEnd();
			}
			while (!pvt->_endofresultset) {
				clearRows();

				// if we're not fetching from a cached result 
				// set tell the server to send one 
				if (!pvt->_cachesource &&
						!pvt->_cachesourceind) {
					pvt->_sqlrc->cs()->write((uint16_t)
							FETCH_RESULT_SET);
					pvt->_sqlrc->cs()->write(
							pvt->_cursorid);
				}

				// parseData should call finishCaching when
				// it hits the end of the result set, but
				// if it or skipAndFetch return a -1 (network
				// error) we'll have to call it ourselves.
				if (!skipAndFetch(true,0) || !parseData()) {
					finishCaching();
					return;
				}
			}
		} else if (closeremote && pvt->_havecursorid) {

			if (pvt->_sqlrc->debug()) {
				pvt->_sqlrc->debugPreStart();
				pvt->_sqlrc->debugPrint("Aborting Result "
							"Set For Cursor: ");
				pvt->_sqlrc->debugPrint(
						(int64_t)pvt->_cursorid);
				pvt->_sqlrc->debugPrint("\n");
				pvt->_sqlrc->debugPreEnd();
			}

			pvt->_sqlrc->cs()->write(
					(uint16_t)ABORT_RESULT_SET);
			pvt->_sqlrc->cs()->write(
					(uint16_t)DONT_NEED_NEW_CURSOR);
			pvt->_sqlrc->cs()->write(pvt->_cursorid);
			pvt->_sqlrc->flushWriteBuffer();
		}
	}
}

void sqlrcursor::clearResultSet() {

	clearCacheDest();
	clearCacheSource();
	clearError();

	// columns is cleared after rows because colcount is used in 
	// clearRows() and set to 0 in clearColumns()
	clearRows();
	clearColumns();

	// clear row counters, since fetchRowIntoBuffer() and clearResultSet()
	// are the only methods that call clearRows() and fetchRowIntoBuffer()
	// needs these values not to be cleared, we'll clear them here...
	pvt->_firstrowindex=0;
	pvt->_previousrowcount=pvt->_rowcount;
	pvt->_rowcount=0;
	pvt->_actualrows=0;
	pvt->_affectedrows=0;
	pvt->_endofresultset=true;
	pvt->_suspendresultsetsent=0;
}

void sqlrcursor::clearError() {
	delete[] pvt->_error;
	pvt->_error=NULL;
	pvt->_errorno=0;
	if (pvt->_sqlrc) {
		pvt->_sqlrc->clearError();
	}
}

void sqlrcursor::clearRows() {

	// delete data in rows for long datatypes
	uint32_t	rowbuffercount=pvt->_rowcount-pvt->_firstrowindex;
	for (uint32_t i=0; i<rowbuffercount; i++) {
	        for (uint32_t j=0; j<pvt->_colcount; j++) {
			if (getColumnInternal(j)->longdatatype) {
				char		*field=getFieldInternal(i,j);
				uint32_t	len=getFieldLengthInternal(i,j);
				// Null lobs might be stored as a NULL or as
				// an empty string.  In either case (and in no
				// other case) the length will be 0.  In the
				// case of a NULL there's nothing to delete.
				// In the case of an empty string, the memory
				// will be allocated from the rowstorage pool
				// and shouldn't be deallocated here.
				if (len) {
					delete[] field;
				}
			}
		}
	}

	// delete linked list storing extra result set fields
	sqlrclientrow	*currentrow;
	if (pvt->_firstextrarow) {
		currentrow=pvt->_firstextrarow;
		while (currentrow) {
			pvt->_firstextrarow=currentrow->next;
			delete currentrow;
			currentrow=pvt->_firstextrarow;
		}
		pvt->_firstextrarow=NULL;
	}
	currentrow=NULL;

	// delete array pointing to linked list items
	delete[] pvt->_extrarows;
	pvt->_extrarows=NULL;

	// delete arrays of fields and field lengths
	if (pvt->_fields) {
		for (uint32_t i=0; i<rowbuffercount; i++) {
			delete[] pvt->_fields[i];
		}
		delete[] pvt->_fields;
		pvt->_fields=NULL;
	}
	if (pvt->_fieldlengths) {
		for (uint32_t i=0; i<rowbuffercount; i++) {
			delete[] pvt->_fieldlengths[i];
		}
		delete[] pvt->_fieldlengths;
		pvt->_fieldlengths=NULL;
	}

	// reset the row storage pool
	pvt->_rowstorage->deallocate();
}

void sqlrcursor::clearColumns() {

	// delete the column type strings (if necessary)
	if (pvt->_sentcolumninfo==SEND_COLUMN_INFO &&
				pvt->_columntypeformat!=COLUMN_TYPE_IDS) {
		for (uint32_t i=0; i<pvt->_colcount; i++) {
			delete[] getColumnInternal(i)->typestring;
		}
	}

	// reset the column storage pool
	pvt->_colstorage->deallocate();

	// reset the column count
	pvt->_previouscolcount=pvt->_colcount;
	pvt->_colcount=0;

	// delete array pointing to each column name
	delete[] pvt->_columnnamearray;
	pvt->_columnnamearray=NULL;
}

char *sqlrcursor::getQueryTree() {

	pvt->_reexecute=false;
	pvt->_validatebinds=false;
	pvt->_resumed=false;
	clearVariables();

	if (!pvt->_endofresultset) {
		closeResultSet(false);
	}
	clearResultSet();

	if (!pvt->_sqlrc->openSession()) {
		return NULL;
	}

	pvt->_cached=false;
	pvt->_endofresultset=false;

	// tell the server we want to get a db list
	pvt->_sqlrc->cs()->write((uint16_t)GET_QUERY_TREE);

	// tell the server whether we'll need a cursor or not
	sendCursorStatus();

	pvt->_sqlrc->flushWriteBuffer();

	uint16_t	err=getErrorStatus();
	if (err!=NO_ERROR_OCCURRED) {
		if (err==TIMEOUT_GETTING_ERROR_STATUS) {
			pvt->_sqlrc->endSession();
			return NULL;
		}
		getErrorFromServer();
		if (err==ERROR_OCCURRED_DISCONNECT) {
			pvt->_sqlrc->endSession();
		}
		return NULL;
	}

	// get the size of the tree
	uint64_t	querytreelen;
	if (pvt->_sqlrc->cs()->read(&querytreelen)!=sizeof(uint64_t)) {
		return NULL;
	}

	// get the tree itself
	char	*querytree=new char[querytreelen+1];
	if ((uint64_t)pvt->_sqlrc->cs()->read(
				querytree,querytreelen)!=querytreelen) {
		delete[] querytree;
		return NULL;
	}
	querytree[querytreelen]='\0';

	return querytree;
}

bool sqlrcursor::endofresultset() {
	return pvt->_endofresultset;
}

void sqlrcursor::sqlrc(sqlrconnection *sqlrc) {
	pvt->_sqlrc=sqlrc;
}

sqlrcursor *sqlrcursor::next() {
	return pvt->_next;
}

void sqlrcursor::havecursorid(bool havecursorid) {
	pvt->_havecursorid=havecursorid;
}
