/*
 * Pidgin Advanced Sound Notification
 * Copyright (C) 2009 Konrad Gräfe
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02111-1301, USA.
 */

#include "config.h"

#ifndef PURPLE_PLUGINS
#define PURPLE_PLUGINS
#endif

#include "internal.h"

#include <gtkplugin.h>
#include <debug.h>
#include <version.h>
#include <request.h>
#include <sound.h>

#include <gtkutils.h>
#include <gtkblist.h>

#define PLUGIN_PREFS_PREFIX "/plugins/gtk/pidgin-advanced-sound-notification"

/*** Daten ***/
static PurplePlugin *plugin;

typedef enum {
	EVENT_TYPE_INCOMING_FILETRANSFER,
	EVENT_TYPE_FILETRANSFER_COMPLETED,
	EVENT_TYPE_FILETRANSFER_ERROR,
	EVENT_TYPE_AUTHORIZATION_REQUEST,
	EVENT_TYPE_INCOMING_MAIL
} EventType;

typedef struct {
	EventType type;
	gboolean enabled;
	gchar *prefs_path;
	gchar *description;
	gchar *filename;
	gchar *default_filename;
} Event;

static GList *events = NULL;

/*** Events ***/
static void event_occured(Event *event) {
	gchar *soundfile = NULL;
	
	if(!event || !event->enabled) return;
	
	if(event->filename && g_file_test(event->filename, G_FILE_TEST_EXISTS)) {
		/* benutzerdefinierter Sound */
		soundfile = g_strdup(event->filename);
	} else if(event->default_filename && event->default_filename != '\0') {
		/* Standardsound, im Homeverzeichnis des Benutzers */
		soundfile = g_build_filename(purple_user_dir(), "sounds", "pidgin", "pidgin-advanced-sound-notification", event->default_filename, NULL);
		if(!g_file_test(soundfile, G_FILE_TEST_EXISTS)) {
			g_free(soundfile);
			
			/* Standardsound, in DATADIR */
			soundfile = g_build_filename(DATADIR, "sounds", "pidgin", "pidgin-advanced-sound-notification", event->default_filename, NULL);
		}
	} else {
		purple_debug_error(PLUGIN_STATIC_NAME, _("No sound file specified for event \"%s\".\n"), event->description);
		return;
	}
	
	if(g_file_test(soundfile, G_FILE_TEST_EXISTS)) {
		purple_sound_play_file(soundfile, NULL);
	} else {
		purple_debug_error(PLUGIN_STATIC_NAME, _("Sound file (%s) does not exist.\n"), soundfile);
	}
	g_free(soundfile);
}

static void catch_event_with_one(gpointer dummy, gpointer event) {
	event_occured((Event *) event);
}

static gint catch_event_with_two(gpointer dummy1, gpointer dummy2, gpointer event) {
	event_occured((Event *) event);
	return 0;
}

static gint catch_event_with_four(gpointer dummy1, gpointer dummy2, gpointer dummy3, gpointer dummy4, gpointer event) {
	event_occured((Event *) event);
	return 0;
}

static gint catch_event_with_five(gpointer dummy1, gpointer dummy2, gpointer dummy3, gpointer dummy4, guint dummy5, gpointer event) {
	event_occured((Event *) event);
	return 0;
}

static void init_events() {
	Event *event = NULL;

	/** incoming filetransfer **/
	event = g_malloc(sizeof(Event));
	event->type = EVENT_TYPE_INCOMING_FILETRANSFER;
	event->enabled = purple_prefs_get_bool(PLUGIN_PREFS_PREFIX "/incoming_filetransfer/enabled"); 
	event->prefs_path = PLUGIN_PREFS_PREFIX "/incoming_filetransfer";
	event->description = _("Incoming Filetransfer");
	event->filename = g_strdup(purple_prefs_get_string(PLUGIN_PREFS_PREFIX "/incoming_filetransfer/filename"));
	event->default_filename = "filetransfer.wav";
	purple_signal_connect(purple_xfers_get_handle(), "file-recv-request", plugin, PURPLE_CALLBACK(catch_event_with_one), event);
	events = g_list_append(events, event);

	/** filetransfer completed **/
	event = g_malloc(sizeof(Event));
	event->type = EVENT_TYPE_FILETRANSFER_COMPLETED;
	event->enabled = purple_prefs_get_bool(PLUGIN_PREFS_PREFIX "/filetransfer_completed/enabled"); 
	event->prefs_path = PLUGIN_PREFS_PREFIX "/filetransfer_completed";
	event->description = _("Filetransfer completed");
	event->filename = g_strdup(purple_prefs_get_string(PLUGIN_PREFS_PREFIX "/filetransfer_completed/filename"));
	event->default_filename = "filetransfer.wav";
	purple_signal_connect(purple_xfers_get_handle(), "file-recv-complete", plugin, PURPLE_CALLBACK(catch_event_with_one), event);
	purple_signal_connect(purple_xfers_get_handle(), "file-send-complete", plugin, PURPLE_CALLBACK(catch_event_with_one), event);
	events = g_list_append(events, event);

	/** filetransfer error **/
	event = g_malloc(sizeof(Event));
	event->type = EVENT_TYPE_FILETRANSFER_ERROR;
	event->enabled = purple_prefs_get_bool(PLUGIN_PREFS_PREFIX "/filetransfer_error/enabled"); 
	event->prefs_path = PLUGIN_PREFS_PREFIX "/filetransfer_error";
	event->description = _("Filetransfer error");
	event->filename = g_strdup(purple_prefs_get_string(PLUGIN_PREFS_PREFIX "/filetransfer_error/filename"));
	event->default_filename = "filetransfer.wav";
	purple_signal_connect(purple_xfers_get_handle(), "file-recv-cancel", plugin, PURPLE_CALLBACK(catch_event_with_one), event);
	purple_signal_connect(purple_xfers_get_handle(), "file-send-cancel", plugin, PURPLE_CALLBACK(catch_event_with_one), event);
	events = g_list_append(events, event);

	/** authorization request **/
	event = g_malloc(sizeof(Event));
	event->type = EVENT_TYPE_AUTHORIZATION_REQUEST;
	event->enabled = purple_prefs_get_bool(PLUGIN_PREFS_PREFIX "/authorization_request/enabled"); 
	event->prefs_path = PLUGIN_PREFS_PREFIX "/authorization_request";
	event->description = _("Authorization request");
	event->filename = g_strdup(purple_prefs_get_string(PLUGIN_PREFS_PREFIX "/authorization_request/filename"));
	event->default_filename = "authorization.wav";
	purple_signal_connect(purple_accounts_get_handle(), "account-authorization-requested", plugin, PURPLE_CALLBACK(catch_event_with_two), event);
	events = g_list_append(events, event);

	/** incoming mail **/
	event = g_malloc(sizeof(Event));
	event->type = EVENT_TYPE_INCOMING_MAIL;
	event->enabled = purple_prefs_get_bool(PLUGIN_PREFS_PREFIX "/incoming_mail/enabled"); 
	event->prefs_path = PLUGIN_PREFS_PREFIX "/incoming_mail";
	event->description = _("Incoming mail");
	event->filename = g_strdup(purple_prefs_get_string(PLUGIN_PREFS_PREFIX "/incoming_mail/filename"));
	event->default_filename = "email.wav";
	purple_signal_connect(purple_notify_get_handle(), "displaying-email-notification", plugin, PURPLE_CALLBACK(catch_event_with_four), event);
	purple_signal_connect(purple_notify_get_handle(), "displaying-emails-notification", plugin, PURPLE_CALLBACK(catch_event_with_five), event);
	events = g_list_append(events, event);
	
	
	
	/* Bei neuen Events "init_plugin()" nicht vergessen! ;) */
}

static void free_events() {
	Event *event;
	GList *iter;

	purple_signals_disconnect_by_handle(plugin);

	iter = events;
	while(iter) {
		event = iter->data;

		g_free(event->filename);
		g_free(event);

		iter = iter->next;
		events = g_list_remove(events, event);
	}
}

/*** Einstellungen ***/
enum notification_list_columns {
	NOTIFICATION_LIST_COL_ENABLED,
	NOTIFICATION_LIST_COL_DESCRIPTION,
	NOTIFICATION_LIST_COL_DATA,
	NOTIFICATION_LIST_COL_TYPE
};

typedef struct {
	GtkWidget *treeview;
	GtkWidget *filename_entry;
	GtkWidget *infobox;

	GtkTreeModel *model;
} ConfigFrame;

static ConfigFrame config_frame;

static gboolean get_selected_event(Event **event) {
	GtkTreeSelection *selection = NULL;
	GtkTreeIter iter;

	if(!config_frame.treeview) return FALSE;

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(config_frame.treeview));
	if(!selection) return FALSE;

	if(!gtk_tree_selection_get_selected(selection, &(config_frame.model), &iter)) return FALSE;

	gtk_tree_model_get(config_frame.model, &iter, NOTIFICATION_LIST_COL_DATA, event, -1);
	return TRUE;
}

static void config_frame_update_filename_entry() {
	Event *event = NULL;

	if(get_selected_event(&event) && event->filename && *(event->filename) != '\0') {
		gtk_entry_set_text(GTK_ENTRY(config_frame.filename_entry), event->filename);
	} else {
		gtk_entry_set_text(GTK_ENTRY(config_frame.filename_entry), _("(Default)"));
	}
}

static void event_toggled(GtkCellRendererToggle *renderer, gchar *path, gpointer data) {
	GtkTreeIter iter;
	GtkTreePath *treepath = gtk_tree_path_new_from_string(path);
	Event *event = NULL;
	gchar *pref;

	gtk_tree_model_get_iter(config_frame.model, &iter, treepath);
	gtk_tree_model_get(config_frame.model, &iter, NOTIFICATION_LIST_COL_DATA, &event, -1);

	event->enabled = !gtk_cell_renderer_toggle_get_active(renderer);
	
	pref = g_strdup_printf("%s/enabled", event->prefs_path);
	purple_prefs_set_bool(pref, event->enabled);
	g_free(pref);

	gtk_list_store_set(GTK_LIST_STORE(config_frame.model), &iter, NOTIFICATION_LIST_COL_ENABLED, event->enabled, -1);
}

static void sound_selected_cb(gpointer data, const gchar* filename) {
	Event *event = (Event *) data;
	gchar *pref;

	if(!event) return;

	if(!event->filename) g_free(event->filename);
	event->filename = g_strdup(filename);

	pref = g_strdup_printf("%s/filename", event->prefs_path);
	purple_prefs_set_string(pref, event->filename);
	g_free(pref);

	config_frame_update_filename_entry();
}

static void select_sound(GtkWidget *button, gpointer data) {
	Event *event = NULL;
	gchar *filename;

	if(!get_selected_event(&event)) return;

	if(event->filename && *(event->filename) != '\0') {
		filename = event->filename;
	} else {
		filename = NULL;
	}
	purple_request_file(plugin, _("Sound Selection"), filename, FALSE, G_CALLBACK(sound_selected_cb), NULL, NULL, NULL, NULL, event);
}

static void test_sound(GtkWidget *button, gpointer data) {
	Event *event = NULL;
	gboolean enabled;

	if(!get_selected_event(&event)) return;
	enabled = event->enabled;
	

	event->enabled = TRUE;
	event_occured(event);
	event->enabled = enabled;
}

static void reset_sound(GtkWidget *button, gpointer data) {
	Event *event = NULL;
	if(!get_selected_event(&event)) return;

	sound_selected_cb(event, "");
}

static GtkWidget *make_info_widget(gchar *markup, gchar *stock_id, gboolean indent) {
	GtkWidget *infobox, *label, *img, *align;

	if(!markup) return NULL;

	infobox = gtk_hbox_new(FALSE, 5);

	if(indent) {
		label = gtk_label_new("");
		gtk_box_pack_start(GTK_BOX(infobox), label, FALSE, FALSE, 10);
	}

	if(stock_id) {
		align = gtk_alignment_new(0.5, 0, 0, 0); /* align img to the top of the space */
		gtk_box_pack_start(GTK_BOX(infobox), align, FALSE, FALSE, 0);

		img = gtk_image_new_from_stock(stock_id, GTK_ICON_SIZE_MENU);
		gtk_container_add(GTK_CONTAINER(align), img);
	}

	label = gtk_label_new(NULL);
	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
	gtk_label_set_markup(GTK_LABEL(label), markup);
	gtk_box_pack_start(GTK_BOX(infobox), label, FALSE, FALSE, 0);

	return infobox;
}

static void update_muted_sound_hint(GtkWidget *hint) {

	if(!hint) return;

	if(purple_prefs_get_bool(PIDGIN_PREFS_ROOT "/sound/mute")) {
		gtk_widget_show(hint);
	} else {
		gtk_widget_hide(hint);
	}
}

static void update_muted_sound_hint_pidgin_pref_cb(const gchar *prefname, PurplePrefType type, gconstpointer value, gpointer data) {
	GtkWidget *hint = (GtkWidget *)data;

	/* this is not a good but a simple way avoid GTK errors (signal should be disconnected instead) */
	if(GTK_IS_WIDGET(hint)) {
		update_muted_sound_hint((GtkWidget *)data);
	}
}

static void update_muted_sound_hint_show_cb(GtkWidget *widget, GdkEventFocus *event, gpointer data) {
	static gboolean inuse = FALSE;

	if(inuse) return;

	inuse = TRUE;
	update_muted_sound_hint(widget);
	inuse = FALSE;
}

static GtkWidget *get_config_frame(PurplePlugin *plugin) {
	GtkWidget *ret = NULL;
	GtkWidget *vbox = NULL;
	GtkWidget *hbox = NULL;
	GtkWidget *scrolled_window = NULL;
	GtkWidget *treeview = NULL;
	GtkWidget *entry = NULL;
	GtkWidget *button = NULL;

	GtkTreeSelection *selection = NULL;
	GtkListStore *model;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
	GtkTreeIter iter;
	GtkTreeSelection *sel;
	GtkTreePath *path;

	Event *event;
	GList *l;

	ret = gtk_vbox_new(FALSE, PIDGIN_HIG_CAT_SPACE);
	gtk_container_set_border_width(GTK_CONTAINER(ret), 12);

	vbox = gtk_vbox_new(FALSE, PIDGIN_HIG_BOX_SPACE);
	gtk_box_pack_start(GTK_BOX(ret), vbox, TRUE, TRUE, 0);

	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_set_size_request(scrolled_window, 500, 200);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled_window), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled_window, TRUE, TRUE, 0);

	model = gtk_list_store_new(4,
				   G_TYPE_BOOLEAN,	/* Sound aktiviert? */
				   G_TYPE_STRING,	/* Beschreibung des Ereignisses */
				   G_TYPE_POINTER,	/* Zeiger auf das Event */
				   G_TYPE_INT		/* Typ des Events -> zum Sortieren */
				 );
	
	config_frame.model = GTK_TREE_MODEL(model);

	gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(model), NOTIFICATION_LIST_COL_TYPE, GTK_SORT_ASCENDING);

	/* Liste füllen... */
	for(l = events; l; l = l->next) {
		event = l->data;
		
		gtk_list_store_append(model, &iter);
		gtk_list_store_set(model, &iter,
				   NOTIFICATION_LIST_COL_ENABLED, event->enabled,
				   NOTIFICATION_LIST_COL_DESCRIPTION, event->description,
				   NOTIFICATION_LIST_COL_DATA, event,
				   NOTIFICATION_LIST_COL_TYPE, event->type,
				   -1
				  );
	}

	treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(model));

	config_frame.treeview = treeview;

	path = gtk_tree_path_new_first();
	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
	gtk_tree_selection_select_path(sel, path);
	gtk_tree_path_free(path);

	/* Spalte "Sound aktivert?" */
	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(column, _("Play"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);

	renderer = gtk_cell_renderer_toggle_new();
	gtk_tree_view_column_pack_start(column, renderer, FALSE);
	gtk_tree_view_column_add_attribute(column, renderer, "active", NOTIFICATION_LIST_COL_ENABLED);

	g_signal_connect(G_OBJECT(renderer), "toggled", G_CALLBACK(event_toggled), NULL);

	/* Spalte "Beschreibung" */
	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(column, _("Event"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(column, renderer, FALSE);
	gtk_tree_view_column_add_attribute(column, renderer, "text", NOTIFICATION_LIST_COL_DESCRIPTION);

	gtk_tree_view_columns_autosize(GTK_TREE_VIEW(treeview));

	gtk_container_add(GTK_CONTAINER(scrolled_window), treeview);

	hbox = gtk_hbox_new(FALSE, PIDGIN_HIG_BOX_SPACE);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	entry = gtk_entry_new();
	config_frame.filename_entry = entry;
	gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
	gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 1);

	button = gtk_button_new_with_label(_("Browse..."));
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(select_sound), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 1);

	button = gtk_button_new_with_label(_("Preview"));
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(test_sound), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 1);

	button = gtk_button_new_with_label(_("Reset"));
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(reset_sound), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 1);

	config_frame_update_filename_entry();

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(config_frame.treeview));
	g_signal_connect(G_OBJECT(selection), "changed", G_CALLBACK(config_frame_update_filename_entry), NULL);

	config_frame.infobox = make_info_widget(_("You have muted sounds in Pidgin!"), GTK_STOCK_DIALOG_WARNING, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), config_frame.infobox, FALSE, FALSE, 1);

	purple_prefs_connect_callback(config_frame.infobox, PIDGIN_PREFS_ROOT "/sound/mute", update_muted_sound_hint_pidgin_pref_cb, config_frame.infobox);
	g_signal_connect(G_OBJECT(config_frame.infobox), "show", G_CALLBACK(update_muted_sound_hint_show_cb), NULL);

	gtk_widget_show_all(ret);
	return ret;
}

/*** Plugin-Gerödel ***/
static gboolean plugin_load(PurplePlugin *_plugin) {
	plugin = _plugin;

	init_events();

	return TRUE;
}

static gboolean plugin_unload(PurplePlugin *plugin) {
	free_events();
	
	return TRUE;
}

static PidginPluginUiInfo ui_info = {
        get_config_frame,
        0,   /* page_num (Reserved) */
        /* Padding */
        NULL,
        NULL,
        NULL,
        NULL
};

static PurplePluginInfo info = {
	PURPLE_PLUGIN_MAGIC,
	PURPLE_MAJOR_VERSION,
	PURPLE_MINOR_VERSION,
	PURPLE_PLUGIN_STANDARD,				/**< type           */
	PIDGIN_PLUGIN_TYPE,				/**< ui_requirement */
	0,						/**< flags          */
	NULL,						/**< dependencies   */
	PURPLE_PRIORITY_DEFAULT,			/**< priority       */

	PLUGIN_ID,					/**< id             */
	NULL,						/**< name           */
	PLUGIN_VERSION,					/**< version        */
	NULL,						/**  summary        */
				
	NULL,						/**  description    */
	PLUGIN_AUTHOR,					/**< author         */
	PLUGIN_WEBSITE,					/**< homepage       */

	plugin_load,					/**< load           */
	plugin_unload,					/**< unload         */
	NULL,						/**< destroy        */

	&ui_info,					/**< ui_info        */
	NULL,						/**< extra_info     */
	NULL,						/**< prefs_info     */
	NULL,						/**< actions        */
	/* padding */
	NULL,
	NULL,
	NULL,
	NULL
};

static void init_plugin(PurplePlugin *plugin) {
	const char *str = "Advanced Sound Notification";
	gchar *plugins_locale_dir;

#ifdef ENABLE_NLS
	plugins_locale_dir = g_build_filename(purple_user_dir(), "locale", NULL);

	bindtextdomain(GETTEXT_PACKAGE, plugins_locale_dir);
	if(str == _(str)) {
		bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	}
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");

	g_free(plugins_locale_dir);
#endif /* ENABLE_NLS */

        info.name        = _("Advanced Sound Notification");
        info.summary     = _("Implements various missing sound notifications.");
        info.description = _("Implements various missing sound notifications.");

	purple_prefs_add_none(PLUGIN_PREFS_PREFIX);

	purple_prefs_add_none(PLUGIN_PREFS_PREFIX "/incoming_filetransfer");
	purple_prefs_add_bool(PLUGIN_PREFS_PREFIX "/incoming_filetransfer/enabled", FALSE);
	purple_prefs_add_string(PLUGIN_PREFS_PREFIX "/incoming_filetransfer/filename", "");

	purple_prefs_add_none(PLUGIN_PREFS_PREFIX "/filetransfer_completed");
	purple_prefs_add_bool(PLUGIN_PREFS_PREFIX "/filetransfer_completed/enabled", FALSE);
	purple_prefs_add_string(PLUGIN_PREFS_PREFIX "/filetransfer_completed/filename", "");

	purple_prefs_add_none(PLUGIN_PREFS_PREFIX "/filetransfer_error");
	purple_prefs_add_bool(PLUGIN_PREFS_PREFIX "/filetransfer_error/enabled", FALSE);
	purple_prefs_add_string(PLUGIN_PREFS_PREFIX "/filetransfer_error/filename", "");

	purple_prefs_add_none(PLUGIN_PREFS_PREFIX "/authorization_request");
	purple_prefs_add_bool(PLUGIN_PREFS_PREFIX "/authorization_request/enabled", FALSE);
	purple_prefs_add_string(PLUGIN_PREFS_PREFIX "/authorization_request/filename", "");

	purple_prefs_add_none(PLUGIN_PREFS_PREFIX "/incoming_mail");
	purple_prefs_add_bool(PLUGIN_PREFS_PREFIX "/incoming_mail/enabled", FALSE);
	purple_prefs_add_string(PLUGIN_PREFS_PREFIX "/incoming_mail/filename", "");
}

PURPLE_INIT_PLUGIN(PLUGIN_STATIC_NAME, init_plugin, info)
