/*
 *      NiceShaper - Dynamic Traffic Management
 *
 *      Copyright 2004 Mariusz Jedwabny <mariusz@jedwabny.net>
 *
 *      This file is subject to the terms and conditions of the GNU General Public
 *      License.  See the file COPYING in the main directory of this archive for
 *      more details.
 */

#include "ifaces.h"

#include <sys/ioctl.h>
#include <net/if.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <unistd.h>

#include <string>
#include <iostream>
#include <cstdio>
#include <cstring>
#include <cstdlib>

#include "main.h"
#include "aux.h"
#include "sys.h"
#include "log.h"

using namespace aux;

Iface::Iface(int iface_index, std::string iface_name) 
{ 
    Index = iface_index; 
    Name = iface_name; 
    Controlled = false;
    DNShapeMethodSafe = true;
    HtbDNWrapperClass = false;
    Speed = 0;
    FallbackRate = 100 * 1000; // 100kb/s
    Sections.clear();
    SectionsSpeedSum = 0;
    TcFilterType = U32;

    // Erase alias from iface name
    if (Name.find_first_of(":") != std::string::npos) {
        Name.erase(Name.find_first_of(":"));
    }
}

Iface::~Iface()
{
    // nothing
}

IfacesMap::IfacesMap () 
{
    HtbFallbackId = 8;
    HtbDNWrapperId = 9;

    // Complete system devices
    struct ifreq *ifr;
    struct ifreq ifr2;
    struct ifconf ifc;
    unsigned int query_devs=0;
    int sd;

    sd = socket (PF_INET, SOCK_STREAM, 0);

    do {
        if (query_devs) delete [] ifr;
        query_devs++;
        ifr = new ifreq[query_devs];
        ifc.ifc_len = query_devs * sizeof (struct ifreq);
        ifc.ifc_req = ifr;
        ioctl (sd, SIOCGIFCONF, &ifc);
    } while (query_devs == ifc.ifc_len / sizeof(struct ifreq));

    for (unsigned int i = 0; i < (ifc.ifc_len / sizeof (struct ifreq)); i++) {
        strncpy(ifr2.ifr_name, ifr[i].ifr_name, sizeof(ifr2.ifr_name));
        ioctl (sd, SIOCGIFINDEX, &ifr2);
        SysNetDevices.push_back(new Iface(ifr2.ifr_ifindex, std::string(ifr[i].ifr_name)));
    }

    // Checking for IMQ devices.
    int n = 0;
    do {
	    int ifindex = 0;

        sprintf(ifr2.ifr_name, "imq%d", n);
        if (ioctl(sd, SIOCGIFINDEX, &ifr2) < 0) break;
     	ifindex = ifr2.ifr_ifindex;
        if (ioctl(sd, SIOCGIFFLAGS, &ifr2) < 0) break;

        SysNetDevices.push_back(new Iface(ifindex, std::string(ifr2.ifr_name)));
        n++;
    } while (n);
 
    close(sd);
}

IfacesMap::~IfacesMap () 
{
    std::string buf;
    Iface *dev;

    // Erase HTB from interfaces
    for (unsigned int n=0; n < SysNetDevices.size(); n++) 
    {   
        dev = SysNetDevices.at(n);
        if (!dev->Controlled) continue;
        if (!g_fallback_to_tc) {
            TcQdiscType qdisc = HTB;
            sys->rtnlOpen();
            sys->tcQdisc(TC_DEL, dev->Index, TC_H_ROOT, 1, qdisc, 0);
            sys->rtnlClose();
        } else {
            buf = "tc qdisc del root dev " + dev->Name + " 2> /dev/null > /dev/null";
            sys->tc(buf);
        }
    }
}

int IfacesMap::ifaceNum(std::string dev)
{
    for (unsigned int n=0; n < SysNetDevices.size(); n++) {
        if (SysNetDevices.at(n)->Name == dev) return n;
    }

    return -1;
}

int IfacesMap::index(std::string dev)
{
    if (ifaceNum(dev) == -1) return -1;

    return SysNetDevices.at(ifaceNum(dev))->Index;
}

bool IfacesMap::isValidSysDev(std::string dev)
{
    if (ifaceNum(dev) == -1) return false;

    return true;
}

void IfacesMap::setAsControlled(std::string dev)
{
    if (ifaceNum(dev) == -1) return;

    SysNetDevices.at(ifaceNum(dev))->Controlled = true;
}

void IfacesMap::setDNShapeMethodSafe(std::string dev, bool arg)
{
    if (ifaceNum(dev) == -1) return;

    SysNetDevices.at(ifaceNum(dev))->DNShapeMethodSafe = arg;
}

bool IfacesMap::isDNShapeMethodSafe(std::string dev)
{
    if (ifaceNum(dev) == -1) return true;

    return SysNetDevices.at(ifaceNum(dev))->DNShapeMethodSafe;
}

void IfacesMap::setHtbDNWrapperClass(std::string dev, bool arg)
{
    if (ifaceNum(dev) == -1) return;

    SysNetDevices.at(ifaceNum(dev))->HtbDNWrapperClass = arg;
}

void IfacesMap::setSpeed(std::string dev, unsigned int iface_speed)
{
    if (ifaceNum(dev) == -1) return;

    SysNetDevices.at(ifaceNum(dev))->Speed = iface_speed;
}

unsigned int IfacesMap::speed(std::string dev)
{
    if (ifaceNum(dev) == -1) return MAX_RATE;

    return SysNetDevices.at(ifaceNum(dev))->Speed;
}

void IfacesMap::setFallbackRate(std::string dev, unsigned int rate)
{
    if (ifaceNum(dev) == -1) return;

    SysNetDevices.at(ifaceNum(dev))->FallbackRate = rate;
}

void IfacesMap::addSection(std::string dev, std::string section)
{
    if (ifaceNum(dev) == -1) return;
    
    if (!is_in_vector(SysNetDevices.at(ifaceNum(dev))->Sections, section)) SysNetDevices.at(ifaceNum(dev))->Sections.push_back(section);
}

bool IfacesMap::isInSections(std::string dev, std::string section) 
{
    if (ifaceNum(dev) == -1) return false;

    return (is_in_vector(SysNetDevices.at(ifaceNum(dev))->Sections, section));
}

int IfacesMap::addToSectionsSpeedSum(std::string dev_name, unsigned int speed)
{
    Iface *dev;

    if (ifaceNum(dev_name) == -1) return -1;

    dev = SysNetDevices.at(ifaceNum(dev_name));
    
    if (dev->HtbDNWrapperClass) {
        if (!dev->Speed) { log.error(103, ""); return -1; }
        if (speed >= (dev->Speed - dev->SectionsSpeedSum - MIN_RATE)) { log.error (803, (dev_name + " speed " + int_to_str(dev->Speed) + "b/s")); return -1; }
    }

    dev->SectionsSpeedSum += speed;

    return 0;
}

void IfacesMap::setTcFilterType(std::string dev, EnumTcFilterType type)    
{
    if (ifaceNum(dev) == -1) return;

    SysNetDevices.at(ifaceNum(dev))->TcFilterType = type;
}

EnumTcFilterType IfacesMap::tcFilterType(std::string dev)
{
    if (ifaceNum(dev) == -1) return U32;

    return SysNetDevices.at(ifaceNum(dev))->TcFilterType;
}

int IfacesMap::initHtbOnControlled()
{
    unsigned int dnwrapper_rate = 0;
    Iface *dev;

    // Clear and create new HTB structure 
    for (unsigned int n=0; n < SysNetDevices.size(); n++) 
    {   
        dev = SysNetDevices.at(n);
        if (!dev->Controlled) continue;
        if (dev->HtbDNWrapperClass) {
            if (!dev->Speed) { log.error(103, ""); return -1; }
            if (dev->Speed <= (dev->SectionsSpeedSum + dev->FallbackRate + MIN_RATE)) { log.error (804, (dev->Name + " speed " + int_to_str(dev->Speed) + "b/s")); return -1; }
            dnwrapper_rate = dev->Speed - (dev->SectionsSpeedSum + dev->FallbackRate);
        }   

        if (!g_fallback_to_tc) {
            //sys->tcQdisc(TC_DEL, dev_id, TC_H_ROOT, 1, HTB, 0);
            // Ugly hack!!
            sys->tc("tc qdisc del dev " + dev->Name + " root 2> /dev/null > /dev/null");
            if (sys->tcQdisc(TC_ADD, dev->Index, TC_H_ROOT, 1, HTB, 0) == -1) return -1;
            // HTB default - initial create
            if (sys->tcClass(TC_ADD, dev->Index, 0, HtbFallbackId, dev->SectionsSpeedSum, dev->SectionsSpeedSum, 7, compute_quantum(dev->SectionsSpeedSum), 0 , 0) == -1) return -1;
            if (sys->tcQdisc(TC_ADD, dev->Index, HtbFallbackId, HtbFallbackId, SFQ, 10) == -1) return -1;
            // HTB for safe do-not-shape nd wrapper classes if exists
            if (dev->HtbDNWrapperClass) {
                if (sys->tcClass(TC_ADD, dev->Index, 0, HtbDNWrapperId, dnwrapper_rate, dnwrapper_rate, 7, compute_quantum(dnwrapper_rate), 0 , 0) == -1 ) return -1;
//                if (sys->tcQdisc(TC_ADD, dev->Index, HtbDNWrapperId, HtbDNWrapperId, SFQ, 10) == -1 ) return -1;
            }
        }
        else {
            sys->tc("tc qdisc del dev " + dev->Name + " root 2> /dev/null > /dev/null");
            sys->tc("tc qdisc add dev " + dev->Name + " root handle 1: htb default " + int_to_str(HtbFallbackId));
            // HTB default
            sys->tc("tc class add dev " + dev->Name + " parent 1: classid 1:" + int_to_str(HtbFallbackId) + " htb rate " + int_to_str(dev->SectionsSpeedSum) + " ceil " + int_to_str(dev->SectionsSpeedSum) + " prio 7 quantum " + int_to_str(compute_quantum(dev->SectionsSpeedSum)));
            sys->tc("tc qdisc add dev " + dev->Name + " parent 1:" + int_to_str(HtbFallbackId) + " handle " + int_to_str(HtbFallbackId) + ": sfq perturb 10");
            // HTB for safe do-not-shape and wrapper classes if exists
            if (dev->HtbDNWrapperClass) {
                sys->tc("tc class add dev " + dev->Name + " parent 1: classid 1:" + int_to_str(HtbDNWrapperId) + " htb rate " + int_to_str(dnwrapper_rate) + " ceil " + int_to_str(dnwrapper_rate) + " prio 7 quantum " + int_to_str(compute_quantum(dnwrapper_rate)));
//                sys->tc("tc qdisc add dev " + dev->Name + " parent 1:" + int_to_str(HtbDNWrapperId) + " handle " + int_to_str(HtbDNWrapperId) + ": sfq perturb 10");
            }
        }
    }

    return 0;
}

int IfacesMap::endUpHtbFallbackOnControlled()
{
    Iface *dev;

    // HTB default - end up initialization
    for (unsigned int n=0; n < SysNetDevices.size(); n++) 
    {   
        dev = SysNetDevices.at(n);
        if (!dev->Controlled) continue;
        if (!g_fallback_to_tc) {
            if (sys->rtnlOpen() == -1) return -1;
            if (sys->tcClass(TC_MOD, dev->Index, 0, HtbFallbackId, dev->FallbackRate, dev->FallbackRate, 7, compute_quantum(dev->FallbackRate), 0 , 0) == -1) return -1;
            sys->rtnlClose();   
        }
        else {
            sys->tc("tc class change dev " + dev->Name + " parent 1: classid 1:" + int_to_str(HtbFallbackId) + " htb rate " + int_to_str(dev->FallbackRate) + " ceil " + int_to_str(dev->FallbackRate) + " prio 7 quantum " + int_to_str(compute_quantum(dev->FallbackRate)));
        } 
    }

    return 0;
}

unsigned int IfacesMap::htbFallbackId()
{
    return HtbFallbackId;
}

unsigned int IfacesMap::htbDNWrapperId()
{
    return HtbDNWrapperId;
}
