/*
 *      NiceShaper - Dynamic Traffic Management
 *
 *      Copyright 2004 Mariusz Jedwabny <mariusz@jedwabny.net>
 *
 *      This file is subject to the terms and conditions of the GNU General Public
 *      License.  See the file COPYING in the main directory of this archive for
 *      more details.
 */

#include "conv.h"

#include <cstdio>
#include <cstring>
#include <cstdlib>  // srand, rand 

#include <iostream>
#include <fstream>

#include "main.h"
#include "aux.h"
#include "log.h"
#include "ifaces.h"
#include "tests.h"

using namespace aux;

Conv::Conv ()
{
    StatsUnit = KBYTES;
    StatsListenIp = "127.0.0.1";
    StatsListenPort = 6423;
    StatsPassword = "";
    StatsFilePath = "";
    StatsFileOwner = "root";
    StatsFileGroup = "root";     
    StatsFileMode = "0644";
    StatsFileRewrite = 30;
    UsersDownloadSection = "";
    UsersUploadSection = "";
    UsersIfaceInet = "";
    UsersResolveHostName = false;
    UsersReplaceClasses = false;

    // Create random password
    std::string random_password_chars = "abcdefghyjklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUWXYZ1234567890";
    for (int n=0; n < 12; n++) {
        StatsPassword += random_password_chars.at(rand() % random_password_chars.size());
    }
}

Conv::~Conv ()
{
}

std::string Conv::getLine(std::ifstream &src_file)
{
    bool state_store = true, potential_block_comment_tag = false, state_block_comment = false;
    int buf_counter = 0;    
    char sbuf[MAX_LONG_BUF_SIZE];
    char cbuf;
    std::string buf;

    sbuf[0]=0;
    while (src_file.read(&cbuf, 1))
    {
        // completing one line
        // ASCII: '<'=60, '#'=35, '>'=62
        if (!state_block_comment) {
            if (potential_block_comment_tag) {
                if ((cbuf == 35) && state_store ) {
                    state_block_comment = true;
                    buf_counter--;
                    sbuf[buf_counter]=0;
                    continue;
                }
                potential_block_comment_tag = false;
            }
            if ( cbuf == 60 ) potential_block_comment_tag = true;       
        }
        else if ( state_block_comment ) {
            if ( potential_block_comment_tag ) {
                if ( cbuf == 62 ) {
                    state_block_comment = false;
                    state_store = true;
                    sbuf[buf_counter]=0;
                    continue;
                }
                potential_block_comment_tag = false;
            }
            if ( cbuf == 35 ) potential_block_comment_tag = true;
        }

        if ( cbuf == 35 ) state_store = false;  // '#'
        if ( cbuf == 59 ) cbuf = 10;        // ';'
        if ( cbuf == 10 ) state_store = true;   // 'NewLine'

        if ( state_store && !state_block_comment)
        {
            sbuf[buf_counter]=cbuf;
            buf_counter++;
        }

        if ( cbuf != 10 ) continue; 
        // end of completing line

        sbuf[buf_counter]=0;
        buf_counter=0;
        state_store = true;             

        buf = trim_strict(std::string(sbuf));
        sbuf[0]=0;              

        if (buf.empty()) continue;
        else return buf;
    }

    sbuf[buf_counter]=0;
    buf = trim_strict(std::string(sbuf));

    return buf; 
}

int Conv::convertToFpv (std::string confdir, std::string src_file, FileType type, std::vector <unsigned int> &protected_marks, std::vector <std::string> &fpv)
{
    std::string option, param, value;
    std::vector < std::string >::iterator fpvi;
    std::vector < std::string >::iterator si;
    std::vector < std::string > fpv_splited;
    bool running_class=false;
    std::ifstream ifd;
    std::string buf;

    if (src_file.substr(0, 1) != "/") src_file = confdir + "/" + src_file; 

    ifd.open(src_file.c_str());
    if (!ifd) {
        if ( type == CONFTYPE ) {
            log.error(46, src_file);
        } 
        if ( type == CLASSTYPE ) {
            log.error(47, src_file);
        }
        return -1;
    }

    while ((buf=getLine(ifd)).size())
    {
        option = awk(buf, 1);

        // Firstly proceed including
        if (option == "include") {
            if ((type==CLASSTYPE) && running_class && fpv_splited.size()) {
                for (unsigned int i = 0; i < fpv_splited.size(); i++) fpv.push_back (fpv_splited[i]);            
                fpv_splited.clear();
            }
            
            if (includeToFpv(confdir, buf, type, protected_marks, fpv) == -1) { ifd.close(); return -1; }
            continue;
        }

        // Secondly proceed directives, depending on config type (main config or classes config)
        if (type==CLASSTYPE) {
            if (option == "class") {
                std::string sec;
                if (running_class && fpv_splited.size()) {
                    for ( unsigned int i = 0; i < fpv_splited.size(); i++ ) fpv.push_back(fpv_splited[i]);
                    fpv_splited.clear();
                }
                running_class=false;                                                                                                                       
                sec = awk( buf, 2);
                si = RunningSections.begin();
                while ( si < RunningSections.end()) {
                    if ( *si == sec) { running_class = true; break; }
                    si++;
                }
            }

            if ( running_class ) {
                if ((option == "match") || (option == "class")) {
                    fpv.push_back (buf);
                }
                else { 
                    if (directiveSplit (buf, fpv_splited) == -1 ) { ifd.close(); return -1; }
                }
            }
        }
        else {
            if (directiveSplit(buf, fpv) == -1) { ifd.close(); return -1; }
        }

        if ((type==CONFTYPE) && (option == "match")) {
            if (value_of_param(buf, "mark").size()) protected_marks.push_back(str_to_uint(value_of_param(buf, "mark")));
        }
        else if ((type==CLASSTYPE) && (option == "match")) {
            if (value_of_param(buf, "mark").size()) protected_marks.push_back(str_to_uint(value_of_param(buf, "mark")));
            if (value_of_param(buf, "set-mark").size()) protected_marks.push_back( str_to_uint(value_of_param( buf, "set-mark")));
        }
        else if ((type==CLASSTYPE) && (option == "set-mark")) protected_marks.push_back( str_to_uint( awk( buf, 2 )));   
    }

    if ((type==CLASSTYPE) && running_class && fpv_splited.size())
    {
        for (unsigned int i = 0; i < fpv_splited.size(); i++) fpv.push_back(fpv_splited[i]); 
        fpv_splited.clear();
    }

    ifd.close();

    fpvi = fpv.begin();                                                
    while (fpvi < fpv.end()) {
        *fpvi = trim_strict(*fpvi);
        fpvi++;
    }

    return 0;
}

int Conv::includeToFpv (std::string confdir, std::string src, FileType type, std::vector <unsigned int> &protected_marks, std::vector <std::string> &fpv)
{
    std::string param, value;
    std::string include_type = "";
    std::string include_file = "";
    std::string include_download_section = UsersDownloadSection;
    std::string include_upload_section = UsersUploadSection;
    std::string include_iface_inet = UsersIfaceInet;
    bool include_resolve_hostname = UsersResolveHostName;
    unsigned int n=1;

    while (awk(src, ++n).size()) {
        param = awk( src, n );
        value = awk( src, ++n );
        if (param.empty() || value.empty()) { log.error( 24, src ); return -1; }
        
        if ((param == "file") || (param == "usersfile")) {
            include_type = param;
            include_file = value;
        }
        else if (param == "download-section") {
            include_download_section = value; 
        }
        else if (param == "upload-section") {
            include_upload_section = value;
        }
        else if (param == "iface-inet") {
            include_iface_inet = value;   
        }
        else if (param == "resolve-hostname") {
            if (value == "yes") include_resolve_hostname=true;
            else if (value == "no") include_resolve_hostname=false;
            else if (value == "true") {
                include_resolve_hostname=true;
                log.warning( 10, src );
            }
            else if ((value == "false") || (value == "none")) {
                include_resolve_hostname=false;
                log.warning( 9, src );
            }
            else { log.error( 24, src ); return -1; }
        }
        else { 
            log.error(24, src);
            return -1;
        }
    }

    if ((include_type != "file") && (include_type != "usersfile")) {
        log.error(24, src);
        return -1;
    }

    if (include_type == "file") {
        if (convertToFpv(confdir, include_file, type, protected_marks, fpv) == -1) { return -1; }
    }
    else if (include_type == "usersfile") {
        if (type != CLASSTYPE) { log.error(24, src); return -1;}
        if (convertUsersFile(confdir, include_file, include_download_section, include_upload_section, include_iface_inet, include_resolve_hostname, false, fpv) == -1) { return -1; }
    }
    return 1;
}

int Conv::convertUsersFile (std::string confdir, std::string src_file, std::string download_section, std::string upload_section, std::string iface_inet, bool resolve_hostname, bool make_comments, std::vector <std::string> &fpv)
{
    std::ifstream ifd;
    std::string usersbuf;
    std::string ip, dev;
    std::string hostname;

    if (src_file.substr(0, 1) != "/") src_file = confdir + "/" + src_file;

    ifd.open(src_file.c_str());
    if (!ifd) {
        log.error(26, src_file);
        return -1;
    }
    
    if (download_section.empty() && upload_section.empty()) {
        log.error(56, "download-section | upload-section");
        return -1;
    }

    while ((usersbuf=getLine(ifd)).size())
    {
        ip = awk( usersbuf, 1);
        dev = awk( usersbuf, 2);

        if ( !test->validIp(ip)) { log.error( 29, usersbuf ); return -1; }   
        if ( !ifaces->isValidSysDev(dev)) { log.error( 16, usersbuf ); return -1; }   

        if ( resolve_hostname ) hostname = ip_to_hostname(ip);
        else hostname = ip;

        if ( make_comments )
        {
            fpv.push_back("\n############### " + hostname + " ###");
        }

        if ( download_section.size()) {
            fpv.push_back( "class " + download_section + " " + dev + " " + hostname);
            fpv.push_back(" match dstip " + ip);
            if (value_of_param( usersbuf, "dl_low").size()) fpv.push_back(" low " + value_of_param( usersbuf, "dl_low"));
            if (value_of_param( usersbuf, "dl_ceil").size()) fpv.push_back(" ceil " + value_of_param( usersbuf, "dl_ceil"));
            if (value_of_param( usersbuf, "dl_rate").size()) fpv.push_back(" rate " + value_of_param( usersbuf, "dl_rate"));
        }

        if ( upload_section.size()) {
            if ( iface_inet.empty()) { log.error(56, "iface-inet"); return -1; }
            fpv.push_back("class " + upload_section + " " + iface_inet + " " + hostname);
            fpv.push_back(" match srcip " + ip);
            if (value_of_param( usersbuf, "ul_low").size()) fpv.push_back(" low " + value_of_param( usersbuf, "ul_low"));
            if (value_of_param( usersbuf, "ul_ceil").size()) fpv.push_back(" ceil " + value_of_param( usersbuf, "ul_ceil"));
            if (value_of_param( usersbuf, "ul_rate").size()) fpv.push_back(" rate " + value_of_param( usersbuf, "ul_rate"));
        }    
    }

    return 0;
}

int Conv::directiveSplit (std::string arg, std::vector < std::string > &fpv)
{
    std::string option, param, value;
    unsigned int pos = 0;

    // type1 directives. 
    // has 1 required value and nothing else, 
    static char t1_src[10][MAX_SHORT_BUF_SIZE] = { "ceil", "hold", "lang", "low", "mode", "rate", "reload", "set-mark", "strict", "type" };
    std::vector <std::string> t1 (t1_src, t1_src + sizeof(t1_src)/sizeof(t1_src[0]));

    // type2 directives.
    // gets all given values as his own. Need at least 1 value.
    static char t2_src[4][MAX_SHORT_BUF_SIZE] = { "debug", "mark-on-ifaces", "run", "fallback" };
    std::vector <std::string> t2 (t2_src, t2_src + sizeof(t2_src)/sizeof(t2_src[0]));

    // type4 directives
    // by iteration, gets pairs of words ( parameter and value ), 
    // it's syntax error if parameter is unknown or one of pair elements is empty.
    static char t4_src[11][12][MAX_SHORT_BUF_SIZE] = {{ "log", "file", "syslog", "terminal" },
        { "users", "replace-classes", "download-section", "upload-section", "iface-inet", "resolve-hostname" },
        { "stats", "unit", "classes", "sum", "listen", "password", "file", "owner", "group", "mode", "rewrite" },
        { "section", "shape", "speed", "htb-burst", "htb-cburst" },
        { "htb", "scheduler", "prio", "burst", "cburst" },
        { "sfq", "perturb" },
        { "esfq", "hash", "perturb" },
        { "iptables", "hook", "hook-mode", "target" },
        { "imq", "autoredirect" },
        { "alter", "low", "ceil", "rate", "time-period" },
        { "quota", "low", "ceil", "rate", "day", "week", "month", "file", "reset-hour", "reset-wday", "reset-mday" }};
    static char t4_iface_src[3][MAX_SHORT_BUF_SIZE] = { "speed", "do-not-shape-method", "fallback-rate" };
    std::vector <std::string> t4;
    for (unsigned int i=0; i<(sizeof(t4_src)/sizeof(t4_src[0])); i++) {
        if (std::string(std::string(t4_src[i][0])).size()) t4.push_back(std::string(t4_src[i][0]));
        else break;
    }

    // type5 directives
    // special directives, copy whole line without changes.             
    static char t5_src[3][MAX_SHORT_BUF_SIZE] = { "class", "match", "include" };
    std::vector <std::string> t5 (t5_src, t5_src + sizeof(t5_src)/sizeof(t5_src[0]));

    pos = 1;
    if (awk(arg, pos) == "default") pos++;
    option = awk(arg, pos);

    for (unsigned int i=0; i<t1.size(); i++) {
        if (option == t1.at(i)) {
            value = awk(arg, ++pos);
            if (!value.size() || awk(arg, pos+1).size()) {
                log.error(24, arg);
                return -1;
            }
            
            fpv.push_back (option + " " + value);

            return 1;
        }
    }
 
    for (unsigned int i=0; i<t2.size(); i++) {
        if (option == t2.at(i)) {
            if (awk(arg, pos+1).empty()) {
                log.error(24, arg);
                return -1;
            }            

            while (awk(arg, ++pos).size()) {
                fpv.push_back(option + " " + awk(arg, pos));
            }

            return 1;
        }
    }

    for (unsigned int i=0; i<=t4.size(); i++) {
        if (i==t4.size()) {
            if ((awk(option, "-", 1) != "iface") || (awk(option, "-", 2).empty()) || (awk(option, "-", 3).size())) break;
            if (!ifaces->isValidSysDev(awk(option, "-", 2))) { log.error(16, arg); return -1; }
        } 
        if ((i == t4.size()) || (option == t4.at(i))) {
            if (i==t4.size()) t4.assign(t4_iface_src, t4_iface_src + sizeof(t4_iface_src)/sizeof(t4_iface_src[0]));
            else t4.assign(t4_src[i]+1, t4_src[i]+sizeof(t4_src[i])/sizeof(t4_src[i][0]));
            do {
                param = awk(arg, ++pos);
                value = awk(arg, ++pos);
                if (param.empty() || value.empty()) {
                    log.error(24, arg);
                    return -1;
                }
                if (!is_in_vector(t4, param)) {
                    log.error(11, arg);
                    return -1;
                }

                fpv.push_back (option + " " + param + " " + value);
            } while (awk(arg, pos+1).size());
            return 1;
        }
    }

    for (unsigned int i=0; i<t5.size(); i++) {
        if (option == t5.at(i)) {
            fpv.push_back(arg);

            return 1;
        }
    }

    // Check for section tag
    option = awk(arg, 1);
    if (awk(arg, 2).size()) {
        log.error(24, arg);
        return -1;
    }

    if ((option.at(0) == '<') && (option.at(option.size()-1) == '>')) 
    {
        fpv.push_back( arg );
        return 1;   
    }

    // Directive not found
    log.error ( 11, arg );
    return -1;
}

int Conv::setStatsListenAddr (std::string arg)
{
    std::string ip_addr = "";
    int ip_port = 0;

    if (split_ip_port (arg, ip_addr, ip_port) == -1) return -1;

    StatsListenIp = ip_addr;
    if (ip_port) StatsListenPort = ip_port;

    return 0;
}
