#ifdef COPYRIGHT_INFORMATION
#include "gplv3.h"
#endif
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

// All this runs in non main threads.
static gboolean
transfer_operations(view_t *view_p, xfdir_t *new_xfdir_p, const gchar *module, gint items); 
static void
new_population_f (view_t * view_p, xfdir_t *new_xfdir_p);

typedef struct reload_t {
    const view_t *view_p;
    record_entry_t *target_en;
    xfdir_t *xfdir_p;
}reload_t;

typedef struct heartbeat_t{
    const view_t *view_p;
    const record_entry_t *target_en;
    xfdir_t *xfdir_p;
    GMutex *xfdir_mutex;
    GCond *xfdir_signal;
    gint heartbeat;
    view_preferences_t *view_preferences_p;
    GThread *thread;
} heartbeat_t;

#define resaturate(X) rfm_context_function (resaturate_f, X);

static void *
resaturate_f(gpointer data){
    view_t *view_p = data;
    if (!rfm_main_context_check(view_p, FALSE)) return NULL;
     population_t *population_p = (population_t *)
	rodent_find_in_population (view_p, view_p->mouse_event.current_mouseX, view_p->mouse_event.current_mouseY);
    if (rodent_valid_population_p(view_p, population_p) ) {
        rodent_saturate_icon (view_p, population_p);	
    }
    return NULL;
}

static xfdir_t *
get_root_xfdir (view_t * view_p) {
    GSList *plugin_list = rfm_find_plugins ();
    GSList *list;
    gint max_elements = 0;
    for(list = plugin_list; list; list = list->next) {
	if (rfm_is_root_plugin ((gchar *) list->data)) {
	    max_elements++;
	} else {
	    NOOP(stderr, "%s is not a root plugin\n", (gchar *) list->data);
	}
    }
    // extra elements: home directory and filesystem icon.
    max_elements += 2;  
    xfdir_t *xfdir_p = (xfdir_t *)malloc(sizeof(xfdir_t));
    if (!xfdir_p) g_error("malloc: %s:", strerror(errno));
    memset (xfdir_p, 0, sizeof(xfdir_t));

    xfdir_p->pathc = max_elements;	
    xfdir_p->gl = (dir_t *)malloc(xfdir_p->pathc*sizeof(dir_t));
    if (!xfdir_p->gl) g_error("malloc: %s", strerror(errno));
    memset(xfdir_p->gl,0,xfdir_p->pathc*sizeof(dir_t));
    
    

    gint item=0;
    /* create filesystem root icon */
    xfdir_p->gl[item].en=rfm_stat_entry("/", 0);
    xfdir_p->gl[item].en->tag = g_strdup(_("This is the root of the filesystem"));
    xfdir_p->gl[item].pathv = g_strdup(_("Root Directory"));
    SET_ROOT_TYPE(xfdir_p->gl[item].en->type);
    /* create homedir icon */
    item++;
    xfdir_p->gl[item].en=rfm_stat_entry(g_get_home_dir(), 0);
    xfdir_p->gl[item].pathv = g_strdup(_("Home Directory"));
    SET_ROOT_TYPE(xfdir_p->gl[item].en->type);

    item++;

    NOOP ("ROOTS: now plugins...\n");
    for(list = plugin_list; list; list = list->next) {
	if (!rfm_is_root_plugin ((gchar *) list->data))continue;
	const gchar *module_name = list->data;
        NOOP ("ROOTS: creating population item for module %s\n", module_name);
	xfdir_p->gl[item].en=rfm_mk_entry(0);
	xfdir_p->gl[item].en->st = NULL;
	xfdir_p->gl[item].en->module = module_name; 
	gchar *module_label = 
	    rfm_void (PLUGIN_DIR, (void *)module_name, "module_label");
   
	xfdir_p->gl[item].en->path= module_label;    
	xfdir_p->gl[item].pathv = g_strdup(module_label);
	SET_ROOT_TYPE(xfdir_p->gl[item].en->type);
        item++;
    }
    rfm_global_t *rfm_global_p = rfm_global();
    rfm_cursor_reset(rfm_global_p->window);
    return xfdir_p;
}


static void *
direct_loadxfdir(gpointer data){
    NOOP(stderr, "direct_loadxfdir\n");
    reload_t *reload_p = data;
    view_preferences_t *view_preferences_p = 
	rfm_get_view_preferences (reload_p->view_p->flags.type,
		(record_entry_t *)reload_p->target_en);
    xfdir_t *xfdir_p = xfdir_get ((view_t *)reload_p->view_p, 
	    view_preferences_p,
	    (record_entry_t *)reload_p->target_en,
	    NULL);
	    
    if(xfdir_p==NULL) {
	NOOP ("load_xfdir_structure: xfdir_p is NULL!\n");
    }
    g_free(view_preferences_p);
    return xfdir_p;
}

static void *
heartbeat_loadxfdir(gpointer data){
    heartbeat_t *heartbeat_p = data;
    xfdir_t *xfdir_p = NULL;

    NOOP(stderr, "at heartbeat_loadxfdir\n");
    xfdir_p = xfdir_get ((view_t *)heartbeat_p->view_p, 
	    heartbeat_p->view_preferences_p,
	    (record_entry_t *)heartbeat_p->target_en,
	    &(heartbeat_p->heartbeat));
    NOOP( "at heartbeat_loadxfdir got xfdir_p\n");
	    
    g_mutex_lock(heartbeat_p->xfdir_mutex);
    heartbeat_p->xfdir_p = xfdir_p;
    g_cond_signal(heartbeat_p->xfdir_signal);
    g_mutex_unlock(heartbeat_p->xfdir_mutex);
    if(xfdir_p==NULL) {
	NOOP ("load_xfdir_structure: xfdir_p is NULL!\n");
    }
    return xfdir_p;

}

static 
void *wait_on_thread(gpointer data){
    heartbeat_t *heartbeat_p = data;
    xfdir_t *xfdir_p = g_thread_join(heartbeat_p->thread);

    rfm_mutex_free(heartbeat_p->xfdir_mutex);
    rfm_cond_free(heartbeat_p->xfdir_signal);
    g_free(heartbeat_p->view_preferences_p);
    g_free (heartbeat_p);
    return xfdir_p;
}


static
xfdir_t *
load_xfdir_with_timeout (reload_t *reload_p, gboolean with_heartbeat) {
    TRACE( "load_xfdir_with_timeout with_heartbeat %d\n",with_heartbeat);
    if (!with_heartbeat){
	return direct_loadxfdir(reload_p);

    }
    heartbeat_t *heartbeat_p = (heartbeat_t *)malloc(sizeof(heartbeat_t));
    if (!heartbeat_p) g_error("malloc heartbeat_p: %s\n",strerror(errno));
    memset(heartbeat_p, 0, sizeof(heartbeat_t));

    // Get user defined preferences for target entry
    heartbeat_p->view_preferences_p = 
	rfm_get_view_preferences (reload_p->view_p->flags.type,
		(record_entry_t *)reload_p->target_en);

    rfm_mutex_init(heartbeat_p->xfdir_mutex);
    rfm_cond_init(heartbeat_p->xfdir_signal);
    heartbeat_p->view_p = reload_p->view_p;
    heartbeat_p->target_en = reload_p->target_en;

    g_mutex_lock(heartbeat_p->xfdir_mutex);
    NOOP( "Creating wait thread for heartbeat_loadxfdir\n");
    heartbeat_p->thread =
	rfm_thread_create ("heartbeat_loadxfdir", heartbeat_loadxfdir, heartbeat_p, TRUE);
    if (!heartbeat_p->thread){
	g_mutex_unlock(heartbeat_p->xfdir_mutex);
	return NULL;
    }
#define RFM_LOAD_TIMEOUT 5
    if (!heartbeat_p->xfdir_p) {
	gint load_timeout = RFM_LOAD_TIMEOUT;
	const gchar *p = getenv ("RFM_LOAD_TIMEOUT");
	if(p && strlen (p)) {
	    errno=0;
	    long value = strtol(p, NULL, 0);
	    if (errno == 0){
		load_timeout = value;
	    }
	}


	// loop will be broken when heartbeat stops.
	// since this is read only, mutex can be skipped.
	while (heartbeat_p->heartbeat >= 0){
	    NOOP("*heartbeat records = %d\n", heartbeat_p->heartbeat);
	    rfm_threadwait();
	}
	// Now that heartbeat has stopped. Wait for the timeout condition.
	if (!rfm_cond_timed_wait(heartbeat_p->xfdir_signal, heartbeat_p->xfdir_mutex, load_timeout))
	{
	    g_mutex_unlock(heartbeat_p->xfdir_mutex);
	    TRACE( "dead heartbeat\n");
	    widgets_t *widgets_p = (widgets_t *)&(heartbeat_p->view_p->widgets);
	    rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-error", 
		g_strdup_printf("%s: %s", reload_p->target_en->path, strerror(ETIMEDOUT)));
	    // Dead heartbeat:
	    // Fire off a wait and cleanup thread.
	    rfm_view_thread_create((view_t *)heartbeat_p->view_p, wait_on_thread, heartbeat_p, "wait_on_thread");
	    return NULL;
	}
    }
    g_mutex_unlock(heartbeat_p->xfdir_mutex);
    xfdir_t *xfdir_p = wait_on_thread(heartbeat_p);
    return xfdir_p;
}


static void
exit_helper_threads(view_t *view_p){
    // Signal monitor to exit.
    xfdir_exit_monitor (view_p);
    // If any thumbnailer is active, signal it to exit:
    if (view_p->flags.thumbnailer_active) {
	view_p->flags.thumbnailer_exit=TRUE;    
    } 
    // FIXME: signal exit to tips threads here (normal or preview)
    // to avoid race collision. This is not yet coded.
    // XXX: Is this still valid?
}


static void
timeout_message(widgets_t *widgets_p, const gchar *path){
    // ETIMEDOUT       Connection timed out (POSIX.1)
    // EDEADLK         Resource deadlock avoided (POSIX.1)
    gchar *msg = g_strdup_printf(_("Unable to load: %s..."), path);
    rfm_threaded_show_text(widgets_p);
    gchar *m = g_strconcat( strerror(ETIMEDOUT), ": ", msg, 
	    " (", strerror(EDEADLK),")\n", NULL);
    rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-error", m);
    g_free(msg);
}

static void
error_trap(xfdir_t *new_xfdir){
    if (new_xfdir->pathc > 0) return;
    DBG("No items were loaded into xfdir structure!\n");
    new_xfdir->pathc = 1;
    new_xfdir->gl = (dir_t *) malloc (sizeof (dir_t));
    if(!new_xfdir->gl) g_error ("malloc: %s", strerror(errno));
    memset (new_xfdir->gl, 0, sizeof (dir_t));
    // root item:
    new_xfdir->gl[0].en = NULL;
    new_xfdir->gl[0].pathv = g_strdup (g_get_host_name ());
    return;
}

static
void
set_deepest_dir (view_t * view_p) {
    if(view_p->en && view_p->en->path && rfm_g_file_test (view_p->en->path, G_FILE_TEST_IS_DIR)) {
        if(!view_p->deepest_dir || !strstr (view_p->deepest_dir, view_p->en->path)) {
            g_free (view_p->deepest_dir);
            view_p->deepest_dir = g_strdup_printf ("%s/.", view_p->en->path);
        }
    } else {
        //DBG("cannot set deepest dir\n");
    }
    
}



static void
transfer_open(view_t *view_p, xfdir_t *new_xfdir){
    NOOP(stderr, "transfer_open...\n");
    // Error trap: if nothing is loaded, then at least put in
    // the localhost icon.
    error_trap(new_xfdir);
    // Set the entry preferences in the view.
    record_entry_t *en = view_p->en;
    view_preferences_t *view_preferences_p = 
	rfm_get_view_preferences (view_p->flags.type, en);
    rfm_set_view_preferences (view_p, view_preferences_p);
    g_free(view_preferences_p);
#ifdef DEBUG
	if(view_p->flags.preferences & __SHOW_HIDDEN){
	    NOOP( "transfer_open: shows hidden...\n");
	} else {
	    NOOP( "transfer_open: does *not* show hidden...\n");
	}
#endif


    // This will set the layout parameters to the correct
    // values and then set the paper widget accordingly.
    gint items = count_xfdir_population(view_p, new_xfdir);
    rfm_layout_configure(view_p, items);
      
    // Non null entry scenario
    if (en) view_p->module = en->module;
    else view_p->module = NULL;

    widgets_t *widgets_p = &(view_p->widgets);
    g_free (widgets_p->workdir);
    if (en && IS_LOCAL_TYPE(en->type)) widgets_p->workdir = g_strdup (en->path);
    else widgets_p->workdir = g_strdup(g_get_home_dir());
    // set the dtfile quick dir buttons (or menu items in our case)
    if(en && !en->module) {
	if(!en->path || 
		!IS_SDIR(en->type)) {
	    DBG ("Invalid entry sent to xfdir_get_local\n");
	} else {
	    set_deepest_dir (view_p);
	}
    }
    GtkWidget *pathbar = g_object_get_data(G_OBJECT(view_p->widgets.paper), "pathbar");
    if (en && en->module) {
        rfm_update_pathbar(pathbar, "RFM_MODULE");
        return;
    }
    rfm_update_pathbar(pathbar, (view_p->en)?view_p->en->path:NULL);
    return;
}

static gboolean
want_monitor(widgets_t *widgets_p, record_entry_t *en){
    NOOP(stderr, "want_monitor...\n");
    if (!en) return FALSE; 
    // Local files monitor:
    if (en->module){
	// Plugin requesting a file monitor thread:
	NOOP(stderr, "Plugin requesting a file monitor thread ...\n");
	if (rfm_void(PLUGIN_DIR, en->module, "module_monitor")) return TRUE;
    } else {
	// Local files monitor:
	if (IS_LOCAL_TYPE(en->type)) return TRUE;
	// Non local types with monitor:
	else if (IS_MONITOR_TYPE(en->type))return TRUE;
    }
    gchar *mount_point = g_strdup(en->path);
    do {
	NOOP(stderr, "mount_point ...--%s--\n", mount_point);
	// sysfs, tmpfs, devpts, debugfs and proc should be considered local.
	gchar *mount_type = rfm_natural(PLUGIN_DIR, "fstab", mount_point, "mnt_type");
	if (mount_type){
	    gboolean ok=FALSE;
	    if (strcmp(mount_type,"sysfs")==0) ok = TRUE;
	    if (strcmp(mount_type,"tmpfs")==0) ok = TRUE;
	    if (strcmp(mount_type,"devpts")==0) ok = TRUE;
	    if (strcmp(mount_type,"debugfs")==0) ok = TRUE;
	    if (strcmp(mount_type,"proc")==0) ok = TRUE;
	    g_free(mount_type);
	     if (ok) {
		g_free(mount_point);
		return TRUE;
	    }
	}
	gchar *g = mount_point;
	mount_point = g_path_get_dirname(g);
	g_free(g);
	if (mount_point && 
		(strcmp(mount_point, "/")==0 || strcmp(mount_point, ".")==0)) {
	    g_free(mount_point);
	    mount_point = NULL;
	}
    } while (mount_point);

    if (g_path_is_absolute(en->path)) {
	// warning 
	rfm_threaded_diagnostics (widgets_p, "xffm/stock_disconnect", NULL);
	rfm_threaded_diagnostics (widgets_p, "xffm-tag/red", 
		g_strconcat(_("Could not initialize monitoring"), 
		    " (", en->path,
		    ": ", _("Remote Connection..."),")\n", NULL));
    }
    return FALSE;
}
static void 
save_environment_desktop_dir(view_t *view_p, record_entry_t *target_en){
    // save the default desktop dir to the environment...
    if(view_p->flags.type != DESKVIEW_TYPE) {
	return;
    }
    g_free (view_p->desktop_dir);
    if (!target_en) {
	view_p->desktop_dir = g_strdup("");
    } else if (target_en->module) {
	if (rfm_void (PLUGIN_DIR, target_en->module, "module_name")) {
	    view_p->desktop_dir = g_strdup_printf ("module:%s",
		(gchar *)rfm_void (PLUGIN_DIR, target_en->module, "module_name"));
	} else {
	    view_p->desktop_dir = g_strdup (target_en->module);
	}
    } else {
	view_p->desktop_dir = g_strdup (target_en->path);
    }
    NOOP(stderr,"+++ save_environment_desktop_dir: RFM_DESKTOP_DIR\n");
    rfm_setenv ( "RFM_DESKTOP_DIR", view_p->desktop_dir, TRUE);
    // save configuration too
    rfm_rational (RFM_MODULE_DIR, "settings", (void *) "RFM_DESKTOP_DIR", (void *) view_p->desktop_dir, "mcs_set_var");
    NOOP(" setting RFM_DESKTOP_DIR to %s\n",view_p->desktop_dir);
    return;
}


// This requires GDK MUTEX and could be sent to thread-queue
// 
static void
transfer_close(view_t *view_p, xfdir_t *new_xfdir_p){
    record_entry_t *en = view_p->en;
    widgets_t *widgets_p = &(view_p->widgets);
	NOOP(stderr, "transfer_close...\n");

    if (en && en->tag){
	rfm_threaded_status (widgets_p, "xffm/stock_dialog-info", g_strdup(en->tag));
    } else if (!en) {
	rfm_threaded_status (widgets_p, "xffm/emblem_computer", g_strdup(_("Installed Plugins")));
    }


    // start background directory monitor thread, or else free
    // xdfir memory block.
    
    if (want_monitor(widgets_p, en)) {
	NOOP(stderr, "--- xfdir_start_monitor...\n");
	g_mutex_lock(view_p->mutexes.monitor_control);
	    view_p->flags.active_monitor = TRUE;
	g_mutex_unlock(view_p->mutexes.monitor_control);
	xfdir_start_monitor (view_p, new_xfdir_p);
    } else {
	// Free xfdir memory block.	
	NOOP(stderr, "Not starting xfdir monitor...\n");
	g_mutex_lock(view_p->mutexes.monitor_control);
	view_p->flags.active_monitor = FALSE;
	g_mutex_unlock(view_p->mutexes.monitor_control);
	xfdir_free_data (new_xfdir_p);
	g_free (new_xfdir_p);

    }
    if (view_p->flags.type == DESKVIEW_TYPE) {
	// Save desktop directory for navigational desktop
	save_environment_desktop_dir(view_p, en);
    } else {
	NOOP(stderr, "rodent_set_scroll_position\n");
	rfm_context_function(rodent_set_scroll_position, view_p);
	NOOP(stderr, "rodent_set_scroll_position done \n");
        rodent_set_view_icon (view_p);
        rodent_set_view_title (view_p);
	rodent_set_toggle_buttons(view_p);
	NOOP(stderr, "rodent_set_toggle_buttons done \n");
 	rfm_update_status_line (view_p);
    }
    // resaturate any icon which was previously saturated...
    
    resaturate(view_p);
	NOOP(stderr, "resaturate done \n");
}

// This should only be done after the write lock is active...
/*static void
increase_population_serial (view_t * view_p) {
    g_mutex_lock (view_p->mutexes.population_serial);
    view_p->flags.population_serial++;
    g_mutex_unlock (view_p->mutexes.population_serial);
    TRACE ("increase_population_serial: %d\n",
	    view_p->flags.population_serial);

}*/

static record_entry_t *
init_view (view_t * view_p, record_entry_t * en) {
    TRACE ("rodent_population: init_view\n");
    // this function is entered with a write lock on population
    // so mutex on population serial is not necessary

    //increase_population_serial (view_p);
    view_p->flags.population_serial++;

    view_p->mouse_event.boxX = -1, view_p->mouse_event.boxY = -1;
    view_p->mouse_event.old_X = -1, view_p->mouse_event.old_Y = -1;
#if 0
    // destroy selection ... not really...
    if(view_p->selection_list) {
	GSList *tmp=view_p->selection_list;
	for (;tmp && tmp->data; tmp=tmp->next){
	    record_entry_t *en=tmp->data;
	    rfm_destroy_entry(en);
	}
        g_slist_free (view_p->selection_list);
        view_p->selection_list = NULL;
    }
#endif
    view_p->mouse_event.mouseX = -1;
    view_p->mouse_event.mouseY = -1;
    view_p->widgets.rename = NULL;
    view_p->mouse_event.label_p = NULL;
    view_p->mouse_event.doing_drag_p = NULL;
    view_p->mouse_event.saturated_p = NULL;
 // nope:   view_p->mouse_event.selected_p = NULL;
    view_p->en = en;

  
    return en;
}


void
rodent_destroy_population (population_t **population_pp) {
    if(!population_pp) return;
    population_t **tmp;
    NOOP (stderr, "rodent_population: rodent_destroy_population: 0x%x\n",
	    GPOINTER_TO_INT(population_pp));
    for(tmp = population_pp; tmp && *tmp; tmp++) {
        population_t *population_p = *tmp;
        destroy_population_item (population_p);
        population_p = NULL;
    }
    //g_free (population_pp);
}

// This is the reload done by user driven callback (also in its own thread).
// Hard reload equivalent.

static void 
update_f(view_t *view_p, record_entry_t *target_en, xfdir_t *new_xfdir) {
    if (!new_xfdir){
	g_error("new_xfdir cannot be null here.");
    }
    g_mutex_lock(view_p->mutexes.status_mutex);
    // This will trigger monitor exit.
    view_p->flags.monitor_id++;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    // This will wake up a sleeping monitor
    xfdir_monitor_control_greenlight(&(view_p->widgets));
    // And here we should wait until monitor indeed has exited
    g_mutex_lock(view_p->mutexes.monitor_run_mutex);
    TRACE("~~~  update_f wait on monitor\n");
    while (view_p->monitor_running) {
        g_cond_wait(view_p->mutexes.monitor_run_signal, view_p->mutexes.monitor_run_mutex);
    }
    TRACE("***  update_f wait complete\n");
    g_mutex_unlock(view_p->mutexes.monitor_run_mutex);
    

    TRACE("~~~  update_f requesting writelock\n");
    if (!rfm_population_write_lock (view_p, "update_f")) {
	DBG ("update_f: !rfm_population_write_lock (view_p)\n");
	return;
    }

    TRACE("+++  update_f got write lock...\n");
    record_entry_t *old_en = view_p->en;
    //const gchar *module = (old_en)?old_en->module:NULL;


    init_view (view_p, target_en);
    NOOP( "init_view done...\n");
    transfer_open(view_p, new_xfdir);
    NOOP( "transfer_open done...\n");
    new_population_f(view_p, new_xfdir);
    rfm_destroy_entry(old_en);
    NOOP(stderr, "rfm_destroy_entry done...\n");
    // This function will do the scroll bar setting.
    transfer_close(view_p, new_xfdir);
    NOOP(stderr, "transfer_close done...\n");
    rfm_population_write_unlock (view_p, "update_f");
    TRACE("---  update_f released write lock...\n");
    rodent_redraw_items(view_p);
    NOOP(stderr, "rodent_redraw_items done...\n");
}

extern gboolean noexpose;

/*static pthread_once_t reload_once = PTHREAD_ONCE_INIT;
static pthread_mutex_t reload_mutex;
static void reload_init() {
    pthread_mutexattr_t attr;
    pthread_mutexattr_init(&attr);
    pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_ERRORCHECK);
    pthread_mutex_init(&reload_mutex, &attr);
    pthread_mutexattr_destroy(&attr);
}*/


// This is a thread function...
static void *
reload_f (gpointer data) {

    // Single mutex for all views is deadly deadlock
    //pthread_once(&reload_once, reload_init);

    NOOP(stderr, "at reload_f...\n");
    // Hold until the setup RW lock is available.
    rfm_global_t *rfm_global_p = rfm_global();
    rfm_rw_lock_reader_lock (&(rfm_global_p->setup_lock));

    reload_t *reload_p = data;
    view_t *view_p = (view_t *)reload_p->view_p;
    rfm_rw_lock_reader_unlock (&(rfm_global_p->setup_lock));

    // This mutex will protect against multiple function calls
    // simultaneous on the same view.
    //
    // No sense in queueing multiple reloads on same universe.
    // (try lock instead of lock)
    TRACE("%p: requesting lock on view_p->mutexes.reload_mutex...\n", g_thread_self());
/*   int errcode = pthread_mutex_trylock(&reload_mutex);
    if (errcode != 0) {
        DBG("FAILED to get lock on reload_mutex...%s\n", strerror(errcode));
	return NULL;
    }*/
    if (!g_mutex_trylock(view_p->mutexes.reload_mutex)) {
        DBG("FAILED to get lock on reload_mutex...\n");
	return NULL;
    }
    TRACE("%p: lock on view_p->mutexes.reload_mutex...\n", g_thread_self());


    widgets_t *widgets_p = &(view_p->widgets);
    record_entry_t *record_entry_p  = (record_entry_t *)reload_p->target_en;
    NOOP(stderr, "target entry = 0x%x\n", GPOINTER_TO_INT(reload_p->target_en));
    xfdir_t *new_xfdir = NULL;

    rodent_set_expose_hold(view_p, TRUE);

    // Step 2.
    // 2. Get new xfdir. This requires no extra locks nor mutexes...
    gboolean aborted_load = FALSE;
    if (record_entry_p == NULL){
	new_xfdir = get_root_xfdir(view_p);
    } else {
	if (IS_LOCAL_TYPE(record_entry_p->type)){
	    NOOP(stderr, "IS_LOCAL_TYPE at reload_f\n");
	    // This is a non-heartbeat controlled load. Will block.
	    new_xfdir = load_xfdir_with_timeout(reload_p, FALSE);
	    //new_xfdir = load_xfdir_with_timeout(reload_p, TRUE);
	} else {
	    // This is a heartbeat controlled load.
            void *result = rfm_void(PLUGIN_DIR, reload_p->target_en->module, "blocking_load");
	    if (reload_p->target_en->module && GPOINTER_TO_INT(result) > 0) {
		new_xfdir = load_xfdir_with_timeout(reload_p, FALSE);
	    } else {
		new_xfdir = load_xfdir_with_timeout(reload_p, TRUE);
	    }
	}

	// On timeout, null is returned.
	if (!new_xfdir){
	    timeout_message(widgets_p, record_entry_p->path);
	    // If this is not the initial load, then abort reload.
	    rfm_destroy_entry(record_entry_p);
	    if(view_p->go_list && g_list_last (view_p->go_list)){
		//rodent_expose_all (view_p);
		rodent_redraw_items (view_p);
		aborted_load = TRUE;
	    } else {
		reload_p->target_en = record_entry_p = NULL;
		new_xfdir = get_root_xfdir(view_p);
	    }
	} 
    }
    TRACE( "reload_f has loaded xfdir structure...\n");
 
    if (!aborted_load){
	// By now we have a new_xfdir, or valid NULL, so we can do a transfer.
	// target_en will substitute view_p->en downstream.
	exit_helper_threads(view_p);
        // update_f will request write lock
	update_f(view_p, reload_p->target_en, new_xfdir);
	//sem_post(rfm_global_p->setup_semaphore);
    } else {
	// Reset layout status so that a simple expose will put previous
	// population on the paper.
    }
    rfm_threaded_cursor_reset(rfm_global_p->window);
    g_free(reload_p);
    g_mutex_unlock(view_p->mutexes.reload_mutex);
    rodent_set_expose_hold(view_p, FALSE); // This will set expose to grab readlock
    //pthread_mutex_unlock(&reload_mutex);
    TRACE("%p: unlock on view_p->mutexes.reload_mutex.\n", g_thread_self());

    return GINT_TO_POINTER(!aborted_load);
}

static void *
mk_grid_elements_f(void *data){
    gint grid_elements = GPOINTER_TO_INT(data);
    NOOP (">> graphics_mk_grid_elements\n");
    population_t **population_pp;
    /* we use (grid_elements+1) to use a NULL pointer as an
     * end of list marker */
    NOOP (stderr, "mk_grid_elements(): grid_elements=%d (elements)\n", grid_elements);
    population_pp = (population_t **) malloc ((grid_elements + 1) * sizeof (population_t *));
    memset ((void *)(population_pp), 0, (grid_elements + 1) * sizeof (population_t *));
    population_pp[grid_elements] = NULL;
	
    NOOP(stderr, "malloc'd 0x%x\n", GPOINTER_TO_INT(population_pp));
    return population_pp;
}
static population_t **
mk_grid_elements (gint grid_elements) {
    //return rfm_context_function(mk_grid_elements_f, GINT_TO_POINTER(grid_elements));
    return mk_grid_elements_f( GINT_TO_POINTER(grid_elements));
}

#if 0
// This was for testing with valgrind.
static void *context_malloc(void *data){
    void *pointer = malloc (GPOINTER_TO_INT(data));
    if (!pointer) g_error("malloc: %s", strerror(errno));
    memset (pointer, 0, GPOINTER_TO_INT(data));
    return pointer;
}

static void *context_free(void *data){
    g_free(data);
    return NULL;
}
#endif

enum {
    TINY_ICON_SIZE_PIXBUF,
    SMALL_ICON_SIZE_PIXBUF,
    MEDIUM_ICON_SIZE_PIXBUF,
    BIG_ICON_SIZE_PIXBUF
};
 
static void reget_pixbuf(const gchar *file, GdkPixbuf **pixbuf_v, gint index, gint size){
        if (pixbuf_v[index] && G_IS_OBJECT(pixbuf_v[index])) 
            g_object_unref(pixbuf_v[index]);
	pixbuf_v[index] = rfm_get_pixbuf (file, size);

}

// this function gets the initial pixbuf icon. If population_p->icon_id is
// not set, then icon will be updated by monitor function.
static void
get_initial_pixbuf(population_t *population_p, const gchar * icon_id){
    static GdkPixbuf *file_pixbuf[4];
    static GdkPixbuf *folder_pixbuf[4];

    static gchar *icon_theme = NULL;
    const gchar *gtk_icon_theme = getenv("RFM_USE_GTK_ICON_THEME");
    static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

    pthread_mutex_lock(&mutex);
    if (!gtk_icon_theme) gtk_icon_theme = "";

    gboolean all_set = (icon_theme && strcmp(icon_theme, gtk_icon_theme)==0);
    if (!all_set){
        icon_theme = g_strdup(gtk_icon_theme);

	//const gchar *file = (strlen(gtk_icon_theme))?"xffm/stock_file":"xffm/emblem_file";
	//const gchar *folder = (strlen(gtk_icon_theme))? "xffm/stock_directory":"xffm/emblem_folder";
        const gchar *file = "xffm/stock_file";
	const gchar *folder = "xffm/stock_directory";
        reget_pixbuf(file, file_pixbuf, TINY_ICON_SIZE_PIXBUF, TINY_ICON_SIZE);
        reget_pixbuf(file, file_pixbuf, SMALL_ICON_SIZE_PIXBUF, SMALL_ICON_SIZE);
        reget_pixbuf(file, file_pixbuf, MEDIUM_ICON_SIZE_PIXBUF, MEDIUM_ICON_SIZE);
        reget_pixbuf(file, file_pixbuf, BIG_ICON_SIZE_PIXBUF, BIG_ICON_SIZE);

        reget_pixbuf(folder, folder_pixbuf, TINY_ICON_SIZE_PIXBUF, TINY_ICON_SIZE);
        reget_pixbuf(folder, folder_pixbuf, SMALL_ICON_SIZE_PIXBUF, SMALL_ICON_SIZE);
        reget_pixbuf(folder, folder_pixbuf, MEDIUM_ICON_SIZE_PIXBUF, MEDIUM_ICON_SIZE);
        reget_pixbuf(folder, folder_pixbuf, BIG_ICON_SIZE_PIXBUF, BIG_ICON_SIZE);


    }
    pthread_mutex_unlock(&mutex);

    view_t *view_p = population_p->view_p;
    record_entry_t *en = population_p->en;
    // Root level pixbuf short circuit
    if (!en) {
	population_p->pixbuf = rfm_get_pixbuf("xffm/emblem_computer", population_p->icon_size);	//refs
	population_p->icon_id = g_strdup("xffm/emblem_computer");
	return;
    }
    // First we let modules define whatever icon id it needs
    // (icon_id is ignored...)
    if (view_p->module || en->module){
	const gchar *id = NULL;
	if (IS_ROOT_TYPE(en->type))
	   id = rfm_natural(PLUGIN_DIR, en->module, en, "module_icon_id");
	else
	    id = rfm_natural(PLUGIN_DIR, view_p->module, en, "item_icon_id");
	NOOP(stderr, "create_population_f(): id = %s\n", id); 
	if (id) {
	    population_p->pixbuf = rfm_get_pixbuf(id, population_p->icon_size);	//refs
	    if (population_p->pixbuf) {
		population_p->icon_id = g_strdup(id);
		return;
	    }
	    DBG("Cannot load pixbuf for module with item_icon_id: \"%s\"\n", id);	
	}
    } 
    // Try a dirent type
    if (IS_UP_TYPE(en->type)){
	population_p->pixbuf = rfm_get_pixbuf ("xffm/stock_go-up", population_p->icon_size); //refs
    } else if (IS_SCHR(en->type)) {
	population_p->pixbuf = rfm_get_pixbuf ("xffm/emblem_chardevice", population_p->icon_size); //refs
    } else if (IS_SBLK(en->type)) {
	population_p->pixbuf = rfm_get_pixbuf ("xffm/emblem_blockdevice", population_p->icon_size); //refs
    } else if (IS_SFIFO(en->type)) {
	population_p->pixbuf = rfm_get_pixbuf ("xffm/emblem_fifo", population_p->icon_size); //refs
    } else if (IS_SSOCK(en->type)) {
	population_p->pixbuf = rfm_get_pixbuf ("xffm/emblem_network/compositeSE/emblem_fifo", population_p->icon_size); //refs
    }
    if (population_p->pixbuf) return; 
    // Folder (dirent) icon.
    // First try the cache...

    //GdkPixbuf **pixbuf_array;
    if (IS_SDIR(en->type)) {
	if (IS_ROOT_TYPE(en->type)){
	    const gchar *home=g_get_home_dir();
	    if (strcmp(en->path, home)==0) 
		population_p->pixbuf = rfm_get_pixbuf ("xffm/stock_home", population_p->icon_size);
	    if (population_p->pixbuf){
		population_p->icon_id = g_strdup("xffm/stock_home");
	       	return;
	    }
	}
        // Try cache to get content icon first...
        gchar *cache_id = rfm_get_icon_id_from_cache(en);
        if (cache_id) {
            population_p->pixbuf = rfm_get_pixbuf (cache_id, population_p->icon_size); //refs
            if (population_p->pixbuf) {
                g_free(cache_id);
                return;
            }
            DBG("Cannot load pixbuf from cache_id: \"%s\" (path=%s)\n", cache_id, en->path);
        }
        g_free(cache_id);

	switch (population_p->icon_size) {
	    case TINY_ICON_SIZE: 
		population_p->pixbuf = folder_pixbuf[TINY_ICON_SIZE_PIXBUF]; break;
	    case SMALL_ICON_SIZE: 
		population_p->pixbuf = folder_pixbuf[SMALL_ICON_SIZE_PIXBUF]; break;
	    case MEDIUM_ICON_SIZE: 
		population_p->pixbuf = folder_pixbuf[MEDIUM_ICON_SIZE_PIXBUF]; break;
	    case BIG_ICON_SIZE:
		population_p->pixbuf = folder_pixbuf[BIG_ICON_SIZE_PIXBUF]; break;
	}
	if (population_p->pixbuf) {
	    // This pixbuf must get an additional reference here.
	    g_object_ref(population_p->pixbuf);
	    population_p->icon_id = g_strdup("xffm/stock_directory");
            /*
	    gboolean gtk_icontheme = getenv("RFM_USE_GTK_ICON_THEME")
				    && strlen(getenv("RFM_USE_GTK_ICON_THEME"));
	    population_p->icon_id = g_strdup((gtk_icontheme)? "xffm/stock_directory":"xffm/emblem_folder");
            */
	    return;
	}
    }
    // Second. Is an icon id specified in function call?
    if (icon_id){
	population_p->pixbuf = rfm_get_pixbuf(icon_id, population_p->icon_size);//refs
	if (population_p->pixbuf) {
	    population_p->icon_id = g_strdup(icon_id);
	    return;
	}
	DBG("Cannot load pixbuf with id: %s\n", icon_id);	
    }
    // Third. Dealing with images we want to check if a thumbnail file is available
    if (rfm_path_is_image(en->path)){
	// Try to load pixbuf from thumbnail file.
	gchar *thumbnail = 
	    rfm_get_thumbnail_path (en->path, population_p->icon_size);
	if (g_file_test(thumbnail, G_FILE_TEST_EXISTS)){
	    NOOP("thumbnail for %s exists\n",en->path);
	    population_p->pixbuf = 
		rfm_get_pixbuf (en->path, population_p->icon_size); //ref
	    if (population_p->pixbuf) {
		g_free(thumbnail);	    
		return;
	    }
	    DBG("Cannot load thumbnail %s for: %s\n", thumbnail, en->path);	
	}
	
	g_free(thumbnail);	    
    }
    // Fourth. Try to retrieve an icon id for path from dbh cache.
    gchar *cache_id = rfm_get_icon_id_from_cache(en);
    if (cache_id) {
	population_p->pixbuf = rfm_get_pixbuf (cache_id, population_p->icon_size); //refs
	if (population_p->pixbuf) {
	    g_free(cache_id);
	    return;
	}
	DBG("Cannot load pixbuf from cache_id: \"%s\" (%s)\n", cache_id, en->path);
    }
    g_free(cache_id);

    // Sixth. Try a simple mimetype (no magic) for icon id
    if (strchr(en->path, '.')){  
	gchar *mimetype =
		    rfm_natural(RFM_MODULE_DIR, "mime", en->path, "mime_type_plain");
	if (mimetype){
	    population_p->pixbuf =
		    rfm_get_pixbuf (mimetype, population_p->icon_size);
	    if (population_p->pixbuf){
		g_free(mimetype);
		return;
	    }
	    DBG("cannot get pixbuf for plain mimetype: \"%s\"\n", mimetype);
	    g_free(mimetype);
	}
    }

    // Final fallback
    switch (population_p->icon_size) {
	case TINY_ICON_SIZE: 
	    population_p->pixbuf = file_pixbuf[TINY_ICON_SIZE_PIXBUF]; break;
	case SMALL_ICON_SIZE: 
	    population_p->pixbuf = file_pixbuf[SMALL_ICON_SIZE_PIXBUF]; break;
	case MEDIUM_ICON_SIZE: 
	    population_p->pixbuf = file_pixbuf[MEDIUM_ICON_SIZE_PIXBUF]; break;
	case BIG_ICON_SIZE:
	    population_p->pixbuf = file_pixbuf[BIG_ICON_SIZE_PIXBUF]; break;
    }
    // OjO: this must get an additional ref.
    g_object_ref(population_p->pixbuf);
    if (!population_p->pixbuf) // should never happen.
	g_error("Cannot get initial pixbuf for %s\n", en->path);
/*
    if (icon_id == NULL) {
	// FALSE means that magic will *not* be done on item, even if module
	// is available. This speeds up initial load. If magic is available,
	// it shall be done in background thread, nanosecond later without
	// using expensive main thread resources.
	//
	// For modules and rodent root we override the cache value here,
	// (secondary icons will not be used)
        if (!view_p->en || !en || en->module) {
	    g_free(population_p->icon_id);
	    population_p->icon_id = rfm_get_entry_icon_id(view_p, en, FALSE);
	    rfm_save_icon_id_to_cache(en, population_p->icon_id);
	}
    } else {
        g_free(population_p->icon_id);
    }*/
    return;
}

static population_t *
create_population_f (view_t * view_p,
	record_entry_t * en, 
	gint element, 
	const gchar * icon_id, 
	const gchar * label) 
{
    population_t *population_p = (population_t *) malloc (sizeof (population_t));
    if (!population_p) {
	g_warning("malloc: %s", strerror(errno));
	return NULL;
    }
    memset (population_p, 0, sizeof (population_t));

    population_p->view_p = view_p;
    population_p->icon_size = ICON_SIZE(view_p);
    population_p->icon_size_id = ICON_SIZE_ID(view_p);
    if(en) {
        population_p->en = rfm_copy_entry (en);
	population_p->module = en->module;
    } else {
        population_p->en = NULL;
	label = g_get_host_name();
    }
    population_p->flags = 0;
    gint pasteboard_status = rfm_pasteboard_status(view_p);
    gint item_status = 0;
    if (pasteboard_status){
        item_status = rfm_in_pasteboard (view_p, population_p->en);
        if(item_status == 2) {
            population_p->flags |= POPULATION_CUT;
        } else if (item_status == 1) {
            population_p->flags |= POPULATION_COPIED;
        }
    }
    population_p->label = g_strdup(label);


    rfm_layout_population_grid_row(view_p, population_p, element);
    ///////////////////////////////////////
    get_initial_pixbuf(population_p, icon_id);
    if (population_p->pixbuf==NULL || !GDK_IS_PIXBUF(population_p->pixbuf)) {
	g_error("cannot get initial pixbuf for %s\n", population_p->en->path);
    }
    return population_p;
}

// This is the equivalent to transfer population for the initial load.
// Population will be recreated.
// st_ino will be set to zero to force the background stat update.
static void
new_population_f (view_t * view_p, xfdir_t *new_xfdir_p) {
    gint i;
    NOOP (stderr, "new_population_f(0x%x): pathc = %d\n", GPOINTER_TO_INT(g_thread_self()),
	    (gint)new_xfdir_p->pathc);


    if (view_p->population_hash) g_hash_table_destroy(view_p->population_hash);
    view_p->population_hash = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);

    population_t **old_population_pp = view_p->population_pp;
    // create new population_pp
    gint items = count_xfdir_population(view_p, new_xfdir_p);
    NOOP(stderr, "------  items = %d, pathc = %d\n", items, new_xfdir_p->pathc);
    population_t **population_pp = mk_grid_elements (items);
    view_p->flags.preferences &= (__INVALIDATE_ICONS ^ 0xffffffff);

    gint count;
    for(i = 0, count = 0; i < new_xfdir_p->pathc; i++) {
	if ( new_xfdir_p->gl[i].en && !new_xfdir_p->gl[i].en->path ){
            g_error("new_population_f(): this is bad.... path cannot be NULL\n");
	}
	if (skip_item(view_p, new_xfdir_p->gl[i].pathv, new_xfdir_p->gl[i].en)) continue;
	    // create a new population item.
	if (new_xfdir_p->gl[i].en && view_p->module){
            population_pp[count] = 
		create_population_f (view_p, new_xfdir_p->gl[i].en, count, NULL, new_xfdir_p->gl[i].pathv);
	} else {
	    NOOP(stderr, "Local item quick icon: %s (module:%s)\n",
		    (new_xfdir_p->gl[i].en)?new_xfdir_p->gl[i].en->path:"null entry",
		    (view_p->module)?view_p->module:"NONE"); 
	    population_pp[count] = 
		create_population_f(view_p, new_xfdir_p->gl[i].en, count, NULL, new_xfdir_p->gl[i].pathv);
	    // hack to force a stat update:
	    if (new_xfdir_p->gl[i].en 
		    && IS_LOCAL_TYPE(new_xfdir_p->gl[i].en->type)
		    && new_xfdir_p->gl[i].en->st) new_xfdir_p->gl[i].en->st->st_ino = 0;
	}
        if (!new_xfdir_p->gl[i].en){
            g_hash_table_insert(view_p->population_hash, 
                    g_strdup("RODENT-ROOT"), population_pp[count]);
        } else {
            g_hash_table_insert(view_p->population_hash, 
                    g_strdup(new_xfdir_p->gl[i].en->path), population_pp[count]);
        }
	count++;
	// New items must get the thumbnail icon updated.
	// (later on)
	
    }
    NOOP(stderr, "exited loop\n");
    // Time to do the switch to the new population pp.
    view_p->population_pp = population_pp;

    // Remove obsolete population items
    
    rodent_destroy_population(old_population_pp);
    NOOP(stderr, "rodent_destroy_population done\n");
    g_free(old_population_pp);

    // This just sets the column width for details view...
    rfm_layout_pango_layout_setup(view_p);

    return ;
    
}

// new way: only for background thread
static void
transfer_population (view_t * view_p, xfdir_t *new_xfdir_p, const gchar *module, gint items) {
     // Construct hash table for reloaded population_pp;
    // (skip up element)
    // (hashed items are dir_t *)
    GHashTable *new_hash = g_hash_table_new(g_str_hash, g_str_equal);
    gint i;
    NOOP (stderr, "transfer(0x%x): items=%d pathc = %d\n", GPOINTER_TO_INT(g_thread_self()), items, (gint)new_xfdir_p->pathc);
    for(i = 0; i < new_xfdir_p->pathc; i++) {
	if (skip_item(view_p, new_xfdir_p->gl[i].pathv, new_xfdir_p->gl[i].en)) continue;
	const gchar *key;
	if (new_xfdir_p->gl[i].en) key = new_xfdir_p->gl[i].en->path;
	else key = "RODENT-ROOT";
	g_hash_table_insert(new_hash, (void *)key, (new_xfdir_p->gl)+i);
    }
    NOOP(stderr, "t_population loop 1 done...\n");

    GHashTable *old_hash = g_hash_table_new(g_str_hash, g_str_equal);
    population_t **old_population_pp = view_p->population_pp;
    // Construct hash table for current view population_pp.
    // (skip up element)
    // (hashed items are population_t *)
    GSList *remove_list = NULL;

    population_t **population_pp = old_population_pp;
    for(; population_pp && *population_pp; population_pp++) {
	population_t *population_p = *population_pp;
	const gchar *key;
	if (!population_p->en || !population_p->en->path) key = "RODENT-ROOT";
	else key = population_p->en->path;
	g_hash_table_insert(old_hash, (void *)key, population_p);
	// Construct list of remove items.
	if (!g_hash_table_lookup(new_hash, (void *)key)){
            NOOP (stderr, "transfer(): adding to remove list: %s\n", key);
	    remove_list = g_slist_prepend(remove_list, population_p);
	}
    }

    // create new population_pp
    population_pp = mk_grid_elements (items);
    NOOP(stderr, "new population_pp 0x%x\n", GPOINTER_TO_INT(population_pp));
    // Transfer transferable items and create new items.
//    if (view_p->population_pp) {population_pp[0] = view_p->population_pp[0]; }
    
    // If iconsize id has changed, we must also get new icons and 
    // new pango layouts!
    // If only iconsize not changed, we can skip new icons.
    gint icon_size_id = ICON_SIZE_ID(view_p);
    gint count = 0;
    for(i = 0; i < new_xfdir_p->pathc; i++) {
	if (skip_item(view_p, new_xfdir_p->gl[i].pathv, new_xfdir_p->gl[i].en)) continue;
	population_t *population_p = NULL;
	const gchar *key = (new_xfdir_p->gl[i].en)?  new_xfdir_p->gl[i].en->path : "RODENT-ROOT";
	population_p = g_hash_table_lookup(old_hash, key);

	gboolean do_magic = TRUE;
	if (new_xfdir_p->gl[i].en == NULL ||
		new_xfdir_p->gl[i].en->st == NULL || 
		(new_xfdir_p->gl[i].en->st->st_mtime == 0 &&
		new_xfdir_p->gl[i].en->st->st_blksize == 0)){
	    do_magic = FALSE;
	}

	if (population_p){
	    // Item is found and will only be updated (if necessary).
            population_pp[count] = population_p;
	    NOOP(stderr, "updating population record for %s \n",
		    (new_xfdir_p->gl[i].en)?new_xfdir_p->gl[i].en->path:"ROOT");
            /*  update the stat record */
            if(population_p->en 
		    && new_xfdir_p->gl[i].en 
		    && population_p->en->st 
		    && new_xfdir_p->gl[i].en->st) {
		gboolean out_of_date = memcmp(population_p->en->st, new_xfdir_p->gl[i].en->st, sizeof(struct stat)); 
		gboolean zap_preview = !(view_p->flags.preferences & __SHOW_IMAGES);
		if (zap_preview || out_of_date) {
		    if (out_of_date) {
			memcpy (population_p->en->st, new_xfdir_p->gl[i].en->st, sizeof (struct stat));
			if (rfm_entry_is_image(population_p->en)){
			    // This will set up the population item for the expose which
			    // takes care of generating the thumbnail.
			    population_p->flags |= POPULATION_IS_IMAGE;
			}
		    }
		    // remove any previous popup preview image, including the thumbnail file.
		    // (see xfdir.i:674)
		    if (population_p->preview_pixbuf) {
			GdkPixbuf *old_pixbuf = population_p->preview_pixbuf;
			population_p->preview_pixbuf = NULL;
			if (G_IS_OBJECT(old_pixbuf)) g_object_unref(old_pixbuf);
			gchar *thumbnail = 
			    rfm_get_thumbnail_path (population_p->en->path, PREVIEW_IMAGE_SIZE);
			if (g_file_test(thumbnail, G_FILE_TEST_EXISTS) && unlink(thumbnail)<0)
			    DBG("unlink(%s): %s\n", thumbnail, strerror(errno));
			    
			g_free(thumbnail);
			// We will now get a new preview here when requested.
		    }
		    // update the icon (or thumbnail) on expose.
		    // For this we mark the pixbuf as unclean.
		    population_p->flags &= (POPULATION_PIXBUF_CLEAN ^ 0xffffffff);

		    if (zap_preview){
			population_p->flags &= (POPULATION_THUMBNAILED ^ 0xffffffff);
			population_p->flags &= (POPULATION_PREVIEW_DONE ^ 0xffffffff);
			population_p->flags &= (POPULATION_PREVIEW_BUSY ^ 0xffffffff);
		    }
		}
		// update the icon if necessary
		// (will also redo content icon)
		// Module defined icons have preference...
		gchar *icon_id = NULL;
		if (view_p->en->module) {
		    const gchar *id = rfm_natural(PLUGIN_DIR, view_p->en->module, 
			    new_xfdir_p->gl[i].en,
			    "item_icon_id");
		    if (id) icon_id = g_strdup(id);
		}
		if (!icon_id && IS_SDIR(new_xfdir_p->gl[i].en->type)){
		    icon_id = folder_icon_id(view_p, new_xfdir_p->gl[i].en);
		} else if (!icon_id) {
		    icon_id = rfm_get_entry_icon_id(view_p, new_xfdir_p->gl[i].en, do_magic);
		}

		  
		// We add the condition for not showing images, since this is necessary
		// to reflect the change when the user switches the toggle.
		if (!population_p->icon_id || (view_p->flags.preferences & __INVALIDATE_ICONS) || (icon_id && strcmp(icon_id, population_p->icon_id))){
		    g_free(population_p->icon_id);
		    population_p->icon_id = icon_id;
		    population_p->flags &= (POPULATION_PIXBUF_CLEAN ^ 0xffffffff);

		    
		    if (view_p->flags.preferences & __INVALIDATE_ICONS) {
			population_p->flags &= (POPULATION_THUMBNAILED ^ 0xffffffff);
			population_p->flags &= (POPULATION_PREVIEW_DONE ^ 0xffffffff);
			population_p->flags &= (POPULATION_PREVIEW_BUSY ^ 0xffffffff);
		    }
		} else {
		    g_free(icon_id);
		}

	    }
#if 10
	    if (population_p->icon_size_id != icon_size_id) {
		population_p->flags &= (POPULATION_PIXBUF_CLEAN ^ 0xffffffff);
		population_p->icon_size_id = icon_size_id;
		rfm_layout_cleanup(population_p);
		NOOP(stderr, "updating iconsize to: %d\n",population_p->icon_size);
	    }
	    if (!icon_size_id){
		// Zap layout2 so it will reflect stat changes.
		// get new layout.
		NOOP(stderr, "We need new layout for %s\n", population_p->en->path);
		if (population_p->layout) {
		    if (G_IS_OBJECT(population_p->layout))
                        g_object_unref(population_p->layout);
		    population_p->layout=NULL;
		}
	    }		
#endif
	    
	} else 
	{
	    // create a new population item.
            TRACE ("transfer(): Creating new population item for %s\n",key);
	    // create mimetype icon (no magic)
	    // bugfree test for doing magic... if stat record is valid, do magic.

	    gchar *icon_id = 
		rfm_get_entry_icon_id(view_p, new_xfdir_p->gl[i].en, do_magic);
            population_pp[count] = 
		create_population_f (view_p, new_xfdir_p->gl[i].en, count, icon_id, new_xfdir_p->gl[i].pathv);
            if (new_xfdir_p->gl[i].en==NULL) {
                g_hash_table_replace(view_p->population_hash, 
                        g_strdup("RODENT-ROOT"), population_pp[count]);
            } else {
                g_hash_table_replace(view_p->population_hash, 
                        g_strdup(new_xfdir_p->gl[i].en->path), population_pp[count]);
            }
	    g_free(icon_id);   

	}
	if (rfm_entry_is_image(population_pp[count]->en)){
	    population_pp[count]->flags |= POPULATION_IS_IMAGE;
	    do_the_thumbnail(population_pp[count]);
	}
	count++;
    }
    view_p->flags.preferences &= (__INVALIDATE_ICONS ^ 0xffffffff);
    // That done, update the population selection list. 

    GSList *tmp;


    // Time to do the switch to the new population pp.
    // old population is now obsolete. Pointer array
    // is no longer necessary (we have the remove list).
    view_p->population_pp = population_pp;

    // Remove obsolete population items
    for (tmp = remove_list; tmp && tmp->data; tmp = tmp->next){
	population_t *population_p = tmp->data;
        NOOP (stderr, "transfer(): destroying  population item: %s\n", 
		(population_p->en)?population_p->en->path:"RODENT-ROOT");
        if (population_p->en) {
            g_hash_table_remove(view_p->population_hash, population_p->en->path);
        } else {
            g_hash_table_remove(view_p->population_hash, "RODENT-ROOT");
        }
	destroy_population_item (population_p);
    }

    NOOP(stderr, "remove old population_pp in transfer: 0x%x\n", GPOINTER_TO_INT(old_population_pp));
    g_free(old_population_pp);



    // Clean up temporary hash tables and remove list.
    g_hash_table_destroy(old_hash);
    g_hash_table_destroy(new_hash);
    g_slist_free(remove_list);

    // This just sets the column width for details view...
    rfm_layout_pango_layout_setup(view_p);

    return ;
}

static gboolean
transfer_operations(view_t *view_p, xfdir_t *new_xfdir_p, const gchar *module, gint items){
    /* make new population_pp and transfer old population_pp elements,
     * creating new elements for those not in old population_pp
     * and eliminating those in old population_pp no longer present
     * */


    // Requires write lock at least:
    transfer_population (view_p, new_xfdir_p, module, items);


    NOOP(stderr, "transfer_population done...\n");
    // recalculate x,y
    // get x,y coordinates for each population_p
    // or correct row, column
    //
    // Requires read lock at least:
    rodent_recalc_population_geometry (view_p);
    //rfm_population_write_unlock (view_p, "transfer_operations");
   
    if(view_p->en && view_p->en->st){
        memcpy (view_p->en->st, new_xfdir_p->en->st, sizeof (struct stat));
    }

    return TRUE;
}

// This is the reload done by the monitor thread.

static void *
thread_reload_view (view_t *view_p, xfdir_t *new_xfdir_p) {
//	return NULL;

    TRACE( "thread_reload_view: ...\n");

    g_mutex_lock(view_p->mutexes.status_mutex);
    gboolean status = view_p->flags.status;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    if(status == STATUS_EXIT) {
	return NULL;
    }
        

    if(!new_xfdir_p) {
        g_error ("short_rodent_thread_reload_view: !new_xfdir_p");
    }
    NOOP (stderr, "starting short_rodent_reload_view...\n");
    /*  monitor in wait state  with view_p->flags.refresh */

    // population semaphore is ours and population serial is valid...
    // Requires read lock at least:
    if (!rfm_population_write_lock (view_p, "thread_reload_view")) {
	TRACE ("thread_reload_view: !rfm_population_write_lock (view_p)\n");
	return NULL;
    }

    // This will set the layout parameters to the correct
    // values and then set the paper widget accordingly.
    TRACE("thread_reload_view: count_xfdir_population\n");
    gint items = count_xfdir_population(view_p, new_xfdir_p);
    TRACE("thread_reload_view: rfm_layout_configure\n");
    rfm_layout_configure(view_p, items);

    TRACE("thread_reload_view: transfer_operations\n");
    if (transfer_operations(view_p, new_xfdir_p, (view_p->en)?view_p->en->module:NULL, items)){ 
	// This requires GDK MUTEX and could be sent to thread-queue
	NOOP( "short_reload_view: reload is done\n");
    TRACE("thread_reload_view: resaturate\n");
	resaturate(view_p);
	NOOP ("population_sem: reload_view(): released!\n");
    }
    rfm_population_write_unlock (view_p, "thread_reload_view");
    rodent_redraw_items(view_p);

    return GINT_TO_POINTER(1);

}


//////////////////////////   janitor    /////////////////////////////
//

static void
destroy_population_item (population_t * population_p) {
    if(!population_p) return;
    /* objects that need to be unreffed: 
     * (tagged as referrence in types.h) */

    //static gint count=1; NOOP(stderr, "%d. Destroying population item, free 0x%x: %s\n", count++, GPOINTER_TO_INT(population_p), population_p->label);

    rfm_destroy_entry (population_p->en);
    g_free(population_p->icon_id);
    g_free(population_p->label);

    if(population_p->preview_pixbuf && G_IS_OBJECT(population_p->preview_pixbuf)){
        g_object_unref (population_p->preview_pixbuf);
    }
    if(population_p->pixbuf && G_IS_OBJECT(population_p->pixbuf)){
        g_object_unref (population_p->pixbuf);
    }
    if(population_p->thumbnail && G_IS_OBJECT(population_p->thumbnail)){
        g_object_unref (population_p->thumbnail);
    }    
    if(population_p->layout && G_IS_OBJECT(population_p->layout)){
        g_object_unref (population_p->layout);
    }
    if(population_p->layout2 && G_IS_OBJECT(population_p->layout2)){
        g_object_unref (population_p->layout2);
    }
    if(population_p->layout_full && G_IS_OBJECT(population_p->layout_full)){
        g_object_unref (population_p->layout_full);
    }
    g_free (population_p);
}

static void *
destroy_view(void *data){
    // XXX leak: race conditions may ensue (output, little run buttons, etc.)
    //           too hacky to account for all.
    return NULL;

    // Cleanup 
    if (rfm_get_gtk_thread() == g_thread_self()) g_error("rfm_destroy_view is a thread function.\n");
    view_t *view_p=data;
    rfm_global_t *rfm_global_p = rfm_global();

    TRACE("disconnecting g_signal handler %ld\n", view_p->signal_handlers[0]);
    if (view_p->signal_handlers[0]) {
	g_signal_handler_disconnect(G_OBJECT(rfm_global_p->window), view_p->signal_handlers[0]);
    }
    	
    //view_p->widgets.status=NULL;
    //view_p->widgets.paper=NULL;


    // Clean up strings
    g_free (view_p->desktop_dir);   
    g_free (view_p->lp_command);
    g_free (view_p->deepest_dir);
    g_free (view_p->xbuffer);
    g_free (view_p->widgets.workdir);
    
    //
    // Clean up lists
    //
    // Cleanup selection list.
    GSList *t;
    for(t = view_p->selection_list; t && t->data; t = t->next) {
	rfm_destroy_entry ((record_entry_t *) (t->data));
    }
    g_slist_free (view_p->selection_list);
    for(t = view_p->f_list; t && t->data; t = t->next) {
	rfm_destroy_entry ((record_entry_t *) (t->data));
    }
    // Cleanup reselect list.
    for(t = view_p->reselect_list; t && t->data; t = t->next) {
	g_free(t->data);
    }
    g_slist_free (view_p->reselect_list);

    // Cleanup go history.go_list,f_list
    GList *tmp;
    for(tmp = view_p->go_list; tmp; tmp = tmp->next) {
	rfm_destroy_entry ((record_entry_t *) (tmp->data));
    }
    g_list_free (view_p->go_list);
    g_slist_free (view_p->f_list);
    // Cleanup sh_command
    GList *list = view_p->sh_command;
    for (; list && list->data; list=list->next) g_free(list->data);
    g_list_free(view_p->sh_command);


    rfm_mutex_free(view_p->mutexes.reload_mutex);
    rfm_mutex_free(view_p->mutexes.status_mutex);

    rfm_mutex_free(view_p->mutexes.monitor_control );
    rfm_mutex_free(view_p->mutexes.monitor_run_mutex );
    rfm_cond_free(view_p->mutexes.monitor_signal);
    rfm_cond_free(view_p->mutexes.monitor_run_signal);

    rfm_rw_lock_clear(&(view_p->mutexes.monitor_lock)); 
    rfm_rw_lock_clear(&(view_p->mutexes.population_rwlock));
    rfm_rw_lock_clear(&(view_p->mutexes.view_lock));

    gtk_target_list_unref(view_p->mouse_event.target_list);
    rfm_destroy_entry(view_p->en);






    g_free (view_p->eyecandy.desktop_bg_file);
    g_free (view_p->eyecandy.desktop_color);
    g_free (view_p->eyecandy.iconview_color);
    // destroy associated widgets
    // FIXME SEE callback.i for items to destroy
    //rfm_context_function(destroy_widget_f, view_p->widgets.page_label);
    g_free(view_p);
    return NULL;
}


gpointer
janitor(gpointer data){
    NOOP("starting janitor...\n");
    // This is a stop light for initial setup:
    rfm_global_t *rfm_global_p = rfm_global();
    rfm_rw_lock_reader_lock (&(rfm_global_p->setup_lock));
    rfm_rw_lock_reader_unlock (&(rfm_global_p->setup_lock));

    //GtkWidget *window=data;
    do {
	g_mutex_lock(rfm_global_p->status_mutex);
	gint status = rfm_global_p->status;
	TRACE("janitor at window wait...exit=%d\n", (status == STATUS_EXIT));
	g_cond_wait(rfm_global_p->janitor_signal, 
		rfm_global_p->status_mutex);
	status = rfm_global_p->status;
	if (status == STATUS_EXIT) {
	    g_cond_broadcast(rfm_global_p->status_signal);
	}
	g_mutex_unlock(rfm_global_p->status_mutex);
	
	TRACE("Janitor is awake...\n");


	if (status == STATUS_EXIT){
            TRACE("No janitation required here...\n");

        }

        GSList **list_p = rfm_view_list_lock(NULL, "Janitor");
	TRACE("Janitor got the view_list_lock (read)\n");

        GSList *tmp_list =  NULL;
        GSList *list = *list_p;
        for (;list && list->data; list=list->next){
            tmp_list = g_slist_prepend(tmp_list, list->data);
        }
        rfm_view_list_unlock("janitor");

	for (list = tmp_list; list && list->data; list=list->next){
	    view_t *view_p = list->data;
	    gint view_status;
	    if (status == STATUS_EXIT) view_status = status;
	    else view_status = view_p->flags.status;
	    
	    if (view_status == STATUS_EXIT) {
		g_cond_broadcast(view_p->mutexes.monitor_signal);


		// View needs to be destroyed.
		// Is the view thread locked?
		if(!rfm_rw_lock_writer_trylock(&(view_p->mutexes.view_lock))){
		    // Go on to the next view to cleanup
		    TRACE( "janitor: will not destroy view 0x%x (still in use)...\n",
			GPOINTER_TO_INT(view_p));
#ifdef DEBUG_TRACE
		    // This makes trace very verbose
		    // rfm_thread_unreference(NULL, NULL);
#endif
		    continue;
		}
		TRACE( "janitor() will now destroy view 0x%x\n", 
		    GPOINTER_TO_INT(view_p));
		rfm_rw_lock_writer_unlock(&(view_p->mutexes.view_lock));

                
		// Cleanup population 
                // This population lock has an implied view list lock (read).
		rfm_population_write_lock (view_p, "janitor");
		TRACE("destroying population_pp %p\n", (void *)(view_p->population_pp));
		rodent_destroy_population (view_p->population_pp);

                if (view_p->population_hash) g_hash_table_destroy(view_p->population_hash);
		g_free(view_p->population_pp);
                view_p->population_pp = NULL;
		rfm_population_write_unlock (view_p, "janitor");

 
		// finish up. Must lock view list for write before 
                // destroying the view.
                // 
                // This here removes the view from the view list:
                rfm_rm_view(view_p); 

                // this will attempt list for write
                // now it is safe to destroy the view.
                // This here will actually destroy the view:
                // (currently disabled within. Accepted leak).
		destroy_view(view_p);

		continue;
	    } 
	}

	gint active_views = g_slist_length(tmp_list);
        g_slist_free(tmp_list);
	if(active_views <= 0 && status == STATUS_EXIT)break;
	if (active_views <= 0 ) {
            TRACE("janitor setting status to STATUS_EXIT )active_views=%d)\n", active_views);
            rfm_global_p->status = STATUS_EXIT;
        }
		    
    } while (TRUE);
    TRACE("Janitor exiting main_loop now.\n");
    g_main_loop_quit(main_loop);
    // final cleanup done by main executable.
    return NULL;
}

static gboolean updating_pixbuf_hash = FALSE;
static void *
update_pixbuf_hash_f(void *data){
    if (updating_pixbuf_hash) return NULL;
    updating_pixbuf_hash=TRUE;
    NOOP("---->   time to update pixbuf hash!\n");
    rfm_change_icontheme();
    //sleep(1);
    updating_pixbuf_hash=FALSE;
    return NULL;
}

