/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rodent.h"
#include "rfm_modules.h"

#include "primary-internal.h"



static GMainLoop *main_loop=NULL;
GThread *main_loop_thread=NULL;

void rodent_main_loop(void){
    main_loop_thread = g_thread_self();
    main_loop = g_main_loop_new (NULL, FALSE);
    g_main_loop_run(main_loop);
    g_main_loop_unref (main_loop);
}

static void
destroy_population_item (population_t * population_p);
static void *
thread_reload_view (view_t *view_p, xfdir_t *new_xfdir_p);

#include "xfdir.i"
#include "rodent_population.i"
#include "rodent_population_threads.i"

gboolean rodent_is_gnu_backup_type(const gchar *file){
    if (!file) return FALSE;
    if(file[strlen (file) - 1] == '~' || file[strlen (file) - 1] == '%'|| file[strlen (file) - 1] == '#') return TRUE;
    return FALSE;
}

gboolean rodent_is_mime_backup_type(const gchar *file){
    if (!file) return FALSE;
    gchar *e = strrchr(file, '.');
    if (e){
	if (strcmp(e,".old")==0) return TRUE;
	else if (strcmp(e,".bak")==0) return TRUE;
	else if (strcmp(e,".sik")==0) return TRUE;
    }
    return FALSE;
}

gboolean rodent_is_backup_type(const gchar *file){
    if (!file) return FALSE;
    if (rodent_is_gnu_backup_type(file)) return TRUE;
    if (rodent_is_mime_backup_type(file)) return TRUE;
    return FALSE;
}

gint
rodent_full_reload_view (view_t *view_p, record_entry_t * target_en) {
    NOOP(stderr, "rodent_full_reload_view --> reload_f thread\n");
    /*if (rfm_get_gtk_thread() != g_thread_self()) {
	DBG("rodent_full_reload_view is a main thread callback! Call aborted\n");
	return 0;
    }*/
    widgets_t *widgets_p = &(view_p->widgets);

    print_loading_msg(widgets_p, target_en);
    
    reload_t *reload_p = (reload_t *)malloc(sizeof(reload_t));
    if (!reload_p) g_error("malloc: %s", strerror(errno));
    memset(reload_p, 0, sizeof(reload_t));
    reload_p->view_p = view_p;
    reload_p->target_en = target_en;
    
    NOOP(stderr, "starting load thread for: %s\n",
	    (target_en)? target_en->path: "rodent-root");
    rfm_view_thread_create(view_p, reload_f, reload_p, "reload_f");
    return 1;
}

void
rodent_clear_cut_icons (view_t * view_p) {
        NOOP("rodent_clear_cut_icons()...\n");
    if(!rfm_population_try_read_lock (view_p, "rodent_clear_cut_icons")) {
        DBG("rodent_clear_cut_icons():!rfm_population_try_read_lock\n"); 
        return;
    }
    if (!view_p->population_hash){
        DBG("rodent_clear_cut_icons(): no population hash. This should not happen.\n");
        goto done;
    }
    // get pasteboard vector
    gchar **pasteboard_v = rfm_pasteboard_v(view_p);
    if (!pasteboard_v) {
        DBG("rodent_clear_cut_icons():done !pasteboard_v\n");
        goto done;
    }
    gchar **p = pasteboard_v;
    for(; p && *p; p++){
        gboolean expose = FALSE;
        population_t *population_p = g_hash_table_lookup(view_p->population_hash, *p);
        if (population_p){
		if (population_p->flags & (POPULATION_COPIED|POPULATION_CUT)) {
		    expose = TRUE;
		}
		population_p->flags &= (POPULATION_COPIED ^ 0xffffffff);
		population_p->flags &= (POPULATION_CUT ^ 0xffffffff);
        } 
        if(expose){
	    //rfm_expose_item(view_p,population_p);
        }
    }
    g_strfreev(pasteboard_v);
done:
    rfm_population_read_unlock (view_p, "rodent_clear_cut_icons");
}

void
rodent_update_cut_icons (view_t * view_p) {
        NOOP("rodent_update_cut_icons()...\n");
    if(!rfm_population_try_read_lock (view_p, "rodent_update_cut_icons")) {
        DBG("rodent_update_cut_icons():!rfm_population_try_read_lock\n"); 
        return;
    }
    if (!view_p->population_hash){
        DBG("rodent_update_cut_icons(): no population hash. This should not happen.\n");
        goto done;
    }

    // 1. get pasteboard status
    int pasteboard_status = rfm_pasteboard_status (view_p);
    if (!pasteboard_status) {
        DBG("rodent_update_cut_icons():done !pasteboard_status\n");
        goto done;
    }


    // 2. get pasteboard vector
    gchar **pasteboard_v = rfm_pasteboard_v(view_p);
    if (!pasteboard_v) {
        DBG("rodent_update_cut_icons():done !pasteboard_v\n");
        goto done;
    }
    
    // clear out cut/copy emblems...
    // XXX this makes the cut/opied icon be reexposed constantly
    //     which I don't like too much.
    population_t **tmp;
    for(tmp = view_p->population_pp; tmp && *tmp; tmp++) {
        population_t *population_p = *tmp;
        population_p->flags &= (POPULATION_COPIED ^ 0xffffffff);
        population_p->flags &= (POPULATION_CUT ^ 0xffffffff);

    }


    // 3. do your thing for each item, using population_hash
    gchar **p = pasteboard_v;
    for(; p && *p; p++){
        gboolean expose = FALSE;
        population_t *population_p = g_hash_table_lookup(view_p->population_hash, *p);
        if (population_p){
            if (pasteboard_status==2) { //cut
                if (!(population_p->flags & POPULATION_CUT)) {
                    expose = TRUE;
                }
                population_p->flags |= POPULATION_CUT;
                population_p->flags &= (POPULATION_COPIED ^ 0xffffffff);

            } else {
                if (!(population_p->flags & POPULATION_COPIED)) {
                    expose = TRUE;
                }
                population_p->flags |= POPULATION_COPIED;
                population_p->flags &= (POPULATION_CUT ^ 0xffffffff);
            }
        } 
        if(expose){
	    rfm_expose_item(view_p,population_p);
            //rodent_redraw_item(view_p, population_p);
        }
    }
    // 4. free pasteboard vector
    g_strfreev(pasteboard_v);
#if 0
    obsolete.
    // find and update record
    population_t **tmp;
    for(tmp = view_p->population_pp; tmp && *tmp; tmp++) {
        population_t *population_p = *tmp;
        gboolean expose = FALSE;
        if(population_p->en && population_p->en->path) {
            int pasteboard_status = rfm_in_pasteboard (view_p, population_p->en);
            if(pasteboard_status) {
		if (pasteboard_status==2) { //cut
		    if (!(population_p->flags & POPULATION_CUT)) {
			expose = TRUE;
		    }
		    population_p->flags |= POPULATION_CUT;
		    population_p->flags &= (POPULATION_COPIED ^ 0xffffffff);

		} else {
		    if (!(population_p->flags & POPULATION_COPIED)) {
			expose = TRUE;
		    }
		    population_p->flags |= POPULATION_COPIED;
		    population_p->flags &= (POPULATION_CUT ^ 0xffffffff);
		}
                NOOP ("CUT: pasteboard: %s\n", population_p->en->path);
            } else {
                NOOP ("CUT: not in pasteboard: %s\n", population_p->en->path);
		if (population_p->flags & (POPULATION_COPIED|POPULATION_CUT)) {
		    expose = TRUE;
		}
		population_p->flags &= (POPULATION_COPIED ^ 0xffffffff);
		population_p->flags &= (POPULATION_CUT ^ 0xffffffff);
            }
            if(expose){
		rodent_redraw_item(view_p, population_p);
	    }
        }
    }

    // release population semaphore
#endif

done:
    rfm_population_read_unlock (view_p, "rodent_update_cut_icons");
}



static void
start_element (GMarkupParseContext * context,
               const gchar * element_name,
               const gchar ** attribute_names, 
	       const gchar ** attribute_values, 
	       gpointer user_data, 
	       GError ** error) 
{
    if(strcmp (element_name, "keybind")) {
        DBG("start_element(): strcmp (element_name, \"keybind\") \n");
        return;
    }
    if (!attribute_names || !attribute_values) {
        DBG("start_element(): strcmp (element_name, (!attribute_names || !attribute_values)) \n");
        return;
    }

    gchar **a = (gchar **)attribute_names;
    gchar **v = (gchar **)attribute_values;

    for(; a && *a && v && *v; a++, v++){
	if (strcasecmp(*a, "function_id")) continue;
	errno = 0;
	glong function_id = strtol(*v,NULL,0);
	if (errno) continue;
	gchar **aa = (gchar **)attribute_names;
	gchar **vv = (gchar **)attribute_values;
	const gchar *key = NULL;
	const gchar *mask = NULL;
	for(; aa && *aa && vv && *vv; aa++, vv++){
	    if (strcasecmp(*aa, "mask") && strcasecmp(*aa, "key")) continue;
	    if (strcasecmp(*aa, "mask")==0) mask = *vv; else key = *vv;
	}
	if (!key || !mask) continue;
        DBG("setting keybind %s+%s\n", mask, key);
 	rfm_complex(RFM_MODULE_DIR, "callbacks", 
		    GINT_TO_POINTER(function_id),
		    (void *)key, (void *)mask,
		    "set_menu_callback_keybind");
   }
    return;
}

static void
glib_parser (const gchar * mimefile) {
    GMarkupParseContext *context;
    GMarkupParser parser = {
        start_element,
        NULL,
        NULL,                   /*text_fun, */
        NULL,
        NULL
    };

    NOOP(stderr, "glib_parser(icon-module): parsing %s\n", mimefile);
    context = g_markup_parse_context_new (&parser, 0, NULL, NULL);
    FILE *f = fopen (mimefile, "r");
    if(!f) {
        DBG ("cannot open %s\n", mimefile);
        return;
    }
    size_t l;
    gchar line[81];
    GError *error = NULL;
    while(!feof (f) && (l = fread (line, 1, 80, f)) != 0) {
        line[l] = 0;
        g_markup_parse_context_parse (context, line, l, &error);
    }
    fclose (f);
    g_markup_parse_context_free (context);
}


void
rodent_load_keybindings(void){
    gchar *keybindings_file = g_build_filename(KEYBINDINGS_FILE, NULL);
    if (!g_file_test(keybindings_file, G_FILE_TEST_EXISTS)) {
	g_free(keybindings_file);
	return;
    }
    static time_t mtime=0;
    struct stat st;
    stat(keybindings_file, &st);
    if (st.st_mtime != mtime){
	//reload
        DBG("Loading user defined keybindings...\n");
	glib_parser(keybindings_file);
	mtime = st.st_mtime;
    }

    g_free(keybindings_file);
    return;
}


void 
rodent_icontheme_test(void){
    // Icon theme switch:
    static gchar *use_gtk_theme_icons = NULL;
    if (!use_gtk_theme_icons){
	if (getenv("RFM_USE_GTK_ICON_THEME")){
	    use_gtk_theme_icons = 
	       g_strdup(getenv("RFM_USE_GTK_ICON_THEME"));
	} else {
	    use_gtk_theme_icons=g_strdup("");
	}
    }
    const gchar *instant_gtk_theme_icons =
	(getenv("RFM_USE_GTK_ICON_THEME"))?
	   getenv("RFM_USE_GTK_ICON_THEME"):"";

    if (strcmp(use_gtk_theme_icons,instant_gtk_theme_icons) != 0){
	g_free(use_gtk_theme_icons);
	use_gtk_theme_icons = g_strdup(instant_gtk_theme_icons);
	rfm_view_thread_create(NULL, update_pixbuf_hash_f, NULL, "update_pixbuf_hash_f");
    }
}

gchar *rodent_get_tabname(const gchar *path){
    gchar *tabname = g_path_get_basename (path);
#if 0
    // Old way, before pathbar...
    gchar *basename = g_path_get_basename (path);
    gchar *dirname = g_path_get_dirname (path);
    gchar *parentname = g_path_get_basename (dirname);
    gchar *tabname;
    if(strcmp (parentname, ".") == 0 || strcmp (basename, "/") == 0) {
        tabname = g_strdup (basename);
    } else {
        tabname = g_build_filename (parentname, basename, NULL);
    }
    g_free (basename);
    g_free (dirname);
    g_free (parentname);
#endif
    
    gchar *chopped_path=g_strdup(tabname);
    rfm_chop_excess (chopped_path);
    gchar *utf_tabname=rfm_utf_string(chopped_path);
    g_free(chopped_path);
    g_free(tabname);
    return utf_tabname;
}

//*** end progress bar stuff ***/
static void *
set_view_icon_f (gpointer data) {
    view_t * view_p = data;

    NOOP (">> set_iconview_title\n");
    if (!rfm_main_context_check(view_p, TRUE)) return NULL;
    record_entry_t *en = view_p->en;
    gchar *icon_id = rfm_get_entry_icon_id(view_p, en, FALSE);

    if (en && rodent_path_has_bookmark(en->path)) {
	gchar *g = g_strconcat(icon_id, "/compositeNE/emblem_bookmark", NULL);
	g_free(icon_id);
	icon_id = g;
    } 
    
    NOOP("icon resolved as %s\n", icon_id);

    set_application_icon (&(view_p->widgets), icon_id);

    // Set tab icon
    GdkPixbuf *pb = rfm_get_pixbuf (icon_id, 20);
    g_free(icon_id);
    GtkWidget *image = gtk_image_new_from_pixbuf (pb);
    if (pb && G_IS_OBJECT(pb)) g_object_unref(pb);
    gtk_widget_show (image);
    GtkWidget *page_label_icon_box = g_object_get_data(G_OBJECT(view_p->widgets.paper), "page_label_icon_box");
    //if (page_label_icon_box && GTK_IS_CONTAINER(page_label_icon_box)) 
    {
	GList *children = gtk_container_get_children (GTK_CONTAINER (page_label_icon_box));
	if(children) {
	    if (children->data && GTK_IS_WIDGET(children->data)) 
                gtk_container_remove (GTK_CONTAINER (page_label_icon_box), GTK_WIDGET (children->data));
	}
	gtk_container_add (GTK_CONTAINER (page_label_icon_box), image);
	g_list_free (children);
    }
    GtkImage *menu_image = g_object_get_data(G_OBJECT(view_p->widgets.paper), "menu_image");
    if (menu_image) {
	gtk_image_set_from_pixbuf(menu_image, pb);
    }

    return NULL;
}

static void *
set_view_title_f (gpointer data) {
    view_t * view_p = data;

    NOOP (">> set_iconview_title\n");
    if (!rfm_main_context_check(view_p, TRUE)) return FALSE;
    record_entry_t *en = view_p->en;
    const gchar *path;

    if(en && en->path && strlen (en->path)) {
        if (en->pseudo_path) path = en->pseudo_path;
	else path = en->path;
    } else {
        path = g_get_host_name ();
    }

#if 0
    // deprecated: not used anymore
    if(view_p->module) {
        if(view_p->module && rfm_natural (PLUGIN_DIR, view_p->module, view_p->en, "window_title")) {
            path = rfm_natural (PLUGIN_DIR, view_p->module, view_p->en, "window_title");
        } else {
	    NOOP("module has no window_title function\n");
	}
    }
#endif

    rfm_global_t *rfm_global_p = rfm_global();
    set_icon_name (rfm_global_p->window, path);
    gchar *utf_tabname=rodent_get_tabname(path);

    gchar *chopped_path=g_strdup(path);
    rfm_chop_excess (chopped_path);
    gchar *utf_path=rfm_utf_string(chopped_path);
    g_free(chopped_path);

    NOOP(stderr, "setting bold title for %s\n", utf_tabname);
    gchar *markup = g_strconcat("<span weight=\"normal\">", utf_tabname, "</span>", NULL);
    GtkWidget *page_label = g_object_get_data(G_OBJECT(view_p->widgets.paper), "page_label");
    gtk_label_set_markup (GTK_LABEL (page_label), markup);
    g_free (markup);
    GtkLabel *menu_label = g_object_get_data(G_OBJECT(view_p->widgets.paper), "menu_label");
    if (menu_label) {
	markup = g_strconcat("<span weight=\"normal\">", utf_path, "</span>", NULL);
	gtk_label_set_markup(menu_label, markup);
	g_free (markup);
    }
 
    g_free (utf_path);
    g_free (utf_tabname);

    return NULL;
}

void
rodent_set_view_title (view_t * view_p) {
    if(view_p->flags.type == DESKVIEW_TYPE) return;
    rfm_context_function(set_view_title_f, view_p);
    return;
}

void
rodent_set_view_icon (view_t * view_p) {
    if(view_p->flags.type == DESKVIEW_TYPE) return;
    rfm_context_function(set_view_icon_f, view_p);
    return;
}


gint rodent_trigger_reload(view_t *view_p){
    NOOP(stderr, "rodent_trigger_reload()...\n");
    if (g_thread_self() == rfm_get_gtk_thread()) {
	record_entry_t *target_en = NULL;
	if (view_p->en) target_en = rfm_copy_entry(view_p->en);
	return (rodent_full_reload_view (view_p, target_en));
    }
    // Called from a non-main thread...
    if (view_p->en && !IS_LOCAL_TYPE(view_p->en->type)){
	// This is currently in test for remote items...XXX
	// non local type, just a plain reload...
	record_entry_t *target_en = NULL;
	if (view_p->en) target_en = rfm_copy_entry(view_p->en);
	return (rodent_full_reload_view (view_p, target_en));    
    }
	// Monitor will take care of reload here...
    return xfdir_monitor_control_greenlight(&(view_p->widgets));

}
gint
rodent_refresh (widgets_t * widgets_p, record_entry_t * en) {
    NOOP (stderr, "rodent_refresh(%s)...\n", (en) ? en->path : "en==NULL");
    if (en && en->st) NOOP ("en->st->st_uid  %d\n", en->st->st_uid );
    g_return_val_if_fail (widgets_p != NULL, 0);
    view_t *view_p = widgets_p->view_p;
    //view_p->redraw_pixmap = TRUE;
    gint retval;
    if(en) {
	if (en->path && g_path_is_absolute(en->path)){
	    gboolean exists;
	    if (IS_LOCAL_TYPE(en->type)) exists = g_file_test(en->path, G_FILE_TEST_IS_DIR);
	    else exists = rfm_g_file_test_with_wait(en->path, G_FILE_TEST_IS_DIR);
	    if (!exists) {
		rfm_show_text(widgets_p);
		rfm_diagnostics(widgets_p, "xffm/stock_dialog-error", NULL);
		rfm_diagnostics(widgets_p, "xffm_tag/stderr", en->path, ": ", 
			strerror(ENOENT), "\n",NULL);
		return 0;
	    }
	}
        UNSET_UP_TYPE (en->type);
        if (en->st && view_p->en && view_p->en->st) NOOP("source inode=%d target inode=%d\n",
	    (gint)view_p->en->st->st_ino,
	    (gint)en->st->st_ino);
    }
    // The purpose of the following lines is to check whether the target 
    // to refresh is the same as the current view, but path differs.
    // XXX Only for local files (modules may trip here until verified).
    TRACE("rodent_refresh()...\n");
    if (!rfm_view_list_lock(view_p, "rodent_refresh")) return FALSE;
    if (en && IS_LOCAL_TYPE(en->type) && !en->module){
	gboolean valid_entry = (en->st != NULL && en->st->st_mtime);
	gboolean valid_view = view_p->en && view_p->en->st;
	if (valid_view && valid_entry) {
	    gboolean same_inode =  view_p->en->st->st_ino == en->st->st_ino &&
			       view_p->en->st->st_dev == en->st->st_dev;
	    gboolean redundant_condition= strcmp(view_p->en->path, en->path)!=0;
	    if (same_inode && redundant_condition){
		// this condition also is true if a reload is attempted while the
		// original load has not completed.
		//rfm_diagnostics(widgets_p, "xffm/stock_dialog-warning", NULL);
		//rfm_diagnostics(widgets_p, "xffm_tag/stderr", 
		//	strerror(EXDEV),"\n",NULL);
		DBG("rodent_refresh cancelled upon redundant_condition. inode=%ld\n", (long)en->st->st_ino);
                rfm_view_list_unlock("rodent_refresh");
		return 0;
	    }
	}
    }
    TRACE("rodent_refresh calling rodent_full_reload_view()\n");
    retval = rodent_full_reload_view ((gpointer) view_p, en);
                
    rfm_view_list_unlock("rodent_refresh");

    return retval;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////  xfdir  //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

/**
 * SECTION:xfdir
 * @short_description: xfdir structure is a location to place the contents of 
 * the rodent view. The primary xfdir functions allow for the loading and
 * monitoring of local files on a POSIX compliant system.
 * 
 * @section_id:	xfdir
 * @stability: Current
 * @include: primary.h
 *
 **/


/**
 * xfdir_exit_monitor:
 * @xfdir_p:  pointer to an xfdir_t structure
 *
 * Schedule monitor to exit as soon as possible. This function should 
 * be accessed from threads other than the monitor thread. For the monitor thread,
 * use the static exit_monitor()  or rfm_increment_monitor() from outside.
 **/
void
xfdir_exit_monitor (view_t * view_p) {
    view_p->flags.monitor_id++;
    NOOP("xfdir monitor: xfdir_exit_monitor->%d\n", view_p->flags.monitor_id);
}


/**
 * xfdir_count_files:
 * @path: directory path
 * Returns: file count within directory, including hidden files
 *
 * This function is non-recursive.
 **/
gint
xfdir_count_files (const gchar * path) {
    if(!rfm_g_file_test (path, G_FILE_TEST_IS_DIR))
        return 0;
    DIR *directory;
    directory = opendir (path);
    if(!directory)
        return 0;
    gint count = 0;
    struct dirent *d;
    while((d = readdir (directory)) != NULL) {
        if(strcmp (d->d_name, ".") == 0)
            continue;
        if(strcmp (d->d_name, "..") == 0)
            continue;
        count++;
    }
    closedir (directory);
    return count;
}

/**
 * xfdir_count_hidden_files:
 * @path: directory path
 * Returns: hidden file count within directory
 *
 * This function is non-recursive.
 **/
gint
xfdir_count_hidden_files (const gchar * path) {
    if(!rfm_g_file_test (path, G_FILE_TEST_IS_DIR))
        return 0;
    DIR *directory;
    directory = opendir (path);
    if(!directory)
        return 0;
    gint count = 0;
    struct dirent *d;
    while((d = readdir (directory)) != NULL) {
        if(strcmp (d->d_name, ".") == 0)
            continue;
        if(strcmp (d->d_name, "..") == 0)
            continue;
        if(d->d_name[0] != '.')
            continue;
        count++;
    }
    closedir (directory);
    return count;
}

// This is obsolete now...
void
xfdir_register_popup(view_t *view_p, GtkWidget *popup){
    if (!view_p || !GTK_IS_WIDGET(popup)){
	DBG("xfdir_register_popup(): !view_p || !GTK_IS_WIDGET(popup)\n");
	return;
    }
    g_object_set_data(G_OBJECT(view_p->widgets.paper), 
		"private_popup_widget", popup);
}

gboolean
xfdir_monitor_control_greenlight(widgets_t *widgets_p){
    if (!widgets_p){
	TRACE("xfdir_monitor_control_greenlight(): widgets_p is NULL\n");
	return FALSE;
    }
    view_t *view_p = widgets_p->view_p;
    if (!view_p) {
	TRACE("xfdir_monitor_control_greenlight(): view_p is NULL\n");
	return FALSE;
    }
    //rfm_threaded_diagnostics(widgets_p,NULL, g_strdup( "signalling with xfdir_monitor_control_greenlight\n"));
    // Just signal here. Monitor thread may even be dead for all that matters.

    TRACE("xfdir_monitor_control_greenlight() preparing to signal\n");
    g_mutex_lock(view_p->mutexes.monitor_control);
    gboolean active_monitor = view_p->flags.active_monitor;
    view_p->flags.monitor_go = TRUE;
    g_cond_broadcast(view_p->mutexes.monitor_signal);
    g_mutex_unlock(view_p->mutexes.monitor_control);
    TRACE("xfdir_monitor_control_greenlight() has signalled\n");
    return active_monitor;
}

#if 0
static void *
destroy_widget_f(void *data){
    if (!data) return NULL;
    GtkWidget *widget = data;
    gtk_widget_destroy(widget);
    return NULL;
}
#endif



view_t *
rodent_new_view(void){
    view_t *view_p = (view_t *) malloc (sizeof (view_t));
    if (!view_p) g_error("malloc: %s", strerror(errno));
    memset (view_p, 0, sizeof (view_t));

    // initialize the mutex collection 

    rfm_mutex_init(view_p->mutexes.reload_mutex);
    rfm_mutex_init(view_p->mutexes.status_mutex);


    rfm_mutex_init(view_p->mutexes.monitor_control);
    rfm_mutex_init(view_p->mutexes.monitor_run_mutex);
    rfm_cond_init(view_p->mutexes.monitor_signal);
    rfm_cond_init(view_p->mutexes.monitor_run_signal);

	
    rfm_rw_lock_init(&(view_p->mutexes.monitor_lock)); 
    rfm_rw_lock_init(&(view_p->mutexes.population_rwlock));
    rfm_rw_lock_init(&(view_p->mutexes.view_lock));

    view_p->mouse_event.boxX = -1;
    view_p->mouse_event.boxY = -1;

    view_p->flags.preferences = __SHOW_IMAGES;
    view_p->flags.sortcolumn = GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID;
    return view_p;
}

void *
rodent_set_toggle_buttons(void *data){
    if (rfm_get_gtk_thread() != g_thread_self()){
	rfm_context_function(rodent_set_toggle_buttons, data);
	return NULL;
    }
    rfm_global_t *rfm_global_p = rfm_global();
    if (!rfm_global_p) return NULL;
    view_t *view_p = data;
    GtkWidget *button;
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "show_hidden_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), view_p->flags.preferences & __SHOW_HIDDEN);
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "show_backup_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), view_p->flags.preferences & __SHOWS_BACKUP);
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "case_insensitive_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), view_p->flags.preferences & __CASE_INSENSITIVE);
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "show_previews_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), view_p->flags.preferences & __SHOW_IMAGES);
    }
    //sort stuff //////////////////////////////////////////////////////////////
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "Name_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), view_p->flags.sortcolumn == NAME_SORT);
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "Type_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), view_p->flags.sortcolumn == TYPE_SORT);
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "Date_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), view_p->flags.sortcolumn == DATE_SORT);
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "Size_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), view_p->flags.sortcolumn == SIZE_SORT);
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "Owner_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), view_p->flags.sortcolumn == OWNER_SORT);
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "Group_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), view_p->flags.sortcolumn == GROUP_SORT);
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "Mode_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), view_p->flags.sortcolumn == MODE_SORT);
    }
    //  ascending/descending
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "Ascending_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), SHOWS_ASCENDING(view_p->flags.preferences));
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "Descending_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), !SHOWS_ASCENDING(view_p->flags.preferences));
    }
    // iconsize buttons  /////////////////////////////////////////////////////////
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "List_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), ICON_SIZE_ID(view_p) == LIST_ICON_SIZE);
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "Tiny_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), ICON_SIZE_ID(view_p) == TINY_ICON_SIZE);
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "Normal_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), ICON_SIZE_ID(view_p) == SMALL_ICON_SIZE);
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "Big_buttonBig_buttonBig_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), ICON_SIZE_ID(view_p) == MEDIUM_ICON_SIZE);
    }
    button =
	g_object_get_data(G_OBJECT(rfm_global_p->window), "Huge_button");
    if (button){
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), ICON_SIZE_ID(view_p) == BIG_ICON_SIZE);
    }

    return NULL;
}


