/*
 * Copyright (C) 2003-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

#define __MODULES_C__
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rfm.h"
#include "rfm_modules.h"

static GHashTable *module_hash = NULL;


static RfmProgramOptions lite_plugin_options[]={
    {"fstab",N_("Mount local disks and devices"),TRUE,NULL},
    {"ps",N_("View current processes and monitor system state"),TRUE,NULL},
    {"dotdesktop",N_("Next-generation application launcher."),TRUE,NULL},

    {"submodule-label",N_("Windows networks (SMB)"),TRUE,NULL},
    {"smb",N_("SMB Browser"),TRUE,NULL},
    {"submodule-indent",NULL,TRUE,NULL},
	{"workgroup",N_("Windows workgroup"),TRUE,NULL},
	{"shares",N_("Windows Shares"),TRUE,NULL},
    {"submodule-unindent",NULL,TRUE,NULL},

    {"submodule-label",N_("FUSE Volume"),TRUE,NULL},
    {"fuse",N_("Mount user-space filesystems (FUSE)"),TRUE,NULL},
    {"submodule-indent",NULL,TRUE,NULL},
	{"nfs",N_("NFS Network Volume"),TRUE,NULL},
	{"sftp",N_("Secure FTP (SSH)"),TRUE,NULL},
#ifdef THIS_IS_LINUX
	{"ecryptfs",N_("Encrypted filesystem"),TRUE,NULL},
	{"obex",N_("Bluetooth Transfer"),TRUE,NULL},
	{"ftp",N_("FTP Client"),TRUE,NULL},
#endif
	{"cifs",N_("CIFS Volume"),TRUE,NULL},
    {"submodule-unindent",NULL,TRUE,NULL},
    {NULL,NULL,FALSE,NULL}
};

static RfmProgramOptions lite_module_options[]={
    {"callbacks",N_("Callbacks"),FALSE,NULL},
    {"bcrypt",N_("Blowfish"),FALSE,NULL},
    {"settings",N_("Settings"),FALSE,NULL},
    {"run",N_("Run program and return its output"),FALSE,NULL},
    {"properties",N_("Properties"),TRUE,NULL},
    {"completion",N_("Text completion"),TRUE,NULL},
    {"combobox",N_("History of combo url."),TRUE,NULL},
    {"mime",N_("Mime Type"),FALSE,NULL},
    {"mimemagic",N_("Use MIME type magic"),TRUE,NULL},
    {"mimezip",N_("ZIP archive plugin"),TRUE,NULL},
    {"icons",N_("Icon Themes"),TRUE,NULL},
     {NULL,NULL,FALSE,NULL}
};

RfmProgramOptions *rfm_get_lite_module_options(void){return lite_module_options;}
RfmProgramOptions *rfm_get_lite_plugin_options(void){return lite_plugin_options;}

// This is to set downstream modules/plugins to be used...
static gchar *plugin_dir=NULL; 

const gchar *rfm_set_module_dir(const gchar *libdir, const gchar *dir){
    DBG("rfm_set_module_dir() is no longer used\n");
    return NULL;
}

const gchar *rfm_set_plugin_dir(const gchar *libdir, const gchar *dir){
    g_free(plugin_dir);
    plugin_dir = g_build_filename (libdir, dir, NULL);
    return (const gchar *) plugin_dir;
}

const gchar *rfm_plugin_dir(void){
    if (!plugin_dir){
        plugin_dir = g_build_filename (LIBDIR, "rfm", "plugins", NULL);

        DBG("rfm_set_plugin_dir() not set. Using default %s\n",plugin_dir);
    }
    return (const gchar *) plugin_dir;
}

static pthread_mutex_t *
get_module_hash_mutex(void){
    static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
    return &mutex;
}


const gchar *
rfm_get_module_popup_id(void){
    return "module_popup_widget";
}
#if 0
static gboolean
get_lite_active(const gchar *module_name, RfmProgramOptions *options_p, gint64 flags, gboolean root_test){
    gint i;
    gboolean indent = FALSE;
    for (i=0;options_p && options_p->option; options_p++,i++){
	if (strcmp(options_p->option,"submodule-indent")==0) {
	    indent = TRUE;
	    continue;
	}
	if (strcmp(options_p->option,"submodule-unindent")==0) {
	    indent = FALSE;
	    continue;
	}
	if (indent && root_test) continue;
	if (strcmp(module_name, options_p->option)==0) {
	    if (flags & (((gint64)0x01)<<i)) return TRUE;
	}
    }
    return FALSE;
}

static 
gint64 get_bin_flag(const gchar *variable, RfmProgramOptions *options_p){
    const gchar *flag = getenv(variable);
    gint64 flags;
    if (!flag || !strlen(flag)) flags = 0L;
    else {
	errno=0;
	flags = (gint64) strtoll(flag, NULL, 16);
	if (errno){
	    if (options_p == lite_module_options) flags = (gint64)0x3L;
	    else flags = 0;
	    DBG("%s: %s\n", variable, strerror(errno));
	}
    }
    return flags;
}
#endif
	
gboolean 
rfm_is_root_plugin(const gchar *module_name){
    if (!module_name) return FALSE;
    return (GPOINTER_TO_INT(rfm_void(PLUGIN_DIR, module_name, "is_root_module")));
}
const gchar * 
rfm_get_plugin_icon(const gchar *module_name){
    if (!module_name) return NULL;
    return (rfm_void(PLUGIN_DIR, module_name, "module_icon_id"));
}
gchar * 
rfm_get_plugin_label(const gchar *module_name){
    if (!module_name) return NULL;
    return (rfm_void(PLUGIN_DIR, module_name, "module_label"));
}

void
rfm_sanity_check (int argc, char **argv, int librfm_serial) {
    if(librfm_serial != LIBRFM_SERIAL) {
        gchar *p = g_strdup_printf ("%s needs to be recompiled \n(has obsolete library headers)", argv[0]);
	rfm_confirm(NULL,GTK_MESSAGE_ERROR,p,NULL,NULL);
	g_assert_not_reached();
	//g_error("%s\n", p);
    }
    return;
}

static gint
compare_strings (gconstpointer a, gconstpointer b) {
    return (strcmp ((char *)a, (char *)b));
}

GSList *
rfm_find_plugins (void) {
    static GSList *plugin_list = NULL;
    static gsize initialized = 0;

    // Only once...
    if (g_once_init_enter(&initialized)){
	GError *error = NULL;

	GDir *dir;
	if((dir = g_dir_open (PLUGIN_DIR, 0, &error)) != NULL) {
	    const gchar *name;
	    while((name = g_dir_read_name (dir)) != NULL) {
		gchar *plugin;
		if(strncmp (name, "lib", strlen ("lib"))){
		    plugin = g_strdup (name);
		} else {
		    plugin = g_strdup (name + strlen ("lib"));
		}
		if(strchr (plugin, '.')) plugin = strtok (plugin, ".");

		if(g_slist_find_custom (plugin_list, plugin, compare_strings)) {
		    g_free (plugin);
		} else {
		    NOOP ("modules: g_slist_append %s\n", plugin);
		    plugin_list = g_slist_prepend (plugin_list, plugin);
		}
	    }
	    g_dir_close (dir);
	}
	g_once_init_leave(&initialized, 1);
    }
    return plugin_list;
}


/*************************************************************/

//void unload_module (const gchar * module_name);

static void *
module_error(const gchar * module_name, gchar *module, gchar *warning){
    pthread_mutex_t *module_hash_mutex = get_module_hash_mutex();
    pthread_mutex_lock(module_hash_mutex);
    g_hash_table_insert (module_hash, (gpointer) module_name, GINT_TO_POINTER(-1));
    pthread_mutex_unlock(module_hash_mutex);
    DBG ("Module %s (%s): %s\n", module, module_name, warning);
    g_free (warning);
    g_free (module);
    return NULL;
}


GModule *
get_module_info (const gchar * librarydir, const gchar * module_name) {
    if (!module_name || !librarydir || !strlen(module_name) ) {
	DBG("Failed to get module info on !module_name (%p) || !librarydir (%p) || !strlen(module_name) \"%s\"\n",
                module_name, librarydir, module_name);
	return NULL;
    }
    rfm_global_t *rfm_global_p = rfm_global();
    if (rfm_global_p && rfm_global_p->status == STATUS_EXIT) {
	TRACE("Refusing to get module info on STATUS_EXIT for %s\n",module_name);
	return NULL;
    }
    GModule *module_cm = NULL;
    void *(*module_sanity) (void);
    int init_version;

    gchar *module = g_module_build_path (librarydir, module_name);

    if(!rfm_g_file_test (module, G_FILE_TEST_EXISTS)) {
	DBG("Module \"%s\" does not exist!\n", module_name);
	g_free (module);
	return NULL;
    }

    if (strstr(module_name,"gridview"))DBG("getting module info for %s (librarydir=%s)\n", module, librarydir);

    if(!module_hash) {
        TRACE("get_module_info: creating module hash\n");
        module_hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
        if(!module_hash) g_assert_not_reached ();
    }
    NOOP ("looking for module in hash: %s\n", module_name);
    pthread_mutex_t *module_hash_mutex = get_module_hash_mutex();
    pthread_mutex_lock(module_hash_mutex);
    module_cm = g_hash_table_lookup (module_hash, module_name);
    pthread_mutex_unlock(module_hash_mutex);
    if (GPOINTER_TO_INT(module_cm) == -1){
	DBG("Module %s cannot be loaded.\n", module);
	g_free(module);
	return NULL;
    }
    if(module_cm != NULL) {
        NOOP("module info for %s found (already loaded) 0x%x\n",
		module,
		GPOINTER_TO_INT(module_cm));
	g_free(module);
        return module_cm;
    } 

    NOOP ("get_module_info(): RSS changes with load of %s\n", module);
    // Here's the rub: lazy mode willfetch a symbol with the same name
    // from another module than that which was specified. 
    // Only use lazy if different modules do not export symbols with the
    // same name (or just do not use lazy).
#ifdef DEBUG
    module_cm = g_module_open (module, G_MODULE_BIND_LAZY | G_MODULE_BIND_LOCAL);
#else
    module_cm = g_module_open (module, G_MODULE_BIND_LOCAL);
#endif

    if(!module_cm) {
	return module_error(module_name, module, 
		g_strdup("g_module_open() == NULL\nmodule cannot be opened: Check if correctly installed or unresolved symbols within...\n****\n"));
    }

    // Check for module sanity 
    if(!g_module_symbol (module_cm, "module_sanity", (gpointer) & (module_sanity))) {
	return module_error(module_name, module, 
		g_strdup ("Module is not sane!\nDoes module specify: \"LIBRFM_MODULE\"?"));
    } 
    
    // Check for construction with correct structure serial (types.h)
    init_version = GPOINTER_TO_INT ((*module_sanity) ());
    if(init_version != LIBRFM_SERIAL) {
	return module_error(module_name, module, 
		g_strdup ("Module is compiled with obsolete headers (not loaded)"));
    }



    NOOP ("adding module to module hash: %s (0x%lx)\n", module_name, (unsigned long)module_cm);
    pthread_mutex_lock(module_hash_mutex);
    g_hash_table_insert (module_hash, g_strdup(module_name), module_cm);
    pthread_mutex_unlock(module_hash_mutex);
    TRACE("get_module_info: module %s successfully loaded\n", module);
    g_free (module);
    return module_cm;
    
}


void rfm_destroy_module_hash(void){
    pthread_mutex_t *module_hash_mutex = get_module_hash_mutex();
    pthread_mutex_lock(module_hash_mutex);
    if (module_hash) g_hash_table_destroy(module_hash);
    pthread_mutex_unlock(module_hash_mutex);
}


void *
rfm_void (const gchar * librarydir, const gchar * module_name, const gchar * function_id) {
    if (!module_name || !strlen(module_name)) return NULL;
    gchar *(*function) (void);
    if(!librarydir || !module_name || !function_id) return NULL;
    TRACE("attempting module function: %s\n", function_id);
    GModule *module_cm = get_module_info (librarydir, module_name);
    if(!module_cm){
	DBG("get_module_info failed: %s\n", module_name);
	return NULL;
    }

    if(!g_module_symbol (module_cm, function_id, (gpointer) & (function))) {
        /*if (strcmp(function_id,"submodule_name")!=0 && strcmp(function_id,"submodule_dir")!=0) */
        {
            TRACE("g_module_symbol(%s) != FALSE in module %s\n", function_id, module_name);
        }
        /*unload_module(module_name); */
        return NULL;
    }
    return (*function) ();
}

void *
rfm_natural (const gchar * librarydir, const gchar * module_name, void *n, const gchar * function_id) {
    gchar *(*function) (void *n);
    if(!librarydir || !module_name || !function_id){
	DBG("!librarydir (%s)|| !module_name (%s)|| !function_id(%s)\n",
		librarydir, module_name, function_id);
        return NULL;
    }
    GModule *module_cm = get_module_info (librarydir, module_name);
    if(!module_cm) return NULL;
    if(!g_module_symbol (module_cm, function_id, (gpointer) & (function))) {
        if(strcmp (function_id, "submodule_name") != 0 && strcmp (function_id, "submodule_dir") != 0) {
            DBG ("g_module_symbol(%s) != FALSE in module %s\n", function_id, module_name);
        }
        /*unload_module(module_name); */
        return NULL;
    }
    return (*function) (n);
}

void *
rfm_rational (const gchar * librarydir, const gchar * module_name, void *p, void *q, const gchar * function_id) {
    gchar *(*function) (void *p, void *q);
    if(!librarydir || !module_name || !function_id) return NULL;
    GModule *module_cm = get_module_info (librarydir, module_name);
    if(!module_cm) return NULL;
    if(!g_module_symbol (module_cm, function_id, (gpointer) & (function))) {
        if(strcmp (function_id, "submodule_name") != 0 && strcmp (function_id, "submodule_dir") != 0) {
            DBG ("g_module_symbol(%s) != FALSE in module %s\n", function_id, module_name);
        }
        /*unload_module(module_name); */
        return NULL;
    }
    return (*function) (p, q);
}

void *
rfm_complex (const gchar * librarydir, const gchar * module_name, void *p, void *q, void *r, const gchar * function_id) {
    gchar *(*function) (void *p, void *q, void *r);
    if(!librarydir || !module_name || !function_id) return NULL;
    GModule *module_cm = get_module_info (librarydir, module_name);
    if(!module_cm) return NULL;
    if(!g_module_symbol (module_cm, function_id, (gpointer) & (function))) {
        if(strcmp (function_id, "submodule_name") != 0 && strcmp (function_id, "submodule_dir") != 0) {
            DBG ("g_module_symbol(%s) != FALSE in module %s\n", function_id, module_name);
        }
        /*unload_module(module_name); */
        return NULL;
    }
    return (*function) (p, q, r);
}

static void **
argument_container_new(gint size, const void **in_vector){
    void **argument_container = (void **)malloc(size * sizeof(void *));
    if (!argument_container) g_error("malloc %s\n", strerror(errno));
    memset(argument_container, 0, size * sizeof(void *));
    if (in_vector){
	gint i;
	for (i=0; i < size; i++) argument_container[i] = (void *)in_vector[i];
    }
    return argument_container;
}

   
G_MODULE_EXPORT
void *
rfm_vector_run(const gchar * librarydir, const gchar * module_name, 
	void *vector_size_p, const void **vector, const gchar *function_id){

    void **arg = argument_container_new(GPOINTER_TO_INT(vector_size_p), vector);
    TRACE("vector run %d: %s\n", GPOINTER_TO_INT(vector_size_p), function_id);
    return rfm_natural(librarydir, module_name, arg, function_id);
}


#if 0
void *
load_module (const gchar * librarydir, const gchar * module_name) {
    GModule *module_cm = get_module_info (librarydir, module_name);
    NOOP ("finished load attempt of module %s (%s)\n", module_name, librarydir);
    if(!module_cm) {
        DBG ("Show stopper: cannot get module info for %s\n", module_name);
        g_assert_not_reached ();
    }
    return GINT_TO_POINTER(1);
}
#endif

void
rfm_unload_module (const gchar * module_name) {

    if(!module_hash) return;
    pthread_mutex_t *module_hash_mutex = get_module_hash_mutex();
    pthread_mutex_lock(module_hash_mutex);
    GModule *module_cm = g_hash_table_lookup (module_hash, module_name);
    pthread_mutex_unlock(module_hash_mutex);
    if(!module_cm) {
        DBG ("module %s is not loaded\n", module_name);
        return;
    }
        TRACE ("unloading module %s\n", module_name);

    if(!g_module_close (module_cm)) {
        DBG ("g_module_close (%s) failed\n", module_name);
    } else {
    pthread_mutex_lock(module_hash_mutex);
        if(!g_hash_table_remove (module_hash, module_name)) {
            DBG ("could not remove %s from module hash\n", module_name);
        }
    pthread_mutex_unlock(module_hash_mutex);
        TRACE("module %s unloaded\n", module_name);
    }
}


/*********************************************************************************************/


