/* Copyright (C) 2014 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/

/*
  All this file does is to provide a singleton mutex to protect
  guile's make-fresh-user-module and a means of initializing the guile
  FFI implementing the locking of that mutex, where CGU_USE_GUILE is
  defined in cgu_config.h.  That is because make-fresh-user-module is
  not of itself thread safe up to at least guile-2.0.11 without
  external locking.  Where CGU_USE_GUILE is defined, no mutex is
  actually constructed unless and until get_mutex() is called (it is
  constructed lazily).

  This file does not include the extension.h header file because that
  is not necessary: it is entirely self-contained.  The public
  interface consists only of two function, get_user_module_mutex() and
  init_mutex.
*/

#include <c++-gtk-utils/cgu_config.h>

#ifdef CGU_USE_GUILE

#include <c++-gtk-utils/lib_defs.h>
#include <c++-gtk-utils/mutex.h>

#include <pthread.h>

static Cgu::Thread::Mutex* module_mutex = 0;

// provide the arguments for pthread_once with both C linkage
// specification and internal linkage
extern "C" {
  static pthread_once_t mutex_once_control = PTHREAD_ONCE_INIT;
  static void mutex_once_func() {
    try {
      module_mutex = new Cgu::Thread::Mutex;
    }
    catch (...) {
      // We could catch either std::bad_alloc or
      // Cgu::Thread::MutexError.  If so, just do nothing.  The error
      // will be signalled by init_mutex() returning false.  It is
      // safe to use a catch-all here, because thread cancellation is
      // blocked whenever init_mutex() is called by code in
      // extension.h
    }
  }
} // extern "C"

namespace Cgu {

namespace Extension {

Cgu::Thread::Mutex* get_user_module_mutex() {
  return module_mutex; // NULL on error
}

bool init_mutex() {
  pthread_once(&mutex_once_control, mutex_once_func);
  return (module_mutex != 0);
}

} // namespace Extension

} // namespace Cgu

#endif // CGU_USE_GUILE
