/*
 * Decompiled with CFR 0.152.
 */
package com.limegroup.gnutella.downloader;

import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;
import com.limegroup.gnutella.RemoteFileDesc;
import com.limegroup.gnutella.URN;
import com.limegroup.gnutella.UrnSet;
import com.limegroup.gnutella.downloader.VerifyingFile;
import com.limegroup.gnutella.downloader.VerifyingFileFactory;
import com.limegroup.gnutella.library.IncompleteFileCollection;
import com.limegroup.gnutella.library.Library;
import com.limegroup.gnutella.tigertree.HashTreeCache;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import org.limewire.bittorrent.Torrent;
import org.limewire.bittorrent.TorrentManager;
import org.limewire.collection.Comparators;
import org.limewire.collection.Range;
import org.limewire.core.settings.SharingSettings;
import org.limewire.io.InvalidDataException;
import org.limewire.util.CommonUtils;
import org.limewire.util.FileUtils;
import org.limewire.util.OSUtils;

@Singleton
public class IncompleteFileManager {
    static final String SEPARATOR = "-";
    public static final String PREVIEW_PREFIX = "Preview-";
    public static final String INCOMPLETE_PREFIX = "T-";
    private final Map<File, VerifyingFile> blocks = new TreeMap<File, VerifyingFile>(Comparators.fileComparator());
    private final Map<URN, File> hashes = new HashMap<URN, File>();
    private final Provider<Library> library;
    private final Provider<IncompleteFileCollection> incompleteFileCollection;
    private final Provider<HashTreeCache> tigerTreeCache;
    private final VerifyingFileFactory verifyingFileFactory;
    private final Provider<TorrentManager> torrentManager;

    @Inject
    public IncompleteFileManager(Provider<Library> library, Provider<IncompleteFileCollection> incompleteFileCollection, Provider<HashTreeCache> tigerTreeCache, VerifyingFileFactory verifyingFileFactory, Provider<TorrentManager> torrentManager) {
        this.library = library;
        this.incompleteFileCollection = incompleteFileCollection;
        this.tigerTreeCache = tigerTreeCache;
        this.verifyingFileFactory = verifyingFileFactory;
        this.torrentManager = torrentManager;
    }

    public synchronized boolean purge() {
        boolean ret = false;
        Iterator<File> iter = this.blocks.keySet().iterator();
        while (iter.hasNext()) {
            File file = iter.next();
            if (file.exists()) continue;
            ret = true;
            this.library.get().remove(file);
            file.delete();
            iter.remove();
        }
        return ret;
    }

    public synchronized boolean initialPurge(Collection<File> activeFiles) {
        File file;
        boolean ret = false;
        Iterator<File> iter = this.blocks.keySet().iterator();
        while (iter.hasNext()) {
            file = iter.next();
            try {
                file = FileUtils.getCanonicalFile(file);
            }
            catch (IOException iox) {
                file = file.getAbsoluteFile();
            }
            if (file.exists() && (!IncompleteFileManager.isOld(file) || activeFiles.contains(file))) continue;
            ret = true;
            this.library.get().remove(file);
            file.delete();
            iter.remove();
        }
        iter = this.hashes.values().iterator();
        while (iter.hasNext()) {
            file = iter.next();
            if (file.exists()) continue;
            iter.remove();
            ret = true;
        }
        return ret;
    }

    private static final boolean isOld(File file) {
        long days = SharingSettings.INCOMPLETE_PURGE_TIME.getValue();
        long purgeTime = System.currentTimeMillis() - days * 24L * 60L * 60L * 1000L;
        return file.lastModified() < purgeTime;
    }

    static boolean same(RemoteFileDesc rfd1, RemoteFileDesc rfd2) {
        return IncompleteFileManager.same(rfd1.getFileName(), rfd1.getSize(), rfd1.getSHA1Urn(), rfd2.getFileName(), rfd2.getSize(), rfd2.getSHA1Urn());
    }

    static boolean same(String name1, long size1, URN hash1, String name2, long size2, URN hash2) {
        if (hash1 != null && hash2 != null) {
            return hash1.equals(hash2);
        }
        return size1 == size2 && name1.equals(name2);
    }

    private static File canonicalize(File f) throws IOException {
        f = f.getAbsoluteFile();
        if (OSUtils.isWindows()) {
            return f;
        }
        return f.getCanonicalFile();
    }

    public synchronized File getFile(RemoteFileDesc rfd) throws IOException {
        return this.getFile(rfd.getFileName(), rfd.getSHA1Urn(), rfd.getSize());
    }

    public synchronized File getFile(String name, URN sha1, long size) throws IOException {
        return this.getFile(name, sha1, size, SharingSettings.INCOMPLETE_DIRECTORY.get());
    }

    public synchronized File getFile(String name, URN sha1, long size, File incDir) throws IOException {
        boolean dirsMade = false;
        File baseFile = null;
        File canonFile = null;
        dirsMade = incDir.mkdirs();
        String convertedName = CommonUtils.convertFileName(name);
        try {
            File f;
            if (sha1 != null) {
                File file = this.hashes.get(sha1);
                if (file != null) {
                    return file;
                }
                int i = 1;
                while (true) {
                    baseFile = file = new File(incDir, IncompleteFileManager.tempName(convertedName, size, i));
                    canonFile = file = IncompleteFileManager.canonicalize(file);
                    if (!this.hashes.values().contains(file)) break;
                    ++i;
                }
                this.hashes.put(sha1, file);
                FileUtils.touch(file);
                return file;
            }
            baseFile = f = new File(incDir, IncompleteFileManager.tempName(convertedName, size, 0));
            canonFile = f = IncompleteFileManager.canonicalize(f);
            return f;
        }
        catch (IOException ioe) {
            IOException ioe2 = new IOException("dirsMade: " + dirsMade + "\ndirExist: " + incDir.exists() + "\nbaseFile: " + baseFile + "\ncannFile: " + canonFile);
            ioe2.initCause(ioe);
            throw ioe2;
        }
    }

    public synchronized File getFileForUrn(URN urn) {
        if (urn == null) {
            throw new NullPointerException("null urn");
        }
        return this.hashes.get(urn);
    }

    private static String tempName(String filename, long size, int suffix) {
        if (suffix <= 1) {
            return INCOMPLETE_PREFIX + size + SEPARATOR + filename;
        }
        int i = filename.lastIndexOf(46);
        if (i < 0) {
            return INCOMPLETE_PREFIX + size + SEPARATOR + filename + " (" + suffix + ")";
        }
        String noExtension = filename.substring(0, i);
        String extension = filename.substring(i);
        return INCOMPLETE_PREFIX + size + SEPARATOR + noExtension + " (" + suffix + ")" + extension;
    }

    public synchronized void removeEntry(File incompleteFile) {
        this.blocks.remove(incompleteFile);
        Iterator<Map.Entry<URN, File>> iter = this.hashes.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<URN, File> entry = iter.next();
            if (!incompleteFile.equals(entry.getValue())) continue;
            iter.remove();
        }
        this.library.get().remove(incompleteFile);
    }

    public synchronized void initEntry(File incompleteFile, List<Range> ranges, URN sha1, boolean publish) throws InvalidDataException {
        VerifyingFile verifyingFile;
        try {
            incompleteFile = IncompleteFileManager.canonicalize(incompleteFile);
        }
        catch (IOException iox) {
            throw new InvalidDataException(iox);
        }
        try {
            verifyingFile = this.verifyingFileFactory.createVerifyingFile(IncompleteFileManager.getCompletedSize(incompleteFile));
        }
        catch (IllegalArgumentException iae) {
            throw new InvalidDataException(iae);
        }
        if (ranges != null) {
            for (Range range : ranges) {
                verifyingFile.addInterval(range);
            }
        }
        if (ranges == null || ranges.isEmpty()) {
            try {
                verifyingFile.setScanForExistingBlocks(true, incompleteFile.length());
            }
            catch (IOException iox) {
                throw new InvalidDataException(iox);
            }
        }
        this.blocks.put(incompleteFile, verifyingFile);
        if (sha1 != null) {
            this.hashes.put(sha1, incompleteFile);
        }
        if (publish) {
            this.registerIncompleteFile(incompleteFile);
        }
    }

    public synchronized void addEntry(File incompleteFile, VerifyingFile vf, boolean publish) {
        try {
            incompleteFile = IncompleteFileManager.canonicalize(incompleteFile);
        }
        catch (IOException ignored) {
            // empty catch block
        }
        this.blocks.put(incompleteFile, vf);
        if (publish) {
            this.registerIncompleteFile(incompleteFile);
        }
    }

    public synchronized VerifyingFile getEntry(File incompleteFile) {
        return this.blocks.get(incompleteFile);
    }

    public synchronized long getBlockSize(File incompleteFile) {
        VerifyingFile vf = this.blocks.get(incompleteFile);
        if (vf == null) {
            return 0L;
        }
        return vf.getBlockSize();
    }

    public synchronized void registerAllIncompleteFiles() {
        for (File file : this.blocks.keySet()) {
            if (!file.exists() || IncompleteFileManager.isOld(file)) continue;
            this.registerIncompleteFile(file);
        }
    }

    private synchronized void registerIncompleteFile(File incompleteFile) {
        Set<URN> completeHashes = this.getAllCompletedHashes(incompleteFile);
        if (completeHashes.size() == 0) {
            return;
        }
        this.incompleteFileCollection.get().addIncompleteFile(incompleteFile, completeHashes, IncompleteFileManager.getCompletedName(incompleteFile), IncompleteFileManager.getCompletedSize(incompleteFile), this.getEntry(incompleteFile));
    }

    public static String getCompletedName(File incompleteFile) throws IllegalArgumentException {
        String name = incompleteFile.getName();
        int i = name.indexOf(SEPARATOR);
        if (i < 0) {
            throw new IllegalArgumentException("Missing separator: " + name);
        }
        int j = name.indexOf(SEPARATOR, i + 1);
        if (j < 0) {
            throw new IllegalArgumentException("Missing separator: " + name);
        }
        if (j == name.length() - 1) {
            throw new IllegalArgumentException("No name after last separator");
        }
        return name.substring(j + 1);
    }

    public static long getCompletedSize(File incompleteFile) throws IllegalArgumentException {
        String name = incompleteFile.getName();
        int i = name.indexOf(SEPARATOR);
        if (i < 0) {
            throw new IllegalArgumentException("Missing separator: " + name);
        }
        int j = name.indexOf(SEPARATOR, i + 1);
        if (j < 0) {
            throw new IllegalArgumentException("Missing separator: " + name);
        }
        try {
            return Long.parseLong(name.substring(i + 1, j));
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("Bad number format: " + name);
        }
    }

    public synchronized URN getCompletedHash(File incompleteFile) {
        for (Map.Entry<URN, File> entry : this.hashes.entrySet()) {
            if (!incompleteFile.equals(entry.getValue()) || !entry.getKey().isSHA1()) continue;
            return entry.getKey();
        }
        return null;
    }

    public synchronized Set<URN> getAllCompletedHashes(File incompleteFile) {
        UrnSet urns = new UrnSet();
        for (Map.Entry<URN, File> entry : this.hashes.entrySet()) {
            if (!incompleteFile.equals(entry.getValue())) continue;
            urns.add(entry.getKey());
            URN ttroot = this.tigerTreeCache.get().getHashTreeRootForSha1(entry.getKey());
            if (ttroot == null) continue;
            urns.add(ttroot);
        }
        return urns;
    }

    public synchronized String toString() {
        StringBuilder buf = new StringBuilder();
        buf.append("{");
        boolean first = true;
        for (File file : this.blocks.keySet()) {
            if (!first) {
                buf.append(", ");
            }
            List<Range> intervals = this.blocks.get(file).getVerifiedBlocksAsList();
            buf.append(file);
            buf.append(":");
            buf.append(intervals.toString());
            first = false;
        }
        buf.append("}");
        return buf.toString();
    }

    public synchronized String dumpHashes() {
        return this.hashes.toString();
    }

    public Collection<File> getUnregisteredIncompleteFilesInDirectory(File value) {
        if (value == null) {
            return Collections.emptyList();
        }
        File[] files = value.listFiles(new FileFilter(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public boolean accept(File incompleteFile) {
                if (!incompleteFile.isFile()) {
                    return false;
                }
                String name = incompleteFile.getName();
                if (IncompleteFileManager.isTorrentFile(incompleteFile)) {
                    Torrent torrent = ((TorrentManager)IncompleteFileManager.this.torrentManager.get()).getTorrent(incompleteFile);
                    return torrent == null;
                }
                if (!name.startsWith(IncompleteFileManager.INCOMPLETE_PREFIX)) {
                    return false;
                }
                int i = name.indexOf(IncompleteFileManager.SEPARATOR);
                if (i < 0 || i == name.length() - 1) {
                    return false;
                }
                int j = name.indexOf(IncompleteFileManager.SEPARATOR, i + 1);
                if (j < 0 || j == name.length() - 1) {
                    return false;
                }
                try {
                    Long.parseLong(name.substring(i + 1, j));
                }
                catch (NumberFormatException e) {
                    return false;
                }
                IncompleteFileManager incompleteFileManager = IncompleteFileManager.this;
                synchronized (incompleteFileManager) {
                    return !IncompleteFileManager.this.blocks.containsKey(FileUtils.canonicalize(incompleteFile));
                }
            }
        });
        if (files == null) {
            return Collections.emptyList();
        }
        return Arrays.asList(files);
    }

    public static boolean isTorrentFile(File incompleteFile) {
        return "torrent".equals(FileUtils.getFileExtension(incompleteFile));
    }
}

