
/*   tubo.c  */

/*  A program independent forking object module for gtk based programs.
 *  
 *  Copyright 2000-2006(C)  Edscott Wilson Garcia under GNU GPL
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif


#include <sys/time.h>


#ifndef HAVE_SYS_TYPES_H 
#error "This program requieres  the <sys/types.h> header file."
#endif
#ifndef HAVE_SYS_WAIT_H
#error "This program requieres  the <sys/wait.h> header file."
#endif
#ifndef HAVE_SIGNAL_H
#error "This program requieres  the <signal.h> header file."
#endif

/* signal handling */
#ifndef HAVE_SIGNAL
#ifndef HAVE_SIGACTION
#error "This program needs signals to work!"
#endif
#endif

#define H(x) 
#define XH(x) 

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <unistd.h>
#include <stdarg.h>
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/select.h>

#include <glib.h>
#include <gmodule.h>

/* don't change for gthreads */
/*#include <pthread.h>*/


#define USE_SHMEM
#ifdef USE_SHMEM
#include <sys/mman.h>

typedef struct arena_t{
    char greenlight;
    char thread_busy;
    char thread_running;
    pid_t pid;
    int src;
}arena_t;
#define PAGESIZE sizeof(arena_t)
#endif

/* public stuff */
#include "tubo.h"

/* private stuff */

/* undefine this variable to leave zombies behind
 * (this will allow you follow program path in debugging): */
#define FORK_OFF

typedef struct fork_structure
{
    pid_t PID;
    int tubo[3][3];
    void *user_data; 
    /* user fork function: */
    void (*fork_function) (void *);
    void (*fork_finished_function) (pid_t, void *);
    /* user parse functions: */
    int (*operate_stdout) (int, void *, void *);
    int (*operate_stderr) (int, void *, void *);
    int read_delay; 
    int input_redlight;
    void *m;
    int shm;
    char shm_name[64];
    gboolean blockread;
}
fork_struct;

#define TUBO_BLK_SIZE 256	/* 256 is the size of my cpu internal cache */

static GList * valid_fork_objects=NULL;
static char line[TUBO_BLK_SIZE];

static gint watch_input(gpointer data);

/* memcpy is necesary  */
#ifdef __GNUC__
/* memcpy is a GNU extension.*/
#define MEMCPY memcpy
#else
static
void *
MEMCPY(void *dest, const void *src, size_t n)
{
    char *destC, *srcC;
    size_t i;

    destC = (char *)dest;
    srcC = (char *)src;
    for(i = 0; i < n; i++)
	destC[i] = srcC[i];
    return dest;
}
#endif

static 
fork_struct *
TuboClosePipes(fork_struct * forkO)
{
    int i;
    H(printf("TuboClosePipes()\n");)
    if(!forkO)
    {
	return NULL;
    }
    for(i = 0; i < 3; i++)
    {
	if(forkO->tubo[i][0] > 0)
	{
	    close(forkO->tubo[i][0]);
	    forkO->tubo[i][0]=0;
	}
	if(forkO->tubo[i][1] > 0)
	{
	    close(forkO->tubo[i][1]);
	    forkO->tubo[i][1]=0;
	}
	if(forkO->tubo[i][2] > 0)
	{
	    XH(printf("removing input signal %d\n", i);)
	    g_source_remove(forkO->tubo[i][2]);
	    forkO->tubo[i][2] = 0;
	}
    }
    return NULL;
}

static 
fork_struct *
TuboFree(fork_struct * forkO)
{
    H(printf("TuboFree()\n");)
    if(!forkO)
    {
	return NULL;
    }
    valid_fork_objects=g_list_remove(valid_fork_objects,(gpointer)forkO);
    free(forkO);
    return NULL;
}


static 
void * 
block_get_line(void *in)
{
    int i;
    int exit_code=0;
    arena_t *arena_p = (arena_t *)in;
    memset(line, 0, TUBO_BLK_SIZE);

    for(i = 0; i < TUBO_BLK_SIZE - 1; i++)
    {
	if(!read(arena_p->src, line + i, 1))
	    break;
	XH(printf("%c",line[i]);fflush(NULL);)
	if(*(line + i) == '\n')
	{
	    *(line + i + 1) = (char)0;
	    goto done;
	}
    }
    if(i) exit_code = i;		/* something has been read */
    else  exit_code = -1;
done:
    return (GINT_TO_POINTER(exit_code));
}
 

static 
int 
TuboInput(fork_struct *newfork, int src, int (*user_parse_function) (int, void *, void *), void *user_data)
{
    int i;
    arena_t *arena_p = (arena_t *)(newfork->m);
    arena_p->src=src;

    i = GPOINTER_TO_INT(block_get_line(arena_p));
    XH(if (i >=0) printf("TUBO i=%d get_line(%d)=%s\n", i,src,line);)
    if (i < 0) return 0;
    if (i==0 && *((char *)line)==0) return 0;
    if (user_parse_function) (*user_parse_function) (i, (void *)line, user_data);
  return 1;

}



static 
gint 
check_read(gpointer data){
    fork_struct *newfork = (fork_struct *) data;
    int retval1=FALSE,retval2=FALSE;
    fd_set rfds;
    struct timeval tv;
    int nfds = ((newfork->tubo[1][0]>newfork->tubo[2][0]) ? newfork->tubo[1][0] : newfork->tubo[2][0]);
    FD_ZERO(&rfds);
	
    if (newfork->tubo[1][0] >= 0) FD_SET(newfork->tubo[1][0], &rfds); /* stdout */
    if (newfork->tubo[2][0] >= 0) FD_SET(newfork->tubo[2][0], &rfds); /* stderr */

    /* Wait up to five microseconds. */
    tv.tv_sec = 0;
    tv.tv_usec = 5;
    
    
    
    XH(printf("TUBO watch input...\n"); )
    if (select(nfds+1, &rfds, NULL, NULL, &tv) <= 0) {
	XH(printf("TUBO  select(nfds+1, &rfds, NULL, NULL, &tv) <= 0 : return TRUE\n");)
	return 2;
    }
    if (!newfork) {
	XH(printf("TUBO  !newfork: return FALSE\n");)
	return 0;
    }
    
    retval1=0;
    if (FD_ISSET(newfork->tubo[1][0], &rfds)){
	XH(printf("TUBO  stdout ready for reading...(%d)\n",newfork->tubo[1][0]);)
	retval1=TuboInput(newfork, newfork->tubo[1][0], newfork->operate_stdout, newfork->user_data);
    }
    if (FD_ISSET(newfork->tubo[2][0], &rfds)){
	XH(printf("TUBO  stderr ready for reading...(%d)\n",newfork->tubo[2][0]);)
	retval2=TuboInput(newfork, newfork->tubo[2][0], newfork->operate_stderr, newfork->user_data);
    }
    return (retval1 | retval2);
	
}

static 
void
Tubo_endit(gpointer fork_object){
    pid_t PID;
    fork_struct *forkO = (fork_struct *) ((long)fork_object);
    void (*user_end_function) (pid_t, void *);
    void *user_data;

    user_data = forkO->user_data;
    user_end_function = forkO->fork_finished_function;
    PID = forkO->PID;
    forkO->PID = 0;
    H(printf("Parent is terminating tubo by Tubo_endit\n"); )
    TuboClosePipes(forkO);

    if(user_end_function) {
        H(printf("Looking for user_end_function...\n"); )
	(*user_end_function) (PID, user_data);
    }
    {
	H(arena_t *arena_p=(arena_t *)forkO->m; )
        H(printf("terminated sibling is %d\n",arena_p->pid); )
	
    }
    munmap(forkO->m,PAGESIZE);
    shm_unlink(forkO->shm_name);
    TuboFree(forkO);
    forkO=NULL;
}


static 
gint 
watch_input(gpointer data)
{
    fork_struct *newfork = (fork_struct *) data;
    int retval;
    arena_t *arena_p = (arena_t *)newfork->m;
 	
    /* pick up any zombies */
    waitpid(-1,&retval,WNOHANG);
    
    if (!data) return FALSE;
    if (arena_p->thread_running){
	return TRUE;
    }
    if (newfork->tubo[1][0] < 0 && newfork->tubo[2][0] < 0) {
	XH(printf("TUBO newfork->tubo[1][0] < 0 && newfork->tubo[2][0] < 0\n");)
	return FALSE;
    }

    if (newfork->input_redlight) return TRUE;
    
    
    newfork->input_redlight=TRUE;
    XH(printf("parent watching for data...  \n");)
 
    /* check read will return TRUE if (pipe not ready) or (pipe ready and data read)
     * and FALSE when pipe ready and error reading pipe (child done) */
  do {  
    retval = check_read(data);
   
    if (!retval && *((char *)(newfork->m))==1) {
	H(fprintf(stderr,"shared memory block unlocked\n");)
	if (newfork->tubo[0][2]) g_source_remove(newfork->tubo[0][2]);
	newfork->tubo[0][2] = 0;
	Tubo_endit((void *)newfork);
    }
  } while (retval==1 && newfork->blockread);

    newfork->input_redlight=FALSE;
    return TRUE;
}


static 
tubo_t *
private_tubo_full (void (*fork_function) (void *), 
	   void *fork_function_data, 
	   void (*fork_finished_function) (pid_t, void *), 
	   int *stdinFD, 
	   int (*operate_stdout) (int, void *, void *), 
	   int (*operate_stderr) (int, void *, void *),
	   void *user_data, 
	   int read_delay,
	   gboolean blockread)
{
    int i;
    fork_struct tmpfork, *newfork = NULL;
    arena_t *arena_p;
    H(fprintf(stderr,"+++++++++++++++++++++++++++++++++++++++++++++++\n");)
     
    for(i = 0; i < 3; i++)
    {
	tmpfork.tubo[i][0] = tmpfork.tubo[i][1] = -1;
	tmpfork.tubo[i][2] = 0;
    
    
	if(pipe(tmpfork.tubo[i]) == -1)
	{
	    TuboClosePipes(&tmpfork);
	    return NULL;
	}
    }

    tmpfork.blockread = blockread;
     
    tmpfork.user_data = user_data;
    tmpfork.operate_stdout = operate_stdout;
    tmpfork.operate_stderr = operate_stderr;
    tmpfork.fork_function = fork_function;
    tmpfork.fork_finished_function = fork_finished_function;


    /* open shared memory block... */
    sprintf(tmpfork.shm_name,"/xffm.%ld",(long)getpid());
    tmpfork.shm=shm_open(tmpfork.shm_name, O_CREAT | O_RDWR, 0700);
    if (tmpfork.shm < 0) {
	H(g_message("share memory item failed: %s",tmpfork.shm_name);)
	sprintf(tmpfork.shm_name,"%s/xffm.%ld",g_get_tmp_dir(),(long)getpid());
	H(g_message("trying shared memory item: %s",tmpfork.shm_name);)
	tmpfork.shm=shm_open(tmpfork.shm_name, O_CREAT | O_RDWR, 0700);
        if (tmpfork.shm < 0) {
	    g_warning("shm_open(%s): %s",tmpfork.shm_name,strerror(errno));
	    return NULL;
	}
    }
    ftruncate(tmpfork.shm,PAGESIZE);
    tmpfork.m=mmap(0, PAGESIZE, PROT_READ | PROT_WRITE , MAP_SHARED, tmpfork.shm, 0);
    if (tmpfork.m == MAP_FAILED) {
	    g_warning("(parent) mmap: %s",strerror(errno));
	    return NULL;
    }
    arena_p=(arena_t *)tmpfork.m;
    memset(tmpfork.m,0,PAGESIZE);
    msync(tmpfork.m, PAGESIZE, MS_SYNC);
    /* end shared memory block */
    
    
    tmpfork.PID = fork();
#ifdef FORK_OFF
    if(tmpfork.PID==0)
#else
    if(tmpfork.PID!=0)
#endif
    {				/* the child */
	/* INPUT PIPES *************** */
	/*sprintf(tmpfork.end_string,"Tubo-END:%d",tmpfork.PID);*/
	setpgid(0,0); /* or setpgrp();*/

	tmpfork.m=mmap(0, PAGESIZE, PROT_READ | PROT_WRITE , MAP_SHARED, tmpfork.shm, 0);
	
	if (tmpfork.m == MAP_FAILED) {
	    g_warning("(child) mmap: %s",strerror(errno));
	    perror(strerror(errno));
	    return NULL;
	}
	arena_p=(arena_t *)tmpfork.m;
    
	usleep(50);		/* first give the child a time slot */
	newfork = (fork_struct *) malloc(sizeof(fork_struct));
	if (!newfork){
	    perror("malloc(sizeof(fork_struct)) == NULL\n");
	    exit(123);
	}
	valid_fork_objects=g_list_append(valid_fork_objects,(gpointer)newfork);
	MEMCPY((void *)newfork, (void *)(&tmpfork), sizeof(fork_struct));

	close(newfork->tubo[0][0]);	/* not used */
	newfork->tubo[0][0] = -1;
	

	/* OUTPUT PIPES ************** */
	if(stdinFD) {
		*stdinFD = newfork->tubo[0][1];	
	}
	else {
		close(newfork->tubo[0][1]);
		newfork->tubo[0][1] = -1;
	}
	close(newfork->tubo[1][1]);	/* not used */
	close(newfork->tubo[2][1]);	/* not used */
	newfork->tubo[1][1] = -1;
	newfork->tubo[2][1] = -1;
	
	newfork->read_delay=read_delay;
    	
	if (newfork->read_delay == 0) newfork->read_delay = 275;
	if (operate_stdout || operate_stderr){
		newfork->tubo[2][2] = 
			g_timeout_add(newfork->read_delay,(GSourceFunc) watch_input, (gpointer) newfork);
	}
	arena_p->greenlight=1;
	msync(tmpfork.m, PAGESIZE, MS_SYNC);
	
	newfork->input_redlight=FALSE;
	H(fprintf(stderr,"child placed greenlight=%d\n",arena_p->greenlight);)

	usleep(50);		/* give the child a time slot again*/
	H(printf("The parent process is returning\n");)
 
	XH(printf ("g_message: parent %d<->%d   %d<->%d   %d<->%d\n",
		newfork->tubo[0][0],
		newfork->tubo[0][1],
		newfork->tubo[1][0],
		newfork->tubo[1][1],
		newfork->tubo[2][0],
		newfork->tubo[2][1]);)
	return newfork;		/* back to user's program flow */
    }
    else
    {				/* the parent */
	
	newfork = (fork_struct *) malloc(sizeof(fork_struct));
	if(!newfork)
	{

	    _exit(1);
	}
	MEMCPY((void *)newfork, (void *)(&tmpfork), sizeof(fork_struct));

	/* INPUT PIPES */
	if(stdinFD) dup2(newfork->tubo[0][0], 0);	/* stdin */	    
	else {
	    close(newfork->tubo[0][0]);		/* stdin */
	    newfork->tubo[0][0] = -1;
	}
	
	/*close(newfork->tubo[1][0]);*/	/* keep open for when child is gone */
	/*close(newfork->tubo[2][0]);*/	/* keep open for when child is gone */

	/* OUTPUT PIPES */
	close(newfork->tubo[0][1]);	/* not used */
	newfork->tubo[1][0] = newfork->tubo[2][0] = newfork->tubo[0][1] = -1;
	
	if(operate_stdout)
	    dup2(newfork->tubo[1][1], 1);	/* stdout */
	else {
	    close(newfork->tubo[1][1]);	/* not used */
	    newfork->tubo[1][1] = -1;
	}
	if(operate_stdout)
	    dup2(newfork->tubo[2][1], 2);	/* stderr */
	else {
	    close(newfork->tubo[2][1]);	/* not used */
	    newfork->tubo[2][1] = -1;
	}
	/*printf ("g_message: child %d<->%d   %d<->%d   %d<->%d\n",
		newfork->tubo[0][0],
		newfork->tubo[0][1],
		newfork->tubo[1][0],
		newfork->tubo[1][1],
		newfork->tubo[2][0],
		newfork->tubo[2][1]);*/
	
     {
	pid_t grandchildPID;
	int status, status2;

	
	do {
	    usleep(500);
	} while (arena_p->greenlight == 0);

	/* reuse greenlight for grandchild */
	arena_p->greenlight = 0;
	msync(tmpfork.m, PAGESIZE, MS_SYNC);
	grandchildPID =  fork();
	
	if (grandchildPID<0) {
	    printf("g_message: could not fork!\n");
	    _exit(1);
	} 
	if (grandchildPID==0) {
	    setpgid(0,0); /* or setpgrp();*/
	    tmpfork.m=mmap(0, PAGESIZE, PROT_READ | PROT_WRITE , MAP_SHARED, tmpfork.shm, 0);	
	    if (tmpfork.m == MAP_FAILED) {
		return NULL;
	    }
	    arena_p=(arena_t *)tmpfork.m;
	    
	    do {
		usleep(500);
	    } while (arena_p->greenlight == 0);
	    H(fprintf(stderr,"The grandchild has green light pid=%d\n",arena_p->pid);)
	    if(newfork->fork_function) (*(newfork->fork_function)) (fork_function_data);
	    fprintf(stderr,"Incorrect usage: fork_function should _exit()\n");
	    _exit(1);
	} 
	
	arena_p->pid = grandchildPID;
	msync(tmpfork.m, PAGESIZE, MS_SYNC);

	XH(fprintf(stderr,"The process has forked: parent=%d child=%d, sibling=%d\n",getpid(),tmpfork.PID, arena_p->pid); )

	if (grandchildPID<0) {
	    printf("g_message: could not fork!\n");
	    _exit(1);
	} 
	
      
	XH(fprintf(stderr,"waiting for grandchild\n");fflush(NULL);)	
	arena_p->greenlight = 1;
	msync(tmpfork.m, PAGESIZE, MS_SYNC);
	usleep(500);
	setpgid(0,grandchildPID);
	
	waitpid(grandchildPID,&status,0);
	waitpid(tmpfork.PID,&status2,WNOHANG);
	if (!WIFEXITED(status2)) 
	{
	    XH(fprintf(stderr,"*wait for grandchild COMPLETE: WIFEXITED(status2)=%d WIFSIGNALED(status2)=%d (%d,%d)\n", WIFEXITED(status2),WIFSIGNALED(status2),
		    grandchildPID,tmpfork.PID);)
	}
	fflush(NULL);
	sleep(1);
	memset(tmpfork.m,1,1);
	msync(tmpfork.m, PAGESIZE, MS_SYNC);
	munmap(tmpfork.m,PAGESIZE);
	shm_unlink(tmpfork.shm_name);	
	_exit(1);
	
     }
	 
    } /* end parent */

}

G_MODULE_EXPORT
tubo_t *
Tubo_full (void (*fork_function) (void *), 
	   void *fork_function_data, 
	   void (*fork_finished_function) (pid_t, void *), 
	   int *stdinFD, 
	   int (*operate_stdout) (int, void *, void *), 
	   int (*operate_stderr) (int, void *, void *),
	   void *user_data, 
	   int read_delay)
{
    return private_tubo_full (fork_function, fork_function_data, 
	   fork_finished_function, 
	   stdinFD, 
	   operate_stdout, 
	   operate_stderr,
	   user_data, 
	   read_delay,
	   FALSE);
}

/* XXX this has problems... output is lost from stdout in xfsamba smb_ws.i */
G_MODULE_EXPORT
tubo_t *
Tubo_full_blockread (void (*fork_function) (void *), 
	   void *fork_function_data, 
	   void (*fork_finished_function) (pid_t, void *), 
	   int *stdinFD, 
	   int (*operate_stdout) (int, void *, void *), 
	   int (*operate_stderr) (int, void *, void *),
	   void *user_data, 
	   int read_delay)
{
    return private_tubo_full (fork_function, fork_function_data, 
	   fork_finished_function, 
	   stdinFD, 
	   operate_stdout, 
	   operate_stderr,
	   user_data, 
	   read_delay,
	   TRUE);
}


G_MODULE_EXPORT
tubo_t *
TuboCancel (void *forkObject, void (*cleanup) (void *), void *user_data)
{
    fork_struct *forkO;
    forkO = (fork_struct *) forkObject;
    if(!forkO || g_list_find(valid_fork_objects,(gconstpointer)forkO)==NULL)
    {
        fprintf(stderr,"ignoring TuboCancel() call on invalid fork object.\n");
	return NULL;
    }
    /* speed up exit: */
    forkO->blockread=TRUE;

#if 10
    {
      arena_t *arena_p;
      arena_p=(arena_t *)forkO->m;
      H(fprintf(stderr,"%s:cancelling fork object, pid=%d\n",forkO->shm_name,arena_p->pid);)
      if(arena_p->pid) {
	  kill(arena_p->pid, SIGKILL);	
	  usleep(250);
      }
    }
#endif
#if 0
    {
      pid_t pid;
      pid = atoi(strchr(forkO->shm_name,'.')+1);
      fprintf(stderr,"%s:cancelling fork object, pid=%d\n",forkO->shm_name,pid);
      if(pid) {
	  kill(pid, SIGHUP);	
	  usleep(250);
      }
    }
#endif

    
    if(cleanup)	(*cleanup) (user_data);

    /*note: fork object freed by TuboWaitDone() function */
    return NULL;
}

G_MODULE_EXPORT
pid_t 
TuboPID (gpointer forkObject)
{
    fork_struct *forkO = (fork_struct *) forkObject;
    arena_t *arena_p;
    XH(g_message("TuboPID");)
    if(!forkO || g_list_find(valid_fork_objects,(gconstpointer)forkO)==NULL)
    {
	XH(g_message("0x%x is crooked",(unsigned)forkO);)
	return 0;
    }
    arena_p=(arena_t *)forkO->m;
    XH(g_message("arena_p= 0x%x looks ok",(unsigned)arena_p);)
    return (arena_p->pid);
}

G_MODULE_EXPORT
tubo_t *
TuboO(	   void (*fork_function) (void *), 
	   void *fork_function_data, 
	   void (*fork_finished_function) (pid_t, void *), 
	   int (*operate_stdout) (int, void *, void *), 
	   int (*operate_stderr) (int, void *, void *),
	   void *user_data){
    return Tubo_full(fork_function, fork_function_data, fork_finished_function, 
	0, operate_stdout,operate_stderr,
	user_data,0);
}


G_MODULE_EXPORT
tubo_t *
TuboIO(	   void (*fork_function) (void *), 
	   void *fork_function_data, 
	   void (*fork_finished_function) (pid_t, void *), 
	   int *operate_stdin, 
	   int (*operate_stdout) (int, void *, void *), 
	   int (*operate_stderr) (int, void *, void *),
	   void *user_data){
    return Tubo_full(fork_function, fork_function_data, fork_finished_function, 
	operate_stdin, operate_stdout,operate_stderr,
	user_data,0);
} 
    
