#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <assert.h>
#include <qapp.h>
#include <qstack.h>
#include <qmsgbox.h>

#include "turtle.hpp"

#define SMALLRAND(a)	(int)((float)(rand()/(float)RAND_MAX)*(float)(a)+1)

/*
 * Turtle command set :
 *		F : forwards
 *		B : backwards
 *		+ : turn 1 unit clockwise
 *		- : turn 1 unit anti-clockwise
 *		[ : save state
 *		] : restore state
 *		< : decrease unit length
 *		> : increase unit length
 *		I : random velocity increase (disabled)
 *		D : random velocity decrease (disabled)
 */

KTurtle::KTurtle
(
	QWidget *p
)
: QWidget(p)
{
	pixmap = 0;
	dirs = 0;
	level = 0;
	axiom = 0;
	rules = 0;
	buf1 = new char[MAXMEM*1024];
	buf2 = new char[MAXMEM*1024];
	device = 0;
	devw = devh = 0;
}

KTurtle::~KTurtle()
{
	delete pixmap;
	delete buf1;
	delete buf2;
}

int KTurtle::strRewrite
(
	char *ax,
	char **ru,
	char *res
)
{
	int len = 0;

	for (int max = MAXMEM*1024, found = 0; *ax; found = 0, ax++)
	{
		for (char **ptr = ru; *ptr; ptr++)
		{
			if (**ptr == *ax)
			{
				for (char *pidx = *ptr+1; *pidx; res++, pidx++, len++)
				{
					if (++len >= max)
						return -1;
					*res = *pidx;
				}
				found = 1;
				break;
			}
		}
		if (!found)
		{
			if (++len >= max)
				return -1;
			*res++ = *ax;
		}
	}
	*res = 0;

	return len;
}

void KTurtle::render
(
	QPaintDevice *pd,	// device to render onto ( NULL means render onto pixm )
	int w,				// width of render canvas
	int h,				// height of render canvas
	char *c,			// command-set
	int d				// number of directions
)
{
	struct sTurtleState
	{
		float x, y, dir, len;

		sTurtleState(float cx, float cy, float cd, float cl)
			{ x = cx; y = cy; dir = cd; len = cl; }
	};

	QPainter p;
	QStack<sTurtleState> states;
	sTurtleState *state;
	time_t s = time(0);
	float dirinc = (2.0*M_PI)/(dirs = d);
	float dir;
	float len = 10.0, leninc = 1.0, lenmax = 20.0;
	float maxy = 0.0, miny = 0.0, maxx = 0.0, minx = 0.0;
	float x = 0.0, y = 0.0, rx, ry;
	float xdim, ydim;
	float scalar, randceil = 3.0;
	char *ptr;

	// NULL means use background pixmap
	if (!pd)
	{
		w = width();
		h = height();
		if (!pixmap)
			pixmap = new QPixmap(w, h);
		pixmap->fill();
		pd = pixmap;
	}

	p.begin(pd);

	p.setPen(black);

	// do the process twice
	// 1 : determine dimensions of rendered command-set using well-known
	//     movement length
	// 2 : use knowledge gained in iteration 1 to scale image to fit dimensions
	//     passed as arguments and draw using qt paint interface
	// would most probably be a bit faster to seperate two phases to avoid if()
	// to discern when to draw but i don't think its worth the added
	// maintenance on 2 seperate pieces of code with virtually the same function
	for (int i = 0; i < 2; i++)
	{
		// start facing east
		dir = 0.0;

		// set seed the same for both iteration otherwise scalar might be void
		srand(s);

		// 2nd iteration, use 1st iteration's results
		if (i == 1)
		{
			// scale entire setting down to fit on canvas (leave 10 unit margin)

			// calc dimensions of imaginary viewport result
			xdim = fabs(maxx-minx);
			ydim = fabs(maxy-miny);

			// calc scalar to go to widget viewport
			scalar = ((float)w-10.0) / xdim;
			if (scalar*ydim > ((float)h-10.0))
				scalar = ((float)h-10.0) / ydim;

			randceil = scalar * 3.0;
			len = scalar * 10.0;
			leninc = scalar * 1.0;
			lenmax = scalar * 20.0;
			x = fabs(minx)*scalar+5+(w-10-scalar*xdim)/2;
			y = fabs(miny)*scalar+5;
			p.moveTo((int)x, h-(int)y);
		}

		// iterate through command-set and execute those familiar to current
		// version of turtle
		for (ptr = c; *ptr; ptr++)
		{
			switch(*ptr)
			{
				// forward len units
				case 'F' :
					rx = len*cos(dir);
					ry = len*sin(dir);
					x += rx;
					y += ry;
					if (i == 0)
					{
						maxx = MAX(maxx, x);
						maxy = MAX(maxy, y);
						minx = MIN(minx, x);
						miny = MIN(miny, y);
					}
					else
						p.lineTo((int)x, h-(int)y);
					break;

				// backwards len units
				case 'B' :
					rx = len*cos(dir+M_PI);
					ry = len*sin(dir+M_PI);
					x += rx;
					y += ry;
					if (i == 0)
					{
						maxx = MAX(maxx, x);
						maxy = MAX(maxy, y);
						minx = MIN(minx, x);
						miny = MIN(miny, y);
					}
					else
						p.lineTo((int)x, h-(int)y);
					break;

				// clockwise 360/#dirs
				case '+' :
					dir -= dirinc;
					break;

				// anti-clockwise 360/#dirs
				case '-' :
					dir += dirinc;
					break;

				// decrease movement length
				case '<' :
					len -= leninc;
					if (len < 1.0)
						len = 1.0;
					break;

				// increase movement length
				case '>' :
					len += leninc;
					if (len > lenmax)
						len = lenmax;
					break;

				// save state on stack
				case '[' :
					states.push(new sTurtleState(x, y, dir, len));
					break;

				// pop state from stack
				case ']' :
					if (!states.isEmpty())
					{
						state = states.pop();
						x = state->x;
						y = state->y;
						dir = state->dir;
						len = state->len;
						delete state;
						p.moveTo((int)x, h-(int)y);
					}
					break;

				// random velocity increase
				case 'I' :
					/*
					len += SMALLRAND((int)randceil);
					if (len > lenmax)
						len = lenmax;
					*/
					break;

				// random velocity decrease
				case 'D' :
					/*
					len -= SMALLRAND((int)randceil);
					if (len < 1.0)
						len = 1.0;
					*/
					break;
			}
		}
	}

	p.end();

	// clean up after misbehaving algorithms
	while (!states.isEmpty())
		delete states.pop();
}

int KTurtle::go
(
	char *ax,			// axiom ( 0 terminated string )
	char **ru,			// rules ( NULL terminated string array )
	int di,				// directions ( 360/di incrementals during render )
	int le				// level ( -1 means as far as possible )
)
{
	assert(ax);
	assert(ru);
	assert(di);

	// init class members with parameters
	axiom = ax;
	rules = ru;
	dirs = di;
	level = le;

	return go();
}

int KTurtle::go()
{
	char *tmp, *result, *axm;
	int prev, now = -1;

	// start wrong site front due to the way the loops work
	axm = buf1;
	result = buf2;
	strcpy(result, axiom);

	// generate maximum possible level. be on lookout for linear algs
	// which possibly doesn't have the ability to reach max. they will
	// cause infinite loop. that's the reason for the prev/now vars
	if (level == -1)
	{
		do
		{
			prev = now;
			tmp = axm;
			axm = result;
			result = tmp;
			qApp->processEvents();
		}
		while ((now = strRewrite(axm, rules, result)) != -1 && now != prev);

		// remember, the the result couldn't be generated so use previous
		// result ( which is this level's axiom )
		result = axm;
	}
	else
	{
		for (int i = 0; i <= level; i++)
		{
			tmp = axm;
			axm = result;
			result = tmp;
			if (strRewrite(axm, rules, result) == -1)
			{
				QMessageBox m(this);
				m.setText("Expansion comsumes more than maximum allowed memory");
				m.show();
				level--;
				return -1;
			}
		}
	}

	// render turtle with level expanded commandset
	render(device, devw, devh, result, dirs);

	// if device is 0 it was on canvas so update display
	if (!device)
		repaint();

	return 0;
}

void KTurtle::reset()
{
	dirs = 0;
	level = 0;
	axiom = 0;
	rules = 0;
}

void KTurtle::save
(
	const char *file
)
{
	if (!pixmap)
	{
		QMessageBox m(this);
		m.setText("Nothing to save");
		m.show();
	}
	else if (pixmap->save(file, "BMP") == FALSE)
	{
		QMessageBox m(this);
		m.setText("Failed to save the image");
		m.show();
	}
}

void KTurtle::resizeEvent
(
	QResizeEvent *
)
{
	delete pixmap;
	pixmap = 0;
	if (axiom && rules)
	{
		go();
		paintEvent(0);
	}
}

void KTurtle::paintEvent
(
	QPaintEvent *
)
{
	QPainter p;

	if (!pixmap)
		return;

	p.begin(this);
	p.drawPixmap(0, 0, *pixmap);
	p.end();
}
