#include "xyplot.h"
#include "xyplot.moc"

#include <iostream.h>
#include <math.h>
#include <stdlib.h>

#include "debug.h"
#include "centre.h"

#include <kiconloader.h>
#include <kapp.h>



static const char* qcolor (const QColor& color)
{
  static char buffer [64];
  sprintf(buffer,"%i %i %i",color.red(),color.green(),color.blue());
  return buffer;
}



XYPlotLabelRangeState::XYPlotLabelRangeState ()
{
  font = QFont("helvetica",12);
  color = black;
  pen = QPen(black);
  mode = 0;
  from = 0.0;
  to = 0.0;
  show_zero = true;
  label_at_zero = false;
  fmt = 0;
}

XYPlotLabelRangeState::XYPlotLabelRangeState (const XYPlotLabelRangeState& rhs)
{
  font = rhs.font;
  color = rhs.color;
  pen = QPen(color); // TODO: eigener PenButton
  mode = rhs.mode;
  from = rhs.from;
  to = rhs.to;
  show_zero = rhs.show_zero;
  label_at_zero = rhs.label_at_zero;
  fmt = rhs.fmt;
}


XYPlotTickState::XYPlotTickState ()
{
  mode = 2;
  size = 0.0;
  steps = 10;
  tick_len = 3;
  grid = false;
  tick_pen = QPen(black);
  grid_pen = QPen(black,0,NoPen);
}

XYPlotTickState::XYPlotTickState (const XYPlotTickState& rhs)
{
  mode = rhs.mode;
  size = rhs.size;
  steps = rhs.steps;
  tick_len = rhs.tick_len;
  grid = rhs.grid;
  tick_pen = rhs.tick_pen;
  grid_pen = rhs.grid_pen;
}


XYAxisRecord::XYAxisRecord ()
{
  log_x = log_y = 0;
}

XYAxisRecord::XYAxisRecord (const XYAxisRecord& rhs)
{
  log_x = rhs.log_x;
  log_y = rhs.log_y;
  x_ticks = rhs.x_ticks;
  x_subticks = rhs.x_subticks;
  x_range = rhs.x_range;
  y_ticks = rhs.y_ticks;
  y_subticks = rhs.y_subticks;
  y_range = rhs.y_range;
}


XYPlotRecord::XYPlotRecord ()
{
  re_calc = true;
  color = blue;
  pen = color;
  marker = NoMarker;
  marker_size = 3;
  worksheet = 0;
}

XYPlotRecord::XYPlotRecord (const XYPlotRecord& rhs)
{
  int i;
  re_calc = rhs.re_calc;
  color = rhs.color;
  pen = rhs.pen;
  marker = rhs.marker;
  marker_size = rhs.marker_size;
  worksheet = rhs.worksheet;
  for (i=0; i<5; i++)
	input[i] = rhs.input[i];
}


REGISTER_PLOT(XYPlot);


XYAxisRecord* XYPlot::defaultAxis = 0;
XYPlotRecord* XYPlot::defaultSet  = 0;


XYPlot::XYPlot () : Plot()
{
  the_menu = 0;
  records.setAutoDelete(true);
  axis = 0;
}


void XYPlot::instantiate (Page* page, QWidget* parent, const char* name,
						  WFlags f)
{
  active_icon = new QPixmap(
	kapp->getIconLoader()->loadIcon("xyplot-a.xpm"));
  inactive_icon = new QPixmap(
	kapp->getIconLoader()->loadIcon("xyplot-i.xpm"));

  Plot::instantiate(page,parent,name,f);

  if (defaultAxis == 0)
	restoreOptions();
  if (defaultAxis == 0)
	axis = new XYAxisRecord();
  else
	axis = new XYAxisRecord(*defaultAxis);

  the_treeitem = new TreeItem();
  the_treeitem->setPixmap(active_icon);
  the_treeitem->setText(description());
  the_treeitem->setObject(this);
}


Plot* XYPlot::clone ()
{
  return new XYPlot();
}


XYPlot::~XYPlot ()
{
  IFDEBUG("korigin.xyplot",4)
	cout << "XYPlot::~XYPlot() starts" << endl;
  ENDDEBUG;

  delete axis;
  axis = 0;
  delete the_menu;
  the_menu = 0;

  IFDEBUG("korigin.xyplot",4)
	cout << "XYPlot::~XYPlot() ends" << endl;
  ENDDEBUG;
}
  

QPopupMenu* XYPlot::menu (QPopupMenu* predefined)
{
  QPopupMenu* menue;
  if (predefined) {
	menue = predefined;
	menue->clear();
  } else {
	menue = new QPopupMenu();
  }
  menue->insertItem("Plot selected",this,SLOT(plotSelectedColumns()));
  menue->insertSeparator();
  menue->insertItem("Pref X Axis...",this,SLOT(optionsXAxis()));
  menue->insertItem("Pref Y Axis...",this,SLOT(optionsYAxis()));
  menue->insertSeparator();
  menue->insertItem("X Axis...",this,SLOT(slotXAxis()));
  menue->insertItem("Y Axis...",this,SLOT(slotYAxis()));
  menue->insertItem("Datasets...",this,SLOT(slotDataset()));
  menue->insertItem("Delete Set...",this,SLOT(slotDelete()));
  return menue;
}


const char* XYPlot::name ()
{
  return "X/Y Plot";
}


const char* XYPlot::description ()
{
  static char buffer [128];
  sprintf(buffer,"X/Y Plot, %i data sets",records.count());
  return buffer;
}


QPixmap* XYPlot::icon ()
{
  if (activePlot == this) {
	return active_icon;
  } else {
	return inactive_icon;
  }
}

  
int XYPlot::paramCount ()
{
  return 5;
}


const char* XYPlot::paramName (int i)
{
  switch(i) {
  case 0: return "X";
  case 1: return "Y";
  case 2: return "X error";
  case 3: return "Y error";
  case 4: return "Label";
  default: return "Martin hates index checks";
  }
}


void XYPlot::optionsDialog ()
{
  QMessageBox::information(0,"Deviation",
						   "This entry does not work yet.\n"\
						   "Please select \"X Axis Options\"\n"\
						   "or \"X Axis Options\" from the \n"\
						   "menu instead.");
}


void XYPlot::optionsXAxis ()
{
  XYPlotAxisDialog* dlg;
  dlg = new XYPlotAxisDialog("X/Y Plot X Axis Defaults",defaultAxis,0);
  dlg->exec();
  storeOptions();
  delete dlg;
}


void XYPlot::optionsYAxis ()
{
  XYPlotAxisDialog* dlg;
  dlg = new XYPlotAxisDialog("X/Y Plot Y Axis Defaults",defaultAxis,1);
  dlg->exec();
  storeOptions();
  delete dlg;
}


void XYPlot::slotXAxis ()
{
  XYPlotAxisDialog* dlg;
  dlg = new XYPlotAxisDialog("X Axis Settings",axis,0);
  connect(dlg,SIGNAL(settingsChanged()),page(),SLOT(update()));
  dlg->exec();
  delete dlg;
}


void XYPlot::slotYAxis ()
{
  XYPlotAxisDialog* dlg;
  dlg = new XYPlotAxisDialog("Y Axis Settings",axis,1);
  connect(dlg,SIGNAL(settingsChanged()),page(),SLOT(update()));
  dlg->exec();
  delete dlg;
}


void XYPlot::slotDataset ()
{
  XYPlotDatasetDialog* dlg;
  dlg = new XYPlotDatasetDialog(records);
  dlg->exec();
  delete dlg;
}
  

void XYPlot::slotDelete ()
{
  if (records.count() == 0) return;

  XYPlotDeleteDialog* dlg;
  dlg = new XYPlotDeleteDialog(records);
  if (dlg->exec() == QDialog::Accepted)
	{
	  IFDEBUG("korigin.xyplot",4)
		cout << "DELETE # " << dlg->current() << endl
			 << "DELETE   " << records.at(dlg->current())->input[0].title() 
			 << " vs " << records.at(dlg->current())->input[1].title()
			 << endl;
	  ENDDEBUG;

	  records.remove(dlg->current());
	}
  delete dlg;
}


void XYPlot::storeOptions ()
{
  KConfig* conf = kapp->getConfig();
  conf->setGroup("XYPlot");
  /*
  conf->writeEntry("x_color",defaultAxis->x_color);
  conf->writeEntry("y_color",defaultAxis->y_color);
  conf->writeEntry("x_font",defaultAxis->x_font);
  conf->writeEntry("y_font",defaultAxis->y_font);
  conf->writeEntry("auto_x",defaultAxis->auto_x);
  conf->writeEntry("auto_y",defaultAxis->auto_y);
  conf->writeEntry("log_x",defaultAxis->log_x);
  conf->writeEntry("log_y",defaultAxis->log_y);
  conf->writeEntry("x_steps",defaultAxis->x_steps);
  conf->writeEntry("y_steps",defaultAxis->y_steps);
  conf->writeEntry("x_zero",(defaultAxis->x_zero?1:0));
  conf->writeEntry("y_zero",(defaultAxis->y_zero?1:0));
  conf->writeEntry("x_format",defaultAxis->x_format);
  conf->writeEntry("y_format",defaultAxis->y_format);
  conf->writeEntry("plot_pen",0);
  conf->writeEntry("plot_color",defaultSet->color);
  conf->writeEntry("plot_marker",defaultSet->marker);
  */
  conf->sync();
}


void XYPlot::restoreOptions ()
{
  if (defaultAxis == 0) defaultAxis = new XYAxisRecord ();
  if (defaultSet == 0) defaultSet = new XYPlotRecord ();
  KConfig* conf = kapp->getConfig();
  conf->setGroup("XYPlot");
  /*
  defaultAxis->x_color = conf->readColorEntry("x_color",&black);
  defaultAxis->y_color = conf->readColorEntry("y_color",&black);
  defaultAxis->x_font = conf->readFontEntry("x_font",&(the_widget->font()));
  defaultAxis->y_font = conf->readFontEntry("y_font",&(the_widget->font()));
  defaultAxis->auto_x = conf->readNumEntry("auto_x",1);
  defaultAxis->auto_y = conf->readNumEntry("auto_y",1);
  defaultAxis->log_x = conf->readNumEntry("log_x",0);
  defaultAxis->log_y = conf->readNumEntry("log_y",0);
  defaultAxis->x_steps = conf->readNumEntry("x_steps",10);
  defaultAxis->y_steps = conf->readNumEntry("y_steps",10);
  defaultAxis->x_zero = (conf->readNumEntry("x_zero",1) == 1);
  defaultAxis->y_zero = (conf->readNumEntry("y_zero",1) == 1);
  strcpy(defaultAxis->x_format,conf->readEntry("x_format","%.2lf"));
  strcpy(defaultAxis->y_format,conf->readEntry("y_format","%.2lf"));
  conf->readNumEntry("plot_pen",0);
  defaultSet->color = conf->readColorEntry("plot_color",&black);
  defaultSet->marker = (XYPlotRecord::XYPlotMarker)conf->readNumEntry(
                       "plot_marker",
					   (int)XYPlotRecord::NoMarker);
  */
}


#define WRITE(string) writeBlock((string),strlen(string))


void XYPlot::save_dataset (QFile& file, XYPlotRecord* set)
{
  int i;
  char buffer [128];

  file.WRITE("<dataset>\n");
  sprintf(buffer,"worksheet = %s\n",set->worksheet->caption());
  file.WRITE(buffer);
  for (i=0; i<paramCount(); i++) {
	if (set->input[i].title()) {
	  sprintf(buffer,"%s = %s\n",paramName(i),set->input[i].title());
	  file.WRITE(buffer);
	}
  }
  sprintf(buffer,"pen = %i %i %s\n",set->pen.style(),
		  set->pen.width(),qcolor(set->pen.color()));
  file.WRITE(buffer);
  sprintf(buffer,"color = %s\n",qcolor(set->color));
  file.WRITE(buffer);
  sprintf(buffer,"marker = %i\n",(int)(set->marker));
  file.WRITE(buffer);
  sprintf(buffer,"markersize = %i\n",set->marker_size);
  file.WRITE(buffer);
  file.WRITE("</dataset>\n");
}


void XYPlot::save_ticks (QFile& file, XYPlotTickState& tick)
{
  //
  // Not yet implemented.
  //
}


void XYPlot::save_range (QFile& file, XYPlotLabelRangeState& range)
{
  //
  // Not yet implemented.
  //
}


void XYPlot::save_axis (QFile& file, int nr)
{
  char buffer [128];
  if (nr == 0) {
	file.WRITE("<axis x>\n");
	save_ticks(file,axis->x_ticks);
	save_ticks(file,axis->x_subticks);
	save_range(file,axis->x_range);
	sprintf(buffer,"log = %i\n",axis->log_x);
	file.WRITE(buffer);
	file.WRITE("</axis>\n");
  } else {
	file.WRITE("<axis y>\n");
	save_ticks(file,axis->y_ticks);
	save_ticks(file,axis->y_subticks);
	save_range(file,axis->y_range);
	sprintf(buffer,"log = %i\n",axis->log_y);
	file.WRITE(buffer);
	file.WRITE("</axis>\n");
  }
}


bool XYPlot::store (QFile& file, bool /*with_data*/)
{
  int i;

  file.writeBlock(start_mark(),strlen(start_mark()));
  PageObject::store(file);

  for (i=0; i<(int)records.count(); i++)
	save_dataset(file,records.at(i));
  save_axis(file,0);
  save_axis(file,1);

  file.writeBlock(end_mark(),strlen(end_mark()));
  return true;
}


void XYPlot::load_skip (QFile& file)
{
  char buffer [128];
  file.readLine(buffer,126);
  while (!file.atEnd() && strcmp(buffer,end_mark()) != 0)
	file.readLine(buffer,126);
}


bool XYPlot::load_dataset (QFile& file)
{
  int i;
  int a, b, c, d, e;
  char buffer [128];
  char namebuffer [128];
  XYPlotRecord* new_record;
  bool rec_ok;
  //
  // Load data set
  //
  new_record = new XYPlotRecord ();
  new_record->worksheet = 0;
  rec_ok = true;
  file.readLine(buffer,126);
  while (!file.atEnd() && strcmp(buffer,end_mark()) != 0) 
	{
	  if (buffer[strlen(buffer)-1] == '\n')
		buffer[strlen(buffer)-1] = 0;
	  if (strncmp(buffer,"</dataset>",10) == 0) {
		//
		// Dataset end tag. Leave inner loop.
		//
		break;
	  } else if  (strncmp(buffer,"worksheet",9) == 0) {
		//
		// Worksheet tag. Look up worksheet.
		//
		new_record->worksheet =
		  Worksheet::findWorksheet(strchr(buffer,'=')+2);
		if (new_record == 0) rec_ok = false;
	  } else if (strncmp(buffer,"pen",3) == 0) {
		//
		// Pen tag. Restore line pen.
		//
		sscanf(strchr(buffer,'=')+1,"%i %i %i %i %i",&a,&b,&c,&d,&e);
		new_record->pen = QPen(QColor(c,d,e),b,(PenStyle)a);
	  } else if (strncmp(buffer,"color",5) == 0) {
		//
		// Color tag. Restore marker color.
		//
		sscanf(strchr(buffer,'=')+1,"%i %i %i",&a,&b,&c);
		new_record->color = QColor(a,b,c);
	  } else if (strncmp(buffer,"markersize",10) == 0) {
		//
		// Marker size tag. Restore marker size.
		//
		sscanf(strchr(buffer,'=')+1,"%i",&a);
		new_record->marker_size = a;
	  } else if (strncmp(buffer,"marker",6) == 0) {
		//
		// Marker tag. Restore marker (diamond, circle etc.)
		//
		sscanf(strchr(buffer,'=')+1,"%i",&a);
		new_record->marker = (enum XYPlotRecord::XYPlotMarker) a;
	  } else {
		//
		// Possibly parameter tag. Check for paramName()s,
		// eventually look up column.
		//
		for (i=paramCount()-1; i>=0; i--) {
		  if (strncmp(buffer,paramName(i),strlen(paramName(i))) == 0) {
			if (new_record->worksheet != 0) {
			  sscanf(strchr(buffer,'=')+1,"%s",namebuffer);
			  new_record->input[i] = 
				new_record->worksheet->findColumn(namebuffer);
			  break;
			}
			//
			// Error: column, but no worksheet ?!
			//
			rec_ok = false;
			break;
		  }
		}
		if (i < 0) {
		  //
		  // Error: unknown tag.
		  //
		  rec_ok = false;
		}
	  }
	  file.readLine(buffer,126);
	}
  if (!rec_ok)
	return false;
  records.append(new_record);
  return true;
}


bool XYPlot::load_ticks (QFile&, XYPlotTickState&)
{
  //
  // Not yet implemented.
  //
  return true;
}

bool XYPlot::load_range (QFile&, XYPlotLabelRangeState&)
{
  //
  // Not yet implemented.
  //
  return true;
}


bool XYPlot::load_axis (QFile& file, int nr)
{
  char buffer [128];
  //
  // Load axis settings.
  //
  file.readLine(buffer,126);
  while (!file.atEnd() && strcmp(buffer,end_mark()) != 0) 
	{
	  if (buffer[strlen(buffer)-1] == '\n')
		buffer[strlen(buffer)-1] = 0;
	  if (strncmp(buffer,"</axis>",7) == 0) {
		return true;
	  } else if (strncmp(buffer,"<ticks>",7) == 0) {
		if (!load_ticks(file,(nr==0 ? axis->x_ticks : axis->y_ticks)))
		  return false;
	  } else if (strncmp(buffer,"<subticks>",10) == 0) {
		if (!load_ticks(file,(nr==0 ? axis->x_subticks : axis->y_subticks)))
		  return false;
	  } else if (strncmp(buffer,"<range>",7) == 0) {
		if(!load_range(file,(nr==0 ? axis->x_range : axis->y_range)))
		  return false;
	  } else if (strncmp(buffer,"log",3) == 0) {
		if (nr == 0)
		  sscanf(strchr(buffer,'=')+1,"%i",&(axis->log_x));
		else
		  sscanf(strchr(buffer,'=')+1,"%i",&(axis->log_y));
	  }
	  file.readLine(buffer,126);
	}
  return (strcmp(buffer,end_mark()) == 0);
}


bool XYPlot::restore (QFile& file, bool /*with_data*/)
{
  char buffer [128];

  PageObject::restore(file);

  file.readLine(buffer,126);
  while (!file.atEnd() && strcmp(buffer,end_mark()) != 0) 
	{
	  if (buffer[strlen(buffer)-1] == '\n')
		buffer[strlen(buffer)-1] = 0;
	  if (strncmp(buffer,"<dataset>",9) == 0) {
		if (!load_dataset(file)) {
		  load_skip(file);
		  return false;
		}
	  } else if (strncmp(buffer,"<axis x>",8) == 0) {
		if (!load_axis(file,0)) {
		  load_skip(file);
		  return false;
		}
	  } else if (strncmp(buffer,"<axis y>",8) == 0) {
		if (!load_axis(file,1)) {
		  load_skip(file);
		  return false;
		}
	  } else {
		//
		// Error: unknown section ?!
		//
		load_skip(file);
		return false;
	  }
	  file.readLine(buffer,126);
	}
  return (strcmp(buffer,end_mark()) == 0);
}


void XYPlot::plotSelectedColumns ()
{
  char msg [1024];
  XYPlotRecord* new_record;
  Column* data;
  int i;

  if (activeWS == 0) {
	QMessageBox::critical(0,"X/Y Plot: cant plot",
						  "I cant plot on this X/Y plot since there\n"\
						  "is no active worksheet.");
	return;
  }
  data = activeWS->getSelected();
  if (data == 0
	  || data[0].type()==Column::columnEmpty
	  || data[1].type()==Column::columnEmpty) {
	sprintf(msg,"The currently active worksheet, %s,\n"\
			"has no columns marked as \"X\" and \"Y\".",
			activeWS->caption());
	QMessageBox::critical(0,"X/Y Plot: cant plot",msg);
	return;
  }
  if (data[0].type() != Column::columnDouble) {
	sprintf(msg,"The column marked as \"X\", %s,\n"\
			"does not contain numerical data as required by\n"\
			"the X/Y plot.",
			data[0].title());
	QMessageBox::critical(0,"X/Y Plot: cant plot",msg);
	return;
  }
  if (data[1].type() != Column::columnDouble) {
	sprintf(msg,"The column marked as \"Y\", %s,\n"\
			"does not contain numerical data as required by\n"\
			"the X/Y plot.",
			data[0].title());
	QMessageBox::critical(0,"X/Y Plot: cant plot",msg);
	return;
  }

  new_record = new XYPlotRecord(*defaultSet);
  new_record->worksheet = activeWS;
  for (i=0; i<5; i++)
	new_record->input[i] = data[i];
  new_record->re_calc = true;
  records.append(new_record);
  paintEvent(0);

  treeitem()->setText(description());
  emit changedTreeItem();
}


void XYPlot::mousePressEvent (QMouseEvent* me)
{
  QPoint global_pos;
  global_pos = the_widget->mapToGlobal(me->pos());
  if (me->button() == RightButton)
	{
	  menu()->popup(global_pos);
	}
  else if (me->button() == LeftButton)
	{
	  QMessageBox::information(0,"Not implemented",
							   "The left mouse in an xy plot currently\n"\
							   "does nothing. This will change (hopefully)\n"\
							   "soon...");
	}
}
