#ifndef __IMPORT_H
#define __IMPORT_H

#include <qlist.h>
#include <qstring.h>
#include <qpopmenu.h>

#include "column.h"
#include "table.h"

class Worksheet;


/**
 *   @short ImportFilter BaseClass
 *
 *   This class defines the interface of all import filters.
 *   The importfilter must return its name, its optionsslot, and the table
 *   opened from the disk.
 *
 *   There are also members to store and get the latest used directory
 *   as well as the default directory.
 *
 *   Very cool structure: only this file has to be included by the main
 *   program. All filters have to be compiled to an object file and this
 *   object file has then to be linked to the main executable.
 *   They dont need to be included at any positon!!!
 *   So leave their includefiles where they are. Dont copy them to
 *   the korigin include directory.
 *
 *   Write new import filters by inheriting the class importfilter
 *   and defining all virtual declared functions. Watch importASCII
 *   for an example.
 *
 *   @author Martin Hfner (mh@ap-dec717c.physik.uni-karlsruhe.de)
 *   @version 0.1
 */
class importFilter
{
  public:
    /**
     *  This is the main ImportFilter procedure. It is called by
     *  centre and worksheet.
     */
    Worksheet* importFromFile ();

    /**
     *  return the popupmenu build from all found import filters.
     */
    static QPopupMenu* getImportFilterPopup ();

    /**
     * returns a options popup built from all found import filters.
     * Already connected to the appropriate slots.
     */
    static QPopupMenu* getImportOptionsPopup ();

    /**
     *  of course each class has to have one
     */
    importFilter () {};

    /**
     *  the same like for the constructor
     */
    virtual ~importFilter () {};

    /**
     *  get Description of the filter. This is the text being displayed
     *  everywhere where the filter will be called in the main program,
     *  for example in the Popup and FileDialog. 
     */
    virtual QString getDescription () = 0;

    /**
     *  a routine which should build a tabdialog and fill it with life.
     *  That means asking for the FileFilter, or specific import file format.
     */
    virtual void getOptionsSlot () = 0;


  protected:

    /**
     *  all import filters only display files the a specific file extension
     *  in the File dialogs. You have to return something like "*.html".
     */
    virtual const char* getFileFilter () = 0;

    /**
     *  This is the main function of import Filter. It will be called from
     *  the main program and reads a specific file from disk.
     */
    virtual Table* readTable (const char* filename) = 0;    

    /**
     *  return the latest used directory (the directory used on the last
     *  import command). This should be the default directory the first
     *  time this function is called in a session.
     *  @see #setLatestDirectory
     */
    virtual QString getLatestDirectory () = 0;

    /**
     *  set the latest used directory.
     */
    virtual void setLatestDirectory (const char* dir) = 0;

    /**
     *  return default directory. Needed by the options dialog which
     *  builds a dialog for setting the default directory. The default
     *  directory is the first directory used when importing a specific
     *  file in a session. All filters should have their own directory.
     *  @see #setDefaultDirectory 
     */
    virtual QString getDefaultDirectory () = 0;

    /**
     *  set the default directory.
     *  @see #setLatestDirectory
     *
     */
    virtual void setDefaultDirectory (const char* dir) = 0;
};



/**
 *  The import Module runner. Do not do anything with this class.
 *  It is a helper class for importFilter.
 *
 */
class runImportFilters : public QObject
{
  Q_OBJECT

  public slots:

    /**
     *  shows and runs the correct options dialog for the importfilter i within
     *  the importfilter QList.
     */
    void importOptions (int i);
};



//
// define the import filters list here
//

typedef QList<importFilter> ImportFilterList;
ImportFilterList& importFilterList ();


#endif








