/*

  The KOrigin Command Centre
  --------------------------

  " This was the wrong thing to do,
    This was the wrong one to be doing.
    This was the road to destiny,
    This was the way to our ruin. "

  (Megadeth, This Was My Life, from the album Countdown to Extinction)

  Written 1997,98 mostly by Patrick Schemitz.
  KOrigin (C) 1997,98 Patrick Schemtz, Martin Hfner.

  History:
  97/11/04  Getting started.
  97/11/06  Semi-stable version.
  97/11/06  KDE version started. (Flat) tree working. Wow!
  97/11/07  Tree substructure implemented. Pages integrated.
  97/11/08  Tree selection works now.
  97/11/10  Worksheet integrated. Wow!
  97/11/12  "Prototype" pattern for plots implemented. See warning.
  97/11/21  "Singleton" now supports "Prototype". What a team!
            Plot of selected data enabled.
  97/11/22  Import filters (by Martin) integrated.
  97/11/24  Options/Import/ASCII integrated. Help menu. (Martin)
  97/11/25  New icons for the tool bar.
  97/12/05  Export filters/options (by Martin) integrated.
  97/12/21  PRELIMINARY: Fitters integrated.
  97/12/30  Tree largely rewritten. Tree is mostly bugfree now.
  98/01/01  Own tree subclass implements right mouse button; KDE icon
            loader; cleanup.

  TODO: (in this order :-)
        - add lots of features.
        - debug those features.

*/


#include <iostream.h>

#include "centre.h"  // obscure: must be included *before* any Qt header ?!
#include "centre.moc"

#include <qarray.h>
#include <qaccel.h>
#include <qmsgbox.h>

#include <kmenubar.h>
#include <kiconloader.h>
#include <kapp.h>

#include "debug.h"
#include "column.h"
#include "table.h"
#include "worksheet.h"
#include "plot.h"
#include "page.h"
#include "pg_options.h"
#include "import.h"
#include "export.h"
#include "fitter.h"
#include "tree.h"
#include "resultwindow.h"


#define PIX_EXIT   "exit.xpm"
#define PIX_PLOT   "tb_plot16.xpm"
#define PIX_TEXT   "tb_text16.xpm"
#define PIX_MOVE   "tb_move16.xpm"


/* Default value for "IlikeWindows" is true. If a user is too stupid
   to switch it off, s/he probably likes it anyway :-)
   Besides, it makes a good debugging tool.
*/

bool KCentre::IlikeWindows = true;


/* KCentre, the main window on Korigin. This KTopLevelWidget is the
   main window of KOrigin. It consists of the menu, a status bar,
   a tool bar and the tree view.
*/

KCentre::KCentre (QWidget*, const char* name)
  : KTopLevelWidget(name)
{
  IFDEBUG("korigin.centre",3)
	cout << "KCentre::KCentre() starts." << endl;
  ENDDEBUG;

  setCaption("Korigin");

  KConfig* conf = kapp->getConfig();
  conf->setGroup("Settings");
  if (strcmp(conf->group(),"Settings") == 0) {
	IlikeWindows = (strcasecmp((const char*)(
			conf->readEntry("IlikeWindows", "yes")), "yes") == 0);
	resize( conf->readNumEntry("width",550),
		conf->readNumEntry("height",150) );
  } else {
	IlikeWindows = true;
  }

  Worksheet::list().setAutoDelete(true);
  Page::list().setAutoDelete(true);

  QPopupMenu* newpop;
  QPopupMenu* openpop;
  QPopupMenu* savepop;

  newpop = new QPopupMenu();
  newpop->insertItem("Session", this, SLOT(clearSession()));
  newpop->insertSeparator();
  newpop->insertItem("Worksheet", this, SLOT(createWorksheet()));
  newpop->insertItem("Page", this, SLOT(createPage()));

  openpop = new QPopupMenu();
  openpop->insertItem("Session", this, SLOT(loadSession()));
  openpop->insertSeparator();
  openpop->insertItem("Worksheet", this, SLOT(openWorksheet()));
  openpop->insertItem("Page", this, SLOT(openPage()));

  savepop = new QPopupMenu();
  savepop->insertItem("Session", this, SLOT(saveSession()));
  savepop->insertSeparator();
  savepop->insertItem("Worksheet", this, SLOT(saveWorksheet()));
  savepop->insertItem("Page", this, SLOT(savePage()));

  import_menu = importFilter::getImportFilterPopup();
  connect(import_menu,SIGNAL(activated(int)),SLOT(importTable(int)));

  file_menu = new QPopupMenu();
  file_menu->insertItem("New", newpop);
  file_menu->insertItem("Open", openpop);
  file_menu->insertItem("Save", savepop);
  file_menu->insertSeparator();
  file_menu->insertItem("Open session", this, SLOT(loadSession()), Key_F3);
  file_menu->insertItem("Save session", this, SLOT(saveSession()), Key_F2);
  file_menu->insertItem("Save session as", this, SLOT(saveSessionAs()));
  file_menu->insertItem("Close", this, SLOT(clearSession()), CTRL+Key_F4);
  file_menu->insertSeparator();
  file_menu->insertItem("Import...", import_menu);
  file_menu->insertItem("Export...", exportFilter::getExportFilterPopup());
  file_menu->insertSeparator();
  file_menu->insertItem("Print page", this, SLOT(printPage()));
  file_menu->insertSeparator();
  file_menu->insertItem("E&xit",  qApp, SLOT(quit()), CTRL+Key_Q);

  edit_menu = buildEditMenu(0);
  worksheet_menu = buildWorksheetMenu(0);

  IFDEBUG("korigin.centre",5)
	cout << "KCentre::KCentre() page starts." << endl;
  ENDDEBUG;

  new_plot_menu = buildNewPlotMenu(0);
  page_menu = buildInitialPageMenu(0);
  plot_menu = buildInitialPlotMenu(0);
  fitter_menu = Fitter::getFitterMenu();
  connect(fitter_menu,SIGNAL(activated(int)),SLOT(executeFitter(int)));

  options_menu = new QPopupMenu();
  options_menu->insertItem("Import", importFilter::getImportOptionsPopup());
  options_menu->insertItem("Export", exportFilter::getExportOptionsPopup());
  options_menu->insertItem("Fitter", Fitter::getFitterOptionsMenu());
  options_menu->insertItem("Page", this, SLOT(pageDefaults()));

  IFDEBUG("korigin.centre",5)
	cout << "KCentre::KCentre() help starts." << endl;
  ENDDEBUG;

  // the window menu
  window_menu = new QPopupMenu;
  
  resultID = window_menu->insertItem ("Show &Results", this, 
				       SLOT(slotResultWindow()));

  // the help menu
  help_menu = new QPopupMenu;
  help_menu->insertItem("&Help", this, SLOT(slotShowHelper()), Key_F1);
  help_menu->insertItem("&Index", this, SLOT(slotShowHelpIndex()));
  help_menu->insertSeparator();
  help_menu->insertItem("About &Qt", this, SLOT(slotShowAboutQt()));
  help_menu->insertItem("&About", this, SLOT(slotShowAboutBox()));
  // the first Item should work automatically, but it does not

  IFDEBUG("korigin.centre",5)
	cout << "KCentre::KCentre() menu assembling." << endl;
  ENDDEBUG;

  menu = new KMenuBar(this);
  menu->insertItem( "&File", file_menu );
  menu->insertItem( "&Edit", edit_menu );
  menu->insertItem( "&Worksheet", worksheet_menu );
  menu->insertItem( "P&age", page_menu );
  menu->insertItem( "&Plot", plot_menu );
  menu->insertItem( "Fi&t", fitter_menu );
  menu->insertItem( "W&indow", window_menu );
  menu->insertItem( "&Options", options_menu );
  menu->insertSeparator();
  menu->insertItem( "&Help", help_menu );

  result = new resultWindow ();

  IFDEBUG("korigin.centre",5)
	cout << "KCentre::KCentre() menu assembled." << endl;
  ENDDEBUG;

  toolbar = new KToolBar(this);
  connect(toolbar,SIGNAL(clicked(int)),SLOT(toolbarClicked(int)));
  toolbar->insertButton(KApplication::getKApplication()->getIconLoader()->loadIcon("exit.xpm"),199,TRUE,"Eggshit - quit");
  toolbar->insertSeparator();
  toolbar->insertButton(KApplication::getKApplication()->getIconLoader()->loadIcon(PIX_MOVE),200,TRUE,"Move/resize Objects");
  toolbar->insertButton(KApplication::getKApplication()->getIconLoader()->loadIcon(PIX_PLOT),202,TRUE,"Edit Plots");
  toolbar->insertButton(KApplication::getKApplication()->getIconLoader()->loadIcon(PIX_TEXT),201,TRUE,"Enter/edit Text");
  toolbar->move(0,menu->height());

  tree = new Tree(this);
  connect(tree,SIGNAL(highlighted(int)),SLOT(slotTreeSelect(int)));
  connect(tree,SIGNAL(collapsed(int)),SLOT(slotTreeFold(int)));
  connect(tree,SIGNAL(expanded(int)),SLOT(slotTreeUnfold(int)));
  connect(tree,SIGNAL(selected(int)),SLOT(slotTreeToggleFold(int)));
  connect(tree,SIGNAL(pageHasChanged(Page*)),SLOT(changeActivePage(Page*)));
  connect(tree,SIGNAL(plotHasChanged(Plot*)),SLOT(changeActivePlot(Plot*)));
  connect(tree,SIGNAL(worksheetHasChanged(Worksheet*)),
		  SLOT(changeActiveWorksheet(Worksheet*)));
  connect(tree,SIGNAL(removePage(Page*)),SLOT(removePage(Page*)));
  connect(tree,SIGNAL(removePageObject(PageObject*)),
		  SLOT(removePageObject(PageObject*)));
  connect(tree,SIGNAL(removeWorksheet(Worksheet*)),
		  SLOT(removeWorksheet(Worksheet*)));
  tree->move(0,menu->height()+toolbar->height());
  tree->resize(width(),height()-menu->height()-toolbar->height());
  tree->setExpandLevel(2);

  status = new KStatusBar(this);
  status->insertItem("Korigin (C) 1997,98 Hfner & Schemitz",0);

  IFDEBUG("korigin.centre",5)
	cout << "KCentre::KCentre() assembling KTopLevelWidget." << endl;
  ENDDEBUG;

  setView(tree);
  setMenu(menu);
  setStatusBar(status);
  addToolBar(toolbar);

  IFDEBUG("korigin.centre",5)
	cout << "KCentre::KCentre() assembling KTopLevelWidget." << endl;
  ENDDEBUG;

  Page::loadDefaults(0);

  IFDEBUG("korigin.centre",3)
	cout << "KCentre::KCentre() ends." << endl;
  ENDDEBUG;
}


KCentre::~KCentre ()
{
  kapp->getConfig()->sync();
  delete result;      // delete the result window
}


/*
  Session Management Section
  --------------------------
*/

QString KCentre::getSaveName (const char* extension)
{
  QString dir;
  QString fileName;
  bool accepted;
  KConfig* conf;
  char buffer [1024];

  conf = kapp->getConfig();
  conf->setGroup("Settings");
  if (strcmp(conf->group(),"Settings") == 0)
	dir = conf->readEntry("directory",getenv("HOME"));

  do
	{
	  fileName = QFileDialog::getSaveFileName(dir,extension);
	  if (fileName.isEmpty())
		return QString();
	  if (QFile::exists(fileName)) {
		sprintf(buffer,"The following file already exists:\n   %s\n"\
				"Replace it?",
				(const char*)fileName);
		accepted = (QMessageBox::warning(0,"Replace Existing File?", buffer,
										 "Yes","No",0,1) == 0);
	  } else
		accepted = true;
	}
  while (!accepted);
  QFile file (fileName);
  file.open(IO_WriteOnly);
  if (!QFileInfo(file).isWritable()) {
	sprintf(buffer,"The following file is not writable for you:\n"\
			"   %s\n"\
			"Most likely, you dont have write permissions for the"\
			"directory you selected. Try to store the file into you"\
			"home directory, %s",
			(const char*)fileName,getenv("HOME"));
	QMessageBox::information(0,"Write Permission Denied",buffer);
	file.close();
	return QString();
  }
  file.close();
  return fileName;
}


QString KCentre::getLoadName (const char* extension)
{
  char buffer [1024];
  QString fileName, dir;
  KConfig* conf;
  bool ok;

  conf = kapp->getConfig();
  conf->setGroup("Settings");
  if (strcmp(conf->group(),"Settings") == 0)
	dir = conf->readEntry("directory",getenv("HOME"));

  do
	{
	  fileName = QFileDialog::getOpenFileName(dir,extension);
	  if (fileName.isEmpty()) 
		return QString();
	  if (!(ok = QFile::exists(fileName))) {
		sprintf(buffer,"The following file was not found:\n"\
				"   %s\n",
				(const char*)fileName);
		if (QMessageBox::information(0,"File Inexistent",buffer,
									 "Retry","Cancel") == 1)
		  return QString();
	  }
	}
  while (!ok);
  QFile file (fileName);
  file.open(IO_ReadOnly);
  if (!QFileInfo(file).isReadable) {
	sprintf(buffer,"The following file is not readable for you:\n"\
			"   %s\n"\
			"You might want to inform the owner of the"\
			"file, %s, to change permissions.",
			(const char*)fileName, QFileInfo(file).owner());
	QMessageBox::information(0,"File Unreadable",buffer);
	file.close();
	return QString();
  }
  file.close();
  return fileName;
}


void KCentre::clearSession ()
{
  int i, ok;
  char buffer [1024];
  if (Worksheet::list().count() > 0 || Page::list().count() > 0) {
	sprintf(buffer,"Clear Session %s",(const char*)sessionName);
	ok = QMessageBox::critical(0,buffer,
					 "This would DELETE all open worksheets and pages!\n"\
					 "Are you sure you want to DELETE them all?",
					 "Yes","No/Cancel",0,1);
	if (ok == 1) return;
  }
  for (i=(int)(tree->count())-1; i>=0; i--)
	tree->takeItem(i);
  Worksheet::list().setAutoDelete(true);
  Worksheet::list().clear();
  Page::list().setAutoDelete(true);
  Page::list().clear();
  result->clear();
  sessionName = QString();
}


void KCentre::loadSession ()
{
  QString dir, fileName;
  char typeString [128];
  char buffer [1024];
  Worksheet* ws;
  Page* page;

  fileName = getLoadName("*.kos");
  if (fileName.isEmpty()) return;

  QFile file (fileName);
  file.open(IO_ReadOnly);

  while (!file.atEnd())
	{
	  file.readLine(typeString,126);
	  if (file.atEnd()) break;

	  if (strcmp(typeString,"<worksheet>\n") == 0) {
		ws = Worksheet::loadFromDisk(&file);
		if (ws == 0) {
		  sprintf(buffer,"The following file contains a corrupted "\
				  "worksheet:\n   %s\n"\
				  "The session has not been fully loaded.",
				  (const char*)fileName);
		  QMessageBox::critical(0,"Session Load Error",buffer);
		  file.close();
		  return;
		}
		propagate_worksheet(ws);
	  } else if (strcmp(typeString,Page::start_mark()) == 0) {
		page = Page::restore_page(file);
		if (page == 0) {
		  sprintf(buffer,"The following file contains a corrupted "\
				  "page:\n   %s\n"\
				  "The session has not been fully loaded.",
				  (const char*)fileName);
		  QMessageBox::critical(0,"Session Load Error",buffer);
		  file.close();
		  return;
		}
		propagate_page(page);
		page->restore_objects(file,true);
	  } else if (strcmp(typeString,"<results>\n") == 0) {
	        result->loadFromDisk(&file);
	  } else {
		sprintf(buffer,"The following file contains corrupted "\
				"data:\n   %s\n"\
				"The session has not been fully loaded.\n"\
				"(Unrecognized type field: %s.)",
				(const char*)fileName,typeString);
		QMessageBox::critical(0,"Session Load Error",buffer);
		file.close();
		return;
	  }
	}
  file.close();
}


void KCentre::saveSessionAs ()
{
  QString fileName;

  fileName = getSaveName("*.kos");
  if (fileName.isEmpty()) return;

  sessionName = fileName;
  saveSession();
}


void KCentre::saveSession ()
{
  uint i;
  char buffer [1024];

  if (sessionName.isEmpty())
	{
	  saveSessionAs();
	  return;
	}

  QFile file (sessionName);
  file.open(IO_WriteOnly);

  sprintf(buffer,"Session file is:\n   %s\n",(const char*)sessionName);
  strcat(buffer,"Worksheets saved:\n");
  for (i=0; i<Worksheet::list().count(); i++)
	{
	  Worksheet::list().at(i)->saveToDisk(&file);
	  strcat(buffer,"   ");
	  strcat(buffer,Worksheet::list().at(i)->caption());
	  strcat(buffer,"\n");
	}
  strcat(buffer,"Pages saved:\n");
  for (i=0; i<Page::list().count(); i++)
	{
	  Page::list().at(i)->store(file,true);
	  strcat(buffer,"   ");
	  strcat(buffer,Page::list().at(i)->caption());
	  strcat(buffer,"\n");
	}
  result->saveToDisk (&file);
  file.close();

  if (IlikeWindows)
	QMessageBox::information(0,"Session Saved.",buffer);
}



/*
  Page Section
  ------------
*/

void KCentre::propagate_page (Page* new_page)
{
  Page::list().append(new_page);
  connect(new_page,SIGNAL(activePlotChanged(Plot*)),
		  this,SLOT(changeActivePlot(Plot*)));
  connect(new_page,SIGNAL(activePageChanged(Page*)),
		  this,SLOT(changeActivePage(Page*)));
  connect(new_page,SIGNAL(addTreeChild(TreeItem*,TreeItem*)),
		  this,SLOT(treeInsertChild(TreeItem*,TreeItem*)));
  connect(new_page,SIGNAL(treeItemChanged()),tree,SLOT(update()));
  new_page->show();
  tree->insertItem(new_page->treeitem());
  changeActivePage(new_page);
}


void KCentre::unpropagate_page (Page* old_page)
{
  uint index;
  tree->takeItem(tree->itemIndex(old_page->treeitem()));
  index = Page::list().find(old_page);
  if (index+1 < Page::list().count())
	changeActivePage(Page::list().at(index+1));
  else if (index > 0)
	changeActivePage(Page::list().at(index-1));
  else
	changeActivePage(0);
  Page::list().remove(old_page);
}


void KCentre::openPage ()
{
  QString fileName = getLoadName("*.kop");
  if (fileName.isEmpty()) return;
  QFile file (fileName);
  file.open(IO_ReadOnly);
  Page* page = Page::restore_page(file);
  if (page == 0) {
	QMessageBox::critical(0,"Page Load Error","????");
	return;
  }
  propagate_page(page);
  page->restore_objects(file,true);
}


void KCentre::savePage ()
{
  if (Page::list().count() == 0) {
	QMessageBox::information(0,"Page Save Error",
							 "You dont have a page opened, so I"\
							 "cant save any.");
	return;
  }
  if (activeWS == 0 && Page::list().count() != 1) {
	QMessageBox::information(0,"Page Save Error",
							 "You havent marked a page active. Mark"\
							 "one active (i.e. click in it) and retry.");
	return;
  }
  if (activePage == 0)
	changeActivePage(Page::list().at(0));

  if (activePage == 0)
	return;

  QString fileName = getSaveName("*.kop");
  if (fileName.isEmpty()) return;
  QFile file (fileName);
  file.open(IO_WriteOnly);
  activePage->store(file,true);
  file.close();
}


void KCentre::createPage ()
{
  Page* new_page = new Page(0,0);

  if (new_page == 0) {
	QMessageBox::information(0,"Cant create page",
							 "I couldnt create an empty page. You might\n"\
							 "be out of memory.");
	return;
  }
  propagate_page(new_page);
}


void KCentre::removePage (Page* the_page)
{
  if (the_page == 0) return;
  unpropagate_page(the_page);
}


void KCentre::removePageObject (PageObject* old_obj)
{
  if (old_obj == 0) return;
  if (old_obj->inherits("Plot")) {
	cout << "This object was a plot: " << old_obj->description() << endl;
	changeActivePlot(0);
  }
  tree->takeItem(tree->itemIndex(old_obj->treeitem()));
  old_obj->page()->remove_object(old_obj);
}


void KCentre::printPage (Page* the_page)
{
  if (the_page == 0) {
	if (activePage == 0) {
	  QMessageBox::information(0,"Cant print page",
							   "There is no page I could print.\n"\
							   "File|New|Page opens a new page.");
	  return;
	}
	activePage->printPage();
  } else {
	the_page->printPage();
  }
}


void KCentre::printPage ()
{
  printPage(activePage);
}


void KCentre::pageDefaults ()
{
  QDialog* dialog = new PageOptionsDialog();
  dialog->exec();
  Page::storeDefaults(0);
  delete dialog;
}


void KCentre::changeActivePage (Page* the_page)
{
  if (activePage) activePage->deactivate();
  activePage = the_page;
  if (activePage == 0) {
	buildInitialPageMenu(page_menu);
	tree->update();
  } else {
	activePage->menu(page_menu);
	activePage->activate();
	tree->pageChanged(activePage);
  }
}


QPopupMenu* KCentre::buildInitialPageMenu (QPopupMenu* predefined)
{
  QPopupMenu* menue;
  if (predefined) {
	menue = predefined;
	menue->clear();
  } else {
	menue = new QPopupMenu();
  }
  menue->insertItem("New plot", new_plot_menu);
  return menue;
}



/*
  Fitter Section
  --------------
*/

void KCentre::executeFitter (int id)
{
  int i, n;
  Fitter* fitter;
  Column* data;
  Column* in_data;
  Column* out_data;
  char msg [1024];

  if ((fitter = Fitters().at(id)) == 0) {
	QMessageBox::critical(0,"Bug: Fitter not found",
						  "You just encountered a Real Bug(tm):"\
						  "The selected fitter was not found in the\n"\
						  "fitter list. Please report this bug.");
	return;
  }

  if (activeWS == 0) {
	QMessageBox::information(0,"Cant fit",
							 "You dont have an active worksheet. Please\n"\
							 "click in a worksheet and retry.");
	return;
  }

  data = activeWS->getSelected();
  if (data == 0) {
	QMessageBox::information(0,"Cant fit",
							 "You dont have selected columns in the\n"\
							 "active worksheet. Please select the columns\n"\
							 "for the fitter and retry.");
	return;
  }

  IFDEBUG("korigin.centre",2)
	cout << "KCentre::executeFitter(" << fitter->name() 
		 << ") prepares itself." << endl;
  ENDDEBUG;

  n = fitter->getInputColumnCount();
  in_data = new Column [n];
  for (i=0; i<n; i++)
	in_data[i] = data[i];

  n = fitter->getOutputColumnCount();
  out_data = new Column [n];
  for (i=0; i<n; i++)
	out_data[i] = Column(Column::columnDouble);

  fitter->setInputColumns(in_data);
  fitter->setOutputColumns(out_data);

  if (fitter->fit() == false) {
	sprintf(msg,"Fitter \"%s\" failed.\nNo plot and worksheet will\n"\
			"be generated.",fitter->name());
	QMessageBox::information(0,"Fitter failed.",msg);
	return;
  }

  IFDEBUG("korigin.centre",3)
	cout << "KCentre::executeFitter() building table..." << endl;
  ENDDEBUG;

  QString title = fitter->name();
  title += " results";
  Table* T = new Table;
  T->setName(title.data());
  n = fitter->getOutputColumnCount();
  for (i=0; i<n; i++)
	T->insertColumn(i,out_data[i]);

  Worksheet* ws = new Worksheet();
  Worksheet::list().append(ws);
  tree->insertItem(ws->treeitem());
  connect(ws,SIGNAL(activeWSChanged(Worksheet*)),
		  this,SLOT(changeActiveWorksheet(Worksheet*)));
  connect(ws,SIGNAL(treeItemChanged()),
		  tree,SLOT(update()));
  changeActiveWorksheet(ws);
  ws->setTable(T);
  ws->show();

  IFDEBUG("korigin.centre",3)
	cout << "KCentre::executeFitter() building plot..." << endl;
  ENDDEBUG;

  if (activePlot == 0) {
	initialPlotDefault();
  } else {
	activePlot->plotSelectedColumns();
  }

  IFDEBUG("korigin.centre",2)
	cout << "KCentre::executeFitter() ends." << endl;
  ENDDEBUG;
}



/*
  Plot Section
  ------------
*/

QPopupMenu* KCentre::buildNewPlotMenu (QPopupMenu* predefined)
{
  Plot* iter;
  QPopupMenu* new_plot;

  if (predefined) {
	new_plot = predefined;
	new_plot->clear();
  } else {
	new_plot = new QPopupMenu();
  }

  for (iter=Plots().first(); iter!=0; iter=Plots().next())
	new_plot->insertItem(iter->name());
  connect(new_plot,SIGNAL(activated(int)),SLOT(initialPlot(int)));

  return new_plot;
}


QPopupMenu* KCentre::buildInitialPlotMenu (QPopupMenu* predefined)
{
  QPopupMenu* menue;
  if (predefined) {
	menue = predefined;
	menue->clear();
  } else {
	menue = new QPopupMenu();
  }
  menue->insertItem("Plot selected data", this, SLOT(initialPlotDefault()));
  return menue;
}


void KCentre::changeActivePlot (Plot* thePlot)
{
  if (activePlot) activePlot->deactivate();
  activePlot = thePlot;
  if (activePlot == 0) {
	buildInitialPlotMenu(plot_menu);
	tree->update();
  } else {
	activePlot->menu(plot_menu);
	activePlot->activate();
	tree->plotChanged(activePlot);
  }
}


void KCentre::initialPlotDefault ()
{
  Plot* prototype;

  IFDEBUG("korigin.centre",2)
	cout << "First Blood, First Plot: default type with plotting." << endl;
  ENDDEBUG;

  if (activeWS == 0) {
	QMessageBox::information(0,"Cant plot",
							 "I cant plot anything because there is\n"\
							 "no active worksheet to take data from.\n"\
							 "Load a worksheet and retry.");
	return;
  }
  prototype = Plot::defaultPlot();
  if (prototype == 0) {
	QMessageBox::critical(0,"Cant create plot",
						  "I cant find a plot prototype. You seem to have\n"\
						  "forgotten the plot modules. This being the\n"\
						  "case, youll have to recompile the program.\n"\
						  "OTOH, if you *have* included the plot modules,\n"\
						  "this is definitive a bug.");
	return;
  }

  if (activePage == 0)
	createPage();
  if (activePage == 0) {
	return;
  }
  activePage->make_plot(prototype,true);
}


void KCentre::initialPlot (int id)
{
  Plot* prototype;

  IFDEBUG("korigin.centre",2)
	cout << "First Blood, First Plot: without plotting." << endl;
  ENDDEBUG;

  if (activeWS == 0) {
	QMessageBox::information(0,"Cant plot",
							 "I cant plot anything because there is\n"\
							 "no active worksheet to take data from.\n"\
							 "Load a worksheet and retry.");
	return;
  }

  prototype = Plots().at(id);
  if (prototype == 0) {
	QMessageBox::critical(0,"Cant create plot",
						  "I cant find a plot prototype. You seem to have\n"\
						  "forgotten the plot modules. This being the\n"\
						  "case, youll have to recompile the program.\n"\
						  "OTOH, if you *have* included the plot modules,\n"\
						  "this is definitive a bug.");
	return;
  }

  if (activePage == 0)
	createPage();
  if (activePage == 0) {
	return;
  }
  activePage->make_plot(prototype);
}



/*
  Worksheet Section
  -----------------
*/

void KCentre::propagate_worksheet (Worksheet* new_ws)
{
  Worksheet::list().append(new_ws);
  new_ws->show();
  tree->insertItem(new_ws->treeitem());
  connect(new_ws,SIGNAL(activeWSChanged(Worksheet*)),
		  this,SLOT(changeActiveWorksheet(Worksheet*)));
  connect(new_ws,SIGNAL(treeItemChanged()),
		  tree,SLOT(update()));
  changeActiveWorksheet(new_ws);
}


void KCentre::unpropagate_worksheet (Worksheet* old_ws)
{
  uint index;
  index = Worksheet::list().find(old_ws);
  if (index+1 < Worksheet::list().count())
	changeActiveWorksheet(Worksheet::list().at(index+1));
  else if (index > 0)
	changeActiveWorksheet(Worksheet::list().at(index-1));
  else
	changeActiveWorksheet(0);
  tree->takeItem(tree->itemIndex(old_ws->treeitem()));
  Worksheet::list().remove(old_ws);
}


void KCentre::createWorksheet ()
{
  Worksheet* ws = new Worksheet();
  if (ws == 0) {
	QMessageBox::critical(0,"Cant create worksheet",
						  "I couldnt create a new worksheet.\n"\
						  "Guess youre out of memory.");
	return;
  }
  propagate_worksheet(ws);
}


void KCentre::openWorksheet ()
{
  Worksheet* ws = Worksheet::openWorksheetFromDisk ();
  if (ws != 0)
	propagate_worksheet(ws);
}


void KCentre::saveWorksheet ()
{
  if (Worksheet::list().count() == 0) {
	QMessageBox::information(0,"Worksheet Save Error",
							 "You dont have a worksheet opened, so I"\
							 "cant save any.");
	return;
  }
  if (activeWS == 0 && Worksheet::list().count() != 1) {
	QMessageBox::information(0,"Worksheet Save Error",
							 "You havent marked a worksheet active. Mark"\
							 "one active (i.e. click in it) and retry.");
	return;
  }
  if (activeWS == 0)
	changeActiveWorksheet(Worksheet::list().at(0));

  if (activeWS)
	activeWS->saveWorksheetAs();
}


void KCentre::removeWorksheet (Worksheet* worksheet)
{
  if (worksheet == 0) return;
  unpropagate_worksheet(worksheet);
}


QPopupMenu* KCentre::buildWorksheetMenu (QPopupMenu* predefined)
{
  QPopupMenu* worksheet;
  if (predefined) {
	worksheet = predefined;
	worksheet->clear();
  } else {
	worksheet = new QPopupMenu();
  }
  worksheet->insertItem("New Worksheet", this, SLOT(createWorksheet()));
  return worksheet;
}


void KCentre::changeActiveWorksheet (Worksheet* theWS)
{
  int oldIndex;
  Worksheet* oldWS;

  oldIndex = findTreeObject(oldWS=activeWS);
  activeWS = theWS;
  if (oldWS && tree->itemAt(oldIndex))
	tree->itemAt(oldIndex)->setPixmap(oldWS->getSymbol());
  if (activeWS) {
	tree->itemAt(findTreeObject(activeWS))->setPixmap(activeWS->getSymbol());
	activeWS->getMenu(worksheet_menu);
	activeWS->getEditMenu(edit_menu);
  } else {
	buildWorksheetMenu(worksheet_menu);
	buildEditMenu(edit_menu);
  }
  tree->update();
}



/*
  Tree Section
  ------------
*/

int KCentre::findTreeObject (QObject* theObject)
{
  uint i;
  for (i=0; i<tree->count(); i++)
	if (((TreeItem*)(tree->itemAt(i)))->object() == theObject)
	  return i;
  return -1;
}


void KCentre::slotTreeSelect (int id)
{
  Page* selectedPage;
  Plot* selectedPlot;
  Worksheet* selectedWS;
  TreeItem* selectedItem;

  IFDEBUG("korigin.centre",3)
	cout << "Tree View: selected item ID: "
		 << id << endl;
  ENDDEBUG;

  if (id<0 || id>=(int)tree->count())
	{
	  IFDEBUG("korigin.centre",1)
		cout << "Tree View: index " << id << " out of range." << endl;
	  ENDDEBUG;
	  return;
	}

  selectedItem = (TreeItem*)(tree->itemAt(id));
  if (strcmp(selectedItem->object()->className(),"Page") == 0)
	{
	  //
	  // Selected a "Page", so:
	  // - set activePage
	  // - set activePlot
	  //
	  selectedPage = (Page*)selectedItem->object();
	  if (activePage == selectedPage)
		return;
	  changeActivePage(selectedPage);

	  //if (activePage->getPlotCount() > 0)
	  //	activePlot = activePage->getPlot(0);
	  //else
	  //	activePlot = 0;
	  IFDEBUG("korigin.centre",3)
		  cout << "Tree View: Page selected: "
			   << activePage->caption() << endl;
	  ENDDEBUG;
	}
  else if (selectedItem->object()->inherits("Plot"))
	{
	  //
	  // Selected a "PlotWidget" stem, so:
	  // - set activePlot
	  // - find correct activePage by "climbing the tree"
	  //
	  selectedPlot = (Plot*)selectedItem->object();
	  if (activePlot == selectedPlot)
		return;

	  changeActivePlot(selectedPlot);

	  selectedItem = (TreeItem*)selectedItem->getParent();
	  selectedPage = (Page*)selectedItem->object();
	  if (activePage == selectedPage)
		return;
	  changeActivePage(selectedPage);

	  IFDEBUG("korigin.centre",3)
		cout << "Tree View: PlotWidget selected: "
			 << activePage->caption() << endl;
	  ENDDEBUG;
	}
  else if (strcmp(selectedItem->object()->className(),"Worksheet") == 0)
	{
	  //
	  // Selected a "Worksheet", so:
	  // - set activeWS
	  //
	  selectedWS = (Worksheet*)selectedItem->object();
	  if (activeWS == selectedWS)
		return;

	  changeActiveWorksheet(selectedWS);

	  IFDEBUG("korigin.centre",3)
		cout << "Tree View: Worksheet selected: " 
			 << activeWS->caption() << endl;
	  ENDDEBUG;
	}
  else
	{
	  IFDEBUG("korigin.centre",1)
		cout << "Tree View: illegal object " 
			 << selectedItem->object()->className()
			 << " in tree list, index " << id << endl;
	  ENDDEBUG;
	}
  tree->update();
}


void KCentre::slotTreeToggleFold (int id)
{
  QObject* object = ((TreeItem*)tree->itemAt(id))->object();
  if (strcmp(object->className(),"Page") == 0 ||
	  strcmp(object->className(),"Worksheet") == 0)
	{
	  if (((QWidget*)object)->isVisible())
		((QWidget*)object)->iconify();
	  else
		((QWidget*)object)->show();
	}
}


void KCentre::slotTreeFold (int id)
{
  QObject* object = ((TreeItem*)tree->itemAt(id))->object();
  if (strcmp(object->className(),"Page") == 0 ||
	  strcmp(object->className(),"Worksheet") == 0)
	{
	  ((QWidget*)object)->iconify();
	}
}


void KCentre::slotTreeUnfold (int id)
{
  QObject* object = ((TreeItem*)tree->itemAt(id))->object();
  if (strcmp(object->className(),"Page") == 0 ||
	  strcmp(object->className(),"Worksheet") == 0)
	{
	  ((QWidget*)object)->show();
	}
}


void KCentre::treeInsertChild (TreeItem* parent, TreeItem* child)
{
  int index = tree->itemIndex(parent);
  IFDEBUG("korigin.centre",2)
	cout << "KCentre::treeInsertChild() inserts " 
		 << child->object()->className()
		 << " at index " << index << endl;
  ENDDEBUG;
  tree->addChildItem(child,index);
}


/*
 *  The window menu
 *
 */

void KCentre::slotResultWindow ()
{
  result->show ();
}


/*
  Help Section
  ------------
*/

void KCentre::slotShowHelper ()
{
  KApplication::getKApplication()->invokeHTMLHelp ("korigin/korigin.html","");
}


void KCentre::slotShowHelpIndex ()
{
  KApplication::getKApplication()->invokeHTMLHelp ("korigin/index.html","");
}


void KCentre::slotShowAboutQt ()
{
  QMessageBox::aboutQt (0);
}


void KCentre::slotShowAboutBox ()
{
  QMessageBox::information (0, "About Korigin", "Korigin 0.1 alpha\n\n"\
							"Scientific Data Visualization\n\n(c)1997,98 by "\
							"Martin Hfner & Patrick Schemitz\n\nHomepage:"\
							"http://ap-dec717c.physik.uni-karlsruhe.de/~ko"\
							"rigin\n\nE-mail:\nmh@ap-dec717c.physi"\
							"k.uni-karlsruhe.de\nab04@rz.uni-karlsruhe.de");
}



/*
  Miscellaneous Section
  ---------------------
*/

QPopupMenu* KCentre::buildEditMenu (QPopupMenu* predefined)
{
  int id;
  QPopupMenu* edit;
  if (predefined) {
	edit = predefined;
	edit->clear();
  } else {
	edit = new QPopupMenu();
  }
  id = edit->insertItem("Cut", this, SLOT(unimplemented()));
  edit->setItemEnabled(id,false);
  id = edit->insertItem("Copy", this, SLOT(unimplemented()));
  edit->setItemEnabled(id,false);
  id = edit->insertItem("Paste", this, SLOT(unimplemented()));
  edit->setItemEnabled(id,false);
  return edit;
}


void KCentre::importTable (int i)
{
  Worksheet* ws = importFilterList().at(i)->importFromFile();
  if (ws != 0)
    {
	  Worksheet::list().append(ws);
      ws->show();
	  tree->insertItem(ws->treeitem());
	  connect(ws,SIGNAL(activeWSChanged(Worksheet*)),
			  this,SLOT(changeActiveWorksheet(Worksheet*)));
	  connect(ws,SIGNAL(treeItemChanged()),
			  tree,SLOT(update()));
      changeActiveWorksheet(ws);
    }
}


void KCentre::toolbarClicked (int id)
{
  IFDEBUG("korigin.centre",3)
	cout << "Toolbar Pixmap " << id << " clicked." << endl;
  ENDDEBUG;
  switch(id) {
  case 199 :
	qApp->quit();
	return; // never reached, I know...
  case 200 :
	Page::setMouseMode(Page::MoveMode);
	break;
  case 201 :
	Page::setMouseMode(Page::TextMode);
	break;
  case 202 :
	Page::setMouseMode(Page::PlotMode);
	break;
  }
}


void KCentre::oblivion ()
{
  // the functions name suggests where the signal goes to...
}


void KCentre::unimplemented ()
{
  QMessageBox::information(0,"Unimplemented!",
						   "Whatever action you just wanted to be done,\n"\
						   "it is not yet implemented.\n\n"\
						   "We, the developers, apologize.");
}

