#include "qwt_sclif.h"

//------------------------------------------------------------
//.H QwtScaleIf|3|27/04/97|Qwt Widget Library|Qwt Programmer's manual
//
//.S	NAME
//	QwtScaleIf - An interface class for widgets containing a scale 
//
//.S	SYNOPSIS
//	#include <qwt_sclif.h>
//
//.S	PUBLIC MEMBERS 
//.t
//	void \R{QwtScaleIf::setScale }(double vmin, double vmax, int logarithmic = 0) --
//				Set user-defined scale
//	void \R{QwtScaleIf::setScale }(double vmin, double vmax, double step, int logarithmic = 0) --
//				Set user-defined scale
//	void \R{QwtScaleIf::setScale}(const QwtScaleDiv &s)	--
//				Set user-defined scale
//	void \R{QwtScaleIf::setScaleMaxMajor}(int ticks) --
//				Set Max. number of major tick intervals
//	void \R{QwtScaleIf::setScaleMaxMinor}(int ticks) --
//				Set max. number of minor rick intervals
//	void \R{QwtScaleIf::autoScale}() -- adjust scale automatically
//	int scaleMaxMinor() const -- return max. number of minor tick intervals
//      int scaleMaxMajor() const -- return max. number of major tick intervals
//
//.S	PROTECTED MEMBERS
//.t
//	 \R{QwtScaleIf::hasUserScale}() -- Check for user-defined scale
//	 \R{QwtScaleIf::scaleChange}() -- Notify change of scale
//
//.S	DESCRIPTION
//	This interface class is used to provide classes
//	with a protected QwtScaleDraw member and a public 
//	interface to access that scale.
//
//.S	INTERNALS
//	The primary purpose of this class is to define
//	a common interface for classes which are supposed to
//	contain a ScaleDraw class. It provides a protected 
//	ScaleDraw member
//      called d_scale and a couple of public member functions 
//      which allow direct but restricted access 
//	to this scale by the user. 
//	Widgets derived from this class have
//	to implement the member function scaleChange(),
//	which is called to notify changes of the
//	scale parameters and usually requires repainting or
//	resizing respectively.
//	In general, a class derived from QwtScaleIf is 
//      expected to manage the division and the position of its scale internally
//	when no user-defined scale is set. It should take the d_maxMinor
//	and d_maxMajor members into account, which can be set by the user.
//	An implementation can check if a user-definde scale is set by calling the
//	\R{QwtScaleIf::hasUserScale}() member.
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.S	QwtScaleIf::QwtScaleIf
//		Construct a QwtScaleIf instance
//
//.s	Syntax
//.f	 QwtScaleIf::QwtScaleIf()
//
//------------------------------------------------------------
QwtScaleIf::QwtScaleIf()
{
    d_userScale = FALSE;
    d_maxMajor = 5;
    d_maxMinor = 3;
    d_scale.setScale(0.0,100.0,d_maxMajor, d_maxMinor);
}

//------------------------------------------------------------
//
//.S	QwtScaleIf::setScale
//		Specify a user-defined scale.
//
//.s	Parameters
//.p	double vmin, double vmax, int logarithmic
//
//.s	Return Type
//		void
//
//.s	Description
//		By default, the widget is supposed to control the range of its scale
//		automatically,	but sometimes it is desirable to have a user-defined
//		scale which is not in sync with
//		the widget's range, e.g. if a logatithmic scale is needed
//		(sliders don't support that) or if the scale is required
//		to have a fixed range (say 0...100%), independent of the
//		widget's range. 
//
//.s	See also
//	\R{QwtScaleIf::autoScale}()
//
//.s	Syntax
//.f	void QwtScaleIf::setScale(double vmin, double vmax, int logarithmic)
//
//------------------------------------------------------------
void QwtScaleIf::setScale(double vmin, double vmax, int logarithmic)
{
    setScale(vmin,vmax,0.0,logarithmic);
}


//------------------------------------------------------------
//
//.S	QwtScaleIf::setScale
//		Specify a user-defined scale.
//
//.s	Parameters
//.p	double vmin, double vmax, int logarithmic
//
//.s	Return Type
//		void
//
//.s	Description
//		By default, the widget is supposed to control the range of its scale
//		automatically,	but sometimes it is desirable to have a user-defined
//		scale which is not in sync with
//		the widget's range, e.g. if a logatithmic scale is needed
//		(sliders don't support that) or if the scale is required
//		to have a fixed range (say 0...100%), independent of the
//		widget's range. 
//
//.s	See also
//	\R{QwtScaleIf::autoScale}()
//
//.s	Syntax
//.f	void QwtScaleIf::setScale(double vmin, double vmax, int logarithmic)
//
//------------------------------------------------------------
void QwtScaleIf::setScale(double vmin, double vmax, double step, int logarithmic)
{
    QwtScaleDiv oldscl(d_scale.scaleDiv());
    
    d_scale.setScale(vmin, vmax, d_maxMajor, d_maxMinor, step,logarithmic);
    d_userScale = TRUE;
    if (oldscl != d_scale.scaleDiv())
       scaleChange();
}






//------------------------------------------------------------
//
//.S	QwtScaleIf::autoScale
//		Advise the widget to control the scale range
//		internally.
//
//.s	Description
//		Autoscaling is on by default. 
//		
//.s	See also
//		\R{QwtScaleIf::setScale()}
//
//.s	Syntax
//.f	void QwtScaleIf::autoScale()
//
//------------------------------------------------------------
void QwtScaleIf::autoScale()
{
	if (!d_userScale) 
	{
		d_userScale = FALSE;
		scaleChange();
	}
}

//------------------------------------------------------------
//
//.S	QwtScaleIf::setScaleMaxMajor
//		Set the maximum number of major tick intervals.
//
//.s	Parameters
//.p	int ticks		--		maximal number of major ticks.
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//		The scale's major ticks are calculated automatically such that
//		the number of major intervals does not exceed <ticks>.
//		The default value is 5.
//.s	See Also
//		ScaleDraw class
//
//.s	Syntax
//.f	void QwtScaleIf::setScaleMaxMajor(int ticks)
//
//------------------------------------------------------------
void QwtScaleIf::setScaleMaxMajor(int ticks)
{
   if (ticks != d_maxMajor)
   {
       d_maxMajor = ticks;
       d_scale.setScale(d_scale.scaleDiv().lBound(), d_scale.scaleDiv().hBound(),
			d_maxMajor, d_maxMinor, 0.0,d_scale.scaleDiv().logScale()); 
       scaleChange();
   }
}

//------------------------------------------------------------
//
//.S	QwtScaleIf::setScaleMaxMinor
//	Set the maximum number of minor tick intervals
//
//.s	Parameters
//.p	int ticks
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//		The scale's minor ticks are calculated automatically such that
//		the number of minor intervals does not exceed <ticks>.
//		The default value is 3.
//
//.s	Syntax
//.f	void QwtScaleIf::setScaleMaxMinor(int ticks)
//
//------------------------------------------------------------
void QwtScaleIf::setScaleMaxMinor(int ticks)
{
    if ( ticks != d_maxMinor)
    {
	d_maxMinor = ticks;
	d_scale.setScale(d_scale.scaleDiv().lBound(), d_scale.scaleDiv().hBound(),
			 d_maxMajor, d_maxMinor, 0.0, d_scale.scaleDiv().logScale()); 
	scaleChange();
    }
}






