#include "qwt_knob.h"

#include <qpainter.h>
#include <qpalette.h>
#include "qwt_math.h"

#include "qwt_knob.moc"

//------------------------------------------------------------
//.H QwtKnob|3|27/04/97|Qwt Widget Library|Qwt Programmer's manual
//.I knob.gif
//.S NAME
//	QwtKnob - The Knob Widget
//
//.S SYNOPSIS 
//	#include <qwt_knob.h>
//
//.S INHERITED CLASSES
//	\l{qwt_sldbase.html}{QwtSliderBase}, \l{qwt_sclif.html}{QwtScaleIf}
//
//.S PUBLIC MEMBERS
//.t
// \R{QwtKnob::QwtKnob}(QWidget* parent = 0, const char *name = 0) -- Constructor
// \R{QwtKnob::~QwtKnob}() -- Destructor
// \R{QwtKnob::setKnobWidth}(int w) -- change the width of the knob
// \R{QwtKnob::setTotalAngle}(double angle) -- change the total angle by which
//					       the knob can be turned
// \R{QwtKnob::setBorderWidth}(int bw)	-- change the border width of the knob
//
//.S PROTECTED MEMBERS 
//
//.S DESCRIPTION
//
//------------------------------------------------------------

//------------------------------------------------------------
//.C	MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------

//------------------------------------------------------------
//
//.S	QwtKnob::QwtKnob
//		Constructor
//
//.s	Parameters
//.p	QWidget *parent	-- parent widget
//	const char *name -- name 
//
//.s	Syntax
//.f	QwtKnob::QwtKnob(QWidget *parent, const char *name )
//
//------------------------------------------------------------
QwtKnob::QwtKnob(QWidget* parent, const char *name)
: QwtSliderBase(parent,name)
{
    d_borderWidth = 2;
    d_borderDist = 4;
    d_totalAngle = 270.0;
    d_scaleDist = 4;
    d_hasScale = 0;
    d_symbol = Line;
    d_maxScaleTicks = 11;
    d_knobWidth = 50;
    d_faceColor = backgroundColor();
    d_markerColor = foregroundColor();
    d_dotWidth = 8;
    
    setMinimumSize(30,30);
    setUpdateTime(50);
   d_scale.setFont(QFont("Helvetica",8));
}

//------------------------------------------------------------
//
//.S	QwtKnob::~QwtKnob
//	Destructor
//.s	Syntax
//.f	 QwtKnob::~QwtKnob()
//
//------------------------------------------------------------
QwtKnob::~QwtKnob() 
{
}

//------------------------------------------------------------
//
//.S	QwtKnob::setTotalAngle
//	Set the total angle by which the knob can be turned
//
//.s	Parameters
//.p	double angle	--	angle in degrees.
//
//.s	Description
//	The default angle is 270 degrees. It is possible to specify
//	an angle of more than 360 degrees so that the knob can be
//	turned several times around its axis.   
//
//.s	Syntax
//.f	void QwtKnob::setTotalAngle(double angle)
//
//------------------------------------------------------------
void QwtKnob::setTotalAngle (double angle) 
{
    if (angle < 10.0)
       d_totalAngle = 10.0;
    else
       d_totalAngle = angle;
    d_scale.setAngleRange( -0.5 * d_totalAngle, 0.5 * d_totalAngle);
}


//------------------------------------------------------------
//.-
//.S    QwtKnob::drawKnob
//	Draw the knob
//.s	Syntax
//.f	void QwtKnob::drawKnob(QPainter &p, QRect &r)
//
//.s    Parameters
//.p		QPainter &p	--		painter
//   	const QRect &r	 --   borders of the knob
//
//------------------------------------------------------------
void QwtKnob::drawKnob(QPainter *p, const QRect &r)
{

    QRect aRect;

    QColorGroup g = colorGroup();
    QPen pn;
    int bw2 = d_borderWidth / 2;
    

    aRect.setRect( r.x() + bw2,
		  r.y() + bw2,
		  r.width() - 2*bw2,
		  r.height() - 2*bw2);
    
    //
    // draw button face
    //
    p->setPen(NoPen);
    p->setBrush(d_faceColor);
    p->drawEllipse(aRect);

    //
    // draw button shades
    //
    pn.setWidth(d_borderWidth);

    
    pn.setColor(g.light());
    p->setPen(pn);
    p->drawArc(aRect, 45*16,180*16);

    pn.setColor(g.dark());
    p->setPen(pn);
    p->drawArc(aRect, 225*16,180*16);

    //
    // draw marker
    //
    drawMarker(p, d_angle, d_markerColor);
    
}


//------------------------------------------------------------
//.-
//.S	QwtSliderBase::valueChange
//	Notify change of value
//
//.s	Parameters
//	double x  --    new value
//
//.s	Description
//	Sets the slider's value to the nearest multiple
//          of the step size.
//    
//------------------------------------------------------------
void QwtKnob::valueChange() 
{
    recalcAngle();
    d_newVal++;
    repaint(d_kRect, FALSE);
    QwtSliderBase::valueChange();
}

//------------------------------------------------------------
//.-
//.S	QwtKnob::getValue
//	Determine the value corresponding to a specified position
//
//.s	Parameters:
//	const QPoint &p -- point 
//
//.s	Return Type
//	double
//
//.s	Description:
//	Called by QwtSliderBase
//------------------------------------------------------------
double QwtKnob::getValue(const QPoint &p)
{

    double newValue;
    double oneTurn;
    double eqValue;
    double arc;
    
    const QRect& r = rect();

    double dx = double((r.x() + r.width() / 2) - p.x() );
    double dy = double((r.y() + r.height() / 2) - p.y() );

    arc = atan2(-dx,dy) * 180.0 / M_PI;

    newValue =  0.5 * (minValue() + maxValue())
       + (arc + d_nTurns * 360.0) * (maxValue() - minValue())
	  / d_totalAngle;
    
    oneTurn = fabs(maxValue() - minValue()) * 360.0 / d_totalAngle;
    eqValue = value() + d_mouseOffset;
    
    if (fabs(newValue - eqValue) > 0.5 * oneTurn) 
    {
	if (newValue < eqValue) 
	   newValue += oneTurn;
	else
	   newValue -= oneTurn;
    }

    return newValue;	

}



//------------------------------------------------------------
//.-
//.S	QwtKnob::setScrollMode
//	Determine the scrolling mode and direction
//	corresponding to a specified position
//
//.s	Parameters
//	const QPoint &p -- point in question
//
//.s	Description
//	Called by QwtSliderBase
//------------------------------------------------------------
void QwtKnob::getScrollMode(const QPoint &p, int &scrollMode, int &direction)
{
    int dx, dy, r;
    double arc;

    r = d_kRect.width() / 2;

    dx = d_kRect.x() + r - p.x();
    dy = d_kRect.y() + r - p.y();
    
    if ( (dx * dx) + (dy * dy) <= (r * r)) // point is inside the knob
    {
	scrollMode = ScrMouse;
	direction = 0;
    }
    else								// point lies outside
    {
	scrollMode = ScrTimer;
	arc = atan2(double(-dx),double(dy)) * 180.0 / M_PI;
	if ( arc < d_angle)
	   direction = -1;
	else if (arc > d_angle)
	   direction = 1;
	else
	   direction = 0;
    }
    return;
}



//------------------------------------------------------------
//.-
//.S  QwtKnob::rangeChange
//	Notify a change of the range
//
//.s  Description
//	Called by QwtSliderBase
//------------------------------------------------------------
void QwtKnob::rangeChange()
{
    if (!hasUserScale())
    {
	d_scale.setScale(minValue(), maxValue(),
			 d_maxMajor, d_maxMinor);
    }
    recalcAngle();
    resize(size());
    repaint(FALSE);
}

//------------------------------------------------------------
//.-
//.s	QwtKnob::resizeEvent
//		Qt Resize Event
//
//.s	Parameters
//.p	QResizeEvent *e
//
//.s	Return Type
//		void
//
//.s	Description
//
//.s	Syntax
//.f	void QwtKnob::resizeEvent(QResizeEvent *e)
//
//------------------------------------------------------------
void QwtKnob::resizeEvent(QResizeEvent *e)
{
    int width, width_2;

    const QRect &r = rect();

    width = qwtMin((qwtMin(r.height(), r.width()) - 1) / 2, d_knobWidth);
    width_2 = width / 2;
    
    d_kRect.setRect(r.x() + r.width() / 2 - width_2,
		    r.y() + r.height() / 2 - width_2,
		    width, width);

    d_scale.setGeometry( d_kRect.x() +  width_2,
			d_kRect.y() + width_2,
			width_2 + d_scaleDist, QwtScaleDraw::Round );
}

//------------------------------------------------------------
//.-
//.s	QwtKnob::paintEvent
//	Repaint the knob
//
//.s	Parameters
//.p	QPaintEvent *e	   -- paint event
//
//.s	Syntax
//.f	void QwtKnob::paintEvent(QPaintEvent *e)
//
//------------------------------------------------------------
void QwtKnob::paintEvent(QPaintEvent *e)
{
    QPainter p;
    const QRect &r = e->rect();

    if( p.begin(this))
    {
	
	if ( (r == d_kRect) && d_newVal ) // event from valueChange() 
	{
	    if (d_newVal > 1)		// lost paintEvents()?
	    {
		drawKnob(&p, d_kRect);
	    }
	    else
	    {
		drawMarker(&p, d_oldAngle, d_faceColor);
		drawMarker(&p, d_angle, d_markerColor);
	    }
	    
	}
	else
	{
	    p.fillRect(rect(), backgroundColor());
	    d_scale.draw(&p);
	    drawKnob(&p, d_kRect);
	}
	
    }
    
    p.end();
    
    d_newVal = 0;
    
}

//------------------------------------------------------------
//.-
//.s	QwtKnob::drawMarker
//	Draw the marker at the knob's front
//
//.s	Parameters
//.p	QPainter *p	--	painter
//	double arc	--	angle of the marker
//	const QColor &c	 --	marker color
//
//.s	Syntax
//				void QwtKnob::drawMarker(QPainter *p)
//
//------------------------------------------------------------
void QwtKnob::drawMarker(QPainter *p, double arc, const QColor &c)
{
    
    QPen pn;
    int radius;
    double rb,re;
    double rarc;
    
    rarc = arc * M_PI / 180.0;
    double ca = cos(rarc);
    double sa = - sin(rarc);
    radius = d_kRect.width() / 2 - d_borderWidth;
    if (radius < 3) radius = 3;	
    int ym = d_kRect.y() + radius + d_borderWidth;
    int xm = d_kRect.x() + radius + d_borderWidth;
    
    switch (d_symbol)
    {
    case Dot:
	
	p->setBrush(c);
	p->setPen(NoPen);
	rb = double(qwtMax(radius - 4 - d_dotWidth / 2, 0));
	p->drawEllipse(xm - int(rint(sa * rb)) - d_dotWidth / 2,
		       ym - int(rint(ca * rb)) - d_dotWidth / 2,
		       d_dotWidth, d_dotWidth);
	
	break;
	
    case Line:
	
	pn.setColor(c);
	pn.setWidth(2);
	p->setPen(pn);
	
	rb = qwtMax(double((radius - 4) / 3.0), 0.0);
	re = qwtMax(double(radius - 4), 0.0);
	
	p->drawLine( xm - int(rint(sa * rb)),
		    ym - int(rint(ca * rb)),
		    xm - int(rint(sa * re)),
		    ym - int(rint(ca * re)));
	
	break;
    }
    
    
}

//------------------------------------------------------------
//
//.S	QwtKnob::setKnobWidth
//		Change the knob's width.
//
//.s	Parameters
//.p	int w     --	new width
//
//.s	Return Type
//		void
//
//.s	Description
//		The specified width must be >= 5, or it will be clipped.
//
//.s	Syntax
//.f	void QwtKnob::setKnobWidth(int w)
//
//------------------------------------------------------------
void QwtKnob::setKnobWidth(int w)
{
    d_knobWidth = qwtMax(w,5);
    resize(size());
    repaint(FALSE);
}

//------------------------------------------------------------
//
//.S	QwtKnob::setBorderWidth
//		Set the Knob's border width
//
//.s	Parameters
//.p	int bw
//
//.s	Syntax
//.f	void QwtKnob::setBorderWidth(int bw)
//
//------------------------------------------------------------
void QwtKnob::setBorderWidth(int bw)
{
    d_borderWidth = qwtMax(bw, 0);
    resize(size());
    repaint(FALSE);
}

//------------------------------------------------------------
//.-
//.S	QwtKnob::recalcAngle
//		Recalculate the marker angle corresponding to the
//		current value
//
//.s	Return Type
//		void
//
//.s	Syntax
//.f	void QwtKnob::recalcAngle()
//
//------------------------------------------------------------
void QwtKnob::recalcAngle()
{
    d_oldAngle = d_angle;
    
    //
    // calculate the angle corresponding to the value
    //
    if (maxValue() == minValue())
    {
	d_angle = 0;
	d_nTurns = 0;
    }
    else 
    {
	d_angle = (value() - 0.5 * (minValue() + maxValue()))
	   / (maxValue() - minValue()) * d_totalAngle;
	d_nTurns = floor((d_angle + 180.0) / 360.0);
	d_angle = d_angle - d_nTurns * 360.0;
	
    }
    
}


void QwtKnob::scaleChange()
{
    repaint(FALSE);
}

void QwtKnob::fontChange()
{
    repaint(FALSE);
}









