// KPPPLoad - a PPP load monitor
// Copyright (C) 1998  Sean Vyain, svyain@mail.tds.net
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <time.h>

#include <qmsgbox.h>

#include <kapp.h>
#include <kpopmenu.h>
#include <kstatusbar.h>

#include "Graph.h"
#include "KPPPLoad.h"
#include "Options.h"
#include "OptionsDlg.h"
#include "PPPStats.h"

#include "KPPPLoad.moc"

KPPPLoad::KPPPLoad()
        : _pppstats( NULL ),
          _graph( NULL )
{
    setCaption( "KPPPLoad" );
    setIconText( "KPPPLoad" );
    
    _popup = new KPopupMenu( "KPPPLoad" );
    _popup->insertItem( i18n( "Options..." ), this, SLOT( doOptions() ) );
    _popup->insertItem( i18n( "Help" ), KApplication::getKApplication()->getHelpMenu( TRUE, i18n( "KPPPLoad is a PPP load monitor.\n\nCopyright (c) 1997 Sean Vyain\nsvyain@mail.tds.net" ) ) );
    _popup->insertSeparator();
    _popup->insertItem( i18n( "Exit" ), this, SLOT( doExit() ) );

    configPPPStats();
    connect( options, SIGNAL( sigStatsInterval()    ), this, SLOT( configPPPStats() ) );
    connect( options, SIGNAL( sigPeriodicInterval() ), this, SLOT( configPPPStats() ) );
    connect( options, SIGNAL( sigPPPCommand()       ), this, SLOT( configPPPStats() ) );
    connect( options, SIGNAL( sigPeriodicCommand()  ), this, SLOT( configPPPStats() ) );
    connect( options, SIGNAL( sigLinkNumber()       ), this, SLOT( configPPPStats() ) );
    connect( options, SIGNAL( sigRetryInterval()    ), this, SLOT( configPPPStats() ) );
    
    configGraph();
    connect( options, SIGNAL( sigStatsInterval() ), this, SLOT( configGraph() ) );
    connect( options, SIGNAL( sigStatsPeriod()   ), this, SLOT( configGraph() ) );

    configStatusBar();
    connect( options, SIGNAL( sigShowTxRate()     ), this, SLOT( configStatusBar() ) );
    connect( options, SIGNAL( sigShowRxRate()     ), this, SLOT( configStatusBar() ) );
    connect( options, SIGNAL( sigShowTxTotal()    ), this, SLOT( configStatusBar() ) );
    connect( options, SIGNAL( sigShowRxTotal()    ), this, SLOT( configStatusBar() ) );

    resize( 400, 100 );
}

KPPPLoad::~KPPPLoad()
{
    delete _pppstats;
}

QString KPPPLoad::toString( uint bytes )
{
    QString str;

    if ( bytes < 1024 ) {
        str.sprintf( "%dB", bytes );
    } else if ( bytes < 1024*1024 ) {
        str.sprintf( "%.1fK", (float)bytes / 1024.0 );
    } else if ( bytes < 1024*1024*1024 ) {
        str.sprintf( "%.1fM", (float)bytes / 1024.0 / 1024.0 );
    } else {
        str.sprintf( "%.1fG", (float)bytes / 1024.0 / 1024.0 / 1024.0 );
    }

    return str;
}

void KPPPLoad::updateStats( uint rxDelta, uint txDelta, uint rxTotal, uint txTotal )
{
    if ( statusBar() ) {
        QString msg;
        
        if ( options->getShowTxRate() ) {
            msg.sprintf( i18n( "TX: %s/s" ), toString( txDelta / options->getStatsInterval() ).data() );
            statusBar()->changeItem( msg, 0 );
        }
        if ( options->getShowRxRate() ) {
            msg.sprintf( i18n( "RX: %s/s" ), toString( rxDelta / options->getStatsInterval() ).data() );
            statusBar()->changeItem( msg, 1 );
        }
        if ( options->getShowTxTotal() ) {
            msg.sprintf( i18n( "TX: %s" ), toString( txTotal ).data() );
            statusBar()->changeItem( msg, 2 );
        }
        if ( options->getShowRxTotal() ) {
            msg.sprintf( i18n( "RX: %s" ), toString( rxTotal ).data() );
            statusBar()->changeItem( msg, 3 );
        }
    }
}

void KPPPLoad::linkDown()
{
    killTimers();
    setCaption( i18n( "DOWN" ) );
    setIconText( i18n( "DOWN" ) );
}

void KPPPLoad::linkUp()
{
    _connectTime = time( NULL );
    startTimer( 30 * 1000 );
    timerEvent( 0 );
}

void KPPPLoad::retryCount( int retries )
{
    QString tmp;
    tmp.sprintf( i18n( "DOWN (%d)" ), retries );
    setCaption( tmp );
    setIconText( tmp );
}

void KPPPLoad::timerEvent( QTimerEvent* )
{
    QString str;
    
    int elapsed = time( NULL ) - _connectTime;
    str.sprintf( i18n( "%02d:%02d" ), elapsed / 3600, elapsed / 60 % 60 );

    setCaption( str.data() );
    setIconText( str.data() );
}

void KPPPLoad::configStatusBar()
{
    KStatusBar* status = statusBar();

    // Remove the old status bar.
    if ( status ) {
        setStatusBar( NULL );
        delete status;
    }

    if ( ( options->getShowTxRate()     ) ||
         ( options->getShowRxRate()     ) ||
         ( options->getShowTxTotal()    ) ||
         ( options->getShowRxTotal()    ) ) {
        status = new KStatusBar( this );
        if ( options->getShowTxRate() ) {
            status->insertItem( i18n( "TX:XX.XK/s" ), 0 );
        }
        if ( options->getShowRxRate() ) {
            status->insertItem( i18n( "RX:XX.XK/s" ), 1 );
        }
        if ( options->getShowTxTotal() ) {
            status->insertItem( i18n( "TX:XXX.XK" ), 2 );
        }
        if ( options->getShowRxTotal() ) {
            status->insertItem( i18n( "RX:XXX.XK" ), 3 );
        }
        status->insertItem( "", 4 );
        
        setStatusBar( status );
        status->show();
        updateRects();
    }
}

void KPPPLoad::configGraph()
{
    setView( NULL );
    delete _graph;

    _graph = new Graph( this );
    setView( _graph );
    _graph->show();
    updateRects();
    
    connect( _graph, SIGNAL( popupMenu( const QPoint& ) ), this, SLOT( popupMenu( const QPoint& ) ) );
    connect( _pppstats, SIGNAL( changeStats( uint, uint, uint, uint ) ), _graph, SLOT( updateStats( uint, uint, uint, uint ) ) );
    connect( _pppstats, SIGNAL( linkDown() ), _graph, SLOT( linkDown() ) );
}

void KPPPLoad::configPPPStats()
{
    delete _pppstats;

    _pppstats = new PPPStats();
    if ( _pppstats->isUp() ) {
        linkUp();
    } else {
        linkDown();
    }
    connect( _pppstats, SIGNAL( changeStats( uint, uint, uint, uint ) ), this, SLOT( updateStats( uint, uint, uint, uint ) ) );
    connect( _pppstats, SIGNAL( linkDown() ), this, SLOT( linkDown() ) );
    connect( _pppstats, SIGNAL( linkUp() ), this, SLOT( linkUp() ) );
    connect( _pppstats, SIGNAL( retryCount( int ) ), this, SLOT( retryCount( int ) ) );

    if ( _graph ) {
        connect( _pppstats, SIGNAL( changeStats( uint, uint, uint, uint ) ), _graph, SLOT( updateStats( uint, uint, uint, uint ) ) );
        connect( _pppstats, SIGNAL( linkDown() ), _graph, SLOT( linkDown() ) );
    }
}

void KPPPLoad::popupMenu( const QPoint& pos )
{
    _popup->popup( pos, KPM_FirstItem );
}

void KPPPLoad::doOptions()
{
    OptionsDlg dlg( this );
    dlg.show();
}

void KPPPLoad::doExit()
{
    KApplication::getKApplication()->quit();
}
