#ifndef __k2url_h__
#define __k2url_h__

#include <string>
#include <list>

/**
 * Mention that K2URL has some restrictions regarding the path 
 * encoding. K2URL works intern with the decoded path and
 * and encoded query. For example in
 * <pre>
 * http://localhost/cgi-bin/test%20me.pl?cmd=Hello%20you
 * </pre>
 * would result in a decoded path "/cgi-bin/test me.pl"
 * and in the encoded query "cmd=Hello%20you".
 * Since path is internally always encoded you may NOT use
 * "%00" in tha path while this is ok for the query.
 */
class K2URL
{
  typedef list<K2URL> List;
public:
  /**
   * @param _url is considered to be encoded. You can pass strings like
   *             "/home/weis", in this case the protocol "file" is assumed.
   *             This is dangerous since even this simple path is assumed to be
   *             encoded. For example "/home/Torben%20Weis" will be decoded to
   *             "/home/Torben Weis". This means: If you have a usual UNIX like
   *             path, you have to use @ref encode first before you pass it to K2URL.
   */
  K2URL();
  K2URL( const char *_url );
  K2URL( string& _url );
  K2URL( const K2URL& _u );
  /**
   * @param _rel_url is considered to be encoded.
   */
  K2URL( const K2URL& _u, const char *_rel_url );
  
  const char* protocol() { return m_strProtocol.c_str(); }
  void setProtocol( const char *_txt ) { m_strProtocol = _txt; }

  const char* user() { return m_strUser.c_str(); }
  void setUser( const char *_txt ) { m_strUser = _txt; }
  bool hasUser() { return !m_strUser.empty(); }

  const char* pass() { return m_strPass.c_str(); }
  void setPass( const char *_txt ) { m_strPass = _txt; }
  bool hasPass() { return !m_strPass.empty(); }

  const char* host() { return m_strHost.c_str(); }
  void setHost( const char *_txt ) { m_strHost = _txt; }
  bool hasHost() { return !m_strHost.empty(); }

  int port() { return m_iPort; }
  void setPort( int _p ) { m_iPort = _p; }
  
  /**
   * @return the current decoded path. This does NOT include the query.
   *
   */
  const char* path() const  { return m_strPath.c_str(); }
  /**
   * @param _trailing may be ( -1, 0 +1 ). -1 strips a trailing '/', +1 adds
   *                  a trailing '/' if there is none yet and 0 returns the
   *                  path unchanged. If the URL has no path, then no '/' is added
   *                  anyways. And on the other side: If the path is "/", then this
   *                  character wont be stripped. Reason: "ftp://weis@host" means something
   *                  completly different than "ftp://weis@host/". So adding or stripping
   *                  the '/' would really alter the URL, while "ftp://host/path" and
   *                  "ftp://host/path/" mean the same directory.
   *
   * @return the current decoded path. This does NOT include the query.
   */
  string path( int _trailing ) const;
  /**
   * _txt is considered to be decoded. This means: %3f does not become decoded
   *      and the ? does not indicate the start of the query part.
   *      The query is not changed by this function.
   */
  void setPath( const char *_txt ) { m_strPath = _txt; }  
  bool hasPath() { return !m_strPath.empty(); }

  /**
   * This is useful for HTTP. It looks first for '?' and decodes then.
   * The encoded path is the concatenation of the current path and the query.
   */
  void setEncodedPathAndQuery( const char *_txt );
  /**
   * @return the concatenation if the encoded path , '?' and the encoded query.
   *
   * @param _no_empty_path If set to true then an empty path is substituted by "/".
   */
  string encodedPathAndQuery( int _trailing = 0, bool _no_empty_path = false );

  /**
   * @param _txt is considered to be encoded. This has a good reason: The query may contain the 0 character.
   */
  void setQuery( const char *_txt ) { m_strQuery_encoded = _txt; }
  /**
   * @return the encoded query. This has a good reason: The query may contain the 0 character.
   */
  const char* query() { return m_strQuery_encoded.c_str(); }
  
  /**
   * The reference is NEVER decoded automatically.
   */
  const char* ref() { return m_strRef_encoded.c_str(); }
  /**
   * @param _txt is considered encoded.
   */
  void setRef( const char *_txt ) { m_strRef_encoded = _txt; }
  bool hasRef() { return !m_strRef_encoded.empty(); }
  
  bool isMalformed() const  { return m_bIsMalformed; }

  bool isLocalFile();
  bool hasSubURL();

  /**
   * Assumes that the current path is a directory. '_txt' is appended to the
   * current path. The function adds '/' if needed while concatenating.
   * This means it does not matter wether the current path has a trailing
   * '/' or not. If there is none, it becomes appended. If '_txt'
   * has a leading '/' then this one is stripped.
   *
   * @param _txt is considered to be decoded
   */
  void addPath( const char *_txt );
  /**
   * In comparison to @ref addPath this function does not assume that the current path
   * is a directory. This is only assumed if the current path ends with '/'.
   *
   * @param _txt is considered to be decoded. If the current path ends with '/'
   *             then '_txt' ist just appended, otherwise all text behind the last '/'
   *             in the current path is erased and '_txt' is appended then. It does
   *             not matter wether '_txt' starts with '/' or not.
   */
  void setFileName( const char *_txt );

  /**
   * @return the filename of the current path. The returned string is decoded.
   *
   * @ref _ignore_trailing_slash_in_path tells wether a trailing '/' should be ignored.
   *                                     This means that the function would return "torben" for
   *                                     <tt>file:/hallo/torben/</tt> and <tt>file:/hallo/torben</tt>.
   *                                     If the flag is set to false, then everything behind the last '/'
   *                                     is considered to be the filename.
   */
  string filename( bool _ignore_trailing_slash_in_path = true );
  /**
   * @return the directory part of the current path. Everything between the last and the second last '/'
   *         is returned. For example <tt>file:/hallo/torben/</tt> would return "/hallo/torben/" while
   *         <tt>file:/hallo/torben</tt> would return "hallo/". The returned string is decoded.
   *
   * @param _strip_trailing_slash_from_result tells wether the returned result should end with '/' or not.
   *                                          If the path is empty or just "/" then this flag has no effect.
   * @param _ignore_trailing_slash_in_path means that <tt>file:/hallo/torben</tt> and 
   *                                       <tt>file:/hallo/torben/"</tt> would both return <tt>/hallo/torben/</tt>
   *                                       or <tt>/hallo/torben</tt> depending on the other flag
   */
  string directory( bool _strip_trailing_slash_from_result = true, bool _ignore_trailing_slash_in_path = true );
  
  /**
   * @return the complete encoded URL.
   */
  string url();
  /**
   * @return the complete encoded URL.
   *
   * @param _trailing may be ( -1, 0 +1 ). -1 strips a trailing '/' from the path, +1 adds
   *                  a trailing '/' if there is none yet and 0 returns the
   *                  path unchanged.
   */
  string url( int _trailing );

  K2URL& K2URL::operator=( const K2URL& _u );
  K2URL& K2URL::operator=( const char* _url );
  K2URL& K2URL::operator=( string& _url );

  bool K2URL::operator==( const K2URL& _u ) const;
  bool K2URL::operator==( const char* _u ) const;
  /**
   * This function should be used if you want to ignore trailing '/' characters.
   *
   * @see path
   */
  bool cmp( K2URL &_u, bool _ignore_trailing = false );
  
  /**
   * Splits nested URLs like file:/home/weis/kde.gz#gzip:/decompress#tar:/kdebase.
   *
   * @return false on parse error and true otherwise.
   *
   * @param _list holds all URLs after the call. The URLs are just appended, so you have
   *              to clean the list usually before calling this function.
   */
  static bool split( const char *_url, List& _list );
  /**
   * Reverses @ref #split.
   */
  static void join( List& _list, string& _dest );
  
  /**
   * Decode the string, this means decoding "%20" into a space for example. Note that "%00" is
   * not handled correctly here.
   */
  static void decode( string& _url );
  /**
   * Reverse of @ref decode
   */
  static void encode( string& _url );
  
protected:
  void reset();
  void parse( const char* _url );
  
  static char hex2int( char _char );

  string m_strProtocol;
  string m_strUser;
  string m_strPass;
  string m_strHost;
  string m_strPath;
  string m_strRef_encoded;
  string m_strQuery_encoded;
  
  bool m_bIsMalformed;
  int m_iPort;
};

typedef K2URL::List K2URLList;

bool urlcmp( K2URLList& _url1, K2URLList& _url2 );
bool urlcmp( const char *_url1, const char *_url2 );
bool urlcmp( const char *_url1, const char *_url2, bool _ignore_trailing, bool _ignore_ref );

#endif
