	/*

	Copyright (C) 1998-1999 Stefan Westerfeld
                            stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include <sys/types.h>
#include <sys/time.h>
#include <math.h>
#include <stdio.h>
#include <ctype.h>
#include <strings.h>
#include <unistd.h>
#include <stdlib.h>
#include <malloc.h>
#include <time.h>
#include <dirent.h>
#include "synth_impl.h"
#include "resources.h"
#include "midi.h"
#include "midibus.h"
#include "structures.h"
#include "ioevent.h"
#include "artsorb.h"
#include "debug.h"
#include "execrequest.h"
#include "receiver_impl.h"
#include "bus.h"
#include "sequenceutils.h"
 
extern "C" {
#include "sound.h"
#include "utils.h"
}

#define Die(a) { fprintf(stderr,"ERROR: %s\n",a); exit(1); }

//****************** Synthesizer *******************

Synthesizer_impl::Synthesizer_impl()
{
	Arts::RealtimeConfig InitRTConfig;

	describe("Synthesizer");

	InitRTConfig.Cycles = 128;		// calculate 128 cycles at once by default
	InitRTConfig.CacheSizeKB = 8192;	// 8 megs of cache are reasonable
	InitRTConfig.RingBufferSize = 64;	// ringbuffer size (2^n)
	InitRTConfig.FragmentCount = 7;		// fragment count
	InitRTConfig.FragmentSize = 256;	// fragment size (2^n)
	InitRTConfig.Debug = false;		// no debugging by default

	RTConfig(InitRTConfig);

	artsdebug("... Synthesizer_impl created\n");
	SynthModules = NULL;
	_Cache = new Cache;
	MCount = 0;
	_StructChangeCount = 0;
	MS_SynthModule = (ModuleServer<SynthModule> *)SynthModule::get_MS();
	nextID = 1;
	nextMID = 1;
	nextRestoreID = 1;

	inScheduler = false;
	ioManager.start();
	schedulerLocks = 0;

	_MidiChannel = new MidiChannel_impl();
	_ModuleBroker = new ModuleBroker_impl(this);
	_AudioManager = new AudioManager_impl(this);

	_ModuleBroker->incRef();
	_AudioManager->incRef();

	initModules();

	_StructureExpander = new StructureExpander(_ModuleBroker,this);
}

void Synthesizer_impl::cleanUp()
{
	unsigned long i;

	// release published structures when being shut down
	for(i=0;i<_Structures.length();i++)
		_Structures[i]->decRef();

	_Structures.length(0);	// should release the references to the structures

	// remove published modules from broker

	_ModuleBroker->serverShutdown(this);
	_ModuleBroker->decRef();

	// close AudioManager

	_AudioManager->decRef();

	// delete StructureExpander, which will free it's references to
	// Synthesizer & ModuleBroker
	delete _StructureExpander;
}

MidiChannel_impl *Synthesizer_impl::getMidiChannel()
{
	return _MidiChannel;
}

long Synthesizer_impl::StructChangeCount()
{
	return _StructChangeCount;
}

Synthesizer_impl::~Synthesizer_impl()
{
	CORBA::release(_MidiChannel);
	artsdebug("... Synthesizer_impl destroyed\n");
}

void Synthesizer_impl::forceTermination()
{
	exit(255);
}

Arts::ModuleBroker *Synthesizer_impl::moduleBroker()
{
	assert(_ModuleBroker);
	return(Arts::ModuleBroker::_duplicate(_ModuleBroker));
}

Arts::AudioManager *Synthesizer_impl::audioManager()
{
	assert(_AudioManager);
	return(Arts::AudioManager::_duplicate(_AudioManager));
}

void Synthesizer_impl::RTConfig(const Arts::RealtimeConfig &newconfig)
{
	assert(newconfig.Cycles > 0);
	_RTConfig = newconfig;
	setartsdebug(_RTConfig.Debug);
}

Arts::RealtimeConfig Synthesizer_impl::RTConfig()
{
	return _RTConfig;
}

Cache *Synthesizer_impl::getCache()
{
	return _Cache;
}

bool Synthesizer_impl::Exec()
{
/** old style (one sample) scheduling **
	long i,cycles = _RTConfig.Cycles;
	while(cycles--)
	{
		for(i=0;i<MCount;i++) SynthModules[i]->Calculate();
	}
****************************************/

/** new style (dynamic size) scheduling **/

	unsigned long cycles = _RTConfig.Cycles;
	unsigned long *done = (unsigned long *)calloc(MCount,sizeof(unsigned long));
	unsigned long i;
	long incomplete, died = 0;

	//printf("entering; cycles = %d\n",cycles);
	do {
		incomplete = 0;		/* assume we have calculated all cycles for all
								consumers, and later increment if some are
								still missing */
		for(i=0;i<MCount;i++)
		{
			if(SynthModules[i]->clients == 0 && SynthModules[i]->enabled)
			{
				//printf("consumber = %s,done = %d, cycles = %d\n",SynthModules[i]->getClassName(),done[i],cycles);
				/* a module whose input is not comsumed from other modules
					is a "push delivery" style module, such as speakers,
					or writing audio to log file, etc. and has to get
					external requests from the scheduling system */

				if(done[i] != cycles)
					done[i] += SynthModules[i]->request(cycles-done[i]);
				assert(done[i] <= cycles);
				if(done[i] != cycles) incomplete++;
				//printf("*scheduler*\n");
				died ++;
				if(died > 10000)
				{
					free(done);
					return false;
				}
			}
		}
	} while(incomplete);

	//printf("<=> done!!\n");
	free(done);
	return true;
}

void Synthesizer_impl::notifyIO(int fd,long)
{
	Run();
}

void Synthesizer_impl::Run()
{
	if(schedulerLocks > 0)
	{
		/*
		 * What happens here is:
		 *
		 *  - the Scheduler is blocked (probably because some algorithms
		 *    modify the flow system right now, in that case scheduling
		 *    is very unwise)
		 *
		 *  - some filedescriptor (probably the soundcard) is requesting
		 *    data
		 *
		 * We need to stop watching our filedescriptors when this situation
		 * happens, because it is likely, that the synthesizer is waiting
		 * for some data from a process without realtime rights (such as the
		 * GUI server).
		 *
		 * If we would continue watching our descriptors, the synthesizer
		 * would try to do the scheduling again and again, using 100% cpu
		 * time, which means that the GUI Server (or whatever) never gets
		 * the chance to complete his request. So give them the CPU power,
		 * so they can do something useful.
		 */
		ioManager.stop();
		artsdebug("run occured while scheduler locked (locks = %ld)\n",
			schedulerLocks);
		return;
	}

	inScheduler = true;
	bool eresult = Exec();
	inScheduler = false;

	if(!eresult)
	{
		status_halted = true;
		fprintf(stderr,"seems to be uncalculatable?\n");
	}

	while(killList.size())
	{
		long mid = *killList.begin();
		// FIXME: this accesses the execution server variables from the flow
		// server, which is ugly
		long id = 0;
		list<ExecRequest *>::iterator ei;
		for(ei=execRequests.begin();ei != execRequests.end(); ei++)
		{
			ExecRequest *er = *ei;
			if(er->ServerID() == ID() && er->MID() == mid)
				id = er->StructureID();
		}
		if(id == 0)
			artsdebug("kill mid %ld failed\n",mid);
		else
			freeStructure(id);
		killList.erase(killList.begin());
	}

	if(status_halted)
	{
		usleep(10000);
		return;
	}

	//fprintf(stderr,"Using %ld modules for synthesis\n",MCount);

	// Calculate cpu usage.
	status_cycles += _RTConfig.Cycles;
	if(status_cycles > 20000)
	{
		// do cache cleanups periodically
		status_cache_usage = _Cache->cleanUp(_RTConfig.CacheSizeKB*1024);

	    float cpu_time = (clock()-status_clocks)/(float)CLOCKS_PER_SEC;
		float real_time = benchmark(BENCH_END);

		if(cpu_time > 0 && real_time > 0) // there may be wraparounds
		{
			status_cpu_usage = cpu_time / real_time;
			if(status_cpu_usage > 0.9)	  // more than 90%  -> not good!
			{
				status_stalled ++;
			}
			else status_stalled = 0;

			if(status_stalled > 5)
			{
				// ok, cancel synthesis due to cpu overload!
				status_halted = true;
			}
		}

		// prepare for next checkpoint
		status_clocks = clock(); 
		status_cycles -= 20000;
		benchmark(BENCH_BEGIN);
	}

// CALLBACKS that can't be handled during the scheduling cycle

	if(callBackList.size())
	{
		// This is for modules that want to perform operations that are
		// not allowed while scheduling, such as freeing or starting
		// structures.
		//
		// Perform max. 1 callback here, as they may be costly in resources,
		// and they may lead to reentrant processing otherwise
		//
		// FIXME: make kill work as callbacks as well, we might get into
		// troube with reentrancy due to the same reasons
		SynthModule *m = *callBackList.begin();
		callBackList.erase(callBackList.begin());
		m->CallBack();
	}
}

Arts::Status Synthesizer_impl::getStatus()
{
	Arts::Status status;

	status.cpu_usage = status_cpu_usage;
	status.cache_usage = status_cache_usage;
	status.halted = status_halted;
	status.module_count = MCount;
	return(status);
}

void Synthesizer_impl::Reset()
{
	status_halted = false;
	status_stalled = 0;
	status_cpu_usage = 0;
	status_clocks = clock();
	status_cycles = 0;	
	benchmark(BENCH_BEGIN);
}

void collectPorts( Arts::PortTypeSeq& Ports, Arts::StringSeq& PortNames,
     SynthModule *module, SynthModule::ArgType mdir, Arts::PortDirection pdir )
{
	string portname;
	char *pn;
	long portcount = 0;
	long len;

	Arts::PortType porttype;

	while((portname = module->getArgument(portcount,mdir)) != "")
	{
		porttype.Direction = pdir;
		porttype.DataType = Arts::audio_data;
		porttype.ConnType = Arts::stream;

		if(portname[0] == '_')		// is there a better way?
		{
			pn = strdup(portname.c_str());
			pn ++;					// cut the first byte
			portname = pn;
			pn --;					// but free the right thing
			free(pn);

			porttype.DataType = Arts::string_data;
			porttype.ConnType = Arts::property;
		}

		artsdebug("#%d: %s\n",portcount,portname.c_str());

		len = Ports.length();
		Ports.length(len+1);
		Ports[len] = porttype;

		len = PortNames.length();
		PortNames.length(len+1);
		PortNames[len] = (const char *)portname.c_str();

		portcount++;
	}
}

// FlowServer

void Synthesizer_impl::initModules()
{
	long nr;

	_ID = _ModuleBroker->mkID();

	for(nr=0;nr < MS_SynthModule->getModuleCount(); nr++)
	{
		char *name=MS_SynthModule->getModuleName(nr);
		if(name)
		{
			SynthModule *m = (SynthModule *)MS_SynthModule->getModule(name);

			if(m)
			{
				Arts::ModuleInfo info;
				info.name = CORBA::string_dup(name);
				info.isInterface = (strncmp(name,"Interface",9) == 0);
				info.isStructure = false;

				collectPorts(info.ports,info.portnames,m,
								SynthModule::arg_in, Arts::input);
				collectPorts(info.ports,info.portnames,m,
								SynthModule::arg_out, Arts::output);

				_ModuleBroker->publishModule(this,info);

				delete m;
			}
		}
	}
}

Arts::StructureDesc *Synthesizer_impl::expandStructureDesc(Arts::StructureDesc
															*structuredesc)
{
	return _StructureExpander->expandStructure(structuredesc);
}

// Factory for StructureDesc
Arts::StructureDesc *Synthesizer_impl::createStructureDesc()
{
	artsdebug("creating new structure desc\n");

	Arts::StructureDesc_ptr sd =
		new StructureDesc_impl(Arts::ModuleBroker::_duplicate(_ModuleBroker));
	return Arts::StructureDesc::_duplicate(sd);
}

// Execution Manager
CORBA::Long Synthesizer_impl::createStructure(Arts::StructureDesc
					*structuredesc, const Arts::ArtsServerSeq& preferredservers)
{
	// no interfaces to the outside world
	//
	// this is just for using the same code for creating structures
	// that are connected with interfaces to the outside world and
	// creating self contained structures
	list<SynthModule *> interfaces;

	return createConnectedStructure(structuredesc,preferredservers,interfaces);
}


// Execution Manager
long Synthesizer_impl::createConnectedStructure(Arts::StructureDesc
			*structuredesc, const Arts::ArtsServerSeq& preferredservers,
			list<SynthModule*> interfaces, CORBA::Long restoreID,
			CORBA::Long oldID)
{
	assert(!inScheduler);

	_StructChangeCount++;
	long currentID = nextID++;

	// first step: create modules, allocate output buffers

	// since "interfaces" are not yet specified in the CORBA interfaces,
	// they are passed to the FlowServer beside CORBA, which is an ugly
	// hack

	_global_interfaces_hack = interfaces;

	Arts::ModuleDescSeq_var modules;
	Arts::StructureDesc_var expandedsd;
	bool needExpansion = structuredesc->containsStructures();

	CORBA::String_var structureName = structuredesc->Name();

	if(!needExpansion)
	{
		artsdebug("Execute 0: no expansion necessary, no substructures used\n");
 		modules = structuredesc->Modules();
	}
	else
	{
		artsdebug("Execute 0 <Expansion>\n");
		expandedsd = expandStructureDesc(structuredesc);
		expandedsd->incRef();
 		modules = expandedsd->Modules();
	}

	list<ExecRequest *> newexlist;
	list<ExecRequest *>::iterator ei;
	ExecRequest *er;

	artsdebug("Execute I <Creation>\n");
	unsigned long m;

	for(m=0;m<modules->length();m++)
	{
		Arts::ModuleDesc *md = (*modules)[m];
		CORBA::String_var modname = md->Name();
		Arts::ArtsServerSeq_var servers =
						_ModuleBroker->whichServerProvides(modname);
		artsdebug("checking whether module %s is present\n",
					(const char *)modname);
		assert(servers->length());

		// FIXME: might do some nicer server selection algorithm
		long serverID = -1;
		Arts::ArtsServer *server = 0;

		if(servers->length() > 1)
		{
			artsdebug("more that one possible server choice for module %s\n",
								(const char *)modname);
			unsigned long i,s;
			for(i=0;i<preferredservers.length() && !server;i++)
			{
				long goodID = preferredservers[i]->ID();
				for(s=0;s<servers->length() && !server;s++)
				{
					if((*servers)[s]->ID() == goodID)
					{
						artsdebug("selected server with ID %d\n",goodID);
						server = (*servers)[s];
					}
				}
			}
			if(!server)
				artsdebug("WARNING: no server due to preferences possible\n");
		}
		if(!server)		// default choice - take the first server that is there
			server = (*servers)[0];

		serverID = server->ID();

		er = 0;

		for(ei=newexlist.begin();!er && ei != newexlist.end();ei++)
		{
			ExecRequest *thiser = *ei;
			if(thiser->ServerID() == serverID)
			{
				// good, we've already got some server doing work
				// for us, just put the module in there
				er = thiser;
			}
		}

		// server not yet connected to do the creation for us; connect
		if(!er)
		{
			er = new ExecRequest(server,currentID,structureName);
			newexlist.push_back(er);
			execRequests.push_back(er);
		}
		er->createModule(md);
	}

	// While the modifications to the flow system are made, no scheduling
	// should be done!
	lockScheduler();

	// second step: establish local connections between the modules
	for(ei=newexlist.begin();ei != newexlist.end();ei++)
		(*ei)->localConnectModules();

	// remote connection setup:
	for(ei=newexlist.begin();ei != newexlist.end();ei++)
	{
		list<ExecRequest *>::iterator ei2;
		for(ei2=newexlist.begin();ei2 != newexlist.end();ei2++)
		{
			if(ei != ei2)
				(*ei)->remoteConnectModules(*ei2);
		}
	}

	// session data restauration

	RestoreInfo *ri = 0;
	if(restoreID != 0)
	{
		// every session restoration information is supposed to be used once,
		// so if we find session data here, we'll free it as well

		list<RestoreInfo *>::iterator rii;

		rii = riList.begin();
		while(ri == 0 && rii != riList.end())
		{
			if((*rii)->restoreID == restoreID && (*rii)->oldID == oldID)
			{
				if((*rii)->ok && (*rii)->haveSessionData)
				{
					ri = *rii;
					riList.erase(rii);
				}
			}

			if(!ri) rii++;
		}
	}

	if(ri != 0)
	{
		for(ei=newexlist.begin();ei != newexlist.end();ei++)
		{
			(*ei)->Server()->restoreSessionModules((*ei)->MID(),
												ri->sessionData, ri->restoreID);
			// FIXME: could check return code here
		}
		delete ri;
	}

	// finalize the modules (does the last initialization before start)

	artsdebug("finalization:\n");
	for(ei=newexlist.begin();ei != newexlist.end();ei++)
		(*ei)->finalizeModules();

	// finally start the modules
	artsdebug("start:\n");
	for(ei=newexlist.begin();ei != newexlist.end();ei++)
		(*ei)->startModules();

	unlockScheduler();

	if(needExpansion)
	{
		artsdebug("Free expanded structure\n");
		expandedsd->decRef();
	}

	artsdebug("Execute ok\n");

	return currentID;
}

void Synthesizer_impl::lockScheduler()
{
	assert(inScheduler == false);
	schedulerLocks++;
}

void Synthesizer_impl::unlockScheduler()
{
	assert(inScheduler == false);
	assert(schedulerLocks > 0);

	schedulerLocks--;
	if(schedulerLocks == 0)
		ioManager.start();
}

// Dynamic Connections? Execution Manager?
void Synthesizer_impl::addDynamicConnection (SynthModule *target, SynthModule *source, unsigned long size)
{
	assert(!inScheduler);
	unsigned long i;
	artsdebug(">>>>>>>>>> ADC: target is %s source is %s\n",target->getClassName(),source->getClassName());
	artsdebug(">>>>>>>>>> ADC: source->inConnCount = %d\n",source->inConnCount);
	artsdebug(">>>>>>>>>> ADC: target->inConnCount = %d\n",target->inConnCount);
	artsdebug(">>>>>>>>>> ADC: add channels: %d\n",size);
	assert(source->inConnCount >= size);
	for(i=0;i < size;i++)
	{
		SynthConnection *newconn = new SynthConnection(target,0);
		newconn->isDynamic = true;
		target->assignInConn(newconn);
		// FIXME: will fail when constant stuff
		newconn->connectToSource(source->inConn[i]->sourceconn);
	}
	artsdebug(">>>>>>>>>> after target->inConnCount = %d\n",target->inConnCount);
}

// Dynamic Connections? Execution Manager?
void Synthesizer_impl::removeDynamicConnections (SynthModule *module)
{
	assert(!inScheduler);

	unsigned long i,remove=0;

	for(i=0;i<module->inConnCount;i++)
	{
		if(module->inConn[i]->isDynamic)
		{
			remove++;
			if(module->inConn[i]->sourceconn)
				module->inConn[i]->disconnectFromSource(module->inConn[i]->sourceconn);
			delete module->inConn[i];
		}
		else
		{
			// dynamic channels should ONLY be attached at the end!
			assert(remove == 0);
		}
	}
	module->inConnCount -= remove;

	module->inConn = (SynthConnection **)realloc(module->inConn,
						sizeof(SynthConnection *)*module->inConnCount);
	// FIXME: free memory here (?)
}

// Flow Server
void Synthesizer_impl::kill(long id)
{
	// Is called by Synth_STRUCT_KILL
	//
	// As freeing while a scheduling cycle is done is not possible, we push
	// that request to a list and execute the kills after scheduling

	killList.push_back(id);
}

// Flow Server
void Synthesizer_impl::requestCallBack(SynthModule *m)
{
	callBackList.push_back(m);
}

// Execution Manager
CORBA::Boolean Synthesizer_impl::isExecuting(CORBA::Long id)
{
	list<ExecRequest *>::iterator ei;
	for(ei=execRequests.begin(); ei != execRequests.end(); ei++)
	{
		if((*ei)->StructureID() == id) return true;
	}
	return false;
}

// Execution Manager + Flow Server
CORBA::Boolean Synthesizer_impl::freeStructure(CORBA::Long id)
{
	artsdebug("freeing structure id = %ld\n",id);

	// Should notify the parent that the structure it is executing
	// no longer exists. Currently there is only polling to check that.

	_StructChangeCount++;

	// kill all those modules that are executed on some server out there
	// and belong to that structure

	// don't do scheduling while modifications to the flow system are made

	lockScheduler();

	list<ExecRequest *>::iterator ei = execRequests.begin();
	while(ei != execRequests.end())
	{
		ExecRequest *er = *ei;
		if(er->StructureID() == id)
		{
			er->deleteModules();
			execRequests.erase(ei);
			delete er;
			ei = execRequests.begin();
		}
		else ei++;
	}

	unlockScheduler();

	return true; // FIXME: return code
}

// Structure Repository
void Synthesizer_impl::publishStructureDesc(Arts::StructureDesc *structuredesc)
{
	Arts::StructureDesc_ptr structure;

	unsigned long l = 0;
	bool found = false;

	// create deep copy by saving and reloading
	Arts::StringSeq_var strings = structuredesc->saveToList();

	// reload below

	while(l < _Structures.length() && !found)
	{
		CORBA::String_var name1=_Structures[l]->Name();
		CORBA::String_var name2=structuredesc->Name();

		found = !strcmp(name1,name2);
		if(!found) l++;
	}

	if(!found) // didn't find a structure with the same name
	{
		_Structures.length(l+1);

		structure = createStructureDesc();
		_Structures[l] = structure;
		structure->incRef();
	}
	else		// if we did, just overwrite this one
	{
		structure = _Structures[l];
		structure->clear();		// clear old contents

		// remove old published module
		_ModuleBroker->removeModule(this,structure->Name());
	}
	structure->loadFromList(strings);

	Arts::ModuleInfo_var mi = structure->ExternalInterface();
	_ModuleBroker->publishModule(this,*mi);
}

// Structure Repository
Arts::StructureDesc *Synthesizer_impl::lookupStructureDesc(const char *name)
{
	unsigned long i;

	for(i=0;i<_Structures.length();i++)
	{
		CORBA::String_var aname = _Structures[i]->Name();
		if(strcmp(aname,name) == 0)
			return Arts::StructureDesc::_duplicate(_Structures[i]);
	}

	return 0;
}

// Structure Repository
Arts::StructureDescSeq *Synthesizer_impl::publishedStructures()
{
	return(new Arts::StructureDescSeq(_Structures));
}

// Flow Server?
Arts::ResourceInfoSeq *Synthesizer_impl::Resources()
{
	Arts::ResourceInfoSeq *resources = new Arts::ResourceInfoSeq;
	list<ResourceMonitor *> &rlist = *ResourceList::the()->getResources();
	list<ResourceMonitor *>::iterator i;
	unsigned long nr = 0;

	resources->length(rlist.size());
	for(i=rlist.begin();i != rlist.end();i++)
	{
		Arts::ResourceInfo& ri = (*resources)[nr++];
		ri.Used = (*i)->Used();
		ri.Description = CORBA::string_dup((*i)->Description());
	}
	return(resources);
}

/************ Flow System Server *************/

CORBA::Long Synthesizer_impl::createModules()
{
	return nextMID++;
}

CORBA::Boolean Synthesizer_impl::createModule(CORBA::Long mid,
												Arts::ModuleDesc *desc)
{
	unsigned long p;

	CORBA::String_var modname = desc->Name();
	Arts::PortDescSeq_var ports = desc->Ports();
	long RBSize = _RTConfig.RingBufferSize;
	artsdebug("creating module %s\n",(char *)modname);
	
	SynthModule *NewModule;

	// since "interfaces" are not yet specified in the CORBA interfaces,
	// they are passed to the FlowServer beside CORBA, which is an ugly
	// hack
	list<SynthModule *>& interfaces = _global_interfaces_hack;
	
	if(desc->isInterface())
	{
		list<SynthModule *>::iterator iface = interfaces.begin();
		bool found = false;
		artsdebug(" => setting up interface connection\n");
	
		while(!found && iface != interfaces.end())
		{
			artsdebug("[%s]<=>[%s]\n",(*iface)->getClassName(),(char *)modname);
			if(strcmp((*iface)->getClassName(),modname) == 0)
			{
				found = true;
				NewModule = (*iface);

				// only use once of course!
				interfaces.remove(NewModule);
			}
			iface++;
		}
		assert(found);
	}	
	else
	{
		NewModule = (SynthModule *)MS_SynthModule->getModule(modname);
		NewModule->FirstInitialize();
	}
	assert(NewModule);
	NewModule->Synthesizer = this;
	
	NewModule->ID = desc->ID();
	NewModule->mID = mid;

/*	FIXME: may use some other solution here
	if(!desc->isInterface())
	{
		...?
	}
*/
	//NewModule->enabled = !desc->isInterface();

	//NewModule->enabled = true;  !!! WARNING: don't enable here, as mico
	// may try to do flow system evaluations in between

	SynthModules = (SynthModule **)realloc(SynthModules,
									sizeof(SynthModule *)*(MCount+1));
	SynthModules[MCount++] = NewModule;

	unsigned long inportcount = 0;
	unsigned long outportcount = 0;

	for(p=0;p<ports->length();p++)
	{
		Arts::PortDesc *pd = (*ports)[p];
		Arts::PortType pt = pd->Type();


		if(pt.Direction == Arts::input)
		{
			assert(pt.ConnType != Arts::event); // no event channels yet

			SynthPort *synthport =
				new SynthPort(NewModule, inportcount++, pd->ID());
			NewModule->inPorts.push_back(synthport);

			if(pt.ConnType == Arts::stream)
			{
				assert(pt.DataType == Arts::audio_data);
				
				SynthConnection *conn =
					new SynthConnection(NewModule,pd->ID());
				NewModule->assignInConn(conn);
				synthport->setConnection(conn);

				if(pd->isConnected())
				{
					Arts::PortDescSeq_var conns = pd->Connections();
					assert(conns->length() == 1);
					Arts::PortDesc *pd2 = (*conns)[0];

					long wantedID = pd2->ID();
					conn->needConnect = true;
					conn->wantedID = wantedID;
				}
				else
				{
					float initialvalue = 0.0;

					if(pd->hasValue())
						initialvalue = pd->FloatValue();

					conn->allocBuffer(initialvalue,RBSize);
				}
			}
			if(pt.ConnType == Arts::property)
			{
				assert(pt.DataType == Arts::string_data);

				if(pd->isConnected())
				{
					// if its connected, find the connection partner
					Arts::PortDescSeq_var conns = pd->Connections();
					assert(conns->length() == 1);
					Arts::PortDesc *pd2 = (*conns)[0];

					long wantedID = pd2->ID();
					synthport->setNeedConnect(true, wantedID);
				}
				else
				{
					// if its not connected simply look up value and create
					// a property that will hold the value

					// FIXME: this fails when no value is given (should it
					// fail then?)

					CORBA::String_var arg = pd->StringValue();
					synthport->setProperty(new SynthProperty(arg));
					artsdebug("got stringproperty; %s\n",(const char *)arg);
				}
			}
		}
		if(pt.Direction == Arts::output)
		{
			artsdebug("creating output port id %ld\n",pd->ID());

			SynthPort *synthport =
				new SynthPort(NewModule, outportcount++, pd->ID());
			NewModule->outPorts.push_back(synthport);

			if(pt.ConnType == Arts::stream)
			{
				SynthConnection *conn =
					new SynthConnection(NewModule,pd->ID());
				conn->allocBuffer(0.0,RBSize);
				NewModule->assignOutConn(conn);
				synthport->setConnection(conn);
			}
			if(pt.ConnType == Arts::property)
			{
				synthport->setProperty(new SynthProperty(""));
			}
		}
	}
	return true;
}

CORBA::Boolean Synthesizer_impl::remoteConnectModules(CORBA::Long mid,
		CORBA::Long remotemid, const Arts::ModuleDescSeq& modules,
		Arts::ArtsServer *server)
{
	map<long,bool> portMap;

	artsdebug("remoteConnectModules to server %d called\n",server->ID());
	unsigned long m,i;

	for(m=0;m<modules.length();m++)
	{
		Arts::ModuleDesc *md = modules[m];
		Arts::PortDescSeq_var ports = md->Ports();

		unsigned long p;
		for(p=0;p<(*ports).length();p++)
		{
			portMap[(*ports)[p]->ID()] = true;		// he has that port
		}
	}
	for(i=0;i<MCount;i++)
	{
		SynthModule *NewModule = SynthModules[i];

		if(NewModule->mID == mid)
		{
			unsigned long p;
			for(p=0;p<NewModule->inConnCount;p++)
			{
				SynthConnection *conn = NewModule->inConn[p];
				if(conn->needConnect)
				{
					artsdebug("needconnect for %s\n",NewModule->getClassName());
					if(portMap[conn->wantedID])  // yes, it's on that server
					{
						artsdebug("found a source for %s\n",
								NewModule->getClassName());
						artsdebug("requesting from server\n");
						Arts::Receiver *rcv = new Receiver_impl(conn);
						server->requestSignal(remotemid,conn->wantedID, rcv);
						conn->needConnect = false;
					}
				}
			}
		}
	}
	return true;
}

CORBA::Boolean Synthesizer_impl::localConnectModules(CORBA::Long mid)
{
	if(!SynthModules) return false;
	
	artsdebug("Execute II localConnectModules\n");

	map<long,SynthPort *> portMap;

	unsigned long i;
	for(i=0;i<MCount;i++)
	{
		SynthModule *NewModule = SynthModules[i];

		if(NewModule->mID == mid)
		{
			vector<SynthPort *>::iterator i;
			for(i=NewModule->outPorts.begin();i!=NewModule->outPorts.end();i++)
			{
				SynthPort *p = *i;
				portMap[p->ID()] = p;
			}
		}
	}
	for(i=0;i<MCount;i++)
	{
		SynthModule *NewModule = SynthModules[i];

		if(NewModule->mID == mid)
		{
			vector<SynthPort *>::iterator i;
			for(i=NewModule->inPorts.begin();i!=NewModule->inPorts.end();i++)
			{
				SynthPort *p = *i;
				SynthConnection *conn = p->connection();

				// FIXME: needs a little rewrite to use SynthPort interface
				// for needConnect,...
				if(conn && conn->needConnect)
				{
					SynthConnection *source = 0;

					SynthPort *sourceport = portMap[conn->wantedID];
					if(sourceport) source=sourceport->connection();

					if(!source)
					{
						artsdebug("NewModule %s missing source (remote?)\n",
							NewModule->getClassName());
					}
					else
					{
						conn->connectToSource(source);
						conn->needConnect = false;
					}
				}
				else if(p->needConnect())
				{
					SynthPort *dest = portMap[p->wantedID()];
					if(!dest)
					{
						artsdebug("NewModule %s missing propsource (remote?)\n",
							NewModule->getClassName());
					}
					else
						p->connectToPort(dest);
				}
			}
		}
	}
	return true;
}

CORBA::Boolean Synthesizer_impl::finalizeModules(CORBA::Long mid)
{
	if(!SynthModules) return false;

	artsdebug("Execute III <Prepare & Initialization> in finalizeModules\n");

	unsigned long i;
	for(i=0;i<MCount;i++)
	{
		SynthModule *NewModule = SynthModules[i];

		if(NewModule->mID == mid)
		{
			artsdebug("NewModule %s characteristics: in %ld, out %ld\n",
						NewModule->getClassName(),NewModule->inConnCount,
						NewModule->outConnCount);
			NewModule->Initialize(); //NewModule->args);
	
			// EXPERIMENTAL: prepareExecution after Initialize
			NewModule->prepareExecution();
		}
	}
	return true;
}

void Synthesizer_impl::startModules(CORBA::Long mid)
{
	if(!SynthModules) return;

	artsdebug("Execute IV <IO Initialization> in startModules\n");

	unsigned long i;
	for(i=0;i<MCount;i++)
	{
		SynthModule *NewModule = SynthModules[i];
		if(NewModule->mID == mid)
		{
			ArtsNotifyIO *handler = this;
			int infd,outfd;
	
			NewModule->enabled = true;
			NewModule->getfds(infd,outfd);

			if(NewModule->ioHandler() != 0) handler = NewModule->ioHandler();
	
			if(infd != -1)
				ioManager.addFD(handler,mid,ArtsIOEvent::ioRead,infd);

			if(outfd != -1)
				ioManager.addFD(handler,mid,ArtsIOEvent::ioWrite,outfd);
		}
	}
}

void Synthesizer_impl::deleteModules(CORBA::Long mid)
{
	long NCount = 0,freed = 0;
	artsdebug("executing deletemodules %ld\n",mid);

	ioManager.remove(mid);

	if(SynthModules)
	{
		unsigned long i;
		// kill those that are local and match the id

		// 1st step: deinitialize them
		for(i=0;i<MCount;i++)
		{
			if(SynthModules[i]->mID == mid)
				SynthModules[i]->DeInitialize();
		}

		// 2nd step: delete them
		for(i=0;i<MCount;i++)
		{
			if(SynthModules[i]->mID == mid)
			{
				delete SynthModules[i];
				freed++;
			}
			else SynthModules[NCount++] = SynthModules[i]; // "compress" modules
		}

		MCount = NCount;
		SynthModules = (SynthModule **)realloc(SynthModules,
									sizeof(SynthModule *)*MCount);
		if(MCount == 0) SynthModules = 0;

		if(freed)
		{
			artsdebug("freed %ld modules\n",freed);
		}
	}
	artsdebug("deletemodules ok\n");
}

CORBA::Long Synthesizer_impl::ID()
{
	return _ID;
}

void Synthesizer_impl::requestSignal(CORBA::Long mid, CORBA::Long portid,
										Arts::Receiver *receiver)
{
}

Arts::StringSeq *Synthesizer_impl::busList()
{
	return BusManager::the()->busList();
}

//--- Path management:

Arts::StringSeq* Synthesizer_impl::artsPath()
{
	return new Arts::StringSeq(_artsPath); 
}

void Synthesizer_impl::addArtsDirectory( const char* directory )
{
	char *dname = strdup(directory);
	unsigned long l;

	// strip slashes which are too much
	while((l=strlen(dname)) > 1 && dname[l] == '/') dname[l] = 0;

	for(l=0;l<_artsPath.length();l++)
	{
		if(strcmp(_artsPath[l],dname) == 0)
		{
			free(dname);
			return;
		}
	}

	l = _artsPath.length();
	_artsPath.length(l+1);
	_artsPath[l++] = CORBA::string_dup(dname);
	free(dname);
}


char* Synthesizer_impl::artsFileName( const char *directory, const char* name )
{
	if(name[0] == '/')
	{
		// it's an absolute filename, so don't use the path to expand it
		return(CORBA::string_dup(name));
	}

	unsigned long l;
	for(l=0;l<_artsPath.length();l++)
	{
		string combined = string(_artsPath[l])
				+ "/" + string(directory) + "/" + string(name);
		FILE *f = fopen(combined.c_str(),"r");
		if(f)
		{
			fclose(f);
			return CORBA::string_dup(combined.c_str());
		}
	}

	// not found
	return CORBA::string_dup("");
}

Arts::StringSeq*
Synthesizer_impl::listFiles( const char* directory, const char* extension )
{
	Arts::StringSeq *result = new Arts::StringSeq;

	unsigned long l, extlen = strlen(extension);
	for(l=0;l<_artsPath.length();l++)
	{
		string pathname = string(_artsPath[l])+"/"+directory;

		DIR *dir = opendir(pathname.c_str());
		if(dir != 0)
		{
			struct dirent *de;
			while((de = readdir(dir)) != 0)
			{
				if(strlen(de->d_name) > extlen &&
						strncmp(&de->d_name[strlen(de->d_name)-extlen],
											extension,extlen) == 0)
				{
					result->length(result->length()+1);
					(*result)[result->length()-1]=CORBA::string_dup(de->d_name);
				}
			}
			closedir(dir);
		}
	}
	return result;
}

// session management

CORBA::Boolean Synthesizer_impl::saveSession(CORBA::Long structureID,
			CORBA::Boolean saveStructureDescs, Arts::StringSeq*& session)
{
	lockScheduler();

	artsdebug("saveSession starting\n");
	session = new Arts::StringSeq;

	list<long> todoIDs;
	map<long,bool> listedIDs;

	/* saving a session
	 *
	 * A session consists of multiple running structures. You start by
	 * giving the sessionmanager an ID of a session to save.
	 *
	 * Then it will save this first running structure (which will be called
	 * the first component of the session). It might be, that this structure
	 * requires other structures to be running as well (e.g. a mixer will
	 * require one additional structure per channel. The saveSessionModules
	 * call will then come back with additional IDs to save as well.
	 *
	 * So the layout of the session file is
	 *
	 * component=<ID>
	 * {
	 *   structure=<name of the structure this component was created from>
	 *   data
	 *   {
	 *     module_id=<module ID>
	 *     {
	 *       [... module specific data, for instance position of a slider ...]
	 *     }
	 *     module_id=<module ID>
	 *     {
	 *       [... module specific data, for instance position of a slider ...]
	 *     }
	 *   }
	 * }
	 * component=<ID>
	 * {
	 *   [... next component, layout like the first one ...]
	 * }
     */
	todoIDs.push_back(structureID);
	listedIDs[structureID] = true;

	while(todoIDs.size())
	{
		long ID = *todoIDs.begin();
		todoIDs.pop_front();

		Arts::StringSeq_var serverData = new Arts::StringSeq;
		Arts::StringSeq_var thisComponent = new Arts::StringSeq;
		string structureName;

		int hits = 0;

		list<ExecRequest *>::iterator ei;
		for(ei=execRequests.begin();ei != execRequests.end(); ei++)
		{
			ExecRequest *er = *ei;
			if(er->StructureID() == ID)
			{
				if(hits == 0)
				{
					structureName = er->structureName();
				}
				else
				{
					assert(structureName == er->structureName());
				}
				Arts::ArtsServer *Server = er->Server();

				Arts::StringSeq_var data;
				Arts::IDSeq_var ids;

				// FIXME: check return code
				Server->saveSessionModules(er->MID(), data, ids);
				appendStringSeq(serverData,data);

				unsigned long l;
				for(l=0;l<ids->length();l++)
				{
					long tosave = (*ids)[l];
					if(!listedIDs[tosave]) todoIDs.push_back(tosave);
				}
				hits++;
			}
		}

		sqprintf(thisComponent,"structure=%s",structureName.c_str());
		sqprintf(thisComponent,"data",structureName.c_str());
		addSubStringSeq(thisComponent,serverData);

		sqprintf(session,"component=%ld",ID);
		addSubStringSeq(session,thisComponent);
	}

	artsdebug("saveSession finished\n");
	unlockScheduler();
	return true;
}

CORBA::Long Synthesizer_impl::restoreStructure(Arts::StructureDesc
		*structuredesc, const Arts::ArtsServerSeq& preferredservers,
		CORBA::Long restoreID, CORBA::Long oldID)
{
	list<SynthModule *> interfaces;

	return createConnectedStructure(structuredesc,preferredservers, interfaces,
										restoreID,oldID);
}

CORBA::Long Synthesizer_impl::restoreSession(const Arts::StringSeq& session,
			const Arts::ArtsServerSeq& preferredservers)
{
	long returnID = 0;

	// first component is restored by looking up structure by name, all other
	// components should be restored by their respective owners

	RestoreInfo *firstRI = 0;

	bool firstComponent = true;

	long theRestoreID = nextRestoreID++;
	unsigned long i;
	char *cmd,*param;

	// first step: collect all components into RestoreInfo data structures

	for(i=0;i<session.length();i++)
	{
		if(parse_line(session[i],cmd,param))	// otherwise: empty or comment
		{
			if(strcmp(cmd,"component") == 0)
			{
				RestoreInfo *ri = new RestoreInfo;
				Arts::StringSeq_var componentData;

				riList.push_back(ri);
				ri->haveStructureName = ri->haveSessionData = false;
				ri->ok = true;
				ri->oldID = atol(param);
				ri->restoreID = theRestoreID;
				componentData = getSubStringSeq(&session,i);

				// parse component restauration information

				unsigned long j;

				for(j=0;j<componentData->length();j++)
				{
					if(parse_line((*componentData)[j],cmd,param))
					{
						artsdebug("RS: load-> cmd was %s\n",cmd);
						if(strcmp(cmd,"structure") == 0)
						{
							if(ri->haveStructureName) ri->ok = false; // twice?
							ri->haveStructureName = true;
							ri->structureName = param;
						}
						else if(strcmp(cmd,"data") == 0)
						{
							if(ri->haveSessionData) ri->ok = false;	  // twice?
							ri->haveSessionData = true;
			
							ri->sessionData = getSubStringSeq(componentData,j);
						}
					}
				}

				if(firstComponent && ri->haveStructureName)
					firstRI = ri;
				firstComponent = false;
			}
		}
	}

	if(firstRI && firstRI->haveStructureName)
	{
		Arts::StructureDesc_var sd;

		sd = lookupStructureDesc(firstRI->structureName.c_str());

		if(sd)
		{
			sd->incRef();
			returnID = restoreStructure(sd,preferredservers,
										firstRI->restoreID, firstRI->oldID);
		}
	}
//--- vars for sending which id has been mapped to which new ID
	return returnID;
}

/*
struct RestoreInfo {
	long oldID, newID;

	bool haveStructureName, haveSessionData, haveStructureDesc;

	string structureName;
	Arts::StructureDesc_var structureDesc;
	Arts::StringSeq_var sessionData;

	bool ok,isFirst;
	bool completed;
};

CORBA::Long Synthesizer_impl::restoreSession(const Arts::StringSeq& session,
			const Arts::ArtsServerSeq& preferredservers)
{
	list<RestoreInfo *> riList;

	long returnID = 0;

	// first component is restored by looking up structure by name, all other
	// components should be restored by their respective owners

	bool firstComponent = true;

	unsigned long i;
	char *cmd,*param;

	// first step: collect all components into RestoreInfo data structures

	for(i=0;i<session.length();i++)
	{
		if(parse_line(session[i],cmd,param))	// otherwise: empty or comment
		{
			if(strcmp(cmd,"component") == 0)
			{
				RestoreInfo *ri = new RestoreInfo;
				Arts::StringSeq_var componentData;

				riList.push_back(ri);
				ri->haveStructureName = ri->haveSessionData = false;

				ri->haveStructureDesc = false;
				ri->ok = true; ri->completed = false;
				ri->oldID = atol(param);
				componentData = getSubStringSeq(&session,i);

				// parse component restauration information

				unsigned long j;

				for(j=0;j<componentData->length();j++)
				{
					if(parse_line((*componentData)[j],cmd,param))
					{
						artsdebug("RS: load-> cmd was %s\n",cmd);
						if(strcmp(cmd,"structure") == 0)
						{
							if(ri->haveStructureName) ri->ok = false; // twice?
							ri->haveStructureName = true;
							ri->structureName = param;
						}
						else if(strcmp(cmd,"data") == 0)
						{
							if(ri->haveSessionData) ri->ok = false;	  // twice?
							ri->haveSessionData = true;
			
							ri->sessionData = getSubStringSeq(componentData,j);
						}
					}
				}

				if(firstComponent && ri->haveStructureName)
				{
    				ri->structureDesc =
						lookupStructureDesc(ri->structureName.c_str());

					if(ri->structureDesc)
					{
						ri->structureDesc->incRef();
						ri->haveStructureDesc = true;
					}
				}

				ri->isFirst = firstComponent;
				firstComponent = false;
			}
		}
	}

//--- vars for sending which id has been mapped to which new ID
	map<long,bool> isRestoredComponent;
	Arts::IDSeq_var oldIDs = new Arts::IDSeq;
	Arts::IDSeq_var newIDs = new Arts::IDSeq;
	unsigned long idCount = 0;


	list<RestoreInfo *>::iterator rii;

	bool active = true;

	while(active)
	{
		active = false;

		for(rii=riList.begin();rii != riList.end();rii++)
		{
			RestoreInfo *ri = *rii;

			if(ri->ok && ri->haveStructureDesc && ri->haveSessionData &&
				!ri->completed)
			{
				list<SynthModule *> interfaces;
				list<Arts::RestoreDescSeq *> rdsList;

				ri->completed = true;
				active = true;

				ri->newID = createConnectedStructure(ri->structureDesc,
					preferredservers,interfaces,ri->sessionData, &rdsList);

				if(ri->isFirst) returnID = ri->newID;

				oldIDs->length(idCount+1);
				newIDs->length(idCount+1);

				(*oldIDs)[idCount] = ri->oldID;
				(*newIDs)[idCount] = ri->newID;

				idCount++;
				isRestoredComponent[ri->newID] = true;

				list<Arts::RestoreDescSeq *>::iterator rdsi;
				for(rdsi = rdsList.begin();rdsi != rdsList.end();rdsi++)
				{
					// autofree the rds thingies here
					Arts::RestoreDescSeq_var rds = (*rdsi);

					for(i=0;i < rds->length();i++)
					{
						list<RestoreInfo *>::iterator rii2;

						for(rii2=riList.begin();rii2 != riList.end();rii2++)
						{
							RestoreInfo *ri2 = *rii2;

							if((*rds)[i].oldID == ri2->oldID)
							{
								artsdebug("* session management: found structureDesc for oldID=%d\n",ri2->oldID);
								ri2->structureDesc =
										Arts::StructureDesc::_duplicate(
												(*rds)[i].structureDesc);

								CORBA::String_var name = ri2->structureDesc->Name();
								artsdebug("* session management: name is %s\n",(const char *)name);
								ri2->haveStructureDesc = true;
							}
						}
					}
				}
			}
		}
	}

	artsdebug("* session management: sending out new IDs of components\n");

	// send out information which ID has been mapped to which ID
	list<ExecRequest *>::iterator ei;
	for(ei=execRequests.begin();ei != execRequests.end(); ei++)
	{
		ExecRequest *er = *ei;
		if(isRestoredComponent[er->StructureID()])
			er->Server()->restoreSessionStructures(er->MID(), oldIDs, newIDs);
	}

	artsdebug("* session management: decRef'ing dtructureDescs\n");
	// decRef ri structureDescs (they are unused now) and free ris (which
	// will also release CORBA style references)
	while(riList.size())
	{
		RestoreInfo *ri = *riList.begin();
		riList.pop_front();

		if(ri->haveStructureDesc)
			ri->structureDesc->decRef();

		delete ri;
	}
	return returnID;
}
*/

CORBA::Boolean Synthesizer_impl::saveSessionModules(CORBA::Long mid,
			Arts::StringSeq*& data, Arts::IDSeq*& structureIDs)
{
	assert(!inScheduler);

	data = new Arts::StringSeq;
	structureIDs = new Arts::IDSeq;

	if(!SynthModules) return true; // no SynthModules, no session information

	list<long> ids;
	unsigned long i;
	// save session in the module implementations

	for(i=0;i<MCount;i++)
	{
		if(SynthModules[i]->mID == mid)
		{
			Arts::StringSeq_var moduledata =
				SynthModules[i]->saveSessionParameters(ids);
			if(moduledata->length() > 0)
			{
				sqprintf(data,"module_id=%d",SynthModules[i]->ID);
				addSubStringSeq(data,moduledata);
			}
		}
	}

	// create list of structureids that are required

	unsigned long l = 0;

	list<long>::iterator ii;
	structureIDs->length(ids.size());
	for(ii=ids.begin();ii != ids.end(); ii++)
		(*structureIDs)[l++] = (*ii);
		
	return true;
}

CORBA::Boolean Synthesizer_impl::restoreSessionModules(CORBA::Long mid,
			const Arts::StringSeq& data, CORBA::Long restoreID)
{
    unsigned long i;
    char *cmd,*param;

	if(!SynthModules) return true;

    for(i=0;i<data.length();i++)
    {
        if(parse_line(data[i],cmd,param))   // otherwise: empty or comment
        {
            if(strcmp(cmd,"module_id") == 0) {
                long id = atol(param);
                Arts::StringSeq_var params = getSubStringSeq(&data,i);

				unsigned long i;
				for(i=0;i<MCount;i++)
				{
					SynthModule *m = SynthModules[i];

					if(m->mID == mid && m->ID == id)
					{
						printf("found module to restore parameters\n");
						m->setRestoreID(restoreID);
						m->restoreSessionParameters(*params);
					}
				}
            }
        }
    }                                                                           
	return true;
}
