    /*

    Copyright (C) 1999 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "guislider.h"
#include "qframe.h"
#include "qbitmap.h"
#include "kiconloader.h"
#include "sequenceutils.h"

MovingLabel::MovingLabel(int minx, int miny, int maxx, int maxy,
							Gui_SLIDER *slider) :QLabel()
{
	isMoving = false;
	this->minx = minx;
	this->miny = miny;
	this->maxx = maxx;
	this->maxy = maxy;
	this->slider = slider;
}

void MovingLabel::mousePressEvent(QMouseEvent *event)
{
	if ( event->button() == LeftButton )
	{
		isMoving = true;
		movestart = event->pos();
		sx = x();
		sy = y();
	}
}

void MovingLabel::mouseMoveEvent(QMouseEvent *event)
{
	if(isMoving)
	{
		int dx = event->x() - movestart.x();
		int dy = event->y() - movestart.y();

		int newx = x()+dx;
		int newy = y()+dy;

		if(newx < minx) newx = minx;
		if(newy < miny) newy = miny;
		if(newx > maxx) newx = maxx;
		if(newy > maxy) newy = maxy;

		move(newx,newy);
		
		float mdx = maxx-minx;
		float mdy = maxy-miny;

		if(mdx < 1) mdx = 1;
		if(mdy < 1) mdy = 1;
		slider->sendPosChange((float)(newx-minx)/mdx,(float)(newy-miny)/mdy);
	}
}

void MovingLabel::mouseReleaseEvent(QMouseEvent *event)
{
	if ( event->button() == LeftButton ) isMoving = false;
}

Gui_SLIDER::Gui_SLIDER()
{
	printf("constructor\n");
	Widget = 0;
	SLWidget = 0;
	ParentWidget = 0;
}

Gui_SLIDER::~Gui_SLIDER()
{
	if(Widget) delete Widget;
	if(SLWidget) delete SLWidget;
}

void Gui_SLIDER::widgetDestroyed(QWidget *widget)
{
	if(widget == Widget) { Widget = 0; return; }
	if(widget == SLWidget) { SLWidget = 0; return; }
	assert(0); // what else?
}

void Gui_SLIDER::publish(Arts::ArtsServer *Server, Arts::ModuleBroker *Broker)
{
	Arts::ModuleInfo info;

	info.name = CORBA::string_dup("Gui_SLIDER");
	info.isInterface = false;
	info.isStructure = false;

	addPort(info,"parent",Arts::input,Arts::audio_data,Arts::stream);
	addPort(info,"x",Arts::input,Arts::audio_data,Arts::property);
	addPort(info,"y",Arts::input,Arts::audio_data,Arts::property);
	addPort(info,"bgpixmap",Arts::input,Arts::string_data,Arts::property);
	addPort(info,"slpixmap",Arts::input,Arts::string_data,Arts::property);
	addPort(info,"min",Arts::input,Arts::audio_data,Arts::property);
	addPort(info,"max",Arts::input,Arts::audio_data,Arts::property);
	addPort(info,"initial",Arts::input,Arts::audio_data,Arts::property);
	addPort(info,"value",Arts::output,Arts::audio_data,Arts::stream);

	Broker->publishModule(Server, info);
}

void *Gui_SLIDER::Creator()
{
	Gui_SLIDER *result = new Gui_SLIDER;
	return (GuiModule *)result;
}

ModuleClient MC_Gui_SLIDER(GuiModule::get_MS,"Gui_SLIDER",Gui_SLIDER::Creator);

void Gui_SLIDER::applyParameters(Arts::ModuleDesc *desc)
{
	float initialvalue;
	KIconLoader iconloader;

	string bg = getStringParam(desc,"bgpixmap");
	if(strlen(bg.c_str()))
	{
		QString iconname = QString(bg.c_str())+".xpm";
		bgpixmap = iconloader.loadIcon(iconname);
	}

	string sl = getStringParam(desc,"slpixmap");
	if(strlen(sl.c_str()))
	{
		QString iconname = QString(sl.c_str())+".xpm";
		slpixmap = iconloader.loadIcon(iconname);
	}

	getPosition(desc,x,y);

	minvalue = getFloatParam(desc,"min");
	printf("got parameter minvalue %f\n",minvalue);
	maxvalue = getFloatParam(desc,"max");
	printf("got parameter minvalue %f\n",maxvalue);

	initialvalue = getFloatParam(desc,"initial");
	if(initialvalue < minvalue) initialvalue = minvalue;
	if(initialvalue > maxvalue) initialvalue = maxvalue;
	currentvalue = initialvalue;
	printf("SLIDER: applyParams; value = %f\n",currentvalue);

	setupParent(desc,"parent");

	if(!Widget)
	{
		Widget = new QLabel;
		watchWidget(Widget);
	}
	Widget->setFixedSize(bgpixmap.width(), bgpixmap.height());

	ysize = bgpixmap.height()-slpixmap.height();
	printf("SLIDER: applyParams; ysize = %d\n",ysize);
	if(!SLWidget)
	{
		int mlx = x+(bgpixmap.width()-slpixmap.width())/2;

		printf("mlc>\n");
		SLWidget = new MovingLabel(mlx,y,mlx,y+ysize,this);
		printf("mlc<\n");
		watchWidget(SLWidget);
	}


	SLWidget->setFixedSize(slpixmap.width(), slpixmap.height());
	SLWidget->setBackgroundPixmap(slpixmap);

	valuePort = lookupPort(desc,"value");
}

void Gui_SLIDER::setParent(QWidget *parent, QBoxLayout *layout)
{
	static QBitmap a,b;
	if(Widget)
	{
		printf("recreating widget...\n");
		Widget->recreate(parent,0,QPoint(x,y));

		ypos = ysize-
			(int)(((currentvalue-minvalue)/(maxvalue-minvalue))*ysize+0.5);

		SLWidget->recreate(parent,0,
					QPoint(x+(bgpixmap.width()-slpixmap.width())/2,y+ypos));
		if(layout)
		{
			printf("adding widget to layout\n");
			layout->addWidget(Widget);
		}
		if(bgpixmap.height())
		{
			a = *bgpixmap.mask();
			Widget->setMask(a);
			Widget->setBackgroundPixmap(bgpixmap);
		}
		else
			printf("missing slider background pixmap!\n");
		if(slpixmap.height())
		{
			b = *slpixmap.mask();
			SLWidget->setMask(b);
			SLWidget->setBackgroundPixmap(slpixmap);
		}
		else
			printf("missing slider pixmap!\n");
		printf("ok.\n");
	}
	ParentWidget = parent;
}

void Gui_SLIDER::finalize()
{
	printf("SLIDER: finalize; value = %f\n",currentvalue);
	emitSignal(valuePort,currentvalue);
}

void Gui_SLIDER::start()
{
	Widget->show();
	SLWidget->show();
}

void Gui_SLIDER::sendPosChange(float x, float y)
{
	currentvalue = ((1-y) * (maxvalue - minvalue))+minvalue;
	emitSignal(valuePort,currentvalue);
}

Arts::StringSeq *Gui_SLIDER::saveSessionParameters(list<long>&)
{
	Arts::StringSeq *params = new Arts::StringSeq;

	sqprintf(params,"value=%f",currentvalue);
	return params;
}

void Gui_SLIDER::restoreSessionParameters(const Arts::StringSeq& params)
{
	unsigned long i;
   	char *cmd,*param;
   	for(i=0;i<params.length();i++)
   	{
   		if(parse_line(params[i],cmd,param))   // otherwise: empty or comment
   		{
   			if(strcmp(cmd,"value") == 0) {
   				currentvalue = atof(param);
			}
		}
   	}                                                                           
}
