#include <limits.h>
#include "engine.h"
#include "math.h"

// Some static functions which are needed
static long lround(double s)
{
  if (s > LONG_MAX)
    return LONG_MAX;
  else {
    if (s < LONG_MIN)
      return LONG_MIN;
    else
      return static_cast<long>(floor(s + 0.5));
  }
}


static long mul4b(unsigned short b1, unsigned short b2, unsigned short b3, unsigned short b4)
{
  long r;

  r = b1;
  r *= b2;
  r *= b3;
  r *= b4;
  return r;
}

Vector::Vector()
{
  x = y = z = 0.0;
}


Vector::Vector(double X, double Y, double Z)
{
  x = X; y = Y; z = Z;
}


double Vector::norm()
{
  return sqrt(x * x + y * y + z * z);
}


void Vector::normalize()
{
  double n;

  n = norm();
  Engine::errEps(n);
  x /= n;
  y /= n;
  z /= n;
}


Vector Vector::operator+(Vector a)
{
  return Vector(x+a.x, y+a.y, z+a.z);
}


Vector Vector::operator-(Vector a)
{
  return Vector(x-a.x, y-a.y, z-a.z);
}


Vector Vector::operator*(double k)
{
  return  Vector(x*k, y*k, z*k);
}


Vector Vector::operator+=(Vector a)
{
  x += a.x; y += a.y; z += a.z;
  return *this;
}


Vector Vector::operator-=(Vector a)
{
  x -= a.x; y -= a.y; z -= a.z;
  return *this;
}


Vector Vector::operator*=(double k)
{
  x = x*k; y = y*k; z = z*k;
  return (*this);
}


double Vector::operator*(Vector a)
{
  return (a.x * x + a.y * y + a.z * z);
}


double Vector::Noise(const LatTyp& lattice)
{
  unsigned short b100, b010, b001, b101, b110, b011, b111, b000, 
    x0, y0, z0, x1, y1,	z1;
  long lx, ly, lz, xs, ys, zs;
  Vector v;

  v =(*this)*(double)(128*LatCnt);
  lx = lround(v.x);
  ly = lround(v.y);
  lz = lround(v.z);
  xs = ((unsigned long)lx) >> 7;
  ys = ((unsigned long)ly) >> 7;
  zs = ((unsigned long)lz) >> 7;
  x0 = xs & LatMask;
  y0 = ys & LatMask;
  z0 = zs & LatMask;
  x1 = (xs + 1) & LatMask;
  y1 = (ys + 1) & LatMask;
  z1 = (zs + 1) & LatMask;
  b000 = lattice[x0][y0][z0];
  b001 = lattice[x0][y0][z1];
  b010 = lattice[x0][y1][z0];
  b011 = lattice[x0][y1][z1];
  b100 = lattice[x1][y0][z0];
  b101 = lattice[x1][y0][z1];
  b110 = lattice[x1][y1][z0];
  b111 = lattice[x1][y1][z1];
  x1 = lx & 0x7f;
  y1 = ly & 0x7f;
  z1 = lz & 0x7f;
  x0 = 0x80 - x1;
  y0 = 0x80 - y1;
  z0 = 0x80 - z1;
  return ((mul4b(b000, x0, y0, z0) + mul4b(b001, x0, y0, z1) 
	   + mul4b(b010, x0, y1, z0) + mul4b(b011, x0, y1, z1)
	   + mul4b(b100, x1, y0, z0) + mul4b(b101, x1, y0, z1) 
	   + mul4b(b110, x1, y1, z0) + mul4b(b111, x1, y1, z1)) 
	  * nscal - 1);
}

struct LOC_dNoise {
  long xs, ys, zs;
} ;

static unsigned short GetDif(unsigned short k, unsigned short i, 
			     struct LOC_dNoise *LINK, const LatTyp& lattice)
{
  static unsigned short oM[4][8][3] = {
    { { 1, 0, 0 },
      { 1, 0, 1 },
      { 1, 1, 0 },
      { 1, 1, 1 },
      { 2, 0, 0 },
      { 2, 0, 1 },
      { 2, 1, 0 },
      { 2, 1, 1 } },
    { { 0, 1, 0 },
      { 0, 1, 1 },
      { 0, 2, 0 },
      { 0, 2, 1 },
      { 1, 1, 0 },
      { 1, 1, 1 },
      { 1, 2, 0 },
      { 1, 2, 1 } },
    { { 0, 0, 1 },
      { 0, 0, 2 },
      { 0, 1, 1 },
      { 0, 1, 2 },
      { 1, 0, 1 },
      { 1, 0, 2 },
      { 1, 1, 1 },
      { 1, 1, 2 } },
    { { 0, 0, 0 },
      { 0, 0, 1 },
      { 0, 1, 0 },
      { 0, 1, 1 },
      { 1, 0, 0 },
      { 1, 0, 1 },
      { 1, 1, 0 },
      { 1, 1, 1 } }
  };

  return ((lattice[(LINK->xs + oM[k][i][0]) & LatMask]
	   [(LINK->ys + oM[k][i][1]) & LatMask]
	   [(LINK->zs + oM[k][i][2]) & LatMask] - lattice[(LINK->xs + oM[3][i]
							   [0]) & LatMask]
	   [(LINK->ys + oM[3][i][1]) & LatMask]
	   [(LINK->zs + oM[3][i][2]) & LatMask]) & 255);
}

Vector Vector::dNoise(const LatTyp& lattice)
{
  struct LOC_dNoise V;
  unsigned short x0, y0, z0, x1, y1, z1, k;
  long lx, ly, lz;
  Vector v, n;
  double h;

  
  v = (*this) * (double)scal;
  lx = lround(v.x);
  ly = lround(v.y);
  lz = lround(v.z);
  V.xs = ((unsigned long)lx) >> 7;
  V.ys = ((unsigned long)ly) >> 7;
  V.zs = ((unsigned long)lz) >> 7;
  x1 = lx & 0x7f;
  y1 = ly & 0x7f;
  z1 = lz & 0x7f;
  x0 = 0x80 - x1;
  y0 = 0x80 - y1;
  z0 = 0x80 - z1;

  for (k = 0; k <= 2; k++) {
    h = (mul4b(GetDif(k, 0, &V, lattice), x0, y0, z0) 
	 + mul4b(GetDif(k, 1, &V, lattice), x0, y0, z1) 
	 + mul4b(GetDif(k, 2, &V, lattice), x0, y1, z0) 
	 + mul4b(GetDif(k, 3, &V, lattice), x0, y1, z1) 
	 + mul4b(GetDif(k, 4, &V, lattice), x1, y0, z0)
	 + mul4b(GetDif(k, 5, &V, lattice), x1, y0, z1) 
	 + mul4b(GetDif(k, 6, &V, lattice), x1, y1, z0) 
	 + mul4b(GetDif(k, 7, &V, lattice), x1, y1, z1)) * nscal - 1;
    if (k == 0)
      n.x = h;
    else if (k == 1)
      n.y = h;
    else
      n.z = h;
  }
  return n;
}


double Vector::turbulence(const LatTyp& lattice)
{
  double t, s, noise;
  Vector p(*this);

  t = 0.0;
  s = 1.0;
  while (s > mins) {
    noise = p.Noise(lattice);
    t += fabs(noise * s);
    s *= 0.5;
    p *= 2.0;
  }
  return t;
}


