/***************************************************************************
                          mapwidget.h
                      -------------------
    description          : Map Display Widget
    begin                : Wed Oct 20 1999
    copyright            : (C) 1999 by John-Paul Stanford
    email                : john-paul.stanford@virgin.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef CMAPWIDGET_H
#define CMAPWIDGET_H

#include <kapp.h>
#include <klocale.h>

#include <qscrollview.h>
#include <qpixmap.h>
#include <qlist.h>
#include <qpopupmenu.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <qcursor.h>

#include "resource.h"
#include "kmuddoc.h"
#include "cmapelement.h"

// PROTOTYPEmaeS

class CMapRoom;

// CLASSES

class CMapText : public CMapElement
{
public:
	
	CMapText (QString str,int x,int y,QFont font);

	~CMapText();

	virtual elementTyp getElementType(void) { return TEXT; }		
	void setText(QString str);
	QString getText(void);
	void setLevel(int level);
	int getLevel(void);
	void setFont(QFont f);
	QFont getFont(void);
	QColor getColour(void);
	void setColour(QColor colour);
	virtual void paint(QPainter *p);
	virtual void dragPaint(QPainter *p,int offsetx,int offsety);
	virtual void lowerPaint(QPainter *p);
	virtual void higherPaint(QPainter *p);
	virtual CMapElement *copy(void);

private:
	QColor col;
	QString text;
	QFont font;
	int lvl;
};

/**This class is used to store the details of each path on
  *the map (The exits from one room to another)
  *@author John-Paul Stanford
  */
class CMapPath : public CMapElement
{
public:

	/** This is the constuctor for the map paths and
	  * used to create a path
	  */
	CMapPath (direction SrcDir,direction DestDir,CMapRoom *DestRoom,CMapRoom *SrcRoom)
	{
		srcDir = SrcDir;
		destDir = DestDir;
		destRoom = DestRoom;
		srcRoom  = SrcRoom;
		setSelected(false);
	}	

	/** Returns true if the path is two way other wise returns false */ 	
        bool isTwoWay(void);

	/** Get the directions of the start and finish directions of the path */
	direction getSrcDir(void)							{ return srcDir; }
	direction getDestDir(void)						{ return destDir; }

	/** Get a pointer to the destination room */
	CMapRoom *getDestRoom(void)						{ return destRoom; }
	CMapRoom *getSrcRoom(void)						{ return srcRoom; }

	/** Set the start and finish directrions of the path */
	void setSrcDir(direction SrcDir)			{ srcDir = SrcDir; }
	void setDestDir(direction DestDir)		{ destDir = DestDir; }

	/** Set a pointer to the destination room */
	void setDestRoom(CMapRoom *DestRoom)	{ destRoom = DestRoom; }	
	void setSrcRoom(CMapRoom *SrcRoom)    { srcRoom = SrcRoom; }
	
	virtual void paint(QPainter *p);
	virtual void dragPaint(QPainter *p,int offsetx,int offsety);
	virtual void lowerPaint(QPainter *p);
	virtual void higherPaint(QPainter *p);
	virtual elementTyp getElementType(void)      { return PATH; }
	virtual CMapElement *copy(void);

private:	
	/** These are used to store the directions the path enters and leaves a room */
	direction srcDir,destDir;	

	/** Used to store a pointer to the desination room */
	CMapRoom *destRoom,*srcRoom;
};

/**This class is used to store the details of each room
  *on the map
  *@author John-Paul Stanford
  */
class CMapRoom : public CMapElement
{
public:                                                                             	
	/** Constructor which is used to create a new room */
	CMapRoom( int type );

	/** Uesd tp set the room type */
	void setType(int type)	{ roomType = type; }
	/** Used to get the room type */
	int  getType(void)	{ return roomType; }

	/** Used to add a exit from the room */
	void addPath(CMapPath *path);
	/** Used to find the path for the give direction. If a path can't be found then
	    null is returned                                                             */
	CMapPath *getPathDirection(direction dir);

	void setLevel(signed int level)	{ mapLevel = level; }
	signed int  getLevel(void)			{ return mapLevel;  }

	void setX(int x);			
	void setY(int y);		
	int  getX(void)				{ return xpos; }
	int  getY(void)				{ return ypos; }

	void setCurrentRoom(bool currentRoom);
	bool getCurrentRoom(void)				{ return current; }
	void setLoginRoom(bool loginRoom)		{ login = loginRoom; }
	bool getLoginRoom(void)				{ return login; }
	
	virtual void paint(QPainter *p);
	virtual void dragPaint(QPainter *p,int offsetx,int offsety);
	virtual void lowerPaint(QPainter *p);
	virtual void higherPaint(QPainter *p);
	virtual elementTyp getElementType(void)      { return ROOM; }
	virtual CMapElement *copy(void);
		
	void moveBy(int x, int y);		
	
	void setMoveTime(signed int time)         { moveTime = time; }
	signed int getMoveTime(void)              { return moveTime; }	
	
	void setLabel(QString str)		{ label = str; }
	QString getLabel(void)			{ return label; }
	void setDescription (QString str)	{ description = str; }
	QString getDescription (void)		{ return description; }
	void setColour(QColor col)              { colour = col; }
	QColor getColour(void)			{ return colour; }
	void setLoginColour(QColor col)		{ loginCol = col; }
	QColor getLoginColour(void)		{ return loginCol; }
	void setUseDefaultCol(bool b)		{ useDefaultCol = b; }
	bool getUseDefaultCol(void)		{ return useDefaultCol; }
	
public:
	/** Used to store details of all the paths from this room */
	QList<CMapPath> pathList;
	QList<CMapPath> connectingPaths;	

private:
	bool useDefaultCol;
	QColor colour,loginCol;
	QString label;
	QString description;
	
	signed int moveTime;
	/** Used to store the position of the room on the map */
	int xpos,ypos;

 	/** Used to store the type of room */
	int roomType;

	bool current;
	signed int  mapLevel;
	bool login;
};

/**This class is used to store all the different maps for the mud.
  *For example, the maps for different levels
  *@author John-Paul Stanford
  */
class CMapList
{
public:
	/** The constructor for the map list */
	CMapList() { }
	/** Get the list of rooms */
	QList<CMapRoom> *getRoomList(void) 		{ return &roomList; }
	QList<CMapText> *getTextList(void)		{ return &textList; }
private:
	QList<CMapRoom> roomList;
	QList<CMapText> textList;
};


struct selectedTyp
{
	CMapElement *element;
	int x;
	int y;
};

/**This class is used to display dynamic tool tips
  *for the maps. This enables the map widget to show
  *room labels as tool tips.
  *@author John-Paul Stanford
  */
class CMapToolTip : public QToolTip
{
public:
    CMapToolTip( QWidget * parent );

protected:
    void maybeTip( const QPoint & );
};

/**This is the widget used to display the map
  *@author John-Paul Stanford
  */
class CMapWidget : public QWidget
{

	Q_OBJECT
public:
	/** The map widget constructor */
	CMapWidget(KmudDoc *document,QString mudTitle,CMudProfile* mudPro,QScrollView *view,QWidget *parent=0);
	/** The Map widget deconstructor */
	~CMapWidget();

	/** Used to get the tool tip for a given location passed in p */
	QString CMapWidget::getTip(const QPoint p);
	
	/** These methods are used to set the colour values */
	void setBackgroundColour(QColor colour);
	void setGridColour(QColor colour);
	void setElementsColour(QColor loPath,QColor defPath,QColor hiPath,QColor loRoom,QColor defRoom,QColor hiRoom,QColor login,QColor sel,QColor defText);

	/** These methods are used to get the colour values being used */
	QColor getBackgroundColour(void);
	QColor getGridColour(void);
	QColor getDefaultRoomColour(void);
	QColor getDefaultPathColour(void);
	QColor getHigherRoomColour(void);
	QColor getHigherPathColour(void);
	QColor getLowerRoomColour(void);
	QColor getLowerPathColour(void);
	QColor getDefaultTextColour(void);	
	QColor getLoginColour(void);	
	QColor getSelectedColour(void);		

	/** Used to set the default path type */
	void setDefaultPathTwoWay(bool set);
	/** Used to return the default path type */
	bool getDefaultPathTwoWay(void);
	/** Used to create a new level of the map */
	void createLevel(direction dir);	
	/** This checks the given size and resizes if needed */		
	void checkSize(int x,int y);
	/** Used to get the current level number */
	int getCurrentLevel(void);
	/** Used to export a map to a file */
	void exportMap(QString filename);
	/** Used to import a map a file */
	int importMap(QString filename);
	/** Used to set  the current tool */
	void setTool(int tool);
	/** Returns the visable state of the grid */
	bool GridVisable(void);
	/** Turns the grid on or off */
	void setGridVisable(bool visable);	
	/** Set overview visable */
	void setViewOverview(bool visiable);
	/** Returns the current room that the player is in */
	CMapRoom *getCurrentRoom(void);	
	/** This method is used to load a map */
	void loadMap(void);
	/** This method is used to save a map */
	void saveMap(void);
	/** This method is used to move the map by the given vector */
	void moveMap(signed int x,signed int y);
	/** Used to create a test map */
	void createDummyMap(void);
	/** This method is used to erase a map from memory */
	void eraseMap(void);
	/** the showLevel method is used to display a level on the map */
	void showLevel(int startx, int starty, signed int level);
	/** move the player relative to the current position of the player */
	void movePlayerBy(direction dir,bool create);
	/** Create a new map */
	void createNewMap();
	/** Turn on and off the display of the higher map */
	void setViewHigherMap(bool visiable);
	/** Turn on and off the display of the lower map */
	void setViewLowerMap(bool visiable);
	/** Get the totals for the map */
	void getTotals(int *lvl,int *room, int *path, int *text);
	/** Get the login room */
	CMapRoom *getLoginRoom(void);
	/** Used to store the max x,y location of the rooms */
	int xMax, yMax;
	/** This method is used to find a room at a given location in the current level of the map.
	  * Null is retured if the room can't be found otherwise a pointer is returned to the room.
	  */
	CMapRoom *findRoomAt(int x,int y,signed int level);
	void setSpeedWalkAbortActive(bool set);
	bool getSpeedWalkAbortActive(void);
	void setSpeedWalkLimit(int limit);
	int getSpeedWalkLimit(void);

public slots:
	/** Used to change the properties of the text elements */
	void slotTextProperties(void);
	/** Used to change a path to a twoway path */
	void slotPathSetOneWay(void);
	/** Used to change a path to one way */
	void slotPathSetTwoWay(void);
	/** Used display the path properties */
	void slotPathProperties(void);
	/** Move the player to the selected position on the map */
	void slotSetCurrentPos(void);
	/** Called when the user selects delete from the room menu */
	void slotDelElement(void);
	/** Called when teh user selects create from the room menu */
	void slotCreateRoom(void);
	/** switch argument for slot selection by menu or toolbar ID */
	void commandCallback(int id_);
	/** switch argument for Statusbar help entries on slot selection */
	void statusCallback(int id_);
	/** change the status message of the whole statusbar temporary */
	void slotStatusHelpMsg(const char *text);
	/** This slot is called when the copy menu option is selected */
	void slotCopy(void);
	/** This slot is called when the paste menu option is selected */
	void slotPaste(void);
	/** This slot is called when the cut menu option is selected */
	void slotCut(void);
	/** This slot is called when the delete menu option is selected */
	void slotDelete(void);
	/** This slot is called when dragging is to be started */
	void slotStartDraging(void);
	/** This slot is used to select all elements */
	void slotSelectAll(void);
	/** This slot is used to deselect all elements */
	void slotDeselectAll(void);
	/** This slot is used to invert the elements selected */
	void slotInvertSelection(void);
	/** This slot is called when a room is to be repositioned */
	void slotMoveElement(QMouseEvent *e);
	/** This is used to display the contex menu for map elements */
	void slotContexMenu(QMouseEvent *e);
	/** The select tool mouse press event */
	void slotSelectMousePress(QMouseEvent *e);
	/** The select tool mouse move event */
	void slotSelectMouseMove(QMouseEvent *e);
	/** The select tool mouse release event */
	void slotSelectMouseRelease(QMouseEvent *e);
	/** Used to speed walk a player */
	void slotMovePlayerTo(void);
	/** Used to view the next level up */
	void slotLevelUp(void);
	/** Used to view the next level down */
	void slotLevelDown(void);
	/** Used to set the login position for a character */
	void slotSetLogin(void);
	/** This slot is used to set the properties of a room */
	void slotRoomProperties(void);
	/** This is called when a charcter is able to move again */
	void slotMoveTimerDone(void);

protected:
	/** A pointer to the mud profile */
	CMudProfile* mudProfile;
	/** draw the map widget */
	void paintEvent(QPaintEvent *pe);
	/** The mouse release event */	
	void mouseReleaseEvent(QMouseEvent *e);
	/** The mouse press event */
	void mousePressEvent(QMouseEvent *e);
	/** Called when the mouse is being moved */
	void mouseMoveEvent(QMouseEvent *e);
	/** Called when the mouse is double clicked */
	void mouseDoubleClickEvent ( QMouseEvent * e );
	/** Called when a key is pressed */
	void keyPressEvent(QKeyEvent *e);
	/** Called when a key is released */
	void keyReleaseEvent(QKeyEvent *e);		
	
private:
	QPixmap *buffer;
	/** number of moves before speed walk is aborted  */
	int speedWalkAbortLimit;
	/** Tells the mapper if speed walk move limit is active */
	bool speedWalkAbortActive;
	QCursor *deleteCursor;
	QCursor *pathStartCursor;
	QCursor *pathEndCursor;
	CMapRoom *pathStartRoom;
	int pathToolMode;
	/** Used to tell the mapper to create two way paths by default */
	bool defaultTwoWayPath;	
	/** Used to tell the mapper if the overview pane needs to be displayed */	
	bool bOverview;
	/** A pointer to the scroll view of the map */
	QScrollView *mapView;
	/** Create a zoomed version of the map */
	void createZoomedPixmap(QPixmap *newPixmap, QPixmap pm,int w,int h);
	/** This is used to make sure there is a delay between each call of the movePlayerBy */
	QTimer *move_timer;
	/** This is used to tell if movePlayerBy has to wait for the previous move to finish */
	bool moveEnabled;
	/** The dynamic tool tip class */
	CMapToolTip *tip;
	/** The kmud document */
	KmudDoc *doc;
	/** Used to read a string from a file */
	QString readStr(QFile *f);	
	/** Used to write a string to a file */
	void writeStr(QFile *f,QString str);
	/** Used to read a int from a file */
        int readInt(QFile *f);
        /** Used to write a int to a file */
        void writeInt(QFile *f,int i);
	/** Delete a text element */
	void deleteText(CMapText *text);
	/** Used to create a text element */
	CMapText *createText(QString str,int x,int y,QFont font,QColor col,int level);
	/** Used to calulate and set the cords of a path */
	void setPathCords(CMapPath *path);
	/** Used to unslected all the elements on the map */
	bool unselectAll(void);
	/** This method is used to convert a direction into a offset */
	void directionToCord(direction dir, int distance,signed int *x,signed int *y);
	void directionToCordInverted(direction dir, int distance,signed int *x,signed int *y);	
	/** Draw the grid if it's visable */
	void drawGrid(QPainter *p);
	/** This method is used to delete a room on the map */
	void deleteRoom(CMapRoom *room);
	/** Delete a path */
	void deletePath(CMapPath *path);
	/** The filename of the map */
	QString mapName;
	/** This method is used to create a new room on the map */
	CMapRoom *createRoom(int x,int y,signed int level,int type);
	
	/** This method is used to create a path between to rooms */
	CMapPath *createPath(int srcX,int srcY,signed int srcLevel,direction srcDir,int destX,int destY,signed int destLevel,direction destDir);
	/** This points to the room that the character is standing in */
	CMapRoom *currentRoom;
	/** This is a pointer to the room that is the login room */
	CMapRoom *loginRoom;
	/** This is a list of all the rooms on the current map */
	QList<CMapRoom> *currentMap;
	/** Used to store the current level number of the map */
	int currentLevelNum;
	/** This is a list of all the maps for this mud */
	QList<CMapList> mapList;
	/** The spaceing of the gird on the Y axis */
	int YGridSpace;
	/** The spaceing of the gird on the X axis */
	int XGridSpace;
	/** Used to store the on/off state of the grid */
	int hasGrid;
	/** The context menu for the rooms */
	QPopupMenu *room_menu;
	/** The context menu for the paths */
	QPopupMenu *path_menu;
	/** The context menu for the text elements */
	QPopupMenu *text_menu;
	/** Used to regiester when the ctrl key is being held down */
	bool bCtrlPressed;
	/* This is a list of pointers pointing to all the elements on the map
	 * can be used to iterate through all the elements
	 */
	QList<CMapElement> elementList;
	/** Same as the elementList but used to paint the level above the current one */
	QList<CMapElement> elementListUpper;
	/** Same as the elementList but used to paint the level below the current one */
	QList<CMapElement> elementListLower;
	/* The element clipboard. This is used in cut/copy/paste operations */
	QList<CMapElement> elementClipboard;

	struct selectedTyp selected;
	/** Used to mesure how long a mouse putton is held down */
	QTimer *mouseDownTimer;	
	/** Used to tell if the mouse is being draged */
	bool bDragging;
	/** Uses to store the current pos for dragging positions */
	int mouseDragX,mouseDragY;
	/** Used to store the last mouse pos for dragging positions */
	int lastDragX,lastDragY;
	/** Used to paint by the move events */
	QPainter *dragPainter;
	/** Is a room being moved by dragging */
	bool moveDrag;
	/** The current tool that is being used */
	int currentTool;
	/** Used to flag if the lower map is to be displayed */
	bool bViewLowerMap;
	/** Used to flag if the upper map is to be displayed */
	bool bViewHigherMap;

	// Used to store the colour values
	QColor gridColour;
	QColor backColour;
	QColor defaultRoomColour;
	QColor defaultPathColour;
	QColor lowerRoomColour;
	QColor lowerPathColour;
	QColor higherRoomColour;
	QColor higherPathColour;
	QColor defaultTextColour;
	QColor selectedColour;
	QColor loginColour;
	
signals:
	/** will be emmited when the mapper wants to move the player */
	void movePlayer(direction dir);
	/** Used to inform of a change in level */
	void levelChange(signed int level);
	/** Used to inform when the help status bar message needs to be changed */
	void statusMsg(const char *text);
	/** Used to inform when the status bar message needs to be changed */
	void statusHelpMsg(const char *text);
};

#endif
