/*
 *   kmamerun - a interface for M.A.M.E
 *   Copyright (C) 1998  Juergen Vigna
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <unistd.h>

#include <qtooltip.h>
#include <qapp.h>
#include <qstring.h>
#include <qfiledlg.h>
#include <qregexp.h>
#include <qmsgbox.h>
#include <qfile.h>
#include <qtstream.h>
#include <qprogressbar.h>

#include <kapp.h>
#include <kkeyconf.h>
#include <ksimpleconfig.h>

#include "kmamerun.h"
#include "myprocess.h"
#include "preferences.h"

#define Inherited KMameRunData
/* Global Variables */

extern QString debugFlag;

#define LEN_GAME	20
#define LEN_NAME	25

KMameRun::KMameRun(QWidget* parent, const char* name, unsigned int sf, int so)
	:Inherited( parent, name )
{
    debugFlag = getenv("DEBUG");
    setCaption( "KMameRun V" VERSION "(loading ...)");
    // add some tooltips
    current_directory = QDir::currentDirPath();
    error_listbox = new LogListBox();
    // set labels for i18n
    labelGames->setAutoResize(true);
    labelGames->setText(i18n("Games"));
    // init defaults
    qml = 0;
    showFlag = sf;
    showLogfile = false;
    loadListbox();
    setSortMode(so);
    gameHighlightCB(gameListBox->text(gameListBox->currentItem()));
}


KMameRun::~KMameRun()
{
}

void KMameRun::RunGameCB()
{
    RunGame();
}

int KMameRun::RunGame(const char *item=0)
{
    static int
        numgame = 0;
    QString
        mamecmd,
        logdir,
        filename,
        logfile,
        cmd;
    QFile
        ff,fl;
    Preferences
        prefs;

    mamecmd = prefs.prefGeneral.mamecmd;
    logdir = prefs.prefGeneral.logdir;
    if (mamecmd.isEmpty()) {
        QString str;
        str = "M.A.M.E program command not specified!";
        QMessageBox::critical(this,"KMameRun",str.data());
        return false;
    }
    if (!item) {
        cmd = gameListBox->text(gameListBox->currentItem());
        if (cmd.isEmpty()) {
            QMessageBox::critical(this,"KMameRun","No game selected");
            return false;
        }
        gameName = cmd.mid(gameNameFullLen+1,gameNameLen);
    } else
        gameName = item;
    numgame++;
    logfile.sprintf("%s/kmamerun%d.log",logdir.data(),getpid());
    fl.setName(logfile);
    cmd = mamecmd.copy();
    cmd.replace(QRegExp("@@Game"),gameName);
    if (cmd == mamecmd) {
        QString str;
        str = "No game selected or missing @@Game-option in\n"
            "M.A.M.E programm command";
        QMessageBox::critical(this,"KMameRun",str.data());
        return false;
    }
    QString
        opt = prefs.optionString();
    if (!opt.isEmpty()) {
        cmd += " ";
        cmd += opt;
    }
    if (!debugFlag.isEmpty())
        fprintf(stderr,"CMD: %s\n",cmd.data());
    MyProcess
        *proc = new MyProcess;

    *proc << cmd.data();
    connect(proc, SIGNAL(processExited(KProcess *)),this,
            SLOT(RunGameDone(KProcess*)));
    connect(proc, SIGNAL(receivedStderr(KProcess *,char*,int)),this, 
            SLOT(ReadStderr(KProcess*,char*,int)));
    connect(proc, SIGNAL(receivedStdout(KProcess *,char*,int)),this, 
            SLOT(ReadStdout(KProcess*,char*,int)));
    if (!error_listbox->isVisible() && showLogfile)
        error_listbox->show();

    bool result = proc->start(KProcess::NotifyOnExit , KProcess::All);
    if(!result){
        QString str;
        str.sprintf("Cannot start a new program\nfork() failed.");
        QMessageBox::critical(this,"KMameRun",str.data());
        if (proc) {
            if(proc->isRunning())
                proc->kill();
            delete proc;
        }
        return false;
    }
    return true;
}


void KMameRun::ReadStderr(KProcess *, char *buffer, int buflen)
{
    static QString
        str;
    QString
        s;
    int
        pos;

    s.sprintf("%.*s",buflen,buffer);
    str += s;
    pos = str.find('\n');
    while(pos >= 0) {
        pos++;
        s = str.left(pos);
        error_listbox->insertItem(s);
        str = str.right(str.length()-pos);
        pos = str.find('\n');
    }
}

void KMameRun::ReadStdout(KProcess *, char *buffer, int buflen)
{
    static QString
        str;
    QString
        s;
    int
        pos;

    s.sprintf("%.*s",buflen,buffer);
    str += s;
    pos = str.find('\n');
    while((str.length() > 1) && pos >= 0) {
        pos++;
        s = str.left(pos);
        error_listbox->insertItem(s);
        str = str.right(str.length()-pos);
        pos = str.find('\n');
    }
}

void KMameRun::RunGameDone(KProcess *proc)
{
    MyProcess *mproc = (MyProcess*)proc;

    if (mproc->localData())
        remove(mproc->localData());
    if (!debugFlag.isEmpty())
        printf("Called RunGameDone\n");
    if (proc) {
        if(proc->isRunning())
            proc->kill();
    }
    delete proc;
}

void KMameRun::resizeEvent (QResizeEvent *rse)
{
    QSize
        sz;
    int
        width,height,minh;

    sz = rse->size();
    minh = gameListBox->minimumSize().height();
    width = sz.width() - 20;
    height = sz.height() - gameListBox->y() - 10;
    if (height < minh)
        height = minh;
    gameListBox->setGeometry(gameListBox->x(), gameListBox->y(), width, height);
}

int KMameRun::totMinimumHeight() 
{
    return gameListBox->y() + gameListBox->minimumSize().height();
}

void KMameRun::loadListbox()
{
    QString
        mamecmd,
        logdir,
	mamedir,
        message = "Error in call to M.A.M.E -listfull:\n",
        filename,
        logfile,
        cmd;
    QFile
        ff,fl;
    Preferences
        prefs;

    if (qml)
        delete qml;
    qml = new QMyList();
    gameListBox->clear();
    mamecmd = prefs.prefGeneral.mamecmd;
    logdir = prefs.prefGeneral.logdir;
    mamedir = prefs.prefGeneral.mamedir;
    if (mamecmd.isEmpty()) {
        QString str;
        str = "M.A.M.E program command not specified!";
        QMessageBox::critical(this,"KMameRun",str.data());
        return;
    }
    logfile.sprintf("%s/kmamerun%d.log",logdir.data(),getpid());
    filename.sprintf("%s/kmamerun%d.tmp",logdir.data(),getpid());
    fl.setName(logfile);
    cmd.sprintf("%s >%s 2>%s",mamecmd.data(),filename.data(),logfile.data());
    cmd.replace(QRegExp("@@Game"),"-listfull");
    ff.setName(filename);
    if (!debugFlag.isEmpty())
        fprintf(stderr,"MAME-CMD: %s\n",(const char *)cmd);
    system(cmd);
    if (ff.size() <= 0) {
        if ((fl.size() > 0) && fl.open(IO_ReadOnly)) {
            QTextStream t(&fl);
            message += t.readLine();
            fl.close();
        }
        QMessageBox::warning(this,"Error in calling MAME-Listfull",message);
        remove(logfile);
        remove(filename);
        return;
    }
    remove(logfile);
    if (!ff.open( IO_ReadOnly ))
        return;
    QTextStream
        t(&ff);
    QString
        gname,
        gnamefull,
        line,
        str;
    QMyListType
        qmt, *item;
    char
        buf[256];
    FILE
        *fp;
    QProgressBar
        qpb(ff.size(),0,0,WStyle_Customize | WStyle_DialogBorder);
    qpb.setCaption(i18n("KMameRun (working...)"));
    qpb.setGeometry(mapToGlobal(QPoint(0,0)).x()+width()/2-100,
                    mapToGlobal(QPoint(0,0)).y()+height()/2-10,
                    200,20);
    qpb.show();
    gameNameLen = gameNameFullLen = 0;
    str = t.readLine(); // read first line
    str = t.readLine();
    qml->setSortMode(QMyList::SortPriority);
    while (!t.eof()) {
        qpb.setProgress(ff.at());
        sscanf(str.data(),"%s",buf);
        gname = buf;
#if VERIFY_ROMS
        cmd.sprintf("%s %s >>/tmp/kmr.log 2>>/tmp/kmr.err",mamecmd.data(),gname.data());
        cmd.replace(QRegExp("@@Game"),"-verifyroms");
        if (!debugFlag.isEmpty())
            fprintf(stderr,"MAME-CMD: %s\n",(const char *)cmd);
        fprintf(stderr,"%s:%d\n",gname.data(),system(cmd));
#else
        if (showFlag) { // showFlag == 0 means all
            cmd.sprintf("%s/roms/%s.zip",mamedir.data(),gname.data());
            if (!(fp=fopen(cmd.data(),"r"))) {
                cmd.sprintf("%s/%s",mamedir.data(),gname.data());
                fp=fopen(cmd.data(),"r");
            }
            if (fp)
                fclose(fp);
            if ((!fp && (showFlag == 1)) || // means all available games
                (fp && (showFlag == 2))) {  // means all not available games
                str = t.readLine();
                continue;
            }
        }
#endif
        if (gameNameLen < gname.length())
            gameNameLen = gname.length();
        sscanf(str.data(),"%*s \"%[^\"]\"",buf);
        gnamefull = buf;
        if (gameNameFullLen < gnamefull.length())
            gameNameFullLen = gnamefull.length();
        qmt.set(gname,gnamefull,getGameType(gname), getGamePriority(gname));
        qml->inSort(&qmt);
        str = t.readLine();
    }
    ff.close();
    remove(filename);
    gameListBox->setAutoUpdate(false);
    gameListBox->clear();
    item = qml->first();
    while(item) {
        str = item->string(gameNameLen,gameNameFullLen);
        gameListBox->insertItem(str.data());
        item = qml->next();
    }
    gameListBox->setAutoUpdate(true);
    gameListBox->repaint();
}

int KMameRun::reloadListbox(unsigned int mode, bool force)
{
    if (qml && (mode == showFlag) && !force)
        return qml->count();
    showFlag = mode;
    loadListbox();
    return qml->count();
}

void KMameRun::reloadListbox()
{
    QMyList
        list;
    QMyListType
        *item;
    QString
        act,
        str;
    int
        current=0,
        i=0;

    list.setSortMode(qml->sortMode());
    item = qml->first();
    act = gameListBox->text(gameListBox->currentItem());
    act = act.mid(gameNameFullLen+1,gameNameLen);
    act = act.stripWhiteSpace();
    while(item) {
        item->set(item->name(),item->fullname(),
                  getGameType(item->name()), getGamePriority(item->name()));
        list.inSort(item);
        item = qml->next();
    }
    gameListBox->setAutoUpdate(false);
    gameListBox->clear();
    item = list.first();
    while(item) {
        str = item->string(gameNameLen,gameNameFullLen);
        if (!current) {
            if (act == item->name())
                current = i;
            i++;
        }
        gameListBox->insertItem(str.data());
        item = list.next();
    }
    gameListBox->setCurrentItem(current);
    gameListBox->centerCurrentItem();
    gameListBox->setAutoUpdate(true);
    gameListBox->repaint();
}

void KMameRun::setSortMode(int mode)
{
    QMyList::SortMode
        sm = qml->sortMode();

    switch(mode) {
      case 0:
          qml->setSortMode(QMyList::SortName);
          break;
      case 1:
          qml->setSortMode(QMyList::SortNameFull);
          break;
      case 2:
          qml->setSortMode(QMyList::SortType);
          break;
      case 3:
          qml->setSortMode(QMyList::SortPriority);
          break;
      default:
          qml->setSortMode(QMyList::SortNameFull);
    }
    if (sm != qml->sortMode())
        reloadListbox();
}

void KMameRun::gameSelectCB(const char *item)
{
    RunGame(QString(item).mid(gameNameFullLen+1,gameNameLen));
}

void KMameRun::gameHighlightCB(const char *item)
{
    emit gameSelected(item, gameNameFullLen);
}

void KMameRun::setPos(QLabel *lab, QLineEdit *le)
{
    int pos, diff;

    pos = lab->x() + lab->width() + 5;
    diff = pos - le->x();
    if (diff > 0)
        le->setGeometry(pos, le->y(), le->width() - diff, le->height());
}

void KMameRun::toggleLogfile()
{
    showLogfile = !error_listbox->isVisible();
    if (showLogfile)
        error_listbox->show();
    else
        error_listbox->hide();
}

int KMameRun::getGameType(const char *cname)
{
    QString
        cgroup = cname,
        aConfigName = KApplication::localconfigdir() + "/kmamerunrc";

    cgroup += "Game";
    KSimpleConfig config(aConfigName);
    config.setGroup(cgroup);
    if (config.hasKey("type"))
        return config.readNumEntry("type");
    return MAXTYPE;
}

int KMameRun::getGamePriority(const char *cname)
{
    QString
        cgroup = cname,
        aConfigName = KApplication::localconfigdir() + "/kmamerunrc";

    cgroup += "Game";
    KSimpleConfig config(aConfigName);
    config.setGroup(cgroup);
    if (config.hasKey("priority"))
        return config.readNumEntry("priority");
    return MAXPRIORITY;
}
