/*
  Copyright(C) 2002-2007 Pierre Mazire
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
  printrom.c
  
  Print roms informations
*/

#include "common.h"
#include <mamory/readinfo.h>
#include <mamory/rommngr.h>
#include <mamory/printrom.h>

void fprintGamesListMamoryHeader(FILE *fp,s_GamesList *GamesList)
{
  unsigned int i;

  for(i=0;i<GamesList->NbrSources;i++)
    {
      fprintf(fp,"mamory (\n");
      switch(GamesList->Sources[i]->Type)
	{
	case TYPE_EXEC:
	  fprintf(fp,"\ttype exec\n");
	  break;
	case TYPE_ADMAMELI:
	  fprintf(fp,"\ttype admameli\n");
	  break;
	case TYPE_DIR:
	  fprintf(fp,"\ttype directory\n");
	  break;
	case TYPE_FILE:
	  fprintf(fp,"\ttype file\n");
	  break;
	};
      
      fprintf(fp,"\tsource \"%s\"\n",GamesList->Sources[i]->Source);
      
      if(GamesList->Sources[i]->Target!=NULL)
	fprintf(fp,"\ttarget \"%s\"\n",GamesList->Sources[i]->Target);
      
      if(GamesList->Sources[i]->TargetVersion!=NULL)
	fprintf(fp,"\ttargetversion \"%s\"\n",GamesList->Sources[i]->TargetVersion);
      
      if(GamesList->Sources[i]->Author!=NULL)
	fprintf(fp,"\tauthor \"%s\"\n",GamesList->Sources[i]->Author);
      
      if(GamesList->Sources[i]->Version!=NULL)
	fprintf(fp,"\tversion \"%s\"\n",GamesList->Sources[i]->Version);
      
      if(GamesList->Sources[i]->Comment!=NULL)
	fprintf(fp,"\tcomment \"%s\"\n",GamesList->Sources[i]->Comment);
      
      switch(GamesList->Sources[i]->StorageMode)
	{
	case STMODE_MERGE:
	  fprintf(fp,"\tstoragemode merge\n");
	  break;
	case STMODE_SPLIT:
	  fprintf(fp,"\tstoragemode split\n");
	  break;   
	case STMODE_FULL:
	  fprintf(fp,"\tstoragemode full\n");
	  break;
	};
/*      
      switch(GamesList->Sources[i]->StorageFormat)
	{
	case STFMT_ZIP:
	  fprintf(fp,"\tstorageformat zip\n");
	  break;
	case STFMT_RAW:
	  fprintf(fp,"\tstorageformat raw\n");
	  break;   
	};
*/    
      if(GamesList->Sources[i]->ForceStorageMode!=STMODE_UNSET ||
	 GamesList->Sources[i]->ForceStorageFormat!=STFMT_UNSET)
	{
	  fprintf(fp,"\tmanager ( ");
	  switch(GamesList->Sources[i]->ForceStorageMode)
	    {
	    case STMODE_MERGE:
	      fprintf(fp,"forcestoragemode merge ");
	      break;
	    case STMODE_SPLIT:
	      fprintf(fp,"forcestoragemode split ");
	      break;   
	    case STMODE_FULL:
	      fprintf(fp,"forcestoragemode full ");
	      break;
	    };
	  
	  switch(GamesList->Sources[i]->ForceStorageFormat)
	    {
	    case STFMT_ZIP:
	      fprintf(fp,"forcestorageformat zip ");
	      break;
	    case STFMT_RAW:
	      fprintf(fp,"forcestorageformat raw ");
	      break;
	    };
	  fprintf(fp,")\n");
	};
      
      fprintf(fp,")\n\n");
    };
};


inline unsigned char *strquote(unsigned char *string)
{
  unsigned char *quotedstring=NULL;
  unsigned int i=0;

  if(!strchr(string,' ') &&
     !strchr(string,'(') &&
     !strchr(string,')') &&
     !strchr(string,'"'))
    return string;

  quotedstring=XCALLOC(unsigned char,strlen(string)+2+1);
  sprintf(quotedstring,"\"%s\"",string);
  XFREE(string);
  return quotedstring;
}
  
void fprintRomInfo(FILE *fp, s_GamesList *GamesList,
		   s_GameInfo *GameInfo, s_RomInfo *Rom)
{
  unsigned int i,j;
  unsigned char *string1=NULL,*string2=NULL;

  fprintf(fp,"\trom ( ");
  
  string1=XSTRDUP(Rom->Name);
  string1=strquote(string1);
  if(Rom->Merge!=NULL && strcmp(Rom->Name,Rom->Merge)!=0)
    fprintf(fp,"name %s merge %s ",
	    string1,
	    Rom->Merge);
  else
    fprintf(fp,"name %s ",
	    string1);

  fprintf(fp,"size %i crc %08x ",
	  Rom->Content->Size,
	  Rom->Content->CRC);

  if(Rom->Content->CompSize!=0)
	fprintf(fp,"compsize %i ",
		Rom->Content->CompSize);
  if(strcmp(GameInfo->Name,Rom->GameName)!=0)
    fprintf(fp,"clone %s ",
	    Rom->GameName);
  
  if(GamesList->NbrSources>1)
    {
      for(i=0;i<GamesList->NbrSources;i++)
	if(Rom->Source==GamesList->Sources[i])
	  break;
      fprintf(fp,"source %i )\n",i);
    }
  else
    fprintf(fp,")\n");
  
  for(i=0;i<Rom->NbrShared;i++)
    {
      fprintf(fp,"\trom ( ");
      if(Rom->Shared[i]->RomName!=NULL)
	{
	  string2=XSTRDUP(Rom->Shared[i]->RomName);
	  string2=strquote(string2);
	  fprintf(fp,"name %s merge %s ",
		  string2,
		  string1);
	  XFREE(string2);
	} 
      else
	fprintf(fp,"name %s ",
		string1);
      XFREE(string2);
      XFREE(string1);

      fprintf(fp,"size %i crc %08x ",
	      Rom->Content->Size,
	      Rom->Content->CRC);
      if(Rom->Content->CompSize!=0)
	fprintf(fp,"compsize %i ",
		Rom->Content->CompSize);

      string2=XSTRDUP(Rom->Shared[i]->CloneName);
      string2=strquote(string2);
      fprintf(fp,"clone %s ",
	      string2);
      XFREE(string2);

      if(GamesList->NbrSources>1)
	{
	  for(j=0;j<GamesList->NbrSources;j++)
	    if(Rom->Source==GamesList->Sources[j])
	      break;
	  fprintf(fp,"source %i )\n",j);
	}
      else
	fprintf(fp,")\n");	      
    };
};

void fprintGameInfo(FILE *fp,s_GamesList *GamesList, s_GameInfo *GameInfo)
{
  unsigned int i;
  unsigned char *string=NULL;

  for(i=0;i<GamesList->NbrResources;i++)
    if(strcmp(GameInfo->Name,GamesList->Resources[i]->Name)==0)
      {
	fprintf(fp,"resource (\n");
	break;
      }
  
  if(i==GamesList->NbrResources)
     fprintf(fp,"game (\n");
  
  string=XSTRDUP(GameInfo->Name);
  string=strquote(string);
  fprintf(fp,"\tname %s\n",string);
  XFREE(string);

  if(GameInfo->More!=NULL)
    {
      if(GameInfo->More->Description!=NULL)
	fprintf(fp,"\tdescription \"%s\"\n",
		GameInfo->More->Description);
      
      if(GameInfo->More->Resource!=NULL)
	{
	  string=XSTRDUP(GameInfo->More->Resource);
	  string=strquote(string);
	  fprintf(fp,"\tresource %s\n",
		  string);
	  XFREE(string);
	};

      if(GameInfo->More->CloneOf!=NULL)
	{
	  string=XSTRDUP(GameInfo->More->CloneOf);
	  string=strquote(string);
	  fprintf(fp,"\tcloneof %s\n",
		  string);
	  XFREE(string);
	};

      if(GameInfo->More->RomOf!=NULL)
	{
	  string=XSTRDUP(GameInfo->More->RomOf);
	  string=strquote(string);
	  fprintf(fp,"\tromof %s\n",
		  string);
	  XFREE(string);
	};

      for(i=0;i<GameInfo->More->NbrClones;i++)
	{
	  string=XSTRDUP(GameInfo->More->Clones[i]->Name);
	  string=strquote(string);
	  fprintf(fp,"\tclone ( name %s\n",
		  string);
	  XFREE(string);

	  fprintf(fp,"\t\tdescription \"%s\"\n",
		  GameInfo->More->Clones[i]->Description);  
	  fprintf(fp,"\t)\n");
	};
    };

  for (i=0;i<GameInfo->NbrRoms;i++)
    fprintRomInfo(fp,GamesList,GameInfo,GameInfo->Roms[i]);

  if(GameInfo->More!=NULL)
    for(i=0;i<GameInfo->More->NbrMergedRoms;i++)
      fprintRomInfo(fp,GamesList,GameInfo,GameInfo->More->MergedRoms[i]);

  fprintf(fp,")\n\n");
  
/*     
      fprintf(fp," %s , %s, %s, %s, %u, %s \n",
	      GameInfo->Name,
	      GameInfo->More->Description,
	      GameInfo->More->CloneOf, 
	      GameInfo->More->RomOf,
	      GameInfo->More->Misc->Year, 
	      GameInfo->More->Misc->Manufacturer);
      
      for (i=0;i<GameInfo->NbrRoms;i++)
	fprintf(fp," %s, %s, %i octets, CRC: %08lX \n",
		GameInfo->Roms[i]->Name, 
		GameInfo->Roms[i]->Merge,
		GameInfo->Roms[i]->Content->Size, 
		GameInfo->Roms[i]->Content->CRC);
      
      for (i=0;i<GameInfo->More->Misc->NbrChips;i++)
	fprintf(fp," %s, %s, %s, %i Hz \n",
		GameInfo->More->Misc->Chips[i]->Type, 
		GameInfo->More->Misc->Chips[i]->Flags,
		GameInfo->More->Misc->Chips[i]->Name, 
		GameInfo->More->Misc->Chips[i]->Clock);
      
      fprintf(fp," %s, %i, X %i, Y %i, Color %i, Freq %f \n",
	      GameInfo->More->Misc->Video->Screen, 
	      GameInfo->More->Misc->Video->Orientation,
	      GameInfo->More->Misc->Video->X, 
	      GameInfo->More->Misc->Video->Y, 
	      GameInfo->More->Misc->Video->Colors,
	      GameInfo->More->Misc->Video->Freq);

      fprintf(fp," %i \n",GameInfo->More->Misc->Sound->Channels);
      fprintf(fp," %i, %s, %i, %i, %s \n",
	      GameInfo->More->Misc->Input->Players, 
	      GameInfo->More->Misc->Input->Control,
	      GameInfo->More->Misc->Input->Buttons, 
	      GameInfo->More->Misc->Input->Coins,
	      GameInfo->More->Misc->Input->Tilt);
      
      for(i=0;i<GameInfo->More->Misc->NbrDipSwitches;i++)
	{
	  fprintf(fp," %s,",
	          GameInfo->More->Misc->DipSwitches[i]->Name);
	  for(k=0;k<GameInfo->More->Misc->DipSwitches[i]->NbrEntries;k++)
	    fprintf(fp," %s,",
                    GameInfo->More->Misc->DipSwitches[i]->Entries[k]);    
	  fprintf(fp," %s\n",
                  GameInfo->More->Misc->DipSwitches[i]->Default);
	};
      
      fprintf(fp," %s, %s, %s, %i \n",
	      GameInfo->More->Misc->Driver->Status, 
	      GameInfo->More->Misc->Driver->Color,
	      GameInfo->More->Misc->Driver->Sound, 
	      GameInfo->More->Misc->Driver->PaletteSize);
*/      
};


int fprintDiff(FILE *fp,s_Diff *Diff)
{
  s_DiffGame *ThisDiffGame;
  s_DiffRom *ThisDiffRom;

  fprintf(fp,"%i games to modify:\n",Diff->NbrGames);

  if(Diff->NbrGames!=0)
    {
      Diff->Games=CLFirst(Diff->Games);
      do
	{
	  ThisDiffGame=DIFF_GAME(CLGetData(Diff->Games));
	  fprintf(fp,"%s:\n",ThisDiffGame->Game->Name);
	  ThisDiffGame->Roms=CLFirst(ThisDiffGame->Roms);
	  do
	    {
	      ThisDiffRom=DIFF_ROM(CLGetData(ThisDiffGame->Roms));
	      if(ThisDiffRom->Type==SAME)
		{
		  if(ThisDiffGame->Roms->Next!=NULL)
		    ThisDiffGame->Roms=ThisDiffGame->Roms->Next;
		  else
		    break;
		  continue;
		};

	      if( !(((ThisDiffRom->Rom->Content->CRC==0 &&
		      ThisDiffRom->Type==ADDEDNEW) ||
		     (ThisDiffRom->Rom->Content->CRC==0 &&
		      ThisDiffRom->Type==ADDEDTRANSFERT &&
		       ThisDiffRom->SourceRom==NULL)) &&
		    Diff->OptionFlags & OPTION_IGNORE_MISSING_NO_DUMP_ROMS))
		{

		  fprintf(fp,"\t%s (CRC: %08X) -> ",
			  ThisDiffRom->Rom->Name,
			  ThisDiffRom->Rom->Content->CRC );
		  switch(ThisDiffRom->Type)
		    {
		    case CHANGEDNAME: 
		      fprintf(fp,"CHANGED NAME FROM %s\n",
			      (unsigned char*)ThisDiffRom->SourceRom->Name);
		      break;
		    case ADDEDTRANSFERT:
		      if(ThisDiffRom->Rom->Content->CRC==0 &&
			 ThisDiffRom->SourceRom==NULL)
			{
			  fprintf(fp,"MISSING (but magic available)\n");
			  break;
			};

		      if(strcmp(ThisDiffRom->SourceRom->Name,ThisDiffRom->Rom->Name)==0)
			fprintf(fp,"TRANSFERED IN\n"); 
		      else
			fprintf(fp,"TRANSFERED IN AND CHANGED NAME FROM %s\n",
				(unsigned char*)ThisDiffRom->SourceRom->Name);
		      break;
		    case ADDEDNEW: 
		      fprintf(fp,"MISSING\n"); 
		      break; 
		    case SUPPRTRANSFERT: fprintf(fp,"TRANSFERED OUT\n"); break; 
		    case SUPPRESSED: fprintf(fp,"UNUSED\n"); break; 
		    case UNKNOWN: fprintf(fp,"UNKNOWN\n"); break; 
		    };
		};

	      if(ThisDiffGame->Roms->Next!=NULL)
		ThisDiffGame->Roms=ThisDiffGame->Roms->Next;
	      else
		break;
	    }while (ThisDiffGame->Roms!=NULL);
	
	  if(Diff->Games->Next!=NULL)
	    Diff->Games=Diff->Games->Next;
	  else
	    break;
	}while (Diff->Games!=NULL);
    };
  return 1;
};

