/*
 * Copyright 2016-2017 Józef Kucia for CodeWeavers
 * Copyright 2020-2021 Philip Rebohle for Valve Corporation
 * Copyright 2020-2021 Joshua Ashton for Valve Corporation
 * Copyright 2020-2021 Hans-Kristian Arntzen for Valve Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#define VKD3D_DBG_CHANNEL VKD3D_DBG_CHANNEL_API
#include "d3d12_crosstest.h"

static uint32_t compute_tile_count(uint32_t resource_size, uint32_t mip, uint32_t tile_size)
{
    uint32_t mip_size = max(resource_size >> mip, 1u);
    return (mip_size / tile_size) + (mip_size % tile_size ? 1 : 0);
}

void test_get_resource_tiling(void)
{
    D3D12_FEATURE_DATA_D3D12_OPTIONS options;
    D3D12_SUBRESOURCE_TILING tilings_alt[16];
    D3D12_PACKED_MIP_INFO packed_mip_info;
    D3D12_SUBRESOURCE_TILING tilings[16];
    UINT num_resource_tiles, num_tilings;
    D3D12_RESOURCE_DESC resource_desc;
    struct test_context_desc desc;
    struct test_context context;
    D3D12_TILE_SHAPE tile_shape;
    ID3D12Resource *resource;
    unsigned int i, j;
    HRESULT hr;

    static const struct
    {
        D3D12_RESOURCE_DIMENSION dim;
        DXGI_FORMAT format;
        UINT width;
        UINT height;
        UINT depth_or_array_layers;
        UINT mip_levels;
        UINT expected_tile_count;
        UINT expected_tiling_count;
        UINT expected_standard_mips;
        UINT tile_shape_w;
        UINT tile_shape_h;
        UINT tile_shape_d;
        D3D12_TILED_RESOURCES_TIER min_tier;
        bool todo_radv;
    }
    tests[] =
    {
        /* Test buffers */
        { D3D12_RESOURCE_DIMENSION_BUFFER,    DXGI_FORMAT_UNKNOWN,            1024,    1,  1,  1,  1,  1,  0, 65536,   1,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_BUFFER,    DXGI_FORMAT_UNKNOWN,        16*65536,    1,  1,  1, 16,  1,  0, 65536,   1,   1, D3D12_TILED_RESOURCES_TIER_1 },
        /* Test small resource behavior */
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8_UNORM,              1,    1,  1,  1,  1,  1,  0,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8_UNORM,              2,    2,  1,  2,  1,  2,  0,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8_UNORM,              4,    4,  1,  3,  1,  3,  0,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8_UNORM,              8,    8,  1,  4,  1,  4,  0,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8_UNORM,             16,   16,  1,  5,  1,  5,  0,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8_UNORM,             32,   32,  1,  6,  1,  6,  0,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8_UNORM,             64,   64,  1,  7,  1,  7,  0,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8_UNORM,            128,  128,  1,  8,  1,  8,  0,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8_UNORM,            256,  256,  1,  9,  2,  9,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        /* Test various image formats */
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8_UNORM,	           512,  512,  1,  1,  4,  1,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8G8_UNORM,          512,  512,  1,  1,  8,  1,  1,   256, 128,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8G8B8A8_UNORM,      512,  512,  1,  1, 16,  1,  1,   128, 128,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R16G16B16A16_UNORM,  512,  512,  1,  1, 32,  1,  1,   128,  64,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R32G32B32A32_FLOAT,  512,  512,  1,  1, 64,  1,  1,    64,  64,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_D16_UNORM,           512,  512,  1,  1,  8,  1,  1,   256, 128,   1, D3D12_TILED_RESOURCES_TIER_1, true },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_D32_FLOAT,           512,  512,  1,  1, 16,  1,  1,   128, 128,   1, D3D12_TILED_RESOURCES_TIER_1, true },

        /* Test rectangular textures */
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8G8B8A8_UNORM,     1024,  256,  1,  1, 16,  1,  1,   128, 128,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8G8B8A8_UNORM,      256, 1024,  1,  1, 16,  1,  1,   128, 128,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8G8B8A8_UNORM,      192,  128,  1,  1,  2,  1,  1,   128, 128,   1, D3D12_TILED_RESOURCES_TIER_2 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8G8B8A8_UNORM,      128,  192,  1,  1,  2,  1,  1,   128, 128,   1, D3D12_TILED_RESOURCES_TIER_2 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8G8B8A8_UNORM,      320,  192,  1,  1,  6,  1,  1,   128, 128,   1, D3D12_TILED_RESOURCES_TIER_2 },
        /* Test array layers and packed mip levels */
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8G8B8A8_UNORM,      128,  128, 16,  1, 16, 16,  1,   128, 128,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8G8B8A8_UNORM,      128,  128,  1,  8,  1,  8,  1,   128, 128,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8G8B8A8_UNORM,      512,  512,  1, 10, 21, 10,  3,   128, 128,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8G8B8A8_UNORM,      512,  512,  4,  3, 84, 12,  3,   128, 128,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_R8G8B8A8_UNORM,       64,   64,  1,  1,  1,  1,  0,   128, 128,   1, D3D12_TILED_RESOURCES_TIER_1 },
        /* Test 3D textures */
        { D3D12_RESOURCE_DIMENSION_TEXTURE3D, DXGI_FORMAT_R8_UNORM,             64,   64, 64,  1,  4,  1,  1,    64,  32,  32, D3D12_TILED_RESOURCES_TIER_3 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE3D, DXGI_FORMAT_R8G8_UNORM,           64,   64, 64,  1,  8,  1,  1,    32,  32,  32, D3D12_TILED_RESOURCES_TIER_3 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE3D, DXGI_FORMAT_R8G8B8A8_UNORM,       64,   64, 64,  1, 16,  1,  1,    32,  32,  16, D3D12_TILED_RESOURCES_TIER_3 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE3D, DXGI_FORMAT_R32G32B32A32_FLOAT,   64,   64, 64,  3, 73,  3,  3,    16,  16,  16, D3D12_TILED_RESOURCES_TIER_3 },

        /* Basic BC configurations. */
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC1_UNORM,           512, 512,  1,  1,  2,  1,  1,   512, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC2_UNORM,           512, 512,  1,  1,  4,  1,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC3_UNORM,           512, 512,  1,  1,  4,  1,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC4_UNORM,           512, 512,  1,  1,  2,  1,  1,   512, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC5_UNORM,           512, 512,  1,  1,  4,  1,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC6H_UF16,           512, 512,  1,  1,  4,  1,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC6H_SF16,           512, 512,  1,  1,  4,  1,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC7_UNORM,           512, 512,  1,  1,  4,  1,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },

        /* Basic mipmapping with obvious tiling layouts. */
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC1_UNORM,           512,  256,  1,  10,  2, 10,  1,   512, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC1_UNORM,          1024,  512,  1,  10,  6, 10,  2,   512, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC1_UNORM,          2048, 1024,  1,  10, 22, 10,  3,   512, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC7_UNORM,           256,  256,  1,   9,  2,  9,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC7_UNORM,           512,  512,  1,   9,  6,  9,  2,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC7_UNORM,          1024, 1024,  1,   9, 22,  9,  3,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },

        /* Wide shapes. On AMD, we keep observing standard mips even when the smallest dimension dips below the tile size.
         * This is not the case on NV however. */
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC1_UNORM,          1024,  256,  1, 10,  3, 10,  1,   512, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC1_UNORM,          2048,  256,  1, 10,  6, 10,  1,   512, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC1_UNORM,          4096,  256,  1, 10, 11, 10,  1,   512, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC7_UNORM,           512,  256,  1,  9,  3,  9,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC7_UNORM,          1024,  256,  1,  9,  6,  9,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC7_UNORM,          2048,  256,  1,  9, 11,  9,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },

        /* Tall shapes. Similar to wide tests. */
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC1_UNORM,           512,  512,  1, 10,  3, 10,  1,   512, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC1_UNORM,           512, 1024,  1, 10,  6, 10,  1,   512, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC1_UNORM,           512, 2048,  1, 10, 11, 10,  1,   512, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC7_UNORM,           256,  512,  1,  9,  3,  9,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC7_UNORM,           256, 1024,  1,  9,  6,  9,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
        { D3D12_RESOURCE_DIMENSION_TEXTURE2D, DXGI_FORMAT_BC7_UNORM,           256, 2048,  1,  9, 11,  9,  1,   256, 256,   1, D3D12_TILED_RESOURCES_TIER_1 },
    };

    memset(&desc, 0, sizeof(desc));
    desc.rt_width = 640;
    desc.rt_height = 480;
    desc.rt_format = DXGI_FORMAT_R8G8B8A8_UNORM;
    if (!init_test_context(&context, &desc))
        return;

    hr = ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_D3D12_OPTIONS, &options, sizeof(options));
    ok(hr == S_OK, "Failed to check feature support, hr %#x.\n", hr);

    if (!options.TiledResourcesTier)
    {
        skip("Tiled resources not supported by device.\n");
        destroy_test_context(&context);
        return;
    }

    /* Test behaviour with various parameter combinations */
    resource_desc.Dimension = D3D12_RESOURCE_DIMENSION_TEXTURE2D;
    resource_desc.Alignment = 0;
    resource_desc.Width = 512;
    resource_desc.Height = 512;
    resource_desc.DepthOrArraySize = 1;
    resource_desc.MipLevels = 10;
    resource_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    resource_desc.SampleDesc.Count = 1;
    resource_desc.SampleDesc.Quality = 0;
    resource_desc.Layout = D3D12_TEXTURE_LAYOUT_64KB_UNDEFINED_SWIZZLE;
    resource_desc.Flags = D3D12_RESOURCE_FLAG_NONE;

    hr = ID3D12Device_CreateReservedResource(context.device, &resource_desc,
        D3D12_RESOURCE_STATE_GENERIC_READ, NULL, &IID_ID3D12Resource, (void **)&resource);
    ok(hr == S_OK, "Failed to create reserved resource, hr %#x.\n", hr);

    /* This is nonsense, but it doesn't crash or generate errors. */
    ID3D12Device_GetResourceTiling(context.device, resource, NULL, NULL, NULL, NULL, 0, NULL);

    /* If num_tilings is NULL, tilings_alt is ignored. */
    memset(tilings, 0, sizeof(tilings));
    memset(tilings_alt, 0, sizeof(tilings_alt));
    ID3D12Device_GetResourceTiling(context.device, resource, NULL, NULL, NULL, NULL, 0, tilings_alt);
    ok(memcmp(tilings, tilings_alt, sizeof(tilings_alt)) == 0, "Mismatch.\n");

    num_tilings = 0;
    ID3D12Device_GetResourceTiling(context.device, resource, NULL, NULL, NULL, &num_tilings, 0, NULL);
    ok(num_tilings == 0, "Unexpected tiling count %u.\n", num_tilings);

    num_tilings = ARRAY_SIZE(tilings);
    ID3D12Device_GetResourceTiling(context.device, resource, NULL, NULL, NULL, &num_tilings, 10, tilings);
    ok(num_tilings == 0, "Unexpected tiling count %u.\n", num_tilings);

    num_tilings = ARRAY_SIZE(tilings);
    ID3D12Device_GetResourceTiling(context.device, resource, NULL, NULL, NULL, &num_tilings, 2, tilings);
    ok(num_tilings == 8, "Unexpected tiling count %u.\n", num_tilings);
    ok(tilings[0].StartTileIndexInOverallResource == 20, "Unexpected start tile index %u.\n", tilings[0].StartTileIndexInOverallResource);

    num_tilings = 1;
    memset(&tilings, 0xaa, sizeof(tilings));
    ID3D12Device_GetResourceTiling(context.device, resource, NULL, NULL, NULL, &num_tilings, 2, tilings);
    ok(num_tilings == 1, "Unexpected tiling count %u.\n", num_tilings);
    ok(tilings[0].StartTileIndexInOverallResource == 20, "Unexpected start tile index %u.\n", tilings[0].StartTileIndexInOverallResource);
    ok(tilings[1].StartTileIndexInOverallResource == 0xaaaaaaaa, "Tiling array got modified.\n");

    ID3D12Resource_Release(resource);

    /* Test actual tiling properties */
    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        unsigned int tile_index = 0;
        vkd3d_test_set_context("test %u", i);

        if (tests[i].min_tier > options.TiledResourcesTier)
        {
            skip("Tiled resources tier %u not supported.\n", tests[i].min_tier);
            continue;
        }

        memset(&packed_mip_info, 0xaa, sizeof(packed_mip_info));
        memset(&tile_shape, 0xaa, sizeof(tile_shape));
        memset(&tilings, 0xaa, sizeof(tilings));

        num_resource_tiles = 0xdeadbeef;
        num_tilings = ARRAY_SIZE(tilings);

        resource_desc.Dimension = tests[i].dim;
        resource_desc.Alignment = 0;
        resource_desc.Width = tests[i].width;
        resource_desc.Height = tests[i].height;
        resource_desc.DepthOrArraySize = tests[i].depth_or_array_layers;
        resource_desc.MipLevels = tests[i].mip_levels;
        resource_desc.Format = tests[i].format;
        resource_desc.SampleDesc.Count = 1;
        resource_desc.SampleDesc.Quality = 0;
        resource_desc.Layout = D3D12_TEXTURE_LAYOUT_64KB_UNDEFINED_SWIZZLE;
        resource_desc.Flags = D3D12_RESOURCE_FLAG_NONE;

        if (tests[i].dim == D3D12_RESOURCE_DIMENSION_BUFFER)
            resource_desc.Layout = D3D12_TEXTURE_LAYOUT_ROW_MAJOR;

        hr = ID3D12Device_CreateReservedResource(context.device, &resource_desc,
            D3D12_RESOURCE_STATE_GENERIC_READ, NULL, &IID_ID3D12Resource, (void **)&resource);
        todo_if(is_radv_device(context.device) && tests[i].todo_radv)
        ok(hr == S_OK, "Failed to create reserved resource, hr %#x.\n", hr);

        if (hr != S_OK)
            continue;

        ID3D12Device_GetResourceTiling(context.device, resource, &num_resource_tiles, &packed_mip_info, &tile_shape, &num_tilings, 0, tilings);

        ok(num_resource_tiles >= tests[i].expected_tile_count, "Unexpected resource tile count %u.\n", num_resource_tiles);
        ok(num_tilings == tests[i].expected_tiling_count, "Unexpected subresource tiling count %u.\n", num_tilings);

        ok(packed_mip_info.NumStandardMips >= tests[i].expected_standard_mips, "Unexpected standard mip count %u.\n", packed_mip_info.NumStandardMips);
        ok(packed_mip_info.NumPackedMips == (tests[i].dim == D3D12_RESOURCE_DIMENSION_BUFFER
                ? 0 : tests[i].mip_levels - packed_mip_info.NumStandardMips),
                "Unexpected packed mip count %u.\n", packed_mip_info.NumPackedMips);
        ok((packed_mip_info.NumTilesForPackedMips == 0) == (packed_mip_info.NumPackedMips == 0),
                "Unexpected packed tile count %u.\n", packed_mip_info.NumTilesForPackedMips);

        /* Docs say that tile shape should be cleared to zero if there is no standard mip, but drivers don't seem to care about this. */
        ok(tile_shape.WidthInTexels == tests[i].tile_shape_w, "Unexpected tile width %u.\n", tile_shape.WidthInTexels);
        ok(tile_shape.HeightInTexels == tests[i].tile_shape_h, "Unexpected tile height %u.\n", tile_shape.HeightInTexels);
        ok(tile_shape.DepthInTexels == tests[i].tile_shape_d, "Unexpected tile depth %u.\n", tile_shape.DepthInTexels);

        for (j = 0; j < tests[i].expected_tiling_count; j++)
        {
            uint32_t mip = j % tests[i].mip_levels;

            if (mip < packed_mip_info.NumStandardMips || !packed_mip_info.NumPackedMips)
            {
                uint32_t expected_w = compute_tile_count(tests[i].width, mip, tests[i].tile_shape_w);
                uint32_t expected_h = compute_tile_count(tests[i].height, mip, tests[i].tile_shape_h);
                uint32_t expected_d = 1;

                if (tests[i].dim == D3D12_RESOURCE_DIMENSION_TEXTURE3D)
                    expected_d = compute_tile_count(tests[i].depth_or_array_layers, mip, tests[i].tile_shape_d);

                ok(tilings[j].WidthInTiles == expected_w, "Unexpected width %u for subresource %u.\n", tilings[j].WidthInTiles, j);
                ok(tilings[j].HeightInTiles == expected_h, "Unexpected width %u for subresource %u.\n", tilings[j].HeightInTiles, j);
                ok(tilings[j].DepthInTiles == expected_d, "Unexpected width %u for subresource %u.\n", tilings[j].DepthInTiles, j);

                ok(tilings[j].StartTileIndexInOverallResource == tile_index, "Unexpected start tile index %u for subresource %u.\n",
                        tilings[j].StartTileIndexInOverallResource, j);

                tile_index += tilings[j].WidthInTiles * tilings[j].HeightInTiles * tilings[j].DepthInTiles;
            }
            else
            {
                ok(!tilings[j].WidthInTiles && !tilings[j].HeightInTiles && !tilings[j].DepthInTiles,
                        "Unexpected tile count (%u,%u,%u) for packed subresource %u.\n",
                        tilings[j].WidthInTiles, tilings[j].HeightInTiles, tilings[j].DepthInTiles, j);
                ok(tilings[j].StartTileIndexInOverallResource == 0xffffffff, "Unexpected start tile index %u for packed subresource %u.\n",
                        tilings[j].StartTileIndexInOverallResource, j);
            }
        }

        ok(num_resource_tiles == tile_index + packed_mip_info.NumTilesForPackedMips,
                "Unexpected resource tile count %u.\n", num_resource_tiles);
        ok(packed_mip_info.StartTileIndexInOverallResource == (packed_mip_info.NumPackedMips ? tile_index : 0),
                "Unexpected mip tail start tile index %u.\n", packed_mip_info.StartTileIndexInOverallResource);

        ID3D12Resource_Release(resource);
    }
    vkd3d_test_set_context(NULL);

    destroy_test_context(&context);
}

static void set_region_offset(D3D12_TILED_RESOURCE_COORDINATE *region, uint32_t x, uint32_t y, uint32_t z, uint32_t subresource)
{
    region->X = x;
    region->Y = y;
    region->Z = z;
    region->Subresource = subresource;
}

static void set_region_size(D3D12_TILE_REGION_SIZE *region, uint32_t num_tiles, bool use_box, uint32_t w, uint32_t h, uint32_t d)
{
    region->NumTiles = num_tiles;
    region->UseBox = use_box;
    region->Width = w;
    region->Height = h;
    region->Depth = d;
}

void test_update_tile_mappings(void)
{
    D3D12_TILED_RESOURCE_COORDINATE region_offsets[8];
    ID3D12PipelineState *check_texture_3d_pipeline;
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    ID3D12PipelineState *clear_texture_pipeline;
    ID3D12PipelineState *check_texture_pipeline;
    ID3D12PipelineState *check_buffer_pipeline;
    ID3D12Resource *resource, *readback_buffer;
    D3D12_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    ID3D12DescriptorHeap *cpu_heap, *gpu_heap;
    ID3D12RootSignature *clear_root_signature;
    D3D12_FEATURE_DATA_D3D12_OPTIONS options;
    D3D12_SHADER_RESOURCE_VIEW_DESC srv_desc;
    D3D12_DESCRIPTOR_RANGE descriptor_range;
    D3D12_ROOT_PARAMETER root_parameters[2];
    D3D12_TILE_REGION_SIZE region_sizes[8];
    D3D12_GPU_VIRTUAL_ADDRESS readback_va;
    D3D12_PACKED_MIP_INFO packed_mip_info;
    D3D12_HEAP_PROPERTIES heap_properties;
    D3D12_SUBRESOURCE_TILING tilings[10];
    D3D12_TILE_RANGE_FLAGS tile_flags[8];
    ID3D12RootSignature *root_signature;
    D3D12_RESOURCE_DESC resource_desc;
    struct test_context_desc desc;
    struct resource_readback rb;
    struct test_context context;
    D3D12_TILE_SHAPE tile_shape;
    unsigned int i, j, x, y, z;
    D3D12_HEAP_DESC heap_desc;
    UINT tile_offsets[8];
    UINT tile_counts[8];
    ID3D12Heap *heap;
    UINT num_tilings;
    D3D12_BOX box;
    HRESULT hr;

#if 0
    StructuredBuffer<uint> tiled_buffer : register(t0);
    RWStructuredBuffer<uint> out_buffer : register(u0);

    [numthreads(64, 1, 1)]
    void main(uint3 thread_id : SV_DispatchThreadID)
    {
        out_buffer[thread_id.x] = tiled_buffer[16384 * thread_id.x];
    }
#endif
    static const DWORD cs_buffer_code[] =
    {
        0x43425844, 0xa8625c41, 0xfd85df89, 0xcedb7945, 0x0e3444ea, 0x00000001, 0x00000108, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x000000b4, 0x00050050, 0x0000002d, 0x0100086a,
        0x040000a2, 0x00107000, 0x00000000, 0x00000004, 0x0400009e, 0x0011e000, 0x00000000, 0x00000004,
        0x0200005f, 0x00020012, 0x02000068, 0x00000001, 0x0400009b, 0x00000040, 0x00000001, 0x00000001,
        0x06000029, 0x00100012, 0x00000000, 0x0002000a, 0x00004001, 0x0000000e, 0x8b0000a7, 0x80002302,
        0x00199983, 0x00100012, 0x00000000, 0x0010000a, 0x00000000, 0x00004001, 0x00000000, 0x00107006,
        0x00000000, 0x080000a8, 0x0011e012, 0x00000000, 0x0002000a, 0x00004001, 0x00000000, 0x0010000a,
        0x00000000, 0x0100003e,
    };
#if 0
    Texture2D<uint> tiled_texture : register(t0);
    RWStructuredBuffer<uint> out_buffer : register(u0);

    [numthreads(28,1,1)]
    void main(uint3 thread_id : SV_DispatchThreadID)
    {
        uint2 tile_size = uint2(128, 128);
        uint tile_index = 0;
        uint tile_count = 4;
        uint mip_count = 10;
        uint mip_level = 0;

        while (thread_id.x >= tile_index + tile_count * tile_count && mip_level < mip_count)
        {
            tile_index += tile_count * tile_count;
            tile_count = max(tile_count / 2, 1);
            mip_level += 1;
        }

        uint2 tile_coord;
        tile_coord.x = (thread_id.x - tile_index) % tile_count;
        tile_coord.y = (thread_id.x - tile_index) / tile_count;

        out_buffer[thread_id.x] = tiled_texture.mips[mip_level][tile_coord * tile_size];
    }
#endif
    static const DWORD cs_texture_code[] =
    {
        0x43425844, 0x03e118db, 0xda7deb90, 0xedb39031, 0x6b646a0b, 0x00000001, 0x00000288, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000234, 0x00050050, 0x0000008d, 0x0100086a,
        0x04001858, 0x00107000, 0x00000000, 0x00004444, 0x0400009e, 0x0011e000, 0x00000000, 0x00000004,
        0x0200005f, 0x00020012, 0x02000068, 0x00000003, 0x0400009b, 0x0000001c, 0x00000001, 0x00000001,
        0x08000036, 0x00100072, 0x00000000, 0x00004002, 0x00000000, 0x00000004, 0x00000000, 0x00000000,
        0x01000030, 0x09000023, 0x00100082, 0x00000000, 0x0010001a, 0x00000000, 0x0010001a, 0x00000000,
        0x0010000a, 0x00000000, 0x06000050, 0x00100012, 0x00000001, 0x0002000a, 0x0010003a, 0x00000000,
        0x0700004f, 0x00100022, 0x00000001, 0x0010002a, 0x00000000, 0x00004001, 0x0000000a, 0x07000001,
        0x00100012, 0x00000001, 0x0010001a, 0x00000001, 0x0010000a, 0x00000001, 0x03000003, 0x0010000a,
        0x00000001, 0x07000055, 0x00100012, 0x00000001, 0x0010001a, 0x00000000, 0x00004001, 0x00000001,
        0x07000053, 0x00100022, 0x00000000, 0x0010000a, 0x00000001, 0x00004001, 0x00000001, 0x0700001e,
        0x00100042, 0x00000000, 0x0010002a, 0x00000000, 0x00004001, 0x00000001, 0x05000036, 0x00100012,
        0x00000000, 0x0010003a, 0x00000000, 0x01000016, 0x05000036, 0x001000c2, 0x00000001, 0x00100aa6,
        0x00000000, 0x0700001e, 0x00100012, 0x00000000, 0x8010000a, 0x00000041, 0x00000000, 0x0002000a,
        0x0900004e, 0x00100012, 0x00000000, 0x00100012, 0x00000002, 0x0010000a, 0x00000000, 0x0010001a,
        0x00000000, 0x05000036, 0x00100022, 0x00000002, 0x0010000a, 0x00000000, 0x0a000029, 0x00100032,
        0x00000001, 0x00100046, 0x00000002, 0x00004002, 0x00000007, 0x00000007, 0x00000000, 0x00000000,
        0x8900002d, 0x800000c2, 0x00111103, 0x00100012, 0x00000000, 0x00100e46, 0x00000001, 0x00107e46,
        0x00000000, 0x080000a8, 0x0011e012, 0x00000000, 0x0002000a, 0x00004001, 0x00000000, 0x0010000a,
        0x00000000, 0x0100003e,
    };

#if 0
    Texture3D<uint> tiled_texture : register(t0);
    RWStructuredBuffer<uint> out_buffer : register(u0);

    [numthreads(9,1,1)]
    void main(uint3 thread_id : SV_DispatchThreadID)
    {
        uint3 tile_size = uint3(32, 32, 16);
        uint tile_index = 0;
        uint tile_count = 2;
        uint mip_count = 2;
        uint mip_level = 0;

        while (thread_id.x >= tile_index + tile_count * tile_count * tile_count && mip_level < mip_count)
        {
            tile_index += tile_count * tile_count * tile_count;
            tile_count = max(tile_count / 2, 1);
            mip_level += 1;
        }

        uint3 tile_coord;
        tile_coord.x = (thread_id.x - tile_index) % tile_count;
        tile_coord.y = ((thread_id.x - tile_index) / tile_count) % tile_count;
        tile_coord.z = (thread_id.x - tile_index) / (tile_count * tile_count);

        out_buffer[thread_id.x] = tiled_texture.mips[mip_level][tile_coord * tile_size];
    }
#endif
    static const DWORD cs_texture_3d_code[] =
    {
        0x43425844, 0x71b4eb36, 0x2c65e68d, 0x7763693f, 0xfd4eafc6, 0x00000001, 0x000002f4, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x000002a0, 0x00050050, 0x000000a8, 0x0100086a,
        0x04002858, 0x00107000, 0x00000000, 0x00004444, 0x0400009e, 0x0011e000, 0x00000000, 0x00000004,
        0x0200005f, 0x00020012, 0x02000068, 0x00000004, 0x0400009b, 0x00000009, 0x00000001, 0x00000001,
        0x08000036, 0x00100032, 0x00000000, 0x00004002, 0x00000000, 0x00000002, 0x00000000, 0x00000000,
        0x05000036, 0x00100082, 0x00000001, 0x00004001, 0x00000000, 0x01000030, 0x08000026, 0x0000d000,
        0x00100042, 0x00000000, 0x0010001a, 0x00000000, 0x0010001a, 0x00000000, 0x09000023, 0x00100042,
        0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0010000a, 0x00000000, 0x06000050,
        0x00100082, 0x00000000, 0x0002000a, 0x0010002a, 0x00000000, 0x0700004f, 0x00100012, 0x00000002,
        0x0010003a, 0x00000001, 0x00004001, 0x00000002, 0x07000001, 0x00100082, 0x00000000, 0x0010003a,
        0x00000000, 0x0010000a, 0x00000002, 0x03000003, 0x0010003a, 0x00000000, 0x07000055, 0x00100082,
        0x00000000, 0x0010001a, 0x00000000, 0x00004001, 0x00000001, 0x07000053, 0x00100022, 0x00000000,
        0x0010003a, 0x00000000, 0x00004001, 0x00000001, 0x0700001e, 0x00100082, 0x00000001, 0x0010003a,
        0x00000001, 0x00004001, 0x00000001, 0x05000036, 0x00100012, 0x00000000, 0x0010002a, 0x00000000,
        0x01000016, 0x0700001e, 0x00100012, 0x00000000, 0x8010000a, 0x00000041, 0x00000000, 0x0002000a,
        0x0900004e, 0x00100012, 0x00000002, 0x00100012, 0x00000003, 0x0010000a, 0x00000000, 0x0010001a,
        0x00000000, 0x0800004e, 0x0000d000, 0x00100022, 0x00000003, 0x0010000a, 0x00000002, 0x0010001a,
        0x00000000, 0x08000026, 0x0000d000, 0x00100022, 0x00000000, 0x0010001a, 0x00000000, 0x0010001a,
        0x00000000, 0x0800004e, 0x00100042, 0x00000003, 0x0000d000, 0x0010000a, 0x00000000, 0x0010001a,
        0x00000000, 0x0a000029, 0x00100072, 0x00000001, 0x00100246, 0x00000003, 0x00004002, 0x00000005,
        0x00000005, 0x00000004, 0x00000000, 0x8900002d, 0x80000142, 0x00111103, 0x00100012, 0x00000000,
        0x00100e46, 0x00000001, 0x00107e46, 0x00000000, 0x080000a8, 0x0011e012, 0x00000000, 0x0002000a,
        0x00004001, 0x00000000, 0x0010000a, 0x00000000, 0x0100003e,
    };

#if 0
    RWTexture3D<uint> uav : register(u0);

    cbuffer clear_args
    {
            uint3 offset;
            uint value;
    };

    [numthreads(4, 4, 4)]
    void main(uint3 coord : SV_DispatchThreadID)
    {
            uav[offset + coord] = value;
    }
#endif
    static const DWORD cs_clear_code[] =
    {
        0x43425844, 0x288d0bcd, 0xbe5e644d, 0x95665c2e, 0xd8f02c36, 0x00000001, 0x000000e0, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x0000008c, 0x00050050, 0x00000023, 0x0100086a,
        0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0400289c, 0x0011e000, 0x00000000, 0x00004444,
        0x0200005f, 0x00020072, 0x02000068, 0x00000001, 0x0400009b, 0x00000004, 0x00000004, 0x00000004,
        0x0700001e, 0x001000f2, 0x00000000, 0x00020a46, 0x00208a46, 0x00000000, 0x00000000, 0x080000a4,
        0x0011e0f2, 0x00000000, 0x00100e46, 0x00000000, 0x00208ff6, 0x00000000, 0x00000000, 0x0100003e,
    };

    static const D3D12_SHADER_BYTECODE cs_texture = { cs_texture_code, sizeof(cs_texture_code) };
    static const D3D12_SHADER_BYTECODE cs_texture_3d = { cs_texture_3d_code, sizeof(cs_texture_3d_code) };
    static const D3D12_SHADER_BYTECODE cs_buffer = { cs_buffer_code, sizeof(cs_buffer_code) };
    static const D3D12_SHADER_BYTECODE cs_clear = { cs_clear_code, sizeof(cs_clear_code) };

    static const uint32_t buffer_region_tiles[] =
    {
    /*     0   1   2   3   4   5   6   7   8   9 */
    /*0*/ 33, 34, 35, 36, 37,  6,  7,  8,  9, 10,
    /*1*/ 11, 12, 38, 39, 40, 41,  1, 18,  2, 20,
    /*2*/ 21, 22, 23,  3,  4,  4,  4,  0,  0, 25,
    /*3*/ 26, 27, 28, 29, 30, 36, 37, 38, 39, 40,
    /*4*/  9, 11, 43, 44, 45, 46, 45, 46, 49, 50,
    /*5*/  0,  0, 17, 18, 19, 20, 21, 22, 23, 24,
    /*6*/ 61, 62, 63, 12,
    };

    static const uint32_t texture_region_tiles[] =
    {
        1, 2, 4, 5, 6, 7, 1, 1, 9, 1, 17, 14, 8, 14, 3, 0,
        17, 18, 19, 18, 19, 22, 23, 24, 25, 26, 27, 28,
    };

    static const uint32_t texture_3d_region_tiles[] =
    {
        3, 2, 0, 7, 8, 2, 4, 5, 6,
    };

    memset(&desc, 0, sizeof(desc));
    desc.rt_width = 640;
    desc.rt_height = 480;
    desc.rt_format = DXGI_FORMAT_R8G8B8A8_UNORM;
    if (!init_test_context(&context, &desc))
        return;

    hr = ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_D3D12_OPTIONS, &options, sizeof(options));
    ok(hr == S_OK, "Failed to check feature support, hr %#x.\n", hr);

    if (!options.TiledResourcesTier)
    {
        skip("Tiled resources not supported by device.\n");
        destroy_test_context(&context);
        return;
    }

    descriptor_range.RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_SRV;
    descriptor_range.NumDescriptors = 1;
    descriptor_range.BaseShaderRegister = 0;
    descriptor_range.RegisterSpace = 0;
    descriptor_range.OffsetInDescriptorsFromTableStart = 0;
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[0].DescriptorTable.NumDescriptorRanges = 1;
    root_parameters[0].DescriptorTable.pDescriptorRanges = &descriptor_range;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
    root_parameters[1].Descriptor.ShaderRegister = 0;
    root_parameters[1].Descriptor.RegisterSpace = 0;
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_signature_desc.NumParameters = ARRAY_SIZE(root_parameters);
    root_signature_desc.pParameters = root_parameters;
    root_signature_desc.NumStaticSamplers = 0;
    root_signature_desc.pStaticSamplers = NULL;
    root_signature_desc.Flags = D3D12_ROOT_SIGNATURE_FLAG_NONE;
    hr = create_root_signature(context.device, &root_signature_desc, &root_signature);
    ok(hr == S_OK, "Failed to create root signature, hr %#x.\n", hr);

    descriptor_range.RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_UAV;
    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_32BIT_CONSTANTS;
    root_parameters[1].Constants.ShaderRegister = 0;
    root_parameters[1].Constants.RegisterSpace = 0;
    root_parameters[1].Constants.Num32BitValues = 4;
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    hr = create_root_signature(context.device, &root_signature_desc, &clear_root_signature);
    ok(hr == S_OK, "Failed to create root signature, hr %#x.\n", hr);

    clear_texture_pipeline = create_compute_pipeline_state(context.device, clear_root_signature, cs_clear);
    check_texture_pipeline = create_compute_pipeline_state(context.device, root_signature, cs_texture);
    check_texture_3d_pipeline = create_compute_pipeline_state(context.device, root_signature, cs_texture_3d);
    check_buffer_pipeline = create_compute_pipeline_state(context.device, root_signature, cs_buffer);

    cpu_heap = create_cpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 11);
    gpu_heap = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 11);

    memset(&heap_properties, 0, sizeof(heap_properties));
    heap_properties.Type = D3D12_HEAP_TYPE_DEFAULT;

    resource_desc.Dimension = D3D12_RESOURCE_DIMENSION_BUFFER;
    resource_desc.Alignment = 0;
    resource_desc.Width = 64 * sizeof(uint32_t);
    resource_desc.Height = 1;
    resource_desc.DepthOrArraySize = 1;
    resource_desc.MipLevels = 1;
    resource_desc.Format = DXGI_FORMAT_UNKNOWN;
    resource_desc.SampleDesc.Count = 1;
    resource_desc.SampleDesc.Quality = 0;
    resource_desc.Layout = D3D12_TEXTURE_LAYOUT_ROW_MAJOR;
    resource_desc.Flags = D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS;
    hr = ID3D12Device_CreateCommittedResource(context.device, &heap_properties, D3D12_HEAP_FLAG_NONE,
            &resource_desc, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, NULL, &IID_ID3D12Resource, (void **)&readback_buffer);
    ok(hr == S_OK, "Failed to create readback buffer, hr %#x.\n", hr);

    readback_va = ID3D12Resource_GetGPUVirtualAddress(readback_buffer);

    /* Test buffer tile mappings */
    heap_desc.Properties = heap_properties;
    heap_desc.Alignment = 0;
    heap_desc.SizeInBytes = 64 * 65536;
    heap_desc.Flags = D3D12_HEAP_FLAG_ALLOW_ONLY_BUFFERS;
    hr = ID3D12Device_CreateHeap(context.device, &heap_desc, &IID_ID3D12Heap, (void **)&heap);
    ok(hr == S_OK, "Failed to create heap, hr %#x.\n", hr);

    resource_desc.Width = 64 * 65536;
    hr = ID3D12Device_CreateReservedResource(context.device, &resource_desc,
        D3D12_RESOURCE_STATE_UNORDERED_ACCESS, NULL, &IID_ID3D12Resource, (void **)&resource);
    ok(hr == S_OK, "Failed to create reserved buffer, hr %#x.\n", hr);

    srv_desc.Format = DXGI_FORMAT_UNKNOWN;
    srv_desc.ViewDimension = D3D12_SRV_DIMENSION_BUFFER;
    srv_desc.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;
    srv_desc.Buffer.FirstElement = 0;
    srv_desc.Buffer.NumElements = resource_desc.Width / sizeof(uint32_t);
    srv_desc.Buffer.StructureByteStride = sizeof(uint32_t);
    srv_desc.Buffer.Flags = D3D12_BUFFER_SRV_FLAG_NONE;
    ID3D12Device_CreateShaderResourceView(context.device, resource, &srv_desc, get_cpu_descriptor_handle(&context, gpu_heap, 0));

    uav_desc.Format = DXGI_FORMAT_R32_UINT;
    uav_desc.ViewDimension = D3D12_UAV_DIMENSION_BUFFER;
    uav_desc.Buffer.FirstElement = 0;
    uav_desc.Buffer.NumElements = resource_desc.Width / sizeof(uint32_t);
    uav_desc.Buffer.StructureByteStride = 0;
    uav_desc.Buffer.CounterOffsetInBytes = 0;
    uav_desc.Buffer.Flags = D3D12_BUFFER_UAV_FLAG_NONE;
    ID3D12Device_CreateUnorderedAccessView(context.device, resource, NULL, &uav_desc, get_cpu_descriptor_handle(&context, cpu_heap, 1));
    ID3D12Device_CreateUnorderedAccessView(context.device, resource, NULL, &uav_desc, get_cpu_descriptor_handle(&context, gpu_heap, 1));

    /* Map entire buffer, linearly, and initialize tile data */
    tile_offsets[0] = 0;
    ID3D12CommandQueue_UpdateTileMappings(context.queue, resource,
        1, NULL, NULL, heap, 1, NULL, tile_offsets, NULL, D3D12_TILE_MAPPING_FLAG_NONE);

    for (i = 0; i < 64; i++)
    {
        UINT clear_value[4] = { 0, 0, 0, 0 };
        D3D12_RECT clear_rect;

        set_rect(&clear_rect, 16384 * i, 0, 16384 * (i + 1), 1);
        clear_value[0] = i + 1;

        ID3D12GraphicsCommandList_ClearUnorderedAccessViewUint(context.list,
                get_gpu_descriptor_handle(&context, gpu_heap, 1),
                get_cpu_descriptor_handle(&context, cpu_heap, 1),
                resource, clear_value, 1, &clear_rect);
    }

    transition_resource_state(context.list, resource, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE);
    ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &gpu_heap);
    ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(context.list, check_buffer_pipeline);
    ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(context.list, 0, get_gpu_descriptor_handle(&context, gpu_heap, 0));
    ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context.list, 1, readback_va);
    ID3D12GraphicsCommandList_Dispatch(context.list, 1, 1, 1);
    transition_resource_state(context.list, readback_buffer, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

    get_buffer_readback_with_command_list(readback_buffer, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

    for (i = 0; i < 64; i++)
    {
        set_box(&box, i, 0, 0, i + 1, 1, 1);
        check_readback_data_uint(&rb, &box, i + 1, 0);
    }

    release_resource_readback(&rb);

    /* Test arbitrary tile mappings */
    set_region_offset(&region_offsets[0], 16, 0, 0, 0);
    set_region_offset(&region_offsets[1], 18, 0, 0, 0);
    set_region_offset(&region_offsets[2], 23, 0, 0, 0);
    set_region_offset(&region_offsets[3], 40, 0, 0, 0);
    set_region_offset(&region_offsets[4], 41, 0, 0, 0);
    set_region_offset(&region_offsets[5], 63, 0, 0, 0);

    tile_offsets[0] = 0;
    tile_offsets[1] = 8;
    tile_offsets[2] = 10;

    tile_counts[0] = 3;
    tile_counts[1] = 1;
    tile_counts[2] = 2;

    ID3D12CommandQueue_UpdateTileMappings(context.queue, resource,
            6, region_offsets, NULL, heap, 3, NULL, tile_offsets, tile_counts,
            D3D12_TILE_MAPPING_FLAG_NONE);

    set_region_offset(&region_offsets[0], 24, 0, 0, 0);
    set_region_offset(&region_offsets[1], 50, 0, 0, 0);
    set_region_offset(&region_offsets[2], 0, 0, 0, 0);
    set_region_offset(&region_offsets[3], 52, 0, 0, 0);
    set_region_offset(&region_offsets[4], 29, 0, 0, 0);

    set_region_size(&region_sizes[0], 5, false, 0, 0, 0);
    set_region_size(&region_sizes[1], 2, false, 0, 0, 0);
    set_region_size(&region_sizes[2], 16, false, 0, 0, 0);
    set_region_size(&region_sizes[3], 8, false, 0, 0, 0);
    set_region_size(&region_sizes[4], 6, false, 0, 0, 0);

    tile_flags[0] = D3D12_TILE_RANGE_FLAG_REUSE_SINGLE_TILE;
    tile_flags[1] = D3D12_TILE_RANGE_FLAG_NULL;
    tile_flags[2] = D3D12_TILE_RANGE_FLAG_NONE;
    tile_flags[3] = D3D12_TILE_RANGE_FLAG_SKIP;
    tile_flags[4] = D3D12_TILE_RANGE_FLAG_NONE;
    tile_flags[5] = D3D12_TILE_RANGE_FLAG_NONE;

    tile_offsets[0] = 3;
    tile_offsets[1] = 0;
    tile_offsets[2] = 32;
    tile_offsets[3] = 0;
    tile_offsets[4] = 37;
    tile_offsets[5] = 16;

    tile_counts[0] = 3;
    tile_counts[1] = 4;
    tile_counts[2] = 5;
    tile_counts[3] = 7;
    tile_counts[4] = 4;
    tile_counts[5] = 14;

    ID3D12CommandQueue_UpdateTileMappings(context.queue, resource,
        5, region_offsets, region_sizes, heap, 6, tile_flags, tile_offsets, tile_counts,
        D3D12_TILE_MAPPING_FLAG_NONE);

    set_region_offset(&region_offsets[0], 46, 0, 0, 0);
    set_region_offset(&region_offsets[1], 44, 0, 0, 0);
    set_region_size(&region_sizes[0], 2, false, 0, 0, 0);

    ID3D12CommandQueue_CopyTileMappings(context.queue,
        resource, &region_offsets[0], resource, &region_offsets[1],
        &region_sizes[0], D3D12_TILE_MAPPING_FLAG_NONE);

    reset_command_list(context.list, context.allocator);

    transition_resource_state(context.list, readback_buffer, D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
    ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &gpu_heap);
    ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(context.list, check_buffer_pipeline);
    ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(context.list, 0, get_gpu_descriptor_handle(&context, gpu_heap, 0));
    ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context.list, 1, readback_va);
    ID3D12GraphicsCommandList_Dispatch(context.list, 1, 1, 1);
    transition_resource_state(context.list, readback_buffer, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

    get_buffer_readback_with_command_list(readback_buffer, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

    for (i = 0; i < ARRAY_SIZE(buffer_region_tiles); i++)
    {
        if (options.TiledResourcesTier > D3D12_TILED_RESOURCES_TIER_2 || buffer_region_tiles[i])
        {
            set_box(&box, i, 0, 0, i + 1, 1, 1);
            check_readback_data_uint(&rb, &box, buffer_region_tiles[i], 0);
        }
    }

    release_resource_readback(&rb);

    ID3D12Resource_Release(resource);
    ID3D12Heap_Release(heap);

    /* Test 2D image tile mappings */
    heap_desc.Properties = heap_properties;
    heap_desc.Alignment = 0;
    heap_desc.SizeInBytes = 64 * 65536;
    heap_desc.Flags = D3D12_HEAP_FLAG_ALLOW_ONLY_NON_RT_DS_TEXTURES;
    hr = ID3D12Device_CreateHeap(context.device, &heap_desc, &IID_ID3D12Heap, (void **)&heap);
    ok(hr == S_OK, "Failed to create heap, hr %#x.\n", hr);

    resource_desc.Dimension = D3D12_RESOURCE_DIMENSION_TEXTURE2D;
    resource_desc.Alignment = 0;
    resource_desc.Width = 512;
    resource_desc.Height = 512;
    resource_desc.DepthOrArraySize = 1;
    resource_desc.MipLevels = 10;
    resource_desc.Format = DXGI_FORMAT_R32_UINT;
    resource_desc.SampleDesc.Count = 1;
    resource_desc.SampleDesc.Quality = 0;
    resource_desc.Layout = D3D12_TEXTURE_LAYOUT_64KB_UNDEFINED_SWIZZLE;
    resource_desc.Flags = D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS;

    hr = ID3D12Device_CreateReservedResource(context.device, &resource_desc,
        D3D12_RESOURCE_STATE_UNORDERED_ACCESS, NULL, &IID_ID3D12Resource, (void **)&resource);
    ok(hr == S_OK, "Failed to create reserved texture, hr %#x.\n", hr);

    num_tilings = resource_desc.MipLevels;
    ID3D12Device_GetResourceTiling(context.device, resource, NULL, &packed_mip_info, &tile_shape, &num_tilings, 0, tilings);
    ok(packed_mip_info.NumStandardMips >= 3, "Unexpected number of standard mips %u.\n", packed_mip_info.NumStandardMips);

    srv_desc.Format = DXGI_FORMAT_R32_UINT;
    srv_desc.ViewDimension = D3D12_SRV_DIMENSION_TEXTURE2D;
    srv_desc.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;
    srv_desc.Texture2D.MostDetailedMip = 0;
    srv_desc.Texture2D.MipLevels = resource_desc.MipLevels;
    srv_desc.Texture2D.PlaneSlice = 0;
    srv_desc.Texture2D.ResourceMinLODClamp = 0.0f;
    ID3D12Device_CreateShaderResourceView(context.device, resource, &srv_desc, get_cpu_descriptor_handle(&context, gpu_heap, 0));

    /* Map entire image */
    tile_offsets[0] = 0;
    ID3D12CommandQueue_UpdateTileMappings(context.queue, resource,
        1, NULL, NULL, heap, 1, NULL, tile_offsets, NULL, D3D12_TILE_MAPPING_FLAG_NONE);

    reset_command_list(context.list, context.allocator);

    for (i = 0, j = 0; i < resource_desc.MipLevels; i++)
    {
        uav_desc.Format = DXGI_FORMAT_R32_UINT;
        uav_desc.ViewDimension = D3D12_UAV_DIMENSION_TEXTURE2D;
        uav_desc.Texture2D.MipSlice = i;
        uav_desc.Texture2D.PlaneSlice = 0;
        ID3D12Device_CreateUnorderedAccessView(context.device, resource, NULL, &uav_desc, get_cpu_descriptor_handle(&context, cpu_heap, 1 + i));
        ID3D12Device_CreateUnorderedAccessView(context.device, resource, NULL, &uav_desc, get_cpu_descriptor_handle(&context, gpu_heap, 1 + i));

        for (y = 0; y < max(1u, tilings[i].HeightInTiles); y++)
        {
            for (x = 0; x < max(1u, tilings[i].WidthInTiles); x++)
            {
                UINT clear_value[4] = { 0, 0, 0, 0 };
                D3D12_RECT clear_rect;

                clear_value[0] = ++j;
                set_rect(&clear_rect,
                    x * tile_shape.WidthInTexels, y * tile_shape.HeightInTexels,
                    min(resource_desc.Width >> i, (x + 1) * tile_shape.WidthInTexels),
                    min(resource_desc.Height >> i, (y + 1) * tile_shape.HeightInTexels));

                ID3D12GraphicsCommandList_ClearUnorderedAccessViewUint(context.list,
                        get_gpu_descriptor_handle(&context, gpu_heap, 1 + i),
                        get_cpu_descriptor_handle(&context, cpu_heap, 1 + i),
                        resource, clear_value, 1, &clear_rect);
            }
        }
    }

    transition_resource_state(context.list, resource, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE);
    transition_resource_state(context.list, readback_buffer, D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
    ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &gpu_heap);
    ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(context.list, check_texture_pipeline);
    ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(context.list, 0, get_gpu_descriptor_handle(&context, gpu_heap, 0));
    ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context.list, 1, readback_va);
    ID3D12GraphicsCommandList_Dispatch(context.list, 1, 1, 1);
    transition_resource_state(context.list, readback_buffer, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

    get_buffer_readback_with_command_list(readback_buffer, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

    for (i = 0; i < j; i++)
    {
        set_box(&box, i, 0, 0, i + 1, 1, 1);
        check_readback_data_uint(&rb, &box, i + 1, 0);
    }

    release_resource_readback(&rb);

    set_region_offset(&region_offsets[0], 2, 0, 0, 0);
    set_region_offset(&region_offsets[1], 1, 1, 0, 0);
    set_region_offset(&region_offsets[2], 1, 1, 0, 1);
    set_region_offset(&region_offsets[3], 0, 3, 0, 0);

    set_region_size(&region_sizes[0], 3, false, 0, 0, 0);
    set_region_size(&region_sizes[1], 4, true, 2, 2, 1);
    set_region_size(&region_sizes[2], 2, false, 0, 0, 0);
    set_region_size(&region_sizes[3], 4, true, 4, 1, 1);

    tile_flags[0] = D3D12_TILE_RANGE_FLAG_NONE;
    tile_flags[1] = D3D12_TILE_RANGE_FLAG_REUSE_SINGLE_TILE;
    tile_flags[2] = D3D12_TILE_RANGE_FLAG_NONE;
    tile_flags[3] = D3D12_TILE_RANGE_FLAG_NONE;
    tile_flags[4] = D3D12_TILE_RANGE_FLAG_SKIP;
    tile_flags[5] = D3D12_TILE_RANGE_FLAG_NONE;
    tile_flags[6] = D3D12_TILE_RANGE_FLAG_NULL;

    tile_offsets[0] = 3;
    tile_offsets[1] = 0;
    tile_offsets[2] = 16;
    tile_offsets[3] = 7;
    tile_offsets[4] = 0;
    tile_offsets[5] = 2;
    tile_offsets[6] = 0;

    tile_counts[0] = 4;
    tile_counts[1] = 2;
    tile_counts[2] = 3;
    tile_counts[3] = 1;
    tile_counts[4] = 1;
    tile_counts[5] = 1;
    tile_counts[6] = 1;

    ID3D12CommandQueue_UpdateTileMappings(context.queue, resource,
        4, region_offsets, region_sizes, heap, 7, tile_flags, tile_offsets, tile_counts,
        D3D12_TILE_MAPPING_FLAG_NONE);

    set_region_offset(&region_offsets[0], 3, 1, 0, 0);
    set_region_offset(&region_offsets[1], 1, 2, 0, 0);
    set_region_size(&region_sizes[0], 2, true, 1, 2, 1);

    ID3D12CommandQueue_CopyTileMappings(context.queue,
        resource, &region_offsets[0], resource, &region_offsets[1],
        &region_sizes[0], D3D12_TILE_MAPPING_FLAG_NONE);

    reset_command_list(context.list, context.allocator);

    transition_resource_state(context.list, readback_buffer, D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
    ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &gpu_heap);
    ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(context.list, check_texture_pipeline);
    ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(context.list, 0, get_gpu_descriptor_handle(&context, gpu_heap, 0));
    ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context.list, 1, readback_va);
    ID3D12GraphicsCommandList_Dispatch(context.list, 1, 1, 1);
    transition_resource_state(context.list, readback_buffer, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

    get_buffer_readback_with_command_list(readback_buffer, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

    for (i = 0; i < j; i++)
    {
        if (options.TiledResourcesTier > D3D12_TILED_RESOURCES_TIER_2 || texture_region_tiles[i])
        {
            set_box(&box, i, 0, 0, i + 1, 1, 1);
            check_readback_data_uint(&rb, &box, texture_region_tiles[i], 0);
        }
    }

    release_resource_readback(&rb);
    ID3D12Resource_Release(resource);

    if (options.TiledResourcesTier >= D3D12_TILED_RESOURCES_TIER_3)
    {
        /* Test 3D image tile mappings */
        resource_desc.Dimension = D3D12_RESOURCE_DIMENSION_TEXTURE3D;
        resource_desc.Alignment = 0;
        resource_desc.Width = 64;
        resource_desc.Height = 64;
        resource_desc.DepthOrArraySize = 32;
        resource_desc.MipLevels = 2;
        resource_desc.Format = DXGI_FORMAT_R32_UINT;
        resource_desc.SampleDesc.Count = 1;
        resource_desc.SampleDesc.Quality = 0;
        resource_desc.Layout = D3D12_TEXTURE_LAYOUT_64KB_UNDEFINED_SWIZZLE;
        resource_desc.Flags = D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS;

        hr = ID3D12Device_CreateReservedResource(context.device, &resource_desc,
            D3D12_RESOURCE_STATE_UNORDERED_ACCESS, NULL, &IID_ID3D12Resource, (void **)&resource);
        ok(hr == S_OK, "Failed to create reserved texture, hr %#x.\n", hr);

        num_tilings = resource_desc.MipLevels;
        ID3D12Device_GetResourceTiling(context.device, resource, NULL, &packed_mip_info, &tile_shape, &num_tilings, 0, tilings);
        ok(packed_mip_info.NumStandardMips == 2, "Unexpected number of standard mips %u.\n", packed_mip_info.NumStandardMips);

        srv_desc.Format = DXGI_FORMAT_R32_UINT;
        srv_desc.ViewDimension = D3D12_SRV_DIMENSION_TEXTURE3D;
        srv_desc.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;
        srv_desc.Texture3D.MostDetailedMip = 0;
        srv_desc.Texture3D.MipLevels = resource_desc.MipLevels;
        srv_desc.Texture3D.ResourceMinLODClamp = 0.0f;
        ID3D12Device_CreateShaderResourceView(context.device, resource, &srv_desc, get_cpu_descriptor_handle(&context, gpu_heap, 0));

        /* Map entire image */
        tile_offsets[0] = 0;
        ID3D12CommandQueue_UpdateTileMappings(context.queue, resource,
            1, NULL, NULL, heap, 1, NULL, tile_offsets, NULL, D3D12_TILE_MAPPING_FLAG_NONE);

        reset_command_list(context.list, context.allocator);

        for (i = 0, j = 0; i < resource_desc.MipLevels; i++)
        {
            uav_desc.Format = DXGI_FORMAT_R32_UINT;
            uav_desc.ViewDimension = D3D12_UAV_DIMENSION_TEXTURE3D;
            uav_desc.Texture3D.MipSlice = i;
            uav_desc.Texture3D.FirstWSlice = 0;
            uav_desc.Texture3D.WSize = resource_desc.DepthOrArraySize >> i;
            ID3D12Device_CreateUnorderedAccessView(context.device, resource, NULL, &uav_desc, get_cpu_descriptor_handle(&context, cpu_heap, 1 + i));
            ID3D12Device_CreateUnorderedAccessView(context.device, resource, NULL, &uav_desc, get_cpu_descriptor_handle(&context, gpu_heap, 1 + i));

            /* ClearUnorderedAccessView only takes 2D coordinates so we have to
             * bring our own shader to initialize portions of a 3D image */
            ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &gpu_heap);
            ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, clear_root_signature);
            ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(context.list, 0, get_gpu_descriptor_handle(&context, gpu_heap, 1 + i));
            ID3D12GraphicsCommandList_SetPipelineState(context.list, clear_texture_pipeline);

            for (z = 0; z < max(1u, tilings[i].DepthInTiles); z++)
            {
                for (y = 0; y < max(1u, tilings[i].HeightInTiles); y++)
                {
                    for (x = 0; x < max(1u, tilings[i].WidthInTiles); x++)
                    {
                        UINT shader_args[4];
                        shader_args[0] = tile_shape.WidthInTexels * x;
                        shader_args[1] = tile_shape.HeightInTexels * y;
                        shader_args[2] = tile_shape.DepthInTexels * z;
                        shader_args[3] = ++j;

                        ID3D12GraphicsCommandList_SetComputeRoot32BitConstants(context.list,
                                1, ARRAY_SIZE(shader_args), shader_args, 0);
                        ID3D12GraphicsCommandList_Dispatch(context.list,
                                tile_shape.WidthInTexels / 4,
                                tile_shape.HeightInTexels / 4,
                                tile_shape.DepthInTexels / 4);
                    }
                }
            }
        }

        transition_resource_state(context.list, resource, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE);
        transition_resource_state(context.list, readback_buffer, D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
        ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &gpu_heap);
        ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, root_signature);
        ID3D12GraphicsCommandList_SetPipelineState(context.list, check_texture_3d_pipeline);
        ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(context.list, 0, get_gpu_descriptor_handle(&context, gpu_heap, 0));
        ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context.list, 1, readback_va);
        ID3D12GraphicsCommandList_Dispatch(context.list, 1, 1, 1);
        transition_resource_state(context.list, readback_buffer, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

        get_buffer_readback_with_command_list(readback_buffer, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

        for (i = 0; i < j; i++)
        {
            set_box(&box, i, 0, 0, i + 1, 1, 1);
            check_readback_data_uint(&rb, &box, i + 1, 0);
        }

        release_resource_readback(&rb);

        set_region_offset(&region_offsets[0], 0, 0, 0, 0);
        set_region_offset(&region_offsets[1], 0, 1, 1, 0);
        set_region_offset(&region_offsets[2], 1, 1, 0, 0);
        set_region_offset(&region_offsets[3], 1, 0, 0, 0);
        set_region_offset(&region_offsets[4], 0, 1, 0, 0);

        set_region_size(&region_sizes[0], 1, false, 0, 0, 0);
        set_region_size(&region_sizes[1], 3, false, 0, 0, 0);
        set_region_size(&region_sizes[2], 2, false, 0, 0, 0);
        set_region_size(&region_sizes[3], 2, true,  1, 1, 2);
        set_region_size(&region_sizes[4], 1, true,  1, 1, 1);

        tile_flags[0] = D3D12_TILE_RANGE_FLAG_NONE;
        tile_flags[1] = D3D12_TILE_RANGE_FLAG_REUSE_SINGLE_TILE;
        tile_flags[2] = D3D12_TILE_RANGE_FLAG_NULL;

        tile_offsets[0] = 2;
        tile_offsets[1] = 1;
        tile_offsets[2] = 0;

        tile_counts[0] = 6;
        tile_counts[1] = 2;
        tile_counts[2] = 1;

        ID3D12CommandQueue_UpdateTileMappings(context.queue, resource,
            5, region_offsets, region_sizes, heap, 3, tile_flags, tile_offsets, tile_counts,
            D3D12_TILE_MAPPING_FLAG_NONE);

        reset_command_list(context.list, context.allocator);

        transition_resource_state(context.list, readback_buffer, D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
        ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &gpu_heap);
        ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, root_signature);
        ID3D12GraphicsCommandList_SetPipelineState(context.list, check_texture_3d_pipeline);
        ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(context.list, 0, get_gpu_descriptor_handle(&context, gpu_heap, 0));
        ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context.list, 1, readback_va);
        ID3D12GraphicsCommandList_Dispatch(context.list, 1, 1, 1);
        transition_resource_state(context.list, readback_buffer, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

        get_buffer_readback_with_command_list(readback_buffer, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

        for (i = 0; i < j; i++)
        {
            set_box(&box, i, 0, 0, i + 1, 1, 1);
            check_readback_data_uint(&rb, &box, texture_3d_region_tiles[i], 0);
        }

        release_resource_readback(&rb);
        ID3D12Resource_Release(resource);
    }
    else
    {
        skip("Tiles resources tier 3 not supported.\n");
    }

    ID3D12Heap_Release(heap);

    ID3D12DescriptorHeap_Release(gpu_heap);
    ID3D12DescriptorHeap_Release(cpu_heap);
    ID3D12Resource_Release(readback_buffer);
    ID3D12PipelineState_Release(clear_texture_pipeline);
    ID3D12PipelineState_Release(check_texture_3d_pipeline);
    ID3D12PipelineState_Release(check_texture_pipeline);
    ID3D12PipelineState_Release(check_buffer_pipeline);
    ID3D12RootSignature_Release(clear_root_signature);
    ID3D12RootSignature_Release(root_signature);
    destroy_test_context(&context);
}

void test_copy_tiles(void)
{
#define TILE_SIZE 65536
    ID3D12Resource *tiled_resource, *dst_buffer, *src_buffer;
    D3D12_TILED_RESOURCE_COORDINATE region_offset;
    D3D12_FEATURE_DATA_D3D12_OPTIONS options;
    uint32_t tile_offset, buffer_offset;
    D3D12_TILE_REGION_SIZE region_size;
    D3D12_RESOURCE_DESC resource_desc;
    struct test_context_desc desc;
    struct resource_readback rb;
    struct test_context context;
    D3D12_HEAP_DESC heap_desc;
    uint32_t *buffer_data;
    unsigned int i, x, y;
    ID3D12Heap *heap;
    D3D12_BOX box;
    HRESULT hr;

    static const struct
    {
        uint32_t x;
        uint32_t y;
        uint32_t tile_idx;
    }
    image_tiles[] =
    {
        {1, 0, 0}, {2, 0, 1}, {1, 1, 2}, {2, 1, 3},
        {3, 1, 4}, {0, 2, 5}, {1, 2, 6},
    };

    memset(&desc, 0, sizeof(desc));
    desc.rt_width = 640;
    desc.rt_height = 480;
    desc.rt_format = DXGI_FORMAT_R8G8B8A8_UNORM;
    if (!init_test_context(&context, &desc))
        return;

    hr = ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_D3D12_OPTIONS, &options, sizeof(options));
    ok(hr == S_OK, "Failed to check feature support, hr %#x.\n");

    if (!options.TiledResourcesTier)
    {
        skip("Tiled resources not supported by device.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&heap_desc, 0, sizeof(heap_desc));
    heap_desc.Properties.Type = D3D12_HEAP_TYPE_DEFAULT;
    heap_desc.SizeInBytes = TILE_SIZE * 16;

    resource_desc.Dimension = D3D12_RESOURCE_DIMENSION_BUFFER;
    resource_desc.Alignment = 0;
    resource_desc.Width = heap_desc.SizeInBytes;
    resource_desc.Height = 1;
    resource_desc.DepthOrArraySize = 1;
    resource_desc.MipLevels = 1;
    resource_desc.Format = DXGI_FORMAT_UNKNOWN;
    resource_desc.SampleDesc.Count = 1;
    resource_desc.SampleDesc.Quality = 0;
    resource_desc.Layout = D3D12_TEXTURE_LAYOUT_ROW_MAJOR;
    resource_desc.Flags = D3D12_RESOURCE_FLAG_NONE;
    hr = ID3D12Device_CreateCommittedResource(context.device, &heap_desc.Properties, D3D12_HEAP_FLAG_NONE,
            &resource_desc, D3D12_RESOURCE_STATE_COPY_DEST, NULL, &IID_ID3D12Resource, (void **)&src_buffer);
    ok(hr == S_OK, "Failed to create buffer, hr %#x.\n", hr);
    hr = ID3D12Device_CreateCommittedResource(context.device, &heap_desc.Properties, D3D12_HEAP_FLAG_NONE,
            &resource_desc, D3D12_RESOURCE_STATE_COPY_DEST, NULL, &IID_ID3D12Resource, (void **)&dst_buffer);
    ok(hr == S_OK, "Failed to create buffer, hr %#x.\n", hr);

    buffer_data = malloc(resource_desc.Width);
    for (i = 0; i < resource_desc.Width / sizeof(*buffer_data); i++)
        buffer_data[i] = i;
    upload_buffer_data(src_buffer, 0, resource_desc.Width, buffer_data, context.queue, context.list);

    reset_command_list(context.list, context.allocator);
    transition_resource_state(context.list, src_buffer,
            D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_COPY_SOURCE);

    /* Test buffer */
    hr = ID3D12Device_CreateReservedResource(context.device, &resource_desc,
            D3D12_RESOURCE_STATE_COPY_DEST, NULL, &IID_ID3D12Resource, (void **)&tiled_resource);
    ok(hr == S_OK, "Failed to create tiled buffer, hr %#x.\n", hr);

    heap_desc.Flags = D3D12_HEAP_FLAG_ALLOW_ONLY_BUFFERS;
    hr = ID3D12Device_CreateHeap(context.device, &heap_desc, &IID_ID3D12Heap, (void **)&heap);
    ok(hr == S_OK, "Failed to create heap, hr %#x.\n", hr);

    tile_offset = 0;
    ID3D12CommandQueue_UpdateTileMappings(context.queue, tiled_resource,
            1, NULL, NULL, heap, 1, NULL, &tile_offset, NULL, D3D12_TILE_MAPPING_FLAG_NONE);

    /* Copy source tiles 0-2 with a 32-byte offset to buffer tiles 4-6 */
    set_region_offset(&region_offset, 4, 0, 0, 0);
    set_region_size(&region_size, 3, false, 0, 0, 0);

    buffer_offset = 32;

    ID3D12GraphicsCommandList_CopyTiles(context.list, tiled_resource,
            &region_offset, &region_size, src_buffer, buffer_offset,
            D3D12_TILE_COPY_FLAG_LINEAR_BUFFER_TO_SWIZZLED_TILED_RESOURCE);

    transition_resource_state(context.list, tiled_resource,
            D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_COPY_SOURCE);

    get_buffer_readback_with_command_list(tiled_resource, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

    for (i = 0; i < 3 * TILE_SIZE / sizeof(*buffer_data); i += 1024)
    {
        uint32_t offset = i + 4 * TILE_SIZE / sizeof(*buffer_data);
        set_box(&box, offset, 0, 0, offset + 1, 1, 1);
        check_readback_data_uint(&rb, &box, buffer_data[i + buffer_offset / sizeof(*buffer_data)], 0);
    }

    release_resource_readback(&rb);

    reset_command_list(context.list, context.allocator);

    /* Read tiles 5-6 from the tiled resource */
    set_region_offset(&region_offset, 5, 0, 0, 0);
    set_region_size(&region_size, 1, false, 0, 0, 0);

    ID3D12GraphicsCommandList_CopyTiles(context.list, tiled_resource,
            &region_offset, &region_size, dst_buffer, 0,
            D3D12_TILE_COPY_FLAG_SWIZZLED_TILED_RESOURCE_TO_LINEAR_BUFFER);

    /* NONE behaves the same as SWIZZLED_TILED_RESOURCE_TO_LINEAR_BUFFER */
    set_region_offset(&region_offset, 6, 0, 0, 0);

    ID3D12GraphicsCommandList_CopyTiles(context.list, tiled_resource,
            &region_offset, &region_size, dst_buffer, TILE_SIZE,
            D3D12_TILE_COPY_FLAG_NONE);

    transition_resource_state(context.list, dst_buffer,
            D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_COPY_SOURCE);

    get_buffer_readback_with_command_list(dst_buffer, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

    for (i = 0; i < 2 * TILE_SIZE / sizeof(*buffer_data); i += 1024)
    {
        uint32_t offset = i + (TILE_SIZE + buffer_offset) / sizeof(*buffer_data);
        set_box(&box, i, 0, 0, i + 1, 1, 1);
        check_readback_data_uint(&rb, &box, buffer_data[offset], 0);
    }

    release_resource_readback(&rb);

    ID3D12Resource_Release(tiled_resource);
    ID3D12Heap_Release(heap);

    /* Test image */
    resource_desc.Dimension = D3D12_RESOURCE_DIMENSION_TEXTURE2D;
    resource_desc.Alignment = 0;
    resource_desc.Width = 512;
    resource_desc.Height = 512;
    resource_desc.DepthOrArraySize = 1;
    resource_desc.MipLevels = 1;
    resource_desc.Format = DXGI_FORMAT_R32_UINT;
    resource_desc.SampleDesc.Count = 1;
    resource_desc.SampleDesc.Quality = 0;
    resource_desc.Layout = D3D12_TEXTURE_LAYOUT_64KB_UNDEFINED_SWIZZLE;
    resource_desc.Flags = D3D12_RESOURCE_FLAG_NONE;

    hr = ID3D12Device_CreateReservedResource(context.device, &resource_desc,
            D3D12_RESOURCE_STATE_COPY_DEST, NULL, &IID_ID3D12Resource, (void **)&tiled_resource);
    ok(hr == S_OK, "Failed to create tiled buffer, hr %#x.\n", hr);

    heap_desc.Flags = D3D12_HEAP_FLAG_ALLOW_ONLY_NON_RT_DS_TEXTURES;
    hr = ID3D12Device_CreateHeap(context.device, &heap_desc, &IID_ID3D12Heap, (void **)&heap);
    ok(hr == S_OK, "Failed to create heap, hr %#x.\n", hr);

    tile_offset = 0;
    ID3D12CommandQueue_UpdateTileMappings(context.queue, tiled_resource,
            1, NULL, NULL, heap, 1, NULL, &tile_offset, NULL, D3D12_TILE_MAPPING_FLAG_NONE);

    reset_command_list(context.list, context.allocator);

    /* Copy source tiles 0-3 to 2x2 region at (1,0) */
    set_region_offset(&region_offset, 1, 0, 0, 0);
    set_region_size(&region_size, 4, true, 2, 2, 1);

    ID3D12GraphicsCommandList_CopyTiles(context.list, tiled_resource,
            &region_offset, &region_size, src_buffer, 0,
            D3D12_TILE_COPY_FLAG_LINEAR_BUFFER_TO_SWIZZLED_TILED_RESOURCE);

    /* Copy source tiles 4-6 to (3,1), (0,2) and (1,2) */
    set_region_offset(&region_offset, 3, 1, 0, 0);
    set_region_size(&region_size, 3, false, 0, 0, 0);

    ID3D12GraphicsCommandList_CopyTiles(context.list, tiled_resource,
            &region_offset, &region_size, src_buffer, 4 * TILE_SIZE,
            D3D12_TILE_COPY_FLAG_LINEAR_BUFFER_TO_SWIZZLED_TILED_RESOURCE);

    transition_resource_state(context.list, tiled_resource,
            D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_COPY_SOURCE);

    get_texture_readback_with_command_list(tiled_resource, 0, &rb, context.queue, context.list);

    for (i = 0; i < ARRAY_SIZE(image_tiles); i++)
    {
        for (y = 0; y < 128; y += 32)
        {
            for (x = 0; x < 128; x += 32)
            {
                uint32_t offset = image_tiles[i].tile_idx * TILE_SIZE / sizeof(*buffer_data) + 128 * y + x;
                set_box(&box, 128 * image_tiles[i].x + x, 128 * image_tiles[i].y + y, 0,
                        128 * image_tiles[i].x + x + 1, 128 * image_tiles[i].y + y + 1, 1);
                check_readback_data_uint(&rb, &box, buffer_data[offset], 0);
            }
        }
    }

    release_resource_readback(&rb);

    reset_command_list(context.list, context.allocator);

    transition_resource_state(context.list, dst_buffer,
            D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_COPY_DEST);

    /* Read 0-3 to 2x2 region at (1,0) */
    set_region_offset(&region_offset, 1, 0, 0, 0);
    set_region_size(&region_size, 4, true, 2, 2, 1);

    ID3D12GraphicsCommandList_CopyTiles(context.list, tiled_resource,
            &region_offset, &region_size, dst_buffer, 0,
            D3D12_TILE_COPY_FLAG_SWIZZLED_TILED_RESOURCE_TO_LINEAR_BUFFER);

    /* Read tiles (3,1), (0,2) and (1,2) */
    set_region_offset(&region_offset, 3, 1, 0, 0);
    set_region_size(&region_size, 3, false, 0, 0, 0);

    ID3D12GraphicsCommandList_CopyTiles(context.list, tiled_resource,
            &region_offset, &region_size, dst_buffer, 4 * TILE_SIZE,
            D3D12_TILE_COPY_FLAG_NONE);

    transition_resource_state(context.list, dst_buffer,
            D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_COPY_SOURCE);

    get_buffer_readback_with_command_list(dst_buffer, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

    for (i = 0; i < ARRAY_SIZE(image_tiles); i++)
    {
        for (x = 0; x < TILE_SIZE / sizeof(uint32_t); x += 1024)
        {
            uint32_t offset = image_tiles[i].tile_idx * TILE_SIZE / sizeof(uint32_t) + x;
            set_box(&box, offset, 0, 0, offset + 1, 1, 1);
            check_readback_data_uint(&rb, &box, buffer_data[offset], 0);
        }
    }

    release_resource_readback(&rb);

    ID3D12Resource_Release(tiled_resource);
    ID3D12Heap_Release(heap);

    ID3D12Resource_Release(src_buffer);
    ID3D12Resource_Release(dst_buffer);

    free(buffer_data);
    destroy_test_context(&context);
#undef TILE_SIZE
}

static void test_buffer_feedback_instructions(bool use_dxil)
{
#define TILE_SIZE 65536
    D3D12_TILED_RESOURCE_COORDINATE tile_regions[2];
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_DESCRIPTOR_RANGE descriptor_ranges[2];
    ID3D12DescriptorHeap *cpu_heap, *gpu_heap;
    ID3D12Resource *tiled_buffer, *out_buffer;
    D3D12_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    D3D12_FEATURE_DATA_D3D12_OPTIONS options;
    D3D12_SHADER_RESOURCE_VIEW_DESC srv_desc;
    D3D12_ROOT_PARAMETER root_parameters[3];
    ID3D12PipelineState *pipeline_state;
    ID3D12RootSignature *root_signature;
    D3D12_RESOURCE_DESC resource_desc;
    struct test_context_desc desc;
    struct resource_readback rb;
    struct test_context context;
    D3D12_HEAP_DESC heap_desc;
    unsigned int i, j;
    ID3D12Heap *heap;
    UINT tile_offset;
    bool test_is_raw;
    HRESULT hr;

#if 0
    Buffer<uint> buf : register(t0);
    RWByteAddressBuffer uav : register(u0);

    cbuffer args : register(b0)
    {
        uint stride;
    };

    [numthreads(64, 1, 1)]
    void main(uint3 tid : SV_DISPATCHTHREADID)
    {
        uint fb;
        uint v = buf.Load(tid.x * stride, fb);
        uint s = CheckAccessFullyMapped(fb) ? 1 : 0;
        uav.Store2(8 * tid.x, uint2(v, s));
    }
#endif
    static const DWORD cs_ld_typed_dxbc[] =
    {
        0x43425844, 0xdf4dd82e, 0x7200a243, 0xf2ca5d6f, 0x9dfaa650, 0x00000001, 0x0000017c, 0x00000004,
        0x00000030, 0x00000040, 0x00000050, 0x0000016c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008,
        0x4e47534f, 0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000114, 0x00050050, 0x00000045,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x04000858, 0x00107000, 0x00000000,
        0x00004444, 0x0300009d, 0x0011e000, 0x00000000, 0x0200005f, 0x00020012, 0x02000068, 0x00000002,
        0x0400009b, 0x00000040, 0x00000001, 0x00000001, 0x08000026, 0x0000d000, 0x00100012, 0x00000000,
        0x0002000a, 0x0020800a, 0x00000000, 0x00000000, 0x8b0000df, 0x80000042, 0x00111103, 0x00100012,
        0x00000000, 0x00100012, 0x00000001, 0x00100006, 0x00000000, 0x00107e46, 0x00000000, 0x050000ea,
        0x00100042, 0x00000000, 0x0010000a, 0x00000001, 0x09000037, 0x00100022, 0x00000000, 0x0010002a,
        0x00000000, 0x00004001, 0x00000001, 0x00004001, 0x00000000, 0x06000029, 0x00100042, 0x00000000,
        0x0002000a, 0x00004001, 0x00000003, 0x070000a6, 0x0011e032, 0x00000000, 0x0010002a, 0x00000000,
        0x00100046, 0x00000000, 0x0100003e, 0x30494653, 0x00000008, 0x00000100, 0x00000000,
    };
    static const BYTE cs_ld_typed_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xdc, 0x3d, 0x27, 0x4b, 0xc3, 0x35, 0x33, 0xac, 0xb2, 0xec, 0xa0, 0x75, 0x16, 0xf3, 0x38, 0xaa, 0x01, 0x00, 0x00, 0x00, 0x04, 0x08, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x6c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x24, 0x07, 0x00, 0x00,
        0x60, 0x00, 0x05, 0x00, 0xc9, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x0c, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00,
        0xc0, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c,
        0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20,
        0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff,
        0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00,
        0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x73, 0x04,
        0x60, 0x30, 0x93, 0x36, 0x0e, 0xec, 0x10, 0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x20, 0x0a, 0xf5, 0x60, 0x0e, 0xe6, 0x50, 0x0e, 0xf2, 0xc0, 0x07, 0xf5, 0xe0, 0x0e, 0xf3, 0x90, 0x0e, 0xe7, 0xe0, 0x0e,
        0xe5, 0x40, 0x0e, 0x60, 0x90, 0x0e, 0xee, 0x40, 0x0f, 0x7e, 0x80, 0x02, 0x40, 0xe1, 0x34, 0x69, 0x8a, 0x28, 0x61, 0xf2, 0x57, 0x78, 0xc3, 0x26, 0x42, 0x1b, 0x86, 0x88, 0x90, 0xa4, 0x8d, 0x2a,
        0x0a, 0x22, 0x42, 0x01, 0xa0, 0x71, 0x04, 0x10, 0x19, 0x12, 0x0a, 0x00, 0x95, 0x11, 0x80, 0x12, 0x1c, 0x42, 0x65, 0x00, 0x00, 0x90, 0x3a, 0x6a, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0xe7, 0x36,
        0xaa, 0x58, 0x89, 0xc9, 0x47, 0x6e, 0x1b, 0x11, 0x00, 0x00, 0x30, 0x47, 0x80, 0x50, 0xbb, 0x67, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x0f, 0x81, 0x66, 0x58, 0x08, 0x14, 0xb8, 0x42, 0x30, 0xf0,
        0x00, 0xc1, 0x9b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56, 0x62, 0xf2, 0x91, 0xdb, 0x46, 0x05, 0x00, 0x00, 0x80, 0x52, 0x48, 0xf0, 0x00, 0xa0, 0x39, 0x47, 0x10, 0x94, 0xa1,
        0x02, 0x40, 0xb6, 0x18, 0x0f, 0x34, 0x00, 0x2a, 0xe1, 0xa2, 0x1c, 0xf0, 0x00, 0x00, 0x00, 0x80, 0x46, 0x7a, 0x20, 0x60, 0x8e, 0x00, 0x14, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87,
        0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07,
        0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76,
        0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21, 0x23, 0x45, 0x44, 0x00, 0x66, 0x00, 0xc0, 0xdc, 0x00, 0x80, 0xd9, 0x01, 0x00, 0x0c, 0x79, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x18, 0xf2, 0x28, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x33, 0x01, 0x01,
        0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xc7, 0x02, 0x02, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x06, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x43, 0x1e, 0x0d, 0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb2, 0x40, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90,
        0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x02, 0x25, 0x30, 0x02, 0x50, 0x08, 0xc5, 0x50, 0x14, 0xa5, 0x50, 0x16, 0x65, 0x50, 0x0e, 0x05, 0x18, 0x40, 0xbd, 0x40, 0x81, 0xa8, 0x8e, 0x00, 0xd0,
        0x98, 0x01, 0x20, 0x32, 0x03, 0x40, 0x65, 0x06, 0x80, 0xcc, 0x0c, 0x00, 0x89, 0x19, 0x00, 0x0a, 0x33, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x5c, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90,
        0x46, 0x02, 0x13, 0xc4, 0x83, 0x0c, 0x6f, 0x0c, 0x24, 0xc6, 0x45, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x00, 0x65, 0x82, 0x00, 0x2c, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x00, 0xcc, 0x06, 0x61, 0x30, 0x28,
        0x8c, 0xcd, 0x4d, 0x10, 0x80, 0x66, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0xd0, 0x38, 0x88, 0xd5, 0x99, 0x4d, 0x10, 0x00, 0x67, 0x82, 0x00, 0x3c, 0x1b, 0x04, 0xc2, 0xd9, 0x90, 0x10, 0x0b, 0x43,
        0x10, 0x43, 0x43, 0x3c, 0x1b, 0x02, 0x68, 0x82, 0x40, 0x5c, 0x1c, 0xea, 0xc2, 0xec, 0x26, 0x08, 0x00, 0x34, 0x41, 0xa8, 0xaa, 0x0d, 0x0b, 0x21, 0x4d, 0x04, 0x31, 0x50, 0x55, 0x55, 0x01, 0x1b,
        0x02, 0x6b, 0x82, 0x60, 0x64, 0x24, 0xc2, 0xe4, 0xce, 0xe6, 0x36, 0x20, 0x04, 0x96, 0x11, 0xc4, 0x60, 0x00, 0x1b, 0x02, 0x6d, 0x03, 0x11, 0x5d, 0x1b, 0x30, 0x41, 0x08, 0x36, 0x06, 0x68, 0x13,
        0x04, 0x20, 0x9a, 0x20, 0x00, 0xd2, 0x06, 0x23, 0xf1, 0x3e, 0x02, 0x0c, 0x9c, 0x0d, 0x82, 0x11, 0x06, 0x13, 0x84, 0xc1, 0xda, 0x60, 0x24, 0xde, 0x47, 0x80, 0x81, 0xb1, 0x41, 0x30, 0xc8, 0x60,
        0x82, 0x50, 0x60, 0x34, 0xe6, 0xe8, 0xe4, 0xd2, 0xc8, 0xca, 0x36, 0x18, 0xc9, 0x19, 0x7c, 0x04, 0x18, 0x38, 0x1b, 0x04, 0x03, 0x0d, 0x36, 0x1c, 0x44, 0x27, 0x06, 0x63, 0x50, 0x06, 0x66, 0x90,
        0x06, 0x13, 0x84, 0x64, 0xd8, 0x00, 0x6c, 0x18, 0x08, 0x36, 0x60, 0x83, 0x0d, 0x41, 0x1b, 0x6c, 0x18, 0x86, 0x35, 0x70, 0x03, 0x12, 0x6d, 0x61, 0x69, 0x6e, 0x13, 0x84, 0x8e, 0x9a, 0x20, 0x00,
        0xd3, 0x86, 0x41, 0x0e, 0x86, 0x61, 0x03, 0x41, 0xc4, 0x81, 0x31, 0x07, 0x1b, 0x8a, 0x35, 0x80, 0x03, 0x80, 0xa3, 0x83, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x53,
        0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c, 0xda, 0x9b, 0xdb, 0x94, 0x80, 0x68, 0x42, 0x86, 0xe7, 0x62, 0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x30, 0xea, 0x90, 0xe1, 0xb9,
        0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95, 0xb1, 0x4d, 0x09, 0x92, 0x32, 0x64, 0x78, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73, 0x53, 0x82, 0xad, 0x12, 0x19,
        0x9e, 0x0b, 0x5d, 0x1e, 0x5c, 0x59, 0x90, 0x9b, 0xdb, 0x1b, 0x5d, 0x18, 0x5d, 0xda, 0x9b, 0xdb, 0xdc, 0x14, 0x21, 0x0d, 0xdc, 0xa0, 0x0e, 0x19, 0x9e, 0x4b, 0x99, 0x1b, 0x9d, 0x5c, 0x1e, 0xd4,
        0x5b, 0x9a, 0x1b, 0xdd, 0xdc, 0x94, 0x80, 0x0e, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88,
        0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce,
        0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48,
        0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e,
        0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b,
        0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78,
        0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1,
        0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39,
        0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70,
        0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72,
        0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x2d, 0x00, 0x00, 0x00, 0x36, 0x40, 0x30, 0x00, 0xd2, 0x58, 0xc1, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x40, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30,
        0x94, 0x84, 0x01, 0x08, 0x98, 0x8f, 0xdc, 0xb6, 0x1d, 0x80, 0xc1, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x07, 0x21, 0x28, 0x9a, 0x20, 0x10, 0x92, 0xf4, 0x51, 0xcb, 0x82, 0x99, 0xc0, 0xf3, 0x10, 0x83,
        0x8f, 0xdc, 0x76, 0x01, 0x58, 0x81, 0x9f, 0xb8, 0x6e, 0x06, 0x02, 0x67, 0xd6, 0x1f, 0x49, 0x58, 0x37, 0x9b, 0xcb, 0x72, 0x60, 0x12, 0x08, 0x0c, 0x9a, 0x00, 0x01, 0x44, 0x86, 0x64, 0x06, 0xd4,
        0x70, 0xf9, 0xce, 0xe3, 0x07, 0x54, 0x51, 0x10, 0x51, 0xe9, 0x00, 0x83, 0x8f, 0xdc, 0x76, 0x05, 0x7c, 0x81, 0x5f, 0x3d, 0x6c, 0x07, 0x02, 0x67, 0x55, 0xe9, 0x55, 0x98, 0xa7, 0x97, 0x83, 0x64,
        0xb2, 0xbc, 0x3c, 0x9f, 0x0b, 0xeb, 0x66, 0x73, 0x59, 0x0e, 0x04, 0x06, 0x0d, 0x41, 0x1a, 0x2e, 0xdf, 0x79, 0x7c, 0x21, 0x22, 0x80, 0x89, 0x08, 0x81, 0x66, 0x58, 0x08, 0x23, 0x90, 0x86, 0xcb,
        0x77, 0x1e, 0x7f, 0x3a, 0x22, 0x02, 0x18, 0xc4, 0xc1, 0x47, 0x6e, 0xdb, 0x12, 0xaa, 0xe1, 0xf2, 0x9d, 0xc7, 0x0f, 0xa8, 0xa2, 0x20, 0x22, 0x76, 0x72, 0x22, 0xc2, 0x47, 0x6e, 0x1b, 0x00, 0x00,
        0x61, 0x20, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x04, 0x66, 0x00, 0x4a, 0xae, 0x74, 0x03, 0xca, 0xae, 0x10, 0x03, 0xca,
        0x31, 0xa0, 0x14, 0x03, 0xa8, 0x95, 0xc0, 0x08, 0x40, 0x11, 0x94, 0x01, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x7d, 0x0a, 0xc1, 0x71, 0xd2, 0x88, 0x41, 0x02, 0x80,
        0x20, 0x18, 0x60, 0x60, 0xb0, 0x10, 0x5d, 0x37, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x16, 0x06, 0x0c, 0xe1, 0x79, 0xd4, 0x88, 0x81, 0x01, 0x80, 0x20, 0x18, 0x28, 0x66, 0xc0, 0x7c, 0x23,
        0x06, 0x07, 0x00, 0x82, 0x60, 0x00, 0x99, 0x01, 0x23, 0x80, 0xc1, 0x68, 0x42, 0x00, 0x54, 0x30, 0xc8, 0x88, 0x01, 0x02, 0x80, 0x20, 0x18, 0x4c, 0x68, 0xe0, 0x18, 0x81, 0x34, 0x9a, 0x10, 0x00,
        0xa3, 0x09, 0x02, 0x31, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x96, 0x1a, 0x40, 0xc1, 0x05, 0x20, 0x2a, 0x11, 0x83, 0x1b, 0x31, 0x68, 0x00, 0x10, 0x04, 0x03, 0x6d, 0x0d, 0xa4, 0x26, 0xc0, 0x0a,
        0x01, 0xc3, 0x1c, 0x0c, 0x07, 0x02, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x86, 0xf2, 0x04, 0x54, 0xf1, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0xf1, 0x07, 0x54, 0x61, 0x29, 0x0c, 0x10, 0x19, 0xd2, 0x2f,
        0x04, 0x54, 0x51, 0x10, 0x91, 0xb9, 0x28, 0x1b, 0x55, 0x14, 0x44, 0x54, 0x3a, 0xc0, 0x60, 0x27, 0x0e, 0x05, 0x54, 0xbf, 0xab, 0xbd, 0x7f, 0x04, 0x58, 0x01, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
    };
#if 0
    /* uint4 to work around what looks like an fxc bug */
    RWBuffer<uint4> buf : register(u1);
    RWByteAddressBuffer uav : register(u0);

    cbuffer args : register(b0)
    {
        uint stride;
    };

    [numthreads(64, 1, 1)]
    void main(uint3 tid : SV_DISPATCHTHREADID)
    {
        uint fb;
        uint v = buf.Load(tid.x * stride, fb).r;
        uint s = CheckAccessFullyMapped(fb) ? 1 : 0;
        uav.Store2(8 * tid.x, uint2(v, s));
    }
#endif
    static const DWORD cs_ld_typed_uav_dxbc[] =
    {
        0x43425844, 0xd7dc177d, 0x5722245f, 0x5af69a70, 0x281d24ee, 0x00000001, 0x0000017c, 0x00000004,
        0x00000030, 0x00000040, 0x00000050, 0x0000016c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008,
        0x4e47534f, 0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000114, 0x00050050, 0x00000045,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300009d, 0x0011e000, 0x00000000,
        0x0400089c, 0x0011e000, 0x00000001, 0x00004444, 0x0200005f, 0x00020012, 0x02000068, 0x00000002,
        0x0400009b, 0x00000040, 0x00000001, 0x00000001, 0x08000026, 0x0000d000, 0x00100012, 0x00000000,
        0x0002000a, 0x0020800a, 0x00000000, 0x00000000, 0x8b0000e1, 0x80000042, 0x00111103, 0x00100012,
        0x00000000, 0x00100012, 0x00000001, 0x00100006, 0x00000000, 0x0011ee46, 0x00000001, 0x050000ea,
        0x00100042, 0x00000000, 0x0010000a, 0x00000001, 0x09000037, 0x00100022, 0x00000000, 0x0010002a,
        0x00000000, 0x00004001, 0x00000001, 0x00004001, 0x00000000, 0x06000029, 0x00100042, 0x00000000,
        0x0002000a, 0x00004001, 0x00000003, 0x070000a6, 0x0011e032, 0x00000000, 0x0010002a, 0x00000000,
        0x00100046, 0x00000000, 0x0100003e, 0x30494653, 0x00000008, 0x00000900, 0x00000000, 
    };
    static const BYTE cs_ld_typed_uav_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x6d, 0xbe, 0x19, 0x57, 0x7a, 0xca, 0x09, 0xe9, 0x8b, 0x07, 0x51, 0x69, 0xd5, 0x03, 0x66, 0x06, 0x01, 0x00, 0x00, 0x00, 0x30, 0x08, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x6c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x50, 0x07, 0x00, 0x00,
        0x60, 0x00, 0x05, 0x00, 0xd4, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x38, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00,
        0xcb, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c,
        0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20,
        0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff,
        0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x3d, 0x00, 0x00, 0x00,
        0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x7c, 0x73, 0x04,
        0x60, 0x30, 0x8c, 0x40, 0x00, 0x33, 0xa1, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3, 0x3c, 0xb8, 0x81, 0x2c, 0xdc, 0x82, 0x28, 0xd4, 0x83, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x03, 0x1f, 0xd8, 0x43,
        0x39, 0x8c, 0x03, 0x3d, 0xbc, 0x83, 0x3c, 0xf0, 0x41, 0x3d, 0xb8, 0xc3, 0x3c, 0xa4, 0xc3, 0x39, 0xb8, 0x43, 0x39, 0x90, 0x03, 0x18, 0xa4, 0x83, 0x3b, 0xd0, 0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8,
        0x01, 0x18, 0xf8, 0x01, 0x0a, 0x02, 0x89, 0xd3, 0xa4, 0x29, 0xa2, 0x84, 0xc9, 0x5f, 0xe1, 0x0d, 0x9b, 0x08, 0x6d, 0x18, 0x22, 0x42, 0x92, 0x36, 0xaa, 0x28, 0x88, 0x08, 0x05, 0x80, 0xc8, 0x11,
        0x40, 0x64, 0x48, 0x28, 0x00, 0x64, 0x46, 0x00, 0x4a, 0x80, 0x28, 0x95, 0x01, 0x00, 0x40, 0xeb, 0xa8, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x9f, 0xdb, 0xa8, 0x62, 0x25, 0x26, 0x1f, 0xb9, 0x6d,
        0x44, 0x00, 0x00, 0xc0, 0x1c, 0x01, 0x42, 0xee, 0x9e, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x3f, 0x04, 0x9a, 0x61, 0x21, 0x50, 0xf0, 0x0a, 0xd1, 0x00, 0x04, 0x14, 0x6f, 0x1a, 0x2e, 0x7f, 0xc2,
        0x1e, 0x42, 0xf2, 0x57, 0x42, 0x5a, 0x89, 0xc9, 0x47, 0x6e, 0x1b, 0x15, 0x00, 0x00, 0x00, 0x4a, 0x31, 0x01, 0x04, 0x80, 0xe8, 0x1c, 0x41, 0x50, 0x06, 0x0b, 0x00, 0xdd, 0x62, 0x40, 0xe0, 0x00,
        0xb0, 0x94, 0x8b, 0x82, 0x00, 0x04, 0x00, 0x00, 0x00, 0x1c, 0xed, 0x81, 0x80, 0x39, 0x02, 0x50, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79,
        0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0,
        0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73,
        0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07,
        0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f,
        0x64, 0x90, 0x21, 0x23, 0x45, 0x44, 0x00, 0x6a, 0x00, 0xc0, 0xe4, 0x00, 0x80, 0xe9, 0x01, 0x00, 0x0c, 0x79, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x2c,
        0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x89, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x43, 0x01, 0x01, 0x20, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xe7, 0x02, 0x02, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x4f, 0x06, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43,
        0x9e, 0x0d, 0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb2, 0x40, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47,
        0xc6, 0x04, 0x43, 0x02, 0x25, 0x30, 0x02, 0x50, 0x08, 0xc5, 0x50, 0x14, 0xa5, 0x50, 0x16, 0x05, 0x52, 0x06, 0xe5, 0x50, 0x80, 0x01, 0xe4, 0x0b, 0x14, 0x38, 0x80, 0xec, 0x08, 0x00, 0x91, 0x19,
        0x00, 0x2a, 0x33, 0x00, 0x64, 0x66, 0x00, 0xe8, 0xcc, 0x00, 0xd0, 0x98, 0x01, 0x20, 0x31, 0x03, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x5c, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90,
        0x46, 0x02, 0x13, 0xc4, 0x83, 0x0c, 0x6f, 0x0c, 0x24, 0xc6, 0x45, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x00, 0x65, 0x82, 0x00, 0x2c, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x00, 0xcc, 0x06, 0x61, 0x30, 0x28,
        0x8c, 0xcd, 0x4d, 0x10, 0x80, 0x66, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x61, 0xd8, 0x38, 0x88, 0xd5, 0x99, 0x4d, 0x10, 0x00, 0x67, 0x82, 0x60, 0x59, 0x13, 0x04, 0xe0, 0xd9, 0x20, 0x10, 0xcf, 0x86,
        0x85, 0x58, 0x18, 0x62, 0x18, 0x1a, 0xc7, 0x71, 0xa0, 0x09, 0x42, 0x81, 0x71, 0xa8, 0x0b, 0xb3, 0x9b, 0x20, 0x00, 0xd0, 0x86, 0x65, 0x90, 0x26, 0x82, 0x18, 0x28, 0xc7, 0x71, 0x80, 0x0d, 0x42,
        0x54, 0x4d, 0x10, 0x0e, 0x8d, 0x44, 0x98, 0xdc, 0xd9, 0xdc, 0x06, 0x84, 0xb8, 0x30, 0x82, 0x18, 0x0c, 0x60, 0x43, 0x90, 0x6d, 0x20, 0x00, 0x4b, 0x03, 0x26, 0x08, 0x02, 0x37, 0x41, 0x00, 0x22,
        0x06, 0x68, 0x13, 0x04, 0x40, 0x9a, 0x20, 0x00, 0xd3, 0x06, 0x23, 0xf1, 0x3e, 0x02, 0x0c, 0x9e, 0x0d, 0x42, 0x17, 0x06, 0x13, 0x04, 0xe2, 0xda, 0x60, 0x24, 0xde, 0x47, 0x80, 0x81, 0xb1, 0x41,
        0x30, 0xc8, 0x60, 0x82, 0x60, 0x64, 0x34, 0xe6, 0xe8, 0xe4, 0xd2, 0xc8, 0xca, 0x36, 0x18, 0xc9, 0x19, 0x7c, 0x04, 0x18, 0x3c, 0x1b, 0x04, 0x03, 0x0d, 0x36, 0x1c, 0x04, 0x27, 0x06, 0x63, 0x50,
        0x06, 0x66, 0x90, 0x06, 0x13, 0x04, 0x65, 0xd8, 0x00, 0x6c, 0x18, 0x08, 0x36, 0x60, 0x83, 0x0d, 0x41, 0x1b, 0x6c, 0x18, 0x86, 0x35, 0x70, 0x03, 0x12, 0x6d, 0x61, 0x69, 0x6e, 0x13, 0x04, 0xaf,
        0x9a, 0x20, 0x00, 0xd4, 0x86, 0x41, 0x0e, 0x86, 0x61, 0x03, 0x41, 0xc4, 0x81, 0x31, 0x07, 0x1b, 0x8a, 0x35, 0x80, 0x03, 0x60, 0xa3, 0x83, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65,
        0x6e, 0x74, 0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c, 0xda, 0x9b, 0xdb, 0x94, 0x80, 0x68, 0x42, 0x86, 0xe7, 0x62, 0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x30, 0xea,
        0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95, 0xb1, 0x4d, 0x09, 0x92, 0x32, 0x64, 0x78, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73, 0x53, 0x02,
        0xad, 0x12, 0x19, 0x9e, 0x0b, 0x5d, 0x1e, 0x5c, 0x59, 0x90, 0x9b, 0xdb, 0x1b, 0x5d, 0x18, 0x5d, 0xda, 0x9b, 0xdb, 0xdc, 0x14, 0x21, 0x0d, 0xdc, 0xa0, 0x0e, 0x19, 0x9e, 0x4b, 0x99, 0x1b, 0x9d,
        0x5c, 0x1e, 0xd4, 0x5b, 0x9a, 0x1b, 0xdd, 0xdc, 0x94, 0x80, 0x0e, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88,
        0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce,
        0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48,
        0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e,
        0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b,
        0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78,
        0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1,
        0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39,
        0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70,
        0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72,
        0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x31, 0x00, 0x00, 0x00, 0x36, 0x40, 0x30, 0x00, 0xd2, 0x58, 0xc1, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x40, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30,
        0x94, 0x84, 0x01, 0x08, 0x98, 0x8f, 0xdc, 0x76, 0x01, 0x9c, 0x41, 0xe0, 0x27, 0xae, 0x9b, 0x81, 0xc0, 0x99, 0xf5, 0x47, 0x92, 0x5e, 0x85, 0x75, 0xb3, 0xb9, 0x2c, 0x07, 0x5a, 0x7f, 0x24, 0x7b,
        0x79, 0x4c, 0x7f, 0xcb, 0x81, 0x49, 0x12, 0x6c, 0x06, 0x04, 0x02, 0x81, 0x41, 0x3b, 0x00, 0x83, 0xe1, 0xf2, 0x9d, 0xc7, 0x17, 0x0e, 0x42, 0x50, 0x34, 0x41, 0x20, 0x24, 0xe9, 0xa3, 0x96, 0x05,
        0x33, 0x81, 0xe7, 0x21, 0x06, 0x1f, 0xb9, 0x6d, 0x13, 0x20, 0x80, 0xc8, 0x90, 0xcc, 0x80, 0x1a, 0x2e, 0xdf, 0x79, 0xfc, 0x80, 0x2a, 0x0a, 0x22, 0x2a, 0x1d, 0x60, 0xf0, 0x91, 0xdb, 0xae, 0x80,
        0x2f, 0xf0, 0xab, 0x87, 0xed, 0x40, 0xe0, 0xac, 0x2a, 0xbd, 0x0a, 0xf3, 0xf4, 0x72, 0x90, 0x4c, 0x96, 0x97, 0xe7, 0x73, 0x61, 0xdd, 0x6c, 0x2e, 0xcb, 0x81, 0xc0, 0xa0, 0x21, 0x48, 0xc3, 0xe5,
        0x3b, 0x8f, 0x2f, 0x44, 0x04, 0x30, 0x11, 0x21, 0xd0, 0x0c, 0x0b, 0x61, 0x04, 0xd2, 0x70, 0xf9, 0xce, 0xe3, 0x4f, 0x47, 0x44, 0x00, 0x83, 0x38, 0xf8, 0xc8, 0x6d, 0x5b, 0x42, 0x35, 0x5c, 0xbe,
        0xf3, 0xf8, 0x01, 0x55, 0x14, 0x44, 0xc4, 0x4e, 0x4e, 0x44, 0xf8, 0xc8, 0x6d, 0x03, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00,
        0x06, 0x00, 0x00, 0x00, 0x04, 0x66, 0x00, 0x4a, 0xae, 0x74, 0x03, 0xca, 0xae, 0x10, 0x03, 0xca, 0x31, 0xa0, 0x14, 0x03, 0xc8, 0x95, 0x40, 0x11, 0x94, 0x01, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00,
        0x82, 0x60, 0x90, 0x7d, 0xc9, 0xd0, 0x71, 0xd1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x64, 0x60, 0xa0, 0x10, 0x9d, 0x27, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x41, 0x16, 0x06, 0x0b, 0xe1, 0x79,
        0xd3, 0x88, 0x81, 0x01, 0x80, 0x20, 0x18, 0x2c, 0x66, 0xb0, 0x7c, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x10, 0x99, 0xc1, 0x22, 0x80, 0xc1, 0x68, 0x42, 0x00, 0x54, 0x30, 0xc8, 0x88, 0x01, 0x02,
        0x80, 0x20, 0x18, 0x50, 0x68, 0xd0, 0x18, 0x41, 0x34, 0x9a, 0x10, 0x00, 0xa3, 0x09, 0x02, 0x31, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x97, 0x1a, 0x3c, 0xc1, 0x05, 0x20, 0x2a, 0x09, 0x83, 0x1b,
        0x31, 0x68, 0x00, 0x10, 0x04, 0x83, 0x6d, 0x0d, 0xa2, 0x26, 0xb8, 0x0a, 0xe1, 0xba, 0x1c, 0x0c, 0x07, 0x02, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x86, 0xb2, 0x04, 0x54, 0xf1, 0xbb, 0xda, 0xfb,
        0x07, 0x54, 0x61, 0x29, 0x0c, 0x10, 0x19, 0xd2, 0x2f, 0x04, 0x54, 0x51, 0x10, 0x91, 0xb9, 0x28, 0x1b, 0x55, 0x14, 0x44, 0x54, 0x3a, 0xc0, 0x60, 0x27, 0x0e, 0x05, 0x54, 0xbf, 0xab, 0xbd, 0x7f,
        0x04, 0x58, 0x01, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
#if 0
    ByteAddressBuffer buf : register(t0);
    RWByteAddressBuffer uav : register(u0);

    cbuffer args : register(b0)
    {
        uint stride;
    };

    [numthreads(64, 1, 1)]
    void main(uint3 tid : SV_DISPATCHTHREADID)
    {
        uint fb;
        uint v = buf.Load(tid.x * stride, fb);
        uint s = CheckAccessFullyMapped(fb) ? 1 : 0;
        uav.Store2(8 * tid.x, uint2(v, s));
    }
#endif
    static const DWORD cs_ld_raw_dxbc[] =
    {
        0x43425844, 0xb925f9c1, 0x36734268, 0x213bf8fc, 0x3e097b37, 0x00000001, 0x00000178, 0x00000004,
        0x00000030, 0x00000040, 0x00000050, 0x00000168, 0x4e475349, 0x00000008, 0x00000000, 0x00000008,
        0x4e47534f, 0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000110, 0x00050050, 0x00000044,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x030000a1, 0x00107000, 0x00000000,
        0x0300009d, 0x0011e000, 0x00000000, 0x0200005f, 0x00020012, 0x02000068, 0x00000002, 0x0400009b,
        0x00000040, 0x00000001, 0x00000001, 0x08000026, 0x0000d000, 0x00100012, 0x00000000, 0x0002000a,
        0x0020800a, 0x00000000, 0x00000000, 0x8b0000e2, 0x800002c2, 0x00199983, 0x00100012, 0x00000000,
        0x00100012, 0x00000001, 0x0010000a, 0x00000000, 0x00107006, 0x00000000, 0x050000ea, 0x00100042,
        0x00000000, 0x0010000a, 0x00000001, 0x09000037, 0x00100022, 0x00000000, 0x0010002a, 0x00000000,
        0x00004001, 0x00000001, 0x00004001, 0x00000000, 0x06000029, 0x00100042, 0x00000000, 0x0002000a,
        0x00004001, 0x00000003, 0x070000a6, 0x0011e032, 0x00000000, 0x0010002a, 0x00000000, 0x00100046,
        0x00000000, 0x0100003e, 0x30494653, 0x00000008, 0x00000100, 0x00000000,
    };
    static const BYTE cs_ld_raw_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xa6, 0x4d, 0x81, 0x45, 0xbb, 0x45, 0x0d, 0x9b, 0x5c, 0xad, 0x97, 0xdb, 0xe7, 0x38, 0x8a, 0x70, 0x01, 0x00, 0x00, 0x00, 0xdc, 0x07, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x6c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xfc, 0x06, 0x00, 0x00,
        0x60, 0x00, 0x05, 0x00, 0xbf, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xe4, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00,
        0xb6, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c,
        0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20,
        0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff,
        0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x31, 0x00, 0x00, 0x00,
        0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x73, 0x04,
        0x60, 0x70, 0x98, 0x34, 0x45, 0x94, 0x30, 0xf9, 0x1b, 0x36, 0x11, 0xda, 0x30, 0x44, 0x84, 0x24, 0x6d, 0x54, 0x51, 0x10, 0x11, 0x0a, 0x00, 0x85, 0xd3, 0xa4, 0x29, 0xa2, 0x84, 0xc9, 0x5f, 0xe1,
        0x0d, 0x9b, 0x08, 0x6d, 0x18, 0x22, 0x42, 0x92, 0x36, 0xaa, 0x28, 0x88, 0x08, 0x05, 0x80, 0xc6, 0x11, 0x40, 0x64, 0x48, 0x28, 0x00, 0x54, 0x46, 0x00, 0x4a, 0x70, 0x08, 0x95, 0x01, 0x00, 0x40,
        0xea, 0xa8, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x9f, 0xdb, 0xa8, 0x62, 0x25, 0x26, 0x1f, 0xb9, 0x6d, 0x44, 0x00, 0x00, 0xc0, 0x1c, 0x01, 0x42, 0xed, 0x9e, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24,
        0x3f, 0x04, 0x9a, 0x61, 0x21, 0x50, 0xe0, 0x0a, 0xc1, 0xc0, 0x03, 0x04, 0xe7, 0x08, 0x82, 0x32, 0x48, 0x00, 0x68, 0x16, 0xe3, 0x81, 0x06, 0x80, 0xa4, 0x7a, 0xd3, 0x70, 0xf9, 0x13, 0xf6, 0x10,
        0x92, 0xbf, 0x12, 0xd2, 0x4a, 0x4c, 0x3e, 0x72, 0xdb, 0xa8, 0x00, 0x00, 0x00, 0x50, 0x8a, 0x0b, 0x1e, 0x00, 0x84, 0x8b, 0x72, 0xc0, 0x03, 0x00, 0x00, 0x00, 0x1a, 0xe9, 0x81, 0x80, 0x39, 0x02,
        0x50, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d,
        0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e,
        0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10,
        0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78,
        0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21, 0x23, 0x45, 0x44, 0x00, 0x66, 0x00, 0xc0, 0xdc, 0x00, 0x80, 0xd9, 0x01, 0x00,
        0x0c, 0x79, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x28, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00, 0x10,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x33, 0x01, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x80, 0x21, 0x0f, 0x06, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e, 0x0d, 0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb2, 0x40, 0x00, 0x00,
        0x0f, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x02, 0x25, 0x30, 0x02, 0x50, 0x08, 0xc5, 0x50, 0x16, 0x65, 0x50, 0x0e, 0xa5,
        0x50, 0x80, 0x01, 0xd4, 0x0b, 0x14, 0x88, 0xe4, 0x08, 0x00, 0x8d, 0x19, 0x00, 0x22, 0x33, 0x00, 0x54, 0x66, 0x00, 0xc8, 0xcc, 0x00, 0x90, 0x98, 0x01, 0xa0, 0x30, 0x03, 0x00, 0x00, 0x00, 0x00,
        0x79, 0x18, 0x00, 0x00, 0x56, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x83, 0x0c, 0x6f, 0x0c, 0x24, 0xc6, 0x45, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x00, 0x65, 0x82, 0x00,
        0x2c, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x00, 0xcc, 0x06, 0x61, 0x30, 0x28, 0x8c, 0xcd, 0x4d, 0x10, 0x80, 0x66, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0xc8, 0x38, 0x88, 0xd5, 0x99, 0x4d, 0x10, 0x00,
        0x67, 0x43, 0x42, 0x2c, 0x0c, 0x41, 0x0c, 0x0d, 0x01, 0x6c, 0x08, 0x9c, 0x09, 0x02, 0x61, 0x71, 0xa8, 0x0b, 0xb3, 0x9b, 0x20, 0x48, 0xd4, 0x86, 0x85, 0x80, 0x22, 0x82, 0x18, 0x1a, 0x49, 0x92,
        0x80, 0x0d, 0xc1, 0x34, 0x41, 0x30, 0x30, 0x12, 0x61, 0x72, 0x67, 0x73, 0x1b, 0x10, 0xa2, 0xb2, 0x08, 0x62, 0x30, 0x80, 0x0d, 0xc1, 0xb5, 0x81, 0x78, 0x28, 0x0c, 0x98, 0x20, 0x04, 0x1a, 0x03,
        0xb4, 0x09, 0x02, 0xf0, 0x4c, 0x10, 0x00, 0x68, 0x83, 0x91, 0x6c, 0x1c, 0xd1, 0x19, 0x1b, 0x04, 0xc3, 0x9b, 0x20, 0x0c, 0xd5, 0x04, 0xa1, 0xb8, 0x68, 0xcc, 0xd1, 0xc9, 0xa5, 0x91, 0x95, 0x4d,
        0x10, 0x80, 0x68, 0x83, 0x91, 0x88, 0x01, 0x47, 0x74, 0x63, 0xb0, 0x41, 0x30, 0xc8, 0x60, 0xc3, 0x41, 0x68, 0x1f, 0x18, 0x7c, 0x61, 0x50, 0x06, 0x13, 0x84, 0x64, 0xd8, 0x00, 0x6c, 0x18, 0x08,
        0x34, 0x40, 0x83, 0x0d, 0x41, 0x1a, 0x6c, 0x18, 0x86, 0x33, 0x50, 0x03, 0x12, 0x6d, 0x61, 0x69, 0x6e, 0x13, 0x84, 0x6e, 0x9a, 0x20, 0x00, 0xd2, 0x86, 0xc1, 0x0d, 0x86, 0x61, 0x03, 0x41, 0xb4,
        0x81, 0xf1, 0x06, 0x1b, 0x8a, 0x33, 0x60, 0x03, 0x20, 0x83, 0x83, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc,
        0x5c, 0xda, 0x9b, 0xdb, 0x94, 0x80, 0x68, 0x42, 0x86, 0xe7, 0x62, 0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x30, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91,
        0x95, 0xb1, 0x4d, 0x09, 0x92, 0x32, 0x64, 0x78, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73, 0x53, 0x02, 0xac, 0x12, 0x19, 0x9e, 0x0b, 0x5d, 0x1e, 0x5c, 0x59, 0x90, 0x9b, 0xdb,
        0x1b, 0x5d, 0x18, 0x5d, 0xda, 0x9b, 0xdb, 0xdc, 0x14, 0xa1, 0x0c, 0xd4, 0xa0, 0x0e, 0x19, 0x9e, 0x4b, 0x99, 0x1b, 0x9d, 0x5c, 0x1e, 0xd4, 0x5b, 0x9a, 0x1b, 0xdd, 0xdc, 0x94, 0x00, 0x0e, 0x00,
        0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73,
        0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b,
        0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20,
        0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61,
        0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87,
        0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98,
        0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61,
        0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b,
        0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8,
        0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x2f, 0x00, 0x00, 0x00,
        0x36, 0x40, 0x30, 0x00, 0xd2, 0x58, 0xc1, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x40, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0x94, 0x84, 0x01, 0x08, 0x98, 0x8f, 0xdc, 0xb6, 0x19, 0x80, 0xc1, 0x70,
        0xf9, 0xce, 0xe3, 0x0b, 0x07, 0x21, 0x28, 0x9a, 0x20, 0x10, 0x92, 0xf4, 0x51, 0xcb, 0x82, 0x99, 0xc0, 0xf3, 0x10, 0x83, 0x8f, 0xdc, 0xb6, 0x09, 0x10, 0x40, 0x64, 0x48, 0x86, 0x40, 0x0d, 0x97,
        0xef, 0x3c, 0x7e, 0x40, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0xf8, 0xc8, 0x6d, 0x57, 0xc0, 0x17, 0xf8, 0xd5, 0xc3, 0x76, 0x20, 0x70, 0x56, 0x95, 0x5e, 0x85, 0x79, 0x7a, 0x39, 0x48, 0x26, 0xcb,
        0xcb, 0xf3, 0xb9, 0xb0, 0x6e, 0x36, 0x97, 0xe5, 0x40, 0x60, 0xd0, 0x08, 0xa4, 0xe1, 0xf2, 0x9d, 0xc7, 0x9f, 0x8e, 0x88, 0x00, 0x06, 0x71, 0xf0, 0x91, 0xdb, 0x2e, 0x80, 0x2e, 0xf0, 0x13, 0xd7,
        0xcd, 0x40, 0xe0, 0xac, 0x2a, 0xcc, 0xd3, 0xcb, 0x41, 0x32, 0x59, 0x5e, 0x9e, 0xcf, 0x85, 0x75, 0xb3, 0xb9, 0x2c, 0x07, 0x02, 0x83, 0x96, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0x7e, 0x40, 0x15, 0x05,
        0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x76, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x04, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00,
        0x35, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x04, 0x66, 0x00, 0x4a, 0xae, 0x74, 0x03, 0xca, 0xae, 0x10, 0x03, 0xca, 0x31, 0xa0, 0x14, 0x03,
        0xa8, 0x95, 0xc0, 0x08, 0x40, 0x11, 0x94, 0x01, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x50, 0x7d, 0x0a, 0xb1, 0x6d, 0xd2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x54, 0x60,
        0xb0, 0x10, 0x1c, 0x37, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x41, 0x15, 0x06, 0x0c, 0xd1, 0x75, 0xd4, 0x88, 0x81, 0x01, 0x80, 0x20, 0x18, 0x28, 0x65, 0xc0, 0x78, 0x23, 0x06, 0x07, 0x00, 0x82,
        0x60, 0x00, 0x95, 0x01, 0x23, 0x7c, 0xa3, 0x09, 0x01, 0x50, 0xc1, 0x20, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x80, 0x95, 0x81, 0x63, 0x04, 0xd2, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc4, 0x88,
        0x81, 0x01, 0x80, 0x20, 0x18, 0x4c, 0x6a, 0x00, 0x05, 0x17, 0x80, 0xa8, 0x64, 0x0c, 0x6e, 0xc4, 0xa0, 0x01, 0x40, 0x10, 0x0c, 0x34, 0x35, 0x90, 0x9a, 0x00, 0x2b, 0x04, 0x0c, 0x73, 0x30, 0x1c,
        0x08, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x76, 0x22, 0x05, 0x54, 0xf1, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0xf1, 0x47, 0x80, 0x15, 0x50, 0x85, 0xa1, 0x30, 0x40, 0x64, 0x48, 0xbf, 0x10, 0x50,
        0x45, 0x41, 0x44, 0x66, 0xe2, 0x50, 0x40, 0xf5, 0xbb, 0xda, 0xfb, 0x47, 0x80, 0x15, 0x50, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
#if 0
    StructuredBuffer<uint> buf : register(t0);
    RWByteAddressBuffer uav : register(u0);

    cbuffer args : register(b0)
    {
        uint stride;
    };

    [numthreads(64, 1, 1)]
    void main(uint3 tid : SV_DISPATCHTHREADID)
    {
        uint fb;
        uint v = buf.Load(tid.x * stride, fb);
        uint s = CheckAccessFullyMapped(fb) ? 1 : 0;
        uav.Store2(8 * tid.x, uint2(v, s));
    }
#endif
    static const DWORD cs_ld_structured_dxbc[] =
    {
        0x43425844, 0x5b283603, 0xf1cbfe03, 0x2e0ecde6, 0x0d858acc, 0x00000001, 0x00000184, 0x00000004,
        0x00000030, 0x00000040, 0x00000050, 0x00000174, 0x4e475349, 0x00000008, 0x00000000, 0x00000008,
        0x4e47534f, 0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x0000011c, 0x00050050, 0x00000047,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x040000a2, 0x00107000, 0x00000000,
        0x00000004, 0x0300009d, 0x0011e000, 0x00000000, 0x0200005f, 0x00020012, 0x02000068, 0x00000002,
        0x0400009b, 0x00000040, 0x00000001, 0x00000001, 0x08000026, 0x0000d000, 0x00100012, 0x00000000,
        0x0002000a, 0x0020800a, 0x00000000, 0x00000000, 0x8d0000e3, 0x80002302, 0x00199983, 0x00100012,
        0x00000000, 0x00100012, 0x00000001, 0x0010000a, 0x00000000, 0x00004001, 0x00000000, 0x00107006,
        0x00000000, 0x050000ea, 0x00100042, 0x00000000, 0x0010000a, 0x00000001, 0x09000037, 0x00100022,
        0x00000000, 0x0010002a, 0x00000000, 0x00004001, 0x00000001, 0x00004001, 0x00000000, 0x06000029,
        0x00100042, 0x00000000, 0x0002000a, 0x00004001, 0x00000003, 0x070000a6, 0x0011e032, 0x00000000,
        0x0010002a, 0x00000000, 0x00100046, 0x00000000, 0x0100003e, 0x30494653, 0x00000008, 0x00000100,
        0x00000000,
    };
    static const BYTE cs_ld_structured_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x1d, 0xe5, 0x6d, 0xa8, 0x2f, 0x74, 0x1a, 0xfa, 0x70, 0xfd, 0x2b, 0xbf, 0x11, 0x95, 0xa0, 0x75, 0x01, 0x00, 0x00, 0x00, 0x10, 0x08, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x6c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x30, 0x07, 0x00, 0x00,
        0x60, 0x00, 0x05, 0x00, 0xcc, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x18, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00,
        0xc3, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c,
        0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20,
        0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff,
        0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00,
        0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x73, 0x04,
        0x60, 0x30, 0x13, 0x19, 0x8c, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xcc, 0x02, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x03, 0x39, 0x88, 0x42, 0x3d, 0x98,
        0x83, 0x39, 0x94, 0x83, 0x3c, 0xf0, 0x41, 0x3d, 0xb8, 0xc3, 0x3c, 0xa4, 0xc3, 0x39, 0xb8, 0x43, 0x39, 0x90, 0x03, 0x18, 0xa4, 0x83, 0x3b, 0xd0, 0x83, 0x1f, 0xa0, 0x00, 0x50, 0x38, 0x4d, 0x9a,
        0x22, 0x4a, 0x98, 0xfc, 0x15, 0xde, 0xb0, 0x89, 0xd0, 0x86, 0x21, 0x22, 0x24, 0x69, 0xa3, 0x8a, 0x82, 0x88, 0x50, 0x00, 0x68, 0x1c, 0x01, 0x44, 0x86, 0x84, 0x02, 0x40, 0x65, 0x04, 0xa0, 0x04,
        0x87, 0x50, 0x19, 0x00, 0x00, 0xa4, 0x8e, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0xb9, 0x8d, 0x2a, 0x56, 0x62, 0xf2, 0x91, 0xdb, 0x46, 0x04, 0x00, 0x00, 0xcc, 0x11, 0x20, 0xd4, 0xee, 0x19,
        0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x43, 0xa0, 0x19, 0x16, 0x02, 0x05, 0xae, 0x10, 0x0c, 0x3c, 0x40, 0x70, 0x8e, 0x20, 0x28, 0x83, 0x04, 0x80, 0x66, 0x31, 0x1e, 0x68, 0x00, 0x48, 0xaa, 0x37,
        0x0d, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x2b, 0x21, 0xad, 0xc4, 0xe4, 0x23, 0xb7, 0x8d, 0x0a, 0x00, 0x00, 0x00, 0xa5, 0xb8, 0xe0, 0x01, 0x40, 0xb8, 0x28, 0x07, 0x3c, 0x00, 0x00, 0x00, 0xa0,
        0x91, 0x1e, 0x08, 0x98, 0x23, 0x00, 0x05, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e,
        0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20,
        0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74,
        0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07,
        0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21, 0x23, 0x45, 0x44, 0x00, 0x66, 0x00, 0xc0, 0xdc, 0x00,
        0x80, 0xd9, 0x01, 0x00, 0x0c, 0x79, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x28, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4,
        0x81, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x33, 0x01, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x40, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x06, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e, 0x0d, 0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb2, 0x40, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x02, 0x25, 0x30, 0x02, 0x50, 0x08, 0xc5, 0x50, 0x18,
        0x65, 0x51, 0x06, 0xe5, 0x50, 0x0a, 0x05, 0x18, 0x40, 0xbd, 0x40, 0x81, 0x48, 0x8e, 0x00, 0xd0, 0x98, 0x01, 0x20, 0x32, 0x03, 0x40, 0x65, 0x06, 0x80, 0xcc, 0x0c, 0x00, 0x89, 0x19, 0x00, 0x0a,
        0x33, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x5a, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x83, 0x0c, 0x6f, 0x0c, 0x24, 0xc6, 0x45, 0x66, 0x43, 0x10, 0x4c, 0x10,
        0x00, 0x65, 0x82, 0x00, 0x2c, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x00, 0xcc, 0x06, 0x61, 0x30, 0x28, 0x8c, 0xcd, 0x4d, 0x10, 0x80, 0x66, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0xd0, 0x38, 0x88, 0xd5,
        0x99, 0x4d, 0x10, 0x00, 0x67, 0x43, 0x42, 0x2c, 0x0c, 0x41, 0x0c, 0x0d, 0x71, 0x6c, 0x08, 0x9c, 0x09, 0x02, 0x71, 0x71, 0xa8, 0x0b, 0xb3, 0x9b, 0x20, 0x00, 0xcf, 0x04, 0x41, 0xaa, 0x36, 0x2c,
        0x04, 0x14, 0x11, 0xc4, 0x20, 0x4d, 0xd3, 0x04, 0x6c, 0x08, 0xa8, 0x09, 0x82, 0x91, 0x91, 0x08, 0x93, 0x3b, 0x9b, 0xdb, 0x80, 0x10, 0xd6, 0x45, 0x10, 0x83, 0x01, 0x6c, 0x08, 0xb0, 0x0d, 0xc4,
        0x53, 0x65, 0xc0, 0x04, 0x21, 0xd8, 0x18, 0xa0, 0x4d, 0x10, 0x00, 0x68, 0x82, 0x00, 0x44, 0x13, 0x04, 0x40, 0xda, 0x60, 0x24, 0x5c, 0x47, 0x78, 0xdf, 0x06, 0xc1, 0x00, 0x83, 0x09, 0xc2, 0x60,
        0x6d, 0x30, 0x12, 0xae, 0x23, 0x3c, 0x63, 0x83, 0x60, 0x8c, 0xc1, 0x04, 0xa1, 0xc0, 0x68, 0xcc, 0xd1, 0xc9, 0xa5, 0x91, 0x95, 0x6d, 0x30, 0x12, 0x33, 0xe8, 0x08, 0xef, 0xdb, 0x20, 0x18, 0x67,
        0xb0, 0xe1, 0x20, 0xb6, 0x30, 0x10, 0x03, 0x32, 0x28, 0x03, 0x34, 0x98, 0x20, 0x24, 0xc3, 0x06, 0x60, 0xc3, 0x40, 0xac, 0xc1, 0x1a, 0x6c, 0x08, 0xd8, 0x60, 0xc3, 0x30, 0xa8, 0x41, 0x1b, 0x90,
        0x68, 0x0b, 0x4b, 0x73, 0x9b, 0x20, 0x74, 0xd4, 0x04, 0x01, 0x98, 0x36, 0x0c, 0x71, 0x30, 0x0c, 0x1b, 0x08, 0x02, 0x0e, 0x0c, 0x39, 0xd8, 0x50, 0xa8, 0xc1, 0x1b, 0x00, 0xda, 0x1c, 0x54, 0x61,
        0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30,
        0x36, 0xbb, 0x32, 0xb9, 0x29, 0x81, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x90, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b,
        0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x64, 0x95, 0xc8, 0xf0, 0x5c, 0xe8, 0xf2, 0xe0, 0xca, 0x82, 0xdc, 0xdc, 0xde, 0xe8, 0xc2, 0xe8, 0xd2, 0xde, 0xdc, 0xe6, 0xa6, 0x08, 0x68, 0xd0, 0x06,
        0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x73, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c,
        0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e,
        0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c,
        0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e,
        0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4,
        0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07,
        0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5,
        0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90,
        0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b,
        0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90,
        0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x2f, 0x00, 0x00, 0x00, 0x05, 0x00, 0x06, 0x81, 0x9f, 0xb8, 0x6e, 0x06, 0x02, 0x67, 0xd6, 0x1f,
        0x69, 0x4a, 0x97, 0xd7, 0xc7, 0xf4, 0xba, 0xbc, 0x4c, 0x16, 0xd6, 0xcd, 0xe6, 0xb2, 0x1c, 0x98, 0x04, 0x02, 0x83, 0x36, 0x40, 0x30, 0x00, 0xd2, 0x58, 0xc1, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x42,
        0x40, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0x94, 0x84, 0x01, 0x08, 0x98, 0x8f, 0xdc, 0xb6, 0x19, 0x80, 0xc1, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x07, 0x21, 0x28, 0x9a, 0x20, 0x10, 0x92, 0xf4, 0x51,
        0xcb, 0x82, 0x99, 0xc0, 0xf3, 0x10, 0x83, 0x8f, 0xdc, 0xb6, 0x09, 0x10, 0x40, 0x64, 0x48, 0x86, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x7e, 0x40, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0xf8, 0xc8, 0x6d,
        0x57, 0xc0, 0x17, 0xf8, 0xd5, 0xc3, 0x76, 0x20, 0x70, 0x56, 0x95, 0x5e, 0x85, 0x79, 0x7a, 0x39, 0x48, 0x26, 0xcb, 0xcb, 0xf3, 0xb9, 0xb0, 0x6e, 0x36, 0x97, 0xe5, 0x40, 0x60, 0xd0, 0x08, 0xa4,
        0xe1, 0xf2, 0x9d, 0xc7, 0x9f, 0x8e, 0x88, 0x00, 0x06, 0x71, 0xf0, 0x91, 0xdb, 0xb6, 0x03, 0x69, 0xb8, 0x7c, 0xe7, 0xf1, 0x85, 0x88, 0x00, 0x26, 0x22, 0x04, 0x9a, 0x61, 0x21, 0x2c, 0xa1, 0x1a,
        0x2e, 0xdf, 0x79, 0xfc, 0x80, 0x2a, 0x0a, 0x22, 0x62, 0x27, 0x27, 0x22, 0x7c, 0xe4, 0xb6, 0x01, 0x61, 0x20, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00,
        0x07, 0x00, 0x00, 0x00, 0x04, 0x4a, 0x6e, 0x06, 0xa0, 0x74, 0x03, 0xca, 0xae, 0x10, 0x03, 0xca, 0x31, 0xa0, 0x14, 0x03, 0xa8, 0x95, 0xc0, 0x08, 0x40, 0x11, 0x94, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x50, 0x81, 0xc1, 0x42, 0x70, 0x9c, 0x34, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x55, 0x18, 0x30, 0x44, 0xd7, 0x4d, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x50,
        0x89, 0x41, 0x43, 0x78, 0x1e, 0x35, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x8a, 0x19, 0x30, 0xdf, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x40, 0x66, 0xc0, 0x08, 0x60, 0x30, 0x9a, 0x10, 0x00, 0x15,
        0x0c, 0x32, 0x62, 0x80, 0x00, 0x20, 0x08, 0x06, 0x98, 0x19, 0x38, 0x46, 0x30, 0x06, 0xa3, 0x09, 0x01, 0x30, 0x9a, 0x20, 0x10, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x30, 0xad, 0x01, 0x14, 0x5c,
        0x00, 0xa2, 0x92, 0x31, 0xb8, 0x11, 0x83, 0x06, 0x00, 0x41, 0x30, 0xd0, 0xd6, 0x40, 0x6a, 0x82, 0xab, 0x10, 0xae, 0xcb, 0xc1, 0x70, 0x20, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x96, 0xc2, 0x00, 0x91,
        0x21, 0xfd, 0x42, 0x40, 0x15, 0x05, 0x11, 0xd9, 0x89, 0x43, 0x01, 0xd5, 0xef, 0x6a, 0xef, 0x1f, 0x01, 0x56, 0x40, 0x15, 0x86, 0x52, 0x05, 0x54, 0xf1, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0xf1, 0x4b,
        0x53, 0x44, 0x09, 0x53, 0x40, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const struct
    {
        D3D12_SHADER_BYTECODE cs_dxbc;
        D3D12_SHADER_BYTECODE cs_dxil;
        bool is_structured;
        bool is_raw;
        bool is_uav;
    }
    tests[] =
    {
        { SHADER_BYTECODE(cs_ld_typed_dxbc),      SHADER_BYTECODE(cs_ld_typed_dxil),      false, false, false },
        { SHADER_BYTECODE(cs_ld_typed_uav_dxbc),  SHADER_BYTECODE(cs_ld_typed_uav_dxil),  false, false, true  },
        { SHADER_BYTECODE(cs_ld_raw_dxbc),        SHADER_BYTECODE(cs_ld_raw_dxil),        false, true,  false },
        { SHADER_BYTECODE(cs_ld_structured_dxbc), SHADER_BYTECODE(cs_ld_structured_dxil), true,  false, false },
    };

    struct shader_args
    {
        uint32_t stride;
    }
    shader_args;

    memset(&desc, 0, sizeof(desc));
    desc.no_render_target = true;
    if (!init_test_context(&context, &desc))
        return;

    if (use_dxil && !context_supports_dxil(&context))
    {
        destroy_test_context(&context);
        return;
    }

    hr = ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_D3D12_OPTIONS, &options, sizeof(options));
    ok(hr == S_OK, "Failed to check feature support, hr %#x.\n", hr);

    if (options.TiledResourcesTier < D3D12_TILED_RESOURCES_TIER_2)
    {
        skip("Tiled resources Tier 2 not supported by device.\n");
        destroy_test_context(&context);
        return;
    }

    descriptor_ranges[0].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_SRV;
    descriptor_ranges[0].NumDescriptors = 1;
    descriptor_ranges[0].BaseShaderRegister = 0;
    descriptor_ranges[0].RegisterSpace = 0;
    descriptor_ranges[0].OffsetInDescriptorsFromTableStart = 0;
    descriptor_ranges[1].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_UAV;
    descriptor_ranges[1].NumDescriptors = 1;
    descriptor_ranges[1].BaseShaderRegister = 1;
    descriptor_ranges[1].RegisterSpace = 0;
    descriptor_ranges[1].OffsetInDescriptorsFromTableStart = 1;
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[0].DescriptorTable.NumDescriptorRanges = ARRAY_SIZE(descriptor_ranges);
    root_parameters[0].DescriptorTable.pDescriptorRanges = descriptor_ranges;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
    root_parameters[1].Descriptor.ShaderRegister = 0;
    root_parameters[1].Descriptor.RegisterSpace = 0;
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[2].ParameterType = D3D12_ROOT_PARAMETER_TYPE_32BIT_CONSTANTS;
    root_parameters[2].Constants.Num32BitValues = sizeof(shader_args) / sizeof(uint32_t);
    root_parameters[2].Constants.ShaderRegister = 0;
    root_parameters[2].Constants.RegisterSpace = 0;
    root_parameters[2].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_signature_desc.NumParameters = ARRAY_SIZE(root_parameters);
    root_signature_desc.pParameters = root_parameters;
    root_signature_desc.NumStaticSamplers = 0;
    root_signature_desc.pStaticSamplers = NULL;
    root_signature_desc.Flags = D3D12_ROOT_SIGNATURE_FLAG_NONE;
    hr = create_root_signature(context.device, &root_signature_desc, &root_signature);
    ok(hr == S_OK, "Failed to create root signature, hr %#x.\n", hr);

    memset(&heap_desc, 0, sizeof(heap_desc));
    heap_desc.Properties.Type = D3D12_HEAP_TYPE_DEFAULT;
    heap_desc.SizeInBytes = TILE_SIZE * 2;

    resource_desc.Dimension = D3D12_RESOURCE_DIMENSION_BUFFER;
    resource_desc.Alignment = 0;
    resource_desc.Width = 128 * sizeof(uint32_t);
    resource_desc.Height = 1;
    resource_desc.DepthOrArraySize = 1;
    resource_desc.MipLevels = 1;
    resource_desc.Format = DXGI_FORMAT_UNKNOWN;
    resource_desc.SampleDesc.Count = 1;
    resource_desc.SampleDesc.Quality = 0;
    resource_desc.Layout = D3D12_TEXTURE_LAYOUT_ROW_MAJOR;
    resource_desc.Flags = D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS;
    hr = ID3D12Device_CreateCommittedResource(context.device, &heap_desc.Properties, D3D12_HEAP_FLAG_NONE,
            &resource_desc, D3D12_RESOURCE_STATE_COPY_SOURCE, NULL, &IID_ID3D12Resource, (void **)&out_buffer);
    ok(hr == S_OK, "Failed to create committed resource, hr %#x.\n", hr);

    resource_desc.Width = 4 * TILE_SIZE;
    hr = ID3D12Device_CreateReservedResource(context.device, &resource_desc,
            D3D12_RESOURCE_STATE_UNORDERED_ACCESS, NULL, &IID_ID3D12Resource, (void **)&tiled_buffer);
    ok(hr == S_OK, "Failed to create reserved resource, hr %#x.\n", hr);

    hr = ID3D12Device_CreateHeap(context.device, &heap_desc, &IID_ID3D12Heap, (void **)&heap);
    ok(hr == S_OK, "Failed to create heap, hr %#x.\n", hr);

    /* Map the 0k-64k range and the 128k-192k range, leave the rest unmapped */
    set_region_offset(&tile_regions[0], 0, 0, 0, 0);
    set_region_offset(&tile_regions[1], 2, 0, 0, 0);
    tile_offset = 0;

    ID3D12CommandQueue_UpdateTileMappings(context.queue, tiled_buffer,
            ARRAY_SIZE(tile_regions), tile_regions, NULL, heap, 1, NULL,
            &tile_offset, NULL, D3D12_TILE_MAPPING_FLAG_NONE);

    gpu_heap = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 2);
    cpu_heap = create_cpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 1);

    uav_desc.ViewDimension = D3D12_UAV_DIMENSION_BUFFER;
    uav_desc.Format = DXGI_FORMAT_R32_UINT;
    uav_desc.Buffer.FirstElement = 0;
    uav_desc.Buffer.NumElements = resource_desc.Width / sizeof(uint32_t);
    uav_desc.Buffer.StructureByteStride = 0;
    uav_desc.Buffer.CounterOffsetInBytes = 0;
    uav_desc.Buffer.Flags = D3D12_BUFFER_UAV_FLAG_NONE;

    ID3D12Device_CreateUnorderedAccessView(context.device, tiled_buffer, NULL, &uav_desc, get_cpu_descriptor_handle(&context, cpu_heap, 0));
    ID3D12Device_CreateUnorderedAccessView(context.device, tiled_buffer, NULL, &uav_desc, get_cpu_descriptor_handle(&context, gpu_heap, 1));

    for (i = 0; i < resource_desc.Width / TILE_SIZE; i++)
    {
        UINT clear_values[] = { i + 1, 0, 0, 0 };
        D3D12_RECT rect;
        set_rect(&rect, i * TILE_SIZE / sizeof(uint32_t), 0, (i + 1) * TILE_SIZE / sizeof(uint32_t), 1);

        ID3D12GraphicsCommandList_ClearUnorderedAccessViewUint(context.list,
                get_gpu_descriptor_handle(&context, gpu_heap, 1),
                get_cpu_descriptor_handle(&context, cpu_heap, 0),
                tiled_buffer, clear_values, 1, &rect);
    };

    transition_resource_state(context.list, tiled_buffer,
            D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE);

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        vkd3d_test_set_context("Test %u", i);
        test_is_raw = (i == 2) || (i == 3);

        todo_if(use_dxil && test_is_raw)
        pipeline_state = create_compute_pipeline_state(context.device,
                root_signature, use_dxil ? tests[i].cs_dxil : tests[i].cs_dxbc);

        /* This will fail for SSBO buffer feedback case on DXIL. */
        todo_if(use_dxil && test_is_raw)
        ok(!!pipeline_state, "Failed to create pipeline state.\n");
        if (!pipeline_state)
            continue;

        srv_desc.ViewDimension = D3D12_SRV_DIMENSION_BUFFER;
        srv_desc.Format = tests[i].is_structured ? DXGI_FORMAT_UNKNOWN :
            (tests[i].is_raw ? DXGI_FORMAT_R32_TYPELESS : DXGI_FORMAT_R32_UINT);
        srv_desc.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;
        srv_desc.Buffer.FirstElement = 0;
        srv_desc.Buffer.NumElements = resource_desc.Width / sizeof(uint32_t);
        srv_desc.Buffer.StructureByteStride = tests[i].is_structured ? sizeof(uint32_t) : 0;
        srv_desc.Buffer.Flags = tests[i].is_raw ? D3D12_BUFFER_SRV_FLAG_RAW : D3D12_BUFFER_SRV_FLAG_NONE;

        uav_desc.ViewDimension = D3D12_UAV_DIMENSION_BUFFER;
        uav_desc.Format = srv_desc.Format;
        uav_desc.Buffer.FirstElement = srv_desc.Buffer.FirstElement;
        uav_desc.Buffer.NumElements = srv_desc.Buffer.NumElements;
        uav_desc.Buffer.StructureByteStride = srv_desc.Buffer.StructureByteStride;
        uav_desc.Buffer.CounterOffsetInBytes = 0;
        uav_desc.Buffer.Flags = tests[i].is_raw ? D3D12_BUFFER_UAV_FLAG_RAW : D3D12_BUFFER_UAV_FLAG_NONE;

        if (tests[i].is_uav)
        {
            ID3D12Device_CreateUnorderedAccessView(context.device, tiled_buffer, NULL, &uav_desc, get_cpu_descriptor_handle(&context, gpu_heap, 1));
            transition_resource_state(context.list, tiled_buffer, D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
        }
        else
            ID3D12Device_CreateShaderResourceView(context.device, tiled_buffer, &srv_desc, get_cpu_descriptor_handle(&context, gpu_heap, 0));

        transition_resource_state(context.list, out_buffer,
        D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

        shader_args.stride = resource_desc.Width / 64;

        if (!tests[i].is_raw)
            shader_args.stride /= sizeof(uint32_t);

        ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &gpu_heap);
        ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, root_signature);
        ID3D12GraphicsCommandList_SetPipelineState(context.list, pipeline_state);
        ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(context.list, 0, get_gpu_descriptor_handle(&context, gpu_heap, 0));
        ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context.list, 1, ID3D12Resource_GetGPUVirtualAddress(out_buffer));
        ID3D12GraphicsCommandList_SetComputeRoot32BitConstants(context.list, 2, sizeof(shader_args) / sizeof(uint32_t), &shader_args, 0);
        ID3D12GraphicsCommandList_Dispatch(context.list, 1, 1, 1);

        transition_resource_state(context.list, out_buffer,
                D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

        if (tests[i].is_uav)
        {
            transition_resource_state(context.list, tiled_buffer,
                    D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE);
        }

        get_buffer_readback_with_command_list(out_buffer, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

        for (j = 0; j < 64; j++)
        {
            D3D12_BOX box;
            UINT tile_index = j / 16;

            set_box(&box, 2 * j, 0, 0, 2 * j + 1, 1, 1);
            todo_if(test_is_raw) check_readback_data_uint(&rb, &box, (tile_index & 1) ? 0 : (tile_index + 1), 0);
            set_box(&box, 2 * j + 1, 0, 0, 2 * j + 2, 1, 1);
            todo_if(test_is_raw) check_readback_data_uint(&rb, &box, (tile_index & 1) ? 0 : 1, 0);
        }

        release_resource_readback(&rb);

        reset_command_list(context.list, context.allocator);

        ID3D12PipelineState_Release(pipeline_state);
    }

    ID3D12DescriptorHeap_Release(cpu_heap);
    ID3D12DescriptorHeap_Release(gpu_heap);
    ID3D12Heap_Release(heap);
    ID3D12Resource_Release(tiled_buffer);
    ID3D12Resource_Release(out_buffer);
    ID3D12RootSignature_Release(root_signature);
    destroy_test_context(&context);
#undef TILE_SIZE
}

void test_buffer_feedback_instructions_sm51(void)
{
    test_buffer_feedback_instructions(false);
}

void test_buffer_feedback_instructions_dxil(void)
{
    test_buffer_feedback_instructions(true);
}

static void test_texture_feedback_instructions(bool use_dxil)
{
#define TILE_SIZE 65536
    ID3D12DescriptorHeap *gpu_heap, *sampler_heap, *rtv_heap;
    ID3D12Resource *tiled_image, *color_rt, *residency_rt;
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pipeline_desc;
    D3D12_TILED_RESOURCE_COORDINATE tile_regions[3];
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_DESCRIPTOR_RANGE descriptor_ranges[3];
    D3D12_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    D3D12_FEATURE_DATA_D3D12_OPTIONS options;
    D3D12_SHADER_RESOURCE_VIEW_DESC srv_desc;
    D3D12_ROOT_PARAMETER root_parameters[3];
    D3D12_RENDER_TARGET_VIEW_DESC rtv_desc;
    ID3D12PipelineState *pipeline_state;
    ID3D12RootSignature *root_signature;
    D3D12_RESOURCE_DESC resource_desc;
    D3D12_SAMPLER_DESC sampler_desc;
    struct test_context_desc desc;
    struct resource_readback rb;
    struct test_context context;
    D3D12_HEAP_DESC heap_desc;
    D3D12_VIEWPORT viewport;
    D3D12_RECT scissor;
    ID3D12Heap *heap;
    UINT tile_offset;
    unsigned int i;
    HRESULT hr;

#if 0
    void main(uint idx : SV_VERTEXID, out float4 pos : SV_POSITION, out float2 uv : UV_TEXCOORD)
    {
        uv = float2((idx << 1) & 2, idx & 2);
        pos = float4(2.0f * uv - 1.0f, 0.0f, 1.0f);
    }
#endif
    static const DWORD vs_code_dxbc[] =
    {
        0x43425844, 0x7964a10a, 0x2d3d08ae, 0x83dfabe2, 0x129879ea, 0x00000001, 0x00000204, 0x00000003,
        0x0000002c, 0x00000060, 0x000000b8, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000006, 0x00000001, 0x00000000, 0x00000101, 0x565f5653, 0x45545245, 0x00444958,
        0x4e47534f, 0x00000050, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x00000044, 0x00000000, 0x00000000, 0x00000003, 0x00000001, 0x00000c03,
        0x505f5653, 0x5449534f, 0x004e4f49, 0x545f5655, 0x4f435845, 0x0044524f, 0x58454853, 0x00000144,
        0x00010050, 0x00000051, 0x0100086a, 0x04000060, 0x00101012, 0x00000000, 0x00000006, 0x04000067,
        0x001020f2, 0x00000000, 0x00000001, 0x03000065, 0x00102032, 0x00000001, 0x02000068, 0x00000001,
        0x07000001, 0x00100012, 0x00000000, 0x0010100a, 0x00000000, 0x00004001, 0x00000002, 0x05000056,
        0x00100022, 0x00000000, 0x0010000a, 0x00000000, 0x08000000, 0x00100042, 0x00000000, 0x8010001a,
        0x00000041, 0x00000000, 0x00004001, 0x3f800000, 0x0b00008c, 0x00100082, 0x00000000, 0x00004001,
        0x00000001, 0x00004001, 0x00000001, 0x0010100a, 0x00000000, 0x00004001, 0x00000000, 0x05000056,
        0x00100012, 0x00000000, 0x0010003a, 0x00000000, 0x0f000032, 0x00102032, 0x00000000, 0x00100086,
        0x00000000, 0x00004002, 0x40000000, 0x40000000, 0x00000000, 0x00000000, 0x00004002, 0xbf800000,
        0xbf800000, 0x00000000, 0x00000000, 0x05000036, 0x00102032, 0x00000001, 0x00100046, 0x00000000,
        0x08000036, 0x001020c2, 0x00000000, 0x00004002, 0x00000000, 0x00000000, 0x00000000, 0x3f800000,
        0x0100003e, 
    };
    static const BYTE vs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xbd, 0xa3, 0x7e, 0x42, 0x8b, 0xca, 0xb7, 0x1e, 0xe1, 0x6a, 0x91, 0xef, 0xe8, 0xfe, 0xd3, 0xa9, 0x01, 0x00, 0x00, 0x00, 0x9c, 0x06, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x7c, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x56, 0x65, 0x72, 0x74, 0x65, 0x78, 0x49, 0x44, 0x00,
        0x4f, 0x53, 0x47, 0x31, 0x60, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x55, 0x56, 0x5f, 0x54,
        0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x50, 0x53, 0x56, 0x30, 0x8c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x01, 0x02, 0x00, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x00, 0x55, 0x56, 0x5f, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x41, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c,
        0x18, 0x05, 0x00, 0x00, 0x60, 0x00, 0x01, 0x00, 0x46, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde,
        0x21, 0x0c, 0x00, 0x00, 0x3d, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39,
        0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x10, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0x84, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x42, 0x88,
        0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x21, 0x46, 0x06,
        0x51, 0x18, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00,
        0x49, 0x18, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x22, 0x08, 0x09, 0x20, 0x64, 0x85, 0x04,
        0x13, 0x22, 0xa4, 0x84, 0x04, 0x13, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x88, 0x8c, 0x0b, 0x84, 0x84, 0x4c, 0x10, 0x30, 0x23, 0x00, 0x25, 0x00, 0x8a, 0x39, 0x02, 0x30, 0x98, 0x23,
        0x40, 0x8a, 0x31, 0x33, 0x43, 0x43, 0x35, 0x03, 0x50, 0x0c, 0x98, 0x19, 0x3a, 0xc2, 0x81, 0x80, 0x1c, 0x18, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79,
        0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0,
        0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73,
        0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07,
        0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x05, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x02, 0x01, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x10, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x9a, 0x12, 0x18, 0x01, 0x28,
        0x84, 0x62, 0x20, 0x2a, 0x85, 0x12, 0x18, 0x01, 0x28, 0x89, 0x32, 0x28, 0x84, 0x22, 0xa0, 0x1d, 0x6b, 0x08, 0x8c, 0x39, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
        0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x83, 0x0c, 0x6f, 0x0c, 0x24, 0xc6, 0x45, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x86, 0x61, 0x82, 0x30, 0x10, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xc5,
        0x06, 0x61, 0x30, 0x28, 0xd8, 0xcd, 0x4d, 0x10, 0x06, 0x63, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x41, 0x00, 0x36, 0x00, 0x1b, 0x06, 0x82, 0x61, 0x36, 0x04, 0xcd, 0x86, 0x61, 0x58, 0x9c, 0x09, 0xc2,
        0xe2, 0x6c, 0x08, 0x20, 0x12, 0x6d, 0x61, 0x69, 0x6e, 0x5c, 0xa6, 0xac, 0xbe, 0xac, 0xca, 0xe4, 0xe8, 0xca, 0xf0, 0x92, 0x88, 0x26, 0x08, 0xc4, 0x31, 0x41, 0x20, 0x90, 0x0d, 0x01, 0x31, 0x41,
        0x20, 0x92, 0x0d, 0x0b, 0x31, 0x51, 0x95, 0x75, 0x0d, 0x15, 0x71, 0x01, 0x1b, 0x02, 0x8c, 0xcb, 0x94, 0xd5, 0x17, 0xd4, 0xdb, 0x5c, 0x1a, 0x5d, 0xda, 0x9b, 0xdb, 0x04, 0x81, 0x50, 0x26, 0x08,
        0xc4, 0x32, 0x41, 0x20, 0x98, 0x0d, 0x0b, 0xa1, 0x6d, 0x9c, 0xd5, 0x0d, 0x1d, 0x71, 0x01, 0x5c, 0xaa, 0xac, 0xbe, 0xa8, 0x8a, 0xb0, 0x86, 0x9e, 0x9e, 0xa4, 0x88, 0x26, 0x08, 0x44, 0xb3, 0x61,
        0x19, 0xbe, 0xed, 0xb2, 0xc0, 0x60, 0x00, 0x83, 0xe1, 0x02, 0x36, 0x08, 0x5e, 0x18, 0x6c, 0x18, 0x32, 0x31, 0x00, 0x36, 0x14, 0x8b, 0x34, 0x06, 0x00, 0x50, 0x85, 0x8d, 0xcd, 0xae, 0xcd, 0x25,
        0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54, 0x21, 0xc3, 0x73, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x9b, 0x12, 0x10, 0x4d, 0xc8, 0xf0, 0x5c, 0xec, 0xc2, 0xd8, 0xec, 0xca, 0xe4, 0xa6,
        0x04, 0x46, 0x1d, 0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30, 0xb2, 0x32, 0xb9, 0xa6, 0x37, 0xb2, 0x32, 0xb6, 0x29, 0x41, 0x52, 0x89, 0x0c, 0xcf, 0x85, 0x2e, 0x0f, 0xae, 0x2c, 0xc8, 0xcd, 0xed, 0x8d,
        0x2e, 0x8c, 0x2e, 0xed, 0xcd, 0x6d, 0x6e, 0x4a, 0xe0, 0xd4, 0x21, 0xc3, 0x73, 0xb1, 0x4b, 0x2b, 0xbb, 0x4b, 0x22, 0x9b, 0xa2, 0x0b, 0xa3, 0x2b, 0x9b, 0x12, 0x40, 0x75, 0xc8, 0xf0, 0x5c, 0xca,
        0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x63, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66,
        0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e,
        0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b,
        0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0,
        0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83,
        0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76,
        0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30,
        0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43,
        0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0x03, 0x00, 0x00,
        0x71, 0x20, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x16, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x23, 0xb7, 0x6d, 0x00, 0x04, 0x03, 0x20, 0x8d, 0x09, 0x54,
        0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x4d, 0x4e, 0x44, 0xa0, 0xd4, 0xf4, 0x50, 0x93, 0x5f, 0xdc, 0x36, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x2f, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c,
        0x10, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x34, 0xa5, 0x50, 0x04, 0x33, 0x00, 0x74, 0x63, 0x04, 0x20, 0x08, 0x82, 0xf8, 0x37, 0x46, 0x00, 0x82, 0x20, 0x08, 0x82, 0xc1, 0x18, 0x01, 0x08,
        0x82, 0x20, 0xfe, 0x0b, 0x23, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x50, 0x54, 0x91, 0x24, 0x35, 0x45, 0x05, 0xd4, 0x55, 0x80, 0xe8, 0x05, 0x57, 0x45, 0x28, 0x7a, 0xc1, 0x95,
        0x29, 0x41, 0x7c, 0x8c, 0x50, 0xe4, 0x63, 0xc2, 0x22, 0x1f, 0x13, 0x16, 0xf8, 0x98, 0xc0, 0xc0, 0x67, 0xc4, 0x20, 0x01, 0x40, 0x10, 0x0c, 0x90, 0x4f, 0xea, 0x3a, 0x4c, 0x18, 0x31, 0x48, 0x00,
        0x10, 0x04, 0x03, 0xe4, 0x93, 0xba, 0x2e, 0x0b, 0x46, 0x0c, 0x12, 0x00, 0x04, 0xc1, 0x00, 0xf9, 0xa4, 0xae, 0xa3, 0x98, 0x11, 0x83, 0x04, 0x00, 0x41, 0x30, 0x40, 0x3e, 0xa9, 0xeb, 0xac, 0x67,
        0xc4, 0x20, 0x01, 0x40, 0x10, 0x0c, 0x90, 0x4f, 0xf2, 0x3a, 0x0c, 0x19, 0x31, 0x48, 0x00, 0x10, 0x04, 0x03, 0xe4, 0x93, 0xbc, 0x2e, 0x33, 0x30, 0x1c, 0x08, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0xd6, 0x31, 0xf8, 0x08, 0x6d, 0x1e, 0x83, 0x8f, 0xd4, 0xf6, 0x41, 0xf8, 0x08, 0x5d, 0x1b, 0x08, 0xe1, 0x23, 0xb5, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    };
#if 0
    SamplerState samp : register(s0);
    Texture2D<float4> tex : register(t0);

    cbuffer args : register(b0)
    {
        float lod_clamp;
    };

    void main(float4 pos : SV_POSITION, float2 uv : UV_TEXCOORD, out float4 o0 : SV_TARGET0, out uint o1 : SV_TARGET1)
    {
        uint fb;
        o0 = tex.Sample(samp, uv, int2(0, 0), lod_clamp, fb);
        o1 = CheckAccessFullyMapped(fb) ? 1 : 0;
    }
#endif
    static const DWORD ps_sample_dxbc[] =
    {
        0x43425844, 0xb83eba7f, 0x7b287256, 0x98a2ccc3, 0x4fe6a5e0, 0x00000001, 0x000001e4, 0x00000004,
        0x00000030, 0x00000088, 0x000000d4, 0x000001d4, 0x4e475349, 0x00000050, 0x00000002, 0x00000008,
        0x00000038, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000,
        0x00000000, 0x00000003, 0x00000001, 0x00000303, 0x505f5653, 0x5449534f, 0x004e4f49, 0x545f5655,
        0x4f435845, 0x0044524f, 0x4e47534f, 0x00000044, 0x00000002, 0x00000008, 0x00000038, 0x00000000,
        0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x00000038, 0x00000001, 0x00000000, 0x00000001,
        0x00000001, 0x00000e01, 0x545f5653, 0x45475241, 0xabab0054, 0x58454853, 0x000000f8, 0x00000050,
        0x0000003e, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300005a, 0x00106000,
        0x00000000, 0x04001858, 0x00107000, 0x00000000, 0x00005555, 0x03001062, 0x00101032, 0x00000001,
        0x03000065, 0x001020f2, 0x00000000, 0x03000065, 0x00102012, 0x00000001, 0x02000068, 0x00000002,
        0x910000e6, 0x800000c2, 0x00155543, 0x001000f2, 0x00000000, 0x00100012, 0x00000001, 0x00101046,
        0x00000001, 0x00107e46, 0x00000000, 0x00106000, 0x00000000, 0x0020800a, 0x00000000, 0x00000000,
        0x00000000, 0x05000036, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x050000ea, 0x00100012,
        0x00000000, 0x0010000a, 0x00000001, 0x09000037, 0x00102012, 0x00000001, 0x0010000a, 0x00000000,
        0x00004001, 0x00000001, 0x00004001, 0x00000000, 0x0100003e, 0x30494653, 0x00000008, 0x00000100,
        0x00000000, 
    };
    static const BYTE ps_sample_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x5b, 0x3d, 0xa0, 0x0c, 0x79, 0x14, 0x7b, 0x9b, 0xfb, 0x3a, 0x98, 0xbe, 0x7a, 0x47, 0xda, 0xc2, 0x01, 0x00, 0x00, 0x00, 0xce, 0x0a, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0x06, 0x01, 0x00, 0x00, 0xf2, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x60, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x55, 0x56, 0x5f, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x52, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54,
        0x61, 0x72, 0x67, 0x65, 0x74, 0x00, 0x50, 0x53, 0x56, 0x30, 0xe4, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x02, 0x02, 0x00, 0x02, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x55, 0x56, 0x5f, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x41, 0x10, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xd4, 0x08, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x35, 0x02,
        0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xbc, 0x08, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x2c, 0x02, 0x00, 0x00, 0x0b, 0x82,
        0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04,
        0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19,
        0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c,
        0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64,
        0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x90, 0xc1, 0x1c, 0x01, 0x18, 0xdc, 0x24, 0x4d, 0x11,
        0x25, 0x4c, 0x3e, 0x0b, 0x30, 0xcf, 0x42, 0x44, 0xec, 0x04, 0x4c, 0x04, 0x0a, 0x00, 0x0a, 0x66, 0x00, 0x86, 0x11, 0x88, 0x61, 0xa6, 0x36, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37,
        0xa0, 0x85, 0x72, 0xc0, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x39, 0x20, 0x05, 0x3e, 0xb0, 0x87, 0x72, 0x18, 0x07, 0x7a, 0x78, 0x07, 0x79, 0xe0, 0x03, 0x73, 0x60, 0x87, 0x77, 0x08, 0x07,
        0x7a, 0x60, 0x03, 0x30, 0xa0, 0x03, 0x3f, 0x00, 0x03, 0x3f, 0xd0, 0x03, 0x3d, 0x68, 0x87, 0x74, 0x80, 0x87, 0x79, 0xf8, 0x05, 0x7a, 0xc8, 0x07, 0x78, 0x28, 0x07, 0x14, 0x00, 0x33, 0x89, 0xc1,
        0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0, 0xc3, 0x3b, 0xc8,
        0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0, 0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x01, 0x12, 0x88, 0x82, 0x8c, 0x23, 0x80, 0xc8, 0x90, 0x50, 0x30, 0x10, 0x32, 0x02,
        0x50, 0x02, 0x85, 0x96, 0x39, 0x02, 0xa4, 0x18, 0x03, 0x00, 0x00, 0x0d, 0x40, 0x4e, 0x31, 0x14, 0x00, 0x00, 0x9a, 0x81, 0xa0, 0x62, 0x28, 0x00, 0x00, 0x34, 0x00, 0x49, 0x47, 0x0d, 0x97, 0x3f,
        0x61, 0x0f, 0x21, 0xf9, 0xdc, 0x46, 0x15, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0x23, 0x62, 0x18, 0x86, 0x81, 0x9a, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68, 0x86, 0x85, 0x40, 0x41,
        0x55, 0x08, 0x0a, 0xb0, 0x00, 0xba, 0x6e, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x57, 0x42, 0x5a, 0x89, 0xc9, 0x2f, 0x6e, 0x1b, 0x15, 0xc3, 0x30, 0x0c, 0x40, 0x61, 0x32, 0xc0, 0xb2, 0x86,
        0x61, 0x18, 0x00, 0x00, 0x18, 0x48, 0x9b, 0x23, 0x08, 0xca, 0xc0, 0x01, 0x00, 0x75, 0xc5, 0xb0, 0x80, 0x06, 0x00, 0x38, 0xfa, 0x06, 0x02, 0x12, 0x01, 0x98, 0x23, 0x00, 0x05, 0x00, 0x13, 0x14,
        0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72,
        0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07,
        0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20,
        0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a,
        0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x84, 0x90, 0x21, 0x23, 0x45, 0x44, 0x00, 0xc6, 0x00, 0x80, 0x69, 0x03, 0x00, 0x26, 0x0e, 0x00, 0x60, 0xc8, 0xb3, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xc7, 0x01, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x04, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x43, 0x1e, 0x09, 0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x17, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x34, 0x20,
        0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x74, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0xf9, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x14, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x02, 0x4a, 0x60, 0x04, 0xa0, 0x10, 0x8a,
        0xa1, 0x08, 0x4a, 0xa2, 0x50, 0xca, 0xa0, 0x1c, 0x0a, 0xa4, 0x14, 0x68, 0x2c, 0x40, 0x20, 0x6a, 0x4a, 0xa0, 0x14, 0x0a, 0xa4, 0x08, 0x46, 0x00, 0x4a, 0xa2, 0x0c, 0x0a, 0x81, 0x92, 0x19, 0x00,
        0x42, 0x66, 0x00, 0xe8, 0x98, 0x01, 0x20, 0x63, 0x06, 0x80, 0x8a, 0x19, 0x00, 0x12, 0xc7, 0x42, 0x8c, 0x02, 0x00, 0x00, 0xe0, 0xfb, 0x48, 0x98, 0x01, 0x00, 0x79, 0x18, 0x00, 0x00, 0x8b, 0x00,
        0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x83, 0x0c, 0x6f, 0x0c, 0x24, 0xc6, 0x45, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x80, 0x65, 0x82, 0x00, 0x30, 0x1b, 0x84, 0x81, 0x98, 0x20,
        0x00, 0xcd, 0x06, 0x61, 0x30, 0x28, 0xc0, 0xcd, 0x4d, 0x10, 0x00, 0x67, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0xe1, 0x08, 0x03, 0x0e, 0x74, 0x65, 0x78, 0x13, 0x04, 0xe0, 0x99, 0x20, 0x00, 0xd0, 0x06,
        0x81, 0x70, 0x36, 0x24, 0xc4, 0xc2, 0x10, 0xc4, 0xd0, 0x10, 0xcf, 0x86, 0x00, 0x9a, 0x20, 0x24, 0x1f, 0x89, 0x30, 0xb9, 0xb3, 0xb9, 0x0d, 0x08, 0x21, 0x4d, 0x04, 0x31, 0x18, 0xc0, 0x86, 0x80,
        0x9a, 0x20, 0x08, 0x65, 0x40, 0x62, 0x2e, 0xac, 0x0d, 0x6e, 0x03, 0x42, 0x58, 0x17, 0x41, 0x0c, 0x04, 0xb0, 0x21, 0xc0, 0x36, 0x10, 0x11, 0x50, 0x65, 0x13, 0x04, 0x43, 0x0c, 0x26, 0x08, 0x40,
        0xc4, 0x00, 0x6d, 0x82, 0x00, 0x48, 0x13, 0x04, 0x60, 0xda, 0x60, 0x24, 0x9d, 0x47, 0x7c, 0x0e, 0x89, 0xb6, 0x34, 0xb8, 0xb9, 0x09, 0x02, 0x40, 0x6d, 0x20, 0x92, 0x30, 0xf0, 0xc4, 0x60, 0xc3,
        0xc0, 0x81, 0xc1, 0x18, 0x4c, 0x10, 0x8a, 0x31, 0xa0, 0x81, 0x16, 0xe6, 0x46, 0xc6, 0x56, 0x36, 0x41, 0x00, 0xaa, 0x0d, 0x46, 0x62, 0x06, 0x1e, 0xf1, 0x9d, 0xc1, 0x06, 0xc1, 0x40, 0x83, 0x09,
        0x02, 0x02, 0x06, 0x4c, 0xd8, 0xde, 0xc8, 0xbe, 0xc6, 0xd8, 0xc2, 0xda, 0xe0, 0x36, 0x18, 0xc9, 0x1a, 0x78, 0xc4, 0xe7, 0x6c, 0x10, 0x0c, 0x36, 0xd8, 0x70, 0x10, 0x1b, 0x19, 0x94, 0x41, 0x1a,
        0xa8, 0x41, 0x1b, 0x4c, 0x10, 0x98, 0x61, 0x03, 0xb0, 0x61, 0x20, 0xe0, 0x00, 0x0e, 0x36, 0x04, 0x71, 0xb0, 0x61, 0x18, 0xde, 0x40, 0x0e, 0x26, 0x08, 0x62, 0x40, 0x06, 0x1b, 0x02, 0x3a, 0x20,
        0xd1, 0x16, 0x96, 0xe6, 0xc6, 0x65, 0xca, 0xea, 0x0b, 0xea, 0x6d, 0x2e, 0x8d, 0x2e, 0xed, 0xcd, 0x6d, 0x82, 0xd0, 0x70, 0x13, 0x84, 0xa6, 0xdb, 0x10, 0x10, 0x13, 0x84, 0xc6, 0x9b, 0x20, 0x34,
        0xdb, 0x86, 0x85, 0xb8, 0x03, 0x3c, 0xc8, 0x03, 0x3d, 0xd8, 0x83, 0x61, 0x0f, 0x08, 0x3e, 0x00, 0xb8, 0x54, 0x59, 0x7d, 0x51, 0x15, 0x61, 0x0d, 0x3d, 0x3d, 0x49, 0x11, 0x4d, 0x10, 0x1a, 0x6d,
        0xc3, 0x32, 0xf8, 0x01, 0x1e, 0xf0, 0x81, 0x1e, 0xfc, 0xc1, 0xf0, 0x07, 0x03, 0x1f, 0x00, 0x1b, 0x84, 0x3e, 0x00, 0x05, 0x26, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74, 0x13, 0x84,
        0x26, 0xdb, 0xb0, 0x10, 0xa2, 0x80, 0x07, 0xa3, 0xa0, 0x07, 0x7c, 0x30, 0xec, 0x01, 0xc1, 0x07, 0xc0, 0x04, 0xa1, 0xc1, 0x36, 0x04, 0xc3, 0x04, 0xa1, 0xb9, 0x36, 0x2c, 0x83, 0x28, 0x94, 0xc2,
        0x28, 0x98, 0x02, 0x1f, 0x0c, 0xa7, 0x30, 0xf0, 0x01, 0xb0, 0x41, 0x20, 0x05, 0x54, 0xd8, 0x30, 0x84, 0x42, 0x2a, 0x00, 0x13, 0x84, 0x31, 0xb0, 0x36, 0x08, 0xc4, 0x2a, 0x6c, 0x28, 0xde, 0xc0,
        0x0e, 0x54, 0x41, 0x63, 0x85, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c, 0xda, 0x9b, 0xdb, 0x94, 0x80,
        0x68, 0x42, 0x86, 0xe7, 0x62, 0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x30, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95, 0xb1, 0x4d, 0x09, 0x92, 0x32,
        0x64, 0x78, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73, 0x53, 0x82, 0xac, 0x12, 0x19, 0x9e, 0x0b, 0x5d, 0x1e, 0x5c, 0x59, 0x90, 0x9b, 0xdb, 0x1b, 0x5d, 0x18, 0x5d, 0xda, 0x9b,
        0xdb, 0xdc, 0x14, 0xa1, 0x0d, 0xe4, 0xa0, 0x0e, 0x19, 0x9e, 0x8b, 0x5d, 0x5a, 0xd9, 0x5d, 0x12, 0xd9, 0x14, 0x5d, 0x18, 0x5d, 0xd9, 0x94, 0x80, 0x0e, 0xea, 0x90, 0xe1, 0xb9, 0x94, 0xb9, 0xd1,
        0xc9, 0xe5, 0x41, 0xbd, 0xa5, 0xb9, 0xd1, 0xcd, 0x4d, 0x09, 0x58, 0x01, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01,
        0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1,
        0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07,
        0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0,
        0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84,
        0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05,
        0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62,
        0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98,
        0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07,
        0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0,
        0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x46, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x07,
        0xdb, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x01, 0x55, 0x14, 0x44, 0x54, 0x3a, 0xc0, 0x50, 0x12, 0x06, 0x20, 0x60, 0x7e, 0x71, 0xdb, 0x15, 0x80, 0x06, 0x81, 0x1f, 0xbd, 0x8c, 0x07, 0x02, 0x67, 0xd6,
        0x1f, 0x89, 0x5a, 0xc6, 0xd3, 0xeb, 0xf2, 0xb2, 0x8c, 0x08, 0xb4, 0xfe, 0x48, 0xf6, 0xf2, 0x98, 0xfe, 0x96, 0x03, 0x9b, 0x24, 0xd8, 0x0c, 0x08, 0x04, 0x02, 0x83, 0x36, 0x40, 0x30, 0x00, 0xd2,
        0x58, 0x02, 0x18, 0x0c, 0x97, 0xef, 0x3c, 0xbe, 0x70, 0x10, 0x82, 0xa2, 0x09, 0x02, 0x21, 0x49, 0x1f, 0xb5, 0x2c, 0x98, 0x09, 0x3c, 0x0f, 0x31, 0xf8, 0xc8, 0x6d, 0x9b, 0x00, 0x01, 0x44, 0x86,
        0x54, 0x00, 0x59, 0xe0, 0x37, 0x0f, 0xb7, 0xe1, 0x40, 0xe0, 0xac, 0x3a, 0x0d, 0xb7, 0xe1, 0xec, 0xb2, 0x7c, 0x4a, 0x0f, 0xd3, 0xcb, 0x40, 0x60, 0xd0, 0x0a, 0xaa, 0xe1, 0xf2, 0x9d, 0xc7, 0x97,
        0x26, 0x27, 0x22, 0x50, 0x6a, 0x7a, 0xa8, 0xc9, 0x2f, 0x6e, 0xdb, 0x0c, 0xaa, 0xe1, 0xf2, 0x9d, 0xc7, 0x97, 0x26, 0x27, 0x22, 0x50, 0x6a, 0x7a, 0xa8, 0xc9, 0x47, 0x6e, 0xdb, 0x10, 0xa0, 0xe1,
        0xf2, 0x9d, 0xc7, 0x97, 0x00, 0xe6, 0x59, 0x08, 0xbf, 0xb8, 0x6d, 0x53, 0x90, 0x86, 0xcb, 0x77, 0x1e, 0x5f, 0x88, 0x08, 0x60, 0x22, 0x42, 0xa0, 0x19, 0x16, 0x02, 0x00, 0x00, 0x00, 0x61, 0x20,
        0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x94, 0xdc, 0x0c, 0x40, 0xd9, 0x15, 0x5e, 0x39, 0x06, 0x10, 0x37, 0x02, 0x40,
        0xc3, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xf0, 0x8d, 0xc1, 0x21, 0x85, 0x41, 0x18, 0x08, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xf0, 0x91, 0x01, 0x12, 0x89, 0x81,
        0x18, 0x0c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xf0, 0x95, 0x41, 0x52, 0x8d, 0xc1, 0x18, 0x10, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xe0, 0xb0, 0xc1, 0x18, 0x94, 0x01, 0x19, 0x54, 0xc9, 0x88,
        0x41, 0x02, 0x80, 0x20, 0x18, 0x38, 0x6d, 0x40, 0x06, 0x66, 0x50, 0x06, 0x9a, 0x32, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0xd7, 0x1a, 0x28, 0x83, 0x19, 0x8c, 0x26, 0x04, 0xc0, 0x88, 0xc1, 0x03,
        0x80, 0x20, 0x18, 0x68, 0x6c, 0xb0, 0x1c, 0x06, 0x31, 0x20, 0x08, 0x1a, 0xa0, 0x41, 0x13, 0x8c, 0x26, 0x04, 0xc0, 0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68, 0x02, 0x31, 0x8c, 0x26, 0x14,
        0xc4, 0x88, 0x81, 0x01, 0x80, 0x20, 0x18, 0x74, 0x71, 0x00, 0x05, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x00, 0xd9, 0x81, 0x19, 0xbc, 0xc1, 0x1b, 0x80, 0x81, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08,
        0x06, 0x90, 0x1d, 0x98, 0xc1, 0x1b, 0xbc, 0x01, 0x19, 0x14, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x00, 0xd9, 0x81, 0x19, 0xbc, 0xc1, 0x1b, 0x84, 0x01, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06,
        0x90, 0x1d, 0x98, 0xc1, 0x1b, 0xbc, 0x81, 0x37, 0x5c, 0x00, 0xc4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x48, 0x76, 0x70, 0x06, 0x71, 0x00, 0x07, 0x61, 0x10, 0x60, 0x38, 0x10, 0x00, 0x09, 0x00,
        0x00, 0x00, 0xe6, 0xc2, 0x48, 0x00, 0xf3, 0xfc, 0x12, 0xc0, 0x3c, 0x0b, 0x11, 0x59, 0x8b, 0x33, 0x11, 0xd7, 0x3f, 0x11, 0xd7, 0x44, 0x45, 0xc4, 0x6f, 0x0f, 0xf6, 0xc2, 0x00, 0x91, 0x21, 0xfd,
        0x42, 0x40, 0x15, 0x05, 0x11, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
#if 0
    SamplerState samp : register(s0);
    Texture2D<float4> tex : register(t0);

    cbuffer args : register(b0)
    {
        float lod_clamp;
        float lod_bias;
    };

    void main(float4 pos : SV_POSITION, float2 uv : UV_TEXCOORD, out float4 o0 : SV_TARGET0, out uint o1 : SV_TARGET1)
    {
        uint fb;
        o0 = tex.SampleBias(samp, uv, lod_bias, int2(0, 0), lod_clamp, fb);
        o1 = CheckAccessFullyMapped(fb) ? 1 : 0;
    }
#endif
    static const DWORD ps_sample_bias_dxbc[] =
    {
        0x43425844, 0xafcb01e1, 0x3f4af08e, 0x56bda673, 0x8e911618, 0x00000001, 0x000001f0, 0x00000004,
        0x00000030, 0x00000088, 0x000000d4, 0x000001e0, 0x4e475349, 0x00000050, 0x00000002, 0x00000008,
        0x00000038, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000,
        0x00000000, 0x00000003, 0x00000001, 0x00000303, 0x505f5653, 0x5449534f, 0x004e4f49, 0x545f5655,
        0x4f435845, 0x0044524f, 0x4e47534f, 0x00000044, 0x00000002, 0x00000008, 0x00000038, 0x00000000,
        0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x00000038, 0x00000001, 0x00000000, 0x00000001,
        0x00000001, 0x00000e01, 0x545f5653, 0x45475241, 0xabab0054, 0x58454853, 0x00000104, 0x00000050,
        0x00000041, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300005a, 0x00106000,
        0x00000000, 0x04001858, 0x00107000, 0x00000000, 0x00005555, 0x03001062, 0x00101032, 0x00000001,
        0x03000065, 0x001020f2, 0x00000000, 0x03000065, 0x00102012, 0x00000001, 0x02000068, 0x00000002,
        0x940000e7, 0x800000c2, 0x00155543, 0x001000f2, 0x00000000, 0x00100012, 0x00000001, 0x00101046,
        0x00000001, 0x00107e46, 0x00000000, 0x00106000, 0x00000000, 0x0020801a, 0x00000000, 0x00000000,
        0x0020800a, 0x00000000, 0x00000000, 0x00000000, 0x05000036, 0x001020f2, 0x00000000, 0x00100e46,
        0x00000000, 0x050000ea, 0x00100012, 0x00000000, 0x0010000a, 0x00000001, 0x09000037, 0x00102012,
        0x00000001, 0x0010000a, 0x00000000, 0x00004001, 0x00000001, 0x00004001, 0x00000000, 0x0100003e,
        0x30494653, 0x00000008, 0x00000100, 0x00000000, 
    };
    static const BYTE ps_sample_bias_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xc2, 0xe2, 0xfa, 0xbc, 0xae, 0x78, 0xac, 0x1b, 0x8b, 0xa7, 0x22, 0xa3, 0xdf, 0xc2, 0xeb, 0x67, 0x01, 0x00, 0x00, 0x00, 0xf6, 0x0a, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0x06, 0x01, 0x00, 0x00, 0xf2, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x60, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x55, 0x56, 0x5f, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x52, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54,
        0x61, 0x72, 0x67, 0x65, 0x74, 0x00, 0x50, 0x53, 0x56, 0x30, 0xe4, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x02, 0x02, 0x00, 0x02, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x55, 0x56, 0x5f, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x41, 0x10, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xfc, 0x08, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x3f, 0x02,
        0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xe4, 0x08, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x36, 0x02, 0x00, 0x00, 0x0b, 0x82,
        0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04,
        0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19,
        0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c,
        0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x52, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64,
        0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x90, 0xc1, 0x1c, 0x01, 0x18, 0xdc, 0x24, 0x4d, 0x11,
        0x25, 0x4c, 0x3e, 0x0b, 0x30, 0xcf, 0x42, 0x44, 0xec, 0x04, 0x4c, 0x04, 0x0a, 0x00, 0x0a, 0x66, 0x00, 0x86, 0x11, 0x88, 0x61, 0xa6, 0x36, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37,
        0xa0, 0x85, 0x72, 0xc0, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x39, 0x20, 0x05, 0x3e, 0xb0, 0x87, 0x72, 0x18, 0x07, 0x7a, 0x78, 0x07, 0x79, 0xe0, 0x03, 0x73, 0x60, 0x87, 0x77, 0x08, 0x07,
        0x7a, 0x60, 0x03, 0x30, 0xa0, 0x03, 0x3f, 0x00, 0x03, 0x3f, 0xd0, 0x03, 0x3d, 0x68, 0x87, 0x74, 0x80, 0x87, 0x79, 0xf8, 0x05, 0x7a, 0xc8, 0x07, 0x78, 0x28, 0x07, 0x14, 0x00, 0x33, 0x89, 0xc1,
        0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0, 0xc3, 0x3b, 0xc8,
        0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0, 0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x01, 0x12, 0x88, 0x82, 0x8c, 0x23, 0x80, 0xc8, 0x90, 0x90, 0x30, 0x0c, 0x84, 0x8c,
        0x00, 0x94, 0x40, 0xa1, 0x65, 0x8e, 0x00, 0x29, 0xc6, 0x00, 0x00, 0x40, 0x03, 0x90, 0x53, 0x0c, 0x05, 0x00, 0x80, 0x66, 0x20, 0xa8, 0x18, 0x0a, 0x00, 0x00, 0x0d, 0x40, 0xd2, 0x51, 0xc3, 0xe5,
        0x4f, 0xd8, 0x43, 0x48, 0x3e, 0xb7, 0x51, 0xc5, 0x4a, 0x4c, 0x7e, 0x71, 0xdb, 0x88, 0x18, 0x86, 0x61, 0xa0, 0xe6, 0x9e, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x3f, 0x04, 0x9a, 0x61, 0x21, 0x50,
        0x50, 0x15, 0x82, 0x02, 0x2c, 0x80, 0xae, 0x9b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56, 0x62, 0xf2, 0x8b, 0xdb, 0x46, 0xc5, 0x30, 0x0c, 0x03, 0x50, 0x9a, 0x0c, 0xb0, 0xac,
        0x61, 0x18, 0x06, 0x00, 0x00, 0x86, 0x81, 0xb4, 0x39, 0x82, 0xa0, 0x0c, 0x1c, 0x00, 0x50, 0x57, 0x0c, 0x0b, 0x68, 0x00, 0x80, 0xa3, 0x6f, 0x20, 0x20, 0x11, 0x80, 0x39, 0x02, 0x50, 0x00, 0x00,
        0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f,
        0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76,
        0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07,
        0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x84, 0x90, 0x21, 0x23, 0x45, 0x44, 0x00, 0xc6, 0x00, 0x80, 0x69, 0x03, 0x00, 0x26, 0x0e, 0x00, 0x60, 0xc8,
        0xb3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xc7, 0x01, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x04, 0x04, 0x80, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e, 0x09, 0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x17, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x0c, 0x79, 0x34, 0x20, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x74, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0xf9, 0x80, 0x00, 0x18,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x15, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x02, 0x4a, 0x60,
        0x04, 0xa0, 0x10, 0x8a, 0xa1, 0x08, 0x4a, 0xa2, 0x20, 0x0a, 0xa5, 0x0c, 0xca, 0xa1, 0x40, 0x4a, 0x81, 0xc6, 0x02, 0x04, 0xa2, 0xa6, 0x04, 0x4a, 0xa1, 0x40, 0x8a, 0x60, 0x04, 0xa0, 0x24, 0xca,
        0xa0, 0x10, 0x28, 0x99, 0x01, 0x20, 0x64, 0x06, 0x80, 0x8e, 0x19, 0x00, 0x32, 0x66, 0x00, 0xa8, 0x98, 0x01, 0x20, 0x71, 0x2c, 0xc4, 0x28, 0x00, 0x00, 0x00, 0xbe, 0x8f, 0x84, 0x19, 0x00, 0x00,
        0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x91, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x83, 0x0c, 0x6f, 0x0c, 0x24, 0xc6, 0x45, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x80, 0x65,
        0x82, 0x00, 0x30, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x00, 0xcd, 0x06, 0x61, 0x30, 0x28, 0xc0, 0xcd, 0x4d, 0x10, 0x00, 0x67, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0xe1, 0x10, 0x03, 0x0e, 0x74, 0x65, 0x78,
        0x13, 0x04, 0xe0, 0x99, 0x20, 0x00, 0xd0, 0x06, 0x81, 0x70, 0x36, 0x24, 0xc4, 0xc2, 0x10, 0xc4, 0xd0, 0x10, 0xcf, 0x86, 0x00, 0x9a, 0x20, 0x24, 0x60, 0x40, 0x22, 0x4c, 0xee, 0x6c, 0x6e, 0x82,
        0x00, 0x44, 0x1b, 0x10, 0x42, 0x9a, 0x08, 0x62, 0xa0, 0x80, 0x0d, 0x41, 0x35, 0x41, 0x10, 0xcc, 0x80, 0xc4, 0x5c, 0x58, 0x1b, 0xdc, 0x06, 0x84, 0xb8, 0x30, 0x82, 0x18, 0x08, 0x60, 0x43, 0x90,
        0x6d, 0x20, 0x22, 0xc0, 0xd2, 0x26, 0x08, 0xc6, 0x18, 0x4c, 0x10, 0x00, 0x89, 0x01, 0xda, 0x04, 0x01, 0x98, 0x26, 0x08, 0x00, 0xb5, 0xc1, 0x48, 0xbc, 0x8f, 0x00, 0x03, 0x87, 0x44, 0x5b, 0x1a,
        0xdc, 0xdc, 0x04, 0x01, 0xa8, 0x36, 0x10, 0x89, 0x18, 0x7c, 0x63, 0xb0, 0x61, 0xe8, 0xc2, 0x80, 0x0c, 0x26, 0x08, 0x05, 0x19, 0xd0, 0x40, 0x0b, 0x73, 0x23, 0x63, 0x2b, 0x9b, 0x20, 0x00, 0xd6,
        0x06, 0x23, 0x39, 0x83, 0x8f, 0x00, 0x03, 0x34, 0xd8, 0x20, 0x18, 0x69, 0x30, 0x41, 0x40, 0xc2, 0x80, 0x09, 0xdb, 0x1b, 0xd9, 0xd7, 0x18, 0x5b, 0x58, 0x1b, 0xdc, 0x06, 0x23, 0x61, 0x83, 0x8f,
        0x00, 0x03, 0x87, 0x08, 0xdb, 0x1b, 0xd9, 0x97, 0x58, 0x5a, 0xd8, 0xdc, 0x06, 0x23, 0x71, 0x83, 0xcf, 0x00, 0x03, 0x67, 0xc3, 0x40, 0xb5, 0xc1, 0x1b, 0x6c, 0x38, 0x08, 0xae, 0x0c, 0xcc, 0x40,
        0x0d, 0xd6, 0x00, 0x0e, 0x26, 0x08, 0xcc, 0xb0, 0x01, 0xd8, 0x30, 0x10, 0x73, 0x30, 0x07, 0x1b, 0x02, 0x3a, 0xd8, 0x30, 0x0c, 0x72, 0x50, 0x07, 0x13, 0x04, 0x31, 0x28, 0x83, 0x0d, 0xc1, 0x1d,
        0x90, 0x68, 0x0b, 0x4b, 0x73, 0xe3, 0x32, 0x65, 0xf5, 0x05, 0xf5, 0x36, 0x97, 0x46, 0x97, 0xf6, 0xe6, 0x36, 0x41, 0x68, 0xba, 0x09, 0x42, 0xe3, 0x6d, 0x08, 0x88, 0x09, 0x42, 0xf3, 0x4d, 0x10,
        0x1a, 0x6e, 0xc3, 0x42, 0xe8, 0xc1, 0x1e, 0xf0, 0x41, 0x1f, 0xf8, 0xc1, 0xe0, 0x07, 0xc4, 0x1f, 0x00, 0x5c, 0xaa, 0xac, 0xbe, 0xa8, 0x8a, 0xb0, 0x86, 0x9e, 0x9e, 0xa4, 0x88, 0x26, 0x08, 0xcd,
        0xb6, 0x61, 0x19, 0x42, 0x61, 0x0f, 0xfe, 0xa0, 0x0f, 0x44, 0x61, 0x10, 0x85, 0xe1, 0x0f, 0x80, 0x0d, 0x02, 0x28, 0x8c, 0x02, 0x93, 0x29, 0xab, 0x2f, 0xaa, 0x30, 0xb9, 0xb3, 0x32, 0xba, 0x09,
        0x42, 0xa3, 0x6d, 0x58, 0x88, 0x52, 0xd8, 0x03, 0x53, 0xe8, 0x83, 0x3f, 0x18, 0xfc, 0x80, 0xf8, 0x03, 0x60, 0x82, 0xd0, 0x64, 0x1b, 0x82, 0x61, 0x82, 0xd0, 0x60, 0x1b, 0x96, 0xa1, 0x14, 0x50,
        0xc1, 0x14, 0x52, 0xe1, 0x0f, 0x06, 0x55, 0x18, 0xfe, 0x00, 0xd8, 0x20, 0x9c, 0xc2, 0x2a, 0x6c, 0x18, 0x48, 0x81, 0x15, 0x80, 0x09, 0xc2, 0x18, 0x5c, 0x1b, 0x04, 0xc2, 0x15, 0x36, 0x14, 0x72,
        0x90, 0x07, 0xad, 0xb0, 0xbd, 0x42, 0x15, 0x36, 0x36, 0xbb, 0x36, 0x97, 0x34, 0xb2, 0x32, 0x37, 0xba, 0x29, 0x41, 0x50, 0x85, 0x0c, 0xcf, 0xc5, 0xae, 0x4c, 0x6e, 0x2e, 0xed, 0xcd, 0x6d, 0x4a,
        0x40, 0x34, 0x21, 0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x18, 0x75, 0xc8, 0xf0, 0x5c, 0xe6, 0xd0, 0xc2, 0xc8, 0xca, 0xe4, 0x9a, 0xde, 0xc8, 0xca, 0xd8, 0xa6, 0x04, 0x49,
        0x19, 0x32, 0x3c, 0x17, 0xb9, 0xb2, 0xb9, 0xb7, 0x3a, 0xb9, 0xb1, 0xb2, 0xb9, 0x29, 0x81, 0x56, 0x89, 0x0c, 0xcf, 0x85, 0x2e, 0x0f, 0xae, 0x2c, 0xc8, 0xcd, 0xed, 0x8d, 0x2e, 0x8c, 0x2e, 0xed,
        0xcd, 0x6d, 0x6e, 0x8a, 0x00, 0x07, 0x75, 0x50, 0x87, 0x0c, 0xcf, 0xc5, 0x2e, 0xad, 0xec, 0x2e, 0x89, 0x6c, 0x8a, 0x2e, 0x8c, 0xae, 0x6c, 0x4a, 0x70, 0x07, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc,
        0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0xaf, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01,
        0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1,
        0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07,
        0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0,
        0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84,
        0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05,
        0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62,
        0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98,
        0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07,
        0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0,
        0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0x46, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x5d, 0x01,
        0x68, 0x10, 0xf8, 0xd1, 0xcb, 0x78, 0x20, 0x70, 0x66, 0xfd, 0x91, 0xa8, 0x65, 0x3c, 0xbd, 0x2e, 0x2f, 0xcb, 0x88, 0x40, 0xeb, 0x8f, 0x64, 0x2f, 0x8f, 0xe9, 0x6f, 0x39, 0xb0, 0x49, 0x82, 0xcd,
        0x80, 0x40, 0x20, 0x30, 0x68, 0x03, 0x04, 0x03, 0x20, 0x8d, 0x1d, 0x6c, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x04, 0x54, 0x51, 0x10, 0x51, 0xe9, 0x00, 0x43, 0x49, 0x18, 0x80, 0x80, 0xf9, 0xc5, 0x6d,
        0x5b, 0x02, 0x18, 0x0c, 0x97, 0xef, 0x3c, 0xbe, 0x70, 0x10, 0x82, 0xa2, 0x09, 0x02, 0x21, 0x49, 0x1f, 0xb5, 0x2c, 0x98, 0x09, 0x3c, 0x0f, 0x31, 0xf8, 0xc8, 0x6d, 0x9b, 0x00, 0x01, 0x44, 0x86,
        0x54, 0x00, 0x59, 0xe0, 0x37, 0x0f, 0xb7, 0xe1, 0x40, 0xe0, 0xac, 0x3a, 0x0d, 0xb7, 0xe1, 0xec, 0xb2, 0x7c, 0x4a, 0x0f, 0xd3, 0xcb, 0x40, 0x60, 0xd0, 0x0a, 0xaa, 0xe1, 0xf2, 0x9d, 0xc7, 0x97,
        0x26, 0x27, 0x22, 0x50, 0x6a, 0x7a, 0xa8, 0xc9, 0x2f, 0x6e, 0xdb, 0x0c, 0xaa, 0xe1, 0xf2, 0x9d, 0xc7, 0x97, 0x26, 0x27, 0x22, 0x50, 0x6a, 0x7a, 0xa8, 0xc9, 0x47, 0x6e, 0xdb, 0x10, 0xa8, 0xe1,
        0xf2, 0x9d, 0xc7, 0x97, 0x00, 0xe6, 0x59, 0x88, 0x0d, 0x01, 0x24, 0xbf, 0xb8, 0x6d, 0x53, 0x90, 0x86, 0xcb, 0x77, 0x1e, 0x5f, 0x88, 0x08, 0x60, 0x22, 0x42, 0xa0, 0x19, 0x16, 0x02, 0x00, 0x00,
        0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x94, 0xdc, 0x0c, 0x40, 0xd9, 0x95, 0x5e, 0x39, 0x06,
        0x10, 0x37, 0x02, 0x40, 0xc3, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xf0, 0x91, 0xc1, 0x21, 0x89, 0x81, 0x18, 0x08, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xf0, 0x95,
        0x01, 0x12, 0x8d, 0xc1, 0x18, 0x0c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xf0, 0x99, 0x41, 0x52, 0x91, 0x01, 0x19, 0x10, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xe0, 0xb4, 0x01, 0x19, 0x98, 0x41,
        0x19, 0x54, 0xc9, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x38, 0x6e, 0x50, 0x06, 0x67, 0x60, 0x06, 0x9a, 0x32, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x17, 0x1b, 0x28, 0xc3, 0x19, 0x8c, 0x26, 0x04,
        0xc0, 0x68, 0x82, 0x10, 0x8c, 0x18, 0x40, 0x00, 0x08, 0x82, 0x81, 0xe6, 0x06, 0x0c, 0x72, 0x14, 0x44, 0x92, 0xa8, 0x81, 0x1a, 0x38, 0x81, 0x30, 0x9a, 0x10, 0x00, 0xa3, 0x09, 0x42, 0x30, 0x9a,
        0x30, 0x08, 0xa3, 0x09, 0xc4, 0x30, 0x9a, 0x50, 0x10, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0xd0, 0xcd, 0x41, 0x14, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x84, 0x07, 0x67, 0x10, 0x07, 0x71,
        0x10, 0x06, 0xc6, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x40, 0x78, 0x70, 0x06, 0x71, 0x10, 0x07, 0x65, 0x50, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x84, 0x07, 0x67, 0x10, 0x07, 0x71, 0x20,
        0x06, 0xc4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x40, 0x78, 0x70, 0x06, 0x71, 0x10, 0x07, 0xdf, 0x70, 0x01, 0x10, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0xe1, 0x01, 0x1a, 0xcc, 0x81, 0x1c,
        0x88, 0x41, 0x80, 0xe1, 0x40, 0x00, 0x09, 0x00, 0x00, 0x00, 0xf6, 0xc2, 0x48, 0x00, 0xf3, 0xfc, 0x12, 0xc0, 0x3c, 0x0b, 0x11, 0x99, 0x8b, 0x33, 0x11, 0xd7, 0x3f, 0x11, 0xd7, 0x44, 0x45, 0xc4,
        0x6f, 0x0f, 0x06, 0xc3, 0x00, 0x91, 0x21, 0xfd, 0x42, 0x40, 0x15, 0x05, 0x11, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    };
#if 0
    SamplerState samp : register(s0);
    Texture2D<float4> tex : register(t0);

    cbuffer args : register(b0)
    {
        float lod_clamp;
        float ddx;
        float ddy;
    };

    void main(float4 pos : SV_POSITION, float2 uv : UV_TEXCOORD, out float4 o0 : SV_TARGET0, out uint o1 : SV_TARGET1)
    {
        uint fb;
        o0 = tex.SampleGrad(samp, uv, ddx, ddy, int2(0, 0), lod_clamp, fb);
        o1 = CheckAccessFullyMapped(fb) ? 1 : 0;
    }
#endif
    static const DWORD ps_sample_grad_dxbc[] =
    {
        0x43425844, 0x3446d50e, 0x26d0c876, 0x2d6ba03a, 0xab028756, 0x00000001, 0x000001fc, 0x00000004,
        0x00000030, 0x00000088, 0x000000d4, 0x000001ec, 0x4e475349, 0x00000050, 0x00000002, 0x00000008,
        0x00000038, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000,
        0x00000000, 0x00000003, 0x00000001, 0x00000303, 0x505f5653, 0x5449534f, 0x004e4f49, 0x545f5655,
        0x4f435845, 0x0044524f, 0x4e47534f, 0x00000044, 0x00000002, 0x00000008, 0x00000038, 0x00000000,
        0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x00000038, 0x00000001, 0x00000000, 0x00000001,
        0x00000001, 0x00000e01, 0x545f5653, 0x45475241, 0xabab0054, 0x58454853, 0x00000110, 0x00000050,
        0x00000044, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300005a, 0x00106000,
        0x00000000, 0x04001858, 0x00107000, 0x00000000, 0x00005555, 0x03001062, 0x00101032, 0x00000001,
        0x03000065, 0x001020f2, 0x00000000, 0x03000065, 0x00102012, 0x00000001, 0x02000068, 0x00000002,
        0x970000e8, 0x800000c2, 0x00155543, 0x001000f2, 0x00000000, 0x00100012, 0x00000001, 0x00101046,
        0x00000001, 0x00107e46, 0x00000000, 0x00106000, 0x00000000, 0x00208556, 0x00000000, 0x00000000,
        0x00208aa6, 0x00000000, 0x00000000, 0x0020800a, 0x00000000, 0x00000000, 0x00002000, 0x05000036,
        0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x050000ea, 0x00100012, 0x00000000, 0x0010000a,
        0x00000001, 0x09000037, 0x00102012, 0x00000001, 0x0010000a, 0x00000000, 0x00004001, 0x00000001,
        0x00004001, 0x00000000, 0x0100003e, 0x30494653, 0x00000008, 0x00000100, 0x00000000,
    };
    static const BYTE ps_sample_grad_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x68, 0xe1, 0x46, 0x09, 0xda, 0xc5, 0x9d, 0x05, 0xe8, 0x4a, 0x2d, 0x42, 0xe3, 0xdb, 0x9c, 0x76, 0x01, 0x00, 0x00, 0x00, 0x12, 0x0b, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0x06, 0x01, 0x00, 0x00, 0xf2, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x60, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x55, 0x56, 0x5f, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x52, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54,
        0x61, 0x72, 0x67, 0x65, 0x74, 0x00, 0x50, 0x53, 0x56, 0x30, 0xe4, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x02, 0x02, 0x00, 0x02, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x55, 0x56, 0x5f, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x41, 0x10, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x18, 0x09, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x46, 0x02,
        0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x3d, 0x02, 0x00, 0x00, 0x0b, 0x82,
        0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04,
        0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19,
        0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c,
        0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x53, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64,
        0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x90, 0xc1, 0x1c, 0x01, 0x18, 0xdc, 0x24, 0x4d, 0x11,
        0x25, 0x4c, 0x3e, 0x0b, 0x30, 0xcf, 0x42, 0x44, 0xec, 0x04, 0x4c, 0x04, 0x0a, 0x00, 0x0a, 0x66, 0x00, 0x86, 0x11, 0x88, 0x61, 0xa6, 0x36, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37,
        0xa0, 0x85, 0x72, 0xc0, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x39, 0x20, 0x05, 0x3e, 0xb0, 0x87, 0x72, 0x18, 0x07, 0x7a, 0x78, 0x07, 0x79, 0xe0, 0x03, 0x73, 0x60, 0x87, 0x77, 0x08, 0x07,
        0x7a, 0x60, 0x03, 0x30, 0xa0, 0x03, 0x3f, 0x00, 0x03, 0x3f, 0xd0, 0x03, 0x3d, 0x68, 0x87, 0x74, 0x80, 0x87, 0x79, 0xf8, 0x05, 0x7a, 0xc8, 0x07, 0x78, 0x28, 0x07, 0x14, 0x00, 0x33, 0x89, 0xc1,
        0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0, 0xc3, 0x3b, 0xc8,
        0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0, 0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x01, 0x12, 0x88, 0x82, 0x8c, 0x23, 0x80, 0xc8, 0x90, 0xd0, 0x30, 0x0c, 0x03, 0x21,
        0x23, 0x00, 0x25, 0x50, 0x68, 0x99, 0x23, 0x40, 0x8a, 0x31, 0x00, 0x00, 0xd0, 0x00, 0xe4, 0x14, 0x43, 0x01, 0x00, 0xa0, 0x19, 0x08, 0x2a, 0x86, 0x02, 0x00, 0x40, 0x03, 0x90, 0x74, 0xd4, 0x70,
        0xf9, 0x13, 0xf6, 0x10, 0x92, 0xcf, 0x6d, 0x54, 0xb1, 0x12, 0x93, 0x5f, 0xdc, 0x36, 0x22, 0x86, 0x61, 0x18, 0xa8, 0xb9, 0x67, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x0f, 0x81, 0x66, 0x58, 0x08,
        0x14, 0x54, 0x85, 0xa0, 0x00, 0x0b, 0xa0, 0xeb, 0xa6, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0xa4, 0x95, 0x98, 0xfc, 0xe2, 0xb6, 0x51, 0x31, 0x0c, 0xc3, 0x00, 0x14, 0x29, 0x03, 0x2c,
        0x6b, 0x18, 0x86, 0x01, 0x00, 0x80, 0x61, 0x18, 0x86, 0x61, 0x18, 0x48, 0x9b, 0x23, 0x08, 0xca, 0xc0, 0x01, 0x00, 0x75, 0xc5, 0xb0, 0x80, 0x06, 0x00, 0x38, 0xfa, 0x06, 0x02, 0x12, 0x01, 0x98,
        0x23, 0x00, 0x05, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50,
        0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06,
        0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0,
        0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x84, 0x90, 0x21, 0x23, 0x45, 0x44, 0x00, 0xc6, 0x00, 0x80, 0x69, 0x03, 0x00, 0x26,
        0x0e, 0x00, 0x60, 0xc8, 0xb3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xc7, 0x01, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x04,
        0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e, 0x09, 0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x17, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x34, 0x20, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x74, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4,
        0xf9, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x15, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04,
        0x43, 0x02, 0x4a, 0x60, 0x04, 0xa0, 0x10, 0x8a, 0xa1, 0x08, 0x4a, 0xa2, 0x30, 0x0a, 0xa5, 0x0c, 0xca, 0xa1, 0x40, 0x4a, 0xa1, 0x20, 0x68, 0x2c, 0x40, 0x20, 0x6a, 0x4a, 0xa0, 0x14, 0x0a, 0xa4,
        0x08, 0x46, 0x00, 0x4a, 0xa2, 0x0c, 0x0a, 0x81, 0x92, 0x19, 0x00, 0x42, 0x66, 0x00, 0xe8, 0x98, 0x01, 0x20, 0x63, 0x06, 0x80, 0x8a, 0x19, 0x00, 0x12, 0xc7, 0x42, 0x8c, 0x02, 0x00, 0x00, 0xe0,
        0xfb, 0x48, 0x98, 0x01, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x95, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x83, 0x0c, 0x6f, 0x0c, 0x24, 0xc6, 0x45, 0x66, 0x43, 0x10,
        0x4c, 0x10, 0x80, 0x65, 0x82, 0x00, 0x30, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x00, 0xcd, 0x06, 0x61, 0x30, 0x28, 0xc0, 0xcd, 0x4d, 0x10, 0x00, 0x67, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0xe1, 0x18, 0x03,
        0x0e, 0x74, 0x65, 0x78, 0x13, 0x04, 0xe0, 0x99, 0x20, 0x00, 0xd0, 0x06, 0x81, 0x70, 0x36, 0x24, 0xc4, 0xc2, 0x10, 0xc4, 0xd0, 0x10, 0xcf, 0x86, 0x00, 0x9a, 0x20, 0x24, 0x61, 0x40, 0x22, 0x4c,
        0xee, 0x6c, 0x6e, 0x82, 0x00, 0x44, 0x1b, 0x10, 0x42, 0x9a, 0x08, 0x62, 0xa0, 0x80, 0x0d, 0x41, 0x35, 0x41, 0x10, 0xce, 0x80, 0xc4, 0x5c, 0x58, 0x1b, 0xdc, 0x06, 0x84, 0xb8, 0x30, 0x82, 0x18,
        0x08, 0x60, 0x43, 0x90, 0x6d, 0x20, 0x22, 0xc0, 0xd2, 0x26, 0x08, 0x06, 0x19, 0x4c, 0x10, 0x00, 0x89, 0x01, 0xda, 0x04, 0x01, 0x98, 0x26, 0x08, 0x00, 0xb5, 0xc1, 0x48, 0xbc, 0x8f, 0x00, 0x03,
        0x87, 0x44, 0x5b, 0x1a, 0xdc, 0xdc, 0x04, 0x01, 0xa8, 0x36, 0x10, 0x89, 0x18, 0x7c, 0x63, 0xb0, 0x61, 0xe8, 0xc2, 0x80, 0x0c, 0x26, 0x08, 0x45, 0x19, 0xd0, 0x40, 0x0b, 0x73, 0x23, 0x63, 0x2b,
        0x9b, 0x20, 0x00, 0xd6, 0x06, 0x23, 0x39, 0x83, 0x8f, 0x00, 0x03, 0x34, 0xd8, 0x20, 0x18, 0x69, 0x30, 0x41, 0x40, 0xc4, 0x80, 0x09, 0xdb, 0x1b, 0xd9, 0xd7, 0x18, 0x5b, 0x58, 0x1b, 0xdc, 0x06,
        0x23, 0x61, 0x83, 0x8f, 0x00, 0x03, 0x87, 0x03, 0x19, 0x19, 0xde, 0x06, 0x23, 0x71, 0x83, 0xcf, 0x00, 0x03, 0x87, 0x03, 0x19, 0x59, 0xde, 0x04, 0x01, 0xb8, 0x36, 0x18, 0x09, 0x1c, 0x7c, 0x71,
        0x00, 0x06, 0xce, 0x06, 0x82, 0x6a, 0x83, 0x37, 0x90, 0x83, 0x0d, 0x07, 0xc1, 0x95, 0x81, 0x19, 0xa8, 0xc1, 0x1a, 0xcc, 0xc1, 0x04, 0x81, 0x19, 0x36, 0x00, 0x1b, 0x06, 0xc2, 0x0e, 0xec, 0x60,
        0x43, 0x70, 0x07, 0x1b, 0x86, 0xa1, 0x0e, 0xf0, 0x60, 0x82, 0x20, 0x06, 0x66, 0xb0, 0x21, 0xd0, 0x03, 0x12, 0x6d, 0x61, 0x69, 0x6e, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8, 0xd2,
        0xde, 0xdc, 0x26, 0x08, 0x8d, 0x37, 0x41, 0x68, 0xbe, 0x0d, 0x01, 0x31, 0x41, 0x68, 0xc0, 0x60, 0x82, 0xd0, 0x74, 0x1b, 0x16, 0xa2, 0x0f, 0xfc, 0xe0, 0x0f, 0x40, 0x21, 0x14, 0x86, 0x50, 0x20,
        0x44, 0x01, 0xe0, 0x52, 0x65, 0xf5, 0x45, 0x55, 0x84, 0x35, 0xf4, 0xf4, 0x24, 0x45, 0x34, 0x41, 0x68, 0xb8, 0x0d, 0xcb, 0x40, 0x0a, 0x7e, 0x20, 0x0a, 0xa0, 0x50, 0x0a, 0x43, 0x29, 0x0c, 0xa2,
        0x00, 0x6c, 0x10, 0x46, 0xc1, 0x14, 0x98, 0x4c, 0x59, 0x7d, 0x51, 0x85, 0xc9, 0x9d, 0x95, 0xd1, 0x4d, 0x10, 0x9a, 0x6d, 0xc3, 0x42, 0xa0, 0x82, 0x1f, 0xa4, 0x02, 0x28, 0x88, 0xc2, 0x10, 0x0a,
        0x84, 0x28, 0x00, 0x13, 0x84, 0x46, 0xdb, 0x10, 0x0c, 0x13, 0x84, 0x26, 0xdb, 0xb0, 0x0c, 0xa8, 0xb0, 0x0a, 0xa9, 0xc0, 0x0a, 0xa2, 0x30, 0xb4, 0xc2, 0x20, 0x0a, 0xc0, 0x06, 0x41, 0x15, 0x5c,
        0x61, 0xc3, 0x70, 0x0a, 0xaf, 0x00, 0x4c, 0x10, 0xc6, 0x00, 0xdb, 0x20, 0x10, 0xb1, 0xb0, 0xa1, 0xa8, 0x03, 0x3e, 0x80, 0x85, 0x4d, 0x16, 0xaa, 0xb0, 0xb1, 0xd9, 0xb5, 0xb9, 0xa4, 0x91, 0x95,
        0xb9, 0xd1, 0x4d, 0x09, 0x82, 0x2a, 0x64, 0x78, 0x2e, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x53, 0x02, 0xa2, 0x09, 0x19, 0x9e, 0x8b, 0x5d, 0x18, 0x9b, 0x5d, 0x99, 0xdc, 0x94, 0xc0, 0xa8,
        0x43, 0x86, 0xe7, 0x32, 0x87, 0x16, 0x46, 0x56, 0x26, 0xd7, 0xf4, 0x46, 0x56, 0xc6, 0x36, 0x25, 0x48, 0xca, 0x90, 0xe1, 0xb9, 0xc8, 0x95, 0xcd, 0xbd, 0xd5, 0xc9, 0x8d, 0x95, 0xcd, 0x4d, 0x09,
        0xb4, 0x4a, 0x64, 0x78, 0x2e, 0x74, 0x79, 0x70, 0x65, 0x41, 0x6e, 0x6e, 0x6f, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x53, 0x84, 0x39, 0xc0, 0x83, 0x3a, 0x64, 0x78, 0x2e, 0x76, 0x69, 0x65,
        0x77, 0x49, 0x64, 0x53, 0x74, 0x61, 0x74, 0x65, 0x53, 0x02, 0x3d, 0xa8, 0x43, 0x86, 0xe7, 0x52, 0xe6, 0x46, 0x27, 0x97, 0x07, 0xf5, 0x96, 0xe6, 0x46, 0x37, 0x37, 0x25, 0x90, 0x05, 0x00, 0x00,
        0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78,
        0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84,
        0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87,
        0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d,
        0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37,
        0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87,
        0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c,
        0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88,
        0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87,
        0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x35, 0x00,
        0x00, 0x00, 0x46, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x07, 0xdb, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x01, 0x55, 0x14, 0x44, 0x54, 0x3a,
        0xc0, 0x50, 0x12, 0x06, 0x20, 0x60, 0x7e, 0x71, 0xdb, 0x15, 0x80, 0x06, 0x81, 0x1f, 0xbd, 0x8c, 0x07, 0x02, 0x67, 0xd6, 0x1f, 0x89, 0x5a, 0xc6, 0xd3, 0xeb, 0xf2, 0xb2, 0x8c, 0x08, 0xb4, 0xfe,
        0x48, 0xf6, 0xf2, 0x98, 0xfe, 0x96, 0x03, 0x9b, 0x24, 0xd8, 0x0c, 0x08, 0x04, 0x02, 0x83, 0x36, 0x40, 0x30, 0x00, 0xd2, 0x58, 0x02, 0x18, 0x0c, 0x97, 0xef, 0x3c, 0xbe, 0x70, 0x10, 0x82, 0xa2,
        0x09, 0x02, 0x21, 0x49, 0x1f, 0xb5, 0x2c, 0x98, 0x09, 0x3c, 0x0f, 0x31, 0xf8, 0xc8, 0x6d, 0x9b, 0x00, 0x01, 0x44, 0x86, 0x64, 0x05, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28,
        0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x08, 0xd4, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x00, 0xf3, 0x2c, 0x04, 0x18, 0x01, 0x83, 0x5f, 0xdc, 0xb6, 0x19, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x4d,
        0x4e, 0x44, 0xa0, 0xd4, 0xf4, 0x50, 0x93, 0x8f, 0xdc, 0x76, 0x01, 0x64, 0x81, 0xdf, 0x3c, 0xdc, 0x86, 0x03, 0x81, 0xb3, 0xea, 0x34, 0xdc, 0x86, 0xb3, 0xcb, 0xf2, 0x29, 0x3d, 0x4c, 0x2f, 0x03,
        0x81, 0x41, 0x53, 0x90, 0x86, 0xcb, 0x77, 0x1e, 0x5f, 0x88, 0x08, 0x60, 0x22, 0x42, 0xa0, 0x19, 0x16, 0x02, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x13, 0x04,
        0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x94, 0xdc, 0x0c, 0x40, 0xd9, 0x95, 0x5f, 0x39, 0x06, 0x10, 0x37, 0x02, 0x40, 0xc3, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06,
        0x09, 0x00, 0x82, 0x60, 0xf0, 0x95, 0xc1, 0x21, 0x8d, 0xc1, 0x18, 0x08, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xf0, 0x99, 0x01, 0x12, 0x91, 0x01, 0x19, 0x0c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60,
        0xf0, 0x9d, 0x41, 0x52, 0x95, 0x41, 0x19, 0x10, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xe0, 0xb8, 0x41, 0x19, 0x9c, 0x81, 0x19, 0x54, 0xc9, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x38, 0x6f, 0x60,
        0x06, 0x68, 0x70, 0x06, 0x9a, 0x32, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x57, 0x1b, 0x28, 0x03, 0x1a, 0x8c, 0x26, 0x04, 0xc0, 0x68, 0x82, 0x20, 0x8c, 0x26, 0x0c, 0xc1, 0x88, 0x41, 0x05, 0x80,
        0x20, 0x18, 0x68, 0x70, 0xd0, 0x24, 0x88, 0x51, 0x28, 0x0a, 0x1b, 0xb0, 0xc1, 0x13, 0x04, 0x8a, 0x20, 0x28, 0xc3, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc1, 0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10,
        0xc3, 0x68, 0x42, 0x41, 0x8c, 0x18, 0x18, 0x00, 0x08, 0x82, 0x41, 0x57, 0x07, 0x52, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x90, 0x1e, 0xa4, 0xc1, 0x1c, 0xcc, 0x81, 0x18, 0x18, 0x23, 0x06,
        0x09, 0x00, 0x82, 0x60, 0x00, 0xe9, 0x41, 0x1a, 0xcc, 0xc1, 0x1c, 0x98, 0x41, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x90, 0x1e, 0xa4, 0xc1, 0x1c, 0xcc, 0xc1, 0x18, 0x10, 0x23, 0x06, 0x09,
        0x00, 0x82, 0x60, 0x00, 0xe9, 0x41, 0x1a, 0xcc, 0xc1, 0x1c, 0x80, 0xc1, 0x70, 0x01, 0x10, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0xe9, 0x81, 0x1a, 0xd4, 0x01, 0x1d, 0x8c, 0x41, 0x80, 0xe1,
        0x40, 0x00, 0x09, 0x00, 0x00, 0x00, 0x06, 0xc3, 0x48, 0x00, 0xf3, 0xfc, 0x12, 0xc0, 0x3c, 0x0b, 0x11, 0xd9, 0x8b, 0x33, 0x11, 0xd7, 0x3f, 0x11, 0xd7, 0x44, 0x45, 0xc4, 0x6f, 0x0f, 0x16, 0xc3,
        0x00, 0x91, 0x21, 0xfd, 0x42, 0x40, 0x15, 0x05, 0x11, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
#if 0
    SamplerState samp : register(s0);
    Texture2D<float4> tex : register(t0);

    cbuffer args : register(b0)
    {
        float lod;
    };

    void main(float4 pos : SV_POSITION, float2 uv : UV_TEXCOORD, out float4 o0 : SV_TARGET0, out uint o1 : SV_TARGET1)
    {
        uint fb;
        o0 = tex.SampleLevel(samp, uv, lod, int2(0, 0), fb);
        o1 = CheckAccessFullyMapped(fb) ? 1 : 0;
    }
#endif
    static const DWORD ps_sample_lod_dxbc[] =
    {
        0x43425844, 0xcde27032, 0xe798d71a, 0x2983b30e, 0xba80d74a, 0x00000001, 0x000001e4, 0x00000004,
        0x00000030, 0x00000088, 0x000000d4, 0x000001d4, 0x4e475349, 0x00000050, 0x00000002, 0x00000008,
        0x00000038, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000,
        0x00000000, 0x00000003, 0x00000001, 0x00000303, 0x505f5653, 0x5449534f, 0x004e4f49, 0x545f5655,
        0x4f435845, 0x0044524f, 0x4e47534f, 0x00000044, 0x00000002, 0x00000008, 0x00000038, 0x00000000,
        0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x00000038, 0x00000001, 0x00000000, 0x00000001,
        0x00000001, 0x00000e01, 0x545f5653, 0x45475241, 0xabab0054, 0x58454853, 0x000000f8, 0x00000050,
        0x0000003e, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300005a, 0x00106000,
        0x00000000, 0x04001858, 0x00107000, 0x00000000, 0x00005555, 0x03001062, 0x00101032, 0x00000001,
        0x03000065, 0x001020f2, 0x00000000, 0x03000065, 0x00102012, 0x00000001, 0x02000068, 0x00000002,
        0x910000e4, 0x800000c2, 0x00155543, 0x001000f2, 0x00000000, 0x00100012, 0x00000001, 0x00101046,
        0x00000001, 0x00107e46, 0x00000000, 0x00106000, 0x00000000, 0x0020800a, 0x00000000, 0x00000000,
        0x00000000, 0x05000036, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x050000ea, 0x00100012,
        0x00000000, 0x0010000a, 0x00000001, 0x09000037, 0x00102012, 0x00000001, 0x0010000a, 0x00000000,
        0x00004001, 0x00000001, 0x00004001, 0x00000000, 0x0100003e, 0x30494653, 0x00000008, 0x00000100,
        0x00000000, 
    };
    static const BYTE ps_sample_lod_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xc8, 0x56, 0xaa, 0x6f, 0x14, 0x0d, 0x7b, 0x3c, 0x13, 0x36, 0xb5, 0x40, 0x42, 0xf7, 0x4c, 0x86, 0x01, 0x00, 0x00, 0x00, 0xca, 0x0a, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0x06, 0x01, 0x00, 0x00, 0xf2, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x60, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x55, 0x56, 0x5f, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x52, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54,
        0x61, 0x72, 0x67, 0x65, 0x74, 0x00, 0x50, 0x53, 0x56, 0x30, 0xe4, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x02, 0x02, 0x00, 0x02, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x55, 0x56, 0x5f, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x41, 0x10, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xd0, 0x08, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x34, 0x02,
        0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xb8, 0x08, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x2b, 0x02, 0x00, 0x00, 0x0b, 0x82,
        0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04,
        0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19,
        0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c,
        0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64,
        0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x90, 0xc1, 0x1c, 0x01, 0x18, 0xdc, 0x24, 0x4d, 0x11,
        0x25, 0x4c, 0x3e, 0x0b, 0x30, 0xcf, 0x42, 0x44, 0xec, 0x04, 0x4c, 0x04, 0x0a, 0x00, 0x0a, 0x66, 0x00, 0x86, 0x11, 0x88, 0x61, 0xa6, 0x36, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37,
        0xa0, 0x85, 0x72, 0xc0, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x39, 0x20, 0x05, 0x3e, 0xb0, 0x87, 0x72, 0x18, 0x07, 0x7a, 0x78, 0x07, 0x79, 0xe0, 0x03, 0x73, 0x60, 0x87, 0x77, 0x08, 0x07,
        0x7a, 0x60, 0x03, 0x30, 0xa0, 0x03, 0x3f, 0x00, 0x03, 0x3f, 0xd0, 0x03, 0x3d, 0x68, 0x87, 0x74, 0x80, 0x87, 0x79, 0xf8, 0x05, 0x7a, 0xc8, 0x07, 0x78, 0x28, 0x07, 0x14, 0x00, 0x33, 0x89, 0xc1,
        0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0, 0xc3, 0x3b, 0xc8,
        0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0, 0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x01, 0x12, 0x88, 0x82, 0x8c, 0x23, 0x80, 0xc8, 0x90, 0x50, 0x30, 0x10, 0x32, 0x02,
        0x50, 0x02, 0x85, 0x96, 0x39, 0x02, 0xa4, 0x18, 0x03, 0x00, 0x00, 0x0d, 0x40, 0x4e, 0x31, 0x14, 0x00, 0x00, 0x9a, 0x81, 0xa0, 0x62, 0x28, 0x00, 0x00, 0x34, 0x00, 0x49, 0x47, 0x0d, 0x97, 0x3f,
        0x61, 0x0f, 0x21, 0xf9, 0xdc, 0x46, 0x15, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0x23, 0x62, 0x18, 0x86, 0x81, 0x9a, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68, 0x86, 0x85, 0x40, 0x41,
        0x55, 0x08, 0x0a, 0xb0, 0x00, 0xba, 0x6e, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x57, 0x42, 0x5a, 0x89, 0xc9, 0x2f, 0x6e, 0x1b, 0x15, 0xc3, 0x30, 0x0c, 0x40, 0x61, 0x32, 0xc0, 0xb2, 0x86,
        0x61, 0x18, 0x00, 0x00, 0x18, 0x48, 0x9b, 0x23, 0x08, 0xca, 0xc0, 0x01, 0x00, 0x75, 0xc5, 0xb0, 0x80, 0x06, 0x00, 0x38, 0xfa, 0x06, 0x02, 0x12, 0x01, 0x98, 0x23, 0x00, 0x05, 0x00, 0x13, 0x14,
        0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72,
        0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07,
        0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20,
        0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a,
        0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x84, 0x90, 0x21, 0x23, 0x45, 0x44, 0x00, 0xc6, 0x00, 0x80, 0x69, 0x03, 0x00, 0x26, 0x0e, 0x00, 0x60, 0xc8, 0xb3, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xc7, 0x01, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x04, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x43, 0x1e, 0x09, 0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x17, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x34, 0x20,
        0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x74, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0xf9, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x14, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x02, 0x4a, 0x60, 0x04, 0xa0, 0x10, 0x8a,
        0xa1, 0x08, 0x4a, 0xa2, 0x50, 0xca, 0xa0, 0x1c, 0x0a, 0xa4, 0x14, 0x68, 0x2c, 0x40, 0x20, 0x6a, 0x4a, 0xa0, 0x14, 0x0a, 0xa4, 0x08, 0x46, 0x00, 0x4a, 0xa2, 0x0c, 0x0a, 0x81, 0x92, 0x19, 0x00,
        0x42, 0x66, 0x00, 0xe8, 0x98, 0x01, 0x20, 0x63, 0x06, 0x80, 0x8a, 0x19, 0x00, 0x12, 0xc7, 0x42, 0x8c, 0x02, 0x00, 0x00, 0xe0, 0xfb, 0x48, 0x98, 0x01, 0x00, 0x79, 0x18, 0x00, 0x00, 0x89, 0x00,
        0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x83, 0x0c, 0x6f, 0x0c, 0x24, 0xc6, 0x45, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x80, 0x65, 0x82, 0x00, 0x30, 0x1b, 0x84, 0x81, 0x98, 0x20,
        0x00, 0xcd, 0x06, 0x61, 0x30, 0x28, 0xc0, 0xcd, 0x4d, 0x10, 0x00, 0x67, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0xe1, 0x08, 0x03, 0x0e, 0x74, 0x65, 0x78, 0x13, 0x04, 0xe0, 0x99, 0x20, 0x00, 0xd0, 0x06,
        0x81, 0x70, 0x36, 0x24, 0xc4, 0xc2, 0x10, 0xc4, 0xd0, 0x10, 0xcf, 0x86, 0x00, 0x9a, 0x20, 0x24, 0x1f, 0x89, 0x30, 0xb9, 0xb3, 0xb9, 0x0d, 0x08, 0x21, 0x4d, 0x04, 0x31, 0x18, 0xc0, 0x86, 0x80,
        0x9a, 0x20, 0x08, 0x65, 0x40, 0x62, 0x2e, 0xac, 0x0d, 0x6e, 0x03, 0x42, 0x58, 0x17, 0x41, 0x0c, 0x04, 0xb0, 0x21, 0xc0, 0x36, 0x10, 0x11, 0x50, 0x65, 0x13, 0x04, 0x43, 0x0c, 0x26, 0x08, 0x40,
        0xc4, 0x00, 0x6d, 0x82, 0x00, 0x48, 0x13, 0x04, 0x60, 0xda, 0x60, 0x24, 0x9d, 0x47, 0x7c, 0x0e, 0x89, 0xb6, 0x34, 0xb8, 0xb9, 0x09, 0x02, 0x40, 0x6d, 0x20, 0x92, 0x30, 0xf0, 0xc4, 0x60, 0xc3,
        0xc0, 0x81, 0xc1, 0x18, 0x4c, 0x10, 0x8a, 0x31, 0xa0, 0x81, 0x16, 0xe6, 0x46, 0xc6, 0x56, 0x36, 0x41, 0x00, 0xaa, 0x0d, 0x46, 0x62, 0x06, 0x1e, 0xf1, 0x9d, 0xc1, 0x06, 0xc1, 0x40, 0x83, 0x09,
        0x02, 0x02, 0x06, 0x1c, 0xd8, 0xde, 0xc8, 0x36, 0x18, 0xc9, 0x1a, 0x78, 0xc4, 0xe7, 0x6c, 0x10, 0x0c, 0x36, 0xd8, 0x70, 0x10, 0x1b, 0x19, 0x94, 0x41, 0x1a, 0xa8, 0x41, 0x1b, 0x4c, 0x10, 0x98,
        0x61, 0x03, 0xb0, 0x61, 0x20, 0xe0, 0x00, 0x0e, 0x36, 0x04, 0x71, 0xb0, 0x61, 0x18, 0xde, 0x40, 0x0e, 0x26, 0x08, 0x62, 0x40, 0x06, 0x1b, 0x02, 0x3a, 0x20, 0xd1, 0x16, 0x96, 0xe6, 0xc6, 0x65,
        0xca, 0xea, 0x0b, 0xea, 0x6d, 0x2e, 0x8d, 0x2e, 0xed, 0xcd, 0x6d, 0x82, 0xd0, 0x70, 0x13, 0x84, 0xa6, 0xdb, 0x10, 0x10, 0x13, 0x84, 0xc6, 0x9b, 0x20, 0x34, 0xdb, 0x86, 0x85, 0xb8, 0x03, 0x3c,
        0xc8, 0x03, 0x3d, 0xd8, 0x83, 0x61, 0x0f, 0x08, 0x3e, 0x00, 0xb8, 0x54, 0x59, 0x7d, 0x51, 0x15, 0x61, 0x0d, 0x3d, 0x3d, 0x49, 0x11, 0x4d, 0x10, 0x1a, 0x6d, 0xc3, 0x32, 0xf8, 0x01, 0x1e, 0xf0,
        0x81, 0x1e, 0xfc, 0xc1, 0xf0, 0x07, 0x03, 0x1f, 0x00, 0x1b, 0x84, 0x3e, 0x00, 0x05, 0x26, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74, 0x13, 0x84, 0x26, 0xdb, 0xb0, 0x10, 0xa2, 0x80,
        0x07, 0xa3, 0xa0, 0x07, 0x7c, 0x30, 0xec, 0x01, 0xc1, 0x07, 0xc0, 0x04, 0xa1, 0xc1, 0x36, 0x04, 0xc3, 0x04, 0xa1, 0xb9, 0x36, 0x2c, 0x83, 0x28, 0x94, 0xc2, 0x28, 0x98, 0x02, 0x1f, 0x0c, 0xa7,
        0x30, 0xf0, 0x01, 0xb0, 0x41, 0x20, 0x05, 0x54, 0xd8, 0x30, 0x84, 0x42, 0x2a, 0x00, 0x13, 0x84, 0x31, 0xb0, 0x36, 0x08, 0xc4, 0x2a, 0x6c, 0x28, 0xde, 0xc0, 0x0e, 0x54, 0x41, 0x63, 0x85, 0x2a,
        0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c, 0xda, 0x9b, 0xdb, 0x94, 0x80, 0x68, 0x42, 0x86, 0xe7, 0x62, 0x17,
        0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x30, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95, 0xb1, 0x4d, 0x09, 0x92, 0x32, 0x64, 0x78, 0x2e, 0x72, 0x65, 0x73,
        0x6f, 0x75, 0x72, 0x63, 0x65, 0x73, 0x53, 0x82, 0xac, 0x12, 0x19, 0x9e, 0x0b, 0x5d, 0x1e, 0x5c, 0x59, 0x90, 0x9b, 0xdb, 0x1b, 0x5d, 0x18, 0x5d, 0xda, 0x9b, 0xdb, 0xdc, 0x14, 0xa1, 0x0d, 0xe4,
        0xa0, 0x0e, 0x19, 0x9e, 0x8b, 0x5d, 0x5a, 0xd9, 0x5d, 0x12, 0xd9, 0x14, 0x5d, 0x18, 0x5d, 0xd9, 0x94, 0x80, 0x0e, 0xea, 0x90, 0xe1, 0xb9, 0x94, 0xb9, 0xd1, 0xc9, 0xe5, 0x41, 0xbd, 0xa5, 0xb9,
        0xd1, 0xcd, 0x4d, 0x09, 0x58, 0x01, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42,
        0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d,
        0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70,
        0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21,
        0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07,
        0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08,
        0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1,
        0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38,
        0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90,
        0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20,
        0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0x46, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x07, 0xdb, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x01, 0x55,
        0x14, 0x44, 0x54, 0x3a, 0xc0, 0x50, 0x12, 0x06, 0x20, 0x60, 0x7e, 0x71, 0xdb, 0x15, 0x80, 0x06, 0x81, 0x1f, 0xbd, 0x8c, 0x07, 0x02, 0x67, 0xd6, 0x1f, 0x89, 0x5a, 0xc6, 0xd3, 0xeb, 0xf2, 0xb2,
        0x8c, 0x08, 0xb4, 0xfe, 0x48, 0xf6, 0xf2, 0x98, 0xfe, 0x96, 0x03, 0x9b, 0x24, 0xd8, 0x0c, 0x08, 0x04, 0x02, 0x83, 0x36, 0x40, 0x30, 0x00, 0xd2, 0x58, 0x02, 0x18, 0x0c, 0x97, 0xef, 0x3c, 0xbe,
        0x70, 0x10, 0x82, 0xa2, 0x09, 0x02, 0x21, 0x49, 0x1f, 0xb5, 0x2c, 0x98, 0x09, 0x3c, 0x0f, 0x31, 0xf8, 0xc8, 0x6d, 0x1b, 0x42, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x12, 0xc0, 0x3c, 0x0b, 0x51, 0x12,
        0x15, 0xb1, 0xf8, 0xc5, 0x6d, 0x9b, 0x00, 0x01, 0x44, 0x86, 0x64, 0x05, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb7, 0x5d, 0x00, 0x59, 0xe0,
        0x37, 0x0f, 0xb7, 0xe1, 0x40, 0xe0, 0xac, 0x3a, 0x0d, 0xb7, 0xe1, 0xec, 0xb2, 0x7c, 0x4a, 0x0f, 0xd3, 0xcb, 0x40, 0x60, 0xd0, 0x0c, 0xaa, 0xe1, 0xf2, 0x9d, 0xc7, 0x97, 0x26, 0x27, 0x22, 0x50,
        0x6a, 0x7a, 0xa8, 0xc9, 0x47, 0x6e, 0xdb, 0x14, 0xa4, 0xe1, 0xf2, 0x9d, 0xc7, 0x17, 0x22, 0x02, 0x98, 0x88, 0x10, 0x68, 0x86, 0x85, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x49, 0x00,
        0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x94, 0xdc, 0x0c, 0x40, 0xd9, 0x15, 0x5f, 0x39, 0x06, 0x10, 0x37, 0x02, 0x40, 0xc3, 0x0c, 0x00, 0x00,
        0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xf0, 0x8d, 0xc1, 0x21, 0x85, 0x41, 0x18, 0x08, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xf0, 0x91, 0x01, 0x12, 0x89, 0x81, 0x18, 0x0c, 0x23, 0x06,
        0x09, 0x00, 0x82, 0x60, 0xf0, 0x95, 0x41, 0x52, 0x8d, 0xc1, 0x18, 0x10, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xe0, 0xb0, 0xc1, 0x18, 0x94, 0x01, 0x19, 0x54, 0xc9, 0x88, 0x41, 0x02, 0x80, 0x20,
        0x18, 0x38, 0x6d, 0x40, 0x06, 0x66, 0x50, 0x06, 0x9a, 0x32, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0xd7, 0x1a, 0x28, 0x83, 0x19, 0x8c, 0x26, 0x04, 0xc0, 0x88, 0xc1, 0x03, 0x80, 0x20, 0x18, 0x68,
        0x6c, 0xb0, 0x1c, 0x06, 0x31, 0x20, 0x08, 0x1a, 0xa0, 0x41, 0x13, 0x8c, 0x26, 0x04, 0xc0, 0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68, 0x02, 0x31, 0x8c, 0x26, 0x14, 0xc4, 0x88, 0x81, 0x01,
        0x80, 0x20, 0x18, 0x74, 0x71, 0x00, 0x05, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x00, 0xd9, 0x81, 0x19, 0xbc, 0xc1, 0x1b, 0x80, 0x81, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x90, 0x1d, 0x98,
        0xc1, 0x1b, 0xbc, 0x01, 0x19, 0x14, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x00, 0xd9, 0x81, 0x19, 0xbc, 0xc1, 0x1b, 0x84, 0x01, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x90, 0x1d, 0x98, 0xc1,
        0x1b, 0xbc, 0x81, 0x37, 0x5c, 0x00, 0xc4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x48, 0x76, 0x70, 0x06, 0x71, 0x00, 0x07, 0x61, 0x10, 0x60, 0x38, 0x10, 0x00, 0x09, 0x00, 0x00, 0x00, 0xe6, 0xc2,
        0x48, 0x00, 0xf3, 0xfc, 0x12, 0xc0, 0x3c, 0x0b, 0x11, 0x59, 0x8b, 0x33, 0x11, 0xd7, 0x3f, 0x11, 0xd7, 0x44, 0x45, 0xc4, 0x6f, 0x0f, 0xf6, 0xc2, 0x00, 0x91, 0x21, 0xfd, 0x42, 0x40, 0x15, 0x05,
        0x11, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    };
#if 0
    SamplerState samp : register(s0);
    Texture2D<float4> tex : register(t0);

    void main(float4 pos : SV_POSITION, float2 uv : UV_TEXCOORD, out float4 o0 : SV_TARGET0, out uint o1 : SV_TARGET1)
    {
        uint fb;
        o0 = tex.Gather(samp, uv, int2(0, 0), fb);
        o1 = CheckAccessFullyMapped(fb) ? 1 : 0;
    }
#endif
    static const DWORD ps_gather_dxbc[] =
    {
        0x43425844, 0xc880fe7b, 0x43ca814e, 0xb2213fd2, 0x0b33a886, 0x00000001, 0x000001c4, 0x00000004,
        0x00000030, 0x00000088, 0x000000d4, 0x000001b4, 0x4e475349, 0x00000050, 0x00000002, 0x00000008,
        0x00000038, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000,
        0x00000000, 0x00000003, 0x00000001, 0x00000303, 0x505f5653, 0x5449534f, 0x004e4f49, 0x545f5655,
        0x4f435845, 0x0044524f, 0x4e47534f, 0x00000044, 0x00000002, 0x00000008, 0x00000038, 0x00000000,
        0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x00000038, 0x00000001, 0x00000000, 0x00000001,
        0x00000001, 0x00000e01, 0x545f5653, 0x45475241, 0xabab0054, 0x58454853, 0x000000d8, 0x00000050,
        0x00000036, 0x0100086a, 0x0300005a, 0x00106000, 0x00000000, 0x04001858, 0x00107000, 0x00000000,
        0x00005555, 0x03001062, 0x00101032, 0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x03000065,
        0x00102012, 0x00000001, 0x02000068, 0x00000002, 0x8d0000db, 0x800000c2, 0x00155543, 0x001000f2,
        0x00000000, 0x00100012, 0x00000001, 0x00101046, 0x00000001, 0x00107e46, 0x00000000, 0x0010600a,
        0x00000000, 0x05000036, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x050000ea, 0x00100012,
        0x00000000, 0x0010000a, 0x00000001, 0x09000037, 0x00102012, 0x00000001, 0x0010000a, 0x00000000,
        0x00004001, 0x00000001, 0x00004001, 0x00000000, 0x0100003e, 0x30494653, 0x00000008, 0x00000100,
        0x00000000, 
    };
    static const BYTE ps_gather_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xb3, 0xf3, 0x2a, 0x90, 0x1a, 0x83, 0x22, 0xdd, 0x58, 0x26, 0xac, 0x7e, 0xc3, 0x74, 0xdf, 0x91, 0x01, 0x00, 0x00, 0x00, 0x06, 0x0a, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0x06, 0x01, 0x00, 0x00, 0xe2, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x60, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x55, 0x56, 0x5f, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x52, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54,
        0x61, 0x72, 0x67, 0x65, 0x74, 0x00, 0x50, 0x53, 0x56, 0x30, 0xd4, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x02, 0x02, 0x00, 0x02, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00,
        0x00, 0x00, 0x00, 0x55, 0x56, 0x5f, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x41, 0x10, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x1c, 0x08, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x07, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x04, 0x08,
        0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xfe, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8,
        0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08,
        0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83,
        0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff,
        0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00,
        0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x47, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c,
        0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x7c, 0x73, 0x04, 0x60, 0x70, 0x93, 0x34, 0x45, 0x94, 0x30, 0xf9, 0x2c, 0xc0, 0x3c, 0x0b, 0x11, 0xb1, 0x13, 0x30, 0x11, 0x28, 0x00, 0x14, 0x66, 0x00,
        0x86, 0x11, 0x88, 0x61, 0xa6, 0x36, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37, 0xa0, 0x85, 0x72, 0xc0, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x39, 0x20, 0x05, 0x3e, 0xb0, 0x87,
        0x72, 0x18, 0x07, 0x7a, 0x78, 0x07, 0x79, 0xe0, 0x03, 0x73, 0x60, 0x87, 0x77, 0x08, 0x07, 0x7a, 0x60, 0x03, 0x30, 0xa0, 0x03, 0x3f, 0x00, 0x03, 0x3f, 0xd0, 0x03, 0x3d, 0x68, 0x87, 0x74, 0x80,
        0x87, 0x79, 0xf8, 0x05, 0x7a, 0xc8, 0x07, 0x78, 0x28, 0x07, 0x14, 0x80, 0x99, 0xc4, 0x60, 0x1c, 0xd8, 0x21, 0x1c, 0xe6, 0x61, 0x1e, 0xdc, 0x80, 0x16, 0xca, 0x01, 0x1f, 0xe8, 0xa1, 0x1e, 0xe4,
        0xa1, 0x1c, 0xe4, 0x80, 0x14, 0xf8, 0xc0, 0x1e, 0xca, 0x61, 0x1c, 0xe8, 0xe1, 0x1d, 0xe4, 0x81, 0x0f, 0xcc, 0x81, 0x1d, 0xde, 0x21, 0x1c, 0xe8, 0x81, 0x0d, 0xc0, 0x80, 0x0e, 0xfc, 0x00, 0x0c,
        0xfc, 0x00, 0x09, 0xa4, 0x90, 0x19, 0x01, 0x28, 0x01, 0xa2, 0x34, 0x47, 0x80, 0x14, 0x63, 0x00, 0x00, 0x16, 0x20, 0x56, 0x0c, 0x04, 0x00, 0x58, 0x83, 0x5c, 0x31, 0x10, 0x00, 0x60, 0x01, 0x82,
        0x37, 0x0d, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x2b, 0x21, 0xad, 0xc4, 0xe4, 0x17, 0xb7, 0x8d, 0x8a, 0x31, 0xc6, 0x00, 0xb4, 0xee, 0x19, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x43, 0xa0, 0x19,
        0x16, 0x02, 0x05, 0xb3, 0x2c, 0x12, 0x50, 0x3a, 0xc6, 0x18, 0x00, 0x00, 0xaa, 0x73, 0x04, 0x41, 0x19, 0x2e, 0x00, 0x84, 0x8b, 0x41, 0xc1, 0x02, 0xe0, 0x92, 0x1e, 0x08, 0x48, 0x04, 0x30, 0x47,
        0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50,
        0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06,
        0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0,
        0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21, 0x23, 0x45, 0x44, 0x00, 0x66, 0x00, 0xc0, 0xf4, 0x00, 0x80, 0x87,
        0x3c, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x18, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x38, 0x40, 0x00, 0x08, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x53, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x90, 0x07, 0x03, 0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x8f, 0x06, 0x04, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x13, 0x00,
        0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x02, 0x25, 0x30, 0x02, 0x50, 0x08, 0xc5, 0x50, 0x04, 0x25, 0x51, 0x28, 0x65, 0x50, 0x0e,
        0x05, 0x52, 0x0a, 0xe4, 0x0b, 0x10, 0x88, 0x56, 0x09, 0x94, 0x42, 0x81, 0x94, 0x44, 0x19, 0x14, 0xc2, 0x08, 0x40, 0x11, 0xd0, 0x99, 0x01, 0x20, 0x33, 0x03, 0x40, 0x65, 0x06, 0x80, 0xfa, 0x58,
        0x88, 0x51, 0x00, 0x00, 0x00, 0x7c, 0x1f, 0x89, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x83, 0x0c,
        0x6f, 0x0c, 0x24, 0xc6, 0x45, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x80, 0x64, 0x82, 0x00, 0x28, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x00, 0xcb, 0x06, 0x61, 0x30, 0x28, 0xc0, 0xcd, 0x4d, 0x10, 0x00, 0x66,
        0xc3, 0x80, 0x24, 0xc4, 0x04, 0xe1, 0xe8, 0x38, 0xd0, 0x95, 0xe1, 0x4d, 0x10, 0x80, 0x66, 0x82, 0x00, 0x38, 0x1b, 0x04, 0xc2, 0xd9, 0x90, 0x10, 0x0b, 0x43, 0x10, 0x43, 0x43, 0x3c, 0x1b, 0x02,
        0x68, 0x82, 0x20, 0x84, 0x01, 0x89, 0xb9, 0xb0, 0x36, 0xb8, 0x0d, 0x08, 0x21, 0x4d, 0x04, 0x31, 0x10, 0xc0, 0x86, 0x80, 0xda, 0x40, 0x44, 0x00, 0x50, 0x4d, 0x10, 0x0c, 0x6f, 0x82, 0x00, 0x3c,
        0x0c, 0xd0, 0x26, 0x08, 0x00, 0x34, 0x41, 0x00, 0xa2, 0x0d, 0x46, 0x92, 0x69, 0xc4, 0xe6, 0x90, 0x68, 0x4b, 0x83, 0x9b, 0x9b, 0x20, 0x00, 0xd2, 0x06, 0x22, 0xe9, 0x34, 0x6f, 0xc3, 0x80, 0x71,
        0xdf, 0x04, 0xa1, 0xf8, 0x68, 0xa0, 0x85, 0xb9, 0x91, 0xb1, 0x95, 0x4d, 0x10, 0x80, 0x69, 0x83, 0x91, 0x88, 0x81, 0x46, 0x6c, 0x63, 0xb0, 0x41, 0x30, 0xc8, 0x60, 0x43, 0x41, 0x5c, 0x60, 0x10,
        0x06, 0x65, 0x30, 0x41, 0x50, 0x84, 0x0d, 0xc0, 0x86, 0x81, 0x40, 0x03, 0x34, 0xd8, 0x10, 0xa4, 0xc1, 0x86, 0x61, 0x38, 0x03, 0x35, 0x98, 0x20, 0x74, 0x60, 0xb0, 0x21, 0x60, 0x03, 0x12, 0x6d,
        0x61, 0x69, 0x6e, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8, 0xd2, 0xde, 0xdc, 0x26, 0x08, 0x0b, 0x36, 0x41, 0x58, 0xb2, 0x0d, 0x01, 0x31, 0x41, 0x58, 0xb4, 0x09, 0xc2, 0xb2, 0x6d,
        0x58, 0x88, 0x37, 0x80, 0x83, 0x38, 0x90, 0x83, 0x39, 0x18, 0xe6, 0x80, 0xa0, 0x03, 0x80, 0x4b, 0x95, 0xd5, 0x17, 0x55, 0x11, 0xd6, 0xd0, 0xd3, 0x93, 0x14, 0xd1, 0x04, 0x61, 0xe1, 0x36, 0x2c,
        0x83, 0x1d, 0xc0, 0x01, 0x1d, 0xc8, 0xc1, 0x1d, 0x0c, 0x77, 0x30, 0xd0, 0x01, 0xb0, 0x41, 0xa8, 0x03, 0x3c, 0x60, 0x32, 0x65, 0xf5, 0x45, 0x15, 0x26, 0x77, 0x56, 0x46, 0x37, 0x41, 0x58, 0xae,
        0x0d, 0x0b, 0xa1, 0x07, 0x70, 0xb0, 0x07, 0x72, 0x40, 0x07, 0xc3, 0x1c, 0x10, 0x74, 0x00, 0x4c, 0x10, 0x16, 0x6b, 0x43, 0x30, 0x4c, 0x10, 0x96, 0x6a, 0xc3, 0x32, 0xe8, 0x41, 0x1f, 0xec, 0x81,
        0x1f, 0xd0, 0xc1, 0xf0, 0x07, 0x03, 0x1d, 0x00, 0x1b, 0x04, 0x3e, 0x00, 0x85, 0x0d, 0x43, 0x1e, 0x84, 0x02, 0x30, 0x41, 0xf0, 0xa8, 0x0d, 0x02, 0x31, 0x0a, 0x1b, 0x8a, 0x33, 0x70, 0x03, 0x51,
        0xb0, 0x48, 0xa1, 0x0a, 0x1b, 0x9b, 0x5d, 0x9b, 0x4b, 0x1a, 0x59, 0x99, 0x1b, 0xdd, 0x94, 0x20, 0xa8, 0x42, 0x86, 0xe7, 0x62, 0x57, 0x26, 0x37, 0x97, 0xf6, 0xe6, 0x36, 0x25, 0x20, 0x9a, 0x90,
        0xe1, 0xb9, 0xd8, 0x85, 0xb1, 0xd9, 0x95, 0xc9, 0x4d, 0x09, 0x8c, 0x3a, 0x64, 0x78, 0x2e, 0x73, 0x68, 0x61, 0x64, 0x65, 0x72, 0x4d, 0x6f, 0x64, 0x65, 0x6c, 0x53, 0x82, 0xa4, 0x0c, 0x19, 0x9e,
        0x8b, 0x5c, 0xd9, 0xdc, 0x5b, 0x9d, 0xdc, 0x58, 0xd9, 0xdc, 0x94, 0xa0, 0xaa, 0x44, 0x86, 0xe7, 0x42, 0x97, 0x07, 0x57, 0x16, 0xe4, 0xe6, 0xf6, 0x46, 0x17, 0x46, 0x97, 0xf6, 0xe6, 0x36, 0x37,
        0x45, 0x28, 0x03, 0x35, 0xa8, 0x43, 0x86, 0xe7, 0x62, 0x97, 0x56, 0x76, 0x97, 0x44, 0x36, 0x45, 0x17, 0x46, 0x57, 0x36, 0x25, 0x60, 0x83, 0x3a, 0x64, 0x78, 0x2e, 0x65, 0x6e, 0x74, 0x72, 0x79,
        0x50, 0x6f, 0x69, 0x6e, 0x74, 0x73, 0x53, 0x02, 0x52, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38,
        0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c,
        0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70,
        0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10,
        0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0,
        0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77,
        0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4,
        0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43,
        0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74,
        0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00,
        0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x00, 0x36, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x5d, 0x01, 0x68, 0x10, 0xf8, 0xd1,
        0xcb, 0x78, 0x20, 0x70, 0x66, 0xfd, 0x91, 0xa8, 0x65, 0x3c, 0xbd, 0x2e, 0x2f, 0xcb, 0x88, 0x40, 0xeb, 0x8f, 0x64, 0x2f, 0x8f, 0xe9, 0x6f, 0x39, 0xb0, 0x49, 0x82, 0xcd, 0x80, 0x40, 0x20, 0x30,
        0x68, 0x02, 0x04, 0x03, 0x20, 0x8d, 0x1d, 0x80, 0xc1, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x07, 0x21, 0x28, 0x9a, 0x20, 0x10, 0x92, 0xf4, 0x51, 0xcb, 0x82, 0x99, 0xc0, 0xf3, 0x10, 0x83, 0x8f, 0xdc,
        0xb6, 0x11, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x4d, 0x4e, 0x44, 0xa0, 0xd4, 0xf4, 0x50, 0x93, 0x5f, 0xdc, 0x76, 0x01, 0x64, 0x81, 0xdf, 0x3c, 0xdc, 0x86, 0x03, 0x81, 0xb3, 0xea, 0x34, 0xdc,
        0x86, 0xb3, 0xcb, 0xf2, 0x29, 0x3d, 0x4c, 0x2f, 0x03, 0x81, 0x41, 0x2b, 0xa8, 0x86, 0xcb, 0x77, 0x1e, 0x5f, 0x9a, 0x9c, 0x88, 0x40, 0xa9, 0xe9, 0xa1, 0x26, 0x1f, 0xb9, 0x6d, 0x43, 0x90, 0x86,
        0xcb, 0x77, 0x1e, 0x5f, 0x88, 0x08, 0x60, 0x22, 0x42, 0xa0, 0x19, 0x16, 0xc2, 0x0c, 0xae, 0xe1, 0xf2, 0x9d, 0xc7, 0x9f, 0x88, 0x6b, 0xa2, 0x22, 0x02, 0x04, 0xa6, 0x83, 0x88, 0xfc, 0xe2, 0xb6,
        0x01, 0x00, 0x61, 0x20, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x4a, 0x6e, 0x06, 0xa0, 0x24, 0x03, 0xca, 0x31, 0x80,
        0xee, 0x08, 0x00, 0x8d, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xa0, 0x81, 0x81, 0xd1, 0x78, 0x9e, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x5a, 0x18, 0x1c,
        0xd0, 0xf7, 0x0d, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xc0, 0x9c, 0xc1, 0x17, 0x06, 0x60, 0xf0, 0x1c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xc0, 0xa0, 0x01, 0x18, 0x88, 0x41, 0x18, 0x58, 0xc8,
        0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x54, 0x66, 0x80, 0x10, 0x83, 0x10, 0x14, 0x85, 0x18, 0x88, 0x81, 0x18, 0x8c, 0x26, 0x04, 0xc0, 0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68, 0x02, 0x31,
        0x8c, 0x26, 0x14, 0xc4, 0x88, 0x81, 0x01, 0x80, 0x20, 0x18, 0x60, 0x6b, 0xd0, 0x04, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xe0, 0xbc, 0x01, 0x18, 0xa4, 0x41, 0x1a, 0x60, 0xc6, 0x88, 0x41, 0x02,
        0x80, 0x20, 0x18, 0x38, 0x6f, 0x00, 0x06, 0x69, 0x90, 0x06, 0x5e, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xce, 0x1b, 0x80, 0x41, 0x1a, 0xa4, 0xc1, 0x45, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82,
        0x81, 0xf3, 0x06, 0x60, 0x90, 0x06, 0x69, 0xa0, 0x0d, 0x17, 0x80, 0x18, 0x31, 0x48, 0x00, 0x10, 0x04, 0x03, 0xe8, 0x0d, 0xc2, 0x60, 0x0d, 0xd4, 0x20, 0x0b, 0x30, 0x1c, 0x08, 0x00, 0x07, 0x00,
        0x00, 0x00, 0x96, 0xc2, 0x48, 0x00, 0xf3, 0xfc, 0x12, 0xc0, 0x3c, 0x0b, 0x11, 0x19, 0x8a, 0x33, 0x11, 0xd7, 0x3f, 0x11, 0xd7, 0x44, 0x45, 0xc4, 0x6f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    };
#if 0
    SamplerState samp : register(s0);
    Texture2D<float4> tex : register(t0);

    cbuffer args : register(b0)
    {
        float2 offset;
    };

    void main(float4 pos : SV_POSITION, float2 uv : UV_TEXCOORD, out float4 o0 : SV_TARGET0, out uint o1 : SV_TARGET1)
    {
        uint fb;
        int2 ofs = int2(offset);
        o0 = tex.GatherRed(samp, uv, ofs, ofs, ofs, ofs, fb);
        o1 = CheckAccessFullyMapped(fb) ? 1 : 0;
    }
#endif
    static const DWORD ps_gather_po_dxbc[] =
    {
        0x43425844, 0x866266cf, 0xf5812e71, 0xe43abe96, 0x391fee80, 0x00000001, 0x000001f4, 0x00000004,
        0x00000030, 0x00000088, 0x000000d4, 0x000001e4, 0x4e475349, 0x00000050, 0x00000002, 0x00000008,
        0x00000038, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x00000044, 0x00000000,
        0x00000000, 0x00000003, 0x00000001, 0x00000303, 0x505f5653, 0x5449534f, 0x004e4f49, 0x545f5655,
        0x4f435845, 0x0044524f, 0x4e47534f, 0x00000044, 0x00000002, 0x00000008, 0x00000038, 0x00000000,
        0x00000000, 0x00000003, 0x00000000, 0x0000000f, 0x00000038, 0x00000001, 0x00000000, 0x00000001,
        0x00000001, 0x00000e01, 0x545f5653, 0x45475241, 0xabab0054, 0x58454853, 0x00000108, 0x00000050,
        0x00000042, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x0300005a, 0x00106000,
        0x00000000, 0x04001858, 0x00107000, 0x00000000, 0x00005555, 0x03001062, 0x00101032, 0x00000001,
        0x03000065, 0x001020f2, 0x00000000, 0x03000065, 0x00102012, 0x00000001, 0x02000068, 0x00000002,
        0x0600001b, 0x00100032, 0x00000000, 0x00208046, 0x00000000, 0x00000000, 0x8f0000dd, 0x800000c2,
        0x00155543, 0x001000f2, 0x00000000, 0x00100012, 0x00000001, 0x00101046, 0x00000001, 0x00100046,
        0x00000000, 0x00107e46, 0x00000000, 0x0010600a, 0x00000000, 0x05000036, 0x001020f2, 0x00000000,
        0x00100e46, 0x00000000, 0x050000ea, 0x00100012, 0x00000000, 0x0010000a, 0x00000001, 0x09000037,
        0x00102012, 0x00000001, 0x0010000a, 0x00000000, 0x00004001, 0x00000001, 0x00004001, 0x00000000,
        0x0100003e, 0x30494653, 0x00000008, 0x00000100, 0x00000000, 
    };
    static const BYTE ps_gather_po_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xfc, 0xc6, 0xf7, 0x22, 0x4d, 0x23, 0xc3, 0x7f, 0x71, 0x61, 0x46, 0x25, 0xb5, 0xc8, 0x98, 0x47, 0x01, 0x00, 0x00, 0x00, 0xee, 0x0a, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0x06, 0x01, 0x00, 0x00, 0xf2, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x60, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x55, 0x56, 0x5f, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x52, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54,
        0x61, 0x72, 0x67, 0x65, 0x74, 0x00, 0x50, 0x53, 0x56, 0x30, 0xe4, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x02, 0x02, 0x00, 0x02, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x55, 0x56, 0x5f, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x41, 0x10, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xf4, 0x08, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x3d, 0x02,
        0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xdc, 0x08, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x34, 0x02, 0x00, 0x00, 0x0b, 0x82,
        0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04,
        0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19,
        0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c,
        0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x52, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64,
        0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x94, 0xc1, 0x1c, 0x01, 0x18, 0xdc, 0x24, 0x4d, 0x11,
        0x25, 0x4c, 0x3e, 0x0b, 0x30, 0xcf, 0x42, 0x44, 0xec, 0x04, 0x4c, 0x04, 0x0a, 0x00, 0x0a, 0x66, 0x00, 0x86, 0x11, 0x88, 0x61, 0xa6, 0x36, 0x18, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37,
        0xa0, 0x85, 0x72, 0xc0, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x39, 0x20, 0x05, 0x3e, 0xb0, 0x87, 0x72, 0x18, 0x07, 0x7a, 0x78, 0x07, 0x79, 0xe0, 0x03, 0x73, 0x60, 0x87, 0x77, 0x08, 0x07,
        0x7a, 0x60, 0x03, 0x30, 0xa0, 0x03, 0x3f, 0x00, 0x03, 0x3f, 0xd0, 0x03, 0x3d, 0x68, 0x87, 0x74, 0x80, 0x87, 0x79, 0xf8, 0x05, 0x7a, 0xc8, 0x07, 0x78, 0x28, 0x07, 0x14, 0x00, 0x33, 0x89, 0xc1,
        0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0, 0xc3, 0x3b, 0xc8,
        0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0, 0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x01, 0x12, 0x88, 0x82, 0x8c, 0x61, 0x04, 0x61, 0x38, 0x02, 0x88, 0x0c, 0x09, 0x05,
        0x08, 0x25, 0x23, 0x00, 0x25, 0x58, 0x88, 0x99, 0x23, 0x40, 0x8a, 0x31, 0x00, 0x00, 0xe0, 0x00, 0xf4, 0x14, 0x63, 0x01, 0x00, 0xc0, 0x19, 0x28, 0x2a, 0xc6, 0x02, 0x00, 0x80, 0x03, 0xd0, 0x74,
        0xd4, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xcf, 0x6d, 0x54, 0xb1, 0x12, 0x93, 0x5f, 0xdc, 0x36, 0x22, 0x86, 0x61, 0x18, 0xc8, 0xb9, 0x67, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x0f, 0x81, 0x66,
        0x58, 0x08, 0x14, 0x58, 0x85, 0xa8, 0x80, 0x0b, 0x20, 0xec, 0xa6, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0xa4, 0x95, 0x98, 0xfc, 0xe2, 0xb6, 0x51, 0x31, 0x0c, 0xc3, 0x00, 0x94, 0x45,
        0x03, 0xae, 0x6b, 0x18, 0x86, 0x01, 0x00, 0x00, 0xda, 0xe6, 0x08, 0x82, 0x32, 0x74, 0x00, 0x40, 0x5e, 0x31, 0x2e, 0xc0, 0x01, 0x80, 0x8e, 0xc0, 0x81, 0x80, 0x44, 0x00, 0xe6, 0x08, 0x40, 0x01,
        0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f,
        0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76,
        0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07,
        0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x84, 0x90, 0x21, 0x23, 0x45, 0x44, 0x00, 0xc6, 0x00, 0x80, 0x69, 0x03, 0x00, 0x66, 0x0e, 0x00, 0x60, 0xc8,
        0xc3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xe7, 0x01, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x4f, 0x04, 0x04, 0x80, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x9e, 0x09, 0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x18, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x0c, 0x79, 0x36, 0x20, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x78, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x01, 0x03, 0x20, 0x00,
        0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x02, 0x01, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04,
        0x43, 0x02, 0x4a, 0x60, 0x04, 0xa0, 0x10, 0x8a, 0xa1, 0x08, 0x4a, 0xa2, 0x20, 0x0a, 0xa5, 0x0c, 0xca, 0xa1, 0x40, 0x4a, 0x81, 0xc8, 0x02, 0x04, 0x22, 0xa7, 0x04, 0x4a, 0xa1, 0x40, 0x8a, 0x60,
        0x04, 0xa0, 0x24, 0xca, 0xa0, 0x10, 0x48, 0x99, 0x01, 0xa0, 0x64, 0x06, 0x80, 0x8e, 0x19, 0x00, 0x32, 0x66, 0x00, 0xa8, 0x98, 0x01, 0xa0, 0x71, 0x2c, 0xc4, 0x28, 0x00, 0x00, 0x00, 0xbe, 0x8f,
        0x84, 0x19, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x8c, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x83, 0x0c, 0x6f, 0x0c, 0x24, 0xc6, 0x45, 0x66, 0x43, 0x10,
        0x4c, 0x10, 0x80, 0x65, 0x82, 0x00, 0x30, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x00, 0xcd, 0x06, 0x61, 0x30, 0x28, 0xc0, 0xcd, 0x4d, 0x10, 0x00, 0x67, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0xe1, 0x10, 0x03,
        0x0e, 0x74, 0x65, 0x78, 0x13, 0x04, 0xe0, 0x99, 0x20, 0x00, 0xd0, 0x06, 0x81, 0x70, 0x36, 0x24, 0xc4, 0xc2, 0x10, 0xc4, 0xd0, 0x10, 0xcf, 0x86, 0x00, 0x9a, 0x20, 0x28, 0x60, 0x40, 0x22, 0x4c,
        0xee, 0x6c, 0x6e, 0x82, 0x00, 0x44, 0x1b, 0x10, 0x42, 0x9a, 0x08, 0x62, 0xa0, 0x80, 0x0d, 0x41, 0x35, 0x41, 0x10, 0xcc, 0x80, 0xc4, 0x5c, 0x58, 0x1b, 0xdc, 0x06, 0x84, 0xb8, 0x30, 0x82, 0x18,
        0x08, 0x60, 0x43, 0x90, 0x6d, 0x20, 0x22, 0xc0, 0xd2, 0x26, 0x08, 0xc6, 0x18, 0x4c, 0x10, 0x00, 0x89, 0x01, 0xda, 0x04, 0x01, 0x98, 0x26, 0x08, 0x00, 0xb5, 0xc1, 0x48, 0xbc, 0x8f, 0x00, 0x03,
        0x87, 0x44, 0x5b, 0x1a, 0xdc, 0xdc, 0x04, 0x01, 0xa8, 0x36, 0x10, 0x89, 0x18, 0x7c, 0x63, 0xb0, 0x61, 0xe8, 0xc2, 0x80, 0x0c, 0x26, 0x08, 0x05, 0x19, 0xd0, 0x40, 0x0b, 0x73, 0x23, 0x63, 0x2b,
        0x9b, 0x20, 0x00, 0xd6, 0x06, 0x23, 0x39, 0x83, 0x8f, 0x00, 0x03, 0x34, 0xd8, 0x20, 0x18, 0x69, 0x30, 0x41, 0x48, 0xc2, 0x80, 0xc6, 0x9b, 0x99, 0xd9, 0x5c, 0x19, 0xdd, 0x06, 0x23, 0x61, 0x83,
        0x8f, 0x00, 0x03, 0x67, 0x83, 0x40, 0xb5, 0xc1, 0x86, 0x83, 0xe0, 0xca, 0xc0, 0x0c, 0xd4, 0x60, 0x0d, 0xdc, 0x60, 0x82, 0xd0, 0x0c, 0x1b, 0x80, 0x0d, 0x03, 0x11, 0x07, 0x71, 0xb0, 0x21, 0x90,
        0x83, 0x0d, 0xc3, 0x00, 0x07, 0x73, 0x30, 0x41, 0x18, 0x83, 0x32, 0xd8, 0x10, 0xd4, 0x01, 0x89, 0xb6, 0xb0, 0x34, 0x37, 0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e,
        0x13, 0x04, 0xa7, 0x9b, 0x20, 0x38, 0xde, 0x86, 0x80, 0x98, 0x20, 0x38, 0xdf, 0x04, 0xc1, 0xe1, 0x36, 0x2c, 0x04, 0x1e, 0xe4, 0x81, 0x1e, 0xec, 0x01, 0x1f, 0x0c, 0x7c, 0x40, 0xf4, 0x01, 0xc0,
        0xa5, 0xca, 0xea, 0x8b, 0xaa, 0x08, 0x6b, 0xe8, 0xe9, 0x49, 0x8a, 0x68, 0x82, 0xe0, 0x6c, 0x1b, 0x96, 0xe1, 0x0f, 0xf2, 0xa0, 0x0f, 0xf6, 0x00, 0x14, 0x06, 0x50, 0x18, 0xfa, 0x00, 0xd8, 0x20,
        0xf8, 0x41, 0x28, 0x30, 0x99, 0xb2, 0xfa, 0xa2, 0x0a, 0x93, 0x3b, 0x2b, 0xa3, 0x9b, 0x20, 0x38, 0xda, 0x86, 0x85, 0x18, 0x85, 0x3c, 0x20, 0x85, 0x3d, 0xe8, 0x83, 0x81, 0x0f, 0x88, 0x3e, 0x00,
        0x26, 0x08, 0x4e, 0xb6, 0x21, 0x18, 0x26, 0x08, 0x0e, 0xb6, 0x61, 0x19, 0x46, 0xc1, 0x14, 0x48, 0xe1, 0x14, 0xfa, 0x60, 0x40, 0x85, 0xa1, 0x0f, 0x80, 0x0d, 0x42, 0x29, 0xa4, 0xc2, 0x86, 0x41,
        0x14, 0x54, 0x01, 0x98, 0x20, 0x90, 0xc1, 0xb5, 0x41, 0x20, 0x58, 0x61, 0x43, 0x01, 0x07, 0x77, 0xb0, 0x0a, 0x5b, 0x2b, 0x54, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b,
        0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x81, 0x51, 0x87, 0x0c, 0xcf,
        0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x90, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x68, 0x95, 0xc8,
        0xf0, 0x5c, 0xe8, 0xf2, 0xe0, 0xca, 0x82, 0xdc, 0xdc, 0xde, 0xe8, 0xc2, 0xe8, 0xd2, 0xde, 0xdc, 0xe6, 0xa6, 0x08, 0x6e, 0x30, 0x07, 0x75, 0xc8, 0xf0, 0x5c, 0xec, 0xd2, 0xca, 0xee, 0x92, 0xc8,
        0xa6, 0xe8, 0xc2, 0xe8, 0xca, 0xa6, 0x04, 0x75, 0x50, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e, 0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0xd0, 0x0a, 0x00, 0x00, 0x00, 0x79, 0x18,
        0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71,
        0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03,
        0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19,
        0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66,
        0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68,
        0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79,
        0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca,
        0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3,
        0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79,
        0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0x46, 0x30,
        0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x07, 0xdb, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x01, 0x55, 0x14, 0x44, 0x54, 0x3a, 0xc0, 0x50, 0x12, 0x06,
        0x20, 0x60, 0x7e, 0x71, 0xdb, 0x15, 0x80, 0x06, 0x81, 0x1f, 0xbd, 0x8c, 0x07, 0x02, 0x67, 0xd6, 0x1f, 0x89, 0x5a, 0xc6, 0xd3, 0xeb, 0xf2, 0xb2, 0x8c, 0x08, 0xb4, 0xfe, 0x48, 0xf6, 0xf2, 0x98,
        0xfe, 0x96, 0x03, 0x9b, 0x24, 0xd8, 0x0c, 0x08, 0x04, 0x02, 0x83, 0x36, 0x40, 0x30, 0x00, 0xd2, 0x58, 0x02, 0x18, 0x0c, 0x97, 0xef, 0x3c, 0xbe, 0x70, 0x10, 0x82, 0xa2, 0x09, 0x02, 0x21, 0x49,
        0x1f, 0xb5, 0x2c, 0x98, 0x09, 0x3c, 0x0f, 0x31, 0xf8, 0xc8, 0x6d, 0x9b, 0x00, 0x01, 0x44, 0x86, 0x54, 0x00, 0x59, 0xe0, 0x37, 0x0f, 0xb7, 0xe1, 0x40, 0xe0, 0xac, 0x3a, 0x0d, 0xb7, 0xe1, 0xec,
        0xb2, 0x7c, 0x4a, 0x0f, 0xd3, 0xcb, 0x40, 0x60, 0xd0, 0x0a, 0xaa, 0xe1, 0xf2, 0x9d, 0xc7, 0x97, 0x26, 0x27, 0x22, 0x50, 0x6a, 0x7a, 0xa8, 0xc9, 0x2f, 0x6e, 0xdb, 0x0c, 0xaa, 0xe1, 0xf2, 0x9d,
        0xc7, 0x97, 0x26, 0x27, 0x22, 0x50, 0x6a, 0x7a, 0xa8, 0xc9, 0x47, 0x6e, 0xdb, 0x14, 0xa4, 0xe1, 0xf2, 0x9d, 0xc7, 0x17, 0x22, 0x02, 0x98, 0x88, 0x10, 0x68, 0x86, 0x85, 0x30, 0x84, 0x6b, 0xb8,
        0x7c, 0xe7, 0xf1, 0x27, 0xe2, 0x9a, 0xa8, 0x88, 0x00, 0x81, 0xe9, 0x20, 0x22, 0xbf, 0xb8, 0x6d, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00,
        0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x94, 0xdc, 0x0c, 0x40, 0xd9, 0x95, 0x64, 0x40, 0x39, 0x06, 0x50, 0x37, 0x02, 0x40, 0xc3, 0x0c, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60,
        0x00, 0x06, 0x64, 0x70, 0x48, 0x62, 0x20, 0x06, 0xc2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x80, 0x41, 0x19, 0x20, 0xd1, 0x18, 0x8c, 0xc1, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x60, 0x60,
        0x06, 0x49, 0x45, 0x06, 0x64, 0x40, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0xc1, 0xd3, 0x06, 0x64, 0x60, 0x06, 0x65, 0x50, 0x25, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xf0, 0xb8, 0x41, 0x19, 0x9c,
        0x81, 0x19, 0x68, 0xca, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x60, 0x6c, 0xa0, 0x0c, 0x67, 0x30, 0x9a, 0x10, 0x00, 0xa3, 0x09, 0x42, 0x70, 0x02, 0x40, 0x27, 0x00, 0x34, 0x62, 0xe0, 0x00, 0x20,
        0x08, 0x06, 0x1b, 0x1c, 0x38, 0x4a, 0x72, 0x18, 0xcb, 0x22, 0x04, 0x6c, 0x30, 0x9a, 0x10, 0x00, 0xa3, 0x09, 0x42, 0x30, 0x9a, 0x30, 0x08, 0xa3, 0x09, 0xc4, 0x30, 0x9a, 0x50, 0x10, 0x23, 0x06,
        0x06, 0x00, 0x82, 0x60, 0xe0, 0xd5, 0xc1, 0x14, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x41, 0xa4, 0x07, 0x69, 0x30, 0x07, 0x73, 0x30, 0x06, 0xc6, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x44, 0x7a,
        0x90, 0x06, 0x73, 0x30, 0x07, 0x67, 0x50, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x41, 0xa4, 0x07, 0x69, 0x30, 0x07, 0x73, 0x40, 0x06, 0xc4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x44, 0x7a, 0x90,
        0x06, 0x73, 0x30, 0x07, 0x61, 0x30, 0x5c, 0x00, 0xc4, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x4c, 0x7a, 0xa0, 0x06, 0x75, 0x40, 0x07, 0x64, 0x10, 0x60, 0x38, 0x10, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x00, 0x00, 0xf6, 0xc2, 0x48, 0x00, 0xf3, 0xfc, 0x12, 0xc0, 0x3c, 0x0b, 0x11, 0x99, 0x8b, 0x33, 0x11, 0xd7, 0x3f, 0x11, 0xd7, 0x44, 0x45, 0xc4, 0x6f, 0x0f, 0x06, 0xc3, 0x00, 0x91, 0x21, 0xfd,
        0x42, 0x40, 0x15, 0x05, 0x11, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    };
#if 0
    Texture2D<float4> tex : register(t0);

    cbuffer args : register(b0)
    {
        float lod;
    };

    void main(float4 pos : SV_POSITION, out float4 o0 : SV_TARGET0, out uint o1 : SV_TARGET1)
    {
        uint fb;
        o0 = tex.Load(int3(pos.xy, int(lod)), int2(0, 0), fb);
        o1 = CheckAccessFullyMapped(fb) ? 1 : 0;
    }
#endif
    static const DWORD ps_ld_dxbc[] =
    {
        0x43425844, 0xe165b596, 0xe7589c76, 0xe38a2020, 0x4ca87505, 0x00000001, 0x000001cc, 0x00000004,
        0x00000030, 0x00000064, 0x000000b0, 0x000001bc, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f,
        0x004e4f49, 0x4e47534f, 0x00000044, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000000,
        0x00000003, 0x00000000, 0x0000000f, 0x00000038, 0x00000001, 0x00000000, 0x00000001, 0x00000001,
        0x00000e01, 0x545f5653, 0x45475241, 0xabab0054, 0x58454853, 0x00000104, 0x00000050, 0x00000041,
        0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000001, 0x04001858, 0x00107000, 0x00000000,
        0x00005555, 0x04002064, 0x00101032, 0x00000000, 0x00000001, 0x03000065, 0x001020f2, 0x00000000,
        0x03000065, 0x00102012, 0x00000001, 0x02000068, 0x00000002, 0x0600001b, 0x001000c2, 0x00000000,
        0x00208006, 0x00000000, 0x00000000, 0x0500001b, 0x00100032, 0x00000000, 0x00101046, 0x00000000,
        0x8b0000df, 0x800000c2, 0x00155543, 0x001000f2, 0x00000000, 0x00100012, 0x00000001, 0x00100e46,
        0x00000000, 0x00107e46, 0x00000000, 0x05000036, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000,
        0x050000ea, 0x00100012, 0x00000000, 0x0010000a, 0x00000001, 0x09000037, 0x00102012, 0x00000001,
        0x0010000a, 0x00000000, 0x00004001, 0x00000001, 0x00004001, 0x00000000, 0x0100003e, 0x30494653,
        0x00000008, 0x00000100, 0x00000000, 
    };
    static const BYTE ps_ld_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xad, 0x5f, 0x86, 0xc6, 0xd4, 0x91, 0x0f, 0xeb, 0xf1, 0xe6, 0xe0, 0xf5, 0x1d, 0xa1, 0x6d, 0xf3, 0x01, 0x00, 0x00, 0x00, 0xee, 0x09, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0xda, 0x00, 0x00, 0x00, 0x8a, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x4f, 0x53, 0x47, 0x31, 0x52, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74, 0x00, 0x50, 0x53, 0x56, 0x30, 0xa8, 0x00,
        0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x02, 0x00, 0x01, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x41, 0x10, 0x01, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x1f, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x5c, 0x08, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x17, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01,
        0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x44, 0x08, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x0e, 0x02, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00,
        0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92,
        0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23,
        0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02,
        0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82,
        0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13,
        0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x88, 0xc1, 0x0c, 0xc0, 0x30, 0x02, 0x01, 0xcc, 0x11, 0x80, 0xc1, 0x4c, 0x6d, 0x30, 0x0e, 0xec, 0x10,
        0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x40, 0x0b, 0xe5, 0x80, 0x0f, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0x7c, 0x60, 0x0f, 0xe5, 0x30, 0x0e, 0xf4, 0xf0, 0x0e, 0xf2, 0xc0, 0x07, 0xe6,
        0xc0, 0x0e, 0xef, 0x10, 0x0e, 0xf4, 0xc0, 0x06, 0x60, 0x40, 0x07, 0x7e, 0x00, 0x06, 0x7e, 0xa0, 0x07, 0x7a, 0xd0, 0x0e, 0xe9, 0x00, 0x0f, 0xf3, 0xf0, 0x0b, 0xf4, 0x90, 0x0f, 0xf0, 0x50, 0x0e,
        0x28, 0x10, 0x66, 0x12, 0x83, 0x71, 0x60, 0x87, 0x70, 0x98, 0x87, 0x79, 0x70, 0x03, 0x5a, 0x28, 0x07, 0x7c, 0xa0, 0x87, 0x7a, 0x90, 0x87, 0x72, 0x90, 0x03, 0x52, 0xe0, 0x03, 0x7b, 0x28, 0x87,
        0x71, 0xa0, 0x87, 0x77, 0x90, 0x07, 0x3e, 0x30, 0x07, 0x76, 0x78, 0x87, 0x70, 0xa0, 0x07, 0x36, 0x00, 0x03, 0x3a, 0xf0, 0x03, 0x30, 0xf0, 0x03, 0x24, 0x04, 0x03, 0x11, 0x47, 0x00, 0x91, 0x21,
        0xa1, 0x00, 0x20, 0x63, 0x04, 0xa0, 0x04, 0x08, 0x25, 0x73, 0x04, 0x48, 0x31, 0x00, 0x41, 0x10, 0x16, 0x81, 0x98, 0x62, 0x20, 0x82, 0x20, 0x2c, 0x00, 0x39, 0xc5, 0x40, 0x04, 0x41, 0x58, 0x04,
        0x82, 0x8e, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0xb9, 0x8d, 0x2a, 0x56, 0x62, 0xf2, 0x8b, 0xdb, 0x46, 0x04, 0x00, 0x00, 0x00, 0x2d, 0xf7, 0x0c, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x21,
        0xd0, 0x0c, 0x0b, 0x81, 0x82, 0xa9, 0x10, 0x92, 0x40, 0x09, 0x54, 0xdd, 0x34, 0x5c, 0xfe, 0x84, 0x3d, 0x84, 0xe4, 0xaf, 0x84, 0xb4, 0x12, 0x93, 0x5f, 0xdc, 0x36, 0x2a, 0x00, 0x00, 0x00, 0x84,
        0xa2, 0x5c, 0x02, 0x25, 0x08, 0x82, 0x20, 0x08, 0x02, 0x61, 0x73, 0x04, 0x41, 0x19, 0x34, 0x41, 0xa0, 0xad, 0x18, 0x94, 0xb0, 0x08, 0x82, 0x46, 0xdd, 0x40, 0x40, 0x1a, 0x08, 0x73, 0x04, 0xa0,
        0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f,
        0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76,
        0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07,
        0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21, 0x23, 0x45, 0x44, 0x00, 0x72, 0x00, 0xc0, 0xf4, 0x00, 0x00, 0x86, 0x3c, 0x09, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x18, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x38, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x53, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x07, 0x03,
        0x02, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0xcf, 0x06, 0x04, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x9e, 0x0e, 0x08, 0x80, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xb2, 0x40, 0x13, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x12, 0x4a, 0x60, 0x04, 0xa0, 0x10, 0x8a,
        0xa1, 0x08, 0x4a, 0xa2, 0x50, 0xca, 0xa0, 0x1c, 0x0a, 0xa4, 0x14, 0x28, 0x2c, 0x40, 0x20, 0x5a, 0x4a, 0xa0, 0x14, 0x0a, 0x64, 0x04, 0xa0, 0x24, 0xca, 0xa0, 0x10, 0xe8, 0x98, 0x01, 0x20, 0x63,
        0x06, 0x80, 0x8a, 0x19, 0x00, 0x22, 0x66, 0x00, 0x08, 0x1c, 0xcb, 0x20, 0x8a, 0xef, 0x03, 0x00, 0x1a, 0x66, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x7b, 0x00, 0x00, 0x00, 0x1a, 0x03,
        0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x83, 0x0c, 0x6f, 0x0c, 0x24, 0xc6, 0x45, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x04, 0x65, 0x82, 0x20, 0x2c, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x08, 0xcc, 0x06, 0x61,
        0x30, 0x28, 0xc0, 0xcd, 0x4d, 0x10, 0x84, 0x66, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0xa1, 0xf8, 0x38, 0xd0, 0x95, 0xe1, 0x4d, 0x10, 0x04, 0x67, 0x82, 0x20, 0x3c, 0x1b, 0x04, 0xc2, 0xd9, 0x90, 0x10,
        0x0b, 0x43, 0x10, 0x43, 0x43, 0x3c, 0x1b, 0x02, 0x68, 0x82, 0x70, 0x74, 0x24, 0xc2, 0xe4, 0xce, 0xe6, 0x36, 0x20, 0x84, 0x34, 0x11, 0xc4, 0x60, 0x00, 0x1b, 0x02, 0x6a, 0x03, 0x11, 0x01, 0x15,
        0x30, 0x41, 0x20, 0xc0, 0x60, 0x82, 0x20, 0x40, 0x0c, 0xd0, 0x26, 0x08, 0x42, 0x34, 0x41, 0x10, 0xa4, 0x0d, 0x46, 0x92, 0x69, 0xc4, 0xe6, 0x90, 0x68, 0x4b, 0x83, 0x9b, 0x9b, 0x20, 0x08, 0xd3,
        0x06, 0x22, 0xe9, 0x34, 0x6f, 0xc3, 0x80, 0x71, 0xdf, 0x04, 0x61, 0x10, 0x03, 0x1a, 0x68, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x13, 0x04, 0x81, 0xda, 0x60, 0x24, 0x62, 0xa0, 0x11, 0xdb, 0x18, 0x6c,
        0x10, 0x0c, 0x32, 0x98, 0x20, 0x18, 0x1e, 0x07, 0xb6, 0x37, 0xb2, 0x0d, 0x46, 0x72, 0x06, 0x1a, 0xb1, 0x39, 0x1b, 0x04, 0x03, 0x0d, 0x36, 0x1c, 0xc4, 0x05, 0x06, 0x61, 0x50, 0x06, 0x66, 0x90,
        0x06, 0x13, 0x04, 0x45, 0xd8, 0x00, 0x6c, 0x18, 0x08, 0x36, 0x60, 0x83, 0x0d, 0x41, 0x1b, 0x6c, 0x18, 0x86, 0x35, 0x70, 0x83, 0x09, 0x02, 0x18, 0x84, 0xc1, 0x86, 0x00, 0x0e, 0x48, 0xb4, 0x85,
        0xa5, 0xb9, 0x71, 0x99, 0xb2, 0xfa, 0x82, 0x7a, 0x9b, 0x4b, 0xa3, 0x4b, 0x7b, 0x73, 0x9b, 0x20, 0x2c, 0xda, 0x04, 0x61, 0xd9, 0x36, 0x04, 0xc4, 0x04, 0x61, 0xe1, 0x26, 0x08, 0x4b, 0xb6, 0x61,
        0x21, 0xe6, 0x80, 0x0e, 0xea, 0xc0, 0x0e, 0xee, 0x60, 0xb8, 0x03, 0x02, 0x0f, 0x80, 0x0d, 0x41, 0x1e, 0x30, 0x99, 0xb2, 0xfa, 0xa2, 0x0a, 0x93, 0x3b, 0x2b, 0xa3, 0x9b, 0x20, 0x2c, 0xd8, 0x86,
        0x85, 0xd8, 0x03, 0x3a, 0xe0, 0x03, 0x3b, 0xc0, 0x83, 0xe1, 0x0e, 0x08, 0x3c, 0x00, 0x26, 0x08, 0xcb, 0xb5, 0x21, 0x18, 0x26, 0x08, 0x8b, 0xb5, 0x61, 0x19, 0xf6, 0xc0, 0x0f, 0xf8, 0xe0, 0x0f,
        0xf0, 0x60, 0x00, 0x85, 0x01, 0x0f, 0x80, 0x0d, 0x42, 0x1f, 0x84, 0xc2, 0x86, 0x41, 0x0f, 0x44, 0x01, 0x98, 0x20, 0x84, 0x41, 0xb5, 0x41, 0x20, 0x48, 0x61, 0x43, 0xb1, 0x06, 0x72, 0x30, 0x0a,
        0x56, 0x29, 0x54, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32,
        0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x81, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x90, 0x94, 0x21, 0xc3, 0x73,
        0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x54, 0x95, 0xc8, 0xf0, 0x5c, 0xe8, 0xf2, 0xe0, 0xca, 0x82, 0xdc, 0xdc, 0xde, 0xe8, 0xc2, 0xe8, 0xd2, 0xde, 0xdc, 0xe6, 0xa6,
        0x08, 0x69, 0xe0, 0x06, 0x75, 0xc8, 0xf0, 0x5c, 0xec, 0xd2, 0xca, 0xee, 0x92, 0xc8, 0xa6, 0xe8, 0xc2, 0xe8, 0xca, 0xa6, 0x04, 0x70, 0x50, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e, 0x0f,
        0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0x50, 0x0a, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38,
        0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c,
        0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70,
        0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10,
        0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0,
        0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77,
        0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4,
        0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43,
        0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74,
        0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00,
        0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x2f, 0x00, 0x00, 0x00, 0x36, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x07, 0xd5, 0x70, 0xf9, 0xce,
        0xe3, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0x51, 0x3a, 0xc0, 0xe0, 0x17, 0xb7, 0x6d, 0x06, 0xdb, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x01, 0x55, 0x14, 0x44, 0x54, 0x3a, 0xc0, 0x50, 0x12, 0x06, 0x20, 0x60,
        0x7e, 0x71, 0xdb, 0x05, 0x80, 0x06, 0x81, 0x1f, 0xbd, 0x8c, 0x07, 0x02, 0x67, 0xd6, 0x1f, 0x89, 0x5a, 0xc6, 0xd3, 0xeb, 0xf2, 0xb2, 0x8c, 0x08, 0xb4, 0xfe, 0x48, 0xf6, 0xf2, 0x98, 0xfe, 0x96,
        0x03, 0x9b, 0x24, 0xd8, 0x0c, 0x08, 0x04, 0x02, 0x83, 0x26, 0x40, 0x30, 0x00, 0xd2, 0x18, 0x02, 0x18, 0x0c, 0x97, 0xef, 0x3c, 0xbe, 0x70, 0x10, 0x82, 0xa2, 0x09, 0x02, 0x21, 0x49, 0x1f, 0xb5,
        0x2c, 0x98, 0x09, 0x3c, 0x0f, 0x31, 0xf8, 0xc8, 0x6d, 0x5b, 0x00, 0x01, 0x44, 0x86, 0x64, 0x04, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb7,
        0x6d, 0x05, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4, 0x23, 0xb7, 0x6d, 0x09, 0xd2, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x11, 0x01, 0x4c, 0x44, 0x08, 0x34,
        0xc3, 0x42, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x47, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x24, 0xcc, 0x00, 0x94, 0x5c, 0xd9,
        0x15, 0x61, 0x40, 0x39, 0x06, 0xd0, 0x52, 0x04, 0xa4, 0x8d, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xd0, 0x85, 0x81, 0x11, 0x7d, 0x5f, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08,
        0x06, 0x9d, 0x18, 0x1c, 0x03, 0x18, 0x80, 0x81, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x4c, 0x1a, 0x80, 0x41, 0x18, 0x84, 0xc1, 0x94, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xa3, 0x06,
        0x61, 0x20, 0x06, 0x62, 0x70, 0x29, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x50, 0xa1, 0x41, 0x32, 0x8c, 0xc1, 0x68, 0x42, 0x00, 0x5c, 0x20, 0xd0, 0x15, 0x02, 0x5d, 0x21, 0xd0, 0x88, 0x41, 0x03,
        0x80, 0x20, 0x18, 0x60, 0x6c, 0xd0, 0x24, 0x83, 0x10, 0x40, 0x68, 0x80, 0x06, 0xd0, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc1, 0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10, 0xc3, 0x68, 0x42, 0x41, 0x8c,
        0x18, 0x18, 0x00, 0x08, 0x82, 0xc1, 0x16, 0x07, 0x52, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x8e, 0x1d, 0x98, 0xc1, 0x1b, 0xbc, 0x41, 0x18, 0x18, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xe0,
        0xd8, 0x81, 0x19, 0xbc, 0xc1, 0x1b, 0x90, 0x41, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x8e, 0x1d, 0x98, 0xc1, 0x1b, 0xbc, 0x81, 0x44, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0x63, 0x07,
        0x66, 0xf0, 0x06, 0x6f, 0xf0, 0x0d, 0x17, 0x08, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x90, 0x1d, 0x9c, 0x41, 0x1c, 0xc0, 0x81, 0x18, 0x04, 0x18, 0x0e, 0x04, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x00, 0x00, 0x36, 0xb3, 0xb4, 0xc4, 0x35, 0x51, 0x11, 0x51, 0x3a, 0xc0, 0x60, 0x2a, 0xce, 0x44, 0x5c, 0xff, 0x44, 0x5c, 0x13, 0x15, 0x11, 0xbf, 0x3d, 0xd8, 0x0a, 0x03, 0x44, 0x86, 0xf4, 0x0b,
        0x01, 0x55, 0x14, 0x44, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    };
#if 0
    RWTexture2D<unorm float4> tex : register(u2);

    void main(float4 pos : SV_POSITION, out float4 o0 : SV_TARGET0, out uint o1 : SV_TARGET1)
    {
        uint fb;
        o0 = tex.Load(int2(pos.xy), fb);
        o1 = CheckAccessFullyMapped(fb) ? 1 : 0;
    }
#endif
    static const DWORD ps_ld_uav_dxbc[] =
    {
        0x43425844, 0x802cf0fb, 0x86d1616b, 0xff4c948c, 0x700640db, 0x00000001, 0x000001a4, 0x00000004,
        0x00000030, 0x00000064, 0x000000b0, 0x00000194, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x5449534f,
        0x004e4f49, 0x4e47534f, 0x00000044, 0x00000002, 0x00000008, 0x00000038, 0x00000000, 0x00000000,
        0x00000003, 0x00000000, 0x0000000f, 0x00000038, 0x00000001, 0x00000000, 0x00000001, 0x00000001,
        0x00000e01, 0x545f5653, 0x45475241, 0xabab0054, 0x58454853, 0x000000dc, 0x00000050, 0x00000037,
        0x0100086a, 0x0400189c, 0x0011e000, 0x00000002, 0x00001111, 0x04002064, 0x00101032, 0x00000000,
        0x00000001, 0x03000065, 0x001020f2, 0x00000000, 0x03000065, 0x00102012, 0x00000001, 0x02000068,
        0x00000002, 0x0500001b, 0x001000f2, 0x00000000, 0x00101546, 0x00000000, 0x8b0000e1, 0x800000c2,
        0x00044443, 0x001000f2, 0x00000000, 0x00100012, 0x00000001, 0x00100e46, 0x00000000, 0x0011ee46,
        0x00000002, 0x05000036, 0x001020f2, 0x00000000, 0x00100e46, 0x00000000, 0x050000ea, 0x00100012,
        0x00000000, 0x0010000a, 0x00000001, 0x09000037, 0x00102012, 0x00000001, 0x0010000a, 0x00000000,
        0x00004001, 0x00000001, 0x00004001, 0x00000000, 0x0100003e, 0x30494653, 0x00000008, 0x00000900,
        0x00000000, 
    };
    static const BYTE ps_ld_uav_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x2c, 0xdf, 0x91, 0xcf, 0xb6, 0x34, 0x1e, 0x7c, 0x91, 0xdc, 0x5d, 0xf3, 0x0d, 0x82, 0x04, 0xb4, 0x01, 0x00, 0x00, 0x00, 0xa2, 0x08, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0xda, 0x00, 0x00, 0x00, 0x7a, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x4f, 0x53, 0x47, 0x31, 0x52, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74, 0x00, 0x50, 0x53, 0x56, 0x30, 0x98, 0x00,
        0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x02, 0x00, 0x01, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x01, 0x01, 0x41, 0x10, 0x01, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x20, 0x07,
        0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xc8, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x08, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c,
        0x00, 0x00, 0xbf, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01,
        0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90,
        0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18,
        0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff,
        0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x30, 0x00,
        0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70,
        0x33, 0x00, 0xc3, 0x08, 0x04, 0x30, 0x13, 0x19, 0x8c, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xc8, 0xc2, 0x2d, 0xd0, 0x42, 0x39, 0xe0, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x83, 0x1c,
        0x90, 0x02, 0x1f, 0xd8, 0x43, 0x39, 0x8c, 0x03, 0x3d, 0xbc, 0x83, 0x3c, 0xf0, 0x81, 0x39, 0xb0, 0xc3, 0x3b, 0x84, 0x03, 0x3d, 0xb0, 0x01, 0x18, 0xd0, 0x81, 0x1f, 0x80, 0x81, 0x1f, 0xa0, 0x20,
        0x90, 0x18, 0x01, 0x28, 0x01, 0xa1, 0x32, 0x47, 0x00, 0x06, 0x73, 0x04, 0x48, 0x31, 0xc0, 0x39, 0x07, 0x1d, 0x4a, 0xc5, 0x20, 0xe7, 0x1c, 0x04, 0x68, 0x15, 0x83, 0x9c, 0x73, 0xd0, 0xa1, 0x76,
        0xd3, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0xd2, 0x4a, 0x4c, 0x7e, 0x71, 0xdb, 0xa8, 0x00, 0x00, 0x80, 0x43, 0xe8, 0x9e, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x3f, 0x04, 0x9a, 0x61,
        0x21, 0x50, 0x00, 0x8b, 0xf2, 0x4e, 0x3c, 0xe7, 0x9c, 0x73, 0x0e, 0xc9, 0x39, 0x82, 0xa0, 0x0c, 0xf4, 0x1c, 0xaa, 0xc5, 0x88, 0x07, 0x9d, 0x43, 0xe9, 0x0e, 0x04, 0xa4, 0xc1, 0x99, 0x23, 0x00,
        0x05, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f,
        0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76,
        0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07,
        0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x44, 0x90, 0x21, 0x23, 0x45, 0x44, 0x00, 0x3a, 0x00, 0xe0, 0x21, 0x4f, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x9e, 0x04, 0x08, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x0b, 0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c,
        0x79, 0x1a, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x48, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0xa9, 0x80, 0x00, 0x18, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x73, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x11, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11,
        0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0xda, 0x05, 0x08, 0x1c, 0x40, 0x74, 0x04, 0x80, 0x4e, 0x09, 0x8c, 0x00, 0x14, 0x42, 0x31, 0x14, 0x41, 0x49, 0x14, 0x48, 0x19, 0x94, 0x03,
        0xa1, 0x92, 0x28, 0x83, 0x42, 0x18, 0x01, 0x28, 0x90, 0x52, 0x28, 0x01, 0x1a, 0x33, 0x00, 0xa4, 0xc7, 0x32, 0x88, 0xe2, 0xfb, 0x00, 0x80, 0xc4, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18,
        0x00, 0x00, 0x65, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x83, 0x0c, 0x6f, 0x0c, 0x24, 0xc6, 0x45, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x0e, 0x65, 0x82, 0x70, 0x2c, 0x1b,
        0x84, 0x81, 0x98, 0x20, 0x1c, 0xcc, 0x06, 0x61, 0x30, 0x28, 0xc0, 0xcd, 0x4d, 0x10, 0x8e, 0x66, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x61, 0xd0, 0x38, 0xd0, 0x95, 0xe1, 0x4d, 0x10, 0x0e, 0x67, 0x82,
        0x40, 0x25, 0x13, 0x84, 0xe3, 0xd9, 0x20, 0x10, 0xcf, 0x86, 0x85, 0x58, 0x18, 0xa2, 0x19, 0x1a, 0xc7, 0x71, 0xa0, 0x0d, 0x41, 0xb4, 0x81, 0x00, 0x24, 0x00, 0x98, 0x20, 0x08, 0xdc, 0x04, 0xe1,
        0x80, 0x18, 0xa0, 0x4d, 0x10, 0x8e, 0x68, 0x82, 0x70, 0x48, 0x1b, 0x8c, 0xc4, 0xba, 0x08, 0xec, 0xd9, 0x20, 0x54, 0xd9, 0x86, 0x81, 0xa0, 0xb4, 0x09, 0x82, 0x11, 0x6c, 0x00, 0x36, 0x0c, 0x44,
        0xd7, 0x6d, 0x08, 0xbc, 0x0d, 0xc3, 0xc0, 0x7d, 0x13, 0x04, 0x6d, 0xdb, 0x10, 0x84, 0x01, 0x89, 0xb6, 0xb0, 0x34, 0x37, 0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e,
        0x13, 0x04, 0x64, 0x9a, 0x20, 0x20, 0xd4, 0x86, 0x80, 0x98, 0x20, 0x20, 0xd5, 0x04, 0x01, 0xb1, 0x36, 0x2c, 0x04, 0x19, 0x94, 0x81, 0x19, 0x9c, 0x01, 0x1a, 0x0c, 0x68, 0x40, 0xa4, 0x01, 0xb0,
        0x21, 0x50, 0x03, 0x26, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74, 0x13, 0x04, 0xe4, 0xda, 0xb0, 0x10, 0x6c, 0x50, 0x06, 0x6d, 0x70, 0x06, 0x69, 0x30, 0xa0, 0x01, 0x91, 0x06, 0xc0,
        0x04, 0x01, 0xc1, 0x36, 0x04, 0xc3, 0x04, 0x01, 0xc9, 0x36, 0x2c, 0x03, 0x1b, 0xbc, 0x41, 0x1b, 0xc0, 0x41, 0x1a, 0x0c, 0x71, 0x30, 0xa4, 0x01, 0xb0, 0x41, 0x70, 0x03, 0x39, 0xd8, 0x30, 0xac,
        0xc1, 0x1c, 0x00, 0x13, 0x84, 0x0d, 0xd9, 0x20, 0x10, 0x75, 0xb0, 0xa1, 0xe0, 0xc6, 0x80, 0x0e, 0x26, 0x3b, 0xa8, 0xc2, 0xc6, 0x66, 0xd7, 0xe6, 0x92, 0x46, 0x56, 0xe6, 0x46, 0x37, 0x25, 0x08,
        0xaa, 0x90, 0xe1, 0xb9, 0xd8, 0x95, 0xc9, 0xcd, 0xa5, 0xbd, 0xb9, 0x4d, 0x09, 0x88, 0x26, 0x64, 0x78, 0x2e, 0x76, 0x61, 0x6c, 0x76, 0x65, 0x72, 0x53, 0x02, 0xa3, 0x0e, 0x19, 0x9e, 0xcb, 0x1c,
        0x5a, 0x18, 0x59, 0x99, 0x5c, 0xd3, 0x1b, 0x59, 0x19, 0xdb, 0x94, 0x20, 0x29, 0x43, 0x86, 0xe7, 0x22, 0x57, 0x36, 0xf7, 0x56, 0x27, 0x37, 0x56, 0x36, 0x37, 0x25, 0x90, 0x2a, 0x91, 0xe1, 0xb9,
        0xd0, 0xe5, 0xc1, 0x95, 0x05, 0xb9, 0xb9, 0xbd, 0xd1, 0x85, 0xd1, 0xa5, 0xbd, 0xb9, 0xcd, 0x4d, 0x11, 0xb4, 0xaf, 0x0e, 0x19, 0x9e, 0x8b, 0x5d, 0x5a, 0xd9, 0x5d, 0x12, 0xd9, 0x14, 0x5d, 0x18,
        0x5d, 0xd9, 0x94, 0x20, 0x0c, 0xea, 0x90, 0xe1, 0xb9, 0x94, 0xb9, 0xd1, 0xc9, 0xe5, 0x41, 0xbd, 0xa5, 0xb9, 0xd1, 0xcd, 0x4d, 0x09, 0xec, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00,
        0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f,
        0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d,
        0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec,
        0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc,
        0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87,
        0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee,
        0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81,
        0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c,
        0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70,
        0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x26, 0x30, 0x0d, 0x97, 0xef, 0x3c,
        0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x04, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4, 0x23, 0xb7, 0x6d, 0x05, 0xd5, 0x70,
        0xf9, 0xce, 0xe3, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0x51, 0x3a, 0xc0, 0xe0, 0x17, 0xb7, 0x6d, 0x01, 0x04, 0x03, 0x20, 0x8d, 0x19, 0x80, 0xc1, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x07, 0x21, 0x28, 0x9a,
        0x20, 0x10, 0x92, 0xf4, 0x51, 0xcb, 0x82, 0x99, 0xc0, 0xf3, 0x10, 0x83, 0x8f, 0xdc, 0xb6, 0x1d, 0x48, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x44, 0x04, 0x30, 0x11, 0x21, 0xd0, 0x0c, 0x0b, 0x61, 0x03,
        0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb7, 0x5d, 0x00, 0x6a, 0x10, 0xf8, 0xd1, 0xcb, 0x78, 0x20, 0x70, 0x66, 0xfd, 0x91, 0xa4, 0x57, 0x6a,
        0x19, 0x4f, 0xaf, 0xcb, 0xcb, 0x32, 0x22, 0xd0, 0xfa, 0x23, 0xd9, 0xcb, 0x63, 0xfa, 0x5b, 0x0e, 0x6c, 0x92, 0x60, 0x33, 0x20, 0x10, 0x08, 0x0c, 0x02, 0x00, 0x61, 0x20, 0x00, 0x00, 0x37, 0x00,
        0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x74, 0x66, 0x00, 0x4a, 0xa1, 0xe4, 0x8a, 0x30, 0xa0, 0x1c, 0x03, 0x08, 0x15, 0x01, 0x00, 0x00, 0x23, 0x06,
        0x09, 0x00, 0x82, 0x60, 0x70, 0x71, 0x84, 0x82, 0x55, 0xda, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x24, 0x62, 0x80, 0x65, 0x99, 0x73, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x41, 0x32, 0x06, 0x99,
        0xa6, 0x31, 0xc8, 0x09, 0x87, 0x4e, 0x38, 0x34, 0x62, 0xd0, 0x00, 0x20, 0x08, 0x06, 0xd2, 0x18, 0x20, 0xc5, 0x22, 0x04, 0xcb, 0xb2, 0x2c, 0xa3, 0x09, 0x01, 0x30, 0x9a, 0x20, 0x04, 0xa3, 0x09,
        0x83, 0x30, 0x9a, 0x40, 0x0c, 0xa3, 0x09, 0x05, 0x31, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x15, 0x1a, 0x34, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x2c, 0x6c, 0x10, 0x91, 0x01, 0x19, 0x64,
        0xc6, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x2c, 0x6c, 0x10, 0x91, 0x01, 0x19, 0x58, 0xc5, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x2c, 0x6c, 0x10, 0x91, 0x01, 0x19, 0x34, 0xc4, 0x88, 0x41, 0x02,
        0x80, 0x20, 0x18, 0x2c, 0x6c, 0x10, 0x91, 0x01, 0x19, 0x6c, 0xc3, 0x05, 0x27, 0x46, 0x0c, 0x12, 0x00, 0x04, 0xc1, 0xa0, 0x61, 0x03, 0xc9, 0x0c, 0xca, 0x40, 0x0b, 0x30, 0x1c, 0x08, 0x06, 0x00,
        0x00, 0x00, 0x36, 0xa2, 0x4c, 0xc4, 0xf5, 0xbb, 0xda, 0xfb, 0xdb, 0x83, 0xa1, 0x2c, 0x2d, 0x71, 0x4d, 0x54, 0x44, 0x94, 0x0e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 
    };

    struct shader_args
    {
        float args[4];
    };

    static const struct
    {
        D3D12_SHADER_BYTECODE ps_dxbc;
        D3D12_SHADER_BYTECODE ps_dxil;
        uint32_t expected_mip;
        struct shader_args args;
        bool is_gather;
    }
    tests[] =
    {
        { SHADER_BYTECODE(ps_sample_dxbc),        SHADER_BYTECODE(ps_sample_dxil),        0, {{0.0f}} },
        { SHADER_BYTECODE(ps_sample_dxbc),        SHADER_BYTECODE(ps_sample_dxil),        1, {{1.0f}} },
        { SHADER_BYTECODE(ps_sample_bias_dxbc),   SHADER_BYTECODE(ps_sample_bias_dxil),   0, {{0.0f, 0.0f}} },
        { SHADER_BYTECODE(ps_sample_bias_dxbc),   SHADER_BYTECODE(ps_sample_bias_dxil),   1, {{1.0f, 0.0f}} },
        { SHADER_BYTECODE(ps_sample_bias_dxbc),   SHADER_BYTECODE(ps_sample_bias_dxil),   1, {{0.0f, 1.0f}} },
        { SHADER_BYTECODE(ps_sample_grad_dxbc),   SHADER_BYTECODE(ps_sample_grad_dxil),   0, {{0.0f, 0.0f, 0.0f}} },
        { SHADER_BYTECODE(ps_sample_grad_dxbc),   SHADER_BYTECODE(ps_sample_grad_dxil),   1, {{0.0f, 1.0f, 1.0f}} },
        { SHADER_BYTECODE(ps_sample_lod_dxbc),    SHADER_BYTECODE(ps_sample_lod_dxil),    0, {{0.0f}} },
        { SHADER_BYTECODE(ps_sample_lod_dxbc),    SHADER_BYTECODE(ps_sample_lod_dxil),    1, {{1.0f}} },
        { SHADER_BYTECODE(ps_gather_dxbc),        SHADER_BYTECODE(ps_gather_dxil),        0, {{0.0f}}, true },
        { SHADER_BYTECODE(ps_gather_po_dxbc),     SHADER_BYTECODE(ps_gather_po_dxil),     0, {{0.0f, 0.0f}}, true },
        { SHADER_BYTECODE(ps_ld_dxbc),            SHADER_BYTECODE(ps_ld_dxil),            0, {{0.0f}} },
        { SHADER_BYTECODE(ps_ld_uav_dxbc),        SHADER_BYTECODE(ps_ld_uav_dxil),        0, {{0.0f}} },
    };

    static const FLOAT clear_colors[2][4] = {
        { 0.2f, 0.4f, 0.6f, 0.8f },
        { 0.5f, 0.5f, 0.5f, 0.5f },
    };

    memset(&desc, 0, sizeof(desc));
    desc.no_render_target = true;
    if (!init_test_context(&context, &desc))
        return;

    if (use_dxil && !context_supports_dxil(&context))
    {
        destroy_test_context(&context);
        return;
    }

    hr = ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_D3D12_OPTIONS, &options, sizeof(options));
    ok(hr == S_OK, "Failed to check feature support, hr %#x.\n", hr);

    if (options.TiledResourcesTier < D3D12_TILED_RESOURCES_TIER_2)
    {
        skip("Tiled resources Tier 2 not supported by device.\n");
        destroy_test_context(&context);
        return;
    }

    descriptor_ranges[0].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_SRV;
    descriptor_ranges[0].NumDescriptors = 1;
    descriptor_ranges[0].BaseShaderRegister = 0;
    descriptor_ranges[0].RegisterSpace = 0;
    descriptor_ranges[0].OffsetInDescriptorsFromTableStart = 0;
    descriptor_ranges[1].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_UAV;
    descriptor_ranges[1].NumDescriptors = 1;
    descriptor_ranges[1].BaseShaderRegister = 2;
    descriptor_ranges[1].RegisterSpace = 0;
    descriptor_ranges[1].OffsetInDescriptorsFromTableStart = 1;
    descriptor_ranges[2].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_SAMPLER;
    descriptor_ranges[2].NumDescriptors = 1;
    descriptor_ranges[2].BaseShaderRegister = 0;
    descriptor_ranges[2].RegisterSpace = 0;
    descriptor_ranges[2].OffsetInDescriptorsFromTableStart = 0;
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[0].DescriptorTable.NumDescriptorRanges = 2;
    root_parameters[0].DescriptorTable.pDescriptorRanges = &descriptor_ranges[0];
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_PIXEL;
    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[1].DescriptorTable.NumDescriptorRanges = 1;
    root_parameters[1].DescriptorTable.pDescriptorRanges = &descriptor_ranges[2];
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_PIXEL;
    root_parameters[2].ParameterType = D3D12_ROOT_PARAMETER_TYPE_32BIT_CONSTANTS;
    root_parameters[2].Constants.Num32BitValues = sizeof(struct shader_args) / sizeof(uint32_t);
    root_parameters[2].Constants.ShaderRegister = 0;
    root_parameters[2].Constants.RegisterSpace = 0;
    root_parameters[2].ShaderVisibility = D3D12_SHADER_VISIBILITY_PIXEL;
    root_signature_desc.NumParameters = ARRAY_SIZE(root_parameters);
    root_signature_desc.pParameters = root_parameters;
    root_signature_desc.NumStaticSamplers = 0;
    root_signature_desc.pStaticSamplers = NULL;
    root_signature_desc.Flags = D3D12_ROOT_SIGNATURE_FLAG_NONE;
    hr = create_root_signature(context.device, &root_signature_desc, &root_signature);
    ok(hr == S_OK, "Failed to create root signature, hr %#x.\n", hr);

    memset(&heap_desc, 0, sizeof(heap_desc));
    heap_desc.Properties.Type = D3D12_HEAP_TYPE_DEFAULT;
    heap_desc.SizeInBytes = TILE_SIZE * 3;

    resource_desc.Dimension = D3D12_RESOURCE_DIMENSION_TEXTURE2D;
    resource_desc.Alignment = 0;
    resource_desc.Width = 256;
    resource_desc.Height = 256;
    resource_desc.DepthOrArraySize = 1;
    resource_desc.MipLevels = 1;
    resource_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    resource_desc.SampleDesc.Count = 1;
    resource_desc.SampleDesc.Quality = 0;
    resource_desc.Layout = D3D12_TEXTURE_LAYOUT_UNKNOWN;
    resource_desc.Flags = D3D12_RESOURCE_FLAG_ALLOW_RENDER_TARGET;
    hr = ID3D12Device_CreateCommittedResource(context.device, &heap_desc.Properties, D3D12_HEAP_FLAG_NONE,
            &resource_desc, D3D12_RESOURCE_STATE_RENDER_TARGET, NULL, &IID_ID3D12Resource, (void **)&color_rt);
    ok(hr == S_OK, "Failed to create committed resource, hr %#x.\n", hr);

    resource_desc.Format = DXGI_FORMAT_R32_UINT;
    hr = ID3D12Device_CreateCommittedResource(context.device, &heap_desc.Properties, D3D12_HEAP_FLAG_NONE,
            &resource_desc, D3D12_RESOURCE_STATE_RENDER_TARGET, NULL, &IID_ID3D12Resource, (void **)&residency_rt);
    ok(hr == S_OK, "Failed to create committed resource, hr %#x.\n", hr);

    resource_desc.MipLevels = 2;
    resource_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    resource_desc.Layout = D3D12_TEXTURE_LAYOUT_64KB_UNDEFINED_SWIZZLE;
    resource_desc.Flags = D3D12_RESOURCE_FLAG_ALLOW_RENDER_TARGET | D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS;
    hr = ID3D12Device_CreateReservedResource(context.device, &resource_desc,
            D3D12_RESOURCE_STATE_RENDER_TARGET, NULL, &IID_ID3D12Resource, (void **)&tiled_image);
    ok(hr == S_OK, "Failed to create reserved resource, hr %#x.\n", hr);

    hr = ID3D12Device_CreateHeap(context.device, &heap_desc, &IID_ID3D12Heap, (void **)&heap);
    ok(hr == S_OK, "Failed to create heap, hr %#x.\n", hr);

    /* Map top-left + bottom-right of mip 0 + entire mip 1 */
    set_region_offset(&tile_regions[0], 0, 0, 0, 0);
    set_region_offset(&tile_regions[1], 1, 1, 0, 0);
    set_region_offset(&tile_regions[2], 0, 0, 0, 1);
    tile_offset = 0;

    ID3D12CommandQueue_UpdateTileMappings(context.queue, tiled_image,
            ARRAY_SIZE(tile_regions), tile_regions, NULL, heap, 1, NULL,
            &tile_offset, NULL, D3D12_TILE_MAPPING_FLAG_NONE);

    rtv_heap = create_cpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_RTV, 4);
    gpu_heap = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 2);
    sampler_heap = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_SAMPLER, 1);

    rtv_desc.Format = DXGI_FORMAT_UNKNOWN;
    rtv_desc.ViewDimension = D3D12_RTV_DIMENSION_TEXTURE2D;
    rtv_desc.Texture2D.MipSlice = 0;
    rtv_desc.Texture2D.PlaneSlice = 0;

    ID3D12Device_CreateRenderTargetView(context.device, color_rt, &rtv_desc, get_cpu_rtv_handle(&context, rtv_heap, 0));
    ID3D12Device_CreateRenderTargetView(context.device, residency_rt, &rtv_desc, get_cpu_rtv_handle(&context, rtv_heap, 1));
    ID3D12Device_CreateRenderTargetView(context.device, tiled_image, &rtv_desc, get_cpu_rtv_handle(&context, rtv_heap, 2));

    rtv_desc.Texture2D.MipSlice = 1;
    ID3D12Device_CreateRenderTargetView(context.device, tiled_image, &rtv_desc, get_cpu_rtv_handle(&context, rtv_heap, 3));

    ID3D12GraphicsCommandList_ClearRenderTargetView(context.list, get_cpu_rtv_handle(&context, rtv_heap, 2), clear_colors[0], 0, NULL);
    ID3D12GraphicsCommandList_ClearRenderTargetView(context.list, get_cpu_rtv_handle(&context, rtv_heap, 3), clear_colors[1], 0, NULL);
    transition_resource_state(context.list, tiled_image, D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_PIXEL_SHADER_RESOURCE);

    srv_desc.ViewDimension = D3D12_SRV_DIMENSION_TEXTURE2D;
    srv_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    srv_desc.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;
    srv_desc.Texture2D.MostDetailedMip = 0;
    srv_desc.Texture2D.MipLevels = ~0u;
    srv_desc.Texture2D.PlaneSlice = 0;
    srv_desc.Texture2D.ResourceMinLODClamp = 0;
    ID3D12Device_CreateShaderResourceView(context.device, tiled_image, &srv_desc, get_cpu_descriptor_handle(&context, gpu_heap, 0));

    uav_desc.ViewDimension = D3D12_UAV_DIMENSION_TEXTURE2D;
    uav_desc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    uav_desc.Texture2D.MipSlice = 0;
    uav_desc.Texture2D.PlaneSlice = 0;
    ID3D12Device_CreateUnorderedAccessView(context.device, tiled_image, NULL, &uav_desc, get_cpu_descriptor_handle(&context, gpu_heap, 1));

    memset(&sampler_desc, 0, sizeof(sampler_desc));
    sampler_desc.Filter = D3D12_FILTER_MIN_MAG_MIP_POINT;
    sampler_desc.AddressU = D3D12_TEXTURE_ADDRESS_MODE_CLAMP;
    sampler_desc.AddressV = D3D12_TEXTURE_ADDRESS_MODE_CLAMP;
    sampler_desc.AddressW = D3D12_TEXTURE_ADDRESS_MODE_CLAMP;
    sampler_desc.MipLODBias = 0.0f;
    sampler_desc.MaxAnisotropy = 1;
    sampler_desc.MinLOD = -16.0f;
    sampler_desc.MaxLOD = 16.0f;
    ID3D12Device_CreateSampler(context.device, &sampler_desc, get_cpu_sampler_handle(&context, sampler_heap, 0));

    memset(&pipeline_desc, 0, sizeof(pipeline_desc));
    pipeline_desc.pRootSignature = root_signature;
    pipeline_desc.VS.BytecodeLength = use_dxil ? sizeof(vs_code_dxil) : sizeof(vs_code_dxbc);
    pipeline_desc.VS.pShaderBytecode = use_dxil ? (const void*)vs_code_dxil : (const void*)vs_code_dxbc;
    pipeline_desc.BlendState.RenderTarget[0].RenderTargetWriteMask = D3D12_COLOR_WRITE_ENABLE_ALL;
    pipeline_desc.BlendState.RenderTarget[1].RenderTargetWriteMask = D3D12_COLOR_WRITE_ENABLE_ALL;
    pipeline_desc.SampleMask = 0xFFFFFFFF;
    pipeline_desc.RasterizerState.FillMode = D3D12_FILL_MODE_SOLID;
    pipeline_desc.RasterizerState.CullMode = D3D12_CULL_MODE_NONE;
    pipeline_desc.PrimitiveTopologyType = D3D12_PRIMITIVE_TOPOLOGY_TYPE_TRIANGLE;
    pipeline_desc.NumRenderTargets = 2;
    pipeline_desc.RTVFormats[0] = DXGI_FORMAT_R8G8B8A8_UNORM;
    pipeline_desc.RTVFormats[1] = DXGI_FORMAT_R32_UINT;
    pipeline_desc.SampleDesc.Count = 1;

    set_viewport(&viewport, 0.0f, 0.0f, 256.0f, 256.0f, 0.0f, 1.0f);
    set_rect(&scissor, 0, 0, 256, 256);

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        D3D12_CPU_DESCRIPTOR_HANDLE rt_handle = get_cpu_rtv_handle(&context, rtv_heap, 0);
        ID3D12DescriptorHeap *descriptor_heaps[2] = { gpu_heap, sampler_heap };
        const FLOAT clear_residency_rt[] = { 100.0f, 0.0f, 0.0f, 0.0f };
        const FLOAT clear_color_rt[] = { 1.0f, 1.0f, 1.0f, 1.0f };
        unsigned int color, expected_a, expected_b;
        vkd3d_test_set_context("Test %u", i);

        expected_a = tests[i].expected_mip ? 0x80808080 : (tests[i].is_gather ? 0x33333333 : 0xcc996633);
        expected_b = tests[i].expected_mip ? expected_a : 0;
        
        pipeline_desc.PS = use_dxil ? tests[i].ps_dxil : tests[i].ps_dxbc;

        hr = ID3D12Device_CreateGraphicsPipelineState(context.device, &pipeline_desc, &IID_ID3D12PipelineState, (void **)&pipeline_state);
        ok(hr == S_OK, "Failed to compile graphics pipeline, hr %#x.\n", hr);

        ID3D12GraphicsCommandList_OMSetRenderTargets(context.list, 2, &rt_handle, true, NULL);
        ID3D12GraphicsCommandList_ClearRenderTargetView(context.list, get_cpu_rtv_handle(&context, rtv_heap, 1), clear_residency_rt, 0, NULL);
        ID3D12GraphicsCommandList_ClearRenderTargetView(context.list, get_cpu_rtv_handle(&context, rtv_heap, 0), clear_color_rt, 0, NULL);
        ID3D12GraphicsCommandList_IASetPrimitiveTopology(context.list, D3D_PRIMITIVE_TOPOLOGY_TRIANGLELIST);
        ID3D12GraphicsCommandList_RSSetViewports(context.list, 1, &viewport);
        ID3D12GraphicsCommandList_RSSetScissorRects(context.list, 1, &scissor);
        ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, ARRAY_SIZE(descriptor_heaps), descriptor_heaps);
        ID3D12GraphicsCommandList_SetGraphicsRootSignature(context.list, root_signature);
        ID3D12GraphicsCommandList_SetPipelineState(context.list, pipeline_state);
        ID3D12GraphicsCommandList_SetGraphicsRootDescriptorTable(context.list, 0, get_gpu_descriptor_handle(&context, gpu_heap, 0));
        ID3D12GraphicsCommandList_SetGraphicsRootDescriptorTable(context.list, 1, get_gpu_sampler_handle(&context, sampler_heap, 0));
        ID3D12GraphicsCommandList_SetGraphicsRoot32BitConstants(context.list, 2, sizeof(tests[i].args) / sizeof(uint32_t), tests[i].args.args, 0);
        ID3D12GraphicsCommandList_DrawInstanced(context.list, 3, 1, 0, 0);

        transition_resource_state(context.list, color_rt, D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_COPY_SOURCE);
        transition_resource_state(context.list, residency_rt, D3D12_RESOURCE_STATE_RENDER_TARGET, D3D12_RESOURCE_STATE_COPY_SOURCE);

        get_texture_readback_with_command_list(color_rt, 0, &rb, context.queue, context.list);

        color = get_readback_uint(&rb, 64, 64, 0);
        ok(compare_color(color, expected_a, 1), "Got color 0x%08x, expected %08x.\n", color, expected_a);
        color = get_readback_uint(&rb, 192, 64, 0);
        ok(compare_color(color, expected_b, 1), "Got color 0x%08x, expected %08x.\n", color, expected_b);
        color = get_readback_uint(&rb, 64, 192, 0);
        ok(compare_color(color, expected_b, 1), "Got color 0x%08x, expected %08x.\n", color, expected_b);
        color = get_readback_uint(&rb, 192, 192, 0);
        ok(compare_color(color, expected_a, 1), "Got color 0x%08x, expected %08x.\n", color, expected_a);

        release_resource_readback(&rb);
        reset_command_list(context.list, context.allocator);

        get_texture_readback_with_command_list(residency_rt, 0, &rb, context.queue, context.list);

        color = get_readback_uint(&rb, 64, 64, 0);
        ok(compare_color(color, !!expected_a, 0), "Got residency %#x, expected %#x.\n", color, !!expected_a);
        color = get_readback_uint(&rb, 192, 64, 0);
        ok(compare_color(color, !!expected_b, 0), "Got residency %#x, expected %#x.\n", color, !!expected_b);
        color = get_readback_uint(&rb, 64, 192, 0);
        ok(compare_color(color, !!expected_b, 0), "Got residency %#x, expected %#x.\n", color, !!expected_b);
        color = get_readback_uint(&rb, 192, 192, 0);
        ok(compare_color(color, !!expected_a, 0), "Got residency %#x, expected %#x.\n", color, !!expected_a);

        release_resource_readback(&rb);
        reset_command_list(context.list, context.allocator);

        transition_resource_state(context.list, color_rt, D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_RENDER_TARGET);
        transition_resource_state(context.list, residency_rt, D3D12_RESOURCE_STATE_COPY_SOURCE, D3D12_RESOURCE_STATE_RENDER_TARGET);

        ID3D12PipelineState_Release(pipeline_state);
    }

    ID3D12DescriptorHeap_Release(rtv_heap);
    ID3D12DescriptorHeap_Release(gpu_heap);
    ID3D12DescriptorHeap_Release(sampler_heap);
    ID3D12Heap_Release(heap);
    ID3D12Resource_Release(tiled_image);
    ID3D12Resource_Release(color_rt);
    ID3D12Resource_Release(residency_rt);
    ID3D12RootSignature_Release(root_signature);
    destroy_test_context(&context);
    #undef TILE_SIZE
}

void test_texture_feedback_instructions_sm51(void)
{
    test_texture_feedback_instructions(false);
}

void test_texture_feedback_instructions_dxil(void)
{
    test_texture_feedback_instructions(true);
}

void test_sparse_buffer_memory_lifetime(void)
{
    /* Attempt to bind sparse memory, then free the underlying heap, but keep the sparse resource
     * alive. This should confuse drivers that attempt to track BO lifetimes. */
    D3D12_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    D3D12_SHADER_RESOURCE_VIEW_DESC srv_desc;
    D3D12_FEATURE_DATA_D3D12_OPTIONS options;
    const UINT values[] = { 42, 42, 42, 42 };
    D3D12_ROOT_PARAMETER root_parameters[2];
    D3D12_TILE_REGION_SIZE region_size;
    D3D12_GPU_DESCRIPTOR_HANDLE h_gpu;
    D3D12_CPU_DESCRIPTOR_HANDLE h_cpu;
    D3D12_ROOT_SIGNATURE_DESC rs_desc;
    D3D12_DESCRIPTOR_RANGE desc_range;
    struct test_context context;
    struct resource_readback rb;
    ID3D12DescriptorHeap *cpu;
    ID3D12DescriptorHeap *gpu;
    D3D12_HEAP_DESC heap_desc;
    D3D12_RESOURCE_DESC desc;
    ID3D12Resource *sparse;
    ID3D12Resource *buffer;
    ID3D12Heap *heap_live;
    ID3D12Heap *heap;
    unsigned int i;
    HRESULT hr;

    static const DWORD cs_sparse_query_dxbc[] =
    {
#if 0
    RWStructuredBuffer<uint> RWBuf : register(u0);
    Buffer<uint> Buf : register(t0);

    [numthreads(1, 1, 1)]
    void main(uint thr : SV_DispatchThreadID)
    {
            uint code;

            // Sample mapped, but freed memory. See what CheckAccessFullyMapped returns.
            uint data = Buf.Load(thr, code);
            uint value = CheckAccessFullyMapped(code) ? (1u << 16) : 0u;
            value |= data & 0xffffu;
            RWBuf[2 * thr + 0] = value;

            // Sample not yet mapped memory. See what CheckAccessFullyMapped returns.
            data = Buf.Load(thr + 1024 * 1024, code);
            value = CheckAccessFullyMapped(code) ? (1u << 16) : 0u;
            value |= data & 0xffffu;

            RWBuf[2 * thr + 1] = value;
    }
#endif
        0x43425844, 0x8c2a40af, 0x2a9b20a6, 0xa99f0977, 0x37daacf5, 0x00000001, 0x00000280, 0x00000004,
        0x00000030, 0x00000040, 0x00000050, 0x00000270, 0x4e475349, 0x00000008, 0x00000000, 0x00000008,
        0x4e47534f, 0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000218, 0x00050050, 0x00000086,
        0x0100086a, 0x04000858, 0x00107000, 0x00000000, 0x00004444, 0x0400009e, 0x0011e000, 0x00000000,
        0x00000004, 0x0200005f, 0x00020012, 0x02000068, 0x00000002, 0x0400009b, 0x00000001, 0x00000001,
        0x00000001, 0x8a0000df, 0x80000042, 0x00111103, 0x00100012, 0x00000000, 0x00100012, 0x00000001,
        0x00020006, 0x00107e46, 0x00000000, 0x050000ea, 0x00100022, 0x00000000, 0x0010000a, 0x00000001,
        0x09000037, 0x00100022, 0x00000000, 0x0010001a, 0x00000000, 0x00004001, 0x00010000, 0x00004001,
        0x00000000, 0x0b00008c, 0x00100012, 0x00000000, 0x00004001, 0x00000010, 0x00004001, 0x00000000,
        0x0010000a, 0x00000000, 0x0010001a, 0x00000000, 0x06000029, 0x00100022, 0x00000000, 0x0002000a,
        0x00004001, 0x00000001, 0x090000a8, 0x0011e012, 0x00000000, 0x0010001a, 0x00000000, 0x00004001,
        0x00000000, 0x0010000a, 0x00000000, 0x1300008c, 0x00100052, 0x00000000, 0x00004002, 0x00000014,
        0x00000000, 0x0000001f, 0x00000000, 0x00004002, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
        0x00020006, 0x00004002, 0x00100000, 0x00000000, 0x00000001, 0x00000000, 0x8b0000df, 0x80000042,
        0x00111103, 0x00100012, 0x00000000, 0x00100012, 0x00000001, 0x00100006, 0x00000000, 0x00107e46,
        0x00000000, 0x050000ea, 0x00100082, 0x00000000, 0x0010000a, 0x00000001, 0x09000037, 0x00100082,
        0x00000000, 0x0010003a, 0x00000000, 0x00004001, 0x00010000, 0x00004001, 0x00000000, 0x0b00008c,
        0x00100012, 0x00000000, 0x00004001, 0x00000010, 0x00004001, 0x00000000, 0x0010000a, 0x00000000,
        0x0010003a, 0x00000000, 0x090000a8, 0x0011e012, 0x00000000, 0x0010002a, 0x00000000, 0x00004001,
        0x00000000, 0x0010000a, 0x00000000, 0x0100003e, 0x30494653, 0x00000008, 0x00000100, 0x00000000,
    };
    static const D3D12_SHADER_BYTECODE cs_sparse_query = SHADER_BYTECODE(cs_sparse_query_dxbc);

    if (!init_compute_test_context(&context))
        return;

    hr = ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_D3D12_OPTIONS, &options, sizeof(options));
    ok(hr == S_OK, "Failed to check feature support, hr %#x.\n", hr);

    if (options.TiledResourcesTier < D3D12_TILED_RESOURCES_TIER_2)
    {
        skip("Tiled resources Tier 2 not supported by device.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&rs_desc, 0, sizeof(rs_desc));
    memset(root_parameters, 0, sizeof(root_parameters));
    memset(&desc_range, 0, sizeof(desc_range));
    rs_desc.NumParameters = ARRAY_SIZE(root_parameters);
    rs_desc.pParameters = root_parameters;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[1].DescriptorTable.NumDescriptorRanges = 1;
    root_parameters[1].DescriptorTable.pDescriptorRanges = &desc_range;
    desc_range.NumDescriptors = 1;
    desc_range.RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_SRV;
    create_root_signature(context.device, &rs_desc, &context.root_signature);
    context.pipeline_state = create_compute_pipeline_state(context.device, context.root_signature, cs_sparse_query);

    memset(&heap_desc, 0, sizeof(heap_desc));
    heap_desc.SizeInBytes = 4 * 1024 * 1024;
    heap_desc.Properties.Type = D3D12_HEAP_TYPE_DEFAULT;
    heap_desc.Alignment = D3D12_DEFAULT_RESOURCE_PLACEMENT_ALIGNMENT;
    heap_desc.Flags = D3D12_HEAP_FLAG_ALLOW_ONLY_BUFFERS;
    hr = ID3D12Device_CreateHeap(context.device, &heap_desc, &IID_ID3D12Heap, (void**)&heap);
    ok(SUCCEEDED(hr), "Failed to create heap, hr #%x.\n", hr);
    hr = ID3D12Device_CreateHeap(context.device, &heap_desc, &IID_ID3D12Heap, (void**)&heap_live);
    ok(SUCCEEDED(hr), "Failed to create heap, hr #%x.\n", hr);

    memset(&desc, 0, sizeof(desc));
    desc.Width = 64 * 1024 * 1024;
    desc.Height = 1;
    desc.DepthOrArraySize = 1;
    desc.SampleDesc.Count = 1;
    desc.Format = DXGI_FORMAT_UNKNOWN;
    desc.Dimension = D3D12_RESOURCE_DIMENSION_BUFFER;
    desc.MipLevels = 1;
    desc.Alignment = D3D12_DEFAULT_RESOURCE_PLACEMENT_ALIGNMENT;
    desc.Layout = D3D12_TEXTURE_LAYOUT_ROW_MAJOR;
    desc.Flags = D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS;
    hr = ID3D12Device_CreateReservedResource(context.device, &desc, D3D12_RESOURCE_STATE_UNORDERED_ACCESS,
            NULL, &IID_ID3D12Resource, (void**)&sparse);
    ok(SUCCEEDED(hr), "Failed to create reserved resource, hr #%x.\n", hr);

    {
        const D3D12_TILED_RESOURCE_COORDINATE region_start_coordinate = { 0 };
        const D3D12_TILE_RANGE_FLAGS range_flag = D3D12_TILE_RANGE_FLAG_NULL;
        const UINT offset = 0;
        const UINT count = desc.Width / (64 * 1024);
        region_size.UseBox = FALSE;
        region_size.NumTiles = desc.Width / (64 * 1024);
        ID3D12CommandQueue_UpdateTileMappings(context.queue, sparse, 1, &region_start_coordinate, &region_size,
                NULL, 1, &range_flag, &offset, &count, D3D12_TILE_MAPPING_FLAG_NONE);
    }

    region_size.UseBox = FALSE;
    region_size.NumTiles = 1;

    for (i = 0; i < 2; i++)
    {
        const D3D12_TILED_RESOURCE_COORDINATE region_start_coordinate = { i, 0, 0, 0 };
        const D3D12_TILE_RANGE_FLAGS range_flag = D3D12_TILE_RANGE_FLAG_NONE;
        const UINT offset = i;
        const UINT count = 1;

        ID3D12CommandQueue_UpdateTileMappings(context.queue, sparse, 1, &region_start_coordinate, &region_size,
                i == 0 ? heap : heap_live, 1, &range_flag, &offset, &count, D3D12_TILE_MAPPING_FLAG_NONE);
    }
    wait_queue_idle(context.device, context.queue);

    buffer = create_default_buffer(context.device, 128 * 1024,
            D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_DEST);
    cpu = create_cpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 1);
    gpu = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 2);
    memset(&uav_desc, 0, sizeof(uav_desc));
    uav_desc.ViewDimension = D3D12_UAV_DIMENSION_BUFFER;
    uav_desc.Format = DXGI_FORMAT_R32_UINT;
    uav_desc.Buffer.NumElements = 128 * 1024 / 4;
    uav_desc.Buffer.FirstElement = 0;
    ID3D12Device_CreateUnorderedAccessView(context.device, sparse, NULL, &uav_desc,
            ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(cpu));
    ID3D12Device_CreateUnorderedAccessView(context.device, sparse, NULL, &uav_desc,
            ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(gpu));

    memset(&srv_desc, 0, sizeof(srv_desc));
    srv_desc.Buffer.FirstElement = 0;
    srv_desc.Buffer.NumElements = 2 * 1024 * 1024;
    srv_desc.ViewDimension = D3D12_SRV_DIMENSION_BUFFER;
    srv_desc.Format = DXGI_FORMAT_R32_UINT;
    srv_desc.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;

    h_cpu = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(gpu);
    h_cpu.ptr += ID3D12Device_GetDescriptorHandleIncrementSize(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);
    ID3D12Device_CreateShaderResourceView(context.device, sparse, &srv_desc, h_cpu);

    ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &gpu);
    ID3D12GraphicsCommandList_ClearUnorderedAccessViewUint(context.list,
            ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(gpu),
            ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(cpu), sparse, values, 0, NULL);
    transition_resource_state(context.list, sparse,
            D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
    ID3D12GraphicsCommandList_CopyBufferRegion(context.list, buffer, 0, sparse, 0, 128 * 1024);
    transition_resource_state(context.list, buffer,
            D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_COPY_SOURCE);
    get_buffer_readback_with_command_list(buffer, DXGI_FORMAT_R32_UINT,
            &rb, context.queue, context.list);
    reset_command_list(context.list, context.allocator);
    ok(get_readback_uint(&rb, 0, 0, 0) == 42, "Got #%x, expected 42.\n", get_readback_uint(&rb, 0, 0, 0));
    ok(get_readback_uint(&rb, 64 * 1024 / 4, 0, 0) == 42, "Got #%x, expected 42.\n", get_readback_uint(&rb, 64 * 1024 / 4, 0, 0));
    release_resource_readback(&rb);

    ID3D12Heap_Release(heap);

    /* Access a resource where we can hypothetically access the freed heap memory. */
    /* On AMD Windows native at least, if we read the freed region, we read garbage, which proves it's not required to unbind explicitly.
     * We'd read 0 in that case. */
    ID3D12GraphicsCommandList_CopyBufferRegion(context.list, buffer, 0, sparse, 64 * 1024, 64 * 1024);

#define EXPLORE_UNDEFINED_BEHAVIOR 0

#if EXPLORE_UNDEFINED_BEHAVIOR
    /* This reads unmapped memory. */
    ID3D12GraphicsCommandList_CopyBufferRegion(context.list, buffer, 1024, sparse, 1024, 1024);
#endif

    transition_resource_state(context.list, buffer, D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

    h_gpu = ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(gpu);
    h_gpu.ptr += ID3D12Device_GetDescriptorHandleIncrementSize(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);
    ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &gpu);
    ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(context.list, context.pipeline_state);
    ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(context.list, 0, ID3D12Resource_GetGPUVirtualAddress(buffer));
    ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(context.list, 1, h_gpu);
#if EXPLORE_UNDEFINED_BEHAVIOR
    ID3D12GraphicsCommandList_Dispatch(context.list, 1, 1, 1);
#endif

    transition_resource_state(context.list, buffer, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

    get_buffer_readback_with_command_list(buffer, DXGI_FORMAT_R32_UINT,
            &rb, context.queue, context.list);

#if EXPLORE_UNDEFINED_BEHAVIOR
    skip("Reading undefined value #%x.\n", get_readback_uint(&rb, 0, 0, 0));
    skip("Reading value #%x (expect 0).\n", get_readback_uint(&rb, 1, 0, 0));
    skip("Reading undefined value #%x.\n", get_readback_uint(&rb, 1024 / 4, 0, 0));
#endif
    ok(get_readback_uint(&rb, 2048 / 4, 0, 0) == 42, "Got #%x, expected 42.\n", get_readback_uint(&rb, 2048 / 4, 0, 0));
    ok(get_readback_uint(&rb, 64 * 1024 / 4, 0, 0) == 42, "Got #%x, expected 42.\n", get_readback_uint(&rb, 64 * 1024 / 4, 0, 0));
    release_resource_readback(&rb);

    ID3D12Resource_Release(buffer);
    ID3D12Resource_Release(sparse);
    ID3D12DescriptorHeap_Release(cpu);
    ID3D12DescriptorHeap_Release(gpu);
    ID3D12Heap_Release(heap_live);
    destroy_test_context(&context);
}

void test_reserved_resource_mapping(void)
{
    struct test_context context;
    D3D12_FEATURE_DATA_D3D12_OPTIONS options;
    D3D12_RESOURCE_DESC desc;
    ID3D12Resource *res;
    HRESULT hr;
    void *ptr;

    if (!init_compute_test_context(&context))
        return;

    hr = ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_D3D12_OPTIONS, &options, sizeof(options));
    ok(hr == S_OK, "Failed to check feature support, hr %#x.\n", hr);

    if (!options.TiledResourcesTier)
    {
        skip("Tiled resources not supported by device.\n");
        destroy_test_context(&context);
        return;
    }


    memset(&desc, 0, sizeof(desc));
    desc.Dimension = D3D12_RESOURCE_DIMENSION_BUFFER;
    desc.MipLevels = 1;
    desc.DepthOrArraySize = 1;
    desc.Height = 1;
    desc.Width = 64 * 1024;
    desc.Layout = D3D12_TEXTURE_LAYOUT_ROW_MAJOR;
    desc.SampleDesc.Count = 1;
    hr = ID3D12Device_CreateReservedResource(context.device, &desc, D3D12_RESOURCE_STATE_COMMON, NULL, &IID_ID3D12Resource, (void **)&res);
    ok(SUCCEEDED(hr), "Failed to create reserved resource, hr #%x.\n", hr);

    hr = ID3D12Resource_Map(res, 0, NULL, &ptr);
    ok(hr == E_INVALIDARG, "Unexpected return value hr #%x.\n", hr);

    ID3D12Resource_Release(res);

    destroy_test_context(&context);
}
