/*
    razertool - Tool for controlling Razer Copperhead(TM) mice
    Copyright (C) 2006  Christopher Lais

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
*/

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include <usb.h>

#include "razertool.h"
#include "razerlib.h"
#include "razerfirmware.h"

#define IS_DEBUG()	(razerlib_debug!=0)
#define DEBUG(fmt,...)	do{if(IS_DEBUG()) fprintf(stderr, fmt , ## __VA_ARGS__);}while(0)

#define USB_REQ_RAZER_FW_PROGRAM	0x81
#define USB_REQ_RAZER_FW_ERASE	0x82
#define USB_REQ_RAZER_FW_ERASE_ALL	0x83
#define USB_REQ_RAZER_FW_READ	0x84
#define USB_REQ_RAZER_FW_VERIFY	0x87
#define USB_REQ_RAZER_FW_STATUS	0x8f

int razer_dummy_flash = 0;

struct usb_device *razer_find_flash(struct usb_device *prev)
{
	struct usb_bus *busses;
	struct usb_bus *bus;
	const char *debug;

	/* FIXME: wrong place for this. */
	debug = getenv("RAZERLIB_DEBUG");
	if (debug) { razerlib_debug = atoi(debug); }

	if (!prev) {
		usb_find_busses();
		usb_find_devices();
	}

	busses = usb_get_busses();

	for (bus = busses; bus; bus = bus->next) {
		struct usb_device *dev;

		for (dev = bus->devices; dev; dev = dev->next) {
			DEBUG("DEVICE: %04x:%04x\n", dev->descriptor.idVendor, dev->descriptor.idProduct);

			if (prev) {
				if (prev == dev)
					prev = NULL;
				continue;
			}

			if (dev->descriptor.idVendor != USB_VENDOR_FREESCALE)
				continue;

			if (dev->descriptor.idProduct != USB_PRODUCT_FREESCALE_JW32_ICP)
				continue;

			return dev;
		}
	}

	return NULL;
}

int razer_firmware_status(struct usb_dev_handle *udev, int *stat, int timeout)
{
	char buf[1];
	int status;

	DEBUG("%s:", __FUNCTION__);

	status = usb_control_msg(
		udev,
		USB_ENDPOINT_IN | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
		USB_REQ_RAZER_FW_STATUS,
		0,
		0,
		buf, 1,
		timeout
	);

	DEBUG("%d:%d\n", status, buf[0]);

	if (status < 0)
		return status;

	*stat = buf[0];

	return 0;
}

int razer_firmware_erase_block(struct usb_dev_handle *udev, int start, int len, int timeout)
{
	int status, stat;

	if (len != RAZER_FIRMWARE_ERASE_BLOCK_LEN)
		return -EINVAL;
	if (start % RAZER_FIRMWARE_ERASE_BLOCK_LEN)
		return -EINVAL;

	if (razer_dummy_flash)
		return 0;

	DEBUG("%s:", __FUNCTION__);

	status = usb_control_msg(
		udev,
		USB_ENDPOINT_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
		USB_REQ_RAZER_FW_ERASE,
		start,
		start + len - 1,
		NULL, 0,
		timeout
	);

	DEBUG("%d\n", status);

	if (status < 0)
		return status;

	status = razer_firmware_status(udev, &stat, timeout);
	if (status < 0)
		return status;

	if (stat != 1)
		return -RAZERLIB_ERR_STATUS;

	return 0;
}

int razer_firmware_program_block(struct usb_dev_handle *udev, int start, /* const */ char *data, int len, int timeout)
{
	int status, stat;

	if (len != RAZER_FIRMWARE_BLOCK_LEN)
		return -EINVAL;

	if (razer_dummy_flash)
		return 0;

	DEBUG("%s:", __FUNCTION__);

	status = usb_control_msg(
		udev,
		USB_ENDPOINT_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
		USB_REQ_RAZER_FW_PROGRAM,
		start,
		start + len - 1,
		data, len,
		timeout
	);

	DEBUG("%d\n", status);

	if (status < 0)
		return status;

	status = razer_firmware_status(udev, &stat, timeout);
	if (status < 0)
		return status;

	if (stat != 1)
		return -RAZERLIB_ERR_STATUS;

	return 0;
}

/* Doesn't work.. */
int razer_firmware_read_block(struct usb_dev_handle *udev, int start, char *data, int len, int timeout)
{
	int status, stat;

	DEBUG("%s:", __FUNCTION__);

	status = usb_control_msg(
		udev,
		USB_ENDPOINT_IN | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
		USB_REQ_RAZER_FW_READ,
		start,
		start + len - 1,
		data, len,
		timeout
	);

	DEBUG("%d\n", status);

	if (status < 0)
		return status;

	status = razer_firmware_status(udev, &stat, timeout);
	if (status < 0)
		return status;

	if (stat != 1)
		return -RAZERLIB_ERR_STATUS;

	return 0;
}

/* Initialize data to 0xff, or to what you think is there. */
int razer_firmware_read_block_slow(struct usb_dev_handle *udev, int start, char *data, int len, int timeout)
{
	int i, guess;
	int status;

	for (i = 0; i < len; i++) {
		for (guess = 0; guess <= 0xff; guess++) {
			unsigned char guess_ch = (guess + data[i]) & 0xff; /* start w/ guess, go up  */

			status = razer_firmware_verify_block(
				udev,
				start + i,
				(char *)&guess_ch,
				1,
				RAZERLIB_DEFAULT_TIMEOUT
			);
			if (status < 0) {
				if (status == -RAZERLIB_ERR_STATUS)
					continue;
				return status;
			}

			data[i] = guess_ch;
			goto next_char;
		}

		return -RAZERLIB_ERR_STATUS;

	  next_char:
		continue;
	}

	return 0;
}

int razer_firmware_verify_block(struct usb_dev_handle *udev, int start, /* const */ char *data, int len, int timeout)
{
	int status, stat;

	DEBUG("%s:", __FUNCTION__);

	status = usb_control_msg(
		udev,
		USB_ENDPOINT_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
		USB_REQ_RAZER_FW_VERIFY,
		start,
		start + len - 1,
		data, len,
		timeout
	);

	DEBUG("%d\n", status);

	if (status < 0)
		return status;

	status = razer_firmware_status(udev, &stat, timeout);
	if (status < 0)
		return status;

	if (stat != 1)
		return -RAZERLIB_ERR_STATUS;

	return 0;
}

static int razer_firmware_add_data(razer_firmware_t *f, int addr, unsigned char *data, int datalen)
{
	int i;

	if (addr < 0 || (addr+datalen) > RAZER_FIRMWARE_LEN)
		return -EINVAL;

	memcpy(&f->mem[addr], data, datalen);

	for (i = addr; i < (addr+datalen+RAZER_FIRMWARE_BLOCK_LEN-1); i += RAZER_FIRMWARE_BLOCK_LEN)
		f->blocks[i / RAZER_FIRMWARE_BLOCK_LEN] = 1;

	return 0;
}

#define ERROR()		DEBUG("%s:%d\n", __FUNCTION__, __LINE__)

int razer_firmware_parse_srec(int fd, razer_firmware_t *f)
{
	unsigned char buf[514];
	int srec_count = 0;
	unsigned int start_addr = 0;

	memset(f->mem, 0xff, sizeof(f->mem));
	memset(f->blocks, 0, sizeof(f->blocks));

	for (;;) {
		ssize_t readlen;
		unsigned char type, csum;
		int len;
		int i;
	
		readlen = read(fd, buf, 1);
		if (readlen == 0) break;
		if (readlen != 1) { ERROR(); return -RAZERLIB_ERR_PARSE; }

		/* Ignore spurious line feeds */
		if (buf[0] == '\n' || buf[0] == '\r')
			continue;

		/* Verify that it is, in fact, an S record */
		if (buf[0] != 'S')
			{ ERROR(); return -RAZERLIB_ERR_PARSE; }

		/* Read the type and length */
		readlen = read(fd, buf, 3);
		if (readlen != 3) { ERROR(); return -RAZERLIB_ERR_PARSE; }

		type = buf[0];
	
		len = razer_2hex((const char*)&buf[1]);
		if (len < 0) { ERROR(); return -RAZERLIB_ERR_PARSE; }

		/* Read the address/data/checksum */
		readlen = read(fd, buf, 2*len + 1);
		if (readlen != (2*len + 1)) { ERROR(); return -RAZERLIB_ERR_PARSE; }

		/* Make sure it's terminated with a line ending */
		if (buf[2*len] != '\r' && buf[2*len] != '\n')
			{ ERROR(); return -RAZERLIB_ERR_PARSE; }

		/* Convert the buffer to binary and verify the checksum */
		csum = len;
		for (i = 0; i < len; i++) {
			int tmp = razer_2hex((const char*)&buf[2*i]);
			if (tmp < 0) { ERROR(); return -RAZERLIB_ERR_PARSE; }
			buf[i] = tmp;
			csum += buf[i];
		}
		if ((csum&0xff) != 0xff) { ERROR(); return -RAZERLIB_ERR_PARSE; }

		--len; /* take the checksum out of the picture */

		switch (type) {
		  case '0': continue; /* ignore it */

		  /* 16-bit address + data */
		  case '1':
			if (len < 2)
				{ ERROR(); return -RAZERLIB_ERR_PARSE; }
			++srec_count;
			if (razer_firmware_add_data(f, (buf[0] << 8) | buf[1], &buf[2], len - 2) < 0)
				{ ERROR(); return -RAZERLIB_ERR_PARSE; }
			break;
		  /* 24-bit address + data */
		  case '2':
			if (len < 3)
				{ ERROR(); return -RAZERLIB_ERR_PARSE; }
			++srec_count;
			if (razer_firmware_add_data(f, (buf[0] << 16) | (buf[1] << 8) | buf[2], &buf[3], len - 3) < 0)
				{ ERROR(); return -RAZERLIB_ERR_PARSE; }
			break;
		  /* 32-bit address + data */
		  case '3':
			if (len < 4)
				{ ERROR(); return -RAZERLIB_ERR_PARSE; }
			++srec_count;
			if (razer_firmware_add_data(f, (buf[0] << 24) | (buf[1] << 16) | (buf[2] << 8) | buf[3], &buf[4], len - 4) < 0)
				{ ERROR(); return -RAZERLIB_ERR_PARSE; }
			break;

		  case '5':
			if (len != 2)
				{ ERROR(); return -RAZERLIB_ERR_PARSE; }
			if (((buf[1] << 8) | buf[0]) != srec_count)
				{ ERROR(); return -RAZERLIB_ERR_PARSE; }
			break;

		  /* 32-bit start address */
		  case '7':
			if (len != 4)
				{ ERROR(); return -RAZERLIB_ERR_PARSE; }
			start_addr = (buf[0] << 24) | (buf[1] << 16) | (buf[2] << 8) | buf[3];
			break;
		  /* 24-bit start address */
		  case '8':
			if (len != 3)
				{ ERROR(); return -RAZERLIB_ERR_PARSE; }
			start_addr = (buf[0] << 16) | (buf[1] << 8) | buf[2];
			break;
		  /* 16-bit start address */
		  case '9':
			if (len != 2)
				{ ERROR(); return -RAZERLIB_ERR_PARSE; }
			start_addr = (buf[0] << 8) | buf[1];
			break;

		  default:
			{ ERROR(); return -RAZERLIB_ERR_PARSE; }
		}
	}

	if (start_addr)
		{ ERROR(); return -RAZERLIB_ERR_PARSE; }

	return 0;
}

int razer_firmware_write_flash(
	struct usb_dev_handle *udev,
	razer_firmware_t *f,
	razer_progress_callback_t progress
)
{
	int i;
	int status;

#if 1
	/* Verify */
	for (i = RAZER_FIRMWARE_FIRST_FLASH; i <= RAZER_FIRMWARE_LAST_FLASH; i += RAZER_FIRMWARE_ERASE_BLOCK_LEN) {
		int block = i / RAZER_FIRMWARE_BLOCK_LEN;
		double fraction = (double)(i-RAZER_FIRMWARE_FIRST_FLASH) / (double)(RAZER_FIRMWARE_LAST_FLASH-RAZER_FIRMWARE_FIRST_FLASH);
		int j;

		for (j = 0; j < (RAZER_FIRMWARE_ERASE_BLOCK_LEN/RAZER_FIRMWARE_BLOCK_LEN); j++) {
			int addr = i + j*RAZER_FIRMWARE_BLOCK_LEN;
			if (!f->blocks[block+j]) continue;

			if (progress) progress(
				RAZER_PROGRESS_VERIFYING,
				addr, addr + RAZER_FIRMWARE_BLOCK_LEN - 1,
				fraction
			);

			status = razer_firmware_verify_block(
				udev,
				addr,
				(char *)&f->mem[addr],
				RAZER_FIRMWARE_BLOCK_LEN,
				RAZERLIB_DEFAULT_TIMEOUT
			);
			if (status < 0) {
				if (status != -RAZERLIB_ERR_STATUS)
					return status;

				/* Block needs flashing. */
				goto needs_flash;
			}
		}

		/* Every piece verified - Doesn't need erased/flashed. */
		for (j = 0; j < (RAZER_FIRMWARE_ERASE_BLOCK_LEN/RAZER_FIRMWARE_BLOCK_LEN); j++)
			f->blocks[block+j] = 0;

	  needs_flash:
		continue;
	}
#endif

	/* Erase */
	for (i = RAZER_FIRMWARE_LAST_FLASH; i >= RAZER_FIRMWARE_FIRST_FLASH; i -= RAZER_FIRMWARE_ERASE_BLOCK_LEN) {
		int block = (i - i%RAZER_FIRMWARE_ERASE_BLOCK_LEN)/RAZER_FIRMWARE_BLOCK_LEN;
		int addr_start = block*RAZER_FIRMWARE_BLOCK_LEN;
		int j;

		for (j = 0; j < (RAZER_FIRMWARE_ERASE_BLOCK_LEN/RAZER_FIRMWARE_BLOCK_LEN); j++)
			if (f->blocks[block+j]) goto has_block;
		continue;

	  has_block:
		if (progress) progress(
			RAZER_PROGRESS_ERASING,
			addr_start, addr_start+RAZER_FIRMWARE_ERASE_BLOCK_LEN-1,
			1.0 - (double)(i-RAZER_FIRMWARE_FIRST_FLASH) / (double)(RAZER_FIRMWARE_LAST_FLASH-RAZER_FIRMWARE_FIRST_FLASH)
		);
		status = razer_firmware_erase_block(
			udev,
			addr_start,
			RAZER_FIRMWARE_ERASE_BLOCK_LEN,
			-1
		);
		if (status < 0)
			return status;
	}

	/* Program */
	for (i = RAZER_FIRMWARE_FIRST_FLASH; i <= RAZER_FIRMWARE_LAST_FLASH; i += RAZER_FIRMWARE_BLOCK_LEN) {
		int block = i / RAZER_FIRMWARE_BLOCK_LEN;

		if (!f->blocks[block]) continue;

		if (progress) progress(
			RAZER_PROGRESS_PROGRAMMING,
			i, i+RAZER_FIRMWARE_BLOCK_LEN-1,
			(double)(i-RAZER_FIRMWARE_FIRST_FLASH) / (double)(RAZER_FIRMWARE_LAST_FLASH-RAZER_FIRMWARE_FIRST_FLASH)
		);
		status = razer_firmware_program_block(
			udev,
			i,
			(char *)&f->mem[i],
			RAZER_FIRMWARE_BLOCK_LEN,
			-1
		);
		if (status < 0)
			return status;
	}

	if (progress) progress(RAZER_PROGRESS_DONE, 0, 0, 1.0);

	return 0;
}

int razer_firmware_read_flash(
	struct usb_dev_handle *udev,
	razer_firmware_t *f,
	razer_progress_callback_t progress
)
{
	int i, j;
	int status;

	memset(f->mem, 0xff, sizeof(f->mem));
	memset(f->blocks, 0, sizeof(f->blocks));

	for (i = RAZER_FIRMWARE_BLOCKS_FIRST; i < RAZER_FIRMWARE_BLOCKS; i++) {
		if (progress) progress(
			RAZER_PROGRESS_READING,
			i*RAZER_FIRMWARE_BLOCK_LEN, (i+1)*RAZER_FIRMWARE_BLOCK_LEN-1,
			(double)(i - RAZER_FIRMWARE_BLOCKS_FIRST) / (double)(RAZER_FIRMWARE_BLOCKS-RAZER_FIRMWARE_BLOCKS_FIRST-1)
		);

		status = razer_firmware_read_block(
			udev,
			i*RAZER_FIRMWARE_BLOCK_LEN,
			(char *)&f->mem[i*RAZER_FIRMWARE_BLOCK_LEN],
			RAZER_FIRMWARE_BLOCK_LEN,
			RAZERLIB_DEFAULT_TIMEOUT
		);
		if (status < 0) {
			if (status == -RAZERLIB_ERR_STATUS)
				return i*RAZER_FIRMWARE_BLOCK_LEN;
			return status;
		}
		for (j = 0; j < RAZER_FIRMWARE_BLOCK_LEN; j++) {
			if (f->mem[i*RAZER_FIRMWARE_BLOCK_LEN+j] != 0xff) {
				f->blocks[i] = 1;
				break;
			}
		}
	}

	if (progress) progress(RAZER_PROGRESS_DONE, 0, 0, 1.0);

	return 0;
}

int razer_firmware_read_flash_slow(
	struct usb_dev_handle *udev,
	razer_firmware_t *f,
	razer_progress_callback_t progress
)
{
	int i, guess;
	int counter = 0, lcount = -1;
	int status;

	memset(f->blocks, 0, sizeof(f->blocks));

	for (i = RAZER_FIRMWARE_FIRST_FLASH; i < RAZER_FIRMWARE_LEN; i++) {
		for (guess = 0; guess <= 0xff; guess++) {
			unsigned char guess_ch = (guess + f->mem[i]) & 0xff; /* start w/ guess, go up  */

			if ((counter / RAZER_FIRMWARE_BLOCK_LEN) != lcount) {
				lcount = counter / RAZER_FIRMWARE_BLOCK_LEN;
				if (progress) progress(
					RAZER_PROGRESS_READING,
					i, i,
					(double)(i - RAZER_FIRMWARE_FIRST_FLASH) / (double)(RAZER_FIRMWARE_LEN-RAZER_FIRMWARE_FIRST_FLASH-1)
				);
			}

			status = razer_firmware_verify_block(
				udev,
				i,
				(char *)&guess_ch,
				1,
				RAZERLIB_DEFAULT_TIMEOUT
			);
			if (status < 0) {
				if (status == -RAZERLIB_ERR_STATUS) {
					counter++;
					continue;
				}
				return status;
			}

			f->mem[i] = guess_ch;
			if (guess_ch != 0xff)
				f->blocks[i / RAZER_FIRMWARE_BLOCK_LEN] = 1;
			goto next_char;
		}

		return i;

	  next_char:
		counter++;
		continue;
	}

	if (progress) progress(RAZER_PROGRESS_DONE, 0, 0, 1.0);

	return 0;
}

int razer_firmware_verify_flash(
	struct usb_dev_handle *udev,
	razer_firmware_t *f,
	razer_progress_callback_t progress
)
{
	int i;
	int status;
	int ret = 0;

	for (i = RAZER_FIRMWARE_FIRST_FLASH; i <= RAZER_FIRMWARE_LAST_FLASH; i += RAZER_FIRMWARE_BLOCK_LEN) {
		double fraction = (double)(i-RAZER_FIRMWARE_FIRST_FLASH) / (double)(RAZER_FIRMWARE_LAST_FLASH-RAZER_FIRMWARE_FIRST_FLASH);

		if (progress) progress(
			RAZER_PROGRESS_VERIFYING,
			i, i+RAZER_FIRMWARE_BLOCK_LEN-1,
			fraction
		);

		status = razer_firmware_verify_block(
			udev,
			i,
			(char *)&f->mem[i],
			RAZER_FIRMWARE_BLOCK_LEN,
			RAZERLIB_DEFAULT_TIMEOUT
		);
		if (status < 0) {
			if (status != -RAZERLIB_ERR_STATUS)
				return status;

			if (!ret)
				ret = i;

			if (progress) progress(
				RAZER_PROGRESS_VERIFY_ERROR,
				i, i+RAZER_FIRMWARE_BLOCK_LEN-1,
				fraction
			);
		}
	}

	if (progress) progress(RAZER_PROGRESS_DONE, 0, 0, 1.0);

	return ret;
}

void razer_firmware_dump(razer_firmware_t *f)
{
	int i, j;
	for (i = 0; i < RAZER_FIRMWARE_BLOCKS; i++) {
		if (!f->blocks[i])
			continue;
		printf("%04x: ", i*RAZER_FIRMWARE_BLOCK_LEN);
		for (j = 0; j < RAZER_FIRMWARE_BLOCK_LEN; j++)
			printf("%02x", f->mem[i*RAZER_FIRMWARE_BLOCK_LEN + j]);
		printf("\n");
	}
}

void razer_firmware_dump_srecord_single(char type, unsigned int addr, int addrlen, const unsigned char *data, int datalen)
{
	int i;
	unsigned char csum = addrlen + datalen + 1;

	for (i = 0; i < addrlen; i++)
		csum += (addr >> (i*8)) & 0xff;
	for (i = 0; i < datalen; i++)
		csum += data[i];

	csum = (csum ^ 0xff) & 0xff;

	printf("S%c%02X", type, (addrlen + datalen + 1) & 0xff);
	for (i = addrlen - 1; i >= 0; i--)
		printf("%02X", (addr >> (i*8)) & 0xff);
	for (i = 0; i < datalen; i++)
		printf("%02X", data[i]);
	printf("%02X\n", csum);
}

static const char razerflash_hdr[] = "dumped with razerflash " VERSION;
void razer_firmware_dump_srecord(razer_firmware_t *f)
{
	int i;
	int srec_count = 0;

	razer_firmware_dump_srecord_single('0', 0, 2, (const unsigned char*)razerflash_hdr, strlen(razerflash_hdr));
	for (i = RAZER_FIRMWARE_FIRST_FLASH; i <= RAZER_FIRMWARE_LAST_FLASH; i += 0x20) {
		int block = i / RAZER_FIRMWARE_BLOCK_LEN;
		if (!f->blocks[block]) continue;
		razer_firmware_dump_srecord_single('1', i, 2, &f->mem[i], 0x20);
		++srec_count;
	}
	razer_firmware_dump_srecord_single('9', 0, 2, NULL, 0);
}

#define RAZER_FIRMWARE_LASER_ADDRESS	0xde00

/* bit 7: VCSEL match, 6-0 power; 0 = 100%, 0x7f = 33.85% */
int razer_firmware_read_laser_power(struct usb_dev_handle *udev, int *power, razer_progress_callback_t progress)
{
	int status;
	unsigned char ch_power;

	ch_power = 0xff;

	if (progress) progress(
		RAZER_PROGRESS_READING,
		RAZER_FIRMWARE_LASER_ADDRESS, RAZER_FIRMWARE_LASER_ADDRESS,
		0.0
	);
	DEBUG("%s:read_block:", __FUNCTION__);
	status = razer_firmware_read_block_slow(
		udev, 0xde00, (char*)&ch_power, 1, RAZERLIB_DEFAULT_TIMEOUT
	);

	*power = ch_power;

	if (status < 0)
		return status;

	if (progress) progress(RAZER_PROGRESS_DONE, 0, 0, 1.0);

	return 0;
}

int razer_firmware_write_laser_power(struct usb_dev_handle *udev, int power, razer_progress_callback_t progress)
{
	int status;
	unsigned char buf[RAZER_FIRMWARE_ERASE_BLOCK_LEN];
	int i;

	if (power < 0 || power > 0xff)
		return -EINVAL;

	memset(buf, 0xff, sizeof(buf));

	if (progress) progress(
		RAZER_PROGRESS_READING,
		RAZER_FIRMWARE_LASER_ADDRESS, RAZER_FIRMWARE_LASER_ADDRESS+RAZER_FIRMWARE_BLOCK_LEN-1,
		0.0
	);
	DEBUG("%s:read_block:", __FUNCTION__);
	status = razer_firmware_read_block_slow(
		udev, RAZER_FIRMWARE_LASER_ADDRESS, (char*)buf, sizeof(buf), RAZERLIB_DEFAULT_TIMEOUT
	);
	if (status < 0)
		return status;

	buf[0] = power;

	if (progress) progress(
		RAZER_PROGRESS_ERASING,
		RAZER_FIRMWARE_LASER_ADDRESS, RAZER_FIRMWARE_LASER_ADDRESS+RAZER_FIRMWARE_ERASE_BLOCK_LEN-1,
		0.5
	);
	DEBUG("%s:erase_block:", __FUNCTION__);
	status = razer_firmware_erase_block(
		udev, RAZER_FIRMWARE_LASER_ADDRESS, sizeof(buf), RAZERLIB_DEFAULT_TIMEOUT
	);
	if (status < 0)
		return status;

	for (i = 0; i < RAZER_FIRMWARE_ERASE_BLOCK_LEN; i += RAZER_FIRMWARE_BLOCK_LEN) {
		int address = RAZER_FIRMWARE_LASER_ADDRESS + i;
		unsigned char *p = buf + i;

		if (progress) progress(
			RAZER_PROGRESS_PROGRAMMING,
			address, address+RAZER_FIRMWARE_BLOCK_LEN-1,
			0.75
		);
		DEBUG("%s:program_block:", __FUNCTION__);
		status = razer_firmware_program_block(
			udev, address, (char*)p, RAZER_FIRMWARE_BLOCK_LEN, RAZERLIB_DEFAULT_TIMEOUT
		);
		if (status < 0)
			return status;
	}

	if (progress) progress(RAZER_PROGRESS_DONE, 0, 0, 1.0);

	return 0;
}
