/* tree tools
   Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003
   Wouter van Ooijen

This file is part of jal.

jal is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

jal is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with jal; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"
#include "global.h"
#include "target.h"
#include "errorlh.h"
#include "stacksg.h"
#include "reswords.h"
#include "cstringf.h"
#include "treerep.h"
#include "treetools.h"
#include "parser.h"
#include "regalloc.h"
#include "assemble.h"
#include "scanner.h"
#include "codegen.h"

tree follow(tree p)
{
    assert_pointer(NULL, p);
    stack_guard;
    for (;;) {
        if ((p->kind == node_chain)
            && (p->next != NULL)
            ) {
            p = p->next;
        } else if ((p->kind == node_chain)
                   && (p->first != NULL)
            ) {
            p = p->first;
        } else {
            return p;
        }
        assert_pointer(NULL, p);
    }
    return p;
}

boolean is_type_compatible(tree p, tree q)
{
    stack_guard;
    assert_pointer(NULL, p);
    assert_pointer(NULL, q);
    p = follow(p);
    q = follow(q);
    assert_pointer(NULL, p);
    assert_pointer(NULL, q);

#ifdef __DEBUG__
    trace_subtree(p);
    trace_subtree(q);
#endif

    if (p->kind != node_type) {
        assert_pointer(NULL, p->type);
        return is_type_compatible(p->type, q);

    } else if (q->kind != node_type) {
        assert_pointer(NULL, q->type);
        return is_type_compatible(p, q->type);

    } else if (p == q) {
        return true;

    } else if (p == type_universal) {
        return is_type_compatible(q, p);

    } else if (q == type_universal) {
        if (p == type_byte)
            return true;
        if (p == type_word)
            return true;
        if (p == type_long)
            return true;

    }
    return false;
}

void check_type_compatible(loc_t loc, tree p, tree q)
{
    stack_guard;
#ifdef __DEBUG__
    trace_subtree(p);
    trace_subtree(q);
#endif
    if (!is_type_compatible(p, q)) {
#ifdef __DEBUG__
        trace_subtree(p);
        trace_subtree(q);
#endif
        fatal(loc, (m, "these types are not compatible"));
    }
}

tree new_assign(loc_t loc, tree d, tree s)
{
    tree p = new_node(loc, node_assign);
    stack_guard;
    assert_pointer(NULL, d);
    assert_kind(d->loc, d, node_ref);
#ifdef __DEBUG__
    trace_subtree(d);
#endif
    assert_kind(d->loc, d->type, node_type);
    d->is_target = true;
    p->first = d;
    p->next = s;
    p->type = d->type;
#ifdef __DEBUG__
    trace_subtree(p);
#endif

    if (!d->first->write_allowed) {
#ifdef __DEBUG__
        trace_subtree(d);
#endif
        if (loc == NULL) {
            loc = d->loc;
        }
        fatal(loc, (m, "can not be assigned"));
    }

    check_type_compatible(loc, d, s);

    if ((d->ref == ref_actual)
        && (d->first->is_volatile)
        && (s->kind == node_ref)
        && (s->first->indirect)
        ) {
        s->indirect = true;
    }
    return p;
}

tree new_assign_actual(loc_t loc, tree d, tree s)
{
    boolean save;
    tree t;
    assert_pointer(NULL, d);
    assert_kind(d->loc, d, node_ref);
    save = d->first->write_allowed;
    d->first->write_allowed = true;
    t = new_assign(loc, d, s);
    d->first->write_allowed = save;

    /* 04-25 added */
    if ((d->first->is_volatile)
        && (s->kind == node_ref)
        ) {
        s->indirect = true;
    }

    return t;
}

/* return the maximum depth of the expression */
int expression_depth(tree p)
{
    int depth = 0;
    stack_guard;
    if (p == NULL)
        return 0;
    assert_pointer(NULL, p);
#ifdef __DEBUG__
    log((m, "expression_depth %d %d", p->nr, p->kind));
#endif
    switch (p->kind) {
    case node_ref:
        return expression_depth(p->first);
    case node_const:
        return 1;
    case node_var:
        return 1;
    case node_decl:
        return 0;
    case node_error:
        return 0;
    case node_w:
        return 0;
    case node_precall:
        return 0;               /* should these two be allowed */
    case node_call:
        return 100;             /* here? */
    case node_op:{
            if (is_monop(p->op)) {
                depth = 1 + expression_depth(p->first);
            } else {
                int a = 1 + expression_depth(p->first);
                int b = 1 + expression_depth(p->next);
                depth = max(a, b);
            }
            break;
        }
    case node_assign:{
            depth = 1 + expression_depth(p->next);
        }
    case node_chain:{
            int a = expression_depth(p->first);
            int b = expression_depth(p->next);
            depth = max(a, b);
            break;
        }
    default:{
            snark_node(p->loc, p);
        }
    }
    return depth;
}

/* return call depth (== stack use) */
void call_depth_2(tree p, int *n)
{
    stack_guard;
    if (p == NULL)
        return;
#ifdef __DEBUG__
    log((m, "call_depth %d %s d=%d", p->nr, node_name[p->kind], *n));
#endif

    /* already known? */
    if (p->generation == generation) {

        /* yes, already known */

    } else {
        int x = 0;

        switch (p->kind) {

            /* do not recurse non-called subprograms */
        case node_return:
        case node_ref:
        case node_decl:{
                break;
            }

            /* depth is max of current depth and sub-depth + 1 */
        case node_call:{
                if (p->indirect) {
                    x = indirect_stack_usage;
                } else {
                    call_depth_2(p->first->first, &x);
                }
                if (!p->is_chained) {
                    x = x + 1;
                }
                break;
            }

            /* recurse subtrees */
        default:{
#undef for_each_subtree
#define for_each_subtree( t ) \
               if( p->t != NULL ){ call_depth_2( p->t, &x ); }
                expand_for_all_subtrees break;
            }
        }

        /* save */
        p->data = x;
        p->generation = generation;
    }

    *n = max(*n, p->data);
}

int call_depth(tree p)
{
    int n = 0;
    generation++;
    call_depth_2(p, &n);
    return n;
}

/* a list of strings */
void string_list_add(string_list * p, char *s)
{
    string_list x = allocate(sizeof(string_list));
    if (*p == NULL) {
        *p = x;
    } else {
        string_list q = *p;
        while (q->next != NULL) {
            q = q->next;
        }
        q->next = x;
    }
    x->s = new_string(s);
}

/* prepare and print call tree */
string_list call_tree_root = NULL;
void call_tree_2(tree p, int *n)
{
    stack_guard;
    if (p == NULL)
        return;
#ifdef __DEBUG__
    log((m, "call_depth %d %s d=%d", p->nr, node_name[p->kind], *n));
#endif
    /* already known? */
    if (p->generation == generation) {

        /* yes, already known */

    } else {
        int x = 0;

        switch (p->kind) {

            /* do not recurse non-called subprograms */
        case node_return:
        case node_ref:
        case node_decl:{
                break;
            }

            /* depth is max of current depth and sub-depth + 1 */
        case node_call:{
                if (p->indirect) {
                    x = indirect_stack_usage;
                } else {
                    call_tree_2(p->first->first, &x);
                }
                if (!p->is_chained) {
                    x = x + 1;
                }
                break;
            }

            /* recurse subtrees */
        default:{
#undef for_each_subtree
#define for_each_subtree( t ) \
               if( p->t != NULL ){ call_tree_2( p->t, n ); }
                expand_for_all_subtrees break;
            }
        }

        /* save */
        p->data = x;
        p->generation = generation;
    }

    *n = max(*n, p->data);
}

int make_call_tree(tree p)
{
    int n = 0;
    generation++;
    call_tree_2(p, &n);
    return n;
}

void print_call_tree(FILE * f)
{
    string_list p = call_tree_root;
    while (p != NULL) {
        fprintf(f, "%s\n", p->s);
        p = p->next;
    }
}

tree clone2(tree p, boolean dup)
{
    stack_guard;
    if (p == NULL)
        return NULL;
    if (!dup) {
        p->duplicate = NULL;
    } else {
        if (p->duplicate == NULL) {
            /* p->duplicate = new_node( p ); */
        }
    }
    return p->duplicate;
}

tree clone(tree p)
{
    (void) clone2(p, false);
    return clone2(p, true);
}

/* count the number of assembler locations */
int code_size(tree p)
{
    int n = 0;
    stack_guard;
    while (p != NULL) {
#ifdef __DEBUG__
        log((m, "nr=%04d n=%d", p->nr, n));
#endif
        switch (p->kind) {
        case node_chain:
        case node_decl:
        case node_procedure:{
#ifdef __DEBUG__
                log((m, "before node=%d n=%d", p->nr, n));
#endif
                n += code_size(p->first);
                p = p->next;
                break;
            }
        case node_asm:{
#ifdef __DEBUG__
                log((m, "node=%d", p->nr));
#endif
                if (p->opcode == opcode_bank) {
                    if (target_cpu == pic_14) {
 /* ((target_chip == t_16f877) || (target_chip == t_16f876) || (target_chip == t_16f873)) {*/
  /* Added 16f876 & 16f873 by Javi 2003-03-01 */
                        return n + 2;
                    }
                }
                return n + 1;
                break;
            }
        case node_test:
        case node_org:
        case node_precall:
        case node_error:
        case node_var:
        case node_const:
        case node_ref:
        case node_w:
        case node_label:{
                return n;
                break;
            }
        default:
            snark_node(p->loc, p);
        }
    }
    return n;
}

/* determine whether the code sequence can be skipped
 * using a skip instruction
 */
int can_be_skipped(tree p)
{
    int n = 0;
    stack_guard;
    while (p != NULL) {
#ifdef __DEBUG__
        log((m, "nr=%04d n=%d", p->nr, n));
#endif
        switch (p->kind) {
        case node_chain:
        case node_decl:
        case node_procedure:{
#ifdef __DEBUG__
                log((m, "before node=%d n=%d", p->nr, n));
#endif
                n += code_size(p->first);
                p = p->next;
                break;
            }
        case node_asm:{
#ifdef __DEBUG__
                log((m, "node=%d", p->nr));
#endif
                if (p->opcode == opcode_bank) {
                    if (target_cpu == pic_14) {
 /* ((target_chip == t_16f877) || (target_chip == t_16f876) || (target_chip == t_16f873)) {*/
  /* Added 16f876 & 16f873 by Javi 2003-03-01 */
                       return n + 2;
                    }
                }
                if (p->opcode == opcode_a2nd) {
                    return n;
                }
                return n + 1;
                break;
            }
        case node_test:
        case node_org:
        case node_precall:
        case node_error:
        case node_var:
        case node_const:
        case node_ref:
        case node_w:
        case node_label:{
                return n;
                break;
            }
        default:
            snark_node(p->loc, p);
        }
    }
    return (n <= 1);
}


/* set register bank bits */
tree code_register_bank(tree p, tree x)
{
#ifdef __DEBUG__
    trace_subtree(p);
#endif
    if (!chip_without_memory_banks) {
        tree q = p;
        loc_t loc = NULL;
        if (x != NULL) {
            loc = x->loc;
        }
        if (q->kind == node_ref) {
            assert_pointer(q->loc, q->first);
            q = q->first;
        }
        return new_chain2(new_asm(loc, opcode_bank, q, 0), x);
    } else {
        return x;
    }
}

/* set code page bits */
tree code_code_page(tree p, tree x)
{
#ifdef __DEBUG__
    trace_subtree(p);
#endif
    if (!chip_without_code_pages) {
        tree q = p;
        if (q->kind == node_ref) {
            q = q->first;
        }
/* can also be a const from linline assembly */
#ifdef __DEBUG__
        assert_kind(p->loc, q, node_label);
#endif
        return new_chain2(new_asm(((x == NULL) ? p->loc : x->loc), opcode_page, q, 0), x);
    } else {
        return x;
    }
}

/********** scoping **********/

/* A declaration with a name which starts with a space
 * is used to delimit the build-in declarations which
 * can not be re-declared.
 *
 * A declaration with first_decl == true marks the beggining
 * of a scope.
 */

tree last_chain = NULL;
tree last_decl = NULL;
boolean first_decl = true;

/* add a node to the current list of statements */
void add(tree p)
{
    stack_guard;
    if (p == NULL)
        return;
    if (last_chain == NULL)
        last_chain = new_chain2(NULL, NULL);
    last_chain->next = new_chain2(p, NULL);
    last_chain = last_chain->next;
    if (p->kind == node_decl) {
        string_to_lowercase(p->first->name);
        p->prev_decl = last_decl;
        last_decl = p;
        p->first_decl = first_decl;
        first_decl = false;
    }
}

tree arg(tree p, int n, boolean must_find)
{
    stack_guard;
#ifdef __DEBUG__
    log((m, "n=%d", n));
    trace_subtree(p);
#endif
    assert_kind(p->loc, p, node_procedure);
/*    jal_assert(p->loc, n > 0); */
    assert_pointer(p->loc, n > 0);
    p = p->first;
    assert_pointer(NULL, p);

    /* skip dummy first if present */
    if (p->first == NULL) {
        p = p->next;
    }

    /* find the request argument */
    while (n > 1) {
        if (must_find) {
            assert_kind(NULL, p->next, node_chain);
        }
        if (p->next == NULL)
            return NULL;
        p = p->next;
#ifdef __DEBUG__
        log((m, "n=%d", n));
        trace_subtree(p);
#endif
        n--;
    }

#ifdef __DEBUG__
    trace_subtree(p);
#endif
    if (must_find) {
        assert_kind(NULL, p->first, node_decl);
    }
    if ((p->first == NULL) || (p->first->kind != node_decl)) {
        return NULL;
    }
    assert_kind(NULL, p->first->first, node_var);
    p = p->first->first;
#ifdef __DEBUG__
    trace_subtree(p);
#endif
    return p;
}

boolean operator_match(op_t op, tree left, tree right, tree p)
{
    stack_guard;
    assert_pointer(NULL, p);
    if ((p->kind != node_procedure) || (op != p->op)) {
        return false;
    }
    if (!is_type_compatible(left, arg(p, 1, true))) {
        return false;
    }
    if (is_monop(op)) {
        return true;
    }
    if (!is_type_compatible(right, arg(p, 2, true))) {
        return false;
    }
    if ((right == type_universal) || (left == type_universal)) {
        if (arg(p, 1, true) != arg(p, 2, true)) {
            return false;
        }
    }
    return true;
}

/* try to find a name in the indicated scope */
tree find_general(string ss, op_t op, tree left, tree right, boolean local, boolean use)
{
    tree q;
    string s;
    stack_guard;
    if (local && first_decl)
        use = false;
#ifdef __DEBUG__
    trace_subtree(last_decl)
        log((m, "find name='%s' op='%s'", s, op_name[op]));
    trace_subtree(left);
    trace_subtree(right);
#endif
    string_copy_lowercase(s, ss);
#ifdef __DEBUG__
    trace
#endif
        if (last_decl != NULL)
        for (q = last_decl; q != NULL; q = q->prev_decl) {
#ifdef __DEBUG__
            trace
#endif
                if (q->kind != node_decl) {
                trace_subtree(q);
                fflush(NULL);
            }
            assert_kind(NULL, q, node_decl);
#ifdef __DEBUG__
            log((m, "find n=%d x='%s'", q->first->nr, q->first->name));
#endif
            cassert((q->first->kind == node_const)
                    || (q->first->kind == node_type)
                    || (q->first->kind == node_var)
                    || (q->first->kind == node_procedure)
                );
            if (use) {
                if (op == 0) {
                    /* for a procedure, function, const or var the name must match */
                    if (string_equal(s, q->first->name)) {
                        return q->first;
                    }

                } else {
                    /* for an operator the op and types must match */
                    if (operator_match(op, left, right, q->first)) {
                        return q->first;
                    }
                }
            }
            if (local && q->first_decl)
                use = false;
            if (q->first->name[0] == ' ')
                use = true;
        }
#ifdef __DEBUG__
    trace
#endif
        return NULL;
}

tree find_local(string s)
{
    return find_general(s, 0, NULL, NULL, true, true);
}

tree find(string s)
{
    return find_general(s, 0, NULL, NULL, false, true);
}

tree find_local_operator(op_t op, tree left, tree right)
{
    return find_general("", op, left, right, true, true);
}

tree find_operator(op_t op, tree left, tree right)
{
    return find_general("", op, left, right, false, true);
}

/* check and add name to the current scope */
void check_and_add(tree p)
{
    tree q;
    stack_guard;
    assert_kind(NULL, p, node_decl);
    assert_pointer(NULL, p->first);
    if (p->first->op == 0) {
        q = find_local(p->first->name);
    } else {
        q = find_local_operator(p->first->op, arg(p->first, 1, false), arg(p->first, 2, false)
            );
    }
    if (q != NULL) {
#ifdef __DEBUG__
        trace_subtree(p);
        trace_subtree(q);
#endif
        /* t0030 */
        fatal(p->loc, (m, "identifier already declared in current scope"));
    }
    add(p);
}


/********** basic jal types **********/

tree type_tree(char *s, int bits, int large)
{
    tree p = new_node(NULL, node_type);
    p->name = new_string(s);
    p->bits = bits;
    p->large = large;
    return p;
}

void init_types(void)
{
    stack_guard;

    /* the jal build-in types */
    type_universal = type_tree(" universal", 0, 0);
    type_bit = type_tree("bit", 1, 1);
    type_byte = type_tree("byte", 8, 255);
    type_char = type_tree("char", 8, 255);
    type_word = type_tree("word", 16, 65535);
    type_long = type_tree("long", 32, 0);
    add(new_decl(NULL, type_universal));
    add(new_decl(NULL, type_bit));
    add(new_decl(NULL, type_byte));
    add(new_decl(NULL, type_char));
    add(new_decl(NULL, type_word));
    add(new_decl(NULL, type_long));

    /* the transfer variables for the indirect calls */
    transfer_bit = new_var(NULL, " transfer_bit", type_bit);
    transfer_byte = new_var(NULL, " transfer_byte", type_byte);
    add(new_decl(NULL, transfer_bit));
    add(new_decl(NULL, transfer_byte));
    transfer_bit->is_argument = true;
    transfer_byte->is_argument = true;

    /* a few usefull constants */
    const_zero = new_const(new_value(type_universal, 0));
    const_one = new_const(new_value(type_universal, 1));
    const_two = new_const(new_value(type_universal, 2));
}

/* return whether p is a basic integer type */
boolean is_buildin_type(tree p)
{
    stack_guard;
    assert_pointer(NULL, p);
    return (p == type_universal)
        || (p == type_bit)
        || (p == type_byte)
        || (p == type_char)
        || (p == type_word)
        || (p == type_long);
}

/* chop a value node according to its type */
void chop(tree p)
{
    stack_guard;
    if (p != NULL) {
        assert_kind(NULL, p, node_value);
        assert_kind(NULL, p->type, node_type);
        if (p->type->large != 0) {
            p->x &= p->type->large;
        }
    }
}
