/* char and string functions
   Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003
   Wouter van Ooijen

This file is part of jal.

jal is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

jal is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with jal; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"
#include "global.h"
#include "target.h"
#include "errorlh.h"
#include "stacksg.h"
#include "jalcomp.h"


/* return length of sring s (without the \0) */
int string_length(char *s)
{
    int i = 0;
    stack_guard;
    while (s[i] != '\0')
        i++;
    return i;
}

/* get one line from the indicated file */
void get_line(FILE * f, char *s, boolean * eof)
{
    stack_guard;
    for (;;) {
        int x = getc(f);
        *s = '\0';
        if (x == EOF) {
            if (eof != NULL)
                *eof = true;
            return;
        } else if (x != 10) {
            *s = x;
            s++;
        } else {
            return;
        }
    }
}

/* return uppercase of c */
char up_char(char c)
{
    stack_guard;
    if (c >= 'a' && (c <= 'z')) {
        return c - 'a' + 'A';
    } else {
        return c;
    }
}

#define up_char(x) ((((x)>='a')&&((x)<='z'))?((x)-'a'+'A'):(x))

/* return lowercase of c */
char low_char(char c)
{
    stack_guard;
    if (c >= 'A' && (c <= 'Z')) {
        return c - 'A' + 'a';
    } else {
        return c;
    }
}

/* report whether c1 and c2 are the same (ignore case) */
boolean char_match(char c1, char c2)
{
    stack_guard;
    return (c1 == c2) || (up_char(c1) == up_char(c2));
}

/* report whether s1 and s2 are the same (ignore case) */
boolean string_match(char *s1, char *s2)
{
    stack_guard;
    for (;;) {
        if ((*s1 == '\0') && (*s2 == '\0')) {
            return true;
        } else if (char_match(*s1, *s2)) {
            s1++;
            s2++;
        } else {
            return false;
        }
    }
}

/* report whether s1 and s2 are the same (watch case) */
boolean string_equal(char *s1, char *s2)
{
    stack_guard;
    for (;;) {
        if (*s1 == *s2) {
            if (*s1 == '\0')
                return true;
            s1++;
            s2++;
        } else {
            return false;
        }
    }
}

/* report whether x is a prefix of s */
boolean string_prefix(char *x, char *s)
{
    stack_guard;
    if (*x == '\0')
        return true;
    if (char_match(*x, *s)) {
        return string_prefix(x + 1, s + 1);
    }
    return false;
}

/* return index of first occurrence of x in s, -1 if none */
int string_pos(char *x, char *s)
{
    int i;
    stack_guard;
    for (i = 0; s[i] != '\0'; i++) {
        if (string_prefix(x, &s[i]))
            return i;
    }
    return -1;
}

/* give index of first appearance of c in x, -1 if none */
int char_pos(char c, char *s)
{
    int i = 0;
    stack_guard;
    while (s[i] != '\0') {
        if (s[i] == c)
            return i;
        i++;
    }
    return -1;
}

/* copy string s to d */
void string_copy(char *d, char *s)
{
    stack_guard;
    for (; *s != '\0';) {
        *d = *s;
        s++;
        d++;
    }
    *d = '\0';
}

/* convert string s in-place to lowercase */
void string_to_lowercase(char *s)
{
    stack_guard;
    for (; *s != '\0';) {
        *s = low_char(*s);
        s++;
    }
}

/* copy string s to d, convert to uppercase */
void string_copy_lowercase(char *d, char *s)
{
    stack_guard;
    for (; *s != '\0';) {
        *d = low_char(*s);
        s++;
        d++;
    }
    *d = '\0';
}

/* copy string s to d, skip chars which appear in x */
void string_copy_skip(char *d, char *s, char *x)
{
    stack_guard;
    for (; *s != '\0';) {
        *d = *s;
        if (char_pos(*s, x) < 0)
            d++;
        s++;
    }
    *d = '\0';
}

void replace_char(char *p, char old, char new)
{
    while (*p != '\0') {
        if (*p == old) {
            *p = new;
        }
        p++;
    }
}

void normalize_directory_separator(char *p)
{
    replace_char(p, '/', directory_separator);
    replace_char(p, '\\', directory_separator);
}

/* return int value of c assuming c is a hex char */
int char_hex_value(char c)
{
    char d = up_char(c);
    stack_guard;
    if ((d >= '0') & (d <= '9')) {
        return d - '0';
    } else if ((d >= 'A') & (d <= 'F')) {
        return d - 'A' + 10;
    } else {
        return -1;
    }
}

/* convert s to an integer value, report success */
boolean string_int_value(char *z, int *x)
{
    char c1, c2;
    int base;
    char *s = allocate(string_length(z) + 1);
    stack_guard;
    string_copy_skip(s, z, "_");
    if ((2 == sscanf(s, "%c%c", &c1, &c2))
        && (c1 == '0')
        && (c2 > '9')
        ) {
        char *p = &s[2];

        /* determine base */
        if (up_char(c2) == 'B') {
            base = 2;
        } else if (up_char(c2) == 'D') {
            base = 10;
        } else if (up_char(c2) == 'H') {
            base = 16;
        } else if (up_char(c2) == 'X') {
            base = 16;
        } else {
            return false;
        }

        /* determine value */
        *x = 0;
        while ((*p != '\0')) {
            int n = char_hex_value(*p);
            *x *= base;
            *x += n;
            if ((n < 0) | (n >= base))
                return false;
            p++;
        }

        /* no explicit value, use sscanf */
    } else {
        int n;
        n = sscanf(s, "%d%c", x, &c1);
        if (n != 1)
            return false;
    }
    return true;
}

/* return alloacted copy of s */
char *new_string(char *s)
{
    char *p = allocate(string_length(s) + 1);
    stack_guard;
    string_copy(p, s);
    return p;
}

/* return alloacted copy of s */
char *new_string_lowercase(char *s)
{
    char *p = allocate(string_length(s) + 1);
    stack_guard;
    string_copy_lowercase(p, s);
    return p;
}

/* return allocated string of n c's */
char *bar(int n, char c)
{
    char *p;
    int i;
    stack_guard;
    if (n < 0) {
        n = 0;
    }
    p = allocate(n + 1);
    for (i = 0; i < n; i++)
        p[i] = c;
    p[n] = '\0';
    return p;
}

/* return allocated string s, padded by c's up to length n */
char *pad(char *s, unsigned n, char c)
{
    char *p;
    unsigned i;
    stack_guard;
#ifdef __DEBUG__
    if (n < 0)
        n = 0;
#endif
    p = allocate(max(strlen(s), n) + 1);
    for (i = 0; i < strlen(s); i++)
        p[i] = s[i];
    for (; i < n; i++)
        p[i] = c;
    p[i] = '\0';
    return p;
}

/* replace '.' with '\0' */
void chop_extension(char *s)
{
    char *p;
    stack_guard;
    
	for (p = s + strlen(s); p != s; p--) {
		if (*p == '.') {
			*p = '\0';
			break;
		}
	}
}

void chop_path_tail(char *s)
{
    char *p;
    stack_guard;
    
	for (p = s + strlen(s); p != s; p--) {
        if (*p == directory_separator) {
			*p = '\0';
			break;
		}
    }
}


void chop_path(char *s)
{
    char *p, *q;
    stack_guard;

    q = s;
    for (p = s; *p != '\0'; p++) {
        if (*p == directory_separator)
            q = p + 1;
    }

    string_copy(s, q);
}

/* report whether file s exists */
boolean file_exists(char *s)
{
    FILE *f;
    stack_guard;
    f = fopen(s, "r");
    if (f == NULL) {
        return false;
    }
    fclose(f);
    return true;
}


/********** char classification functions **********/

/* report whether c is a separator */
boolean is_separator(char c)
{
    stack_guard;
    return c <= ' ';
}

/* report whether c is a letter */
boolean is_letter(char c)
{
    stack_guard;
    return (((c >= 'a') && (c <= 'z'))
            || ((c >= 'A') && (c <= 'Z'))
        );
}

/* report whether c is a number */
boolean is_number(char c)
{
    stack_guard;
    return ((c >= '0')
            && (c <= '9')
        );
}

/* report whether c is valid as first character of an identifier */
boolean is_id_start(char c)
{
    stack_guard;
    return (is_letter(c)
            || (c == '_')
            /* || ( c == '.' ) removed 14-APR-1999 */
        );
}

/* report whether c is valid as next character of an identifier */
boolean is_id_next(char c)
{
    stack_guard;
    return (is_id_start(c)
            || is_number(c)
        );
}

/* report whether c is valid as first character of a constant */
boolean is_number_start(char c)
{
    stack_guard;
    return (is_number(c)
        );
}

/* report whether c is valid as next character of a constant */
boolean is_number_next(char c)
{
    stack_guard;
    return (is_letter(c)
            || is_number(c)
            || (c == '_')
            || (c == '.')
        );
}

/* report whether c is a special (operator) character */
boolean is_special(char c)
{
    string s = "!:=></+-*%^|&._\\";
    int i;
    stack_guard;
    for (i = 0;; i++) {
        if (s[i] == c)
            return true;
        if (s[i] == '\0')
            return false;
    }
}


/* search list handling */
#define slist struct _slist
struct _slist {
    slist *next;
    char *s;
};
slist slist_first = {
    NULL,
    ""
};
void add_to_searchlist(string s)
{
    string m;
    slist *p = allocate(sizeof(slist));
    sprintf(m, "%s%c", s, directory_separator);
    p->s = new_string(m);
    normalize_directory_separator(p->s);
    p->next = slist_first.next;
    slist_first.next = p;
}
boolean find_name2(char *dest, char *source)
{

    /* try unchanged name */
    sprintf(dest, "%s", source);
    if (file_exists(dest)) {
        return true;
    }

    /* try name.jal */
    sprintf(dest, "%s.jal", source);
    if (file_exists(dest)) {
        return true;
    }

    return false;
}

boolean find_name(char *dest, char *source)
{
    string s;

    /* try all names in the searchlist */
    slist *p = &slist_first;
    normalize_directory_separator(source);
    while (p != NULL) {
        sprintf(s, "%s%s", p->s, source);
        if (find_name2(dest, s)) {
            return true;
        }
        p = p->next;
    }

    return false;
}


/********** date and time **********/

float current_time(void)
{
#ifdef _MSC_VER
    clock_t x = clock();
    return (float) ((1.0 * x) / CLOCKS_PER_SEC);
#else
    clock_t x = clock();
    return (float) ((1.0 * x) / CLOCKS_PER_SEC);
#endif
}


string _date_and_time;
char *date_and_time(void)
{
    time_t t;
#ifdef __DEBUG__
    return "(NULL)";
#endif
    t = time(NULL);
    strftime(_date_and_time, sizeof(_date_and_time) - 1, "%d-%b-%Y %H:%M:%S", localtime(&t)
        );
    return _date_and_time;
}
