// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package merchantapi provides access to the Merchant API.
//
// For product documentation, see: https://developers.devsite.corp.google.com/merchant/api
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/merchantapi/accounts_v1beta"
//	...
//	ctx := context.Background()
//	merchantapiService, err := merchantapi.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	merchantapiService, err := merchantapi.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	merchantapiService, err := merchantapi.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package merchantapi // import "google.golang.org/api/merchantapi/accounts_v1beta"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "merchantapi:accounts_v1beta"
const apiName = "merchantapi"
const apiVersion = "accounts_v1beta"
const basePath = "https://merchantapi.googleapis.com/"
const basePathTemplate = "https://merchantapi.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://merchantapi.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// Manage your product listings and accounts for Google Shopping
	ContentScope = "https://www.googleapis.com/auth/content"
)

// NewService creates a new APIService.
func NewService(ctx context.Context, opts ...option.ClientOption) (*APIService, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/content",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &APIService{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Accounts = NewAccountsService(s)
	s.TermsOfService = NewTermsOfServiceService(s)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new APIService. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*APIService, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.Background(), option.WithHTTPClient(client))
}

type APIService struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Accounts *AccountsService

	TermsOfService *TermsOfServiceService
}

func (s *APIService) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewAccountsService(s *APIService) *AccountsService {
	rs := &AccountsService{s: s}
	rs.AutofeedSettings = NewAccountsAutofeedSettingsService(s)
	rs.BusinessIdentity = NewAccountsBusinessIdentityService(s)
	rs.BusinessInfo = NewAccountsBusinessInfoService(s)
	rs.EmailPreferences = NewAccountsEmailPreferencesService(s)
	rs.Homepage = NewAccountsHomepageService(s)
	rs.Issues = NewAccountsIssuesService(s)
	rs.OnlineReturnPolicies = NewAccountsOnlineReturnPoliciesService(s)
	rs.Programs = NewAccountsProgramsService(s)
	rs.Regions = NewAccountsRegionsService(s)
	rs.ShippingSettings = NewAccountsShippingSettingsService(s)
	rs.TermsOfServiceAgreementStates = NewAccountsTermsOfServiceAgreementStatesService(s)
	rs.Users = NewAccountsUsersService(s)
	return rs
}

type AccountsService struct {
	s *APIService

	AutofeedSettings *AccountsAutofeedSettingsService

	BusinessIdentity *AccountsBusinessIdentityService

	BusinessInfo *AccountsBusinessInfoService

	EmailPreferences *AccountsEmailPreferencesService

	Homepage *AccountsHomepageService

	Issues *AccountsIssuesService

	OnlineReturnPolicies *AccountsOnlineReturnPoliciesService

	Programs *AccountsProgramsService

	Regions *AccountsRegionsService

	ShippingSettings *AccountsShippingSettingsService

	TermsOfServiceAgreementStates *AccountsTermsOfServiceAgreementStatesService

	Users *AccountsUsersService
}

func NewAccountsAutofeedSettingsService(s *APIService) *AccountsAutofeedSettingsService {
	rs := &AccountsAutofeedSettingsService{s: s}
	return rs
}

type AccountsAutofeedSettingsService struct {
	s *APIService
}

func NewAccountsBusinessIdentityService(s *APIService) *AccountsBusinessIdentityService {
	rs := &AccountsBusinessIdentityService{s: s}
	return rs
}

type AccountsBusinessIdentityService struct {
	s *APIService
}

func NewAccountsBusinessInfoService(s *APIService) *AccountsBusinessInfoService {
	rs := &AccountsBusinessInfoService{s: s}
	return rs
}

type AccountsBusinessInfoService struct {
	s *APIService
}

func NewAccountsEmailPreferencesService(s *APIService) *AccountsEmailPreferencesService {
	rs := &AccountsEmailPreferencesService{s: s}
	return rs
}

type AccountsEmailPreferencesService struct {
	s *APIService
}

func NewAccountsHomepageService(s *APIService) *AccountsHomepageService {
	rs := &AccountsHomepageService{s: s}
	return rs
}

type AccountsHomepageService struct {
	s *APIService
}

func NewAccountsIssuesService(s *APIService) *AccountsIssuesService {
	rs := &AccountsIssuesService{s: s}
	return rs
}

type AccountsIssuesService struct {
	s *APIService
}

func NewAccountsOnlineReturnPoliciesService(s *APIService) *AccountsOnlineReturnPoliciesService {
	rs := &AccountsOnlineReturnPoliciesService{s: s}
	return rs
}

type AccountsOnlineReturnPoliciesService struct {
	s *APIService
}

func NewAccountsProgramsService(s *APIService) *AccountsProgramsService {
	rs := &AccountsProgramsService{s: s}
	return rs
}

type AccountsProgramsService struct {
	s *APIService
}

func NewAccountsRegionsService(s *APIService) *AccountsRegionsService {
	rs := &AccountsRegionsService{s: s}
	return rs
}

type AccountsRegionsService struct {
	s *APIService
}

func NewAccountsShippingSettingsService(s *APIService) *AccountsShippingSettingsService {
	rs := &AccountsShippingSettingsService{s: s}
	return rs
}

type AccountsShippingSettingsService struct {
	s *APIService
}

func NewAccountsTermsOfServiceAgreementStatesService(s *APIService) *AccountsTermsOfServiceAgreementStatesService {
	rs := &AccountsTermsOfServiceAgreementStatesService{s: s}
	return rs
}

type AccountsTermsOfServiceAgreementStatesService struct {
	s *APIService
}

func NewAccountsUsersService(s *APIService) *AccountsUsersService {
	rs := &AccountsUsersService{s: s}
	return rs
}

type AccountsUsersService struct {
	s *APIService
}

func NewTermsOfServiceService(s *APIService) *TermsOfServiceService {
	rs := &TermsOfServiceService{s: s}
	return rs
}

type TermsOfServiceService struct {
	s *APIService
}

// Accepted: Describes the accepted terms of service.
type Accepted struct {
	// AcceptedBy: The account where the acceptance was recorded. This can be the
	// account itself or, in the case of subaccounts, the MCA account.
	AcceptedBy string `json:"acceptedBy,omitempty"`
	// TermsOfService: The accepted termsOfService
	// (google.shopping.merchant.accounts.v1main.TermsOfService).
	TermsOfService string `json:"termsOfService,omitempty"`
	// ValidUntil: When set, it states that the accepted TermsOfService
	// (google.shopping.merchant.accounts.v1main.TermsOfService) is only valid
	// until the end of this date (in UTC). A new one must be accepted before then.
	// The information of the required TermsOfService
	// (google.shopping.merchant.accounts.v1main.TermsOfService) is found in the
	// Required (Required) message.
	ValidUntil *Date `json:"validUntil,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AcceptedBy") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AcceptedBy") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Accepted) MarshalJSON() ([]byte, error) {
	type NoMethod Accepted
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Account: The `Account` message represents a merchant's account within
// Shopping Ads. It's the primary entity for managing product data, settings,
// and interactions with Google's services and external providers. Accounts can
// operate as standalone entities or be part of a multi-client account (MCA)
// structure. In an MCA setup the parent account manages multiple sub-accounts.
// Establishing an account involves configuring attributes like the account
// name, time zone, and language preferences. The `Account` message is the
// parent entity for many other resources, for example, `AccountRelationship`,
// `Homepage`, `BusinessInfo` and so on.
type Account struct {
	// AccountId: Output only. The ID of the account.
	AccountId int64 `json:"accountId,omitempty,string"`
	// AccountName: Required. A human-readable name of the account. See store name
	// (https://support.google.com/merchants/answer/160556) and business name
	// (https://support.google.com/merchants/answer/12159159) for more information.
	AccountName string `json:"accountName,omitempty"`
	// AdultContent: Whether this account contains adult content.
	AdultContent bool `json:"adultContent,omitempty"`
	// LanguageCode: Required. The account's BCP-47 language code
	// (https://tools.ietf.org/html/bcp47), such as `en-US` or `sr-Latn`.
	LanguageCode string `json:"languageCode,omitempty"`
	// Name: Identifier. The resource name of the account. Format:
	// `accounts/{account}`
	Name string `json:"name,omitempty"`
	// TestAccount: Output only. Whether this is a test account.
	TestAccount bool `json:"testAccount,omitempty"`
	// TimeZone: Required. The time zone of the account. On writes, `time_zone`
	// sets both the `reporting_time_zone` and the `display_time_zone`. For reads,
	// `time_zone` always returns the `display_time_zone`. If `display_time_zone`
	// doesn't exist for your account, `time_zone` is empty.
	TimeZone *TimeZone `json:"timeZone,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccountId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccountId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Account) MarshalJSON() ([]byte, error) {
	type NoMethod Account
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AccountAggregation: `AccountAggregation` payload.
type AccountAggregation struct {
}

// AccountIssue: An `AccountIssue`
// (https://support.google.com/merchants/answer/12153802?sjid=17798438912526418908-EU#account).
type AccountIssue struct {
	// Detail: Further localized details about the issue.
	Detail string `json:"detail,omitempty"`
	// DocumentationUri: Link to Merchant Center Help Center providing further
	// information about the issue and how to fix it.
	DocumentationUri string `json:"documentationUri,omitempty"`
	// ImpactedDestinations: The impact this issue has on various destinations.
	ImpactedDestinations []*ImpactedDestination `json:"impactedDestinations,omitempty"`
	// Name: Identifier. The resource name of the account issue. Format:
	// `accounts/{account}/issues/{id}`
	Name string `json:"name,omitempty"`
	// Severity: The overall severity of the issue.
	//
	// Possible values:
	//   "SEVERITY_UNSPECIFIED" - The severity is unknown.
	//   "CRITICAL" - The issue causes offers to not serve.
	//   "ERROR" - The issue might affect offers (in the future) or might be an
	// indicator of issues with offers.
	//   "SUGGESTION" - The issue is a suggestion for improvement.
	Severity string `json:"severity,omitempty"`
	// Title: The localized title of the issue.
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Detail") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Detail") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccountIssue) MarshalJSON() ([]byte, error) {
	type NoMethod AccountIssue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AddAccountService: Additional instructions to add account services during
// creation of the account.
type AddAccountService struct {
	// AccountAggregation: The provider is an aggregator
	// (https://support.google.com/merchants/answer/188487) for the account.
	// Payload for service type Account Aggregation.
	AccountAggregation *AccountAggregation `json:"accountAggregation,omitempty"`
	// Provider: Required. The provider of the service. Either the reference to an
	// account such as `providers/123` or a well-known service provider (one of
	// `providers/GOOGLE_ADS` or `providers/GOOGLE_BUSINESS_PROFILE`).
	Provider string `json:"provider,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AccountAggregation") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccountAggregation") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AddAccountService) MarshalJSON() ([]byte, error) {
	type NoMethod AddAccountService
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Address: Shipping address of the warehouse.
type Address struct {
	// AdministrativeArea: Required. Top-level administrative subdivision of the
	// country. For example, a state like California ("CA") or a province like
	// Quebec ("QC").
	AdministrativeArea string `json:"administrativeArea,omitempty"`
	// City: Required. City, town or commune. May also include dependent localities
	// or sublocalities (For example neighborhoods or suburbs).
	City string `json:"city,omitempty"`
	// PostalCode: Required. Postal code or ZIP (For example "94043").
	PostalCode string `json:"postalCode,omitempty"`
	// RegionCode: Required. CLDR country code
	// (http://www.unicode.org/repos/cldr/tags/latest/common/main/en.xml) (For
	// example "US").
	RegionCode string `json:"regionCode,omitempty"`
	// StreetAddress: Street-level part of the address. For example: `111w 31st
	// Street`.
	StreetAddress string `json:"streetAddress,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdministrativeArea") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdministrativeArea") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Address) MarshalJSON() ([]byte, error) {
	type NoMethod Address
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AutofeedSettings: Collection of information related to the autofeed
// (https://support.google.com/merchants/answer/7538732) settings.
type AutofeedSettings struct {
	// Eligible: Output only. Determines whether merchant is eligible for being
	// enrolled into an autofeed.
	Eligible bool `json:"eligible,omitempty"`
	// EnableProducts: Required. Enables or disables product crawling through the
	// autofeed for the given account. Autofeed accounts must meet certain
	// conditions
	// (https://support.google.com/merchants/answer/7538732#Configure_automated_feeds_Standard_Experience),
	// which can be checked through the `eligible` field. The account must **not**
	// be a marketplace. When the autofeed is enabled for the first time, the
	// products usually appear instantly. When re-enabling, it might take up to 24
	// hours for products to appear.
	EnableProducts bool `json:"enableProducts,omitempty"`
	// Name: Identifier. The resource name of the autofeed settings. Format:
	// `accounts/{account}/autofeedSettings`.
	Name string `json:"name,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Eligible") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Eligible") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AutofeedSettings) MarshalJSON() ([]byte, error) {
	type NoMethod AutofeedSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// BusinessDayConfig: Business days of the warehouse.
type BusinessDayConfig struct {
	// BusinessDays: Required. Regular business days. May not be empty.
	//
	// Possible values:
	//   "WEEKDAY_UNSPECIFIED"
	//   "MONDAY"
	//   "TUESDAY"
	//   "WEDNESDAY"
	//   "THURSDAY"
	//   "FRIDAY"
	//   "SATURDAY"
	//   "SUNDAY"
	BusinessDays []string `json:"businessDays,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BusinessDays") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BusinessDays") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s BusinessDayConfig) MarshalJSON() ([]byte, error) {
	type NoMethod BusinessDayConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// BusinessIdentity: Collection of information related to the identity of a
// business (https://support.google.com/merchants/answer/12564247).
type BusinessIdentity struct {
	// BlackOwned: Optional. Specifies whether the business identifies itself as
	// being black-owned. This optional field will only be available for merchants
	// with a business country set to `US`. It is also not applicable for
	// marketplaces or marketplace sellers.
	BlackOwned *IdentityAttribute `json:"blackOwned,omitempty"`
	// LatinoOwned: Optional. Specifies whether the business identifies itself as
	// being latino-owned. This optional field will only be available for merchants
	// with a business country set to `US`. It is also not applicable for
	// marketplaces or marketplace sellers.
	LatinoOwned *IdentityAttribute `json:"latinoOwned,omitempty"`
	// Name: Identifier. The resource name of the business identity. Format:
	// `accounts/{account}/businessIdentity`
	Name string `json:"name,omitempty"`
	// PromotionsConsent: Required. Whether the identity attributes may be used for
	// promotions.
	//
	// Possible values:
	//   "PROMOTIONS_CONSENT_UNSPECIFIED" - Default value indicating that no
	// selection was made.
	//   "PROMOTIONS_CONSENT_GIVEN" - Indicates that the account consented to
	// having their business identity used for promotions.
	//   "PROMOTIONS_CONSENT_DENIED" - Indicates that the account did not consent
	// to having their business identity used for promotions.
	PromotionsConsent string `json:"promotionsConsent,omitempty"`
	// SmallBusiness: Optional. Specifies whether the business identifies itself as
	// a small business. This optional field will only be available for merchants
	// with a business country set to `US`. It is also not applicable for
	// marketplaces.
	SmallBusiness *IdentityAttribute `json:"smallBusiness,omitempty"`
	// VeteranOwned: Optional. Specifies whether the business identifies itself as
	// being veteran-owned. This optional field will only be available for
	// merchants with a business country set to `US`. It is also not applicable for
	// marketplaces or marketplace sellers.
	VeteranOwned *IdentityAttribute `json:"veteranOwned,omitempty"`
	// WomenOwned: Optional. Specifies whether the business identifies itself as
	// being women-owned. This optional field will only be available for merchants
	// with a business country set to `US`. It is also not applicable for
	// marketplaces or marketplace sellers.
	WomenOwned *IdentityAttribute `json:"womenOwned,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "BlackOwned") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BlackOwned") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s BusinessIdentity) MarshalJSON() ([]byte, error) {
	type NoMethod BusinessIdentity
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// BusinessInfo: The `BusinessInfo` message contains essential information
// about a merchant's business. This message captures key business details such
// as physical address, customer service contacts, and region-specific
// identifiers.
type BusinessInfo struct {
	// Address: Optional. The address of the business. Only `region_code`,
	// `address_lines`, `postal_code`, `administrative_area` and `locality` fields
	// are supported. All other fields are ignored.
	Address *PostalAddress `json:"address,omitempty"`
	// CustomerService: Optional. The customer service of the business.
	CustomerService *CustomerService `json:"customerService,omitempty"`
	// KoreanBusinessRegistrationNumber: Optional. The 10-digit Korean business
	// registration number (https://support.google.com/merchants/answer/9037766)
	// separated with dashes in the format: XXX-XX-XXXXX.
	KoreanBusinessRegistrationNumber string `json:"koreanBusinessRegistrationNumber,omitempty"`
	// Name: Identifier. The resource name of the business info. Format:
	// `accounts/{account}/businessInfo`
	Name string `json:"name,omitempty"`
	// Phone: Output only. The phone number of the business.
	Phone *PhoneNumber `json:"phone,omitempty"`
	// PhoneVerificationState: Output only. The phone verification state of the
	// business.
	//
	// Possible values:
	//   "PHONE_VERIFICATION_STATE_UNSPECIFIED" - Default value. This value is
	// unused.
	//   "PHONE_VERIFICATION_STATE_VERIFIED" - The phone is verified.
	//   "PHONE_VERIFICATION_STATE_UNVERIFIED" - The phone is unverified
	PhoneVerificationState string `json:"phoneVerificationState,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Address") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Address") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s BusinessInfo) MarshalJSON() ([]byte, error) {
	type NoMethod BusinessInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CarrierRate: A list of carrier rates that can be referred to by `main_table`
// or `single_value`. Supported carrier services are defined in
// https://support.google.com/merchants/answer/12577710?ref_topic=12570808&sjid=10662598224319463032-NC#zippy=%2Cdelivery-cost-rate-type%2Ccarrier-rate-au-de-uk-and-us-only.
type CarrierRate struct {
	// Carrier: Required. Carrier service, such as "UPS" or "Fedex".
	Carrier string `json:"carrier,omitempty"`
	// CarrierService: Required. Carrier service, such as "ground" or "2 days".
	CarrierService string `json:"carrierService,omitempty"`
	// FlatAdjustment: Optional. Additive shipping rate modifier. Can be negative.
	// For example `{ "amount_micros": 1, "currency_code" : "USD" }` adds $1 to the
	// rate, `{ "amount_micros": -3, "currency_code" : "USD" }` removes $3 from the
	// rate.
	FlatAdjustment *Price `json:"flatAdjustment,omitempty"`
	// Name: Required. Name of the carrier rate. Must be unique per rate group.
	Name string `json:"name,omitempty"`
	// OriginPostalCode: Required. Shipping origin for this carrier rate.
	OriginPostalCode string `json:"originPostalCode,omitempty"`
	// PercentageAdjustment: Optional. Multiplicative shipping rate modifier as a
	// number in decimal notation. Can be negative. For example "5.4" increases
	// the rate by 5.4%, "-3" decreases the rate by 3%.
	PercentageAdjustment string `json:"percentageAdjustment,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Carrier") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Carrier") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CarrierRate) MarshalJSON() ([]byte, error) {
	type NoMethod CarrierRate
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ClaimHomepageRequest: Request message for the `ClaimHomepage` method.
type ClaimHomepageRequest struct {
	// Overwrite: Optional. When set to `true`, this option removes any existing
	// claim on the requested website and replaces it with a claim from the account
	// that makes the request.
	Overwrite bool `json:"overwrite,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Overwrite") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Overwrite") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ClaimHomepageRequest) MarshalJSON() ([]byte, error) {
	type NoMethod ClaimHomepageRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CreateAndConfigureAccountRequest: Request message for the
// `CreateAndConfigureAccount` method.
type CreateAndConfigureAccountRequest struct {
	// Account: Required. The account to be created.
	Account *Account `json:"account,omitempty"`
	// Service: Required. An account service between the account to be created and
	// the provider account is initialized as part of the creation. At least one
	// such service needs to be provided. Currently exactly one of these needs to
	// be `account_aggregation`, which means you can only create sub accounts, not
	// standalone account through this method. Additional `account_management` or
	// `product_management` services may be provided.
	Service []*AddAccountService `json:"service,omitempty"`
	// Users: Optional. Users to be added to the account.
	Users []*CreateUserRequest `json:"users,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Account") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Account") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CreateAndConfigureAccountRequest) MarshalJSON() ([]byte, error) {
	type NoMethod CreateAndConfigureAccountRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CreateUserRequest: Request message for the `CreateUser` method.
type CreateUserRequest struct {
	// Parent: Required. The resource name of the account for which a user will be
	// created. Format: `accounts/{account}`
	Parent string `json:"parent,omitempty"`
	// User: Optional. The user to create.
	User *User `json:"user,omitempty"`
	// UserId: Required. The email address of the user (for example,
	// `john.doe@gmail.com`).
	UserId string `json:"userId,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Parent") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Parent") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CreateUserRequest) MarshalJSON() ([]byte, error) {
	type NoMethod CreateUserRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CustomerService: Customer service information.
type CustomerService struct {
	// Email: Optional. The email address where customer service may be reached.
	Email string `json:"email,omitempty"`
	// Phone: Optional. The phone number where customer service may be called.
	Phone *PhoneNumber `json:"phone,omitempty"`
	// Uri: Optional. The URI where customer service may be found.
	Uri string `json:"uri,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Email") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Email") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CustomerService) MarshalJSON() ([]byte, error) {
	type NoMethod CustomerService
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CutoffConfig: Configs related to local delivery ends for the day.
type CutoffConfig struct {
	// LocalCutoffTime: Time that local delivery ends for the day.
	LocalCutoffTime *LocalCutoffTime `json:"localCutoffTime,omitempty"`
	// NoDeliveryPostCutoff: Merchants can opt-out of showing n+1 day local
	// delivery when they have a shipping service configured to n day local
	// delivery. For example, if the shipping service defines same-day delivery,
	// and it's past the cut-off, setting this field to `true` results in the
	// calculated shipping service rate returning `NO_DELIVERY_POST_CUTOFF`. In the
	// same example, setting this field to `false` results in the calculated
	// shipping time being one day. This is only for local delivery.
	NoDeliveryPostCutoff bool `json:"noDeliveryPostCutoff,omitempty"`
	// StoreCloseOffsetHours: Only valid with local delivery fulfillment.
	// Represents cutoff time as the number of hours before store closing. Mutually
	// exclusive with `local_cutoff_time`.
	StoreCloseOffsetHours int64 `json:"storeCloseOffsetHours,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "LocalCutoffTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LocalCutoffTime") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CutoffConfig) MarshalJSON() ([]byte, error) {
	type NoMethod CutoffConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CutoffTime: Business days cutoff time definition.
type CutoffTime struct {
	// Hour: Required. Hour of the cutoff time until which an order has to be
	// placed to be processed in the same day.
	Hour int64 `json:"hour,omitempty"`
	// Minute: Required. Minute of the cutoff time until which an order has to be
	// placed to be processed in the same day.
	Minute int64 `json:"minute,omitempty"`
	// TimeZone: Required. Timezone identifier
	// (https://developers.google.com/adwords/api/docs/appendix/codes-formats#timezone-ids)
	// For example "Europe/Zurich".
	TimeZone string `json:"timeZone,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Hour") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Hour") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CutoffTime) MarshalJSON() ([]byte, error) {
	type NoMethod CutoffTime
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Date: Represents a whole or partial calendar date, such as a birthday. The
// time of day and time zone are either specified elsewhere or are
// insignificant. The date is relative to the Gregorian Calendar. This can
// represent one of the following: * A full date, with non-zero year, month,
// and day values. * A month and day, with a zero year (for example, an
// anniversary). * A year on its own, with a zero month and a zero day. * A
// year and month, with a zero day (for example, a credit card expiration
// date). Related types: * google.type.TimeOfDay * google.type.DateTime *
// google.protobuf.Timestamp
type Date struct {
	// Day: Day of a month. Must be from 1 to 31 and valid for the year and month,
	// or 0 to specify a year by itself or a year and month where the day isn't
	// significant.
	Day int64 `json:"day,omitempty"`
	// Month: Month of a year. Must be from 1 to 12, or 0 to specify a year without
	// a month and day.
	Month int64 `json:"month,omitempty"`
	// Year: Year of the date. Must be from 1 to 9999, or 0 to specify a date
	// without a year.
	Year int64 `json:"year,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Day") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Day") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Date) MarshalJSON() ([]byte, error) {
	type NoMethod Date
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DeliveryTime: Time spent in various aspects from order to the delivery of
// the product.
type DeliveryTime struct {
	// CutoffTime: Business days cutoff time definition. If not configured the
	// cutoff time will be defaulted to 8AM PST.
	CutoffTime *CutoffTime `json:"cutoffTime,omitempty"`
	// HandlingBusinessDayConfig: The business days during which orders can be
	// handled. If not provided, Monday to Friday business days will be assumed.
	HandlingBusinessDayConfig *BusinessDayConfig `json:"handlingBusinessDayConfig,omitempty"`
	// MaxHandlingDays: Maximum number of business days spent before an order is
	// shipped. 0 means same day shipped, 1 means next day shipped. Must be greater
	// than or equal to `min_handling_days`. 'min_handling_days' and
	// 'max_handling_days' should be either set or not set at the same time.
	MaxHandlingDays int64 `json:"maxHandlingDays,omitempty"`
	// MaxTransitDays: Maximum number of business days that is spent in transit. 0
	// means same day delivery, 1 means next day delivery. Must be greater than or
	// equal to `min_transit_days`.
	MaxTransitDays int64 `json:"maxTransitDays,omitempty"`
	// MinHandlingDays: Minimum number of business days spent before an order is
	// shipped. 0 means same day shipped, 1 means next day shipped.
	// 'min_handling_days' and 'max_handling_days' should be either set or not set
	// at the same time.
	MinHandlingDays int64 `json:"minHandlingDays,omitempty"`
	// MinTransitDays: Minimum number of business days that is spent in transit. 0
	// means same day delivery, 1 means next day delivery. Either
	// `min_transit_days`, `max_transit_days` or `transit_time_table` must be set,
	// but not both.
	MinTransitDays int64 `json:"minTransitDays,omitempty"`
	// TransitBusinessDayConfig: The business days during which orders can be
	// in-transit. If not provided, Monday to Friday business days will be assumed.
	TransitBusinessDayConfig *BusinessDayConfig `json:"transitBusinessDayConfig,omitempty"`
	// TransitTimeTable: Transit time table, number of business days spent in
	// transit based on row and column dimensions. Either `min_transit_days`,
	// `max_transit_days` or `transit_time_table` can be set, but not both.
	TransitTimeTable *TransitTable `json:"transitTimeTable,omitempty"`
	// WarehouseBasedDeliveryTimes: Optional. Indicates that the delivery time
	// should be calculated per warehouse (shipping origin location) based on the
	// settings of the selected carrier. When set, no other transit time related
	// field in delivery time should be set.
	WarehouseBasedDeliveryTimes []*WarehouseBasedDeliveryTime `json:"warehouseBasedDeliveryTimes,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CutoffTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CutoffTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DeliveryTime) MarshalJSON() ([]byte, error) {
	type NoMethod DeliveryTime
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DisableProgramRequest: Request message for the DisableProgram method.
type DisableProgramRequest struct {
}

// Distance: Maximum delivery radius. This is only required for the local
// delivery shipment type.
type Distance struct {
	// Unit: Unit can differ based on country, it is parameterized to include miles
	// and kilometers.
	//
	// Possible values:
	//   "UNIT_UNSPECIFIED" - Unit unspecified
	//   "MILES" - Unit in miles
	//   "KILOMETERS" - Unit in kilometers
	Unit string `json:"unit,omitempty"`
	// Value: Integer value of distance.
	Value int64 `json:"value,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Distance) MarshalJSON() ([]byte, error) {
	type NoMethod Distance
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// EmailPreferences: The categories of notifications the user opted into /
// opted out of. The email preferences do not include mandatory announcements
// as users can't opt out of them.
type EmailPreferences struct {
	// Name: Identifier. The name of the EmailPreferences. The endpoint is only
	// supported for the authenticated user.
	Name string `json:"name,omitempty"`
	// NewsAndTips: Optional. Updates on new features, tips and best practices.
	//
	// Possible values:
	//   "OPT_IN_STATE_UNSPECIFIED" - Opt-in status is not specified.
	//   "OPTED_OUT" - User has opted out of receiving this type of email.
	//   "OPTED_IN" - User has opted in to receiving this type of email.
	//   "UNCONFIRMED" - User has opted in to receiving this type of email and the
	// confirmation email has been sent, but user has not yet confirmed the opt in
	// (applies only to certain countries).
	NewsAndTips string `json:"newsAndTips,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Name") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Name") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s EmailPreferences) MarshalJSON() ([]byte, error) {
	type NoMethod EmailPreferences
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// EnableProgramRequest: Request message for the EnableProgram method.
type EnableProgramRequest struct {
}

// GeoTargetArea: A list of geotargets that defines the region area.
type GeoTargetArea struct {
	// GeotargetCriteriaIds: Required. A non-empty list of location IDs
	// (https://developers.google.com/adwords/api/docs/appendix/geotargeting). They
	// must all be of the same location type (for example, state).
	GeotargetCriteriaIds googleapi.Int64s `json:"geotargetCriteriaIds,omitempty"`
	// ForceSendFields is a list of field names (e.g. "GeotargetCriteriaIds") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "GeotargetCriteriaIds") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GeoTargetArea) MarshalJSON() ([]byte, error) {
	type NoMethod GeoTargetArea
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Headers: A non-empty list of row or column headers for a table. Exactly one
// of `prices`, `weights`, `num_items`, `postal_code_group_names`, or
// `location` must be set.
type Headers struct {
	// Locations: Required. A list of location ID sets. Must be non-empty. Can only
	// be set if all other fields are not set.
	Locations []*LocationIdSet `json:"locations,omitempty"`
	// NumberOfItems: Required. A list of inclusive number of items upper bounds.
	// The last value can be "infinity". For example `["10", "50", "infinity"]`
	// represents the headers "<= 10 items", "<= 50 items", and "> 50 items". Must
	// be non-empty. Can only be set if all other fields are not set.
	NumberOfItems []string `json:"numberOfItems,omitempty"`
	// PostalCodeGroupNames: Required. A list of postal group names. The last value
	// can be "all other locations". Example: `["zone 1", "zone 2", "all other
	// locations"]`. The referred postal code groups must match the delivery
	// country of the service. Must be non-empty. Can only be set if all other
	// fields are not set.
	PostalCodeGroupNames []string `json:"postalCodeGroupNames,omitempty"`
	// Prices: Required. A list of inclusive order price upper bounds. The last
	// price's value can be infinity by setting price amount_micros = -1. For
	// example `[{"amount_micros": 10000000, "currency_code": "USD"},
	// {"amount_micros": 500000000, "currency_code": "USD"}, {"amount_micros": -1,
	// "currency_code": "USD"}]` represents the headers "<= $10", "<= $500", and ">
	// $500". All prices within a service must have the same currency. Must be
	// non-empty. Must be positive except -1. Can only be set if all other fields
	// are not set.
	Prices []*Price `json:"prices,omitempty"`
	// Weights: Required. A list of inclusive order weight upper bounds. The last
	// weight's value can be infinity by setting price amount_micros = -1. For
	// example `[{"amount_micros": 10000000, "unit": "kg"}, {"amount_micros":
	// 50000000, "unit": "kg"}, {"amount_micros": -1, "unit": "kg"}]` represents
	// the headers "<= 10kg", "<= 50kg", and "> 50kg". All weights within a service
	// must have the same unit. Must be non-empty. Must be positive except -1. Can
	// only be set if all other fields are not set.
	Weights []*Weight `json:"weights,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Locations") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Locations") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Headers) MarshalJSON() ([]byte, error) {
	type NoMethod Headers
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Homepage: The `Homepage` message represents a merchant's store homepage
// within the system. A merchant's homepage is the primary domain where
// customers interact with their store. The homepage can be claimed and
// verified as a proof of ownership and allows the merchant to unlock features
// that require a verified website. For more information, see Understanding
// online store URL verification
// (//support.google.com/merchants/answer/176793).
type Homepage struct {
	// Claimed: Output only. Whether the homepage is claimed. See
	// https://support.google.com/merchants/answer/176793.
	Claimed bool `json:"claimed,omitempty"`
	// Name: Identifier. The resource name of the store's homepage. Format:
	// `accounts/{account}/homepage`
	Name string `json:"name,omitempty"`
	// Uri: Required. The URI (typically a URL) of the store's homepage.
	Uri string `json:"uri,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Claimed") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Claimed") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Homepage) MarshalJSON() ([]byte, error) {
	type NoMethod Homepage
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IdentityAttribute: All information related to an identity attribute.
type IdentityAttribute struct {
	// IdentityDeclaration: Required. The declaration of identity for this
	// attribute.
	//
	// Possible values:
	//   "IDENTITY_DECLARATION_UNSPECIFIED" - Default value indicating that no
	// selection was made.
	//   "SELF_IDENTIFIES_AS" - Indicates that the account identifies with the
	// attribute.
	//   "DOES_NOT_SELF_IDENTIFY_AS" - Indicates that the account does not identify
	// with the attribute.
	IdentityDeclaration string `json:"identityDeclaration,omitempty"`
	// ForceSendFields is a list of field names (e.g. "IdentityDeclaration") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "IdentityDeclaration") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IdentityAttribute) MarshalJSON() ([]byte, error) {
	type NoMethod IdentityAttribute
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Impact: The impact of the issue on a region.
type Impact struct {
	// RegionCode: The CLDR region code (https://cldr.unicode.org/) where this
	// issue applies.
	RegionCode string `json:"regionCode,omitempty"`
	// Severity: The severity of the issue on the destination and region.
	//
	// Possible values:
	//   "SEVERITY_UNSPECIFIED" - The severity is unknown.
	//   "CRITICAL" - The issue causes offers to not serve.
	//   "ERROR" - The issue might affect offers (in the future) or might be an
	// indicator of issues with offers.
	//   "SUGGESTION" - The issue is a suggestion for improvement.
	Severity string `json:"severity,omitempty"`
	// ForceSendFields is a list of field names (e.g. "RegionCode") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RegionCode") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Impact) MarshalJSON() ([]byte, error) {
	type NoMethod Impact
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ImpactedDestination: The impact of the issue on a destination.
type ImpactedDestination struct {
	// Impacts: The (negative) impact for various regions on the given destination.
	Impacts []*Impact `json:"impacts,omitempty"`
	// ReportingContext: The impacted reporting context.
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - [YouTube Checkout](
	ReportingContext string `json:"reportingContext,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Impacts") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Impacts") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ImpactedDestination) MarshalJSON() ([]byte, error) {
	type NoMethod ImpactedDestination
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListAccountIssuesResponse: Response message for the `ListAccountIssues`
// method.
type ListAccountIssuesResponse struct {
	// AccountIssues: The issues from the specified account.
	AccountIssues []*AccountIssue `json:"accountIssues,omitempty"`
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccountIssues") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccountIssues") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListAccountIssuesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListAccountIssuesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListAccountsResponse: Response message for the `ListAccounts` method.
type ListAccountsResponse struct {
	// Accounts: The accounts matching the `ListAccountsRequest`.
	Accounts []*Account `json:"accounts,omitempty"`
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Accounts") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Accounts") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListAccountsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListAccountsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListOnlineReturnPoliciesResponse: Response message for the
// `ListOnlineReturnPolicies` method.
type ListOnlineReturnPoliciesResponse struct {
	// NextPageToken: A token, which can be sent as `pageToken` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// OnlineReturnPolicies: The retrieved return policies.
	OnlineReturnPolicies []*OnlineReturnPolicy `json:"onlineReturnPolicies,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListOnlineReturnPoliciesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListOnlineReturnPoliciesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListProgramsResponse: Response message for the ListPrograms method.
type ListProgramsResponse struct {
	// NextPageToken: A token that can be sent as `page_token` to retrieve the next
	// page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Programs: The programs for the given account.
	Programs []*Program `json:"programs,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListProgramsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListProgramsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListRegionsResponse: Response message for the `ListRegions` method.
type ListRegionsResponse struct {
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Regions: The regions from the specified merchant.
	Regions []*Region `json:"regions,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListRegionsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListRegionsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListSubAccountsResponse: Response message for the `ListSubAccounts` method.
type ListSubAccountsResponse struct {
	// Accounts: The accounts for which the given parent account is an aggregator.
	Accounts []*Account `json:"accounts,omitempty"`
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Accounts") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Accounts") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListSubAccountsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListSubAccountsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListUsersResponse: Response message for the `ListUsers` method.
type ListUsersResponse struct {
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Users: The users from the specified account.
	Users []*User `json:"users,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListUsersResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListUsersResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LocalCutoffTime: Time that local delivery ends for the day.
type LocalCutoffTime struct {
	// Hour: Hour local delivery orders must be placed by to process the same day.
	Hour int64 `json:"hour,omitempty,string"`
	// Minute: Minute local delivery orders must be placed by to process the same
	// day.
	Minute int64 `json:"minute,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "Hour") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Hour") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LocalCutoffTime) MarshalJSON() ([]byte, error) {
	type NoMethod LocalCutoffTime
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LocationIdSet: A list of location ID sets. Must be non-empty. Can only be
// set if all other fields are not set.
type LocationIdSet struct {
	// LocationIds: Required. A non-empty list of location IDs
	// (https://developers.google.com/adwords/api/docs/appendix/geotargeting). They
	// must all be of the same location type (For example, state).
	LocationIds []string `json:"locationIds,omitempty"`
	// ForceSendFields is a list of field names (e.g. "LocationIds") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LocationIds") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LocationIdSet) MarshalJSON() ([]byte, error) {
	type NoMethod LocationIdSet
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LoyaltyProgram: Loyalty program
// (https://support.google.com/merchants/answer/12922446) provided by a
// merchant.
type LoyaltyProgram struct {
	// LoyaltyProgramTiers: Optional. Loyalty program tier of this shipping
	// service.
	LoyaltyProgramTiers []*LoyaltyProgramTiers `json:"loyaltyProgramTiers,omitempty"`
	// ProgramLabel: This is the loyalty program label set in your loyalty program
	// settings in Merchant Center. This sub-attribute allows Google to map your
	// loyalty program to eligible offers.
	ProgramLabel string `json:"programLabel,omitempty"`
	// ForceSendFields is a list of field names (e.g. "LoyaltyProgramTiers") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LoyaltyProgramTiers") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LoyaltyProgram) MarshalJSON() ([]byte, error) {
	type NoMethod LoyaltyProgram
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LoyaltyProgramTiers: Subset of a merchants loyalty program.
type LoyaltyProgramTiers struct {
	// TierLabel: The tier label [tier_label] sub-attribute differentiates offer
	// level benefits between each tier. This value is also set in your program
	// settings in Merchant Center, and is required for data source changes even if
	// your loyalty program only has 1 tier.
	TierLabel string `json:"tierLabel,omitempty"`
	// ForceSendFields is a list of field names (e.g. "TierLabel") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "TierLabel") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LoyaltyProgramTiers) MarshalJSON() ([]byte, error) {
	type NoMethod LoyaltyProgramTiers
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MinimumOrderValueTable: Table of per store minimum order values for the
// pickup fulfillment type.
type MinimumOrderValueTable struct {
	// StoreCodeSetWithMovs: Required. A list of store code sets sharing the same
	// minimum order value (MOV). At least two sets are required and the last one
	// must be empty, which signifies 'MOV for all other stores'. Each store code
	// can only appear once across all the sets. All prices within a service must
	// have the same currency.
	StoreCodeSetWithMovs []*StoreCodeSetWithMov `json:"storeCodeSetWithMovs,omitempty"`
	// ForceSendFields is a list of field names (e.g. "StoreCodeSetWithMovs") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "StoreCodeSetWithMovs") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MinimumOrderValueTable) MarshalJSON() ([]byte, error) {
	type NoMethod MinimumOrderValueTable
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// OnlineReturnPolicy: Online return policy
// (https://support.google.com/merchants/answer/10220642) object. This is
// currently used to represent return policies for ads and free listings
// programs.
type OnlineReturnPolicy struct {
	// AcceptDefectiveOnly: This field specifies if merchant only accepts defective
	// products for returns, and this field is required.
	AcceptDefectiveOnly bool `json:"acceptDefectiveOnly,omitempty"`
	// AcceptExchange: This field specifies if merchant allows customers to
	// exchange products, this field is required.
	AcceptExchange bool `json:"acceptExchange,omitempty"`
	// Countries: Required. Immutable. The countries of sale where the return
	// policy applies. The values must be a valid 2 letter ISO 3166 code.
	Countries []string `json:"countries,omitempty"`
	// ItemConditions: The item conditions accepted for returns must not be empty
	// unless the type of return policy is 'noReturns'.
	//
	// Possible values:
	//   "ITEM_CONDITION_UNSPECIFIED" - Default value. This value is unused.
	//   "NEW" - New.
	//   "USED" - Used.
	ItemConditions []string `json:"itemConditions,omitempty"`
	// Label: Required. Immutable. This field represents the unique user-defined
	// label of the return policy. It is important to note that the same label
	// cannot be used in different return policies for the same country. Unless a
	// product specifies a specific label attribute, policies will be automatically
	// labeled as 'default'. To assign a custom return policy to certain product
	// groups, follow the instructions provided in the [Return policy label]
	// (https://support.google.com/merchants/answer/9445425). The label can contain
	// up to 50 characters.
	Label string `json:"label,omitempty"`
	// Name: Identifier. The name of the `OnlineReturnPolicy` resource. Format:
	// `accounts/{account}/onlineReturnPolicies/{return_policy}`
	Name string `json:"name,omitempty"`
	// Policy: The return policy.
	Policy *Policy `json:"policy,omitempty"`
	// ProcessRefundDays: The field specifies the number of days it takes for
	// merchants to process refunds, field is optional.
	ProcessRefundDays int64 `json:"processRefundDays,omitempty"`
	// RestockingFee: The restocking fee that applies to all return reason
	// categories. This would be treated as a free restocking fee if the value is
	// not set.
	RestockingFee *RestockingFee `json:"restockingFee,omitempty"`
	// ReturnMethods: The return methods of how customers can return an item. This
	// value is required to not be empty unless the type of return policy is
	// noReturns.
	//
	// Possible values:
	//   "RETURN_METHOD_UNSPECIFIED" - Default value. This value is unused.
	//   "BY_MAIL" - Return by mail.
	//   "IN_STORE" - Return in store.
	//   "AT_A_KIOSK" - Return at a kiosk.
	ReturnMethods []string `json:"returnMethods,omitempty"`
	// ReturnPolicyId: Output only. Return policy ID generated by Google.
	ReturnPolicyId string `json:"returnPolicyId,omitempty"`
	// ReturnPolicyUri: Required. The return policy uri. This can used by Google to
	// do a sanity check for the policy. It must be a valid URL.
	ReturnPolicyUri string `json:"returnPolicyUri,omitempty"`
	// ReturnShippingFee: The return shipping fee. Should be set only when customer
	// need to download and print the return label.
	ReturnShippingFee *ReturnShippingFee `json:"returnShippingFee,omitempty"`
	// SeasonalOverrides: Optional. Overrides to the general policy for orders
	// placed during a specific set of time intervals.
	SeasonalOverrides []*SeasonalOverride `json:"seasonalOverrides,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AcceptDefectiveOnly") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AcceptDefectiveOnly") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s OnlineReturnPolicy) MarshalJSON() ([]byte, error) {
	type NoMethod OnlineReturnPolicy
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PhoneNumber: An object representing a phone number, suitable as an API wire
// format. This representation: - should not be used for locale-specific
// formatting of a phone number, such as "+1 (650) 253-0000 ext. 123" - is not
// designed for efficient storage - may not be suitable for dialing -
// specialized libraries (see references) should be used to parse the number
// for that purpose To do something meaningful with this number, such as format
// it for various use-cases, convert it to an `i18n.phonenumbers.PhoneNumber`
// object first. For instance, in Java this would be:
// com.google.type.PhoneNumber wireProto =
// com.google.type.PhoneNumber.newBuilder().build();
// com.google.i18n.phonenumbers.Phonenumber.PhoneNumber phoneNumber =
// PhoneNumberUtil.getInstance().parse(wireProto.getE164Number(), "ZZ"); if
// (!wireProto.getExtension().isEmpty()) {
// phoneNumber.setExtension(wireProto.getExtension()); } Reference(s): -
// https://github.com/google/libphonenumber
type PhoneNumber struct {
	// E164Number: The phone number, represented as a leading plus sign ('+'),
	// followed by a phone number that uses a relaxed ITU E.164 format consisting
	// of the country calling code (1 to 3 digits) and the subscriber number, with
	// no additional spaces or formatting. For example: - correct: "+15552220123" -
	// incorrect: "+1 (555) 222-01234 x123". The ITU E.164 format limits the latter
	// to 12 digits, but in practice not all countries respect that, so we relax
	// that restriction here. National-only numbers are not allowed. References: -
	// https://www.itu.int/rec/T-REC-E.164-201011-I -
	// https://en.wikipedia.org/wiki/E.164. -
	// https://en.wikipedia.org/wiki/List_of_country_calling_codes
	E164Number string `json:"e164Number,omitempty"`
	// Extension: The phone number's extension. The extension is not standardized
	// in ITU recommendations, except for being defined as a series of numbers with
	// a maximum length of 40 digits. Other than digits, some other dialing
	// characters such as ',' (indicating a wait) or '#' may be stored here. Note
	// that no regions currently use extensions with short codes, so this field is
	// normally only set in conjunction with an E.164 number. It is held separately
	// from the E.164 number to allow for short code extensions in the future.
	Extension string `json:"extension,omitempty"`
	// ShortCode: A short code. Reference(s): -
	// https://en.wikipedia.org/wiki/Short_code
	ShortCode *ShortCode `json:"shortCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "E164Number") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "E164Number") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PhoneNumber) MarshalJSON() ([]byte, error) {
	type NoMethod PhoneNumber
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Policy: The available policies.
type Policy struct {
	// Days: The number of days items can be returned after delivery, where one day
	// is defined as 24 hours after the delivery timestamp. Required for
	// `NUMBER_OF_DAYS_AFTER_DELIVERY` returns.
	Days int64 `json:"days,omitempty,string"`
	// Type: Policy type.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value. This value is unused.
	//   "NUMBER_OF_DAYS_AFTER_DELIVERY" - The number of days within which a return
	// is valid after delivery.
	//   "NO_RETURNS" - No returns.
	//   "LIFETIME_RETURNS" - Life time returns.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Days") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Days") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Policy) MarshalJSON() ([]byte, error) {
	type NoMethod Policy
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PostalAddress: Represents a postal address. For example for postal delivery
// or payments addresses. Given a postal address, a postal service can deliver
// items to a premise, P.O. Box or similar. It is not intended to model
// geographical locations (roads, towns, mountains). In typical usage an
// address would be created by user input or from importing existing data,
// depending on the type of process. Advice on address input / editing: - Use
// an internationalization-ready address widget such as
// https://github.com/google/libaddressinput) - Users should not be presented
// with UI elements for input or editing of fields outside countries where that
// field is used. For more guidance on how to use this schema, see:
// https://support.google.com/business/answer/6397478
type PostalAddress struct {
	// AddressLines: Unstructured address lines describing the lower levels of an
	// address. Because values in address_lines do not have type information and
	// may sometimes contain multiple values in a single field (For example
	// "Austin, TX"), it is important that the line order is clear. The order of
	// address lines should be "envelope order" for the country/region of the
	// address. In places where this can vary (For example Japan), address_language
	// is used to make it explicit (For example "ja" for large-to-small ordering
	// and "ja-Latn" or "en" for small-to-large). This way, the most specific line
	// of an address can be selected based on the language. The minimum permitted
	// structural representation of an address consists of a region_code with all
	// remaining information placed in the address_lines. It would be possible to
	// format such an address very approximately without geocoding, but no semantic
	// reasoning could be made about any of the address components until it was at
	// least partially resolved. Creating an address only containing a region_code
	// and address_lines, and then geocoding is the recommended way to handle
	// completely unstructured addresses (as opposed to guessing which parts of the
	// address should be localities or administrative areas).
	AddressLines []string `json:"addressLines,omitempty"`
	// AdministrativeArea: Optional. Highest administrative subdivision which is
	// used for postal addresses of a country or region. For example, this can be a
	// state, a province, an oblast, or a prefecture. Specifically, for Spain this
	// is the province and not the autonomous community (For example "Barcelona"
	// and not "Catalonia"). Many countries don't use an administrative area in
	// postal addresses. For example in Switzerland this should be left
	// unpopulated.
	AdministrativeArea string `json:"administrativeArea,omitempty"`
	// LanguageCode: Optional. BCP-47 language code of the contents of this address
	// (if known). This is often the UI language of the input form or is expected
	// to match one of the languages used in the address' country/region, or their
	// transliterated equivalents. This can affect formatting in certain countries,
	// but is not critical to the correctness of the data and will never affect any
	// validation or other non-formatting related operations. If this value is not
	// known, it should be omitted (rather than specifying a possibly incorrect
	// default). Examples: "zh-Hant", "ja", "ja-Latn", "en".
	LanguageCode string `json:"languageCode,omitempty"`
	// Locality: Optional. Generally refers to the city/town portion of the
	// address. Examples: US city, IT comune, UK post town. In regions of the world
	// where localities are not well defined or do not fit into this structure
	// well, leave locality empty and use address_lines.
	Locality string `json:"locality,omitempty"`
	// Organization: Optional. The name of the organization at the address.
	Organization string `json:"organization,omitempty"`
	// PostalCode: Optional. Postal code of the address. Not all countries use or
	// require postal codes to be present, but where they are used, they may
	// trigger additional validation with other parts of the address (For example
	// state/zip validation in the U.S.A.).
	PostalCode string `json:"postalCode,omitempty"`
	// Recipients: Optional. The recipient at the address. This field may, under
	// certain circumstances, contain multiline information. For example, it might
	// contain "care of" information.
	Recipients []string `json:"recipients,omitempty"`
	// RegionCode: Required. CLDR region code of the country/region of the address.
	// This is never inferred and it is up to the user to ensure the value is
	// correct. See https://cldr.unicode.org/ and
	// https://www.unicode.org/cldr/charts/30/supplemental/territory_information.html
	// for details. Example: "CH" for Switzerland.
	RegionCode string `json:"regionCode,omitempty"`
	// Revision: The schema revision of the `PostalAddress`. This must be set to 0,
	// which is the latest revision. All new revisions **must** be backward
	// compatible with old revisions.
	Revision int64 `json:"revision,omitempty"`
	// SortingCode: Optional. Additional, country-specific, sorting code. This is
	// not used in most regions. Where it is used, the value is either a string
	// like "CEDEX", optionally followed by a number (For example "CEDEX 7"), or
	// just a number alone, representing the "sector code" (Jamaica), "delivery
	// area indicator" (Malawi) or "post office indicator" (For example Côte
	// d'Ivoire).
	SortingCode string `json:"sortingCode,omitempty"`
	// Sublocality: Optional. Sublocality of the address. For example, this can be
	// neighborhoods, boroughs, districts.
	Sublocality string `json:"sublocality,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AddressLines") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AddressLines") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PostalAddress) MarshalJSON() ([]byte, error) {
	type NoMethod PostalAddress
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PostalCodeArea: A list of postal codes that defines the region area. Note:
// All regions defined using postal codes are accessible through the account's
// `ShippingSettings.postalCodeGroups` resource.
type PostalCodeArea struct {
	// PostalCodes: Required. A range of postal codes.
	PostalCodes []*PostalCodeRange `json:"postalCodes,omitempty"`
	// RegionCode: Required. CLDR territory code
	// (http://www.unicode.org/repos/cldr/tags/latest/common/main/en.xml) or the
	// country the postal code group applies to.
	RegionCode string `json:"regionCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "PostalCodes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PostalCodes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PostalCodeArea) MarshalJSON() ([]byte, error) {
	type NoMethod PostalCodeArea
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PostalCodeRange: A range of postal codes that defines the region area.
type PostalCodeRange struct {
	// Begin: Required. A postal code or a pattern of the form prefix* denoting the
	// inclusive lower bound of the range defining the area. Examples values:
	// `94108`, `9410*`, `9*`.
	Begin string `json:"begin,omitempty"`
	// End: Optional. A postal code or a pattern of the form `prefix*` denoting the
	// inclusive upper bound of the range defining the area. It must have the same
	// length as postalCodeRangeBegin: if postalCodeRangeBegin is a postal code
	// then postalCodeRangeEnd must be a postal code too; if postalCodeRangeBegin
	// is a pattern then postalCodeRangeEnd must be a pattern with the same prefix
	// length. Optional: if not set, then the area is defined as being all the
	// postal codes matching postalCodeRangeBegin.
	End string `json:"end,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Begin") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Begin") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PostalCodeRange) MarshalJSON() ([]byte, error) {
	type NoMethod PostalCodeRange
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Price: The price represented as a number and currency.
type Price struct {
	// AmountMicros: The price represented as a number in micros (1 million micros
	// is an equivalent to one's currency standard unit, for example, 1 USD =
	// 1000000 micros).
	AmountMicros int64 `json:"amountMicros,omitempty,string"`
	// CurrencyCode: The currency of the price using three-letter acronyms
	// according to ISO 4217 (http://en.wikipedia.org/wiki/ISO_4217).
	CurrencyCode string `json:"currencyCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AmountMicros") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AmountMicros") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Price) MarshalJSON() ([]byte, error) {
	type NoMethod Price
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductChange: The change that happened to the product including old value,
// new value, country code as the region code and reporting context.
type ProductChange struct {
	// NewValue: The new value of the changed resource or attribute.
	NewValue string `json:"newValue,omitempty"`
	// OldValue: The old value of the changed resource or attribute.
	OldValue string `json:"oldValue,omitempty"`
	// RegionCode: Countries that have the change (if applicable)
	RegionCode string `json:"regionCode,omitempty"`
	// ReportingContext: Reporting contexts that have the change (if applicable)
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - [YouTube Checkout](
	ReportingContext string `json:"reportingContext,omitempty"`
	// ForceSendFields is a list of field names (e.g. "NewValue") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NewValue") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductChange) MarshalJSON() ([]byte, error) {
	type NoMethod ProductChange
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductStatusChangeMessage: The message that the merchant will receive to
// notify about product status change event
type ProductStatusChangeMessage struct {
	// Account: The target account that owns the entity that changed. Format :
	// `accounts/{merchant_id}`
	Account string `json:"account,omitempty"`
	// Attribute: The attribute in the resource that changed, in this case it will
	// be always `Status`.
	//
	// Possible values:
	//   "ATTRIBUTE_UNSPECIFIED" - Unspecified attribute
	//   "STATUS" - Status of the changed entity
	Attribute string `json:"attribute,omitempty"`
	// Changes: A message to describe the change that happened to the product
	Changes []*ProductChange `json:"changes,omitempty"`
	// ExpirationTime: The product expiration time. This field will not bet set if
	// the notification is sent for a product deletion event.
	ExpirationTime string `json:"expirationTime,omitempty"`
	// ManagingAccount: The account that manages the merchant's account. can be the
	// same as merchant id if it is standalone account. Format :
	// `accounts/{service_provider_id}`
	ManagingAccount string `json:"managingAccount,omitempty"`
	// Resource: The product name. Format:
	// `{product.name=accounts/{account}/products/{product}}`
	Resource string `json:"resource,omitempty"`
	// ResourceId: The product id.
	ResourceId string `json:"resourceId,omitempty"`
	// ResourceType: The resource that changed, in this case it will always be
	// `Product`.
	//
	// Possible values:
	//   "RESOURCE_UNSPECIFIED" - Unspecified resource
	//   "PRODUCT" - Resource type : product
	ResourceType string `json:"resourceType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Account") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Account") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductStatusChangeMessage) MarshalJSON() ([]byte, error) {
	type NoMethod ProductStatusChangeMessage
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Program: Defines participation in a given program for the specified account.
// Programs provide a mechanism for adding functionality to merchant accounts.
// A typical example of this is the Free product listings
// (https://support.google.com/merchants/topic/9240261?ref_topic=7257954,7259405,&sjid=796648681813264022-EU)
// program, which enables products from a merchant's store to be shown across
// Google for free.
type Program struct {
	// ActiveRegionCodes: Output only. The regions in which the account is actively
	// participating in the program. Active regions are defined as those where all
	// program requirements affecting the regions have been met. Region codes are
	// defined by CLDR (https://cldr.unicode.org/). This is either a country where
	// the program applies specifically to that country or `001` when the program
	// applies globally.
	ActiveRegionCodes []string `json:"activeRegionCodes,omitempty"`
	// DocumentationUri: Output only. The URL of a Merchant Center help page
	// describing the program.
	DocumentationUri string `json:"documentationUri,omitempty"`
	// Name: Identifier. The resource name of the program. Format:
	// `accounts/{account}/programs/{program}`
	Name string `json:"name,omitempty"`
	// State: Output only. The participation state of the account in the program.
	//
	// Possible values:
	//   "STATE_UNSPECIFIED" - Default value. This value is unused.
	//   "NOT_ELIGIBLE" - The account is not eligible to participate in the
	// program.
	//   "ELIGIBLE" - The account is eligible to participate in the program.
	//   "ENABLED" - The program is enabled for the account.
	State string `json:"state,omitempty"`
	// UnmetRequirements: Output only. The requirements that the account has not
	// yet satisfied that are affecting participation in the program.
	UnmetRequirements []*Requirement `json:"unmetRequirements,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ActiveRegionCodes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ActiveRegionCodes") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Program) MarshalJSON() ([]byte, error) {
	type NoMethod Program
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RateGroup: Shipping rate group definitions. Only the last one is allowed to
// have an empty `applicable_shipping_labels`, which means "everything else".
// The other `applicable_shipping_labels` must not overlap.
type RateGroup struct {
	// ApplicableShippingLabels: Required. A list of shipping labels
	// (https://support.google.com/merchants/answer/6324504) defining the products
	// to which this rate group applies to. This is a disjunction: only one of the
	// labels has to match for the rate group to apply. May only be empty for the
	// last rate group of a service.
	ApplicableShippingLabels []string `json:"applicableShippingLabels,omitempty"`
	// CarrierRates: Optional. A list of carrier rates that can be referred to by
	// `main_table` or `single_value`.
	CarrierRates []*CarrierRate `json:"carrierRates,omitempty"`
	// MainTable: A table defining the rate group, when `single_value` is not
	// expressive enough. Can only be set if `single_value` is not set.
	MainTable *Table `json:"mainTable,omitempty"`
	// Name: Optional. Name of the rate group. If set has to be unique within
	// shipping service.
	Name string `json:"name,omitempty"`
	// SingleValue: The value of the rate group (For example flat rate $10). Can
	// only be set if `main_table` and `subtables` are not set.
	SingleValue *Value `json:"singleValue,omitempty"`
	// Subtables: Optional. A list of subtables referred to by `main_table`. Can
	// only be set if `main_table` is set.
	Subtables []*Table `json:"subtables,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ApplicableShippingLabels")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ApplicableShippingLabels") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RateGroup) MarshalJSON() ([]byte, error) {
	type NoMethod RateGroup
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Region: Represents a geographic region that you can use as a target with
// both the `RegionalInventory` and `ShippingSettings` services. You can define
// regions as collections of either postal codes or, in some countries, using
// predefined geotargets. For more information, see Set up regions
// (https://support.google.com/merchants/answer/7410946#zippy=%2Ccreate-a-new-region)
// for more information.
type Region struct {
	// DisplayName: Optional. The display name of the region.
	DisplayName string `json:"displayName,omitempty"`
	// GeotargetArea: Optional. A list of geotargets that defines the region area.
	GeotargetArea *GeoTargetArea `json:"geotargetArea,omitempty"`
	// Name: Identifier. The resource name of the region. Format:
	// `accounts/{account}/regions/{region}`
	Name string `json:"name,omitempty"`
	// PostalCodeArea: Optional. A list of postal codes that defines the region
	// area.
	PostalCodeArea *PostalCodeArea `json:"postalCodeArea,omitempty"`
	// RegionalInventoryEligible: Output only. Indicates if the region is eligible
	// for use in the Regional Inventory configuration.
	RegionalInventoryEligible bool `json:"regionalInventoryEligible,omitempty"`
	// ShippingEligible: Output only. Indicates if the region is eligible for use
	// in the Shipping Services configuration.
	ShippingEligible bool `json:"shippingEligible,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "DisplayName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DisplayName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Region) MarshalJSON() ([]byte, error) {
	type NoMethod Region
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Required: Describes the terms of service which are required to be accepted.
type Required struct {
	// TermsOfService: The termsOfService
	// (google.shopping.merchant.accounts.v1main.TermsOfService) that need to be
	// accepted.
	TermsOfService string `json:"termsOfService,omitempty"`
	// TosFileUri: Full URL to the terms of service file. This field is the same as
	// TermsOfService.file_uri (TermsOfService.file_uri), it is added here for
	// convenience only.
	TosFileUri string `json:"tosFileUri,omitempty"`
	// ForceSendFields is a list of field names (e.g. "TermsOfService") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "TermsOfService") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Required) MarshalJSON() ([]byte, error) {
	type NoMethod Required
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Requirement: Defines a requirement specified for participation in the
// program.
type Requirement struct {
	// AffectedRegionCodes: Output only. The regions that are currently affected by
	// this requirement not being met. Region codes are defined by CLDR
	// (https://cldr.unicode.org/). This is either a country where the program
	// applies specifically to that country or `001` when the program applies
	// globally.
	AffectedRegionCodes []string `json:"affectedRegionCodes,omitempty"`
	// DocumentationUri: Output only. The URL of a help page describing the
	// requirement.
	DocumentationUri string `json:"documentationUri,omitempty"`
	// Title: Output only. Name of the requirement.
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AffectedRegionCodes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AffectedRegionCodes") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Requirement) MarshalJSON() ([]byte, error) {
	type NoMethod Requirement
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RestockingFee: The restocking fee. This can be a flat fee or a micro
// percent.
type RestockingFee struct {
	// FixedFee: Fixed restocking fee.
	FixedFee *Price `json:"fixedFee,omitempty"`
	// MicroPercent: Percent of total price in micros. 15,000,000 means 15% of the
	// total price would be charged.
	MicroPercent int64 `json:"microPercent,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FixedFee") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FixedFee") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RestockingFee) MarshalJSON() ([]byte, error) {
	type NoMethod RestockingFee
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReturnShippingFee: The return shipping fee. This can either be a fixed fee
// or a boolean to indicate that the customer pays the actual shipping cost.
type ReturnShippingFee struct {
	// FixedFee: Fixed return shipping fee amount. This value is only applicable
	// when type is `FIXED`. We will treat the return shipping fee as free if type
	// is `FIXED` and this value is not set.
	FixedFee *Price `json:"fixedFee,omitempty"`
	// Type: Required. Type of return shipping fee.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value. This value is unused.
	//   "FIXED" - The return shipping fee is a fixed value.
	//   "CUSTOMER_PAYING_ACTUAL_FEE" - Customers will pay the actual return
	// shipping fee.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FixedFee") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FixedFee") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReturnShippingFee) MarshalJSON() ([]byte, error) {
	type NoMethod ReturnShippingFee
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Row: Include a list of cells.
type Row struct {
	// Cells: Required. The list of cells that constitute the row. Must have the
	// same length as `columnHeaders` for two-dimensional tables, a length of 1 for
	// one-dimensional tables.
	Cells []*Value `json:"cells,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Cells") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Cells") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Row) MarshalJSON() ([]byte, error) {
	type NoMethod Row
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SeasonalOverride: Next: 5
type SeasonalOverride struct {
	// Begin: Required. Defines the date range when this seasonal override applies.
	// Both begin and end are inclusive and should be in date decimal format,
	// example 20250115. The dates of the seasonal overrides should not overlap.
	Begin *Date `json:"begin,omitempty"`
	// End: Required. seasonal override end date (inclusive).
	End *Date `json:"end,omitempty"`
	// Label: Required. Display name of this seasonal override in Merchant Center.
	Label string `json:"label,omitempty"`
	// Policy: Required. The return policy for the given date range.
	Policy *Policy `json:"policy,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Begin") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Begin") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SeasonalOverride) MarshalJSON() ([]byte, error) {
	type NoMethod SeasonalOverride
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Service: Shipping service.
type Service struct {
	// Active: Required. A boolean exposing the active status of the shipping
	// service.
	Active bool `json:"active,omitempty"`
	// CurrencyCode: The CLDR code of the currency to which this service applies.
	// Must match that of the prices in rate groups.
	CurrencyCode string `json:"currencyCode,omitempty"`
	// DeliveryCountries: Required. The CLDR territory code of the countries to
	// which the service applies.
	DeliveryCountries []string `json:"deliveryCountries,omitempty"`
	// DeliveryTime: Required. Time spent in various aspects from order to the
	// delivery of the product.
	DeliveryTime *DeliveryTime `json:"deliveryTime,omitempty"`
	// LoyaltyPrograms: Optional. Loyalty programs that this shipping service is
	// limited to.
	LoyaltyPrograms []*LoyaltyProgram `json:"loyaltyPrograms,omitempty"`
	// MinimumOrderValue: Minimum order value for this service. If set, indicates
	// that customers will have to spend at least this amount. All prices within a
	// service must have the same currency. Cannot be set together with
	// minimum_order_value_table.
	MinimumOrderValue *Price `json:"minimumOrderValue,omitempty"`
	// MinimumOrderValueTable: Table of per store minimum order values for the
	// pickup fulfillment type. Cannot be set together with minimum_order_value.
	MinimumOrderValueTable *MinimumOrderValueTable `json:"minimumOrderValueTable,omitempty"`
	// RateGroups: Optional. Shipping rate group definitions. Only the last one is
	// allowed to have an empty `applicable_shipping_labels`, which means
	// "everything else". The other `applicable_shipping_labels` must not overlap.
	RateGroups []*RateGroup `json:"rateGroups,omitempty"`
	// ServiceName: Required. Free-form name of the service. Must be unique within
	// target account.
	ServiceName string `json:"serviceName,omitempty"`
	// ShipmentType: Type of locations this service ships orders to.
	//
	// Possible values:
	//   "SHIPMENT_TYPE_UNSPECIFIED" - This service did not specify shipment type.
	//   "DELIVERY" - This service ships orders to an address chosen by the
	// customer.
	//   "LOCAL_DELIVERY" - This service ships orders to an address chosen by the
	// customer. The order is shipped from a local store near by.
	//   "COLLECTION_POINT" - This service ships orders to an address chosen by the
	// customer. The order is shipped from a collection point.
	ShipmentType string `json:"shipmentType,omitempty"`
	// StoreConfig: A list of stores your products are delivered from. This is only
	// valid for the local delivery shipment type.
	StoreConfig *StoreConfig `json:"storeConfig,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Active") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Active") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Service) MarshalJSON() ([]byte, error) {
	type NoMethod Service
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ShippingSettings: The merchant account's shipping setting
// (https://support.google.com/merchants/answer/6069284).
type ShippingSettings struct {
	// Etag: Required. This field is used for avoid async issue. Make sure shipping
	// setting data didn't change between get call and insert call. The user should
	// do following steps： 1. Set etag field as empty string for initial shipping
	// setting creation. 2. After initial creation, call get method to obtain an
	// etag and current shipping setting data before call insert. 3. Modify to
	// wanted shipping setting information. 4. Call insert method with the wanted
	// shipping setting information with the etag obtained from step 2. 5. If
	// shipping setting data changed between step 2 and step 4. Insert request will
	// fail because the etag changes every time the shipping setting data changes.
	// User should repeate step 2-4 with the new etag.
	Etag string `json:"etag,omitempty"`
	// Name: Identifier. The resource name of the shipping setting. Format:
	// `accounts/{account}/shippingSetting`
	Name string `json:"name,omitempty"`
	// Services: Optional. The target account's list of services.
	Services []*Service `json:"services,omitempty"`
	// Warehouses: Optional. A list of warehouses which can be referred to in
	// `services`.
	Warehouses []*Warehouse `json:"warehouses,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Etag") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Etag") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ShippingSettings) MarshalJSON() ([]byte, error) {
	type NoMethod ShippingSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ShortCode: An object representing a short code, which is a phone number that
// is typically much shorter than regular phone numbers and can be used to
// address messages in MMS and SMS systems, as well as for abbreviated dialing
// (For example "Text 611 to see how many minutes you have remaining on your
// plan."). Short codes are restricted to a region and are not internationally
// dialable, which means the same short code can exist in different regions,
// with different usage and pricing, even if those regions share the same
// country calling code (For example: US and CA).
type ShortCode struct {
	// Number: Required. The short code digits, without a leading plus ('+') or
	// country calling code. For example "611".
	Number string `json:"number,omitempty"`
	// RegionCode: Required. The BCP-47 region code of the location where calls to
	// this short code can be made, such as "US" and "BB". Reference(s): -
	// http://www.unicode.org/reports/tr35/#unicode_region_subtag
	RegionCode string `json:"regionCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Number") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Number") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ShortCode) MarshalJSON() ([]byte, error) {
	type NoMethod ShortCode
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// StoreCodeSetWithMov: A list of store code sets sharing the same minimum
// order value. At least two sets are required and the last one must be empty,
// which signifies 'MOV for all other stores'. Each store code can only appear
// once across all the sets. All prices within a service must have the same
// currency.
type StoreCodeSetWithMov struct {
	// StoreCodes: Optional. A list of unique store codes or empty for the catch
	// all.
	StoreCodes []string `json:"storeCodes,omitempty"`
	// Value: The minimum order value for the given stores.
	Value *Price `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "StoreCodes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "StoreCodes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s StoreCodeSetWithMov) MarshalJSON() ([]byte, error) {
	type NoMethod StoreCodeSetWithMov
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// StoreConfig: A list of stores your products are delivered from. This is only
// valid for the local delivery shipment type.
type StoreConfig struct {
	// CutoffConfig: Configs related to local delivery ends for the day.
	CutoffConfig *CutoffConfig `json:"cutoffConfig,omitempty"`
	// ServiceRadius: Maximum delivery radius. This is only required for the local
	// delivery shipment type.
	ServiceRadius *Distance `json:"serviceRadius,omitempty"`
	// StoreCodes: Optional. A list of store codes that provide local delivery. If
	// empty, then `all_stores` must be true.
	StoreCodes []string `json:"storeCodes,omitempty"`
	// StoreServiceType: Indicates whether all stores, or selected stores, listed
	// by this merchant provide local delivery.
	//
	// Possible values:
	//   "STORE_SERVICE_TYPE_UNSPECIFIED" - Did not specify store service type.
	//   "ALL_STORES" - Indicates whether all stores, current and future, listed by
	// this merchant provide local delivery.
	//   "SELECTED_STORES" - Indicates that only the stores listed in `store_codes`
	// are eligible for local delivery.
	StoreServiceType string `json:"storeServiceType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CutoffConfig") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CutoffConfig") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s StoreConfig) MarshalJSON() ([]byte, error) {
	type NoMethod StoreConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Table: A table defining the rate group, when `single_value` is not
// expressive enough.
type Table struct {
	// ColumnHeaders: Headers of the table's columns. Optional: if not set then the
	// table has only one dimension.
	ColumnHeaders *Headers `json:"columnHeaders,omitempty"`
	// Name: Name of the table. Required for subtables, ignored for the main table.
	Name string `json:"name,omitempty"`
	// RowHeaders: Required. Headers of the table's rows.
	RowHeaders *Headers `json:"rowHeaders,omitempty"`
	// Rows: Required. The list of rows that constitute the table. Must have the
	// same length as `row_headers`.
	Rows []*Row `json:"rows,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ColumnHeaders") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ColumnHeaders") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Table) MarshalJSON() ([]byte, error) {
	type NoMethod Table
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TermsOfService: The `TermsOfService` message represents a specific version
// of the terms of service that merchants must accept to access certain
// features or services (see
// https://support.google.com/merchants/answer/160173). This message is
// important for the onboarding process, ensuring that merchants agree to the
// necessary legal agreements for using the service. Merchants can retrieve the
// latest terms of service for a given `kind` and `region` through
// `RetrieveLatestTermsOfService`, and accept them as required through
// `AcceptTermsOfService`.
type TermsOfService struct {
	// External: Whether this terms of service version is external. External terms
	// of service versions can only be agreed through external processes and not
	// directly by the merchant through UI or API.
	External bool `json:"external,omitempty"`
	// FileUri: URI for terms of service file that needs to be displayed to signing
	// users.
	FileUri string `json:"fileUri,omitempty"`
	// Kind: The Kind this terms of service version applies to.
	//
	// Possible values:
	//   "TERMS_OF_SERVICE_KIND_UNSPECIFIED" - Default value. This value is unused.
	//   "MERCHANT_CENTER" - Merchant Center application.
	Kind string `json:"kind,omitempty"`
	// Name: Identifier. The resource name of the terms of service version. Format:
	// `termsOfService/{version}`
	Name string `json:"name,omitempty"`
	// RegionCode: Region code as defined by CLDR (https://cldr.unicode.org/). This
	// is either a country where the ToS applies specifically to that country or
	// `001` when the same `TermsOfService` can be signed in any country. However
	// note that when signing a ToS that applies globally we still expect that a
	// specific country is provided (this should be merchant business country or
	// program country of participation).
	RegionCode string `json:"regionCode,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "External") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "External") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TermsOfService) MarshalJSON() ([]byte, error) {
	type NoMethod TermsOfService
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TermsOfServiceAgreementState: This resource represents the agreement state
// for a given account and terms of service kind. The state is as follows: * If
// the merchant has accepted a terms of service: accepted
// (TermsOfServiceAggrementState.accepted) will be populated, otherwise it will
// be empty * If the merchant must sign a terms of service: required
// (TermsOfServiceAggrementState.required) will be populated, otherwise it will
// be empty. Note that both required (TermsOfServiceAggrementState.required)
// and accepted (TermsOfServiceAggrementState.accepted) can be present. In this
// case the `accepted` terms of services will have an expiration date set in
// the valid_until (Accepted.valid_until) field. The `required` terms of
// services need to be accepted before `valid_until` in order for the account
// to continue having a valid agreement. When accepting new terms of services
// we expect 3Ps to display the text associated with the given terms of service
// agreement (the url to the file containing the text is added in the Required
// message below as tos_file_uri (Accepted.tos_file_uri). The actual acceptance
// of the terms of service is done by calling accept on the TermsOfService
// (TermsOfService) resource.
type TermsOfServiceAgreementState struct {
	// Accepted: The accepted terms of service of this kind and for the associated
	// region_code
	Accepted *Accepted `json:"accepted,omitempty"`
	// Name: Identifier. The resource name of the terms of service version. Format:
	// `accounts/{account}/termsOfServiceAgreementState/{identifier}` The
	// identifier format is: `{TermsOfServiceKind}-{country}` For example, an
	// identifier could be: `MERCHANT_CENTER-US`
	Name string `json:"name,omitempty"`
	// RegionCode: Region code as defined by https://cldr.unicode.org/. This is the
	// country the current state applies to.
	RegionCode string `json:"regionCode,omitempty"`
	// Required: The required terms of service
	Required *Required `json:"required,omitempty"`
	// TermsOfServiceKind: Terms of Service kind associated with the particular
	// version.
	//
	// Possible values:
	//   "TERMS_OF_SERVICE_KIND_UNSPECIFIED" - Default value. This value is unused.
	//   "MERCHANT_CENTER" - Merchant Center application.
	TermsOfServiceKind string `json:"termsOfServiceKind,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Accepted") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Accepted") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TermsOfServiceAgreementState) MarshalJSON() ([]byte, error) {
	type NoMethod TermsOfServiceAgreementState
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TimeZone: Represents a time zone from the IANA Time Zone Database
// (https://www.iana.org/time-zones).
type TimeZone struct {
	// Id: IANA Time Zone Database time zone. For example "America/New_York".
	Id string `json:"id,omitempty"`
	// Version: Optional. IANA Time Zone Database version number. For example
	// "2019a".
	Version string `json:"version,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Id") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Id") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TimeZone) MarshalJSON() ([]byte, error) {
	type NoMethod TimeZone
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TransitTable: Transit time table, number of business days spent in transit
// based on row and column dimensions. Either `min_transit_days`,
// `max_transit_days` or `transit_time_table` can be set, but not both.
type TransitTable struct {
	// PostalCodeGroupNames: Required. A list of region names Region.name . The
	// last value can be "all other locations". Example: `["zone 1", "zone 2",
	// "all other locations"]`. The referred postal code groups must match the
	// delivery country of the service.
	PostalCodeGroupNames []string `json:"postalCodeGroupNames,omitempty"`
	// Rows: Required. If there's only one dimension set of
	// `postal_code_group_names` or `transit_time_labels`, there are multiple rows
	// each with one value for that dimension. If there are two dimensions, each
	// row corresponds to a `postal_code_group_names`, and columns (values) to a
	// `transit_time_labels`.
	Rows []*TransitTimeRow `json:"rows,omitempty"`
	// TransitTimeLabels: Required. A list of transit time labels. The last value
	// can be "all other labels". Example: `["food", "electronics", "all other
	// labels"]`.
	TransitTimeLabels []string `json:"transitTimeLabels,omitempty"`
	// ForceSendFields is a list of field names (e.g. "PostalCodeGroupNames") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PostalCodeGroupNames") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TransitTable) MarshalJSON() ([]byte, error) {
	type NoMethod TransitTable
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TransitTimeRow: If there's only one dimension set of
// `postal_code_group_names` or `transit_time_labels`, there are multiple rows
// each with one value for that dimension. If there are two dimensions, each
// row corresponds to a `postal_code_group_names`, and columns (values) to a
// `transit_time_labels`.
type TransitTimeRow struct {
	// Values: Required. Transit time range (min-max) in business days.
	Values []*TransitTimeValue `json:"values,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Values") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Values") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TransitTimeRow) MarshalJSON() ([]byte, error) {
	type NoMethod TransitTimeRow
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TransitTimeValue: Transit time range (min-max) in business days.
type TransitTimeValue struct {
	// MaxTransitDays: Must be greater than or equal to `min_transit_days`.
	MaxTransitDays int64 `json:"maxTransitDays,omitempty"`
	// MinTransitDays: Minimum transit time range in business days. 0 means same
	// day delivery, 1 means next day delivery.
	MinTransitDays int64 `json:"minTransitDays,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MaxTransitDays") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MaxTransitDays") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TransitTimeValue) MarshalJSON() ([]byte, error) {
	type NoMethod TransitTimeValue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UnclaimHomepageRequest: Request message for the `UnclaimHomepage` method.
type UnclaimHomepageRequest struct {
}

// User: The `User` message represents a user associated with a Merchant Center
// account. It is used to manage user permissions and access rights within the
// account. For more information, see Frequently asked questions about people
// and access levels (//support.google.com/merchants/answer/12160472).
type User struct {
	// AccessRights: Optional. The access rights
	// (https://support.google.com/merchants/answer/12160472?sjid=6789834943175119429-EU#accesstypes)
	// the user has.
	//
	// Possible values:
	//   "ACCESS_RIGHT_UNSPECIFIED" - Default value. This value is unused.
	//   "STANDARD" - Standard access rights.
	//   "ADMIN" - Admin access rights.
	//   "PERFORMANCE_REPORTING" - Users with this right have access to performance
	// and insights.
	AccessRights []string `json:"accessRights,omitempty"`
	// Name: Identifier. The resource name of the user. Format:
	// `accounts/{account}/user/{email}` Use `me` to refer to your own email
	// address, for example `accounts/{account}/users/me`.
	Name string `json:"name,omitempty"`
	// State: Output only. The state of the user.
	//
	// Possible values:
	//   "STATE_UNSPECIFIED" - Default value. This value is unused.
	//   "PENDING" - The user is pending confirmation. In this state, the user
	// first needs to accept the invitation before performing other actions.
	//   "VERIFIED" - The user is verified.
	State string `json:"state,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccessRights") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessRights") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s User) MarshalJSON() ([]byte, error) {
	type NoMethod User
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Value: The single value of a rate group or the value of a rate group table's
// cell. Exactly one of `no_shipping`, `flat_rate`, `price_percentage`,
// `carrier_rateName`, `subtable_name` must be set.
type Value struct {
	// CarrierRate: The name of a carrier rate referring to a carrier rate defined
	// in the same rate group. Can only be set if all other fields are not set.
	CarrierRate string `json:"carrierRate,omitempty"`
	// FlatRate: A flat rate. Can only be set if all other fields are not set.
	FlatRate *Price `json:"flatRate,omitempty"`
	// NoShipping: If true, then the product can't be shipped. Must be true when
	// set, can only be set if all other fields are not set.
	NoShipping bool `json:"noShipping,omitempty"`
	// PricePercentage: A percentage of the price represented as a number in
	// decimal notation (For example, "5.4"). Can only be set if all other fields
	// are not set.
	PricePercentage string `json:"pricePercentage,omitempty"`
	// Subtable: The name of a subtable. Can only be set in table cells (For
	// example, not for single values), and only if all other fields are not set.
	Subtable string `json:"subtable,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CarrierRate") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CarrierRate") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Value) MarshalJSON() ([]byte, error) {
	type NoMethod Value
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Warehouse: A fulfillment warehouse, which stores and handles inventory.
type Warehouse struct {
	// BusinessDayConfig: Business days of the warehouse. If not set, will be
	// Monday to Friday by default.
	BusinessDayConfig *BusinessDayConfig `json:"businessDayConfig,omitempty"`
	// CutoffTime: Required. The latest time of day that an order can be accepted
	// and begin processing. Later orders will be processed in the next day. The
	// time is based on the warehouse postal code.
	CutoffTime *WarehouseCutoffTime `json:"cutoffTime,omitempty"`
	// HandlingDays: Required. The number of days it takes for this warehouse to
	// pack up and ship an item. This is on the warehouse level, but can be
	// overridden on the offer level based on the attributes of an item.
	HandlingDays int64 `json:"handlingDays,omitempty,string"`
	// Name: Required. The name of the warehouse. Must be unique within account.
	Name string `json:"name,omitempty"`
	// ShippingAddress: Required. Shipping address of the warehouse.
	ShippingAddress *Address `json:"shippingAddress,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BusinessDayConfig") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BusinessDayConfig") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Warehouse) MarshalJSON() ([]byte, error) {
	type NoMethod Warehouse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// WarehouseBasedDeliveryTime: Indicates that the delivery time should be
// calculated per warehouse (shipping origin location) based on the settings of
// the selected carrier. When set, no other transit time related field in
// `delivery_time` should be set.
type WarehouseBasedDeliveryTime struct {
	// Carrier: Required. Carrier, such as "UPS" or "Fedex". supported carriers
	// (https://support.google.com/merchants/answer/7050921#zippy=%2Ccarrier-rates-au-de-uk-and-us-only)
	Carrier string `json:"carrier,omitempty"`
	// CarrierService: Required. Carrier service, such as "ground" or "2 days".
	// The name of the service must be in the eddSupportedServices list.
	CarrierService string `json:"carrierService,omitempty"`
	// Warehouse: Required. Warehouse name. This should match warehouse
	Warehouse string `json:"warehouse,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Carrier") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Carrier") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s WarehouseBasedDeliveryTime) MarshalJSON() ([]byte, error) {
	type NoMethod WarehouseBasedDeliveryTime
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// WarehouseCutoffTime: The latest time of day that an order can be accepted
// and begin processing. Later orders will be processed in the next day. The
// time is based on the warehouse postal code.
type WarehouseCutoffTime struct {
	// Hour: Required. Hour of the cutoff time until which an order has to be
	// placed to be processed in the same day by the warehouse. Hour is based on
	// the timezone of warehouse.
	Hour int64 `json:"hour,omitempty"`
	// Minute: Required. Minute of the cutoff time until which an order has to be
	// placed to be processed in the same day by the warehouse. Minute is based on
	// the timezone of warehouse.
	Minute int64 `json:"minute,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Hour") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Hour") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s WarehouseCutoffTime) MarshalJSON() ([]byte, error) {
	type NoMethod WarehouseCutoffTime
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Weight: The weight represented as the value in string and the unit.
type Weight struct {
	// AmountMicros: Required. The weight represented as a number in micros (1
	// million micros is an equivalent to one's currency standard unit, for
	// example, 1 kg = 1000000 micros). This field can also be set as infinity by
	// setting to -1. This field only support -1 and positive value.
	AmountMicros int64 `json:"amountMicros,omitempty,string"`
	// Unit: Required. The weight unit. Acceptable values are: kg and lb
	//
	// Possible values:
	//   "WEIGHT_UNIT_UNSPECIFIED" - unit unspecified
	//   "POUND" - lb unit.
	//   "KILOGRAM" - kg unit.
	Unit string `json:"unit,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AmountMicros") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AmountMicros") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Weight) MarshalJSON() ([]byte, error) {
	type NoMethod Weight
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AccountsCreateAndConfigureCall struct {
	s                                *APIService
	createandconfigureaccountrequest *CreateAndConfigureAccountRequest
	urlParams_                       gensupport.URLParams
	ctx_                             context.Context
	header_                          http.Header
}

// CreateAndConfigure: Creates a standalone Merchant Center account with
// additional configuration. Adds the user that makes the request as an admin
// for the new account.
func (r *AccountsService) CreateAndConfigure(createandconfigureaccountrequest *CreateAndConfigureAccountRequest) *AccountsCreateAndConfigureCall {
	c := &AccountsCreateAndConfigureCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.createandconfigureaccountrequest = createandconfigureaccountrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsCreateAndConfigureCall) Fields(s ...googleapi.Field) *AccountsCreateAndConfigureCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsCreateAndConfigureCall) Context(ctx context.Context) *AccountsCreateAndConfigureCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsCreateAndConfigureCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsCreateAndConfigureCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.createandconfigureaccountrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/accounts:createAndConfigure")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.createAndConfigure", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.createAndConfigure" call.
// Any non-2xx status code is an error. Response headers are in either
// *Account.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsCreateAndConfigureCall) Do(opts ...googleapi.CallOption) (*Account, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Account{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.createAndConfigure", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsDeleteCall struct {
	s          *APIService
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes the specified account regardless of its type: standalone,
// MCA or sub-account. Deleting an MCA leads to the deletion of all of its
// sub-accounts. Executing this method requires admin access. The deletion
// succeeds only if the account does not provide services to any other account
// and has no processed offers. You can use the `force` parameter to override
// this.
//
// - name: The name of the account to delete. Format: `accounts/{account}`.
func (r *AccountsService) Delete(name string) *AccountsDeleteCall {
	c := &AccountsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Force sets the optional parameter "force": If set to `true`, the account is
// deleted even if it provides services to other accounts or has processed
// offers.
func (c *AccountsDeleteCall) Force(force bool) *AccountsDeleteCall {
	c.urlParams_.Set("force", fmt.Sprint(force))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsDeleteCall) Fields(s ...googleapi.Field) *AccountsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsDeleteCall) Context(ctx context.Context) *AccountsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsGetCall struct {
	s            *APIService
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves an account from your Merchant Center account. After
// inserting, updating, or deleting an account, it may take several minutes
// before changes take effect.
//
// - name: The name of the account to retrieve. Format: `accounts/{account}`.
func (r *AccountsService) Get(name string) *AccountsGetCall {
	c := &AccountsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsGetCall) Fields(s ...googleapi.Field) *AccountsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsGetCall) IfNoneMatch(entityTag string) *AccountsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsGetCall) Context(ctx context.Context) *AccountsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Account.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsGetCall) Do(opts ...googleapi.CallOption) (*Account, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Account{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsListCall struct {
	s            *APIService
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists accounts accessible to the calling user and matching the
// constraints of the request such as page size or filters. This is not just
// listing the sub-accounts of an MCA, but all accounts the calling user has
// access to including other MCAs, linked accounts, standalone accounts and so
// on. If no filter is provided, then it returns accounts the user is directly
// added to.
func (r *AccountsService) List() *AccountsListCall {
	c := &AccountsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Filter sets the optional parameter "filter": Returns only accounts that
// match the filter (/merchant/api/guides/accounts/filter). For more details,
// see the filter syntax reference
// (/merchant/api/guides/accounts/filter-syntax).
func (c *AccountsListCall) Filter(filter string) *AccountsListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// accounts to return. The service may return fewer than this value. If
// unspecified, at most 250 accounts are returned. The maximum value is 500;
// values above 500 are coerced to 500.
func (c *AccountsListCall) PageSize(pageSize int64) *AccountsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListAccounts` call. Provide this to retrieve the subsequent
// page. When paginating, all other parameters provided to `ListAccounts` must
// match the call that provided the page token.
func (c *AccountsListCall) PageToken(pageToken string) *AccountsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsListCall) Fields(s ...googleapi.Field) *AccountsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsListCall) IfNoneMatch(entityTag string) *AccountsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsListCall) Context(ctx context.Context) *AccountsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/accounts")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListAccountsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsListCall) Do(opts ...googleapi.CallOption) (*ListAccountsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListAccountsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsListCall) Pages(ctx context.Context, f func(*ListAccountsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsListSubaccountsCall struct {
	s            *APIService
	provider     string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// ListSubaccounts: List all sub-accounts for a given multi client account.
// This is a convenience wrapper for the more powerful `ListAccounts` method.
// This method will produce the same results as calling `ListsAccounts` with
// the following filter: `relationship(providerId={parent} AND
// service(type="ACCOUNT_AGGREGATION"))`
//
//   - provider: The aggregation service provider. Format:
//     `providers/{providerId}`.
func (r *AccountsService) ListSubaccounts(provider string) *AccountsListSubaccountsCall {
	c := &AccountsListSubaccountsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.provider = provider
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// accounts to return. The service may return fewer than this value. If
// unspecified, at most 250 accounts are returned. The maximum value is 500;
// values above 500 are coerced to 500.
func (c *AccountsListSubaccountsCall) PageSize(pageSize int64) *AccountsListSubaccountsCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListAccounts` call. Provide this to retrieve the subsequent
// page. When paginating, all other parameters provided to `ListAccounts` must
// match the call that provided the page token.
func (c *AccountsListSubaccountsCall) PageToken(pageToken string) *AccountsListSubaccountsCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsListSubaccountsCall) Fields(s ...googleapi.Field) *AccountsListSubaccountsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsListSubaccountsCall) IfNoneMatch(entityTag string) *AccountsListSubaccountsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsListSubaccountsCall) Context(ctx context.Context) *AccountsListSubaccountsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsListSubaccountsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsListSubaccountsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+provider}:listSubaccounts")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"provider": c.provider,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.listSubaccounts", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.listSubaccounts" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListSubAccountsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsListSubaccountsCall) Do(opts ...googleapi.CallOption) (*ListSubAccountsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListSubAccountsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.listSubaccounts", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsListSubaccountsCall) Pages(ctx context.Context, f func(*ListSubAccountsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsPatchCall struct {
	s          *APIService
	name       string
	account    *Account
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates an account regardless of its type: standalone, MCA or
// sub-account. Executing this method requires admin access.
//
//   - name: Identifier. The resource name of the account. Format:
//     `accounts/{account}`.
func (r *AccountsService) Patch(name string, account *Account) *AccountsPatchCall {
	c := &AccountsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.account = account
	return c
}

// UpdateMask sets the optional parameter "updateMask": List of fields being
// updated. The following fields are supported (in both `snake_case` and
// `lowerCamelCase`): - `account_name` - `adult_content` - `language_code` -
// `time_zone`
func (c *AccountsPatchCall) UpdateMask(updateMask string) *AccountsPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsPatchCall) Fields(s ...googleapi.Field) *AccountsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsPatchCall) Context(ctx context.Context) *AccountsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.account)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Account.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsPatchCall) Do(opts ...googleapi.CallOption) (*Account, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Account{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsAutofeedSettingsGetAutofeedSettingsCall struct {
	s            *APIService
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetAutofeedSettings: Retrieves the autofeed settings of an account.
//
//   - name: The resource name of the autofeed settings. Format:
//     `accounts/{account}/autofeedSettings`.
func (r *AccountsAutofeedSettingsService) GetAutofeedSettings(name string) *AccountsAutofeedSettingsGetAutofeedSettingsCall {
	c := &AccountsAutofeedSettingsGetAutofeedSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAutofeedSettingsGetAutofeedSettingsCall) Fields(s ...googleapi.Field) *AccountsAutofeedSettingsGetAutofeedSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAutofeedSettingsGetAutofeedSettingsCall) IfNoneMatch(entityTag string) *AccountsAutofeedSettingsGetAutofeedSettingsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAutofeedSettingsGetAutofeedSettingsCall) Context(ctx context.Context) *AccountsAutofeedSettingsGetAutofeedSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAutofeedSettingsGetAutofeedSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAutofeedSettingsGetAutofeedSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.autofeedSettings.getAutofeedSettings", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.autofeedSettings.getAutofeedSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *AutofeedSettings.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsAutofeedSettingsGetAutofeedSettingsCall) Do(opts ...googleapi.CallOption) (*AutofeedSettings, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AutofeedSettings{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.autofeedSettings.getAutofeedSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsAutofeedSettingsUpdateAutofeedSettingsCall struct {
	s                *APIService
	name             string
	autofeedsettings *AutofeedSettings
	urlParams_       gensupport.URLParams
	ctx_             context.Context
	header_          http.Header
}

// UpdateAutofeedSettings: Updates the autofeed settings of an account.
//
//   - name: Identifier. The resource name of the autofeed settings. Format:
//     `accounts/{account}/autofeedSettings`.
func (r *AccountsAutofeedSettingsService) UpdateAutofeedSettings(name string, autofeedsettings *AutofeedSettings) *AccountsAutofeedSettingsUpdateAutofeedSettingsCall {
	c := &AccountsAutofeedSettingsUpdateAutofeedSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.autofeedsettings = autofeedsettings
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. List of
// fields being updated.
func (c *AccountsAutofeedSettingsUpdateAutofeedSettingsCall) UpdateMask(updateMask string) *AccountsAutofeedSettingsUpdateAutofeedSettingsCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAutofeedSettingsUpdateAutofeedSettingsCall) Fields(s ...googleapi.Field) *AccountsAutofeedSettingsUpdateAutofeedSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAutofeedSettingsUpdateAutofeedSettingsCall) Context(ctx context.Context) *AccountsAutofeedSettingsUpdateAutofeedSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAutofeedSettingsUpdateAutofeedSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAutofeedSettingsUpdateAutofeedSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.autofeedsettings)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.autofeedSettings.updateAutofeedSettings", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.autofeedSettings.updateAutofeedSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *AutofeedSettings.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsAutofeedSettingsUpdateAutofeedSettingsCall) Do(opts ...googleapi.CallOption) (*AutofeedSettings, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AutofeedSettings{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.autofeedSettings.updateAutofeedSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsBusinessIdentityGetBusinessIdentityCall struct {
	s            *APIService
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetBusinessIdentity: Retrieves the business identity of an account.
//
//   - name: The resource name of the business identity. Format:
//     `accounts/{account}/businessIdentity`.
func (r *AccountsBusinessIdentityService) GetBusinessIdentity(name string) *AccountsBusinessIdentityGetBusinessIdentityCall {
	c := &AccountsBusinessIdentityGetBusinessIdentityCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsBusinessIdentityGetBusinessIdentityCall) Fields(s ...googleapi.Field) *AccountsBusinessIdentityGetBusinessIdentityCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsBusinessIdentityGetBusinessIdentityCall) IfNoneMatch(entityTag string) *AccountsBusinessIdentityGetBusinessIdentityCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsBusinessIdentityGetBusinessIdentityCall) Context(ctx context.Context) *AccountsBusinessIdentityGetBusinessIdentityCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsBusinessIdentityGetBusinessIdentityCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsBusinessIdentityGetBusinessIdentityCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.businessIdentity.getBusinessIdentity", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.businessIdentity.getBusinessIdentity" call.
// Any non-2xx status code is an error. Response headers are in either
// *BusinessIdentity.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsBusinessIdentityGetBusinessIdentityCall) Do(opts ...googleapi.CallOption) (*BusinessIdentity, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &BusinessIdentity{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.businessIdentity.getBusinessIdentity", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsBusinessIdentityUpdateBusinessIdentityCall struct {
	s                *APIService
	name             string
	businessidentity *BusinessIdentity
	urlParams_       gensupport.URLParams
	ctx_             context.Context
	header_          http.Header
}

// UpdateBusinessIdentity: Updates the business identity of an account.
// Executing this method requires admin access.
//
//   - name: Identifier. The resource name of the business identity. Format:
//     `accounts/{account}/businessIdentity`.
func (r *AccountsBusinessIdentityService) UpdateBusinessIdentity(name string, businessidentity *BusinessIdentity) *AccountsBusinessIdentityUpdateBusinessIdentityCall {
	c := &AccountsBusinessIdentityUpdateBusinessIdentityCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.businessidentity = businessidentity
	return c
}

// UpdateMask sets the optional parameter "updateMask": List of fields being
// updated. The following fields are supported (in both `snake_case` and
// `lowerCamelCase`): - `black_owned` - `latino_owned` - `promotions_consent` -
// `small_business` - `veteran_owned` - `women_owned`
func (c *AccountsBusinessIdentityUpdateBusinessIdentityCall) UpdateMask(updateMask string) *AccountsBusinessIdentityUpdateBusinessIdentityCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsBusinessIdentityUpdateBusinessIdentityCall) Fields(s ...googleapi.Field) *AccountsBusinessIdentityUpdateBusinessIdentityCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsBusinessIdentityUpdateBusinessIdentityCall) Context(ctx context.Context) *AccountsBusinessIdentityUpdateBusinessIdentityCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsBusinessIdentityUpdateBusinessIdentityCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsBusinessIdentityUpdateBusinessIdentityCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.businessidentity)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.businessIdentity.updateBusinessIdentity", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.businessIdentity.updateBusinessIdentity" call.
// Any non-2xx status code is an error. Response headers are in either
// *BusinessIdentity.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsBusinessIdentityUpdateBusinessIdentityCall) Do(opts ...googleapi.CallOption) (*BusinessIdentity, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &BusinessIdentity{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.businessIdentity.updateBusinessIdentity", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsBusinessInfoGetBusinessInfoCall struct {
	s            *APIService
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetBusinessInfo: Retrieves the business info of an account.
//
//   - name: The resource name of the business info. Format:
//     `accounts/{account}/businessInfo`.
func (r *AccountsBusinessInfoService) GetBusinessInfo(name string) *AccountsBusinessInfoGetBusinessInfoCall {
	c := &AccountsBusinessInfoGetBusinessInfoCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsBusinessInfoGetBusinessInfoCall) Fields(s ...googleapi.Field) *AccountsBusinessInfoGetBusinessInfoCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsBusinessInfoGetBusinessInfoCall) IfNoneMatch(entityTag string) *AccountsBusinessInfoGetBusinessInfoCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsBusinessInfoGetBusinessInfoCall) Context(ctx context.Context) *AccountsBusinessInfoGetBusinessInfoCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsBusinessInfoGetBusinessInfoCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsBusinessInfoGetBusinessInfoCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.businessInfo.getBusinessInfo", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.businessInfo.getBusinessInfo" call.
// Any non-2xx status code is an error. Response headers are in either
// *BusinessInfo.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsBusinessInfoGetBusinessInfoCall) Do(opts ...googleapi.CallOption) (*BusinessInfo, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &BusinessInfo{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.businessInfo.getBusinessInfo", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsBusinessInfoUpdateBusinessInfoCall struct {
	s            *APIService
	name         string
	businessinfo *BusinessInfo
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// UpdateBusinessInfo: Updates the business info of an account. Executing this
// method requires admin access.
//
//   - name: Identifier. The resource name of the business info. Format:
//     `accounts/{account}/businessInfo`.
func (r *AccountsBusinessInfoService) UpdateBusinessInfo(name string, businessinfo *BusinessInfo) *AccountsBusinessInfoUpdateBusinessInfoCall {
	c := &AccountsBusinessInfoUpdateBusinessInfoCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.businessinfo = businessinfo
	return c
}

// UpdateMask sets the optional parameter "updateMask": List of fields being
// updated. The following fields are supported (in both `snake_case` and
// `lowerCamelCase`): - `address` - `customer_service` -
// `korean_business_registration_number`
func (c *AccountsBusinessInfoUpdateBusinessInfoCall) UpdateMask(updateMask string) *AccountsBusinessInfoUpdateBusinessInfoCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsBusinessInfoUpdateBusinessInfoCall) Fields(s ...googleapi.Field) *AccountsBusinessInfoUpdateBusinessInfoCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsBusinessInfoUpdateBusinessInfoCall) Context(ctx context.Context) *AccountsBusinessInfoUpdateBusinessInfoCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsBusinessInfoUpdateBusinessInfoCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsBusinessInfoUpdateBusinessInfoCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.businessinfo)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.businessInfo.updateBusinessInfo", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.businessInfo.updateBusinessInfo" call.
// Any non-2xx status code is an error. Response headers are in either
// *BusinessInfo.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsBusinessInfoUpdateBusinessInfoCall) Do(opts ...googleapi.CallOption) (*BusinessInfo, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &BusinessInfo{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.businessInfo.updateBusinessInfo", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsEmailPreferencesGetEmailPreferencesCall struct {
	s            *APIService
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetEmailPreferences: Returns the email preferences for a Merchant Center
// account user. This service only permits retrieving and updating email
// preferences for the authenticated user. Use the
// name=accounts/*/users/me/emailPreferences alias to get preferences for the
// authenticated user.
//
//   - name: The name of the `EmailPreferences` resource. Format:
//     `accounts/{account}/users/{email}/emailPreferences`.
func (r *AccountsEmailPreferencesService) GetEmailPreferences(name string) *AccountsEmailPreferencesGetEmailPreferencesCall {
	c := &AccountsEmailPreferencesGetEmailPreferencesCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsEmailPreferencesGetEmailPreferencesCall) Fields(s ...googleapi.Field) *AccountsEmailPreferencesGetEmailPreferencesCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsEmailPreferencesGetEmailPreferencesCall) IfNoneMatch(entityTag string) *AccountsEmailPreferencesGetEmailPreferencesCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsEmailPreferencesGetEmailPreferencesCall) Context(ctx context.Context) *AccountsEmailPreferencesGetEmailPreferencesCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsEmailPreferencesGetEmailPreferencesCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsEmailPreferencesGetEmailPreferencesCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.emailPreferences.getEmailPreferences", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.emailPreferences.getEmailPreferences" call.
// Any non-2xx status code is an error. Response headers are in either
// *EmailPreferences.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsEmailPreferencesGetEmailPreferencesCall) Do(opts ...googleapi.CallOption) (*EmailPreferences, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &EmailPreferences{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.emailPreferences.getEmailPreferences", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsEmailPreferencesUpdateEmailPreferencesCall struct {
	s                *APIService
	name             string
	emailpreferences *EmailPreferences
	urlParams_       gensupport.URLParams
	ctx_             context.Context
	header_          http.Header
}

// UpdateEmailPreferences: Updates the email preferences for a Merchant Center
// account user. MCA users should specify the MCA account rather than a
// sub-account of the MCA. Preferences which are not explicitly selected in the
// update mask will not be updated. It is invalid for updates to specify an
// UNCONFIRMED opt-in status value. Use the
// name=accounts/*/users/me/emailPreferences alias to update preferences for
// the authenticated user.
//
//   - name: Identifier. The name of the EmailPreferences. The endpoint is only
//     supported for the authenticated user.
func (r *AccountsEmailPreferencesService) UpdateEmailPreferences(name string, emailpreferences *EmailPreferences) *AccountsEmailPreferencesUpdateEmailPreferencesCall {
	c := &AccountsEmailPreferencesUpdateEmailPreferencesCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.emailpreferences = emailpreferences
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. List of
// fields being updated. The following fields are supported (in both
// `snake_case` and `lowerCamelCase`): - `news_and_tips`
func (c *AccountsEmailPreferencesUpdateEmailPreferencesCall) UpdateMask(updateMask string) *AccountsEmailPreferencesUpdateEmailPreferencesCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsEmailPreferencesUpdateEmailPreferencesCall) Fields(s ...googleapi.Field) *AccountsEmailPreferencesUpdateEmailPreferencesCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsEmailPreferencesUpdateEmailPreferencesCall) Context(ctx context.Context) *AccountsEmailPreferencesUpdateEmailPreferencesCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsEmailPreferencesUpdateEmailPreferencesCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsEmailPreferencesUpdateEmailPreferencesCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.emailpreferences)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.emailPreferences.updateEmailPreferences", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.emailPreferences.updateEmailPreferences" call.
// Any non-2xx status code is an error. Response headers are in either
// *EmailPreferences.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsEmailPreferencesUpdateEmailPreferencesCall) Do(opts ...googleapi.CallOption) (*EmailPreferences, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &EmailPreferences{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.emailPreferences.updateEmailPreferences", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsHomepageClaimCall struct {
	s                    *APIService
	name                 string
	claimhomepagerequest *ClaimHomepageRequest
	urlParams_           gensupport.URLParams
	ctx_                 context.Context
	header_              http.Header
}

// Claim: Claims a store's homepage. Executing this method requires admin
// access. If the homepage is already claimed, this will recheck the
// verification (unless the merchant is exempted from claiming, which also
// exempts from verification) and return a successful response. If ownership
// can no longer be verified, it will return an error, but it won't clear the
// claim. In case of failure, a canonical error message will be returned: *
// PERMISSION_DENIED: user doesn't have the necessary permissions on this MC
// account; * FAILED_PRECONDITION: - The account is not a Merchant Center
// account; - MC account doesn't have a homepage; - claiming failed (in this
// case the error message will contain more details).
//
//   - name: The name of the homepage to claim. Format:
//     `accounts/{account}/homepage`.
func (r *AccountsHomepageService) Claim(name string, claimhomepagerequest *ClaimHomepageRequest) *AccountsHomepageClaimCall {
	c := &AccountsHomepageClaimCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.claimhomepagerequest = claimhomepagerequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsHomepageClaimCall) Fields(s ...googleapi.Field) *AccountsHomepageClaimCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsHomepageClaimCall) Context(ctx context.Context) *AccountsHomepageClaimCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsHomepageClaimCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsHomepageClaimCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.claimhomepagerequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}:claim")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.homepage.claim", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.homepage.claim" call.
// Any non-2xx status code is an error. Response headers are in either
// *Homepage.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsHomepageClaimCall) Do(opts ...googleapi.CallOption) (*Homepage, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Homepage{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.homepage.claim", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsHomepageGetHomepageCall struct {
	s            *APIService
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetHomepage: Retrieves a store's homepage.
//
//   - name: The name of the homepage to retrieve. Format:
//     `accounts/{account}/homepage`.
func (r *AccountsHomepageService) GetHomepage(name string) *AccountsHomepageGetHomepageCall {
	c := &AccountsHomepageGetHomepageCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsHomepageGetHomepageCall) Fields(s ...googleapi.Field) *AccountsHomepageGetHomepageCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsHomepageGetHomepageCall) IfNoneMatch(entityTag string) *AccountsHomepageGetHomepageCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsHomepageGetHomepageCall) Context(ctx context.Context) *AccountsHomepageGetHomepageCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsHomepageGetHomepageCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsHomepageGetHomepageCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.homepage.getHomepage", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.homepage.getHomepage" call.
// Any non-2xx status code is an error. Response headers are in either
// *Homepage.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsHomepageGetHomepageCall) Do(opts ...googleapi.CallOption) (*Homepage, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Homepage{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.homepage.getHomepage", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsHomepageUnclaimCall struct {
	s                      *APIService
	name                   string
	unclaimhomepagerequest *UnclaimHomepageRequest
	urlParams_             gensupport.URLParams
	ctx_                   context.Context
	header_                http.Header
}

// Unclaim: Unclaims a store's homepage. Executing this method requires admin
// access.
//
//   - name: The name of the homepage to unclaim. Format:
//     `accounts/{account}/homepage`.
func (r *AccountsHomepageService) Unclaim(name string, unclaimhomepagerequest *UnclaimHomepageRequest) *AccountsHomepageUnclaimCall {
	c := &AccountsHomepageUnclaimCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.unclaimhomepagerequest = unclaimhomepagerequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsHomepageUnclaimCall) Fields(s ...googleapi.Field) *AccountsHomepageUnclaimCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsHomepageUnclaimCall) Context(ctx context.Context) *AccountsHomepageUnclaimCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsHomepageUnclaimCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsHomepageUnclaimCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.unclaimhomepagerequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}:unclaim")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.homepage.unclaim", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.homepage.unclaim" call.
// Any non-2xx status code is an error. Response headers are in either
// *Homepage.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsHomepageUnclaimCall) Do(opts ...googleapi.CallOption) (*Homepage, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Homepage{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.homepage.unclaim", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsHomepageUpdateHomepageCall struct {
	s          *APIService
	name       string
	homepage   *Homepage
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// UpdateHomepage: Updates a store's homepage. Executing this method requires
// admin access.
//
//   - name: Identifier. The resource name of the store's homepage. Format:
//     `accounts/{account}/homepage`.
func (r *AccountsHomepageService) UpdateHomepage(name string, homepage *Homepage) *AccountsHomepageUpdateHomepageCall {
	c := &AccountsHomepageUpdateHomepageCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.homepage = homepage
	return c
}

// UpdateMask sets the optional parameter "updateMask": List of fields being
// updated. The following fields are supported (in both `snake_case` and
// `lowerCamelCase`): - `uri`
func (c *AccountsHomepageUpdateHomepageCall) UpdateMask(updateMask string) *AccountsHomepageUpdateHomepageCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsHomepageUpdateHomepageCall) Fields(s ...googleapi.Field) *AccountsHomepageUpdateHomepageCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsHomepageUpdateHomepageCall) Context(ctx context.Context) *AccountsHomepageUpdateHomepageCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsHomepageUpdateHomepageCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsHomepageUpdateHomepageCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.homepage)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.homepage.updateHomepage", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.homepage.updateHomepage" call.
// Any non-2xx status code is an error. Response headers are in either
// *Homepage.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsHomepageUpdateHomepageCall) Do(opts ...googleapi.CallOption) (*Homepage, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Homepage{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.homepage.updateHomepage", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsIssuesListCall struct {
	s            *APIService
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all account issues of a Merchant Center account.
//
//   - parent: The parent, which owns this collection of issues. Format:
//     `accounts/{account}`.
func (r *AccountsIssuesService) List(parent string) *AccountsIssuesListCall {
	c := &AccountsIssuesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// LanguageCode sets the optional parameter "languageCode": The issues in the
// response will have human-readable fields in the given language. The format
// is BCP-47 (https://tools.ietf.org/html/bcp47), such as `en-US` or `sr-Latn`.
// If not value is provided, `en-US` will be used.
func (c *AccountsIssuesListCall) LanguageCode(languageCode string) *AccountsIssuesListCall {
	c.urlParams_.Set("languageCode", languageCode)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// issues to return. The service may return fewer than this value. If
// unspecified, at most 50 users will be returned. The maximum value is 100;
// values above 100 will be coerced to 100
func (c *AccountsIssuesListCall) PageSize(pageSize int64) *AccountsIssuesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListAccountIssues` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListAccountIssues` must match the call that provided the page token.
func (c *AccountsIssuesListCall) PageToken(pageToken string) *AccountsIssuesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// TimeZone sets the optional parameter "timeZone": The IANA
// (https://www.iana.org/time-zones) timezone used to localize times in
// human-readable fields. For example 'America/Los_Angeles'. If not set,
// 'America/Los_Angeles' will be used.
func (c *AccountsIssuesListCall) TimeZone(timeZone string) *AccountsIssuesListCall {
	c.urlParams_.Set("timeZone", timeZone)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsIssuesListCall) Fields(s ...googleapi.Field) *AccountsIssuesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsIssuesListCall) IfNoneMatch(entityTag string) *AccountsIssuesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsIssuesListCall) Context(ctx context.Context) *AccountsIssuesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsIssuesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsIssuesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+parent}/issues")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.issues.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.issues.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListAccountIssuesResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsIssuesListCall) Do(opts ...googleapi.CallOption) (*ListAccountIssuesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListAccountIssuesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.issues.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsIssuesListCall) Pages(ctx context.Context, f func(*ListAccountIssuesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsOnlineReturnPoliciesGetCall struct {
	s            *APIService
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets an existing return policy for a given merchant.
//
//   - name: The name of the return policy to retrieve. Format:
//     `accounts/{account}/onlineReturnPolicies/{return_policy}`.
func (r *AccountsOnlineReturnPoliciesService) Get(name string) *AccountsOnlineReturnPoliciesGetCall {
	c := &AccountsOnlineReturnPoliciesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsOnlineReturnPoliciesGetCall) Fields(s ...googleapi.Field) *AccountsOnlineReturnPoliciesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsOnlineReturnPoliciesGetCall) IfNoneMatch(entityTag string) *AccountsOnlineReturnPoliciesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsOnlineReturnPoliciesGetCall) Context(ctx context.Context) *AccountsOnlineReturnPoliciesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsOnlineReturnPoliciesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsOnlineReturnPoliciesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.onlineReturnPolicies.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.onlineReturnPolicies.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *OnlineReturnPolicy.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsOnlineReturnPoliciesGetCall) Do(opts ...googleapi.CallOption) (*OnlineReturnPolicy, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &OnlineReturnPolicy{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.onlineReturnPolicies.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsOnlineReturnPoliciesListCall struct {
	s            *APIService
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all existing return policies for a given merchant.
//
//   - parent: The merchant account for which to list return policies. Format:
//     `accounts/{account}`.
func (r *AccountsOnlineReturnPoliciesService) List(parent string) *AccountsOnlineReturnPoliciesListCall {
	c := &AccountsOnlineReturnPoliciesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// `OnlineReturnPolicy` resources to return. The service returns fewer than
// this value if the number of return policies for the given merchant is less
// that than the `pageSize`. The default value is 10. The maximum value is 100;
// If a value higher than the maximum is specified, then the `pageSize` will
// default to the maximum
func (c *AccountsOnlineReturnPoliciesListCall) PageSize(pageSize int64) *AccountsOnlineReturnPoliciesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListOnlineReturnPolicies` call. Provide the page token to
// retrieve the subsequent page. When paginating, all other parameters provided
// to `ListOnlineReturnPolicies` must match the call that provided the page
// token. The token returned as nextPageToken in the response to the previous
// request.
func (c *AccountsOnlineReturnPoliciesListCall) PageToken(pageToken string) *AccountsOnlineReturnPoliciesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsOnlineReturnPoliciesListCall) Fields(s ...googleapi.Field) *AccountsOnlineReturnPoliciesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsOnlineReturnPoliciesListCall) IfNoneMatch(entityTag string) *AccountsOnlineReturnPoliciesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsOnlineReturnPoliciesListCall) Context(ctx context.Context) *AccountsOnlineReturnPoliciesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsOnlineReturnPoliciesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsOnlineReturnPoliciesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+parent}/onlineReturnPolicies")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.onlineReturnPolicies.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.onlineReturnPolicies.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListOnlineReturnPoliciesResponse.ServerResponse.Header or (if a response
// was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsOnlineReturnPoliciesListCall) Do(opts ...googleapi.CallOption) (*ListOnlineReturnPoliciesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListOnlineReturnPoliciesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.onlineReturnPolicies.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsOnlineReturnPoliciesListCall) Pages(ctx context.Context, f func(*ListOnlineReturnPoliciesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsProgramsDisableCall struct {
	s                     *APIService
	name                  string
	disableprogramrequest *DisableProgramRequest
	urlParams_            gensupport.URLParams
	ctx_                  context.Context
	header_               http.Header
}

// Disable: Disable participation in the specified program for the account.
// Executing this method requires admin access.
//
//   - name: The name of the program for which to disable participation for the
//     given account. Format: `accounts/{account}/programs/{program}`.
func (r *AccountsProgramsService) Disable(name string, disableprogramrequest *DisableProgramRequest) *AccountsProgramsDisableCall {
	c := &AccountsProgramsDisableCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.disableprogramrequest = disableprogramrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProgramsDisableCall) Fields(s ...googleapi.Field) *AccountsProgramsDisableCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProgramsDisableCall) Context(ctx context.Context) *AccountsProgramsDisableCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProgramsDisableCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProgramsDisableCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.disableprogramrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}:disable")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.programs.disable", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.programs.disable" call.
// Any non-2xx status code is an error. Response headers are in either
// *Program.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsProgramsDisableCall) Do(opts ...googleapi.CallOption) (*Program, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Program{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.programs.disable", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsProgramsEnableCall struct {
	s                    *APIService
	name                 string
	enableprogramrequest *EnableProgramRequest
	urlParams_           gensupport.URLParams
	ctx_                 context.Context
	header_              http.Header
}

// Enable: Enable participation in the specified program for the account.
// Executing this method requires admin access.
//
//   - name: The name of the program for which to enable participation for the
//     given account. Format: `accounts/{account}/programs/{program}`.
func (r *AccountsProgramsService) Enable(name string, enableprogramrequest *EnableProgramRequest) *AccountsProgramsEnableCall {
	c := &AccountsProgramsEnableCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.enableprogramrequest = enableprogramrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProgramsEnableCall) Fields(s ...googleapi.Field) *AccountsProgramsEnableCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProgramsEnableCall) Context(ctx context.Context) *AccountsProgramsEnableCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProgramsEnableCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProgramsEnableCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.enableprogramrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}:enable")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.programs.enable", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.programs.enable" call.
// Any non-2xx status code is an error. Response headers are in either
// *Program.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsProgramsEnableCall) Do(opts ...googleapi.CallOption) (*Program, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Program{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.programs.enable", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsProgramsGetCall struct {
	s            *APIService
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves the specified program for the account.
//
//   - name: The name of the program to retrieve. Format:
//     `accounts/{account}/programs/{program}`.
func (r *AccountsProgramsService) Get(name string) *AccountsProgramsGetCall {
	c := &AccountsProgramsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProgramsGetCall) Fields(s ...googleapi.Field) *AccountsProgramsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsProgramsGetCall) IfNoneMatch(entityTag string) *AccountsProgramsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProgramsGetCall) Context(ctx context.Context) *AccountsProgramsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProgramsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProgramsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.programs.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.programs.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Program.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsProgramsGetCall) Do(opts ...googleapi.CallOption) (*Program, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Program{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.programs.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsProgramsListCall struct {
	s            *APIService
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Retrieves all programs for the account.
//
//   - parent: The name of the account for which to retrieve all programs.
//     Format: `accounts/{account}`.
func (r *AccountsProgramsService) List(parent string) *AccountsProgramsListCall {
	c := &AccountsProgramsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// programs to return in a single response. If unspecified (or 0), a default
// size of 1000 is used. The maximum value is 1000; values above 1000 will be
// coerced to 1000.
func (c *AccountsProgramsListCall) PageSize(pageSize int64) *AccountsProgramsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A continuation token,
// received from a previous `ListPrograms` call. Provide this to retrieve the
// next page.
func (c *AccountsProgramsListCall) PageToken(pageToken string) *AccountsProgramsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProgramsListCall) Fields(s ...googleapi.Field) *AccountsProgramsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsProgramsListCall) IfNoneMatch(entityTag string) *AccountsProgramsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProgramsListCall) Context(ctx context.Context) *AccountsProgramsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProgramsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProgramsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+parent}/programs")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.programs.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.programs.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListProgramsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsProgramsListCall) Do(opts ...googleapi.CallOption) (*ListProgramsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListProgramsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.programs.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsProgramsListCall) Pages(ctx context.Context, f func(*ListProgramsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsRegionsCreateCall struct {
	s          *APIService
	parent     string
	region     *Region
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a region definition in your Merchant Center account.
// Executing this method requires admin access.
//
// - parent: The account to create a region for. Format: `accounts/{account}`.
func (r *AccountsRegionsService) Create(parent string, region *Region) *AccountsRegionsCreateCall {
	c := &AccountsRegionsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.region = region
	return c
}

// RegionId sets the optional parameter "regionId": Required. The identifier
// for the region, unique over all regions of the same account.
func (c *AccountsRegionsCreateCall) RegionId(regionId string) *AccountsRegionsCreateCall {
	c.urlParams_.Set("regionId", regionId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsRegionsCreateCall) Fields(s ...googleapi.Field) *AccountsRegionsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsRegionsCreateCall) Context(ctx context.Context) *AccountsRegionsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsRegionsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsRegionsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.region)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+parent}/regions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.regions.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.regions.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Region.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsRegionsCreateCall) Do(opts ...googleapi.CallOption) (*Region, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Region{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.regions.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsRegionsDeleteCall struct {
	s          *APIService
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a region definition from your Merchant Center account.
// Executing this method requires admin access.
//
//   - name: The name of the region to delete. Format:
//     `accounts/{account}/regions/{region}`.
func (r *AccountsRegionsService) Delete(name string) *AccountsRegionsDeleteCall {
	c := &AccountsRegionsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsRegionsDeleteCall) Fields(s ...googleapi.Field) *AccountsRegionsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsRegionsDeleteCall) Context(ctx context.Context) *AccountsRegionsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsRegionsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsRegionsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.regions.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.regions.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsRegionsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.regions.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsRegionsGetCall struct {
	s            *APIService
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves a region defined in your Merchant Center account.
//
//   - name: The name of the region to retrieve. Format:
//     `accounts/{account}/regions/{region}`.
func (r *AccountsRegionsService) Get(name string) *AccountsRegionsGetCall {
	c := &AccountsRegionsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsRegionsGetCall) Fields(s ...googleapi.Field) *AccountsRegionsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsRegionsGetCall) IfNoneMatch(entityTag string) *AccountsRegionsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsRegionsGetCall) Context(ctx context.Context) *AccountsRegionsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsRegionsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsRegionsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.regions.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.regions.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Region.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsRegionsGetCall) Do(opts ...googleapi.CallOption) (*Region, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Region{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.regions.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsRegionsListCall struct {
	s            *APIService
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the regions in your Merchant Center account.
//
// - parent: The account to list regions for. Format: `accounts/{account}`.
func (r *AccountsRegionsService) List(parent string) *AccountsRegionsListCall {
	c := &AccountsRegionsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// regions to return. The service may return fewer than this value. If
// unspecified, at most 50 regions will be returned. The maximum value is 1000;
// values above 1000 will be coerced to 1000.
func (c *AccountsRegionsListCall) PageSize(pageSize int64) *AccountsRegionsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListRegions` call. Provide this to retrieve the subsequent
// page. When paginating, all other parameters provided to `ListRegions` must
// match the call that provided the page token.
func (c *AccountsRegionsListCall) PageToken(pageToken string) *AccountsRegionsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsRegionsListCall) Fields(s ...googleapi.Field) *AccountsRegionsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsRegionsListCall) IfNoneMatch(entityTag string) *AccountsRegionsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsRegionsListCall) Context(ctx context.Context) *AccountsRegionsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsRegionsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsRegionsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+parent}/regions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.regions.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.regions.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListRegionsResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsRegionsListCall) Do(opts ...googleapi.CallOption) (*ListRegionsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListRegionsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.regions.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsRegionsListCall) Pages(ctx context.Context, f func(*ListRegionsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsRegionsPatchCall struct {
	s          *APIService
	name       string
	region     *Region
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates a region definition in your Merchant Center account.
// Executing this method requires admin access.
//
//   - name: Identifier. The resource name of the region. Format:
//     `accounts/{account}/regions/{region}`.
func (r *AccountsRegionsService) Patch(name string, region *Region) *AccountsRegionsPatchCall {
	c := &AccountsRegionsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.region = region
	return c
}

// UpdateMask sets the optional parameter "updateMask": The comma-separated
// field mask indicating the fields to update. Example:
// "displayName,postalCodeArea.regionCode".
func (c *AccountsRegionsPatchCall) UpdateMask(updateMask string) *AccountsRegionsPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsRegionsPatchCall) Fields(s ...googleapi.Field) *AccountsRegionsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsRegionsPatchCall) Context(ctx context.Context) *AccountsRegionsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsRegionsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsRegionsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.region)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.regions.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.regions.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Region.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsRegionsPatchCall) Do(opts ...googleapi.CallOption) (*Region, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Region{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.regions.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsShippingSettingsGetShippingSettingsCall struct {
	s            *APIService
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetShippingSettings: Retrieve shipping setting information.
//
//   - name: The name of the shipping setting to retrieve. Format:
//     `accounts/{account}/shippingsetting`.
func (r *AccountsShippingSettingsService) GetShippingSettings(name string) *AccountsShippingSettingsGetShippingSettingsCall {
	c := &AccountsShippingSettingsGetShippingSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsShippingSettingsGetShippingSettingsCall) Fields(s ...googleapi.Field) *AccountsShippingSettingsGetShippingSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsShippingSettingsGetShippingSettingsCall) IfNoneMatch(entityTag string) *AccountsShippingSettingsGetShippingSettingsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsShippingSettingsGetShippingSettingsCall) Context(ctx context.Context) *AccountsShippingSettingsGetShippingSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsShippingSettingsGetShippingSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsShippingSettingsGetShippingSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.shippingSettings.getShippingSettings", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.shippingSettings.getShippingSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *ShippingSettings.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsShippingSettingsGetShippingSettingsCall) Do(opts ...googleapi.CallOption) (*ShippingSettings, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ShippingSettings{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.shippingSettings.getShippingSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsShippingSettingsInsertCall struct {
	s                *APIService
	parent           string
	shippingsettings *ShippingSettings
	urlParams_       gensupport.URLParams
	ctx_             context.Context
	header_          http.Header
}

// Insert: Replace the shipping setting of a merchant with the request shipping
// setting. Executing this method requires admin access.
//
//   - parent: The account where this product will be inserted. Format:
//     accounts/{account}.
func (r *AccountsShippingSettingsService) Insert(parent string, shippingsettings *ShippingSettings) *AccountsShippingSettingsInsertCall {
	c := &AccountsShippingSettingsInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.shippingsettings = shippingsettings
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsShippingSettingsInsertCall) Fields(s ...googleapi.Field) *AccountsShippingSettingsInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsShippingSettingsInsertCall) Context(ctx context.Context) *AccountsShippingSettingsInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsShippingSettingsInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsShippingSettingsInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.shippingsettings)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+parent}/shippingSettings:insert")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.shippingSettings.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.shippingSettings.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *ShippingSettings.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsShippingSettingsInsertCall) Do(opts ...googleapi.CallOption) (*ShippingSettings, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ShippingSettings{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.shippingSettings.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsTermsOfServiceAgreementStatesGetCall struct {
	s            *APIService
	nameid       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns the state of a terms of service agreement.
//
//   - name: The resource name of the terms of service version. Format:
//     `accounts/{account}/termsOfServiceAgreementState/{identifier}` The
//     identifier format is: `{TermsOfServiceKind}-{country}`.
func (r *AccountsTermsOfServiceAgreementStatesService) Get(nameid string) *AccountsTermsOfServiceAgreementStatesGetCall {
	c := &AccountsTermsOfServiceAgreementStatesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.nameid = nameid
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsTermsOfServiceAgreementStatesGetCall) Fields(s ...googleapi.Field) *AccountsTermsOfServiceAgreementStatesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsTermsOfServiceAgreementStatesGetCall) IfNoneMatch(entityTag string) *AccountsTermsOfServiceAgreementStatesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsTermsOfServiceAgreementStatesGetCall) Context(ctx context.Context) *AccountsTermsOfServiceAgreementStatesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsTermsOfServiceAgreementStatesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsTermsOfServiceAgreementStatesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.nameid,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.termsOfServiceAgreementStates.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.termsOfServiceAgreementStates.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *TermsOfServiceAgreementState.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsTermsOfServiceAgreementStatesGetCall) Do(opts ...googleapi.CallOption) (*TermsOfServiceAgreementState, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TermsOfServiceAgreementState{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.termsOfServiceAgreementStates.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsTermsOfServiceAgreementStatesRetrieveForApplicationCall struct {
	s            *APIService
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// RetrieveForApplication: Retrieves the state of the agreement for the
// application terms of service.
//
//   - parent: The account for which to get a TermsOfServiceAgreementState
//     Format: `accounts/{account}`.
func (r *AccountsTermsOfServiceAgreementStatesService) RetrieveForApplication(parent string) *AccountsTermsOfServiceAgreementStatesRetrieveForApplicationCall {
	c := &AccountsTermsOfServiceAgreementStatesRetrieveForApplicationCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsTermsOfServiceAgreementStatesRetrieveForApplicationCall) Fields(s ...googleapi.Field) *AccountsTermsOfServiceAgreementStatesRetrieveForApplicationCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsTermsOfServiceAgreementStatesRetrieveForApplicationCall) IfNoneMatch(entityTag string) *AccountsTermsOfServiceAgreementStatesRetrieveForApplicationCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsTermsOfServiceAgreementStatesRetrieveForApplicationCall) Context(ctx context.Context) *AccountsTermsOfServiceAgreementStatesRetrieveForApplicationCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsTermsOfServiceAgreementStatesRetrieveForApplicationCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsTermsOfServiceAgreementStatesRetrieveForApplicationCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+parent}/termsOfServiceAgreementStates:retrieveForApplication")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.termsOfServiceAgreementStates.retrieveForApplication", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.termsOfServiceAgreementStates.retrieveForApplication" call.
// Any non-2xx status code is an error. Response headers are in either
// *TermsOfServiceAgreementState.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsTermsOfServiceAgreementStatesRetrieveForApplicationCall) Do(opts ...googleapi.CallOption) (*TermsOfServiceAgreementState, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TermsOfServiceAgreementState{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.termsOfServiceAgreementStates.retrieveForApplication", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsUsersCreateCall struct {
	s          *APIService
	parent     string
	user       *User
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a Merchant Center account user. Executing this method
// requires admin access.
//
//   - parent: The resource name of the account for which a user will be created.
//     Format: `accounts/{account}`.
func (r *AccountsUsersService) Create(parent string, user *User) *AccountsUsersCreateCall {
	c := &AccountsUsersCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.user = user
	return c
}

// UserId sets the optional parameter "userId": Required. The email address of
// the user (for example, `john.doe@gmail.com`).
func (c *AccountsUsersCreateCall) UserId(userId string) *AccountsUsersCreateCall {
	c.urlParams_.Set("userId", userId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsUsersCreateCall) Fields(s ...googleapi.Field) *AccountsUsersCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsUsersCreateCall) Context(ctx context.Context) *AccountsUsersCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsUsersCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsUsersCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.user)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+parent}/users")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.users.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.users.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *User.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsUsersCreateCall) Do(opts ...googleapi.CallOption) (*User, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &User{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.users.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsUsersDeleteCall struct {
	s          *APIService
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a Merchant Center account user. Executing this method
// requires admin access. The user to be deleted can't be the last admin user
// of that account. Also a user is protected from deletion if it is managed by
// Business Manager"
//
//   - name: The name of the user to delete. Format:
//     `accounts/{account}/users/{email}` It is also possible to delete the user
//     corresponding to the caller by using `me` rather than an email address as
//     in `accounts/{account}/users/me`.
func (r *AccountsUsersService) Delete(name string) *AccountsUsersDeleteCall {
	c := &AccountsUsersDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsUsersDeleteCall) Fields(s ...googleapi.Field) *AccountsUsersDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsUsersDeleteCall) Context(ctx context.Context) *AccountsUsersDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsUsersDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsUsersDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.users.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.users.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsUsersDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.users.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsUsersGetCall struct {
	s            *APIService
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves a Merchant Center account user.
//
//   - name: The name of the user to retrieve. Format:
//     `accounts/{account}/users/{email}` It is also possible to retrieve the
//     user corresponding to the caller by using `me` rather than an email
//     address as in `accounts/{account}/users/me`.
func (r *AccountsUsersService) Get(name string) *AccountsUsersGetCall {
	c := &AccountsUsersGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsUsersGetCall) Fields(s ...googleapi.Field) *AccountsUsersGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsUsersGetCall) IfNoneMatch(entityTag string) *AccountsUsersGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsUsersGetCall) Context(ctx context.Context) *AccountsUsersGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsUsersGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsUsersGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.users.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.users.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *User.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsUsersGetCall) Do(opts ...googleapi.CallOption) (*User, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &User{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.users.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsUsersListCall struct {
	s            *APIService
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all users of a Merchant Center account.
//
//   - parent: The parent, which owns this collection of users. Format:
//     `accounts/{account}`.
func (r *AccountsUsersService) List(parent string) *AccountsUsersListCall {
	c := &AccountsUsersListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of users
// to return. The service may return fewer than this value. If unspecified, at
// most 50 users will be returned. The maximum value is 100; values above 100
// will be coerced to 100
func (c *AccountsUsersListCall) PageSize(pageSize int64) *AccountsUsersListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListUsers` call. Provide this to retrieve the subsequent
// page. When paginating, all other parameters provided to `ListUsers` must
// match the call that provided the page token.
func (c *AccountsUsersListCall) PageToken(pageToken string) *AccountsUsersListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsUsersListCall) Fields(s ...googleapi.Field) *AccountsUsersListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsUsersListCall) IfNoneMatch(entityTag string) *AccountsUsersListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsUsersListCall) Context(ctx context.Context) *AccountsUsersListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsUsersListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsUsersListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+parent}/users")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.users.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.users.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListUsersResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsUsersListCall) Do(opts ...googleapi.CallOption) (*ListUsersResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListUsersResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.users.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsUsersListCall) Pages(ctx context.Context, f func(*ListUsersResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsUsersPatchCall struct {
	s          *APIService
	name       string
	user       *User
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates a Merchant Center account user. Executing this method
// requires admin access.
//
//   - name: Identifier. The resource name of the user. Format:
//     `accounts/{account}/user/{email}` Use `me` to refer to your own email
//     address, for example `accounts/{account}/users/me`.
func (r *AccountsUsersService) Patch(name string, user *User) *AccountsUsersPatchCall {
	c := &AccountsUsersPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.user = user
	return c
}

// UpdateMask sets the optional parameter "updateMask": List of fields being
// updated. The following fields are supported (in both `snake_case` and
// `lowerCamelCase`): - `access_rights`
func (c *AccountsUsersPatchCall) UpdateMask(updateMask string) *AccountsUsersPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsUsersPatchCall) Fields(s ...googleapi.Field) *AccountsUsersPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsUsersPatchCall) Context(ctx context.Context) *AccountsUsersPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsUsersPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsUsersPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.user)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.users.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.users.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *User.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsUsersPatchCall) Do(opts ...googleapi.CallOption) (*User, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &User{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.users.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TermsOfServiceAcceptCall struct {
	s            *APIService
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Accept: Accepts a `TermsOfService`. Executing this method requires admin
// access.
//
//   - name: The resource name of the terms of service version. Format:
//     `termsOfService/{version}`.
func (r *TermsOfServiceService) Accept(name string) *TermsOfServiceAcceptCall {
	c := &TermsOfServiceAcceptCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Account sets the optional parameter "account": Required. The account for
// which to accept the ToS.
func (c *TermsOfServiceAcceptCall) Account(account string) *TermsOfServiceAcceptCall {
	c.urlParams_.Set("account", account)
	return c
}

// RegionCode sets the optional parameter "regionCode": Required. Region code
// as defined by CLDR (https://cldr.unicode.org/). This is either a country
// when the ToS applies specifically to that country or 001 when it applies
// globally.
func (c *TermsOfServiceAcceptCall) RegionCode(regionCode string) *TermsOfServiceAcceptCall {
	c.urlParams_.Set("regionCode", regionCode)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TermsOfServiceAcceptCall) Fields(s ...googleapi.Field) *TermsOfServiceAcceptCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *TermsOfServiceAcceptCall) IfNoneMatch(entityTag string) *TermsOfServiceAcceptCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TermsOfServiceAcceptCall) Context(ctx context.Context) *TermsOfServiceAcceptCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TermsOfServiceAcceptCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TermsOfServiceAcceptCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}:accept")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.termsOfService.accept", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.termsOfService.accept" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TermsOfServiceAcceptCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.termsOfService.accept", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TermsOfServiceGetCall struct {
	s            *APIService
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves the `TermsOfService` associated with the provided version.
//
//   - name: The resource name of the terms of service version. Format:
//     `termsOfService/{version}`.
func (r *TermsOfServiceService) Get(name string) *TermsOfServiceGetCall {
	c := &TermsOfServiceGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TermsOfServiceGetCall) Fields(s ...googleapi.Field) *TermsOfServiceGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *TermsOfServiceGetCall) IfNoneMatch(entityTag string) *TermsOfServiceGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TermsOfServiceGetCall) Context(ctx context.Context) *TermsOfServiceGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TermsOfServiceGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TermsOfServiceGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.termsOfService.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.termsOfService.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *TermsOfService.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TermsOfServiceGetCall) Do(opts ...googleapi.CallOption) (*TermsOfService, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TermsOfService{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.termsOfService.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TermsOfServiceRetrieveLatestCall struct {
	s            *APIService
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// RetrieveLatest: Retrieves the latest version of the `TermsOfService` for a
// given `kind` and `region_code`.
func (r *TermsOfServiceService) RetrieveLatest() *TermsOfServiceRetrieveLatestCall {
	c := &TermsOfServiceRetrieveLatestCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Kind sets the optional parameter "kind": Required. The Kind this terms of
// service version applies to.
//
// Possible values:
//
//	"TERMS_OF_SERVICE_KIND_UNSPECIFIED" - Default value. This value is unused.
//	"MERCHANT_CENTER" - Merchant Center application.
func (c *TermsOfServiceRetrieveLatestCall) Kind(kind string) *TermsOfServiceRetrieveLatestCall {
	c.urlParams_.Set("kind", kind)
	return c
}

// RegionCode sets the optional parameter "regionCode": Required. Region code
// as defined by CLDR (https://cldr.unicode.org/). This is either a country
// when the ToS applies specifically to that country or 001 when it applies
// globally.
func (c *TermsOfServiceRetrieveLatestCall) RegionCode(regionCode string) *TermsOfServiceRetrieveLatestCall {
	c.urlParams_.Set("regionCode", regionCode)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TermsOfServiceRetrieveLatestCall) Fields(s ...googleapi.Field) *TermsOfServiceRetrieveLatestCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *TermsOfServiceRetrieveLatestCall) IfNoneMatch(entityTag string) *TermsOfServiceRetrieveLatestCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TermsOfServiceRetrieveLatestCall) Context(ctx context.Context) *TermsOfServiceRetrieveLatestCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TermsOfServiceRetrieveLatestCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TermsOfServiceRetrieveLatestCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "accounts/v1beta/termsOfService:retrieveLatest")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.termsOfService.retrieveLatest", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.termsOfService.retrieveLatest" call.
// Any non-2xx status code is an error. Response headers are in either
// *TermsOfService.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TermsOfServiceRetrieveLatestCall) Do(opts ...googleapi.CallOption) (*TermsOfService, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TermsOfService{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.termsOfService.retrieveLatest", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
