// Copyright 2017 clair authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package rpm

import (
	"testing"

	"github.com/stretchr/testify/require"

	"github.com/coreos/clair/database"
	"github.com/coreos/clair/ext/featurefmt"
	"github.com/coreos/clair/ext/versionfmt/rpm"
)

var expectedBigCaseInfo = []database.LayerFeature{
	{Feature: database.Feature{"libmount", "2.32.1-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libffi", "3.1-16.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libunistring", "0.9.10-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"fedora-repos", "28-5", "rpm", "binary"}},
	{Feature: database.Feature{"libarchive", "3.3.1-4.fc28", "rpm", "source"}},
	{Feature: database.Feature{"langpacks", "1.0-12.fc28", "rpm", "source"}},
	{Feature: database.Feature{"readline", "7.0-11.fc28", "rpm", "source"}},
	{Feature: database.Feature{"gzip", "1.9-3.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libverto", "0.3.0-5.fc28", "rpm", "source"}},
	{Feature: database.Feature{"ncurses-base", "6.1-5.20180224.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libfdisk", "2.32.1-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libselinux", "2.8-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"nss-util", "3.38.0-1.0.fc28", "rpm", "source"}},
	{Feature: database.Feature{"mpfr", "3.1.6-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libunistring", "0.9.10-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libpcap", "14:1.9.0-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libarchive", "3.3.1-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"gmp", "1:6.1.2-7.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"crypto-policies", "20180425-5.git6ad4018.fc28", "rpm", "source"}},
	{Feature: database.Feature{"gzip", "1.9-3.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"fedora-release", "28-2", "rpm", "source"}},
	{Feature: database.Feature{"zlib", "1.2.11-8.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"crypto-policies", "20180425-5.git6ad4018.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"lz4", "1.8.1.2-4.fc28", "rpm", "source"}},
	{Feature: database.Feature{"keyutils", "1.5.10-6.fc28", "rpm", "source"}},
	{Feature: database.Feature{"gpgme", "1.10.0-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libgpg-error", "1.31-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"gnutls", "3.6.3-4.fc28", "rpm", "source"}},
	{Feature: database.Feature{"coreutils", "8.29-7.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libsepol", "2.8-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libssh", "0.8.2-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libpwquality", "1.4.0-7.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"dnf-conf", "2.7.5-12.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"basesystem", "11-5.fc28", "rpm", "source"}},
	{Feature: database.Feature{"setup", "2.11.4-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libmetalink", "0.1.3-6.fc28", "rpm", "source"}},
	{Feature: database.Feature{"texinfo", "6.5-4.fc28", "rpm", "source"}},
	{Feature: database.Feature{"expat", "2.2.5-3.fc28", "rpm", "source"}},
	{Feature: database.Feature{"ncurses", "6.1-5.20180224.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libpwquality", "1.4.0-7.fc28", "rpm", "source"}},
	{Feature: database.Feature{"pcre", "8.42-3.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"sssd", "1.16.3-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"basesystem", "11-5.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"systemd-pam", "238-9.git0e0aa59.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"python3-six", "1.11.0-3.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libcurl", "7.59.0-6.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"qrencode", "3.4.4-5.fc28", "rpm", "source"}},
	{Feature: database.Feature{"xz", "5.2.4-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libpkgconf", "1.4.2-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libzstd", "1.3.5-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"bash", "4.4.23-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"cyrus-sasl", "2.1.27-0.2rc7.fc28", "rpm", "source"}},
	{Feature: database.Feature{"ncurses-libs", "6.1-5.20180224.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"xz-libs", "5.2.4-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"dbus", "1.12.10-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"grep", "3.1-5.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libusbx", "1.0.22-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"audit", "2.8.4-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"sed", "4.5-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"sqlite", "3.22.0-4.fc28", "rpm", "source"}},
	{Feature: database.Feature{"openldap", "2.4.46-3.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"gawk", "4.2.1-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"gpgme", "1.10.0-4.fc28", "rpm", "source"}},
	{Feature: database.Feature{"lvm2", "2.02.177-5.fc28", "rpm", "source"}},
	{Feature: database.Feature{"nspr", "4.19.0-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libsolv", "0.6.35-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"info", "6.5-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"openssl-libs", "1:1.1.0h-3.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libxcrypt", "4.1.2-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libselinux", "2.8-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libgcc", "8.1.1-5.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"cracklib", "2.9.6-13.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"python3-libs", "3.6.6-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"glibc-langpack-en", "2.27-32.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"json-c", "0.13.1-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"gnupg2", "2.2.8-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"openssl", "1:1.1.0h-3.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"glibc-common", "2.27-32.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"p11-kit-trust", "0.23.12-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"zstd", "1.3.5-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libxml2", "2.9.8-4.fc28", "rpm", "source"}},
	{Feature: database.Feature{"dbus", "1:1.12.10-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"ca-certificates", "2018.2.24-1.0.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libcomps", "0.1.8-11.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"nss", "3.38.0-1.0.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libcom_err", "1.44.2-0.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"keyutils-libs", "1.5.10-6.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libseccomp", "2.3.3-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"elfutils-libs", "0.173-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libuuid", "2.32.1-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"pkgconf", "1.4.2-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"grep", "3.1-5.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libpcap", "1.9.0-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"deltarpm", "3.6-25.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"krb5-libs", "1.16.1-13.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"glibc", "2.27-32.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libseccomp", "2.3.3-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libsemanage", "2.8-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"openssl-pkcs11", "0.4.8-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libxml2", "2.9.8-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"e2fsprogs", "1.44.2-0.fc28", "rpm", "source"}},
	{Feature: database.Feature{"file-libs", "5.33-7.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"elfutils-default-yama-scope", "0.173-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"glibc", "2.27-32.fc28", "rpm", "source"}},
	{Feature: database.Feature{"publicsuffix-list-dafsa", "20180514-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"popt", "1.16-14.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libnsl2", "1.2.0-2.20180605git4a062cf.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"lua-libs", "5.3.4-10.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libsemanage", "2.8-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"glibc-minimal-langpack", "2.27-32.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"attr", "2.4.48-3.fc28", "rpm", "source"}},
	{Feature: database.Feature{"gdbm", "1.14.1-4.fc28", "rpm", "source"}},
	{Feature: database.Feature{"pkgconf", "1.4.2-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"acl", "2.2.53-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"gnutls", "3.6.3-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"fedora-repos", "28-5", "rpm", "source"}},
	{Feature: database.Feature{"python3-pip", "9.0.3-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libnsl2", "1.2.0-2.20180605git4a062cf.fc28", "rpm", "source"}},
	{Feature: database.Feature{"rpm", "4.14.1-9.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libutempter", "1.1.6-14.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libdnf", "0.11.1-3.fc28", "rpm", "source"}},
	{Feature: database.Feature{"vim-minimal", "2:8.1.328-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"tzdata", "2018e-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"nettle", "3.4-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"python-pip", "9.0.3-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"python-six", "1.11.0-3.fc28", "rpm", "source"}},
	{Feature: database.Feature{"diffutils", "3.6-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"rpm-plugin-selinux", "4.14.1-9.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"shadow-utils", "2:4.6-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"pkgconf-pkg-config", "1.4.2-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"cracklib-dicts", "2.9.6-13.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libblkid", "2.32.1-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"python-setuptools", "39.2.0-6.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libsss_idmap", "1.16.3-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libksba", "1.3.5-7.fc28", "rpm", "source"}},
	{Feature: database.Feature{"sssd-client", "1.16.3-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"curl", "7.59.0-6.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"pam", "1.3.1-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libsigsegv", "2.11-5.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"langpacks-en", "1.0-12.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"nss-softokn-freebl", "3.38.0-1.0.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"glib2", "2.56.1-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"python3-gobject-base", "3.28.3-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libffi", "3.1-16.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libmodulemd", "1.6.2-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"openssl", "1.1.0h-3.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libyaml", "0.1.7-5.fc28", "rpm", "source"}},
	{Feature: database.Feature{"pam", "1.3.1-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"iptables", "1.6.2-3.fc28", "rpm", "source"}},
	{Feature: database.Feature{"util-linux", "2.32.1-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libsmartcols", "2.32.1-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"dnf", "2.7.5-12.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"glib2", "2.56.1-4.fc28", "rpm", "source"}},
	{Feature: database.Feature{"lua", "5.3.4-10.fc28", "rpm", "source"}},
	{Feature: database.Feature{"nss-softokn", "3.38.0-1.0.fc28", "rpm", "source"}},
	{Feature: database.Feature{"python3-dnf", "2.7.5-12.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"filesystem", "3.8-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libsss_nss_idmap", "1.16.3-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"pcre2", "10.31-10.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libyaml", "0.1.7-5.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"python3-rpm", "4.14.1-9.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"zlib", "1.2.11-8.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libutempter", "1.1.6-14.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"pcre2", "10.31-10.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libtirpc", "1.0.3-3.rc2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"pkgconf-m4", "1.4.2-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libreport", "2.9.5-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"vim", "8.1.328-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"file", "5.33-7.fc28", "rpm", "source"}},
	{Feature: database.Feature{"shadow-utils", "4.6-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"sqlite-libs", "3.22.0-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"setup", "2.11.4-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"gcc", "8.1.1-5.fc28", "rpm", "source"}},
	{Feature: database.Feature{"mpfr", "3.1.6-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"device-mapper", "1.02.146-5.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"p11-kit", "0.23.12-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"fedora-release", "28-2", "rpm", "binary"}},
	{Feature: database.Feature{"libnghttp2", "1.32.1-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libcap-ng", "0.7.9-4.fc28", "rpm", "source"}},
	{Feature: database.Feature{"iptables-libs", "1.6.2-3.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"audit-libs", "2.8.4-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libsigsegv", "2.11-5.fc28", "rpm", "source"}},
	{Feature: database.Feature{"rootfiles", "8.1-22.fc28", "rpm", "source"}},
	{Feature: database.Feature{"kmod-libs", "25-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"lz4-libs", "1.8.1.2-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libassuan", "2.5.1-3.fc28", "rpm", "source"}},
	{Feature: database.Feature{"p11-kit", "0.23.12-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"nss-sysinit", "3.38.0-1.0.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libcap-ng", "0.7.9-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"bash", "4.4.23-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"pygobject3", "3.28.3-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"dnf-yum", "2.7.5-12.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"nss-softokn", "3.38.0-1.0.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"expat", "2.2.5-3.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libassuan", "2.5.1-3.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libdb", "5.3.28-30.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"tar", "2:1.30-3.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"sed", "4.5-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libmetalink", "0.1.3-6.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"python-smartcols", "0.3.0-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"systemd", "238-9.git0e0aa59.fc28", "rpm", "source"}},
	{Feature: database.Feature{"python-iniparse", "0.4-30.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libsepol", "2.8-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libattr", "2.4.48-3.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"python3-smartcols", "0.3.0-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libdb", "5.3.28-30.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libmodulemd", "1.6.2-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"python3-hawkey", "0.11.1-3.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"dbus-libs", "1:1.12.10-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"chkconfig", "1.10-4.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libargon2", "20161029-5.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"openssl-pkcs11", "0.4.8-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libusbx", "1.0.22-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"python3-setuptools", "39.2.0-6.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"chkconfig", "1.10-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"openldap", "2.4.46-3.fc28", "rpm", "source"}},
	{Feature: database.Feature{"bzip2", "1.0.6-26.fc28", "rpm", "source"}},
	{Feature: database.Feature{"npth", "1.5-4.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libtirpc", "1.0.3-3.rc2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"util-linux", "2.32.1-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"nss", "3.38.0-1.0.fc28", "rpm", "source"}},
	{Feature: database.Feature{"elfutils", "0.173-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libcomps", "0.1.8-11.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libxcrypt", "4.1.2-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"gnupg2", "2.2.8-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libdnf", "0.11.1-3.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"cracklib", "2.9.6-13.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libidn2", "2.0.5-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"bzip2-libs", "1.0.6-26.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"json-c", "0.13.1-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"gdbm", "1:1.14.1-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"pcre", "8.42-3.fc28", "rpm", "source"}},
	{Feature: database.Feature{"systemd", "238-9.git0e0aa59.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"cryptsetup-libs", "2.0.4-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"dnf", "2.7.5-12.fc28", "rpm", "source"}},
	{Feature: database.Feature{"ca-certificates", "2018.2.24-1.0.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libidn2", "2.0.5-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libpsl", "0.20.2-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"gdbm-libs", "1:1.14.1-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"kmod", "25-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libreport-filesystem", "2.9.5-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"ima-evm-utils", "1.1-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"nghttp2", "1.32.1-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"cyrus-sasl-lib", "2.1.27-0.2rc7.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libsolv", "0.6.35-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"cryptsetup", "2.0.4-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"filesystem", "3.8-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libcap", "2.25-9.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libpsl", "0.20.2-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"deltarpm", "3.6-25.fc28", "rpm", "source"}},
	{Feature: database.Feature{"fedora-gpg-keys", "28-5", "rpm", "binary"}},
	{Feature: database.Feature{"ima-evm-utils", "1.1-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"nss-tools", "3.38.0-1.0.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libtasn1", "4.13-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"elfutils-libelf", "0.173-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"device-mapper-libs", "1.02.146-5.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"gobject-introspection", "1.56.1-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"publicsuffix-list", "20180514-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libcap", "2.25-9.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"librepo", "1.8.1-7.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"rpm-sign-libs", "4.14.1-9.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"coreutils-single", "8.29-7.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libacl", "2.2.53-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"popt", "1.16-14.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libtasn1", "4.13-2.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"gawk", "4.2.1-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"diffutils", "3.6-4.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libgpg-error", "1.31-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libdb-utils", "5.3.28-30.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"python3-iniparse", "0.4-30.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"acl", "2.2.53-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"libssh", "0.8.2-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"python3-librepo", "1.8.1-7.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"gobject-introspection", "1.56.1-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"rpm", "4.14.1-9.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libgcrypt", "1.8.3-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"curl", "7.59.0-6.fc28", "rpm", "source"}},
	{Feature: database.Feature{"tzdata", "2018e-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"krb5", "1.16.1-13.fc28", "rpm", "source"}},
	{Feature: database.Feature{"librepo", "1.8.1-7.fc28", "rpm", "source"}},
	{Feature: database.Feature{"python3-gpg", "1.10.0-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"nettle", "3.4-2.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libgcrypt", "1.8.3-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"python3", "3.6.6-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"python3-libcomps", "0.1.8-11.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"rpm-libs", "4.14.1-9.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"nspr", "4.19.0-1.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"argon2", "20161029-5.fc28", "rpm", "source"}},
	{Feature: database.Feature{"tar", "1.30-3.fc28", "rpm", "source"}},
	{Feature: database.Feature{"qrencode-libs", "3.4.4-5.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"gmp", "6.1.2-7.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libverto", "0.3.0-5.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"python3", "3.6.6-1.fc28", "rpm", "source"}},
	{Feature: database.Feature{"libksba", "1.3.5-7.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"readline", "7.0-11.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"rpm-build-libs", "4.14.1-9.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"npth", "1.5-4.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"rootfiles", "8.1-22.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"rpm-plugin-systemd-inhibit", "4.14.1-9.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"systemd-libs", "238-9.git0e0aa59.fc28", "rpm", "binary"}},
	{Feature: database.Feature{"nss-util", "3.38.0-1.0.fc28", "rpm", "binary"}},
}

func TestRpmFeatureDetection(t *testing.T) {
	for _, test := range []featurefmt.TestCase{
		{
			"valid small case",
			map[string]string{"var/lib/rpm/Packages": "rpm/testdata/valid"},
			[]database.LayerFeature{
				{Feature: database.Feature{"centos-release", "7-1.1503.el7.centos.2.8", "rpm", "binary"}},
				{Feature: database.Feature{"filesystem", "3.2-18.el7", "rpm", "binary"}},
				{Feature: database.Feature{"centos-release", "7-1.1503.el7.centos.2.8", "rpm", "source"}},
				{Feature: database.Feature{"filesystem", "3.2-18.el7", "rpm", "source"}},
			},
		},
		{
			"valid big case",
			map[string]string{"var/lib/rpm/Packages": "rpm/testdata/valid_big"},
			expectedBigCaseInfo,
		},
	} {
		featurefmt.RunTest(t, test, lister{}, rpm.ParserName)
	}
}

func TestParseSourceRPM(t *testing.T) {
	for _, test := range [...]struct {
		sourceRPM string

		expectedName    string
		expectedVersion string
		expectedErr     string
	}{
		// valid cases
		{"publicsuffix-list-20180514-1.fc28.src.rpm", "publicsuffix-list", "20180514-1.fc28", ""},
		{"libreport-2.9.5-1.fc28.src.rpm", "libreport", "2.9.5-1.fc28", ""},
		{"lua-5.3.4-10.fc28.src.rpm", "lua", "5.3.4-10.fc28", ""},
		{"crypto-policies-20180425-5.git6ad4018.fc28.src.rpm", "crypto-policies", "20180425-5.git6ad4018.fc28", ""},

		// invalid cases
		{"crypto-policies-20180425-5.git6ad4018.fc28.src.dpkg", "", "", "unexpected package type, expect: 'rpm', got: 'dpkg'"},
		{"crypto-policies-20180425-5.git6ad4018.fc28.debian-8.rpm", "", "", "unexpected package architecture, expect: 'src' or 'nosrc', got: 'debian-8'"},
		{"fc28.src.rpm", "", "", "unexpected termination while parsing 'Release Token'"},
		{"...", "", "", "unexpected package type, expect: 'rpm', got: ''"},

		// impossible case
		// This illustrates the limitation of this parser, it will not find the
		// error cased by extra '-' in the intended version/expect token. Based
		// on the documentation, this case should never happen and indicates a
		// corrupted rpm database.
		// actual expected: name="lua", version="5.3.4", release="10.fc-28"
		{"lua-5.3.4-10.fc-28.src.rpm", "lua-5.3.4", "10.fc-28", ""},
	} {
		name, version, release, _, err := parseSourceRPM(test.sourceRPM)
		if test.expectedErr != "" {
			require.EqualError(t, err, test.expectedErr)
			continue
		}

		require.Nil(t, err)
		require.Equal(t, test.expectedName, name)
		require.Equal(t, test.expectedVersion, version+"-"+release)
	}
}
