/*
  hxd.c -- binary -> ascii (hexadecimal) converter
  Copyright (C) 1998,1999 Steffen Solyga <solyga@absinth.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 *	$Id: hxd.c,v 1.5 2004/04/08 21:01:41 solyga Exp $
 */


#include	"hxd.h"


int
display_help( char* pn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s): ", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( HELP_CHANNEL, "Binary -> ascii (hex) conversion\n" );
  fprintf( HELP_CHANNEL, "Flowers & bug reports to %s.\n", MY_EMAIL_ADDRESS );
  fprintf( HELP_CHANNEL, "Usage: %s [options] [in-file [out-file]]\n", pn );
  fprintf( HELP_CHANNEL, "switches:\n" );
  fprintf( HELP_CHANNEL, "  -h\t write this info to %s and exit sucessfully\n",HELP_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -r\t print relative offsets\n" );
  fprintf( HELP_CHANNEL, "  -v\t raise verbosity level on %s\n", VERBOSE_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -V\t write version and compilation info to %s and exit sucessfully\n",VERSION_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "controllers:\n" );
  fprintf( HELP_CHANNEL, "  -b BYTES\t number of bytes per line (default is %d)\n", DEFAULT_NBPL );
  fprintf( HELP_CHANNEL, "  -n BYTES\t number of bytes to dump\n" );
  fprintf( HELP_CHANNEL, "  -o OFF\t offset (default is 0)\n" );
  fprintf( HELP_CHANNEL, "  -s TYPE\t switch on/off display TYPE, with TYPE a mixture of\n" );
  fprintf( HELP_CHANNEL, "\t\t a - ascii\n" );
  fprintf( HELP_CHANNEL, "\t\t n - numbers (actual dump)\n" );
  fprintf( HELP_CHANNEL, "\t\t o - offsets\n" );
  return( 0 );
}


int
display_version( char* pn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( VERSION_CHANNEL, "compilation settings:\n" );
  fprintf( VERSION_CHANNEL, "  DEFAULT_NBPL       :\t%d\n", DEFAULT_NBPL );
  fprintf( VERSION_CHANNEL, "  MAX_NBPL           :\t%d\n", MAX_NBPL );
  fprintf( VERSION_CHANNEL, "  BEGIN_CHAR         :\t%c\n", BEGIN_CHAR );
  fprintf( VERSION_CHANNEL, "  END_CHAR           :\t%c\n", END_CHAR );
  fprintf( VERSION_CHANNEL, "  REMARKER           :\t%c\n", REMARKER );
  fprintf( VERSION_CHANNEL, "  HAVE_LONGLONG      :\t" );
#ifdef	HAVE_LONGLONG
  fprintf( VERSION_CHANNEL, "defined\n" );
#else
  fprintf( VERSION_CHANNEL, "undefined\n" );
#endif
  fprintf( VERSION_CHANNEL, "  HAVE_LITTLE_ENDIAN :\t" );
#ifdef	HAVE_LITTLE_ENDIAN
  fprintf( VERSION_CHANNEL, "defined\n" );
#else
  fprintf( VERSION_CHANNEL, "undefined\n" );
#endif
  return( 0 );
}


char
*ss_str_dsc( char* str1, char* str2) {
/* returns pointer to first char of str1 not contained in str2 */
/* 1997-02-09, 1999-07-10 */
  char *p1, *p2;
  for ( p1=str1; ; p1++ ) {
    if ( *p1 == '\0' ) return( p1 );
    for ( p2=str2; ; p2++ ) {
      if ( *p2 == '\0' ) return( p1 );
      if ( *p2 == *p1 ) break;
    }
  }
}


OFF_T
getoptarg( char opt, char* arg, OFF_T min, OFF_T max, char* pn ) {
  char* p;
  OFF_T val= STRTOL( arg, &p, 0 );
  if( p == arg ) { /* nothing read */
    fprintf( ERROR_CHANNEL,
             "%s: Argument of option %c (`%s') must be a number.\n",
             pn, opt, arg );
    exit( RETVAL_ERROR );
  }
  if( val<min || val>max ) {
#ifdef	HAVE_LONGLONG
    fprintf( ERROR_CHANNEL,
             "%s: Argument value %Ld of option %c must be in [%Ld,%Ld].\n",
             pn, val, opt, min, max );
#else
    fprintf( ERROR_CHANNEL,
             "%s: Argument value %ld of option %c must be in [%ld,%ld].\n",
             pn, val, opt, min, max );
#endif
    exit( RETVAL_ERROR );
  }
  if( *ss_str_dsc(p," \t") != '\0' ) {
    fprintf( ERROR_CHANNEL,
             "%s: Argument `%s' of option %c must be an integer.\n",
             pn, arg, opt );
    exit( RETVAL_ERROR );
  }
  return( val ) ;
}


ssize_t
my_read( int fd, void* buf, size_t nbtr ) {
/* returns number of bytes read or -1 on error */
/* like read(2) but tnbr<nbtr only if eof reached */
  ssize_t nbr;
  ssize_t tnbr= 0;
  size_t rem= nbtr;
  unsigned char *p= (unsigned char*)buf;
  do {
    if( (nbr=read(fd,p+tnbr,rem)) == -1 ) return( -1 );
    tnbr+= nbr;
    rem= nbtr - tnbr;
  }
  while( nbr>0 && rem>0 );
  return( tnbr );
}


ssize_t
my_write( int fd, void *buf, size_t count ) {
/* write exactly count bytes to fd (else error) */
  ssize_t nbw;
  ssize_t tnbw= 0;
  size_t rem= count;
  unsigned char *p= (unsigned char*)buf;
  do {
    if( (nbw= write( fd, p+tnbw, rem )) == -1 ) return( -1 );
    tnbw+= nbw;
    rem-= nbw;
  } while( nbw>0 && rem>0 );
  return( tnbw );
}


int
main( int argc, char** argv) {
/*
SOLYGA --------------------
SOLYGA main( int argc, char** argv) hxd
SOLYGA hexadecimal dump utility

SOLYGA started      : 1996/11/11 @beast
*/
  int retval= RETVAL_OK;
  long nbpl= DEFAULT_NBPL;		/* bytes per line */
  int c;				/* option char */
  int dsp_off= 1;			/* flag: display offsets */
  int dsp_num= 1;			/* flag: display numbers */
  int dsp_asc= 1;			/* flag: display characters */
  int verbose= 0;
  int rel= 0;				/* flag: relative offsets */
  int add_blank= 0;			/* flag: add. blank after 4 bytes */
  char* in_fn= NULL;
  int in_fd= STDIN_FILENO;
  char* out_fn= NULL;
  int out_fd= STDOUT_FILENO;
  int read_until_eof= 1;		/* flag */
  ssize_t nbr;				/* number of bytes read */
  size_t nbtr;				/* number of bytes to read */
  ssize_t nbw;				/* number of bytes written */
  size_t nbtw;				/* number of bytes to write */
  unsigned char in_buffer[IN_BUFFER_SIZE];
  unsigned char* ibuf= in_buffer;
  long ibuf_size;			/* int. mult. of nbpl */
  char* obuf= NULL;
  long obuf_size= 0l;
  OFF_T	offset= 0;			/* offset from option */
  OFF_T off;				/* actual offset */
  OFF_T tnbr;				/* total number of bytes read */
  OFF_T tnbw;				/* total number of bytes written */
  OFF_T tnbtr;				/* total number of bytes to read */
  OFF_T num;				/* offset for printing purposes */
  unsigned char* nump= (unsigned char*)(&num);
#if defined OS_TYPE_AIX
  extern int errno;
  extern char* sys_errlist[];
#endif

  /* process options */
  while( (c=getopt(argc,argv,"hrvVb:n:o:s:")) != EOF )
    switch( c )
    {
      case 'h': /* display help */
        display_help( *argv );
        retval= RETVAL_OK; goto DIE_NOW;
      case 'r': /* relative offsets */
        FLIP( rel );
        break;
      case 'v': /* raise verbosity level */
        verbose++;
        break;
      case 'V': /* display version */
        display_version( *argv );
        retval= RETVAL_OK; goto DIE_NOW;
      case 'b': /* number of bytes per line */
        nbpl= (long) getoptarg( c, optarg, (OFF_T)1, (OFF_T)MAX_NBPL, *argv );
        break;
      case 'n': /* number of bytes to dump */
        read_until_eof= 0;
        tnbtr= getoptarg( c, optarg, SIZE_MIN, SIZE_MAX, *argv );
        break;
      case 'o': /* offset */
        offset= getoptarg( c, optarg, SIZE_MIN, SIZE_MAX, *argv );
        break;
      case 's': /* switch display */
        {
          char* p= optarg;
          while( *(p=ss_str_dsc(p," \t")) != '\0' ) {
            switch( tolower(*p) ) {
              case 'a': FLIP( dsp_asc ); break;
              case 'n': FLIP( dsp_num ); break;
              case 'o': FLIP( dsp_off ); break;
              default :
                fprintf( ERROR_CHANNEL,
                         "%s: Unknown argument `%c' of option `s'.\n",
                         *argv, *p );
                retval= RETVAL_ERROR; goto DIE_NOW;
                break;
            }
            p++;
          }
        }
        break;
      case '?': /* refer to -h */
        fprintf( ERROR_CHANNEL, "%s: Try `%s -h' for more information.\n",
                 *argv, *argv );
        retval= RETVAL_ERROR; goto DIE_NOW;
      default: /* program error */
        fprintf( ERROR_CHANNEL,
                 "%s: Bug detected! Please send a mail to %s.\n",
                 *argv, MY_EMAIL_ADDRESS );
        retval= RETVAL_BUG; goto DIE_NOW;
    }


  /* set additional flags */
  if( nbpl%4 == 0 ) add_blank= 1;


  /* setup input and output buffers */
  /* ibuf: integer multiple of nbpl but <= IN_BUFFER_SIZE	*/
  /* obuf: 5*num_of_in_bytes + 22 fixed per line, i.e.		*/
  /*       2 numbers, 2 spaces, 1 ascii per ibyte		*/
  /*       16 offsets, 2 dividers, 4 spaces per line		*/
  ibuf_size= nbpl * ( IN_BUFFER_SIZE / nbpl );
  obuf_size= 5*ibuf_size + 22 * ( IN_BUFFER_SIZE / nbpl );
#ifdef	DEBUG
  fprintf( DEBUG_CHANNEL, "DEBUG: ibuf_size= %ld (0x%08lx)\n",
           ibuf_size, ibuf_size );
  fprintf( DEBUG_CHANNEL, "DEBUG: obuf_size= %ld (0x%08lx)\n",
           obuf_size, obuf_size );
#endif
  if( (obuf=malloc(obuf_size)) == NULL ) {
    fprintf( ERROR_CHANNEL,
             "%s: Cannot allocate %ld bytes for output buffer.\n",
             *argv, obuf_size );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }


  /* open input and output */
  if( argc>optind && strcmp(argv[optind],"-")!=0 ) {
    in_fn= argv[optind];
    if( (in_fd=open(in_fn,O_RDONLY)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot open `%s' read-only. %s.\n",
               *argv, in_fn, strerror(errno) );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }
  if( argc>optind+1 && strcmp(argv[optind+1],"-")!=0 ) {
    out_fn= argv[optind+1];
    if( (out_fd=open(out_fn,O_CREAT|O_WRONLY|O_TRUNC,PERMISSIONS)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot open `%s' write-only. %s.\n",
               *argv, out_fn, strerror(errno) );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }


  /* lseek input */
  if( in_fd != STDIN_FILENO ) {
    if( (off=LSEEK(in_fd,offset,SEEK_SET)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot lseek `%s'. %s.\n",
               *argv, in_fn, strerror(errno) );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }
  else {
    off= 0;
    nbtr= (size_t) MIN( IN_BUFFER_SIZE, offset-off );
    while( nbtr > 0 ) {
      if( (nbr=my_read(in_fd,ibuf,nbtr)) == -1 ) {
        fprintf( ERROR_CHANNEL, "%s: Cannot read from stdin. %s.\n",
                 *argv, strerror(errno) );
        retval= RETVAL_ERROR; goto DIE_NOW;
      }
      off+= nbr;
      if( nbr < nbtr ) break;
      nbtr= (size_t) MIN( IN_BUFFER_SIZE, offset-off );
    }
  }


  /* dump */
  tnbr= 0; tnbw= 0;
  nbtr= read_until_eof ? (size_t)ibuf_size : (size_t)MIN(ibuf_size,tnbtr);
  while( nbtr > 0 ) {
    long line, lines;		/* line counter */
    if( (nbr=my_read(in_fd,ibuf,nbtr)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot read from ", *argv );
      if( in_fd == STDIN_FILENO ) fprintf( ERROR_CHANNEL, "stdin" );
      else fprintf( ERROR_CHANNEL, "`%s'", in_fn );
#ifdef	HAVE_LONGLONG
      fprintf( ERROR_CHANNEL, "at offset 0x%08Lx", off );
#else
      fprintf( ERROR_CHANNEL, "at offset 0x%08lx", off );
#endif
      fprintf( ERROR_CHANNEL, ". %s.\n", strerror(errno) );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    lines= nbr / nbpl;
    if( nbr > lines*nbpl ) lines++;
    nbtw= 0;
    /* fill output buffer */
    for( line=0; line<lines; line++ ) { /* process one line */
      if( dsp_off ) { /* write offset */
        int i= sizeof(OFF_T);
#ifdef	HAVE_LITTLE_ENDIAN
        int imax;
        num= rel ? tnbr+line*nbpl : off+line*nbpl;
        for( imax=i-1; imax>=4; imax-- ) if( nump[imax] ) break;
        for( i=imax; i>=0; i-- ) {
          obuf[nbtw++]= HIG_UCHAR( nump[i] );
          obuf[nbtw++]= LOW_UCHAR( nump[i] );
        }
#else	/* tested on mips IRIX 6.2 */
        int imin;
        num= rel ? tnbr+line*nbpl : off+line*nbpl;
        for( imin=0; imin<i-4; imin++ ) if( nump[imin] ) break;
        for( i=imin; i<sizeof(OFF_T); i++ ) {
          obuf[nbtw++]= HIG_UCHAR( nump[i] );
          obuf[nbtw++]= LOW_UCHAR( nump[i] );
        }
#endif
      }
      if( dsp_num ) {
        long i;
        obuf[nbtw++]= BEGIN_CHAR;
        obuf[nbtw++]= ' ';
        for( i=0; i<nbpl; i++ ) {
          if( add_blank && i && i%4==0 ) obuf[nbtw++]= ' ';
          if( line*nbpl+i < nbr ) {
            obuf[nbtw++]= HIG_UCHAR( ibuf[line*nbpl+i] );
            obuf[nbtw++]= LOW_UCHAR( ibuf[line*nbpl+i] );
          }
          else {
            obuf[nbtw++]= ' ';
            obuf[nbtw++]= ' ';
          }
          if( i<nbpl-1) obuf[nbtw++]= ' ';
        }
      }
      if( dsp_asc ) {
        long i;
        unsigned char c;
        obuf[nbtw++]= ' ';
        obuf[nbtw++]= END_CHAR;
        obuf[nbtw++]= ' ';
        for( i=0; i<nbpl; i++ ) {
          if( line*nbpl+i < nbr ) {
            c= ibuf[line*nbpl+i];
#ifdef	NON_PRINTABLE_1_DEF
#if	NON_PRINTABLE_1_LOW == 0	/* only to avoid compiler warning */
            if( c<=NON_PRINTABLE_1_HIG )
#else
            if( c>=NON_PRINTABLE_1_LOW && c<=NON_PRINTABLE_1_HIG )
#endif
              c= NON_PRINTABLE_1_CHAR;
#endif
#ifdef	NON_PRINTABLE_2_DEF
            if( c>=NON_PRINTABLE_2_LOW && c<=NON_PRINTABLE_2_HIG )
              c= NON_PRINTABLE_2_CHAR;
#endif
#ifdef	NON_PRINTABLE_3_DEF
            if( c>=NON_PRINTABLE_3_LOW && c<=NON_PRINTABLE_3_HIG )
              c= NON_PRINTABLE_3_CHAR;
#endif
            obuf[nbtw++]= c;
          }
          else
            obuf[nbtw++]= ' ';
        }
      }
      if( !dsp_num && !dsp_asc) { /* possible but stupid :-) */
        obuf[nbtw++]= ' ';
        obuf[nbtw++]= END_CHAR;
        obuf[nbtw++]= ' ';
        obuf[nbtw++]= ':';
        obuf[nbtw++]= '-';
        obuf[nbtw++]= ')';
      }
      obuf[nbtw++]= '\n';
    }	/* end: fill output buffer */
    if( nbtw > obuf_size ) {
      fprintf( ERROR_CHANNEL,
               "%s: Bug detected! Please send a mail to %s.\n",
               *argv, MY_EMAIL_ADDRESS );
      retval= RETVAL_BUG; goto DIE_NOW;
    }
    /* write output */
    if( (nbw=my_write(out_fd,obuf,nbtw)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot write to ", *argv );
      if( out_fd == STDOUT_FILENO ) fprintf( ERROR_CHANNEL, "stdout" );
      else fprintf( ERROR_CHANNEL, "`%s'", in_fn );
      fprintf( ERROR_CHANNEL, ". %s.\n", strerror(errno) );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    off+= nbr;
    tnbr+= nbr;
    tnbw+= nbw;
    nbtr= read_until_eof?(size_t)ibuf_size:(size_t)MIN(ibuf_size,tnbtr-tnbr);
    if( nbr < nbtr ) nbtr= 0;
  }	/* end: process one input buffer */


  /* close input and output */
  if( out_fd!=STDOUT_FILENO && close(out_fd)==-1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot close `%s'. %s.\n",
             *argv, out_fn, strerror(errno) );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  if( in_fd!=STDIN_FILENO && close(in_fd)==-1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot close `%s'. %s.\n",
             *argv, in_fn, strerror(errno) );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }


  if( verbose )
#ifdef	HAVE_LONGLONG
    fprintf( VERBOSE_CHANNEL, "%s: %Ld/%Ld bytes read/written.\n",
             *argv, tnbr, tnbw );
#else
    fprintf( VERBOSE_CHANNEL, "%s: %ld/%ld bytes read/written.\n",
             *argv, tnbr, tnbw );
#endif

DIE_NOW:
  close( out_fd );
  close( in_fd );
  exit( retval );
}
