/* filecopy.c
  filecopy.c,v 1.1 2000/03/02 11:46:36 gwiley Exp
  Glen Wiley, <gwiley@ieee.org>

	Copyright (c)2000,2001 Glen Wiley
   
   Permission is hereby granted, free of charge, to any person obtaining a copy
   of this software and associated documentation files (the "Software"), to deal
   in the Software without restriction, including without limitation the rights
   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
   copies of the Software, and to permit persons to whom the Software is
   furnished to do so, subject to the following conditions:
   
   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.
   
   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
   THE SOFTWARE.

  this function will reliably copy a file, detecting
  whether source/destination are the same file
  returns 0 on success
*/

#ifdef HAVE_CONFIG_H
# include "config.h"
#else
# define HAVE_ERRNO_H     1
# define HAVE_SYS_TYPES_H 1
# define HAVE_SYS_STAT_H  1
# define HAVE_UNISTD_H    1
#endif
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

/* size of the buffer used for copying the file */
static const int COPYBUFFSZ = 4096;

/*-------------------------------------- filecopy
 copy file fn_src to a separate file, fn_dst

 if setown != 0 then we will chown/chgrp to uid/gid otherwise the owner and
 group of the source file will be used

 if setmode != 0 then we will chmod to match mode otherwise we use the mode
 on the source file

 if append != 0 then the destination file will be appended, otherwise it is
 replaced, if append is != 0 then setown and setmode are ignored!

 return 0 on success, errno on failure
*/
int
filecopy(const char *fn_src, const char *fn_dst, int setown, uid_t uid, gid_t gid, int setmode, mode_t mode, int append)
{
	int   bytesr;
	int   retval  = 0;
	int   done    = 0;
	FILE  *fh_src = NULL;
	FILE  *fh_dst = NULL;
	char  *buff   = NULL;
	char  *p      = NULL;
	struct stat stat_s;
	struct stat stat_d;

	buff = (char *) malloc(COPYBUFFSZ);
	if(buff == NULL)
		return errno;

	if(stat(fn_src, &stat_s) != 0)
		retval = errno;

	/* if we can stat the dest file then make sure it is not the same as src */

	if(stat(fn_dst, &stat_d) == 0)
	{
		if(stat_s.st_dev == stat_d.st_dev && stat_s.st_ino == stat_d.st_ino)
			retval = EEXIST;
	}

	if(!retval)
	{
		fh_src = fopen(fn_src, "r");
		if(fh_src == NULL)
			retval = errno;
	}

	if(!retval)
	{
		if(append)
		 	fh_dst = fopen(fn_dst, "a+");
		else
		 	fh_dst = fopen(fn_dst, "w+");

	 	if(fh_dst == NULL || stat(fn_dst, &stat_d) != 0)
			retval = errno;
		else
		{
			if(append == 0)
			{
				/* we make a best effort to change the user/group, if called
					by non-root user this will fail but they should expect that */

				if(setown)
					fchown(fileno(fh_dst), uid, gid);
				else
					fchown(fileno(fh_dst), stat_s.st_uid, stat_s.st_gid);

				/* we make a best effort to change the mode, this should
					probably not end up failing */

				if(setmode)
					fchmod(fileno(fh_dst), mode);
				else
					fchmod(fileno(fh_dst), stat_s.st_mode);
				errno = 0;
			}

			while(!retval)
			{
				bytesr = fread(buff, 1, COPYBUFFSZ, fh_src);
				if(bytesr < 1)
				{
					if(feof(fh_src))
						break;
					else
						retval = ferror(fh_src);
				}
				else
				{
					if(fwrite(buff, 1, bytesr, fh_dst) != bytesr)
					{
						retval = ferror(fh_dst);
						break;
					}
				}
			}
		} /* if fh_dst */

		if(fh_dst)
			fclose(fh_dst);

	} /* if !retval */

	if(fh_src)
		fclose(fh_src);

	if(buff)
		free(buff);

	return retval;
} /* filecopy */

/* filecopy.c */
