/*-------------------------------------------------------------------------- 
  Filename        : sgmode.c
  Abstract        : SCSI Generic Disk Mode Page tool
  Operation System: Linux 2.2 or greater
  Author  	  : Andy Cress   <arcress@users.sourceforge.net>
  Copyright (c) Intel Corporation 2001-2008
  Copyright (c) 2009, Kontron America, Inc.
  ----------- Change History -----------------------------------------------
  05/01/00 v0.10 ARCress  created 
  06/11/00 v0.70 ARCress  long pages (page len > 0x12) fixed in parsing code.
  08/14/01 v0.80 ARCress  added -r option
  09/05/01 v0.84 ARCress  cleanup
  09/28/01 v1.00 ARCress  turn on sg_debug flags
  10/30/01 v1.0  ARCress  cleanup includes & indent -kr 
  11/09/01 v1.0  ARCress  enlarged devstat2 from 80 to 120 for overrun 
  11/14/01 v1.0  ARCress  added default dir for mdf files
  03/07/02 v1.1  ARCress  special case for mode param byte 3 w some disks
  04/11/02 v1.2  ARCress  path changes for log & mdf files, 
                          set do_numeric=1 default
  05/08/02 v1.3  ARCress  tune sg_debug level down to 5 by default
  08/15/02 v1.4  ARCress  moved common routines to sgcommon
			  added more usage messages 
  09/03/02 v1.5  ARCress  streamline display for errors in get_scsi_info 
  04/10/03 v1.6  ARCress  relinked with DEVFS changes from sgcommon.c
  09/03/03 v1.7  ARCress  handle changes in blocksize (fchgblksz)
  10/02/03 v1.8  ARCress  make sure getchar() values are int for 64-bit compile
  01/31/05 v1.9  ARCress  added decodemodebuf routine
  05/11/05 v1.10 ARCress  included patch for larger devlist from Nate Dailey 
  08/06/07 v1.14 ARCress  added -w to set WriteCacheEnable for any disk
  ----------- Description --------------------------------------------------
  sgmode  
Sequence of events for this utility:
  * List each device on the system with firmware versions.
  * User selects a device for mode page (automatic if using -m or -a)
  * Read the mode page file for the selected disk 
  * Verify that the disk is present and ready
  * Write the mode pages via mode select, (if using -s)
  * If the '-m' option was used, repeat writing the pages for each disk of 
    this model.
sgmode    [-a -e -f modefile -m diskmodel -r -x] 
Options:
  -a  	Automatically modepage all drives.
  -e  	Do not write to any files. Usually a log file (sgmode.log) is created
    	and written to.
  -f 	Specify this filename for the modepages.  Normally, this option is 
	not used and the filename is formed using the first 8 characters of the 
	model, with the ".mdf" extension.  For example:  "st39140w.mdf".   
  -m  	Automatically modepage all drives that match the given model string.
  -r    Read-only.  Do not try to write any mode pages.  By default, if 
        -a or -m is specified, the utility will attempt to open an .mdf file
        for each drive, and write modepages if a matching file exists.
  -x    Outputs extra debug messages
----------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------
Copyright (c) 2002-2005, Intel Corporation
Copyright (c) 2009, Kontron America, Inc.
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  -------------------------------------------------------------------------*/
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <time.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>
#include "sgsub.h"
#include "sgcommon.h"

/* External Functions prototypes */
// extern int errno;
extern void setbuf(FILE * stream, char *buf);
extern char fdebugsub;		/* used in sgsub.c */
extern FILE *dbgout;		/* used in sgsub.c */
extern char HeaderStr[1];       /* from sgcommon.c */
#define  EBUFF_LEN   80
#define  MAX_ERRORS   4
#define  bool   unsigned char
#define  SZ_MODEL    20
#define  SZ_MDFILE  132 

/* Global data definitions */
char *progver  = "1.66";		/* program version */
char *progname = "sgmode";	/* program name */
char logfile[] = "/var/log/sgmode.log";	/* log filename */
char mdsufx[] = ".mdf";		/* suffix for mode format filenames */
char model[SZ_MODEL] = "";		/* default model string */
char fsetfile = 0;
char mdfile[SZ_MDFILE] = "default";	/* default mdf filename (8char.mdf) */
char defdir[30] = "/usr/share/scsirastools/"; /* default dir for mdf files*/
char device[80] = "/dev/sda";	/* UNIX device name (w path) */
int sdelay = 10;		/* num sec to delay before TUR */
FILE *fdlog = NULL;		/* log file descriptor */
FILE *fdmsg;			/* file descriptor for messages  */
FILE *fdtmp = NULL;		/* tmp mdf file descriptor */
char fautomodel = 0;		/* =1 to automatically do all of a model */
char fauto = 0;			/* =1 to automatically do all disks */
char freadonly = 0;		/* =1 dont try to write any mode pages */
char fsmartoff = 0;		/* =1 to turn off SMART  */
char filesok = 1;		/* =1 if ok to write to files */
char fdebug = 0;		/* =1 for debug messages */
char finteractive = 0;		/* =1 for interactive mode */
char fdoreset = 0;		/* =1 if need to do reset for this drive */
char flogopen = 0;		/* =1 if log file is open */
char foverwrite = 0;		/* =1 to overwrite/update capacity */
int do_numeric = 1;		/* NUMERIC_SCAN_DEF; */
char fsavemdf = 0;		/* =1 to save mdf file from modesense */
char fchgwcache = 0;		/* =1 to specify disk write cache value */
uchar bwcache = 0;		/* disk write cache value 0=off, 1=on */
char fonedev = 0;		/* =1 for autorun w a certain disk */
int  ionedev = 0;
uchar sense_buffer[80];		/* buffer for sense data */
char output[132];		/* message output buffer */
char output2[80];		/* extra message output buffer */
uchar devstat[80];		/* used for inquiry & device status */
uchar devstat2[120];		/* used for Seagate inquiry2, etc.  */
uchar disk_buffer[0x200];	/* 512 bytes, used for misc scsi data */
uchar modebuf[300];		/* ~240 bytes, used for mode sense/select */
int sgdbglvl = 5;		/* set sg_debug_level to 5 or 10 while here */
static ulong onemeg = 1024 * 1024;
int idev = 0;
int ndev = 0;
DEVLIST_TYPE devlist[MAX_DEVLIST_DEVS];

/* Local subroutines */
int do_modesense(int idx);
int do_modeselect(int idx);
int beforemd(int idx, int *pnumrdy);
int aftermd(void);

int main(int argc, char **argv)
{
    int i, c, idx;
    char response = 0;
    int runl;
    short sts, rc;
    ulong ltime1;
    char fname[64];
    char ebuff[EBUFF_LEN];
    int sg_fd;
    int flags;
    int res, k;
    int eacces_err = 0;
    int num_errors = 0;
    char fmodesense = 1;
    char fmodeselect = 0;
    DEVFS_LIST devfs_components = {0,0,0,0,0};
    fdmsg = stdout;		/* set default fdmsg */
    // progname = argv[0];
    while ((c = getopt(argc, argv, "aei:norsw:xf:m:I?")) != EOF)
	switch (c) {
	case 'e':
	    filesok = 0;
	    break;
	case 'w':
	    fchgwcache = 1;
	    bwcache = atoi(optarg);
	    fsavemdf = 1;
	    break;
	case 'x':
	    fdebug = 1;
	    break;
	case 'i':
	    fonedev = 1;
            ionedev = atoi(optarg);
	    break;
	case 'n':
	    do_numeric = 0;
	    break;
	case 'f':
	    i = strlen(optarg);
	    if (i >= SZ_MDFILE) 
		printf("filename too long (%d >= %d)\n",i,SZ_MDFILE);
	    else {
		strcpy(mdfile, optarg);
		fsetfile = 1;
	    }
	    break;
	case 'a':
	    fauto = 1;		/* automatically do all disks */
	    break;
	case 's':
	    fauto = 1;		/* automatically do all disks */
	    fsmartoff = 1;      /* turn SMART off */
	    fsavemdf = 1;
	    break;
	case 'I':
	    finteractive = 1;	/* Interactive mode */
	    break;
	case 'o':
	    foverwrite = 1;	/* force overwrite pages, capacity too */
	    break;
	case 'r':
	    freadonly = 1;	/* dont try to write any mode pages */
	    break;
	case 'm':
	    i = strlen(optarg);
	    if (i >= SZ_MODEL) {
		i = SZ_MODEL - 1;
		optarg[i] = 0;  /*truncate input for model*/
	    }
	    fautomodel = 1;	/* automatically do all of this model */
	    strcpy(model, optarg);
	    break;
	case '?':
	default:
	    printf("Usage: %s [-aeinorswxI -m diskmodel -f modefile]\n",
		   progname);
	    printf(" -a  Automatically modepage all drives\n");
	    printf(" -e  Do not write to any files, not even the log file\n");
	    printf(" -i  Autorun for the disk at index N\n");
	    printf(" -n  Turn off Numeric, use alpha device names instead\n");
	    printf(" -o  Overwrite mode pages and capacity if needed\n");
	    printf(" -r  Read-only.  Do not try to write any mode pages\n");
	    printf(" -x  Outputs extra debug messages\n");
	    printf(" -I  Interactive mode, prompt for input.\n");
	    printf(" -f  Specify this filename for the modepages to override default name\n");
	    printf(" -m  Automatically modepage all drives that match the given model string\n");
	    printf(" -s  Automatically turn off SMART on all mode pages\n");
	    printf(" -wN Automatically set WriteCache to N (0=off, 1=on)\n");
	    exit(1);
	}
    /* only run this as superuser */
    i = geteuid();
    if (i > 1) {
	printf("Not superuser (%d)\n", i);
	exit(1);
    }
    /* check for run-level 1 */
    {
	runl = 1;
	/* how to check this? */
	if (fdebug) {
	    /*DEBUG*/ printf("fdebug = %d\n", fdebug);
	    printf("sdelay = %d sec\n", sdelay);
	    printf("run-level = %d\n", runl);
	}
    }
    if (!freadonly && (runl != 1)) {	/* show warning if not run level 1 */
	printf
	    ("Warning: Make sure disks are inactive while writing modepages\n");
	// dont exit, just give a warning.
    }
    if (fdebug) sgdbglvl = 10;  /* max */
    else sgdbglvl = 5;
    /* initialize devlist */
    for (i = 0; i < MAX_DEVLIST_DEVS; i++ ) {
	devlist[i].sgfd = 0;
	devlist[i].fdoit = 0;
	devlist[i].dcap = 0;
    }
    /* open log file */
    if (filesok)
	fdlog = fopen(logfile, "a+");
    if (fdlog == NULL) {
	flogopen = 0;
	filesok = 0;
	fdlog = stderr;
    } else {			/* logfile open successful */
	flogopen = 1;
	time((time_t *) & ltime1);
	fprintf(fdlog, "\n--- %s v%s log started at %s", progname, progver,
		ctime((long *) &ltime1));   // ctime outputs a '\n' at end
	fprintf(fdlog, "model:   %s\n", model);
	fprintf(fdlog, "mode file: %s\n", mdfile);
    }
    fdebugsub = fdebug;
    dbgout = fdlog;
    /* loop to display disk choices and do modepages */
    while (1) {
	/* display header */
	printf("\n");
	printf("                %s utility v%s for SCSI disks\n", 
	       progname, progver);
	printf
	    ("              ******************************************\n");
	if (flogopen)
	    printf("Log file %s is open\n", logfile);
	printf(HeaderStr);
	/* get SCSI Device Info */
	idev = 0;
	flags = O_RDWR;		/* could use OPEN_FLAG if read-only. */
	num_errors = 0;
        for (k = 0; (k < MAX_SG_DEVS) && (idev < MAX_DEVLIST_DEVS) &&
                    (num_errors < MAX_ERRORS); ++k) {
	    make_dev_name(fname, k, do_numeric, &devfs_components);
	    if (devfs_components.all_leaves_searched != 0) {
		devfs_components.all_leaves_searched=0;
		break;
	    }
	    sg_fd = open(fname, flags | O_NONBLOCK);
	    if (sg_fd < 0) {
		if (EBUSY == errno) {
		    printf("%s: device busy (O_EXCL lock), skipping\n",
			   fname);
		    continue;
		}
		    else if ((ENODEV == errno) || (ENOENT == errno) ||
			     (ENXIO == errno)) {
		    ++num_errors;
		    continue;
		} else {
		    if (EACCES == errno)
			eacces_err = 1;
		    sprintf(ebuff, "Error opening %s ", fname);
		    perror(ebuff);
		    ++num_errors;
		    continue;
		}
	    }
	    sts = get_scsi_info(sg_fd, idev, fname,0);
	    if (sts) {
		  showlog("device %s failed with sts = %d, skip\n",
			fname, sts);
	    } 
	    {
		if (sg_fd > 0) {
		    res = close(sg_fd);
		    if (res < 0)
			fprintf(fdmsg, "Error closing %s, errno = %d\n",
				fname, errno);
		    else
			devlist[idev].sgfd = 0;	/* set in get_scsi_info() */
		}
		idev++;
	    }
	}			/*end loop */
	ndev = idev;
	if (ndev == 0 && errno == ENODEV) {	/* CONFIG_CHR_DEV_SG != y ? */
	    sprintf(output,
		    "Cant open any SCSI devices.\nMake sure CONFIG_CHR_DEV_SG is set in /usr/src/linux/.config\n");
	    showit(output);
	    quit(1);
	}
        else if (ndev == 0) {
            sprintf(output,"Cant open any sg SCSI devices, errno = %d\n",errno);
	    if (errno == ENOENT) 
	       strcat(output,"Try option -n to change /dev/sg* device names\n");
            showit(output);
        }
	if (finteractive) {
	    printf("\nEnter Selection ('w' to write pages, 'r' to read pages, 'q' to quit) : ");
	    response = get_func();
	} else if (fautomodel || fauto || fonedev) {
	    if (freadonly)
		response = 'r';
	    else
		response = 'w';
	} else {
	    response = 'r';
	}
	if (response == 'q' || response == 'Q') {
	    printf("\n");
	    quit(0);
	} else if (response == 'w' || response == 'W') {  /* do mode selects */
	    fmodesense = 1;	/* do both */
	    fmodeselect = 1;
	    printf("Write Mode Select ...\n");
	} else if (response == 'r' || response == 'R') {
	    fmodesense = 1;
	    fmodeselect = 0;
	    printf("Read Mode Sense ...\n");
	} else {		/* response wrong */
	    printf("Invalid input [%c], enter 'w', 'r', or 'q'.\n",
		   response);
	}
	if (fautomodel)
	    i = get_mdev(model);	// get first matching dev
	else if (finteractive)
	    i = get_idev();	/* get the device index that the user picks */
	else if (fauto)
	    i = 0;		// start with first dev
	else if (fonedev)
	    i = ionedev;
	else
	    i = 0;		// start with first dev
	if (i == 0xff) {
	    if (finteractive) {
	        printf("Index out of range, try again.\n");
	    } else { // if (fautomodel || fauto || fonedev) * then done *
	        printf("Device not found.\n");
		quit(1);	/* error exit */
	    }
	} else {			/*index ok */
	    int numrdy;
	    idx = i;		/* first disk to modepage */
	    rc = beforemd(idx, &numrdy);
	    if (numrdy == 0)
		idx = ndev;
	    /* Get modepages for each matching disk */
	    for (i = idx; i < ndev; i++) {
		if (devlist[i].fdoit) {
		    if (fmodesense) {
			rc = do_modesense(i);
			if (fdebug && rc != 0)
			    quit(rc);
		    }
		    if (fmodeselect) {
			rc = do_modeselect(i);
			if (fdebug && rc != 0)
			    quit(rc);
		    }
		}
		if (finteractive || fonedev)
		    i = ndev;	/* only do one if interactive */
	    }			/*endfor each disk */
	    rc = aftermd();	/* closes each open sgfd */
	    if (finteractive) {	/* interactive, go to menu again */
		do_pause();
	    } else {     /* if (fautomodel || fauto || fonedev) * then done */
		quit(0);	/* normal exit, successful */
	    }
	}			/*index ok */
    }				/*end while(1) */
}				/*end main() */

ulong cap_report(uchar * buf)
{
    ulong addr_1, addr_2, addr_3, addr_4;
    ulong num_blk, blk_len, cap;
    addr_4 = ((((ulong) buf[4]) << 24) & 0xff000000);
    addr_3 = ((((ulong) buf[5]) << 16) & 0x00ff0000);
    addr_2 = ((((ulong) buf[6]) << 8) & 0x0000ff00);
    addr_1 = (((ulong) buf[7]) & 0x000000ff);
    num_blk = (addr_1 | addr_2 | addr_3 | addr_4);
    addr_3 = ((((ulong) buf[9]) << 16) & 0x00ff0000);
    addr_2 = ((((ulong) buf[10]) << 8) & 0x0000ff00);
    addr_1 = (((ulong) buf[11]) & 0x000000ff);
    blk_len = (addr_1 | addr_2 | addr_3);
    cap = (blk_len * num_blk);	/* get overflow if >8G */
    return (num_blk);
}

void get_blksz(uchar * buf, ulong *nblk, ulong *blksz)
{
    ulong addr_1, addr_2, addr_3, addr_4;
    ulong num_blk, blk_len;
    addr_4 = ((((ulong) buf[4]) << 24) & 0xff000000);
    addr_3 = ((((ulong) buf[5]) << 16) & 0x00ff0000);
    addr_2 = ((((ulong) buf[6]) << 8) & 0x0000ff00);
    addr_1 = (((ulong) buf[7]) & 0x000000ff);
    num_blk = (addr_1 | addr_2 | addr_3 | addr_4);
    *nblk = num_blk;
    addr_3 = ((((ulong) buf[9]) << 16) & 0x00ff0000);
    addr_2 = ((((ulong) buf[10]) << 8) & 0x0000ff00);
    addr_1 = (((ulong) buf[11]) & 0x000000ff);
    blk_len = (addr_1 | addr_2 | addr_3);
    *blksz = blk_len;
    return;
}

static void build_mdf_name(int idev, char *mdname)
{
    struct SCSI_INQUIRY *scsi_inq;
    char modelx[16];
    int i;

    scsi_inq = (struct SCSI_INQUIRY *) devlist[idev].inq;
	if (fautomodel)
	    strncpy(modelx, model, 8);
	else
	    strncpy(modelx, (char *) scsi_inq->productid, 8);
	modelx[8] = 0;		/*stringify */
	for (i = 0; i < 8; i++) {	/*limit filename to 8 chars */
	    if (modelx[i] == ' ')
		break;
	    else
		mdname[i] = tolower(modelx[i]);
	}
	mdname[i] = 0;
	strcat(mdname, mdsufx);
}

int do_modeselect(int idx)
{
    int sts = 0;
    int i, j, k, m, n, l, l1, l2, mlen;
    int good_cmp;
    int fchgblksz = 0;
    int devfd;
    ulong cap_1, cap_2;
    uchar pagecode; 
    int c;
    FILE  *fp1;
    char  *result;
    uchar *out_buffer;
    uchar sel_buffer[0x110];
    uchar tch[60];
    char line[160];
    char scratch[80];
    char *pmdfile;
    ulong nb1, sb1, nb2, sb2;

    devfd = devlist[idx].sgfd;
    /* if fsetfile, mdfile has been set by user param */
    /* else build the mdf filename */
    if (!fsetfile) {
        build_mdf_name(idx,mdfile);
    }
    pmdfile = mdfile;
    if ((fp1 = fopen(pmdfile, "r")) == NULL) {  /* try current directory */
	strcpy(line,defdir);
	strcat(line,mdfile);
	pmdfile = line;
        fp1 = fopen(pmdfile, "r");  /* try default directory */
    }
    if (fp1 == NULL) {  /* mdfile not opened */
       sprintf(output, "Cannot open %s, errno = %d\n", pmdfile, errno);
       showit(output);
       return (errno);
    }
    else {			/* mdf file opened */
	if (fdebug)
	    printf("opened %s file, ready to read, fp = %p.\n", pmdfile, fp1);
	out_buffer = modebuf;	/* mode sense/select buffer */
	/* set up MODE SELECT buffer header */
	/* First 4 bytes are the mode parameters */
	sel_buffer[0] = 0x00;	/* Reserved during MODE SELECT */
	for (j = 1; j < 4; j++) 
	    sel_buffer[j] = out_buffer[j];
	/* SCSI Block spec 7.1.3 says that DPOFUA bit (0x10) in byte 3 
           from mode sense is reserved for mode select so mask it out. */
	 sel_buffer[2] = sel_buffer[2] & 0xEF;
	/* Set up MODE SELECT Block Descriptor (BD) */
	result = fgets(line, sizeof(line), fp1);
	if (strchr(result, ':') != NULL) {
	    /* if ':' is found, there is BD in Mode data file */
	    /* Assumption: all BD data are in one line */
	    for (j = 4; j < (4 + out_buffer[3]); j++) {
		if (sscanf
		    (result + 3 * (j - 4), "%02x",
		     (uint *) (tch + j - 4)) != 1) {
		     showlog( "Error in reading Block Descriptor data. ");
		     showlog( "Aborting MODE SELECT for this page.\n");
		    break;	/* from this for() loop */
		}
		sel_buffer[j] = tch[j - 4];
	    }
	    if (j != (4 + out_buffer[3])) {	/* error in reading BD data */
		fclose(fp1);
		return (0x801);	/* go to next Device */
	    }
	} else {
	    rewind(fp1);	/* No BD in Mode data file */
	    /* use the BD the drive returned */
	    for (j = 4; j < (4 + out_buffer[3]); j++)
		sel_buffer[j] = out_buffer[j];
	}
	good_cmp = memcmp(sel_buffer + 4, out_buffer + 4, 8);
	if (good_cmp != 0) {	/* capacity is different */
	    cap_1 = cap_report(out_buffer);
	    cap_2 = cap_report(sel_buffer);
	    sprintf(scratch,
		    "Capacity is %luMB (0x%lx) --> Update Capacity to %luMB (0x%lx)\n",
		    cap_1 / (onemeg/512), cap_1, cap_2 / (onemeg/512), cap_2);
	    printf("\n%s", scratch);
	    showlog("%s\n", scratch);
	    if (!foverwrite) {
		/* use existing capacity from mode sense */
		memcpy(sel_buffer + 4, out_buffer + 4, 8);
		sprintf(scratch, "Using existing capacity instead\n");
		printf(scratch);
		showlog( scratch);
	    } else {  /* foverwrite==1, changing something */
		/* usually only change num blks, if change blk size, flag it. */
	        get_blksz(out_buffer,&nb1,&sb1);
	        get_blksz(sel_buffer,&nb2,&sb2);
		if ((sb1 != sb2) && (nb2 != 0)) fchgblksz = 1;
	    }
	}
	if (fauto || fautomodel || foverwrite || fonedev) {
	    /* automatic, or specified to update */
	    showlog( "Update=Y\n");
	} else if (finteractive) {		/* interactive */
	    printf("\nEnter Y to Update or another letter to skip\n");
	    c = getchar();
	    if (tolower((uchar)c) != 'y') {
		showlog( "Update=N, skipped\n");
		return (0x803);	/* go to next Device */
	    }
	    showlog( "Update=Y\n");
	    c = getchar();	/* get return char too */
	} else {
	    showlog( "Update=N\n");
	    return (0x803);	/* go to next Device */
	}
	printf("Writing mode pages ");
	/* Read in the pages from the definition file */
	while ((result = fgets(line, sizeof(line), fp1)) != NULL) {
	    int maxl;
	    maxl = 0x40; /* =64. bytes max per line */
	    /* set i to start of mode page */
	    i = 4 + sel_buffer[3];   /* mode param hdr =4 + BD size, usu =8 */
	    for (j = i; j < 0x100; j++)
		sel_buffer[j] = ' ';
	    k = sscanf(result,"%02x %02x", (uint *)tch, (uint *)(tch + 1));
	    if (k != 2) {
		fprintf(fdmsg, "Error reading next Page Code & Length. ");
		fprintf(fdmsg, "Aborting MODE SELECT.\n");
		break;	/* from this while() loop, hence go to next target */
	    }
	    sel_buffer[i] = tch[0] & 0x3f;  /* page code with PS bit reset */
	    i += 1;		/* set i to byte #1 of page */
	    sel_buffer[i] = tch[1];	/* page length */
	    i += 1;		/* set i to byte #2 of page */
	    j = 2;
	    m = 5 + sel_buffer[3];	/* set m to get byte #1 of page */
	    for (k = 0; (uchar)k < sel_buffer[m]; k++, i++, j++) {
		if ((j % maxl) == 0)  /*MODE SENSE page data exceeds one line*/
		    result = fgets(line, sizeof(line), fp1);
		l1 = j % maxl;
		if (sscanf(result + 3 * l1, "%02x", (uint *) (tch + l1)) != 1) 
		{
		    showlog( "Error in reading data on page 0x%02x. ",
			    sel_buffer[m - 1]);
		    showlog( "Aborting MODE SELECT for this page.\n");
		    break;	/* from this for() loop */
		}
		sel_buffer[i] = tch[(j % maxl)];
	    }
	    if ((uchar) k < sel_buffer[m]) {
		/* Error in reading all bytes of the page */
		if (sel_buffer[m] > maxl) {	/* page data exceeds one line */
		    l1 = (j / maxl) + 1;
		    /* line # for this page */
		    l2 = ((sel_buffer[m] + 2 - 1) / maxl) + 1;
		    /* # of lines for this page */
		    for (l = l1; l < l2; l++)
			result = fgets(line, sizeof(line), fp1); /* get unread lines */
		}
		continue;	/* the while() loop, hence go to next page */
	    }
	    /* Now perform MODE SELECT */
	    n = 4 + sel_buffer[3];
	    mlen = n + sel_buffer[n + 1] + 2;
	    pagecode = sel_buffer[n] & 0x3f;	/* with PS bit reset */
	    if (fdebug) {	/* debug, output the buffer */
		showlog( "\nsel_buffer pg %02x, n=%d, mlen=%d (0x%x)",
			pagecode,n, mlen,mlen);
		if (filesok && flogopen) 
		    dump_log(sel_buffer,mlen,"sel_buffer",0);
	    }
	    if (fchgblksz && (pagecode == 3)) { /* fixup format dev page */
	      if (n == 4) { /* no BD, insert 8-byte Block Descriptor header */
	         sel_buffer[3] = 8;  
	         n = 4 + sel_buffer[3];
		 /* shift existing data out 8 bytes (in reverse order) */
		 for (i = (mlen-3); i >= 0; i--) 
		     sel_buffer[n+i] = sel_buffer[4+i]; 
	      }
	      /* nb2 & sb2 are the new values, as set above. */
		/* set the 8 block descriptor bytes */
		sel_buffer[4] = (uchar)(nb2 >>24);   /*numblk MSB*/
		sel_buffer[5] = (uchar)(nb2 >>16); 
		sel_buffer[6] = (uchar)(nb2 >>8); 
		sel_buffer[7] = (uchar) nb2 ;        /*numblk LSB*/
		sel_buffer[8]  = 0;   		   /*Density code*/
		sel_buffer[9]  = (uchar)(sb2 >>16);   /*blksz MSB*/
		sel_buffer[10] = (uchar)(sb2 >>8); 
		sel_buffer[11] = (uchar) sb2 ;        /*blksz LSB*/
		/* Adjust default format page data (blksz) */
		/* n+12 = 24(0x18): Data bytes per physical sector */
		sel_buffer[n+12] = (uchar)(sb2 >>8);    
		sel_buffer[n+13] = (uchar) sb2 ;

	      if (fdebug && filesok && flogopen) 
		    dump_log(sel_buffer,mlen,"adjusted sel_buffer",0);
		/* This could be handled by setting this in the mdf file too,
		 * but the block sizes must match here.  */
	    }
	    if ((pagecode == 8) && fchgwcache) {  /* writecache specified */
		if (bwcache == 0)  
		   sel_buffer[n+2] &= 0xFB;
		else  /*enable write cache*/
		   sel_buffer[n+2] |= 0x04;
            }
            if ((pagecode == 0x1c) && fsmartoff) { /* turn SMART off */
		   sel_buffer[n+2] = 0x88;
		   sel_buffer[n+3] = 0x00;
	       if (fdebug && filesok && flogopen) 
		    dump_log(sel_buffer,mlen,"adjusted sel_buffer",0);
            }
	    showlog( "Writing mode page 0x%02x.\n", pagecode);
	    printf(" 0x%02x", pagecode);
#ifdef DBG
	    if (fdebug || (mlen >= 36)) {	/* 0x16 in sel_buffer[n+1] */
		/* RH6.2 aic driver can't handle pages this long, 
		 * so skip this one. */
		printf("(skip)");
		showlog( "(skip - length=%d)\n", mlen);
	    } /* endif page length bug */
	    else
#endif
	    {			/* normal path */
		sts = mode_select(devfd, sel_buffer, mlen);
		if (sts) {
		    sprintf(output,
			    "[%d]: Error %d setting Mode Page %x\n", idx,
			    sts, pagecode);
		    showit(output);
		}
	    }
	}			/*end while */
	printf("\n");
	fclose(fp1);		/* close mdf file */
    }				/* end-else mdf opened */
    return (sts);
}

/*
 * decodemodebuf 
 *
 * Decode & display interpreted mode page data.
 * The intent is not to interpret everything, but to 
 * display the more commonly changed mode parameters.
 */
void decodemodebuf(FILE *fdout, uchar *buf)
{
    int i, n, sz, v;
    bool b0, b1, b2, b3, b4, b5, b6, b7;
    uchar mlen;
    char *pstr;
    
    mlen = buf[0];		/*mode header has total mlen */
    sz = 0;
    /* traverse the mode page buffer */
    for (i = 12; i < mlen; i += sz) {
	n = buf[i] & 0x7f;       /*mode page number */
	sz = buf[i+1] + 2;  /* size of this mode page */
	switch(n)
	{
	   case 0x01:   /* error recovery page */
		fprintf(fdout,"page %02x: ",n);
		if (buf[i+2] & 0x80) b7 = 1; else b7 = 0;
		if (buf[i+2] & 0x40) b6 = 1; else b6 = 0;
		if (buf[i+2] & 0x20) b5 = 1; else b5 = 0;
		if (buf[i+2] & 0x10) b4 = 1; else b4 = 0;
		if (buf[i+2] & 0x08) b3 = 1; else b3 = 0;
		if (buf[i+2] & 0x04) b2 = 1; else b2 = 0;
		if (buf[i+2] & 0x02) b1 = 1; else b1 = 0;
		if (buf[i+2] & 0x01) b0 = 1; else b0 = 0;
		fprintf(fdout,"AWRE=%d ARRE=%d EER=%d PER=%d DTE=%d DCR=%d ",
			b7,b6,b3,b2,b1,b0);
		v = (buf[i+10] << 8) + buf[i+11];  /*recovery time*/
		fprintf(fdout,"Rretries=%d Wretries=%d time=%d\n",
			buf[i+3], buf[i+8], v);
		break;
	   case 0x07:   /* verify error recovery page */
		fprintf(fdout,"page %02x: ",n);
		if (buf[i+2] & 0x08) b3 = 1; else b3 = 0;
		if (buf[i+2] & 0x04) b2 = 1; else b2 = 0;
		if (buf[i+2] & 0x02) b1 = 1; else b1 = 0;
		if (buf[i+2] & 0x01) b0 = 1; else b0 = 0;
		v = (buf[i+10] << 8) + buf[i+11];  /*recovery time*/
		fprintf(fdout,"verify EER=%d PER=%d DTE=%d DCR=%d time=%d\n",
			b3,b2,b1,b0,v);
		break;
	   case 0x08:    /* Caching page */
		fprintf(fdout,"page %02x: ",n);
		if (buf[i+2] & 0x40) b6 = 1; else b6 = 0;
		if (buf[i+2] & 0x04) b2 = 1; else b2 = 0;
		if (buf[i+2] & 0x01) b0 = 1; else b0 = 0;
		fprintf(fdout,"CAP=%d WCE=%d RCD=%d\n",b6,b2,b0);
		break;
	   case 0x0a:    /* Control mode page */
		fprintf(fdout,"page %02x: ",n);
		if (buf[i+4] & 0x08) b3 = 1; else b3 = 0;
		if (buf[i+4] & 0x04) b2 = 1; else b2 = 0;
		if (buf[i+4] & 0x02) b1 = 1; else b1 = 0;
		if (buf[i+4] & 0x01) b0 = 1; else b0 = 0;
		v = (buf[i+8] << 8) + buf[i+9];  /*busy timeout*/
		fprintf(fdout,
			"SWP=%d RAERP=%d UAAERP=%d EAERP=%d BusyTime=%d\n",
			b3,b2,b1,b0,v);
		break;
	   case 0x1c:    /* Info exceptions (SMART) */
		fprintf(fdout,"page %02x: ",n);
		if (buf[i+2] == 0x88) pstr = "SMART=off,";
		else pstr = "SMART=on,";
		if (buf[i+2] & 0x80) b7 = 1; else b7 = 0;
		if (buf[i+2] & 0x08) b3 = 1; else b3 = 0;
		if (buf[i+2] & 0x01) b0 = 1; else b0 = 0;
		v = (buf[i+8] << 24) + (buf[i+9] << 16) + (buf[i+10] << 8) +
			buf[i+11]; 
		fprintf(fdout,"%s Perf=%d DExcpt=%d LogErr=%d MRIE=%x cnt=%d\n",
			pstr,b7,b3,b0,(buf[i+3]&0x0f),v);
		break;
	   default:
		break;
	}
    }
    return;
}  /*end decodemodebuf*/

/* 
 * showmodebuf
 * Show the mode page buffer as hex bytes.
 */
void showmodebuf(FILE * fdout, uchar * bufp, char flimit)
{
    int i, j, k, cnt;
    uchar mlen, pg;
    ulong cap;
    uchar fbdzero;
    mlen = bufp[0];		/*mode header has total mlen */
    if (!flimit) {
       if (fdebug) {
 	  fprintf(fdout, "mlen = %d\n", mlen);
	  fprintf(fdout, "bufp: %02x %02x %02x %02x\n",
		  bufp[0], bufp[1], bufp[2], bufp[3]);
       }
       // if (mlen > 280) return;  /* uchar max is 256 anyway */
       cap = cap_report(bufp);
       fprintf(fdout, "Capacity: %luMB (MB=1024*1024)\n"
		"Block Descriptor & Mode Pages:\n",
		 cap / (onemeg/512));
    } 
    fbdzero = 1;  
    for (i = 0; i < 8; i++) {
        if (bufp[4 + i] != 0) fbdzero = 0;  /*if any block desc is non-zero*/
	fprintf(fdout, "%02x ", bufp[4 + i]);
    }
    fprintf(fdout, "   :Block Descriptor");
    if (fbdzero) {
       fprintf(fdout, "\nInvalid block descriptor (mlen=%d)\n",mlen);
    } else {
       cnt = 0;
       for (i = 12; i < mlen; cnt++) {
	  char abuf[5];
          pg = bufp[i] & 0x7F;
	  fprintf(fdout, "\n%02x ", pg);
	  k = bufp[i + 1] + 2;
	  for (j = 1; j < k; j++) {
	    itoh(&bufp[i + j], 1, abuf);
	    fprintf(fdout, "%s", abuf);
	  }
	  i += k;
          if (flimit && (pg >= 0x1c)) break;
      }
      fprintf(fdout, "\n");
      if (!flimit)
        decodemodebuf(fdout,bufp);  /*show interpreted*/
    }
    return;
}				/*end showmodebuf() */

int do_modesense(int idx)
{
    int sts;
    int devfd;
    int isave;
    int i, rc, rcdl;
    int k, a, q;
    uchar ch, dv;
    uchar *bufp;
    ulong cap;
    isave = idx;
    rcdl = 0;
    /* Get Mode pages for each disk */
    {
	rcdl = 0;
	fdoreset = 0;
	if (finteractive || fonedev) 
	    idx = isave;	/* only do selected disk */
	devfd = devlist[idx].sgfd;
	ch = devlist[idx].chn;
	dv = devlist[idx].tgt;
	if (devlist[idx].fdoit == 1) {
	    if (fautomodel) {	/* if (fautomode || fauto) -- why ?? */
		/* Sometimes next device may have a unit attention/reset check 
		 * condition pending, so clear that first before the mode command.
		 */
		sts = test_unit_ready(devfd, 0);
		if (fdebug)
		    printf("Extra test_unit_ready sts = %x\n", sts);
	    }
	    /* start the modepage command */
	    sprintf(output, "Getting mode sense pages from disk %d\n",
		    idx);
	    showit(output);
	    bufp = modebuf;
	    sts = mode_sense(devfd, 0x3F, bufp);
	    if (sts) {
		rcdl = sts;
		sprintf(output, "[%d] Error %d in mode_sense\n", idx, sts);
		showit(output);
		if (fdebug || !fautomodel) {
		    return (rcdl);	//  (only if debug on)
		}
	    } else {		/* good results */
	        cap = cap_report(bufp);
		devlist[idx].dcap = cap / (onemeg/512);
		if (fdebug) showmsg("showmode1 fdmsg=%x fdlog=%x\n",fdmsg,fdlog);
		showmodebuf(fdmsg, bufp,0);
		if (fdebug) showmsg("showmode2 fdmsg=%x fdlog=%x\n",fdmsg,fdlog);
		if (filesok && flogopen)
		    showmodebuf(fdlog, bufp,0);
		/* reuse some of the modebuf in the mode select also. */
                if (fsavemdf && filesok) {  /*want to save mdf if not exist */
    		   if (!fsetfile) {  /*not a user-specified file */
                      char mdpath[100];
		      build_mdf_name(idx,mdfile);
		      strcpy(mdpath,defdir);
		      strcat(mdpath,mdfile);
	              fdtmp = fopen(mdpath, "w");
		      if (fdebug) 
		         showmsg("mdfile %s open fd =%p\n",mdpath,fdtmp);
                      if (fdtmp != NULL) {
		         showmodebuf(fdtmp, bufp,1);
		         fclose(fdtmp);
		      }
                   } 
		}
	    }			/*endif good */
	    if ((fauto || fautomodel) && rcdl != 0)
		return (rcdl);	/*skip to next one if error */
	    sts = test_unit_ready(devfd, 0);
	    for (i = 0; i < 30 && sts; i++) {
		sts = test_unit_ready(devfd, 0);
		if (sts == SCHECK_CND) {	/* check for certain key/asc/ascq values */
		    rc = get_sense(sts, sense_buffer);
		    k = sense_buffer[2] & 0x0f;	/*Sense Key */
		    a = sense_buffer[12];
		    /*ASC*/ q = sense_buffer[13];
		    /*ASCQ*/ if (k == 2 && a == 4 && q == 2) {
			/* 02-04-02 means it needs a start command, so issue it. */
			sts = start_unit(devfd);
			if (fdebug) {
			    /*DEBUG*/
				printf("\nStart Unit: sts=%d, ", sts);
			}
		    }		/*endif 02-04-02 */
		} /*endif sts */
		else {
		    k = 0;
		    a = 0;
		    q = 0;
		}
		if (fdebug)
		    fprintf(fdmsg, "[%d: %d %d %d]", sts, k, a, q);
		fprintf(fdmsg, ".");
		sleep(3);
	    }			/* end TUR for loop */
	    printf("\n");
	    sprintf(output,
		    "[%d] ModeSense commands complete, status = 0x%x\n",
		    idx, rcdl);
	    showit(output);
	    if ((rcdl & 0x0FFF) == 2) {
		sprintf(output, "   Sense data: %02x %02x %02x\n",
			sense_buffer[2] & 0x0f, sense_buffer[12],
			sense_buffer[13]);
		showit(output);
	    }
	    if (fdebug && rcdl != 0) {
		return (rcdl);	/* return (only if debug on) */
	    }
	    if (finteractive || fonedev) 
		idx = ndev;	/* end loop */
	}			/*endif fdoit */
    }				/*endfor each disk */
    return (rcdl);
}

int beforemd(int idx, int *pnumrdy)
{
    struct SCSI_INQUIRY *scsi_inq;
    int sts;
    int sgfd;
    int isave, nrdy;
    char *pl;
    int i, rc, rcdl;
    int k, a, q;
    int openflags;
    char *fname;
    char ftry;
    isave = idx;
    nrdy = 0;
    sts = 0;
    rcdl = 0;
    openflags = O_RDWR;		/* | O_NONBLOCK; */
    /* if fautomodel, use global model from optarg, otherwise use any. */
    if (!fautomodel) {
	scsi_inq = (struct SCSI_INQUIRY *) devlist[idx].inq;
	strncpy(model, (char *) scsi_inq->productid, 8);
	model[8] = 0;		/*stringify */
    }
    for (idx = 0; idx < ndev; idx++) {
	if (finteractive || fonedev)
	    idx = isave;	/* only do selected disk */
	fname = devlist[idx].fname;
	sgfd = open(fname, openflags);	/* open blocking */
	if (sgfd < 0) {
	    printf("%s: open error, errno = %d\n", fname, errno);
	    return (errno);
	}
	devlist[idx].sgfd = sgfd;
        sts = set_sg_debug(sgfd, sgdbglvl);
        if (sts) 
	    showlog( "[%d] cant set debug level %d, sts = %d\n", 
                    idx, sgdbglvl, sts);
	scsi_inq = (struct SCSI_INQUIRY *) devlist[idx].inq;
	ftry = 0;
	if (fautomodel
	    && strncmp(model, (char *) scsi_inq->productid, 8) == 0)
	    ftry = 1;
	else
	    ftry = 1;
	if (fdebug && ftry)
	    /*DEBUG*/
		printf("Doing Test Unit Ready on idx = %d, model=%s\n",
		       idx, model);
	if (ftry) {		/* do only ones of the same disk product id */
	    sts = test_unit_ready(sgfd, 0);	/* dont show errors */
	    if (sts == SCHECK_CND) {	/* not ready initially, try to make it ready */
		i = get_sense(sts, sense_buffer);
		k = sense_buffer[2] & 0x0f;	/*Sense Key */
		a = sense_buffer[12];
		/*ASC*/ q = sense_buffer[13];
		/*ASCQ*/ if (k == 2 && a == 4 && q == 2) {
		    /* 02-04-02 means it needs a start command, so issue it. */
		    sts = start_unit(sgfd);
		    if (fdebug) {
			/*DEBUG*/ printf("\nStart Unit: sts=%d, ", sts);
			if (sts == SCHECK_CND) {
			    rc = get_sense(sts, sense_buffer);
			    printf("sense data: %x %x %x\n",
				   sense_buffer[2] & 0x0f,
				   sense_buffer[12], sense_buffer[13]);
			}
		    }
		}		/*endif 02-04-02 */
		sts = test_unit_ready(sgfd, 1);	/* try again & show errors */
	    }
	    if (sts) {
		sprintf(output, "[%d] Error %d from Test Unit Ready\n",
			idx, sts);
		showit(output);
		devlist[idx].fdoit = 0;
	    } else {
		switch(devlist[idx].devtype) {
		   case DEVT_PROC:
		   case DEVT_ENCL:
		   case DEVT_EMUL:
		       devlist[idx].fdoit = 0;
		       break;
		   default:  /* fdoit=1 for Disk, Tape, etc. */
		       devlist[idx].fdoit = 1;
		       break;
		}
		if (devlist[idx].fdoit) {
		   sprintf(output,"Device [%d] is ready for modepage\n",idx);
		   showit(output);
		}
		nrdy++;		/* number ready to modepage */
	    }
	} else
	    devlist[idx].fdoit = 0;	/* not same product id */
	if (finteractive || fonedev)
	    idx = ndev;		/* end loop */
    }				/* end for devlist TUR loop */
    *pnumrdy = nrdy;		/* set return value */
    if (nrdy == 0) {
	sprintf(output, "There are no ready devices for model %s.\n", model);
        strcat(output,"Try 'modprobe sg' if not already loaded\n"); 
	showit(output);
	return (sts);
    } else {
	if (nrdy > 1)
	    pl = "s";
	else
	    pl = "";
	sprintf(output, "Starting modepage process for %d disk%s.\n", nrdy,
		pl);
	showit(output);
    }
    setbuf(fdmsg, NULL);	/* set for unbuffered writes */
    return (rcdl);
}				/*end beforemd() */

int aftermd(void)
{
    int sts, i;
    sts = 0;
    for (i = 0; i < ndev; i++) {
	if (devlist[i].sgfd > 0) {
	    sts = set_sg_debug(devlist[i].sgfd, 0);
	    if (sts)
		showlog( "[%d] cant clear debug flag, sts = %d\n", i, sts);
	    close(devlist[i].sgfd);	/*opened in beforedl() */
	    devlist[i].sgfd = 0;
	}
    }
    /* done, all drives should be ready now */
    return (sts);
}				/*end aftermd() */

/* end sgmode.c */
