pub const ext = @import("ext.zig");
const handy = @This();

const std = @import("std");
const compat = @import("compat");
const gtk = @import("gtk3");
const xlib = @import("xlib2");
const gdk = @import("gdk3");
const cairo = @import("cairo1");
const gobject = @import("gobject2");
const glib = @import("glib2");
const pango = @import("pango1");
const harfbuzz = @import("harfbuzz0");
const freetype2 = @import("freetype22");
const gio = @import("gio2");
const gmodule = @import("gmodule2");
const gdkpixbuf = @import("gdkpixbuf2");
const atk = @import("atk1");
/// A `gtk.ListBoxRow` used to present actions.
///
/// The `HdyActionRow` widget can have a title, a subtitle and an icon. The row
/// can receive additional widgets at its end, or prefix widgets at its start.
///
/// It is convenient to present a preference and its related actions.
///
/// `HdyActionRow` is unactivatable by default, giving it an activatable widget
/// will automatically make it activatable, but unsetting it won't change the
/// row's activatability.
///
/// ## HdyActionRow as GtkBuildable
///
/// The `HdyActionRow` implementation of the `gtk.Buildable` interface
/// supports adding a child at its end by specifying “suffix” or omitting the
/// “type” attribute of a <child> element.
///
/// It also supports adding a child as a prefix widget by specifying “prefix” as
/// the “type” attribute of a <child> element.
///
/// ## CSS nodes
///
/// `HdyActionRow` has a main CSS node with name `row`.
///
/// It contains the subnode `box.header` for its main horizontal box, and
/// `box.title` for the vertical box containing the title and subtitle labels.
///
/// It contains subnodes `label.title` and `label.subtitle` representing
/// respectively the title label and subtitle label.
pub const ActionRow = extern struct {
    pub const Parent = handy.PreferencesRow;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Actionable, gtk.Buildable };
    pub const Class = handy.ActionRowClass;
    f_parent_instance: handy.PreferencesRow,

    pub const virtual_methods = struct {
        /// Activates `self`.
        pub const activate = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(ActionRow.Class, p_class).f_activate.?(gobject.ext.as(ActionRow, p_self));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(ActionRow.Class, p_class).f_activate = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The activatable widget for this row.
        ///
        /// The widget is activated, either by clicking on it, by calling
        /// `ActionRow.activate`, or via mnemonics in the title or the subtitle.
        /// See the `ActionRow.properties.use_underline` property to enable mnemonics.
        ///
        /// The target widget will be activated by emitting the
        /// `gtk.Widget.signals.mnemonic_activate` signal on it.
        pub const activatable_widget = struct {
            pub const name = "activatable-widget";

            pub const Type = ?*gtk.Widget;
        };

        /// The icon name for this row.
        pub const icon_name = struct {
            pub const name = "icon-name";

            pub const Type = ?[*:0]u8;
        };

        /// The subtitle for this row.
        pub const subtitle = struct {
            pub const name = "subtitle";

            pub const Type = ?[*:0]u8;
        };

        /// The number of lines at the end of which the subtitle label will be
        /// ellipsized.
        ///
        /// If the value is 0, the number of lines won't be limited.
        pub const subtitle_lines = struct {
            pub const name = "subtitle-lines";

            pub const Type = c_int;
        };

        /// The number of lines at the end of which the title label will be ellipsized.
        ///
        /// If the value is 0, the number of lines won't be limited.
        pub const title_lines = struct {
            pub const name = "title-lines";

            pub const Type = c_int;
        };

        /// Whether embedded underlines in the title or subtitle indicates a mnemonic.
        ///
        /// If true, an underline in the text of the title or subtitle labels indicates
        /// the next character should be used for the mnemonic accelerator key.
        pub const use_underline = struct {
            pub const name = "use-underline";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// This signal is emitted after the row has been activated.
        pub const activated = struct {
            pub const name = "activated";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ActionRow, p_instance))),
                    gobject.signalLookup("activated", ActionRow.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `HdyActionRow`.
    extern fn hdy_action_row_new() *handy.ActionRow;
    pub const new = hdy_action_row_new;

    /// Activates `self`.
    extern fn hdy_action_row_activate(p_self: *ActionRow) void;
    pub const activate = hdy_action_row_activate;

    /// Adds a prefix widget to `self`.
    extern fn hdy_action_row_add_prefix(p_self: *ActionRow, p_widget: *gtk.Widget) void;
    pub const addPrefix = hdy_action_row_add_prefix;

    /// Gets the widget activated when `self` is activated.
    extern fn hdy_action_row_get_activatable_widget(p_self: *ActionRow) ?*gtk.Widget;
    pub const getActivatableWidget = hdy_action_row_get_activatable_widget;

    /// Gets the icon name for `self`.
    extern fn hdy_action_row_get_icon_name(p_self: *ActionRow) [*:0]const u8;
    pub const getIconName = hdy_action_row_get_icon_name;

    /// Gets the subtitle for `self`.
    extern fn hdy_action_row_get_subtitle(p_self: *ActionRow) ?[*:0]const u8;
    pub const getSubtitle = hdy_action_row_get_subtitle;

    /// Gets the number of lines at the end of which the subtitle label will be
    /// ellipsized.
    ///
    /// If the value is 0, the number of lines won't be limited.
    extern fn hdy_action_row_get_subtitle_lines(p_self: *ActionRow) c_int;
    pub const getSubtitleLines = hdy_action_row_get_subtitle_lines;

    /// Gets the number of lines at the end of which the title label will be
    /// ellipsized.
    ///
    /// If the value is 0, the number of lines won't be limited.
    extern fn hdy_action_row_get_title_lines(p_self: *ActionRow) c_int;
    pub const getTitleLines = hdy_action_row_get_title_lines;

    /// Gets whether an embedded underline in the title or subtitle indicates a
    /// mnemonic.
    extern fn hdy_action_row_get_use_underline(p_self: *ActionRow) c_int;
    pub const getUseUnderline = hdy_action_row_get_use_underline;

    /// Sets the widget to activate when `self` is activated.
    extern fn hdy_action_row_set_activatable_widget(p_self: *ActionRow, p_widget: ?*gtk.Widget) void;
    pub const setActivatableWidget = hdy_action_row_set_activatable_widget;

    /// Sets the icon name for `self`.
    extern fn hdy_action_row_set_icon_name(p_self: *ActionRow, p_icon_name: [*:0]const u8) void;
    pub const setIconName = hdy_action_row_set_icon_name;

    /// Sets the subtitle for `self`.
    extern fn hdy_action_row_set_subtitle(p_self: *ActionRow, p_subtitle: ?[*:0]const u8) void;
    pub const setSubtitle = hdy_action_row_set_subtitle;

    /// Sets the number of lines at the end of which the subtitle label will be
    /// ellipsized.
    ///
    /// If the value is 0, the number of lines won't be limited.
    extern fn hdy_action_row_set_subtitle_lines(p_self: *ActionRow, p_subtitle_lines: c_int) void;
    pub const setSubtitleLines = hdy_action_row_set_subtitle_lines;

    /// Sets the number of lines at the end of which the title label will be
    /// ellipsized.
    ///
    /// If the value is 0, the number of lines won't be limited.
    extern fn hdy_action_row_set_title_lines(p_self: *ActionRow, p_title_lines: c_int) void;
    pub const setTitleLines = hdy_action_row_set_title_lines;

    /// Sets whether an embedded underline in the title or subtitle indicates a
    /// mnemonic.
    extern fn hdy_action_row_set_use_underline(p_self: *ActionRow, p_use_underline: c_int) void;
    pub const setUseUnderline = hdy_action_row_set_use_underline;

    extern fn hdy_action_row_get_type() usize;
    pub const getGObjectType = hdy_action_row_get_type;

    extern fn g_object_ref(p_self: *handy.ActionRow) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.ActionRow) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ActionRow, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A freeform application window.
///
/// `HdyApplicationWindow` is a `gtk.ApplicationWindow` subclass providing
/// the same features as `Window`.
///
/// See `Window` for details.
///
/// Using `gtk.Application.setAppMenu` and
/// `gtk.Application.setMenubar` is not supported and may result in
/// visual glitches.
pub const ApplicationWindow = extern struct {
    pub const Parent = gtk.ApplicationWindow;
    pub const Implements = [_]type{ atk.ImplementorIface, gio.ActionGroup, gio.ActionMap, gtk.Buildable };
    pub const Class = handy.ApplicationWindowClass;
    f_parent_instance: gtk.ApplicationWindow,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `HdyApplicationWindow`.
    extern fn hdy_application_window_new() *handy.ApplicationWindow;
    pub const new = hdy_application_window_new;

    extern fn hdy_application_window_get_type() usize;
    pub const getGObjectType = hdy_application_window_get_type;

    extern fn g_object_ref(p_self: *handy.ApplicationWindow) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.ApplicationWindow) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ApplicationWindow, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A widget displaying an image, with a generated fallback.
///
/// `HdyAvatar` is a widget to display a round avatar.
///
/// A provided image is made round before displaying, if no image is given this
/// widget generates a round fallback with the initials of the
/// `Avatar.properties.text` on top of a colored background.
///
/// The color is picked based on the hash of the `Avatar.properties.text`.
///
/// If `Avatar.properties.show_initials` is set to `FALSE`,
/// `avatar-default-symbolic` is shown instead of the initials.
///
/// Use `Avatar.setLoadableIcon` or `Avatar.properties.loadable_icon` to
/// set a custom image.
///
/// ## CSS nodes
///
/// `HdyAvatar` has a single CSS node with name `avatar`.
pub const Avatar = opaque {
    pub const Parent = gtk.DrawingArea;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.AvatarClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The name of an icon to use as a fallback.
        ///
        /// If no name is set, the avatar-default-symbolic icon will be used. If the
        /// name doesn't match a valid icon, it is an error and no icon will be
        /// displayed. If the icon theme is changed, the image will be updated
        /// automatically.
        pub const icon_name = struct {
            pub const name = "icon-name";

            pub const Type = ?[*:0]u8;
        };

        /// A `gio.LoadableIcon` used to load the avatar.
        pub const loadable_icon = struct {
            pub const name = "loadable-icon";

            pub const Type = ?*gio.LoadableIcon;
        };

        /// Whether to show the initials or the fallback icon on the generated avatar.
        pub const show_initials = struct {
            pub const name = "show-initials";

            pub const Type = c_int;
        };

        /// The avatar size of the avatar.
        pub const size = struct {
            pub const name = "size";

            pub const Type = c_int;
        };

        /// Sets the text used to generate the fallback initials and color.
        ///
        /// It's only used to generate the color if `Avatar.properties.show_initials` is
        /// `FALSE`.
        pub const text = struct {
            pub const name = "text";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyAvatar`.
    extern fn hdy_avatar_new(p_size: c_int, p_text: ?[*:0]const u8, p_show_initials: c_int) *handy.Avatar;
    pub const new = hdy_avatar_new;

    /// Renders `self` into a `gdkpixbuf.Pixbuf` at `size` and `scale_factor`.
    ///
    /// This can be used to export the fallback avatar.
    extern fn hdy_avatar_draw_to_pixbuf(p_self: *Avatar, p_size: c_int, p_scale_factor: c_int) *gdkpixbuf.Pixbuf;
    pub const drawToPixbuf = hdy_avatar_draw_to_pixbuf;

    /// Renders asynchronously `self` into a pixbuf at `size` and `scale_factor`.
    ///
    /// This can be used to export the fallback avatar.
    extern fn hdy_avatar_draw_to_pixbuf_async(p_self: *Avatar, p_size: c_int, p_scale_factor: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const drawToPixbufAsync = hdy_avatar_draw_to_pixbuf_async;

    /// Finishes an asynchronous draw of an avatar to a pixbuf.
    extern fn hdy_avatar_draw_to_pixbuf_finish(p_self: *Avatar, p_async_result: *gio.AsyncResult) *gdkpixbuf.Pixbuf;
    pub const drawToPixbufFinish = hdy_avatar_draw_to_pixbuf_finish;

    /// Gets the name of an icon to use as a fallback.
    extern fn hdy_avatar_get_icon_name(p_self: *Avatar) ?[*:0]const u8;
    pub const getIconName = hdy_avatar_get_icon_name;

    /// Gets the `gio.LoadableIcon` set via `Avatar.setLoadableIcon`.
    extern fn hdy_avatar_get_loadable_icon(p_self: *Avatar) ?*gio.LoadableIcon;
    pub const getLoadableIcon = hdy_avatar_get_loadable_icon;

    /// Gets whether initials are used instead of an icon on the fallback avatar.
    extern fn hdy_avatar_get_show_initials(p_self: *Avatar) c_int;
    pub const getShowInitials = hdy_avatar_get_show_initials;

    /// Gets the size of the avatar.
    extern fn hdy_avatar_get_size(p_self: *Avatar) c_int;
    pub const getSize = hdy_avatar_get_size;

    /// Gets the text used to generate the fallback initials and color.
    extern fn hdy_avatar_get_text(p_self: *Avatar) ?[*:0]const u8;
    pub const getText = hdy_avatar_get_text;

    /// Sets the name of an icon to use as a fallback.
    ///
    /// If no name is set, `avatar-default-symbolic` will be used.
    extern fn hdy_avatar_set_icon_name(p_self: *Avatar, p_icon_name: ?[*:0]const u8) void;
    pub const setIconName = hdy_avatar_set_icon_name;

    /// A callback which is called when the custom image needs to be reloaded.
    ///
    /// It will be called on `Avatar.properties.size` or
    /// `gtk.Widget.properties.scale_factor` changes.
    extern fn hdy_avatar_set_image_load_func(p_self: *Avatar, p_load_image: ?handy.AvatarImageLoadFunc, p_user_data: ?*anyopaque, p_destroy: ?glib.DestroyNotify) void;
    pub const setImageLoadFunc = hdy_avatar_set_image_load_func;

    /// Sets the `gio.LoadableIcon` to use as an avatar.
    ///
    /// The previous avatar is displayed till the new avatar is loaded, to
    /// immediately remove the custom avatar set the loadable-icon to `NULL`.
    ///
    /// The `gio.LoadableIcon` set via this function is preferred over a set
    /// `AvatarImageLoadFunc`.
    extern fn hdy_avatar_set_loadable_icon(p_self: *Avatar, p_icon: ?*gio.LoadableIcon) void;
    pub const setLoadableIcon = hdy_avatar_set_loadable_icon;

    /// Sets whether to use initials instead of an icon on the fallback avatar.
    extern fn hdy_avatar_set_show_initials(p_self: *Avatar, p_show_initials: c_int) void;
    pub const setShowInitials = hdy_avatar_set_show_initials;

    /// Sets the size of the avatar.
    extern fn hdy_avatar_set_size(p_self: *Avatar, p_size: c_int) void;
    pub const setSize = hdy_avatar_set_size;

    /// Set the text used to generate the fallback initials color.
    extern fn hdy_avatar_set_text(p_self: *Avatar, p_text: ?[*:0]const u8) void;
    pub const setText = hdy_avatar_set_text;

    extern fn hdy_avatar_get_type() usize;
    pub const getGObjectType = hdy_avatar_get_type;

    extern fn g_object_ref(p_self: *handy.Avatar) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.Avatar) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Avatar, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A paginated scrolling widget.
///
/// The `HdyCarousel` widget can be used to display a set of pages with
/// swipe-based navigation between them.
///
/// `CarouselIndicatorDots` and `CarouselIndicatorLines` can be used
/// to provide page indicators for `HdyCarousel`.
///
/// ## CSS nodes
///
/// `HdyCarousel` has a single CSS node with name `carousel`.
pub const Carousel = opaque {
    pub const Parent = gtk.EventBox;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable, handy.Swipeable };
    pub const Class = handy.CarouselClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether to allow swiping for more than one page at a time.
        ///
        /// If the value is `FALSE`, each swipe can only move to the adjacent pages.
        pub const allow_long_swipes = struct {
            pub const name = "allow-long-swipes";

            pub const Type = c_int;
        };

        /// Sets whether the `Carousel` can be dragged with mouse pointer.
        ///
        /// If the value is `FALSE`, dragging is only available on touch.
        pub const allow_mouse_drag = struct {
            pub const name = "allow-mouse-drag";

            pub const Type = c_int;
        };

        /// Whether the widget will respond to scroll wheel events.
        ///
        /// If the value is `FALSE`, wheel events will be ignored.
        pub const allow_scroll_wheel = struct {
            pub const name = "allow-scroll-wheel";

            pub const Type = c_int;
        };

        /// Animation duration used by `Carousel.scrollTo`, in milliseconds.
        pub const animation_duration = struct {
            pub const name = "animation-duration";

            pub const Type = c_uint;
        };

        /// Whether the carousel can be navigated.
        ///
        /// This can be used to temporarily disable a `HdyCarousel` to only allow
        /// navigating it in a certain state.
        pub const interactive = struct {
            pub const name = "interactive";

            pub const Type = c_int;
        };

        /// The number of pages in a `Carousel`.
        pub const n_pages = struct {
            pub const name = "n-pages";

            pub const Type = c_uint;
        };

        /// Current scrolling position, unitless.
        ///
        /// 1 matches 1 page. Use `Carousel.scrollTo` for changing it.
        pub const position = struct {
            pub const name = "position";

            pub const Type = f64;
        };

        /// Page reveal duration, in milliseconds.
        pub const reveal_duration = struct {
            pub const name = "reveal-duration";

            pub const Type = c_uint;
        };

        /// Spacing between pages in pixels.
        pub const spacing = struct {
            pub const name = "spacing";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {
        /// This signal is emitted after a page has been changed.
        ///
        /// It can be used to implement "infinite scrolling" by amending the pages
        /// after every scroll.
        pub const page_changed = struct {
            pub const name = "page-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_index: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Carousel, p_instance))),
                    gobject.signalLookup("page-changed", Carousel.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `HdyCarousel`.
    extern fn hdy_carousel_new() *handy.Carousel;
    pub const new = hdy_carousel_new;

    /// Gets whether to allow swiping for more than one page at a time.
    extern fn hdy_carousel_get_allow_long_swipes(p_self: *Carousel) c_int;
    pub const getAllowLongSwipes = hdy_carousel_get_allow_long_swipes;

    /// Sets whether `self` can be dragged with mouse pointer.
    extern fn hdy_carousel_get_allow_mouse_drag(p_self: *Carousel) c_int;
    pub const getAllowMouseDrag = hdy_carousel_get_allow_mouse_drag;

    /// Gets whether `self` will respond to scroll wheel events.
    extern fn hdy_carousel_get_allow_scroll_wheel(p_self: *Carousel) c_int;
    pub const getAllowScrollWheel = hdy_carousel_get_allow_scroll_wheel;

    /// Gets animation duration used by `Carousel.scrollTo`.
    extern fn hdy_carousel_get_animation_duration(p_self: *Carousel) c_uint;
    pub const getAnimationDuration = hdy_carousel_get_animation_duration;

    /// Gets whether `self` can be navigated.
    extern fn hdy_carousel_get_interactive(p_self: *Carousel) c_int;
    pub const getInteractive = hdy_carousel_get_interactive;

    /// Gets the number of pages in `self`.
    extern fn hdy_carousel_get_n_pages(p_self: *Carousel) c_uint;
    pub const getNPages = hdy_carousel_get_n_pages;

    /// Gets current scroll position in `self`. It's unitless, 1 matches 1 page.
    extern fn hdy_carousel_get_position(p_self: *Carousel) f64;
    pub const getPosition = hdy_carousel_get_position;

    /// Gets duration of the animation used when adding or removing pages, in
    /// milliseconds.
    extern fn hdy_carousel_get_reveal_duration(p_self: *Carousel) c_uint;
    pub const getRevealDuration = hdy_carousel_get_reveal_duration;

    /// Gets spacing between pages in pixels.
    extern fn hdy_carousel_get_spacing(p_self: *Carousel) c_uint;
    pub const getSpacing = hdy_carousel_get_spacing;

    /// Inserts `child` into `self` at position `position`.
    ///
    /// If position is -1, or larger than the number of pages, `child` will be
    /// appended to the end.
    extern fn hdy_carousel_insert(p_self: *Carousel, p_child: *gtk.Widget, p_position: c_int) void;
    pub const insert = hdy_carousel_insert;

    /// Prepends `child` to `self`.
    extern fn hdy_carousel_prepend(p_self: *Carousel, p_child: *gtk.Widget) void;
    pub const prepend = hdy_carousel_prepend;

    /// Moves `child` into position `position`.
    ///
    /// If position is -1, or larger than the number of pages, `child` will be moved
    /// to the end.
    extern fn hdy_carousel_reorder(p_self: *Carousel, p_child: *gtk.Widget, p_position: c_int) void;
    pub const reorder = hdy_carousel_reorder;

    /// Scrolls to `widget` position with an animation.
    ///
    /// `Carousel.properties.animation_duration` property can be used for controlling
    /// the duration.
    extern fn hdy_carousel_scroll_to(p_self: *Carousel, p_widget: *gtk.Widget) void;
    pub const scrollTo = hdy_carousel_scroll_to;

    /// Scrolls to `widget` position with an animation.
    extern fn hdy_carousel_scroll_to_full(p_self: *Carousel, p_widget: *gtk.Widget, p_duration: i64) void;
    pub const scrollToFull = hdy_carousel_scroll_to_full;

    /// Sets whether to allow swiping for more than one page at a time.
    extern fn hdy_carousel_set_allow_long_swipes(p_self: *Carousel, p_allow_long_swipes: c_int) void;
    pub const setAllowLongSwipes = hdy_carousel_set_allow_long_swipes;

    /// Sets whether `self` can be dragged with mouse pointer.
    ///
    /// If `allow_mouse_drag` is `FALSE`, dragging is only available on touch.
    extern fn hdy_carousel_set_allow_mouse_drag(p_self: *Carousel, p_allow_mouse_drag: c_int) void;
    pub const setAllowMouseDrag = hdy_carousel_set_allow_mouse_drag;

    /// Sets whether `self` will respond to scroll wheel events.
    extern fn hdy_carousel_set_allow_scroll_wheel(p_self: *Carousel, p_allow_scroll_wheel: c_int) void;
    pub const setAllowScrollWheel = hdy_carousel_set_allow_scroll_wheel;

    /// Sets animation duration used by `Carousel.scrollTo`.
    extern fn hdy_carousel_set_animation_duration(p_self: *Carousel, p_duration: c_uint) void;
    pub const setAnimationDuration = hdy_carousel_set_animation_duration;

    /// Sets whether `self` can be navigated.
    ///
    /// This can be used to temporarily disable a `Carousel` to only allow
    /// swiping in a certain state.
    extern fn hdy_carousel_set_interactive(p_self: *Carousel, p_interactive: c_int) void;
    pub const setInteractive = hdy_carousel_set_interactive;

    /// Sets duration of the animation used when adding or removing pages, in
    /// milliseconds.
    extern fn hdy_carousel_set_reveal_duration(p_self: *Carousel, p_reveal_duration: c_uint) void;
    pub const setRevealDuration = hdy_carousel_set_reveal_duration;

    /// Sets spacing between pages in pixels.
    extern fn hdy_carousel_set_spacing(p_self: *Carousel, p_spacing: c_uint) void;
    pub const setSpacing = hdy_carousel_set_spacing;

    extern fn hdy_carousel_get_type() usize;
    pub const getGObjectType = hdy_carousel_get_type;

    extern fn g_object_ref(p_self: *handy.Carousel) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.Carousel) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Carousel, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A dots indicator for `Carousel`.
///
/// The `HdyCarouselIndicatorDots` widget shows a set of dots for each page of a
/// given `Carousel`. The dot representing the carousel's active page is
/// larger and more opaque than the others, the transition to the active and
/// inactive state is gradual to match the carousel's position.
///
/// See also `CarouselIndicatorLines`.
///
/// ## CSS nodes
///
/// `HdyCarouselIndicatorDots` has a single CSS node with name
/// `carouselindicatordots`.
pub const CarouselIndicatorDots = opaque {
    pub const Parent = gtk.DrawingArea;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = handy.CarouselIndicatorDotsClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `Carousel` the indicator uses.
        pub const carousel = struct {
            pub const name = "carousel";

            pub const Type = ?*handy.Carousel;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyCarouselIndicatorDots`.
    extern fn hdy_carousel_indicator_dots_new() *handy.CarouselIndicatorDots;
    pub const new = hdy_carousel_indicator_dots_new;

    /// Get the `Carousel` the indicator uses.
    extern fn hdy_carousel_indicator_dots_get_carousel(p_self: *CarouselIndicatorDots) ?*handy.Carousel;
    pub const getCarousel = hdy_carousel_indicator_dots_get_carousel;

    /// Sets the `Carousel` to use.
    extern fn hdy_carousel_indicator_dots_set_carousel(p_self: *CarouselIndicatorDots, p_carousel: ?*handy.Carousel) void;
    pub const setCarousel = hdy_carousel_indicator_dots_set_carousel;

    extern fn hdy_carousel_indicator_dots_get_type() usize;
    pub const getGObjectType = hdy_carousel_indicator_dots_get_type;

    extern fn g_object_ref(p_self: *handy.CarouselIndicatorDots) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.CarouselIndicatorDots) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CarouselIndicatorDots, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A lines indicator for `Carousel`.
///
/// The `HdyCarouselIndicatorLines` widget shows a set of lines for each page of
/// a given `Carousel`. The carousel's active page is shown as another line
/// that moves between them to match the carousel's position.
///
/// See also `CarouselIndicatorDots`.
///
/// ## CSS nodes
///
/// `HdyCarouselIndicatorLines` has a single CSS node with name
/// `carouselindicatorlines`.
pub const CarouselIndicatorLines = opaque {
    pub const Parent = gtk.DrawingArea;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = handy.CarouselIndicatorLinesClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The displayed carousel.
        pub const carousel = struct {
            pub const name = "carousel";

            pub const Type = ?*handy.Carousel;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyCarouselIndicatorLines`.
    extern fn hdy_carousel_indicator_lines_new() *handy.CarouselIndicatorLines;
    pub const new = hdy_carousel_indicator_lines_new;

    /// Gets the displayed carousel.
    extern fn hdy_carousel_indicator_lines_get_carousel(p_self: *CarouselIndicatorLines) ?*handy.Carousel;
    pub const getCarousel = hdy_carousel_indicator_lines_get_carousel;

    /// Sets the `Carousel` to use.
    extern fn hdy_carousel_indicator_lines_set_carousel(p_self: *CarouselIndicatorLines, p_carousel: ?*handy.Carousel) void;
    pub const setCarousel = hdy_carousel_indicator_lines_set_carousel;

    extern fn hdy_carousel_indicator_lines_get_type() usize;
    pub const getGObjectType = hdy_carousel_indicator_lines_get_type;

    extern fn g_object_ref(p_self: *handy.CarouselIndicatorLines) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.CarouselIndicatorLines) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CarouselIndicatorLines, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A widget constraining its child to a given size.
///
/// The `HdyClamp` widget constrains the size of the widget it contains to a
/// given maximum size. It will constrain the width if it is horizontal, or the
/// height if it is vertical. The expansion of the child from its minimum to its
/// maximum size is eased out for a smooth transition.
///
/// If the child requires more than the requested maximum size, it will be
/// allocated the minimum size it can fit in instead.
///
/// ## CSS nodes
///
/// `HdyClamp` has a single CSS node with name `clamp`.
///
/// The node will get the style classes `.large` when its child reached its
/// maximum size, `.small` when the clamp allocates its full size to its child,
/// `.medium` in-between, or none if it didn't compute its size yet.
pub const Clamp = opaque {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = handy.ClampClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The maximum size to allocate the children.
        ///
        /// It is the width if the clamp is horizontal, or the height if it is
        /// vertical.
        pub const maximum_size = struct {
            pub const name = "maximum-size";

            pub const Type = c_int;
        };

        /// The size above which the child is clamped.
        ///
        /// Starting from this size, the layout will tighten its grip on the children,
        /// slowly allocating less and less of the available size up to the maximum
        /// allocated size. Below that threshold and below the maximum size, the
        /// children will be allocated all the available size.
        ///
        /// If the threshold is greater than the maximum size to allocate to the
        /// children, they will be allocated the whole size up to the maximum. If the
        /// threshold is lower than the minimum size to allocate to the children, that
        /// size will be used as the tightening threshold.
        ///
        /// Effectively, tightening the grip on a child before it reaches its maximum
        /// size makes transitions to and from the maximum size smoother when resizing.
        pub const tightening_threshold = struct {
            pub const name = "tightening-threshold";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyClamp`.
    extern fn hdy_clamp_new() *handy.Clamp;
    pub const new = hdy_clamp_new;

    /// Gets the maximum size allocated to the children.
    extern fn hdy_clamp_get_maximum_size(p_self: *Clamp) c_int;
    pub const getMaximumSize = hdy_clamp_get_maximum_size;

    /// Gets the size above which the children are clamped.
    extern fn hdy_clamp_get_tightening_threshold(p_self: *Clamp) c_int;
    pub const getTighteningThreshold = hdy_clamp_get_tightening_threshold;

    /// Sets the maximum size allocated to the children.
    extern fn hdy_clamp_set_maximum_size(p_self: *Clamp, p_maximum_size: c_int) void;
    pub const setMaximumSize = hdy_clamp_set_maximum_size;

    /// Sets the size above which the children are clamped.
    extern fn hdy_clamp_set_tightening_threshold(p_self: *Clamp, p_tightening_threshold: c_int) void;
    pub const setTighteningThreshold = hdy_clamp_set_tightening_threshold;

    extern fn hdy_clamp_get_type() usize;
    pub const getGObjectType = hdy_clamp_get_type;

    extern fn g_object_ref(p_self: *handy.Clamp) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.Clamp) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Clamp, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gtk.ListBoxRow` used to choose from a list of items.
///
/// The `HdyComboRow` widget allows the user to choose from a list of valid
/// choices. The row displays the selected choice. When activated, the row
/// displays a popover which allows the user to make a new choice.
///
/// The `ComboRow` uses the model-view pattern; the list of valid choices
/// is specified in the form of a `gio.ListModel`, and the display of the
/// choices can be adapted to the data in the model via widget creation
/// functions.
///
/// `HdyComboRow` is `gtk.ListBoxRow.properties.activatable` if a model is set.
///
/// ## CSS nodes
///
/// `HdyComboRow` has a main CSS node with name `row`.
///
/// Its popover has the node name popover with the `.combo` style class, it
/// contains a `gtk.ScrolledWindow`, which in turn contains a
/// `gtk.ListBox`, both are accessible via their regular nodes.
///
/// A checkmark of node and style class `image.checkmark` in the popover denotes
/// the current item.
pub const ComboRow = extern struct {
    pub const Parent = handy.ActionRow;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Actionable, gtk.Buildable };
    pub const Class = handy.ComboRowClass;
    f_parent_instance: handy.ActionRow,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The index of the selected item in its `gio.ListModel`.
        pub const selected_index = struct {
            pub const name = "selected-index";

            pub const Type = c_int;
        };

        /// Whether to use the current value as the subtitle.
        ///
        /// If you use a custom widget creation function, you will need to give the row
        /// a name conversion closure with `ComboRow.setGetNameFunc`.
        ///
        /// If `TRUE`, you should not access `ActionRow.properties.subtitle`.
        pub const use_subtitle = struct {
            pub const name = "use-subtitle";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyComboRow`.
    extern fn hdy_combo_row_new() *handy.ComboRow;
    pub const new = hdy_combo_row_new;

    /// Binds `model` to `self`.
    ///
    /// If `self` was already bound to a model, that previous binding is destroyed.
    ///
    /// The contents of `self` are cleared and then filled with widgets that represent
    /// items from `model`. `self` is updated whenever `model` changes. If `model` is
    /// `NULL`, `self` is left empty.
    extern fn hdy_combo_row_bind_model(p_self: *ComboRow, p_model: ?*gio.ListModel, p_create_list_widget_func: ?gtk.ListBoxCreateWidgetFunc, p_create_current_widget_func: ?gtk.ListBoxCreateWidgetFunc, p_user_data: ?*anyopaque, p_user_data_free_func: ?glib.DestroyNotify) void;
    pub const bindModel = hdy_combo_row_bind_model;

    /// Binds `model` to `self`.
    ///
    /// If `self` was already bound to a model, that previous binding is destroyed.
    ///
    /// The contents of `self` are cleared and then filled with widgets that represent
    /// items from `model`. `self` is updated whenever `model` changes. If `model` is
    /// `NULL`, `self` is left empty.
    ///
    /// This is more convenient to use than `ComboRow.bindModel` if you want
    /// to represent items of the model with names.
    extern fn hdy_combo_row_bind_name_model(p_self: *ComboRow, p_model: ?*gio.ListModel, p_get_name_func: ?handy.ComboRowGetNameFunc, p_user_data: ?*anyopaque, p_user_data_free_func: ?glib.DestroyNotify) void;
    pub const bindNameModel = hdy_combo_row_bind_name_model;

    /// Gets the model bound to `self`.
    extern fn hdy_combo_row_get_model(p_self: *ComboRow) ?*gio.ListModel;
    pub const getModel = hdy_combo_row_get_model;

    /// Gets the index of the selected item in its `gio.ListModel`.
    extern fn hdy_combo_row_get_selected_index(p_self: *ComboRow) c_int;
    pub const getSelectedIndex = hdy_combo_row_get_selected_index;

    /// Gets whether the current value of `self` should be displayed as its subtitle.
    extern fn hdy_combo_row_get_use_subtitle(p_self: *ComboRow) c_int;
    pub const getUseSubtitle = hdy_combo_row_get_use_subtitle;

    /// Creates a model for `enum_type` and binds it to `self`.
    ///
    /// The items of the model will be `EnumValueObject` objects.
    ///
    /// If `self` was already bound to a model, that previous binding is destroyed.
    ///
    /// The contents of `self` are cleared and then filled with widgets that represent
    /// items from `model`. `self` is updated whenever `model` changes. If `model` is
    /// `NULL`, `self` is left empty.
    ///
    /// This is more convenient to use than `ComboRow.bindNameModel` if you
    /// want to represent values of an enumeration with names.
    ///
    /// See `enumValueRowName`.
    extern fn hdy_combo_row_set_for_enum(p_self: *ComboRow, p_enum_type: usize, p_get_name_func: ?handy.ComboRowGetEnumValueNameFunc, p_user_data: ?*anyopaque, p_user_data_free_func: ?glib.DestroyNotify) void;
    pub const setForEnum = hdy_combo_row_set_for_enum;

    /// Sets a closure to convert items into names.
    ///
    /// See `ComboRow.properties.use_subtitle`.
    extern fn hdy_combo_row_set_get_name_func(p_self: *ComboRow, p_get_name_func: ?handy.ComboRowGetNameFunc, p_user_data: ?*anyopaque, p_user_data_free_func: ?glib.DestroyNotify) void;
    pub const setGetNameFunc = hdy_combo_row_set_get_name_func;

    /// Sets the index of the selected item in its `gio.ListModel`.
    extern fn hdy_combo_row_set_selected_index(p_self: *ComboRow, p_selected_index: c_int) void;
    pub const setSelectedIndex = hdy_combo_row_set_selected_index;

    /// Sets whether the current value of `self` should be displayed as its subtitle.
    ///
    /// If `TRUE`, you should not access `ActionRow.properties.subtitle`.
    extern fn hdy_combo_row_set_use_subtitle(p_self: *ComboRow, p_use_subtitle: c_int) void;
    pub const setUseSubtitle = hdy_combo_row_set_use_subtitle;

    extern fn hdy_combo_row_get_type() usize;
    pub const getGObjectType = hdy_combo_row_get_type;

    extern fn g_object_ref(p_self: *handy.ComboRow) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.ComboRow) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ComboRow, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A swipeable widget showing one of the visible children at a time.
///
/// The `HdyDeck` widget displays one of the visible children, similar to a
/// `gtk.Stack`. The children are strictly ordered and can be navigated
/// using swipe gestures.
///
/// The “over” and “under” stack the children one on top of the other, while the
/// “slide” transition puts the children side by side. While navigating to a
/// child on the side or below can be performed by swiping the current child
/// away, navigating to an upper child requires dragging it from the edge where
/// it resides. This doesn't affect non-dragging swipes.
///
/// The “over” and “under” transitions can draw their shadow on top of the
/// window's transparent areas, like the rounded corners. This is a side-effect
/// of allowing shadows to be drawn on top of OpenGL areas. It can be mitigated
/// by using `Window` or `ApplicationWindow` as they will crop
/// anything drawn beyond the rounded corners.
///
/// ## CSS nodes
///
/// `HdyDeck` has a single CSS node with name `deck`.
pub const Deck = extern struct {
    pub const Parent = gtk.Container;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable, handy.Swipeable };
    pub const Class = handy.DeckClass;
    f_parent_instance: gtk.Container,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether swipe gestures allow switching to the previous child.
        pub const can_swipe_back = struct {
            pub const name = "can-swipe-back";

            pub const Type = c_int;
        };

        /// Whether swipe gestures allow switching to the next child.
        pub const can_swipe_forward = struct {
            pub const name = "can-swipe-forward";

            pub const Type = c_int;
        };

        /// Horizontally homogeneous sizing.
        pub const hhomogeneous = struct {
            pub const name = "hhomogeneous";

            pub const Type = c_int;
        };

        /// Whether or not the size should smoothly change when changing between
        /// differently sized children.
        pub const interpolate_size = struct {
            pub const name = "interpolate-size";

            pub const Type = c_int;
        };

        /// The transition animation duration, in milliseconds.
        pub const transition_duration = struct {
            pub const name = "transition-duration";

            pub const Type = c_uint;
        };

        /// Whether or not the transition is currently running.
        pub const transition_running = struct {
            pub const name = "transition-running";

            pub const Type = c_int;
        };

        /// The type of animation that will be used for transitions between children.
        ///
        /// The transition type can be changed without problems at runtime, so it is
        /// possible to change the animation based on the child that is about to become
        /// current.
        pub const transition_type = struct {
            pub const name = "transition-type";

            pub const Type = handy.DeckTransitionType;
        };

        /// Vertically homogeneous sizing.
        pub const vhomogeneous = struct {
            pub const name = "vhomogeneous";

            pub const Type = c_int;
        };

        /// The widget currently visible.
        ///
        /// The transition is determined by `Deck.properties.transition_type` and
        /// `Deck.properties.transition_duration`. The transition can be cancelled by the
        /// user, in which case visible child will change back to the previously
        /// visible child.
        pub const visible_child = struct {
            pub const name = "visible-child";

            pub const Type = ?*gtk.Widget;
        };

        /// The name of the widget currently visible.
        pub const visible_child_name = struct {
            pub const name = "visible-child-name";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyDeck`.
    extern fn hdy_deck_new() *handy.Deck;
    pub const new = hdy_deck_new;

    /// Finds the previous or next navigatable child.
    ///
    /// Gets the previous or next child. This will be the same widget
    /// `Deck.navigate` will navigate to.
    ///
    /// If there's no child to navigate to, `NULL` will be returned instead.
    extern fn hdy_deck_get_adjacent_child(p_self: *Deck, p_direction: handy.NavigationDirection) ?*gtk.Widget;
    pub const getAdjacentChild = hdy_deck_get_adjacent_child;

    /// Gets whether swipe gestures for navigating backward are enabled.
    extern fn hdy_deck_get_can_swipe_back(p_self: *Deck) c_int;
    pub const getCanSwipeBack = hdy_deck_get_can_swipe_back;

    /// Gets whether swipe gestures for navigating forward enabled.
    extern fn hdy_deck_get_can_swipe_forward(p_self: *Deck) c_int;
    pub const getCanSwipeForward = hdy_deck_get_can_swipe_forward;

    /// Finds the child of `self` with `name`.
    ///
    /// Returns `NULL` if there is no child with this name.
    extern fn hdy_deck_get_child_by_name(p_self: *Deck, p_name: [*:0]const u8) ?*gtk.Widget;
    pub const getChildByName = hdy_deck_get_child_by_name;

    /// Gets whether `self` is homogeneous for the given orientation.
    extern fn hdy_deck_get_homogeneous(p_self: *Deck, p_orientation: gtk.Orientation) c_int;
    pub const getHomogeneous = hdy_deck_get_homogeneous;

    /// Gets whether `self` will interpolate its size when changing the visible child.
    extern fn hdy_deck_get_interpolate_size(p_self: *Deck) c_int;
    pub const getInterpolateSize = hdy_deck_get_interpolate_size;

    /// Gets the mode transition animation duration for `self`.
    extern fn hdy_deck_get_transition_duration(p_self: *Deck) c_uint;
    pub const getTransitionDuration = hdy_deck_get_transition_duration;

    /// Gets whether a transition is currently running for `self`.
    extern fn hdy_deck_get_transition_running(p_self: *Deck) c_int;
    pub const getTransitionRunning = hdy_deck_get_transition_running;

    /// Gets the type of animation used for transitions between children.
    extern fn hdy_deck_get_transition_type(p_self: *Deck) handy.DeckTransitionType;
    pub const getTransitionType = hdy_deck_get_transition_type;

    /// Gets the visible child widget.
    extern fn hdy_deck_get_visible_child(p_self: *Deck) *gtk.Widget;
    pub const getVisibleChild = hdy_deck_get_visible_child;

    /// Gets the name of the currently visible child widget.
    extern fn hdy_deck_get_visible_child_name(p_self: *Deck) [*:0]const u8;
    pub const getVisibleChildName = hdy_deck_get_visible_child_name;

    /// Inserts `child` in the position after `sibling` in the list of children.
    ///
    /// If `sibling` is `NULL`, inserts `child` at the first position.
    extern fn hdy_deck_insert_child_after(p_self: *Deck, p_child: *gtk.Widget, p_sibling: ?*gtk.Widget) void;
    pub const insertChildAfter = hdy_deck_insert_child_after;

    /// Navigates to the previous or next child.
    ///
    /// The switch is similar to performing a swipe gesture to go in `direction`.
    extern fn hdy_deck_navigate(p_self: *Deck, p_direction: handy.NavigationDirection) c_int;
    pub const navigate = hdy_deck_navigate;

    /// Inserts `child` at the first position in `self`.
    extern fn hdy_deck_prepend(p_self: *Deck, p_child: *gtk.Widget) void;
    pub const prepend = hdy_deck_prepend;

    /// Moves `child` to the position after `sibling` in the list of children.
    ///
    /// If `sibling` is `NULL`, move `child` to the first position.
    extern fn hdy_deck_reorder_child_after(p_self: *Deck, p_child: *gtk.Widget, p_sibling: ?*gtk.Widget) void;
    pub const reorderChildAfter = hdy_deck_reorder_child_after;

    /// Sets whether swipe gestures for navigating backward are enabled.
    extern fn hdy_deck_set_can_swipe_back(p_self: *Deck, p_can_swipe_back: c_int) void;
    pub const setCanSwipeBack = hdy_deck_set_can_swipe_back;

    /// Sets whether swipe gestures for navigating forward are enabled.
    extern fn hdy_deck_set_can_swipe_forward(p_self: *Deck, p_can_swipe_forward: c_int) void;
    pub const setCanSwipeForward = hdy_deck_set_can_swipe_forward;

    /// Sets whether `self` is homogeneous for a given orientation.
    ///
    /// If set to `FALSE`, different children can have different size along the
    /// opposite orientation.
    extern fn hdy_deck_set_homogeneous(p_self: *Deck, p_orientation: gtk.Orientation, p_homogeneous: c_int) void;
    pub const setHomogeneous = hdy_deck_set_homogeneous;

    /// Sets whether `self` will interpolate its size when changing the visible child.
    ///
    /// `self` will interpolate its size between the current one and the one it'll
    /// take after changing the visible child, according to the set transition
    /// duration.
    extern fn hdy_deck_set_interpolate_size(p_self: *Deck, p_interpolate_size: c_int) void;
    pub const setInterpolateSize = hdy_deck_set_interpolate_size;

    /// Sets the mode transition animation duration for `self`.
    extern fn hdy_deck_set_transition_duration(p_self: *Deck, p_duration: c_uint) void;
    pub const setTransitionDuration = hdy_deck_set_transition_duration;

    /// Sets the type of animation used for transitions between children.
    ///
    /// The transition type can be changed without problems at runtime, so it is
    /// possible to change the animation based on the child that is about to become
    /// current.
    extern fn hdy_deck_set_transition_type(p_self: *Deck, p_transition: handy.DeckTransitionType) void;
    pub const setTransitionType = hdy_deck_set_transition_type;

    /// Sets the currently visible widget.
    extern fn hdy_deck_set_visible_child(p_self: *Deck, p_visible_child: *gtk.Widget) void;
    pub const setVisibleChild = hdy_deck_set_visible_child;

    /// Makes the child with the name `name` visible.
    ///
    /// See `Deck.setVisibleChild` for more details.
    extern fn hdy_deck_set_visible_child_name(p_self: *Deck, p_name: [*:0]const u8) void;
    pub const setVisibleChildName = hdy_deck_set_visible_child_name;

    extern fn hdy_deck_get_type() usize;
    pub const getGObjectType = hdy_deck_get_type;

    extern fn g_object_ref(p_self: *handy.Deck) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.Deck) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Deck, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An object representing an `gobject.EnumValue`.
///
/// The `HdyEnumValueObject` object represents a `gobject.EnumValue`,
/// allowing it to be used with `gio.ListModel`.
pub const EnumValueObject = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = handy.EnumValueObjectClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `HdyEnumValueObject`.
    extern fn hdy_enum_value_object_new(p_enum_value: *gobject.EnumValue) *handy.EnumValueObject;
    pub const new = hdy_enum_value_object_new;

    /// Gets the name of `self`.
    extern fn hdy_enum_value_object_get_name(p_self: *EnumValueObject) [*:0]const u8;
    pub const getName = hdy_enum_value_object_get_name;

    /// Gets the nick of `self`.
    extern fn hdy_enum_value_object_get_nick(p_self: *EnumValueObject) [*:0]const u8;
    pub const getNick = hdy_enum_value_object_get_nick;

    /// Gets the value of `self`.
    extern fn hdy_enum_value_object_get_value(p_self: *EnumValueObject) c_int;
    pub const getValue = hdy_enum_value_object_get_value;

    extern fn hdy_enum_value_object_get_type() usize;
    pub const getGObjectType = hdy_enum_value_object_get_type;

    extern fn g_object_ref(p_self: *handy.EnumValueObject) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.EnumValueObject) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EnumValueObject, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gtk.ListBoxRow` used to reveal widgets.
///
/// The `HdyExpanderRow` widget allows the user to reveal or hide widgets below
/// it. It also allows the user to enable the expansion of the row, allowing to
/// disable all that the row contains.
///
/// ## HdyExpanderRow as GtkBuildable
///
/// The `HdyExpanderRow` implementation of the `gtk.Buildable` interface
/// supports adding a child as an action widget by specifying “action” as the
/// “type” attribute of a <child> element.
///
/// It also supports adding it as a prefix widget by specifying “prefix” as the
/// “type” attribute of a <child> element.
///
/// ## CSS nodes
///
/// `HdyExpanderRow` has a main CSS node with name `row`, and the `.expander`
/// style class. It has the `.empty` style class when it contains no children.
///
/// It contains the subnodes `row.header` for its main embedded row,
/// `list.nested` for the list it can expand, and `image.expander-row-arrow` for
/// its arrow.
///
/// When expanded, `HdyExpanderRow` will add the
/// `.checked-expander-row-previous-sibling` style class to its previous sibling,
/// and remove it when retracted.
pub const ExpanderRow = extern struct {
    pub const Parent = handy.PreferencesRow;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Actionable, gtk.Buildable };
    pub const Class = handy.ExpanderRowClass;
    f_parent_instance: handy.PreferencesRow,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether expansion is enabled.
        pub const enable_expansion = struct {
            pub const name = "enable-expansion";

            pub const Type = c_int;
        };

        /// Whether the row is expanded.
        pub const expanded = struct {
            pub const name = "expanded";

            pub const Type = c_int;
        };

        /// The icon name for this row.
        pub const icon_name = struct {
            pub const name = "icon-name";

            pub const Type = ?[*:0]u8;
        };

        /// Whether the switch enabling the expansion is visible.
        pub const show_enable_switch = struct {
            pub const name = "show-enable-switch";

            pub const Type = c_int;
        };

        /// The subtitle for this row.
        pub const subtitle = struct {
            pub const name = "subtitle";

            pub const Type = ?[*:0]u8;
        };

        /// Whether an embedded underline in the title or subtitle labels indicates a
        /// mnemonic.
        pub const use_underline = struct {
            pub const name = "use-underline";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyExpanderRow`.
    extern fn hdy_expander_row_new() *handy.ExpanderRow;
    pub const new = hdy_expander_row_new;

    /// Adds an action widget to `self`.
    extern fn hdy_expander_row_add_action(p_self: *ExpanderRow, p_widget: *gtk.Widget) void;
    pub const addAction = hdy_expander_row_add_action;

    /// Adds a prefix widget to `self`.
    extern fn hdy_expander_row_add_prefix(p_self: *ExpanderRow, p_widget: *gtk.Widget) void;
    pub const addPrefix = hdy_expander_row_add_prefix;

    /// Gets whether the expansion of `self` is enabled.
    extern fn hdy_expander_row_get_enable_expansion(p_self: *ExpanderRow) c_int;
    pub const getEnableExpansion = hdy_expander_row_get_enable_expansion;

    extern fn hdy_expander_row_get_expanded(p_self: *ExpanderRow) c_int;
    pub const getExpanded = hdy_expander_row_get_expanded;

    /// Gets the icon name for `self`.
    extern fn hdy_expander_row_get_icon_name(p_self: *ExpanderRow) [*:0]const u8;
    pub const getIconName = hdy_expander_row_get_icon_name;

    /// Gets whether the switch enabling the expansion of `self` is visible.
    extern fn hdy_expander_row_get_show_enable_switch(p_self: *ExpanderRow) c_int;
    pub const getShowEnableSwitch = hdy_expander_row_get_show_enable_switch;

    /// Gets the subtitle for `self`.
    extern fn hdy_expander_row_get_subtitle(p_self: *ExpanderRow) ?[*:0]const u8;
    pub const getSubtitle = hdy_expander_row_get_subtitle;

    /// Gets whether an embedded underline in the title or subtitle labels indicates
    /// a mnemonic.
    extern fn hdy_expander_row_get_use_underline(p_self: *ExpanderRow) c_int;
    pub const getUseUnderline = hdy_expander_row_get_use_underline;

    /// Sets whether the expansion of `self` is enabled.
    extern fn hdy_expander_row_set_enable_expansion(p_self: *ExpanderRow, p_enable_expansion: c_int) void;
    pub const setEnableExpansion = hdy_expander_row_set_enable_expansion;

    extern fn hdy_expander_row_set_expanded(p_self: *ExpanderRow, p_expanded: c_int) void;
    pub const setExpanded = hdy_expander_row_set_expanded;

    /// Sets the icon name for `self`.
    extern fn hdy_expander_row_set_icon_name(p_self: *ExpanderRow, p_icon_name: [*:0]const u8) void;
    pub const setIconName = hdy_expander_row_set_icon_name;

    /// Sets whether the switch enabling the expansion of `self` is visible.
    extern fn hdy_expander_row_set_show_enable_switch(p_self: *ExpanderRow, p_show_enable_switch: c_int) void;
    pub const setShowEnableSwitch = hdy_expander_row_set_show_enable_switch;

    /// Sets the subtitle for `self`.
    extern fn hdy_expander_row_set_subtitle(p_self: *ExpanderRow, p_subtitle: ?[*:0]const u8) void;
    pub const setSubtitle = hdy_expander_row_set_subtitle;

    /// Sets whether an embedded underline in the title or subtitle labels indicates
    /// a mnemonic.
    extern fn hdy_expander_row_set_use_underline(p_self: *ExpanderRow, p_use_underline: c_int) void;
    pub const setUseUnderline = hdy_expander_row_set_use_underline;

    extern fn hdy_expander_row_get_type() usize;
    pub const getGObjectType = hdy_expander_row_get_type;

    extern fn g_object_ref(p_self: *handy.ExpanderRow) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.ExpanderRow) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ExpanderRow, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An adaptive container acting like a box or an overlay.
///
/// The `HdyFlap` widget can display its children like a `gtk.Box` does or
/// like a `gtk.Overlay` does, according to the
/// `Flap.properties.fold_policy` value.
///
/// `HdyFlap` has at most three children: `Flap.properties.content`,
/// `Flap.properties.flap` and `Flap.properties.separator`. Content is the primary
/// child, flap is displayed next to it when unfolded, or overlays it when
/// folded. Flap can be shown or hidden by changing the
/// `Flap.properties.reveal_flap` value, as well as via swipe gestures if
/// `Flap.properties.swipe_to_open` and/or `Flap.properties.swipe_to_close` are set
/// to `TRUE`.
///
/// Optionally, a separator can be provided, which would be displayed between the
/// content and the flap when there's no shadow to separate them, depending on
/// the transition type.
///
/// `Flap.properties.flap` is transparent by default; add the `.background` style
/// class to it if this is unwanted.
///
/// If `Flap.properties.modal` is set to `TRUE`, content becomes completely
/// inaccessible when the flap is revealed while folded.
///
/// The position of the flap and separator children relative to the content is
/// determined by orientation, as well as the `Flap.properties.flap_position`
/// value.
///
/// Folding the flap will automatically hide the flap widget, and unfolding it
/// will automatically reveal it. If this behavior is not desired, the
/// `Flap.properties.locked` property can be used to override it.
///
/// Common use cases include sidebars, header bars that need to be able to
/// overlap the window content (for example, in fullscreen mode) and bottom
/// sheets.
///
/// ## HdyFlap as GtkBuildable
///
/// The `HdyFlap` implementation of the `gtk.Buildable` interface supports
/// setting the flap child by specifying “flap” as the “type” attribute of a
/// <child> element, and separator by specifying “separator”. Specifying
/// “content” child type or omitting it results in setting the content child.
///
/// ## CSS nodes
///
/// `HdyFlap` has a single CSS node with name `flap`. The node will get the style
/// classes `.folded` when it is folded, and `.unfolded` when it's not.
pub const Flap = opaque {
    pub const Parent = gtk.Container;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable, handy.Swipeable };
    pub const Class = handy.FlapClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The content widget.
        ///
        /// It's always displayed when unfolded, and partially visible
        /// when folded.
        pub const content = struct {
            pub const name = "content";

            pub const Type = ?*gtk.Widget;
        };

        /// The flap widget.
        ///
        /// It's only visible when `Flap.properties.reveal_progress` is greater than 0.
        pub const flap = struct {
            pub const name = "flap";

            pub const Type = ?*gtk.Widget;
        };

        /// The flap position.
        ///
        /// If `GTK_PACK_START`, the flap is displayed before the content, if
        /// `GTK_PACK_END`, it's displayed after the content.
        pub const flap_position = struct {
            pub const name = "flap-position";

            pub const Type = gtk.PackType;
        };

        /// The fold transition animation duration, in milliseconds.
        pub const fold_duration = struct {
            pub const name = "fold-duration";

            pub const Type = c_uint;
        };

        /// The current fold policy.
        ///
        /// See `FlapFoldPolicy` for available policies.
        pub const fold_policy = struct {
            pub const name = "fold-policy";

            pub const Type = handy.FlapFoldPolicy;
        };

        /// Whether the flap is currently folded.
        ///
        /// See `Flap.properties.fold_policy`.
        pub const folded = struct {
            pub const name = "folded";

            pub const Type = c_int;
        };

        /// Whether the flap is locked.
        ///
        /// If `FALSE`, folding when the flap is revealed automatically closes it, and
        /// unfolding it when the flap is not revealed opens it. If `TRUE`,
        /// `Flap.properties.reveal_flap` value never changes on its own.
        pub const locked = struct {
            pub const name = "locked";

            pub const Type = c_int;
        };

        /// Whether the flap is modal.
        ///
        /// If `TRUE`, clicking the content widget while flap is revealed, as well as
        /// pressing the <kbd>Esc</kbd> key, will close the flap. If `FALSE`, clicks
        /// are passed through to the content widget.
        pub const modal = struct {
            pub const name = "modal";

            pub const Type = c_int;
        };

        /// The reveal transition animation duration, in milliseconds.
        pub const reveal_duration = struct {
            pub const name = "reveal-duration";

            pub const Type = c_uint;
        };

        /// Whether the flap widget is revealed.
        pub const reveal_flap = struct {
            pub const name = "reveal-flap";

            pub const Type = c_int;
        };

        /// The current reveal transition progress.
        ///
        /// 0 means fully hidden, 1 means fully revealed. See
        /// `Flap.properties.reveal_flap`.
        pub const reveal_progress = struct {
            pub const name = "reveal-progress";

            pub const Type = f64;
        };

        /// The separator widget.
        ///
        /// It's displayed between content and flap when there's no shadow to display.
        /// When exactly it's visible depends on the `Flap.properties.transition_type`
        /// value. If `NULL`, no separator will be used.
        pub const separator = struct {
            pub const name = "separator";

            pub const Type = ?*gtk.Widget;
        };

        /// Whether the flap can be closed with a swipe gesture.
        ///
        /// The area that can be swiped depends on the `Flap.properties.transition_type`
        /// value.
        pub const swipe_to_close = struct {
            pub const name = "swipe-to-close";

            pub const Type = c_int;
        };

        /// Whether the flap can be opened with a swipe gesture.
        ///
        /// The area that can be swiped depends on the `Flap.properties.transition_type`
        /// value.
        pub const swipe_to_open = struct {
            pub const name = "swipe-to-open";

            pub const Type = c_int;
        };

        /// the type of animation used for reveal and fold transitions.
        ///
        /// `Flap.properties.flap` is transparent by default, which means the content
        /// will be seen through it with `HDY_FLAP_TRANSITION_TYPE_OVER` transitions;
        /// add the `.background` style class to it if this is unwanted.
        pub const transition_type = struct {
            pub const name = "transition-type";

            pub const Type = handy.FlapTransitionType;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyFlap`.
    extern fn hdy_flap_new() *handy.Flap;
    pub const new = hdy_flap_new;

    /// Gets the content widget for `self`
    extern fn hdy_flap_get_content(p_self: *Flap) ?*gtk.Widget;
    pub const getContent = hdy_flap_get_content;

    /// Gets the flap widget for `self`
    extern fn hdy_flap_get_flap(p_self: *Flap) ?*gtk.Widget;
    pub const getFlap = hdy_flap_get_flap;

    /// Gets the flap position for `self`.
    extern fn hdy_flap_get_flap_position(p_self: *Flap) gtk.PackType;
    pub const getFlapPosition = hdy_flap_get_flap_position;

    /// Gets the amount of time that fold transitions will take.
    extern fn hdy_flap_get_fold_duration(p_self: *Flap) c_uint;
    pub const getFoldDuration = hdy_flap_get_fold_duration;

    /// Gets the current fold policy of `self`.
    extern fn hdy_flap_get_fold_policy(p_self: *Flap) handy.FlapFoldPolicy;
    pub const getFoldPolicy = hdy_flap_get_fold_policy;

    /// Gets whether `self` is currently folded.
    extern fn hdy_flap_get_folded(p_self: *Flap) c_int;
    pub const getFolded = hdy_flap_get_folded;

    /// Gets whether `self` is locked.
    extern fn hdy_flap_get_locked(p_self: *Flap) c_int;
    pub const getLocked = hdy_flap_get_locked;

    /// Gets whether the `self` is modal.
    extern fn hdy_flap_get_modal(p_self: *Flap) c_int;
    pub const getModal = hdy_flap_get_modal;

    /// Gets the amount of time that reveal transitions will take.
    extern fn hdy_flap_get_reveal_duration(p_self: *Flap) c_uint;
    pub const getRevealDuration = hdy_flap_get_reveal_duration;

    /// Gets whether the flap widget is revealed for `self`.
    extern fn hdy_flap_get_reveal_flap(p_self: *Flap) c_int;
    pub const getRevealFlap = hdy_flap_get_reveal_flap;

    /// Gets the current reveal transition progress for `self`.
    extern fn hdy_flap_get_reveal_progress(p_self: *Flap) f64;
    pub const getRevealProgress = hdy_flap_get_reveal_progress;

    /// Gets the separator widget for `self`.
    extern fn hdy_flap_get_separator(p_self: *Flap) ?*gtk.Widget;
    pub const getSeparator = hdy_flap_get_separator;

    /// Gets whether `self` can be closed with a swipe gesture.
    extern fn hdy_flap_get_swipe_to_close(p_self: *Flap) c_int;
    pub const getSwipeToClose = hdy_flap_get_swipe_to_close;

    /// Gets whether `self` can be opened with a swipe gesture.
    extern fn hdy_flap_get_swipe_to_open(p_self: *Flap) c_int;
    pub const getSwipeToOpen = hdy_flap_get_swipe_to_open;

    /// Gets the type of animation used for reveal and fold transitions in `self`.
    extern fn hdy_flap_get_transition_type(p_self: *Flap) handy.FlapTransitionType;
    pub const getTransitionType = hdy_flap_get_transition_type;

    /// Sets the content widget for `self`.
    ///
    /// It is always displayed when unfolded, and partially visible when folded.
    extern fn hdy_flap_set_content(p_self: *Flap, p_content: ?*gtk.Widget) void;
    pub const setContent = hdy_flap_set_content;

    /// Sets the flap widget for `self`.
    extern fn hdy_flap_set_flap(p_self: *Flap, p_flap: ?*gtk.Widget) void;
    pub const setFlap = hdy_flap_set_flap;

    /// Sets the flap position for `self`.
    extern fn hdy_flap_set_flap_position(p_self: *Flap, p_position: gtk.PackType) void;
    pub const setFlapPosition = hdy_flap_set_flap_position;

    /// Sets the duration that fold transitions will take.
    extern fn hdy_flap_set_fold_duration(p_self: *Flap, p_duration: c_uint) void;
    pub const setFoldDuration = hdy_flap_set_fold_duration;

    /// Sets the current fold policy for `self`.
    extern fn hdy_flap_set_fold_policy(p_self: *Flap, p_policy: handy.FlapFoldPolicy) void;
    pub const setFoldPolicy = hdy_flap_set_fold_policy;

    /// Sets whether `self` is locked.
    ///
    /// If `FALSE`, folding `self` when the flap is revealed automatically closes it,
    /// and unfolding it when the flap is not revealed opens it. If `TRUE`,
    /// `Flap.properties.reveal_flap` value never changes on its own.
    extern fn hdy_flap_set_locked(p_self: *Flap, p_locked: c_int) void;
    pub const setLocked = hdy_flap_set_locked;

    /// Sets whether the `self` can be closed with a click.
    ///
    /// If `modal` is `TRUE`, clicking the content widget while flap is revealed, or
    /// pressing the <kbd>Esc</kbd> key, will close the flap. If `FALSE`, clicks are
    /// passed through to the content widget.
    extern fn hdy_flap_set_modal(p_self: *Flap, p_modal: c_int) void;
    pub const setModal = hdy_flap_set_modal;

    /// Sets the duration that reveal transitions in `self` will take.
    extern fn hdy_flap_set_reveal_duration(p_self: *Flap, p_duration: c_uint) void;
    pub const setRevealDuration = hdy_flap_set_reveal_duration;

    /// Sets whether the flap widget is revealed for `self`.
    extern fn hdy_flap_set_reveal_flap(p_self: *Flap, p_reveal_flap: c_int) void;
    pub const setRevealFlap = hdy_flap_set_reveal_flap;

    /// Sets the separator widget for `self`.
    extern fn hdy_flap_set_separator(p_self: *Flap, p_separator: ?*gtk.Widget) void;
    pub const setSeparator = hdy_flap_set_separator;

    /// Sets whether `self` can be closed with a swipe gesture.
    ///
    /// The area that can be swiped depends on the `Flap.properties.transition_type` value.
    extern fn hdy_flap_set_swipe_to_close(p_self: *Flap, p_swipe_to_close: c_int) void;
    pub const setSwipeToClose = hdy_flap_set_swipe_to_close;

    /// Sets whether `self` can be opened with a swipe gesture.
    ///
    /// The area that can be swiped depends on the `Flap.properties.transition_type`
    /// value.
    extern fn hdy_flap_set_swipe_to_open(p_self: *Flap, p_swipe_to_open: c_int) void;
    pub const setSwipeToOpen = hdy_flap_set_swipe_to_open;

    /// Sets the type of animation used for reveal and fold transitions in `self`.
    extern fn hdy_flap_set_transition_type(p_self: *Flap, p_transition_type: handy.FlapTransitionType) void;
    pub const setTransitionType = hdy_flap_set_transition_type;

    extern fn hdy_flap_get_type() usize;
    pub const getGObjectType = hdy_flap_get_type;

    extern fn g_object_ref(p_self: *handy.Flap) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.Flap) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Flap, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A title bar widget.
///
/// `HdyHeaderBar` is similar to `gtk.HeaderBar` but is designed to fix
/// some of its shortcomings for adaptive applications.
///
/// `HdyHeaderBar` doesn't force the custom title widget to be vertically
/// centered, hence allowing it to fill up the whole height, which is e.g. needed
/// for `ViewSwitcher`.
///
/// When used in a mobile dialog, `HdyHeaderBar` will replace its window
/// decorations by a back button allowing to close it. It doesn't have to be its
/// direct child and you can use any complex contraption you like as the dialog's
/// titlebar.
///
/// `HdyHeaderBar` can be used in window's content area rather than titlebar, and
/// will still be draggable and will handle right click, middle click and double
/// click as expected from a titlebar. This is particularly useful with
/// `Window` or `ApplicationWindow`.
///
/// ## CSS nodes
///
/// `HdyHeaderBar` has a single CSS node with name `headerbar`.
pub const HeaderBar = extern struct {
    pub const Parent = gtk.Container;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.HeaderBarClass;
    f_parent_instance: gtk.Container,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The policy for aligning the center widget.
        pub const centering_policy = struct {
            pub const name = "centering-policy";

            pub const Type = handy.CenteringPolicy;
        };

        /// Custom title widget to display.
        pub const custom_title = struct {
            pub const name = "custom-title";

            pub const Type = ?*gtk.Widget;
        };

        /// The decoration layout for buttons.
        ///
        /// If this property is not set, the
        /// `gtk.Settings.properties.gtk_decoration_layout` setting is used.
        ///
        /// There can be valid reasons for overriding the setting, such as a header bar
        /// design that does not allow for buttons to take room on the right, or only
        /// offers room for a single close button. Split header bars are another example
        /// for overriding the setting.
        ///
        /// The format of the string is button names, separated by commas. A colon
        /// separates the buttons that should appear on the start from those on the
        /// end. Recognized button names are minimize, maximize, close, icon (the
        /// window icon) and menu (a menu button for the fallback app menu).
        ///
        /// For example, “menu:minimize,maximize,close” specifies a menu on the left, and
        /// minimize, maximize and close buttons on the right.
        pub const decoration_layout = struct {
            pub const name = "decoration-layout";

            pub const Type = ?[*:0]u8;
        };

        /// Whether `HeaderBar.properties.decoration_layout` is set.
        pub const decoration_layout_set = struct {
            pub const name = "decoration-layout-set";

            pub const Type = c_int;
        };

        /// Whether to reserve space for a subtitle, even if none is currently set.
        pub const has_subtitle = struct {
            pub const name = "has-subtitle";

            pub const Type = c_int;
        };

        /// Whether the size should smoothly change when changing between children.
        ///
        /// If `TRUE`, the header bar will interpolate its size between the one of the
        /// previous visible child and the one of the new visible child, according to
        /// the set transition duration and the orientation, e.g. if the orientation is
        /// horizontal, it will interpolate the its height.
        pub const interpolate_size = struct {
            pub const name = "interpolate-size";

            pub const Type = c_int;
        };

        /// Whether to show window decorations.
        ///
        /// Which buttons are actually shown and where is determined by the
        /// `HeaderBar.properties.decoration_layout` property, and by the state of the
        /// window (e.g. a close button will not be shown if the window can't be
        /// closed).
        pub const show_close_button = struct {
            pub const name = "show-close-button";

            pub const Type = c_int;
        };

        /// The amount of space between children.
        pub const spacing = struct {
            pub const name = "spacing";

            pub const Type = c_int;
        };

        /// The subtitle to display.
        pub const subtitle = struct {
            pub const name = "subtitle";

            pub const Type = ?[*:0]u8;
        };

        /// The title to display.
        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };

        /// The transition duration, in milliseconds.
        pub const transition_duration = struct {
            pub const name = "transition-duration";

            pub const Type = c_uint;
        };

        /// Whether or not the transition is currently running.
        pub const transition_running = struct {
            pub const name = "transition-running";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyHeaderBar`.
    extern fn hdy_header_bar_new() *handy.HeaderBar;
    pub const new = hdy_header_bar_new;

    /// Gets the policy `self` follows to horizontally align its center widget.
    extern fn hdy_header_bar_get_centering_policy(p_self: *HeaderBar) handy.CenteringPolicy;
    pub const getCenteringPolicy = hdy_header_bar_get_centering_policy;

    /// Retrieves the custom title widget of the header.
    extern fn hdy_header_bar_get_custom_title(p_self: *HeaderBar) ?*gtk.Widget;
    pub const getCustomTitle = hdy_header_bar_get_custom_title;

    /// Gets the decoration layout.
    extern fn hdy_header_bar_get_decoration_layout(p_self: *HeaderBar) [*:0]const u8;
    pub const getDecorationLayout = hdy_header_bar_get_decoration_layout;

    /// Gets whether space is reserved for a subtitle, regardless if one is currently
    /// set or not.
    extern fn hdy_header_bar_get_has_subtitle(p_self: *HeaderBar) c_int;
    pub const getHasSubtitle = hdy_header_bar_get_has_subtitle;

    /// Gets whether `self` should interpolate its size on visible child change.
    extern fn hdy_header_bar_get_interpolate_size(p_self: *HeaderBar) c_int;
    pub const getInterpolateSize = hdy_header_bar_get_interpolate_size;

    /// Gets whether this header bar shows the standard window decorations.
    extern fn hdy_header_bar_get_show_close_button(p_self: *HeaderBar) c_int;
    pub const getShowCloseButton = hdy_header_bar_get_show_close_button;

    /// Gets the subtitle of the header.
    extern fn hdy_header_bar_get_subtitle(p_self: *HeaderBar) ?[*:0]const u8;
    pub const getSubtitle = hdy_header_bar_get_subtitle;

    /// Retrieves the title of the header.
    extern fn hdy_header_bar_get_title(p_self: *HeaderBar) ?[*:0]const u8;
    pub const getTitle = hdy_header_bar_get_title;

    /// Gets the amount of time that transitions between pages will take.
    extern fn hdy_header_bar_get_transition_duration(p_self: *HeaderBar) c_uint;
    pub const getTransitionDuration = hdy_header_bar_get_transition_duration;

    /// Gets whether the `self` is currently in a transition from one page to another.
    extern fn hdy_header_bar_get_transition_running(p_self: *HeaderBar) c_int;
    pub const getTransitionRunning = hdy_header_bar_get_transition_running;

    /// Adds `child` to `self`, packed with reference to the end of the `self`.
    extern fn hdy_header_bar_pack_end(p_self: *HeaderBar, p_child: *gtk.Widget) void;
    pub const packEnd = hdy_header_bar_pack_end;

    /// Adds `child` to `self`, packed with reference to the start of the `self`.
    extern fn hdy_header_bar_pack_start(p_self: *HeaderBar, p_child: *gtk.Widget) void;
    pub const packStart = hdy_header_bar_pack_start;

    /// Sets the policy `self` must follow to horizontally align its center widget.
    extern fn hdy_header_bar_set_centering_policy(p_self: *HeaderBar, p_centering_policy: handy.CenteringPolicy) void;
    pub const setCenteringPolicy = hdy_header_bar_set_centering_policy;

    /// Sets a custom title for the header bar.
    ///
    /// The title should help a user identify the current view. This supersedes any
    /// title set by `HeaderBar.setTitle` or `HeaderBar.setSubtitle`.
    /// To achieve the same style as the builtin title and subtitle, use the `.title`
    /// and `.subtitle` style classes.
    ///
    /// You should set the custom title to `NULL`, for the header title label to be
    /// visible again.
    extern fn hdy_header_bar_set_custom_title(p_self: *HeaderBar, p_title_widget: ?*gtk.Widget) void;
    pub const setCustomTitle = hdy_header_bar_set_custom_title;

    /// Sets the decoration layout for this header bar.
    extern fn hdy_header_bar_set_decoration_layout(p_self: *HeaderBar, p_layout: ?[*:0]const u8) void;
    pub const setDecorationLayout = hdy_header_bar_set_decoration_layout;

    /// Sets whether space is reserved for a subtitle, even if none is currently set.
    extern fn hdy_header_bar_set_has_subtitle(p_self: *HeaderBar, p_setting: c_int) void;
    pub const setHasSubtitle = hdy_header_bar_set_has_subtitle;

    /// Sets whether `self` should interpolate its size on visible child change.
    extern fn hdy_header_bar_set_interpolate_size(p_self: *HeaderBar, p_interpolate_size: c_int) void;
    pub const setInterpolateSize = hdy_header_bar_set_interpolate_size;

    /// Sets whether this header bar shows the standard window decorations.
    extern fn hdy_header_bar_set_show_close_button(p_self: *HeaderBar, p_setting: c_int) void;
    pub const setShowCloseButton = hdy_header_bar_set_show_close_button;

    /// Sets the subtitle of the header bar.
    ///
    /// The title should give a user an additional detail to help them identify the
    /// current view.
    ///
    /// Note that `HeaderBar` by default reserves room for the subtitle, even
    /// if none is currently set. If this is not desired, set the
    /// `HeaderBar.properties.has_subtitle` property to `FALSE`.
    extern fn hdy_header_bar_set_subtitle(p_self: *HeaderBar, p_subtitle: ?[*:0]const u8) void;
    pub const setSubtitle = hdy_header_bar_set_subtitle;

    /// Sets the title of the `HeaderBar`.
    ///
    /// The title should help a user identify the current view. A good title should
    /// not include the application name.
    extern fn hdy_header_bar_set_title(p_self: *HeaderBar, p_title: ?[*:0]const u8) void;
    pub const setTitle = hdy_header_bar_set_title;

    /// Sets the duration that transitions between pages will take.
    extern fn hdy_header_bar_set_transition_duration(p_self: *HeaderBar, p_duration: c_uint) void;
    pub const setTransitionDuration = hdy_header_bar_set_transition_duration;

    extern fn hdy_header_bar_get_type() usize;
    pub const getGObjectType = hdy_header_bar_get_type;

    extern fn g_object_ref(p_self: *handy.HeaderBar) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.HeaderBar) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HeaderBar, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An object handling composite title bars.
///
/// The `HdyHeaderGroup` object handles the header bars of a composite title bar.
/// It splits the window decoration across the header bars, giving the left side
/// of the decorations to the leftmost header bar, and the right side of the
/// decorations to the rightmost header bar. See
/// `HeaderBar.setDecorationLayout`.
///
/// The `HeaderGroup.properties.decorate_all` property can be used in conjunction
/// with `Leaflet.properties.folded` when the title bar is split across the pages
/// of a `Leaflet` to automatically display the decorations on all the
/// pages when the leaflet is folded.
///
/// You can nest header groups, which is convenient when you nest leaflets too:
///
/// ```xml
/// <object class="HdyHeaderGroup" id="inner_header_group">
///   <property name="decorate-all" bind-source="inner_leaflet" bind-property="folded" bind-flags="sync-create"/>
///   <headerbars>
///     <headerbar name="inner_header_bar_1"/>
///     <headerbar name="inner_header_bar_2"/>
///   </headerbars>
/// </object>
/// <object class="HdyHeaderGroup" id="outer_header_group">
///   <property name="decorate-all" bind-source="outer_leaflet" bind-property="folded" bind-flags="sync-create"/>
///   <headerbars>
///     <headerbar name="inner_header_group"/>
///     <headerbar name="outer_header_bar"/>
///   </headerbars>
/// </object>
/// ```
pub const HeaderGroup = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gtk.Buildable};
    pub const Class = handy.HeaderGroupClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether the elements of the group should all receive the full decoration.
        ///
        /// This is useful in conjunction with `Leaflet.properties.folded` when the
        /// leaflet contains the header bars of the group, as you want them all to
        /// display the complete decoration when the leaflet is folded.
        pub const decorate_all = struct {
            pub const name = "decorate-all";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// This signal is emitted before updating the decoration layouts.
        pub const update_decoration_layouts = struct {
            pub const name = "update-decoration-layouts";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(HeaderGroup, p_instance))),
                    gobject.signalLookup("update-decoration-layouts", HeaderGroup.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `HdyHeaderGroup`.
    extern fn hdy_header_group_new() *handy.HeaderGroup;
    pub const new = hdy_header_group_new;

    /// Adds `header_bar` to `self`.
    ///
    /// When the widget is destroyed or no longer referenced elsewhere, it will be
    /// removed from the header group.
    extern fn hdy_header_group_add_gtk_header_bar(p_self: *HeaderGroup, p_header_bar: *gtk.HeaderBar) void;
    pub const addGtkHeaderBar = hdy_header_group_add_gtk_header_bar;

    /// Adds `header_bar` to `self`.
    ///
    /// When the widget is destroyed or no longer referenced elsewhere, it will be
    /// removed from the header group.
    extern fn hdy_header_group_add_header_bar(p_self: *HeaderGroup, p_header_bar: *handy.HeaderBar) void;
    pub const addHeaderBar = hdy_header_group_add_header_bar;

    /// Adds `header_group` to `self`.
    ///
    /// When the nested group is no longer referenced elsewhere, it will be removed
    /// from the header group.
    extern fn hdy_header_group_add_header_group(p_self: *HeaderGroup, p_header_group: *handy.HeaderGroup) void;
    pub const addHeaderGroup = hdy_header_group_add_header_group;

    /// Returns the list of children associated with `self`.
    extern fn hdy_header_group_get_children(p_self: *HeaderGroup) *glib.SList;
    pub const getChildren = hdy_header_group_get_children;

    /// Gets whether the elements of the group should all receive the full
    /// decoration.
    extern fn hdy_header_group_get_decorate_all(p_self: *HeaderGroup) c_int;
    pub const getDecorateAll = hdy_header_group_get_decorate_all;

    /// Removes `child` from `self`.
    extern fn hdy_header_group_remove_child(p_self: *HeaderGroup, p_child: *handy.HeaderGroupChild) void;
    pub const removeChild = hdy_header_group_remove_child;

    /// Removes `header_bar` from `self`.
    extern fn hdy_header_group_remove_gtk_header_bar(p_self: *HeaderGroup, p_header_bar: *gtk.HeaderBar) void;
    pub const removeGtkHeaderBar = hdy_header_group_remove_gtk_header_bar;

    /// Removes `header_bar` from `self`.
    extern fn hdy_header_group_remove_header_bar(p_self: *HeaderGroup, p_header_bar: *handy.HeaderBar) void;
    pub const removeHeaderBar = hdy_header_group_remove_header_bar;

    /// Removes a nested `HdyHeaderGroup` from `self`.
    extern fn hdy_header_group_remove_header_group(p_self: *HeaderGroup, p_header_group: *handy.HeaderGroup) void;
    pub const removeHeaderGroup = hdy_header_group_remove_header_group;

    /// Sets whether the elements of the group should all receive the full
    /// decoration.
    extern fn hdy_header_group_set_decorate_all(p_self: *HeaderGroup, p_decorate_all: c_int) void;
    pub const setDecorateAll = hdy_header_group_set_decorate_all;

    extern fn hdy_header_group_get_type() usize;
    pub const getGObjectType = hdy_header_group_get_type;

    extern fn g_object_ref(p_self: *handy.HeaderGroup) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.HeaderGroup) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HeaderGroup, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A child object for `HeaderGroup`.
pub const HeaderGroupChild = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = handy.HeaderGroupChildClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Gets the child type.
    extern fn hdy_header_group_child_get_child_type(p_self: *HeaderGroupChild) handy.HeaderGroupChildType;
    pub const getChildType = hdy_header_group_child_get_child_type;

    /// Gets the child `gtk.HeaderBar`.
    ///
    /// Use `HeaderGroupChild.getChildType` to check the child type.
    extern fn hdy_header_group_child_get_gtk_header_bar(p_self: *HeaderGroupChild) *gtk.HeaderBar;
    pub const getGtkHeaderBar = hdy_header_group_child_get_gtk_header_bar;

    /// Gets the child `HeaderBar`.
    ///
    /// Use `HeaderGroupChild.getChildType` to check the child type.
    extern fn hdy_header_group_child_get_header_bar(p_self: *HeaderGroupChild) *handy.HeaderBar;
    pub const getHeaderBar = hdy_header_group_child_get_header_bar;

    /// Gets the child `HeaderGroup`.
    ///
    /// Use `HeaderGroupChild.getChildType` to check the child type.
    extern fn hdy_header_group_child_get_header_group(p_self: *HeaderGroupChild) *handy.HeaderGroup;
    pub const getHeaderGroup = hdy_header_group_child_get_header_group;

    extern fn hdy_header_group_child_get_type() usize;
    pub const getGObjectType = hdy_header_group_child_get_type;

    extern fn g_object_ref(p_self: *handy.HeaderGroupChild) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.HeaderGroupChild) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HeaderGroupChild, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A keypad for dialing numbers
///
/// The `HdyKeypad` widget is a keypad for entering numbers such as phone numbers
/// or PIN codes.
///
/// ## CSS nodes
///
/// `HdyKeypad` has a single CSS node with name `keypad`.
pub const Keypad = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.KeypadClass;
    f_parent_instance: gtk.Bin,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The amount of space between two consecutive columns.
        pub const column_spacing = struct {
            pub const name = "column-spacing";

            pub const Type = c_uint;
        };

        /// The widget for the lower end corner of `self`.
        pub const end_action = struct {
            pub const name = "end-action";

            pub const Type = ?*gtk.Widget;
        };

        /// The entry widget connected to the keypad.
        ///
        /// The entry will block any input not possible to type with the keypad.
        pub const entry = struct {
            pub const name = "entry";

            pub const Type = ?*gtk.Entry;
        };

        /// Whether standard letters should be displayed below the digits on the
        /// buttons.
        pub const letters_visible = struct {
            pub const name = "letters-visible";

            pub const Type = c_int;
        };

        /// The amount of space between two consecutive rows.
        pub const row_spacing = struct {
            pub const name = "row-spacing";

            pub const Type = c_uint;
        };

        /// The widget for the lower start corner of `self`.
        pub const start_action = struct {
            pub const name = "start-action";

            pub const Type = ?*gtk.Widget;
        };

        /// Whether to display symbols.
        ///
        /// This includes hash and asterisk buttons, and the plus symbol at the bottom
        /// of its 0 button.
        pub const symbols_visible = struct {
            pub const name = "symbols-visible";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyKeypad`.
    extern fn hdy_keypad_new(p_symbols_visible: c_int, p_letters_visible: c_int) *handy.Keypad;
    pub const new = hdy_keypad_new;

    /// Returns the amount of space between the columns of `self`.
    extern fn hdy_keypad_get_column_spacing(p_self: *Keypad) c_uint;
    pub const getColumnSpacing = hdy_keypad_get_column_spacing;

    /// Gets the widget for the lower right corner (or left, in RTL locales).
    extern fn hdy_keypad_get_end_action(p_self: *Keypad) ?*gtk.Widget;
    pub const getEndAction = hdy_keypad_get_end_action;

    /// Gets the connected entry.
    extern fn hdy_keypad_get_entry(p_self: *Keypad) *gtk.Entry;
    pub const getEntry = hdy_keypad_get_entry;

    /// Gets whether standard letters are displayed below the digits on the buttons.
    extern fn hdy_keypad_get_letters_visible(p_self: *Keypad) c_int;
    pub const getLettersVisible = hdy_keypad_get_letters_visible;

    /// Returns the amount of space between the rows of `self`.
    extern fn hdy_keypad_get_row_spacing(p_self: *Keypad) c_uint;
    pub const getRowSpacing = hdy_keypad_get_row_spacing;

    /// Gets the widget for the lower left corner (or right, in RTL locales).
    extern fn hdy_keypad_get_start_action(p_self: *Keypad) ?*gtk.Widget;
    pub const getStartAction = hdy_keypad_get_start_action;

    /// Gets whether symbols are displayed.
    extern fn hdy_keypad_get_symbols_visible(p_self: *Keypad) c_int;
    pub const getSymbolsVisible = hdy_keypad_get_symbols_visible;

    /// Sets the amount of space between columns of `self`.
    extern fn hdy_keypad_set_column_spacing(p_self: *Keypad, p_spacing: c_uint) void;
    pub const setColumnSpacing = hdy_keypad_set_column_spacing;

    /// Sets the widget for the lower right corner (or left, in RTL locales).
    extern fn hdy_keypad_set_end_action(p_self: *Keypad, p_end_action: ?*gtk.Widget) void;
    pub const setEndAction = hdy_keypad_set_end_action;

    /// Binds `entry` to `self`.
    extern fn hdy_keypad_set_entry(p_self: *Keypad, p_entry: ?*gtk.Entry) void;
    pub const setEntry = hdy_keypad_set_entry;

    /// Sets whether standard letters are displayed below the digits on the buttons.
    extern fn hdy_keypad_set_letters_visible(p_self: *Keypad, p_letters_visible: c_int) void;
    pub const setLettersVisible = hdy_keypad_set_letters_visible;

    /// Sets the amount of space between rows of `self`.
    extern fn hdy_keypad_set_row_spacing(p_self: *Keypad, p_spacing: c_uint) void;
    pub const setRowSpacing = hdy_keypad_set_row_spacing;

    /// Sets the widget for the lower left corner (or right, in RTL locales).
    extern fn hdy_keypad_set_start_action(p_self: *Keypad, p_start_action: ?*gtk.Widget) void;
    pub const setStartAction = hdy_keypad_set_start_action;

    /// Sets whether standard letters are displayed below the digits on the buttons.
    extern fn hdy_keypad_set_symbols_visible(p_self: *Keypad, p_symbols_visible: c_int) void;
    pub const setSymbolsVisible = hdy_keypad_set_symbols_visible;

    extern fn hdy_keypad_get_type() usize;
    pub const getGObjectType = hdy_keypad_get_type;

    extern fn g_object_ref(p_self: *handy.Keypad) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.Keypad) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Keypad, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An adaptive container acting like a box or a stack.
///
/// The `HdyLeaflet` widget can display its children like a `gtk.Box` does
/// or like a `gtk.Stack` does, adapting to size changes by switching
/// between the two modes.
///
/// When there is enough space the children are displayed side by side, otherwise
/// only one is displayed and the leaflet is said to be “folded”. The threshold
/// is dictated by the preferred minimum sizes of the children. When a leaflet is
/// folded, the children can be navigated using swipe gestures.
///
/// The “over” and “under” transition types stack the children one on top of the
/// other, while the “slide” transition puts the children side by side. While
/// navigating to a child on the side or below can be performed by swiping the
/// current child away, navigating to an upper child requires dragging it from
/// the edge where it resides. This doesn't affect non-dragging swipes.
///
/// The “over” and “under” transitions can draw their shadow on top of the
/// window's transparent areas, like the rounded corners. This is a side-effect
/// of allowing shadows to be drawn on top of OpenGL areas. It can be mitigated
/// by using `Window` or `ApplicationWindow` as they will crop
/// anything drawn beyond the rounded corners.
///
/// The child property `navigatable` can be set on `HdyLeaflet` children to
/// determine whether they can be navigated to when folded. If `FALSE`, the child
/// will be ignored by `Leaflet.getAdjacentChild`,
/// `Leaflet.navigate`, and swipe gestures. This can be used used to
/// prevent switching to widgets like separators.
///
/// ## CSS nodes
///
/// `HdyLeaflet` has a single CSS node with name `leaflet`. The node will get the
/// style classes `.folded` when it is folded, `.unfolded` when it's not, or none
/// if it didn't compute its fold yet.
pub const Leaflet = extern struct {
    pub const Parent = gtk.Container;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable, handy.Swipeable };
    pub const Class = handy.LeafletClass;
    f_parent_instance: gtk.Container,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether swipe gestures allow switching to the previous navigatable child.
        pub const can_swipe_back = struct {
            pub const name = "can-swipe-back";

            pub const Type = c_int;
        };

        /// Whether swipe gestures allow switching to the next navigatable child.
        pub const can_swipe_forward = struct {
            pub const name = "can-swipe-forward";

            pub const Type = c_int;
        };

        /// The child transition animation duration, in milliseconds.
        pub const child_transition_duration = struct {
            pub const name = "child-transition-duration";

            pub const Type = c_uint;
        };

        /// Whether a child transition is currently running.
        pub const child_transition_running = struct {
            pub const name = "child-transition-running";

            pub const Type = c_int;
        };

        /// Whether the leaflet is folded.
        ///
        /// The leaflet will be folded if the size allocated to it is smaller than the
        /// sum of the natural size of its children, it will be unfolded otherwise.
        pub const folded = struct {
            pub const name = "folded";

            pub const Type = c_int;
        };

        /// Whether to allocate the same width for all children when folded.
        pub const hhomogeneous_folded = struct {
            pub const name = "hhomogeneous-folded";

            pub const Type = c_int;
        };

        /// Whether to allocate the same width for all children when unfolded.
        pub const hhomogeneous_unfolded = struct {
            pub const name = "hhomogeneous-unfolded";

            pub const Type = c_int;
        };

        /// Whether the size should smoothly change when changing between children.
        pub const interpolate_size = struct {
            pub const name = "interpolate-size";

            pub const Type = c_int;
        };

        /// The mode transition animation duration, in milliseconds.
        pub const mode_transition_duration = struct {
            pub const name = "mode-transition-duration";

            pub const Type = c_uint;
        };

        /// The animation type used for transitions between modes and children.
        ///
        /// The transition type can be changed without problems at runtime, so it is
        /// possible to change the animation based on the mode or child that is about
        /// to become current.
        pub const transition_type = struct {
            pub const name = "transition-type";

            pub const Type = handy.LeafletTransitionType;
        };

        /// Whether to allocates the same height for all children when folded.
        pub const vhomogeneous_folded = struct {
            pub const name = "vhomogeneous-folded";

            pub const Type = c_int;
        };

        /// Whether to allocate the same height for all children when unfolded.
        pub const vhomogeneous_unfolded = struct {
            pub const name = "vhomogeneous-unfolded";

            pub const Type = c_int;
        };

        /// The widget currently visible when the leaflet is folded.
        ///
        /// The transition is determined by `Leaflet.properties.transition_type` and
        /// `Leaflet.properties.child_transition_duration`. The transition can be
        /// cancelled by the user, in which case visible child will change back to the
        /// previously visible child.
        pub const visible_child = struct {
            pub const name = "visible-child";

            pub const Type = ?*gtk.Widget;
        };

        /// The name of the widget currently visible when the leaflet is folded.
        ///
        /// See `Leaflet.properties.visible_child`.
        pub const visible_child_name = struct {
            pub const name = "visible-child-name";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyLeaflet`.
    extern fn hdy_leaflet_new() *handy.Leaflet;
    pub const new = hdy_leaflet_new;

    /// Finds the previous or next navigatable child.
    ///
    /// This will be the same widget `Leaflet.navigate` will navigate to.
    ///
    /// If there's no child to navigate to, `NULL` will be returned instead.
    extern fn hdy_leaflet_get_adjacent_child(p_self: *Leaflet, p_direction: handy.NavigationDirection) ?*gtk.Widget;
    pub const getAdjacentChild = hdy_leaflet_get_adjacent_child;

    /// Gets whether swipe gestures switch to the previous navigatable child.
    extern fn hdy_leaflet_get_can_swipe_back(p_self: *Leaflet) c_int;
    pub const getCanSwipeBack = hdy_leaflet_get_can_swipe_back;

    /// Gets whether swipe gestures switch to the next navigatable child.
    extern fn hdy_leaflet_get_can_swipe_forward(p_self: *Leaflet) c_int;
    pub const getCanSwipeForward = hdy_leaflet_get_can_swipe_forward;

    /// Finds the child of `self` with the name given as the argument.
    ///
    /// Returns `NULL` if there is no child with this name.
    extern fn hdy_leaflet_get_child_by_name(p_self: *Leaflet, p_name: [*:0]const u8) ?*gtk.Widget;
    pub const getChildByName = hdy_leaflet_get_child_by_name;

    /// Gets the amount of time that transitions between children will take.
    extern fn hdy_leaflet_get_child_transition_duration(p_self: *Leaflet) c_uint;
    pub const getChildTransitionDuration = hdy_leaflet_get_child_transition_duration;

    /// Returns whether `self` is currently in a transition from one page to another.
    extern fn hdy_leaflet_get_child_transition_running(p_self: *Leaflet) c_int;
    pub const getChildTransitionRunning = hdy_leaflet_get_child_transition_running;

    /// Gets whether `self` is folded.
    extern fn hdy_leaflet_get_folded(p_self: *Leaflet) c_int;
    pub const getFolded = hdy_leaflet_get_folded;

    /// Gets whether `self` is homogeneous for the given fold and orientation.
    extern fn hdy_leaflet_get_homogeneous(p_self: *Leaflet, p_folded: c_int, p_orientation: gtk.Orientation) c_int;
    pub const getHomogeneous = hdy_leaflet_get_homogeneous;

    /// Gets whether to interpolate between the sizes of children on page switches.
    extern fn hdy_leaflet_get_interpolate_size(p_self: *Leaflet) c_int;
    pub const getInterpolateSize = hdy_leaflet_get_interpolate_size;

    /// Gets the amount of time that transitions between modes in `self` will take.
    extern fn hdy_leaflet_get_mode_transition_duration(p_self: *Leaflet) c_uint;
    pub const getModeTransitionDuration = hdy_leaflet_get_mode_transition_duration;

    /// Gets the animation type that will be used for transitions between modes and
    /// children.
    extern fn hdy_leaflet_get_transition_type(p_self: *Leaflet) handy.LeafletTransitionType;
    pub const getTransitionType = hdy_leaflet_get_transition_type;

    /// Gets the visible child widget.
    extern fn hdy_leaflet_get_visible_child(p_self: *Leaflet) *gtk.Widget;
    pub const getVisibleChild = hdy_leaflet_get_visible_child;

    /// Gets the name of the currently visible child widget.
    extern fn hdy_leaflet_get_visible_child_name(p_self: *Leaflet) [*:0]const u8;
    pub const getVisibleChildName = hdy_leaflet_get_visible_child_name;

    /// Inserts `child` in the position after `sibling` in the list of children.
    ///
    /// If `sibling` is `NULL`, inserts `child` at the first position.
    extern fn hdy_leaflet_insert_child_after(p_self: *Leaflet, p_child: *gtk.Widget, p_sibling: ?*gtk.Widget) void;
    pub const insertChildAfter = hdy_leaflet_insert_child_after;

    /// Navigates to the previous or next navigatable child.
    ///
    /// The switch is similar to performing a swipe gesture to go in `direction`.
    extern fn hdy_leaflet_navigate(p_self: *Leaflet, p_direction: handy.NavigationDirection) c_int;
    pub const navigate = hdy_leaflet_navigate;

    /// Inserts `child` at the first position in `self`.
    extern fn hdy_leaflet_prepend(p_self: *Leaflet, p_child: *gtk.Widget) void;
    pub const prepend = hdy_leaflet_prepend;

    /// Moves `child` to the position after `sibling` in the list of children.
    ///
    /// If `sibling` is `NULL`, move `child` to the first position.
    extern fn hdy_leaflet_reorder_child_after(p_self: *Leaflet, p_child: *gtk.Widget, p_sibling: ?*gtk.Widget) void;
    pub const reorderChildAfter = hdy_leaflet_reorder_child_after;

    /// Sets whether swipe gestures switch to the previous navigatable child.
    extern fn hdy_leaflet_set_can_swipe_back(p_self: *Leaflet, p_can_swipe_back: c_int) void;
    pub const setCanSwipeBack = hdy_leaflet_set_can_swipe_back;

    /// Sets whether swipe gestures switch to the next navigatable child.
    extern fn hdy_leaflet_set_can_swipe_forward(p_self: *Leaflet, p_can_swipe_forward: c_int) void;
    pub const setCanSwipeForward = hdy_leaflet_set_can_swipe_forward;

    /// Sets the duration that transitions between children in `self` will take.
    extern fn hdy_leaflet_set_child_transition_duration(p_self: *Leaflet, p_duration: c_uint) void;
    pub const setChildTransitionDuration = hdy_leaflet_set_child_transition_duration;

    /// Sets whether to be homogeneous for the given fold and orientation.
    ///
    /// If it is homogeneous, the `Leaflet` will request the same
    /// width or height for all its children depending on the orientation. If it
    /// isn't and it is folded, the leaflet may change width or height when a
    /// different child becomes visible.
    extern fn hdy_leaflet_set_homogeneous(p_self: *Leaflet, p_folded: c_int, p_orientation: gtk.Orientation, p_homogeneous: c_int) void;
    pub const setHomogeneous = hdy_leaflet_set_homogeneous;

    /// Sets whether `self` will interpolate its size when changing the visible child.
    ///
    /// If the `Leaflet.properties.interpolate_size` property is set to `TRUE`, `self`
    /// will interpolate its size between the current one and the one it'll take
    /// after changing the visible child, according to the set transition duration.
    extern fn hdy_leaflet_set_interpolate_size(p_self: *Leaflet, p_interpolate_size: c_int) void;
    pub const setInterpolateSize = hdy_leaflet_set_interpolate_size;

    /// Sets the duration that transitions between modes in `self` will take.
    extern fn hdy_leaflet_set_mode_transition_duration(p_self: *Leaflet, p_duration: c_uint) void;
    pub const setModeTransitionDuration = hdy_leaflet_set_mode_transition_duration;

    /// Sets the animation type that will be used for transitions between modes and
    /// children.
    ///
    /// The transition type can be changed without problems at runtime, so it is
    /// possible to change the animation based on the mode or child that is about to
    /// become current.
    extern fn hdy_leaflet_set_transition_type(p_self: *Leaflet, p_transition: handy.LeafletTransitionType) void;
    pub const setTransitionType = hdy_leaflet_set_transition_type;

    /// Sets the currently visible widget when the leaflet is folded.
    extern fn hdy_leaflet_set_visible_child(p_self: *Leaflet, p_visible_child: *gtk.Widget) void;
    pub const setVisibleChild = hdy_leaflet_set_visible_child;

    /// Makes the child with the name `name` visible.
    ///
    /// See `Leaflet.setVisibleChild` for more details.
    extern fn hdy_leaflet_set_visible_child_name(p_self: *Leaflet, p_name: [*:0]const u8) void;
    pub const setVisibleChildName = hdy_leaflet_set_visible_child_name;

    extern fn hdy_leaflet_get_type() usize;
    pub const getGObjectType = hdy_leaflet_get_type;

    extern fn g_object_ref(p_self: *handy.Leaflet) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.Leaflet) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Leaflet, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A group of preference rows.
///
/// A `HdyPreferencesGroup` represents a group or tightly related preferences,
/// which in turn are represented by `PreferencesRow`.
///
/// To summarize the role of the preferences it gathers, a group can have both a
/// title and a description. The title will be used by `PreferencesWindow`
/// to let the user look for a preference.
///
/// ## CSS nodes
///
/// `HdyPreferencesGroup` has a single CSS node with name `preferencesgroup`.
pub const PreferencesGroup = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.PreferencesGroupClass;
    f_parent_instance: gtk.Bin,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The description for this group of preferences.
        pub const description = struct {
            pub const name = "description";

            pub const Type = ?[*:0]u8;
        };

        /// The title for this group of preferences.
        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };

        /// Whether to use markup for the title and description.
        pub const use_markup = struct {
            pub const name = "use-markup";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyPreferencesGroup`.
    extern fn hdy_preferences_group_new() *handy.PreferencesGroup;
    pub const new = hdy_preferences_group_new;

    extern fn hdy_preferences_group_get_description(p_self: *PreferencesGroup) [*:0]const u8;
    pub const getDescription = hdy_preferences_group_get_description;

    /// Gets the title of `self`.
    extern fn hdy_preferences_group_get_title(p_self: *PreferencesGroup) [*:0]const u8;
    pub const getTitle = hdy_preferences_group_get_title;

    /// Gets whether `self` uses markup for the title and description.
    extern fn hdy_preferences_group_get_use_markup(p_self: *PreferencesGroup) c_int;
    pub const getUseMarkup = hdy_preferences_group_get_use_markup;

    /// Sets the description for `self`.
    extern fn hdy_preferences_group_set_description(p_self: *PreferencesGroup, p_description: [*:0]const u8) void;
    pub const setDescription = hdy_preferences_group_set_description;

    /// Sets the title for `self`.
    extern fn hdy_preferences_group_set_title(p_self: *PreferencesGroup, p_title: [*:0]const u8) void;
    pub const setTitle = hdy_preferences_group_set_title;

    /// Sets whether `self` uses markup for the title and description.
    extern fn hdy_preferences_group_set_use_markup(p_self: *PreferencesGroup, p_use_markup: c_int) void;
    pub const setUseMarkup = hdy_preferences_group_set_use_markup;

    extern fn hdy_preferences_group_get_type() usize;
    pub const getGObjectType = hdy_preferences_group_get_type;

    extern fn g_object_ref(p_self: *handy.PreferencesGroup) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.PreferencesGroup) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *PreferencesGroup, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A page from `PreferencesWindow`.
///
/// The `HdyPreferencesPage` widget gathers preferences groups into a single page
/// of a preferences window.
///
/// ## CSS nodes
///
/// `HdyPreferencesPage` has a single CSS node with name `preferencespage`.
pub const PreferencesPage = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.PreferencesPageClass;
    f_parent_instance: gtk.Bin,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The icon name for this page of preferences.
        pub const icon_name = struct {
            pub const name = "icon-name";

            pub const Type = ?[*:0]u8;
        };

        /// The title for this page of preferences.
        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyPreferencesPage`.
    extern fn hdy_preferences_page_new() *handy.PreferencesPage;
    pub const new = hdy_preferences_page_new;

    /// Gets the icon name for `self`.
    extern fn hdy_preferences_page_get_icon_name(p_self: *PreferencesPage) ?[*:0]const u8;
    pub const getIconName = hdy_preferences_page_get_icon_name;

    /// Gets the title of `self`.
    extern fn hdy_preferences_page_get_title(p_self: *PreferencesPage) ?[*:0]const u8;
    pub const getTitle = hdy_preferences_page_get_title;

    /// Sets the icon name for `self`.
    extern fn hdy_preferences_page_set_icon_name(p_self: *PreferencesPage, p_icon_name: ?[*:0]const u8) void;
    pub const setIconName = hdy_preferences_page_set_icon_name;

    /// Sets the title of `self`.
    extern fn hdy_preferences_page_set_title(p_self: *PreferencesPage, p_title: ?[*:0]const u8) void;
    pub const setTitle = hdy_preferences_page_set_title;

    extern fn hdy_preferences_page_get_type() usize;
    pub const getGObjectType = hdy_preferences_page_get_type;

    extern fn g_object_ref(p_self: *handy.PreferencesPage) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.PreferencesPage) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *PreferencesPage, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gtk.ListBoxRow` used to present preferences.
///
/// The `HdyPreferencesRow` widget has a title that `PreferencesWindow`
/// will use to let the user look for a preference. It doesn't present the title
/// in any way and lets you present the preference as you please.
///
/// `ActionRow` and its derivatives are convenient to use as preference
/// rows as they take care of presenting the preference's title while letting you
/// compose the inputs of the preference around it.
pub const PreferencesRow = extern struct {
    pub const Parent = gtk.ListBoxRow;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Actionable, gtk.Buildable };
    pub const Class = handy.PreferencesRowClass;
    f_parent_instance: gtk.ListBoxRow,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The title of the preference represented by this row.
        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };

        /// Whether an embedded underline in the title indicates a mnemonic.
        pub const use_underline = struct {
            pub const name = "use-underline";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyPreferencesRow`.
    extern fn hdy_preferences_row_new() *handy.PreferencesRow;
    pub const new = hdy_preferences_row_new;

    /// Gets the title of the preference represented by `self`.
    extern fn hdy_preferences_row_get_title(p_self: *PreferencesRow) ?[*:0]const u8;
    pub const getTitle = hdy_preferences_row_get_title;

    /// Gets whether an embedded underline in the title indicates a mnemonic.
    extern fn hdy_preferences_row_get_use_underline(p_self: *PreferencesRow) c_int;
    pub const getUseUnderline = hdy_preferences_row_get_use_underline;

    /// Sets the title of the preference represented by `self`.
    extern fn hdy_preferences_row_set_title(p_self: *PreferencesRow, p_title: ?[*:0]const u8) void;
    pub const setTitle = hdy_preferences_row_set_title;

    /// Sets whether an embedded underline in the title indicates a mnemonic.
    extern fn hdy_preferences_row_set_use_underline(p_self: *PreferencesRow, p_use_underline: c_int) void;
    pub const setUseUnderline = hdy_preferences_row_set_use_underline;

    extern fn hdy_preferences_row_get_type() usize;
    pub const getGObjectType = hdy_preferences_row_get_type;

    extern fn g_object_ref(p_self: *handy.PreferencesRow) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.PreferencesRow) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *PreferencesRow, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A window to present an application's preferences.
///
/// The `HdyPreferencesWindow` widget presents an application's preferences
/// gathered into pages and groups. The preferences are searchable by the user.
///
/// ## CSS nodes
///
/// `HdyPreferencesWindow` has a main CSS node with the name `window` and the
/// style class `.preferences`.
pub const PreferencesWindow = extern struct {
    pub const Parent = handy.Window;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.PreferencesWindowClass;
    f_parent_instance: handy.Window,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether the window allows closing the subpage via a swipe gesture.
        pub const can_swipe_back = struct {
            pub const name = "can-swipe-back";

            pub const Type = c_int;
        };

        /// Whether search is enabled.
        pub const search_enabled = struct {
            pub const name = "search-enabled";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyPreferencesWindow`.
    extern fn hdy_preferences_window_new() *handy.PreferencesWindow;
    pub const new = hdy_preferences_window_new;

    /// Closes the current subpage.
    ///
    /// If there is no presented subpage, this does nothing.
    extern fn hdy_preferences_window_close_subpage(p_self: *PreferencesWindow) void;
    pub const closeSubpage = hdy_preferences_window_close_subpage;

    /// Gets whether swipe gestures allow switching from a subpage to the
    /// preferences.
    extern fn hdy_preferences_window_get_can_swipe_back(p_self: *PreferencesWindow) c_int;
    pub const getCanSwipeBack = hdy_preferences_window_get_can_swipe_back;

    /// Gets whether search is enabled for `self`.
    extern fn hdy_preferences_window_get_search_enabled(p_self: *PreferencesWindow) c_int;
    pub const getSearchEnabled = hdy_preferences_window_get_search_enabled;

    /// Sets `subpage` as the window's subpage and opens it.
    ///
    /// The transition can be cancelled by the user, in which case visible child will
    /// change back to the previously visible child.
    extern fn hdy_preferences_window_present_subpage(p_self: *PreferencesWindow, p_subpage: *gtk.Widget) void;
    pub const presentSubpage = hdy_preferences_window_present_subpage;

    /// Sets whether swipe gestures allow switching from a subpage to the
    /// preferences.
    extern fn hdy_preferences_window_set_can_swipe_back(p_self: *PreferencesWindow, p_can_swipe_back: c_int) void;
    pub const setCanSwipeBack = hdy_preferences_window_set_can_swipe_back;

    /// Sets whether search is enabled for `self`.
    extern fn hdy_preferences_window_set_search_enabled(p_self: *PreferencesWindow, p_search_enabled: c_int) void;
    pub const setSearchEnabled = hdy_preferences_window_set_search_enabled;

    extern fn hdy_preferences_window_get_type() usize;
    pub const getGObjectType = hdy_preferences_window_get_type;

    extern fn g_object_ref(p_self: *handy.PreferencesWindow) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.PreferencesWindow) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *PreferencesWindow, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A toolbar to integrate a search entry with.
///
/// `HdySearchBar` is a container made to have a search entry (possibly with
/// additional connex widgets, such as drop-down menus, or buttons) built-in. The
/// search bar would appear when a search is started through typing on the
/// keyboard, or the application’s search mode is toggled on.
///
/// For keyboard presses to start a search, events will need to be forwarded from
/// the top-level window that contains the search bar. See
/// `SearchBar.handleEvent` for example code. Common shortcuts such as
/// <kbd>Ctrl</kbd>+<kbd>F</kbd> should be handled as an application action, or
/// through the menu items.
///
/// You will also need to tell the search bar about which entry you are using as
/// your search entry using `SearchBar.connectEntry`. The following
/// example shows you how to create a more complex search entry.
///
/// `HdySearchBar` is very similar to `gtk.SearchBar`, the main difference
/// being that it allows the search entry to fill all the available space. This
/// allows you to control your search entry's width with a `Clamp`.
///
/// ## CSS nodes
///
/// `HdySearchBar` has a single CSS node with name `searchbar`.
pub const SearchBar = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.SearchBarClass;
    f_parent_instance: gtk.Bin,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether the search mode is on and the search bar shown.
        pub const search_mode_enabled = struct {
            pub const name = "search-mode-enabled";

            pub const Type = c_int;
        };

        /// Whether to show the close button in the toolbar.
        pub const show_close_button = struct {
            pub const name = "show-close-button";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdySearchBar.
    ///
    /// You will need to tell it about which widget is going to be your text entry
    /// using `SearchBar.connectEntry`.
    extern fn hdy_search_bar_new() *handy.SearchBar;
    pub const new = hdy_search_bar_new;

    /// Sets the entry widget passed as the one to be used in this search bar.
    ///
    /// The entry should be a descendant of the search bar. This is only required if
    /// the entry isn’t the direct child of the search bar (as in our main example).
    extern fn hdy_search_bar_connect_entry(p_self: *SearchBar, p_entry: *gtk.Entry) void;
    pub const connectEntry = hdy_search_bar_connect_entry;

    /// Gets whether the search mode is on.
    extern fn hdy_search_bar_get_search_mode(p_self: *SearchBar) c_int;
    pub const getSearchMode = hdy_search_bar_get_search_mode;

    /// Gets whether the close button is shown.
    extern fn hdy_search_bar_get_show_close_button(p_self: *SearchBar) c_int;
    pub const getShowCloseButton = hdy_search_bar_get_show_close_button;

    /// Handles key press events.
    ///
    /// This function should be called when the top-level window which contains the
    /// search bar received a key event.
    ///
    /// If the key event is handled by the search bar, the bar will be shown, the
    /// entry populated with the entered text and `GDK_EVENT_STOP` will be returned.
    /// The caller should ensure that events are not propagated further.
    ///
    /// If no entry has been connected to the search bar, using
    /// `SearchBar.connectEntry`, this function will return immediately with
    /// a warning.
    ///
    /// ## Showing the search bar on key presses
    ///
    /// ```c
    /// static gboolean
    /// on_key_press_event (GtkWidget *widget,
    ///                     GdkEvent  *event,
    ///                     gpointer   user_data)
    /// {
    ///   HdySearchBar *bar = HDY_SEARCH_BAR (user_data);
    ///   return hdy_search_bar_handle_event (self, event);
    /// }
    ///
    /// static void
    /// create_toplevel (void)
    /// {
    ///   GtkWidget *window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    ///   GtkWindow *search_bar = hdy_search_bar_new ();
    ///
    ///   // Add more widgets to the window...
    ///
    ///   g_signal_connect (window,
    ///                    "key-press-event",
    ///                     G_CALLBACK (on_key_press_event),
    ///                     search_bar);
    /// }
    /// ```
    extern fn hdy_search_bar_handle_event(p_self: *SearchBar, p_event: *gdk.Event) c_int;
    pub const handleEvent = hdy_search_bar_handle_event;

    /// Switches the search mode on or off.
    extern fn hdy_search_bar_set_search_mode(p_self: *SearchBar, p_search_mode: c_int) void;
    pub const setSearchMode = hdy_search_bar_set_search_mode;

    /// Shows or hides the close button.
    ///
    /// Applications that already have a “search” toggle button should not show a
    /// close button in their search bar, as it duplicates the role of the toggle
    /// button.
    extern fn hdy_search_bar_set_show_close_button(p_self: *SearchBar, p_visible: c_int) void;
    pub const setShowCloseButton = hdy_search_bar_set_show_close_button;

    extern fn hdy_search_bar_get_type() usize;
    pub const getGObjectType = hdy_search_bar_get_type;

    extern fn g_object_ref(p_self: *handy.SearchBar) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.SearchBar) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SearchBar, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A best fit container.
///
/// The `HdySqueezer` widget is a container which only shows the first of its
/// children that fits in the available size. It is convenient to offer different
/// widgets to represent the same data with different levels of detail, making
/// the widget seem to squeeze itself to fit in the available space.
///
/// Transitions between children can be animated as fades. This can be controlled
/// with `Squeezer.setTransitionType`.
///
/// ## CSS nodes
///
/// `HdySqueezer` has a single CSS node with name `squeezer`.
pub const Squeezer = opaque {
    pub const Parent = gtk.Container;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = handy.SqueezerClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether all children have the same size for the opposite orientation.
        ///
        /// For example, if a squeezer is horizontal and is homogeneous, it will request
        /// the same height for all its children. If it isn't, the squeezer may change
        /// size when a different child becomes visible.
        pub const homogeneous = struct {
            pub const name = "homogeneous";

            pub const Type = c_int;
        };

        /// Whether the squeezer interpolates its size when changing the visible child.
        ///
        /// If `TRUE`, the squeezer will interpolate its size between the one of the
        /// previous visible child and the one of the new visible child, according to
        /// the set transition duration and the orientation, e.g. if the squeezer is
        /// horizontal, it will interpolate the its height.
        pub const interpolate_size = struct {
            pub const name = "interpolate-size";

            pub const Type = c_int;
        };

        /// The animation duration, in milliseconds.
        pub const transition_duration = struct {
            pub const name = "transition-duration";

            pub const Type = c_uint;
        };

        /// Whether a transition is currently running.
        pub const transition_running = struct {
            pub const name = "transition-running";

            pub const Type = c_int;
        };

        /// The type of animation used for transitions between children.
        ///
        /// Available types include various kinds of fades and slides.
        ///
        /// The transition type can be changed without problems at runtime, so it is
        /// possible to change the animation based on the child that is about to become
        /// current.
        pub const transition_type = struct {
            pub const name = "transition-type";

            pub const Type = handy.SqueezerTransitionType;
        };

        /// The currently visible child.
        pub const visible_child = struct {
            pub const name = "visible-child";

            pub const Type = ?*gtk.Widget;
        };

        /// The horizontal alignment, from 0 (start) to 1 (end).
        ///
        /// The xalign property determines the horizontal alignment of the children
        /// inside the squeezer's size allocation. Compare this to
        /// `gtk.Widget.properties.halign`, which determines how the squeezer's size
        /// allocation is positioned in the space available for the squeezer.
        ///
        /// This will affect the position of children too wide to fit in the squeezer
        /// as they are fading out.
        pub const xalign = struct {
            pub const name = "xalign";

            pub const Type = f32;
        };

        /// The vertical alignment, from 0 (start) to 1 (end).
        ///
        /// The yalign property determines the vertical alignment of the children
        /// inside the squeezer's size allocation. Compare this to
        /// `gtk.Widget.properties.valign`, which determines how the squeezer's size
        /// allocation is positioned in the space available for the squeezer.
        ///
        /// This will affect the position of children too tall to fit in the squeezer
        /// as they are fading out.
        pub const yalign = struct {
            pub const name = "yalign";

            pub const Type = f32;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdySqueezer`.
    extern fn hdy_squeezer_new() *handy.Squeezer;
    pub const new = hdy_squeezer_new;

    /// Gets whether `child` is enabled.
    ///
    /// See `Squeezer.setChildEnabled`.
    extern fn hdy_squeezer_get_child_enabled(p_self: *Squeezer, p_child: *gtk.Widget) c_int;
    pub const getChildEnabled = hdy_squeezer_get_child_enabled;

    /// Gets whether `self` is homogeneous.
    extern fn hdy_squeezer_get_homogeneous(p_self: *Squeezer) c_int;
    pub const getHomogeneous = hdy_squeezer_get_homogeneous;

    /// Gets whether `self` should interpolate its size on visible child change.
    extern fn hdy_squeezer_get_interpolate_size(p_self: *Squeezer) c_int;
    pub const getInterpolateSize = hdy_squeezer_get_interpolate_size;

    /// Gets the amount of time that transitions between children will take.
    extern fn hdy_squeezer_get_transition_duration(p_self: *Squeezer) c_uint;
    pub const getTransitionDuration = hdy_squeezer_get_transition_duration;

    /// Gets whether a transition is currently running for `self`.
    extern fn hdy_squeezer_get_transition_running(p_self: *Squeezer) c_int;
    pub const getTransitionRunning = hdy_squeezer_get_transition_running;

    /// Gets the animation type that will be used for transitions between children.
    extern fn hdy_squeezer_get_transition_type(p_self: *Squeezer) handy.SqueezerTransitionType;
    pub const getTransitionType = hdy_squeezer_get_transition_type;

    /// Gets the currently visible child of `self`.
    extern fn hdy_squeezer_get_visible_child(p_self: *Squeezer) ?*gtk.Widget;
    pub const getVisibleChild = hdy_squeezer_get_visible_child;

    /// Gets the horizontal alignment.
    extern fn hdy_squeezer_get_xalign(p_self: *Squeezer) f32;
    pub const getXalign = hdy_squeezer_get_xalign;

    /// Gets the vertical alignment.
    extern fn hdy_squeezer_get_yalign(p_self: *Squeezer) f32;
    pub const getYalign = hdy_squeezer_get_yalign;

    /// Sets whether `child` is enabled.
    ///
    /// If a child is disabled, it will be ignored when looking for the child fitting
    /// the available size best. This allows to programmatically and prematurely hide
    /// a child of `self` even if it fits in the available space.
    ///
    /// This can be used e.g. to ensure a certain child is hidden below a certain
    /// window width, or any other constraint you find suitable.
    extern fn hdy_squeezer_set_child_enabled(p_self: *Squeezer, p_child: *gtk.Widget, p_enabled: c_int) void;
    pub const setChildEnabled = hdy_squeezer_set_child_enabled;

    /// Sets whether all children have the same size for the opposite orientation.
    extern fn hdy_squeezer_set_homogeneous(p_self: *Squeezer, p_homogeneous: c_int) void;
    pub const setHomogeneous = hdy_squeezer_set_homogeneous;

    /// Sets whether `self` should interpolate its size on visible child change.
    extern fn hdy_squeezer_set_interpolate_size(p_self: *Squeezer, p_interpolate_size: c_int) void;
    pub const setInterpolateSize = hdy_squeezer_set_interpolate_size;

    /// Sets the duration that transitions between children in `self` will take.
    extern fn hdy_squeezer_set_transition_duration(p_self: *Squeezer, p_duration: c_uint) void;
    pub const setTransitionDuration = hdy_squeezer_set_transition_duration;

    /// Sets the animation type that will be used for transitions between children.
    extern fn hdy_squeezer_set_transition_type(p_self: *Squeezer, p_transition: handy.SqueezerTransitionType) void;
    pub const setTransitionType = hdy_squeezer_set_transition_type;

    /// Sets the horizontal alignment.
    extern fn hdy_squeezer_set_xalign(p_self: *Squeezer, p_xalign: f32) void;
    pub const setXalign = hdy_squeezer_set_xalign;

    /// Sets the vertical alignment.
    extern fn hdy_squeezer_set_yalign(p_self: *Squeezer, p_yalign: f32) void;
    pub const setYalign = hdy_squeezer_set_yalign;

    extern fn hdy_squeezer_get_type() usize;
    pub const getGObjectType = hdy_squeezer_get_type;

    extern fn g_object_ref(p_self: *handy.Squeezer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.Squeezer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Squeezer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A page used for empty/error states and similar use-cases.
///
/// The `HdyStatusPage` widget can have an icon, a title, a description and a
/// custom widget which is displayed below them.
///
/// ## CSS nodes
///
/// `HdyStatusPage` has a main CSS node with name `statuspage`.
pub const StatusPage = opaque {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.StatusPageClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The description to be displayed below the title.
        pub const description = struct {
            pub const name = "description";

            pub const Type = ?[*:0]u8;
        };

        /// The name of the icon to be used.
        pub const icon_name = struct {
            pub const name = "icon-name";

            pub const Type = ?[*:0]u8;
        };

        /// The title to be displayed below the icon.
        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyStatusPage`.
    extern fn hdy_status_page_new() *handy.StatusPage;
    pub const new = hdy_status_page_new;

    /// Gets the description for `self`.
    extern fn hdy_status_page_get_description(p_self: *StatusPage) ?[*:0]const u8;
    pub const getDescription = hdy_status_page_get_description;

    /// Gets the icon name for `self`.
    extern fn hdy_status_page_get_icon_name(p_self: *StatusPage) ?[*:0]const u8;
    pub const getIconName = hdy_status_page_get_icon_name;

    /// Gets the title for `self`.
    extern fn hdy_status_page_get_title(p_self: *StatusPage) ?[*:0]const u8;
    pub const getTitle = hdy_status_page_get_title;

    /// Sets the description for `self`.
    extern fn hdy_status_page_set_description(p_self: *StatusPage, p_description: ?[*:0]const u8) void;
    pub const setDescription = hdy_status_page_set_description;

    /// Sets the icon name for `self`.
    extern fn hdy_status_page_set_icon_name(p_self: *StatusPage, p_icon_name: ?[*:0]const u8) void;
    pub const setIconName = hdy_status_page_set_icon_name;

    /// Sets the title for `self`.
    extern fn hdy_status_page_set_title(p_self: *StatusPage, p_title: ?[*:0]const u8) void;
    pub const setTitle = hdy_status_page_set_title;

    extern fn hdy_status_page_get_type() usize;
    pub const getGObjectType = hdy_status_page_get_type;

    extern fn g_object_ref(p_self: *handy.StatusPage) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.StatusPage) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *StatusPage, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A class for managing application-wide styling.
///
/// `HdyStyleManager` provides a way to query and influence the application
/// styles, such as whether to use dark or high contrast appearance.
///
/// It allows to set the color scheme via the
/// `StyleManager.properties.color_scheme` property, and to query the current
/// appearance, as well as whether a system-wide color scheme preference exists.
///
/// Important: `gtk.Settings.properties.gtk_application_prefer_dark_theme` should
/// not be used together with `HdyStyleManager` and will result in a warning.
/// Color schemes should be used instead.
pub const StyleManager = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = handy.StyleManagerClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The requested application color scheme.
        ///
        /// The effective appearance will be decided based on the application color
        /// scheme and the system preferred color scheme. The
        /// `StyleManager.properties.dark` property can be used to query the current
        /// effective appearance.
        ///
        /// The `HDY_COLOR_SCHEME_PREFER_LIGHT` color scheme results in the application
        /// using light appearance unless the system prefers dark colors. This is the
        /// default value.
        ///
        /// The `HDY_COLOR_SCHEME_PREFER_DARK` color scheme results in the application
        /// using dark appearance, but can still switch to the light appearance if the
        /// system can prefers it, for example, when the high contrast preference is
        /// enabled.
        ///
        /// The `HDY_COLOR_SCHEME_FORCE_LIGHT` and `HDY_COLOR_SCHEME_FORCE_DARK` values
        /// ignore the system preference entirely, they are useful if the application
        /// wants to match its UI to its content or to provide a separate color scheme
        /// switcher.
        ///
        /// If a per-`gdk.Display` style manager has its color scheme set to
        /// `HDY_COLOR_SCHEME_DEFAULT`, it will inherit the color scheme from the
        /// default style manager.
        ///
        /// For the default style manager, `HDY_COLOR_SCHEME_DEFAULT` is equivalent to
        /// `HDY_COLOR_SCHEME_FORCE_LIGHT`.
        ///
        /// The `StyleManager.properties.system_supports_color_schemes` property can be
        /// used to check if the current environment provides a color scheme
        /// preference.
        pub const color_scheme = struct {
            pub const name = "color-scheme";

            pub const Type = handy.ColorScheme;
        };

        /// Whether the application is using dark appearance.
        ///
        /// This property can be used to query the current appearance, as requested via
        /// `StyleManager.properties.color_scheme`.
        pub const dark = struct {
            pub const name = "dark";

            pub const Type = c_int;
        };

        /// The display the style manager is associated with.
        ///
        /// The display will be `NULL` for the style manager returned by
        /// `StyleManager.getDefault`.
        pub const display = struct {
            pub const name = "display";

            pub const Type = ?*gdk.Display;
        };

        /// Whether the application is using high contrast appearance.
        ///
        /// This cannot be overridden by applications.
        pub const high_contrast = struct {
            pub const name = "high-contrast";

            pub const Type = c_int;
        };

        /// Whether the system supports color schemes.
        ///
        /// This property can be used to check if the current environment provides a
        /// color scheme preference. For example, applications might want to show a
        /// separate appearance switcher if it's set to `FALSE`.
        ///
        /// It's only set at startup and cannot change its value later.
        ///
        /// See `StyleManager.properties.color_scheme`.
        pub const system_supports_color_schemes = struct {
            pub const name = "system-supports-color-schemes";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Gets the default `StyleManager` instance.
    ///
    /// It manages all `gdk.Display` instances unless the style manager for
    /// that display has an override.
    ///
    /// See `StyleManager.getForDisplay`.
    extern fn hdy_style_manager_get_default() *handy.StyleManager;
    pub const getDefault = hdy_style_manager_get_default;

    /// Gets the `StyleManager` instance managing `display`.
    ///
    /// It can be used to override styles for that specific display instead of the
    /// whole application.
    ///
    /// Most applications should use `StyleManager.getDefault` instead.
    extern fn hdy_style_manager_get_for_display(p_display: *gdk.Display) *handy.StyleManager;
    pub const getForDisplay = hdy_style_manager_get_for_display;

    /// Gets the requested application color scheme.
    extern fn hdy_style_manager_get_color_scheme(p_self: *StyleManager) handy.ColorScheme;
    pub const getColorScheme = hdy_style_manager_get_color_scheme;

    /// Gets whether the application is using dark appearance.
    extern fn hdy_style_manager_get_dark(p_self: *StyleManager) c_int;
    pub const getDark = hdy_style_manager_get_dark;

    /// Gets the display the style manager is associated with.
    ///
    /// The display will be `NULL` for the style manager returned by
    /// `StyleManager.getDefault`.
    extern fn hdy_style_manager_get_display(p_self: *StyleManager) *gdk.Display;
    pub const getDisplay = hdy_style_manager_get_display;

    /// Gets whether the application is using high contrast appearance.
    extern fn hdy_style_manager_get_high_contrast(p_self: *StyleManager) c_int;
    pub const getHighContrast = hdy_style_manager_get_high_contrast;

    /// Gets whether the system supports color schemes.
    extern fn hdy_style_manager_get_system_supports_color_schemes(p_self: *StyleManager) c_int;
    pub const getSystemSupportsColorSchemes = hdy_style_manager_get_system_supports_color_schemes;

    /// Sets the requested application color scheme.
    ///
    /// The effective appearance will be decided based on the application color
    /// scheme and the system preferred color scheme. The
    /// `StyleManager.properties.dark` property can be used to query the current
    /// effective appearance.
    extern fn hdy_style_manager_set_color_scheme(p_self: *StyleManager, p_color_scheme: handy.ColorScheme) void;
    pub const setColorScheme = hdy_style_manager_set_color_scheme;

    extern fn hdy_style_manager_get_type() usize;
    pub const getGObjectType = hdy_style_manager_get_type;

    extern fn g_object_ref(p_self: *handy.StyleManager) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.StyleManager) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *StyleManager, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An object for syncing swipeable widgets.
///
/// The `HdySwipeGroup` object can be used to sync multiple swipeable widgets
/// that implement the `Swipeable` interface, such as `Carousel`, so
/// that animating one of them also animates all the other widgets in the group.
///
/// This can be useful for syncing widgets between a window's titlebar and
/// content area.
///
/// ## HdySwipeGroup as GtkBuildable
///
/// `HdySwipeGroup` can be created in an UI definition. The list of swipeable
/// widgets is specified with a &lt;swipeables&gt; element containing multiple
/// &lt;swipeable&gt; elements with their ”name” attribute specifying the id of
/// the widgets.
///
/// ```xml
/// <object class="HdySwipeGroup">
///   <swipeables>
///     <swipeable name="carousel1"/>
///     <swipeable name="carousel2"/>
///   </swipeables>
/// </object>
/// ```
///
/// `HdySwipeGroup` has been deprecated, `Window` and
/// `ApplicationWindow` allow using a single leaflet for both content and
/// header bar, without the need to sync them.
pub const SwipeGroup = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gtk.Buildable};
    pub const Class = handy.SwipeGroupClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `HdySwipeGroup`.
    extern fn hdy_swipe_group_new() *handy.SwipeGroup;
    pub const new = hdy_swipe_group_new;

    /// Adds a swipeable to `self`.
    ///
    /// When the widget is destroyed or no longer referenced elsewhere, it will be
    /// removed from the swipe group.
    extern fn hdy_swipe_group_add_swipeable(p_self: *SwipeGroup, p_swipeable: *handy.Swipeable) void;
    pub const addSwipeable = hdy_swipe_group_add_swipeable;

    /// Gets the list of swipeables associated with `self`.
    extern fn hdy_swipe_group_get_swipeables(p_self: *SwipeGroup) *glib.SList;
    pub const getSwipeables = hdy_swipe_group_get_swipeables;

    /// Removes a widget from a `SwipeGroup`.
    extern fn hdy_swipe_group_remove_swipeable(p_self: *SwipeGroup, p_swipeable: *handy.Swipeable) void;
    pub const removeSwipeable = hdy_swipe_group_remove_swipeable;

    extern fn hdy_swipe_group_get_type() usize;
    pub const getGObjectType = hdy_swipe_group_get_type;

    extern fn g_object_ref(p_self: *handy.SwipeGroup) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.SwipeGroup) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SwipeGroup, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Swipe tracker used in `Carousel` and `Leaflet`.
///
/// The `HdySwipeTracker` object can be used for implementing widgets with swipe
/// gestures. It supports touch-based swipes, pointer dragging, and touchpad
/// scrolling.
///
/// The widgets will probably want to expose `SwipeTracker.properties.enabled`
/// property. If they expect to use horizontal orientation,
/// `SwipeTracker.properties.reversed` property can be used for supporting RTL text
/// direction.
pub const SwipeTracker = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gtk.Orientable};
    pub const Class = handy.SwipeTrackerClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether to allow swiping for more than one snap point at a time.
        ///
        /// If the value is `FALSE`, each swipe can only move to the adjacent snap
        /// points.
        pub const allow_long_swipes = struct {
            pub const name = "allow-long-swipes";

            pub const Type = c_int;
        };

        /// Whether to allow dragging with mouse pointer.
        ///
        /// This should usually be `FALSE`.
        pub const allow_mouse_drag = struct {
            pub const name = "allow-mouse-drag";

            pub const Type = c_int;
        };

        /// Whether the swipe tracker is enabled.
        ///
        /// When it's not enabled, no events will be processed. Usually widgets will
        /// want to expose this via a property.
        pub const enabled = struct {
            pub const name = "enabled";

            pub const Type = c_int;
        };

        /// Whether to reverse the swipe direction.
        ///
        /// If the swipe tracker is horizontal, it can be used for supporting RTL text
        /// direction.
        pub const reversed = struct {
            pub const name = "reversed";

            pub const Type = c_int;
        };

        /// The widget the swipe tracker is attached to. Must not be `NULL`.
        pub const swipeable = struct {
            pub const name = "swipeable";

            pub const Type = ?*handy.Swipeable;
        };
    };

    pub const signals = struct {
        /// This signal is emitted when a possible swipe is detected.
        ///
        /// The `direction` value can be used to restrict the swipe to a certain
        /// direction.
        pub const begin_swipe = struct {
            pub const name = "begin-swipe";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_direction: handy.NavigationDirection, p_direct: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(SwipeTracker, p_instance))),
                    gobject.signalLookup("begin-swipe", SwipeTracker.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted as soon as the gesture has stopped.
        pub const end_swipe = struct {
            pub const name = "end-swipe";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_duration: i64, p_to: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(SwipeTracker, p_instance))),
                    gobject.signalLookup("end-swipe", SwipeTracker.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted every time the progress value changes.
        pub const update_swipe = struct {
            pub const name = "update-swipe";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_progress: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(SwipeTracker, p_instance))),
                    gobject.signalLookup("update-swipe", SwipeTracker.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `HdySwipeTracker` object on `widget`.
    extern fn hdy_swipe_tracker_new(p_swipeable: *handy.Swipeable) *handy.SwipeTracker;
    pub const new = hdy_swipe_tracker_new;

    /// Whether to allow swiping for more than one snap point at a time.
    ///
    /// If the value is `FALSE`, each swipe can only move to the adjacent snap
    /// points.
    extern fn hdy_swipe_tracker_get_allow_long_swipes(p_self: *SwipeTracker) c_int;
    pub const getAllowLongSwipes = hdy_swipe_tracker_get_allow_long_swipes;

    /// Get whether `self` can be dragged with mouse pointer.
    extern fn hdy_swipe_tracker_get_allow_mouse_drag(p_self: *SwipeTracker) c_int;
    pub const getAllowMouseDrag = hdy_swipe_tracker_get_allow_mouse_drag;

    /// Get whether `self` is enabled.
    extern fn hdy_swipe_tracker_get_enabled(p_self: *SwipeTracker) c_int;
    pub const getEnabled = hdy_swipe_tracker_get_enabled;

    /// Get whether `self` is reversing the swipe direction.
    extern fn hdy_swipe_tracker_get_reversed(p_self: *SwipeTracker) c_int;
    pub const getReversed = hdy_swipe_tracker_get_reversed;

    /// Get `self`'s swipeable widget.
    extern fn hdy_swipe_tracker_get_swipeable(p_self: *SwipeTracker) *handy.Swipeable;
    pub const getSwipeable = hdy_swipe_tracker_get_swipeable;

    /// Sets whether to allow swiping for more than one snap point at a time.
    ///
    /// If the value is `FALSE`, each swipe can only move to the adjacent snap
    /// points.
    extern fn hdy_swipe_tracker_set_allow_long_swipes(p_self: *SwipeTracker, p_allow_long_swipes: c_int) void;
    pub const setAllowLongSwipes = hdy_swipe_tracker_set_allow_long_swipes;

    /// Set whether `self` can be dragged with mouse pointer.
    ///
    /// This should usually be `FALSE`.
    extern fn hdy_swipe_tracker_set_allow_mouse_drag(p_self: *SwipeTracker, p_allow_mouse_drag: c_int) void;
    pub const setAllowMouseDrag = hdy_swipe_tracker_set_allow_mouse_drag;

    /// Set whether `self` is enabled.
    extern fn hdy_swipe_tracker_set_enabled(p_self: *SwipeTracker, p_enabled: c_int) void;
    pub const setEnabled = hdy_swipe_tracker_set_enabled;

    /// Set whether to reverse the swipe direction.
    ///
    /// If `self` is horizontal, can be used for supporting RTL text direction.
    extern fn hdy_swipe_tracker_set_reversed(p_self: *SwipeTracker, p_reversed: c_int) void;
    pub const setReversed = hdy_swipe_tracker_set_reversed;

    /// Move the current progress value by `delta`.
    ///
    /// This can be used to adjust the current position if snap points move during
    /// the gesture.
    extern fn hdy_swipe_tracker_shift_position(p_self: *SwipeTracker, p_delta: f64) void;
    pub const shiftPosition = hdy_swipe_tracker_shift_position;

    extern fn hdy_swipe_tracker_get_type() usize;
    pub const getGObjectType = hdy_swipe_tracker_get_type;

    extern fn g_object_ref(p_self: *handy.SwipeTracker) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.SwipeTracker) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SwipeTracker, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A tab bar for `TabView`.
///
/// The `HdyTabBar` widget is a tab bar that can be used with conjunction with
/// `TabView`.
///
/// `HdyTabBar` can autohide and can optionally contain action widgets on both
/// sides of the tabs.
///
/// When there's not enough space to show all the tabs, `HdyTabBar` will scroll
/// them. Pinned tabs always stay visible and aren't a part of the scrollable
/// area.
///
/// ## CSS nodes
///
/// `HdyTabBar` has a single CSS node with name `tabbar`.
pub const TabBar = opaque {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.TabBarClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether tabs automatically hide.
        ///
        /// If set to `TRUE`, the tab bar disappears when the associated
        /// `TabView` has 0 or 1 tab, no pinned tabs, and no tab is being
        /// transferred.
        ///
        /// See `TabBar.properties.tabs_revealed`.
        pub const autohide = struct {
            pub const name = "autohide";

            pub const Type = c_int;
        };

        /// The widget shown after the tabs.
        pub const end_action_widget = struct {
            pub const name = "end-action-widget";

            pub const Type = ?*gtk.Widget;
        };

        /// Whether tabs should expand.
        ///
        /// If set to `TRUE`, the tabs will always vary width filling the whole width
        /// when possible, otherwise tabs will always have the minimum possible size.
        pub const expand_tabs = struct {
            pub const name = "expand-tabs";

            pub const Type = c_int;
        };

        /// Extra drag destination targets.
        ///
        /// Allows to drag arbitrary content onto tabs, for example URLs in a web
        /// browser.
        ///
        /// If a tab is hovered for a certain period of time while dragging the
        /// content, it will be automatically selected.
        ///
        /// After content is dropped, the `TabBar.signals.extra_drag_data_received`
        /// signal can be used to retrieve and process the drag data.
        pub const extra_drag_dest_targets = struct {
            pub const name = "extra-drag-dest-targets";

            pub const Type = ?*gtk.TargetList;
        };

        /// Whether tabs use inverted layout.
        ///
        /// If set to `TRUE`, non-pinned tabs will have the close button at the
        /// beginning and the indicator at the end rather than the opposite.
        pub const inverted = struct {
            pub const name = "inverted";

            pub const Type = c_int;
        };

        /// Whether the tab bar is overflowing.
        ///
        /// If set to `TRUE`, all tabs cannot be displayed at once and require
        /// scrolling.
        pub const is_overflowing = struct {
            pub const name = "is-overflowing";

            pub const Type = c_int;
        };

        /// The widget shown before the tabs.
        pub const start_action_widget = struct {
            pub const name = "start-action-widget";

            pub const Type = ?*gtk.Widget;
        };

        /// Whether tabs are currently revealed.
        ///
        /// See `TabBar.properties.autohide`.
        pub const tabs_revealed = struct {
            pub const name = "tabs-revealed";

            pub const Type = c_int;
        };

        /// The `TabView` the tab bar controls.
        pub const view = struct {
            pub const name = "view";

            pub const Type = ?*handy.TabView;
        };
    };

    pub const signals = struct {
        /// Emitted when content allowed via `TabBar.properties.extra_drag_dest_targets`
        /// is dropped onto a tab.
        ///
        /// See `gtk.Widget.signals.drag_data_received`.
        pub const extra_drag_data_received = struct {
            pub const name = "extra-drag-data-received";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_page: *handy.TabPage, p_context: *gdk.DragContext, p_data: *gtk.SelectionData, p_info: c_uint, p_time: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(TabBar, p_instance))),
                    gobject.signalLookup("extra-drag-data-received", TabBar.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `HdyTabBar` widget.
    extern fn hdy_tab_bar_new() *handy.TabBar;
    pub const new = hdy_tab_bar_new;

    /// Gets whether the tabs automatically hide.
    extern fn hdy_tab_bar_get_autohide(p_self: *TabBar) c_int;
    pub const getAutohide = hdy_tab_bar_get_autohide;

    /// Gets the widget shown after the tabs.
    extern fn hdy_tab_bar_get_end_action_widget(p_self: *TabBar) ?*gtk.Widget;
    pub const getEndActionWidget = hdy_tab_bar_get_end_action_widget;

    /// Gets whether tabs should expand.
    extern fn hdy_tab_bar_get_expand_tabs(p_self: *TabBar) c_int;
    pub const getExpandTabs = hdy_tab_bar_get_expand_tabs;

    /// Gets extra drag destination targets.
    extern fn hdy_tab_bar_get_extra_drag_dest_targets(p_self: *TabBar) ?*gtk.TargetList;
    pub const getExtraDragDestTargets = hdy_tab_bar_get_extra_drag_dest_targets;

    /// Gets whether tabs use inverted layout.
    extern fn hdy_tab_bar_get_inverted(p_self: *TabBar) c_int;
    pub const getInverted = hdy_tab_bar_get_inverted;

    /// Gets whether `self` is overflowing.
    extern fn hdy_tab_bar_get_is_overflowing(p_self: *TabBar) c_int;
    pub const getIsOverflowing = hdy_tab_bar_get_is_overflowing;

    /// Gets the widget shown before the tabs.
    extern fn hdy_tab_bar_get_start_action_widget(p_self: *TabBar) ?*gtk.Widget;
    pub const getStartActionWidget = hdy_tab_bar_get_start_action_widget;

    /// Gets the value of the `TabBar.properties.tabs_revealed` property.
    extern fn hdy_tab_bar_get_tabs_revealed(p_self: *TabBar) c_int;
    pub const getTabsRevealed = hdy_tab_bar_get_tabs_revealed;

    /// Gets the `TabView` `self` controls.
    extern fn hdy_tab_bar_get_view(p_self: *TabBar) ?*handy.TabView;
    pub const getView = hdy_tab_bar_get_view;

    /// Sets whether the tabs automatically hide.
    ///
    /// If `autohide` is `TRUE`, the tab bar disappears when the associated
    /// `TabView` has 0 or 1 tab, no pinned tabs, and no tab is being
    /// transferred.
    ///
    /// Autohide is enabled by default.
    ///
    /// See `TabBar.properties.tabs_revealed`.
    extern fn hdy_tab_bar_set_autohide(p_self: *TabBar, p_autohide: c_int) void;
    pub const setAutohide = hdy_tab_bar_set_autohide;

    /// Sets the widget to show after the tabs.
    extern fn hdy_tab_bar_set_end_action_widget(p_self: *TabBar, p_widget: ?*gtk.Widget) void;
    pub const setEndActionWidget = hdy_tab_bar_set_end_action_widget;

    /// Sets whether tabs should expand.
    ///
    /// If `expand_tabs` is `TRUE`, the tabs will always vary width filling the whole
    /// width when possible, otherwise tabs will always have the minimum possible
    /// size.
    ///
    /// Expand is enabled by default.
    extern fn hdy_tab_bar_set_expand_tabs(p_self: *TabBar, p_expand_tabs: c_int) void;
    pub const setExpandTabs = hdy_tab_bar_set_expand_tabs;

    /// Sets extra drag destination targets.
    ///
    /// This allows to drag arbitrary content onto tabs, for example URLs in a web
    /// browser.
    ///
    /// If a tab is hovered for a certain period of time while dragging the content,
    /// it will be automatically selected.
    ///
    /// After content is dropped, the `TabBar.signals.extra_drag_data_received`
    /// signal can be used to retrieve and process the drag data.
    extern fn hdy_tab_bar_set_extra_drag_dest_targets(p_self: *TabBar, p_extra_drag_dest_targets: ?*gtk.TargetList) void;
    pub const setExtraDragDestTargets = hdy_tab_bar_set_extra_drag_dest_targets;

    /// Sets whether tabs tabs use inverted layout.
    ///
    /// If `inverted` is `TRUE`, non-pinned tabs will have the close button at the
    /// beginning and the indicator at the end rather than the opposite.
    extern fn hdy_tab_bar_set_inverted(p_self: *TabBar, p_inverted: c_int) void;
    pub const setInverted = hdy_tab_bar_set_inverted;

    /// Sets the widget to show before the tabs.
    extern fn hdy_tab_bar_set_start_action_widget(p_self: *TabBar, p_widget: ?*gtk.Widget) void;
    pub const setStartActionWidget = hdy_tab_bar_set_start_action_widget;

    /// Sets the `TabView` `self` controls.
    extern fn hdy_tab_bar_set_view(p_self: *TabBar, p_view: ?*handy.TabView) void;
    pub const setView = hdy_tab_bar_set_view;

    extern fn hdy_tab_bar_get_type() usize;
    pub const getGObjectType = hdy_tab_bar_get_type;

    extern fn g_object_ref(p_self: *handy.TabBar) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.TabBar) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *TabBar, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An auxiliary class used by `TabView`.
pub const TabPage = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = handy.TabPageClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The child of the page.
        pub const child = struct {
            pub const name = "child";

            pub const Type = ?*gtk.Widget;
        };

        /// The icon of the page.
        ///
        /// `TabBar` displays the icon next to the title.
        ///
        /// It will not show the icon if `TabPage.properties.loading` is set to `TRUE`,
        /// or if the page is pinned and [propertyTabPage:indicator-icon] is set.
        pub const icon = struct {
            pub const name = "icon";

            pub const Type = ?*gio.Icon;
        };

        /// Whether the indicator icon is activatable.
        ///
        /// If set to `TRUE`, `TabView.signals.indicator_activated` will be emitted
        /// when the indicator icon is clicked.
        ///
        /// If `TabPage.properties.indicator_icon` is not set, does nothing.
        pub const indicator_activatable = struct {
            pub const name = "indicator-activatable";

            pub const Type = c_int;
        };

        /// An indicator icon for the page.
        ///
        /// A common use case is an audio or camera indicator in a web browser.
        ///
        /// `TabPage` will show it at the beginning of the tab, alongside icon
        /// representing `TabPage.properties.icon` or loading spinner.
        ///
        /// If the page is pinned, the indicator will be shown instead of icon or
        /// spinner.
        ///
        /// If `TabPage.properties.indicator_activatable` is set to `TRUE`, the indicator
        /// icon can act as a button.
        pub const indicator_icon = struct {
            pub const name = "indicator-icon";

            pub const Type = ?*gio.Icon;
        };

        /// Whether the page is loading.
        ///
        /// If set to `TRUE`, `TabBar` will display a spinner in place of icon.
        ///
        /// If the page is pinned and `TabPage.properties.indicator_icon` is set, the
        /// loading status will not be visible.
        pub const loading = struct {
            pub const name = "loading";

            pub const Type = c_int;
        };

        /// Whether the page needs attention.
        ///
        /// `TabBar` will display a glow under the tab representing the page if
        /// set to `TRUE`. If the tab is not visible, the corresponding edge of the tab
        /// bar will be highlighted.
        pub const needs_attention = struct {
            pub const name = "needs-attention";

            pub const Type = c_int;
        };

        /// The parent page of the page.
        ///
        /// See `TabView.addPage` and `TabView.closePage`.
        pub const parent = struct {
            pub const name = "parent";

            pub const Type = ?*handy.TabPage;
        };

        /// Whether the page is pinned.
        ///
        /// See `TabView.setPagePinned`.
        pub const pinned = struct {
            pub const name = "pinned";

            pub const Type = c_int;
        };

        /// Whether the page is selected.
        pub const selected = struct {
            pub const name = "selected";

            pub const Type = c_int;
        };

        /// The title of the page.
        ///
        /// `TabBar` will display it in the center of the tab unless it's pinned,
        /// and will use it as a tooltip unless `TabPage.properties.tooltip` is set.
        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };

        /// The tooltip of the page.
        ///
        /// The tooltip can be marked up with the Pango text markup language.
        ///
        /// If not set, `TabBar` will use `TabPage.properties.title` as a tooltip
        /// instead.
        pub const tooltip = struct {
            pub const name = "tooltip";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Gets the child of `self`.
    extern fn hdy_tab_page_get_child(p_self: *TabPage) *gtk.Widget;
    pub const getChild = hdy_tab_page_get_child;

    /// Gets the icon of `self`.
    extern fn hdy_tab_page_get_icon(p_self: *TabPage) ?*gio.Icon;
    pub const getIcon = hdy_tab_page_get_icon;

    /// Gets whether the indicator of `self` is activatable.
    extern fn hdy_tab_page_get_indicator_activatable(p_self: *TabPage) c_int;
    pub const getIndicatorActivatable = hdy_tab_page_get_indicator_activatable;

    /// Gets the indicator icon of `self`.
    extern fn hdy_tab_page_get_indicator_icon(p_self: *TabPage) ?*gio.Icon;
    pub const getIndicatorIcon = hdy_tab_page_get_indicator_icon;

    /// Gets whether `self` is loading.
    extern fn hdy_tab_page_get_loading(p_self: *TabPage) c_int;
    pub const getLoading = hdy_tab_page_get_loading;

    /// Gets whether `self` needs attention.
    extern fn hdy_tab_page_get_needs_attention(p_self: *TabPage) c_int;
    pub const getNeedsAttention = hdy_tab_page_get_needs_attention;

    /// Gets the parent page of `self`.
    extern fn hdy_tab_page_get_parent(p_self: *TabPage) ?*handy.TabPage;
    pub const getParent = hdy_tab_page_get_parent;

    /// Gets whether `self` is pinned.
    extern fn hdy_tab_page_get_pinned(p_self: *TabPage) c_int;
    pub const getPinned = hdy_tab_page_get_pinned;

    /// Gets whether `self` is selected.
    extern fn hdy_tab_page_get_selected(p_self: *TabPage) c_int;
    pub const getSelected = hdy_tab_page_get_selected;

    /// Gets the title of `self`.
    extern fn hdy_tab_page_get_title(p_self: *TabPage) ?[*:0]const u8;
    pub const getTitle = hdy_tab_page_get_title;

    /// Gets the tooltip of `self`.
    extern fn hdy_tab_page_get_tooltip(p_self: *TabPage) ?[*:0]const u8;
    pub const getTooltip = hdy_tab_page_get_tooltip;

    /// Sets the icon of `self`.
    extern fn hdy_tab_page_set_icon(p_self: *TabPage, p_icon: ?*gio.Icon) void;
    pub const setIcon = hdy_tab_page_set_icon;

    /// Sets whether the indicator of `self` is activatable.
    extern fn hdy_tab_page_set_indicator_activatable(p_self: *TabPage, p_activatable: c_int) void;
    pub const setIndicatorActivatable = hdy_tab_page_set_indicator_activatable;

    /// Sets the indicator icon of `self`.
    extern fn hdy_tab_page_set_indicator_icon(p_self: *TabPage, p_indicator_icon: ?*gio.Icon) void;
    pub const setIndicatorIcon = hdy_tab_page_set_indicator_icon;

    /// Sets whether `self` is loading.
    extern fn hdy_tab_page_set_loading(p_self: *TabPage, p_loading: c_int) void;
    pub const setLoading = hdy_tab_page_set_loading;

    /// Sets whether `self` needs attention.
    extern fn hdy_tab_page_set_needs_attention(p_self: *TabPage, p_needs_attention: c_int) void;
    pub const setNeedsAttention = hdy_tab_page_set_needs_attention;

    /// Sets the title of `self`.
    extern fn hdy_tab_page_set_title(p_self: *TabPage, p_title: ?[*:0]const u8) void;
    pub const setTitle = hdy_tab_page_set_title;

    /// Sets the tooltip of `self`.
    extern fn hdy_tab_page_set_tooltip(p_self: *TabPage, p_tooltip: ?[*:0]const u8) void;
    pub const setTooltip = hdy_tab_page_set_tooltip;

    extern fn hdy_tab_page_get_type() usize;
    pub const getGObjectType = hdy_tab_page_get_type;

    extern fn g_object_ref(p_self: *handy.TabPage) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.TabPage) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *TabPage, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A dynamic tabbed container.
///
/// `HdyTabView` is a container which shows one child at a time. While it
/// provides keyboard shortcuts for switching between pages, it does not provide
/// a visible tab bar and relies on external widgets for that, such as
/// `TabBar`.
///
/// `HdyTabView` maintains a `TabPage` object for each page,which holds
/// additional per-page properties. You can obtain the `TabPage` for a page
/// with `TabView.getPage`, and as return value for
/// `TabView.append` and other functions for adding children.
///
/// `HdyTabView` only aims to be useful for dynamic tabs in multi-window
/// document-based applications, such as web browsers, file managers, text
/// editors or terminals. It does not aim to replace `gtk.Notebook` for use
/// cases such as tabbed dialogs.
///
/// As such, it does not support disabling page reordering or detaching, or
/// adding children via `gtk.Buildable`.
///
/// ## CSS nodes
///
/// `HdyTabView` has a main CSS node with the name `tabview`.
///
/// It contains the subnode overlay, which contains subnodes stack and widget.
/// The stack subnode contains the added pages.
///
/// ```
/// tabview
/// ╰── overlay
///     ├── stack
///     │   ╰── [ Children ]
///     ╰── widget
/// ```
pub const TabView = opaque {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.TabViewClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Default page icon.
        ///
        /// If a page doesn't provide its own icon via `TabPage.properties.icon`, default
        /// icon may be used instead for contexts where having an icon is necessary.
        ///
        /// `TabBar` will use default icon for pinned tabs in case the page is
        /// not loading, doesn't have an icon and an indicator. Default icon is never
        /// used for tabs that aren't pinned.
        pub const default_icon = struct {
            pub const name = "default-icon";

            pub const Type = ?*gio.Icon;
        };

        /// Whether a page is being transferred.
        ///
        /// This property will be set to `TRUE` when a drag-n-drop tab transfer starts
        /// on any `TabView`, and to `FALSE` after it ends.
        ///
        /// During the transfer, children cannot receive pointer input and a tab can be
        /// safely dropped on the tab view.
        pub const is_transferring_page = struct {
            pub const name = "is-transferring-page";

            pub const Type = c_int;
        };

        /// Tab context menu model.
        ///
        /// When a context menu is shown for a tab, it will be constructed from the
        /// provided menu model. Use `TabView.signals.setup_menu` signal to set up the
        /// menu actions for the particular tab.
        pub const menu_model = struct {
            pub const name = "menu-model";

            pub const Type = ?*gio.MenuModel;
        };

        /// The number of pages in the tab view.
        pub const n_pages = struct {
            pub const name = "n-pages";

            pub const Type = c_int;
        };

        /// The number of pinned pages in the tab view.
        ///
        /// See `TabView.setPagePinned`.
        pub const n_pinned_pages = struct {
            pub const name = "n-pinned-pages";

            pub const Type = c_int;
        };

        /// The currently selected page.
        pub const selected_page = struct {
            pub const name = "selected-page";

            pub const Type = ?*handy.TabPage;
        };

        /// Tab shortcut widget.
        ///
        /// Has the following shortcuts:
        ///
        /// * <kbd>Ctrl</kbd>+<kbd>Page Up</kbd> - switch to the previous page
        /// * <kbd>Ctrl</kbd>+<kbd>Page Down</kbd> - switch to the next page
        /// * <kbd>Ctrl</kbd>+<kbd>Home</kbd> - switch to the first page
        /// * <kbd>Ctrl</kbd>+<kbd>End</kbd> - switch to the last page
        /// * <kbd>Ctrl</kbd>+<kbd>Shift</kbd>+<kbd>Page Up</kbd> - move the current page
        ///     backward
        /// * <kbd>Ctrl</kbd>+<kbd>Shift</kbd>+<kbd>Page Down</kbd> - move the current
        ///     page forward
        /// * <kbd>Ctrl</kbd>+<kbd>Shift</kbd>+<kbd>Home</kbd> - move the current page at
        ///     the start
        /// * <kbd>Ctrl</kbd>+<kbd>Shift</kbd>+<kbd>End</kbd> - move the current page at
        ///      the end
        /// * <kbd>Ctrl</kbd>+<kbd>Tab</kbd> - switch to the next page, with looping
        /// * <kbd>Ctrl</kbd>+<kbd>Shift</kbd>+<kbd>Tab</kbd> - switch to the previous
        ///     page, with looping
        /// * <kbd>Alt</kbd>+<kbd>1</kbd>⋯<kbd>9</kbd> - switch to pages 1-9
        /// * <kbd>Alt</kbd>+<kbd>0</kbd> - switch to page 10
        ///
        /// These shortcuts are always available on `self`, this property is useful if
        /// they should be available globally.
        pub const shortcut_widget = struct {
            pub const name = "shortcut-widget";

            pub const Type = ?*gtk.Widget;
        };
    };

    pub const signals = struct {
        /// Emitted after `TabView.closePage` has been called for `page`.
        ///
        /// The handler is expected to call `TabView.closePageFinish` to
        /// confirm or reject the closing.
        ///
        /// The default handler will immediately confirm closing for non-pinned pages,
        /// or reject it for pinned pages, equivalent to the following example:
        ///
        /// ```c
        /// static gboolean
        /// close_page_cb (HdyTabView *view,
        ///                HdyTabPage *page,
        ///                gpointer    user_data)
        /// {
        ///   hdy_tab_view_close_page_finish (view, page, !hdy_tab_page_get_pinned (page));
        ///
        ///   return GDK_EVENT_STOP;
        /// }
        /// ```
        ///
        /// The `TabView.closePageFinish` doesn't have to happen during the
        /// handler, so can be used to do asynchronous checks before confirming the
        /// closing.
        ///
        /// A typical reason to connect to this signal is to show a confirmation dialog
        /// for closing a tab.
        pub const close_page = struct {
            pub const name = "close-page";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_page: *handy.TabPage, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(TabView, p_instance))),
                    gobject.signalLookup("close-page", TabView.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a tab should be transferred into a new window.
        ///
        /// This can happen after a tab has been dropped on desktop.
        ///
        /// The signal handler is expected to create a new window, position it as
        /// needed and return its `HdyTabView`that the page will be transferred into.
        pub const create_window = struct {
            pub const name = "create-window";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) ?*handy.TabView, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(TabView, p_instance))),
                    gobject.signalLookup("create-window", TabView.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted after the indicator icon on `page` has been activated.
        ///
        /// See `TabPage.properties.indicator_icon` and
        /// `TabPage.properties.indicator_activatable`.
        pub const indicator_activated = struct {
            pub const name = "indicator-activated";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_page: *handy.TabPage, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(TabView, p_instance))),
                    gobject.signalLookup("indicator-activated", TabView.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a page has been created or transferred to the view.
        ///
        /// A typical reason to connect to this signal would be to connect to page
        /// signals for things such as updating window title.
        pub const page_attached = struct {
            pub const name = "page-attached";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_page: *handy.TabPage, p_position: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(TabView, p_instance))),
                    gobject.signalLookup("page-attached", TabView.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a page has been removed or transferred to another view.
        ///
        /// A typical reason to connect to this signal would be to disconnect signal
        /// handlers connected in the `TabView.signals.page_attached` handler.
        ///
        /// It is important not to try and destroy the page child in the handler of
        /// this function as the child might merely be moved to another window; use
        /// child dispose handler for that or do it in sync with your
        /// `TabView.closePageFinish` calls.
        pub const page_detached = struct {
            pub const name = "page-detached";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_page: *handy.TabPage, p_position: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(TabView, p_instance))),
                    gobject.signalLookup("page-detached", TabView.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted after `page` has been reordered to `position`.
        pub const page_reordered = struct {
            pub const name = "page-reordered";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_page: *handy.TabPage, p_position: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(TabView, p_instance))),
                    gobject.signalLookup("page-reordered", TabView.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a context menu is opened or closed for `page`.
        ///
        /// If the menu has been closed, `page` will be set to `NULL`.
        ///
        /// It can be used to set up menu actions before showing the menu, for example
        /// disable actions not applicable to `page`.
        pub const setup_menu = struct {
            pub const name = "setup-menu";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_page: *handy.TabPage, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(TabView, p_instance))),
                    gobject.signalLookup("setup-menu", TabView.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `HdyTabView`.
    extern fn hdy_tab_view_new() *handy.TabView;
    pub const new = hdy_tab_view_new;

    /// Adds `child` to `self` with `parent` as the parent.
    ///
    /// This function can be used to automatically position new pages, and to select
    /// the correct page when this page is closed while being selected (see
    /// `TabView.closePage`.
    ///
    /// If `parent` is `NULL`, this function is equivalent to `TabView.append`.
    extern fn hdy_tab_view_add_page(p_self: *TabView, p_child: *gtk.Widget, p_parent: ?*handy.TabPage) *handy.TabPage;
    pub const addPage = hdy_tab_view_add_page;

    /// Inserts `child` as the last non-pinned page.
    extern fn hdy_tab_view_append(p_self: *TabView, p_child: *gtk.Widget) *handy.TabPage;
    pub const append = hdy_tab_view_append;

    /// Inserts `child` as the last pinned page.
    extern fn hdy_tab_view_append_pinned(p_self: *TabView, p_child: *gtk.Widget) *handy.TabPage;
    pub const appendPinned = hdy_tab_view_append_pinned;

    /// Requests to close all pages other than `page`.
    extern fn hdy_tab_view_close_other_pages(p_self: *TabView, p_page: *handy.TabPage) void;
    pub const closeOtherPages = hdy_tab_view_close_other_pages;

    /// Requests to close `page`.
    ///
    /// Calling this function will result in `TabView.signals.close_page` signal
    /// being emitted for `page`. Closing the page can then be confirmed or denied via
    /// `TabView.closePageFinish`.
    ///
    /// If the page is waiting for a `TabView.closePageFinish` call, this
    /// function will do nothing.
    ///
    /// The default handler for `TabView.signals.close_page` will immediately confirm
    /// closing the page if it's non-pinned, or reject it if it's pinned. This
    /// behavior can be changed by registering your own handler for that signal.
    ///
    /// If `page` was selected, another page will be selected instead:
    ///
    /// If the `TabPage.properties.parent` value is `NULL`, the next page will be
    /// selected when possible, or if the page was already last, the previous page
    /// will be selected instead.
    ///
    /// If it's not `NULL`, the previous page will be selected if it's a descendant
    /// (possibly indirect) of the parent. If both the previous page and the parent
    /// are pinned, the parent will be selected instead.
    extern fn hdy_tab_view_close_page(p_self: *TabView, p_page: *handy.TabPage) void;
    pub const closePage = hdy_tab_view_close_page;

    /// Completes a `TabView.closePage` call for `page`.
    ///
    /// If `confirm` is `TRUE`, `page` will be closed. If it's `FALSE`, ite will be
    /// reverted to its previous state and `TabView.closePage` can be called
    /// for it again.
    ///
    /// This function should not be called unless a custom handler for
    /// `TabView.signals.close_page` is used.
    extern fn hdy_tab_view_close_page_finish(p_self: *TabView, p_page: *handy.TabPage, p_confirm: c_int) void;
    pub const closePageFinish = hdy_tab_view_close_page_finish;

    /// Requests to close all pages after `page`.
    extern fn hdy_tab_view_close_pages_after(p_self: *TabView, p_page: *handy.TabPage) void;
    pub const closePagesAfter = hdy_tab_view_close_pages_after;

    /// Requests to close all pages before `page`.
    extern fn hdy_tab_view_close_pages_before(p_self: *TabView, p_page: *handy.TabPage) void;
    pub const closePagesBefore = hdy_tab_view_close_pages_before;

    /// Gets default icon of `self`.
    extern fn hdy_tab_view_get_default_icon(p_self: *TabView) *gio.Icon;
    pub const getDefaultIcon = hdy_tab_view_get_default_icon;

    /// Whether a page is being transferred.
    ///
    /// Gets the value of `TabView.properties.is_transferring_page` property.
    extern fn hdy_tab_view_get_is_transferring_page(p_self: *TabView) c_int;
    pub const getIsTransferringPage = hdy_tab_view_get_is_transferring_page;

    /// Gets the tab context menu model for `self`.
    extern fn hdy_tab_view_get_menu_model(p_self: *TabView) ?*gio.MenuModel;
    pub const getMenuModel = hdy_tab_view_get_menu_model;

    /// Gets the number of pages in `self`.
    extern fn hdy_tab_view_get_n_pages(p_self: *TabView) c_int;
    pub const getNPages = hdy_tab_view_get_n_pages;

    /// Gets the number of pinned pages in `self`.
    ///
    /// See `TabView.setPagePinned`.
    extern fn hdy_tab_view_get_n_pinned_pages(p_self: *TabView) c_int;
    pub const getNPinnedPages = hdy_tab_view_get_n_pinned_pages;

    /// Gets the `TabPage` representing the child at `position`.
    extern fn hdy_tab_view_get_nth_page(p_self: *TabView, p_position: c_int) *handy.TabPage;
    pub const getNthPage = hdy_tab_view_get_nth_page;

    /// Gets the `TabPage` object representing `child`.
    extern fn hdy_tab_view_get_page(p_self: *TabView, p_child: *gtk.Widget) *handy.TabPage;
    pub const getPage = hdy_tab_view_get_page;

    /// Finds the position of `page` in `self`, starting from 0.
    extern fn hdy_tab_view_get_page_position(p_self: *TabView, p_page: *handy.TabPage) c_int;
    pub const getPagePosition = hdy_tab_view_get_page_position;

    /// Returns a `gio.ListModel` containing the pages of `self`.
    ///
    /// This model can be used to keep an up to date view of the pages.
    extern fn hdy_tab_view_get_pages(p_self: *TabView) *gio.ListModel;
    pub const getPages = hdy_tab_view_get_pages;

    /// Gets the currently selected page in `self`.
    extern fn hdy_tab_view_get_selected_page(p_self: *TabView) ?*handy.TabPage;
    pub const getSelectedPage = hdy_tab_view_get_selected_page;

    /// Gets the shortcut widget for `self`.
    extern fn hdy_tab_view_get_shortcut_widget(p_self: *TabView) ?*gtk.Widget;
    pub const getShortcutWidget = hdy_tab_view_get_shortcut_widget;

    /// Inserts a non-pinned page at `position`.
    ///
    /// It's an error to try to insert a page before a pinned page, in that case
    /// `TabView.insertPinned` should be used instead.
    extern fn hdy_tab_view_insert(p_self: *TabView, p_child: *gtk.Widget, p_position: c_int) *handy.TabPage;
    pub const insert = hdy_tab_view_insert;

    /// Inserts a pinned page at `position`.
    ///
    /// It's an error to try to insert a pinned page after a non-pinned page, in that
    /// case `TabView.insert` should be used instead.
    extern fn hdy_tab_view_insert_pinned(p_self: *TabView, p_child: *gtk.Widget, p_position: c_int) *handy.TabPage;
    pub const insertPinned = hdy_tab_view_insert_pinned;

    /// Inserts `child` as the first non-pinned page.
    extern fn hdy_tab_view_prepend(p_self: *TabView, p_child: *gtk.Widget) *handy.TabPage;
    pub const prepend = hdy_tab_view_prepend;

    /// Inserts `child` as the first pinned page.
    extern fn hdy_tab_view_prepend_pinned(p_self: *TabView, p_child: *gtk.Widget) *handy.TabPage;
    pub const prependPinned = hdy_tab_view_prepend_pinned;

    /// Reorders `page` to before its previous page if possible.
    extern fn hdy_tab_view_reorder_backward(p_self: *TabView, p_page: *handy.TabPage) c_int;
    pub const reorderBackward = hdy_tab_view_reorder_backward;

    /// Reorders `page` to the first possible position.
    extern fn hdy_tab_view_reorder_first(p_self: *TabView, p_page: *handy.TabPage) c_int;
    pub const reorderFirst = hdy_tab_view_reorder_first;

    /// Reorders `page` to after its next page if possible.
    extern fn hdy_tab_view_reorder_forward(p_self: *TabView, p_page: *handy.TabPage) c_int;
    pub const reorderForward = hdy_tab_view_reorder_forward;

    /// Reorders `page` to the last possible position.
    extern fn hdy_tab_view_reorder_last(p_self: *TabView, p_page: *handy.TabPage) c_int;
    pub const reorderLast = hdy_tab_view_reorder_last;

    /// Reorders `page` to `position`.
    ///
    /// It's a programmer error to try to reorder a pinned page after a non-pinned
    /// one, or a non-pinned page before a pinned one.
    extern fn hdy_tab_view_reorder_page(p_self: *TabView, p_page: *handy.TabPage, p_position: c_int) c_int;
    pub const reorderPage = hdy_tab_view_reorder_page;

    /// Selects the page after the currently selected page.
    ///
    /// If the last page was already selected, this function does nothing.
    extern fn hdy_tab_view_select_next_page(p_self: *TabView) c_int;
    pub const selectNextPage = hdy_tab_view_select_next_page;

    /// Selects the page before the currently selected page.
    ///
    /// If the first page was already selected, this function does nothing.
    extern fn hdy_tab_view_select_previous_page(p_self: *TabView) c_int;
    pub const selectPreviousPage = hdy_tab_view_select_previous_page;

    /// Sets default page icon for `self`.
    ///
    /// If a page doesn't provide its own icon via `TabPage.properties.icon`, default
    /// icon may be used instead for contexts where having an icon is necessary.
    ///
    /// `TabBar` will use default icon for pinned tabs in case the page is not
    /// loading, doesn't have an icon and an indicator. Default icon is never used
    /// for tabs that aren't pinned.
    ///
    /// By default, `hdy-tab-icon-missing-symbolic` icon is used.
    extern fn hdy_tab_view_set_default_icon(p_self: *TabView, p_default_icon: *gio.Icon) void;
    pub const setDefaultIcon = hdy_tab_view_set_default_icon;

    /// Sets the tab context menu model for `self`.
    ///
    /// When a context menu is shown for a tab, it will be constructed from the
    /// provided menu model. Use `TabView.signals.setup_menu` signal to set up the
    /// menu actions for the particular tab.
    extern fn hdy_tab_view_set_menu_model(p_self: *TabView, p_menu_model: ?*gio.MenuModel) void;
    pub const setMenuModel = hdy_tab_view_set_menu_model;

    /// Pins or unpins `page`.
    ///
    /// Pinned pages are guaranteed to be placed before all non-pinned pages; at any
    /// given moment the first `TabView.properties.n_pinned_pages` pages in `self` are
    /// guaranteed to be pinned.
    ///
    /// When a page is pinned or unpinned, it's automatically reordered: pinning a
    /// page moves it after other pinned pages; unpinning a page moves it before
    /// other non-pinned pages.
    ///
    /// Pinned pages can still be reordered between each other.
    ///
    /// `TabBar` will display pinned pages in a compact form, never showing the
    /// title or close button, and only showing a single icon, selected in the
    /// following order:
    ///
    /// 1. `TabPage.properties.indicator_icon`
    /// 2. A spinner if `TabPage.properties.loading` is `TRUE`
    /// 3. `TabPage.properties.icon`
    /// 4. `TabView.properties.default_icon`
    ///
    /// Pinned pages cannot be closed by default, see `TabView.signals.close_page`
    /// for how to override that behavior.
    extern fn hdy_tab_view_set_page_pinned(p_self: *TabView, p_page: *handy.TabPage, p_pinned: c_int) void;
    pub const setPagePinned = hdy_tab_view_set_page_pinned;

    /// Sets the currently selected page in `self`.
    extern fn hdy_tab_view_set_selected_page(p_self: *TabView, p_selected_page: *handy.TabPage) void;
    pub const setSelectedPage = hdy_tab_view_set_selected_page;

    /// Sets the shortcut widget for `self`.
    extern fn hdy_tab_view_set_shortcut_widget(p_self: *TabView, p_widget: ?*gtk.Widget) void;
    pub const setShortcutWidget = hdy_tab_view_set_shortcut_widget;

    /// Transfers `page` from `self` to `other_view`.
    ///
    /// The `page` object will be reused.
    ///
    /// It's a programmer error to try to insert a pinned page after a non-pinned
    /// one, or a non-pinned page before a pinned one.
    extern fn hdy_tab_view_transfer_page(p_self: *TabView, p_page: *handy.TabPage, p_other_view: *handy.TabView, p_position: c_int) void;
    pub const transferPage = hdy_tab_view_transfer_page;

    extern fn hdy_tab_view_get_type() usize;
    pub const getGObjectType = hdy_tab_view_get_type;

    extern fn g_object_ref(p_self: *handy.TabView) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.TabView) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *TabView, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A simple title bar container.
///
/// `HdyTitleBar` is meant to be used as the top-level widget of your window's
/// title bar. It will be drawn with the same style as a `gtk.HeaderBar`
/// but it won't force a widget layout on you: you can put whatever widget you
/// want in it, including a `gtk.HeaderBar`.
///
/// `HdyTitleBar` becomes really useful when you want to animate header bars,
/// like an adaptive application using `Leaflet` would do.
///
/// `HdyTitleBar` has been deprecated, header bars can be animated without it
/// when placed inside `Window` or `ApplicationWindow`.
///
/// ## CSS nodes
///
/// `HdyTitleBar` has a single CSS node with name `headerbar`.
pub const TitleBar = opaque {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.TitleBarClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether or not the title bar is in selection mode.
        pub const selection_mode = struct {
            pub const name = "selection-mode";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyTitleBar`.
    extern fn hdy_title_bar_new() *handy.TitleBar;
    pub const new = hdy_title_bar_new;

    /// Returns whether whether `self` is in selection mode.
    extern fn hdy_title_bar_get_selection_mode(p_self: *TitleBar) c_int;
    pub const getSelectionMode = hdy_title_bar_get_selection_mode;

    /// Sets whether `self` is in selection mode.
    extern fn hdy_title_bar_set_selection_mode(p_self: *TitleBar, p_selection_mode: c_int) void;
    pub const setSelectionMode = hdy_title_bar_set_selection_mode;

    extern fn hdy_title_bar_get_type() usize;
    pub const getGObjectType = hdy_title_bar_get_type;

    extern fn g_object_ref(p_self: *handy.TitleBar) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.TitleBar) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *TitleBar, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An object representing a `gobject.Value`.
///
/// The `HdyValueObject` object represents a `gobject.Value`, allowing it
/// to be used with `gio.ListModel`.
pub const ValueObject = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = handy.ValueObjectClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The contained value.
        pub const value = struct {
            pub const name = "value";

            pub const Type = ?*gobject.Value;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyValueObject`.
    extern fn hdy_value_object_new(p_value: *const gobject.Value) *handy.ValueObject;
    pub const new = hdy_value_object_new;

    /// Creates a new `HdyValueObject`.
    ///
    /// This is a convenience method which uses the `G_VALUE_COLLECT` macro
    /// internally.
    extern fn hdy_value_object_new_collect(p_type: usize, ...) *handy.ValueObject;
    pub const newCollect = hdy_value_object_new_collect;

    /// Creates a new `HdyValueObject`.
    ///
    /// This is a convenience method to create a `ValueObject` that stores a
    /// string.
    extern fn hdy_value_object_new_string(p_string: [*:0]const u8) *handy.ValueObject;
    pub const newString = hdy_value_object_new_string;

    /// Creates a new `HdyValueObject`.
    ///
    /// This is a convenience method to create a `ValueObject` that stores a
    /// string taking ownership of it.
    extern fn hdy_value_object_new_take_string(p_string: [*:0]u8) *handy.ValueObject;
    pub const newTakeString = hdy_value_object_new_take_string;

    /// Copy data from the contained `gobject.Value` into `dest`.
    extern fn hdy_value_object_copy_value(p_value: *ValueObject, p_dest: *gobject.Value) void;
    pub const copyValue = hdy_value_object_copy_value;

    /// Gets a copy of the contained string if the value is of type `G_TYPE_STRING`.
    extern fn hdy_value_object_dup_string(p_value: *ValueObject) [*:0]u8;
    pub const dupString = hdy_value_object_dup_string;

    /// Returns the contained string if the value is of type `G_TYPE_STRING`.
    extern fn hdy_value_object_get_string(p_value: *ValueObject) [*:0]const u8;
    pub const getString = hdy_value_object_get_string;

    /// Return the contained value.
    extern fn hdy_value_object_get_value(p_value: *ValueObject) *const gobject.Value;
    pub const getValue = hdy_value_object_get_value;

    extern fn hdy_value_object_get_type() usize;
    pub const getGObjectType = hdy_value_object_get_type;

    extern fn g_object_ref(p_self: *handy.ValueObject) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.ValueObject) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ValueObject, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An adaptive view switcher.
///
/// An adaptive view switcher, designed to switch between multiple views in a
/// similar fashion than a `gtk.StackSwitcher`.
///
/// Depending on the available width, the view switcher can adapt from a wide
/// mode showing the view's icon and title side by side, to a narrow mode showing
/// the view's icon and title one on top of the other, in a more compact way.
/// This can be controlled via the policy property.
///
/// To look good in a header bar, an `HdyViewSwitcher` requires to fill its full
/// height. Contrary to `gtk.HeaderBar`, `HeaderBar` doesn't force a
/// vertical alignment on its title widget, so we recommend it over
/// `gtk.HeaderBar`.
///
/// ## CSS nodes
///
/// `HdyViewSwitcher` has a single CSS node with name `viewswitcher`.
pub const ViewSwitcher = opaque {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.ViewSwitcherClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The preferred place to ellipsize the string.
        ///
        /// If the narrow mode label does not have enough room to display the entire
        /// string, specified as a `pango.EllipsizeMode`.
        ///
        /// Note that setting this property to a value other than
        /// `PANGO_ELLIPSIZE_NONE` has the side-effect that the label requests only
        /// enough space to display the ellipsis.
        pub const narrow_ellipsize = struct {
            pub const name = "narrow-ellipsize";

            pub const Type = pango.EllipsizeMode;
        };

        /// The policy to determine which mode to use.
        pub const policy = struct {
            pub const name = "policy";

            pub const Type = handy.ViewSwitcherPolicy;
        };

        /// The `gtk.Stack` the view switcher controls.
        pub const stack = struct {
            pub const name = "stack";

            pub const Type = ?*gtk.Stack;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyViewSwitcher`.
    extern fn hdy_view_switcher_new() *handy.ViewSwitcher;
    pub const new = hdy_view_switcher_new;

    /// Get the ellipsizing position of the narrow mode label.
    extern fn hdy_view_switcher_get_narrow_ellipsize(p_self: *ViewSwitcher) pango.EllipsizeMode;
    pub const getNarrowEllipsize = hdy_view_switcher_get_narrow_ellipsize;

    /// Gets the policy of `self`.
    extern fn hdy_view_switcher_get_policy(p_self: *ViewSwitcher) handy.ViewSwitcherPolicy;
    pub const getPolicy = hdy_view_switcher_get_policy;

    /// Gets the stack controlled by `self`.
    extern fn hdy_view_switcher_get_stack(p_self: *ViewSwitcher) ?*gtk.Stack;
    pub const getStack = hdy_view_switcher_get_stack;

    /// Sets the mode used to ellipsize the text in narrow mode.
    extern fn hdy_view_switcher_set_narrow_ellipsize(p_self: *ViewSwitcher, p_mode: pango.EllipsizeMode) void;
    pub const setNarrowEllipsize = hdy_view_switcher_set_narrow_ellipsize;

    /// Sets the policy of `self`.
    extern fn hdy_view_switcher_set_policy(p_self: *ViewSwitcher, p_policy: handy.ViewSwitcherPolicy) void;
    pub const setPolicy = hdy_view_switcher_set_policy;

    /// Sets the `gtk.Stack` to control.
    extern fn hdy_view_switcher_set_stack(p_self: *ViewSwitcher, p_stack: ?*gtk.Stack) void;
    pub const setStack = hdy_view_switcher_set_stack;

    extern fn hdy_view_switcher_get_type() usize;
    pub const getGObjectType = hdy_view_switcher_get_type;

    extern fn g_object_ref(p_self: *handy.ViewSwitcher) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.ViewSwitcher) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ViewSwitcher, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A view switcher action bar.
///
/// An action bar letting you switch between multiple views offered by a
/// `gtk.Stack`, via an `ViewSwitcher`. It is designed to be put at
/// the bottom of a window and to be revealed only on really narrow windows e.g.
/// on mobile phones. It can't be revealed if there are less than two pages.
///
/// `HdyViewSwitcherBar` is intended to be used together with
/// `ViewSwitcherTitle`.
///
/// A common use case is to bind the `ViewSwitcherBar.properties.reveal` property
/// to `ViewSwitcherTitle.properties.title_visible` to automatically reveal the
/// view switcher bar when the title label is displayed in place of the view
/// switcher, as follows:
///
/// ```xml
/// <object class="GtkWindow"/>
///   <child type="titlebar">
///     <object class="HdyHeaderBar">
///       <property name="centering-policy">strict</property>
///       <child type="title">
///         <object class="HdyViewSwitcherTitle"
///                 id="view_switcher_title">
///           <property name="stack">stack</property>
///         </object>
///       </child>
///     </object>
///   </child>
///   <child>
///     <object class="GtkBox">
///       <child>
///         <object class="GtkStack" id="stack"/>
///       </child>
///       <child>
///         <object class="HdyViewSwitcherBar">
///           <property name="stack">stack</property>
///           <property name="reveal"
///                     bind-source="view_switcher_title"
///                     bind-property="title-visible"
///                     bind-flags="sync-create"/>
///         </object>
///       </child>
///     </object>
///   </child>
/// </object>
/// ```
///
/// ## CSS nodes
///
/// `HdyViewSwitcherBar` has a single CSS node with name `viewswitcherbar`.
pub const ViewSwitcherBar = opaque {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.ViewSwitcherBarClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The policy used to determine which mode to use.
        pub const policy = struct {
            pub const name = "policy";

            pub const Type = handy.ViewSwitcherPolicy;
        };

        /// Whether the bar should be revealed or hidden.
        pub const reveal = struct {
            pub const name = "reveal";

            pub const Type = c_int;
        };

        /// The `gtk.Stack` the `ViewSwitcher` controls.
        pub const stack = struct {
            pub const name = "stack";

            pub const Type = ?*gtk.Stack;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyViewSwitcherBar`.
    extern fn hdy_view_switcher_bar_new() *handy.ViewSwitcherBar;
    pub const new = hdy_view_switcher_bar_new;

    /// Gets the policy of `self`.
    extern fn hdy_view_switcher_bar_get_policy(p_self: *ViewSwitcherBar) handy.ViewSwitcherPolicy;
    pub const getPolicy = hdy_view_switcher_bar_get_policy;

    /// Gets whether `self` should be revealed or hidden.
    extern fn hdy_view_switcher_bar_get_reveal(p_self: *ViewSwitcherBar) c_int;
    pub const getReveal = hdy_view_switcher_bar_get_reveal;

    /// Get the `gtk.Stack` being controlled by the `ViewSwitcher`.
    extern fn hdy_view_switcher_bar_get_stack(p_self: *ViewSwitcherBar) ?*gtk.Stack;
    pub const getStack = hdy_view_switcher_bar_get_stack;

    /// Sets the policy of `self`.
    extern fn hdy_view_switcher_bar_set_policy(p_self: *ViewSwitcherBar, p_policy: handy.ViewSwitcherPolicy) void;
    pub const setPolicy = hdy_view_switcher_bar_set_policy;

    /// Sets whether `self` should be revealed or not.
    extern fn hdy_view_switcher_bar_set_reveal(p_self: *ViewSwitcherBar, p_reveal: c_int) void;
    pub const setReveal = hdy_view_switcher_bar_set_reveal;

    /// Sets the `gtk.Stack` to control.
    extern fn hdy_view_switcher_bar_set_stack(p_self: *ViewSwitcherBar, p_stack: ?*gtk.Stack) void;
    pub const setStack = hdy_view_switcher_bar_set_stack;

    extern fn hdy_view_switcher_bar_get_type() usize;
    pub const getGObjectType = hdy_view_switcher_bar_get_type;

    extern fn g_object_ref(p_self: *handy.ViewSwitcherBar) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.ViewSwitcherBar) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ViewSwitcherBar, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A view switcher title.
///
/// A widget letting you switch between multiple views contained by a
/// `gtk.Stack`, via an `ViewSwitcher`.
///
/// It is designed to be used as the title widget of a `HeaderBar`, and
/// will display the window's title when the window is too narrow to fit the view
/// switcher e.g. on mobile phones, or if there are less than two views.
///
/// `HdyViewSwitcherTitle` is intended to be used together with
/// `ViewSwitcherBar`.
///
/// A common use case is to bind the `ViewSwitcherBar.properties.reveal` property
/// to `ViewSwitcherTitle.properties.title_visible` to automatically reveal the
/// view switcher bar when the title label is displayed in place of the view
/// switcher, as follows:
///
/// ```xml
/// <object class="GtkWindow"/>
///   <child type="titlebar">
///     <object class="HdyHeaderBar">
///       <property name="centering-policy">strict</property>
///       <child type="title">
///         <object class="HdyViewSwitcherTitle"
///                 id="view_switcher_title">
///           <property name="stack">stack</property>
///         </object>
///       </child>
///     </object>
///   </child>
///   <child>
///     <object class="GtkBox">
///       <child>
///         <object class="GtkStack" id="stack"/>
///       </child>
///       <child>
///         <object class="HdyViewSwitcherBar">
///           <property name="stack">stack</property>
///           <property name="reveal"
///                     bind-source="view_switcher_title"
///                     bind-property="title-visible"
///                     bind-flags="sync-create"/>
///         </object>
///       </child>
///     </object>
///   </child>
/// </object>
/// ```
///
/// ## CSS nodes
///
/// `HdyViewSwitcherTitle` has a single CSS node with name `viewswitchertitle`.
pub const ViewSwitcherTitle = opaque {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.ViewSwitcherTitleClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The policy used to determine which mode to use.
        pub const policy = struct {
            pub const name = "policy";

            pub const Type = handy.ViewSwitcherPolicy;
        };

        /// The `gtk.Stack` the `ViewSwitcher` controls.
        pub const stack = struct {
            pub const name = "stack";

            pub const Type = ?*gtk.Stack;
        };

        /// The subtitle of the `ViewSwitcher`.
        ///
        /// The subtitle should give a user additional details.
        pub const subtitle = struct {
            pub const name = "subtitle";

            pub const Type = ?[*:0]u8;
        };

        /// The title of the `ViewSwitcher`.
        ///
        /// The title should give a user additional details. A good title should not
        /// include the application name.
        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };

        /// Whether the bar should be revealed or hidden.
        pub const title_visible = struct {
            pub const name = "title-visible";

            pub const Type = c_int;
        };

        /// Whether the bar should be revealed or hidden.
        ///
        /// If it is disabled, the title will be displayed instead. This allows to
        /// programmatically hide the view switcher even if it fits in the available
        /// space.
        ///
        /// This can be used e.g. to ensure the view switcher is hidden below a certain
        /// window width, or any other constraint you find suitable.
        pub const view_switcher_enabled = struct {
            pub const name = "view-switcher-enabled";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `HdyViewSwitcherTitle`.
    extern fn hdy_view_switcher_title_new() *handy.ViewSwitcherTitle;
    pub const new = hdy_view_switcher_title_new;

    /// Gets the policy of `self`.
    extern fn hdy_view_switcher_title_get_policy(p_self: *ViewSwitcherTitle) handy.ViewSwitcherPolicy;
    pub const getPolicy = hdy_view_switcher_title_get_policy;

    /// Gets the stack controlled by `self`.
    extern fn hdy_view_switcher_title_get_stack(p_self: *ViewSwitcherTitle) ?*gtk.Stack;
    pub const getStack = hdy_view_switcher_title_get_stack;

    /// Gets the subtitle of `self`.
    extern fn hdy_view_switcher_title_get_subtitle(p_self: *ViewSwitcherTitle) ?[*:0]const u8;
    pub const getSubtitle = hdy_view_switcher_title_get_subtitle;

    /// Gets the title of `self`.
    extern fn hdy_view_switcher_title_get_title(p_self: *ViewSwitcherTitle) ?[*:0]const u8;
    pub const getTitle = hdy_view_switcher_title_get_title;

    /// Gets whether the title of `self` is currently visible.
    extern fn hdy_view_switcher_title_get_title_visible(p_self: *ViewSwitcherTitle) c_int;
    pub const getTitleVisible = hdy_view_switcher_title_get_title_visible;

    /// Gets whether `self`'s view switcher is enabled.
    extern fn hdy_view_switcher_title_get_view_switcher_enabled(p_self: *ViewSwitcherTitle) c_int;
    pub const getViewSwitcherEnabled = hdy_view_switcher_title_get_view_switcher_enabled;

    /// Sets the policy of `self`.
    extern fn hdy_view_switcher_title_set_policy(p_self: *ViewSwitcherTitle, p_policy: handy.ViewSwitcherPolicy) void;
    pub const setPolicy = hdy_view_switcher_title_set_policy;

    /// Sets the `gtk.Stack` to control.
    extern fn hdy_view_switcher_title_set_stack(p_self: *ViewSwitcherTitle, p_stack: ?*gtk.Stack) void;
    pub const setStack = hdy_view_switcher_title_set_stack;

    /// Sets the subtitle of `self`.
    extern fn hdy_view_switcher_title_set_subtitle(p_self: *ViewSwitcherTitle, p_subtitle: ?[*:0]const u8) void;
    pub const setSubtitle = hdy_view_switcher_title_set_subtitle;

    /// Sets the title of `self`.
    extern fn hdy_view_switcher_title_set_title(p_self: *ViewSwitcherTitle, p_title: ?[*:0]const u8) void;
    pub const setTitle = hdy_view_switcher_title_set_title;

    /// Sets whether `self`'s view switcher is enabled.
    extern fn hdy_view_switcher_title_set_view_switcher_enabled(p_self: *ViewSwitcherTitle, p_enabled: c_int) void;
    pub const setViewSwitcherEnabled = hdy_view_switcher_title_set_view_switcher_enabled;

    extern fn hdy_view_switcher_title_get_type() usize;
    pub const getGObjectType = hdy_view_switcher_title_get_type;

    extern fn g_object_ref(p_self: *handy.ViewSwitcherTitle) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.ViewSwitcherTitle) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ViewSwitcherTitle, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A freeform window.
///
/// The `HdyWindow` widget is a subclass of `gtk.Window` which has no
/// titlebar area and provides rounded corners on all sides, ensuring they can
/// never be overlapped by the content. This makes it safe to use headerbars in
/// the content area as follows:
///
/// ```xml
/// <object class="HdyWindow"/>
///   <child>
///     <object class="GtkBox">
///       <property name="visible">True</property>
///       <property name="orientation">vertical</property>
///       <child>
///         <object class="HdyHeaderBar">
///           <property name="visible">True</property>
///           <property name="show-close-button">True</property>
///         </object>
///       </child>
///       <child>
///         <!-- ... -->
///       </child>
///     </object>
///   </child>
/// </object>
/// ```
///
/// It's recommended to use `HeaderBar` with `HdyWindow`, as unlike
/// `gtk.HeaderBar` it remains draggable inside the window. Otherwise,
/// `WindowHandle` can be used.
///
/// `HdyWindow` allows to easily implement titlebar autohiding by putting the
/// headerbar inside a `gtk.Revealer`, and to show titlebar above content
/// by putting it into a `gtk.Overlay` instead of `gtk.Box`.
///
/// If the window has a `gtk.GLArea`, it may bring a slight performance
/// regression when the window is not fullscreen, tiled or maximized.
///
/// Using `gtk.Window.getTitlebar` and `gtk.Window.setTitlebar`
/// is not supported and will result in a crash.
///
/// ## CSS nodes
///
/// `HdyWindow` has a main CSS node with the name `window` and style classes
/// `.background`, `.csd` and `.unified`.
///
/// The `.solid-csd` style class on the main node is used for client-side
/// decorations without invisible borders.
///
/// `HdyWindow` also represents window states with the following style classes on
/// the main node: `.tiled`, `.maximized`, `.fullscreen`.
///
/// It contains the subnodes decoration for window shadow and/or border,
/// decoration-overlay for the sheen on top of the window, `widget.titlebar`, and
/// deck, which contains the child inside the window.
pub const Window = extern struct {
    pub const Parent = gtk.Window;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.WindowClass;
    f_parent_instance: gtk.Window,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `HdyWindow`.
    extern fn hdy_window_new() *handy.Window;
    pub const new = hdy_window_new;

    extern fn hdy_window_get_type() usize;
    pub const getGObjectType = hdy_window_get_type;

    extern fn g_object_ref(p_self: *handy.Window) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.Window) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Window, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A bin that acts like a titlebar.
///
/// `HdyWindowHandle` is a `gtk.Bin` subclass that can be dragged to move
/// its `gtk.Window`, and handles right click, middle click and double
/// click as expected from a titlebar. This is particularly useful with
/// `Window` or `ApplicationWindow`.
///
/// It isn't necessary to use `HdyWindowHandle` if you use `HeaderBar`.
///
/// It can be safely nested or used in the actual window titlebar.
///
/// ## CSS nodes
///
/// `HdyWindowHandle` has a single CSS node with name `windowhandle`.
pub const WindowHandle = opaque {
    pub const Parent = gtk.EventBox;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = handy.WindowHandleClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `HdyWindowHandle`.
    extern fn hdy_window_handle_new() *handy.WindowHandle;
    pub const new = hdy_window_handle_new;

    extern fn hdy_window_handle_get_type() usize;
    pub const getGObjectType = hdy_window_handle_get_type;

    extern fn g_object_ref(p_self: *handy.WindowHandle) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.WindowHandle) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *WindowHandle, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An interface for swipeable widgets.
///
/// The `HdySwipeable` interface is implemented by all swipeable widgets. They
/// can be synced using `SwipeGroup`.
///
/// See `SwipeTracker` for details about implementing it.
pub const Swipeable = opaque {
    pub const Prerequisites = [_]type{gtk.Widget};
    pub const Iface = handy.SwipeableInterface;
    pub const virtual_methods = struct {
        /// Gets the progress `self` will snap back to after the gesture is canceled.
        pub const get_cancel_progress = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) f64 {
                return gobject.ext.as(Swipeable.Iface, p_class).f_get_cancel_progress.?(gobject.ext.as(Swipeable, p_self));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) f64) void {
                gobject.ext.as(Swipeable.Iface, p_class).f_get_cancel_progress = @ptrCast(p_implementation);
            }
        };

        /// Gets the swipe distance of `self`.
        ///
        /// This corresponds to how many pixels 1 unit represents.
        pub const get_distance = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) f64 {
                return gobject.ext.as(Swipeable.Iface, p_class).f_get_distance.?(gobject.ext.as(Swipeable, p_self));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) f64) void {
                gobject.ext.as(Swipeable.Iface, p_class).f_get_distance = @ptrCast(p_implementation);
            }
        };

        /// Gets the current progress of `self`.
        pub const get_progress = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) f64 {
                return gobject.ext.as(Swipeable.Iface, p_class).f_get_progress.?(gobject.ext.as(Swipeable, p_self));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) f64) void {
                gobject.ext.as(Swipeable.Iface, p_class).f_get_progress = @ptrCast(p_implementation);
            }
        };

        /// Gets the snap points of `self`.
        ///
        /// Each snap point represents a progress value that is considered acceptable to
        /// end the swipe on.
        pub const get_snap_points = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_n_snap_points: *c_int) [*]f64 {
                return gobject.ext.as(Swipeable.Iface, p_class).f_get_snap_points.?(gobject.ext.as(Swipeable, p_self), p_n_snap_points);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_n_snap_points: *c_int) callconv(.C) [*]f64) void {
                gobject.ext.as(Swipeable.Iface, p_class).f_get_snap_points = @ptrCast(p_implementation);
            }
        };

        /// Gets the area `self` can start a swipe from for the given direction and
        /// gesture type.
        ///
        /// This can be used to restrict swipes to only be possible from a certain area,
        /// for example, to only allow edge swipes, or to have a draggable element and
        /// ignore swipes elsewhere.
        ///
        /// Swipe area is only considered for direct swipes (as in, not initiated by
        /// `SwipeGroup`).
        ///
        /// If not implemented, the default implementation returns the allocation of
        /// `self`, allowing swipes from anywhere.
        pub const get_swipe_area = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_navigation_direction: handy.NavigationDirection, p_is_drag: c_int, p_rect: *gdk.Rectangle) void {
                return gobject.ext.as(Swipeable.Iface, p_class).f_get_swipe_area.?(gobject.ext.as(Swipeable, p_self), p_navigation_direction, p_is_drag, p_rect);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_navigation_direction: handy.NavigationDirection, p_is_drag: c_int, p_rect: *gdk.Rectangle) callconv(.C) void) void {
                gobject.ext.as(Swipeable.Iface, p_class).f_get_swipe_area = @ptrCast(p_implementation);
            }
        };

        /// Gets the `SwipeTracker` used by this swipeable widget.
        pub const get_swipe_tracker = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *handy.SwipeTracker {
                return gobject.ext.as(Swipeable.Iface, p_class).f_get_swipe_tracker.?(gobject.ext.as(Swipeable, p_self));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *handy.SwipeTracker) void {
                gobject.ext.as(Swipeable.Iface, p_class).f_get_swipe_tracker = @ptrCast(p_implementation);
            }
        };

        /// Switches to child with index `index`.
        ///
        /// See `Swipeable.signals.child_switched`.
        pub const switch_child = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_index: c_uint, p_duration: i64) void {
                return gobject.ext.as(Swipeable.Iface, p_class).f_switch_child.?(gobject.ext.as(Swipeable, p_self), p_index, p_duration);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_index: c_uint, p_duration: i64) callconv(.C) void) void {
                gobject.ext.as(Swipeable.Iface, p_class).f_switch_child = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        /// Emitted when the widget's visible child is changed.
        ///
        /// `duration` can be 0 if the child is switched without animation.
        ///
        /// This is used by `SwipeGroup`, applications should not connect to it.
        pub const child_switched = struct {
            pub const name = "child-switched";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_index: c_uint, p_duration: i64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Swipeable, p_instance))),
                    gobject.signalLookup("child-switched", Swipeable.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Emits `Swipeable.signals.child_switched` signal.
    ///
    /// This should be called when the widget switches visible child widget.
    ///
    /// `duration` can be 0 if the child is switched without animation.
    extern fn hdy_swipeable_emit_child_switched(p_self: *Swipeable, p_index: c_uint, p_duration: i64) void;
    pub const emitChildSwitched = hdy_swipeable_emit_child_switched;

    /// Gets the progress `self` will snap back to after the gesture is canceled.
    extern fn hdy_swipeable_get_cancel_progress(p_self: *Swipeable) f64;
    pub const getCancelProgress = hdy_swipeable_get_cancel_progress;

    /// Gets the swipe distance of `self`.
    ///
    /// This corresponds to how many pixels 1 unit represents.
    extern fn hdy_swipeable_get_distance(p_self: *Swipeable) f64;
    pub const getDistance = hdy_swipeable_get_distance;

    /// Gets the current progress of `self`.
    extern fn hdy_swipeable_get_progress(p_self: *Swipeable) f64;
    pub const getProgress = hdy_swipeable_get_progress;

    /// Gets the snap points of `self`.
    ///
    /// Each snap point represents a progress value that is considered acceptable to
    /// end the swipe on.
    extern fn hdy_swipeable_get_snap_points(p_self: *Swipeable, p_n_snap_points: *c_int) [*]f64;
    pub const getSnapPoints = hdy_swipeable_get_snap_points;

    /// Gets the area `self` can start a swipe from for the given direction and
    /// gesture type.
    ///
    /// This can be used to restrict swipes to only be possible from a certain area,
    /// for example, to only allow edge swipes, or to have a draggable element and
    /// ignore swipes elsewhere.
    ///
    /// Swipe area is only considered for direct swipes (as in, not initiated by
    /// `SwipeGroup`).
    ///
    /// If not implemented, the default implementation returns the allocation of
    /// `self`, allowing swipes from anywhere.
    extern fn hdy_swipeable_get_swipe_area(p_self: *Swipeable, p_navigation_direction: handy.NavigationDirection, p_is_drag: c_int, p_rect: *gdk.Rectangle) void;
    pub const getSwipeArea = hdy_swipeable_get_swipe_area;

    /// Gets the `SwipeTracker` used by this swipeable widget.
    extern fn hdy_swipeable_get_swipe_tracker(p_self: *Swipeable) *handy.SwipeTracker;
    pub const getSwipeTracker = hdy_swipeable_get_swipe_tracker;

    /// Switches to child with index `index`.
    ///
    /// See `Swipeable.signals.child_switched`.
    extern fn hdy_swipeable_switch_child(p_self: *Swipeable, p_index: c_uint, p_duration: i64) void;
    pub const switchChild = hdy_swipeable_switch_child;

    extern fn hdy_swipeable_get_type() usize;
    pub const getGObjectType = hdy_swipeable_get_type;

    extern fn g_object_ref(p_self: *handy.Swipeable) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *handy.Swipeable) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Swipeable, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ActionRowClass = extern struct {
    pub const Instance = handy.ActionRow;

    /// the parent class
    f_parent_class: gtk.ListBoxRowClass,
    /// activates the row to trigger its main action
    f_activate: ?*const fn (p_self: *handy.ActionRow) callconv(.C) void,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *ActionRowClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ApplicationWindowClass = extern struct {
    pub const Instance = handy.ApplicationWindow;

    f_parent_class: gtk.ApplicationWindowClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *ApplicationWindowClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const AvatarClass = extern struct {
    pub const Instance = handy.Avatar;

    f_parent_class: gtk.DrawingAreaClass,

    pub fn as(p_instance: *AvatarClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CarouselClass = extern struct {
    pub const Instance = handy.Carousel;

    f_parent_class: gtk.EventBoxClass,

    pub fn as(p_instance: *CarouselClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CarouselIndicatorDotsClass = extern struct {
    pub const Instance = handy.CarouselIndicatorDots;

    f_parent_class: gtk.DrawingAreaClass,

    pub fn as(p_instance: *CarouselIndicatorDotsClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CarouselIndicatorLinesClass = extern struct {
    pub const Instance = handy.CarouselIndicatorLines;

    f_parent_class: gtk.DrawingAreaClass,

    pub fn as(p_instance: *CarouselIndicatorLinesClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ClampClass = extern struct {
    pub const Instance = handy.Clamp;

    f_parent_class: gtk.BinClass,

    pub fn as(p_instance: *ClampClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ComboRowClass = extern struct {
    pub const Instance = handy.ComboRow;

    /// the parent class
    f_parent_class: handy.ActionRowClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *ComboRowClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const DeckClass = extern struct {
    pub const Instance = handy.Deck;

    /// the parent class
    f_parent_class: gtk.ContainerClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *DeckClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const EnumValueObjectClass = extern struct {
    pub const Instance = handy.EnumValueObject;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *EnumValueObjectClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ExpanderRowClass = extern struct {
    pub const Instance = handy.ExpanderRow;

    /// the parent class
    f_parent_class: handy.PreferencesRowClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *ExpanderRowClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const FlapClass = extern struct {
    pub const Instance = handy.Flap;

    f_parent_class: gtk.ContainerClass,

    pub fn as(p_instance: *FlapClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const HeaderBarClass = extern struct {
    pub const Instance = handy.HeaderBar;

    /// the parent class
    f_parent_class: gtk.ContainerClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *HeaderBarClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const HeaderGroupChildClass = extern struct {
    pub const Instance = handy.HeaderGroupChild;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *HeaderGroupChildClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const HeaderGroupClass = extern struct {
    pub const Instance = handy.HeaderGroup;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *HeaderGroupClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const KeypadClass = extern struct {
    pub const Instance = handy.Keypad;

    /// the parent class
    f_parent_class: gtk.BinClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *KeypadClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const LeafletClass = extern struct {
    pub const Instance = handy.Leaflet;

    /// the parent class
    f_parent_class: gtk.ContainerClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *LeafletClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const PreferencesGroupClass = extern struct {
    pub const Instance = handy.PreferencesGroup;

    /// the parent class
    f_parent_class: gtk.BinClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *PreferencesGroupClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const PreferencesPageClass = extern struct {
    pub const Instance = handy.PreferencesPage;

    /// the parent class
    f_parent_class: gtk.BinClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *PreferencesPageClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const PreferencesRowClass = extern struct {
    pub const Instance = handy.PreferencesRow;

    /// the parent class
    f_parent_class: gtk.ListBoxRowClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *PreferencesRowClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const PreferencesWindowClass = extern struct {
    pub const Instance = handy.PreferencesWindow;

    /// the parent class
    f_parent_class: handy.WindowClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *PreferencesWindowClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SearchBarClass = extern struct {
    pub const Instance = handy.SearchBar;

    f_parent_class: gtk.BinClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *SearchBarClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SqueezerClass = extern struct {
    pub const Instance = handy.Squeezer;

    f_parent_class: gtk.ContainerClass,

    pub fn as(p_instance: *SqueezerClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const StatusPageClass = extern struct {
    pub const Instance = handy.StatusPage;

    f_parent_class: gtk.BinClass,

    pub fn as(p_instance: *StatusPageClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const StyleManagerClass = extern struct {
    pub const Instance = handy.StyleManager;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *StyleManagerClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SwipeGroupClass = extern struct {
    pub const Instance = handy.SwipeGroup;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *SwipeGroupClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SwipeTrackerClass = extern struct {
    pub const Instance = handy.SwipeTracker;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *SwipeTrackerClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An interface for swipeable widgets.
pub const SwipeableInterface = extern struct {
    pub const Instance = handy.Swipeable;

    /// the parent interface
    f_parent: gobject.TypeInterface,
    /// switches visible child
    f_switch_child: ?*const fn (p_self: *handy.Swipeable, p_index: c_uint, p_duration: i64) callconv(.C) void,
    /// gets the swipe tracker
    f_get_swipe_tracker: ?*const fn (p_self: *handy.Swipeable) callconv(.C) *handy.SwipeTracker,
    /// gets the swipe distance
    f_get_distance: ?*const fn (p_self: *handy.Swipeable) callconv(.C) f64,
    /// gets the snap points
    f_get_snap_points: ?*const fn (p_self: *handy.Swipeable, p_n_snap_points: *c_int) callconv(.C) [*]f64,
    /// gets the current progress
    f_get_progress: ?*const fn (p_self: *handy.Swipeable) callconv(.C) f64,
    /// gets the cancel progress
    f_get_cancel_progress: ?*const fn (p_self: *handy.Swipeable) callconv(.C) f64,
    /// gets the swipeable rectangle
    f_get_swipe_area: ?*const fn (p_self: *handy.Swipeable, p_navigation_direction: handy.NavigationDirection, p_is_drag: c_int, p_rect: *gdk.Rectangle) callconv(.C) void,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *SwipeableInterface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const TabBarClass = extern struct {
    pub const Instance = handy.TabBar;

    f_parent_class: gtk.BinClass,

    pub fn as(p_instance: *TabBarClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const TabPageClass = extern struct {
    pub const Instance = handy.TabPage;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *TabPageClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const TabViewClass = extern struct {
    pub const Instance = handy.TabView;

    f_parent_class: gtk.BinClass,

    pub fn as(p_instance: *TabViewClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const TitleBarClass = extern struct {
    pub const Instance = handy.TitleBar;

    f_parent_class: gtk.BinClass,

    pub fn as(p_instance: *TitleBarClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ValueObjectClass = extern struct {
    pub const Instance = handy.ValueObject;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *ValueObjectClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ViewSwitcherBarClass = extern struct {
    pub const Instance = handy.ViewSwitcherBar;

    f_parent_class: gtk.BinClass,

    pub fn as(p_instance: *ViewSwitcherBarClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ViewSwitcherClass = extern struct {
    pub const Instance = handy.ViewSwitcher;

    f_parent_class: gtk.BinClass,

    pub fn as(p_instance: *ViewSwitcherClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ViewSwitcherTitleClass = extern struct {
    pub const Instance = handy.ViewSwitcherTitle;

    f_parent_class: gtk.BinClass,

    pub fn as(p_instance: *ViewSwitcherTitleClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const WindowClass = extern struct {
    pub const Instance = handy.Window;

    f_parent_class: gtk.WindowClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *WindowClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const WindowHandleClass = extern struct {
    pub const Instance = handy.WindowHandle;

    f_parent_class: gtk.EventBoxClass,

    pub fn as(p_instance: *WindowHandleClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Describes title centering behavior of a `HeaderBar` widget.
pub const CenteringPolicy = enum(c_int) {
    loose = 0,
    strict = 1,
    _,

    extern fn hdy_centering_policy_get_type() usize;
    pub const getGObjectType = hdy_centering_policy_get_type;
};

/// Application color schemes for `StyleManager.properties.color_scheme`.
pub const ColorScheme = enum(c_int) {
    default = 0,
    force_light = 1,
    prefer_light = 2,
    prefer_dark = 3,
    force_dark = 4,
    _,

    extern fn hdy_color_scheme_get_type() usize;
    pub const getGObjectType = hdy_color_scheme_get_type;
};

/// Describes the possible transitions in a `Deck` widget.
///
/// New values may be added to this enumeration over time.
pub const DeckTransitionType = enum(c_int) {
    over = 0,
    under = 1,
    slide = 2,
    _,

    extern fn hdy_deck_transition_type_get_type() usize;
    pub const getGObjectType = hdy_deck_transition_type_get_type;
};

/// Describes the possible folding behavior of a `Flap` widget.
pub const FlapFoldPolicy = enum(c_int) {
    never = 0,
    always = 1,
    auto = 2,
    _,

    extern fn hdy_flap_fold_policy_get_type() usize;
    pub const getGObjectType = hdy_flap_fold_policy_get_type;
};

/// Describes transitions types of a `Flap` widget.
///
/// These enumeration values describe the possible transitions between children
/// in a `Flap` widget, as well as which areas can be swiped via
/// `Flap.properties.swipe_to_open` and `Flap.properties.swipe_to_close`.
///
/// New values may be added to this enum over time.
pub const FlapTransitionType = enum(c_int) {
    over = 0,
    under = 1,
    slide = 2,
    _,

    extern fn hdy_flap_transition_type_get_type() usize;
    pub const getGObjectType = hdy_flap_transition_type_get_type;
};

/// Describes the child types handled by `HeaderGroup`.
///
/// New values may be added to this enumeration over time.
pub const HeaderGroupChildType = enum(c_int) {
    header_bar = 0,
    gtk_header_bar = 1,
    header_group = 2,
    _,

    extern fn hdy_header_group_child_type_get_type() usize;
    pub const getGObjectType = hdy_header_group_child_type_get_type;
};

/// Describes the possible transitions in a `Leaflet` widget.
///
/// New values may be added to this enumeration over time.
pub const LeafletTransitionType = enum(c_int) {
    over = 0,
    under = 1,
    slide = 2,
    _,

    extern fn hdy_leaflet_transition_type_get_type() usize;
    pub const getGObjectType = hdy_leaflet_transition_type_get_type;
};

/// Describes the direction of a swipe navigation gesture.
pub const NavigationDirection = enum(c_int) {
    back = 0,
    forward = 1,
    _,

    extern fn hdy_navigation_direction_get_type() usize;
    pub const getGObjectType = hdy_navigation_direction_get_type;
};

/// Describes the possible transitions in a `Squeezer` widget.
pub const SqueezerTransitionType = enum(c_int) {
    none = 0,
    crossfade = 1,
    _,

    extern fn hdy_squeezer_transition_type_get_type() usize;
    pub const getGObjectType = hdy_squeezer_transition_type_get_type;
};

/// Describes the adaptive modes of `ViewSwitcher`.
pub const ViewSwitcherPolicy = enum(c_int) {
    auto = 0,
    narrow = 1,
    wide = 2,
    _,

    extern fn hdy_view_switcher_policy_get_type() usize;
    pub const getGObjectType = hdy_view_switcher_policy_get_type;
};

/// Computes the ease out for a value.
extern fn hdy_ease_out_cubic(p_t: f64) f64;
pub const easeOutCubic = hdy_ease_out_cubic;

/// Returns the name of a `EnumValueObject`.
///
/// This is a default implementation of `ComboRowGetEnumValueNameFunc`
/// to be used with `ComboRow.setForEnum`. If the enumeration has a
/// nickname, it will return it, otherwise it will return its name.
extern fn hdy_enum_value_row_name(p_value: *handy.EnumValueObject, p_user_data: ?*anyopaque) [*:0]u8;
pub const enumValueRowName = hdy_enum_value_row_name;

/// Checks whether animations are enabled for `widget`.
///
/// This should be used when implementing an animated widget to know whether to
/// animate it or not.
extern fn hdy_get_enable_animations(p_widget: *gtk.Widget) c_int;
pub const getEnableAnimations = hdy_get_enable_animations;

/// Initializes Libhandy.
///
/// Call this function just after initializing GTK, if you are using
/// `gtk.Application` it means it must be called when the
/// `gio.Application.signals.startup` signal is emitted.
///
/// If Libhandy has already been initialized, the function will simply return.
///
/// This makes sure translations, types, themes, and icons for the Handy library
/// are set up properly.
extern fn hdy_init() void;
pub const init = hdy_init;

/// Callback for loading an `Avatar`'s image.
///
/// The returned `gdkpixbuf.Pixbuf` is expected to be square with width and
/// height set to `size`. The image is cropped to a circle without any scaling or
/// transformation.
pub const AvatarImageLoadFunc = *const fn (p_size: c_int, p_user_data: ?*anyopaque) callconv(.C) ?*gdkpixbuf.Pixbuf;

/// Callback for getting the name of a row from an enum.
///
/// Called for combo rows that are bound to an enumeration with
/// `ComboRow.setForEnum` for each value from that enumeration.
///
/// See also: `enumValueRowName`.
pub const ComboRowGetEnumValueNameFunc = *const fn (p_value: *handy.EnumValueObject, p_user_data: ?*anyopaque) callconv(.C) [*:0]u8;

/// Callback for getting the name of a row.
///
/// Called for combo rows that are bound to a `gio.ListModel` with
/// `ComboRow.bindNameModel` for each item that gets added to the model.
pub const ComboRowGetNameFunc = *const fn (p_item: *gobject.Object, p_user_data: ?*anyopaque) callconv(.C) [*:0]u8;
