/* wmpasman
 * Copyright © 1999-2014  Brad Jorsch <anomie@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "config.h"

#include <stdlib.h>

#include <glib-unix.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gio/gio.h>

#include "wmpasman.h"
#include "die.h"
#include "dock.h"
#include "clipboard.h"
#include "keyring.h"

/* Global variables */
appstate_t appstate = STARTING;
GCancellable *app_cancel = NULL;

static gint timeout = 5;
gboolean allow_insecure_memory = FALSE;
static gboolean allow_core_files = FALSE;
static gchar *config_file = NULL;
static gchar *commands[7] = { NULL, NULL, NULL, NULL, NULL, NULL, NULL };

/* Signal handling */
int pipe2(int pipefd[2], int flags);
static int sig_fd[2];

static void sighandler(int sig) {
    if (write(sig_fd[1], &sig, sizeof(sig)) < 0) {
        // Nothing much we can do about it inside a signal handler, but gcc
        // complains if we don't test it.
    }
    signal(sig, sighandler);
}

static gboolean sig_caught(GIOChannel *source G_GNUC_UNUSED, GIOCondition cond G_GNUC_UNUSED, gpointer data G_GNUC_UNUSED){
    int sig;
    while (read(sig_fd[0], &sig, sizeof(sig)) > 0) {
        switch (sig) {
          case SIGINT:
          case SIGQUIT:
          case SIGKILL: // Useless, but...
          case SIGPIPE:
          case SIGTERM:
          case SIGABRT:
            gtk_main_quit();
            break;
          case SIGHUP:
            // Reload something?
            break;
          case SIGUSR1:
            clock_flags ^= CLOCK_UTC;
            redraw_dock();
            break;
          case SIGUSR2:
            clock_flags ^= CLOCK_24HR;
            redraw_dock();
            break;
          default:
            if (sig == SIGRTMIN+0) {
                clock_flags ^= CLOCK_BEATS;
                redraw_dock();
            } else if (sig == SIGRTMIN+1) {
                clock_flags ^= CLOCK_DISCORDIAN;
                redraw_dock();
            }
            break;
        }
    }
    return TRUE;
}

/* Activation/deactivation */
static gboolean app_timeout(gpointer p G_GNUC_UNUSED) {
    app_deactivate();
    return FALSE;
}

static void app_activated(GError *err) {
    g_clear_object(&app_cancel);
    if (err) {
        error_alert("Keyring activation failed: %s", err);
    } else {
        appstate = ACTIVE;
        if (timeout > 0) {
            g_timeout_add_seconds(timeout, app_timeout, NULL);
        }
    }
    app_finish_work();
}

static void app_keyring_unlocked(GError *err) {
    if (err) {
        error_alert("Keyring unlock failed: %s", err);
    }
    app_finish_work();
    app_update_state();
}

static void app_keyring_locked(GError *err) {
    if (err) {
        error_alert("Keyring lock failed: %s", err);
    }
    app_finish_work();
    app_update_state();
}

void app_unlock(void) {
    if (appstate == LOCKED) {
        keyring_unlock(app_start_work(), app_keyring_unlocked);
    }
}

void app_lock(void) {
    if (appstate != STARTING && appstate != LOCKED) {
        app_deactivate();
        keyring_lock(app_start_work(), app_keyring_locked);
    }
}

void app_activate(void) {
    if (appstate == WORKING || appstate == STARTING || appstate == LOCKED || appstate == NOPASS || !current_password) {
        // Ignore
        return;
    }

    GError *err = NULL;
    if (!clipboard_activate(&err)) {
        error_alert("Clipboard activation failed: %s", err);
        g_clear_error(&err);
        return;
    }
    keyring_activate(app_start_work(), app_activated);
}

void app_deactivate(void) {
    if (appstate == ACTIVE) {
        clipboard_deactivate();
        keyring_deactivate();
        appstate = INACTIVE;
        app_update_state();
    }
}

void app_update_state(void) {
    appstate_t oldstate = appstate;

    if (!keyring_ready()) {
        appstate = STARTING;
    } else if (appstate == WORKING && app_cancel) {
        appstate = WORKING;
    } else if (!keyring_unlocked()) {
        appstate = LOCKED;
    } else if (!current_password) {
        appstate = NOPASS;
    } else if (appstate != ACTIVE) {
        appstate = INACTIVE;
    }

    if (oldstate == ACTIVE && appstate != ACTIVE) {
        clipboard_deactivate();
        keyring_deactivate();
    }
    dock_state_changed();
}

GCancellable *app_start_work(void) {
    if (appstate == WORKING) {
        return NULL;
    }

    appstate = WORKING;
    dock_state_changed();
    app_cancel = g_cancellable_new();
    return app_cancel;
}

void app_cancel_work() {
    if (app_cancel) {
        g_cancellable_cancel(app_cancel);
        g_clear_object(&app_cancel);
    }
}

void app_finish_work() {
    g_clear_object(&app_cancel);
    app_update_state();
}


/* Misc */

GQuark app_generic_error_quark(void) {
    return g_quark_from_static_string("app-generic-error-quark");
}

GQuark app_no_error_quark(void) {
    return g_quark_from_static_string("app-no-error-quark");
}

static gboolean parse_opt(const gchar *option_name, const gchar *value G_GNUC_UNUSED, gpointer data G_GNUC_UNUSED, GError **err G_GNUC_UNUSED) {
    if (!strcmp(option_name, "-V") || !strcmp(option_name, "--version")) {
        printf("%s\n", VERSION);
        exit(0);
    } else if (!strcmp(option_name, "-v") || !strcmp(option_name, "--verbose")) {
        warn_level++;
    } else if (!strcmp(option_name, "-q") || !strcmp(option_name, "--quiet")) {
        warn_level--;
    } else if (!strcmp(option_name, "-U") || !strcmp(option_name, "--utc")) {
        clock_flags ^= CLOCK_UTC;
    } else if (!strcmp(option_name, "-M") || !strcmp(option_name, "--24hr")) {
        clock_flags ^= CLOCK_24HR;
    } else if (!strcmp(option_name, "-B") || !strcmp(option_name, "--beats")) {
        clock_flags ^= CLOCK_BEATS;
    } else if (!strcmp(option_name, "-D") || !strcmp(option_name, "--discordian") || !strcmp(option_name, "--ddate")) {
        clock_flags ^= CLOCK_DISCORDIAN;
    }
    return TRUE;
}

void exec_command(int button) {
    if (button < 1 || button > 7 || !commands[button-1]) {
        return;
    }

    warn(DEBUG_DEBUG, "Executing command %d: %s", button, commands[button-1]);
    GError *err = NULL;
    if (!g_spawn_command_line_async(commands[button-1], &err)) {
        warn(DEBUG_ERROR, "Could not execute command %s: %s", commands[button-1], err ? err->message : "<unknown error>");
    }
    g_clear_error(&err);
}

static GOptionEntry commandline_opts1[] = {
    {"version", 'V', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "Print the version number and exit", NULL},
    {"conf", 'c', 0, G_OPTION_ARG_FILENAME, &config_file, "", NULL},
    {"verbose", 'v', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "Print more messages (can be repeated)", NULL},
    {"quiet", 'q', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "Print fewer messages (can be repeated)", NULL},
    {NULL}
};

static GOptionEntry commandline_opts2[] = {
    {"version", 'V', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "Print the version number and exit", NULL},
    {"conf", 'c', 0, G_OPTION_ARG_FILENAME, NULL, "Specify a configuration file to use instead of the default", "<file>"},
    {"verbose", 'v', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "Print more messages (can be repeated)", NULL},
    {"quiet", 'q', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "Print fewer messages (can be repeated)", NULL},
    {"timeout", 't', 0, G_OPTION_ARG_INT, &timeout, "Seconds to hold the clipboard before dropping the selection (-1 to never drop)", "<seconds>"},

    {"24hr", 'M', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "Toggle time display in 24-hour format", NULL},
    {"utc", 'U', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "Toggle time display in UTC instead of the local timezone", NULL},
    {"beats", 'B', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "Toggle Swatch beats \"Internet time\" (see <http://www.timeanddate.com/time/internettime.html>)", NULL},
    {"discordian", 'D', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "Toggle Discordian dates (see <https://en.wikipedia.org/wiki/Discordian_calendar>, or consult your pineal gland)", NULL},
    {"ddate", 0, G_OPTION_FLAG_HIDDEN|G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "", NULL},

    {"allow-insecure-memory", 0, 0, G_OPTION_ARG_NONE, &allow_insecure_memory, "Allow passphrases to be temporarily stored in insecure memory", NULL},
    {"no-allow-insecure-memory", 0, G_OPTION_FLAG_REVERSE, G_OPTION_ARG_NONE, &allow_insecure_memory, "Exit if secure memory cannot be allocated", NULL},
    {"allow-core-files", 0, 0, G_OPTION_ARG_NONE, &allow_core_files, "Allow execution even if core file dumping cannot be disabled", NULL},
    {"no-allow-core-files", 0, G_OPTION_FLAG_REVERSE, G_OPTION_ARG_NONE, &allow_core_files, "Exit if core file dumping cannot be disabled", NULL},

    {"non-wmaker", 0, 0, G_OPTION_ARG_NONE, &nonwmaker, "Use in a non-Window Maker window manager", NULL},

    {NULL}
};

static GOptionEntry conffile_opts[] = {
    {"timeout", 0, 0, G_OPTION_ARG_INT, &timeout, "", NULL},

    {"24hr", 0, G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "", NULL},
    {"utc", 0, G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "", NULL},
    {"beats", 0, G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "", NULL},
    {"discordian", 0, G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "", NULL},
    {"ddate", 0, G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK, parse_opt, "", NULL},

    {"allow-insecure-memory", 0, 0, G_OPTION_ARG_NONE, &allow_insecure_memory, "", NULL},
    {"no-allow-insecure-memory", 0, G_OPTION_FLAG_REVERSE, G_OPTION_ARG_NONE, &allow_insecure_memory, "", NULL},
    {"allow-core-files", 0, 0, G_OPTION_ARG_NONE, &allow_core_files, "", NULL},
    {"no-allow-core-files", 0, G_OPTION_FLAG_REVERSE, G_OPTION_ARG_NONE, &allow_core_files, "", NULL},

    {"left", 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_STRING, &commands[0], "", NULL},
    {"middle", 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_STRING, &commands[1], "", NULL},
    {"right", 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_STRING, &commands[2], "", NULL},
    {"wheel-up", 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_STRING, &commands[3], "", NULL},
    {"wheel-down", 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_STRING, &commands[4], "", NULL},
    {"button1", 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_STRING, &commands[0], "", NULL},
    {"button2", 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_STRING, &commands[1], "", NULL},
    {"button3", 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_STRING, &commands[2], "", NULL},
    {"button4", 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_STRING, &commands[3], "", NULL},
    {"button5", 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_STRING, &commands[4], "", NULL},
    {"button6", 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_STRING, &commands[5], "", NULL},
    {"button7", 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_STRING, &commands[6], "", NULL},

    {"non-wmaker", 0, 0, G_OPTION_ARG_NONE, &nonwmaker, "", NULL},

    {NULL}
};


int main(int argc, char *argv[]) {
    warn(DEBUG_INFO, "Parsing command line for --conf");
    GOptionContext *context;
    context = g_option_context_new("");
    g_option_context_add_group(context, gtk_get_option_group(TRUE));
    g_option_context_add_main_entries(context, commandline_opts1, NULL);
    g_option_context_set_help_enabled(context, FALSE);
    g_option_context_set_ignore_unknown_options(context, TRUE);
    GError *err = NULL;
    if (!g_option_context_parse(context, &argc, &argv, &err)) {
        die("%s", err ? err->message : "Failed to parse command line options");
    }
    g_clear_error(&err);
    g_option_context_free(context);

    warn(DEBUG_INFO, "Parsing config file");
    gboolean conf_specified = config_file != NULL;
    if (config_file == NULL) {
        const char *homedir = g_getenv("HOME");
        if (!homedir) {
            homedir = g_get_home_dir();
        }
        if (homedir) {
            config_file = g_strjoin("/", homedir, ".wmpasmanrc", NULL);
        }
    }
    if (config_file) {
        GFile *file = g_file_new_for_path(config_file);
        GInputStream *inputstream = (GInputStream *)g_file_read(file, NULL, &err);
        if (inputstream) {
            g_clear_error(&err);
            GDataInputStream *s = g_data_input_stream_new(inputstream);
            char *line, *v;
            gint xargc;
            gchar *xargv[4] = { argv[0], NULL, NULL, NULL }, **xxargv;
            context = g_option_context_new("");
            g_option_context_add_group(context, gtk_get_option_group(TRUE));
            g_option_context_add_main_entries(context, conffile_opts, NULL);
            g_option_context_set_help_enabled(context, FALSE);
            while ((line =  g_data_input_stream_read_line(s, NULL, NULL, &err))) {
                g_strstrip(line);
                if (!*line || *line == '#') {
                    // blank line or comment
                    g_free(line);
                    continue;
                }
                v = strchr(line, ':');
                if (v) {
                    *v++ = '\0';
                    g_strchomp(line);
                    g_strchug(v);
                    xargc = 3;
                    xargv[2] = v;
                } else {
                    xargc = 2;
                    xargv[2] = NULL;
                }
                xargv[1] = g_strconcat("--", line, NULL);
                xxargv = xargv;
                if (!g_option_context_parse(context, &xargc, &xxargv, &err)) {
                    die("Failed to parse conf file %s: %s", config_file, err ? err->message : "<unknown error>");
                }
                g_clear_error(&err);
                g_free(xargv[1]);
                g_free(line);
            }
            if (err) {
                die("Failed to read conf file %s: %s", config_file, err ? err->message : "<unknown error>");
            }
            g_option_context_free(context);
            g_object_unref(s);
            g_object_unref(inputstream);
        } else if (conf_specified) {
            die("Failed to open conf file %s: %s", config_file, err ? err->message : "<unknown error>");
        }
        g_clear_error(&err);
        g_object_unref(file);
    }

    warn(DEBUG_INFO, "Parsing command line for all options");
    context = g_option_context_new("");
    g_option_context_add_group(context, gtk_get_option_group(TRUE));
    g_option_context_add_main_entries(context, commandline_opts2, NULL);
    if (!g_option_context_parse(context, &argc, &argv, &err)) {
        die("%s", err ? err->message : "Failed to parse command line options");
    }
    g_clear_error(&err);
    g_option_context_free(context);

    warn(DEBUG_DEBUG, "Setting up clipboard access");
    clipboard_init();

    warn(DEBUG_DEBUG, "Selecting and initializing keyring");
    keyring_init(allow_core_files);

    warn(DEBUG_DEBUG, "Setting up signal handling");
    errno=0;
    if(!pipe2(sig_fd,O_NONBLOCK)){
        GIOChannel *channel = g_io_channel_unix_new(sig_fd[0]);
        g_io_add_watch(channel, G_IO_IN, sig_caught, NULL);
        g_io_channel_unref(channel);
        signal(SIGINT, sighandler);
        signal(SIGQUIT, sighandler);
        signal(SIGKILL, sighandler); // Useless, but...
        signal(SIGPIPE, sighandler);
        signal(SIGTERM, sighandler);
        signal(SIGABRT, sighandler);
        signal(SIGHUP, sighandler);
        signal(SIGUSR1, sighandler);
        signal(SIGUSR2, sighandler);
        signal(SIGRTMIN+0, sighandler);
        signal(SIGRTMIN+1, sighandler);
        warn(DEBUG_INFO, "SIGRTMIN+0 is %d", SIGRTMIN+0);
        warn(DEBUG_INFO, "SIGRTMIN+1 is %d", SIGRTMIN+1);
    } else {
        warn(DEBUG_WARN, "Signal handling initialization failed (%s), will probably die on any signal", strerror(errno));
    }

    warn(DEBUG_DEBUG, "Creating dock icon");
    create_dock_icon();

    warn(DEBUG_DEBUG, "Starting GTK main loop");
    gtk_main();
    warn(DEBUG_DEBUG, "GTK main loop exited");

    keyring_lock_sync();

    return 0;
}
