/*
Program describing the Mendelian rules of inheritance of the color of pea
plants. It considers a family of two parents and a child.
The problem is, given the alleles of the parents, predict the
probability of the color (or of its alleles) of a pea plant.
From
H. Blockeel. Probabilistic logical models for mendel's experiments:
An exercise.
In Inductive Logic Programming (ILP 2004), Work in Progress Track, 2004.
*/
:- use_module(library(kbest)).

:- if(current_predicate(use_rendering/1)).
:- use_rendering(c3).
:- endif.

:- kbest.

:- begin_lpad.

mother(m,s).
father(f,s).
% family with 3 members: m is the mother of s and f is the father of s

% cg(I,C,A) means that individual I has color allele A on chromosome C
% the color alleles are p and w and the chromosomes are 1 and 2
% color(I,Col) means that individual I has color Col
% Col can be purple or white

cg(m,1,p).
cg(m,2,w).
cg(f,1,w).
cg(f,2,p).
% we know with certainty the alleles of the parants of s

cg(X,1,A):0.5 ; cg(X,1,B):0.5 :- mother(Y,X),cg(Y,1,A), cg(Y,2,B).
% the color allele of an individual on chromosome 1 is inherited from its
% mother. The two alleles of the mother have equal probability of being
% transmitted

cg(X,2,A):0.5 ; cg(X,2,B):0.5 :- father(Y,X),cg(Y,1,A), cg(Y,2,B).
% the color allele of an individual on chromosome 2 is inherited from its
% father. The two alleles of the mother have equal probability of being
% transmitted


color(X,purple) :- cg(X,_,p).
% if an individual has a p allele its color is purple, i.e., purple is
% dominant

color(X,white) :- cg(X,1,w), cg(X,2,w).
% if an individual has two w alleles its color is white, i.e., white is
% recessive

:- end_lpad.

/** <examples>

?- kbest(color(s,purple),1,Prob,Exp).
Prob = 0.5,
Exp = [0.5-[rule(0, cg(s, 1, p), [cg(s, 1, p):0.5, cg(s, 1, w):0.5], [mother(m, s), cg(m, 1, p), cg(m, 2, w)])].

?- kbest(color(s,white),1,Prob,Exp).
Prob = 0.25,
Exp = [rule(0, cg(s, 1, w), [cg(s, 1, p):0.5, cg(s, 1, w):0.5], [mother(m, s), cg(m, 1, p), cg(m, 2, w)]), rule(1, cg(s, 2, w), [cg(s, 2, w):0.5, cg(s, 2, p):0.5], [father(f, s), cg(f, 1, w), cg(f, 2, p)])].

?- kbest(color(s,purple),2,Prob,Exp).
Prob = 0.75,
Exp = [0.5-[rule(1, cg(s, 2, p), [cg(s, 2, w):0.5, cg(s, 2, p):0.5],
[father(f, s), cg(f, 1, w), cg(f, 2, p)])],
0.5-[rule(0, cg(s, 1, p), [cg(s, 1, p):0.5, cg(s, 1, w):0.5], [mother(m, s), cg(m, 1, p), cg(m, 2, w)])]].

?- kbest(color(s,white),2,Prob,Exp).
Prob = 0.25,
Exp = [0.25-[rule(0, cg(s, 1, w), [cg(s, 1, p):0.5, cg(s, 1, w):0.5], [mother(m, s), cg(m, 1, p), cg(m, 2, w)]),
rule(1, cg(s, 2, w), [cg(s, 2, w):0.5, cg(s, 2, p):0.5], [father(f, s), cg(f, 1, w), cg(f, 2, p)])]]


*/
