:- compiler_options([xpp_on,sysmod]).

%%#include "celltags_xsb.h"

/* Documentation for this module is in the subdirectory prolog_db_doc
in file prolog_db.pdf, which can be re-generated by getting into that
directory, and running:

xsb -e "[xsbdoc], xsbdoc(prolog_db_format,pdf),halt."

************/

/***
For better performance, could do:

1. implement pdb_unnumbervars in C.  This would speed up retrieving
rules (since only bodies need unnumbervarring), important when calling
rule-defined predicates in DB.

2. implement find_in_radix_tree for lookup (i.e. when key is given) as
a builtin.

New pred to add:

select_db(+PrefixTerm,+TemplPrefixTerm,+DB0,?DB): takes a prefix term, a
template prefix term, and a DB0 and produces a new DB that contains
terms that are instances of TemplPrefixTerm.  The arities of
PrefixTerm and TemplPrefixTerm (i.e., the number of variables in each)
must be the same.  The resulting database contains terms that are
instances of TemplPrefixTerm whose variables are instantiated by
instances of PrefixTerm in DB0.

For example, select_db(r(a,f(_,_),_),res(_,_,_),DB0,DB) would produce
the same DB as
findall_db(res(A,B,C),fact_in_db(r(a,f(A,B),C),DB0),DB).
But it runs in log time, since it just finds the point in the trie of
PrefixTerm and then constructs the TemplPrefixTerm sequence and points
it to that trie.

Or maybe more useful, a minor variant that doesn't require PrefixTerm
to be a prefix term, but allows variables in the "prefix" portion, and
then is nondeterministic; for each binding those variables, produce
the res(_,_,_) form for that (now instantiated) prefix term.

So select(r(X,f(_,_),_),DB0,DB) would, for X such that there is
r(X,f(_,_),_) in DB0, bind DB to a prolog_db representing ans(_,_,_)
for the tails...


Another possibly useful predicate might be in_proj_db(Pref,N,DB) where
Pref is a prefix term with M variables, N is an integer =< M, and DB
is a DB. E.g., in_proj_db(p(A,B,_,_),2,DB) succeeds for instances of
p/4 in DB, binding A and B to values of those terms in DB.  It
succeeds only once for each distinct pair of values A and B.

***/

:- comment(title,"Trie-Terms for Sets and Prolog Databases").
:- comment(subtitle,"Efficient Representation of Sets of Clauses and Terms").
:- comment(author,"David S. Warren").

:- comment(summary, "The predicates of the @tt{prolog_db} module
support an abstract datatype for sets of Prolog clauses.  The
underlying implementation uses a @em{ground} Prolog term that encodes
an indexed ``trie,'' which represents the set of clauses. The trie is
built using the pre-order traversals of the heads of clauses of the
set, and a variant of a radix tree at each branch point of the trie.
It thus provides complete indexing (to clause heads) for terms that
are bound on an initial sequence of symbols in their pre-order
traversal.  The representation is @em{canonical} in that the same set
of clauses is always represented by the same trie-term, no matter the
add and delete operations, and the order of their application, used to
construct it.

These trie-terms represent sets of Prolog clauses (a.k.a., a Prolog
Database, or PDB.)  We call a set of clauses in this representation a
``trie-term database.''  Operations are provided to assert clauses to,
and retract clauses from, a given trie-term database to generate a new
one reflecting the change.  Also there is an operation to evaluate (or
prove) a goal using the clauses of a given trie-term database.
Trie-terms do not support duplicates or clause order, so cuts (and
duplicate clauses) cannot be used in the clauses of trie-term
databases.  Trie-term databases can support Prolog meta-programming in
a pure way, since trie-terms are simply Prolog terms and so earlier
trie-term database states are automatically recovered on backtracking
(unlike with Prolog's built-in assert and retract operations).

Trie-terms can be used to represent sets of ground Prolog terms (i.e.,
ground facts), which can be interpreted and manipulated as sets of
relations over Prolog terms.  Operations are provided for adding
elements to sets/relations generating new sets/relations, and for
deletion and lookup.  Also several set operations, e.g., union,
intersection, and difference, are provided.  These operations take
advantage of the indexed structure of a trie-term to sometimes provide
much better performance than more naive tuple-at-a-time algorithms.
This module also provides several restricted relational operators that
can also take advantage of the indexed data structure.

The complexity of adding, deleting, and looking up single elements in
a trie-term is approximately linear in the the number of symbols in
the clause being processed.  In the constant is a log factor based on
the size of the embedded radix trees.
	   
Since trie-terms are ground, they can be (and are) interned (a.k.a.
hash-consed). I.e., they are copied to a global space and each
distinct subterm is represented only once.  With this representation,
an interned trie-term is represented in Prolog as a specially named
atom, which in fact encodes a pointer to its global explicit term
representation.  Every predicate that manipulates trie-terms converts
the name to the pointer (or a pointer back to a name) whenever it uses
(or generates) a trie-term.  So a program that uses these trie-terms
passes around ``names'' of the clause sets, instead of the possibly
large terms representing the clause sets themselves.  These interned
trie-terms can be efficiently tabled since only the atomic name of a
trie-term will be copied to a table.  Also since these trie-terms are
canonical, two trie-term names are equal if and only if the sets
(trie-terms) that they name are equal.  Also using answer subsumption
with the subset operation on interned trie-terms may prove useful for
some applications.  ").

:- comment(module," The @tt{prolog_db} module defines a data type that
supports a very general facility for manipulating sets of Prolog
clauses represented as a single (interned, ground) Prolog term that
can be useful for a variety of purposes.  We will describe the
module's functionality though a series of examples, which use
progressively more complicated aspects of the data type.

@section{Sets of Atoms (or Terms)}

In its simplest form, the @tt{prolog_db} data type can be used to
manipulate sets of (ground) atoms efficiently.  For many applications
basic Prolog nondeterminism (perhaps with tabling) will effectively
handle sets of terms, but occasionally there are applications in which
a programmer wishes to maintain a term that explicitly represents a
set of atoms.  This is often done using a sorted list in Prolog;
sorting eliminates duplicates and canonicalizes a list to a set, so
then list equality implements set equality.  This works well for small
sets but, when sets get larger than a few 10's of items, performance
can suffer signicantly.  This can be avoided by using the
@tt{prolog_db} data type.  A new empty set, @var{_S0}, is created
with @pred{new_db(_S0)}; an element, say 'George', is added to it
with @pred{assert_in_db('George',_S0,_S1)} creating the set with
'George' added in @var{_S1}.  For example,

@begin{verbatim}
| ?- new_db(_S0),assert_in_db('George',_S0,_S1),dump_db(userout,_S1).
'George'.
@end{verbatim}

@noindent Dumping out the contents of @var{_S1}, we see it contains
just the atom 'George'.

Sets can be compared for identity with the Prolog operation @pred{==}
(or @pred{=}.)

@begin{verbatim}
| ?- new_db(_S0),assert_in_db('George',_S0,_S1),assert_in_db('Mary',_S1,_S2),
     assert_in_db('Mary',_S0,_S1B),assert_in_db('George',_S1B,_S2B),
     _S2 == _S2B, dump_db(userout,_S2).
'Mary'.
'George'.
@end{verbatim}

@noindent Here we have added 'George' to the empty set and then added
'Mary' to that set to obtain @var{_S2}.  Then we added 'Mary' to the
empty set and added 'George' to that set to obtain @var{_S2B}, then
checked that those two sets were equal (which they were, so it
continued) and dumped the contents of one.

A set can be checked to see if it contains a constant 'George' using
@pred{in_db('George',Set1)}.

@begin{verbatim}
| ?- new_db(_S0),assert_in_db('George',_S0,_S1),assert_in_db('Mary',_S1,_S2),
   (in_db('George',_S2) -> writeln('contains George') ; writeln(oops)),
   (in_db('William',_S2) -> writeln(oops) ; writeln('but not William')).
contains George
but not William
@end{verbatim}

@var{Set1} can be checked to see if it is a subset of @var{Set2} using
@pred{subset_db(Set1,Set2)}.  For example,

@begin{verbatim}
| ?- new_db(_S0),assert_in_db('George',_S0,_S1),assert_in_db('Mary',_S1,_S2),
   (subset_db(_S1,_S2) -> writeln('correctly subset') ; writeln(oops)),
   (subset_db(_S2,_S0) -> writeln('oops') ; writeln('correctly not subset')).
correctly subset
correctly not subset
@end{verbatim}

The union of sets @var{Set1} and @var{Set2} can be computed into
@var{Set3} using @pred{union_db(Set1,Set2,Set3)}.  For example,

@begin{verbatim}
| ?- new_db(_S0),assert_in_db('George',_S0,_S1),assert_in_db('Mary',_S0,_S2),
     union_db(_S1,_S2,_S3),dump_db(userout,_S3).
'Mary'.
'George'.
@end{verbatim}

Other similar set operations are also provided.  See the
@em{Documentation of exports} section below for details.  The time and
space complexities of all these operations are at or near optimal.
Note that ``names'' of sets, rather than the (possibly large) explicit
trie-terms themselves, become the values of the Prolog set variables.
This means that they can be efficiently tabled, since only the names
are copied into and out of a table, rather than the explicit
trie-terms themselves.  To see the names of the sets, we can remove
the underscores that prefix the set variable names in our previous
queries.  So repeating the union query above, we get:

@begin{verbatim}
| ?- new_db(S0),assert_in_db('George',S0,S1),assert_in_db('Mary',S0,S2),
     union_db(S1,S2,S3),dump_db(userout,S3).
'Mary'.
'George'.

S0 = []
S1 = }]]}0000028A1F52C2A1
S2 = }]]}0000028A1F52C2C9
S3 = }]]}0000028A1F567C11
@end{verbatim}

@noindent We see that a named set is represented as an atom whose name
encodes the address of the trie-term structure in interned space.
(The prefix '}]]}' is to try to avoid name conflicts with user
strings.)  The empty set is a special case, since it is not a
structured term, but just the constant [ ].

Of course, elements can be deleted from sets.  This is done using
@tt{retractall_in_db(Atom,Set0,Set)}.  For example:

@begin{verbatim}
| ?- new_db(_S0),assert_in_db(['George','Mary'],_S0,_S1),dump_db(userout,_S1),
     retractall_in_db('George',_S1,_S2),nl,dump_db(userout,_S2).
'Mary'.
'George'.

'Mary'.
@end{verbatim}

@noindent Here we insert both 'George' and 'Mary' into the empty set
(@pred{assert_in_db/3} will accept a list in its first argument), print
out that set, then remove 'George' from that set, and print out the
result.

The functionality of the @tt{prolog_db} data type that is required to
efficiently support sets of atoms is the indexing it provides at a
single point in a trie, and not in the multiple levels of the trie.
Since for a set of atoms, there is really no trie.

@subsection{Example: Paths in Graphs}

We can use @tt{prolog_db} trie-term sets to collect path edges when
computing paths in a graph.  Consider the following program:

@begin{verbatim}
:- import subset_db/2, assert_in_db/3, assert_in_db/2, dump_db/2 from prolog_db.

:- table path(_,_,po(subset_db(_,_))).
path(X,Y,S) :- edge(X,Y), assert_in_db((X->Y),S).
path(X,Y,S) :- path(X,Z,S1),edge(Z,Y),assert_in_db((Z->Y),S1,S).

edge(a,b).    edge(e,f).
edge(a,c).    edge(d,f).
edge(c,d).    edge(f,c).
edge(b,e).    edge(f,g).
@end{verbatim}

Here @pred{path(X,Y,S)} is defined for paths in the @pred{edge/2}
graph where node @var{Y} is reachable from node @var{X} along a path
consisting of minimal sets of edges in set @var{S}.  The @tt{table}
declaration uses answer subsumption to ensure that only the shortest
paths (those with the minimal edge sets) are kept.  So this will work
for all graphs returning, for desired nodes, all minimal paths between
them, e.g., those containing no loops.  As an example, to find the
minimal paths from @tt{a} to @tt{d} in the indicated graph, we can do:

@begin{verbatim}
| ?- X=a,Y=d,path(X,Y,S),nl,writeln((X-->Y)),prolog_db:dump_db(userout,S),fail.

(a --> d)
->(a,c).
->(c,d).

(a --> d)
->(a,b).
->(e,f).
->(b,e).
->(f,c).
->(c,d).

no
| ?- 
@end{verbatim}

@noindent Note that we have two such minimal paths from @tt{a} to
@tt{d} in this graph.  There are, of course, infinitely many distinct
paths from @tt{a} to @tt{d} of the form @tt{acd(fcd)*} in this graph.

By changing the answer subsumption aggregation predicate from
@pred{subset_db/2} to, say, @pred{smaller_db/2} (which we could write
using @pred{compare_size_db/3}), we could compute just the shortest
path(s).

@subsection{Example: Too Many Parses} There are context-free grammars
which for some input strings have many distinct parses, exponentially
many in the length of the input string.  If we are not careful, in
trying to compute the parses, we may take exponential time
constructing parses for substrings of the input string when the full
input string is not itself in the language.

Consider the grammar:
@begin{verbatim}
a --> a, a.
a --> [a].
@end{verbatim}

@noindent This grammar recognizes all strings consisting of one or
more @tt{a}'s.  The natural way to construct parses for this grammar
is with the following Prolog program:

@begin{verbatim}
:- table a/3.
a(a(P1,P2)) --> a(P1), a(P2).
a(a) --> [a].
@end{verbatim}

@noindent in which we've added an argument to hold the Prolog term
that represents the parse.  The exponential behavior arises from the
first rule: P1 may have @tt{m} different values for a given substring
it covers, and P2 may have @tt{n} different values for its substring;
thus a(P1,P2), the parse of the full string, would take on @tt{m * n}
different values.

We can short-circuit this combinatorial explosion by, instead of
representing the parses explicitly, representing, as a trie-term, the
set of parses at a position, and using that set in larger parses. We
can thereby avoid the multiplication that causes the combinatorial
explosion.  In this representation, in the above program @var{P1} and
@var{P2} would take on as values @em{names} of sets of terms.

We can use answer subsumption to carry this out.  Consider the
following program:
@begin{verbatim}
:- import assert_in_db/3, new_db/1, is_db/1, in_db/2 from prolog_db.

:- table a(fold(collect/3,new_db/1),_,_).
a(a(P1,P2)) --> a(P1), a(P2).
a(a) --> [a].

collect(DB0,Parse,DB) :- assert_in_db(Parse,DB0,DB).

term_from_setterm(Term,ETerm) :-
    (is_db(Term)
     ->	in_db(RTerm,Term),
        term_from_setterm(RTerm,ETerm)
     ; atomic(Term)
     ->	ETerm = Term
     ;	Term =.. [F|Args],
        term_from_setterm_list(Args,EArgs),
        ETerm =.. [F|EArgs]
    ).

term_from_setterm_list([],[]).
term_from_setterm_list([T|Ts],[ET|ETs]) :-
    term_from_setterm(T,ET),
    term_from_setterm_list(Ts,ETs).
@end{verbatim}

@noindent The answer-subsumptive aggregation predicate is
@pred{collect/3}, which simply adds a newly computed parse to the
current set.  The predicate @pred{term_from_setterm(+Term,-ETerm)}
takes a term, @var{Term}, that may contain named trie-terms that
represent sets as subcomponents, and replaced each such trie-term
instance with the terms in that set, in turn.  So now we can parse a
string and generate its parses as follows:

@begin{verbatim}
| ?- a(_T,[a,a,a,a],[]),term_from_setterm(_T,P).
P = a(a,a(a,a(a,a)));
P = a(a,a(a(a,a),a));
P = a(a(a,a),a(a,a));
P = a(a(a,a(a,a)),a);
P = a(a(a(a,a),a),a);
no
| ?- 
@end{verbatim}

@noindent So here, the generation of the term @var{_T} by calling the
grammar rule @pred{a/3} takes polynomial time; it is the
@pred{term_from_setterm(_T,P)} that will take exponential time to
produce the combinatorially many parses.

@section{Sets of Relations}

Another way the @tt{prolog_db} data type can be used is to store an
extensional relational database in a Prolog variable.  A set of ground
terms can be interpreted as defining a set of relations by
interpreting the main functor symbol of each term as a relation name,
and the subterms of that structure as the field values of a tuple of
that relation.  This is exactly how Prolog would treat such a set of
terms were they to be asserted into Prolog's clause database.

For example, one can create a trie-term containing two r/2 tuples and
three q/3 tuples in @var{_DB} with
@begin{verbatim}
| ?- assert_in_db([r(a,b),r(b,c),q(a,2,3),q(b,3,4),q(b,4,5)],_DB),
     dump_db(userout,_DB).
q(a,2,3).
q(b,4,5).
q(b,3,4).
r(a,b).
r(b,c).
@end{verbatim}
Given a database name, we can ask queries of that database.  For
example,
@begin{verbatim}
| ?- assert_in_db([r(a,b),r(b,c),q(a,2,3),q(b,3,4),q(b,4,5)],_DB),
     in_db(q(b,X,Y),_DB).

X = 4
Y = 5;

X = 3
Y = 4;

no
| ?- 
@end{verbatim}

@noindent creates the desired trie database name and then evaluates
the query @pred{q(b,X,Y)} in the indicated database and
nondeterministically returns the two answer pairs as shown.
	   
We can also ask more complex queries, as seen with the following join:
@begin{verbatim}
| ?- assert_in_db([r(a,b),r(b,c),q(a,2,3),q(b,3,4),q(b,4,5)],_DB),
     in_db((r(a,Y),q(Y,Z,W)),_DB).

Y = b
Z = 4
W = 5;

Y = b
Z = 3
W = 4;

no
| ?- 
@end{verbatim}

The set operations of @pred{union_db/3}, @pred{intersect_db/3},
etc., can be used with sets representing relational databases.  For
example the union of two relational databases can be directly
computed.  And the operations can be very efficient.  For example, if
the two databases do not contain a table name in common, then their
union is computed with a complexity that is dependent on the number of
distinct tables but independent of their sizes.

The @tt{prolog_db} data type provides operations designed to be used
with extensional relational databases stored as @tt{prolog_db} trie-terms.
These operations take an input database, probably containing multiple
relations, and return an output database that (usually) contains a new
relation added by the operation.  These operations depend on, and take
advantage of, the structure of the trie-terms that represent sets of
(tuples of) terms for their efficient implementation.

Just as the data type contains set operations on full databases, it
contains set operations on tables within a single database.  So, for
example:

@begin{verbatim}
| ?- assert_in_db([r(a,b),r(b,c),s(1,2),s(2,3)],_DB1),
     union_in_db(r/2,s/2,t/2,_DB1,_DB), dump_db(userout,_DB).
t(1,2).
t(a,b).
t(2,3).
t(b,c).
r(a,b).
r(b,c).
s(1,2).
s(2,3).
@end{verbatim}

@noindent initializes a @var{DB1} with tables @pred{r/2} and
@pred{s/2}, then takes the two tables @pred{r/2} and @pred{s/2} in
database @var{DB1} and uses @pred{union_in_db/5} (@em{not}
@pred{union_db/3}) to produce a new database in @var{DB} that contains
all tables in @var{DB1} plus a new table @pred{t/2} that contains the
union of tables @pred{r/2} and @pred{s/2}.  Had @pred{t/2} been in
DB1, it would have been replaced with the new definition.

@subsection{Example: Semi-Naive Transitive Closure}

We'll introduce several other single database operators that are
supported by the @tt{prolog_db} abstract data type with the following
more complicated example.  Assume we have a trie-term relational
database that contains an edge relation @pred{e/2}.  We want to
compute (the converse of) the transitive closure of this @pred{e/2}
relation into a new relation @pred{p/2}.  It turns out that with the
operations we have, the converse of the transitive closure is easier
to compute directly, so that's what we will compute.

Consider the program:
@begin{verbatim}
:- import retractall_in_db/3, reorder_in_db/4, difference_in_db/5, is_empty_in_db/3,
    union_in_db/5, join_in_db/6, project_in_db/5 from prolog_db.
:- import assert_in_db/2, dump_db/2 from prolog_db.

test(Edges) :-
    assert_in_db(Edges,DB0),
    transclose(DB0,DB),
    dump_db(userout,DB).


transclose -->
    retractall_in_db(p(_,_)),
    reorder_in_db(e(A,B),delta(B,A)),
    transclose_loop.

transclose_loop -->
    difference_in_db(delta/2,p/2,delta/2),
    (is_empty_in_db(delta/2)
     ->	[]
     ;  union_in_db(delta/2,p/2,p/2),
        join_in_db(e/2,delta/2,1,delta/3),
        project_in_db(delta/3,1,delta/2),
        retractall_in_db(delta(_,_,_)),
        transclose_loop
    ).
@end{verbatim}

The imports give us access to the @tt{prolog_db} predicates we
need. The predicate @pred{test/1} is a simple driver that takes a list
of @pred{e/2} terms, initializes a @var{DB0} with them, calls
@pred{transclose/2} to compute the (converse of the) transitive
closure, and then prints out the @var{DB} returned.  The work of
computing the transitive closure of @pred{e/2} is all done in
predicates @pred{transclose/2} and @pred{transclose_loop/2}.

Note first that we use the DCG transformation in our definitions of
these two predicates, i.e., the rules are defined using the operator
@pred{-->}.  The DCG transformation adds two arguments to each subgoal
in the body of a DCG rule and chains those added variables.  This in
effect treats each subgoal in a DCG rule as a DB transformer, which
takes a DB as input and produces a DB as output.  This is exactly what
each of these @tt{prolog_db} predicates does, and so the DCG notation
is natural for this specification.

So @pred{transclose/2} transforms its input DB by first deleting the
p/2 table (if any).  It then uses @pred{reorder_in_db/4} to generate a
new table for @pred{delta/2} consisting of the @pred{e/2} tuples with
their arguments reordered (i.e., the converse of @pred{e/2}.)  This in
effect generates a new version of the @pred{e/2} table with different
indexing.  Then it calls @pred{transclose_loop/2} to iterate to
compute the transitive closure of @pred{e/2}.

The predicate @pred{transclose_loop/2} iteratively transforms the
database by accumulating the @pred{delta/2} table into the @pred{p/2}
table and joining the @pred{e/2} table to the new delta table at each
iteration to get a new set of pairs in the transitive closure of
@pred{e/2}.  So the @pred{difference_in_db(delta/2,p/2,delta/2)}
transformation removes from @pred{delta/2} those pairs that are
already in the transitive closure, since we want to continue with
@pred{delta/2} containing only new pairs.  Then if there are no new
pairs, the computation has converged and it returns the current DB.
Note that we import @pred{is_empty_in_db/3}, which is defined in
@tt{prolog_db} as @pred{is_empty_in_db/2} as an identity DB
transformer for use in a DCG rule.  But if there are new tuples in
@pred{delta/2}, we union (accumulate) them into @pred{p/2}.  Then we
perform a join of those tuples with the @pred{e/2} tuples to create
the new @pred{delta} tuples.  The @pred{join_in_db/6} predicate takes
two tables to join and performs an equi-join on them using an initial
sequence of their arguments (here, just 1) and produces a table with
one copy of the join argument(s) and all the other arguments of the
input tables.  This is a very restricted form of join and it is
supported because, with this form of join, the join algorithm can take
advantage of the trie-term data structure to perform efficiently.
Other join forms (and sometimes even this one) can be more efficiently
implemented using @pred{in_db/2}.  After the join, the
@pred{project_in_db(delta/3,1,delta/2)} projects out the first
argument of the @pred{delta/3} table generating the new @pred{delta/2}
table.  Finally the @pred{delta/3} table is deleted, and the loop is
again invoked with the PDB containing the updated @pred{p/2} table and
the new @pred{delta/2} table.

We can see the results of the computation on a given edge relation:
@begin{verbatim}
| ?- test([e(1,2),e(2,3),e(3,4),e(3,1)]).
p(4,1).
p(4,2).
p(4,3).
p(1,1).
p(1,2).
p(1,3).
p(2,1).
p(2,2).
p(2,3).
p(3,1).
p(3,2).
p(3,3).
e(1,2).
e(2,3).
e(3,4).
e(3,1).
@end{verbatim}
@noindent Remember that @pred{p/2} contains the @em{converse} of the
transitive closure of @pred{e/2}, so e.g., @pred{p(4,1)} indicates
that the pair @tt{[1,4]} is in the transitive closure of @pred{e/2}.

@section{Full Prolog Databases}

So far we have only stored ground terms in @tt{prolog_db} trie-term
databases.  But one can add rules to these databases as well.
Consider the following example:

@begin{verbatim}
| ?- assert_in_db([e(1,2),e(2,3),e(3,4),e(3,1)],DB0),
     assert_in_db([(p(X,Y) :- e(X,Y)), (p(X,Y) :- table(p(X,Z)),e(Z,Y))],DB0,DB1),
     in_db(table(p(X,Y)),DB1),writeln(p(X,Y)),fail.
p(3,3)
p(3,2)
p(3,1)
p(3,4)
p(2,2)
p(2,1)
p(2,4)
p(2,3)
p(1,1)
p(1,4)
p(1,3)
p(1,2)
@end{verbatim}

@noindent Here we initially created a database with four @pred{e/2}
facts.  Then we added two rules to it, to define @pred{p/2} as the
transitive closure of @pred{e/2}.  The recursive rule is
left-recursive, so we indicate that calls to that left-recursive goal
should be table-d.  Then in this database with the facts and the
rules, we call the goal @pred{table(p(X,Y))}, which calls @pred{p/2},
tabling the call.  This query then returns the answers to the
rule-defined @pred{p(X,Y)}.

A few caveats: As seen in this example, tabling must be specified at
the call level by wrapping the call of a goal to be tabled with the
functor @tt{table}.  The trie-term databases of @tt{prolog_db} do not
support ordering or duplicates, so rules whose evalutation depend on
those facilities are not supported.  In particular, cuts (!) in rules
are not allowed.  See the detailed documentation on
@pred{in_db/2} for more particulars.

@subsection{Example: STRIPS Planning}

As an example of search through a graph of databases, consider a block
planning problem as specified in STRIPS (ref?).  The situation is a
set of same-sized blocks in a configuration on a table, with some of
the blocks perhaps stacked on top of others in towers.  The problem
is: given an initial state of the blocks configuration and a final
state, what are the operations of moving the blocks that can turn the
initial state into the final state?  A block configuration is described
by a set of facts using the predicates: @pred{on(X,Y)} indicating
block @var{X} is immediately on top of block @var{Y}; @pred{onTable(X)}
indicating that block @var{X} is resting directly on the table;
@pred{clear(X)} indicating that block has no block resting on top of
it; @pred{holds(X)} indicating that the hand is holding block @var{X};
and @pred{handEmpty} indicating that the hand is empty.  We note that
@pred{handEmpty} can be defined in terms of @pred{holds(X)}.  The hand
has four actions it can perform: pickup a block from the table, put
down a block on the table, stack a block on top of another block, and
unstack a block from the top of another block.

Consider the following program:

@begin{verbatim}
:- import in_db/3, assert_in_db/3, retractall_in_db/3,
    dump_db/3, assert_in_db/2 from prolog_db.

pickup(X) -->
    in_db((clear(X),onTable(X),handEmpty)),
    assert_in_db([holds(X)]),
    retractall_in_db([clear(X),onTable(X)]).

putdown(X) -->
    in_db((holds(X))),
    assert_in_db([clear(X),onTable(X)]),
    retractall_in_db([holds(X)]).

stack(X,Y) -->
    in_db((holds(X),clear(Y))),
    assert_in_db([clear(X),on(X,Y)]),
    retractall_in_db([clear(Y),holds(X)]).

unstack(X,Y) -->
    in_db((clear(X),on(X,Y),handEmpty)),
    assert_in_db([clear(Y),holds(X)]),
    retractall_in_db([clear(X),on(X,Y)]).

action --> pickup(_X).
action --> putdown(_X).
action --> stack(_X,_Y).
action --> unstack(_X,_Y).

:- table action_sequence/2.
action_sequence --> action, dump_db(userout).
action_sequence --> action_sequence, action, dump_db(userout).

test(Source,Target) :-
    assert_in_db([(handEmpty :- \\+ holds(_))|Source],SDB),
    assert_in_db([(handEmpty :- \\+ holds(_))|Target],TDB),
    action_sequence(SDB,TDB).
@end{verbatim}

Actions can transform the current state to a new state, but actions
can be performed only if certain preconditions hold.  So each action
is associated with 1) a precondition, 2) a set of facts to add to the
current state, and 3) a set of facts to remove from the current state.
So, for example the @pred{pickup(X)} action, as defined in this
program, requires that block @var{X} be clear in the curent state,
that it be on the table, and that the hand be empty.  If these are
true, then the action can be performed and a new state is generated by
adding the fact that the hand holds block X, and by deleting the facts
that @var{X} is clear, and that @var{X} is on the table.  Similarly,
the other three actions have their own preconditions, facts to add,
and facts to delete to generate the state after application of the
action.  Again, these actions are specified as database transformers
and so we use the DCG notation with its implicit database arguments.

The predicate @pred{action_sequence/2} is defined to be the transitive
closure of the action DB transformer, so it will search paths through
the graph of database states whose edges are actions that transform
the preceding states into the succeeding ones.  We need to table this
predicate since this graph clearly contains cycles.  (Ref. Reza's
thesis).

The predicate @pred{test/2} sets up a desired initial state (including
the rule to define @pred{handEmpty}) and final state and calls
@pred{action_sequence/2} to carry out a sequence of actions that gets
the system from the initial state to the target state.  This program
does not save that path, and does not search for a minimal path.
These functionalities would not be difficult to add (e.g. see
C.R. Ramakrishnan's paper on search.)

We give partial output for a query to find a sequence of actions to
turn an initial state into a final state.  We elide many intermediate
states.  (A total of 500 were generated.)

@begin{verbatim}
| ?- test([onTable(a),on(b,c),onTable(c),onTable(d),clear(a),clear(b),clear(d)],
    [onTable(b),on(c,b),on(a,c),on(d,a),clear(d)]).
clear(a).
clear(b).
holds(d).
on(b,c).
onTable(a).
onTable(c).
:-(handEmpty,\\+(holds(A))).

clear(d).
clear(b).
holds(a).
on(b,c).
onTable(d).
onTable(c).
:-(handEmpty,\\+(holds(A))).

.......

clear(a).
holds(d).
on(a,c).
on(c,b).
onTable(b).
:-(handEmpty,\\+(holds(A))).

clear(d).
on(d,a).
on(a,c).
on(c,b).
onTable(b).
:-(handEmpty,\\+(holds(A))).

yes
| ?- 
@end{verbatim}

@section{Representation Details}

We turn to how a database is represented as a term in the
@tt{prolog_db} data type.  A @tt{prolog_db} database is represented as
a ground Prolog term (or the name of such an interned term).  The term
represents a trie that is indexed at each potential branch point.

The first problem is how to store a set of key-value pairs so that
given a key and value, we can efficiently add the pair to the table,
and given a key, efficiently access its paired value in the table.
There are many data structures that support these operations.  We use
a variant of the radix tree data structure.  We'll describe the
variant we use by starting by assuming that the keys are n-bit
integers.  We use a 4-way tree to store the index.  The last 2 bits of
the key determine which of the four subtrees of the root contain the
value for the key.  The next 2 bits of the key determine which subtree
of that subtree contains the value, etc.

So as a first approximation, if we have the following 3 6-bit integers

@begin{verbatim}
15: 00 11 11
24: 01 10 00
34: 10 00 10
@end{verbatim}

@noindent they would be stored as in the following index tree:

@begin{verbatim}
rt( rt([],[],rt([],V24,[],[]),[]),
    []
    rt(rt([],[],V34,[]),[],[],[]),
    rt([],[],[],rt(V15,[],[],[]))
 )
@end{verbatim}

@begin{enumerate}
@item The 15 (00 11 11) position is found by taking the fourth (11)
subtree of the main tree, then the fourth (11) subtree of that, and
then the first (00) subtree of that.

@item The 24 (or 01 10 00) position is found by taking the first (00)
subtree of the main tree, and then the third subtree (10) of that, and
then the second (01) subtree of that.

@item The 34 (10 00 10) position is found by taking the third (10)
subtree of the main tree, then the first (00) subtree of that, and
then the third (10) subtree of that.

@end{enumerate}

We normally want to use a bitsize much larger than 6, maybe 32.  In
this case every node would be 16 hops from the root, which would incur
more overhead than desired.  So we shrink this data structure by
(iteratively) merging a leaf node with its parent, if the parent has
just one child.  So the actual tree we would store for the above
example is:

@begin{verbatim}
rt( rtf(V24),
    []
    rtf(V34),
    rtf(V15)
 )
@end{verbatim}

We use the rtf-term to indicate a leaf node.  A full radix tree would
merge all only-children (i.e. including internal ones) with their
parents.  That would require a much more complex algorithm and it
seems unlikely that in most of our use cases, that full collapsing
would be worth the effort.

Any subtree that contains no values is represented by [ ].  

The position accessed by a key integer is used to store the list of
values associated with that key (using a second field of the rtf
record).

We will call these trees, index trees.

For our trie-terms, we want to index on the symbols in a Prolog term,
and not n-bit integers.  But we can turn symbols into n-bit integers
by choosing a hash function whose output value is an appropriately
sized integer. 

Now a trie-term is just a set of nested index trees, where the value
associated with a symbol key is an index tree for the remainder of the
trie.  (The last value is the body(s) of rule(s) whose preordered head
generated the symbol sequence through the trie.)

@subsection{Prefix Terms}

In the above section describing the @tt{prolog_db} predicates that
treat a single trie-term as representing a set of extensional
relational tables, we identified the table operands by using a simple
predicate/arity notation.  But in fact there is a more general way to
specify a set of tuples for those table operations to manipulate.

We need to look a little more closely at the trie
data structure that represents sets of terms.  Consider the set of
terms:
@begin{verbatim}
   r(a,b,c).
   r(a,b,d).
   r(a,e,f).
   r(a,f(a,b),c).
   r(b,g,c).
@end{verbatim}
@noindent and their pre-order traversals in a trie:
@begin{verbatim}
   r/3 - a - b - c
               - d	   
             e - f
             f/2 - a - b - c
       - b - g - c
@end{verbatim}

@noindent The position of each symbol in the trie can be thought of as
the root of a sub-trie.  We can indicate such a sub-trie by using a
``prefix term,'' which is a Prolog term whose pre-ordering consists of
an initial sequence of ground symbols followed a sequence of distinct
variables.  For example, @tt{r(_,_,_)} and @tt{r(a,f(a,_),_)} are
prefix terms, but @tt{r(_,b,_)} is not.  Associated with a prefix term
(and with its identified sub-trie) is an integer (called the arity)
that indicates the number of variables in the prefix term and thus the
number of terms that would be retrieved by a pass through the
determined sub-trie.  The full trie always has arity 1, i.e. a full
traversal through the entire trie will produce a single term.  In our
example, the sub-trie determined by the prefix term r(_,_,_), which is
rooted after the @tt{r/3} has arity 3, since starting at that node
will retrieve 3-tuples from the sub-trie; in our example, [a,b,c],
[a,b,d], [a,e,f], [a,f(a,b),c], and [b,g,c].  So the prefix term
r(_,_,_) is natural way to indicate the sub-trie that determines
tuples in the @tt{r/3} table.  As another example, the prefix term
@tt{r(a,f(_,_),_)} determines a sub-trie that contains triples, here
just the single triple, [a,b,c].

The arguments in the call to, say, @pred{union_in_db(P1,P2,P3,DB0,DB)}
that indicate tables, i.e., @var{P1}, @var{P2}, and @var{P3}, can
actually be prefix terms, and they refer to a set of tuples in a
trie-term as described in the previous paragraph.  So, in fact, a
table specification of @pred{r/3}, say, is simply an accepted
short-hand for the actual prefix term @pred{r(_,_,_)}.

").    

/***********************************************************/

:- comment(assert_in_db/3, "@pred{assert_in_db(+Clause,+DB0,?DB)} adds
a clause, @var{Clause}, (or a sequence of clauses, if @var{Clause} is
a list of clauses) to @var{DB0} to generate @var{DB}. ").
:- mode assert_in_db(?,+,?).
assert_in_db(Clauses,H0,H) :-
    (H0 == module
     ->	H = H0,
	(Clauses = [Clause|MClauses]
	 -> assert(Clause),
	    assert_list(MClauses)
	 ;  assert(Clauses)
	)
     ;  tab_intern_termhash(DB0,H0),
	assert_in_db_b(Clauses,DB0,DB),
	tab_intern_termhash(DB,H)
    ).

%% simpler alias if desired.
add(Clauses,H0,H) :- assert_in_db(Clauses,H0,H).

%% bug if do this, for very large metainterp??
%%:- import (table)/1 from standard.
%%:- table tab_intern_termhash/2 as intern.
tab_intern_termhash(DB0,H0) :-
    intern_termhash(DB0,H0).

assert_list([]).
assert_list([Clause|Clauses]) :-
    assert(Clause),
    assert_list(Clauses).

assert_in_db_b(Clauses,DB0,DB) :-
    (Clauses == []
     ->	DB0 = DB
     ; Clauses = [Clause|MClauses]
     ->	assert1_in_db_b(Clause,DB0,DB1),
	assert_in_db_b(MClauses,DB1,DB)
     ;	assert1_in_db_b(Clauses,DB0,DB)
    ).

:- comment(assert1_in_db/3, "@pred{assert1_in_db(+Clause,+DB0,?DB)}
asserts a single clause, @var{Clause}, into @var{DB0}, generating
@var{DB}.  Normally @pred{assert_in_db/3} should be used instead of
this predicate.  This predicate is needed @em{only} in cases in which
the ``clause'' to be asserted is itself a list, and so interpreting it
as a list of clauses would be incorrect. ").
:- mode assert1_in_db(?,+,?).
assert1_in_db(Clause,H0,H) :-
    tab_intern_termhash(DB0,H0),
    assert1_in_db_b(Clause,DB0,DB),
    tab_intern_termhash(DB,H).

assert1_in_db_b(Clause,DB0,DB) :-
    (Clause = (Head :- Body)
     ->	true
     ;	Head = Clause,
	Body = true
    ),
    add_to_trie(Head,Body,DB0,DB).

:- comment(assert_in_db/2, "@pred{assert_in_db(+Clause,?DB)} adds a
clause, @var{Clause}, (or a sequence of clauses, if @var{Clause} is a
list) to the empty DB to generate @var{DB}.").
:- mode assert_in_db(?,?).
assert_in_db(Clauses,H) :-
    new_db(DB0),
    assert_in_db(Clauses,DB0,DB),
    tab_intern_termhash(DB,H).

%% simpler alias if desired.
add(Clauses,H) :- assert_in_db(Clauses,H).

add_to_trie(Head,Body,Trie0,Trie) :-
    (ground([Head|Body])
     ->	add_tuple_to_trie([Head],Body,Trie0,Trie)
     ;	copy_term([Head|Body],[Head1|Body1]),
	my_numbervars([Head1|Body1],0,_),
	add_tuple_to_trie([Head1],Body1,Trie0,Trie)
    ).

add_tuple_to_trie(Terms,Body,Trie0,Trie) :-
    (Terms \== []
     ->	get_key_new_terms(Terms,Key,NTerms),
	add_to_radix_tree(Key,NewSubTrie,OldSubTrie,Trie0,Trie),
	add_tuple_to_trie(NTerms,Body,OldSubTrie,NewSubTrie)
     ;	order_insert(Trie0,Body,Trie)
    ).

order_insert(ACVs,V,NVs) :-
    (ACVs == []
     ->	NVs = [V]
     ;	ACVs = [CV|CVs],
	term_compare(CV,V,Ord),
	(Ord < 0
	 -> NVs = [CV|NVs1],
	    order_insert(CVs,V,NVs1)
	 ; Ord > 0
	 -> NVs = [V|ACVs]
	 ;  NVs = ACVs
	)
    ).

%%:- table hash_key/2. % breaks it, why? Certainly looks like a
%%bug. Maybe complications with in_db metainterp
%%#define HASHBASE 536870027

hash_key(Key,HashCode) :-
    (Key @= functor(_,_,_)
     ->	Key = functor(_Mod,Fun,Ari),
	term_hash(Fun,536870027,HashCode0),
	HashCode is (HashCode0 + Ari) mod 536870027
     ;	term_hash(Key,536870027,HashCode)
    ).
%%    term_hash(Key,13,HashCode).

get_key_new_terms([Term|Terms],Key,NTerms) :-
    (var(Term)
     ->	NTerms = Terms,
	Key = Term
     ; atomic(Term)
     ->	NTerms = Terms,
	Key = Term
     ; get_key_new_terms1(Term,Terms,Key,NTerms)
    ).

get_key_new_terms1(Term,Terms,Key,NTerms) :-
    (varno(Term)
     ->	NTerms = Terms,
	Key = Term
     ;	functor(Term,Mod,Fun,Ari),
	Key = functor(Mod,Fun,Ari),
	Term =.. [Fun|SubTerms],
	append(SubTerms,Terms,NTerms)
    ).    

match_key_new_terms([Term|Terms],Key,NTerms) :-
    (Key = functor(Mod,Fun,Ari)
     ->	my_functor(Term,Mod,Fun,Ari),
	Term =.. [_|SubTerms],
	append(SubTerms,Terms,NTerms)
     ;	Term = Key,
	NTerms = Terms
    ).

% add_to_radix_tree/5
add_to_radix_tree(Key,NVal,OVal,Tree0,Tree) :-
    hash_key(Key,HashCode),
    add_to_radix_tree(Tree0,Key,NVal,HashCode,0,OVal,Tree).
    
%add_to_radix_tree(Tree0,Key,NVal,HashCode,Depth,OVal,Tree)
add_to_radix_tree([],Key,Val,_,_,[],rtl(Key,Val,[])).
add_to_radix_tree(ORTL,Key,Val,HashCode,Depth,OVal,Tree) :- ORTL = rtl(OKey,_,_),
    hash_key(OKey,HashOCode),
    (HashCode =:= HashOCode
     ->	rtl_insert(ORTL,Key,Val,OVal,Tree)
     ;	OVal = [], % adding a new entry so old is []
	add_pair_to_empty_radix_tree(HashCode,Key,Val,HashOCode,ORTL,Depth,Tree)
    ).
add_to_radix_tree(RT,Key,ValTran,HashCode,Depth,OVal,Tree) :- RT = rt(_,_,_,_),
    Index is (HashCode >> Depth) /\ 3,
    NDepth is Depth + 2,
    add_to_right_radix_tree(RT,Key,ValTran,HashCode,NDepth,OVal,Tree,Index).

:- index add_to_right_radix_tree/8-8.
add_to_right_radix_tree(rt(RT1,RT2,RT3,RT4),Key,VT,HashCode,D,OVal,rt(NRT,RT2,RT3,RT4),0) :-
    add_to_radix_tree(RT1,Key,VT,HashCode,D,OVal,NRT).
add_to_right_radix_tree(rt(RT1,RT2,RT3,RT4),Key,VT,HashCode,D,OVal,rt(RT1,NRT,RT3,RT4),1) :-
    add_to_radix_tree(RT2,Key,VT,HashCode,D,OVal,NRT).
add_to_right_radix_tree(rt(RT1,RT2,RT3,RT4),Key,VT,HashCode,D,OVal,rt(RT1,RT2,NRT,RT4),2) :-
    add_to_radix_tree(RT3,Key,VT,HashCode,D,OVal,NRT).
add_to_right_radix_tree(rt(RT1,RT2,RT3,RT4),Key,VT,HashCode,D,OVal,rt(RT1,RT2,RT3,NRT),3) :-
    add_to_radix_tree(RT4,Key,VT,HashCode,D,OVal,NRT).

add_pair_to_empty_radix_tree(HashCode,Key,Val,OHashCode,ORTL,Depth,Tree) :-
    (HashCode =:= OHashCode
     ->	rtl_insert(ORTL,Key,Val,_OVal,Tree)
     ;	Index is (HashCode >> Depth) /\ 3,
	OIndex is (OHashCode >> Depth) /\ 3,
	(Index =\= OIndex
	 -> PIndex is 4*Index + OIndex,
	    add_pair(PIndex,rtl(Key,Val,[]),ORTL,Tree)
	 ;  Depth1 is Depth + 2,
	    build_and_add(HashCode,Key,Val,OHashCode,ORTL,Depth1,Tree,Index)
	)
    ).

:- index build_and_add/8-8.
build_and_add(HashCode,Key,Val,OHashCode,ORTL,D,rt(NT,[],[],[]),0) :-
    add_pair_to_empty_radix_tree(HashCode,Key,Val,OHashCode,ORTL,D,NT).
build_and_add(HashCode,Key,Val,OHashCode,ORTL,D,rt([],NT,[],[]),1) :-
    add_pair_to_empty_radix_tree(HashCode,Key,Val,OHashCode,ORTL,D,NT).
build_and_add(HashCode,Key,Val,OHashCode,ORTL,D,rt([],[],NT,[]),2) :-
    add_pair_to_empty_radix_tree(HashCode,Key,Val,OHashCode,ORTL,D,NT).
build_and_add(HashCode,Key,Val,OHashCode,ORTL,D,rt([],[],[],NT),3) :-
    add_pair_to_empty_radix_tree(HashCode,Key,Val,OHashCode,ORTL,D,NT).

add_pair(1/*0,1*/,NRTL,ORTL,rt(NRTL,ORTL,[],[])).
add_pair(2/*0,2*/,NRTL,ORTL,rt(NRTL,[],ORTL,[])).
add_pair(3/*0,3*/,NRTL,ORTL,rt(NRTL,[],[],ORTL)).
add_pair(4/*1,0*/,NRTL,ORTL,rt(ORTL,NRTL,[],[])).
add_pair(6/*1,2*/,NRTL,ORTL,rt([],NRTL,ORTL,[])).
add_pair(7/*1,3*/,NRTL,ORTL,rt([],NRTL,[],ORTL)).
add_pair(8/*2,0*/,NRTL,ORTL,rt(ORTL,[],NRTL,[])).
add_pair(9/*2,1*/,NRTL,ORTL,rt([],ORTL,NRTL,[])).
add_pair(11/*2,3*/,NRTL,ORTL,rt([],[],NRTL,ORTL)).
add_pair(12/*3,0*/,NRTL,ORTL,rt(ORTL,[],[],NRTL)).
add_pair(13/*3,1*/,NRTL,ORTL,rt([],ORTL,[],NRTL)).
add_pair(14/*3,2*/,NRTL,ORTL,rt([],[],ORTL,NRTL)).

/***********************************************************/

%% for now
:- comment(clause_in_db/3,"@pred{clause_in_db(?Head,?Body,+DB)}
unifies, in turn, @var{Head} and @var{Body} with the head and body of
rules in @var{DB}.").
:- mode clause_in_db(?,?,+).
clause_in_db(Head,Body,H) :-
    (H == module
     ->	clause(Head,Body)
     ;	tab_intern_termhash(DB,H),
	clause_in_db_b(Head,Body,DB)
    ).

:- comment(clause_in_db/4,"@pred{clause_in_db(?Head,?Body,-Vars,+DB)}
unifies, in turn, @var{Head} and @var{Body} with the head and body of
rules in @var{DB}.  @var{Body} contains numbervarred variables, and
@var{Vars} is a log_list of variable values.").
:- mode clause_in_db(?,?,?,+).
clause_in_db(Head,Body,Vars,H) :-
    (H == module
     ->	clause(Head,Body)
     ;	tab_intern_termhash(DB,H),
	find_tuple_in_trie([Head],Body,Vars,DB)
    ).

:- comment(fact_in_db/2, "@pred{fact_in_db(?Fact,+DB)} unifies, in
turn, @var{Fact} with each fact in DB, @var{DB}.  The order of
returned facts is indeterminate.").
:- mode fact_in_db(?,+).
fact_in_db(Head,H) :-
    tab_intern_termhash(DB,H),
    (DB == internalDB
     ->	call_c(Head)
     ;	find_tuple_in_trie([Head],true,_Vars,DB)
    ).

clause_in_db_b(Head,Body,Trie) :-
    find_tuple_in_trie([Head],Body0,Vars,Trie),
    (Body0 == true
     ->	Body = true
     ;	pdb_unnumbervars(Body0,0,Body,Vars)
    ).

find_tuple_in_trie([],Body,_Vars,Trie) :- 
    member(Body,Trie).
%% Use nonstandard '_$Var$' to not conflict with normal numbervar (hack)
find_tuple_in_trie(Terms,Body,Vars,Trie) :- Terms = [Term|RTerms],
    (find_in_radix_tree_b('_$Var$'(VNo),Val,Trie),
     integer(VNo),
     log_ith(VNo,Vars,Term),
     find_tuple_in_trie(RTerms,Body,Vars,Val)
     ;
     get_key_new_terms(Terms,Key,NTerms),
     (nonvar(Key), \+ varno(Key)
      -> find_in_radix_tree_b(Key,Val,Trie),
	 find_tuple_in_trie(NTerms,Body,Vars,Val)
      ;	 find_in_radix_tree_b(Key,Val,Trie),
	 Key =.. [_|NArgs],
	 append(NArgs,NTerms,NNTerms),
	 find_tuple_in_trie(NNTerms,Body,Vars,Val)
     )
    ).

variant_clause_in_db(Head,Body,H) :-
    tab_intern_termhash(DB,H),
    copy_term((Head,Body),(HeadC,BodyC)),
    my_numbervars((HeadC,BodyC),0,_),
    find_variant_tuple_in_trie([HeadC],BodyC,DB).

find_variant_tuple_in_trie([],Body,Trie) :- 
    member(Body,Trie).
%% Use nonstandard '_$Var$' to not conflict with normal numbervar (hack)
find_variant_tuple_in_trie(Terms,Body,Trie) :-
    get_key_new_terms(Terms,Key,NTerms),
    (nonvar(Key)
     ->	find_in_radix_tree_b(Key,Val,Trie),
	find_variant_tuple_in_trie(NTerms,Body,Val)
     ;	find_in_radix_tree_b(Key,Val,Trie),
	Key =.. [_|NArgs],
	append(NArgs,NTerms,NNTerms),
	find_variant_tuple_in_trie(NNTerms,Body,Val)
    ).

varno(Var) :- Var = '_$Var$'(VNo), integer(VNo).

varno(Var,VNo) :- Var = '_$Var$'(VNo), integer(VNo).


find_in_radix_tree_b(Key,Val,Tree) :-
    (Tree @= rtl(_,_,_)
     ->	rtl_member(Key,Val,Tree)
     ;	nonvar(Key)
     ->	hash_key(Key,HashCode),
	find_in_radix_tree(Tree,HashCode,Key,Val) % key bound, det (make builtin?)
     ;	find_in_radix_tree(Tree,HashCode,Key,Val)
    ).

find_in_radix_tree(RTL,_,Key,Val) :- RTL = rtl(_,_,_),
    rtl_member(Key,Val,RTL).
find_in_radix_tree(RT,HashCode,Key,Val) :- RT = rt(_,_,_,_),
    (nonvar(Key)
     ->	Index is HashCode /\ 3,
	NHashCode is HashCode >> 2,
	%% much better code generated w/this dupl, should fix compiler
	find_in_right_radix_tree(RT,NHashCode,Key,Val,Index) 
     ;	find_in_right_radix_tree(RT,NHashCode,Key,Val,Index)
    ).

%% lookahead hackery to avoid laying down choicepoint in most cases
%% rtl_member('_$Var$'(_h691),_h694,rtl(functor(usermod,'_$Var$',1),rtl(A,[true],[]),[])) fails,
%%    should succeed binding _h691 to value gotten from V1...
%%    needed for handling:
%%  assert_in_db(p('_$Var$'(X)),DB),trace,clause_in_db(p('_$Var$'(1)),true,DB).
/*
rtl_member('_$Var$'(X),V,rtl(functor(usermod,'_$Var$',1),SubTrie,[]|RTL)) :-
    find_in_radix_tree(X,V,SubTrie).  doesn't work..... need higher access...
*/
rtl_member(K,V,rtl(K1,V1,RTL)) :-
    (RTL == []
     ->	rtl_member1(K,V,K1,V1)
     ;	rtl_member2(K,V,K1,V1,RTL)
    ).

rtl_member1(K,V,K1,V1) :-
    (var(K), K1 = functor(Mod,Fun,Ari)
     ->	V = V1, my_functor(K,Mod,Fun,Ari)
     ;	K = K1, V = V1
    ).

rtl_member2(K,V,K1,V1,RTL) :-
    (rtl_member1(K,V,K1,V1)
     ;	
     rtl_member(K,V,RTL)
    ).

:- index find_in_right_radix_tree/5-5.
find_in_right_radix_tree(rt(RT1,_,_,_),HashCode,Key,Val,0) :-
    find_in_radix_tree(RT1,HashCode,Key,Val).
find_in_right_radix_tree(rt(_,RT2,_,_),HashCode,Key,Val,1) :-
    find_in_radix_tree(RT2,HashCode,Key,Val).
find_in_right_radix_tree(rt(_,_,RT3,_),HashCode,Key,Val,2) :-
    find_in_radix_tree(RT3,HashCode,Key,Val).
find_in_right_radix_tree(rt(_,_,_,RT4),HashCode,Key,Val,3) :-
    find_in_radix_tree(RT4,HashCode,Key,Val).

/***********************************************************/

:- comment(retractall_in_db/3,"@pred{retractall_in_db(+Goal,+DB0,?DB)}
retracts all clauses whose heads unify with @var{Goal} (or a term in
@var{Goal} if @var{Goal} is a list) from @var{DB0} generating
@var{DB1}.").
:- mode retractall_in_db(?,+,?).
retractall_in_db(Clauses,H0,H) :-
    (H0 == module
     ->	(Clauses = [Clause|MClauses]
	 -> retractall(Clause),
	    retractall_list(MClauses)
	 ;  retractall(Clauses)
	)
     ;	tab_intern_termhash(DB0,H0),
	retractall_in_db_b(Clauses,DB0,DB),
	tab_intern_termhash(DB,H)
    ).

%% simpler alias if desired
remove(Clauses,H0,H) :- retractall_in_db(Clauses,H0,H).

retractall_list([]).
retractall_list([Clause|Clauses]) :-
    retractall(Clause),
    retractall_list(Clauses).

retractall_in_db_b(Clauses0,DB0,DB) :-
    (Clauses0 == []
     ->	DB = DB0
     ; Clauses0 = [Clause|Clauses]
     ->	retractall1_in_db(Clause,DB0,DB1),
	retractall_in_db_b(Clauses,DB1,DB)
     ;	retractall1_in_db(Clauses0,DB0,DB)
    ).

retractall1_in_db(Clause,DB0,DB) :-
    (Clause = (Head:-Body0)
     ->	(ground(Head)
	 -> (ground(Body0)
	     ->	Body = Body0
	     ;	copy_term(Body0,Body),
		my_numbervars(Body,0,_)
	    )
	 ;  misc_error(('Body of rule in retractall_in_db/3 must be empty if Head is nonground:',Clause))
	)
     ;	Head = Clause
    ),
    delete_from_trie_b(Head,Body,DB0,_Vars,DB).

delete_from_trie_b(Head,Body,DB0,Vars,DB) :-
    delete_tuple_from_trie([Head],Body,Vars,DB0,DB).

delete_tuple_from_trie([],Body,Vars,Trie0,Trie) :-
    (var(Body)
     ->	Trie = []		% delete all
     ;	order_delete(Trie0,Body,Vars,Trie)
    ).
delete_tuple_from_trie(Terms,Body,Vars,Trie0,Trie) :- Terms = [Frst|NTermsT],
    (is_most_general_term(Terms)
     ->	Trie = []
     ;	findall(VNo,find_in_radix_tree_b('_$Var$'(VNo),_,Trie0),VNos),
	delete_list_vns_from_trie_list(VNos,Frst,NTermsT,Body,Vars,Trie0,Trie1),
        get_key_new_terms(Terms,VKey,NTerms),
	(nonvar(Key), varno(VKey,VNo)
	 -> log_ith(VNo,Vars,Key)
	 ;  Key = VKey
	),
	(var(Key)
	 -> findall(Terms,find_tuple_in_trie(Terms,Body,Vars,Trie0),TermsList),
	    TermsList \== [],
	    delete_list_from_trie_list(TermsList,Body,Vars,Trie0,Trie)
	 ;  ValTrans = lambda(OSTrie,STrie,delete_tuple_from_trie(NTerms,Body,Vars,OSTrie,STrie)),
	    delete_from_radix_tree_b(Key,ValTrans,Vars,Trie1,Trie)
	)
    ).

:- mode order_delete(+,?,?,?).
order_delete([CV|CVs],V,Vars,NVs) :-
    (varno(CV,VNo)
     ->	log_ith(VNo,Vars,VV),
	(VV \= V
	 -> NVs = [CV|NVsr],
	    order_delete(CVs,V,Vars,NVsr)
	 ;  order_delete(CVs,V,Vars,NVs)
	)
     ;  term_compare(CV,V,Cmp),
	(Cmp =:= 0 
	 -> NVs = CVs
	 ;  Cmp < 0
	 -> NVs = [CV|NVsr],
	    order_delete(CVs,V,Vars,NVsr)
	 ;  fail
	)
    ).

delete_list_from_trie_list([],_,_,Trie,Trie).
delete_list_from_trie_list([Terms|TermsList],Body,Vars,Trie0,Trie) :-
    copy_term(t(Terms,Body,Vars),t(NTerms,NBody,NVars)),
    NTerms = [Key|TermsTail],
    (var(Key)
     ->	ValTrans = lambda(OSTrie,STrie,delete_tuple_from_trie(TermsTail,NBody,NVars,OSTrie,STrie)),
	delete_from_radix_tree_b(Key,ValTrans,Vars,Trie0,Trie1)
     ;	delete_tuple_from_trie(NTerms,NBody,NVars,Trie0,Trie1)
     ),
    delete_list_from_trie_list(TermsList,Body,Vars,Trie1,Trie).

delete_list_vns_from_trie_list([],_Term,_NTerms,_Body,_Vars,Trie,Trie).
delete_list_vns_from_trie_list([VNo|VNos],Term,NTerms,Body,Vars,Trie0,Trie) :-
    (copy_term(t(Term,NTerms,Body,Vars),t(NTerm,NNTerms,NBody,NVars)),
     log_ith(VNo,NVars,NTerm),
     ValTrans = lambda(OSTrie,STrie,delete_tuple_from_trie(NNTerms,NBody,NVars,OSTrie,STrie)),
     delete_from_radix_tree_b('_$Var$'(VNo),ValTrans,NVars,Trie0,Trie1)
     ->	true			% must always succeed
     ;	Trie1 = Trie0
    ),
    delete_list_vns_from_trie_list(VNos,Term,NTerms,Body,Vars,Trie1,Trie).

delete_from_radix_tree_b(Key,ValTran,Vars,Tree0,Tree) :-
    (nonvar(Key)
     ->	hash_key(Key,HashCode)
     ;	true
    ),
    (delete_from_radix_tree(Tree0,Key,ValTran,Vars,Tree,HashCode)
     ->	true
     ;	Tree = Tree0
    ).

raise_subtree(rt([],[],[],[]),[]) :- !.
raise_subtree(rt(RT,[],[],[]),RT) :- RT = rtl(_,_,_), !.
raise_subtree(rt([],RT,[],[]),RT) :- RT = rtl(_,_,_), !.
raise_subtree(rt([],[],RT,[]),RT) :- RT = rtl(_,_,_), !.
raise_subtree(rt([],[],[],RT),RT) :- RT = rtl(_,_,_), !.
raise_subtree(RT,RT).

delete_from_radix_tree(RTL,Key,ValTrans,Vars,Tree,_HashCode) :- RTL = rtl(_,_,_),
    rtl_delete(RTL,Key,ValTrans,Vars,Tree).
delete_from_radix_tree(RT,Key,ValTrans,Vars,Tree,HashCode) :- RT = rt(_,_,_,_),
    (nonvar(Key)
     ->	Index is HashCode /\ 3,
	NHashCode is HashCode >> 2,
	delete_from_right_rt(RT,Key,ValTrans,Vars,Tree1,NHashCode,Index),
	raise_subtree(Tree1,Tree)
     ;	delete_from_right_rt(RT,Key,ValTrans,Vars,Tree1,NHashCode,Index),
	raise_subtree(Tree1,Tree)
    ).

rtl_delete(rtl(OK,OVs,RTLr),K,ValTrans,Vars,NRTL) :-
    (varno(OK,VNo)
     ->	((var(K), log_ith(VNo,Vars,K)
	  ;
	  varno(K,VNo1),
	  log_ith(VNo,Vars,X), log_ith(VNo1,Vars,X)
	 )
	 -> appl(ValTrans,OVs,NVs),
	    (NVs == []
	     ->	NRTL = RTLr
	     ;	NRTL = rtl(OK,NVs,RTLr)
	    )
	 ;  NRTL = rtl(OK,OVs,NRTLr),
	    rtl_delete(RTLr,K,ValTrans,Vars,NRTLr)
	)
     ;  term_compare(OK,K,Cmp),
	(Cmp < 0
	 -> NRTL = rtl(OK,OVs,NRTLr),
	    rtl_delete(RTLr,K,ValTrans,Vars,NRTLr)
	 ;  Cmp =:= 0
	 -> appl(ValTrans,OVs,NVs), % HiLog, for old Vals compute new Vals
	    (NVs == []	       % raising is nec here!
	     ->	NRTL = RTLr
	     ;	NRTL = rtl(OK,NVs,RTLr)
	    )
	    ;  fail
	)
    ).
     
:- index delete_from_right_rt/7-7.
delete_from_right_rt(rt(RT1,RT2,RT3,RT4),Key,VT,Vs,rt(ST,RT2,RT3,RT4),HashCode,0) :-
    delete_from_radix_tree(RT1,Key,VT,Vs,ST,HashCode).
delete_from_right_rt(rt(RT1,RT2,RT3,RT4),Key,VT,Vs,rt(RT1,ST,RT3,RT4),HashCode,1) :-
    delete_from_radix_tree(RT2,Key,VT,Vs,ST,HashCode).
delete_from_right_rt(rt(RT1,RT2,RT3,RT4),Key,VT,Vs,rt(RT1,RT2,ST,RT4),HashCode,2) :-
    delete_from_radix_tree(RT3,Key,VT,Vs,ST,HashCode).
delete_from_right_rt(rt(RT1,RT2,RT3,RT4),Key,VT,Vs,rt(RT1,RT2,RT3,ST),HashCode,3) :-
    delete_from_radix_tree(RT4,Key,VT,Vs,ST,HashCode).

/***********************************************************/
:- comment(union_db/3, " @pred{union_db(+DB1,+DB2,-DB)} takes two
prolog databases, in @var{DB1} and @var{DB2} and produces a prolog DB,
in @var{DB}, that is the union of the two input databases.  This is
done by traversing the two input tries in concert and in the process
generating the output trie.  This allows the output trie to share
subtries with the input tries, and to perhaps avoid traversing large
portions of the input tries.  For example, if the two input databases
contain facts for disjoint predicates, the union just creates a new
trie that points to entire definitions of the predicates in the input
tries, so such a union is done in time proportional to the number of
predicates, and completely independent of the size (i.e., the number
of tuples) of the predicates.").
:- mode union_db(+,+,?).
union_db(H1,H2,H) :-
    tab_intern_termhash(DB1,H1),
    tab_intern_termhash(DB2,H2),
    union_db_b(DB1,DB2,DB),
    tab_intern_termhash(DB,H).

union_db_b(DB1,DB2,DB3) :-
    union_tries(DB1,DB2,0,DB3).

:- comment(union_in_db/5, "@pred{union_in_db(+P1,+P2,+P3,+DB0,?DB)}
unions two subtries in @var{DB0} producing a new (or replaced) subtrie
in @var{DB}.  The input tries are determined by the prefix terms of
@var{P1} and @var{P2}, and the place of the resulting trie is
determined by prefix term @var{P3}.  (See @pred{is_prefix_term/2} and
@pred{apply_op_in_db/6} for discussions of prefix terms.)").
:- mode union_in_db(?,?,?,+,?).
union_in_db(P1,P2,P3,H0,H) :-
    apply_op_in_db(lambda(A,B,C,union_db_b(A,B,C)),P1,P2,P3,H0,H).

union_tries(T1,T2,Depth,T) :-
    (T1 \== T2
     ->	union_tries0(T1,T2,Depth,T)
     ;	T = T1
    ).

union_tries0([],T,_,T).
union_tries0(RTLA,RTLB,Depth,T) :- RTLA = rtl(K1,_,_),
    union_with_rtl(RTLA,RTLB,Depth,T,K1).
union_tries0(RTA,RTLB,Depth,T) :- RTA = rt(_,_,_,_),
    union_with_rt(RTA,RTLB,Depth,T).

:- index union_with_rtl/5-2.
union_with_rtl(T,[],_Depth,T,_K1).
union_with_rtl(RTLA,RTLB,Depth,T,K1) :- RTLB = rtl(K2,_,_),
    hash_key(K1,Hash1),
    hash_key(K2,Hash2),
    (Hash1 =:= Hash2
     ->	rtl_union(RTLA,RTLB,T)
     ;	Ind1 is (Hash1 >> Depth) /\ 3,
	Ind2 is (Hash2 >> Depth) /\ 3,
	(Ind1 =:= Ind2
	 -> Depth1 is Depth + 2,
	    union_tries_ind_recur_b(RTLA,RTLB,Depth1,T,Ind1)
	 ;  CInd is 4 * Ind1 + Ind2,
	    union_tries_build_pair(CInd,RTLA,RTLB,T)
	)
    ).
union_with_rtl(RTLA,RTB,Depth,T,Key) :- RTB = rt(_,_,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    union_tries_ind_recur_l(RTLA,RTB,Depth1,T,NInd).
    
:- index union_with_rt/4-2.
union_with_rt(T,[],_Depth,T).
union_with_rt(RTA,RTLB,Depth,T) :- RTLB = rtl(Key,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    union_tries_ind_recur_r(RTA,RTLB,Depth1,T,NInd).
union_with_rt(rt(RT01,RT02,RT03,RT04),rt(RT11,RT12,RT13,RT14),Depth,rt(RT1,RT2,RT3,RT4)) :-
    Depth1 is Depth + 2,
    union_tries(RT01,RT11,Depth1,RT1),
    union_tries(RT02,RT12,Depth1,RT2),
    union_tries(RT03,RT13,Depth1,RT3),
    union_tries(RT04,RT14,Depth1,RT4).

%% union rtl lists and val-lists of equal keys.
rtl_union([],RTL,RTL).
rtl_union(RTL1,RTL2,RTL3) :- RTL1 = rtl(K1,Vs1,RTL1r),
    (RTL2 == []
     ->	RTL3 = RTL1
     ;	RTL2 = rtl(K2,Vs2,RTL2r),
	term_compare(K1,K2,Cmp),
	(Cmp < 0
	 -> RTL3 = rtl(K1,Vs1,RTL3r),
	    rtl_union(RTL1r,RTL2,RTL3r)
	 ;  Cmp =:= 0
	 -> RTL3 = rtl(K1,Vs3,RTL3r),
 	    order_union(Vs1,Vs2,Vs3),
	    rtl_union(RTL1r,RTL2r,RTL3r)
	 ;  RTL3 = rtl(K2,Vs2,RTL3r),
	    rtl_union(RTL1,RTL2r,RTL3r)
	)
    ).

order_union([],L,L).
order_union(L1,L2,L3) :- L1 = [_|_],
    order_union_list(L1,L2,L3).
order_union(Trie1,Trie2,Trie3) :- Trie1 = rtl(_,_,_),
    union_db_b(Trie1,Trie2,Trie3).
order_union(Trie1,Trie2,Trie3) :- Trie1 = rt(_,_,_,_),
    union_db_b(Trie1,Trie2,Trie3).

order_union_list([],L,L).
order_union_list(L1,L2,L3) :- L1 = [V1|L1r],
    (L2 == []
     ->	L3 = L1
     ;	L2 = [V2|L2r],
	L3 = [V|L3r],
	(V1 @< V2
	 -> V = V1,
	    order_union_list(L1r,L2,L3r)
	 ;  V1 == V2
	 -> V = V1,
	    order_union_list(L1r,L2r,L3r)
	 ;  V = V2,
	    order_union_list(L1,L2r,L3r)
	)
    ).

:- index union_tries_ind_recur_l/5-5.
union_tries_ind_recur_l(RTL,rt(RT1,RT2,RT3,RT4),D,rt(ST,RT2,RT3,RT4),0) :-
    union_tries(RTL,RT1,D,ST).
union_tries_ind_recur_l(RTL,rt(RT1,RT2,RT3,RT4),D,rt(RT1,ST,RT3,RT4),1) :-
    union_tries(RTL,RT2,D,ST).
union_tries_ind_recur_l(RTL,rt(RT1,RT2,RT3,RT4),D,rt(RT1,RT2,ST,RT4),2) :-
    union_tries(RTL,RT3,D,ST).
union_tries_ind_recur_l(RTL,rt(RT1,RT2,RT3,RT4),D,rt(RT1,RT2,RT3,ST),3) :-
    union_tries(RTL,RT4,D,ST).

:- index union_tries_ind_recur_r/5-5.
union_tries_ind_recur_r(rt(RT1,RT2,RT3,RT4),RTL,D,rt(ST,RT2,RT3,RT4),0) :-
    union_tries(RT1,RTL,D,ST).
union_tries_ind_recur_r(rt(RT1,RT2,RT3,RT4),RTL,D,rt(RT1,ST,RT3,RT4),1) :-
    union_tries(RT2,RTL,D,ST).
union_tries_ind_recur_r(rt(RT1,RT2,RT3,RT4),RTL,D,rt(RT1,RT2,ST,RT4),2) :-
    union_tries(RT3,RTL,D,ST).
union_tries_ind_recur_r(rt(RT1,RT2,RT3,RT4),RTL,D,rt(RT1,RT2,RT3,ST),3) :-
    union_tries(RT4,RTL,D,ST).

union_tries_build_pair(1,RT1,RT2,rt(RT1,RT2,[],[])).
union_tries_build_pair(2,RT1,RT2,rt(RT1,[],RT2,[])).
union_tries_build_pair(3,RT1,RT2,rt(RT1,[],[],RT2)).
union_tries_build_pair(4,RT1,RT2,rt(RT2,RT1,[],[])).
union_tries_build_pair(6,RT1,RT2,rt([],RT1,RT2,[])).
union_tries_build_pair(7,RT1,RT2,rt([],RT1,[],RT2)).
union_tries_build_pair(8,RT1,RT2,rt(RT2,[],RT1,[])).
union_tries_build_pair(9,RT1,RT2,rt([],RT2,RT1,[])).
union_tries_build_pair(11,RT1,RT2,rt([],[],RT1,RT2)).
union_tries_build_pair(12,RT1,RT2,rt(RT2,[],[],RT1)).
union_tries_build_pair(13,RT1,RT2,rt([],RT2,[],RT1)).
union_tries_build_pair(14,RT1,RT2,rt([],[],RT2,RT1)).

:- index union_tries_ind_recur_b/5-5.
union_tries_ind_recur_b(RT1,RT2,D,rt(RT,[],[],[]),0) :-
    union_tries(RT1,RT2,D,RT).
union_tries_ind_recur_b(RT1,RT2,D,rt([],RT,[],[]),1) :-
    union_tries(RT1,RT2,D,RT).
union_tries_ind_recur_b(RT1,RT2,D,rt([],[],RT,[]),2) :-
    union_tries(RT1,RT2,D,RT).
union_tries_ind_recur_b(RT1,RT2,D,rt([],[],[],RT),3) :-
    union_tries(RT1,RT2,D,RT).

/**********************************************************/

:- comment(copy_in_db/4, "@pred{copy_in_db(+PrefI,+PrefO,+DB0,?DB)}
makes a copy of @var{PrefI} into @var{Pref0} generating @var{DB} from
@var{DB0}.  @var{PrefI} and @var{PrefO} must be prefix terms and have
the same number of variables.  (See the discussin of Prefix Terms
above.)  This is a very fast operation since it simply copies the
pointer to the appropriate subtrie.").
:- mode copy_in_db(?,?,+,?).
/** now requires variables to match */
copy_in_db(PrefI,PrefO,DB0,DB) :-
    cvt_pa(PrefI,PrefO,PrefIs,PrefOs),
    reorder_in_db(PrefIs,PrefOs,DB0,DB).

/**********************************************************/

:- comment(move_in_db/4, " @pred{move_in_db(+Pref1,+Pref2,+DB0,-DB)}
generates @var{DB} that is the same as @var{DB0} except that the
tuples of @var{Pref1} are copied to @var{Pref2} and the tuples of
@var{Pref1} are deleted.  ").
:- mode move_in_db(?,?,+,?).
move_in_db(Pref1,Pref2,DB0,DB) :-
    copy_in_db(Pref1,Pref2,DB0,DB1),
    retractall_in_db(Pref1,DB1,DB).

/**********************************************************/

:- comment(is_empty_in_db/2, " @pred{is_empty_in_db(+Pref,+DB)}
succeeds if the relation determined by prefix term @var{Pref} in
database @var{DB} is empty.  It fails otherwise.").
:- mode is_empty_in_db(?,+).
is_empty_in_db(Pref,H) :-
    tab_intern_termhash(DB,H),
    cvt_pa(Pref,Prefa),
    \+ find_subtrie_for_prefix([Prefa],DB,_TL,_HT).

:- comment(is_empty_in_db/3, " @pred{is_empty_in_db(+Pref,+DB0,-DB)}
is a version of @pred{is_empty_in_db/2} that returns its input
database so that it can be used as an identity transformation in a
DCG rule.").
:- mode is_empty_in_db(?,+,?).
is_empty_in_db(Pref,H,H) :-
    is_empty_in_db(Pref,H).

/**********************************************************/

:- comment(equal_in_db/3, "@pred{equal_in_db(P1,P2,DB)} succeeds if
the set of tuples specified by prefix term @var{P1} is the same as the
set of tuples specified by prefix term @var{P2} in @var{DB}.").
:- mode equal_in_db(?,?,+).
equal_in_db(P1,P2,DB) :-
    apply_rel_op_in_db(lambda(A,B,(A==B)),P1,P2,DB).


/***********************************************************/

:- comment(count_in_db/3, " @pred{count_in_db(+Pref,+DB,-Count)}
returns in @var{Count} the number of tuples specified by prefix term
@var{Pref} in database @var{DB}.  ").
%% should it use clause_in_db instead of in_db?
%% this is simple way...
%% might be faster if done by traversing the DB trie.
:- mode count_in_db(?,+,?).
count_in_db(Pref,DB,Count) :-
    cvt_pa(Pref,Prefa),
    gensym(ctr,Ctr),
    conset(Ctr,0),
    (do_all
     clause_in_db(Prefa,_,DB),
     coninc(Ctr)
    ),
    conget(Ctr,Count).

/***********************************************************/

:- comment(size_db/2," @pred{size_db(+DB,?Count)} returns the number
of clauses in @var{DB} in @var{Count}.  ").
:- mode size_db(+,?).
size_db(H,Cnt) :-
    tab_intern_termhash(DB,H),
    size_db_b(DB,Cnt).

size_db_b(DB,Cnt) :-
    size_db_b(DB,0,Cnt).

size_db_b([],CC,CC).
size_db_b(rt(RT1,RT2,RT3,RT4),CC0,CC) :-
    size_db_b(RT1,CC0,CC1),
    size_db_b(RT2,CC1,CC2),
    size_db_b(RT3,CC2,CC3),
    size_db_b(RT4,CC3,CC).
size_db_b(rtl(_K,Vs,RTL),CC0,CC) :-
    size_db_b(Vs,CC0,CC1),
    size_db_b(RTL,CC1,CC).
size_db_b([_T|Ts],CC0,CC) :-
    CC1 is CC0+1,
    size_db_b(Ts,CC1,CC).

/***********************************************************/


/***********************************************************/

:- comment(intersect_db/3, " @pred{intersect_db(+DB1,+DB2,?DB)} takes
two prolog databases (tries), in @var{DB1} and @var{DB2}, and produces
a prolog DB, in @var{DB}, that is the intersection of the two input
databases.  This is done by traversing the two input tries in concert
and in the process generating the output trie.  This allows the output
trie to share subterms with the input tries, and to perhaps avoid
traversing large portions of the input tries.  ").
:- mode intersect_db(+,+,?).
intersect_db(H1,H2,H) :-
    tab_intern_termhash(DB1,H1),
    tab_intern_termhash(DB2,H2),
    intersect_db_b(DB1,DB2,DB),
    tab_intern_termhash(DB,H).

:- comment(intersect_in_db/5,
"@pred{intersect_in_db(+P1,+P2,+P3,+DB0,?DB)} intersects two subtries
in @var{DB0} producing a new (or replaced) subtrie in @var{DB}.  The
input tries are determined by the prefix terms of @var{P1} and
@var{P2}, and the place of the resulting trie is determined by prefix
term @var{P3}.  (See @pred{is_prefix_term/2} and
@pred{apply_op_in_db/6} for discussions of prefix terms.)").
:- mode intersect_in_db(?,?,?,+,?).
intersect_in_db(P1,P2,P3,DB0,DB) :-
    apply_op_in_db(lambda(A,B,C,intersect_db_b(A,B,C)),P1,P2,P3,DB0,DB).

intersect_db_b(T0,T1,T) :-
    intersect_tries(T0,T1,0,T).

intersect_tries(T1,T2,Depth,T) :-
    (T1 \== T2
     ->	intersect_tries0(T1,T2,Depth,T)
     ;	T = T1
    ).

intersect_tries0([],_T,_,[]).
intersect_tries0(RTLA,RTLB,Depth,T) :- RTLA = rtl(K1,_,_),
    intersect_with_rtl(RTLA,RTLB,Depth,T,K1).
intersect_tries0(RTA,RTLB,Depth,T) :- RTA = rt(_,_,_,_),
    intersect_with_rt(RTA,RTLB,Depth,T).

:- index intersect_with_rtl/5-2.
intersect_with_rtl(_T,[],_Depth,[],_K1).
intersect_with_rtl(RTLA,RTLB,_Depth,T,K1) :- RTLB = rtl(K2,_,_),
    hash_key(K1,Hash1),
    hash_key(K2,Hash2),
    (Hash1 =:= Hash2
     ->	rtl_intersect(RTLA,RTLB,T)
     ;	T = []
    ).
intersect_with_rtl(RTLA,RTB,Depth,T,Key) :- RTB = rt(_,_,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    intersect_tries_ind_recur_l(RTLA,RTB,Depth1,T0,NInd),
    raise_subtree(T0,T).
    
:- index intersect_with_rt/4-2.
intersect_with_rt(_T,[],_Depth,[]).
intersect_with_rt(RTA,RTLB,Depth,T) :- RTLB = rtl(Key,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    intersect_tries_ind_recur_r(RTA,RTLB,Depth1,T0,NInd),
    raise_subtree(T0,T).
intersect_with_rt(rt(RT01,RT02,RT03,RT04),rt(RT11,RT12,RT13,RT14),Depth,T) :-
    Depth1 is Depth + 2,
    intersect_tries(RT01,RT11,Depth1,RT1),
    intersect_tries(RT02,RT12,Depth1,RT2),
    intersect_tries(RT03,RT13,Depth1,RT3),
    intersect_tries(RT04,RT14,Depth1,RT4),
    raise_subtree(rt(RT1,RT2,RT3,RT4),T).


order_intersect([],L,L).
order_intersect(L1,L2,L3) :- L1 = [_|_],
    order_intersect_list(L1,L2,L3).
order_intersect(Trie1,Trie2,Trie3) :- Trie1 = rtl(_,_,_),
    intersect_db_b(Trie1,Trie2,Trie3).
order_intersect(Trie1,Trie2,Trie3) :- Trie1 = rt(_,_,_,_),
    intersect_db_b(Trie1,Trie2,Trie3).


order_intersect_list([],_L,[]).
order_intersect_list(L1,L2,L3) :- L1 = [V1|L1r],
    (L2 == []
     ->	L3 = []
     ;	L2 = [V2|L2r],
	term_compare(V1,V2,Ord),
	(Ord < 0
	 -> order_intersect_list(L1r,L2,L3)
	 ;  Ord =:= 0
	 -> L3 = [V1|L3r],
	    order_intersect_list(L1r,L2r,L3r)
	 ;  order_intersect_list(L1,L2r,L3)
	)
    ).

rtl_intersect([],_,[]).
rtl_intersect(RTL1,RTL2,RTL3) :- RTL1 = rtl(K1,Vs1,RTL1r),
    (RTL2 == []
     ->	RTL3 = []
     ;	RTL2 = rtl(K2,Vs2,RTL2r),
	term_compare(K1,K2,Cmp),
	(Cmp < 0
	 -> rtl_intersect(RTL1r,RTL2,RTL3)
	 ;  Cmp =:= 0
	 -> order_intersect(Vs1,Vs2,Vs3),
	    (Vs3 == []
	     ->	rtl_intersect(RTL1r,RTL2r,RTL3)
	     ;	RTL3 = rtl(K1,Vs3,RTL3r),
		rtl_intersect(RTL1r,RTL2r,RTL3r)
	    )
	 ;  rtl_intersect(RTL1,RTL2r,RTL3)
	)
    ).

:- index intersect_tries_ind_recur_l/5-5.
intersect_tries_ind_recur_l(RTL,rt(RT1,_,_,_),D,rt(ST,[],[],[]),0) :-
    intersect_tries(RTL,RT1,D,ST).
intersect_tries_ind_recur_l(RTL,rt(_,RT2,_,_),D,rt([],ST,[],[]),1) :-
    intersect_tries(RTL,RT2,D,ST).
intersect_tries_ind_recur_l(RTL,rt(_,_,RT3,_),D,rt([],[],ST,[]),2) :-
    intersect_tries(RTL,RT3,D,ST).
intersect_tries_ind_recur_l(RTL,rt(_,_,_,RT4),D,rt([],[],[],ST),3) :-
    intersect_tries(RTL,RT4,D,ST).

:- index intersect_tries_ind_recur_r/5-5.
intersect_tries_ind_recur_r(rt(RT1,_,_,_),RTL,D,rt(ST,[],[],[]),0) :-
    intersect_tries(RT1,RTL,D,ST).
intersect_tries_ind_recur_r(rt(_,RT2,_,_),RTL,D,rt([],ST,[],[]),1) :-
    intersect_tries(RT2,RTL,D,ST).
intersect_tries_ind_recur_r(rt(_,_,RT3,_),RTL,D,rt([],[],ST,[]),2) :-
    intersect_tries(RT3,RTL,D,ST).
intersect_tries_ind_recur_r(rt(_,_,_,RT4),RTL,D,rt([],[],[],ST),3) :-
    intersect_tries(RT4,RTL,D,ST).

/***********************************************************/

:- comment(difference_db/3, " @var{difference_db(+DB1,+DB2,?DB)} takes
two @tt{prolog_db} trie-term databases, in @var{DB1} and @var{DB2} and
produces a prolog DB, in @var{DB}, that is the set difference of the
two input databases, i.e., all clauses in @var{DB1} that are not in
@var{DB2}.  This is done by traversing the two input tries (and their
index trees) in concert and in the process generating the output trie.
This allows the output trie (and index trees) to share subterms (and
sub index trees) with the input tries, and to perhaps avoid traversing
large portions of the input tries.  For example, if the two input
databases contain the same sets of tuples, the difference immediately
detects that and returns the empty database.").
:- mode difference_db(+,+,?).
difference_db(H1,H2,H) :-
    tab_intern_termhash(DB1,H1),
    tab_intern_termhash(DB2,H2),
    difference_db_b(DB1,DB2,DB),
    tab_intern_termhash(DB,H).

:- comment(difference_in_db/5,
"@pred{difference_in_db(+P1,+P2,+P3,+DB0,?DB)} computes the set
difference of two subtries in @var{DB0} producing a new (or replaced)
subtrie in @var{DB}.  The input tries are determined by the prefix
terms of @var{P1} and @var{P2}, and the place of the resulting trie is
determined by prefix term @var{P3}.  (See @pred{is_prefix_term/2} and
@pred{apply_op_in_db/6} for discussions of prefix terms.)").
:- mode difference_in_db(?,?,?,+,?).
difference_in_db(P1,P2,P3,DB0,DB) :-
    apply_op_in_db(lambda(A,B,C,difference_db_b(A,B,C)),P1,P2,P3,DB0,DB).

difference_db_b(T0,T1,T) :-
    difference_tries(T0,T1,0,T).

difference_tries(T1,T2,Depth,T) :-
    (T1 \== T2
     ->	difference_tries0(T1,T2,Depth,T)
     ;	T = []
    ).

difference_tries0([],_T,_,[]).
difference_tries0(RTLA,RTLB,Depth,T) :- RTLA = rtl(K1,_,_),
    difference_with_rtl(RTLA,RTLB,Depth,T,K1).
difference_tries0(RTA,RTLB,Depth,T) :- RTA = rt(_,_,_,_),
    difference_with_rt(RTA,RTLB,Depth,T).

:- index difference_with_rtl/5-2.
difference_with_rtl(T,[],_Depth,T,_K1).
difference_with_rtl(RTLA,RTLB,_Depth,T,K1) :- RTLB = rtl(K2,_,_),
    hash_key(K1,Hash1),
    hash_key(K2,Hash2),
    (Hash1 =:= Hash2
     ->	rtl_difference(RTLA,RTLB,T)
     ;	T = RTLA
    ).
difference_with_rtl(RTLA,RTB,Depth,T,Key) :- RTB = rt(_,_,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    difference_tries_ind_recur_l(RTLA,RTB,Depth1,T0,NInd),
    raise_subtree(T0,T).
    
:- index difference_with_rt/4-2.
difference_with_rt(T,[],_Depth,T).
difference_with_rt(RTA,RTLB,Depth,T) :- RTLB = rtl(Key,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    difference_tries_ind_recur_r(RTA,RTLB,Depth1,T0,NInd),
    raise_subtree(T0,T).
difference_with_rt(rt(RT01,RT02,RT03,RT04),rt(RT11,RT12,RT13,RT14),Depth,T) :-
    Depth1 is Depth + 2,
    difference_tries(RT01,RT11,Depth1,RT1),
    difference_tries(RT02,RT12,Depth1,RT2),
    difference_tries(RT03,RT13,Depth1,RT3),
    difference_tries(RT04,RT14,Depth1,RT4),
    raise_subtree(rt(RT1,RT2,RT3,RT4),T).

rtl_difference([],_RTL1,[]).
rtl_difference(RTL1,RTL2,RTL3) :- RTL1 = rtl(K1,Vs1,RTL1r),
    (RTL2 == []
     ->	RTL3 = RTL1
     ;	RTL2 = rtl(K2,Vs2,RTL2r),
	term_compare(K1,K2,Cmp),
	(Cmp < 0
	 -> RTL3 = rtl(K1,Vs1,RTL3r),
	    rtl_difference(RTL1r,RTL2,RTL3r)
	 ;  Cmp =:= 0
	 -> order_difference(Vs1,Vs2,Vs3),
	    (Vs3 == []
	     ->	rtl_difference(RTL1r,RTL2r,RTL3)
	     ;	RTL3 = rtl(K1,Vs3,RTL3r),
		rtl_difference(RTL1r,RTL2r,RTL3r)
	    )
	    ;  rtl_difference(RTL1,RTL2r,RTL3)
	)
    ).

order_difference([],_L,[]).
order_difference(L1,L2,L3) :- L1 = [_|_],
    order_difference_list(L1,L2,L3).
order_difference(Trie1,Trie2,Trie3) :- Trie1 = rtl(_,_,_),
    difference_db_b(Trie1,Trie2,Trie3).
order_difference(Trie1,Trie2,Trie3) :- Trie1 = rt(_,_,_,_),
    difference_db_b(Trie1,Trie2,Trie3).

order_difference_list([],_,[]).
order_difference_list(L1,L2,L3) :- L1 = [V1|L1r],
    (L2 == []
     ->	L3 = L1
     ;	L2 = [V2|L2r],
	term_compare(V1,V2,Ord),
	(Ord < 0
	 -> L3 = [V1|L3r],
	    order_difference_list(L1r,L2,L3r)
	 ;  Ord =:= 0
	 -> order_difference_list(L1r,L2r,L3)
	 ;  order_difference_list(L1,L2r,L3)
	)
    ).

:- index difference_tries_ind_recur_l/5-5.
difference_tries_ind_recur_l(RT,rt(RT1,_,_,_),D,rt(ST,[],[],[]),0) :-
    difference_tries(RT,RT1,D,ST).
difference_tries_ind_recur_l(RT,rt(_,RT2,_,_),D,rt([],ST,[],[]),1) :-
    difference_tries(RT,RT2,D,ST).
difference_tries_ind_recur_l(RT,rt(_,_,RT3,_),D,rt([],[],ST,[]),2) :-
    difference_tries(RT,RT3,D,ST).
difference_tries_ind_recur_l(RT,rt(_,_,_,RT4),D,rt([],[],[],ST),3) :-
    difference_tries(RT,RT4,D,ST).

:- index difference_tries_ind_recur_r/5-5.
difference_tries_ind_recur_r(rt(RT1,RT2,RT3,RT4),RT,D,rt(ST,RT2,RT3,RT4),0) :-
    difference_tries(RT1,RT,D,ST).
difference_tries_ind_recur_r(rt(RT1,RT2,RT3,RT4),RT,D,rt(RT1,ST,RT3,RT4),1) :-
    difference_tries(RT2,RT,D,ST).
difference_tries_ind_recur_r(rt(RT1,RT2,RT3,RT4),RT,D,rt(RT1,RT2,ST,RT4),2) :-
    difference_tries(RT3,RT,D,ST).
difference_tries_ind_recur_r(rt(RT1,RT2,RT3,RT4),RT,D,rt(RT1,RT2,RT3,ST),3) :-
    difference_tries(RT4,RT,D,ST).


/***********************************************************/

:- comment(subset_db/2, "@pred{subset_db(+DB1,+DB2)} succeeds if the
terms in prolog DB, @var{DB1}, are a subset of the terms in prolog DB,
@pred{DB2}, and fails otherwise.").
:- mode subset_db(+,+).
subset_db(H1,H2) :-
    tab_intern_termhash(DB1,H1),
    tab_intern_termhash(DB2,H2),
    subset_db_b(DB1,DB2).

:- comment(subset_in_db/3, "@pred{subset_in_db(+Pref1,+Pref2,+DB)}
succeeds if the tuples specified by prefix term @var{Pref1} are a
subset of those specified by @var{Pref2} in @var{DB}.  The number of
variables in @var{Pref1} and @var{Pref2} must be the same.").
:- mode subset_in_db(?,?,+).
subset_in_db(P1,P2,DB) :-
    apply_rel_op_in_db(lambda(A,B,subset_db_b(A,B)),P1,P2,DB).

subset_db_b(T0,T1) :-
    subset_tries(T0,T1,0).

subset_tries(T1,T2,Depth) :-
    (T1 \== T2
     ->	subset_tries0(T1,T2,Depth)
     ;	true
    ).

subset_tries0([],_T,_D).
subset_tries0(RTLA,RTLB,Depth) :- RTLA = rtl(K1,_,_),
    subset_with_rtl(RTLA,RTLB,Depth,K1).
subset_tries0(RTA,RTLB,Depth) :- RTA = rt(_,_,_,_),
    subset_with_rt(RTA,RTLB,Depth).

:- index subset_with_rtl/4-2.
subset_with_rtl(RTLA,RTLB,_Depth,K1) :- RTLB = rtl(K2,_,_),
    hash_key(K1,Hash1),
    hash_key(K2,Hash2),
    (Hash1 =:= Hash2
     ->	rtl_subset(RTLA,RTLB)
     ;	fail
    ).
subset_with_rtl(RTLA,RTB,Depth,Key) :- RTB = rt(_,_,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    subset_tries_ind_recur_l(RTLA,RTB,Depth1,NInd).
    
:- index subset_with_rt/3-2.
subset_with_rt(RTA,RTLB,Depth) :- RTLB = rtl(Key,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    subset_tries_ind_recur_r(RTA,RTLB,Depth1,NInd).
subset_with_rt(rt(RT01,RT02,RT03,RT04),rt(RT11,RT12,RT13,RT14),Depth) :-
    Depth1 is Depth + 2,
    subset_tries(RT01,RT11,Depth1),
    subset_tries(RT02,RT12,Depth1),
    subset_tries(RT03,RT13,Depth1),
    subset_tries(RT04,RT14,Depth1).

order_subset([],_L).
order_subset(L1,L2) :- L1 = [_|_],
    order_subset_list(L1,L2).
order_subset(Trie1,Trie2) :- Trie1 = rtl(_,_,_),
    subset_db_b(Trie1,Trie2).
order_subset(Trie1,Trie2) :- Trie1 = rt(_,_,_,_),
    subset_db_b(Trie1,Trie2).

order_subset_list([],_).
order_subset_list(L1,L2) :- L1 = [V1|L1r],
    (L1 == []
     ->	fail
     ;	L2 = [V2|L2r],
	term_compare(V1,V2,Ord),
	(Ord < 0
	 -> fail
	 ;  Ord =:= 0
	 -> order_subset_list(L1r,L2r)
	 ;  order_subset_list(L1,L2r)
	)
    ).

rtl_subset([],_).
rtl_subset(RTL1,RTL2) :- RTL1 = rtl(K1,V1,RTL1r),
    RTL1 \== [],
    RTL2 = rtl(K2,V2,RTL2r),
    term_compare(K1,K2,Cmp),
    Cmp >= 0,
    (Cmp =:= 0
     -> order_subset(V1,V2),
        rtl_subset(RTL1r,RTL2r)
     ;  rtl_subset(RTL1,RTL2r)
    ).
/***rtl_subset(RTL1,RTL2) :- RTL1 = rtl(K1,V1,RTL1r),
    (RTL1 == []
     ->	fail
     ;	RTL2 = rtl(K2,V2,RTL2r),
	term_compare(K1,K2,Cmp),
	(Cmp < 0
	 -> fail
	 ;  Cmp =:= 0
	 -> order_subset(V1,V2),
	    rtl_subset(RTL1r,RTL2r)
	 ;  rtl_subset(RTL1,RTL2r)
	)
    ).***/

:- index subset_tries_ind_recur_l/4-4.
subset_tries_ind_recur_l(RTL,rt(RT1,_,_,_),D,0) :-
    subset_tries(RTL,RT1,D).
subset_tries_ind_recur_l(RTL,rt(_,RT2,_,_),D,1) :-
    subset_tries(RTL,RT2,D).
subset_tries_ind_recur_l(RTL,rt(_,_,RT3,_),D,2) :-
    subset_tries(RTL,RT3,D).
subset_tries_ind_recur_l(RTL,rt(_,_,_,RT4),D,3) :-
    subset_tries(RTL,RT4,D).

:- index subset_tries_ind_recur_r/4-4.
subset_tries_ind_recur_r(rt(RT1,_,_,_),RTL,D,0) :-
    subset_tries(RT1,RTL,D).
subset_tries_ind_recur_r(rt(_,RT2,_,_),RTL,D,1) :-
    subset_tries(RT2,RTL,D).
subset_tries_ind_recur_r(rt(_,_,RT3,_),RTL,D,2) :-
    subset_tries(RT3,RTL,D).
subset_tries_ind_recur_r(rt(_,_,_,RT4),RTL,D,3) :-
    subset_tries(RT4,RTL,D).

/********************************************************/

:- comment(sym_diff_db/3, " @var{sym_diff_db(+DB1,+DB2,?DB)} takes two
databases (trie-terms), in @var{DB1} and @var{DB2} and produces a DB,
in @var{DB}, that is the symmetric difference of the two input
databases, i.e., all tuples in one but not both of @var{DB1} and
@var{DB2}.  This is done by traversing the two input tries in concert
and in the process generating the output trie.  This allows the output
trie to share subterms with the input tries, and to perhaps avoid
traversing large portions of the input tries.").
:- mode sym_diff_db(+,+,-).
sym_diff_db(H1,H2,H) :-
    tab_intern_termhash(DB1,H1),
    tab_intern_termhash(DB2,H2),
    sym_diff_db_b(DB1,DB2,DB),
    tab_intern_termhash(DB,H).

:- comment(sym_diff_in_db/5,
"@pred{sym_diff_in_db(+P1,+P2,+P3,+DB0,?DB)} computes the symmetric
difference of two subtries in @var{DB0} producing a new (or replaced)
subtrie in @var{DB}.  The input tries are determined by the prefix
terms of @var{P1} and @var{P2}, and the place of the resulting trie is
determined by prefix term @var{P3}.  (See @pred{is_prefix_term/2} and
@pred{apply_op_in_db/6} for discussions of prefix terms.)").
:- mode sym_diff_in_db(?,?,?,+,?).
sym_diff_in_db(P1,P2,P3,DB0,DB) :-
    apply_op_in_db(lambda(A,B,C,sym_diff_db_b(A,B,C)),P1,P2,P3,DB0,DB).

sym_diff_db_b(T0,T1,T) :-
    sym_diff_tries(T0,T1,0,T).

sym_diff_tries(T1,T2,Depth,T) :-
    (T1 \== T2
     ->	sym_diff_tries0(T1,T2,Depth,T)
     ;	T = []
    ).

sym_diff_tries0([],T,_,T).
sym_diff_tries0(RTLA,RTLB,Depth,T) :- RTLA = rtl(K1,_,_),
    sym_diff_with_rtl(RTLA,RTLB,Depth,T,K1).
sym_diff_tries0(RTA,RTLB,Depth,T) :- RTA = rt(_,_,_,_),
    sym_diff_with_rt(RTA,RTLB,Depth,T).

:- index sym_diff_with_rtl/5-2.
sym_diff_with_rtl(T,[],_Depth,T,_K1).
sym_diff_with_rtl(RTLA,RTLB,Depth,T,K1) :- RTLB = rtl(K2,_,_),
    hash_key(K1,Hash1),
    hash_key(K2,Hash2),
    (Hash1 =:= Hash2
     ->	rtl_sym_diff(RTLA,RTLB,T)
     ;	union_with_rtl(RTLA,RTLB,Depth,T,K1)
    ).
sym_diff_with_rtl(RTLA,RTB,Depth,T,Key) :- RTB = rt(_,_,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    sym_diff_tries_ind_recur_l(RTLA,RTB,Depth1,T0,NInd),
    raise_subtree(T0,T).
    
:- index sym_diff_with_rt/4-2.
sym_diff_with_rt(T,[],_Depth,T).
sym_diff_with_rt(RTA,RTLB,Depth,T) :- RTLB = rtl(Key,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    sym_diff_tries_ind_recur_r(RTA,RTLB,Depth1,T0,NInd),
    raise_subtree(T0,T).
sym_diff_with_rt(rt(RT01,RT02,RT03,RT04),rt(RT11,RT12,RT13,RT14),Depth,T) :-
    Depth1 is Depth + 2,
    sym_diff_tries(RT01,RT11,Depth1,RT1),
    sym_diff_tries(RT02,RT12,Depth1,RT2),
    sym_diff_tries(RT03,RT13,Depth1,RT3),
    sym_diff_tries(RT04,RT14,Depth1,RT4),
    raise_subtree(rt(RT1,RT2,RT3,RT4),T).

rtl_sym_diff([],RTL2,RTL2).
rtl_sym_diff(RTL1,RTL2,RTL3) :- RTL1 = rtl(K1,Vs1,RTL1r),
    (RTL2 == []
     ->	RTL3 = RTL1
     ;	RTL2 = rtl(K2,Vs2,RTL2r),
	term_compare(K1,K2,Cmp),
	(Cmp < 0
	 -> RTL3 = rtl(K1,Vs1,RTL3r),
	    rtl_sym_diff(RTL1r,RTL2,RTL3r)
	 ;  Cmp =:= 0
	 -> order_sym_diff(Vs1,Vs2,Vs3),
	    (Vs3 == []		% raise!
	     ->	rtl_sym_diff(RTL1r,RTL2r,RTL3)
	     ;	RTL3 = rtl(K1,Vs3,RTL3r),
		rtl_sym_diff(RTL1r,RTL2r,RTL3r)
	    )
	    ;  RTL3 = rtl(K2,Vs2,RTL3r),
	    rtl_sym_diff(RTL1,RTL2r,RTL3r)
	)
    ).

order_sym_diff([],L,L).
order_sym_diff(L1,L2,L3) :- L1 = [_|_],
    order_sym_diff_list(L1,L2,L3).
order_sym_diff(Trie1,Trie2,Trie3) :- Trie1 = rtl(_,_,_),
    sym_diff_db_b(Trie1,Trie2,Trie3).
order_sym_diff(Trie1,Trie2,Trie3) :- Trie1 = rt(_,_,_,_),
    sym_diff_db_b(Trie1,Trie2,Trie3).

order_sym_diff_list([],L,L).
order_sym_diff_list(L1,L2,L3) :- L1 = [V1|L1r],
    (L2 == []
     ->	L3 = L1
     ;	L2 = [V2|L2r],
	term_compare(V1,V2,Ord),
	(Ord < 0
	 -> L3 = [V1|L3r],
	    order_sym_diff_list(L1r,L2,L3r)
	 ;  Ord =:= 0
	 -> order_sym_diff_list(L1r,L2r,L3)
	 ;  L3 = [V2|L3r],
	    order_sym_diff_list(L1,L2r,L3r)
	)
    ).

:- index sym_diff_tries_ind_recur_l/5-5.
sym_diff_tries_ind_recur_l(RT,rt(RT1,RT2,RT3,RT4),D,rt(ST,RT2,RT3,RT4),0) :-
    sym_diff_tries(RT,RT1,D,ST).
sym_diff_tries_ind_recur_l(RT,rt(RT1,RT2,RT3,RT4),D,rt(RT1,ST,RT3,RT4),1) :-
    sym_diff_tries(RT,RT2,D,ST).
sym_diff_tries_ind_recur_l(RT,rt(RT1,RT2,RT3,RT4),D,rt(RT1,RT2,ST,RT4),2) :-
    sym_diff_tries(RT,RT3,D,ST).
sym_diff_tries_ind_recur_l(RT,rt(RT1,RT2,RT3,RT4),D,rt(RT1,RT2,RT3,ST),3) :-
    sym_diff_tries(RT,RT4,D,ST).

:- index sym_diff_tries_ind_recur_r/5-5.
sym_diff_tries_ind_recur_r(rt(RT1,RT2,RT3,RT4),RT,D,rt(ST,RT2,RT3,RT4),0) :-
    sym_diff_tries(RT1,RT,D,ST).
sym_diff_tries_ind_recur_r(rt(RT1,RT2,RT3,RT4),RT,D,rt(RT1,ST,RT3,RT4),1) :-
    sym_diff_tries(RT2,RT,D,ST).
sym_diff_tries_ind_recur_r(rt(RT1,RT2,RT3,RT4),RT,D,rt(RT1,RT2,ST,RT4),2) :-
    sym_diff_tries(RT3,RT,D,ST).
sym_diff_tries_ind_recur_r(rt(RT1,RT2,RT3,RT4),RT,D,rt(RT1,RT2,RT3,ST),3) :-
    sym_diff_tries(RT4,RT,D,ST).

/***********************************************************/

:- comment(disjoint_db/2, "@pred{disjoint_db(+DB1,+DB2)} succeeds if
the prolog databases @var{DB1} and @var{DB2} are disjoint.").
:- mode disjoint_db(+,+).
disjoint_db(H1,H2) :-
    tab_intern_termhash(DB1,H1),
    tab_intern_termhash(DB2,H2),
    disjoint_db_b(DB1,DB2).

:- comment(disjoint_in_db/3, "@pred{disjoint_in_db(+Pref1,+Pref2,+DB)}
succeeds if the tuples specified by prefix term @var{Pref1} are
disjoint from those specified by @var{Pref2} in @var{DB}.  The number
of variables in @var{Pref1} and @var{Pref2} must be the same.").
:- mode disjoint_in_db(?,?,+).
disjoint_in_db(P1,P2,DB) :-
    apply_rel_op_in_db(lambda(A,B,disjoint_db_b(A,B)),P1,P2,DB).

disjoint_db_b(T0,T1) :-
    disjoint_tries(T0,T1,0).

disjoint_tries(T1,T2,Depth) :-
    (T1 \== T2
     ->	disjoint_tries0(T1,T2,Depth)
     ;	T1 == [], T2 == [] % disjoint only if both empty
    ).

disjoint_tries0([],_T,_D).
disjoint_tries0(RTLA,RTLB,Depth) :- RTLA = rtl(K1,_,_),
    disjoint_with_rtl(RTLA,RTLB,Depth,K1).
disjoint_tries0(RTA,RTLB,Depth) :- RTA = rt(_,_,_,_),
    disjoint_with_rt(RTA,RTLB,Depth).

:- index disjoint_with_rtl/4-2.
disjoint_with_rtl(_T,[],_Depth,_Key).
disjoint_with_rtl(RTLA,RTLB,_Depth,K1) :- RTLB = rtl(K2,_,_),
    hash_key(K1,Hash1),
    hash_key(K2,Hash2),
    (Hash1 =:= Hash2
     ->	rtl_disjoint(RTLA,RTLB)
     ;	true
    ).
disjoint_with_rtl(RTLA,RTB,Depth,Key) :- RTB = rt(_,_,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    disjoint_tries_ind_recur_l(RTLA,RTB,Depth1,NInd).
    
:- index disjoint_with_rt/3-2.
disjoint_with_rt(_T,[],_Depth).
disjoint_with_rt(RTA,RTLB,Depth) :- RTLB = rtl(Key,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    disjoint_tries_ind_recur_r(RTA,RTLB,Depth1,NInd).
disjoint_with_rt(rt(RT01,RT02,RT03,RT04),rt(RT11,RT12,RT13,RT14),Depth) :-
    Depth1 is Depth + 2,
    disjoint_tries(RT01,RT11,Depth1),
    disjoint_tries(RT02,RT12,Depth1),
    disjoint_tries(RT03,RT13,Depth1),
    disjoint_tries(RT04,RT14,Depth1).

order_disjoint([],L) :- L == [].
order_disjoint(L1,L2) :-  L1 = [_|_],
    order_disjoint_list(L1,L2).
order_disjoint(Trie1,Trie2) :- Trie1 = rtl(_,_,_),
    disjoint_db_b(Trie1,Trie2).
order_disjoint(Trie1,Trie2) :- Trie1 = rt(_,_,_,_),
    disjoint_db_b(Trie1,Trie2).

order_disjoint_list([],_L).
order_disjoint_list(L1,L2) :- L1 = [V1|L1r],
    (L2 == []
     ->	true
     ;	L2 = [V2|L2r],
	term_compare(V1,V2,Ord),
	(Ord < 0
	 -> order_disjoint_list(L1r,L2)
	 ;  Ord =:= 0
	 -> fail
	 ;  order_disjoint_list(L1,L2r)
	)
    ).

rtl_disjoint([],_).
rtl_disjoint(RTL1,RTL2) :- RTL1 = rtl(K1,V1,RTL1r),
    (RTL2 == []
     ->	true
     ;	RTL2 = rtl(K2,V2,RTL2r),
	term_compare(K1,K2,Cmp),
	(Cmp < 0
	 -> rtl_disjoint(RTL1r,RTL2)
	 ;  Cmp =:= 0
	 -> order_disjoint(V1,V2)
	 ;  rtl_disjoint(RTL1,RTL2r)
	)
    ).

:- index disjoint_tries_ind_recur_l/4-4.
disjoint_tries_ind_recur_l(RTL,rt(RT1,_,_,_),D,0) :-
    disjoint_tries(RTL,RT1,D).
disjoint_tries_ind_recur_l(RTL,rt(_,RT2,_,_),D,1) :-
    disjoint_tries(RTL,RT2,D).
disjoint_tries_ind_recur_l(RTL,rt(_,_,RT3,_),D,2) :-
    disjoint_tries(RTL,RT3,D).
disjoint_tries_ind_recur_l(RTL,rt(_,_,_,RT4),D,3) :-
    disjoint_tries(RTL,RT4,D).

:- index disjoint_tries_ind_recur_r/4-4.
disjoint_tries_ind_recur_r(rt(RT1,_,_,_),RTL,D,0) :-
    disjoint_tries(RT1,RTL,D).
disjoint_tries_ind_recur_r(rt(_,RT2,_,_),RTL,D,1) :-
    disjoint_tries(RT2,RTL,D).
disjoint_tries_ind_recur_r(rt(_,_,RT3,_),RTL,D,2) :-
    disjoint_tries(RT3,RTL,D).
disjoint_tries_ind_recur_r(rt(_,_,_,RT4),RTL,D,3) :-
    disjoint_tries(RT4,RTL,D).

:- comment(xprod_in_db/5,"
@pred{xprod_in_db(+Pref1,+Pref2,+Pref3,+DB0,?DB)} computes the cross
product of the tries determined by @var{Pref1} and @var{Pref2} and
puts the result at the position determined by @var{Pref3}.  The number
of variables in @var{Pref3} must be the sum of the numbers of
variables in @var{Pref1} and @var{Pref2}.  For example, assuming the
call to build_db constructs a prolog database, @var{DB0}, the query:
@begin{verbatim}	   
| ?- build_db(DB),xprod_in_db(p(_,_),q(_,_,_),r(_,_,_,_,_),DB0,DB).
@end{verbatim}	   
would compute the cross product of relations p/2 and q/3 in @var{DB0}
putting the result in r/5 (with r tuples having the p values preceding
the q values). The predicate r/5 is added or replaced and the new DB
@var{DB} is generated. (Note that, since the form p/2 is converted to
prefix term p(_,_), p/2, q/3 and r/5 could be used in place of the
explicit prefix terms in this query.)

The tuples of r/5 would be as though it were defined by the following
Prolog rule:
@begin{verbatim}
r(P1,P2,Q1,Q2,Q3) :- r(P1,P2), q(Q1,Q2,Q3).
@end{verbatim}

The @var{Pref1} values must be facts (i.e., clauses without (non-true)
bodies.)  The complexity of this operation is the size of the
@var{Pref1} trie, since every leaf of the @var{Pref1} trie is
essentially extended with a pointer to the @var{Pref2} trie.").
:- mode xprod_in_db(?,?,?,+,?).
xprod_in_db(Pref1,Pref2,Pref3,H0,H) :-
    tab_intern_termhash(DB0,H0),
    xprod_in_db_b(Pref1,Pref2,Pref3,DB0,DB),
    tab_intern_termhash(DB,H).

xprod_in_db_b(Pref1,Pref2,Pref3,DB0,DB) :-
    cvt_pa(Pref1,Pref2,Pref1a,Pref2a),
    cvt_pa(Pref3,Pref3a),
    (find_subtrie_for_prefix([Pref2a],DB0,TLen2,HT2),
     find_subtrie_for_prefix([Pref1a],DB0,TLen1,HT1)
     ->	check_prefix_term([Pref3a],TLen3,'xprod_in_db/5'),
	(TLen3 =\= TLen1+TLen2
	 -> misc_error(('[prolog_db:xprod_in_db] ',Pref1,', ',Pref2,', and ',
			Pref3,' are inconsistent for cross product.'))
	 ;  update_subtrie_for_prefix([Pref3a],NSHT,_OSHT,TLen3,DB0,DB),
	    xprod_tries(HT1,HT2,NSHT)
	)
     ;  retractall_in_db_b(Pref3a,DB0,DB) % one or other component is empty
    ).

xprod_tries([],_T,[]).
xprod_tries(RTLA,RTB,T) :- RTLA = rtl(_,_,_),
    (RTB == []
     ->	T = []
     ;	xprod_tries_extend(RTLA,RTB,T)
    ).
xprod_tries(rt(RT01,RT02,RT03,RT04),RTB,T) :-
    (RTB == []
     ->	T = []
     ;	xprod_tries(RT01,RTB,RT1),
	xprod_tries(RT02,RTB,RT2),
	xprod_tries(RT03,RTB,RT3),
	xprod_tries(RT04,RTB,RT4),
	T = rt(RT1,RT2,RT3,RT4)
    ).

xprod_tries_extend([],_RTB,[]).
xprod_tries_extend(rtl(K,V,RTL),RTB,rtl(K,NV,NRTL)) :-
    (V = [true]
     ->	NV = RTB
     ;	xprod_tries(V,RTB,NV)
    ),
    xprod_tries_extend(RTL,RTB,NRTL).

/***********************************************************/

:- comment(materialize_in_db/4,
"@pred{materialize_in_db(+PrefI,+PrefO,+DB0,?DB)} calls the query
@var{PrefI} in @var{DB0} and asserts the term @var{Pref0} into
@var{DB} for every answer.  Normally every variable in @var{Pref0}
will be in the prefix term @var{PrefI}.").
:- mode materialize_in_db(?,?,+,?).
materialize_in_db(PrefI,PrefO,H0,H) :-
    tab_intern_termhash(DB0,H0),
    materialize_in_db_b(PrefI,PrefO,DB0,DB),
    tab_intern_termhash(DB,H).

materialize_in_db_b(PrefI,PrefO,DB0,DB) :-
    cvt_pa(PrefI,PrefO,PrefIa,PrefOa),
    prefix_term([PrefIa],_,Tail),
    prefix_term([PrefOa],_,Tail),
    findall(PrefOa,in_db_b(PrefIa,DB0),Facts),
    retractall_in_db_b(PrefOa,DB0,DB1),
    assert_in_db_b(Facts,DB1,DB).


:- comment(subsumed_db/2," @pred{subsumed_db(+DB1,+DB2)} succeeds if
database @var{DB1} is subsumed by database @var{DB2}.  If the
databases contain only facts, then this succeeds if the set of ground
instances of @var{DB1} is a subset of the set of ground instances of
@var{DB2}. If the databases contain rules, then an approximation is
attempted by comparing individual rules. ").
:- mode subsumed_db(+,+).
subsumed_db(H1,H2) :-
    tab_intern_termhash(DB1,H1),
    tab_intern_termhash(DB2,H2),
    subsumed_db_b(DB1,DB2).

/* Note: This implementation of subsumed_db requires in_db_b, and
so the usual implementation of subsumed_in_db doesn't work, since a
"tail trie" doesn't support that operation.  */

subsumed_db_b(DB1,DB2) :-
    difference_db_b(DB1,DB2,DDB),
    findall((H:-B),clause_in_db_b(H,B,DDB),Clauses),
    all_subsumed(Clauses,DB2).

all_subsumed([],_).
all_subsumed([(Head:-Body)|Clauses],DB) :-
    (Body = true
     ->	subsumed_fact_in_db(Head,DB)
     ;	subsumed_rule_in_db(Head,Body,DB)
    ),
    all_subsumed(Clauses,DB).

subsumed_fact_in_db(G,DB) :-
    term_variables(G,Vars),
    in_db_b(G,DB),
    is_most_general_term(Vars).

%% Hack for rules; must exist another single rule with eq-shorter body
%% with all subgoals subsuming this body
subsumed_rule_in_db(Head,Body,DB) :-
    term_variables(Head,Vars),
    clause_in_db_b(Head,GBody,DB),
    is_most_general_term(Vars),
    subsumes_goal_lists(GBody,Body).

subsumes_goal_lists(GBody,Body) :-
    (GBody = (G1,Gs)
     ->	subsumes_some(G1,Body),
	subsumes_goal_lists(Gs,Body)
     ;	subsumes_some(GBody,Body)
    ).

subsumes_some(G,Body) :-
    (Body = (B1,Bs)
     ->	(subsumes_chk(G,B1)
	 -> true
	 ;  subsumes_some(G,Bs)
	)
     ;	subsumes_chk(G,Body)
    ).


/***********************************************************/
:- comment(apply_op_in_db/6,
"@pred{apply_op_in_db(+OpLambda,+P1,+P2,+P3,+DB0,?DB)} applies a
binary database operation (union, intersect, subtract, etc.) to two
predicates in a database and puts the result into another predicate in
the database.  This is an internal predicate.  For example, to union
the tuples of p/3 and q/3 and put the result into r/3, one could do:

@begin{verbatim}
| ?- assert_in_db([p(a,c,d),p(a,d,e),q(a,c,d),q(b,c,d)],DB1),
	   apply_op_in_db(lambda(A,B,C,union_db_b(A,B,C)),p/3,q/3,r/3,DB1,DB2),
	   dump_db(userout,DB2).
@end{verbatim}

Here we have put tuples for @tt{p/3} and @tt{q/3} into @var{DB1}.
Then we union those two predicates, putting the result into a
predicate @tt{r/3} in the new database @var{DB2}.  And then we print
out the resulting database.  We must use the *_b form for union, since
it takes an unwrapped trie-term as required.

@var{Pref1}, @var{Pref2}, and @var{Pref3} must be prefix terms.  (For
convenience, the form @tt{Pred/Arity} is converted to a prefix term
using @pred{functor/3}.  Clearly, this is the most commonly used form
of prefix terms.)  The operation of @var{OpLambda} is of the form
@tt{lambda(Var1,...,Varn,Goal)}, which is called by binding the
arguments to @var{Var2} through @var{Varn} and then @tt{call}-ing
@var{Goal}.  The operation must take 2 input trie (databases) and
produce an output trie (database.)  The operation will be applied to
the two tries determined in the input database, @var{DB0}, by the two
prefix terms, and the resulting trie will replace the trie determined
by the target prefix term, creating a new database @var{DB}.  (The
target prefix term will be added if it is not in the input database,
@var{DB0}.)  ").

apply_op_in_db(Lambda,Pref1,Pref2,Pref3,H0,H) :-
    tab_intern_termhash(DB0,H0),
    apply_op_in_db_b(Lambda,Pref1,Pref2,Pref3,DB0,DB),
    tab_intern_termhash(DB,H).

apply_op_in_db_b(lambda(HT1,HT2,HT3,Op),Pref1,Pref2,Pref3,DB0,DB) :-
    cvt_pa(Pref1,Pref2,Pref1a,Pref2a),
    cvt_pa(Pref3,Pref3a),
    (find_subtrie_for_prefix([Pref1a],DB0,TL,HT1)
     ->	true
     ;	HT1 = [],
	check_prefix_term([Pref1a],TL,'*_in_db/5')
    ),
    (find_subtrie_for_prefix([Pref2a],DB0,TL2,HT2)
     ->	check_taillen(TL,TL2,Pref2a,'*_in_db/5')
     ;	HT2 = [],
	check_prefix_term([Pref2a],TL,'*_in_db/5')
    ),
    call(Op),
    (update_subtrie_for_prefix([Pref3a],HT3,_,TL,DB0,DB)
     ->	true
     ;	misc_error(('[*_in_db/5] Predicate selection pattern: ',Pref3a,
		    ' is inconsistent with other patterns.'))
    ).


:- comment(apply_rel_op_in_db/4, "
@pred{apply_rel_op_in_db(+RelOpLambdaTerm,+Pref1,+Pref2,+DB)} applies
the relational operator specified in @var{RelOpLambdaTerm} to the
subtries determined by @var{Pref1} and @var{Pref2} in @var{DB} and
succeeds or fails accordingly.  This is an internal predicate.
@var{RelOpLambdaTerm} is of the form @tt{lambda(HT1,HT2,Goal)}, where
@var{HT1} and @var{HT2} are variables and @var{Goal} is a binary goal
whose arguments are those varaibles.  The goal must define the desired
relational operator on subtries.  ").

apply_rel_op_in_db(Lambda,Pref1,Pref2,H) :-
    tab_intern_termhash(DB,H),
    apply_rel_op_in_db_b(Lambda,Pref1,Pref2,DB).

apply_rel_op_in_db_b(lambda(HT1,HT2,Op),Pref1,Pref2,DB) :-
    cvt_pa(Pref1,Pref2,Pref1a,Pref2a),
    (find_subtrie_for_prefix([Pref1a],DB,TL,HT1)
     -> true
     ;	HT1 = [],
	check_prefix_term([Pref1a],TL,'*_in_db/3')
    ),
    (find_subtrie_for_prefix([Pref2a],DB,TL,HT2)
     ->	true
     ;	HT2 = [],
	check_prefix_term([Pref2a],TL,'*_in_db/3')
    ),
    call(Op).


:- comment(find_subtrie_for_prefix/4,
"@pred{find_subtrie_for_prefix(+HeadTail,+HT,-TailLen,-SHT)}
searches for the ht-trie associated in the ht-trie @var{HT} with the
first subterm of @var{HeadTail} that is a variable.  @var{HeadTail}
must consist (when sequentialized) of a sequence of bound terms
followed by a sequence of distinct variables.  If this condition does
not hold, the predicate fails.  The ht-trie that is found is returned
in @var{SHT}, and the number of variables remaining in the tail is
returned in @var{TailLen}.").

find_subtrie_for_prefix([HeadTerm|HeadTail],HT,TailLen,SHT) :-
    nonvar(HeadTerm),
    get_key_new_terms([HeadTerm],Key,HTerms),
    find_in_radix_tree_b(Key,SHT0,HT),
    append(HTerms,HeadTail,NHeadTail),
    (is_most_general_term(NHeadTail)
     ->	length(NHeadTail,TailLen),
	SHT = SHT0
     ;	find_subtrie_for_prefix(NHeadTail,SHT0,TailLen,SHT)
    ).

%% rename to update_subtrie/4.
:- comment(update_subtrie_for_prefix/6,
"@pred{update_subtrie_for_prefix(+HeadTail,+NSHT,+OSHT,-TailLen,+HT0,-HT)}
replaces the ht-trie that is associated in the ht-trie @var{HT0} with
the first variable subterm of @var{HeadTail} with the ht-trie
@var{NSHT}.  The ht-trie that is replaced is returned in @var{OSHT}.
The new ht-trie with the replaced subterm is returned in @var{HT}.
@var{HeadTail} must consist (when sequentialized) of a sequence of
bound terms followed by a sequence of distinct variables.  If this
condition does not hold, the predicate fails.  ").

%% DSW: can we combine this with assert_ref_with_trie, and have both efficient?
:- mode update_subtrie_for_prefix(?,+,?,?,+,?).
update_subtrie_for_prefix(HeadTail,NSHT,OSHT,TailLen,HT0,HT) :-
    (NSHT == []
     ->	check_prefix_term(HeadTail,TailLen,'*_in_db/5'),
	delete_tuple_from_trie(HeadTail,_,_,HT0,HT)
     ;	update_subtrie_for_prefix_1(HeadTail,NSHT,OSHT,TailLen,HT0,HT)
    ).

update_subtrie_for_prefix_1([HeadTerm|HeadTail],NSHT,OSHT,TailLen,HT0,HT) :-
    nonvar(HeadTerm),
    NSHT \== [],
    get_key_new_terms([HeadTerm],Key,HTerms),
    append(HTerms,HeadTail,NHeadTail),
    (is_most_general_term(NHeadTail)
     ->	length(NHeadTail,TailLen),
	update_radix_tree(Key,NSHT,OSHT,HT0,HT)
     ;	update_radix_tree(Key,NSHT0,OSHT0,HT0,HT),
	update_subtrie_for_prefix_1(NHeadTail,NSHT,OSHT,TailLen,OSHT0,NSHT0)
    ).

:- comment(update_radix_tree/5,
"update_radix_tree(+Term,?NewVal,-OldVal,+HashTree0,-HashTree) adds a
new key-value pair (or finds and replaces an existing pair),
@var{Term} and @var{NewVal}, to the hash-tree, @var{HashTree0},
producing the new hash-tree, @var{HashTree}.  @var{OldVal} is bound to
the previous value for this key in the hash table (or to [] if this is
a new key.)").

% update_radix_tree/5
update_radix_tree(Key,NVal,OVal,Tree0,Tree) :-
    hash_key(Key,HashCode),
    update_radix_tree(Tree0,Key,NVal,HashCode,0,OVal,Tree).
    
%update_radix_tree(Tree0,Key,NVal,HashCode,Depth,OVal,Tree)
update_radix_tree([],Key,Val,_,_,[],rtl(Key,Val,[])).  % add if not there
update_radix_tree(ORTL,Key,Val,HashCode,Depth,OVal,Tree) :- ORTL = rtl(OKey,_,_),
    hash_key(OKey,HashOCode),
    (HashCode =:= HashOCode
     ->	rtl_insert(ORTL,Key,Val,OVal,Tree) % ?? why insert?
     ;	OVal = [], % adding a new entry so old is []
	add_pair_to_empty_radix_tree(HashCode,Key,Val,HashOCode,ORTL,Depth,Tree)
    ).
update_radix_tree(RT,Key,ValTran,HashCode,Depth,OVal,Tree) :- RT = rt(_,_,_,_),
    Index is (HashCode >> Depth) /\ 3,
    NDepth is Depth + 2,

    update_right_radix_tree(RT,Key,ValTran,HashCode,NDepth,OVal,Tree,Index).

:- index update_right_radix_tree/8-8.
update_right_radix_tree(rt(RT1,RT2,RT3,RT4),Key,VT,HashCode,D,OVal,rt(NRT,RT2,RT3,RT4),0) :-
    update_radix_tree(RT1,Key,VT,HashCode,D,OVal,NRT).
update_right_radix_tree(rt(RT1,RT2,RT3,RT4),Key,VT,HashCode,D,OVal,rt(RT1,NRT,RT3,RT4),1) :-
    update_radix_tree(RT2,Key,VT,HashCode,D,OVal,NRT).
update_right_radix_tree(rt(RT1,RT2,RT3,RT4),Key,VT,HashCode,D,OVal,rt(RT1,RT2,NRT,RT4),2) :-
    update_radix_tree(RT3,Key,VT,HashCode,D,OVal,NRT).
update_right_radix_tree(rt(RT1,RT2,RT3,RT4),Key,VT,HashCode,D,OVal,rt(RT1,RT2,RT3,NRT),3) :-
    update_radix_tree(RT4,Key,VT,HashCode,D,OVal,NRT).

check_taillen(T1,T2,HeadTail,MsgPred) :-
    (T1 = T2
     ->	true
     ;	misc_error(('[',MsgPred,'] Predicate selection pattern: ',HeadTail,
		    ' is inconsistent with other patterns.'))
    ).

check_prefix_term(HeadTail,TailLen,MsgPred) :-
    (is_prefix_term(HeadTail,TailLen0)
     ->	check_taillen(TailLen,TailLen0,HeadTail,MsgPred)
     ;	misc_error(('[',MsgPred,'] Predicate selection pattern: ',HeadTail,' is illegal.'))
    ).

:- comment(is_prefix_term/2,"
@pred{is_prefix_term(+HeadTail,?TailLen)} checks that @var{HeadTail}
is a @tt{prefix-term}.  A @tt{prefix term} is a prolog term that, when
viewed in prefix order, consists of a sequence of constants followed
by a sequence of distinct variables.  Such a term uniquely determines
a position in a term trie (i.e., prolog DB). ").

is_prefix_term(HeadTail,TailLen) :-
    prefix_term(HeadTail,_Pre,Tail),
    length(Tail,TailLen).

%% fix for radix functor rep
/*prefix_term(PT,Prefix,TailVars) :-
    PT = [Head|Tail],
    (var(Head)
     ->	is_most_general_term(PT),
	TailVars = PT,
	Prefix = []
     ; atomic(Head)
     ->	Prefix = [Head|PrefixT],
	prefix_term(Tail,PrefixT,TailVars)
     ;	Head =.. [_Functor|Args],
	append(Args,Tail,ArgsTail),
	functor(Head,Mod,Functor,Arity),
	Prefix = [functor(Mod,Functor,Arity)|PrefixT],
	prefix_term(ArgsTail,PrefixT,TailVars)
  ). */

prefix_term(PT,Prefix,TailVars) :-
    PT = [Head|_],
    (var(Head)
     ->	is_most_general_term(PT),
	TailVars = PT,
	Prefix = []
     ;	get_key_new_terms(PT,Key,NTerms),
	Prefix = [Key|PrefixT],
	prefix_term(NTerms,PrefixT,TailVars)
    ).


/*** Seems of no use; idea was to subset DB to only needed part to
maximize table use for incremental computation, but would only table
the DB lookup which is about as fast as table lookup anyway.

:- comment(split_prefix_term_b/4,"
	   ").
split_prefix_term_b(Term,Tail,DB0,DB) :-
    split_prefix_term_b0([Term],Tail,DB0,DB).

split_prefix_term_b0([],[],DB,DB).
split_prefix_term_b0(Terms,Tail,DB0,DB) :- Terms = [Term|Terms1],
    (var(Term)
     ->	Tail = Terms, DB = DB0
     ;	get_key_new_terms(Terms,Key,NTerms),
	find_in_radix_tree_b(Key,SDB,DB0),
	split_prefix_term_b0(NTerms,Tail,SDB,DB)
    ).
    
split_clause_in_db_b(Terms,Body,DB) :-
    find_tuple_in_trie(Terms,Body0,Vars,DB),
    pdb_unnumbervars(Body0,0,Body,Vars).
***/

/***********************************************************/

rtl_insert([],Key,Val,[],rtl(Key,Val,[])).
rtl_insert(rtl(OK,OVal0,RTLr),K,Val,OVal,NRTL) :-
    (OK @< K
     ->	NRTL = rtl(OK,OVal0,NRTLr),
	rtl_insert(RTLr,K,Val,OVal,NRTLr)
     ; OK == K
     ->	OVal = OVal0,
	NRTL = rtl(OK,Val,RTLr)	
     ;	OVal = [],
	NRTL = rtl(K,Val,rtl(OK,OVal0,RTLr))
    ).

appl(lambda(X,Y,G),X,Y) :- call_c(G). % HiLog
%%appl(lambda(X,G),X) :- call_c(G).	 % HiLog


cvt_pa(PA,PSkel) :-
    (PA = P/A
     ->	functor(PSkel,P,A)
     ;	PSkel = PA
    ).

cvt_pa(PA1,PA2,PSkel1,PSkel2) :- 
    (PA1 = P1/A1
     ->	functor(PSkel1,P1,A1),
	PSkel1 =.. [P1|Args],
	(PA2 = P2/A2
	 -> A1 =:= A2,
	    PSkel2 =.. [P2|Args]
	 ; prefix_term([PA2],_,Args)
	 -> PSkel2 = PA2
	 ;  misc_error('ERROR: *_in_db: Bad or inconsistent prefix terms')
	)
     ; PA2 = P2/A2
     ->	functor(PSkel2,P2,A2),
	PSkel2 =.. [P2|Args],
	(PA1 = P1/A1
	 -> A2 =:= A1,
	    PSkel1 =.. [P1|Args]
	 ; prefix_term([PA1],_,Args)
	 -> PSkel1 = PA1
	 ;  misc_error('ERROR: *_in_db: Bad or inconsistent prefix terms')
	)
     ; prefix_term([PA1],_,_),prefix_term([PA2],_,_)
     ->	PSkel1 = PA1,
	PSkel2 = PA2
     ;	misc_error('ERROR: *_in_db: Bad or inconsistent prefix terms')
    ).


my_numbervars(Y,I,J) :-
    term_type(Y,T),
    my_numbervars(Y,I,J,T).

:- index my_numbervars/4-4.
my_numbervars(Y,I,J,/*XSB_FREE*/0) :-
    Y='_$Var$'(I),
    J is I+1.
my_numbervars(Y,I,J,/*XSB_ATTV*/7) :-
    Y='_$Var$'(I),
    J is I+1.
my_numbervars(Y,I,J,/*XSB_LIST*/3) :-
    Y=[A1|A2],
    my_numbervars(A1,I,I1),
    my_numbervars(A2,I1,J).
my_numbervars(Y,I,J,/*XSB_STRUCT*/1) :-
    term_psc(Y,PSC),
    psc_arity(PSC,N),
    my_numbervars(Y,I,J,1,N).
my_numbervars(_,I,I,/*XSB_STRING*/5).
my_numbervars(_,I,I,/*XSB_INT*/2).
my_numbervars(_,I,I,/*XSB_FLOAT*/6).

my_numbervars(Y,I,J,N,A) :-
    (N > A
     ->	J=I
     ;	term_arg(Y,N,Arg),
	my_numbervars(Arg,I,I1),
	N1 is N+1,
	my_numbervars(Y,I1,J,N1,A)
    ).


pdb_unnumbervars(TermIn,FirstN,TermOut,Vars) :-
    term_type(TermIn,Type),
    (Type =:= 1/*XSB_STRUCT*/
     ->	(TermIn = '_$Var$'(I), integer(I)
	 -> (I >= FirstN
	     ->	log_ith(I,Vars,TermOut)
	     ;	TermOut = TermIn
	    )
         ;  term_psc(TermIn,PSC),
	    term_new(PSC,NTermOut),
	    NTermOut=TermOut,
	    psc_arity(PSC,Arity),
	    pdb_unnumbervars(TermIn,FirstN,TermOut,Vars,1,Arity)
	)
     ;	Type =:= 3/*XSB_LIST*/
     ->	TermIn = [A1|A2],
	TermOut = [B1|B2],
	pdb_unnumbervars_list(A1,FirstN,B1,Vars,A2,B2) % avoid allocate
     ;	TermOut = TermIn
    ).

pdb_unnumbervars_list(A1,FirstN,B1,Vars,A2,B2) :-
    pdb_unnumbervars(A1,FirstN,B1,Vars),
    pdb_unnumbervars(A2,FirstN,B2,Vars).

pdb_unnumbervars(TermIn,FirstN,TermOut,Vars,I,Arity) :-
	(I =< Arity
         ->     term_arg(TermIn,I,ArgIn),
		term_arg(TermOut,I,ArgOut),
	        pdb_unnumbervars(ArgIn,FirstN,ArgOut,Vars),
		I1 is I+1,
		pdb_unnumbervars(TermIn,FirstN,TermOut,Vars,I1,Arity)
	 ;      true
        ).

my_functor(Term,Mod,Fun,Ari) :-
    (var(Term)
     ->	my_tab_functor(Term,Mod,Fun,Ari)
     ;	functor(Term,Mod,Fun,Ari)
    ).

:- table my_tab_functor/4.
my_tab_functor(Term,Mod,Fun,Ari) :-
    functor(Term,Mod,Fun,Ari).

:- comment(in_db/2, "@pred{in_db(+Goal,+DB)} calls the goal @var{Goal}
in the database @var{DB}, binding variables in @var{Goal} for each
instance of @var{Goal} that is proved using the clauses of
@var{DB}.").
:- mode in_db(?,+).
in_db(Goal,H) :-
    (var(Goal)
     ->	misc_error('[prolog_db] in_db/2, Illegal argument 1, must not be a variable')
     ; H == module
     ->	call(Goal)
     ;	tab_intern_termhash(DB,H),
	in_db_b(Goal,DB)
    ).

%% simpler alias, if desired.
in(Goal,H) :- in_db(Goal,H).

:- comment(in_db/3, "@pred{in_db(+Goal,+DB0,-DB)} is a
version of @pred{in_db/2} that returns the input database so that
it can be used as an identity transformation in a DCG rule.").
:- mode in_db(?,+,?).
in_db(Goal,H,H) :-
    in_db(Goal,H).

in_db_b(true,_DB) :- !.
in_db_b((A,B),DB) :- !, in_db_b(A,DB), in_db_b(B,DB).
in_db_b(\+(G),DB) :- !, \+ in_db_b(G,DB).
in_db_b(tnot(G),DB) :- !, tnot db_table_call(DB,G).
in_db_b(findall(T,G,L),DB) :- !, findall(T,in_db_b(G,DB),L).
in_db_b((do_all G),DB) :- !, (do_all in_db_b(G,DB)).
in_db_b((C->A;B),DB) :- !,
    (in_db_b(C,DB) -> in_db_b(A,DB) ; in_db_b(B,DB)).
in_db_b((A;B),DB) :- !, (in_db_b(A,DB) ; in_db_b(B,DB)).
in_db_b(table(Goal),DB) :- !, db_table_call(DB,Goal).
in_db_b(xact(G),DB) :- !, phrase(G,DB,DB).  % for transaction logic, if useful.
in_db_b(G,_DB) :- functor(G,F,A), standard_symbol(F,A,_), !,
    call_c(G).
in_db_b(G,DB) :-
    clause_in_db_b(G,B,DB),
    in_db_b(B,DB).
   
:- table db_table_call/2.
db_table_call(DB,Goal) :-
    in_db_b(Goal,DB).


:- comment(db_to_list/2,"@var{db_to_list(+DB,?List)} returns the
clauses in @var{DB} in the list @var{List}, in sorted order.  A fact
is represented as its term; a rule is represented as a @tt{:-/2}
term.").

db_to_list(HDB,L) :-
    tab_intern_termhash(DB,HDB),
    findall(C,(clause_in_db_b(H,B,DB),
	       (B == true
		-> C = H
		;  C = (H:-B)
	       )),UL),
    sort(UL,L).

:- comment(dump_db/2,"@var{dump_db(+Filename,+DB)} writes the clauses
in DB, @var{DB}, to the file @var{Filename}.  If @var{Filename} is
'userout', the clauses will be written to userout. ").
:- mode dump_db(+,+).
dump_db(FileName,H) :-
    tab_intern_termhash(DB,H),
    dump_db_b(FileName,DB).


:- comment(dump_db/3,"@var{dump_db(+Filename,+DB0,+DB)} is a version
of @pred{dump_db/2} that returns the input database, so that it can be
used as an identity transformation in a DCG rule.").
:- mode dump_db(+,+,?).
dump_db(FileName,H,H) :-
    dump_db(FileName,H).

dump_db_b(Filename,DB) :-
    (Filename == userout
     ->	OStr = userout
     ;	open(Filename,write,OStr)
    ),
    (do_all
     clause_in_db_b(Head,Body,DB),
     (Body == true
      -> write_canonical_lettervar(OStr,Head)
      ;	 write_canonical_lettervar(OStr,(Head:-Body))
     ),
     writeln(OStr,'.')
    ),
    (Filename == userout
     ->	true
     ;	close(OStr)
    ).

write_db(DB) :- write_db(userout,DB).

write_label_db(Label,DB) :-
    write(Label), write(': '),
    write_db(userout,DB).

write_label_db(Filename,Label,DB) :-
    write(Filename,Label), write(Filename,': '),
    write_db(Filename,DB).

writeln_db(DB) :- writeln_db(userout,DB).

writeln_label_db(Label,DB) :-
    write(Label), write(': '),
    writeln_db(DB).

writeln_label_db(Filename,Label,DB) :-
    write(Filename,Label), write(Filename,': '),
    writeln_db(Filename,DB).

write_db(Filename,DB) :-
    (Filename == userout
     ->	OStr = userout
     ;	open(Filename,write,OStr)
    ),
    findall((Head:-Body),clause_in_db(Head,Body,DB),Clauses),
    (Clauses = [Clause1|ClauseR]
     ->	write_clause(OStr,Clause1),
	write_db_list(OStr,ClauseR),
	write(OStr,'.')
     ;	write(OStr,'(empty).')
    ),    
    (Filename == userout
     ->	true
     ;	close(OStr)
    ).
    
writeln_db(Filename,DB) :-
    (Filename == userout
     ->	OStr = userout
     ;	open(Filename,write,OStr)
    ),
    findall((Head:-Body),clause_in_db(Head,Body,DB),Clauses),
    (Clauses = [Clause1|ClauseR]
     ->	write_clause(OStr,Clause1),
	write_db_list(OStr,ClauseR),
	writeln(OStr,'.')
     ;	writeln(OStr,'(empty).')
    ),    
    (Filename == userout
     ->	true
     ;	close(OStr)
    ).

:- index write_db_list/2-2.
write_db_list(_,[]).
write_db_list(OStr,[Clause|Clauses]) :-
    write(OStr,', '),
    write_clause(OStr,Clause),
    write_db_list(OStr,Clauses).

write_clause(OStr,Clause) :-
    (Clause = (Head :- true)
     ->	write(OStr,Head)
     ;	write(OStr,Clause)
    ).

:- comment(pure_in_db/2,"@pred{pure_in_db(+G,+DB)} has the
same functionality as @pred{in_db(G,DB)}, but it is implemented
without cuts, so it can be used when this file is loaded into a DB by
@pred{load_in_db/2}. ").
:- mode pure_in_db(?,+).
pure_in_db(G,H) :-
    (H == module
     ->	call(G)
     ;	tab_intern_termhash(DB,H),
	pure_in_db_b(G,_,DB)
    ).

pure_in_db_b(G,Vars,DB) :-
    (ctrl_str(G)
     ->	pure_ctrl_in_db_b(G,Vars,DB)
     ;	G \== true,functor(G,F,A),std_xsb:standard_symbol(F,A,_)
     ->	pdb_unnumbervars(G,0,G0,Vars), call_c(G0)
%%     ;	clause_in_db_b(G,B,DB),
     ;  pdb_unnumbervars(G,0,G0,Vars), find_tuple_in_trie([G0],B,Vars1,DB),
	pure_in_db_b(B,Vars1,DB)
    ).

ctrl_str(true).
ctrl_str((_,_)).
ctrl_str(\+(_)).
ctrl_str(xact(_)).
ctrl_str((_;_)).

pure_ctrl_in_db_b(true,_,_).
pure_ctrl_in_db_b((A,B),Vs,DB) :- pure_in_db_b(A,Vs,DB), pure_in_db_b(B,Vs,DB).
pure_ctrl_in_db_b(\+(G),Vs,DB) :- \+ pure_in_db_b(G,Vs,DB).
pure_ctrl_in_db_b(xact(G),Vs,DB) :- !, pdb_unnumbervars(G,0,G0,Vs),phrase(G0,DB,DB).  % for transaction logic, if useful.
pure_ctrl_in_db_b((A;B),Vs,DB) :-
    (A = (C->A1)
     ->	(pure_in_db_b(C,Vs,DB) -> pure_in_db_b(A1,Vs,DB) ; pure_in_db_b(B,Vs,DB))
     ;	(pure_in_db_b(A,Vs,DB) ; pure_in_db_b(B,Vs,DB))
    ).
pure_ctrl_in_db_b(M:G,Vs,_DB) :- pdb_unnumbervars(M:G,0,M0:G0,Vs), call(M0:G0).

:- comment(new_db/1, "@pred{new_db(?DB)} generates (or tests for) and
empty trie-term database, @var{DB}. ").

new_db([]).

:- comment(is_db/1, "@pred{is_db(+DB)} succeeds if @var{DB} looks to
have the form of a valid database reference."  ).
:- mode is_db(+).	   
is_db(H) :-
    atom(H),
    (H == []
     ->	true
     ;	tab_intern_termhash(DB,H),
	DB \== H
    ).
	       
:- comment(load_in_db/2, "@pred{load_in_db(+FileName,-DB)} reads the
clauses in @file{FileName} and creates the database @var{DB} that
contains them. ").
:- mode load_in_db(+,?).
load_in_db(FileName,DB) :-
    new_db(DB0),
    load_in_db(FileName,DB0,DB).

:- comment(loadc_in_db/2, "@pred{loadc_in_db(+FileName,-DB)} uses
@tt{read_canonical} to read the clauses in @file{FileName} and creates
the database @var{DB} that contains them. The clauses in the file must
be in canonical form.  ").
:- mode loadc_in_db(+,-).
loadc_in_db(FileName,DB) :-
    new_db(DB0),
    loadc_in_db(FileName,DB0,DB).

:- comment(load_in_db/3, "@pred{load_in_db(+FileName,+DB0,-DB)} reads
the clauses in @file{FileName} and adds them to the database
@var{DB0}, creating a new database @var{DB}. ").
:- mode load_in_db(+,+,?).
load_in_db(FileName,H0,H) :-
    tab_intern_termhash(DB0,H0),
    load_in_db_b(FileName,DB0,DB),
    tab_intern_termhash(DB,H).

:- comment(loadc_in_db/3, "@pred{loadc_in_db(+FileName,+DB0,-DB)} uses
@tt{read_canonical} to read the clauses in @file{FileName} and adds
them to the database @var{DB0}, creating a new database @var{DB}. The
clauses in the file must be in canonical vorm.").
:- mode loadc_in_db(+,+,?).
loadc_in_db(FileName,H0,H) :-
    tab_intern_termhash(DB0,H0),
    loadc_in_db_b(FileName,DB0,DB),
    tab_intern_termhash(DB,H).

:- comment(load_in_db/4, "
@pred{load_in_db(+FileName,+ExclSet,+DB0,?DB)} reads the clauses in
@var{FileName} and, except for those that unify with clauses in the DB
@var{ExclSet}, asserts them into @var{DB0} generating @var{DB}.  The
most common use of the @var{ExclSet} is to avoid adding unwanted
directives.  ").
:- mode load_in_db(+,+,+,-).
load_in_db(FileName,HExcl,H0,H) :-
    tab_intern_termhash(DB0,H0),
    tab_intern_termhash(EDB,HExcl),
    load_in_db_b(FileName,term,EDB,DB0,DB),
    tab_intern_termhash(DB,H).

:- comment(loadc_in_db/4, "
@pred{loadc_in_db(+FileName,+ExclSet,+DB0,?DB)} uses
@tt{read_canonical} to read the clauses in @var{FileName} and, except
for those that unify with clauses in the DB @var{ExclSet}, asserts
them into @var{DB0} generating @var{DB}.  The most common use of the
@var{ExclSet} is to avoid adding unwanted directives.  The clauses in
the file must be in canonical form.  ").
:- mode loadc_in_db(+,+,+,?).
loadc_in_db(FileName,HExcl,H0,H) :-
    tab_intern_termhash(DB0,H0),
    tab_intern_termhash(EDB,HExcl),
    load_in_db_b(FileName,canon,EDB,DB0,DB),
    tab_intern_termhash(DB,H).

load_in_db_b(FileName,DB0,DB) :-
    new_db(EDB),
    load_in_db_b(FileName,term,EDB,DB0,DB).

loadc_in_db_b(FileName,DB0,DB) :-
    new_db(EDB),
    load_in_db_b(FileName,canon,EDB,DB0,DB).

load_in_db_b(FileName,IfCan,Excludes,DB0,DB) :-
    (concat_atom([Base,'.P'],FileName)
     ->	concat_atom([Base,'.H'],HFileName),
	(file_exists(HFileName)
	 -> open(HFileName,read,HStr),
	    (IfCan == canon
	     ->	read_canons_to_trie(HStr,Excludes,DB0,DB1)
	     ;  read_terms_to_trie(HStr,Excludes,DB0,DB1)
	    ),
	    close(HStr)
	 ;  DB1 = DB0
	)
     ;  DB1 = DB0
    ),
    open(FileName,read,IStr),
    (IfCan == canon
     ->	read_canons_to_trie(IStr,Excludes,DB1,DB)
     ;	read_terms_to_trie(IStr,Excludes,DB1,DB)
    ),
    close(IStr).

read_terms_to_trie(IStr,Excludes,DB0,DB) :-
    read(IStr,Term),
    (Term == end_of_file
     ->	DB = DB0
     ;	(clause_in_db_b(Term,true,Excludes)
	 -> DB1 = DB0
	 ; Term = :-(Directive)
	 -> process_directive(Directive,DB0,DB1)
	 ;  assert_in_db_b(Term,DB0,DB1)
	),
	read_terms_to_trie(IStr,Excludes,DB1,DB)
    ).

read_canons_to_trie(IStr,Excludes,DB0,DB) :-
    read_canonical(IStr,Term),
    (Term == end_of_file
     ->	DB = DB0
     ;	(clause_in_db_b(Term,true,Excludes)
	 -> DB1 = DB0
	 ; Term = :-(Directive)
	 -> process_directive(Directive,DB0,DB1)
	 ;  assert_in_db_b(Term,DB0,DB1)
	),
	read_canons_to_trie(IStr,Excludes,DB1,DB)
    ).

process_directive(Term,DB0,DB) :-
    (Term = (import PAs from Module)
     ->	process_import(PAs,Module,DB0,DB)
     ;	DB = DB0
    ).

process_import(PAs,Module,DB0,DB) :-
    (PAs = (P/A,PAs1)
     ->	functor(Call,P,A),
	(standard_symbol(P,A,_)
	 -> DB1 = DB0
	 ;  assert_in_db_b((Call :- Module:Call),DB0,DB1)
	),
	process_import(PAs1,Module,DB1,DB)
     ;	PAs = P/A,
	(standard_symbol(P,A,_)
	 -> DB = DB0
	 ;  functor(Call,P,A),
	    assert_in_db_b((Call :- Module:Call),DB0,DB)
	)
    ).
    


/********************************************************/

:- comment(join_in_db/6,
"@pred{join_in_db(+Pref1,+Pref2,+NJoin,+Pref3,+DB0,?DB)} performs an
equi-join of tuples of terms specified by prefix terms @var{Pref1} and
@var{Pref2} producing a new set of terms as specified in prefix term
@var{Pref3}.  @var{NJoin} is the number of join variables.  The join
arguments are the first @var{NJoin} fields of the tuples specified by
@var{Pref1} and @var{Pref2}.  The number of variables in @var{Pref3}
must be the sum of the numbers of variables in @var{Pref1} and
@var{Pref2} minus @var{NJoin}.  Assuming the call to @pred{build_db/1}
builds a prolog DB in @var{DB0}, an example query might be:
@begin{verbatim} | ?- build_db(DB0),join_in_db(p/3,q/4,1,r/6,DB0,DB).
@end{verbatim}

This query would materialize the predicate r/6 into @var{DB}, where
the r/6 definition in Prolog would be:
@begin{verbatim}
r(J,P2,P3,Q2,Q3,Q4) :- p(J,P2,P3), q(J,Q2,Q3,Q4).
@end{verbatim}

The p/3 clauses must be ground facts. (q/4 could have rules with
non-empty bodies, but I don't see any application for this.)  ").
:- mode join_in_db(?,?,+,?,+,?).
join_in_db(Pref1,Pref2,NJoin,Pref3,H0,H) :-
    tab_intern_termhash(DB0,H0),
    join_in_db_b(Pref1,Pref2,NJoin,Pref3,DB0,DB),
    tab_intern_termhash(DB,H).

join_in_db_b(Pref1,Pref2,NJoin,Pref3,DB0,DB) :-
%%    writeln(userout,DB0),
    cvt_pa(Pref1,Pref1a),
    cvt_pa(Pref2,Pref2a),
    cvt_pa(Pref3,Pref3a),
    find_subtrie_for_prefix([Pref1a],DB0,TLen1,HT1),
    find_subtrie_for_prefix([Pref2a],DB0,TLen2,HT2),
    OutLen is TLen1+TLen2-NJoin,
    check_prefix_term([Pref3a],OutLen,'join_in_db/6'),
%%    writeln(userout,HT1),
%%    writeln(userout,HT2),
    join_tries(HT1,HT2,NJoin,HT3),
    (HT3 == []
     ->	retractall_in_db_b(Pref3a,DB0,DB)
     ;	update_subtrie_for_prefix([Pref3a],HT3,_,_,DB0,DB)
    ).

join_tries(T1,T2,NJoin,T) :-
    join_tries0(T1,T2,NJoin,0,T).

join_tries0([],_T,_,_,[]).
join_tries0(RTLA,RTLB,NJoin,Depth,T) :- RTLA = rtl(K1,_,_),
    join_with_rtl(RTLA,RTLB,NJoin,Depth,T,K1).
join_tries0(RTA,RTLB,NJoin,Depth,T) :- RTA = rt(_,_,_,_),
    join_with_rt(RTA,RTLB,NJoin,Depth,T).

:- index join_with_rtl/6-2.
join_with_rtl(_T,[],_NJoin,_Depth,[],_K1).
join_with_rtl(RTLA,RTLB,NJoin,_Depth,T,K1) :- RTLB = rtl(K2,_,_),
    hash_key(K1,Hash1),
    hash_key(K2,Hash2),
    (Hash1 =:= Hash2
     ->	rtl_join(RTLA,RTLB,NJoin,T)
     ;	T = []
    ).
join_with_rtl(RTLA,RTB,NJoin,Depth,T,Key) :- RTB = rt(_,_,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    join_tries_ind_recur_l(RTLA,RTB,NJoin,Depth1,T0,NInd),
    raise_subtree(T0,T).
    
:- index join_with_rt/5-2.
join_with_rt(_T,[],_NJoin,_Depth,[]).
join_with_rt(RTA,RTLB,NJoin,Depth,T) :- RTLB = rtl(Key,_,_),
    hash_key(Key,Hash),
    NInd is (Hash >> Depth) /\ 3,
    Depth1 is Depth + 2,
    join_tries_ind_recur_r(RTA,RTLB,NJoin,Depth1,T0,NInd),
    raise_subtree(T0,T).
join_with_rt(rt(RT01,RT02,RT03,RT04),rt(RT11,RT12,RT13,RT14),NJoin,Depth,T) :-
    Depth1 is Depth + 2,
    join_tries0(RT01,RT11,NJoin,Depth1,RT1),
    join_tries0(RT02,RT12,NJoin,Depth1,RT2),
    join_tries0(RT03,RT13,NJoin,Depth1,RT3),
    join_tries0(RT04,RT14,NJoin,Depth1,RT4),
    raise_subtree(rt(RT1,RT2,RT3,RT4),T).

:- index join_tries_ind_recur_l/6-6.
join_tries_ind_recur_l(RTL,rt(RT1,_,_,_),N,D,rt(ST,[],[],[]),0) :-
    join_tries0(RTL,RT1,N,D,ST).
join_tries_ind_recur_l(RTL,rt(_,RT2,_,_),N,D,rt([],ST,[],[]),1) :-
    join_tries0(RTL,RT2,N,D,ST).
join_tries_ind_recur_l(RTL,rt(_,_,RT3,_),N,D,rt([],[],ST,[]),2) :-
    join_tries0(RTL,RT3,N,D,ST).
join_tries_ind_recur_l(RTL,rt(_,_,_,RT4),N,D,rt([],[],[],ST),3) :-
    join_tries0(RTL,RT4,N,D,ST).

:- index join_tries_ind_recur_r/6-6.
join_tries_ind_recur_r(rt(RT1,_,_,_),RTL,N,D,rt(ST,[],[],[]),0) :-
    join_tries0(RT1,RTL,N,D,ST).
join_tries_ind_recur_r(rt(_,RT2,_,_),RTL,N,D,rt([],ST,[],[]),1) :-
    join_tries0(RT2,RTL,N,D,ST).
join_tries_ind_recur_r(rt(_,_,RT3,_),RTL,N,D,rt([],[],ST,[]),2) :-
    join_tries0(RT3,RTL,N,D,ST).
join_tries_ind_recur_r(rt(_,_,_,RT4),RTL,N,D,rt([],[],[],ST),3) :-
    join_tries0(RT4,RTL,N,D,ST).

rtl_join([],_RTL2,_NJoin,[]).
rtl_join(RTL1,RTL2,NJoin,RTL3) :- RTL1 = rtl(K1,Vs1,RTL1r),
    (RTL2 == []
     ->	RTL3 = []
     ;	RTL2 = rtl(K2,Vs2,RTL2r),
	term_compare(K1,K2,Cmp),
	(Cmp < 0
	 -> rtl_join(RTL1r,RTL2,NJoin,RTL3)
	 ; Cmp =:= 0
	 -> functor(K1,_,Arity), % no $vars
	    NJoin1 is NJoin + Arity,
	    order_join(Vs1,Vs2,NJoin1,Vs3),
	    RTL3 = rtl(K1,Vs3,RTL3r),
	    rtl_join(RTL1r,RTL2r,NJoin,RTL3r)
	 ;  rtl_join(RTL1,RTL2r,NJoin,RTL3)
	)
    ).

order_join([],L2,_NJoin,L2).
order_join([_|_],L2,_NJoin,L2).
%order_join(L1,_L2,_NJoin,_L3) :- L1 = [_|_],
%    misc_error('ERROR: Illegal join').
order_join(L1,L2,NJoin,L3) :- L1 = rtl(_,_,_),
    order_join_tr(L1,L2,NJoin,L3).
order_join(L1,L2,NJoin,L3) :- L1 = rt(_,_,_,_),
    order_join_tr(L1,L2,NJoin,L3).

order_join_tr(L1,L2,NJoin,L3) :-
    NJoin1 is NJoin - 1,
    (NJoin1 =:= 0
     ->	xprod_tries(L1,L2,L3)
     ;	join_tries(L1,L2,NJoin1,L3)
    ).

/********************************************************/

:- comment(reorder_in_db/4,
"@pred{reorder_in_db(+PrefI,+PrefO,+DB0,DB)} generates a new version
of the set of tuples determined by the @var{PrefI} prefix term by
re-ordering its arguments, putting its result in the predicate
determined by @var{PrefO}.  Note that one can use this (carefully) to
change the index on @var{Pref1} since tuples are indexed in
left-to-right order.  The variables in @var{PrefO} must be a
permutation of the variables in @var{PrefI}, which defines the
reordering of the arguments.  Note that the tries that correspond to
an unchanged final sequence of variables in the two prefix terms will
not be traversed in the reordering.

The following example interchanges the first two arguments of @tt{p/4}
in @var{DB0} to create @tt{q/4} in @var{DB}:

@begin{verbatim}	   
| ?- build_db(DB0),reorder_in_db(p(A,B,C,D),q(B,A,C,D),DB0,DB).
@end{verbatim}	   

Note that reordering can bring a later argument to the beginning to
make it available for use in a call to @pred{join_in_db/6}. ").
:- mode reorder_in_db(?,?,+,?).
reorder_in_db(PrefI,PrefO,H0,H) :-
    tab_intern_termhash(DB0,H0),
    reorder_in_db_b(PrefI,PrefO,DB0,DB),
    tab_intern_termhash(DB,H).

reorder_in_db_b(PrefI,PrefO,DB0,DB) :-
    cvt_pa(PrefI,PrefIa),
    cvt_pa(PrefO,PrefOa),
    prefix_term([PrefIa],_IPr,IVars),
    is_most_general_term(IVars),
    prefix_term([PrefOa],_OPr,OVars),
    is_most_general_term(OVars),
    num_common_tail_vars(IVars,OVars,ReOrdVars,NCom),
    find_subtrie_for_prefix([PrefIa],DB0,TLen,IHT),
    NArgs is TLen - NCom,
    length(IArgs,NArgs),
    append(IArgs,_,IVars),
    check_prefix_term([PrefOa],TLen,'reorder_in_db/4'),
    reorder_trie(IHT,IVars,ReOrdVars,OVars,[],OutTupTrie),
    update_subtrie_for_prefix([PrefO],OutTupTrie,_,_,DB0,DB).

num_common_tail_vars(IV,OV,ODiffVars,NCom) :-
    (IV == OV
     ->	length(IV,NCom),
	ODiffVars = []
     ;	IV = [_|IVt],
	OV = [OV1|OVt],
	ODiffVars = [OV1|ODiffVars1],
	num_common_tail_vars(IVt,OVt,ODiffVars1,NCom)
    ).

/*reorder_trie(InpTupleTrie,InpTerms,ReordVars,OutPrefTail,OutTupTrie0,OutTupTrie).
InpTupleTrie: the input tuple trie
InpTerms: list of variables for InpTupleTrie
ReordVars: List of variables being reordered (must become ground to assert)
OutPrefTail: Terms of out prefix term subtrie (to add to OutTupleTrie)
OutTupTrie0: input tuple trie collecting output tuples
OutTupTrie: output tuple trie collecting output tuples
*/

reorder_trie([],_Terms,_Reord,_OutPref,OutSubTrie,OutSubTrie).
reorder_trie(RTL,Terms,ReordVars,OutPrefTail,OutSubTrie0,OutSubTrie) :-
    RTL = rtl(_,_,_),
    reorder_trie_rtl(RTL,Terms,ReordVars,OutPrefTail,OutSubTrie0,OutSubTrie).
reorder_trie(InpSubTrie,InpTuple,ReordVars,OutPrefTail,OutSubTrie0,OutSubTrie) :-
    InpSubTrie = rt(RT1,RT2,RT3,RT4),
    reorder_trie(RT1,InpTuple,ReordVars,OutPrefTail,OutSubTrie0,OutSubTrie1),
    reorder_trie(RT2,InpTuple,ReordVars,OutPrefTail,OutSubTrie1,OutSubTrie2),
    reorder_trie(RT3,InpTuple,ReordVars,OutPrefTail,OutSubTrie2,OutSubTrie3),
    reorder_trie(RT4,InpTuple,ReordVars,OutPrefTail,OutSubTrie3,OutSubTrie).

reorder_trie_rtl([],_Terms,_ReordVars,_OutPrefTail,OutSubTrie,OutSubTrie).
reorder_trie_rtl(rtl(K,Trie,RTLs),Terms,ReordVars,OutPrefTail,OutSubTrie0,OutSubTrie) :-
    copy_term(t(Terms,ReordVars,OutPrefTail),t(Terms1,ReordVars1,OutPrefTail1)),
    match_key_new_terms(Terms1,K,NTerms),
    (ground(ReordVars1)
     ->	update_subtrie_for_prefix(OutPrefTail1,Trie,_,_,OutSubTrie0,OutSubTrie1)
     ;	reorder_trie(Trie,NTerms,ReordVars1,OutPrefTail1,OutSubTrie0,OutSubTrie1)
    ),
    reorder_trie_rtl(RTLs,Terms,ReordVars,OutPrefTail,OutSubTrie1,OutSubTrie).


/********************************************************/

:- comment(project_in_db/5,
"@pred{project_in_db(+PrefI,+NArgs,+PrefO,+DB0,?DB)} creates a new
relation that projects away the first @var{NArgs} of the tuples of
@var{PrefI} in @var{DB0}, adding the result according to @var{PrefO}
in @var{DB}.  @var{PrefI} and @var{PrefO} must be prefix terms.  The
number of variables in @var{PrefI} minus @var{NArgs} must equal the
number of variables in @var{PrefO}.  An example of its use is:
	   
@begin{verbatim}	   
| ?- build_db(DB0),project_in_db(p(_,_,_,_),2,q(_,_),DB0,DB).
@end{verbatim}

which projects away the first two arguments of the @tt{p/4} relation
in @var{DB0} to generate the @tt{q/2} relation in @var{DB}. ").
:- mode project_in_db(?,+,?,+,?).
project_in_db(PrefI,NArgs,PrefO,H0,H) :-
    tab_intern_termhash(DB0,H0),
    project_in_db_b(PrefI,NArgs,PrefO,DB0,DB),
    tab_intern_termhash(DB,H).

project_in_db_b(PrefI,NArgs,PrefO,DB0,DB) :-
    cvt_pa(PrefI,PrefIa),
    cvt_pa(PrefO,PrefOa),
    (find_subtrie_for_prefix([PrefIa],DB0,TLen,IHT)
     ->	TLenO is TLen - NArgs,
	check_prefix_term([PrefOa],TLenO,'project_in_db/5'),
%%	project_ht(IHT,NArgs,OHT),
	project_trie(IHT,NArgs,OHT),
	(OHT == []
	 -> delete_from_trie_b(PrefOa,_,DB0,_,DB)
	 ;  update_subtrie_for_prefix([PrefOa],OHT,_,_,DB0,DB)
	)
     ;  DB = DB0
    ).


project_trie(Trie0,NArgs,Trie) :-
    project_trie(Trie0,NArgs,[],Trie).

project_trie([],_NArgs,PTrie,PTrie).
project_trie(Trie0,NArgs,PTrie0,PTrie) :-
    Trie0 = rtl(_,_,_),
    project_rtl(Trie0,NArgs,PTrie0,PTrie).
project_trie(Trie0,NArgs,PTrie0,PTrie) :-
    Trie0 = rt(RT1,RT2,RT3,RT4),
    project_trie(RT1,NArgs,PTrie0,PTrie1),
    project_trie(RT2,NArgs,PTrie1,PTrie2),
    project_trie(RT3,NArgs,PTrie2,PTrie3),
    project_trie(RT4,NArgs,PTrie3,PTrie).

project_rtl([],_,PTrie,PTrie).
project_rtl(rtl(_K,STrie,RTF),NArgs,PTrie0,PTrie) :-
    (NArgs =< 1
     ->	union_tries(PTrie0,STrie,0,PTrie1),
	project_rtl(RTF,NArgs,PTrie1,PTrie)
     ;	NArgs1 is NArgs - 1,
	project_trie(STrie,NArgs1,PTrie0,PTrie1),
	project_rtl(RTF,NArgs,PTrie1,PTrie)
    ).

/********************************************************/

:- comment(compare_size_db/3," @pred{compare_size_db(+DB1,+DB2,Order)}
succeeds if DB1 has fewer elements than DB2.  Often operations on two
sets can be done more efficiently if one knows which set is the smaller
one.  This predicate can be used to determine the smaller set.  It
runs (at worst) in time proportional to the size of the smaller set. "
).
:- mode compare_size_db(+,+,?).
compare_size_db(H1,H2,Order) :-
    tab_intern_termhash(DB1,H1),
    tab_intern_termhash(DB2,H2),
    compare_size_db_b(DB1,DB2,Order).

compare_size_db_b(DB1,DB2,Order) :-
    compare_size([DB1],0,[DB2],0,Order).

compare_size(T1s,C1,[],C2,Order) :- !,
    (C1 > C2
     ->	Order = 1
     ; T1s == []
     ->	(C1 =:= C2
	 -> Order = 0
	 ;  Order = -1
	)
     ;	T1s = [T1|T1sr],
	process_next_tree_comp(T1,T1sr,NTs,C1,NC1),
	compare_size(NTs,NC1,[],C2,Order)
    ).
compare_size([],C1,T2s,C2,Order) :- !,
    (C2 > C1
     ->	Order = -1
     ; T2s == []
     ->	(C1 =:= C2
	 -> Order = 0
	 ;  Order = 1
	)
     ;	T2s = [T2|T2sr],
	process_next_tree_comp(T2,T2sr,NTs,C2,NC2),
	compare_size([],C1,NTs,NC2,Order)
    ).
compare_size(T1s,C1,T2s,C2,Order) :-
    T1s = [T1|T1sr],
    T2s = [T2|T2sr],
    (T1 == T2
     ->	compare_size(T1sr,C1,T2sr,C2,Order)
     ; C1 =< C2
     ->	process_next_tree_comp(T1,T1sr,NTs,C1,NC1),
	compare_size(NTs,NC1,T2s,C2,Order)
     ;	process_next_tree_comp(T2,T2sr,NTs,C2,NC2),
	compare_size(T1s,C1,NTs,NC2,Order)
    ).

%%process_next_tree_comp(OT,OTs,NTs,C,NC)
process_next_tree_comp([],OTs,OTs,C,C) :- !.
process_next_tree_comp(rt(TT1,TT2,TT3,TT4),OTs,[TT1,TT2,TT3,TT4|OTs],C,C) :- !.
process_next_tree_comp(rtl(_K,V,RTL),OTs,[V,RTL|OTs],C,C) :- !.
process_next_tree_comp([T1|T1s],OTs,NTs,C,C) :- !, append_ne([T1|T1s],OTs,NTs).
process_next_tree_comp(_T,OTS,OTS,C,NC) :- NC is C + 1.

%% filter out []'s early.
append_ne([],L,L).
append_ne([X|L1],L2,L3) :-
    (X == []
     ->	append(L1,L2,L3)
     ;	L3 = [X|L3a],
	append_ne(L1,L2,L3a)
    ).


/********************************************************/

:- comment(findall_db/3, "@pred{findall_db(Template,Goal,DB)} is like
Prolog's findall/3 except it returns the found answers in an internal
prolog_db, DB, instead of in a list.  It uses tabling to avoid having
to build the list of (possibly duplicated) answers.  DOES NOT NEST! (but should)").
:- mode findall_db(?,?,?).
findall_db(Template,Goal,DB) :-
    copy_term((Template,Goal),(NTemplate,NGoal)), % protect incoming vars
    (findall_db0(NTemplate,NGoal,DB0)
     ->	abolish_table_pred(findall_db0(_,_,_)),
	DB = DB0
     ;	abolish_table_pred(findall_db0(_,_,_)),
	new_db(DB)
    ).
    
:- table findall_db0(_,_,fold(add_to_db(_,_,_),[])).
findall_db0(Template,Goal,NTemplate) :-
    copy_term((Template,Goal),(NTemplate,NGoal)), % need copy to call
    numbervars((Template,Goal),0,_), % aggr doesnt handle vars in group-by vars
    call(NGoal).

:- export add_to_db/3.
:- mode add_to_db(+,+,?).
add_to_db(DB0,Ans,DB) :-
    assert1_in_db(Ans,DB0,DB).

/********************************************************/

:- comment(sgdb/1,"@pred{sgdb(+DB)} makes @var{DB} the global prolog
DB. ").

:- dynamic '_$global_db'/1.
:- mode sgdb(+).
sgdb(DB) :-
    ('_$global_db'(DB)
     ->	true
     ;	retractall('_$global_db'(_)),
	assert('_$global_db'(DB))
    ).

:- comment(gdb/1,"@pred{sgdb(?DB)} unifies @var{DB} with the global
DB.  If no global DB has been set (by sgdb/1), it returns an empty
DB.  ").
:- mode gdb(?).
gdb(DB) :-
    ('_$global_db'(DB)
     ->	true
     ;	new_db(DB)
    ).




/********************************************************/

end_of_file.

Extra stuff that may come in handy??

radix_tree_size(T,Size) :-
     gensym('_ctr',Ctr),
     conset(Ctr,0),
     (do_all
      find_in_radix_tree_b(_,_,T),
      coninc(Ctr)
     ),
     conget(Ctr,Size).

:- table radix_tree_max_depth(_,lattice(max/3)).
max(A,B,C) :- (A > B -> C = A ; C = B).
radix_tree_max_depth(rtl(_,_,_),1).
radix_tree_max_depth(rt(R1,R2,R3,R4),K) :-
    (radix_tree_max_depth(R1,K0)
     ;
     radix_tree_max_depth(R2,K0)
     ;
     radix_tree_max_depth(R3,K0)
     ;
     radix_tree_max_depth(R4,K0)
    ),
    K is K0 + 1.


hash_bin(Term,BinList0,BinList) :-
    hash_key(Term,X),
    (X =\= 0
     ->	Bit is X /\ 3,
	X1 is X >> 2,
	tobin(X1,[Bit|BinList0],BinList)
     ;	BinList = BinList0
    ).

delete_from_radix_tree(Key,Val,Vars,Tree0,Tree) :-
    ValTrans = lambda(OV,NV,order_delete(OV,Val,NV)),
    delete_from_radix_tree_b(Key,ValTrans,Vars,Tree0,Tree).

%% unused??
delete_list_from_radix_tree([],T,T).
delete_list_from_radix_tree([K-V|KVs],T0,T) :-
    delete_from_radix_tree(K,V,_Vars,T0,T1),
    delete_list_from_radix_tree(KVs,T1,T).


dump_radix_tree(T) :-
    (valid_radix_tree(T)
     ->	(do_all
	 find_in_trie(K,V,T),
	 writeln((K:-V))
	)
     ;	misc_error('Cannot dump; Invalid radix tree')
    ).

valid_radix_tree([]).
valid_radix_tree(rtl(_,_,RTL)) :-
    valid_rtl(RTL).
valid_radix_tree(T) :- T = rt(RT1,RT2,RT3,RT4),
    raise_subtree(T,TT),
    T == TT,
    valid_radix_tree(RT1),
    valid_radix_tree(RT2),
    valid_radix_tree(RT3),
    valid_radix_tree(RT4).

valid_rtl([]).
valid_rtl(rtl(_,_,RTL)) :- valid_rtl(RTL).

delete_list_from_trie([],Trie,Trie).
delete_list_from_trie([H-B|List],Trie0,Trie) :-
    delete_from_trie_b(H,B,Trie0,Trie1),
    delete_list_from_trie(List,Trie1,Trie).

