/* File:      ora_call.P
** Author(s): Hasan Davulcu
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1993-1998
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: ora_call.P,v 1.12 2010-08-19 15:03:38 spyrosh Exp $
** 
*/


:- dynamic (attribute)/4.
:- dynamic '$inact_cursor_table$'/1.
%:- retractall('$inact_cursor_table$'(_)).
:- dynamic '$act_cursor_table$'/1.
%:- retractall('$act_cursor_table$'(_)).
:- dynamic '$select_stack$'/1.
%:- retractall('$select_stack$'(_)).
:- dynamic '$ora_err$'/1.
%:- retractall('$ora_err$'(_)).
:- dynamic table_arity/2.
:- dynamic '$show_query$'/1.
%:- retractall('$show_query$'(_)).
:- dynamic relation/3.

db_open(oracle(Name,Passwd)) :- 
	conset(count,1),init_gensym(table1),init_gensym(null),
	assert('$ora_err$'(off)),
	assert('$show_query$'(on)),
	assert('$act_cursor_table$'([])),
	assert('$inact_cursor_table$'([])),
	assert('$select_stack$'([])),
	oracle_connect(Name,Passwd,Status),
	(   Status =:= 0,!;
	    writeln('ERR - DB: Connection failed.'),fail).

db_close :- 
	retractall('$ora_err$'(_)), retractall('$show_query$'(_)), 
	( '$act_cursor_table$'(_) ->
	    retractall('$act_cursor_table$'(_)); true),
	( '$inact_cursor_table$'(_) ->
	    retractall('$inact_cursor_table$'(_)); true),
	( '$select_stack$'(_) ->retractall('$select_stack$'(_)); true),
	oracle_disconnect(_).

db_flag(fail_on_error,on,off) :-
	retract('$ora_err$'(on)), assert('$ora_err$'(off)).

db_flag(fail_on_error, off, on) :- 
	retract('$ora_err$'(off)), assert('$ora_err$'(on)).

db_flag(show_query, off, on) :- 
	retract('$show_query$'(off)), assert('$show_query$'(on)).

db_flag(show_query, on, off) :- 
	retract('$show_query$'(on)), assert('$show_query$'(off)).

:- mode db_link(?,+).
db_link(ProjectionTerm, DatabaseGoal) :- 
	translate(ProjectionTerm, DatabaseGoal,_, SQLAtom),!,
	conget(count, Num),
	Num1 is Num +1,
	conset(count, Num1),
	assert_call_rule_del(ProjectionTerm, SQLAtom,Num).

:- mode db_link_test(?,+).
db_link_test(ProjectionTerm, DatabaseGoal) :- 
	translate(ProjectionTerm, DatabaseGoal,_, SQLAtom),!,
	conget(count, Num),
	Num1 is Num +1,
	conset(count, Num1),
	assert_call_rule_test(ProjectionTerm, SQLAtom,Num).


db_query(Head,Body) :-
	copy_term(Head,Head1),
	Head =.. [H|A],
	assert((Head1 :- 
		  Head1 =..[_|Args] ',' 
		  get_mode(Args,Mode,A) ',' 
		  NewHead =.. [H| Mode] ','
		  db_link(NewHead,Body) ',' ! ','
		  Head1)).

db_query_test(Head,Body) :-
	copy_term(Head,Head1),
	Head =.. [H|A],
	assert((Head1 :- 
		  Head1 =..[_|Args] ',' 
		  get_mode(Args,Mode,A) ',' 
		  NewHead =.. [H| Mode] ','
		  db_link_test(NewHead,Body) ',' ! ','
		  Head1)).


db_SQLCA(Comm, Res) :-
	Comm == 'SQLCAID',
		db_sqlcaid(Res);
	Comm == 'SQLCABC',
		db_sqlcabc(Res);
	Comm == 'SQLCODE',
		db_sqlcode(Res);
	Comm == 'SQLERRML',
		db_sqlerrml(Res);
	Comm == 'SQLERRMC',
	 	db_sqlerrmc(Res);
	Comm == 'SQLERRP',
		db_sqlerrp(Res);
	Comm == 'SQLERRD'(0),
		db_sqlerrd0(Res);
	Comm == 'SQLERRD'(1),
		db_sqlerrd1(Res);
	Comm == 'SQLERRD'(2),
		db_sqlerrd2(Res);
	Comm == 'SQLERRD'(3),
		db_sqlerrd3(Res);
	Comm == 'SQLERRD'(4),
		db_sqlerrd4(Res);
	Comm == 'SQLERRD'(5),
		db_sqlerrd5(Res);
	Comm == 'SQLWARN'(0),
		db_sqlwarn0(Res);
	Comm == 'SQLWARN'(1),
		db_sqlwarn1(Res);
	Comm == 'SQLWARN'(3),
		db_sqlwarn3(Res);
	Comm == 'SQLWARN'(4),
		db_sqlwarn4(Res);
	Comm == 'SQLWARN'(5),
		db_sqlwarn5(Res);
	Comm == 'SQLEXT',
		db_sqlext(Res).
	
	

db_import(DbPredicate, SymName) :-
    DbPredicate =.. [TableName|Attributes],
    conset(col,1),
    assert_attributes_info(TableName),
             %asserts attribute(Position,Functor,Attribute,Type)
    conget(col,_N), 
    functor(DbPredicate, TableName, Arity),
    assert(relation(SymName,Arity,TableName)),
				 % asserts relation(Functor,Arity,Functor)
    assert_import_attributes(SymName, Attributes, TableName,1),
    functor(SymPred, SymName, Arity),
    db_query(SymPred, (SymPred)).

db_import_test(DbPredicate, SymName) :-
    DbPredicate =.. [TableName|Attributes],
    conset(col,1),
    assert_attributes_info(TableName),
             %asserts attribute(Position,Functor,Attribute,Type)
    conget(col,_N), 
    functor(DbPredicate, TableName, Arity),
    assert(relation(SymName,Arity,TableName)),
				 % asserts relation(Functor,Arity,Functor)
    assert_import_attributes(SymName, Attributes, TableName,1),
    functor(SymPred, SymName, Arity),
    db_query_test(SymPred, (SymPred)).



:- mode get_mode(?,?,?).
get_mode([HA],['NULL'(HR)],[HR]) :- nonvar(HA), HA = 'NULL'(X),var(X),!.
get_mode([HA],[bind(HR)],[HR]) :- nonvar(HA),
	HA == 'NULL',!,
	writeln('WAR - DB: ''NULL'' as a bind variable.'),!.
get_mode([HA],[bind(HR)],[HR]) :- nonvar(HA), HA = 'NULL'(X), nonvar(X),!.
get_mode([HA],[HR],[HR]) :-
	var(HA).
get_mode([HA],[bind(HR)],[HR]) :-
	nonvar(HA).

get_mode([HA|TA],[bind(HR)|TM],[HR|TR]) :-
	nonvar(HA),
	HA = 'NULL',!,
	writeln('WAR - DB: ''NULL'' as a bind variable.'),
	get_mode(TA,TM,TR).

get_mode([HA|TA],['NULL'(HR)|TM],[HR|TR]) :-
	nonvar(HA),
	HA = 'NULL'(X),var(X),!,
	get_mode(TA,TM,TR).

get_mode([HA|TA],[bind(HR)|TM],[HR|TR]) :-
	nonvar(HA),
	HA = 'NULL'(X),nonvar(X),!,
	get_mode(TA,TM,TR).
get_mode([HA|TA],[HR|TM],[HR|TR]) :-
	var(HA),!,
	get_mode(TA,TM,TR).
get_mode([HA|TA],[bind(HR)|TM],[HR|TR]):-
	nonvar(HA),
	get_mode(TA,TM,TR).

:- mode get_mode_ins(?,?,?).
get_mode_ins([HA],['NULL'(HR)],[HR]) :- nonvar(HA), HA = 'NULL'(X),var(X),!.
get_mode_ins([HA],[bind(HR)],[HR]) :- nonvar(HA),
	HA == 'NULL',!,
	writeln('WAR - DB: ''NULL'' as a bind variable.'),!.
get_mode_ins([HA],['NULL'(HR)],[HR]) :- nonvar(HA), HA = 'NULL'(X), nonvar(X),!.
get_mode_ins([HA],[HR],[HR]) :-
	var(HA).
get_mode_ins([HA],[bind(HR)],[HR]) :-
	nonvar(HA).

get_mode_ins([HA|TA],[bind(HR)|TM],[HR|TR]) :-
	nonvar(HA),
	HA = 'NULL',!,
	writeln('WAR - DB: ''NULL'' as a bind variable.'),
	get_mode_ins(TA,TM,TR).

get_mode_ins([HA|TA],['NULL'(HR)|TM],[HR|TR]) :-
	nonvar(HA),
	HA = 'NULL'(X),var(X),!,
	get_mode_ins(TA,TM,TR).

get_mode_ins([HA|TA],['NULL'(HR)|TM],[HR|TR]) :-
	nonvar(HA),
	HA = 'NULL'(X),nonvar(X),!,
	get_mode_ins(TA,TM,TR).
get_mode_ins([HA|TA],[HR|TM],[HR|TR]) :-
	var(HA),!,
	get_mode_ins(TA,TM,TR).
get_mode_ins([HA|TA],[bind(HR)|TM],[HR|TR]):-
	nonvar(HA),
	get_mode_ins(TA,TM,TR).


assert_import_attributes(_,[],_,_).
assert_import_attributes(SymName, [HAttr|RAttr], TableName,Num) :-
	attribute(_,TableName,HAttr,Type),
	assert(attribute(Num,SymName,HAttr,Type)),
	Num1 is Num + 1,
	assert_import_attributes(SymName,RAttr,TableName,Num1).

locate_cursor(StmtNo, SQLAtom, NumBindVars, SQLType, Index, RetVal) :-
   ( cursor_table(StmtNo, Index) ->
	(RetVal = 0, SQLType = 1);
	(allocate_cursor(StmtNo, SQLAtom, NumBindVars, SQLType, Index, RetVal),
	( SQLType =\= 0 ->
		add_cursor_table(StmtNo, Index)
	  ; push_select_table(Index)))
   ).

push_select_table(Index) :-
	retract('$select_stack$'(List)),
	NList = [Index|List],
	assert('$select_stack$'(NList)).

deallocate_cursor_stack(Index) :-
	retract('$select_stack$'(List)),
	deallocate_stack(Index,List,NList),
	assert('$select_stack$'(NList)).

deallocate_stack(Index,[Index|Rest],Rest) :- !,
	deallocate_cursor(Index, _RetVal).
deallocate_stack(Index,[Index1|Rest],Rest1) :- 
	Index =\= Index1,
	deallocate_cursor(Index1, _RetVal),
	deallocate_stack(Index,Rest,Rest1).

cursor_table(StmtNo, Index) :-
	'$act_cursor_table$'(List),
	search_cursors(List, StmtNo, Index).

inact_cursor_table(_StmtNo, Index) :-
	'$inact_cursor_table$'(List),
	member(Index,List).

add_cursor_table(StmtNo, Index) :-
	retract('$act_cursor_table$'(List)),
	append(List,[StmtNo '/' Index], NList),
	assert('$act_cursor_table$'(NList)),
	(inact_cursor_table(_, Index) ->
		true; ( retract('$inact_cursor_table$'(List1)),
			append(List1,[Index], NList1),
			assert('$inact_cursor_table$'(NList1))
			)
	).

search_cursors([StmtNo '/' Index|_], StmtNo, Index).
search_cursors([_|Rest], StmtNo, Index) :-
	search_cursors(Rest, StmtNo, Index).

/*
free :-
	flush,
	retract('$act_cursor_table$'(List)),
	deallocate_list(List),
	assert('$act_cursor_table$'([])).
 */
 
flush :- 
	'$act_cursor_table$'(List),
	flush_list(List).

close_inserts :-
	retract('$inact_cursor_table$'(List)),
	retract('$act_cursor_table$'(_List1)),
	close_list(List),
	assert('$act_cursor_table$'([])),
	assert('$inact_cursor_table$'([])).

close_list([]).
close_list([Index|Rest]):- 
	close_cursor(Index,_),
	close_list(Rest).
	
flush_list([]).
flush_list([_ '/' Index|Rest]):- 
	flush_cursor(Index,_),
	flush_list(Rest).

/*
deallocate_list([]).
deallocate_list([_ '/' Index|Rest]):-
	flush_cursor(Index,_),
	deallocate_cursor(Index,_),
	deallocate_list(Rest).
 */

db_all(Call, SQLAtom, ProjectionTerm, StmtNo) :-
	
	Call =.. [SymbolicName|CallArgs],
	ProjectionTerm =.. [_Head|ProjectionTermArgs],
	count_b_f(ProjectionTermArgs, NumBindVars, NumSLIVars),
        locate_cursor(StmtNo, SQLAtom, NumBindVars, SQLType, Index, RetVal),
	((RetVal =:= 0) ->
	(
	set_global_bind_table(Index,ProjectionTermArgs,CallArgs),
	execute_cursor(Index, NumSLIVars, RetVal1),
	(
	    ((RetVal1 =:= 0,SQLType =:= 0),
		( next(Row, Index),
	  	  skip_bind(Row,ProjectionTermArgs,Row1),
	 	  append([SymbolicName],Row1,CallList),
	 	  Call =.. CallList;
		deallocate_cursor_stack(Index), fail)
	);
	(
	   (RetVal1 =:= 0, SQLType =\= 0),
	   true 
	);

	 RetVal1 < 0,
	 writeln('ERR - DB: Parse error.'),deallocate_cursor(Index,_),
		('$ora_err$'(on) -> fail;true)
	));
		writeln('ERR - DB: No more cursors left.'),fail). 

:- mode db_all_test(?,+,?,+).
db_all_test(Call, SQLAtom, ProjectionTerm, StmtNo) :-
	
	Call =.. [SymbolicName|CallArgs],
	ProjectionTerm =.. [_Head|ProjectionTermArgs],
	count_b_f(ProjectionTermArgs, NumBindVars, NumSLIVars),
        allocate_cursor(StmtNo, SQLAtom, NumBindVars, SQLType, Index, RetVal),
	((RetVal =:= 0) ->
	(
	set_global_bind_table(Index,ProjectionTermArgs,CallArgs),
	execute_cursor(Index, NumSLIVars, RetVal1),
	(
	    ((RetVal1 =:= 0,SQLType =:= 0),
		( next(Row, Index) ->
	  	  	(skip_bind(Row,ProjectionTermArgs,Row1),
	 	  	append([SymbolicName],Row1,CallList),
	 	  	Call =.. CallList,
			deallocate_cursor(Index, _RetVal2));
			deallocate_cursor(Index, _RetVal2),fail)
	);
	(
	   (RetVal1 =:= 0, SQLType =\= 0),
	    writeln('ERR - DB: Non select test stmt.'),
	    deallocate_cursor(Index, _RetVal2), fail 
	);

	 RetVal1 < 0,
	 writeln('ERR - DB: Parse error.'),deallocate_cursor(Index,_),
		('$ora_err$'(on) -> fail;true)
	));
		writeln('ERR - DB: No more cursors left.'),fail). 


db_sql(Sql) :-
	execute_immediate(Sql,Status),
	(Status =:= 0 -> true; 
	  (writeln('ERR - DB: SQL stmy error.'),
	   ('$ora_err$'(on) -> fail;true) 
 	  )
	 ).

db_transaction(commit) :-
	flush, % so that it flushes arrays upon disconnnect
	close_inserts,
	execute_immediate('commit',Status),
	(Status =:= 0 -> true; 
	  (writeln('ERR - DB: SQL stmy error.'),
	   ('$ora_err$'(on) -> fail;true) 
 	  )
	 ).

db_transaction(rollback) :-
	close_inserts,
	execute_immediate('rollback work',Status),
	(Status =:= 0 -> true; 
	  (writeln('ERR - DB: SQL stmy error.'),
	   ('$ora_err$'(on) -> fail;true) 
 	  )
	 ).



db_sql_select(Select,Row) :-
	conget(count, Num),
	Num1 is  Num + 1,
	conset(count, Num1),
	allocate_cursor(Num, Select, 0, SQLType, Index, RetVal),
	( (RetVal =:= 0) ->
	(
	execute_cursor(Index, 10, RetVal1),
	(
	    ((RetVal1 =:= 0,SQLType =:= 0),
		( next(Row, Index);
		close_cursor(Index, _RetVal2),fail) 
	);
	(
	   (RetVal1 =:= 0, SQLType =\= 1),
	   fail 
	);

	 RetVal1 < 0,
	 writeln('ERR - DB: Parse error.'),deallocate_cursor(Index,_),
		('$ora_err$'(on) -> fail;true)
	));
		writeln('ERR - DB: No more cursors left.'),fail).

:- mode db_link_del(?,+).
db_link_del(ProjectionTerm, DatabaseGoal) :- 
	translate_del(ProjectionTerm, DatabaseGoal,_, SQLAtom),!,
	conget(count, Num),
	Num1 is Num +1,
	conset(count, Num1),
	assert_call_rule_del(ProjectionTerm, SQLAtom,Num).


db_delete(Head, Body) :-
	copy_term(Head,Head1),
	Head =.. [H|A],
	assert((Head1 :- 
		  Head1 =..[_|Args] ',' 
		  get_mode(Args,Mode,A) ',' 
		  NewHead =.. [H| Mode] ','
		  db_link_del(NewHead,Body) ',' ! ','
		  Head1)).

:- mode db_link_ins(?,+).
db_link_ins(ProjectionTerm, DatabaseGoal) :- 
	translate_ins(ProjectionTerm, DatabaseGoal,_, SQLAtom),!,
	conget(count, Num),
	Num1 is Num +1,
	conset(count, Num1),
	assert_call_rule(ProjectionTerm, SQLAtom,Num).

	
db_insert(Head, Body) :-
	copy_term(Head,Head1),
	Head =.. [H|A],
	assert((Head1 :- 
		  Head1 =..[_|Args] ',' 
		  get_mode_ins(Args,Mode,A) ',' 
		  NewHead =.. [H| Mode] ','
		  db_link_ins(NewHead,Body) ',' ! ','
		  Head1)).


/*db_SQL_DDL(SQL,Name) :-
	conget(count, Num),
	Num1 is Num + 1,
	conset(count,Num1),
	assert_SQL_call_rule(SQL,Num,Name).

db_SQL(SQL,StmtNo) :-
	find_free_cursor(StmtNo, Index, Parse),
	(Index =\= -1 ->
	(
	parse(SQL, RetVal,Index, Parse),
	RetVal =:= 0,
	set_cursor_close(Index));
	writeln('ERR - DB: No more cursors left.'),fail).


assert_SQL_call_rule(SQL,Num,Name) :-
	assert((Name :- db_SQL(SQL,Num))),listing.

*/ % Do we realy need this ?????

writelist([H]) :- writeln(H),!.
writelist([H|T]) :-
	write(H),write(' '), writelist(T).

writelistln([H]) :- writeln(H),!.
writelistln([H|T]) :-
	writeln(H), writelistln(T).

         
% -----------------------------------------------------------------------
db_show_schema(tuples(Table)) :-
	check_atom(Table,'db_show_schema/1)','1.1'),
	name(Table, Tlist),
	append("select * from ", Tlist, L),
	name(Sql, L),
	db_sql_select(Sql, Row),
	writelist(Row), fail.
db_show_schema(tuples(_Table)).
%--------------------------------
db_get_schema(user, List) :-
	bagof(A, user_tables(A), List).
db_show_schema(user) :-
	db_get_schema(user, List),
	writelistln(List).


user_tables(A) :-
    var(A),
	conget(count, Num),
	Num1 is Num +1,
	conset(count, Num1),

    db_all(user_tables(A),'SELECT rel1.TABLE_NAME  FROM USER_TABLES rel1' ,user_tables('$var$'(var1)),Num).

%---------------------------
db_get_schema(accessible, List) :-
	bagof(A, accessible_tables(A), List).
db_show_schema(accessible) :-
	db_get_schema(accessible, List),
	writelistln(List).


accessible_tables(A) :-
    var(A),
	conget(count, Num),
	Num1 is Num +1,
	conset(count, Num1),

    db_all(accessible_tables(A),'SELECT rel1.TABLE_NAME  FROM ALL_TABLES rel1' ,accessible_tables('$var$'(var1)),Num).

%----------------------------
db_get_schema(arity(Table),N1) :-
	conset(col,1),
	assert_attributes_info(Table),
	table_arity(Table, N1).
db_show_schema(arity(Table)) :-
	db_get_schema(arity(Table), N1),
	write('Table '), write( Table), write(' has arity '), writeln(N1).
%------------------------------

db_get_schema(columns(Table),List) :-
	conset(col,1),
	assert_attributes_info(Table),
	bagof(Col, N1^Type1^attribute(N1,Table,Col,Type1), List).
db_show_schema(columns(Table)) :-
	db_get_schema(columns(Table), List), writelistln(List).
%--------------------------------
db_attach(Pname, table(Table)) :-
	conset(col, 1),	
	assert_attributes_info(Table),
	bagof(Col, N1^Type1^attribute(N1,Table,Col,Type1), List),
	DBPred =.. [Table|List],
	db_import(DBPred, Pname).
		
%-------------------------------
db_attach(Pname, table(Table, List)) :-
	conset(col, 1),	
	assert_attributes_info(Table),
	DBPred =.. [Table|List],
	db_import(DBPred, Pname).
%--------------------------------
db_record(Table, List) :-
	gensym(table1,Name),
	db_attach(Name,table(Table)),
	db_get_schema(arity(Table),N1),
	
	(var(List) ->
	 functor(B,Name,N1);
	 B =.. [Name|List]),!,
	 B,
	 B =.. [Name|List].
%--------------------------------
db_create_table(Table,Fields) :-	
    check_atom(Table,'db_create_table/2',1),
    name(Table,Tl),
    check_atom(Fields,'db_create_table/2',2),
    name(Fields,Fl),
    append("CREATE TABLE ",Tl,L1),
    append(L1," (",L2),
    append(L2,Fl,L3),
    append(L3," )",Res), name(Query,Res),
    db_sql(Query).
%--------------------------------
db_create_index(Tname,Iname,index(_,Fields)) :-
	check_atom(Tname,'db_create_index/3',1),
	check_atom(Iname,'db_create_index/3',2),
	check_atom(Fields,'db_create_index/3','3.2'),
	name(Tname,Tlist),
	name(Iname,Ilist),
	name(Fields,Flist),
	append([67,82,69,65,84,69,32,73,78,68,69,88,32],Ilist,L1),
	append(L1,[32,79,78,32],L2),
	append(L2,Tlist,L3),
	append(L3, [32,40],L4),
	append(L4,Flist,L5),
	append(L5,[32,41],Res), name(Query,Res),
	db_sql(Query).
%--------------------------------
db_delete_table(Table) :-
	check_atom(Table,'db_delete_table/1',1),
	name(Table, Tlist),
	append([68,82,79,80,32,84,65,66,76,69,32],Tlist,Res),name(Query,Res),
	db_sql(Query).

%--------------------------------
db_delete_view(View) :-
	check_atom(View,'db_delete_view/1',1),
	name(View, Vlist),
	append([68,82,79,80,32,86,73,69,87,32],Vlist,Res),name(Query,Res),
	db_sql(Query).
%--------------------------------
db_delete_index(Index) :-
	check_atomic(Index,'db_delete_index/1',1),
	name(Index,Ilist),
	append([68,82,79,80,32,73,78,68,69,88,32],Ilist,Res),name(Query,Res),
	db_sql(Query).
%--------------------------------
db_add_record(Table,VList) :-
	check_atom(Table,'db_add_record/2',1),
	conset(col,1),
	assert_attributes_info(Table),
	conget(col,N), N1 is N - 1,
	name(Table, Tl), append(Tl,[95,105,110,115],Tl1), name(Table1,Tl1),
	assert(relation(Table,N1,Table)),
	functor(Head1, Table1, N1),
	Head1 =..[Table1|Tail],
	Head =..[Table|Tail],
	db_insert(Head1, (Head)),
	asserta((db_add_record(Table,Values) :- Query =..[Table1|Values] ',' ! ',' Query)), 
	Query =.. [Table1|VList],!,
	Query.
%--------------------------------
delete_record(Table,VList) :-
	check_atom(Table,'delete_record/2',1),
	conset(col,1),
	assert_attributes_info(Table),
	conget(col,N), N1 is N - 1,
	name(Table, Tl), append(Tl,[95,100,101,108],Tl1), name(Table1,Tl1),
	assert(relation(Table,N1,Table)),
	functor(Head1, Table1, N1),
	Head1 =..[Table1|Tail],
	Head =..[Table|Tail],
	db_delete(Head1, (Head)),
	asserta((delete_record(Table,Values) :- Query =..[Table1|Values] ',' ! ',' Query)), 
	Query =.. [Table1|VList],!,
	Query.
%--------------------------------
db_error_message(_DBSys,Num,Msg) :-
	
	db_SQLCA('SQLCODE',Num),
	db_SQLCA('SQLERRMC',Msg).



% -----------------------------------------------------------------------
next(Row, Index) :-
        fetch_from_cursor(Index,RetVal),
	(RetVal =:= 0 ->
	(get_list(X1,Index), 
           Row=X1;
	 next(Row, Index));
	 (RetVal < 0) ->
	 (writeln('ERR - DB: FETCH failed.'),fail)).


get_list([X|Y], Index) :- 
	get_output_value(Index,X,RetVal),
		          RetVal=:=0,
 			  get_list(Y, Index),!.
get_list([], _Index).		    		 


skip_head(_R,'$bind$'(_),_).
skip_head('NULL','$var$'(_),'NULL'(X)) :- !, gensym(null, X).
skip_head(R,'$var$'(_),R).
skip_head(_R,'$const$'('NULL'), 'NULL'(X)) :- !, gensym(null, X).
skip_head(_R,'$const$'(_),_).



skip_bind([],[],[]).
skip_bind([R|R1],[P|P1],[H|H1]) :-
	skip_head(R,P,H),
	skip_bind(R1,P1,H1).
/* cursor table */

/*get_next(N) :-
	not '$curs$'(N),!,
	assert('$curs$'(N)),
	_Index=N.
'$curs$'(10000).*/


	

/* set_bind_table */

set_global_bind_table(_Index,[],[]).
set_global_bind_table(Index, [H|T], [H1|T1]) :-
	H = '$bind$'(BindId),!,
	(integer(H1), 
	set_input_value(Index,BindId,H1,1,Status);
        atom(H1), 
	set_input_value(Index,BindId,H1,0,Status);
	H1 = 'NULL'(_),
	set_input_value(Index,BindId,'NULL',0,Status)),
	set_global_bind_table(Index,T,T1).
set_global_bind_table(Index, [_H|T], [_H1|T1]) :-
	set_global_bind_table(Index,T,T1).



%------------------------------------------------------------------------------
% Modifications : Hasan Davulcu
% Date : 19 February, 1994
% 
% The copiler has been extended to generate queries with bind variables.
% assert_call_rule has been added to provide an interface for database calls.
%
%------------------------------------------------------------------------------



% -----------------------------------------------------------------------------
%
% This Prolog to SQL compiler may be distributed free of charge provided that it is
% not used in commercial applications without written consent of the author, and
% that the copyright notice remains unchanged.
%
%                    (C) Copyright by Christoph Draxler, Munich
%                        Version 1.1 of Dec. 21st 1992
%
% I would like to keep in my hands the further development and distribution of the
% compiler. This does not mean that I don't want other people to suggest or even
% implement improvements - quite on the contrary: I greatly appreciate contributions 
% and if they make sense to me I will incorporate them into the compiler (with due
% credits given!). 
% 
% For further development of the compiler, address your requests, comments and
% criticism to the author:
%
%                    Christoph Draxler
%                    CIS Centre for Information and Speech Processing
%                    Ludwig-Maximilians-University Munich
%                    Wagmuellerstr. 23 
%                    D 80538 Munich
%                    Tel : ++49 / +89 / 211 06 64 (-60)
%                    Fax : ++49 / +89 / 211 06 74
%                    Mail: draxler@cis.uni-muenchen.de
%
%
% A report describing the implementation is available upon request from the
% author. 
%
%
% RELEASE INFORMATION
% ===================
% Current version is v. 1.1 of Dec. 21st 1992.
% Version 1.0 Sept. 3 1992
% -----------------------------------------------------------------------------
 




% --------------------------------------------------------------------------------------
%
% Top level predicate translate/3 organizes the compilation and constructs a
% Prolog term representation of the SQL query. 
%
% --------------------------------------------------------------------------------------


translate(ProjectionTerm,DatabaseGoal,SQLQueryTerm,SQLAtom):-
   % --- initialize variable identifiers and range variables for relations -----
   init_gensym(var),
   init_gensym(rel),
   init_gensym('BIND'),

   % --- tokenize projection term and database goal ----------------------------
      tokenize_term(ProjectionTerm,TokenProjectionTerm),
      tokenize_term(DatabaseGoal,TokenDatabaseGoal),
   % --- lexical analysis: reordering of goals for disjunctive normalized form -
   disjunction(TokenDatabaseGoal,Disjunction),

   % --- code generation ---------------------------------------------------------------
   query_generation(Disjunction,TokenProjectionTerm,SQLQueryTerm),
   ('$show_query$'(on) -> printqueries(SQLQueryTerm);true),
  queries_atom(SQLQueryTerm,SQLAtom).

translate_del(DeleteTerm, DatabaseGoal,SQLQueryTerm,SQLAtom):-
   % --- initialize variable identifiers and range variables for relations -----
   init_gensym(var),
   init_gensym(rel),
   init_gensym('BIND'),

   % --- tokenize projection term and database goal ----------------------------
      tokenize_term(DeleteTerm,TokenDeleteTerm),
      tokenize_term(DatabaseGoal,TokenDatabaseGoal),

   % --- lexical analysis: reordering of goals for disjunctive normalized form -
   disjunction(TokenDatabaseGoal,Disjunction),

   % --- code generation ---------------------------------------------------------------
   query_generation_del(Disjunction,TokenDeleteTerm,SQLQueryTerm),
   /*printqueries(SQLQueryTerm),*/
  queries_atom(SQLQueryTerm,SQLAtom).

translate_ins(InsertTerm, DatabaseGoal,SQLQueryTerm,SQLAtom):-
   % --- initialize variable identifiers and range variables for relations -----
   init_gensym(var),
   init_gensym(rel),
   init_gensym('BIND'),

   % --- tokenize projection term and database goal ----------------------------
      tokenize_term(InsertTerm,TokenInsertTerm),
      tokenize_term(DatabaseGoal,TokenDatabaseGoal),
      
   % --- lexical analysis: reordering of goals for disjunctive normalized form -
   disjunction(TokenDatabaseGoal,Disjunction),

   % --- code generation ---------------------------------------------------------------
   query_generation_ins(Disjunction,TokenInsertTerm,SQLQueryTerm),
   /*printqueries(SQLQueryTerm),*/
  queries_atom(SQLQueryTerm,SQLAtom).



% --- disjunction(Goal,Disjunction) ----------------------------------------------------
%
% turns original goal into disjunctive normalized form by computing all conjunctions
% and collecting them in a list
%
% --------------------------------------------------------------------------------------

disjunction(Goal,Disjunction):-
   findall(Conjunction,linearize(Goal,Conjunction),Disjunction).




% --- linearize(Goal,ConjunctionList) --------------------------------------------------
%
% Returns a conjunction of base goals for a complex disjunctive or conjunctive goal
% Yields several solutions upon backtracking for disjunctive goals
%
% --------------------------------------------------------------------------------------


linearize(((A,B),C),(LinA,(LinB,LinC))):-
   % --- transform left-linear to right-linear conjunction (',' is associative) ----
   linearize(A,LinA),
   linearize(B,LinB),
   linearize(C,LinC).

linearize((A,B),(LinA,LinB)):-
   A \= (_,_),
   % --- make sure A is not a conjunction ------------------------------------------
   linearize(A,LinA),
   linearize(B,LinB).

linearize((A;_B),LinA):-
   linearize(A,LinA).

linearize((_A;B),LinB):-
   linearize(B,LinB).

linearize(not A, not LinA):-
   linearize(A,LinA).

linearize(Var^A, Var^LinA):-
   linearize(A,LinA).

linearize(A,A):-
   A \= (_,_),
   A \= (_;_),
   A \= _^_,
   A \= not(_).




% --- tokenize_term(Term,TokenizedTerm) -------------------------------------------------
%
% If Term is a 
%
%  - variable, then this variable is instantiated with a unique identifier 
%    of the form '$var$'(VarId), and TokenizedTerm is bound to the same 
%    term '$var$'(VarId). 
%
%  - constant, then TokenizedTerm is bound to '$const$'(Term).
%
%  - complex term, then the term is decomposed, its arguments are tokenized,
%    and TokenizedTerm is bound to the result of the composition of the original
%    functor and the tokenized arguments.
%
% --------------------------------------------------------------------------------------


tokenize_term('$var$'(VarId),'$var$'(VarId)):-
   var(VarId),
   % --- uninstantiated variable: instantiate it with unique identifier.
   gensym(var,VarId).

tokenize_term('$var$'(VarId),'$var$'(VarId)):-
   nonvar(VarId).

tokenize_term(bind(BindId),'$bind$'(BindId)):-
   var(BindId),
   gensym('BIND', BindId).

tokenize_term('$bind$'(BindId),'$bind$'(BindId)):-
   nonvar(BindId).

tokenize_term('NULL'(HR),'$const$'('NULL')):-
  HR = '$const$'('NULL').
tokenize_term('$const$'('NULL'),'$const$'('NULL')).

tokenize_term(bind(BindId),'$bind$'(BindId)):-
   nonvar(BindId).

tokenize_term(A,'$bind$'(A)) :- 
    atom(A),
    name(A,[66,73,78,68|_]).


tokenize_term(Constant,'$const$'(Constant)):-
   nonvar(Constant),
   functor(Constant,_,0).

tokenize_term(Term,TokenizedTerm):-
   nonvar(Term),
   Term \= '$var$'(_),
   Term \= '$const$'(_),
   Term \= '$bind$'(_),
   Term =.. [Functor|Arguments],
   Arguments \= [],
   tokenize_arguments(Arguments,TokenArguments),
   TokenizedTerm =.. [Functor|TokenArguments].



% --- tokenize_arguments(Arguments,TokenizedArguments) ---------------------------------
%
% organizes tokenization of arguments by traversing list and calling tokenize_term
% for each element of the list.
%
% --------------------------------------------------------------------------------------


tokenize_arguments([],[]).

tokenize_arguments([FirstArg|RestArgs],[TokFirstArg|TokRestArgs]):-
   tokenize_term(FirstArg,TokFirstArg),
   tokenize_arguments(RestArgs,TokRestArgs).







% --- query_generation(ListOfConjunctions, ProjectionTerm, ListOfQueries) -------------- 
%
% For each Conjunction translate the pair (ProjectionTerm,Conjunction) to an SQL query
% and connect each such query through a UNION-operator to result in the ListOfQueries.
%
% A Conjunction consists of positive or negative subgoals. Each subgoal is translated 
% as follows:
%  - the functor of a goal that is not a comparison operation is translated to
%    a relation name with a range variable
%  - negated goals are translated to NOT EXISTS-subqueries with * projection
%  - comparison operations are translated to comparison operations in the WHERE-clause
%  - aggregate function terms are translated to aggregate function (sub)queries
% 
% The arguments of a goal are translated as follows:
%  - variables of a goal are translated to qualified attributes
%  - variables occurring in several goals are translated to equality comparisons
%    (equi join) in the WHERE-clause
%  - constant arguments are translated to equality comparisons in the WHERE-clause
% 
% Special treatment of arithmetic functions:
%  - arithmetic functions are identified through the Prolog is/2 operator
%  - an arithmetic function may contain an unbound variable only on its left side
%  - the right side of the is/2 operator may consist of 
%    * bound variables (bound through occurrence within a positive database goal, or 
%      bound through preceeding arithmetic function), or of 
%    * constants (numbers, i.e. integers, reals)
% 
% The following RESTRICTION holds:
%
%  - the binding of variables follows Prolog: variables are bound by positive base goals
%    and on the left side of the is/2 predicate - comparison operations, negated goals
%    and right sides of the is/2 predicate do not return variable bindings and may even 
%    require all arguments to be bound for a safe evaluation.
%
% --------------------------------------------------------------------------------------

query_generation([],_,[]).

query_generation([Conjunction|Conjunctions],ProjectionTerm,[Query|Queries]):-
   projection_term_variables(ProjectionTerm,InitDict),
   translate_conjunction(Conjunction,SQLFrom,SQLWhere,InitDict,Dict),
   translate_projection(ProjectionTerm,Dict,SQLSelect),
   Query = query(SQLSelect,SQLFrom,SQLWhere),
   query_generation(Conjunctions,ProjectionTerm,Queries).

query_generation_del([],_,[]).

query_generation_del([Conjunction|Conjunctions],ProjectionTerm,[Query|Queries]):-
   projection_term_variables(ProjectionTerm,InitDict),
   translate_conjunction(Conjunction,SQLFrom,SQLWhere,InitDict,Dict),
   translate_projection(ProjectionTerm,Dict,_SQLSelect),
   Query = del_query(SQLFrom,SQLWhere),
   query_generation(Conjunctions,ProjectionTerm,Queries).

query_generation_ins([],_,[]).

query_generation_ins([Conjunction|Conjunctions],ProjectionTerm,[Query|Queries]):-
   projection_term_variables(ProjectionTerm,InitDict),
   translate_conjunction(Conjunction,SQLFrom,SQLWhere,InitDict,Dict),
   translate_projection(ProjectionTerm,Dict,_SQLSelect),
   Query = ins_query(SQLFrom,SQLWhere),
   query_generation(Conjunctions,ProjectionTerm,Queries).

% --- translate_goal(Goal,SQLFrom,SQLWhere,Dict,NewDict) -------------------------------
%
% translates a
%
%   - positive database goal to the associated FROM- and WHERE clause of an SQL query
%   - a negated goal to a negated existential subquery
%   - an arithmetic goal to an arithmetic expression or an aggregate function query
%   - a comparison goal to a comparison expression
%   - a negated comparison goal to a comparison expression with the opposite comparison
%     operator
%
% --------------------------------------------------------------------------------------

attr(A,B,C) :-
    nonvar(A),
    var(B),
    var(C),
	conget(count, Num),
	Num1 is Num +1,
	conset(count, Num1),

    db_all(attr(A,B,C),'SELECT rel1.TABLE_NAME , rel1.COLUMN_NAME , rel1.DATA_TYPE FROM USER_TAB_COLUMNS rel1 WHERE rel1.TABLE_NAME = :BIND1' ,attr('$bind$'('BIND1'),'$var$'(var1),'$var$'(var2)),Num).



translate_goal(SimpleGoal,[SQLFrom],SQLWhere,Dict,NewDict):-
   % --- positive goal binds variables - these bindings are held in the dictionary -----
   
   functor(SimpleGoal,Functor,Arity),
   translate_functor(Functor,Arity,SQLFrom),
   SimpleGoal =.. [Functor|Arguments],
   translate_arguments(Functor,Arguments,SQLFrom,1,SQLWhere,Dict,NewDict).


assert_attributes_info(Table) :- 
	not attribute(_,Table,_,_),
	attr(Table,Col,Type),
	get_type(Type,Type1),
	conget(col,N),
	N1 is N + 1,
	conset(col,N1),
	copy_term(attribute(N,Table,Col,Type1),Term),
	assert(Term),fail.
assert_attributes_info(Table) :- conget(col, N), N1 is N-1, N1 =\= 0,!,
			         assert(table_arity(Table, N1)). 
assert_attributes_info(_Table).




get_type(Type,integer) :-
	atomic(Type),
	name(Type,[78,85,77,66,69,82|_]).
get_type(Type,string) :- atomic(Type), not name(Type,[78,85,77,66,69,82|_]).


translate_goal(Result is Expression,[],SQLWhere,Dict,NewDict):-
   translate_arithmetic_function(Result,Expression,SQLWhere,Dict,NewDict).

translate_goal(not NegatedGoals,[],SQLNegatedSubquery,Dict,Dict):-
   % --- negated goals do not bind variables - hence Dict is returned unchanged --------

   functor(NegatedGoals,Functor,_),
   not comparison(Functor,_),
   translate_conjunction(NegatedGoals,SQLFrom,SQLWhere,Dict,_),
   SQLNegatedSubquery = [negated_existential_subquery([*],SQLFrom,SQLWhere)].

translate_goal(not ComparisonGoal,[],SQLCompOp,Dict,Dict):-
   % --- comparison operations do not bind variables - Dict is returned unchanged ------
   ComparisonGoal =.. [ComparisonOperator,LeftArg,RightArg],
   comparison(ComparisonOperator,SQLOperator),
   negated_comparison(SQLOperator,SQLNegOperator),
   translate_comparison(LeftArg,RightArg,SQLNegOperator,Dict,SQLCompOp).

translate_goal(ComparisonGoal,[],SQLCompOp,Dict,Dict):-
   % --- comparison operations do not bind variables - Dict is returned unchanged ------
   ComparisonGoal =.. [ComparisonOperator,LeftArg,RightArg],
   comparison(ComparisonOperator,SQLOperator),
   translate_comparison(LeftArg,RightArg,SQLOperator,Dict,SQLCompOp).




% --- translate_conjunction(Conjunction,SQLFrom,SQLWhere,Dict,NewDict) -----------------
% 
% translates a conjunction of goals (represented as a list of goals preceeded by 
% existentially quantified variables) to FROM- and WHERE-clause of an SQL query.  
% A dictionary containing the associated SQL table and attribute names is built up
% as an accumulator pair (arguments Dict and NewDict)
%
% --------------------------------------------------------------------------------------

translate_conjunction('$var$'(VarId)^Goal,SQLFrom,SQLWhere,Dict,NewDict):-
   % --- add info on existentially quantified variables to dictionary here -------------
   add_to_dictionary(VarId,_,_,_,existential,Dict,TmpDict),
   translate_conjunction(Goal,SQLFrom,SQLWhere,TmpDict,NewDict).

translate_conjunction(Goal,SQLFrom,SQLWhere,Dict,NewDict):-
   Goal \= (_,_),
   translate_goal(Goal,SQLFrom,SQLWhere,Dict,NewDict).

translate_conjunction((Goal,Conjunction),SQLFrom,SQLWhere,Dict,NewDict):-
   translate_goal(Goal,FromBegin,WhereBegin,Dict,TmpDict),
   translate_conjunction(Conjunction,FromRest,WhereRest,TmpDict,NewDict),
   append(FromBegin,FromRest,SQLFrom),
   append(WhereBegin,WhereRest,SQLWhere).




% --- translate_arithmetic_function(Result,Expression,SQLWhere,Dict,NewDict) -----------
%
% Arithmetic functions (left side of is/2 operator is bound to value of expression on
% right side) may be called with either
%
% - Result unbound: then Result is bound to the value of the evaluation of Expression
% - Result bound: then an equality condition is returned between the value of Result
%   and the value of the evaluation of Expression.
%
% Only the equality test shows up in the WHERE clause of an SQLquery.
%
% --------------------------------------------------------------------------------------

translate_arithmetic_function('$var$'(VarId),Expression,[],Dict,NewDict):-
   % assigment of value of arithmetic expression to variable - does not
   % show up in WHERE-part, but expression corresponding to
   % variable must be stored in Dict for projection translation

   evaluable_expression(Expression,Dict,ArithExpression,Type),
   add_to_dictionary(VarId,is,ArithExpression,Type,all,Dict,NewDict).


translate_arithmetic_function('$var$'(VarId),Expression,ArithComparison,Dict,Dict):-
   % --- test whether left side evaluates to right side: return equality comparison ----
   % Left side consists of qualified attribute, i.e. range variable must not be
   % arithmetic operator is/2 

   lookup(VarId,Dict,PrevRangeVar,PrevAtt,PrevType),
   not (PrevRangeVar = is),

   % test whether type of attribute is numeric - if not, there's no sense in 
   % continuing the translation

   type_compatible(PrevType,number),
   evaluable_expression(Expression,Dict,ArithExpression,ExprType),
   type_compatible(ExprType,number),
   ArithComparison = [comp(att(PrevRangeVar,PrevAtt),'=',ArithExpression)].


translate_arithmetic_function('$var$'(VarId),Expression,ArithComparison,Dict,Dict):-
   % --- test whether left side evaluates to right side: return equality comparison ----
   % Left side consists of arithmetic expression, i.e. VarId is stored in Dict as 
   % belonging to arithmetic expression which is expressed as RangeVar-argument 
   % of lookup returning is/2. Type information is implicit through the is/2 functor

   lookup(VarId,Dict,is,LeftExpr,Type),
   type_compatible(Type,number),
   evaluable_expression(Expression,Dict,RightExpr,ExprType),
   type_compatible(ExprType,number),
   ArithComparison = [comp(LeftExpr,'=',RightExpr)].


translate_arithmetic_function('$const$'(Constant),Expression,ArithComparison,Dict,Dict):-
   % --- is/2 used to test whether left side evaluates to right side -------------------
   get_type('$const$'(Constant),ConstantType),
   type_compatible(ConstantType,number),
   evaluable_expression(Expression,Dict,ArithExpression,ExprType),
   type_compatible(ExprType,number),
   ArithComparison = [comp('$const$'(Constant),'=',ArithExpression)].

translate_arithmetic_function('$bind$'(BindId),Expression,ArithComparison,Dict,Dict):-
   % --- is/2 used to test whether left side evaluates to right side -------------------
   lookup(BindId,Dict,_,_,Type),
   type_compatible(Type,number),
   evaluable_expression(Expression,Dict,ArithExpression,ExprType),
   type_compatible(ExprType,number),
   ArithComparison = [comp('$bind$'(BindId),'=',ArithExpression)].


% --- translate_comparison(LeftArg,RightArg,CompOp,Dict,SQLComparison) ---------
%
% translates the left and right arguments of a comparison term into the
% appropriate comparison operation in SQL. The result type of each 
% argument expression is checked for type compatibility
%
% ------------------------------------------------------------------------------

translate_comparison(LeftArg,RightArg,CompOp,Dict,Comparison):-
   evaluable_expression(LeftArg,Dict,LeftTerm,LeftArgType),
   evaluable_expression(RightArg,Dict,RightTerm,RightArgType),
   type_compatible(LeftArgType,RightArgType),
   Comparison = [comp(LeftTerm,CompOp,RightTerm)].







% --- translate_functor(Functor,QualifiedTableName) ------------------------------------
%
% translate_functor searches for the matching relation table name for
% a given functor and creates a unique range variable to result in
% a unique qualified relation table name.
%
% --------------------------------------------------------------------------------------


translate_functor(Functor,Arity,rel(TableName,RangeVariable)):-
   relation(Functor,Arity,TableName),
   gensym(rel,RangeVariable).




% --- translate_arguments(Arguments,RelTable,ArgPos,Conditions,Dict) -------------------
%
% translate_arguments organizes the translation of term arguments. One
% term argument after the other is taken from the list of term arguments
% until the list is exhausted. 
%
% --------------------------------------------------------------------------------------

translate_arguments(_,[],_,_,[],Dict,Dict).

translate_arguments(Functor,[Arg|Args],SQLTable,Position,SQLWhere,Dict,NewDict):-
   translate_argument(Functor,Arg,SQLTable,Position,Where,Dict,TmpDict),
   NewPosition is Position + 1,
   translate_arguments(Functor,Args,SQLTable,NewPosition,RestWhere,TmpDict,NewDict),
   append(Where,RestWhere,SQLWhere).




% --- translate_argument(Argument,RelTable,Position,Condition,Dict) --------------------
%
% The first occurrence of a variable leads to its associated SQL attribute information
% to be recorded in the Dict. Any further occurrence creates an equi-join condition 
% between the current attribute and the previously recorded attribute.
% Constant arguments always translate to equality comparisons between an attribute and 
% the constant value.
%
% --------------------------------------------------------------------------------------

translate_argument(Functor,'$bind$'(BindId),rel(_SQLTable,RangeVar),Position,Bind_Cond,Dict,NewDict):-
   attribute(Position,Functor,Attribute,Type),
   add_to_dictionary(BindId,RangeVar,Attribute,Type,all,Dict,NewDict),
   Bind_Cond = [comp(att(RangeVar,Attribute),=,'$bind$'(BindId))].

translate_argument(Functor,'$bind$'(BindId),rel(_SQLTable,RangeVar),Position,AttComparison,Dict,Dict):-
   % --- Variable occurred previously - retrieve first occurrence data from dictionary -
   lookup(BindId,Dict,PrevRangeVar,PrevAtt,PrevType),
   attribute(Position,Functor,Attribute,Type),
   type_compatible(PrevType,Type),
   AttComparison = [comp(att(RangeVar,Attribute),=,att(PrevRangeVar,PrevAtt))].

translate_argument(Functor,'$var$'(VarId),rel(_SQLTable,RangeVar),Position,[],Dict,NewDict):-
   attribute(Position,Functor,Attribute,Type),
   add_to_dictionary(VarId,RangeVar,Attribute,Type,all,Dict,NewDict).

translate_argument(Functor,'$var$'(VarId),rel(_SQLTable,RangeVar),Position,AttComparison,Dict,Dict):-
   % --- Variable occurred previously - retrieve first occurrence data from dictionary -
   lookup(VarId,Dict,PrevRangeVar,PrevAtt,PrevType),
   attribute(Position,Functor,Attribute,Type),
   type_compatible(PrevType,Type),
   AttComparison = [comp(att(RangeVar,Attribute),=,att(PrevRangeVar,PrevAtt))].

translate_argument(Functor,'$const$'('NULL'),rel(_SQLTable,RangeVar),Position,ConstComparison,Dict,Dict):- !,
   % --- Equality comparison of constant value and attribute in table ------------------
   attribute(Position,Functor,Attribute,_Type),
   get_type('$const$'('NULL'),_ConstType),
   ConstComparison = [comp(att(RangeVar,Attribute),'IS','$const$'('NULL'))].

translate_argument(Functor,'$const$'(Constant),rel(_SQLTable,RangeVar),Position,ConstComparison,Dict,Dict):-
   % --- Equality comparison of constant value and attribute in table ------------------
   attribute(Position,Functor,Attribute,Type),
   get_type('$const$'(Constant),ConstType),
   type_compatible(ConstType,Type),
   ConstComparison = [comp(att(RangeVar,Attribute),=,'$const$'(Constant))].




% --- projection_term_variables(ProjectionTerm,Dict) -----------------------------------
%
% extracts all variables from the ProjectionTerm and places them into the
% Dict as a dict/4 term with their Identifier, a non instantiated RangeVar and 
% Attribute argument, and the keyword existential for the type of quantification
%
% --------------------------------------------------------------------------------------

projection_term_variables('$const(_)$',[]).

projection_term_variables('$var$'(VarId),[dict(VarId,_,_,_,existential)]).

projection_term_variables('$bind$'(BindId),[dict(BindId,_,_,_,existential)]).

projection_term_variables(ProjectionTerm,ProjectionTermVariables):-
   ProjectionTerm =.. [Functor|ProjectionTermList],
   not (Functor = '$var$'),
   not (Functor = '$bind$'),
   not (ProjectionTermList = []),
   projection_list_vars(ProjectionTermList,ProjectionTermVariables).


projection_list_vars([],[]).
projection_list_vars(['$var$'(VarId)|RestArgs],[dict(VarId,_,_,_,existential)|RestVars]):-
   projection_list_vars(RestArgs,RestVars).
projection_list_vars(['$const$'(_)|RestArgs],Vars):-
   projection_list_vars(RestArgs,Vars).
projection_list_vars(['$bind$'(BindId)|RestArgs],[dict(BindId,_,_,_,existential)|RestVars]):-
   projection_list_vars(RestArgs,RestVars).







% --------------------------------------------------------------------------------------
% RESTRICTION! ProjectionTerm underlies the following restrictions:
%
%  - ProjectionTerm must have a functor other than the built-in
%    operators, i.e. ',',';', etc. are not allowed
%
%  - only variables and constants are allowed as arguments,
%    i.e. no structured terms
%
% --------------------------------------------------------------------------------------

translate_projection('$var$'(VarId),Dict,SelectList):-
   projection_arguments(['$var$'(VarId)],SelectList,Dict).

translate_projection('$bind$'(BindId),Dict,SelectList):-
   projection_arguments(['$bind$'(BindId)],SelectList,Dict).

translate_projection('$const$'(Const),_,['$const$'(Const)]).

translate_projection(ProjectionTerm,Dict,SelectList):-
   ProjectionTerm =.. [Functor|Arguments],
   not (Functor = '$var$'),
   not (Functor = '$const$'),
   not (Functor = '$bind$'),
   not (Arguments = []),
   projection_arguments(Arguments,SelectList,Dict).



projection_arguments([],[],_).

projection_arguments([Arg|RestArgs],[Att|RestAtts],Dict):-
   retrieve_argument(Arg,Att,Dict),
   projection_arguments(RestArgs,RestAtts,Dict).




% - retrieve_argument(Argument,SQLAttribute,Dictionary) --------------------------------
%
% retrieves the mapping of an argument to the appropriate SQL construct, i.e.
%
%  - qualified attribute names for variables in base goals
%  - arithmetic expressions for variables in arithmetic goals
%  - constant values for constants
% 
% --------------------------------------------------------------------------------------

retrieve_argument('$var$'(VarId),Attribute,Dict):-
   lookup(VarId,Dict,TableName,AttName,_),
   (
    TableName = is ->
      Attribute = AttName
   ;
      Attribute = att(TableName,AttName)
   ).

retrieve_argument('$bind$'(BindId),Attribute,Dict):-
   lookup(BindId,Dict,TableName,AttName,_),!,
   Attribute = att(TableName,AttName).

retrieve_argument('$bind$'(BindId),'$bind$'(BindId),Dict):-
   lookup(BindId,Dict,TableName,AttName,_),
   var(TableName), var(AttName).
   

retrieve_argument('$const$'(Constant),'$const$'(Constant),_).





% --- lookup(Key,Dict,Value) -----------------------------------------------------------

lookup(VarId,Dict,RangeVar,Attribute,Type):-
   member(dict(VarId,RangeVar,Attribute,Type,Quant),Dict),
   (
    Quant = all ->
      true
   ;
      nonvar(RangeVar),
      nonvar(Attribute)
   ).



% --- add_to_dictionary(Key,RangeVar,Attribute,Quantifier,Dict,NewDict) ----------------

add_to_dictionary(Key,RangeVar,Attribute,Type,_,Dict,Dict):-
   member(dict(Key,RangeVar,Attribute,Type,existential),Dict).

add_to_dictionary(Key,RangeVar,Attribute,Type,Quantifier,Dict,NewDict):-
   not member(dict(Key,_,_,_,_),Dict),
   NewDict = [dict(Key,RangeVar,Attribute,Type,Quantifier)|Dict].




% --- aggregate_function(AggregateFunctionTerm,Dict,AggregateFunctionQuery) ------------
%
% aggregate_function discerns five Prolog aggregate function terms: count, avg, min,
% max, and sum. Each such term is has two arguments: a variable indicating the attribute
% over which the function is to be computed, and a goal argument which must contain in 
% at least one argument position the variable:
%
%    e.g.  avg(Seats,plane(Type,Seats))
%
% These aggregate function terms correspond to the SQL built-in aggregate functions.
% 
% RESTRICTION: AggregateGoal may only be conjunction of (positive or negative) base 
% goals
% 
% --------------------------------------------------------------------------------------

aggregate_function(AggregateFunctionTerm,Dict,AggregateFunctionExpression):-
   AggregateFunctionTerm =..[AggFunctor,AggVar,AggGoal],
   aggregate_functor(AggFunctor,SQLFunction),
   conjunction(AggGoal,AggConjunction),
   aggregate_query_generation(SQLFunction,AggVar,AggConjunction,Dict,AggregateFunctionExpression).


conjunction(Goal,Conjunction):-
   disjunction(Goal,[Conjunction]).




% --- aggregate_query_generation(Function,FunctionVariable,AggGoal,Dict,AggregateQuery) 
%
% compiles the function variable (representing the attribute over which the aggregate 
% function is to be computed) and the aggregate goal (representing the selection and 
% join conditions for the computation of the aggregate function) to an SQL aggregate 
% function subquery.
% 
% --------------------------------------------------------------------------------------

aggregate_query_generation(count,'$const$'('*'),AggGoal,Dict,AggregateQuery):-
   translate_conjunction(AggGoal,SQLFrom,SQLWhere,Dict,_TmpDict),

   % ATTENTION! It is assumed that in count(*) aggregate query terms there cannot be
   % free variables because '*' stands for "all arguments"

   AggregateQuery = agg_query(_Function,(count,['$const$'(*)]),SQLFrom,SQLWhere,[]).


aggregate_query_generation(Function,FunctionVariable,AggGoal,Dict,AggregateQuery):-
   translate_conjunction(AggGoal,SQLFrom,SQLWhere,Dict,TmpDict),

   % --- only variables occurring in the aggregate goal are relevant to the translation
   % of the function variable and the free variables in the goal.
   % Thus subtract from TmpDict all entries of Dict
   set_difference(TmpDict,Dict,AggDict),
 
   translate_projection(FunctionVariable,AggDict,SQLSelect),
   translate_grouping(FunctionVariable,AggDict,SQLGroup),
   AggregateQuery = agg_query(Function,SQLSelect,SQLFrom,SQLWhere,SQLGroup).





% --- translate_grouping(FunctionVariable,Dict,SQLGroup) -------------------------------
%
% finds the free variables in the aggregate function term and collects their
% corresponding SQL qualified attributes in the SQLGroup list.
% 
% --------------------------------------------------------------------------------------

translate_grouping(FunctionVariable,Dict,SQLGroup):-
   free_vars(FunctionVariable,Dict,FreeVariables),
   translate_free_vars(FreeVariables,SQLGroup).




% --- free_vars(FunctionVariable,Dict,FreeVarList) -------------------------------------
%
% A Variable is free if it neither occurs as the FunctionVariable, nor is stored as
% existentially quantified (through ^/2 in the original goal) in the dictionary
% 
% FreeVars contains for each variable the relevant attribute and relation information 
% contained in the dictionary
% 
% --------------------------------------------------------------------------------------

free_vars(FunctionVariable,Dict,FreeVarList):-
  projection_term_variables(FunctionVariable,FunctionVariableList),
  findall((Var,Table,Attribute),
      (member(dict(Var,Table,Attribute,_Type,all),Dict),
       not member(dict(Var,_,_,_,_),FunctionVariableList)
      ),
      FreeVarList).


% --- function_variable_list(FunctionVariable,FunctionVariableList) --------------------
%
% extracts the list of variables which occur in the function variable term
%
% RESTRICTION: FunctionVariable may only contain one single variable.
% 
% --------------------------------------------------------------------------------------

/*function_variable_list('$var$'(VarId),[VarId]).*/



% --- translate_free_vars(FreeVars,SQLGroup) -------------------------------------------
%
% translates dictionary information on free variables to SQLGroup of aggregate
% function query
% 
% --------------------------------------------------------------------------------------

translate_free_vars([],[]).
translate_free_vars([(_VarId,Table,Attribute)|FreeVars],[att(Table,Attribute)|SQLGroups]):-
   translate_free_vars(FreeVars,SQLGroups).




% --- evaluable_expression(ExpressionTerm,Dictionary,Expression,Type) --------------------
%
% evaluable_expression constructs SQL arithmetic expressions with qualified attribute names
% from the Prolog arithmetic expression term and the information stored in the dictionary.
%
% The type of an evaluable function is returned in the argument Type.
%
% The dictionary is not changed because it is used for lookup only. 
% 

evaluable_expression(AggregateFunctionTerm,Dictionary,AggregateFunctionExpression,number):-
   aggregate_function(AggregateFunctionTerm,Dictionary,AggregateFunctionExpression).

evaluable_expression(LeftExp + RightExp,Dictionary,LeftAr + RightAr,number):-
   evaluable_expression(LeftExp,Dictionary,LeftAr,number),
   evaluable_expression(RightExp,Dictionary,RightAr,number).

evaluable_expression(LeftExp - RightExp,Dictionary,LeftAr - RightAr,number):-
   evaluable_expression(LeftExp,Dictionary,LeftAr,number),
   evaluable_expression(RightExp,Dictionary,RightAr,number).

evaluable_expression(LeftExp * RightExp,Dictionary,LeftAr * RightAr,number):-
   evaluable_expression(LeftExp,Dictionary,LeftAr,number),
   evaluable_expression(RightExp,Dictionary,RightAr,number).

evaluable_expression(LeftExp / RightExp,Dictionary, LeftAr / RightAr,number):-
   evaluable_expression(LeftExp,Dictionary,LeftAr,number),
   evaluable_expression(RightExp,Dictionary,RightAr,number).

evaluable_expression('$var$'(VarId),Dictionary,att(RangeVar,Attribute),Type):-
   lookup(VarId,Dictionary,RangeVar,Attribute,Type),
   RangeVar \= is.

evaluable_expression('$var$'(VarId),Dictionary,ArithmeticExpression,Type):-
   lookup(VarId,Dictionary,is,ArithmeticExpression,Type).
evaluable_expression('$const$'(Const),_,'$const$'(Const),ConstType):-
   get_type('$const$'(Const),ConstType).

evaluable_expression('$bind$'(BindId),_,'$bind$'(BindId),number).





% --------------------------------------------------------------------------------------
%
% Output to screen predicates - rather crude at the moment
%
% --------------------------------------------------------------------------------------


% --- printqueries(Code) ---------------------------------------------------------------

printqueries([Query]):-
   nl,
   print_query(Query),
   write(';'),
   nl,
   nl.

printqueries([Query|Queries]):-
   not (Queries = []),
   nl,
   print_query(Query),
   nl,
   write('UNION'),
   nl,
   printqueries(Queries).



% --- print_query(QueryCode) -----------------------------------------------------------

print_query(query([agg_query(Function,Select,From,Where,Group)],_,_)):-
   % --- ugly rule here: aggregate function only in SELECT Part of query ----
   !,
   print_query(agg_query(Function,Select,From,Where,Group)).

print_query(query(Select,From,Where)):-
   print_clause('SELECT',Select,','),
   nl,
   print_clause('FROM',From,','),
   nl,
   print_clause('WHERE',Where,'AND'),
   nl.

print_query(del_query(From,Where)):-
   print_clause('DELETE',From,','),
   nl,
   print_clause('WHERE',Where,'AND'),
   nl.

print_query(ins_query(From,Where)):-
   print_ins_clause('INSERT INTO',From,','),
   nl,
   print_ins_clause('VALUES (',Where,','),
   nl,
   write(')').


print_query(agg_query(Function,Select,From,Where,Group)):-
   print_clause('SELECT',Function,Select,','),
   nl,
   print_clause('FROM',From,','),
   nl,
   print_clause('WHERE',Where,'AND'),
   nl,
   print_clause('GROUP BY',Group,',').

print_query(negated_existential_subquery(Select,From,Where)):-
   write('NOT EXISTS'),
   nl,
   write('('),
   print_clause('SELECT',Select,','),
   nl,
   print_clause('FROM',From,','),
   nl,
   print_clause('WHERE',Where,'AND'),
   nl,
   write(')').




% --- print_clause(Keyword,ClauseCode,Separator) ---------------------------------------
%
% with 
% Keyword    one of SELECT, FROM, WHERE, or GROUP BY, 
% ClauseCode the code corresponding to the appropriate clause of an SQL query, and 
% Separator  indicating the character(s) through which the items of a clause
%            are separated from each other (',' or 'AND').
% 
% --------------------------------------------------------------------------------------

print_clause(_Keyword,[],_).

print_clause(Keyword,[Column|RestColumns],Separator):-
   write(Keyword),
   write(' '),
   print_clause([Column|RestColumns],Separator).

print_ins_clause(_Keyword,[],_).

print_ins_clause(Keyword,[Column|RestColumns],Separator):-
   write(Keyword),
   write(' '),
   print_ins_clause([Column|RestColumns],Separator).


print_clause(Keyword,Function,[Column],Separator):-
   write(Keyword),
   write(' '),
   write(Function),
   write('('),
   print_clause([Column],Separator),
   write(')').





% --- print_clause(ClauseCode,Separator) -----------------------------------------------

print_clause([Item],_):-
   print_column(Item).

print_clause([Item,NextItem|RestItems],Separator):-
   print_column(Item),
   write(' '),
   write(Separator),
   write(' '),
   print_clause([NextItem|RestItems],Separator).

print_ins_clause([Item],_):-
   print_ins_column(Item).

print_ins_clause([Item,NextItem|RestItems],Separator):-
   print_ins_column(Item),
   write(' '),
   write(Separator),
   write(' '),
   print_ins_clause([NextItem|RestItems],Separator).



% --- print_column(ColumnCode) --------------------------------

print_column('*'):-
   write('*').

print_column(att(RangeVar,Attribute)):-
   write(RangeVar),
   write('.'),
   write(Attribute).

print_column(rel(Relation,RangeVar)):-
   write(Relation),
   write(' '),
   write(RangeVar).

print_ins_column(rel(Relation,_RangeVar)):-
   write(Relation).
   
print_column('$const$'('NULL')):-
   !,
   write('NULL').

print_column('$const$'(String)):-
   get_type('$const$'(String),string),
   write(''''),
   write(String),
   write('''').

print_column('$const$'(Number)):-
   get_type('$const$'(Number),NumType),
   type_compatible(NumType,number),
   write(Number).

print_column('$bind$'(BindId)):-
   write(':'),
   write(BindId).

print_column(comp(LeftArg,Operator,RightArg)):-
   print_column(LeftArg),
   write(' '),
   write(Operator),
   write(' '),
   print_column(RightArg).

print_ins_column(comp(_LeftArg,_Operator,RightArg)):-
      print_column(RightArg).


print_column(LeftExpr * RightExpr):-
   print_column(LeftExpr),
   write('*'),
   print_column(RightExpr).

print_column(LeftExpr / RightExpr):-
   print_column(LeftExpr),
   write('/'),
   print_column(RightExpr).

print_column(LeftExpr + RightExpr):-
   print_column(LeftExpr),
   write('+'),
   print_column(RightExpr).

print_column(LeftExpr - RightExpr):-
   print_column(LeftExpr),
   write('-'),
   print_column(RightExpr).

print_column(agg_query(Function,Select,From,Where,Group)):-
   nl,
   write('('),
   print_query(agg_query(Function,Select,From,Where,Group)),
   write(')').

print_column(negated_existential_subquery(Select,From,Where)):-
   print_query(negated_existential_subquery(Select,From,Where)).





% --- queries_atom(Queries,QueryAtom) ---------------------------- 
%
% queries_atom(Queries,QueryAtom) returns in its second argument
% the SQL query as a Prolog atom. For efficiency reasons, a list
% of ASCII codes is ceated as a difference list, and it is then 
% transformed to an atom by name/2
% ---------------------------------------------------------------- 


queries_atom(Queries,QueryAtom):-
   queries_atom(Queries,QueryList,[]),
   name(QueryAtom,QueryList).



queries_atom([Query],QueryList,Diff):-
   query_atom(Query,QueryList,Diff).

queries_atom([Query|Queries],QueryList,Diff):-
   Queries \= [],
   query_atom(Query,QueryList,X1),
   column_atom('UNION ',X1,X2),
   queries_atom(Queries,X2,Diff).


% --- query_atom(QueryCode) --------------------------------

query_atom(query([agg_query(Function,Select,From,Where,Group)],_,_),QueryList,Diff):-
   % --- ugly rule here: aggregate function only in SELECT Part of query ----
   !,
   query_atom(agg_query(Function,Select,From,Where,Group),QueryList,Diff).

query_atom(query(Select,From,Where),QueryList,Diff):-
   clause_atom('SELECT',Select,',',QueryList,X1),
   clause_atom('FROM',From,',',X1,X2),
   clause_atom('WHERE',Where,'AND',X2,Diff).

query_atom(del_query(From,Where),QueryList,Diff):-
   clause_atom('DELETE',From,',',QueryList,X1),
   clause_atom('WHERE',Where,'AND',X1,Diff).

query_atom(ins_query(From,Where),QueryList,Diff):-
   clause_ins_atom('INSERT INTO',From,',',QueryList,X1),
   clause_ins_atom('VALUES (',Where,',',X1,X2),
   column_atom(')',X2,Diff).


query_atom(agg_query(Function,Select,From,Where,Group),QueryList,Diff):-
   clause_atom('SELECT',Function,Select,',',QueryList,X1),
   clause_atom('FROM',From,',',X1,X2),
   clause_atom('WHERE',Where,'AND',X2,X3),
   clause_atom('GROUP BY',Group,',',X3,Diff).

query_atom(negated_existential_subquery(Select,From,Where),QueryList,Diff):-
   column_atom('NOT EXISTS(',QueryList,X1),   
   clause_atom('SELECT',Select,',',X1,X2),
   clause_atom('FROM',From,',',X2,X3),
   clause_atom('WHERE',Where,'AND',X3,X4),
   column_atom(')',X4,Diff).





% --- clause_atom(Keyword,ClauseCode,Junctor,CurrAtom,QueryAtom) -------------
%
% with 
% Keyword    one of SELECT, FROM, WHERE, or GROUP BY, 
% ClauseCode the code corresponding to the appropriate clause of an SQL query, and 
% Junctor    indicating the character(s) through which the items of a clause
%            are separated from each other (',' or 'AND').

clause_atom(_Keyword,[],_,QueryList,QueryList).

clause_atom(Keyword,[Column|RestColumns],Junctor,QueryList,Diff):-
   column_atom(Keyword,QueryList,X1),
   column_atom(' ',X1,X2),
   clause_atom([Column|RestColumns],Junctor,X2,X3),
   column_atom(' ',X3,Diff).

clause_ins_atom(_Keyword,[],_,QueryList,QueryList).

clause_ins_atom(Keyword,[Column|RestColumns],Junctor,QueryList,Diff):-
   column_atom(Keyword,QueryList,X1),
   column_atom(' ',X1,X2),
   clause_ins_atom([Column|RestColumns],Junctor,X2,X3),
   column_atom(' ',X3,Diff).


clause_atom(Keyword,Function,[Column],Junctor,QueryList,Diff):-
   column_atom(Keyword,QueryList,X1),
   column_atom(' ',X1,X2),
   column_atom(Function,X2,X3),
   column_atom('(',X3,X4),
   clause_atom([Column],Junctor,X4,X5),
   column_atom(') ',X5,Diff).






% --- clause_atom(ClauseCode,Junctor) --------------------------------

clause_atom([Item],_,QueryList,Diff):-
   column_atom(Item,QueryList,Diff).

clause_atom([Item,NextItem|RestItems],Junctor,QueryList,Diff):-
   column_atom(Item,QueryList,X1),
   column_atom(' ',X1,X2),
   column_atom(Junctor,X2,X3),
   column_atom(' ',X3,X4),
   clause_atom([NextItem|RestItems],Junctor,X4,Diff).

clause_ins_atom([Item],_,QueryList,Diff):-
   column_ins_atom(Item,QueryList,Diff).

clause_ins_atom([Item,NextItem|RestItems],Junctor,QueryList,Diff):-
   column_ins_atom(Item,QueryList,X1),
   column_atom(' ',X1,X2),
   column_atom(Junctor,X2,X3),
   column_atom(' ',X3,X4),
   clause_ins_atom([NextItem|RestItems],Junctor,X4,Diff).



column_atom(att(RangeVar,Attribute),QueryList,Diff):-
   column_atom(RangeVar,QueryList,X1),
   column_atom('.',X1,X2),
   column_atom(Attribute,X2,Diff).

column_atom(rel(Relation,RangeVar),QueryList,Diff):-
   column_atom(Relation,QueryList,X1),
   column_atom(' ',X1,X2),
   column_atom(RangeVar,X2,Diff).

column_ins_atom(rel(Relation,_RangeVar),QueryList,Diff):-
   column_atom(Relation,QueryList,Diff).

column_atom('$const$'('NULL'),QueryList,Diff):- 
   get_type('$const$'('NULL'),string),!,
   column_atom('NULL',QueryList,Diff).

column_atom('$const$'(String),QueryList,Diff):-
   get_type('$const$'(String),string),
   column_atom('''',QueryList,X1),
   column_atom(String,X1,X2),
   column_atom('''',X2,Diff).

column_atom('$const$'(Number),QueryList,Diff):-
   get_type('$const$'(Number),NumType),
   type_compatible(NumType,number),
   column_atom(Number,QueryList,Diff).


column_atom('$bind$'(BindId),QueryList,Diff):-
   column_atom(':',QueryList,X1),
   column_atom(BindId,X1,Diff).

column_atom(comp(LeftArg,Operator,RightArg),QueryList,Diff):-
   column_atom(LeftArg,QueryList,X1),
   column_atom(' ',X1,X2),
   column_atom(Operator,X2,X3),
   column_atom(' ',X3,X4),
   column_atom(RightArg,X4,Diff).

column_ins_atom(comp(_LeftArg,_Operator,RightArg),QueryList,Diff):-
      column_atom(RightArg,QueryList,Diff).


column_atom(LeftExpr * RightExpr,QueryList,Diff):-
   column_atom(LeftExpr,QueryList,X1),
   column_atom('*',X1,X2),
   column_atom(RightExpr,X2,Diff).

column_atom(LeftExpr + RightExpr,QueryList,Diff):-
   column_atom(LeftExpr,QueryList,X1),
   column_atom('+',X1,X2),
   column_atom(RightExpr,X2,Diff).

column_atom(LeftExpr - RightExpr,QueryList,Diff):-
   column_atom(LeftExpr,QueryList,X1),
   column_atom('-',X1,X2),
   column_atom(RightExpr,X2,Diff).

column_atom(LeftExpr / RightExpr,QueryList,Diff):-
   column_atom(LeftExpr,QueryList,X1),
   column_atom('/',X1,X2),
   column_atom(RightExpr,X2,Diff).

column_atom(agg_query(Function,Select,From,Where,Group),QueryList,Diff):-
   column_atom('(',QueryList,X1),
   query_atom(agg_query(Function,Select,From,Where,Group),X1,X2),
   column_atom(')',X2,Diff).

column_atom(negated_existential_subquery(Select,From,Where),QueryList,Diff):-
   query_atom(negated_existential_subquery(Select,From,Where),QueryList,Diff).


column_atom(Atom,List,Diff):-
   check_atomic(Atom,'column_atom/3',1),
   name(Atom,X1),
   append(X1,Diff,List).


% --- gensym(Root,Symbol) ----------------------------------------------------
%
% SEPIA 3.2. version - other Prolog implementations provide gensym/2
% and init_gensym/1 as built-ins. */
%
% (C) Christoph Draxler, Aug. 1992
%
% 

gensym(Root,Symbol):-
   nonvar(Root),
   var(Symbol),
   conget(Root,Counter),
   NewCounter is Counter + 1,
   name(NewCounter,L), atom_codes(Atom,L),
   str_cat(Root,Atom,Symbol),
   conset(Root,NewCounter).
   

init_gensym(Root):-
   nonvar(Root),
   conset(Root,0). 




% --- set_difference(SetA,SetB,Difference) -----------------------------------
%
% SetA - SetB = Difference

set_difference([],_,[]).

set_difference([Element|RestSet],Set,[Element|RestDifference]):-
   not member(Element,Set),
   set_difference(RestSet,Set,RestDifference).

set_difference([Element|RestSet],Set,RestDifference):-
   member(Element,Set),
   set_difference(RestSet,Set,RestDifference).


% --- banchmarks of sample queries ---------



% --- Meta Database for schema definition of SQL DB in Prolog ----------------
%
% maps Prolog predicates to SQL table names, Prolog predicate argument positions to SQL
% attributes, and Prolog operators to SQL operators. 
%
% ATTENTION! It is assumed that the arithmetic operators in Prolog and SQL are the same,
% i.e. + is addition in Prolog and in SQL, etc. If this is not the case, then a mapping
% function for arithmetic operators is necessary too.
% ---------------------------------------------------------------------off -----------------


% --- relation(PrologFunctor,Arity,SQLTableName) ---------------------------------------
/*
relation(flight,4,'FLIGHT').
*/


% --- attribute(PrologArgumentPosition,SQLTableName,SQLAttributeName) ------------------
/*
attribute(1,'FLIGHT','FLIGHT_NO',string).
*/





% --- Mapping of Prolog operators to SQL operators -------------------------------------

comparison(=,=).
comparison(<,<).
comparison(>,>).
comparison(@<,<).
comparison(@>,>).
comparison('IS','IS').


negated_comparison(=,'<>').
negated_comparison(\=,=).
negated_comparison(>,=<).
negated_comparison(=<,>).
negated_comparison(<,>=).
negated_comparison(>=,<).


% --- aggregate_function(PrologFunctor,SQLFunction) -----------------

aggregate_functor(avg,'AVG').
aggregate_functor(min,'MIN').
aggregate_functor(max,'MAX').
aggregate_functor(sum,'SUM').
aggregate_functor(count,'COUNT').



% --- type system -------------------------------------------------------------
%
% A rudimentary type system is provided for consistency checking during the
% translation and for output formatting
%
% The basic types are string and number. number has the subtypes integer and
% real.
%
% -----------------------------------------------------------------------------


type_compatible(Type,Type):-
   is_type(Type).
type_compatible(SubType,Type):-
   subtype(SubType,Type).
type_compatible(Type,SubType):-
   subtype(SubType,Type).


% --- subtype(SubType,SuperType) -----------------------------------------------
%
% Simple type hierarchy checking
%
% ------------------------------------------------------------------------------

subtype(SubType,SuperType):-
   is_subtype(SubType,SuperType).

subtype(SubType,SuperType):-
   is_subtype(SubType,InterType),
   subtype(InterType,SuperType).



% --- is_type(Type) ------------------------------------------------------------
%
% Type names
%
% ------------------------------------------------------------------------------

is_type(number).
is_type(integer).
is_type(real).
is_type(string).
is_type(natural).


% --- is_subtype(SubType,SuperType) --------------------------------------------
%
% Simple type hierarchy for numeric types
%
% ------------------------------------------------------------------------------

is_subtype(integer,number).
is_subtype(real,number).
is_subtype(natural,integer).


% --- get_type(Constant,Type) --------------------------------------------------
%
% Prolog implementation specific definition of type retrieval
% sepia Prolog version given here
%
% ------------------------------------------------------------------------------

get_type('$const$'(Constant),integer):-
   number(Constant).

get_type('$const$'(Constant),string):-
   atom(Constant).


assert_call_rule(ProjectionTerm, SQLAtom,Num) :-
      	init_gensym(var),
   	init_gensym(rel),
   	init_gensym('BIND'),
	tokenize_term(ProjectionTerm, Tokens),!,
	functor(ProjectionTerm, Head, Arity),
	Tokens =.. [Head|Args],
	functor(RuleHead, Head, Arity),
	RuleHead =..[_RuleH|NewArgs],
	converts(Args, NewArgs, RuleBody),
	asserta((RuleHead :- (RuleBody ',' ! ',' db_all(RuleHead, SQLAtom, Tokens,Num)))).
	/*listing.*/


convert('$var$'(_), A, var(A)).
convert('$bind$'(_),A, ','(nonvar(A), not(A = 'NULL'(_X)))).
convert('$const$'('NULL'), A, ','(nonvar(A),A = 'NULL'(_))) :- !.
convert('$const$'(C),A, A == C).


converts([H,H2|Rest], [H1|Rest1], ','(X, RestX)) :-
	convert(H,H1, X),
	converts([H2|Rest], Rest1, RestX).

converts([H],[H1],X) :- 
	convert(H,H1,X).

assert_call_rule_del(ProjectionTerm, SQLAtom,Num) :-
      	init_gensym(var),
   	init_gensym(rel),
   	init_gensym('BIND'),
	tokenize_term(ProjectionTerm, Tokens),!,
	functor(ProjectionTerm, Head, Arity),
	Tokens =.. [Head|Args],
	functor(RuleHead, Head, Arity),
	RuleHead =..[_RuleH|NewArgs],
	converts_del(Args, NewArgs, RuleBody),
	asserta((RuleHead :- (RuleBody ',' ! ',' db_all(RuleHead, SQLAtom, Tokens,Num)))).
	/*listing.*/

assert_call_rule_test(ProjectionTerm, SQLAtom,Num) :-
      	init_gensym(var),
   	init_gensym(rel),
   	init_gensym('BIND'),
	tokenize_term(ProjectionTerm, Tokens),!,
	functor(ProjectionTerm, Head, Arity),
	Tokens =.. [Head|Args],
	functor(RuleHead, Head, Arity),
	RuleHead =..[_RuleH|NewArgs],
	converts_del(Args, NewArgs, RuleBody),
	asserta((RuleHead :- (RuleBody ',' ! ',' db_all_test(RuleHead, SQLAtom, Tokens,Num)))).





convert_del('$var$'(_), A, var(A)).
convert_del('$bind$'(_),A,','(nonvar(A), not((A = 'NULL'(X),var(X))))).
convert_del('$const$'('NULL'), A,','(nonvar(A), ','(A = 'NULL'(X), var(X)))) :- !.
convert_del('$const$'(C),A, A == C).


converts_del([H,H2|Rest], [H1|Rest1], ','(X, RestX)) :-
	convert_del(H,H1, X),
	converts_del([H2|Rest], Rest1, RestX).

converts_del([H],[H1],X) :- 
	convert_del(H,H1,X).
%----------------------------------------

	
count_b_f(Term_List, B, S) :- count_bind_free(Term_List, 0, B, 0, S).

count_bind_free([], B,B, S,S).

count_bind_free(['$bind$'(_)|T], Prev_Bind, Next_Bind, Prev_SLI, Next_SLI) :-
	!,
	Next is Prev_Bind + 1,
	count_bind_free(T, Next, Next_Bind, Prev_SLI, Next_SLI).

count_bind_free([_|T], Prev_Bind, Next_Bind, Prev_SLI, Next_SLI) :-
	Next is Prev_SLI + 1,
	count_bind_free(T, Prev_Bind, Next_Bind, Next, Next_SLI).


/* Oracle Interface Primitives (Builtins) */
/* -------------------------------------- */

oracle_connect(Name,Password,Status) :-
	oracle_query(1,Name,Password,Status).

oracle_disconnect(Status) :- oracle_query(2,Status).

allocate_cursor(Template,StmtString,NumBindVars,SQLType,CursorHandle,Status) :-
	oracle_query(3,Template,StmtString,NumBindVars,SQLType,
	             CursorHandle,Status).

deallocate_cursor(CursorHandle,Status) :-
	oracle_query(4,CursorHandle,Status).

close_cursor(CursorHandle,Status) :-
	oracle_query(5,CursorHandle,Status).

% reclaim_cursors(Status) :- oracle_query(6,Status).

set_input_value(CursHndl,BindVarName,BindVarVal,BindVarValType,Status) :-
	oracle_query(7,CursHndl,BindVarName,BindVarVal,BindVarValType,Status).

execute_cursor(CursorHandle,NumSLIs,Status) :-
	oracle_query(8,CursorHandle,NumSLIs,Status).

flush_cursor(CursorHandle,Status) :-
	oracle_query(9,CursorHandle,Status).

fetch_from_cursor(CursorHandle,Status) :-
	oracle_query(10,CursorHandle,Status).

get_output_value(CursorHandle,BindVarValue,Status) :-
	oracle_query(11,CursorHandle,BindVarValue,Status).

execute_immediate(StmtString,Status) :-
	oracle_query(12,StmtString,Status).

:- assert(attribute('$','$','$','$')).

db_sqlcaid(Res) :- oracle_query(16,Res).
db_sqlcabc(Res) :- oracle_query(17,Res).
db_sqlcode(Res) :- oracle_query(18,Res).
db_sqlerrml(Res) :- oracle_query(19,Res).
db_sqlerrmc(Res) :- oracle_query(20,Res).
db_sqlerrp(Res) :- oracle_query(21,Res).
db_sqlerrd0(Res) :- oracle_query(22,Res).
db_sqlerrd1(Res) :- oracle_query(23,Res).
db_sqlerrd2(Res) :- oracle_query(24,Res).
db_sqlerrd3(Res) :- oracle_query(25,Res).
db_sqlerrd4(Res) :- oracle_query(26,Res).
db_sqlerrd5(Res) :- oracle_query(27,Res).
db_sqlwarn0(Res) :- oracle_query(28,Res).
db_sqlwarn1(Res) :- oracle_query(29,Res).
db_sqlwarn3(Res) :- oracle_query(30,Res).
db_sqlwarn4(Res) :- oracle_query(31,Res).
db_sqlwarn5(Res) :- oracle_query(32,Res).
db_sqlext(Res) :- oracle_query(33,Res).

% oracle_query(A) :- oracle_query(A,_,_,_,_,_,_).
oracle_query(A,B) :- oracle_query(A,B,_,_,_,_,_).
oracle_query(A,B,C) :- oracle_query(A,B,C,_,_,_,_).
oracle_query(A,B,C,D) :- oracle_query(A,B,C,D,_,_,_).
% oracle_query(A,B,C,D,E) :- oracle_query(A,B,C,D,E,_,_).
oracle_query(A,B,C,D,E,F) :- oracle_query(A,B,C,D,E,F,_).
