
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>

#include "gamepick.h"
#include "read_line.h"
#include "add_games.h"
#include "edits.h"


void maybe_game_entry (gamepick_info *info, game_struct *game, gint this_tab)
{
  if (!text_here (game->name))
    return;
    
  add_game (info, game, this_tab);
  zero_game_entry (game);
}


int
result
(gamepick_info *info, const gchar *file, gboolean error, const gchar *message)
{
  if (file != NULL)
    g_string_assign (info->conf_file, file);
  info->bad_conf = error;
  if (message)
  {
    g_string_append (info->warnings, message);
    g_string_append (info->warnings, "\n");
  }
  return 0;
}


int find_conf (gamepick_info *info)
{
  const gchar *text;
  static const gchar example[] = "example.conf";

  info->conf_file = g_string_new (NULL);
  info->pwd = g_string_new (NULL);
  info->home_conf = g_string_new (NULL);
  info->debug_window = FALSE;
  info->options_enabled = TRUE;

  text = getenv ("HOME");
  if ((text) && (strcmp (text, "") != 0))
    g_string_sprintf (info->home_conf, "%s/%s",
      text, default_gamepick_home_conf);
  
  text = getenv ("PWD");
  if ((text) && (strcmp (text, "") != 0))
    g_string_assign (info->pwd, text);
  
  text = cmdline_option (info, "-c");
  if (text)
  {
    if (strcmp (text, "") == 0)
      return result (info, example, TRUE, "-c option with no filename");
    if (!file_readable (text))
      return result (info, example, TRUE, "specified file not readable");
    return result (info, text, FALSE, NULL);
  }
  
  if (file_readable (info->home_conf->str))
    return result (info, info->home_conf->str, FALSE, NULL);
  
  if (file_readable (default_gamepick_global_conf))
    return result (info, default_gamepick_global_conf, FALSE, NULL);
  
  if (file_readable (example))
    return result (info, example, FALSE, NULL);

  if (text_here (info->home_conf))
    return result (info, info->home_conf->str, TRUE,
      "No configuration found. Please run auto-games to\n"
      "detect known installed games.\n\n"
      "Visit this program\'s website for icons to use\n"
      "(options, about)");
  
  return result (info, example, TRUE,
    "Warning: no configuration file available.");
}


gboolean keyword_valid (const gchar *text)
{
  static const char* keyword_array[] = 
  {
    "tab", "name", "icon", "cmd", "fsaa", "aitf", "icon dir",
    "options", "auto find", "config fallback", "debug",
    "start dir", "syncvblank", ""
  };
  int count = 0;
  
  while (TRUE)
  {
    if (strcmp (keyword_array [count], "") == 0) return FALSE;
    if (strcmp (text, keyword_array [count]) == 0) return TRUE;
    count++;
  }
}


gboolean keyword_test (const gchar *left, const gchar* keyword)
{
  if (!keyword_valid (keyword))
  {
    g_print ("internal error: unknown keyword %s\n", keyword);
    return FALSE;
  }
  if (strcmp (left, keyword) == 0)
    return TRUE;
  else
    return FALSE;
}


gboolean test_bool (const gchar *text)
{
  if (strcmp (text, "0") == 0)
    return FALSE;
  if (strcmp (text, "") == 0)
    return FALSE;
  return TRUE;
}


void load_lists (gamepick_info *info)
{
  readline *context;
  gchar *left, *right;
  GString *line, *error_text;
  gint count = 0, index, this_tab = -1;
  gboolean errors_present = FALSE;
  game_struct game;

  zero_game_entry (&game);
    
  if (info->bad_conf)
  {
    add_tab (info, "Games 1");
    g_string_assign (game.name, "glxgears");
    g_string_assign (game.cmd, "glxgears");
    add_game (info, &game, 0);
    return;
  }
  
  line = g_string_new (NULL);
  error_text = g_string_new (NULL);
  context = read_start (info->conf_file->str, 256);
  zero_game_entry (&game);

  for (;;)
  {
    if (text_here (error_text))
    {
      if (errors_present == FALSE)
      {
        g_print ("gamepick: errors in %s\n", info->conf_file->str);
        errors_present = TRUE;
      }
      g_print ("line %d: %s\n", count, error_text->str);
      g_string_assign (error_text, "");
    }
    count++;
    if (!read_next (context, &line))
    {
      maybe_game_entry (info, &game, this_tab);
      return;
    }
    if (!text_here (line))
      continue;
    if ((line->str)[0] == '#')
      continue;
    index = char_find (line->str, ':');
    if (index < 0)
      continue;
    if ((line->len < 3) || (index < 1))
      continue;
    (line->str) [index] = '\0';
    left = line->str;
    right = line->str + index + 1; /* yeah a bit iffy */
    while (*right == ' ')
      right++; /* same */
    
    if (strcmp (left, "tab") == 0)
    {
      maybe_game_entry (info, &game, this_tab);
      this_tab = add_tab (info, right);
    }
    else
    if (keyword_test (left, "name"))
    {
      maybe_game_entry (info, &game, this_tab);
      g_string_assign (game.name, right);
    }
    else
    if (keyword_test (left, "cmd"))
      g_string_assign (game.cmd, right);
    else
    if (keyword_test (left, "icon"))
      g_string_assign (game.icon, right);
    else
    if (keyword_test (left, "fsaa"))
      game.fsaa = atoi (right);
    else
    if (keyword_test (left, "aitf"))
      game.aitf = atoi (right);
    else
    if (keyword_test (left, "icon dir"))
      g_string_assign (info->icon_dir, right);
    else
    if (keyword_test (left, "start dir"))
      g_string_assign (game.start_dir, right);
    else
    if (keyword_test (left, "syncvblank"))
      game.syncvblank = test_bool (right);
  }
}


void first_parse (gamepick_info *info)
{
  readline *context;
  gchar *left, *right;
  GString *line, *error_text;
  gint count = 0, index = 0;
  gboolean errors_present = FALSE;
    
  if (info->bad_conf)
  {
    g_print ("can't load settings\n");
    return;
  }
  
  line = g_string_new (NULL);
  error_text = g_string_new (NULL);
  context = read_start (info->conf_file->str, 256);

  for (;;)
  {
    if (text_here (error_text))
    {
      if (errors_present == FALSE)
      {
        g_print ("gamepick: errors in %s\n", info->conf_file->str);
        errors_present = TRUE;
      }
      g_print ("line %d: %s\n", count, error_text->str);
      g_string_assign (error_text, "");
    }
    count++;
    if (!read_next (context, &line))
      return;
    if (!text_here (line))
      continue;
    if ((line->str)[0] == '#')
      continue;
    index = char_find (line->str, ':');
    if (index < 0)
    {
      g_string_assign (error_text, "no : divider");
      continue;
    }
    if ((line->len < 3) || (index < 1))
    {
      g_string_assign (error_text, "line too short");
      continue;
    }
    (line->str) [index] = '\0';
    left = line->str;
    right = line->str + index + 1; /* yeah a bit iffy */
    while (*right == ' ')
      right++; /* same */
    if (strcmp (left, "") == 0)
    {
      g_string_assign (error_text, "no keyword");
      continue;
    }
    if (!keyword_valid (left))
    {
      g_string_sprintf (error_text, "unknown keyword: %s", left);
      continue;
    }
    if (strcmp (right, "") == 0)
    {
      g_string_assign (error_text, "no value");
      continue;
    }
    
    if (keyword_test (left, "options"))
    {
      info->options_enabled = test_bool (right);
      continue;
    }

    if (keyword_test (left, "debug"))
    {
      info->debug_window = test_bool (right);
      continue;
    }

    if (keyword_test (left, "config fallback"))
    {
      info->conf_fallback_enabled = test_bool (right);
      continue;
    }

    if (keyword_test (left, "auto find"))
    {
      info->auto_find_list = g_list_append (info->auto_find_list,
        g_string_new (right));
      continue;
    }
  }
}


void reset_gamepick_info (gamepick_info *info)
{
  info->tab_list = NULL;
  info->tab_count = 0;
  info->current_game = NULL;
  info->button_spacing = 5;
  info->icon_spacing = 10;
  info->x_icons = 5;
  info->y_icons = 3;
  info->max_line = 10;
  info->cmd_dir = g_string_new (default_cmd_dir);
  info->icon_dir = g_string_new (default_icon_dir);
  info->options_enabled = TRUE;
  info->conf_fallback_enabled = 0;
  info->auto_find_list = NULL;
}


void warnings (gamepick_info *info, const gchar *actions)
{
  gint index;
  
  for (index = 0; index < strlen (actions); index++)
  {
    switch (actions [index])
    {
      case 'c':
        if (text_here (info->warnings))
          g_print (info->warnings->str);
        break;
      
      case 'x':
        if (text_here (info->warnings))
          show_notice (info->warnings->str, info->main_window);
        break;
      
      case '0':
        g_string_assign (info->warnings, "");
        return; /* must: actions [index] corrupt */
      
      default:
        g_print ("gamepick: unknown warning action: %c\n", actions [index]);
    }
  }
}
