/*
 * nmon - network statistics monitor
 *
 * Hmm... ya, should write some infos here.
 *
 * Copyright (C) 2002 Thomas Graf <tgr@reeler.org>
 *
 * This file belongs to the nstats package, see COPYING for more information.
 *
 */

#include <stdio.h>
#include <signal.h>
#include <curses.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <errno.h>
#include <time.h>

#include "../config.h"
#include "packet.h"
#include "pcap.h"
#include "conn.h"
#include "util.h"
#include "curs_util.h"
#include "stats.h"


extern struct stat_s stats;

       int              do_resolve = 0;
static int              promisc = 1;
static int              no_pcap = 0;
static int              quit_done = 0;
static char             n_filter[4096];
static char             device[256];
       char             dname[256];
       time_t           now;


/*
 * empty implementation because we don't need connection
 * tracking stuff in nmon
 */
void
handle_port(int sp, int dp, char *saddr, char *daddr, int type,
            const struct pcap_pkthdr *hdr)
{
}


RETSIGTYPE
finish(int sig)
{
    exit(0);
}

void
doexit(void)
{
    if (!quit_done) {
        if (!no_pcap) {
            pcap_stop_all();
        }
        endwin();
    }
    quit_done=1;
}

void
quit(const char *fmt, ...)
{
    va_list args;

    doexit();

    va_start(args, fmt);
    vfprintf(stderr, fmt, args);
    va_end(args);

    exit(-1);
}


void
reset(void)
{
    memset(&stats, 0, sizeof(stats));
}

void
draw(void)
{
    row = 0;
    stats_draw();
    print_status("nmon");
    refresh();
}



int main(int argc, char *argv[])
{
    time_t bt=0;

    memset(&stats, 0, sizeof(stats));
    memset(n_filter, 0, sizeof(n_filter));

    if ( geteuid() != 0 ) {
        fprintf(stderr, "You need to be root.\n");
        exit(0);
    }

    dname[0] = '.';
    if (getdomainname(&dname[1], sizeof(dname)-1) < 0) {
        perror("getdomainname");
        exit(-1);
    }

    while ( 1 ) {
        int c;

        if ( (c = getopt(argc, argv, "pf:i:nx")) == -1)
            break;

        switch (c) {
            case 'p':
                promisc = 0;
                break;
            case 'f':
                strncpy(n_filter, optarg, sizeof(n_filter));
                break;
            case 'i':
                strncpy(device, optarg, sizeof(device));
                break;
            case 'x':
                no_pcap = 1;
                break;
            default:
                exit(-1);
        }
    }

    atexit(&doexit);

    if (signal(SIGINT, finish) < 0) {   /* clean exit upon interrupt */
        perror("signal");
        finish(-1);
    }

    curs_init();

    if (!no_pcap) {

        if (device[0])
            pcap_listen_on_single(device, promisc);
        else
            pcap_listen_on_all(promisc);
    }

    for (;;) {

        now = time(0);

        for (;;) {

            int ch = getch();

            /*
             * handle global commands 
             */
            switch( ch ) {

                case 'q':             /* quit */
                    finish(0);
                    break;

                case 'r':             /* reset */
                    reset();
                    draw();
                    break;

                case 12:
                    clear();
                    draw();
                    break;
            }

            if ( stats_handle_input(ch) )
                draw();

            if (ch == -1)
                break;
        }

        if (!no_pcap) {
            pcap_handle();
        }

        /*
         * do some stuff every second
         */
        if (!bt || (now - bt) >= 1) {
            stats_calc_rate();
            draw();
            bt = now;
        }

        usleep(20000);
    }

    return 0; /* will never be reached */
}
