/*
 * aconn.c
 *
 * Copyright (C) 2002 Thomas Graf <tgr@reeler.org>
 *
 * This file belongs to the nstats package, see COPYING for more information.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <pcap.h>
#include <netdb.h>

#include <netinet/in.h>
#include <netinet/ether.h>
#include <netinet/ip.h>
#include <netinet/ip6.h>
#include <netinet/icmp6.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <sys/ioctl.h>

#include "aconn.h"
#include "stats.h"
#include "packet.h"
#include "util.h"
#include "curs_util.h"

extern time_t now;
extern int do_resolve;

extern int row;

static struct aconn_s *aconns = NULL;

enum {
    c_all,
    c_established,
    c_listen,
    c_time_wait
} m_conn_show;

static char *m_conn_text[] = {
    "ALL",
    "ESTABLISHED",
    "LISTEN",
    "TIME_WAIT"
};

enum {
    s_rate,
    s_state,
    s_srcip,
    s_dstip,
    s_srcport,
    s_dstport
} m_conn_sort;

const char *sort_text[] =
{
    "Rate",
    "State",
    "Source IP",
    "Dest IP",
    "Source Port",
    "Dest Port"
};

enum {
    i_rate,
    i_idle,
    i_total
} m_conn_info;

const char *info_text[] =
{
    "Rate",
    "Idle Time",
    "Total Bytes"
};

const char *tcp_state[] =
{
    "  ",
    "E ",      /* ESTABLISHED   */
    "SS",      /* SYN_SENT      */
    "SR",      /* SYN_RECV      */
    "F1",      /* FIN_WAIT1     */
    "F2",      /* FIN_WAIT2     */
    "TW",      /* TIME_WAIT     */
    "C ",      /* CLOSE         */
    "CW",      /* CLOSE_WAIT    */
    "LA",      /* LAST_ACK      */
    "L ",      /* LISTEN        */
    "CL"       /* CLOSING       */
};



/*
 * most of the code in here is stolen from netstat.c
 */


static int i=1;

extern void quit(const char *);

void
aconn_mark(void)
{
    struct aconn_s *ac;

    ac = aconns;
    while (ac) {
        ac->to_remove = 1;
        ac = ac->next;
    }
}

void
aconn_delete_marked(void)
{
    struct aconn_s *ac, *n, *last=NULL;

    ac = aconns;
    while (ac) {
        n = ac->next;

        if (ac->to_remove) {

            if (last)
                last->next = ac->next;
            else
                aconns = ac->next;

            if (ac->sname)
                free(ac->sname);

            if (ac->dname)
                free(ac->dname);

            if (ac->sip)
                free(ac->sip);

            if (ac->dip)
                free(ac->dip);

            free(ac);
        } else
            last = ac;
        ac = n;
    }
}

void
aconn_calc_rate(void)
{
    struct aconn_s *ac;

    ac = aconns;
    while (ac) {

        if (!ac->t)
            ac->t = now;

        if ( (now - ac->t) >= 3 ) {

            ac->r_bytes += (ac->bytes - ac->bytes_old);
            ac->r_bytes /= ( (now -ac->t) + 1);

            ac->bytes_old = ac->bytes;

            ac->t = now;
        }

        ac = ac->next;
    }
}

void
aconn_update(const char *proto, struct sockaddr *src, struct sockaddr *dst,
             int bytes, int state)
{
    struct aconn_s *ac;

    ac = aconns;
    while (ac) {
        if ( ! strcmp(proto, ac->proto) &&
               src->sa_family == ((struct sockaddr *) &aconns->src)->sa_family) {

            switch ( ((struct sockaddr *) src)->sa_family ) {

                case AF_INET:
                    if (!memcmp((void *) &ac->src, (void *) src, sizeof(struct sockaddr_in)) &&
                        !memcmp((void *) &ac->dst, (void *) dst, sizeof(struct sockaddr_in)))
                        goto found;
                    break;

                case AF_INET6:
                    if (!memcmp((void *) &ac->src, (void *) src, sizeof(struct sockaddr_in6)) &&
                        !memcmp((void *) &ac->dst, (void *) dst, sizeof(struct sockaddr_in6)))
                        goto found;
                    break;

                default:
                    return;
            }
        }
        ac = ac->next;
    }

    if ( !(ac = (struct aconn_s *) calloc(1, sizeof(struct aconn_s))) )
        quit("Out of memory!\n");
    strncpy(ac->proto, proto, sizeof(ac->proto));

    ac->next = aconns;
    aconns = ac;
    ac->last_activity = now;
    ac->tcp_state = state;

    switch ( ((struct sockaddr *) src)->sa_family ) {

        case AF_INET:
            memcpy(&ac->src, src, sizeof(struct sockaddr_in));
            memcpy(&ac->dst, dst, sizeof(struct sockaddr_in));
            break;

        case AF_INET6:
            memcpy(&ac->src, src, sizeof(struct sockaddr_in6));
            memcpy(&ac->dst, dst, sizeof(struct sockaddr_in6));
            break;

        default:
            /* unsupported */
            return;
    }

found:

    ac->to_remove = 0;
    ac->bytes += bytes;

    if (!ac->bytes_old)
        ac->bytes_old = ac->bytes;
    
}

void
aconn_tcp_do_one(char *line)
{
    int num, d, local_port, dest_port, state, timer_run, uid, timeout;
    char local_addr[128], dest_addr[128], more[512];
    unsigned long rxq, txq, time_len, retr, inode;

    struct sockaddr_in6 localaddr, remaddr;

    memset(&localaddr, 0, sizeof(localaddr));
    memset(&remaddr, 0, sizeof(remaddr));

    num = sscanf(line, "%d: %64[0-9A-Fa-f]:%X %64[0-9A-Fa-f]:%X %X %lX:%lX %X:%lX %lX %d %d %ld %512s\n",
            &d, local_addr, &local_port, dest_addr, &dest_port, &state,
            &rxq, &txq, &timer_run, &time_len, &retr, &uid, &timeout, &inode, more);

    if (num < 11)
        return;

    if (strlen(local_addr) > 8) {
        /* XXX */

#if 0
        /* Demangle what the kernel gives us */
        sscanf(local_addr, "%08X%08X%08X%08X",
               &in6.s6_addr32[0], &in6.s6_addr32[1],
               &in6.s6_addr32[2], &in6.s6_addr32[3]);


        sscanf(dest_addr, "%08X%08X%08X%08X",
               &in6.s6_addr32[0], &in6.s6_addr32[1],
               &in6.s6_addr32[2], &in6.s6_addr32[3]);

        localaddr.sin6_family = AF_INET6;
        remaddr.sin6_family = AF_INET6;
#endif

    } else {
        sscanf(local_addr, "%X",
               &((struct sockaddr_in *) &localaddr)->sin_addr.s_addr);
        sscanf(dest_addr, "%X",
               &((struct sockaddr_in *) &remaddr)->sin_addr.s_addr);
        ((struct sockaddr *) &localaddr)->sa_family = AF_INET;
        ((struct sockaddr *) &remaddr)->sa_family = AF_INET;
    }

    if (state == TCP_LISTEN) {
        time_len = 0;
        retr = 0L; 
        rxq = 0L;
        txq = 0L;
    }

   ((struct sockaddr_in *) &localaddr)->sin_port = local_port;
   ((struct sockaddr_in *) &remaddr)->sin_port = dest_port;

   aconn_update("tcp", (struct sockaddr *) &localaddr,
        (struct sockaddr *) &remaddr, 0, state);
}

void
aconn_read_tcp(const char *file)
{
    FILE *f;
    char line[1024];
    i=1;

    if ( !(f = fopen(file, "r")) )
        return;

    memset(line, 0, sizeof(line));
    while ( fgets(line, sizeof(line), f) ) {
        aconn_tcp_do_one(line);
        memset(line, 0, sizeof(line));
    }

    fclose(f);
}


void
handle_port(int sp, int dp, char *saddr, char *daddr, int type,
            const struct pcap_pkthdr *hdr)
{
    struct aconn_s *ac;

    ac = aconns;
    while (ac) {

        /*
         * we need the protocol type as well to match XXX
         */
        switch (type) {
            case ETHERTYPE_IP: /* ip4 */
                if ( (sp == ((struct sockaddr_in *) &ac->src)->sin_port &&
                      dp == ((struct sockaddr_in *) &ac->dst)->sin_port &&
                      !memcmp(saddr, &((struct sockaddr_in *) &ac->src)->sin_addr, 4) &&
                      !memcmp(daddr, &((struct sockaddr_in *) &ac->dst)->sin_addr, 4) )
                    ||
                     (sp == ((struct sockaddr_in *) &ac->dst)->sin_port &&
                      dp == ((struct sockaddr_in *) &ac->src)->sin_port &&
                      !memcmp(saddr, &((struct sockaddr_in *) &ac->dst)->sin_addr, 4) &&
                      !memcmp(daddr, &((struct sockaddr_in *) &ac->src)->sin_addr, 4) ) )
                    goto found;
                break;

            case 0x86DD: /* ip6 */
                if ( (sp == ((struct sockaddr_in6 *) &ac->src)->sin6_port &&
                      dp == ((struct sockaddr_in6 *) &ac->dst)->sin6_port &&
                      !memcmp(saddr, &((struct sockaddr_in6 *) &ac->src)->sin6_addr, 16) &&
                      !memcmp(daddr, &((struct sockaddr_in6 *) &ac->dst)->sin6_addr, 16) )
                    ||
                     (sp == ((struct sockaddr_in6 *) &ac->dst)->sin6_port &&
                      dp == ((struct sockaddr_in6 *) &ac->src)->sin6_port &&
                      !memcmp(saddr, &((struct sockaddr_in6 *) &ac->dst)->sin6_addr, 16) &&
                      !memcmp(daddr, &((struct sockaddr_in6 *) &ac->src)->sin6_addr, 16) ) )
                    goto found;
                break;

            default:
                return;
        }
        
        ac = ac->next;
    }

    return;

found:

    ac->last_activity = now;
    ac->bytes += hdr->len;
    
}

void
aconn_print_connections(void)
{
    char addrbuf[INET6_ADDRSTRLEN];
    char serv1[256], serv2[256], *u;
    struct servent *s;
    float f = 0;
    struct aconn_s *ac;
    int sport, dport, i;
    time_t t;
    char dstr[30];

    print_top("Connections    f:Change Filter  s:Change Sort  i:Change Info");

    if (COLS < 80) {
        print_full_line("Need 80 chars wide screen");
        return;
    }

    if (LINES < 6) {
        print_full_line("Need at least 5 lines");
        return;
    }

    print_full_line("Show: %-12s Sort: %-12s Info: %-12s",
        m_conn_text[m_conn_show],
        sort_text[m_conn_sort],
        info_text[m_conn_info]);

    NEXT_ROW;

    print_full_line("");

    NEXT_ROW;

    switch (m_conn_info) {

        case i_rate:
            print_full_line("S  Src Addr                 Dst Addr                   SPort    DPort       Rate");
            break;

        case i_idle:
            print_full_line("S  Src Addr                 Dst Addr                   SPort    DPort       Idle");
            break;

        case i_total:
            print_full_line("S  Src Addr                 Dst Addr                   SPort    DPort      Total");
            break;
    }

    NEXT_ROW;
    hline(ACS_HLINE, COLS);


    for (i=0,ac = aconns; ac; ac = ac->next,i++) {

        switch (m_conn_show) {
            case c_all:
                break;

            case c_established:
                if (ac->tcp_state != 1)
                    continue;
                break;

            case c_listen:
                if (ac->tcp_state != 10)
                    continue;
                break;

            case c_time_wait:
                if (ac->tcp_state != 6)
                    continue;
                break;
        }

        if (row >= LINES-2)
            break;

        NEXT_ROW;

        memset(serv1, 0, sizeof(serv1));
        memset(serv2, 0, sizeof(serv2));

        switch ( ((struct sockaddr *) &ac->src)->sa_family ) {

            case AF_INET:
                sport = ((struct sockaddr_in *) &ac->src)->sin_port;
                dport = ((struct sockaddr_in *) &ac->dst)->sin_port;
                break;

            case AF_INET6:
                sport = ((struct sockaddr_in6 *) &ac->src)->sin6_port;
                dport = ((struct sockaddr_in6 *) &ac->dst)->sin6_port;
                break;

            default:
                continue;
        }

        if ( (s = getservbyport(htons(sport), NULL)) )
            strncpy(serv1, s->s_name, sizeof(serv1));
        else
            snprintf(serv1, sizeof(serv1), "%d", sport);

        serv1[8] = '\0';

        if ( (s = getservbyport(htons(dport), NULL)) )
            strncpy(serv2, s->s_name, sizeof(serv2));
        else
            snprintf(serv2, sizeof(serv2), "%d", dport);

        serv2[8] = '\0';


        switch ( ((struct sockaddr *) &ac->src)->sa_family ) {

            case AF_INET:

                if ( !ac->sname ) {
                    memset(addrbuf, 0, sizeof(addrbuf));

                    resolveipbyaddr(addrbuf, sizeof(addrbuf), 
                        (char *) &( (struct sockaddr_in *) &ac->src)->sin_addr,
                        sizeof(( (struct sockaddr_in *) &ac->src)->sin_addr), AF_INET);

                    if (!strcmp(addrbuf, "0.0.0.0"))
                        ac->sip = strdup("*");
                    else
                        ac->sip = strdup(addrbuf);
                    

                    if ( resolvebyaddr(addrbuf, sizeof(addrbuf), 
                            (char *) &( (struct sockaddr_in *) &ac->src)->sin_addr,
                            sizeof(( (struct sockaddr_in *) &ac->src)->sin_addr), AF_INET) ) {

                        ac->sname = strdup(addrbuf);
                    }

                }

                if ( !ac->dname ) {
                    memset(addrbuf, 0, sizeof(addrbuf));

                    resolveipbyaddr(addrbuf, sizeof(addrbuf), 
                        (char *) &( (struct sockaddr_in *) &ac->dst)->sin_addr,
                        sizeof(( (struct sockaddr_in *) &ac->dst)->sin_addr), AF_INET);

                    if (!strcmp(addrbuf, "0.0.0.0"))
                        ac->dip = strdup("*");
                    else
                        ac->dip = strdup(addrbuf);
                    

                    if ( resolvebyaddr(addrbuf, sizeof(addrbuf), 
                            (char *) &( (struct sockaddr_in *) &ac->dst)->sin_addr,
                            sizeof(( (struct sockaddr_in *) &ac->dst)->sin_addr), AF_INET) ) {

                        ac->dname = strdup(addrbuf);
                    }
                }

                switch (m_conn_info) {
                    
                    case i_rate:
                        f = sumup(ac->r_bytes, &u);
                        break;

                    case i_idle:
                        f = time_sumup( (now - ac->last_activity), &u);
                        break;

                    case i_total:
                        f = sumup(ac->bytes, &u);
                        break;
                }

                print_full_line("%s %-23s  %-23s %8s %8s %8.1f%s\n",
                        tcp_state[ac->tcp_state],
                        (do_resolve && ac->sname) ? ac->sname : ac->sip,
                        (do_resolve && ac->dname) ? ac->dname : ac->dip,
                        serv1, serv2,
                        f, u);

            case AF_INET6:
                break;

            default:
                continue;
        }
    }

    print_to_end();


    row = LINES-1;
    move(row,0);
    attrset(A_REVERSE);

    print_full_line(" nmon 0.3.5  [ResolveDNS: %s]",
        (do_resolve) ? "Yes" : "No");

    move(row,COLS-16);
    t = time(0);
    strftime(dstr, sizeof(dstr), "%b %d %H:%M:%S", localtime(&t));
    printw("%s", dstr);
    
    attroff(A_REVERSE);
}

int
aconn_handle_input(int ch)
{
    switch (ch) {

        case 's':
            switch (m_conn_sort) {
                case s_rate:
                    m_conn_sort = s_state;
                    break;

                case s_state:
                    m_conn_sort = s_srcip;
                    break;

                case s_srcip:
                    m_conn_sort = s_dstip;
                    break;

                case s_dstip:
                    m_conn_sort = s_srcport;
                    break;

                case s_srcport:
                    m_conn_sort = s_dstport;
                    break;

                case s_dstport:
                    m_conn_sort = s_rate;
                    break;

                default:
                    m_conn_sort = s_rate;
                    break;
            }
            return 1;

        case 'f':
            switch (m_conn_show) {
                case c_all:
                    m_conn_show = c_established;
                    break;
                case c_established:
                    m_conn_show = c_listen;
                    break;
                case c_listen:
                    m_conn_show = c_time_wait;
                    break;
                case c_time_wait:
                    m_conn_show = c_all;
                    break;
                default:
                    m_conn_show = c_all;
                    break;
            }
            return 1;

        case 'i':
            switch (m_conn_info) {
                case i_rate:
                    m_conn_info = i_idle;
                    break;

                case i_idle:
                    m_conn_info = i_total;
                    break;

                case i_total:
                    m_conn_info = i_rate;
                    break;

                default:
                    m_conn_info = i_rate;
                    break;
            }
            return 1;
    }

    return 0;
}

void
aconn_sort(void)
{
    struct aconn_s *p, *q, *e, *tail, *list;
    unsigned int insize, nmerges, psize, qsize, i, f;

    list = aconns;
    insize = 1;

    if (!list)
        return;

    while (1) {
        p = list;
        list = NULL;
        tail = NULL;

        nmerges = 0;

        while (p) {
            nmerges++;
            q = p;
            
            for (psize=0,i=0; i < insize; i++) {
                psize++;
                q = q->next;
                if (!q)
                    break;
            }

            for (qsize=insize; psize > 0 || (qsize > 0 && q);) {

                if (psize == 0) {
                    e = q;
                    q = q->next;
                    qsize--;
                } else if (qsize == 0 || !q) {
                    e = p;
                    p = p->next;
                    psize--;
                } else {

                    f = 0;

                    /*
                     * XXX:
                     * Note: mixed ip4 and ip6 can easly cause problems by overruns
                     * have to fix this
                     */

                    switch ( m_conn_sort ) {

                        case s_rate:
                            if (p->r_bytes >= q->r_bytes)
                                f = 1;
                            break;

                        case s_state:
                            if(p->tcp_state <= q->tcp_state)
                                f = 1;
                            break;

                        case s_srcip:
                            switch ( ((struct sockaddr *) &p->src)->sa_family ) {
                                case AF_INET:
                                    if ( ((struct sockaddr_in *) &p->src)->sin_addr.s_addr >=
                                         ((struct sockaddr_in *) &q->src)->sin_addr.s_addr )
                                        f = 1;
                                    break;

                                case AF_INET6:
                                    /* XXX */
                                    break;
                            }
                            break;

                        case s_dstip:
                            switch ( ((struct sockaddr *) &p->src)->sa_family ) {
                                case AF_INET:
                                    if ( ((struct sockaddr_in *) &p->dst)->sin_addr.s_addr >=
                                         ((struct sockaddr_in *) &q->dst)->sin_addr.s_addr )
                                        f = 1;
                                    break;

                                case AF_INET6:
                                    /* XXX */
                                    break;
                            }
                            break;

                        case s_srcport:
                            switch ( ((struct sockaddr *) &p->src)->sa_family ) {
                                case AF_INET:
                                    if ( ((struct sockaddr_in *) &p->src)->sin_port >=
                                         ((struct sockaddr_in *) &q->src)->sin_port )
                                        f = 1;
                                    break;

                                case AF_INET6:
                                    /* XXX */
                                    break;
                            }
                            break;

                        case s_dstport:
                            switch ( ((struct sockaddr *) &p->src)->sa_family ) {
                                case AF_INET:
                                    if ( ((struct sockaddr_in *) &p->dst)->sin_port >=
                                         ((struct sockaddr_in *) &q->dst)->sin_port )
                                        f = 1;
                                    break;

                                case AF_INET6:
                                    /* XXX */
                                    break;
                            }
                            break;
                    }

                    if ( f )  {
                        e = p;
                        p = p->next;
                        psize--;
                    } else {
                        e = q;
                        q = q->next;
                        qsize--;
                    }
                }

                if (tail)
                    tail->next = e;
                else
                    list = e;

                tail = e;
            }

            p = q;
        }

	    tail->next = NULL;

        if (nmerges <= 1) {
            aconns = list;
            return;
        }

        insize *= 2;
    }
}
