/*
 *   mwmipc.c -- Mwave Modem AT Command Parser
 *
 *  Written By: Paul Schroeder IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <mwmspcfc.h>
#include <unidiagx.h>
#include <mwmcntnd.h>
#include <mwblapi.h>
static char szThisFile[] = "MWMIPC.C";


// Calls to begin and end time critical sections are used to lower the execution
// priority of apps 
//
// TIME_CRITCAL_TIMEOUT is in milliseconds

#define TIME_CRITICAL_TIMEOUT  20000L
#define BEGINTIMECRITICALSECTION
#define ENDTIMECRITICALSECTION

 

ULONG MWM_ENTRY mwmIPC(PMWM_DSPINFO pmwmDspInfo, ULONG ulParm1, ULONG ulParm2 )
{
  ULONG   ulRC;

  USHORT       usTempDRVRSTAT;  // MTS 2416 ... used in case 15...
  USHORT       usTempValue = 0;
  ULONG        ulAddress = 0;
  static ULONG ulDialHandle;      // used in new blacklist
  static ULONG ulBlcklistHandle;  // used in new blacklist
  ULONG        ulDialStatus;      // used in new blacklist
  USHORT       usBlackListStatus = 0;
  USHORT       usParserMode = 0;
  REGISTERS    Registers;
  char         achNumberDialed[AT_CMD_BUFFER_SIZE]; // DCR 3444
  ULONG        ulAddressSTATUS = 0;
  MWM_MODEM_STATUS ModemStatus;
  ULONG        ulTempFeatures = 0;
  MEIO_TAIOCONNARG MeioReturnedInfo; /* mts 5224 */
  USHORT       usFCLASS;
  USHORT       usCurrentVLS;
  USHORT       usHookStatus;
  USHORT       usHandsetStatus;
  USHORT       usFaxCmd;
  USHORT       usFaxXrat;
  USHORT       usUIMessage = 0;
  USHORT       usUseConnectSpeed = 0;
  BOOL         bCountryNumberIsOK = 0;
  ULONG        ulMeioData = 0;         // init since older MEIOs don't
  ULONG        ulMeioDataSize = sizeof (ulMeioData);
  ULONG        ulDelayTime=0;

  MWM_FCLASS_SUBCLASS  ClassStruct; //$1
  
  { MW_SYSLOG_3(TRACE_MWMLW32,"mwmipc::mwmIPC entry ulParm1 %lx ulParm2 %lx\n",ulParm1,ulParm2);  }
  
  if (pmwmDspInfo->bSuspending) {
    return(0);
  } 
  
  switch (ulParm1) {
  case 6:   // used in standard ModemToLineSpeed case 
  case 70:               // used only in %TT command. 
    
    if ((pmwmDspInfo->ulFeaturesToLoad & FEATURE_V34) |
	(pmwmDspInfo->ulFeaturesToLoad & FEATURE_V32TER) |
	(pmwmDspInfo->ulFeaturesToLoad & FEATURE_V32BIS) ) {
      // Kick everyone out now.  Chances are we will have to kick them out 
      // later anyway.  This gives us a better chance at preventing memory 
      // fragmentation problems                                            
      mwmCntndContend(pmwmDspInfo, 0);
    }
    
    if (ulParm1 == 6) {
      ulRC = mwmLoadDpCyclesToLineSpeed(pmwmDspInfo,0); //$8
    } else {
      // This is the IPC 70.  It is only used in the %TT command to keep   
      //  from loading the protocol tasks even if \n is non-zero.           
      ulRC = mwmLoadDpCyclesToLineSpeed(pmwmDspInfo,1);   //$8
    } 
    if (ulRC != DSP_NOERROR) {
      return ulRC;
    }
    
    if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_GETSIZE) {
      if (ulParm1 == 6) {
	(*(pfnWriteDSPSnapshot))(pmwmDspInfo->hDSP,"linspeed.snp");
      } else {
	// used only in the %TT case.                                      
	(*(pfnWriteDSPSnapshot))(pmwmDspInfo->hDSP,"perctt.snp");
      }
    }
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 13: // ECC Protocols Selected 
    UniDiagsSaveProtocolInfo(pmwmDspInfo);       // Save info for #ud command
    UniDiagsSaveDataFlowInfo(pmwmDspInfo);       // Save info for #ud command
    
    //This interrupt is received when the modem has negotiated which ECC  
    // protocols will be used.                                             
    // We will move GPC connections, and possibly load/unload necessary    
    // modules.                                                            
    if ((pmwmDspInfo->ulFeaturesToLoad & FEATURE_V34) |
	(pmwmDspInfo->ulFeaturesToLoad & FEATURE_V32TER) |
	(pmwmDspInfo->ulFeaturesToLoad & FEATURE_V32BIS) ) {
      // Kick everyone out now.  Chances are we will have to kick them out 
      // later anyway.  This gives us a better chance at preventing memory 
      // fragmentation problems                                            
      mwmCntndContend(pmwmDspInfo, 0);
    }
    
    ulRC = mwmLoadCarrierDetectToProtocol(pmwmDspInfo);
    
    // If we failed to allocate enough memory for V.42bis, don't create  
    // a pop-up.  Simply report NO CARRIER.  Chances are good that more  
    // memory will be available if we try to connect again.              
    
    if (ulRC != DSP_NOERROR  &&
	ulRC != DSP_INSUFF_DATA_MEM )
      return ulRC;
    
    if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_GETSIZE) {
      (*(pfnWriteDSPSnapshot))(pmwmDspInfo->hDSP,"protocol.snp");
    }
    
    if ( ulRC == DSP_INSUFF_DATA_MEM ) {
      mwmParseSetDRVRSTAT(10);
      DPF("V42BIS ERROR: Insufficient data memory available");
      ulRC=DSP_NOERROR;              // ignore this error 
    } else {
      mwmParseSetDRVRSTAT(2);
    }
    break;
    
  case 15:  //Load Modem to DSP 
    { MW_SYSLOG_1(TRACE_MWMLW32,"mwmipc::mwmipc, handling case 0x0f\n");  }
    // Save off the registers in case we are suspended during a call
    ulRC = mwmParseQueryModemRegisters(&(pmwmDspInfo->SavedRegisters));
    if(ulRC)
      return ulRC;
    
    pmwmDspInfo->bCarrierDetected = FALSE;
    pmwmDspInfo->bSuspendingDuringCall = TRUE;
    
    if (pmwmDspInfo->usMEIOToggle) {
      ulRC = dspMeioQuery( hMEIO, QADAPTER, &ulMeioDataSize, &ulMeioData);
      if (!ulRC && !(ulMeioData & MACF_V96)) {
	pmwmDspInfo->hConnect3 = 0;
	pmwmDspInfo->hConnect4 = 0;
	ulRC = mwmCntndMEIOConnect(&pmwmDspInfo->hConnect3,CDDAC_S1,LINEOUT_S1,
				   0,ON, 0,
				   pmwmDspInfo,
				   MWM_STR_PHONE_LINE_IN_USE, szThisFile,__LINE__);
	if (ulRC)
	  return ulRC;
	
	ulRC = dspMeioConnect(hMEIO,&pmwmDspInfo->hConnect4, CDDAC_S1,INTSPKROUT_L1,
			      0,ON,0);
      } 
    } 
    
    ulRC = mwmParseQueryParserMode(&usParserMode);
    if (ulRC)
      return ulRC;
    
    // MTS 5151 Move S28/Com Port Max Speed selection to Driver            
    ulRC = mwmSpcfcSetMaxS28ComPortSpeed(pmwmDspInfo);
    if (ulRC)
      return ulRC;
    
    // Initialize V.34 Enhanced Mode Flags and determine if the
    // conditions are there for a V.34 Enhanced Mode connection.
    pmwmDspInfo->bV34EMResponseReceived = FALSE;
    pmwmDspInfo->bV34EMConnectionEstablished = FALSE;
    ulRC = mwmSpcfcIsV34EMConnectionPossible(pmwmDspInfo);
    if (ulRC)
      return ulRC;
    
    pmwmDspInfo->bPCMResponseReceived = FALSE;
    pmwmDspInfo->usPCMConnection = 0;
    pmwmDspInfo->bFirstIPC83Received = FALSE;
    ulRC = mwmSpcfcIsPCMConnectionPossible(pmwmDspInfo);
    if (ulRC)
      return ulRC;
    
    // Get and store the QuoteN value in case we change
    // it during the call.
    if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_PCM) {
      ulRC = mwmParseQueryModemRegisters(&Registers);
      if (ulRC)
	return (ulRC);
      pmwmDspInfo->usQuoteNValue = Registers.QuoteN;
    } 
    
    if ( pmwmDspInfo->ulFeaturesToLoad & FEATURE_SLIMBASE) {    // MTS=SB
      ulRC = mwmLoadDiscrimToDial(pmwmDspInfo);
      if (ulRC != DSP_NOERROR) {
	return ulRC;
      }
    }
    
    // Set the DRVRSTAT=2 value here...so that MCTL can go ahead and dial. 
    mwmParseSetDRVRSTAT(2);
    
    // Unload the V8/V21 pair                     MTS 7757 - 02/12/97 DR
    if (pmwmDspInfo->bV80HandsetSnoop) {
      ulRC = mwmV80LV8V21Unload(pmwmDspInfo);
      if (ulRC) {
	return(ulRC);
      } 
    } 
    // Get the Class/Subclass                                        $7    
    ulRC = mwmFaxGetFCLASS(pmwmDspInfo,&ClassStruct.usFCLASS, &ClassStruct.usSubclass);
    if (ulRC)
      return (ulRC);
    
    if ( pmwmDspInfo->ulFeaturesToLoad & FEATURE_SLIMBASE) {    // MTS=SB
      switch(ClassStruct.usFCLASS) {                                     //$1
      case 0:
	if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_V80) {             //$1
	  // Check SUBCLASS to see which mode we are in...          //$2 begin 
          
	  if (ClassStruct.usSubclass) {                              //$1
	    ulRC = mwmV80LDialToV80nPumps(pmwmDspInfo);            //$1 $8
	    if (ulRC != DSP_NOERROR) {                               //$1
	      return ulRC;                                         //$1
	    }                                                      //$1
	  } else {                                                   //$1
	    ulRC = mwmLoadDialToModem(pmwmDspInfo);                //$1
	    if (ulRC != DSP_NOERROR) {                               //$1
	      return ulRC;                                         //$1
	    }                                                      //$1
	  }                                              //$1
	} else {                                                     //$1
	  ulRC = mwmLoadDialToModem(pmwmDspInfo);
	  if (ulRC != DSP_NOERROR) {
	    return ulRC;
	  }
	} 
	break;
      case 1:
	ulRC = mwmFaxDialToClass1(pmwmDspInfo);
	if (ulRC != DSP_NOERROR) {
	  return ulRC;
	}
	break;
      case 2:
	ulRC = mwmFaxDialToClass2(pmwmDspInfo);
	if (ulRC != DSP_NOERROR) {
	  return ulRC;
	}
	break;
      default:
	break;
      }
    }
    else
      {
        // Set LOADDONE flag                                                
        // This is done in  DialToModem, so it is not executed in the non   
        // discriminator state.                                             
        ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "LOADDONE", &ulAddress);
        if (ulRC!=DSP_NOERROR) {
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC);
          return(ulRC);
        }
	
        usTempValue = 1;
	
        ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                              &usTempValue, 1,
                              DSP_MEMXFER_DATA_WRITE);
        if (ulRC!=DSP_NOERROR) {
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC);
          return(ulRC);
        }
      }
    
    pmwmDspInfo->ulDiscrimState = 0;
    
    // For Class 1 or Class 2, we're done loading everything,
    // so give bumped tasks the opportunity to get back in.
    if ( ClassStruct.usFCLASS == 1 || ClassStruct.usFCLASS == 2 ) {
      mwmCntndLoadComplete(pmwmDspInfo, 1);
    } 
    
    mwmParsePostMessage(WM_MWM_DISCRIM_STATUS, FALSE, 0);
    break;
    
    
  case 16:  // Report Dial Status to Blacklisting Driver 
    // Also, move to discrim state if the      
    // discriminator is running.               
    { MW_SYSLOG_1(TRACE_MWMLW32,"mwmipc::mwmipc, handling case 0x10\n");  }
    UniDiagsSaveSetupInfo(pmwmDspInfo);
    UniDiagsSaveCallSetupResultInfo(pmwmDspInfo);// Save info for #ud command
    UniDiagsUpdateCallStatus(pmwmDspInfo);       // Save info for #ud command
    UniDiagsFinalCallStatus(pmwmDspInfo);
    pmwmDspInfo->bCarrierDetected = FALSE;
    
    pmwmDspInfo->bSuspendingDuringCall = FALSE;
    
    if (pmwmDspInfo->usMEIOToggle) {
      if (pmwmDspInfo->hConnect3) {
	ulRC = dspMeioDisconnect(pmwmDspInfo->hConnect3, 0);
	if (ulRC != MEIO_NOERROR) {
	  mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
			 MWM_MEIO_ERROR, ulRC);
	  return(ulRC);
	} 
	pmwmDspInfo->hConnect3 = 0;
      } 
      
      if (pmwmDspInfo->hConnect4) {
	ulRC = dspMeioDisconnect(pmwmDspInfo->hConnect4, 0);
	if (ulRC != MEIO_NOERROR) {
	  mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
			 MWM_MEIO_ERROR, ulRC);
	  return(ulRC);
	} 
	pmwmDspInfo->hConnect4 = 0;
      } 
    } 
    
    // Check to see if the cause of the hangup was an
    // over-current/over-voltage condition.
    if (pmwmDspInfo->bOverVoltageTestingEnabled) {
      ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "OVER_VOLT_ERROR", &ulAddress);
      if (ulRC != DSP_NOERROR) {
	if (ulRC != DSP_LABEL_NOT_FOUND) {
	  mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
			 MWM_DSP_ERROR, ulRC);
	  return(ulRC);
	} else {
	  DPF("OVER_VOLT_ERROR label not found in modem control!");
	} 
      } else {
	ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
			      &usTempValue, 1, DSP_MEMXFER_DATA_READ);
	if (ulRC!=DSP_NOERROR) {
	  mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
			 MWM_DSP_ERROR, ulRC);
	  return(ulRC);
	} 
	
	if (usTempValue) {
	  mwmParsePostMessage(WM_MWM_VOLTAGE_ERROR, 0, 0);
	  
	  // Clear the flag 
	  usTempValue = 0;
	  
	  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
				&usTempValue, 1, DSP_MEMXFER_DATA_WRITE);
	  if (ulRC!=DSP_NOERROR) {
	    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
			   MWM_DSP_ERROR, ulRC);
	    return(ulRC);
	  } 
	} 
      } 
    } 
    
    // Restore the QuoteN value in case we changed it
    // during the call.
    if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_PCM) {
      mwmParseSetQuoteN(pmwmDspInfo->usQuoteNValue);
    } 
    
    // Need to add report of call status here....                         
    //  MTS 3444                                                          
    //  if the previous dial wasn't blacklisted...                        
    //    -Query the call status.                                         
    //    -Tell the blacklisting code what happened on this call.         
    //    -Clear the wtDialHandle for the next dial.                      
    if (ulDialHandle > 0) {
      // Query the call status                                            
      ulRC = mwmParseQueryBlacklistStatus(&usBlackListStatus);
      if (ulRC)
	return ulRC;
      
      switch(usBlackListStatus) {
      case 1:
	ulDialStatus = USER_ABORT;
	break;
      case 2:
	ulDialStatus = SUCCESSFUL;
	break;
      case 3:
	ulDialStatus = NO_CONNECT;
	break;
      case 4:
	ulDialStatus = NO_DIAL_TONE;
	break;
      case 5:
	ulDialStatus = LINE_BUSY;
	break;
      default:
	ulDialStatus = NO_CONNECT;
	break;
      }
      
      // Tell the blacklisting server what happened..                      
      ulRC=BL_DialResults(ulBlcklistHandle,
			  ulDialHandle,
			  ulDialStatus,
			  0L); 
      
      if (ulRC) {
	mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		       MWM_BL_ERROR,
		       ulRC );
	return ulRC;
      }
      
      ulDialHandle = 0;
    }
    
    ulRC = mwmParseQueryParserMode(&usParserMode);
    if (ulRC)
      return ulRC;
    
    // Move to the discriminator state.                                    
    // if we are in Class 2 mode.                                          
    if ( pmwmDspInfo->ulFeaturesToLoad & FEATURE_SLIMBASE ) {   // MTS=SB
      if (usParserMode & MWM_MODE_CLASS2_FAX) {
	// Move to the dial state...                                       
	// Remove the class 2 function...leaves only base + call progress..
	ulRC = mwmFaxClass2ToDial(pmwmDspInfo);
	if (ulRC != DSP_NOERROR) {
	  return ulRC;
	}
	
	ulRC = mwmLoadDialToDiscrim(pmwmDspInfo);
	if (ulRC != DSP_NOERROR) {
	  return ulRC;
	}
	pmwmDspInfo->ulDiscrimState = 1;
      }
    }
    
    // Discriminator or not, if not class 2, unload the code...            
    if (!(usParserMode & MWM_MODE_CLASS2_FAX)) {
      BEGINTIMECRITICALSECTION;
      
      ulRC = mwmAnywhereToDiscrim(pmwmDspInfo);
      
      ENDTIMECRITICALSECTION;
      
      if (ulRC)
	return ulRC;
    }
    
    { MW_SYSLOG_2(TRACE_MWMLW32,"mwmipc::mwmipc, ulFeaturesToLoad %lx\n",pmwmDspInfo->ulFeaturesToLoad);  }
    
    if ( ! (pmwmDspInfo->ulFeaturesToLoad & FEATURE_SLIMBASE) ) {    // MTS=SB
      // We are not unloading Class 2 stuff, so we don't need to reload it 
      // here.                                                             
      
      // Get the Class/Subclass                                        $7    
      ulRC = mwmFaxGetFCLASS(pmwmDspInfo,&ClassStruct.usFCLASS, &ClassStruct.usSubclass);
      if (ulRC)
	return (ulRC);
      
      // Check FCLASS to see which mode we are in...                       
      if (ClassStruct.usFCLASS == 0) {                                 //$1
	ulRC = mwmLoadDiscrimToDial(pmwmDspInfo);
	if (ulRC)
	  return ulRC;
	if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_V80) {                //$1
	  if (ClassStruct.usSubclass) {
	    ulRC = mwmV80LDialToV80nPumps(pmwmDspInfo);               //$8
	    if (ulRC != DSP_NOERROR) {
	      return ulRC;
	    }
	    
	    ulRC = mwmV80LCyclesAdjustToLineSpeed(pmwmDspInfo, 0, 0); //$6 $8 $9
	    if (ulRC != DSP_NOERROR)
	      return ulRC;
	  } else {
	    ulRC = mwmLoadDialToModem(pmwmDspInfo);
	    if (ulRC != DSP_NOERROR) {
	      return ulRC;
	    }
	  } 
	} else {
	  ulRC = mwmLoadDialToModem(pmwmDspInfo);
	  if (ulRC != DSP_NOERROR) {
	    return ulRC;
	  }
	} 
      }
      else if (ClassStruct.usFCLASS == 1) {                            //$1
	ulRC = mwmLoadDiscrimToDial(pmwmDspInfo);
	if (ulRC)
	  return ulRC;
	
	ulRC = mwmFaxDialToClass1(pmwmDspInfo);
	if (ulRC)
	  return ulRC;
	
      } else if (ClassStruct.usFCLASS == 80) {                           //$1
	ulRC = mwmLoadDiscrimToDial(pmwmDspInfo);
	if (ulRC)
	  return ulRC;
      }
    }
    
    pmwmDspInfo->usDialOrAnswerFlag = ANSWER;
    
    mwmParsePostMessage(WM_MWM_UPDATE_STATUS, MWM_STATE_READY, 0);
    
    mwmParsePostMessage( WM_MWM_DISCRIM_STATUS, TRUE, 0);
    
    mwmParseSetDRVRSTAT(2);
    
    pmwmDspInfo->usContendRetry = 1;                       
    
    BEGINTIMECRITICALSECTION;
    
    mwmCntndLoadComplete(pmwmDspInfo, 1);                   
    
    ENDTIMECRITICALSECTION;
    break;
    
  case 17:
    // 11/29/94                                                            
    // "Deregister" after timeout on ATA                                   
    // This interrupt is received only when we timed out after an ATA      
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 19: // Autoanswer state may be changing. 
    // This is a change of S0...perhaps a change in ring count  
    //            or autoanswer.                                           
    break;
    
  case 20:
    //MCTL has received an ATA command.                                   
    // See if the line is being used by another Mwave app.                 
    //                                                                     
    // If not...                                                           
    // Artificially register with a ring count of 1 to get the call as     
    // quickly as possible.                                                
    usTempDRVRSTAT = 2;
    pmwmDspInfo->usWTChangeDRVRSTATValue = 0;
    
    UniDiagsClearCallStatus();       // Save info for #ud command
    mwmParseSetWTChangeFlag(0);
    
    // For earlier NT builds dynamically changing the country has not been
    // implemented so this INI/REG settting will allow for the old method.
    if(GetPrivateProfileInt("OPTIONS","WTCHANGEOPT", 0,  INIFile)) {
      ulRC = WTT_CheckCountryNumber(NULL, &bCountryNumberIsOK, FALSE, 0L);
    } else
      ulRC = WTT_CheckCountryNumber(NULL, &bCountryNumberIsOK, TRUE, 0L);
    
    if (ulRC) {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    } else if (!bCountryNumberIsOK) {
      DPF("Country is not OK!");
      if(GetPrivateProfileInt("OPTIONS","WTCHANGEOPT", 0,  INIFile)) {
	mwmParseSetWTChangeFlag(1);
	pmwmDspInfo->usWTChangeDRVRSTATValue = 14;
	usTempDRVRSTAT = 0;
      } else
	usTempDRVRSTAT = 5;
    }
    
    if (usTempDRVRSTAT==2) {
      // Determine dialing or answering, because V.23 is not available      
      // when answering.                                                     
      pmwmDspInfo->ulDialingModem=0;
      
      ulRC = mwmParseQueryParserMode(&usParserMode);
      if (ulRC)
	return ulRC;
      
      // Not discriminator....check to see if the coupler ID matches.      
      if (dspMeioQueryConnection( pmwmDspInfo->hConnect1,
				  MEIO_CONNECTION_CHARACTERISTICS,
				  &MeioReturnedInfo.lArg ) == MEIO_NOERROR) {
	if (MeioReturnedInfo.sVal.CouplerMismatch == 1) { // wrong phone
	  usTempDRVRSTAT=5;
	    DPF("Coupler/Country Mismatch!\n");
	}
      }
    }
    if(usTempDRVRSTAT)
      mwmParseSetDRVRSTAT(usTempDRVRSTAT);
    break;
    
  case 21:
    // Enter FAX Connected state.                                          
    mwmParsePostMessage(WM_MWM_UPDATE_STATUS, MWM_STATE_FAXSEND,
			MWM_STATUS_USE_FAX_PROTOCOL);
    
    // NOTE!!! For this interrupt, modem control is NOT waiting for
    // DRVSTAT to be set by the driver. The AT commands being processed
    // by modem control are very time-sensitive, and waiting for
    // DRVSTAT after the interrupt would hurt performance.
    // Therefore, we will not set DRVSTAT so that we do not cause
    // problems with any subsequent interrupts.
    // mwmParseSetDRVRSTAT(2);
    
    break;
    
  case 22:
    // Leave FAX Connected state.                                          
    mwmParsePostMessage(WM_MWM_UPDATE_STATUS, MWM_STATE_READY, 0);
    
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 23:
    // Load the %TT Code Segment.                                          
    if (!pmwmDspInfo->hsegPercentTT) {
      // Get the Address of the %TT Segment Pointer.                       
      ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "ADRTTSEG", &ulAddress);
      if (ulRC!=DSP_NOERROR) {
	mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		       MWM_DSP_ERROR, ulRC);
	return(ulRC);
      }
      
      // Load the %TT Segment                                              
      ulRC = mwmCntndLoadSegment(pmwmDspInfo,
				 "MAINFILE",
				 "MDMTTSEG",
				 "MDMTTSEG",
				 pmwmDspInfo->hmctlTask,
				 &pmwmDspInfo->hsegPercentTT,
				 ulAddress,
				 szThisFile , __LINE__, MWM_MUSTHAVE );
      if (ulRC)
	return (ulRC);
    }
    
    pmwmDspInfo->bPercentTTMode = TRUE;
    
    pmwmDspInfo->usDynaSegmValue = 1;
    
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 24:
    // Unload the %TT Segment.                                             
    if (pmwmDspInfo->hsegPercentTT) {
      ulRC = dspFreeSegment(pmwmDspInfo->hsegPercentTT);
      if (ulRC) {
	mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		       MWM_DSP_ERROR, ulRC);
	return(ulRC);
      }
      pmwmDspInfo->hsegPercentTT = 0;
    }
    
    pmwmDspInfo->bPercentTTMode = FALSE;
    
    // Restore DYNASEGM value from INI file.  
    pmwmDspInfo->usDynaSegmValue = GetPrivateProfileInt("FEATURES",
							"DYNASEGM", 0,
							INIFile);
    
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 25:
    ulRC = mwmSpcfcQueryModemStatus(pmwmDspInfo,&ModemStatus,&ulAddressSTATUS);
    if (ulRC!=DSP_NOERROR) {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }
    
    mwmCntndLoadComplete(pmwmDspInfo, 1);                    
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 26:
    ulRC = mwmSpcfcQueryModemStatus(pmwmDspInfo,&ModemStatus,&ulAddressSTATUS);
    if (ulRC!=DSP_NOERROR) {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }
    
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 29:
    // Enter FAX Connected state.                                          
    mwmParsePostMessage(WM_MWM_UPDATE_STATUS, MWM_STATE_FAXRECEIVE,
			MWM_STATUS_USE_FAX_PROTOCOL);
    
    // NOTE!!! For this interrupt, modem control is NOT waiting for
    // DRVSTAT to be set by the driver. The AT commands being processed
    // by modem control are very time-sensitive, and waiting for
    // DRVSTAT after the interrupt would hurt performance.
    // Therefore, we will not set DRVSTAT so that we do not cause
    // problems with any subsequent interrupts.
    // mwmParseSetDRVRSTAT(2);
    
    break;
    
    
  case 30:
    // Request permission to Dial                                          
    //  - Check to see if the line is in use.                              
    //  - Check for Blacklisting                                           
    { MW_SYSLOG_1(TRACE_MWMLW32,"mwmipc::mwmipc, handling case 0x1e\n");  }
    usTempDRVRSTAT=2;
    pmwmDspInfo->usWTChangeDRVRSTATValue = 0;
    
    UniDiagsClearCallStatus();       // Save info for #ud command
    mwmParseSetWTChangeFlag(0);
    
    // For earlier NT builds dynamically changing the country has not been
    // implemented so this INI/REG settting will allow for the old method.
    if(GetPrivateProfileInt("OPTIONS","WTCHANGEOPT", 0,  INIFile)) {
      ulRC = WTT_CheckCountryNumber(NULL, &bCountryNumberIsOK, FALSE, 0L);
    } else
      ulRC = WTT_CheckCountryNumber(NULL, &bCountryNumberIsOK, TRUE, 0L);
    
    if (ulRC) {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    } else if (!bCountryNumberIsOK) {
      DPF("Country is not OK!");
      if(GetPrivateProfileInt("OPTIONS","WTCHANGEOPT", 0,  INIFile)) {
	mwmParseSetWTChangeFlag(1);
	pmwmDspInfo->usWTChangeDRVRSTATValue = 12;
	usTempDRVRSTAT = 0;
      } else
	usTempDRVRSTAT = 9;
    }
    
    if (usTempDRVRSTAT == 2) {
      // Determine dialing or answering, because V.23 is not available      
      // when answering.                                                    
      pmwmDspInfo->ulDialingModem=1;
      
      ulRC = mwmParseQueryNumberDialed(achNumberDialed);
      if (ulRC)
	return ulRC;
      
      // Get the Class/Subclass                                        $7    
      ulRC = mwmFaxGetFCLASS(pmwmDspInfo,&ClassStruct.usFCLASS, &ClassStruct.usSubclass);
      if (ulRC)
	return (ulRC);
      
      // Check FCLASS to see which mode we are in...                       
      ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "FCLASS", &ulAddress);
      if (ulRC!=DSP_NOERROR) {
	mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		       MWM_DSP_ERROR, ulRC);
	return(ulRC);
      }
      
      // Not discriminator....check to see if the coupler ID matches.      
      if (dspMeioQueryConnection( pmwmDspInfo->hConnect1,
				  MEIO_CONNECTION_CHARACTERISTICS,
				  &MeioReturnedInfo.lArg ) == MEIO_NOERROR) {
	if (MeioReturnedInfo.sVal.CouplerMismatch == 1)  { // wrong phone     
	  usTempDRVRSTAT=5;
	  DPF("Coupler/Country Mismatch!\n");
	}
      }
      
      
      // Check here to see if the number has already been blacklisted...     
      if (usTempDRVRSTAT == 2) {
	// Only do the blacklisting stuff if the blacklister is active, and  
	// if the number to dial has been filled in.                         
	// If the achNumberDialed string is empty, then we can assume this   
	// is an answer instead of a dial.                                   
	
	// Check the class and request dial permission                     
	if (strlen(achNumberDialed)) {
	  if ((ClassStruct.usFCLASS == 1) || (ClassStruct.usFCLASS == 2))
	    ulBlcklistHandle = pmwmDspInfo->ulBlcklistFaxHandle;
	  else if ((ClassStruct.usFCLASS == 8))
	    ulBlcklistHandle = pmwmDspInfo->ulBlcklistVoiceHandle;
	  else
	    ulBlcklistHandle = pmwmDspInfo->ulBlcklistModemHandle;
	  
	  ulRC=BL_RequestDialPermission(ulBlcklistHandle, AUTOMATIC_DIAL,
					achNumberDialed, &ulDialStatus,
					&ulDialHandle, &ulDelayTime, 0L);

	  if (ulRC != BL_SUCCESSFUL) {
	    mwmHandleError(pInstanceInfo, szThisFile, __LINE__,
			   MWM_BL_ERROR, ulRC);
	    return ulRC;
	  }
	  
	  switch(ulDialStatus) {
	  case BL_DIAL_GRANTED:
	    break;
	  case BL_NUMBER_DELAYED:
	    usTempDRVRSTAT = 7;
	    break;
	  default :
	    usTempDRVRSTAT = 4;     // blacklisted
	  }
	}
      }
      
      
      // if everything above this succeeded, then set the user               
      // message to "Dialing"...                                             
      if (usTempDRVRSTAT == 2) {
	mwmParsePostMessage(WM_MWM_UPDATE_STATUS, MWM_STATE_DIALING, 0);
      }
      
      // Clear the number Dialed variable.                                   
      ulRC = mwmParseSetNumberDialed("\0");
      if (ulRC)
	return ulRC;
    }
    if(usTempDRVRSTAT)
      mwmParseSetDRVRSTAT(usTempDRVRSTAT);
    break;
    
    
  case 31:
    UniDiagsSaveSetupInfo(pmwmDspInfo);       // Save info for #ud command
    UniDiagsSaveOctets(pmwmDspInfo);          // Save info for #ud command
    
    // Do DP CPF Adjustment to "startup"                                   
    ulRC = mwmLoadModemToDpCycles(pmwmDspInfo);   //$8
    if (ulRC)
      return ulRC;
    
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 32:
    pmwmDspInfo->bCarrierDetected = TRUE;
    UniDiagsClearUARTCounters(pmwmDspInfo);
    UniDiagsSaveConnectSpeed(pmwmDspInfo); // Save info for #ud command
    UniDiagsSaveLineInfo(pmwmDspInfo);     // Save info for #ud command
    UniDiagsSaveCarrierInfo(pmwmDspInfo);  // Save info for #ud command
    
    // Carrier has just been detected...Get rid of C96, and get MNP ready  
    // for service....                                                     
    ulRC = mwmLoadLineSpeedToCarrierDetect(pmwmDspInfo);
    if (ulRC != DSP_NOERROR) {
      return ulRC;
    }
    
    if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_GETSIZE) {
      (*(pfnWriteDSPSnapshot))(pmwmDspInfo->hDSP,"carrier.snp");
    }
    
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 33:
    // This IPC is caused by the -sse command.                            
    ulRC = mwmParseQueryModemRegisters(&Registers);
    if (ulRC)
      return (ulRC);
    
    // Make a temporary copy of FeaturesToLoad.                            
    // Copy current LongTermFeatures to FeaturesToLoad.                    
    // SetFeatures specifying LongTerm (NOT Temporary).                    
    // Copy TempFeatures back to FeaturesToLoad.                           
    // SetFeatures specifying Temporary to regain Temporary settings with  
    ulTempFeatures = pmwmDspInfo->ulFeaturesToLoad;
    pmwmDspInfo->ulFeaturesToLoad = pmwmDspInfo->ulLongTermFeatures;
    
    ulRC = mwmSpcfcSetFeatures(pmwmDspInfo, 0);   /* tell parser and mctl about change */
    if (ulRC)
      return (ulRC);
    
    pmwmDspInfo->ulFeaturesToLoad = ulTempFeatures;
    ulRC = mwmSpcfcSetFeatures(pmwmDspInfo, 1);   /* tell parser and mctl about change */
    if (ulRC)
      return (ulRC);
    break;
    
    
  case 34:
    // Query status so that we can check the new line speed.             
    ulRC = mwmSpcfcQueryModemStatus(pmwmDspInfo,&ModemStatus,&ulAddressSTATUS);
    if (ulRC!=DSP_NOERROR) {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }
    
    ulRC = mwmcmsgWriteConnectMessage(pmwmDspInfo,&ModemStatus);
    if (ulRC)
      return (ulRC);
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 35:
    // Guess How much V.42bis dictionary would be available.              
    //  - allocate and leave allocated as much as we can get              
    //    (up to the max configured by AT Commands...)                    
    if ((pmwmDspInfo->ulFeaturesToLoad & FEATURE_V34) |
	(pmwmDspInfo->ulFeaturesToLoad & FEATURE_V32TER) |
	(pmwmDspInfo->ulFeaturesToLoad & FEATURE_V32BIS) ) {
      // Kick everyone out now.  Chances are we will have to kick them out 
      // later anyway.  This gives us a better chance at preventing memory 
      // fragmentation problems                                            
      mwmCntndContend(pmwmDspInfo, 0);
    }
    ulRC = mwmPoolGuessMaxV42bisEntries(pmwmDspInfo);
    if (ulRC)
      return ulRC;
    
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 36:
    ulRC = mwmFaxReceivedFCLASS0(pmwmDspInfo);
    if (ulRC)
      return ulRC;
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 37:
    ulRC = mwmFaxReceivedFCLASS1(pmwmDspInfo);
    if (ulRC)
      return ulRC;
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 38:
    ulRC = mwmFaxReceivedFCLASS2(pmwmDspInfo);
    if (ulRC)
      return ulRC;
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 39:
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 40:
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 41: //$1
    ulRC = mwmV80LReceivedSubclassV80(pmwmDspInfo); //$8
    if (ulRC)
      return ulRC;
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 42:
    // Get the Subclass to see if we're in v.80 mode or not.  We may have  
    // already switched out of v.80 mode if a #CLS or +FCLASS command was  
    // received.                                                         $7
    ulRC = mwmV80LGetSUBCLASS(pmwmDspInfo, &ClassStruct.usSubclass);   //$8
    if (ulRC)
      return (ulRC);
    
    if (ClassStruct.usSubclass == 1) {  // if we're in v.80 mode
      ulRC = mwmFaxClassXToDiscrim(pmwmDspInfo, 0);                    //$4
      if (ulRC)
	return ulRC;
    }
    mwmParseSetDRVRSTAT(2);
    break;
    
  
/* @TBD @MIKE  
  case 50:
  case 51:
  case 52:
  case 53:
  case 54:
  case 55:
  case 56:
  case 57:
  case 58:
  case 59:
    mwmParseSetDRVRSTAT(2);
    break;
    @TBD @MIKE */
    
  case 60:
  case 61:
  case 62:
    if( (pmwmDspInfo->ulDiscrimState) && (pmwmDspInfo->bCIDCapability) ) { 
      // we only (un)load caller ID in DISCRIM state and capable
      ulRC = mwmCidlLoad(pmwmDspInfo,ulParm1-60);
      if (ulRC)
	return ulRC;
    }
    mwmParseSetDRVRSTAT(2);   // always release AT commands 08/05/96
    break;
    
  case 71:
    break;
    
  case 72:
    break;
    
  case 73:
    break;
    
  case 74:
    break;
    
  case 75:
  case 76:
    // Turn V.23 on/off...from +MS Command                                 
    // Make a temporary copy of FeaturesToLoad.                            
    // Copy current LongTermFeatures to FeaturesToLoad.                    
    // Add (or subtract) V.23 to both the TempFeatures and FeaturesToLoad  
    // SetFeatures specifying LongTerm (NOT Temporary).                    
    // Copy TempFeatures back to FeaturesToLoad.                           
    // SetFeatures specifying Temporary to regain Temporary settings with  
    //    new V.23 setting.                                                
    ulTempFeatures = pmwmDspInfo->ulFeaturesToLoad;
    pmwmDspInfo->ulFeaturesToLoad = pmwmDspInfo->ulLongTermFeatures;
    
    if (ulParm1 == 75 ) {
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V23;
      ulTempFeatures |= FEATURE_V23;
    } else {
      pmwmDspInfo->ulFeaturesToLoad &= ~FEATURE_V23;
      ulTempFeatures &= ~FEATURE_V23;
    }
    
    ulRC = mwmSpcfcSetFeatures(pmwmDspInfo, 0);   // tell parser and mctl about change 
    if (ulRC)
      return (ulRC);
    
    pmwmDspInfo->ulFeaturesToLoad = ulTempFeatures;
    ulRC = mwmSpcfcSetFeatures(pmwmDspInfo, 1);   // tell parser and mctl about change 
    if (ulRC)
      return (ulRC);
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 77:
    break;
    
  case 78:
    break;
    
    
  case 79:
    break;
    
  case 90:
  case 91:
    ulRC = mwmCidlLoadDRing(pmwmDspInfo, ulParm1-90);
    if (ulRC)
      return (ulRC);
    mwmParseSetDRVRSTAT(2); // 07/17/96 send in IPC1 for speed
    break;
    
    // 0x92 - #UD command
  case 92:
    if( !pmwmDspInfo->ulDiscrimState ) {   // Does this indicate that we are offhook?
      UniDiagsSaveSetupInfo(pmwmDspInfo);
      UniDiagsSaveLineInfo(pmwmDspInfo);
      UniDiagsSaveCarrierInfo(pmwmDspInfo);
      UniDiagsUpdateCallStatus(pmwmDspInfo);
    }
    mwmPoundUDFillBuffer();
    mwmParseSetDRVRSTAT(2);
    break;
    
    
  case 80:
  case 81:
  case 82:
  case 83:
  case 84:
  case 85:
  case 86:
  case 87:
  case 88:
  case 89:
    ulRC = mwmPumpsV34IPC(pmwmDspInfo,ulParm1,ulParm2 );
    if (ulRC)
      return ulRC;
    mwmParseSetDRVRSTAT(2);
    break;


  case 1030:
    // ATDT;                                                              
    //  - Check to see if the line is in use.                             
    //  - Check for Blacklisting                                          
    usTempDRVRSTAT=2;
    pmwmDspInfo->usWTChangeDRVRSTATValue = 0;
    
    UniDiagsClearCallStatus();       // Save info for #ud command
    mwmParseSetWTChangeFlag(0);
    
    // For earlier NT builds dynamically changing the country has not been
    // implemented so this INI/REG settting will allow for the old method.
    if(GetPrivateProfileInt("OPTIONS","WTCHANGEOPT", 0,  INIFile)) {
      ulRC = WTT_CheckCountryNumber(NULL, &bCountryNumberIsOK, FALSE, 0L);
    } else
      ulRC = WTT_CheckCountryNumber(NULL, &bCountryNumberIsOK, TRUE, 0L);
    
    if (ulRC) {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    } else if (!bCountryNumberIsOK) {
      DPF("Country is not OK!");
      if(GetPrivateProfileInt("OPTIONS","WTCHANGEOPT", 0,  INIFile)) {
	mwmParseSetWTChangeFlag(1);
	pmwmDspInfo->usWTChangeDRVRSTATValue = 11;
	usTempDRVRSTAT = 0;
      } else
	usTempDRVRSTAT = 9;
    }
    
    if (usTempDRVRSTAT == 2) {
      // Determine dialing or answering, because V.23 is not available       
      // when answering.                                                     
      pmwmDspInfo->ulDialingModem=1;
      
      ulRC = mwmParseQueryNumberDialed(achNumberDialed);
      if (ulRC)
	return ulRC;
      
      // Get the Class/Subclass                                        $7    
      ulRC = mwmFaxGetFCLASS(pmwmDspInfo,&ClassStruct.usFCLASS, &ClassStruct.usSubclass);
      if (ulRC)
	return (ulRC);
      
      // Check FCLASS to see which mode we are in...                       
      ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "FCLASS", &ulAddress);
      if (ulRC!=DSP_NOERROR) {
	mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		       MWM_DSP_ERROR, ulRC);
	return(ulRC);
      }
      
      // Not discriminator....check to see if the coupler ID matches.      
      if (dspMeioQueryConnection( pmwmDspInfo->hConnect1,
				  MEIO_CONNECTION_CHARACTERISTICS,
				  &MeioReturnedInfo.lArg ) == MEIO_NOERROR) {
	  if (MeioReturnedInfo.sVal.CouplerMismatch == 1) { // wrong phone     
	    usTempDRVRSTAT=5;
	    DPF("Coupler/Country Mismatch!\n");
	  }
      }
      
      //Check here to see if the number has already been blacklisted...     
      if (usTempDRVRSTAT == 2) {
	// Only do the blacklisting stuff if the blacklister is active, and  
	// if the number to dial has been filled in.                        
	// If the achNumberDialed string is empty, then we can assume this  
	// is an answer instead of a dial.                                  
	// 03/14/97 - Change to use the new Blacklister which is in a       
	// separate library.                                                    
	
	// Check the class and request dial permission                     
	if (strlen(achNumberDialed)) {
	  if ((ClassStruct.usFCLASS == 1) || (ClassStruct.usFCLASS == 2))
	    ulBlcklistHandle = pmwmDspInfo->ulBlcklistFaxHandle;
	  else if ((ClassStruct.usFCLASS == 8))
	    ulBlcklistHandle = pmwmDspInfo->ulBlcklistVoiceHandle;
	  else
	    ulBlcklistHandle = pmwmDspInfo->ulBlcklistModemHandle;
	  
	  ulRC = BL_RequestDialPermission(ulBlcklistHandle, AUTOMATIC_DIAL,
					  achNumberDialed, &ulDialStatus, 
					  &ulDialHandle, &ulDelayTime, 0L);

	  if (ulRC != BL_SUCCESSFUL) {
	    mwmHandleError(pInstanceInfo, szThisFile, __LINE__,
			   MWM_BL_ERROR, ulRC);
	    return ulRC;
	  }
	  
	  switch(ulDialStatus) {
	  case BL_DIAL_GRANTED:
	    break;
	  case BL_NUMBER_DELAYED:
	    usTempDRVRSTAT = 7;
	    break;
	  default :
	    usTempDRVRSTAT = 4;     // blacklisted
	  }
	}
      }
      
      //if everything above this succeeded, then set the user               
      // message to "Dialing"...                                             
      if (usTempDRVRSTAT == 2) {
	mwmParsePostMessage(WM_MWM_UPDATE_STATUS, MWM_STATE_DIALING, 0);
      } 
      
      
      // Clear the number Dialed variable.                                  
      ulRC = mwmParseSetNumberDialed("\0");
      if (ulRC)
	return ulRC;
    }
    if(usTempDRVRSTAT)
      mwmParseSetDRVRSTAT(usTempDRVRSTAT);
    break;
    
    // IPCs 0x100 and 0x101 added for V.80 Handset snooping.
    // 0x100 - Handset up - V.8/V.21 pair need to be loaded for
    //         CI Detection.
  case 0x100:
    if (pmwmDspInfo->bV80HandsetSnoop) {
      ulRC = mwmV80LLoadV8AndV21(pmwmDspInfo, FALSE);
      if (ulRC)
	return (ulRC);
    }
    mwmParseSetDRVRSTAT(2);
    break;
    
    // 0x101 - Handset down - Unload V.8/V.21 pair.
  case 0x101:
    if (pmwmDspInfo->bV80HandsetSnoop) {
      ulRC = mwmV80LV8V21Unload(pmwmDspInfo);
      if (ulRC)
	return (ulRC);
    }
    mwmParseSetDRVRSTAT(2);
    break;
    
    // 0x110 - Modem control has parsed one of the following
    // Class 1 AT commands: +FRH, +FTH, +FRM, +FTM.
    // We should now get information from modem control so
    // we can update the GUI with the correct state and speed.
  case 0x110:
    ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "FAXCMD", &ulAddress);
    if (ulRC!=DSP_NOERROR) {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    }
    ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
			 &usFaxCmd, 1, DSP_MEMXFER_DATA_READ);
    if (ulRC!=DSP_NOERROR) {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    } 
    
    ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress+2,
			  &usFaxXrat, 1, DSP_MEMXFER_DATA_READ);
    if (ulRC!=DSP_NOERROR) {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
		     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    } 
    
    DPF("FAXCMD = 0x%X, FAXXRAT = 0x%X", usFaxCmd, usFaxXrat);
    
    // The FAXXRAT value is the value we used to pass down
    // to modem control in a PP command after we parsed the
    // command. Modem control has already parsed the value
    // for us now. Any negative number will indicate
    // 'Negotiating', and other values are translated into
    // speeds.
    if ((SHORT)usFaxXrat < 0) {
      usUIMessage = MWM_STATE_NEGOTIATING;
      usUseConnectSpeed = 0;
    } else {
      usUseConnectSpeed = 1;
      
      switch (usFaxCmd) {
      case 0x09:
      case 0x01:
	usUIMessage = MWM_STATE_FAXSEND;
	break;
	
      case 0x0D:
      case 0x05:
	usUIMessage = MWM_STATE_FAXRECEIVE;
	break;
      } 
      
      ulRC = mwmClss1ConvertXratToSpeed(usFaxXrat);
      if (ulRC)
	return(ulRC);
    }
    
    mwmParsePostMessage(WM_MWM_UPDATE_STATUS, usUIMessage, usUseConnectSpeed);
    
    // NOTE!!! For this interrupt, modem control is NOT waiting for
    // DRVSTAT to be set by the driver. The AT commands being processed
    // by modem control are very time-sensitive, and waiting for
    // DRVSTAT after the interrupt would hurt performance.
    // Therefore, we will not set DRVSTAT so that we do not cause
    // problems with any subsequent interrupts.
    // mwmParseSetDRVRSTAT(2);
    break;
    
    // 0x130 - VCON received - Update GUI.
    /* @TBD @MJS 
  case 0x130:
    mwmParsePostMessage( WM_MWM_UPDATE_STATUS,
			 MWM_STATE_LINEINUSE,
			 (MAKELPARAM(MWM_STATUS_USE_TEXT_CONSTANT,MWM_STATE_TAM)));
    break;
    @MIKE @TBD */
    
    // 0x140 - ATA issued.
    // 0x141 - ATD issued.
  case 0x140:
    { MW_SYSLOG_1(TRACE_MWMLW32,"mwmipc::mwmipc, handling case 0x140\n");  }
    pmwmDspInfo->usDialOrAnswerFlag = ANSWER;
    if (!pmwmDspInfo->ulDiscrimState) {
      ulRC = mwmLoadPSV8ITCBSettings(pmwmDspInfo);
      if (ulRC)
	return(ulRC);
    }
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 0x141:
    pmwmDspInfo->usDialOrAnswerFlag = DIAL;
    if (!pmwmDspInfo->ulDiscrimState) {
      ulRC = mwmLoadPSV8ITCBSettings(pmwmDspInfo);
      if (ulRC)
	return(ulRC);
    }
    mwmParseSetDRVRSTAT(2);
    break;
    
    
  case 0x1000:
  case 0x1001:
  case 0x1002:
  case 0x1003:
  case 0x1004:
  case 0x1005:
  case 0x1006:
  case 0x1007:
  case 0x1008:
  case 0x1009:
  case 0x100A:
  case 0x100B:
  case 0x100C:
  case 0x100D:
  case 0x100F:
    ulRC = mwmrspEchoFAXResponse(pmwmDspInfo,(USHORT)ulParm1);
    if (ulRC != DSP_NOERROR) {
      return ulRC;
    }
    mwmParseSetDRVRSTAT(2);
    break;
    
  case 0x2000:  //$1 received A8E
    // This interrupt simply says that the parser has received a valid    
    // +A8E command.  We have to decide what to do with it here.          
    // We also have to unload the V8/V21 support if the command has been  
    // disabled.                                                          
    
    // First, query the registers to see if A8E is enabled or disabled.    
    ulRC = mwmParseQueryModemRegisters(&Registers);
    if (ulRC)
      return (ulRC);
    
    // Query the Hook status to see if we are offhook.                     
    ulRC = mwmParseQueryHookStatus(&usHandsetStatus,&usHookStatus);
    if (ulRC)
      return (ulRC);
    
    // Get the Class/Subclass of the modem to see if we need to load V8/V21
    ulRC = mwmFaxGetFCLASS(pmwmDspInfo,&usFCLASS, &usCurrentVLS);
    if (ulRC)
      return (ulRC);
    
    if ((Registers.PlusA8E[0] == 6) ||
	(Registers.PlusA8E[1] == 5)) {
      /* not supported */
    } else {
      // The AT Commands are set so that V.8/V.21 should be unloaded.      
      // Unload these two regardless of the current VLS setting.           
      if ((pmwmDspInfo->hmodV22) && (pmwmDspInfo->hmodV8)) {
	ulRC = mwmV80LV8V21Unload(pmwmDspInfo);
	if (ulRC)
	  return ulRC;
      }
    }
    mwmParseSetDRVRSTAT(2);
    break;
    
    
  case 0:
    MW_SYSLOG_1(TRACE_MWMLW32,"mwmipc::mwmIPC ERROR case 0 not expected\n");
    break;
    
    
  default:
    // Add this to make sure that unrecognized IPCs don't hang the system. 
    MW_SYSLOG_ERROR(LOG_ERR,"mwmipc::mwmIPC ERROR case %lx not recognized\n",ulParm1);
    mwmParseSetDRVRSTAT(2);
  }
  
  { MW_SYSLOG_1(TRACE_MWMLW32,"mwmipc::mwmIPC exit\n");  }
  
  return 0;
}

