/*
 * main.c --- main program.
 *
 * Copyright (c) 1997, 98, 99, 2000, 2001, 2002, 2004 by Pascal Wassong All Rights Reserved.
 *
 * Time-stamp: <2004-06-29 23:27:20 pascal>
 *
 * This file is part of Natch.
 *
 * Natch is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Natch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include	"common.h"

#define		EXTERN_MAIN	/* Uniquement defini ici !!! */
#include	"main.h"

#include	"langues.h"
#include	"cmdline.h"
#include	"pcpj.h"
#include	"pcpjtool.h"
#include	"hash.h"	/* for hashSetSize(...) */

#include	<stdlib.h>	/* for getenv(...), exit(...) */
#include	<ctype.h>	/* for tolower(...) */
#include	<string.h>	/* for memset(...), str...() */
#include	<sys/stat.h>	/* for stat(...) */
#include	<assert.h>

/*--------------------------------------------------------------------------*/

#define		MAX_FILE_LENGTH		250

/*--------------------------------------------------------------------------*/

int			main( int argc, char** argv );

static int		getArguments( int argc, char** argv );
static void		initialise_globals();
static void		examine_arguments(
    struct gengetopt_args_info*	args_info       ,
    unsigned int*		hashSize        ,
    int*			number_of_moves ,
    eLanguage*			langue          );
static void		decode_watch_option( const char* options );

extern int		reportbugs_text_length;
extern char*		reportbugs_text[];
extern int		copyright_text_length;
extern char*		copyright_text[];

static void		print_copyright();
static void		print_reportbugs();


static unsigned int	read_file( const char* filename );
static unsigned int	readForsythe( const char* filename );

#ifdef HAS_CHLOE
static unsigned int	readChl( const char* filename );
static piece_type_t	convert_type_chloe_to_Natch(
    unsigned char	piece_type_chloe );
#endif /* HAS_CHLOE */

/*--------------------------------------------------------------------------*/

int
main( int argc, char** argv )
{
    int		nbHalfMoves = getArguments( argc, argv );

    assert( sizeof( piece_index_t ) == 1 );
    assert( sizeof( colour_t      ) == 1 );
    assert( sizeof( square_t      ) == 1 );
    assert( sizeof( castling_t    ) == 1 );
    assert( sizeof( piece_type_t  ) == 1 );
    assert( sizeof( piece_t ) % 4   == 0 );

    initAssociation( nbHalfMoves );

    InitialePieces = langGetString( eStrInitialOfPieces );

    mainBeginning( nbHalfMoves );

    creeAssociation( nbHalfMoves );

    mainEnd( eStrEmpty, 0 );

    return 0 ;
}

/*--------------------------------------------------------------------------*/

static int
getArguments( int argc, char** argv )
{
    struct gengetopt_args_info	args_info ;

    char*			LC_ALL		 = getenv( "LC_ALL" );
    char*			NATCH 		 = getenv( "NATCH"  );

    int				number_of_moves  = -1       ;
    eLanguage			langue           = eEnglish ;
    unsigned int		hashSize         = 6        ;

    initialise_globals();

    if ( LC_ALL != NULL && strlen( LC_ALL ) >= 2 )
    {
	char	lc_all[ 3 ];
	lc_all[ 0 ] = tolower( LC_ALL[ 0 ] );
	lc_all[ 1 ] = tolower( LC_ALL[ 1 ] );
	lc_all[ 2 ] = '\0' ;

	if ( strcmp( lc_all, "en" ) == 0 )
	{
	    langue = eEnglish ;
	}
	else if ( strcmp( lc_all, "fr" ) == 0 )
	{
	    langue = eFrench ;
	}
	else if ( strcmp( lc_all, "de" ) == 0 )
	{
	    langue = eGerman ;
	}
    }

    args_info.position_arg = 1; /* 1 is default value if arg is given. */

    if ( NATCH != NULL )
    {
	size_t		NATCH_length 	 = strlen( NATCH );
	char*		NATCH_copy 	 = (char*)malloc( NATCH_length + 1 );
	char**		arguments            ;
	int		nb_arguments         ;
	int		nb_max_arguments = 1 ;
	unsigned int	i 	             ;
	int		in_text = FALSE      ;

	strcpy( NATCH_copy, NATCH );
	for ( i = 0 ; i < NATCH_length ; i++ )
	{
	    if ( NATCH[ i ] == ' ' )
	    {
		nb_max_arguments++;
	    }
	}

	nb_max_arguments += argc + 1 ;
	arguments = (char**)( malloc( nb_max_arguments * sizeof( char* ) ) );

	arguments[ 0 ] = "NATCH variable";
	nb_arguments = 1 ;

	for ( i = 0 ; i < NATCH_length ; i++ )
	{
	    if ( NATCH_copy[ i ] == ' ' )
	    {
		NATCH_copy[ i ] = 0 ;
		in_text = FALSE ;
	    }
	    else if ( ! in_text )
	    {
		arguments[ nb_arguments ] = & NATCH_copy[ i ];
		nb_arguments++;
		in_text = TRUE ;
	    }
	}

	if ( cmdline_parser( nb_arguments, arguments, &args_info ) != 0 )
	{
	    exit( EXIT_FAILURE );
	}

	examine_arguments( &args_info, &hashSize, &number_of_moves, &langue );

	free( NATCH_copy );
	free( arguments  );
    }

    if ( cmdline_parser( argc, argv, &args_info ) != 0 )
    {
	exit( EXIT_FAILURE );
    }
    if ( args_info.help_given )
    {
	cmdline_parser_print_help ();
	print_reportbugs ();
	exit( EXIT_SUCCESS );
    }

    if ( args_info.version_given )
    {
	cmdline_parser_print_version ();
	print_copyright ();
	exit( EXIT_SUCCESS );
    }

    examine_arguments( &args_info, &hashSize, &number_of_moves, &langue );

    hashSetSize( hashSize );

    if ( args_info.inputs_num > 0 )
    {
	int	nb = read_file( args_info.inputs[ 0 ] );
	if ( number_of_moves == -1 )
	{
	    number_of_moves = nb ;
	}
    }
    else
    {
	fprintf( stderr, "Error : Missing input filename.\n" );
	fprintf( stderr, "Try `Natch --help' for more information.\n" );
	exit( EXIT_FAILURE );
    }

    if ( number_of_moves < 1 )
    {
	fprintf( stderr ,
		 "Error : Bad number of single moves : %d\n",
		 number_of_moves );
	exit( EXIT_FAILURE );
    }

    langSetLanguage( langue );	/* Must be after read_file */

    return number_of_moves ;

}

static void
initialise_globals()
{
    MainExplorationFlag    = TRUE  ;
    MainPrintPositionLevel = 0 ;
    MainVerboseFlag        = FALSE ;
    MainVisualFlag         = FALSE ;
    MainContinueValue      = 0 ;
    MainStopValue          = 0 ;
    MainBeepFlag	   = TRUE ;

    MainLimiteTrajet       = 0 ;
    MainNombreNouvellesCasesInterdites = 0 ;

    MainFD                = stdout ;

    MainExploTicks        = 0 ;
    MainTicks1            = 0 ;
    MainTicks2            = 0 ;
}

static void
examine_arguments(
    struct gengetopt_args_info*	args_info       ,
    unsigned int*		hashSize        ,
    int*			number_of_moves ,
    eLanguage*			langue          )
{
    /* Boolean flags */
    if ( args_info->explore_given )
    {
	MainExplorationFlag = !MainExplorationFlag ;
    }
    if ( args_info->position_given )
    {
	MainPrintPositionLevel = args_info->position_arg;
    }
    if ( args_info->verbose_given )
    {
	MainVerboseFlag = !MainVerboseFlag ;
    }
    if ( args_info->visual_given )
    {
	MainVisualFlag = !MainVisualFlag ;
    }
    if ( args_info->beep_given )
    {
	MainBeepFlag = !MainBeepFlag ;
    }

    /* Other arguments */
    if ( args_info->continue_given )
    {
	MainContinueValue = args_info->continue_arg ;
    }
    if ( args_info->stop_given )
    {
	MainStopValue = args_info->stop_arg ;
    }

    if ( args_info->output_given && MainFD == stdout )
    {
	MainFD = fopen( args_info->output_arg, "w" );
	if ( MainFD == NULL )
	{
	    fprintf( stderr ,
		     "Error : Unable to open file : %s\n",
		     args_info->output_arg );
	    exit( EXIT_FAILURE );
	}
    }

    if ( args_info->watch_given )
    {
	decode_watch_option( args_info->watch_arg );
    }

    if ( args_info->language_given )
    {
	char	c = tolower( args_info->language_arg[ 0 ] );
	switch(c)
	{
	case 'e':
	    *langue = eEnglish ;
	    break;
	case 'f':
	    *langue = eFrench ;
	    break;
	case 'd':
	case 'g':
	    *langue = eGerman ;
	    break;
	default:
	    fprintf( stderr,
		     "Warning : Unknown language. Argument ignored : -l %s\n",
		     args_info->language_arg );
	    break;
	}
    }

    if ( args_info->hash_size_given )
    {
	*hashSize = args_info->hash_size_arg ;
    }

    if ( args_info->moves_given )
    {
	*number_of_moves = args_info->moves_arg ;
    }

    /* Verify the compatibility between different arguments */
    if ( MainVisualFlag )
    {
	MainVerboseFlag = FALSE ;
    }
}

static void
decode_watch_option( const char* options )
{
    const char*	separators = ".,;:/+-";
    const char*	current    = options ;

    while ( *current != '\0' )
    {
	while ( *current != '\0' && strchr( separators, *current ) != NULL )
	{
	    current ++ ;
	}
	if ( isdigit( *current ) )
	{
	    MainLimiteTrajet = abs( atoi( current ) );
	}
	else if ( *current != '\0' )
	{
	    char	column = tolower( *current );
	    char	row    = *( current + 1 );
	    if ( column >= 'a' && column <= 'h' &&
		 row    >= '1' && row    <= '8' )
	    {
		square_t	square = column - 'a'
		    + ( ( row - '1' ) << 4 );
		unsigned int	i ;
		bool_t		already_given = FALSE ;
		for ( i = 0 ; i < MainNombreNouvellesCasesInterdites ; i++ )
		{
		    if ( MainNouvellesCasesInterdites[ i ] == square )
		    {
			already_given = TRUE ;
			break ;
		    }
		}
		if ( already_given )
		{
		    fprintf( stderr,
			     "Warning : square %c%c %s\n",
			     *current, row,
			     "already given in watch option" );
		}
		else
		{
		    if ( MainNombreNouvellesCasesInterdites
			 < MAX_NOUVELLES_CASES_INTERDITES )
		    {
			MainNouvellesCasesInterdites[
			    MainNombreNouvellesCasesInterdites ] = square ;
			MainNombreNouvellesCasesInterdites++;
		    }
		    else
		    {
			fprintf(
			    stderr,
			    "Warning : too much squares given in watch option\n" );
		    }
		}
	    }
	    else
	    {
		fprintf( stderr,
			 "Warning : bad square for watch option ignored : " );
		fprintf( stderr, "%c%c\n", *current, row );
	    }
	}
	while ( *current != '\0' && strchr( separators, *current ) == NULL )
	{
	    current++;
	}
    }

    if ( MainLimiteTrajet == 0 )
    {
	fprintf(
	    stderr,
	    "Warning : watch option given without (or bad) number of moves\n" );
    }
}

void
print_copyright()
{
    int i;

    for ( i = 1; i <= copyright_text_length ; ++i )
    {
	printf( "%s\n", copyright_text[ i ] );
    }
}

void
print_reportbugs()
{
    int i;

    for ( i = 1; i <= reportbugs_text_length ; ++i )
    {
	printf("%s\n", reportbugs_text[ i ] );
    }
}

static unsigned int
read_file( const char* filename )
{
    char*   ptr = strrchr( filename, '.' );

    memset( Echiquier, 0, sizeof( Echiquier ) );

    if ( ptr != NULL )
    {
#ifdef HAS_CHLOE
	char	lowercaseExtension[ 5 ];
	if ( strlen( ptr ) == 4 )
	{
	    int	i;
	    strcpy( lowercaseExtension, ptr );
	    for ( i = 0 ; i < 4 ; i++ )
	    {
		lowercaseExtension[ i ] = tolower( lowercaseExtension[ i ] );
	    }
	}
	else
	{
	    lowercaseExtension[ 0 ] = '\0' ;
	}
	if ( strcmp( lowercaseExtension, ".chl" ) == 0 )
	{
	    return readChl( filename );
	}
	else
	{
	    return readForsythe( filename );
	}
#else /* not HAS_CHLOE */
	return readForsythe( filename );
#endif /* not HAS_CHLOE */
    }
    else
    {
	struct stat	stat_buffer ;
	if ( stat( filename, &stat_buffer ) == 0)
	{
	    return readForsythe( filename );
	}
	else
	{
	    char    new_file_name[ MAX_FILE_LENGTH + 5 ];
	    strcpy( new_file_name, filename );
	    strcat( new_file_name, ".txt" );
	    if ( stat( new_file_name, &stat_buffer ) == 0 )
	    {
		return readForsythe( new_file_name );
	    }
#ifdef HAS_CHLOE
	    strcpy( new_file_name, filename );
	    strcat( new_file_name, ".chl" );
	    if ( stat( new_file_name, &stat_buffer ) == 0 )
	    {
		return readChl( new_file_name );
	    }
#endif /* HAS_CHLOE */
	    printf( "Error : Unable to read file %s\n", filename );
	    exit( EXIT_FAILURE );
	}
    }
}

static unsigned int
readForsythe( const char* filename )
{
    FILE*		    fd;
#define		MAX_LINE_SIZE		250
    char		buffer[ MAX_LINE_SIZE ];

    char*		allowed_characters = "12345678/";
    piece_type_t	typePieces[] =
    {
	ROI      ,
	DAME     ,
	TOUR     ,
	FOU      ,
	CAVALIER ,
	CAVALIER ,
	PION     };

    fd = fopen( filename, "rt" );
    if ( fd == NULL )
    {
	perror( filename );
	exit( EXIT_FAILURE );
    }

    while ( fgets( buffer, MAX_LINE_SIZE, fd ) != NULL )
    {
	eLanguage	langue ;
	unsigned int	length = strlen( buffer );

	/* Suppress blanck caracters at end of line */
	while ( length > 0 &&
		( buffer[length-1]    == '\n'
		  || buffer[length-1] == ' '
		  || buffer[length-1] == '\t' ) )
	{
	    length--;
	}
	buffer[ length ] = '\0';

	if ( length < 3 )
	{
	    continue;
	}

	for ( langue = eFrench; langue < numberOfLanguages ; langue++ )
	{
	    char		all_characters[ 50 ];
	    unsigned		buffer_index ;
	    const char*		forsythe_initials ;
	    int			ligne	  = 0x70 ;
	    int			colonne	  = 0 ;
	    piece_index_t	index_blanche = INDEX_TOUR_DAME_BLANC ;
	    piece_index_t	index_noire   = INDEX_TOUR_DAME_NOIR ;
	    int			nb_rois	  = 0 ;

	    langSetLanguage( langue );
	    forsythe_initials = langGetString( eStrForsytheInitials );
	    strcpy( all_characters, forsythe_initials );
	    strcat( all_characters, allowed_characters );
	    for ( buffer_index = 0 ; buffer_index < length ; buffer_index++ )
	    {
		int	c = toupper( buffer[ buffer_index ] );
		if ( c == forsythe_initials[ 0 ] )
		{
		    nb_rois++;
		}
		if ( strchr( all_characters, c) == NULL )
		{
		    break;
		}
	    }

	    if ( buffer_index < length || nb_rois != 2 )
	    {
		continue;
	    }

	    /* We have found a good line. */
	    for ( buffer_index = 0 ; buffer_index < length ; buffer_index++ )
	    {
		int	c = buffer[ buffer_index ];
		if ( c >= '1' && c <= '8' )
		{
		    colonne += c - '0';
		}
		else if ( c == '/' )
		{
		    ligne   -= 0x10 ;
		    colonne = 0 ;
		}
		else if ( c == '\n' )
		{
		}
		else if ( colonne < 8 )
		{
		    int		couleur	= isupper( c ) ? BLANC : NOIR ;
		    int		index_piece = 0 ;
		    int		type_piece ;
		    c = toupper( c );
		    while ( forsythe_initials[ index_piece ] != c )
		    {
			index_piece++;
		    }
		    type_piece = typePieces[ index_piece ];
		    if ( couleur == BLANC )
		    {
			if ( type_piece == ROI )
			{
			    Position_Pieces[ INDEX_ROI_BLANC ] =
				ligne + colonne ;
			    Type_Pieces[ INDEX_ROI_BLANC ] = ROI ;
			}
			else
			{
			    Position_Pieces[ index_blanche ] = ligne + colonne ;
			    Type_Pieces[ index_blanche ] =
				typePieces[ index_piece ];
			    index_blanche++;
			}
		    }
		    else
		    {
			if ( type_piece == ROI )
			{
			    Position_Pieces[ INDEX_ROI_NOIR ] =
				ligne + colonne ;
			    Type_Pieces[ INDEX_ROI_NOIR ] = ROI ;
			}
			else
			{
			    Position_Pieces[ index_noire ] = ligne + colonne ;
			    Type_Pieces[ index_noire ] =
				typePieces[ index_piece ];
			    index_noire++;
			}
		    }
		    colonne++;
		}
	    }
	    while ( index_blanche <= INDEX_PION_BLANC_H )
	    {
		Position_Pieces[ index_blanche ] = CASE_D_UNE_PIECE_CAPTUREE;
		Type_Pieces[ index_blanche ]     = PION;
		index_blanche++;
	    }
	    while ( index_noire <= INDEX_PION_NOIR_H )
	    {
		Position_Pieces[ index_noire ] = CASE_D_UNE_PIECE_CAPTUREE;
		Type_Pieces[ index_noire ]     = PION;
		index_noire++;
	    }
	    majEchiquier();

	    fgets( buffer, MAX_LINE_SIZE, fd );
	    fclose( fd );

	    return atoi( buffer );
	}
    }

    fclose( fd );

    fprintf( stderr,
	     "Error : Found no legal Forsythe position in file : %s\n",
	     filename );
    exit( EXIT_FAILURE );
}

#ifdef HAS_CHLOE
static unsigned int
readChl( const char* filename )
{
    FILE*	    fd ;
    int		    nb ;
    unsigned char   buffer[ 512 ];
    piece_index_t   i ;
    int		    index_buffer ;

    fd = fopen( filename, "rb" );
    if ( fd == NULL )
    {
	perror( filename );
	exit( EXIT_FAILURE );
    }

    nb = fread( buffer, 1, sizeof( buffer ), fd );
    if ( nb != sizeof( buffer ) )
    {
	if ( nb > 0 )
	{
	    fprintf( stderr, "Error : %s : corrupted file\n", filename );
	}
	else
	{
	    perror( filename );
	}
	exit( EXIT_FAILURE );
    }
    fclose( fd );

    /* Blanc */
    i = INDEX_ROI_BLANC ;
    index_buffer = 0x40 * 2 ;
    while( buffer[ index_buffer ] != 0xFF )
    {
	Position_Pieces[ i ] = buffer[ index_buffer ];
	index_buffer++;
	Type_Pieces[ i ] =
	    convert_type_chloe_to_Natch( buffer[ index_buffer ] );
	i++;
	index_buffer++;
    }
    while ( i <= INDEX_PION_BLANC_H )
    {
	Position_Pieces[ i ] = CASE_D_UNE_PIECE_CAPTUREE ;
	Type_Pieces[ i ] = PION ;
	i++;
    }

    /* Noir */
    i = INDEX_ROI_NOIR ;
    index_buffer = 0x80 * 2 ;
    while( buffer[ index_buffer ] != 0xFF )
    {
	Position_Pieces[ i ] = buffer[ index_buffer ];
	index_buffer++;
	Type_Pieces[ i ] =
	    convert_type_chloe_to_Natch( buffer[ index_buffer ] );
	i++;
	index_buffer++;
    }
    while ( i <= INDEX_PION_NOIR_H )
    {
	Position_Pieces[ i ] = CASE_D_UNE_PIECE_CAPTUREE ;
	Type_Pieces[ i ]     = PION ;
	i++;
    }

    majEchiquier();

    /* Lit le nombre de demi-coups dans le fichier .CHL */
    nb = (buffer[ 12 ] - '0');
    if ( buffer[ 13 ] != ' ')
    {
	nb = nb * 10 + buffer[13] - '0';
    }

    return nb;
}

static piece_type_t
convert_type_chloe_to_Natch(unsigned char piece_type_chloe)
{
    static piece_type_t	    conversion_table[] = {
	ROI  ,
	PION ,
	PION ,
	DAME ,
	TOUR ,
	FOU  ,
	CAVALIER
    };

    if ( piece_type_chloe > 6 )
    {
	return piece_type_chloe;
    }
    return conversion_table[ piece_type_chloe ];
}

#endif /* HAS_CHLOE */
