#include "bool.h"
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include "state.h"
#include "shared.h"
#include "util.h"
#include "exec.h"
#include "main.h"

struct IDENTS {
	char *username;
	char *peername;
	char *pid;
};

static bool makecommand(char *buf, size_t maxlen, const char *fmt,
		struct IDENTS ident);
static bool quad_out_of_range(u_char peer[4], u_char host[4], u_char mask[4]);
static bool host_to_quad(char *host, u_char quad[4]);

#define MAX_COMMAND_LENGTH 1024
bool onaction(bool connect, pid_t *firstlastp, struct clients *item)
{
	char real_command[MAX_COMMAND_LENGTH];
	char *command;
	struct IDENTS ident;
	pid_t *pidpt;
	pid_t pid;

	if (QIS_RUNNING(item->onconnect) || QIS_RUNNING(item->ondisconnect))
		return FALSE; /* Do when it dies. */
	if (connect) {
		command = set.onconnect;
		pidpt = &item->onconnect;
	} else {
		command = set.ondisconnect;
		pidpt = &item->ondisconnect;
	}
	if (!command) {
		if (firstlastp)
			*firstlastp = NOT_RUN;
		return FALSE;
	}
	item->onconnect = item->ondisconnect = NOT_RUN;

	/* evaluate codes in user-supplied command */
	ident.peername = item->peername;
	ident.username = item->username;
	asprintf(&ident.pid, "%lu", (u_long)item->pid);
	if (!ident.pid) {
		notice_err("malloc failed");
		return TRUE;
	}
	if (makecommand(real_command, MAX_COMMAND_LENGTH, command, ident)) {
		notice_debug("makecommand failed\n");
		free(ident.pid);
		return TRUE;
	}
	free(ident.pid);

	/* execute the evaluated command */
	pid = run_fd3open(real_command);
	if (pid==-1)
		pid = NOT_RUN;
	if (firstlastp)
		*firstlastp = pid;
	if (pid==NOT_RUN)
		return TRUE;
	*pidpt = pid;
	return FALSE;
}

bool makehostmasks(char **hostmasks)
{
	u_char *mask;
	int i;

	set.hmdata = xmalloc(set.num_hms * sizeof(*set.hmdata));
	if (!set.hmdata)
		return TRUE;
	/* hostmask is e.g. 127.0.0.1/255.255.255.0 */
	for (i=0; i < set.num_hms; ++i) {
		if (!(mask = strchr(hostmasks[i], '/'))) {
			notice("invalid hostmask '%s'\n", hostmasks[i]);
			return TRUE;
		}
		*mask++ = '\0';
		if (host_to_quad(hostmasks[i], set.hmdata[i].host))
			return TRUE;
		if (host_to_quad(mask, set.hmdata[i].mask))
			return TRUE;
	}
	return FALSE;
}
	 
bool ip_out_of_range(char *peername)
{
	u_char peerquad[4];
	int i;

	if (!set.hmdata)
		return FALSE;
	if (host_to_quad(peername, peerquad))
		return TRUE;
	for (i=0; i < set.num_hms; ++i) {
		if (!quad_out_of_range(peerquad, set.hmdata[i].host,
					set.hmdata[i].mask)) {
				return FALSE;
		}
	}
	return TRUE;
}
			
bool host_to_quad(char *host, u_char quad[4])
{
	int i;
	char *sep = host;

	if (!host)
		return TRUE;
	if (mystrtouc(sep, &quad[0]))
		goto err;
	for (i=1; i < 4; ++i) {
		sep = strchr(sep, '.');
		if (!sep || mystrtouc(++sep, &quad[i]))
			goto err;
	}
	return FALSE;
err:
	notice("IP address '%s' has invalid format\n", host);
	return TRUE;
}
	
bool quad_out_of_range(u_char peer[4], u_char host[4], u_char mask[4])
{
        int i;
        for (i=0; i < 4; ++i) {
		if ((peer[i] & mask[i]) != (host[i] & mask[i]))
			return TRUE;
	}
	return FALSE;
}

/* TODO: audit this function */
bool makecommand(char *buf, size_t maxlen, const char *fmt, struct IDENTS ident)
{
	char *str;
	char *s;
	size_t i;
	size_t len;
	ssize_t left = (ssize_t)maxlen;
	
	if (!fmt || !ident.peername || !ident.username || !ident.pid)
		return TRUE;
	--left; /* \0 */
	for (str=buf; *fmt && left > 0; ++fmt) {
		if (*fmt!='%') {
			*str++ = *fmt;
			--left;
			continue;
		}
		++fmt;
		switch(*fmt) {
	 	case 'b': 
			s = ident.peername;
			break;
		case 'u': 
	    		s = ident.username;
    			break;
		case 'p':
			s = ident.pid;
			break;
		case '%':
    			*str++='%';
			--left;
			continue;
		default: continue;
		}
		len = strlen(s);
		for (i=0; i < len && left > 0; ++i) {
			*str++=*s++;
			--left;
		}
	}
	if (left==0) { /* XXX <= 0? */
		notice("length of generated command is greater than %d "
				"characters\n", maxlen - 1);
		return TRUE;
	}
	*str = '\0';
	return FALSE;
}

pid_t run_raw(int fd3, const char *command)
{
	pid_t pid;
#if 0 /* Won't generate SIGCHLD; adjust() won't be called. */
	if (!command)
		return HAVE_RUN; /* As far as we care, we -have- */
#endif
	if (!command) {
		notice_debug("run: command==NULL\n");
		return -1; /* caller should check */
	}
	pid = xfork();
	if (pid==-1) {
		notice_err("fork failed");
	       	return -1;
	} else if (pid==0) {
		if (cleanup_signals() || cleanup_fds_except(fd3)) {
			notice("cleanup failed: not executing '%s'\n", command);
			_exit(EXIT_FAILURE);
		}
		unblock_all_signals(NULL);
		if (command_logfile())
			_exit(EXIT_FAILURE);
		/* can send messages to users via fd 3 */
		if (fd3!=-1)
			dup2(fd3, 3);
		close(fd3);
		runexec(command);
		_exit(EXIT_FAILURE);
	}
	return pid;
}

void runexec(const char *command)
{
	notice("running /bin/sh -c '%s'\n", command);
	execlp("/bin/sh", "sh", "-c", command, NULL);
	notice_err("running /bin/sh -c '%s' failed", command);
}

/* FIXME: Having all processes write to one file is bad. */
bool command_logfile(void)
{
	const char *file = set.command_logfile;
	if (!file)
		file = "/dev/null";
	MYLOG_OPTIONS = MYLOG_STDERR_DATEPID;
	if (redirect_std(STDOUT_FILENO, file, O_WRONLY | O_APPEND | O_CREAT) ||
	    redirect_std(STDERR_FILENO, file, O_WRONLY | O_APPEND | O_CREAT)) {
		notice_err("opening command logfile '%s' for writing failed",
				file);
		return TRUE;
	}
	return FALSE;
}
