/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <libsbp2/sbp2byteswap.h>
#include <libsbp2/sbp2constants.h>
#include <libsbp2/sbp2main.h>
#include <libsbp2/sbp2manager.h>
#include <libsbp2/sbp2raw1394.h>

#include "app.h"
#include "cleanup.h"
#include "manager.h"
#include "util.h"
#include "wire.h"


typedef enum
{
  LOGIN_EMPTY,
  LOGIN_PENDING,
  LOGIN_ACTIVE,
  LOGIN_RECONNECT,
  LOGIN_DEAD
} LoginStatus;


typedef struct _ManagerState ManagerState;
typedef struct _Port Port;
typedef struct _Login Login;

struct _ManagerState
{
  EndpointApp *app;

  gint         num_ports;
  Port        *ports;

  gint         num_logins;
  Login       *logins;

  gboolean     exclusive;

  GIOChannel  *reader;
  GIOChannel  *writer;
};

struct _Port
{
  ManagerState *state;

  gint          number;

  SBP2Manager  *manager;

  gint          num_logins;
};

struct _Login
{
  LoginStatus  status;

  guint        login_ID;
  Port        *port;

  guint        timeout;

  guint64      EUI_64;

  nodeid_t     node;
  SBP2Pointer  status_FIFO;

  guint8       sbp2_status[8];
};


static gboolean query_logins       (SBP2ManagerAction       *action,
				    SBP2QueryLoginsORB      *orb,
				    SBP2QueryLoginsResponse *resp);
static gboolean login              (SBP2ManagerAction       *action,
				    SBP2LoginORB            *orb,
				    SBP2LoginResponse       *resp);
static gboolean reconnect          (SBP2ManagerAction       *action,
				    SBP2ReconnectORB        *orb);
static gboolean logout             (SBP2ManagerAction       *action,
				    SBP2LogoutORB           *orb);

static gboolean worker_active      (EndpointApp             *app,
				    WireWorkerActive        *message);
static gboolean worker_died        (EndpointApp             *app,
                                    WireWorkerDied          *message);

static gboolean find_free_login_ID (ManagerState            *state,
                                    guint                   *login_ID);
static gboolean expire_login       (Login                   *login);

static void     bus_reset          (SBP2Manager             *manager,
				    gpointer                 user_data);


static SBP2ManagerFuncs manager_funcs = {
  .bus_reset    = bus_reset,

  .login        = login,
  .query_logins = query_logins,
  .reconnect    = reconnect,
  .logout       = logout
};

static WireMsgFuncs wire_funcs = {
  .worker_active = worker_active,
  .worker_died   = worker_died
};


void
manager_process (EndpointApp *app,
		 Process     *process)
{
  ManagerState    *state;
  gint             num_ports, i;
  Port            *port;

  g_return_if_fail (app != NULL);
  g_return_if_fail (process != NULL);

  num_ports = sbp2_raw1394_get_num_ports ();

  if (num_ports < 1)
    exit (1);

  app->wire_funcs = &wire_funcs;

  state = g_new (ManagerState, 1);

  state->app = app;

  state->num_ports = num_ports;
  state->ports = g_new0 (Port, num_ports);

  state->num_logins = 0;
  state->logins = g_new0 (Login, app->max_logins);

  state->reader = process->reader;
  state->writer = process->writer;

  app->user_data = state;

  for (i = 0; i < num_ports; i++)
    {
       port = &state->ports[i];

       port->state = state;

       port->number = i;

       port->manager = sbp2_manager_new (&manager_funcs, app->context, i, port);
       
       if (!port->manager)
         exit (2);

       port->num_logins = 0;
    }
}

static gboolean
query_logins (SBP2ManagerAction       *action,
              SBP2QueryLoginsORB      *orb,
	      SBP2QueryLoginsResponse *resp)
{
  Port *port;

  g_return_val_if_fail (action != NULL, FALSE);
  g_return_val_if_fail (action->user_data != NULL, FALSE);
  g_return_val_if_fail (orb != NULL, FALSE);
  g_return_val_if_fail (resp != NULL, FALSE);

  port = action->user_data;

  resp->length = 4 + 12 * port->num_logins;
  resp->max_logins = port->state->app->max_logins_per_port;

  return TRUE;
}

static gboolean
login (SBP2ManagerAction *action,
       SBP2LoginORB      *orb,
       SBP2LoginResponse *resp)
{
  Port         *port;
  ManagerState *state;
  EndpointApp  *app;
  guint         login_ID;
  Login        *login;
  gpointer      message;

  g_return_val_if_fail (action != NULL, FALSE);
  g_return_val_if_fail (action->user_data != NULL, FALSE);
  g_return_val_if_fail (orb != NULL, FALSE);
  g_return_val_if_fail (resp != NULL, FALSE);

  port = action->user_data;

  state = port->state;
  app = state->app;

  if (app->honor_exclusive && state->num_logins > 0 &&
      (orb->exclusive || state->exclusive))
    {
      action->status->sbp_status = SBP2_STATUS_REQUEST_ACCESS_DENIED;
      return TRUE;
    }

  if (port->num_logins == app->max_logins_per_port)
    {
      action->status->sbp_status = SBP2_STATUS_REQUEST_ACCESS_DENIED;
      return TRUE;
    }

  if (!find_free_login_ID (state, &login_ID))
    {
      action->status->sbp_status = SBP2_STATUS_REQUEST_RESOURCES_UNAVAILABLE;
      return TRUE;
    }

  orb->lun = 0; /* TODO: multiple LUN */

  if (!g_hash_table_lookup (app->disks, GINT_TO_POINTER (orb->lun)))
    {
      action->status->sbp_status = SBP2_STATUS_REQUEST_LU_NOT_SUPPORTED;
      return TRUE;
    }

  if (app->honor_exclusive)
    state->exclusive = orb->exclusive;

  login = &state->logins[login_ID];

  login->status = LOGIN_PENDING;

  login->login_ID = login_ID;
  login->port = port;

  login->node = action->node;
  memcpy (&login->status_FIFO, &orb->status_FIFO, sizeof (login->status_FIFO));

  memcpy (&login->sbp2_status, action->status, sizeof (login->sbp2_status));

  port->num_logins++;
  state->num_logins++;

  message = wire_message_login (login_ID, port->number,
                                action->node, action->pointer, orb);
  wire_message_send (state->writer, message, app);

  return FALSE;
}

static gboolean
reconnect (SBP2ManagerAction *action,
	   SBP2ReconnectORB  *orb)
{
  Port         *port;
  ManagerState *state;
  EndpointApp  *app;
  guint         login_ID;
  Login        *login;

  g_return_val_if_fail (action != NULL, FALSE);
  g_return_val_if_fail (action->user_data != NULL, FALSE);
  g_return_val_if_fail (orb != NULL, FALSE);

  port = action->user_data;

  state = port->state;
  app = state->app;

  login_ID = orb->login_ID;

  if (login_ID >= app->max_logins)
    {
      action->status->sbp_status = SBP2_STATUS_REQUEST_LOGIN_ID_NOT_RECOGNIZED;
      return TRUE;
    }

  login = &state->logins[login_ID];

  if (login->status != LOGIN_RECONNECT)
    {
      action->status->sbp_status = SBP2_STATUS_REQUEST_LOGIN_ID_NOT_RECOGNIZED;
      return TRUE;
    }

  if (login->timeout != 0)
    {
      util_source_remove (app->context, login->timeout);
      login->timeout = 0;
    }

  login->status = LOGIN_ACTIVE;

  return TRUE;
}

static gboolean
logout (SBP2ManagerAction *action,
	SBP2LogoutORB     *orb)
{
  Port         *port;
  ManagerState *state;
  EndpointApp  *app;
  guint         login_ID;
  Login        *login;

  g_return_val_if_fail (action != NULL, FALSE);
  g_return_val_if_fail (action->user_data != NULL, FALSE);
  g_return_val_if_fail (orb != NULL, FALSE);

  port = action->user_data;

  state = port->state;
  app = state->app;
  
  login_ID = orb->login_ID;

  if (login_ID >= app->max_logins)
    {
      action->status->sbp_status = SBP2_STATUS_REQUEST_LOGIN_ID_NOT_RECOGNIZED;
      return TRUE;
    }

  login = &state->logins[login_ID];

  if (login->status == LOGIN_EMPTY)
    {
      action->status->sbp_status = SBP2_STATUS_REQUEST_LOGIN_ID_NOT_RECOGNIZED;
      return TRUE;
    }

  if (login->timeout != 0)
    {
      util_source_remove (app->context, login->timeout);
      login->timeout = 0;
    }

  expire_login (login);

  return TRUE;
}

static gboolean
worker_active (EndpointApp      *app,
	       WireWorkerActive *message)
{
  ManagerState *state;
  Login        *login;

  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (app->user_data != NULL, FALSE);
  g_return_val_if_fail (message != NULL, FALSE);
  g_return_val_if_fail (message->login_ID < app->max_logins, FALSE);

  state = app->user_data;

  login = &state->logins[message->login_ID];

  login->status = LOGIN_ACTIVE;

  return TRUE;
}

static gboolean
worker_died (EndpointApp    *app,
             WireWorkerDied *message)
{
  ManagerState    *state;
  Login           *login;
  Port            *port;
  SBP2Status      *status;

  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (app->user_data != NULL, FALSE);
  g_return_val_if_fail (message != NULL, FALSE);
  g_return_val_if_fail (message->login_ID < app->max_logins, FALSE);

  state = app->user_data;

  login = &state->logins[message->login_ID];
 
  port = login->port;

  switch (login->status)
    {
    case LOGIN_PENDING:
      status = g_memdup (&login->sbp2_status, sizeof (login->sbp2_status));
      status->sbp_status = SBP2_STATUS_REQUEST_RESOURCES_UNAVAILABLE;

      sbp2_manager_write_status (port->manager,
	                         login->node, &login->status_FIFO,
				 status);
      break;

    case LOGIN_ACTIVE:
      /* sbp2_manager_reset_bus (port->manager); */
      break;

    case LOGIN_RECONNECT:
      /*
      status = g_memdup (&login->sbp2_status, sizeof (login->sbp2_status));
      status->sbp_status = SBP2_STATUS_REQUEST_RESOURCES_UNAVAILABLE;

      sbp2_manager_write_status (port->manager,
	                         login->node, &login->status_FIFO,
				 status);
      */
      break;

    case LOGIN_DEAD:
      /* do nothing */
      break;

    case LOGIN_EMPTY:
    default:
      g_assert_not_reached ();
      break;
    }

  login->status = LOGIN_EMPTY;

  port->num_logins--;
  state->num_logins--;

  login->port = NULL;

  return TRUE;
}

static gboolean
find_free_login_ID (ManagerState *state,
                    guint        *login_ID)
{
  gint i;

  g_return_val_if_fail (state != NULL, FALSE);
  g_return_val_if_fail (login_ID != NULL, FALSE);

  for (i = 0; i < state->app->max_logins; i++)
    {
      if (state->logins[i].status == LOGIN_EMPTY)
	{
	  *login_ID = i;
	  return TRUE;
	}
    } 

  return FALSE;
}

static gboolean
expire_login (Login *login)
{
  ManagerState *state;
  gpointer      message;

  g_return_val_if_fail (login != NULL, FALSE);

  state = login->port->state;

  login->status = LOGIN_DEAD;

  message = wire_message_logout (login->login_ID);
  wire_message_send (state->writer, message, state->app);

  login->timeout = 0;

  return FALSE;
}

static void
bus_reset (SBP2Manager *manager,
           gpointer     user_data)
{
  Port         *port = user_data;
  ManagerState *state;
  EndpointApp  *app;
  gint          i;
  Login        *login;

  g_return_if_fail (manager != NULL);
  g_return_if_fail (port != NULL);

  state = port->state;
  app = state->app;

  for (i = 0; i < app->max_logins; i++)
    {
      login = &state->logins[i];

      if (login->port != port)
	continue;

      if (login->status == LOGIN_ACTIVE)
	{
	  login->status = LOGIN_RECONNECT;

	  login->timeout = util_timeout_add (app->context,
	                                     (SBP2_RECONNECT_HOLD + 1) * 1000,
					     (GSourceFunc) expire_login, login);
	}
      else if (login->status == LOGIN_PENDING)
	expire_login (login);
    }
}
