import { IIterator } from '@lumino/algorithm';
import { Poll } from '@lumino/polling';
import { ISignal } from '@lumino/signaling';
import { ServerConnection } from '..';
import * as Terminal from './terminal';
import { BaseManager } from '../basemanager';
/**
 * A terminal session manager.
 */
export declare class TerminalManager extends BaseManager implements Terminal.IManager {
    /**
     * Construct a new terminal manager.
     */
    constructor(options?: TerminalManager.IOptions);
    /**
     * The server settings of the manager.
     */
    readonly serverSettings: ServerConnection.ISettings;
    /**
     * Test whether the manager is ready.
     */
    get isReady(): boolean;
    /**
     * A promise that fulfills when the manager is ready.
     */
    get ready(): Promise<void>;
    /**
     * A signal emitted when the running terminals change.
     */
    get runningChanged(): ISignal<this, Terminal.IModel[]>;
    /**
     * A signal emitted when there is a connection failure.
     */
    get connectionFailure(): ISignal<this, Error>;
    /**
     * Dispose of the resources used by the manager.
     */
    dispose(): void;
    /**
     * Whether the terminal service is available.
     */
    isAvailable(): boolean;
    connectTo(options: Omit<Terminal.ITerminalConnection.IOptions, 'serverSettings'>): Terminal.ITerminalConnection;
    /**
     * Create an iterator over the most recent running terminals.
     *
     * @returns A new iterator over the running terminals.
     */
    running(): IIterator<Terminal.IModel>;
    /**
     * Force a refresh of the running terminals.
     *
     * @returns A promise that with the list of running terminals.
     *
     * #### Notes
     * This is intended to be called only in response to a user action,
     * since the manager maintains its internal state.
     */
    refreshRunning(): Promise<void>;
    /**
     * Create a new terminal session.
     *
     * @param options - The options used to create the terminal.
     *
     * @returns A promise that resolves with the terminal connection instance.
     *
     * #### Notes
     * The manager `serverSettings` will be used unless overridden in the
     * options.
     */
    startNew(options?: Terminal.ITerminal.IOptions): Promise<Terminal.ITerminalConnection>;
    /**
     * Shut down a terminal session by name.
     */
    shutdown(name: string): Promise<void>;
    /**
     * Shut down all terminal sessions.
     *
     * @returns A promise that resolves when all of the sessions are shut down.
     */
    shutdownAll(): Promise<void>;
    /**
     * Execute a request to the server to poll running terminals and update state.
     */
    protected requestRunning(): Promise<void>;
    /**
     * Handle a session starting.
     */
    private _onStarted;
    /**
     * Handle a session terminating.
     */
    private _onDisposed;
    private _isReady;
    private _names;
    private get _models();
    private _pollModels;
    private _terminalConnections;
    private _ready;
    private _runningChanged;
    private _connectionFailure;
}
/**
 * The namespace for TerminalManager statics.
 */
export declare namespace TerminalManager {
    /**
     * The options used to initialize a terminal manager.
     */
    interface IOptions extends BaseManager.IOptions {
        /**
         * When the manager stops polling the API. Defaults to `when-hidden`.
         */
        standby?: Poll.Standby;
    }
}
