/*  job_printreductioninfo.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_printreductioninfo.h"

#include "functions.h"
#include "rspoint.h"
#include "equationlist.h"
#include "files.h"
#include "filedata.h"
#include "sectormappings.h"
#include "yamlutils.h"
#include "filedata.h"

using namespace std;

namespace Reduze {

// register job types at JobFactory
namespace {
JobProxy<PrintReductionInfoFile> dummy1;
JobProxy<PrintReductionInfoSectors> dummy2;
}

void PrintReductionInfoFile::search_masters(
		const Sector& sector, const map<RSPoint, int>& num_unreduced,
		const set<INT>& rhsints) const {
	const list<RSFiniteGenericSelection>& range = check_for_masters;
	YAML::Emitter ye;
	ye << range;
	LOG("Checking " << sector << " for masters in range:\n" << ye.c_str()
			<< "\n");
	set<RSPoint> border;
	RSFiniteGenericSelection::find_upper_border(range, sector.t(), border);
	set<RSPoint>::const_iterator p;
	for (p = border.begin(); p != border.end(); ++p)
		if (num_unreduced.find(*p) == num_unreduced.end()
				|| num_unreduced.find(*p)->second != 0)
			break;
	bool border_reducible = (p == border.end());

	size_t num_masters = 0;
	bool interreduced = true;
	bool missing_rs = false;
	set<RSPoint> all;
	list<RSFiniteGenericSelection>::const_iterator r;
	for (r = range.begin(); r != range.end(); ++r)
		r->find_points(sector.t(), all);
	for (p = all.begin(); p != all.end(); ++p) {
		if (num_unreduced.find(*p) == num_unreduced.end()) {
			missing_rs = true;
			break;
		}
		int unred = num_unreduced.find(*p)->second;
		if (unred < 0)
			interreduced = false;
		num_masters += unred;
	}
	if (missing_rs) {
		LOG("Missing information for at least one (r,s) in the range");
	} else {
		LOG("Upper border is reducible:                "
				<< (border_reducible ? "true" : "false"));
		LOG("Range is reduced w.r.t. other identities: "
				<< (interreduced ? "true" : "false"));
		LOG("Number of unreduced integrals in range:   "
				<< num_masters);
		LOG("Masters check summary " << sector << ":"
		    << " bordred=" << (border_reducible ? "true" : "false")
			<< " interred=" << (interreduced ? "true" : "false")
			<< " masters=" << num_masters);
	}

	LOG("\nIntegrals appearing on r.h.s. of reductions in the range:");
	int num = 0;
	for (set<INT>::const_iterator i = rhsints.begin(); i != rhsints.end();) {
		++num;
		const Sector& sec = i->get_sector();
		if (++i == rhsints.end() || i->get_sector() != sec) {
			LOG(sec << ": " << num);
			num = 0;
		}
	}
	LOG("in total: " << rhsints.size());

	if (!masters_file.empty()) {
		LOG("\nWriting master candidates to file \'" << masters_file << "\'");
		OutFileINTs out(masters_file);
		for (set<INT>::const_iterator i = rhsints.begin(); i != rhsints.end(); ++i)
			out << *i;
		out.finalize();
	}
}

void PrintReductionInfoFile::run_serial() {
	LOG("\nAnalyzing reduction file " << input_file << "\n");

	long int num_eqs = 0;
	double coeff_len = 0.;
	map<Sector, map<pair<int, int> , set<INT> > > reduced_by_rs;
	map<Sector, int> s_max; // sector->s_max
	map<Sector, int> r_max; // sector->r_max
	map<Sector, map<pair<int, int> , bool> > is_interreduced;
	set<INT> reduced;
	map<Sector, set<INT> > rhsints;

	InFileEquations in(input_file);
	EquationHLight id;
	while (in.get(id)) {// load line by line
		INT i(id.max_INT());
		int r = i.r();
		int s = i.s();
		const Sector& sec = i.get_sector();
		reduced_by_rs[sec][make_pair(r, s)].insert(i);
		reduced.insert(i);
		set<INT> all;
		id.find_INT(all);
		for (set<INT>::const_iterator a = all.begin(); a != all.end(); ++a) {
			Sector sec = a->get_sector();
			s_max[sec] = max(s_max[sec], static_cast<int> (a->s()));
			r_max[sec] = max(r_max[sec], static_cast<int> (a->r()));
		}
		coeff_len += id.average_coeff_length();
		++num_eqs;
	}
	in.close();
	LOG("Number of equations: " << num_eqs);
	LOG("Average coefficient length [bytes]: " << fixed << setprecision(0)
			<< coeff_len / num_eqs << resetiosflags(ios::fixed));

	LOG("\nExtracting status of reduction for each r,s point");
	set<INT> unreduced;
	InFileEquations inagain(input_file);
	while (inagain.get(id)) {// load line by line
		INT i(id.max_INT());
		int r = i.r();
		int s = i.s();
		pair<int, int> rs(r, s);
		const Sector& sec(i.get_sector());
		set<INT> sub;
		id.find_subleading_INT(sub);
		is_interreduced[sec][rs] = true;
		for (set<INT>::const_iterator j = sub.begin(); j != sub.end(); ++j) {
			if (reduced.find(*j) != reduced.end())
				is_interreduced[sec][rs] = false;
			else
				unreduced.insert(*j);
			if (rsranges_contain(check_for_masters, i.t(), i))
				rhsints[sec].insert(*j);
		}
	}
	inagain.close();

	LOG("Showing statistics for reductions found in each sector");
	LOG("Number of sectors for which this file provides reductions: "
			<< reduced_by_rs.size());
	LOG("\n");
	LOG("First table shows the following entries:");
	LOG("A: Number of all integrals");
	LOG("B: Number of unreduced integrals");
	LOG("C: Number of different integrals");
	LOG("D: Number of different unreduced integrals");
	LOG("E: Identities are interreduced w.r.t. this file (.) or not (x)");
	LOG("");
	LOG("Second table shows different unreduced integrals in r-s diagram");

	map<Sector, map<pair<int, int> , set<INT> > >::const_iterator it;
	for (it = reduced_by_rs.begin(); it != reduced_by_rs.end(); ++it) {
		ostringstream plot_masters;
		const Sector& sector = it->first;
		const IntegralFamily* ic = sector.integralfamily();
		// Files* files = Files::instance();
		// const SectorMappings* mappings = files->sectormappings(ic->name());
		size_t n = 8;
		// size_t m = min((size_t) 12, to_string(sector).length() + 1);
		size_t m = to_string(sector).length() + 1;
		LOG("\n\n" << setw(m) << left << to_string(sector)
				<< setw(n) << right << "A" << setw(n) << "B"
				<< setw(n) << "C" << setw(n) << "D" << setw(n) << "E");
		LOG(setw(3 + m + 5 * n) << left << setfill('.') << "");
		LOGN(setfill(' '));
		map<RSPoint, int> num_unreduced_by_rs; // neg. val.: not fully reduced
		SeedGeneratorOptions seed_options;
		seed_options.set_reduce_to_equivalent(true);
		SeedGenerator seed_generator(ic, seed_options);
		for (int r = sector.t(); r <= r_max[sector]; ++r) {
			plot_masters << '\n' << setw(m) << left << "r=" + to_string(r)
					+ ":";
			for (int s = 0; s <= s_max[sector]; ++s) {
				const set<INT>& red = reduced_by_rs[sector][make_pair(r, s)];
				set<INT> red_equiv;
				set<INT>::const_iterator i;
				for (i = red.begin(); i != red.end(); ++i) {
					INT e(SectorMappings::get_equivalent_or_unique_zero(*i));
					if (e.get_sector() == sector && e.r() == r && e.s() == s)
						red_equiv.insert(e);
				}
				set<INT> seeds;
				seed_generator.generate(sector.id(), r, s, seeds);
				int num_equiv = seeds.size();
				int num = sector.get_num_integrals(r, s);
				bool inter = true;
				if (is_interreduced[sector].find(make_pair(r, s))
						!= is_interreduced[sector].end())
					inter = is_interreduced[sector][make_pair(r, s)];
				LOG(setw(m) << left << "r=" + to_string(r)
						+ ", s=" + to_string(s) + ":"
						<< setw(n) << right << num
						<< setw(n) << right << (num - static_cast<int>(red.size()))
						<< setw(n) << right << num_equiv
						<< setw(n) << right << (num_equiv - static_cast<int>(red_equiv.size()))
						<< setw(n) << right << (inter ? "." : "x"));
				size_t num_unreduced = num_equiv
						- static_cast<int> (red_equiv.size());
				num_unreduced_by_rs[RSPoint(r, s)] = //
						(inter ? num_unreduced : -num_unreduced);
				string ms = to_string(num_unreduced);
				ms += (inter ? '.' : 'x');
				plot_masters << setw(n) << right << ms;
			}
		}
		LOGN("\n" << setw(m) << left << to_string(sector));
		for (int s = 0; s <= s_max[sector]; ++s)
			LOGN(setw(n) << right << "s=" + to_string(s));
		LOGN("\n" << setw(3 + m + (s_max[sector] + 1)*n) << left << setfill('.') << "");
		LOGN(setfill(' '));
		LOG(plot_masters.str() << "\n\n");
		if (!check_for_masters.empty())
			search_masters(sector, num_unreduced_by_rs, rhsints[sector]);
	}
}

bool PrintReductionInfoFile::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	//find_dependencies_all_sectormappings(outothers, in, auxjobs);
	find_dependencies_reductions(outothers, in);
	in.push_back(input_file);
	if (!check_for_masters.empty() && !masters_file.empty())
		out.push_back(masters_file);
	return true;
}

void PrintReductionInfoFile::init() {
	if (input_file.empty())
		throw runtime_error("empty input file name");
}

std::string PrintReductionInfoFile::get_description() const {
	return "print reduction info: " + short_filename(input_file);
}

// PrintReductionInfoSectors

bool PrintReductionInfoSectors::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	set<Sector> sectors;
	find_dependencies_all_sectormappings(outothers, in, auxjobs);
	try {
		sectors = sector_selection.find_sectors();
	} catch (exception& e) {
		return false;
	}
	for (set<Sector>::const_iterator s = sectors.begin(); s != sectors.end(); ++s) {
		PrintReductionInfoFile *job = new PrintReductionInfoFile;
		job->input_file = Files::instance()->get_filename_sectorreduction(*s);
		job->check_for_masters = check_for_masters;
		if (!check_for_masters.empty() && !masters_file_prefix.empty())
			job->masters_file = masters_file_prefix +
				get_filename_without_directory(job->input_file) + "-rhs";
		auxjobs.push_back(job);
		//in.push_back(Files::instance()->get_filename_sectorreduction(*s));
	}
	return true;
}

std::string PrintReductionInfoSectors::get_description() const {
	return "print reduction info for sectors";
}

}
